// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_feedback_buttons/cr_feedback_buttons.js';
import './auto_tab_groups_group.js';
import './auto_tab_groups_results_actions.js';
import { CrFeedbackOption } from 'chrome://resources/cr_elements/cr_feedback_buttons/cr_feedback_buttons.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './auto_tab_groups_results.css.js';
import { getHtml } from './auto_tab_groups_results.html.js';
const MINIMUM_SCROLLABLE_MAX_HEIGHT = 204;
const NON_SCROLLABLE_VERTICAL_SPACING = 198;
export class AutoTabGroupsResultsElement extends CrLitElement {
    static get is() {
        return 'auto-tab-groups-results';
    }
    static get properties() {
        return {
            session: { type: Object },
            availableHeight: { type: Number },
            feedbackSelectedOption_: { type: Number },
        };
    }
    #session_accessor_storage;
    get session() { return this.#session_accessor_storage; }
    set session(value) { this.#session_accessor_storage = value; }
    #availableHeight_accessor_storage = 0;
    get availableHeight() { return this.#availableHeight_accessor_storage; }
    set availableHeight(value) { this.#availableHeight_accessor_storage = value; }
    #feedbackSelectedOption__accessor_storage = CrFeedbackOption.UNSPECIFIED;
    get feedbackSelectedOption_() { return this.#feedbackSelectedOption__accessor_storage; }
    set feedbackSelectedOption_(value) { this.#feedbackSelectedOption__accessor_storage = value; }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('session')) {
            const changedSession = changedProperties.get('session');
            if (changedSession &&
                (!this.session ||
                    changedSession.sessionId !== this.session.sessionId)) {
                this.feedbackSelectedOption_ = CrFeedbackOption.UNSPECIFIED;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('availableHeight')) {
            this.onAvailableHeightChange_();
        }
        if (changedProperties.has('session')) {
            this.updateScroll_();
        }
    }
    firstUpdated() {
        this.$.scrollable.addEventListener('scroll', this.updateScroll_.bind(this));
    }
    focusInput() {
        const group = this.shadowRoot.querySelector('auto-tab-groups-group');
        if (!group) {
            return;
        }
        group.focusInput();
    }
    updateScroll_() {
        const scrollable = this.$.scrollable;
        scrollable.classList.toggle('can-scroll', scrollable.clientHeight < scrollable.scrollHeight);
        scrollable.classList.toggle('is-scrolled', scrollable.scrollTop > 0);
        scrollable.classList.toggle('scrolled-to-bottom', scrollable.scrollTop + this.getMaxScrollableHeight_() >=
            scrollable.scrollHeight);
    }
    getOrganizations_() {
        if (!this.session) {
            return [];
        }
        return this.session.organizations;
    }
    hasMultipleOrganizations_() {
        return this.getOrganizations_().length > 1;
    }
    getName_(organization) {
        return organization.name;
    }
    getMaxScrollableHeight_() {
        return Math.max(MINIMUM_SCROLLABLE_MAX_HEIGHT, (this.availableHeight - NON_SCROLLABLE_VERTICAL_SPACING));
    }
    onAvailableHeightChange_() {
        const maxHeight = this.getMaxScrollableHeight_();
        this.$.scrollable.style.maxHeight = maxHeight + 'px';
        this.updateScroll_();
    }
    onCreateAllGroupsClick_(event) {
        event.stopPropagation();
        event.preventDefault();
        const groups = [...this.shadowRoot.querySelectorAll('auto-tab-groups-group')];
        const organizations = groups.map((group) => {
            return {
                organizationId: group.organizationId,
                tabs: group.tabs,
            };
        });
        this.fire('create-all-groups-click', { organizations });
    }
    onLearnMoreClick_() {
        this.fire('learn-more-click');
    }
    onLearnMoreKeyDown_(event) {
        if (event.key === 'Enter') {
            this.onLearnMoreClick_();
        }
    }
    onFeedbackKeyDown_(event) {
        if ((event.key !== 'ArrowLeft' && event.key !== 'ArrowRight')) {
            return;
        }
        const feedbackButtons = this.$.feedbackButtons.shadowRoot.querySelectorAll(`cr-icon-button`);
        const focusableElements = [
            this.$.learnMore,
            feedbackButtons[0],
            feedbackButtons[1],
        ];
        const focusableElementCount = focusableElements.length;
        const focusedIndex = focusableElements.findIndex((element) => element.matches(':focus'));
        if (focusedIndex < 0) {
            return;
        }
        let nextFocusedIndex = 0;
        if (event.key === 'ArrowLeft') {
            nextFocusedIndex =
                (focusedIndex + focusableElementCount - 1) % focusableElementCount;
        }
        else if (event.key === 'ArrowRight') {
            nextFocusedIndex = (focusedIndex + 1) % focusableElementCount;
        }
        focusableElements[nextFocusedIndex].focus();
    }
    onFeedbackSelectedOptionChanged_(event) {
        this.feedbackSelectedOption_ = event.detail.value;
        this.fire('feedback', { value: this.feedbackSelectedOption_ });
    }
}
customElements.define(AutoTabGroupsResultsElement.is, AutoTabGroupsResultsElement);
