// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import {ReadAloudNode} from './read_aloud_types.js';
import type {WordBoundaryState} from './word_boundaries.js';

export enum PauseActionSource {
  DEFAULT,
  BUTTON_CLICK,
  VOICE_PREVIEW,
  VOICE_SETTINGS_CHANGE,
  ENGINE_INTERRUPT,
  SPEECH_FINISHED,
}

export interface SpeechPlayingState {
  // True when the user presses play, regardless of if audio has actually
  // started yet. This will be false when speech is paused.
  isSpeechActive: boolean;
  // When `isSpeechActive` is false, this indicates how it became false. e.g.
  // via pause button click or because other speech settings were changed.
  pauseSource: PauseActionSource;
  // Indicates that audio is currently playing.
  // When a user presses the play button, isSpeechActive becomes true, but
  // `isAudioCurrentlyPlaying` will tell us whether audio actually started
  // playing yet. This is a separate state because audio starting has a delay.
  isAudioCurrentlyPlaying: boolean;
  // Indicates if speech has been triggered on the current page by a play
  // button press. This will be true throughout the lifetime of reading
  // the content on the page. It will only be reset when speech has completely
  // stopped from reaching the end of content or changing pages. Pauses will
  // not update it.
  hasSpeechBeenTriggered: boolean;
  // If we're in the middle of repositioning speech, as this could cause a
  // this.speech_.cancel() that shouldn't update the UI for the speech playing
  // state.
  isSpeechBeingRepositioned: boolean;
}

export enum SpeechEngineState {
  NONE,
  LOADING,
  LOADED,
}

interface ReadingPosition {
  node: ReadAloudNode;
  offset: number;
}

export class SpeechModel {
  private speechPlayingState_: SpeechPlayingState = {
    isSpeechActive: false,
    pauseSource: PauseActionSource.DEFAULT,
    isAudioCurrentlyPlaying: false,
    hasSpeechBeenTriggered: false,
    isSpeechBeingRepositioned: false,
  };

  private speechEngineState_: SpeechEngineState = SpeechEngineState.NONE;
  private previewVoicePlaying_: SpeechSynthesisVoice|null = null;

  // With minor page changes, we redistill or redraw sometimes and end up losing
  // our reading position if read aloud has started. This keeps track of the
  // last position so we can check if it's still in the new page.
  private lastReadingPosition_: ReadingPosition|null = null;
  private savedSpeechPlayingState_: SpeechPlayingState|null = null;
  private savedWordBoundaryState_: WordBoundaryState|null = null;

  // Used for logging play time.
  private playSessionStartTime_: number|null = null;
  // Used to log the number of words heard by a user via read aloud on a given
  // page.
  private wordsHeard_: number = 0;

  // The context node used to initialize Read Aloud. This is null if it has
  // not been set. When TS-based segmentation is enabled, this is the root
  // node, and when V8-based segmentation is enabled, this is the first text
  // node.
  private readAloudContextNode_: ReadAloudNode|null|undefined = null;

  private resumeSpeechOnVoiceMenuClose_: boolean = false;
  private speechVolume_: number = 1.0;

  setVolume(volume: number): void {
    this.speechVolume_ = volume;
  }

  getVolume(): number {
    return this.speechVolume_;
  }

  getSavedSpeechPlayingState(): SpeechPlayingState|null {
    return this.savedSpeechPlayingState_;
  }

  setSavedSpeechPlayingState(state: SpeechPlayingState|null): void {
    this.savedSpeechPlayingState_ = state;
  }

  getSavedWordBoundaryState(): WordBoundaryState|null {
    return this.savedWordBoundaryState_;
  }

  setSavedWordBoundaryState(state: WordBoundaryState|null): void {
    this.savedWordBoundaryState_ = state;
  }

  getResumeSpeechOnVoiceMenuClose(): boolean {
    return this.resumeSpeechOnVoiceMenuClose_;
  }

  setResumeSpeechOnVoiceMenuClose(shouldResume: boolean) {
    this.resumeSpeechOnVoiceMenuClose_ = shouldResume;
  }

  getContextNode(): ReadAloudNode|null|undefined {
    return this.readAloudContextNode_;
  }

  setContextNode(node: Node|null) {
    if (!node) {
      this.readAloudContextNode_ = node;
      return;
    }

    this.readAloudContextNode_ = ReadAloudNode.create(node);
  }

  getPlaySessionStartTime(): number|null {
    return this.playSessionStartTime_;
  }

  setPlaySessionStartTime(time: number|null): void {
    this.playSessionStartTime_ = time;
  }

  getLastPosition(): ReadingPosition|null {
    return this.lastReadingPosition_;
  }

  setLastPosition(position: ReadingPosition|null) {
    this.lastReadingPosition_ = position;
  }

  getEngineState(): SpeechEngineState {
    return this.speechEngineState_;
  }

  setEngineState(state: SpeechEngineState): void {
    this.speechEngineState_ = state;
  }

  getPreviewVoicePlaying(): SpeechSynthesisVoice|null {
    return this.previewVoicePlaying_;
  }

  setPreviewVoicePlaying(voice: SpeechSynthesisVoice|null) {
    this.previewVoicePlaying_ = voice;
  }

  getState(): SpeechPlayingState {
    return this.speechPlayingState_;
  }

  setState(state: SpeechPlayingState): void {
    this.speechPlayingState_ = {...state};
  }

  isSpeechActive(): boolean {
    return this.speechPlayingState_.isSpeechActive;
  }

  setIsSpeechActive(value: boolean): void {
    this.speechPlayingState_.isSpeechActive = value;
  }

  getPauseSource(): PauseActionSource {
    return this.speechPlayingState_.pauseSource;
  }

  setPauseSource(value: PauseActionSource): void {
    this.speechPlayingState_.pauseSource = value;
  }

  isAudioCurrentlyPlaying(): boolean {
    return this.speechPlayingState_.isAudioCurrentlyPlaying;
  }

  setIsAudioCurrentlyPlaying(value: boolean): void {
    this.speechPlayingState_.isAudioCurrentlyPlaying = value;
  }

  hasSpeechBeenTriggered(): boolean {
    return this.speechPlayingState_.hasSpeechBeenTriggered;
  }

  setHasSpeechBeenTriggered(value: boolean): void {
    this.speechPlayingState_.hasSpeechBeenTriggered = value;
  }

  isSpeechBeingRepositioned(): boolean {
    return this.speechPlayingState_.isSpeechBeingRepositioned;
  }

  setIsSpeechBeingRepositioned(value: boolean): void {
    this.speechPlayingState_.isSpeechBeingRepositioned = value;
  }

  getWordsHeard(): number {
    return this.wordsHeard_;
  }

  setWordsHeard(words: number): void {
    this.wordsHeard_ = words;
  }

  incrementWordsHeard(): void {
    this.wordsHeard_++;
  }
}
