// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'settings-import-data-dialog' is a component for importing
 * bookmarks and other data from other sources.
 */
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_dialog/cr_dialog.js';
import 'chrome://resources/cr_elements/cr_icon/cr_icon.js';
import 'chrome://resources/cr_elements/cr_spinner_style.css.js';
import 'chrome://resources/cr_elements/icons.html.js';
import 'chrome://resources/cr_elements/md_select.css.js';
import '../controls/settings_checkbox.js';
import '../controls/settings_toggle_button.js';
import '../settings_vars.css.js';
import '../i18n_setup.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { I18nMixin } from 'chrome://resources/cr_elements/i18n_mixin.js';
import { WebUiListenerMixin } from 'chrome://resources/cr_elements/web_ui_listener_mixin.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { ImportDataBrowserProxyImpl, ImportDataStatus } from './import_data_browser_proxy.js';
import { getTemplate } from './import_data_dialog.html.js';
const SettingsImportDataDialogElementBase = WebUiListenerMixin(I18nMixin(PrefsMixin(PolymerElement)));
export class SettingsImportDataDialogElement extends SettingsImportDataDialogElementBase {
    static get is() {
        return 'settings-import-data-dialog';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            browserProfiles_: Array,
            selected_: {
                type: Object,
                observer: 'updateImportDataTypesSelected_',
            },
            /**
             * Whether none of the import data categories is selected.
             */
            noImportDataTypeSelected_: {
                type: Boolean,
                value: false,
            },
            importStatus_: {
                type: String,
                value: ImportDataStatus.INITIAL,
            },
            /**
             * Mirroring the enum so that it can be used from HTML bindings.
             */
            importStatusEnum_: {
                type: Object,
                value: ImportDataStatus,
            },
        };
    }
    browserProxy_ = ImportDataBrowserProxyImpl.getInstance();
    ready() {
        super.ready();
        this.addEventListener('settings-boolean-control-change', this.updateImportDataTypesSelected_);
    }
    connectedCallback() {
        super.connectedCallback();
        this.browserProxy_.initializeImportDialog().then(data => {
            if (!this.isConnected) {
                // Element was disconnected while waiting for the backend call to
                // return. Do nothing.
                return;
            }
            this.browserProfiles_ = data;
            this.selected_ = this.browserProfiles_[0];
            // Show the dialog only after the browser profiles data is populated
            // to avoid UI flicker.
            this.$.dialog.showModal();
        });
        this.addWebUiListener('import-data-status-changed', (importStatus) => {
            this.importStatus_ = importStatus;
            if (this.hasImportStatus_(ImportDataStatus.FAILED)) {
                this.closeDialog_();
            }
        });
    }
    getProfileDisplayName_(name, profileName) {
        return profileName ? `${name} - ${profileName}` : name;
    }
    updateImportDataTypesSelected_() {
        const checkboxes = this.shadowRoot.querySelectorAll('settings-checkbox[checked]:not([hidden])');
        this.noImportDataTypeSelected_ = checkboxes.length === 0;
    }
    /**
     * @return Whether |status| is the current status.
     */
    hasImportStatus_(status) {
        return this.importStatus_ === status;
    }
    isImportFromFileSelected_() {
        // The last entry in |browserProfiles_| always refers to dummy profile for
        // importing from a bookmarks file.
        return this.selected_.index === this.browserProfiles_.length - 1;
    }
    getActionButtonText_() {
        return this.i18n(this.isImportFromFileSelected_() ? 'importChooseFile' : 'importCommit');
    }
    onBrowserProfileSelectionChange_() {
        this.selected_ = this.browserProfiles_[this.$.browserSelect.selectedIndex];
    }
    onActionButtonClick_() {
        const checkboxes = this.shadowRoot.querySelectorAll('settings-checkbox');
        if (this.isImportFromFileSelected_()) {
            this.browserProxy_.importFromBookmarksFile();
        }
        else {
            const types = {};
            checkboxes.forEach(checkbox => {
                types[checkbox.pref.key] = checkbox.checked && !checkbox.hidden;
            });
            this.browserProxy_.importData(this.$.browserSelect.selectedIndex, types);
        }
        checkboxes.forEach(checkbox => checkbox.sendPrefChange());
    }
    closeDialog_() {
        this.$.dialog.close();
    }
    /**
     * @return Whether the import button should be disabled.
     */
    shouldDisableImport_() {
        return this.hasImportStatus_(ImportDataStatus.IN_PROGRESS) ||
            this.noImportDataTypeSelected_;
    }
}
customElements.define(SettingsImportDataDialogElement.is, SettingsImportDataDialogElement);
