// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'settings-autofill-entries-list-element' contains configuration
 * options for Autofill AI.
 */
import 'chrome://resources/cr_elements/cr_action_menu/cr_action_menu.js';
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_icon/cr_icon.js';
import 'chrome://resources/cr_elements/cr_icons.css.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_lazy_render/cr_lazy_render.js';
import 'chrome://resources/cr_elements/cr_shared_style.css.js';
import '/shared/settings/prefs/prefs.js';
import 'chrome://resources/cr_elements/icons.html.js';
import '../icons.html.js';
import '../settings_shared.css.js';
import '../simple_confirmation_dialog.js';
import './autofill_ai_add_or_edit_dialog.js';
// 
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { AnchorAlignment } from 'chrome://resources/cr_elements/cr_action_menu/cr_action_menu.js';
import { I18nMixin } from 'chrome://resources/cr_elements/i18n_mixin.js';
import { WebUiListenerMixin } from 'chrome://resources/cr_elements/web_ui_listener_mixin.js';
import { assert } from 'chrome://resources/js/assert.js';
import { OpenWindowProxyImpl } from 'chrome://resources/js/open_window_proxy.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { loadTimeData } from '../i18n_setup.js';
import { SettingsViewMixin } from '../settings_page/settings_view_mixin.js';
import { getTemplate } from './autofill_ai_entries_list.html.js';
import { EntityDataManagerProxyImpl } from './entity_data_manager_proxy.js';
const SettingsAutofillAiEntriesListElementBase = SettingsViewMixin(WebUiListenerMixin(I18nMixin(PrefsMixin(PolymerElement))));
export class SettingsAutofillAiEntriesListElement extends SettingsAutofillAiEntriesListElementBase {
    static get is() {
        return 'settings-autofill-ai-entries-list-element';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
               If a user is not eligible for Autofill with Ai, but they have data
               saved, the code allows them only to edit and delete their data. They
               are not allowed to add new data, or to opt-in or opt-out of Autofill
               with Ai using the toggle at the top of this page.
               If a user is not eligible for Autofill with Ai and they also have no
               data saved, then they cannot access this page at all.
             */
            ineligibleUser: {
                type: Boolean,
                value() {
                    return !loadTimeData.getBoolean('userEligibleForAutofillAi');
                },
            },
            allowEditing_: {
                type: Object,
                value: false,
            },
            /**
               The corresponding `EntityInstance` model for any entity instance
               related action menus or dialogs.
             */
            activeEntityInstance_: {
                type: Object,
                value: null,
            },
            /**
               Complete list of entity types that exist. When the user wants to add a
               new entity instance, this list is displayed.
             */
            completeEntityTypesList_: {
                type: Array,
                value: () => [],
            },
            /**
               The same dialog can be used for both adding and editing entity
               instances.
             */
            showAddOrEditEntityInstanceDialog_: {
                type: Boolean,
                value: false,
            },
            addOrEditEntityInstanceDialogTitle_: {
                type: String,
                value: '',
            },
            showRemoveEntityInstanceDialog_: {
                type: Boolean,
                value: false,
            },
            entityInstances_: {
                type: Array,
                value: () => [],
            },
        };
    }
    static get observers() {
        return [
            'onAutofillAiPrefChanged_(' +
                'prefs.autofill.profile_enabled.value)',
            'onOptInStatusChanged_(' +
                'prefs.autofill.autofill_ai.opt_in_status.value)',
        ];
    }
    entityInstancesChangedListener_ = null;
    entityDataManager_ = EntityDataManagerProxyImpl.getInstance();
    connectedCallback() {
        super.connectedCallback();
        this.entityDataManager_.getOptInStatus().then(optedIn => this.allowEditing_ = !this.ineligibleUser && optedIn);
        this.entityInstancesChangedListener_ = (entityInstances => {
            this.entityInstances_ =
                entityInstances.sort(this.entityInstancesWithLabelsComparator_);
        });
        this.entityDataManager_.addEntityInstancesChangedListener(this.entityInstancesChangedListener_);
        this.entityDataManager_.getWritableEntityTypes().then((entityTypes) => {
            this.completeEntityTypesList_ =
                entityTypes.sort(this.entityTypesComparator_);
        });
        this.entityDataManager_.loadEntityInstances().then((entityInstances) => {
            this.entityInstances_ =
                entityInstances.sort(this.entityInstancesWithLabelsComparator_);
        });
        this.addWebUiListener('sync-status-changed', this.onSyncStatusChanged_.bind(this));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        assert(this.entityInstancesChangedListener_);
        this.entityDataManager_.removeEntityInstancesChangedListener(this.entityInstancesChangedListener_);
        this.entityInstancesChangedListener_ = null;
    }
    /*
     * This comparator purposefully uses sensitivity 'base', not to differentiate
     * between different capitalization or diacritics.
     */
    entityTypesComparator_(a, b) {
        if (a.supportsWalletStorage !== b.supportsWalletStorage) {
            return a.supportsWalletStorage ? 1 : -1;
        }
        return a.typeNameAsString.localeCompare(b.typeNameAsString, undefined, { sensitivity: 'base' });
    }
    /**
     * This comparator compares the labels alphabetically, and, in case of
     * equality, the sublabels.
     * This comparator purposefully uses sensitivity 'base', not to differentiate
     * between different capitalization or diacritics.
     */
    entityInstancesWithLabelsComparator_(a, b) {
        return (a.entityInstanceLabel + a.entityInstanceSubLabel)
            .localeCompare(b.entityInstanceLabel + b.entityInstanceSubLabel, undefined, { sensitivity: 'base' });
    }
    /**
     * Handles tapping on the "Add" entity instance button.
     */
    onAddEntityInstanceClick_(e) {
        const addButton = e.target;
        this.$.addMenu.get().showAt(addButton, {
            anchorAlignmentX: AnchorAlignment.BEFORE_END,
            anchorAlignmentY: AnchorAlignment.AFTER_END,
            noOffset: true,
        });
    }
    onAddEntityInstanceFromDropdownClick_(e) {
        e.preventDefault();
        // Create a new entity instance with no attribute instances and guid. A guid
        // will be assigned after saving, on the C++ side.
        this.activeEntityInstance_ = {
            type: e.model.item,
            attributeInstances: [],
            guid: '',
            nickname: '',
        };
        this.addOrEditEntityInstanceDialogTitle_ =
            this.activeEntityInstance_.type.addEntityTypeString;
        this.showAddOrEditEntityInstanceDialog_ = true;
        this.$.addMenu.get().close();
    }
    /**
     * Open the action menu.
     */
    async onMoreButtonClick_(e) {
        const moreButton = e.target;
        this.activeEntityInstance_ =
            await this.entityDataManager_.getEntityInstanceByGuid(e.model.item.guid);
        this.$.actionMenu.get().showAt(moreButton);
    }
    /**
     * Handles tapping on the "Edit" entity instance button in the action menu.
     */
    onMenuEditEntityInstanceClick_(e) {
        e.preventDefault();
        assert(this.activeEntityInstance_);
        this.addOrEditEntityInstanceDialogTitle_ =
            this.activeEntityInstance_.type.editEntityTypeString;
        this.showAddOrEditEntityInstanceDialog_ = true;
        this.$.actionMenu.get().close();
    }
    /**
     * Handles tapping on the "Delete" entity instance button in the action menu.
     */
    onMenuRemoveEntityInstanceClick_(e) {
        e.preventDefault();
        this.showRemoveEntityInstanceDialog_ = true;
        this.$.actionMenu.get().close();
    }
    onAutofillAiAddOrEditDone_(e) {
        e.stopPropagation();
        this.entityDataManager_.addOrUpdateEntityInstance(e.detail);
    }
    onAddOrEditEntityInstanceDialogClose_(e) {
        e.stopPropagation();
        this.showAddOrEditEntityInstanceDialog_ = false;
        this.activeEntityInstance_ = null;
    }
    onRemoveEntityInstanceDialogClose_() {
        const wasDeletionConfirmed = this.shadowRoot
            .querySelector('#removeEntityInstanceDialog').wasConfirmed();
        if (wasDeletionConfirmed) {
            assert(this.activeEntityInstance_);
            this.entityDataManager_.removeEntityInstance(this.activeEntityInstance_.guid);
        }
        this.showRemoveEntityInstanceDialog_ = false;
        this.activeEntityInstance_ = null;
    }
    // Adjusts the opt-in state when address autofill status changes.
    //
    // This covers the case where a user disables address autofill and then checks
    // the AutofillAI opt-in status. In this case, we do not remove the AutofillAI
    // entry, but just set the opt-in to false. Note that other
    // preconditions (e.g., sync) are not covered.
    async onAutofillAiPrefChanged_(prefValue) {
        const optedIn = await this.entityDataManager_.getOptInStatus();
        this.allowEditing_ = !this.ineligibleUser && optedIn && prefValue;
    }
    onRemoteWalletPassesLinkClick_() {
        OpenWindowProxyImpl.getInstance().openUrl(loadTimeData.getString('walletPassesPageUrl'));
    }
    async onOptInStatusChanged_() {
        const optedIn = await this.entityDataManager_.getOptInStatus();
        this.allowEditing_ = !this.ineligibleUser && optedIn;
    }
    // Refreshes the entity types list when the sync status changes.
    //
    // Updates the list to reflect whether the user is signed in (allowing the
    // creation of entity instances for types stored on the server) or signed
    // out (disallowing it).
    onSyncStatusChanged_(_) {
        this.entityDataManager_.getWritableEntityTypes().then((entityTypes) => {
            this.completeEntityTypesList_ =
                entityTypes.sort(this.entityTypesComparator_);
        });
    }
}
customElements.define(SettingsAutofillAiEntriesListElement.is, SettingsAutofillAiEntriesListElement);
