// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '/strings.m.js';
import 'chrome://resources/cr_elements/cr_menu_selector/cr_menu_selector.js';
import 'chrome://resources/cr_elements/cr_toolbar/cr_toolbar.js';
import '//resources/cr_elements/icons.html.js';
import './color_data.js';
import { ColorChangeUpdater } from 'chrome://resources/cr_components/color_change_listener/colors_css_updater.js';
import { CrContainerShadowMixinLit } from 'chrome://resources/cr_elements/cr_container_shadow_mixin_lit.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
import { ALL_SECTIONS } from './color_data.js';
const CSS_PREFIX = '--color-sys-';
const CC_PREFIX = 'kColorSys';
const ColorPipelineInternalsAppElementBase = CrContainerShadowMixinLit(CrLitElement);
export class ColorPipelineInternalsAppElement extends ColorPipelineInternalsAppElementBase {
    static get is() {
        return 'color-pipeline-internals-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Substring filter that (when set) shows only entries containing
             * `filter`.
             */
            currentColor_: { type: String },
            filter_: { type: String },
            narrow_: { type: Boolean },
        };
    }
    #currentColor__accessor_storage = '';
    get currentColor_() { return this.#currentColor__accessor_storage; }
    set currentColor_(value) { this.#currentColor__accessor_storage = value; }
    #filter__accessor_storage = '';
    get filter_() { return this.#filter__accessor_storage; }
    set filter_(value) { this.#filter__accessor_storage = value; }
    #narrow__accessor_storage = false;
    get narrow_() { return this.#narrow__accessor_storage; }
    set narrow_(value) { this.#narrow__accessor_storage = value; }
    sections_ = ALL_SECTIONS;
    red_ = '';
    green_ = '';
    blue_ = '';
    alpha_ = '';
    cssName_ = '';
    cppName_ = '';
    firstUpdated() {
        ColorChangeUpdater.forDocument().start();
    }
    onSearchChanged_(e) {
        this.filter_ = e.detail.toLowerCase();
    }
    entryFilter_(entry) {
        return this.filter_ === '' ||
            entry.background.toLowerCase().includes(this.filter_) ||
            !!(entry.foreground?.toLowerCase().includes(this.filter_));
    }
    /**
     * Prevent clicks on sidebar items from navigating.
     */
    onLinkClick_(event) {
        event.preventDefault();
    }
    onSelectorActivate_(event) {
        const url = event.detail.selected;
        this.$.menu.selected = url;
        const idx = url.lastIndexOf('#');
        const el = this.$.content.querySelector(url.substring(idx));
        el.scrollIntoView(true);
    }
    onNarrowChanged_(e) {
        this.narrow_ = e.detail.value;
    }
    getEntryName_(entry) {
        // For display purposes only, strip the prefix.
        let name = entry.background;
        if (name.startsWith(CSS_PREFIX)) {
            name = name.substring(CSS_PREFIX.length);
        }
        return name;
    }
    getEntryStyle_(entry) {
        const foreground = entry.foreground ?? '--color-sys-on-surface';
        return `background-color: var(${entry.background});` +
            ` color: var(${foreground});` +
            ` border: 1px solid var(${foreground});`;
    }
    updateColorInfo_(e) {
        const el = e.target;
        this.currentColor_ = el.querySelector('p').innerText;
        this.cssName_ = CSS_PREFIX + this.currentColor_;
        this.cppName_ = CC_PREFIX +
            this.currentColor_.substring(0, 1).toUpperCase() +
            this.currentColor_.substring(1).replace(/-./g, x => x.substring(1).toUpperCase());
        const colorText = getComputedStyle(el).backgroundColor;
        const toHex = (v) => {
            const result = parseInt(v).toString(16).toUpperCase();
            return (result.length === 1 ? '0' : '') + result;
        };
        let m = colorText.match(/^rgb\s*\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)$/i);
        if (m) {
            this.red_ = `#${toHex(m[1])} / ${m[1]}`;
            this.green_ = `#${toHex(m[2])} / ${m[2]}`;
            this.blue_ = `#${toHex(m[3])} / ${m[3]}`;
            this.alpha_ = '#FF / 255';
        }
        else {
            m = colorText.match(/^rgba\s*\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)$/i);
            if (m) {
                this.red_ = `#${toHex(m[1])} / ${m[1]}`;
                this.green_ = `#${toHex(m[2])} / ${m[2]}`;
                this.blue_ = `#${toHex(m[3])} / ${m[3]}`;
                this.alpha_ = `#${toHex(m[4])} / ${m[4]}`;
                return;
            }
            else {
                this.red_ = colorText;
                this.green_ = '';
                this.blue_ = '';
                this.alpha_ = '';
            }
        }
    }
    clearColorInfo_() {
        this.currentColor_ = '';
    }
}
customElements.define(ColorPipelineInternalsAppElement.is, ColorPipelineInternalsAppElement);
