/**
 * @fileoverview
 * Allows a WebUI page to report the visibility and bounds of HTML elements to
 * the browser process. This is the frontend counterpart to
 * ui::TrackedElementWebUI.
 *
 * The primary use case is to anchor secondary UIs (e.g. App Menu) to a HTML
 * element in WebUI.
 * TODO(crbug.com/40243115): Use TrackedElementManager in Help Bubbles.
 *
 * ## Usage
 *
 * In C++, declare your ui::ElementIdentifier. Make sure it is registered as a
 * known identifier.
 *
 * ```C++
 * DECLARE_ELEMENT_IDENTIFIER_VALUE(kMyElementIdentifier);
 * // TODO(crbug.com/40243115): explain how to register as a known identifier.
 * ```
 *
 * In your WebUI component:
 *
 * 1.  Get the singleton instance of `TrackedElementManager` in your component
 *     class.
 *
 *     ```ts
 *     // in your component class:
 *     private trackedElementManager: TrackedElementManager;
 *
 *     constructor() {
 *       super();
 *       this.trackedElementManager = TrackedElementManager.getInstance();
 *       // ...
 *     }
 *     ```
 *
 * 2.  Call `startTracking()` to begin tracking an element. You probably want
 *     to do this in `connectedCallback()`.
 *
 *     ```ts
 *     override connectedCallback() {
 *       super.connectedCallback();
 *     this.trackedElementManager_.startTracking(
 *         this.$.myElement,
 *         'kMyElementIdentifier',
 *         {...options});
 *     ```
 *
 *     The first parameter is the HTMLElement to track. The second is the
 *     string identifier name that C++ uses. The third is an optional
 *     options object. See `Options` in this file.
 *
 * 3.  Call `stopTracking()` to stop tracking an element.
 *
 *     ```ts
 *     this.trackedElementManager_.stopTracking(this.$.myElement);
 *     ```
 *
 * 4.  To report that the user has activated an element (e.g. by clicking on
 *     it), call `notifyElementActivated()`.
 *
 *     ```ts
 *     this.trackedElementManager_.notifyElementActivated(this.$.myElement);
 *     ```
 *
 * 5.  To report a custom event, call `notifyCustomEvent()`.
 *
 *     ```ts
 *     this.trackedElementManager_.notifyCustomEvent(
 *         this.$.myElement,
 *         'my-custom-event-name');
 *     ```
 */
import type { RectF } from '//resources/mojo/ui/gfx/geometry/mojom/geometry.mojom-webui.js';
/**
 * Options for `TrackedElementManager.startTracking()`.
 */
export interface Options {
    /**
     * Padding added to the element bounds.
     * These values are clamped in the range [0, 20].
     */
    paddingTop?: number;
    paddingLeft?: number;
    paddingBottom?: number;
    paddingRight?: number;
    /**
     * Set this to true if the element is fixed positioned.
     * By default, this class detects tracked elements when they are rendered
     * within the document. This breaks with fix-positioned elements since they
     * are not in the regular flow of the document but they are always visible.
     */
    fixed?: boolean;
}
export declare class TrackedElementManager {
    private static instance_;
    static getInstance(): TrackedElementManager;
    private trackedElementHandler_;
    private trackedElements_;
    private fixedElementObserver_;
    private resizeObserver_;
    private debouncedUpdateAllBoundsCallback_;
    private constructor();
    reset(): void;
    /**
     * Starts tracking an element.
     * A visibility update event will be sent immediately.
     *
     * @param element The element to track.
     * @param nativeId The ElementIdentifier name that C++ uses.
     * @param options Optional options. See `Options` in this file.
     * @param onVisibilityChanged Optional callback that is called when the
     *     visibility of the element changes. The callback is called with two
     *     parameters:
     *       - visible: Whether the element is visible.
     *       - bounds: The bounds of the element.
     */
    startTracking(element: HTMLElement, nativeId: string, options?: Options, onVisibilityChanged?: (visible: boolean, bounds: RectF) => void): void;
    /**
     * Stops tracking an element.
     * A visibility event with `visible: false` will be sent immediately.
     *
     * @param element The element to stop tracking.
     */
    stopTracking(element: HTMLElement): void;
    notifyElementActivated(element: HTMLElement): void;
    notifyCustomEvent(element: HTMLElement, customEventName: string): void;
    private onElementVisibilityChanged_;
    private updateAllBounds_;
    private getElementBounds_;
}
