// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_VECTOR_ICONS_VECTOR_ICONS_UNITTEST_H_
#define COMPONENTS_VECTOR_ICONS_VECTOR_ICONS_UNITTEST_H_

// Define a macro to stamp out the test cases. We do this via a dynamically
// generated header vs generating the cc file directly so that the cc file can
// live in the repo and can provide a test location to infra that needs it.

#define VECTOR_ICON_TEST_CASES \
TEST_F(VectorIconsTest, ParseProductIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 12, 7.5,\n" \
"R_H_LINE_TO, 8.9,\n" \
"CUBIC_TO, 19.3, 4.2, 15.9, 2, 12, 2,\n" \
"CUBIC_TO, 8.9, 2, 6.1, 3.4, 4.3, 5.6,\n" \
"R_LINE_TO, 3.3, 5.7,\n" \
"R_CUBIC_TO, 0.3, -2.1, 2.2, -3.8, 4.4, -3.8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 9,\n" \
"R_CUBIC_TO, -1.7, 0, -3.1, -0.9, -3.9, -2.3,\n" \
"LINE_TO, 3.6, 6.5,\n" \
"CUBIC_TO, 2.6, 8.1, 2, 10, 2, 12,\n" \
"R_CUBIC_TO, 0, 5, 3.6, 9.1, 8.4, 9.9,\n" \
"R_LINE_TO, 3.3, -5.7,\n" \
"R_CUBIC_TO, -0.6, 0.2, -1.1, 0.3, -1.7, 0.3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.5, -4.5,\n" \
"R_CUBIC_TO, 0, 0.8, -0.2, 1.6, -0.6, 2.2,\n" \
"LINE_TO, 11.4, 22,\n" \
"R_H_LINE_TO, 0.6,\n" \
"R_CUBIC_TO, 5.5, 0, 10, -4.5, 10, -10,\n" \
"R_CUBIC_TO, 0, -1.2, -0.2, -2.4, -0.6, -3.5,\n" \
"R_H_LINE_TO, -6.6,\n" \
"R_CUBIC_TO, 1, 0.8, 1.7, 2.1, 1.7, 3.5,\n" \
"CLOSE,\n" \
"CIRCLE, 12, 12, 3.5\n" \
; \
              CheckThatParsedElementsMatch(s, kProductIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseProductRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"// Transparent\n" \
"PATH_COLOR_ARGB, 0xFF, 0xFF, 0xFF, 0xFF,\n" \
"MOVE_TO, 12, 17.99,\n" \
"CUBIC_TO, 15.31, 17.99, 18, 15.31, 18, 11.99,\n" \
"CUBIC_TO, 18, 8.68, 15.31, 5.99, 12, 5.99,\n" \
"CUBIC_TO, 8.69, 5.99, 6, 8.68, 6, 11.99,\n" \
"CUBIC_TO, 6, 15.31, 8.69, 17.99, 12, 17.99,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"FILL_RULE_NONZERO,\n" \
"// Right color\n" \
"PATH_COLOR_ARGB, 0xFF, 0xAE, 0xCB, 0xFA,\n" \
"MOVE_TO, 17.2, 15,\n" \
"LINE_TO, 12.01, 23.99,\n" \
"CUBIC_TO, 14.11, 23.99, 16.18, 23.44, 18, 22.39,\n" \
"CUBIC_TO, 19.83, 21.33, 21.34, 19.82, 22.4, 18,\n" \
"CUBIC_TO, 23.45, 16.17, 24, 14.1, 24, 12,\n" \
"CUBIC_TO, 24, 9.89, 23.44, 7.82, 22.38, 6,\n" \
"LINE_TO, 12, 6,\n" \
"LINE_TO, 11.99, 6.01,\n" \
"CUBIC_TO, 13.05, 6, 14.08, 6.28, 15, 6.8,\n" \
"CUBIC_TO, 15.91, 7.33, 16.67, 8.08, 17.2, 9,\n" \
"CUBIC_TO, 17.73, 9.91, 18, 10.94, 18, 12,\n" \
"CUBIC_TO, 18, 13.05, 17.72, 14.09, 17.2, 15,\n" \
"V_LINE_TO, 15,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"FILL_RULE_NONZERO,\n" \
"// Left color\n" \
"PATH_COLOR_ARGB, 0xFF, 0x66, 0x9D, 0xF6,\n" \
"MOVE_TO, 6.8, 15,\n" \
"LINE_TO, 1.61, 6.01,\n" \
"CUBIC_TO, 0.56, 7.83, 0, 9.9, 0, 12.01,\n" \
"CUBIC_TO, 0, 14.11, 0.56, 16.18, 1.61, 18.01,\n" \
"CUBIC_TO, 2.66, 19.83, 4.18, 21.34, 6.01, 22.39,\n" \
"CUBIC_TO, 7.83, 23.44, 9.9, 23.99, 12.01, 23.99,\n" \
"LINE_TO, 17.2, 15,\n" \
"V_LINE_TO, 14.99,\n" \
"CUBIC_TO, 16.67, 15.9, 15.92, 16.66, 15, 17.19,\n" \
"CUBIC_TO, 14.09, 17.72, 13.06, 18, 12, 18,\n" \
"CUBIC_TO, 10.95, 18, 9.91, 17.73, 9, 17.2,\n" \
"CUBIC_TO, 8.09, 16.67, 7.33, 15.91, 6.8, 15,\n" \
"V_LINE_TO, 15,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"FILL_RULE_NONZERO,\n" \
"// Central color\n" \
"PATH_COLOR_ARGB, 0xFF, 0x1A, 0x73, 0xE8,\n" \
"MOVE_TO, 12, 16.75,\n" \
"CUBIC_TO, 14.62, 16.75, 16.75, 14.62, 16.75, 12,\n" \
"CUBIC_TO, 16.75, 9.38, 14.62, 7.25, 12, 7.25,\n" \
"CUBIC_TO, 9.38, 7.25, 7.25, 9.38, 7.25, 12,\n" \
"CUBIC_TO, 7.25, 14.62, 9.38, 16.75, 12, 16.75,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"FILL_RULE_NONZERO,\n" \
"// Upper color\n" \
"PATH_COLOR_ARGB, 0xFF, 0x19, 0x67, 0xD2,\n" \
"MOVE_TO, 12, 6,\n" \
"H_LINE_TO, 22.38,\n" \
"CUBIC_TO, 21.33, 4.18, 19.82, 2.66, 17.99, 1.61,\n" \
"CUBIC_TO, 16.17, 0.55, 14.1, 0, 11.99, 0,\n" \
"CUBIC_TO, 9.89, 0, 7.82, 0.56, 6, 1.61,\n" \
"CUBIC_TO, 4.17, 2.67, 2.66, 4.18, 1.61, 6.01,\n" \
"LINE_TO, 6.8, 15,\n" \
"LINE_TO, 6.81, 15,\n" \
"CUBIC_TO, 6.28, 14.09, 6, 13.06, 6, 12,\n" \
"CUBIC_TO, 6, 10.95, 6.27, 9.91, 6.8, 9,\n" \
"CUBIC_TO, 7.33, 8.09, 8.08, 7.33, 9, 6.8,\n" \
"CUBIC_TO, 9.91, 6.28, 10.95, 6, 12, 6,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE,\n" \
"NEW_PATH\n" \
; \
              CheckThatParsedElementsMatch(s, kProductRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseAccountCircleIcon) { \
              std::string s = \
"// Copyright 2022 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 5.85, 17.1,\n" \
"R_QUADRATIC_TO, 1.27, -0.97, 2.85, -1.54,\n" \
"QUADRATIC_TO, 10.28, 15, 12, 15,\n" \
"R_QUADRATIC_TO, 1.73, 0, 3.3, 0.56,\n" \
"R_QUADRATIC_TO, 1.58, 0.56, 2.85, 1.54,\n" \
"R_QUADRATIC_TO, 0.88, -1.02, 1.36, -2.33,\n" \
"QUADRATIC_TO, 20, 13.48, 20, 12,\n" \
"R_QUADRATIC_TO, 0, -3.32, -2.34, -5.66,\n" \
"QUADRATIC_TO, 15.33, 4, 12, 4,\n" \
"QUADRATIC_TO_SHORTHAND, 6.34, 6.34,\n" \
"QUADRATIC_TO, 4, 8.68, 4, 12,\n" \
"R_QUADRATIC_TO, 0, 1.48, 0.49, 2.78,\n" \
"R_QUADRATIC_TO, 0.49, 1.3, 1.36, 2.33,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 13,\n" \
"R_QUADRATIC_TO, -1.47, 0, -2.49, -1.01,\n" \
"QUADRATIC_TO, 8.5, 10.98, 8.5, 9.5,\n" \
"R_QUADRATIC_TO, 0, -1.48, 1.01, -2.49,\n" \
"QUADRATIC_TO, 10.53, 6, 12, 6,\n" \
"R_QUADRATIC_TO, 1.47, 0, 2.49, 1.01,\n" \
"QUADRATIC_TO, 15.5, 8.03, 15.5, 9.5,\n" \
"R_QUADRATIC_TO, 0, 1.47, -1.01, 2.49,\n" \
"QUADRATIC_TO, 13.48, 13, 12, 13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 9,\n" \
"R_QUADRATIC_TO, -2.08, 0, -3.9, -0.79,\n" \
"R_QUADRATIC_TO, -1.82, -0.79, -3.17, -2.14,\n" \
"R_QUADRATIC_TO, -1.35, -1.35, -2.14, -3.17,\n" \
"QUADRATIC_TO, 2, 14.08, 2, 12,\n" \
"R_QUADRATIC_TO, 0, -2.08, 0.79, -3.9,\n" \
"R_QUADRATIC_TO, 0.79, -1.82, 2.14, -3.17,\n" \
"R_QUADRATIC_TO, 1.35, -1.35, 3.18, -2.14,\n" \
"QUADRATIC_TO, 9.93, 2, 12, 2,\n" \
"R_QUADRATIC_TO, 2.07, 0, 3.9, 0.79,\n" \
"R_QUADRATIC_TO, 1.83, 0.79, 3.18, 2.14,\n" \
"R_QUADRATIC_TO, 1.35, 1.35, 2.14, 3.18,\n" \
"QUADRATIC_TO, 22, 9.93, 22, 12,\n" \
"R_QUADRATIC_TO, 0, 2.07, -0.79, 3.9,\n" \
"R_QUADRATIC_TO, -0.79, 1.83, -2.14, 3.18,\n" \
"R_QUADRATIC_TO, -1.35, 1.35, -3.17, 2.14,\n" \
"QUADRATIC_TO, 14.08, 22, 12, 22,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2,\n" \
"R_QUADRATIC_TO, 1.33, 0, 2.5, -0.39,\n" \
"R_QUADRATIC_TO, 1.18, -0.39, 2.15, -1.11,\n" \
"R_QUADRATIC_TO, -0.97, -0.72, -2.15, -1.11,\n" \
"QUADRATIC_TO, 13.33, 17, 12, 17,\n" \
"R_QUADRATIC_TO, -1.33, 0, -2.5, 0.39,\n" \
"R_QUADRATIC_TO, -1.17, 0.39, -2.15, 1.11,\n" \
"R_QUADRATIC_TO, 0.98, 0.73, 2.15, 1.11,\n" \
"QUADRATIC_TO, 10.68, 20, 12, 20,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -9,\n" \
"R_QUADRATIC_TO, 0.65, 0, 1.08, -0.42,\n" \
"R_QUADRATIC_TO, 0.43, -0.42, 0.43, -1.07,\n" \
"R_QUADRATIC_TO, 0, -0.65, -0.42, -1.07,\n" \
"QUADRATIC_TO, 12.65, 8, 12, 8,\n" \
"R_QUADRATIC_TO, -0.65, 0, -1.07, 0.43,\n" \
"QUADRATIC_TO, 10.5, 8.85, 10.5, 9.5,\n" \
"R_QUADRATIC_TO, 0, 0.65, 0.43, 1.08,\n" \
"QUADRATIC_TO, 11.35, 11, 12, 11,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 9,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kAccountCircleIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseAccountCircleChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.94, 14.06,\n" \
"R_ARC_TO, 8.24, 8.24, 0, 0, 1, 2.41, -1.18,\n" \
"ARC_TO, 9.14, 9.14, 0, 0, 1, 10, 12.5,\n" \
"R_CUBIC_TO, 0.92, 0, 1.8, 0.13, 2.66, 0.39,\n" \
"R_ARC_TO, 8.24, 8.24, 0, 0, 1, 2.41, 1.18,\n" \
"R_ARC_TO, 5.78, 5.78, 0, 0, 0, 1.08, -1.89,\n" \
"ARC_TO, 6.64, 6.64, 0, 0, 0, 16.5, 10,\n" \
"R_CUBIC_TO, 0, -1.8, -0.63, -3.34, -1.9, -4.6,\n" \
"CUBIC_TO, 13.33, 4.13, 11.8, 3.5, 10, 3.5,\n" \
"R_CUBIC_TO, -1.8, 0, -3.34, 0.63, -4.6, 1.9,\n" \
"CUBIC_TO, 4.13, 6.66, 3.5, 8.2, 3.5, 10,\n" \
"R_CUBIC_TO, 0, 0.75, 0.12, 1.47, 0.36, 2.17,\n" \
"R_ARC_TO, 5.78, 5.78, 0, 0, 0, 1.08, 1.9,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 11.5,\n" \
"R_ARC_TO, 2.9, 2.9, 0, 0, 1, -2.12, -0.87,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 1, 7, 8.5,\n" \
"R_CUBIC_TO, 0, -0.83, 0.29, -1.54, 0.88, -2.12,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 1, 10, 5.5,\n" \
"R_CUBIC_TO, 0.83, 0, 1.54, 0.29, 2.13, 0.88,\n" \
"R_CUBIC_TO, 0.58, 0.58, 0.88, 1.29, 0.88, 2.13,\n" \
"R_CUBIC_TO, 0, 0.83, -0.29, 1.54, -0.87, 2.13,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 1, 10, 11.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 6.5,\n" \
"R_ARC_TO, 7.8, 7.8, 0, 0, 1, -3.11, -0.62,\n" \
"R_ARC_TO, 8.07, 8.07, 0, 0, 1, -2.55, -1.72,\n" \
"R_ARC_TO, 8.07, 8.07, 0, 0, 1, -1.72, -2.55,\n" \
"ARC_TO, 7.8, 7.8, 0, 0, 1, 2, 9.99,\n" \
"R_CUBIC_TO, 0, -1.1, 0.21, -2.14, 0.63, -3.1,\n" \
"R_ARC_TO, 8.12, 8.12, 0, 0, 1, 1.72, -2.54,\n" \
"R_CUBIC_TO, 0.73, -0.73, 1.58, -1.3, 2.55, -1.72,\n" \
"ARC_TO, 7.8, 7.8, 0, 0, 1, 10.01, 2,\n" \
"R_CUBIC_TO, 1.11, 0, 2.14, 0.21, 3.11, 0.63,\n" \
"R_ARC_TO, 8.12, 8.12, 0, 0, 1, 2.54, 1.72,\n" \
"R_CUBIC_TO, 0.73, 0.73, 1.3, 1.58, 1.72, 2.55,\n" \
"R_CUBIC_TO, 0.42, 0.97, 0.63, 2, 0.63, 3.11,\n" \
"R_ARC_TO, 7.8, 7.8, 0, 0, 1, -0.62, 3.11,\n" \
"R_ARC_TO, 8.07, 8.07, 0, 0, 1, -1.72, 2.55,\n" \
"R_CUBIC_TO, -0.73, 0.73, -1.58, 1.3, -2.55, 1.72,\n" \
"R_ARC_TO, 7.75, 7.75, 0, 0, 1, -3.1, 0.63,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 16.5,\n" \
"R_CUBIC_TO, 0.72, 0, 1.42, -0.11, 2.08, -0.34,\n" \
"R_ARC_TO, 6.62, 6.62, 0, 0, 0, 1.88, -1.01,\n" \
"R_ARC_TO, 7.36, 7.36, 0, 0, 0, -1.89, -0.85,\n" \
"ARC_TO, 7.31, 7.31, 0, 0, 0, 10, 14,\n" \
"R_CUBIC_TO, -0.71, 0, -1.4, 0.09, -2.07, 0.28,\n" \
"R_CUBIC_TO, -0.67, 0.19, -1.3, 0.48, -1.88, 0.86,\n" \
"R_ARC_TO, 6.62, 6.62, 0, 0, 0, 1.88, 1.01,\n" \
"R_CUBIC_TO, 0.66, 0.23, 1.36, 0.34, 2.08, 0.34,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -6.5,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, -0.14, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.29, 0.44, -0.64, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.42, -0.14, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 10, 7,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, 0.14, -1.06, 0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 8.5, 8.5,\n" \
"R_CUBIC_TO, 0, 0.42, 0.14, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.64, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 6.65,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.97, 11.19,\n" \
"R_ARC_TO, 6.59, 6.59, 0, 0, 1, 1.92, -0.92,\n" \
"R_CUBIC_TO, 0.68, -0.2, 1.38, -0.3, 2.11, -0.3,\n" \
"R_CUBIC_TO, 0.73, 0, 1.44, 0.1, 2.12, 0.31,\n" \
"R_CUBIC_TO, 0.68, 0.21, 1.32, 0.51, 1.91, 0.93,\n" \
"R_CUBIC_TO, 0.38, -0.46, 0.66, -0.96, 0.84, -1.5,\n" \
"ARC_TO, 5.23, 5.23, 0, 0, 0, 13.15, 8,\n" \
"R_CUBIC_TO, 0, -1.43, -0.5, -2.64, -1.51, -3.64,\n" \
"R_CUBIC_TO, -1.01, -1, -2.22, -1.51, -3.65, -1.51,\n" \
"R_CUBIC_TO, -1.43, 0, -2.64, 0.5, -3.64, 1.51,\n" \
"CUBIC_TO, 3.35, 5.36, 2.85, 6.58, 2.85, 8,\n" \
"R_CUBIC_TO, 0, 0.59, 0.09, 1.15, 0.27, 1.69,\n" \
"R_CUBIC_TO, 0.18, 0.54, 0.47, 1.04, 0.85, 1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 9.23,\n" \
"R_CUBIC_TO, -0.68, 0, -1.25, -0.23, -1.72, -0.7,\n" \
"R_ARC_TO, 2.35, 2.35, 0, 0, 1, -0.71, -1.72,\n" \
"R_CUBIC_TO, 0, -0.68, 0.24, -1.25, 0.71, -1.72,\n" \
"ARC_TO, 2.35, 2.35, 0, 0, 1, 8, 4.38,\n" \
"R_CUBIC_TO, 0.68, 0, 1.25, 0.24, 1.72, 0.71,\n" \
"R_CUBIC_TO, 0.47, 0.47, 0.71, 1.04, 0.71, 1.72,\n" \
"R_CUBIC_TO, 0, 0.68, -0.24, 1.25, -0.71, 1.72,\n" \
"ARC_TO, 2.36, 2.36, 0, 0, 1, 8, 9.23,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 5.3,\n" \
"R_ARC_TO, 6.38, 6.38, 0, 0, 1, -2.54, -0.51,\n" \
"R_ARC_TO, 6.59, 6.59, 0, 0, 1, -3.49, -3.49,\n" \
"R_ARC_TO, 6.39, 6.39, 0, 0, 1, -0.51, -2.54,\n" \
"R_CUBIC_TO, 0, -0.9, 0.17, -1.75, 0.51, -2.53,\n" \
"R_ARC_TO, 6.6, 6.6, 0, 0, 1, 3.49, -3.48,\n" \
"R_ARC_TO, 6.39, 6.39, 0, 0, 1, 2.54, -0.51,\n" \
"R_CUBIC_TO, 0.9, 0, 1.75, 0.17, 2.54, 0.51,\n" \
"R_ARC_TO, 6.6, 6.6, 0, 0, 1, 3.48, 3.49,\n" \
"R_CUBIC_TO, 0.34, 0.79, 0.51, 1.64, 0.51, 2.54,\n" \
"R_ARC_TO, 6.38, 6.38, 0, 0, 1, -0.51, 2.54,\n" \
"R_ARC_TO, 6.59, 6.59, 0, 0, 1, -3.49, 3.49,\n" \
"R_CUBIC_TO, -0.79, 0.34, -1.64, 0.51, -2.53, 0.51,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 13.15,\n" \
"R_CUBIC_TO, 0.57, 0, 1.11, -0.09, 1.62, -0.26,\n" \
"R_ARC_TO, 5.38, 5.38, 0, 0, 0, 1.48, -0.78,\n" \
"R_ARC_TO, 6.2, 6.2, 0, 0, 0, -1.49, -0.66,\n" \
"ARC_TO, 5.74, 5.74, 0, 0, 0, 8, 11.22,\n" \
"R_CUBIC_TO, -0.56, 0, -1.1, 0.07, -1.61, 0.21,\n" \
"R_ARC_TO, 5.29, 5.29, 0, 0, 0, -1.48, 0.67,\n" \
"R_CUBIC_TO, 0.46, 0.34, 0.96, 0.61, 1.47, 0.78,\n" \
"ARC_TO, 4.99, 4.99, 0, 0, 0, 8, 13.15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -5.17,\n" \
"R_CUBIC_TO, 0.32, 0, 0.6, -0.11, 0.82, -0.34,\n" \
"R_CUBIC_TO, 0.23, -0.23, 0.34, -0.5, 0.34, -0.82,\n" \
"R_CUBIC_TO, 0, -0.32, -0.11, -0.6, -0.34, -0.83,\n" \
"R_CUBIC_TO, -0.23, -0.23, -0.5, -0.34, -0.82, -0.34,\n" \
"R_CUBIC_TO, -0.32, 0, -0.6, 0.11, -0.82, 0.34,\n" \
"R_CUBIC_TO, -0.23, 0.23, -0.34, 0.5, -0.34, 0.83,\n" \
"R_CUBIC_TO, 0, 0.32, 0.11, 0.6, 0.34, 0.82,\n" \
"R_CUBIC_TO, 0.23, 0.23, 0.5, 0.34, 0.82, 0.34,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.17,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 5.3,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kAccountCircleChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseAccountCircleOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 13.55, 10.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.97, 2.97,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 19.93,\n" \
"R_CUBIC_TO, 0.82, 0, 1.62, -0.12, 2.39, -0.38,\n" \
"R_ARC_TO, 8.23, 8.23, 0, 0, 0, 2.16, -1.07,\n" \
"R_ARC_TO, 8.18, 8.18, 0, 0, 0, -2.16, -1.07,\n" \
"ARC_TO, 7.67, 7.67, 0, 0, 0, 12, 17.04,\n" \
"R_CUBIC_TO, -0.82, 0, -1.62, 0.13, -2.39, 0.38,\n" \
"R_CUBIC_TO, -0.76, 0.25, -1.48, 0.61, -2.16, 1.07,\n" \
"R_CUBIC_TO, 0.67, 0.47, 1.39, 0.82, 2.16, 1.07,\n" \
"R_ARC_TO, 7.54, 7.54, 0, 0, 0, 2.39, 0.38,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.78, -8.21,\n" \
"R_LINE_TO, -1.53, -1.54,\n" \
"R_ARC_TO, 1.51, 1.51, 0, 0, 0, 0.15, -0.32,\n" \
"R_CUBIC_TO, 0.04, -0.12, 0.06, -0.23, 0.06, -0.34,\n" \
"R_CUBIC_TO, 0, -0.41, -0.14, -0.75, -0.43, -1.03,\n" \
"ARC_TO, 1.39, 1.39, 0, 0, 0, 12, 8.06,\n" \
"R_CUBIC_TO, -0.12, 0, -0.23, 0.02, -0.35, 0.06,\n" \
"R_CUBIC_TO, -0.11, 0.04, -0.22, 0.09, -0.32, 0.16,\n" \
"LINE_TO, 9.8, 6.74,\n" \
"R_ARC_TO, 3.12, 3.12, 0, 0, 1, 1.03, -0.58,\n" \
"ARC_TO, 3.58, 3.58, 0, 0, 1, 12, 5.97,\n" \
"R_CUBIC_TO, 0.98, 0, 1.82, 0.34, 2.51, 1.03,\n" \
"R_CUBIC_TO, 0.69, 0.69, 1.03, 1.53, 1.03, 2.51,\n" \
"R_CUBIC_TO, 0, 0.41, -0.06, 0.8, -0.19, 1.18,\n" \
"R_CUBIC_TO, -0.12, 0.37, -0.32, 0.71, -0.58, 1.02,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.8, 5.81,\n" \
"R_LINE_TO, -1.67, -1.67,\n" \
"R_ARC_TO, 7.47, 7.47, 0, 0, 0, 0.77, -1.86,\n" \
"ARC_TO, 7.94, 7.94, 0, 0, 0, 19.93, 12,\n" \
"R_CUBIC_TO, 0, -2.21, -0.77, -4.08, -2.31, -5.62,\n" \
"CUBIC_TO, 16.08, 4.84, 14.21, 4.07, 12, 4.07,\n" \
"R_CUBIC_TO, -0.68, 0, -1.35, 0.09, -2, 0.25,\n" \
"R_ARC_TO, 7.66, 7.66, 0, 0, 0, -1.86, 0.76,\n" \
"LINE_TO, 6.48, 3.42,\n" \
"R_ARC_TO, 10.73, 10.73, 0, 0, 1, 2.64, -1.21,\n" \
"ARC_TO, 10.07, 10.07, 0, 0, 1, 12, 1.8,\n" \
"R_CUBIC_TO, 1.41, 0, 2.74, 0.27, 3.98, 0.8,\n" \
"R_ARC_TO, 10.35, 10.35, 0, 0, 1, 3.24, 2.18,\n" \
"R_CUBIC_TO, 0.92, 0.92, 1.65, 2, 2.18, 3.24,\n" \
"R_CUBIC_TO, 0.54, 1.24, 0.8, 2.57, 0.8, 3.98,\n" \
"R_CUBIC_TO, 0, 0.99, -0.14, 1.95, -0.42, 2.88,\n" \
"R_CUBIC_TO, -0.28, 0.93, -0.68, 1.81, -1.21, 2.64,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 22.2,\n" \
"R_ARC_TO, 9.92, 9.92, 0, 0, 1, -3.98, -0.8,\n" \
"R_ARC_TO, 10.35, 10.35, 0, 0, 1, -3.24, -2.18,\n" \
"R_CUBIC_TO, -0.92, -0.92, -1.65, -2, -2.18, -3.24,\n" \
"ARC_TO, 9.92, 9.92, 0, 0, 1, 1.8, 12,\n" \
"R_CUBIC_TO, 0, -1, 0.14, -1.98, 0.43, -2.94,\n" \
"R_CUBIC_TO, 0.28, -0.96, 0.7, -1.86, 1.25, -2.71,\n" \
"LINE_TO, 0.65, 3.5,\n" \
"R_LINE_TO, 1.48, -1.48,\n" \
"LINE_TO, 22.06, 21.94,\n" \
"R_LINE_TO, -1.48, 1.49,\n" \
"LINE_TO, 5.14, 8.02,\n" \
"ARC_TO, 7.85, 7.85, 0, 0, 0, 4.07, 12,\n" \
"R_CUBIC_TO, 0, 0.93, 0.16, 1.82, 0.47, 2.67,\n" \
"R_ARC_TO, 7.95, 7.95, 0, 0, 0, 1.34, 2.34,\n" \
"R_ARC_TO, 10.51, 10.51, 0, 0, 1, 2.9, -1.54,\n" \
"ARC_TO, 9.99, 9.99, 0, 0, 1, 12, 14.95,\n" \
"R_CUBIC_TO, 0.65, 0, 1.29, 0.07, 1.94, 0.2,\n" \
"R_CUBIC_TO, 0.64, 0.14, 1.26, 0.33, 1.87, 0.57,\n" \
"R_LINE_TO, 3.45, 3.45,\n" \
"ARC_TO, 10.15, 10.15, 0, 0, 1, 12, 22.2,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 11.06, 8.77,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.33, 2.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 16.44,\n" \
"R_CUBIC_TO, 0.7, 0, 1.38, -0.11, 2.04, -0.34,\n" \
"R_ARC_TO, 6.72, 6.72, 0, 0, 0, 1.84, -0.97,\n" \
"R_ARC_TO, 7.35, 7.35, 0, 0, 0, -1.86, -0.82,\n" \
"ARC_TO, 7.4, 7.4, 0, 0, 0, 10, 14.03,\n" \
"R_CUBIC_TO, -0.69, 0, -1.36, 0.09, -2.02, 0.27,\n" \
"R_CUBIC_TO, -0.66, 0.18, -1.28, 0.46, -1.85, 0.83,\n" \
"R_CUBIC_TO, 0.57, 0.43, 1.18, 0.75, 1.84, 0.97,\n" \
"R_ARC_TO, 6.24, 6.24, 0, 0, 0, 2.03, 0.34,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.5, -6.22,\n" \
"R_LINE_TO, -1.17, -1.18,\n" \
"R_ARC_TO, 0.84, 0.84, 0, 0, 0, 0.11, -0.26,\n" \
"R_CUBIC_TO, 0.02, -0.09, 0.03, -0.18, 0.03, -0.27,\n" \
"R_CUBIC_TO, 0, -0.41, -0.14, -0.75, -0.43, -1.04,\n" \
"R_ARC_TO, 1.41, 1.41, 0, 0, 0, -1.31, -0.4,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, -0.26, 0.11,\n" \
"LINE_TO, 8.29, 6.01,\n" \
"R_CUBIC_TO, 0.26, -0.17, 0.53, -0.3, 0.82, -0.4,\n" \
"R_CUBIC_TO, 0.29, -0.09, 0.59, -0.14, 0.89, -0.14,\n" \
"R_CUBIC_TO, 0.84, 0, 1.56, 0.3, 2.15, 0.89,\n" \
"R_CUBIC_TO, 0.59, 0.59, 0.89, 1.31, 0.89, 2.15,\n" \
"R_ARC_TO, 3.02, 3.02, 0, 0, 1, -0.54, 1.71,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.29, 4.29,\n" \
"R_LINE_TO, -1.26, -1.26,\n" \
"R_ARC_TO, 6.37, 6.37, 0, 0, 0, 0.68, -1.56,\n" \
"ARC_TO, 6.16, 6.16, 0, 0, 0, 16.44, 10,\n" \
"R_CUBIC_TO, 0, -1.78, -0.63, -3.3, -1.88, -4.56,\n" \
"CUBIC_TO, 13.31, 4.19, 11.79, 3.56, 10, 3.56,\n" \
"R_ARC_TO, 6.16, 6.16, 0, 0, 0, -1.69, 0.23,\n" \
"R_ARC_TO, 6.51, 6.51, 0, 0, 0, -1.56, 0.67,\n" \
"LINE_TO, 5.48, 3.21,\n" \
"ARC_TO, 8.01, 8.01, 0, 0, 1, 10, 1.83,\n" \
"R_ARC_TO, 7.93, 7.93, 0, 0, 1, 3.18, 0.64,\n" \
"R_ARC_TO, 8.29, 8.29, 0, 0, 1, 2.6, 1.75,\n" \
"R_ARC_TO, 8.29, 8.29, 0, 0, 1, 1.75, 2.6,\n" \
"R_ARC_TO, 7.93, 7.93, 0, 0, 1, 0.64, 3.18,\n" \
"R_ARC_TO, 8.01, 8.01, 0, 0, 1, -1.37, 4.52,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 18.17,\n" \
"R_ARC_TO, 7.93, 7.93, 0, 0, 1, -3.17, -0.64,\n" \
"R_ARC_TO, 8.23, 8.23, 0, 0, 1, -2.6, -1.75,\n" \
"R_ARC_TO, 8.23, 8.23, 0, 0, 1, -1.75, -2.6,\n" \
"ARC_TO, 7.93, 7.93, 0, 0, 1, 1.83, 10,\n" \
"R_CUBIC_TO, 0, -0.82, 0.12, -1.62, 0.36, -2.39,\n" \
"ARC_TO, 8.48, 8.48, 0, 0, 1, 3.25, 5.42,\n" \
"LINE_TO, 1.85, 4,\n" \
"LINE_TO, 2.96, 2.89,\n" \
"R_LINE_TO, 14.23, 14.23,\n" \
"R_LINE_TO, -1.11, 1.11,\n" \
"LINE_TO, 4.49, 6.66,\n" \
"R_ARC_TO, 6.37, 6.37, 0, 0, 0, -0.7, 1.6,\n" \
"ARC_TO, 6.42, 6.42, 0, 0, 0, 3.56, 10,\n" \
"R_CUBIC_TO, 0, 0.73, 0.12, 1.44, 0.35, 2.12,\n" \
"R_CUBIC_TO, 0.23, 0.68, 0.58, 1.31, 1.06, 1.87,\n" \
"R_ARC_TO, 8.37, 8.37, 0, 0, 1, 2.39, -1.16,\n" \
"R_ARC_TO, 9.27, 9.27, 0, 0, 1, 2.64, -0.37,\n" \
"R_CUBIC_TO, 0.53, 0, 1.05, 0.04, 1.56, 0.14,\n" \
"R_CUBIC_TO, 0.51, 0.09, 1.01, 0.23, 1.5, 0.4,\n" \
"R_LINE_TO, 2.75, 2.75,\n" \
"R_ARC_TO, 8.05, 8.05, 0, 0, 1, -2.65, 1.79,\n" \
"R_CUBIC_TO, -1, 0.42, -2.05, 0.63, -3.15, 0.63,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 8.95, 6.92,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.97, 9.03,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 13.09,\n" \
"R_CUBIC_TO, 0.54, 0, 1.06, -0.09, 1.57, -0.25,\n" \
"R_ARC_TO, 5.39, 5.39, 0, 0, 0, 1.43, -0.74,\n" \
"ARC_TO, 5.86, 5.86, 0, 0, 0, 8, 11.26,\n" \
"R_ARC_TO, 6.01, 6.01, 0, 0, 0, -1.56, 0.2,\n" \
"R_CUBIC_TO, -0.51, 0.14, -0.99, 0.35, -1.44, 0.63,\n" \
"R_CUBIC_TO, 0.45, 0.32, 0.92, 0.57, 1.43, 0.74,\n" \
"R_ARC_TO, 4.95, 4.95, 0, 0, 0, 1.57, 0.25,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.11, -4.98,\n" \
"LINE_TO, 9.08, 7.06,\n" \
"R_ARC_TO, 0.29, 0.29, 0, 0, 0, 0.04, -0.12,\n" \
"R_CUBIC_TO, 0, -0.04, 0, -0.08, 0, -0.12,\n" \
"R_CUBIC_TO, 0, -0.32, -0.11, -0.59, -0.33, -0.8,\n" \
"R_ARC_TO, 1.09, 1.09, 0, 0, 0, -0.93, -0.32,\n" \
"R_ARC_TO, 0.31, 0.31, 0, 0, 0, -0.12, 0.05,\n" \
"LINE_TO, 6.72, 4.7,\n" \
"R_CUBIC_TO, 0.2, -0.12, 0.41, -0.21, 0.63, -0.27,\n" \
"ARC_TO, 2.53, 2.53, 0, 0, 1, 8, 4.34,\n" \
"R_CUBIC_TO, 0.69, 0, 1.27, 0.24, 1.75, 0.72,\n" \
"R_CUBIC_TO, 0.48, 0.48, 0.72, 1.06, 0.72, 1.75,\n" \
"R_CUBIC_TO, 0, 0.22, -0.03, 0.43, -0.09, 0.66,\n" \
"R_ARC_TO, 2.68, 2.68, 0, 0, 1, -0.27, 0.63,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.53, 3.53,\n" \
"R_LINE_TO, -1.21, -1.21,\n" \
"R_CUBIC_TO, 0.21, -0.36, 0.38, -0.75, 0.48, -1.16,\n" \
"R_CUBIC_TO, 0.11, -0.41, 0.17, -0.83, 0.17, -1.26,\n" \
"R_CUBIC_TO, 0, -1.41, -0.5, -2.6, -1.49, -3.6,\n" \
"CUBIC_TO, 10.61, 3.41, 9.41, 2.91, 8, 2.91,\n" \
"R_CUBIC_TO, -0.43, 0, -0.85, 0.06, -1.26, 0.17,\n" \
"R_CUBIC_TO, -0.41, 0.11, -0.8, 0.27, -1.16, 0.47,\n" \
"LINE_TO, 4.37, 2.36,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 1, 1.74, -0.8,\n" \
"ARC_TO, 6.53, 6.53, 0, 0, 1, 8, 1.28,\n" \
"R_ARC_TO, 6.48, 6.48, 0, 0, 1, 2.61, 0.52,\n" \
"ARC_TO, 6.69, 6.69, 0, 0, 1, 12.75, 3.25,\n" \
"R_ARC_TO, 6.69, 6.69, 0, 0, 1, 1.45, 2.14,\n" \
"ARC_TO, 6.48, 6.48, 0, 0, 1, 14.72, 8,\n" \
"R_ARC_TO, 6.53, 6.53, 0, 0, 1, -0.27, 1.89,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 1, -0.8, 1.74,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 14.72,\n" \
"R_ARC_TO, 6.52, 6.52, 0, 0, 1, -2.61, -0.52,\n" \
"R_ARC_TO, 6.7, 6.7, 0, 0, 1, -2.14, -1.44,\n" \
"R_ARC_TO, 6.7, 6.7, 0, 0, 1, -1.44, -2.14,\n" \
"ARC_TO, 6.52, 6.52, 0, 0, 1, 1.28, 8,\n" \
"R_CUBIC_TO, 0, -0.68, 0.1, -1.33, 0.31, -1.97,\n" \
"R_CUBIC_TO, 0.21, -0.64, 0.5, -1.24, 0.88, -1.8,\n" \
"LINE_TO, 1.45, 3.2,\n" \
"R_LINE_TO, 0.95, -0.94,\n" \
"R_LINE_TO, 11.5, 11.5,\n" \
"R_LINE_TO, -0.95, 0.95,\n" \
"R_LINE_TO, -9.32, -9.3,\n" \
"R_ARC_TO, 5.17, 5.17, 0, 0, 0, -0.54, 1.25,\n" \
"R_CUBIC_TO, -0.12, 0.43, -0.18, 0.89, -0.18, 1.35,\n" \
"R_CUBIC_TO, 0, 0.57, 0.09, 1.12, 0.27, 1.65,\n" \
"R_CUBIC_TO, 0.18, 0.53, 0.45, 1.02, 0.81, 1.46,\n" \
"R_ARC_TO, 6.58, 6.58, 0, 0, 1, 1.9, -0.9,\n" \
"R_ARC_TO, 7.36, 7.36, 0, 0, 1, 2.1, -0.3,\n" \
"R_CUBIC_TO, 0.44, 0, 0.86, 0.04, 1.28, 0.11,\n" \
"R_CUBIC_TO, 0.42, 0.08, 0.82, 0.19, 1.22, 0.34,\n" \
"R_LINE_TO, 2.31, 2.31,\n" \
"R_ARC_TO, 6.71, 6.71, 0, 0, 1, -2.19, 1.51,\n" \
"R_CUBIC_TO, -0.83, 0.35, -1.7, 0.53, -2.62, 0.53,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kAccountCircleOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseAddIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 13, 7.2,\n" \
"H_LINE_TO, 8.8,\n" \
"V_LINE_TO, 3,\n" \
"H_LINE_TO, 7.2,\n" \
"R_V_LINE_TO, 4.2,\n" \
"H_LINE_TO, 3,\n" \
"R_V_LINE_TO, 1.7,\n" \
"R_H_LINE_TO, 4.2,\n" \
"V_LINE_TO, 13,\n" \
"R_H_LINE_TO, 1.7,\n" \
"V_LINE_TO, 8.8,\n" \
"H_LINE_TO, 13,\n" \
"V_LINE_TO, 7.2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kAddIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseAddLinkIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 17, 20,\n" \
"R_V_LINE_TO, -3,\n" \
"R_H_LINE_TO, -3,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, -3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -3,\n" \
"R_V_LINE_TO, 3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -6, -3,\n" \
"H_LINE_TO, 7,\n" \
"R_CUBIC_TO, -1.38, 0, -2.56, -0.49, -3.54, -1.46,\n" \
"CUBIC_TO, 2.49, 14.56, 2, 13.38, 2, 12,\n" \
"R_CUBIC_TO, 0, -1.38, 0.49, -2.56, 1.46, -3.54,\n" \
"CUBIC_TO, 4.44, 7.49, 5.62, 7, 7, 7,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 7,\n" \
"R_CUBIC_TO, -0.83, 0, -1.54, 0.29, -2.12, 0.88,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 0, 4, 12,\n" \
"R_CUBIC_TO, 0, 0.83, 0.29, 1.54, 0.88, 2.13,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 0, 7, 15,\n" \
"R_H_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3, -4,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 14, -1,\n" \
"R_H_LINE_TO, -2,\n" \
"R_CUBIC_TO, 0, -0.83, -0.29, -1.54, -0.87, -2.12,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 0, 17, 9,\n" \
"R_H_LINE_TO, -4,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, 4,\n" \
"R_CUBIC_TO, 1.38, 0, 2.56, 0.49, 3.54, 1.46,\n" \
"R_CUBIC_TO, 0.97, 0.98, 1.46, 2.16, 1.46, 3.54,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kAddLinkIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseAdsIcon) { \
              std::string s = \
"// Copyright 2016 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 4, 2,\n" \
"R_ARC_TO, 2, 2, 0, 0, 0, -2, 2,\n" \
"R_V_LINE_TO, 8,\n" \
"R_ARC_TO, 2, 2, 0, 0, 0, 2, 2,\n" \
"R_H_LINE_TO, 8,\n" \
"R_ARC_TO, 2, 2, 0, 0, 0, 2, -2,\n" \
"V_LINE_TO, 4,\n" \
"R_ARC_TO, 2, 2, 0, 0, 0, -2, -2,\n" \
"H_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8, 10,\n" \
"H_LINE_TO, 4,\n" \
"V_LINE_TO, 6,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kAdsIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseAdsChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 17,\n" \
"R_ARC_TO, 1.42, 1.42, 0, 0, 1, -1.06, -0.45,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 3, 15.5,\n" \
"R_V_LINE_TO, -11,\n" \
"R_CUBIC_TO, 0, -0.4, 0.15, -0.75, 0.44, -1.05,\n" \
"CUBIC_TO, 3.73, 3.15, 4.09, 3, 4.5, 3,\n" \
"R_H_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.45,\n" \
"R_CUBIC_TO, 0.29, 0.3, 0.44, 0.65, 0.44, 1.05,\n" \
"R_V_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0, 0.4, -0.15, 0.75, -0.44, 1.05,\n" \
"R_CUBIC_TO, -0.29, 0.3, -0.64, 0.45, -1.06, 0.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 11,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, -11,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kAdsChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseAdsClickIcon) { \
              std::string s = \
"// Copyright 2022 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 11.71, 17.99,\n" \
"CUBIC_TO, 8.53, 17.84, 6, 15.22, 6, 12,\n" \
"R_CUBIC_TO, 0, -3.31, 2.69, -6, 6, -6,\n" \
"R_CUBIC_TO, 3.22, 0, 5.84, 2.53, 5.99, 5.71,\n" \
"R_LINE_TO, -2.1, -0.63,\n" \
"CUBIC_TO, 15.48, 9.31, 13.89, 8, 12, 8,\n" \
"R_CUBIC_TO, -2.21, 0, -4, 1.79, -4, 4,\n" \
"R_CUBIC_TO, 0, 1.89, 1.31, 3.48, 3.08, 3.89,\n" \
"LINE_TO, 11.71, 17.99,\n" \
"CLOSE,\n" \
"MOVE_TO, 22, 12,\n" \
"R_CUBIC_TO, 0, 0.3, -0.01, 0.6, -0.04, 0.9,\n" \
"R_LINE_TO, -1.97, -0.59,\n" \
"CUBIC_TO, 20, 12.21, 20, 12.1, 20, 12,\n" \
"R_CUBIC_TO, 0, -4.42, -3.58, -8, -8, -8,\n" \
"R_CUBIC_TO, -4.42, 0, -8, 3.58, -8, 8,\n" \
"R_CUBIC_TO, 0, 4.42, 3.58, 8, 8, 8,\n" \
"R_CUBIC_TO, 0.1, 0, 0.21, 0, 0.31, -0.01,\n" \
"R_LINE_TO, 0.59, 1.97,\n" \
"CUBIC_TO, 12.6, 21.99, 12.3, 22, 12, 22,\n" \
"CUBIC_TO, 6.48, 22, 2, 17.52, 2, 12,\n" \
"CUBIC_TO, 2, 6.48, 6.48, 2, 12, 2,\n" \
"CUBIC_TO_SHORTHAND, 22, 6.48, 22, 12,\n" \
"CLOSE,\n" \
"MOVE_TO, 18.23, 16.26,\n" \
"LINE_TO, 22, 15,\n" \
"R_LINE_TO, -10, -3,\n" \
"R_LINE_TO, 3, 10,\n" \
"R_LINE_TO, 1.26, -3.77,\n" \
"R_LINE_TO, 4.27, 4.27,\n" \
"R_LINE_TO, 1.98, -1.98,\n" \
"LINE_TO, 18.23, 16.26,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kAdsClickIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseAdsOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16.02, 18.13,\n" \
"LINE_TO, 14.88, 17,\n" \
"H_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3, 15.5,\n" \
"V_LINE_TO, 5.13,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.15,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5, 15.5,\n" \
"R_H_LINE_TO, 8.88,\n" \
"LINE_TO, 4.5, 6.63,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12.5, -0.62,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 7,\n" \
"H_LINE_TO, 9.13,\n" \
"R_LINE_TO, -4, -4,\n" \
"H_LINE_TO, 15.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kAdsOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseArrowBackIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 13.33, 7.33,\n" \
"H_LINE_TO, 5.22,\n" \
"R_LINE_TO, 3.73, -3.73,\n" \
"LINE_TO, 8, 2.67,\n" \
"LINE_TO, 2.67, 8,\n" \
"LINE_TO, 8, 13.33,\n" \
"R_LINE_TO, 0.94, -0.94,\n" \
"R_LINE_TO, -3.72, -3.73,\n" \
"R_H_LINE_TO, 8.11,\n" \
"V_LINE_TO, 7.33,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kArrowBackIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseArrowBackChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 6.88, 10.75,\n" \
"R_LINE_TO, 4.19, 4.19,\n" \
"LINE_TO, 10, 16,\n" \
"R_LINE_TO, -6, -6,\n" \
"R_LINE_TO, 6, -6,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"LINE_TO, 6.88, 9.25,\n" \
"H_LINE_TO, 16,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kArrowBackChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseArrowRightAltIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 12, 14,\n" \
"R_LINE_TO, -1.06, -1.06,\n" \
"LINE_TO, 13.13, 10.75,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 9.13,\n" \
"R_LINE_TO, -2.19, -2.19,\n" \
"LINE_TO, 12, 6,\n" \
"R_LINE_TO, 4, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kArrowRightAltIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBackArrowIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"// TODO(crbug.com/40186140): Rename this to toolbar_back_arrow and move it to\n" \
"// //chrome/app/resources.\n" \
"\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"MOVE_TO, 4.22, 14.03,\n" \
"CUBIC_TO, 3.53, 14.15, 3, 14.77, 3, 15.5,\n" \
"CUBIC_TO, 3, 16.25, 3.53, 16.86, 4.24, 16.98,\n" \
"LINE_TO, 4.02, 17.19,\n" \
"LINE_TO, 14.36, 27.51,\n" \
"CUBIC_TO, 14.65, 27.81, 15.05, 28, 15.5, 28,\n" \
"CUBIC_TO, 16.33, 28, 17, 27.33, 17, 26.5,\n" \
"CUBIC_TO, 17, 26.05, 16.8, 25.65, 16.49, 25.38,\n" \
"LINE_TO, 16.55, 25.32,\n" \
"LINE_TO, 8.22, 17,\n" \
"LINE_TO, 27.51, 17,\n" \
"CUBIC_TO, 28.33, 17, 29, 16.33, 29, 15.5,\n" \
"CUBIC_TO, 29, 14.67, 28.33, 14, 27.51, 14,\n" \
"LINE_TO, 8.2, 14,\n" \
"LINE_TO, 16.55, 5.66,\n" \
"CUBIC_TO, 16.81, 5.34, 17, 4.94, 17, 4.5,\n" \
"CUBIC_TO, 17, 3.68, 16.33, 3, 15.5, 3,\n" \
"CUBIC_TO, 15.06, 3, 14.66, 3.2, 14.39, 3.5,\n" \
"LINE_TO, 14.36, 3.47,\n" \
"LINE_TO, 4, 13.81,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 9, 13.51,\n" \
"LINE_TO, 4.49, 9,\n" \
"LINE_TO, 14, 9,\n" \
"CUBIC_TO, 14.55, 9, 15, 8.55, 15, 8,\n" \
"CUBIC_TO, 15, 7.45, 14.55, 7, 14, 7,\n" \
"LINE_TO, 4.49, 7,\n" \
"LINE_TO, 9, 2.49,\n" \
"CUBIC_TO, 8.97, 2.28, 9, 2.14, 9, 2,\n" \
"CUBIC_TO, 9, 1.45, 8.55, 1, 8, 1,\n" \
"CUBIC_TO, 7.86, 1, 7.72, 1.03, 7.59, 1.09,\n" \
"LINE_TO, 7.51, 1,\n" \
"LINE_TO, 1, 7.51,\n" \
"CUBIC_TO, 1.03, 7.72, 1, 7.85, 1, 8,\n" \
"CUBIC_TO, 1, 8.15, 1.03, 8.28, 1.09, 8.41,\n" \
"LINE_TO, 1, 8.49,\n" \
"LINE_TO, 7.51, 15,\n" \
"CUBIC_TO, 7.72, 14.97, 7.86, 15, 8, 15,\n" \
"CUBIC_TO, 8.55, 15, 9, 14.55, 9, 14,\n" \
"CUBIC_TO, 9, 13.86, 8.97, 13.72, 8.91, 13.59,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBackArrowIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBackArrowChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 6.88, 10.75,\n" \
"R_LINE_TO, 4.19, 4.19,\n" \
"LINE_TO, 10, 16,\n" \
"R_LINE_TO, -6, -6,\n" \
"R_LINE_TO, 6, -6,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"LINE_TO, 6.88, 9.25,\n" \
"H_LINE_TO, 16,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBackArrowChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBackToTabIcon) { \
              std::string s = \
"// Copyright 2022 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 5.2, 12.14,\n" \
"V_LINE_TO, 13.4,\n" \
"R_ARC_TO, 1, 1, 0, 0, 0, 1, 1,\n" \
"R_H_LINE_TO, 6,\n" \
"R_ARC_TO, 1, 1, 0, 0, 0, 1, -1,\n" \
"V_LINE_TO, 9,\n" \
"R_ARC_TO, 1, 1, 0, 0, 0, -1, -1,\n" \
"R_H_LINE_TO, -6,\n" \
"R_ARC_TO, 1, 1, 0, 0, 0, -1, 1,\n" \
"R_V_LINE_TO, 1.84,\n" \
"H_LINE_TO, 2.13,\n" \
"V_LINE_TO, 7.6,\n" \
"H_LINE_TO, 0.8,\n" \
"R_V_LINE_TO, 3.21,\n" \
"R_CUBIC_TO, 0, 0.73, 0.6, 1.33, 1.33, 1.33,\n" \
"R_H_LINE_TO, 3.07,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6.67, -9.21,\n" \
"V_LINE_TO, 6.8,\n" \
"H_LINE_TO, 13.2,\n" \
"V_LINE_TO, 2.93,\n" \
"R_CUBIC_TO, 0, -0.73, -0.59, -1.33, -1.33, -1.33,\n" \
"H_LINE_TO, 6.8,\n" \
"R_V_LINE_TO, 1.33,\n" \
"R_H_LINE_TO, 5.07,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"MOVE_TO, 5.48, 2.93,\n" \
"V_LINE_TO, 1.6,\n" \
"H_LINE_TO, 0.82,\n" \
"R_V_LINE_TO, 4.67,\n" \
"H_LINE_TO, 2.15,\n" \
"V_LINE_TO, 3.87,\n" \
"R_LINE_TO, 4.05, 4.05,\n" \
"R_CUBIC_TO, 0.2, -0.41, 0.53, -0.74, 0.95, -0.93,\n" \
"LINE_TO, 3.09, 2.93,\n" \
"R_H_LINE_TO, 2.39,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBackToTabIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBackToTabChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 5.67, 7.71,\n" \
"LINE_TO, 2.33, 4.38,\n" \
"LINE_TO, 2.33, 7.33,\n" \
"LINE_TO, 1.33, 7.33,\n" \
"LINE_TO, 1.33, 2.67,\n" \
"LINE_TO, 6, 2.67,\n" \
"LINE_TO, 6, 3.67,\n" \
"LINE_TO, 3.04, 3.67,\n" \
"LINE_TO, 6.38, 7,\n" \
"CLOSE,\n" \
"MOVE_TO, 2.33, 13.33,\n" \
"CUBIC_TO, 2.07, 13.33, 1.83, 13.23, 1.63, 13.04,\n" \
"CUBIC_TO, 1.43, 12.83, 1.33, 12.6, 1.33, 12.33,\n" \
"LINE_TO, 1.33, 8.5,\n" \
"LINE_TO, 2.33, 8.5,\n" \
"LINE_TO, 2.33, 12.33,\n" \
"LINE_TO, 8, 12.33,\n" \
"LINE_TO, 8, 13.33,\n" \
"CLOSE,\n" \
"MOVE_TO, 13.67, 8.67,\n" \
"LINE_TO, 13.67, 3.67,\n" \
"LINE_TO, 7.17, 3.67,\n" \
"LINE_TO, 7.17, 2.67,\n" \
"LINE_TO, 13.67, 2.67,\n" \
"CUBIC_TO, 13.93, 2.67, 14.17, 2.77, 14.37, 2.96,\n" \
"CUBIC_TO, 14.57, 3.17, 14.67, 3.4, 14.67, 3.67,\n" \
"LINE_TO, 14.67, 8.67,\n" \
"CLOSE,\n" \
"MOVE_TO, 14.67, 9.67,\n" \
"LINE_TO, 14.67, 13.33,\n" \
"LINE_TO, 9, 13.33,\n" \
"LINE_TO, 9, 9.67,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBackToTabChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBlockedBadgeIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"PATH_MODE_CLEAR,\n" \
"MOVE_TO, 32, 32,\n" \
"LINE_TO, 16, 32,\n" \
"LINE_TO, 16, 18,\n" \
"CUBIC_TO, 16, 16.9, 16.9, 16, 18, 16,\n" \
"LINE_TO, 32, 16,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"PATH_COLOR_ARGB, 0xFF, 0xDB, 0x44, 0x37,\n" \
"MOVE_TO, 30, 32,\n" \
"LINE_TO, 20, 32,\n" \
"CUBIC_TO, 18.9, 32, 18, 31.1, 18, 30,\n" \
"LINE_TO, 18, 20,\n" \
"CUBIC_TO, 18, 18.9, 18.9, 18, 20, 18,\n" \
"LINE_TO, 30, 18,\n" \
"CUBIC_TO, 31.1, 18, 32, 18.9, 32, 20,\n" \
"LINE_TO, 32, 30,\n" \
"CUBIC_TO, 32, 31.1, 31.1, 32, 30, 32,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"PATH_COLOR_ARGB, 0xFF, 0xFF, 0xFF, 0xFF,\n" \
"MOVE_TO, 26.8, 22,\n" \
"LINE_TO, 25, 23.8,\n" \
"LINE_TO, 23.2, 22,\n" \
"CUBIC_TO, 22.87, 21.66, 22.33, 21.66, 22, 22,\n" \
"CUBIC_TO, 21.66, 22.33, 21.66, 22.87, 22, 23.2,\n" \
"LINE_TO, 23.8, 25,\n" \
"LINE_TO, 22, 26.8,\n" \
"CUBIC_TO, 21.66, 27.13, 21.66, 27.67, 22, 28,\n" \
"CUBIC_TO, 22.33, 28.34, 22.87, 28.34, 23.2, 28,\n" \
"LINE_TO, 25, 26.2,\n" \
"LINE_TO, 26.8, 28,\n" \
"CUBIC_TO, 27.13, 28.34, 27.67, 28.34, 28, 28,\n" \
"CUBIC_TO, 28.34, 27.67, 28.34, 27.13, 28, 26.8,\n" \
"LINE_TO, 26.2, 25,\n" \
"LINE_TO, 28, 23.2,\n" \
"CUBIC_TO, 28.34, 22.87, 28.34, 22.33, 28, 22,\n" \
"CUBIC_TO, 27.67, 21.66, 27.13, 21.66, 26.8, 22,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBlockedBadgeIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBluetoothIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 17.71, 7.71,\n" \
"LINE_TO, 12, 2,\n" \
"R_H_LINE_TO, -1,\n" \
"R_V_LINE_TO, 7.59,\n" \
"LINE_TO, 6.41, 5,\n" \
"LINE_TO, 5, 6.41,\n" \
"LINE_TO, 10.59, 12,\n" \
"LINE_TO, 5, 17.59,\n" \
"LINE_TO, 6.41, 19,\n" \
"LINE_TO, 11, 14.41,\n" \
"V_LINE_TO, 22,\n" \
"R_H_LINE_TO, 1,\n" \
"R_LINE_TO, 5.71, -5.71,\n" \
"R_LINE_TO, -4.3, -4.29,\n" \
"R_LINE_TO, 4.3, -4.29,\n" \
"CLOSE,\n" \
"MOVE_TO, 13, 5.83,\n" \
"R_LINE_TO, 1.88, 1.88,\n" \
"LINE_TO, 13, 9.59,\n" \
"V_LINE_TO, 5.83,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.88, 10.46,\n" \
"LINE_TO, 13, 18.17,\n" \
"R_V_LINE_TO, -3.76,\n" \
"R_LINE_TO, 1.88, 1.88,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBluetoothIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBluetoothChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 9, 18,\n" \
"R_V_LINE_TO, -5.87,\n" \
"R_LINE_TO, -3.56, 3.56,\n" \
"R_LINE_TO, -1.06, -1.06,\n" \
"LINE_TO, 9, 10,\n" \
"LINE_TO, 4.38, 5.38,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"LINE_TO, 9, 7.88,\n" \
"V_LINE_TO, 2,\n" \
"R_H_LINE_TO, 1.13,\n" \
"R_LINE_TO, 4.5, 4.5,\n" \
"R_LINE_TO, -3.5, 3.5,\n" \
"R_LINE_TO, 3.5, 3.5,\n" \
"R_LINE_TO, -4.5, 4.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.5, -9.5,\n" \
"R_LINE_TO, 2, -2,\n" \
"R_LINE_TO, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 7,\n" \
"R_LINE_TO, 2, -2,\n" \
"R_LINE_TO, -2, -2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBluetoothChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBluetoothConnectedIcon) { \
              std::string s = \
"// Copyright 2016 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 14, 24,\n" \
"R_LINE_TO, -4, -4,\n" \
"R_LINE_TO, -4, 4,\n" \
"R_LINE_TO, 4, 4,\n" \
"R_LINE_TO, 4, -4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 21.41, -8.59,\n" \
"LINE_TO, 24, 4,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 15.17,\n" \
"LINE_TO, 12.83, 10,\n" \
"LINE_TO, 10, 12.83,\n" \
"LINE_TO, 21.17, 24,\n" \
"LINE_TO, 10, 35.17,\n" \
"LINE_TO, 12.83, 38,\n" \
"LINE_TO, 22, 28.83,\n" \
"V_LINE_TO, 44,\n" \
"R_H_LINE_TO, 2,\n" \
"R_LINE_TO, 11.41, -11.41,\n" \
"LINE_TO, 26.83, 24,\n" \
"R_LINE_TO, 8.58, -8.59,\n" \
"CLOSE,\n" \
"MOVE_TO, 26, 11.66,\n" \
"R_LINE_TO, 3.76, 3.76,\n" \
"LINE_TO, 26, 19.17,\n" \
"R_V_LINE_TO, -7.51,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.76, 20.93,\n" \
"LINE_TO, 26, 36.34,\n" \
"R_V_LINE_TO, -7.52,\n" \
"R_LINE_TO, 3.76, 3.77,\n" \
"CLOSE,\n" \
"MOVE_TO, 38, 20,\n" \
"R_LINE_TO, -4, 4,\n" \
"R_LINE_TO, 4, 4,\n" \
"R_LINE_TO, 4, -4,\n" \
"R_LINE_TO, -4, -4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBluetoothConnectedIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBluetoothOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 16, 18.13,\n" \
"R_LINE_TO, -3, -3,\n" \
"LINE_TO, 10.13, 18,\n" \
"H_LINE_TO, 9,\n" \
"R_V_LINE_TO, -5.87,\n" \
"R_LINE_TO, -3.56, 3.56,\n" \
"R_LINE_TO, -1.06, -1.06,\n" \
"R_LINE_TO, 4.06, -4.06,\n" \
"LINE_TO, 1.88, 3.98,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.15,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.5, 15.5,\n" \
"R_LINE_TO, 1.44, -1.44,\n" \
"R_LINE_TO, -1.44, -1.44,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.13, -6,\n" \
"LINE_TO, 10.5, 8.38,\n" \
"R_LINE_TO, 2, -1.87,\n" \
"R_LINE_TO, -2, -2,\n" \
"R_V_LINE_TO, 3.88,\n" \
"LINE_TO, 9, 6.88,\n" \
"V_LINE_TO, 2,\n" \
"R_H_LINE_TO, 1.13,\n" \
"R_LINE_TO, 4.5, 4.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBluetoothOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBluetoothScanningIcon) { \
              std::string s = \
"// Copyright 2019 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 14.24, 12.01,\n" \
"R_LINE_TO, 2.32, 2.32,\n" \
"R_CUBIC_TO, 0.28, -0.72, 0.44, -1.51, 0.44, -2.33,\n" \
"R_CUBIC_TO, 0, -0.82, -0.16, -1.59, -0.43, -2.31,\n" \
"R_LINE_TO, -2.33, 2.32,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.29, -5.3,\n" \
"R_LINE_TO, -1.26, 1.26,\n" \
"R_CUBIC_TO, 0.63, 1.21, 0.98, 2.57, 0.98, 4.02,\n" \
"R_CUBIC_TO, 0, 1.45, -0.36, 2.82, -0.98, 4.02,\n" \
"R_LINE_TO, 1.2, 1.2,\n" \
"R_ARC_TO, 9.94, 9.94, 0, 0, 0, 1.54, -5.31,\n" \
"R_CUBIC_TO, -0.01, -1.89, -0.55, -3.67, -1.48, -5.19,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3.82, 1,\n" \
"LINE_TO, 10, 2,\n" \
"H_LINE_TO, 9,\n" \
"R_V_LINE_TO, 7.59,\n" \
"LINE_TO, 4.41, 5,\n" \
"LINE_TO, 3, 6.41,\n" \
"LINE_TO, 8.59, 12,\n" \
"LINE_TO, 3, 17.59,\n" \
"LINE_TO, 4.41, 19,\n" \
"LINE_TO, 9, 14.41,\n" \
"V_LINE_TO, 22,\n" \
"R_H_LINE_TO, 1,\n" \
"R_LINE_TO, 5.71, -5.71,\n" \
"R_LINE_TO, -4.3, -4.29,\n" \
"R_LINE_TO, 4.3, -4.29,\n" \
"CLOSE,\n" \
"MOVE_TO, 11, 5.83,\n" \
"R_LINE_TO, 1.88, 1.88,\n" \
"LINE_TO, 11, 9.59,\n" \
"V_LINE_TO, 5.83,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.88, 10.46,\n" \
"LINE_TO, 11, 18.17,\n" \
"R_V_LINE_TO, -3.76,\n" \
"R_LINE_TO, 1.88, 1.88,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBluetoothScanningIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBluetoothScanningChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 7.5, 18,\n" \
"R_V_LINE_TO, -5.87,\n" \
"R_LINE_TO, -3.56, 3.56,\n" \
"R_LINE_TO, -1.06, -1.06,\n" \
"LINE_TO, 7.5, 10,\n" \
"LINE_TO, 2.88, 5.38,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"LINE_TO, 7.5, 7.88,\n" \
"V_LINE_TO, 2,\n" \
"R_H_LINE_TO, 1.13,\n" \
"R_LINE_TO, 4.5, 4.5,\n" \
"R_LINE_TO, -3.5, 3.5,\n" \
"R_LINE_TO, 3.5, 3.5,\n" \
"R_LINE_TO, -4.5, 4.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 9, 8.5,\n" \
"R_LINE_TO, 2, -2,\n" \
"R_LINE_TO, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 7,\n" \
"R_LINE_TO, 2, -2,\n" \
"R_LINE_TO, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.69, -3.81,\n" \
"LINE_TO, 12, 10,\n" \
"R_LINE_TO, 1.69, -1.69,\n" \
"R_CUBIC_TO, 0.11, 0.28, 0.19, 0.56, 0.24, 0.83,\n" \
"R_CUBIC_TO, 0.05, 0.28, 0.07, 0.56, 0.07, 0.86,\n" \
"R_CUBIC_TO, 0, 0.29, -0.02, 0.58, -0.07, 0.86,\n" \
"R_ARC_TO, 4.17, 4.17, 0, 0, 1, -0.24, 0.83,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.27, 2.27,\n" \
"R_LINE_TO, -1.12, -1.12,\n" \
"R_CUBIC_TO, 0.24, -0.44, 0.41, -0.91, 0.51, -1.38,\n" \
"R_ARC_TO, 6.75, 6.75, 0, 0, 0, 0, -2.9,\n" \
"R_ARC_TO, 5.32, 5.32, 0, 0, 0, -0.51, -1.38,\n" \
"R_LINE_TO, 1.13, -1.12,\n" \
"R_ARC_TO, 6.82, 6.82, 0, 0, 1, 0.79, 1.89,\n" \
"R_CUBIC_TO, 0.17, 0.67, 0.25, 1.36, 0.25, 2.06,\n" \
"R_CUBIC_TO, 0, 0.71, -0.08, 1.4, -0.25, 2.06,\n" \
"R_ARC_TO, 6.82, 6.82, 0, 0, 1, -0.79, 1.89,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 5.95, 14.45,\n" \
"R_LINE_TO, 0, -4.67,\n" \
"R_LINE_TO, -2.82, 2.82,\n" \
"R_LINE_TO, -0.89, -0.89,\n" \
"LINE_TO, 5.94, 8,\n" \
"LINE_TO, 2.24, 4.3,\n" \
"R_LINE_TO, 0.9, -0.89,\n" \
"R_LINE_TO, 2.82, 2.82,\n" \
"R_LINE_TO, 0, -4.67,\n" \
"R_H_LINE_TO, 0.93,\n" \
"R_LINE_TO, 3.64, 3.64,\n" \
"LINE_TO, 7.73, 8,\n" \
"R_LINE_TO, 2.8, 2.8,\n" \
"R_LINE_TO, -3.64, 3.65,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.26, -7.72,\n" \
"R_LINE_TO, 1.52, -1.53,\n" \
"R_LINE_TO, -1.52, -1.53,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 5.6,\n" \
"R_LINE_TO, 1.52, -1.53,\n" \
"R_LINE_TO, -1.52, -1.53,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.77, -2.92,\n" \
"LINE_TO, 9.59, 8,\n" \
"R_LINE_TO, 1.4, -1.41,\n" \
"R_CUBIC_TO, 0.1, 0.23, 0.16, 0.47, 0.2, 0.7,\n" \
"R_ARC_TO, 4.34, 4.34, 0, 0, 1, 0, 1.43,\n" \
"R_CUBIC_TO, -0.04, 0.23, -0.1, 0.46, -0.2, 0.69,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.81, 1.81,\n" \
"R_LINE_TO, -0.93, -0.94,\n" \
"R_ARC_TO, 4.31, 4.31, 0, 0, 0, 0.41, -1.11,\n" \
"R_CUBIC_TO, 0.09, -0.39, 0.13, -0.77, 0.13, -1.16,\n" \
"R_CUBIC_TO, 0, -0.39, -0.04, -0.78, -0.12, -1.16,\n" \
"R_ARC_TO, 4.28, 4.28, 0, 0, 0, -0.41, -1.11,\n" \
"R_LINE_TO, 0.94, -0.95,\n" \
"R_CUBIC_TO, 0.3, 0.49, 0.51, 1, 0.65, 1.54,\n" \
"R_ARC_TO, 6.85, 6.85, 0, 0, 1, 0, 3.36,\n" \
"R_ARC_TO, 5.62, 5.62, 0, 0, 1, -0.66, 1.55,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBluetoothScanningChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBusinessIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 24, 14,\n" \
"V_LINE_TO, 6,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 36,\n" \
"R_H_LINE_TO, 40,\n" \
"V_LINE_TO, 14,\n" \
"H_LINE_TO, 24,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 38,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8, 24,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 20, 24,\n" \
"H_LINE_TO, 24,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 16,\n" \
"R_V_LINE_TO, 20,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4, -16,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 8,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBusinessIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseBusinessChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 1.48, 13.79,\n" \
"V_LINE_TO, 2.34,\n" \
"R_H_LINE_TO, 6.65,\n" \
"R_V_LINE_TO, 2.4,\n" \
"R_H_LINE_TO, 6.4,\n" \
"R_V_LINE_TO, 9.05,\n" \
"CLOSE,\n" \
"MOVE_TO, 2.8, 12.46,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -1.2,\n" \
"H_LINE_TO, 2.8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.53,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -1.2,\n" \
"H_LINE_TO, 2.8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.53,\n" \
"H_LINE_TO, 4,\n" \
"V_LINE_TO, 6.19,\n" \
"H_LINE_TO, 2.8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.54,\n" \
"H_LINE_TO, 4,\n" \
"V_LINE_TO, 3.66,\n" \
"H_LINE_TO, 2.8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.8, 7.6,\n" \
"R_H_LINE_TO, 1.2,\n" \
"R_V_LINE_TO, -1.2,\n" \
"H_LINE_TO, 5.6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.53,\n" \
"R_H_LINE_TO, 1.2,\n" \
"R_V_LINE_TO, -1.2,\n" \
"H_LINE_TO, 5.6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.53,\n" \
"R_H_LINE_TO, 1.2,\n" \
"V_LINE_TO, 6.19,\n" \
"H_LINE_TO, 5.6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.54,\n" \
"R_H_LINE_TO, 1.2,\n" \
"V_LINE_TO, 3.66,\n" \
"H_LINE_TO, 5.6,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 12.46,\n" \
"R_H_LINE_TO, 5.2,\n" \
"V_LINE_TO, 6.06,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 1.34,\n" \
"R_H_LINE_TO, 1.2,\n" \
"R_V_LINE_TO, 1.2,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 1.33,\n" \
"R_H_LINE_TO, 1.2,\n" \
"R_V_LINE_TO, 1.2,\n" \
"H_LINE_TO, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.8, -3.87,\n" \
"R_V_LINE_TO, -1.2,\n" \
"H_LINE_TO, 12,\n" \
"R_V_LINE_TO, 1.2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 2.53,\n" \
"R_V_LINE_TO, -1.2,\n" \
"H_LINE_TO, 12,\n" \
"R_V_LINE_TO, 1.2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kBusinessChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCallIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"// Original SVG image: https://material.io/tools/icons/?icon=call\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 20.01, 15.38,\n" \
"R_CUBIC_TO, -1.23, 0, -2.42, -0.2, -3.53, -0.56,\n" \
"R_CUBIC_TO, -0.35, -0.12, -0.74, -0.03, -1.01, 0.24,\n" \
"R_LINE_TO, -1.57, 1.97,\n" \
"R_CUBIC_TO, -2.83, -1.35, -5.48, -3.9, -6.89, -6.83,\n" \
"R_LINE_TO, 1.95, -1.66,\n" \
"R_CUBIC_TO, 0.27, -0.28, 0.35, -0.67, 0.24, -1.02,\n" \
"R_CUBIC_TO, -0.37, -1.11, -0.56, -2.3, -0.56, -3.53,\n" \
"R_CUBIC_TO, 0, -0.54, -0.45, -0.99, -0.99, -0.99,\n" \
"H_LINE_TO, 4.19,\n" \
"CUBIC_TO, 3.65, 3, 3, 3.24, 3, 3.99,\n" \
"CUBIC_TO, 3, 13.28, 10.73, 21, 20.01, 21,\n" \
"R_CUBIC_TO, 0.71, 0, 0.99, -0.63, 0.99, -1.18,\n" \
"R_V_LINE_TO, -3.45,\n" \
"R_CUBIC_TO, 0, -0.54, -0.45, -0.99, -0.99, -0.99,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCallIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCallEndIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 23.62, 11.27,\n" \
"R_CUBIC_TO, -2.03, -1.72, -4.46, -3, -7.12, -3.69,\n" \
"CUBIC_TO, 15.06, 7.21, 13.56, 7, 12, 7,\n" \
"R_CUBIC_TO, -1.56, 0, -3.06, 0.21, -4.5, 0.58,\n" \
"R_CUBIC_TO, -2.66, 0.69, -5.08, 1.96, -7.12, 3.69,\n" \
"R_CUBIC_TO, -0.45, 0.38, -0.5, 1.07, -0.08, 1.49,\n" \
"R_LINE_TO, 0.67, 0.67,\n" \
"R_LINE_TO, 2.26, 2.26,\n" \
"R_CUBIC_TO, 0.33, 0.33, 0.85, 0.39, 1.25, 0.13,\n" \
"R_LINE_TO, 2.56, -1.64,\n" \
"R_CUBIC_TO, 0.29, -0.18, 0.46, -0.5, 0.46, -0.84,\n" \
"V_LINE_TO, 9.65,\n" \
"CUBIC_TO, 8.93, 9.23, 10.44, 9, 12, 9,\n" \
"R_CUBIC_TO, 1.56, 0, 3.07, 0.23, 4.5, 0.65,\n" \
"R_V_LINE_TO, 3.68,\n" \
"R_CUBIC_TO, 0, 0.34, 0.17, 0.66, 0.46, 0.84,\n" \
"R_LINE_TO, 2.56, 1.64,\n" \
"R_CUBIC_TO, 0.4, 0.25, 0.92, 0.2, 1.25, -0.13,\n" \
"R_LINE_TO, 2.26, -2.26,\n" \
"R_LINE_TO, 0.67, -0.67,\n" \
"R_CUBIC_TO, 0.41, -0.41, 0.37, -1.1, -0.08, -1.48,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCallEndIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCallRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 15.9, 17.13,\n" \
"R_ARC_TO, 13.89, 13.89, 0, 0, 1, -4.9, -1.3,\n" \
"R_ARC_TO, 14.17, 14.17, 0, 0, 1, -4.04, -2.84,\n" \
"R_ARC_TO, 14.69, 14.69, 0, 0, 1, -2.84, -4.05,\n" \
"R_ARC_TO, 13.54, 13.54, 0, 0, 1, -1.28, -4.89,\n" \
"R_ARC_TO, 1.11, 1.11, 0, 0, 1, 0.3, -0.87,\n" \
"R_CUBIC_TO, 0.23, -0.25, 0.52, -0.37, 0.86, -0.37,\n" \
"R_H_LINE_TO, 2.83,\n" \
"R_CUBIC_TO, 0.31, 0, 0.57, 0.09, 0.79, 0.25,\n" \
"R_CUBIC_TO, 0.21, 0.17, 0.36, 0.4, 0.43, 0.71,\n" \
"R_LINE_TO, 0.49, 2.13,\n" \
"R_CUBIC_TO, 0.04, 0.22, 0.04, 0.43, -0.02, 0.63,\n" \
"R_ARC_TO, 1.11, 1.11, 0, 0, 1, -0.32, 0.53,\n" \
"R_LINE_TO, -2.05, 2,\n" \
"R_CUBIC_TO, 0.56, 1, 1.23, 1.89, 2.03, 2.69,\n" \
"R_ARC_TO, 12.11, 12.11, 0, 0, 0, 2.67, 2.01,\n" \
"R_LINE_TO, 2.1, -2.02,\n" \
"R_CUBIC_TO, 0.18, -0.17, 0.36, -0.28, 0.56, -0.32,\n" \
"R_CUBIC_TO, 0.2, -0.04, 0.4, -0.04, 0.61, 0,\n" \
"R_LINE_TO, 2.09, 0.47,\n" \
"R_CUBIC_TO, 0.31, 0.07, 0.54, 0.22, 0.71, 0.43,\n" \
"R_CUBIC_TO, 0.17, 0.22, 0.25, 0.48, 0.25, 0.79,\n" \
"R_V_LINE_TO, 2.87,\n" \
"R_CUBIC_TO, 0, 0.42, -0.15, 0.73, -0.44, 0.91,\n" \
"R_CUBIC_TO, -0.29, 0.18, -0.57, 0.27, -0.82, 0.25,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.34, 7.41,\n" \
"LINE_TO, 6.8, 6,\n" \
"R_LINE_TO, -0.35, -1.46,\n" \
"H_LINE_TO, 4.61,\n" \
"R_CUBIC_TO, 0.06, 0.5, 0.15, 0.99, 0.27, 1.47,\n" \
"R_CUBIC_TO, 0.12, 0.48, 0.27, 0.95, 0.46, 1.41,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7.18, 7.18,\n" \
"R_CUBIC_TO, 0.47, 0.19, 0.95, 0.34, 1.43, 0.45,\n" \
"R_CUBIC_TO, 0.49, 0.11, 0.98, 0.21, 1.48, 0.28,\n" \
"V_LINE_TO, 13.48,\n" \
"R_LINE_TO, -1.46, -0.32,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -7.18, -7.18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7.18, 7.18,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCallRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCancelIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 12, 2,\n" \
"CUBIC_TO, 6.47, 2, 2, 6.47, 2, 12,\n" \
"R_CUBIC_TO, 0, 5.53, 4.47, 10, 10, 10,\n" \
"R_CUBIC_TO, 5.53, 0, 10, -4.47, 10, -10,\n" \
"CUBIC_TO, 22, 6.47, 17.53, 2, 12, 2,\n" \
"CLOSE,\n" \
"MOVE_TO, 17, 15.59,\n" \
"LINE_TO, 15.59, 17,\n" \
"LINE_TO, 12, 13.41,\n" \
"LINE_TO, 8.41, 17,\n" \
"LINE_TO, 7, 15.59,\n" \
"LINE_TO, 10.59, 12,\n" \
"LINE_TO, 7, 8.41,\n" \
"LINE_TO, 8.41, 7,\n" \
"LINE_TO, 12, 10.59,\n" \
"LINE_TO, 15.59, 7,\n" \
"LINE_TO, 17, 8.41,\n" \
"LINE_TO, 13.41, 12,\n" \
"LINE_TO, 17, 15.59,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCancelIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCancelChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 7.06, 14,\n" \
"LINE_TO, 10, 11.06,\n" \
"LINE_TO, 12.94, 14,\n" \
"LINE_TO, 14, 12.94,\n" \
"LINE_TO, 11.06, 10,\n" \
"LINE_TO, 14, 7.06,\n" \
"LINE_TO, 12.94, 6,\n" \
"LINE_TO, 10, 8.94,\n" \
"LINE_TO, 7.06, 6,\n" \
"LINE_TO, 6, 7.06,\n" \
"LINE_TO, 8.94, 10,\n" \
"LINE_TO, 6, 12.94,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 18,\n" \
"R_ARC_TO, 7.78, 7.78, 0, 0, 1, -3.1, -0.62,\n" \
"R_ARC_TO, 8.07, 8.07, 0, 0, 1, -2.55, -1.72,\n" \
"R_ARC_TO, 8.07, 8.07, 0, 0, 1, -1.72, -2.55,\n" \
"ARC_TO, 7.78, 7.78, 0, 0, 1, 2, 10,\n" \
"R_CUBIC_TO, 0, -1.11, 0.21, -2.15, 0.63, -3.11,\n" \
"R_ARC_TO, 8.12, 8.12, 0, 0, 1, 1.72, -2.54,\n" \
"R_CUBIC_TO, 0.73, -0.73, 1.58, -1.3, 2.55, -1.72,\n" \
"ARC_TO, 7.78, 7.78, 0, 0, 1, 10, 2,\n" \
"R_ARC_TO, 7.75, 7.75, 0, 0, 1, 3.11, 0.63,\n" \
"R_ARC_TO, 8.12, 8.12, 0, 0, 1, 2.54, 1.72,\n" \
"R_CUBIC_TO, 0.73, 0.73, 1.3, 1.58, 1.72, 2.54,\n" \
"ARC_TO, 7.75, 7.75, 0, 0, 1, 18, 10,\n" \
"R_ARC_TO, 7.78, 7.78, 0, 0, 1, -0.62, 3.11,\n" \
"R_ARC_TO, 8.07, 8.07, 0, 0, 1, -1.72, 2.55,\n" \
"R_CUBIC_TO, -0.73, 0.73, -1.58, 1.3, -2.54, 1.72,\n" \
"ARC_TO, 7.75, 7.75, 0, 0, 1, 10, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_CUBIC_TO, 1.81, 0, 3.34, -0.63, 4.61, -1.89,\n" \
"CUBIC_TO, 15.87, 13.34, 16.5, 11.81, 16.5, 10,\n" \
"R_CUBIC_TO, 0, -1.8, -0.63, -3.34, -1.89, -4.61,\n" \
"CUBIC_TO, 13.34, 4.13, 11.81, 3.5, 10, 3.5,\n" \
"R_CUBIC_TO, -1.8, 0, -3.34, 0.63, -4.61, 1.9,\n" \
"CUBIC_TO, 4.13, 6.66, 3.5, 8.2, 3.5, 10,\n" \
"R_CUBIC_TO, 0, 1.81, 0.63, 3.34, 1.9, 4.61,\n" \
"CUBIC_TO, 6.66, 15.87, 8.2, 16.5, 10, 16.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -6.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCancelChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCaptureIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 1.83, 16.17,\n" \
"V_LINE_TO, 3.83,\n" \
"R_H_LINE_TO, 16.34,\n" \
"R_V_LINE_TO, 12.34,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.73, -1.73,\n" \
"H_LINE_TO, 16.44,\n" \
"V_LINE_TO, 5.56,\n" \
"H_LINE_TO, 3.56,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.5, -1.5,\n" \
"R_H_LINE_TO, 9.88,\n" \
"V_LINE_TO, 7.06,\n" \
"H_LINE_TO, 5.06,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCaptureIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCardboardIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 4.29, 13.12,\n" \
"R_H_LINE_TO, 3.3,\n" \
"R_V_LINE_TO, -0.01,\n" \
"R_LINE_TO, 0.66, -1.26,\n" \
"R_CUBIC_TO, 0.18, -0.33, 0.42, -0.6, 0.73, -0.79,\n" \
"ARC_TO, 1.91, 1.91, 0, 0, 1, 10, 10.77,\n" \
"R_CUBIC_TO, 0.38, 0, 0.72, 0.09, 1.03, 0.28,\n" \
"R_CUBIC_TO, 0.31, 0.19, 0.56, 0.45, 0.73, 0.78,\n" \
"R_LINE_TO, 0.66, 1.28,\n" \
"R_V_LINE_TO, 0.01,\n" \
"R_H_LINE_TO, 3.3,\n" \
"V_LINE_TO, 6.88,\n" \
"H_LINE_TO, 4.29,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.1, -1.56,\n" \
"R_CUBIC_TO, 0.43, 0, 0.8, -0.15, 1.1, -0.46,\n" \
"R_CUBIC_TO, 0.31, -0.31, 0.46, -0.68, 0.46, -1.11,\n" \
"R_CUBIC_TO, 0, -0.43, -0.16, -0.8, -0.46, -1.1,\n" \
"R_ARC_TO, 1.52, 1.52, 0, 0, 0, -1.11, -0.45,\n" \
"R_CUBIC_TO, -0.43, 0, -0.8, 0.15, -1.1, 0.46,\n" \
"R_ARC_TO, 1.52, 1.52, 0, 0, 0, -0.46, 1.11,\n" \
"R_CUBIC_TO, 0, 0.43, 0.15, 0.8, 0.46, 1.1,\n" \
"R_CUBIC_TO, 0.31, 0.3, 0.68, 0.45, 1.11, 0.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.1, 3.13,\n" \
"R_ARC_TO, 1.51, 1.51, 0, 0, 1, -1.1, -0.46,\n" \
"R_ARC_TO, 1.52, 1.52, 0, 0, 1, -0.46, -1.1,\n" \
"V_LINE_TO, 6.88,\n" \
"R_CUBIC_TO, 0, -0.42, 0.16, -0.78, 0.46, -1.09,\n" \
"R_CUBIC_TO, 0.31, -0.31, 0.68, -0.47, 1.09, -0.47,\n" \
"R_H_LINE_TO, 11.43,\n" \
"R_CUBIC_TO, 0.42, 0, 0.79, 0.16, 1.09, 0.47,\n" \
"R_CUBIC_TO, 0.31, 0.31, 0.46, 0.68, 0.46, 1.1,\n" \
"R_V_LINE_TO, 6.24,\n" \
"R_CUBIC_TO, 0, 0.42, -0.16, 0.79, -0.46, 1.09,\n" \
"R_ARC_TO, 1.5, 1.5, 0, 0, 1, -1.1, 0.47,\n" \
"R_H_LINE_TO, -3.29,\n" \
"R_CUBIC_TO, -0.3, 0, -0.58, -0.08, -0.83, -0.24,\n" \
"R_ARC_TO, 1.52, 1.52, 0, 0, 1, -0.57, -0.61,\n" \
"R_LINE_TO, -0.66, -1.26,\n" \
"R_ARC_TO, 0.54, 0.54, 0, 0, 0, -0.16, -0.17,\n" \
"R_ARC_TO, 0.37, 0.37, 0, 0, 0, -0.21, -0.07,\n" \
"R_ARC_TO, 0.46, 0.46, 0, 0, 0, -0.38, 0.24,\n" \
"R_LINE_TO, -0.64, 1.27,\n" \
"R_CUBIC_TO, -0.12, 0.25, -0.31, 0.46, -0.57, 0.61,\n" \
"R_CUBIC_TO, -0.25, 0.16, -0.53, 0.24, -0.83, 0.24,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 9.34, -3.12,\n" \
"R_CUBIC_TO, 0.43, 0, 0.8, -0.15, 1.1, -0.46,\n" \
"R_CUBIC_TO, 0.3, -0.31, 0.46, -0.68, 0.46, -1.11,\n" \
"R_CUBIC_TO, 0, -0.43, -0.15, -0.8, -0.46, -1.1,\n" \
"R_ARC_TO, 1.52, 1.52, 0, 0, 0, -1.11, -0.45,\n" \
"R_CUBIC_TO, -0.43, 0, -0.8, 0.15, -1.1, 0.46,\n" \
"R_ARC_TO, 1.52, 1.52, 0, 0, 0, -0.46, 1.11,\n" \
"R_CUBIC_TO, 0, 0.43, 0.16, 0.8, 0.46, 1.1,\n" \
"R_CUBIC_TO, 0.31, 0.3, 0.68, 0.45, 1.11, 0.45,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCardboardIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCaretDownIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"// A variant of ic_expand_more with rounded corners.\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"STROKE, 3,\n" \
"MOVE_TO, 8.5, 12.5,\n" \
"R_LINE_TO, 7.5, 8,\n" \
"R_LINE_TO, 7.5, -8,\n" \
"// Hard clip at path points +- half the stroke.\n" \
"CLIP, 7, 11, 18, 11,\n" \
"\n" \
"// A variant of ic_expand_more with rounded corners and optimized for 1x\n" \
"// scale factor devices.\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"STROKE, 1.765,\n" \
"MOVE_TO, 4, 6,\n" \
"R_LINE_TO, 4, 4,\n" \
"R_LINE_TO, 4, -4,\n" \
; \
              CheckThatParsedElementsMatch(s, kCaretDownIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCaretUpIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"// A variant of ic_expand_less with rounded corners.\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"// Hard clip at path points +- half the stroke.\n" \
"CLIP, 7, 10, 18, 11,\n" \
"STROKE, 3,\n" \
"MOVE_TO, 8.5, 19.5,\n" \
"R_LINE_TO, 7.5, -8,\n" \
"R_LINE_TO, 7.5, 8,\n" \
"\n" \
"// A variant of ic_expand_less with rounded corners and optimized for 1x\n" \
"// scale factor devices.\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"STROKE, 1.765,\n" \
"MOVE_TO, 4, 10,\n" \
"R_LINE_TO, 4, -4,\n" \
"R_LINE_TO, 4, 4,\n" \
; \
              CheckThatParsedElementsMatch(s, kCaretUpIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCastIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 480, 480,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 320, 320,\n" \
"H_LINE_TO, 600,\n" \
"R_QUADRATIC_TO, 0, -20, -1.5, -40,\n" \
"R_QUADRATIC_TO, -1.5, -20, -4.5, -40,\n" \
"R_H_LINE_TO, 206,\n" \
"R_V_LINE_TO, -480,\n" \
"H_LINE_TO, 160,\n" \
"R_V_LINE_TO, 46,\n" \
"R_QUADRATIC_TO, -20, -3, -40, -4.5,\n" \
"QUADRATIC_TO_SHORTHAND, 80, 280,\n" \
"R_V_LINE_TO, -40,\n" \
"R_QUADRATIC_TO, 0, -33, 23.5, -56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 160, 160,\n" \
"R_H_LINE_TO, 640,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 880, 240,\n" \
"R_V_LINE_TO, 480,\n" \
"R_QUADRATIC_TO, 0, 33, -23.5, 56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 800, 800,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -720, 0,\n" \
"R_V_LINE_TO, -120,\n" \
"R_QUADRATIC_TO, 50, 0, 85, 35,\n" \
"R_QUADRATIC_TO, 35, 35, 35, 85,\n" \
"H_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 200, 0,\n" \
"R_QUADRATIC_TO, 0, -83, -58.5, -141.5,\n" \
"QUADRATIC_TO_SHORTHAND, 80, 600,\n" \
"R_V_LINE_TO, -80,\n" \
"R_QUADRATIC_TO, 117, 0, 198.5, 81.5,\n" \
"QUADRATIC_TO_SHORTHAND, 360, 800,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 160, 0,\n" \
"R_QUADRATIC_TO, 0, -75, -28.5, -140.5,\n" \
"R_QUADRATIC_TO, -28.5, -65.5, -77, -114,\n" \
"R_QUADRATIC_TO, -48.5, -48.5, -114, -77,\n" \
"QUADRATIC_TO_SHORTHAND, 80, 440,\n" \
"R_V_LINE_TO, -80,\n" \
"R_QUADRATIC_TO, 91, 0, 171, 34.5,\n" \
"QUADRATIC_TO_SHORTHAND, 391, 489,\n" \
"R_QUADRATIC_TO, 60, 60, 94.5, 140,\n" \
"QUADRATIC_TO_SHORTHAND, 520, 800,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCastIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCastWarningIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -8, 4,\n" \
"R_CUBIC_TO, 0.56, 0, 1.03, 0.2, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"H_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"R_CUBIC_TO, 1.39, 0, 2.57, 0.48, 3.54, 1.46,\n" \
"CUBIC_TO, 6.52, 13.43, 7, 14.61, 7, 16,\n" \
"H_LINE_TO, 5.5,\n" \
"R_CUBIC_TO, 0, -0.97, -0.34, -1.8, -1.02, -2.48,\n" \
"ARC_TO, 3.38, 3.38, 0, 0, 0, 2, 12.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"R_CUBIC_TO, 1.11, 0, 2.15, 0.21, 3.12, 0.63,\n" \
"ARC_TO, 8.09, 8.09, 0, 0, 1, 7.66, 10.34,\n" \
"R_ARC_TO, 8.09, 8.09, 0, 0, 1, 1.72, 2.54,\n" \
"CUBIC_TO, 9.79, 13.85, 10, 14.89, 10, 16,\n" \
"H_LINE_TO, 8.5,\n" \
"R_ARC_TO, 6.3, 6.3, 0, 0, 0, -0.51, -2.53,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 0, -1.39, -2.07,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 0, -2.07, -1.39,\n" \
"ARC_TO, 6.3, 6.3, 0, 0, 0, 2, 9.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 14.5, 8,\n" \
"R_H_LINE_TO, -5,\n" \
"R_CUBIC_TO, 0, -0.25, -0.01, -0.5, -0.03, -0.75,\n" \
"R_CUBIC_TO, -0.02, -0.25, -0.05, -0.5, -0.09, -0.75,\n" \
"H_LINE_TO, 16.5,\n" \
"R_V_LINE_TO, -3.87,\n" \
"R_CUBIC_TO, 0.28, -0.1, 0.54, -0.23, 0.79, -0.37,\n" \
"ARC_TO, 4.05, 4.05, 0, 0, 0, 18, 9.75,\n" \
"R_V_LINE_TO, 4.75,\n" \
"R_CUBIC_TO, 0, 0.42, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 16,\n" \
"CLOSE,\n" \
"MOVE_TO, 2, 6.5,\n" \
"R_V_LINE_TO, -1,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 4,\n" \
"R_H_LINE_TO, 5.71,\n" \
"R_ARC_TO, 5.64, 5.64, 0, 0, 0, -0.16, 0.73,\n" \
"CUBIC_TO, 9.02, 4.98, 9, 5.24, 9, 5.5,\n" \
"H_LINE_TO, 3.5,\n" \
"R_V_LINE_TO, 1.13,\n" \
"R_CUBIC_TO, -0.25, -0.04, -0.5, -0.07, -0.75, -0.09,\n" \
"R_CUBIC_TO, -0.25, -0.02, -0.5, -0.03, -0.75, -0.03,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12.5, 3,\n" \
"R_CUBIC_TO, -1.11, 0, -2.05, -0.39, -2.83, -1.17,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 1, 10.5, 5.5,\n" \
"R_CUBIC_TO, 0, -1.11, 0.39, -2.05, 1.17, -2.83,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 1, 14.5, 1.5,\n" \
"R_CUBIC_TO, 1.11, 0, 2.05, 0.39, 2.83, 1.17,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 1, 18.5, 5.5,\n" \
"R_CUBIC_TO, 0, 1.11, -0.39, 2.05, -1.17, 2.83,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 1, 14.5, 9.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2,\n" \
"R_CUBIC_TO, 0.13, 0, 0.25, -0.05, 0.35, -0.15,\n" \
"ARC_TO, 0.49, 0.49, 0, 0, 0, 15, 7,\n" \
"R_CUBIC_TO, 0, -0.13, -0.05, -0.25, -0.15, -0.35,\n" \
"ARC_TO, 0.49, 0.49, 0, 0, 0, 14.5, 6.5,\n" \
"R_CUBIC_TO, -0.13, 0, -0.25, 0.05, -0.35, 0.15,\n" \
"ARC_TO, 0.49, 0.49, 0, 0, 0, 14, 7,\n" \
"R_CUBIC_TO, 0, 0.13, 0.05, 0.25, 0.15, 0.35,\n" \
"R_ARC_TO, 0.49, 0.49, 0, 0, 0, 0.35, 0.15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.5, -2,\n" \
"R_H_LINE_TO, 1,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, -1,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCastWarningIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCelebrationIcon) { \
              std::string s = \
"// Copyright 2022 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"NEW_PATH,\n" \
"MOVE_TO, 2, 22,\n" \
"R_LINE_TO, 14, -5,\n" \
"LINE_TO, 7, 8,\n" \
"LINE_TO, 2, 22,\n" \
"CLOSE,\n" \
"MOVE_TO, 12.35, 16.18,\n" \
"LINE_TO, 5.3, 18.7,\n" \
"R_LINE_TO, 2.52, -7.05,\n" \
"LINE_TO, 12.35, 16.18,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"MOVE_TO, 14.53, 12.53,\n" \
"R_LINE_TO, 5.59, -5.59,\n" \
"R_CUBIC_TO, 0.49, -0.49, 1.28, -0.49, 1.77, 0,\n" \
"R_LINE_TO, 0.59, 0.59,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, -0.59, -0.59,\n" \
"R_CUBIC_TO, -1.07, -1.07, -2.82, -1.07, -3.89, 0,\n" \
"R_LINE_TO, -5.59, 5.59,\n" \
"LINE_TO, 14.53, 12.53,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"MOVE_TO, 10.06, 6.88,\n" \
"LINE_TO, 9.47, 7.47,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"R_LINE_TO, 0.59, -0.59,\n" \
"R_CUBIC_TO, 1.07, -1.07, 1.07, -2.82, 0, -3.89,\n" \
"R_LINE_TO, -0.59, -0.59,\n" \
"LINE_TO, 9.47, 4.53,\n" \
"R_LINE_TO, 0.59, 0.59,\n" \
"CUBIC_TO, 10.54, 5.6, 10.54, 6.4, 10.06, 6.88,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"MOVE_TO, 17.06, 11.88,\n" \
"R_LINE_TO, -1.59, 1.59,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"R_LINE_TO, 1.59, -1.59,\n" \
"R_CUBIC_TO, 0.49, -0.49, 1.28, -0.49, 1.77, 0,\n" \
"R_LINE_TO, 1.61, 1.61,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, -1.61, -1.61,\n" \
"CUBIC_TO, 19.87, 10.81, 18.13, 10.81, 17.06, 11.88,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"MOVE_TO, 15.06, 5.88,\n" \
"R_LINE_TO, -3.59, 3.59,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"R_LINE_TO, 3.59, -3.59,\n" \
"R_CUBIC_TO, 1.07, -1.07, 1.07, -2.82, 0, -3.89,\n" \
"R_LINE_TO, -1.59, -1.59,\n" \
"R_LINE_TO, -1.06, 1.06,\n" \
"R_LINE_TO, 1.59, 1.59,\n" \
"CUBIC_TO, 15.54, 4.6, 15.54, 5.4, 15.06, 5.88,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCelebrationIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCertificateIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.5, 16,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 2, 14.5,\n" \
"V_LINE_TO, 5.49,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.76, 0.44, -1.05,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 4,\n" \
"R_H_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 9.01,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.76, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 16,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 13,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, -13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.63, -0.83,\n" \
"R_LINE_TO, -2.83, -2.84,\n" \
"LINE_TO, 7.36, 9.77,\n" \
"R_LINE_TO, 1.77, 1.77,\n" \
"LINE_TO, 12.65, 8,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.5, 14.5,\n" \
"R_V_LINE_TO, -9,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCertificateIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCertificateOffIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16.73, 18.83,\n" \
"LINE_TO, 13.88, 16,\n" \
"H_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, -0.14, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 2, 14.5,\n" \
"R_V_LINE_TO, -9,\n" \
"R_CUBIC_TO, 0, -0.42, 0.15, -0.77, 0.46, -1.06,\n" \
"ARC_TO, 1.55, 1.55, 0, 0, 1, 3.56, 4,\n" \
"R_H_LINE_TO, 0.23,\n" \
"R_V_LINE_TO, 1.9,\n" \
"LINE_TO, 1.17, 3.27,\n" \
"LINE_TO, 2.23, 2.23,\n" \
"R_LINE_TO, 15.54, 15.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.5, 14.5,\n" \
"R_H_LINE_TO, 8.88,\n" \
"LINE_TO, 4.88, 7,\n" \
"H_LINE_TO, 3.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 14.13, 1,\n" \
"LINE_TO, 16.5, 14.38,\n" \
"V_LINE_TO, 7,\n" \
"H_LINE_TO, 9.13,\n" \
"R_LINE_TO, -3, -3,\n" \
"H_LINE_TO, 16.5,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, 0.14, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.64, 0.44, 1.06,\n" \
"R_V_LINE_TO, 9,\n" \
"R_CUBIC_TO, 0, 0.2, -0.04, 0.38, -0.1, 0.56,\n" \
"R_CUBIC_TO, -0.07, 0.18, -0.16, 0.33, -0.27, 0.44,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5.17, -5.17,\n" \
"LINE_TO, 11.4, 9.27,\n" \
"LINE_TO, 12.65, 8,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.06, 1.06,\n" \
"R_LINE_TO, -2.27, 2.27,\n" \
"R_LINE_TO, -2.83, -2.84,\n" \
"LINE_TO, 7.36, 9.77,\n" \
"R_LINE_TO, 1.77, 1.77,\n" \
"R_LINE_TO, 1.21, -1.21,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCertificateOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseChatIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 5, 12,\n" \
"R_H_LINE_TO, 7,\n" \
"R_V_LINE_TO, -1.5,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.75,\n" \
"R_H_LINE_TO, 10,\n" \
"R_V_LINE_TO, -1.5,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"MOVE_TO, 5, 6.5,\n" \
"R_H_LINE_TO, 10,\n" \
"V_LINE_TO, 5,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"MOVE_TO, 2, 18,\n" \
"V_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 2,\n" \
"R_H_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 15,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.38, -4.5,\n" \
"H_LINE_TO, 16.5,\n" \
"R_V_LINE_TO, -10,\n" \
"R_H_LINE_TO, -13,\n" \
"R_V_LINE_TO, 10.88,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.87, 0,\n" \
"R_V_LINE_TO, -10,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kChatIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseChatSparkIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 14.5, 10,\n" \
"R_CUBIC_TO, 0, -1.25, -0.44, -2.32, -1.31, -3.19,\n" \
"CUBIC_TO, 12.32, 5.94, 11.25, 5.5, 10, 5.5,\n" \
"R_CUBIC_TO, 1.25, 0, 2.32, -0.44, 3.19, -1.31,\n" \
"CUBIC_TO, 14.06, 3.32, 14.5, 2.25, 14.5, 1,\n" \
"R_CUBIC_TO, 0, 1.25, 0.44, 2.32, 1.31, 3.19,\n" \
"CUBIC_TO, 16.68, 5.06, 17.75, 5.5, 19, 5.5,\n" \
"R_CUBIC_TO, -1.25, 0, -2.32, 0.44, -3.19, 1.31,\n" \
"CUBIC_TO, 14.94, 7.68, 14.5, 8.75, 14.5, 10,\n" \
"CLOSE,\n" \
"MOVE_TO, 2, 18,\n" \
"V_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0, -0.4, 0.15, -0.75, 0.44, -1.05,\n" \
"CUBIC_TO, 2.73, 2.15, 3.09, 2, 3.5, 2,\n" \
"H_LINE_TO, 11,\n" \
"R_V_LINE_TO, 1.5,\n" \
"H_LINE_TO, 3.5,\n" \
"R_V_LINE_TO, 10.88,\n" \
"R_LINE_TO, 0.88, -0.87,\n" \
"H_LINE_TO, 16.5,\n" \
"V_LINE_TO, 9,\n" \
"H_LINE_TO, 18,\n" \
"R_V_LINE_TO, 4.51,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.76, -0.45, 1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 16.5, 15,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3, -6,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 8,\n" \
"V_LINE_TO, 12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.75,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 7,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 5, 6.5,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kChatSparkIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCheckCircleIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CIRCLE, 24, 24, 20,\n" \
"MOVE_TO, 20, 34,\n" \
"LINE_TO, 10, 24,\n" \
"R_LINE_TO, 2.83, -2.83,\n" \
"LINE_TO, 20, 28.34,\n" \
"R_LINE_TO, 15.17, -15.17,\n" \
"LINE_TO, 38, 16,\n" \
"LINE_TO, 20, 34\n" \
; \
              CheckThatParsedElementsMatch(s, kCheckCircleIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseChecklistIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 5.55, 19,\n" \
"LINE_TO, 2, 15.45,\n" \
"R_LINE_TO, 1.4, -1.4,\n" \
"R_LINE_TO, 2.13, 2.13,\n" \
"R_LINE_TO, 4.25, -4.25,\n" \
"R_LINE_TO, 1.4, 1.43,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"LINE_TO, 2, 7.45,\n" \
"R_LINE_TO, 1.4, -1.4,\n" \
"R_LINE_TO, 2.13, 2.13,\n" \
"R_LINE_TO, 4.25, -4.25,\n" \
"R_LINE_TO, 1.4, 1.43,\n" \
"CLOSE,\n" \
"MOVE_TO, 13, 17,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 9,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, 9,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kChecklistIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCloseIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 19, 6.41,\n" \
"LINE_TO, 17.59, 5,\n" \
"LINE_TO, 12, 10.59,\n" \
"LINE_TO, 6.41, 5,\n" \
"LINE_TO, 5, 6.41,\n" \
"LINE_TO, 10.59, 12,\n" \
"LINE_TO, 5, 17.59,\n" \
"LINE_TO, 6.41, 19,\n" \
"LINE_TO, 12, 13.41,\n" \
"LINE_TO, 17.59, 19,\n" \
"LINE_TO, 19, 17.59,\n" \
"LINE_TO, 13.41, 12,\n" \
"CLOSE\n" \
"\n" \
"// A variant of ic_close with rounded corners and optimized for 1x scale factor\n" \
"// devices.\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"STROKE, 1.85,\n" \
"MOVE_TO, 4, 4,\n" \
"R_LINE_TO, 8, 8,\n" \
"MOVE_TO, 4, 12,\n" \
"R_LINE_TO, 8, -8\n" \
; \
              CheckThatParsedElementsMatch(s, kCloseIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCloseChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 6.06, 15,\n" \
"LINE_TO, 5, 13.94,\n" \
"LINE_TO, 8.94, 10,\n" \
"LINE_TO, 5, 6.06,\n" \
"LINE_TO, 6.06, 5,\n" \
"LINE_TO, 10, 8.94,\n" \
"LINE_TO, 13.94, 5,\n" \
"LINE_TO, 15, 6.06,\n" \
"LINE_TO, 11.06, 10,\n" \
"LINE_TO, 15, 13.94,\n" \
"LINE_TO, 13.94, 15,\n" \
"LINE_TO, 10, 11.06,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCloseChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCloseRoundedIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"// A variant of ic_close with rounded corners.\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"STROKE, 3.5,\n" \
"MOVE_TO, 8.75, 8.75,\n" \
"R_LINE_TO, 14.5, 14.5,\n" \
"MOVE_TO, 8.75, 23.25,\n" \
"R_LINE_TO, 14.5, -14.5\n" \
"\n" \
"// A variant of ic_close with rounded corners and optimized for 1x scale factor\n" \
"// devices.\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"STROKE, 1.85,\n" \
"MOVE_TO, 4, 4,\n" \
"R_LINE_TO, 8, 8,\n" \
"MOVE_TO, 4, 12,\n" \
"R_LINE_TO, 8, -8\n" \
; \
              CheckThatParsedElementsMatch(s, kCloseRoundedIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCloseSmallIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 7.04, 14,\n" \
"LINE_TO, 6, 12.96,\n" \
"LINE_TO, 8.94, 10,\n" \
"LINE_TO, 6, 7.06,\n" \
"LINE_TO, 7.04, 6.02,\n" \
"LINE_TO, 10, 8.96,\n" \
"R_LINE_TO, 2.94, -2.94,\n" \
"R_LINE_TO, 1.04, 1.04,\n" \
"LINE_TO, 11.04, 10,\n" \
"R_LINE_TO, 2.94, 2.96,\n" \
"LINE_TO, 12.94, 14,\n" \
"LINE_TO, 10, 11.06,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCloseSmallIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCodeIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 18.8, 33.2,\n" \
"LINE_TO, 9.7, 24,\n" \
"R_LINE_TO, 9.2, -9.2,\n" \
"LINE_TO, 16, 12,\n" \
"LINE_TO, 4, 24,\n" \
"R_LINE_TO, 12, 12,\n" \
"R_LINE_TO, 2.8, -2.8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 10.4, 0,\n" \
"R_LINE_TO, 9.2, -9.2,\n" \
"R_LINE_TO, -9.2, -9.2,\n" \
"LINE_TO, 32, 12,\n" \
"R_LINE_TO, 12, 12,\n" \
"R_LINE_TO, -12, 12,\n" \
"R_LINE_TO, -2.8, -2.8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCodeIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCodeChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 7, 15,\n" \
"R_LINE_TO, -5, -5,\n" \
"R_LINE_TO, 5, -5,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"LINE_TO, 4.12, 10,\n" \
"R_LINE_TO, 3.94, 3.94,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6, 0,\n" \
"R_LINE_TO, -1.06, -1.06,\n" \
"LINE_TO, 15.88, 10,\n" \
"R_LINE_TO, -3.94, -3.94,\n" \
"LINE_TO, 13, 5,\n" \
"R_LINE_TO, 5, 5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCodeChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCodeOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 16, 18.13,\n" \
"R_LINE_TO, -10, -10,\n" \
"LINE_TO, 4.13, 10,\n" \
"R_LINE_TO, 3.94, 3.94,\n" \
"LINE_TO, 7, 15,\n" \
"R_LINE_TO, -5, -5,\n" \
"R_LINE_TO, 2.94, -2.94,\n" \
"LINE_TO, 1.87, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.94, -5.19,\n" \
"LINE_TO, 14, 11.88,\n" \
"LINE_TO, 15.88, 10,\n" \
"R_LINE_TO, -3.94, -3.94,\n" \
"LINE_TO, 13, 5,\n" \
"R_LINE_TO, 5, 5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCodeOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseContentCopyIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 9, 18,\n" \
"R_QUADRATIC_TO, -0.82, 0, -1.41, -0.59,\n" \
"QUADRATIC_TO, 7, 16.83, 7, 16,\n" \
"V_LINE_TO, 4,\n" \
"R_QUADRATIC_TO, 0, -0.82, 0.59, -1.41,\n" \
"QUADRATIC_TO, 8.18, 2, 9, 2,\n" \
"R_H_LINE_TO, 9,\n" \
"R_QUADRATIC_TO, 0.83, 0, 1.41, 0.59,\n" \
"QUADRATIC_TO, 20, 3.18, 20, 4,\n" \
"R_V_LINE_TO, 12,\n" \
"R_QUADRATIC_TO, 0, 0.83, -0.59, 1.41,\n" \
"QUADRATIC_TO, 18.83, 18, 18, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2,\n" \
"R_H_LINE_TO, 9,\n" \
"V_LINE_TO, 4,\n" \
"H_LINE_TO, 9,\n" \
"R_V_LINE_TO, 12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4, 6,\n" \
"R_QUADRATIC_TO, -0.82, 0, -1.41, -0.59,\n" \
"QUADRATIC_TO, 3, 20.83, 3, 20,\n" \
"V_LINE_TO, 6,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 14,\n" \
"R_H_LINE_TO, 11,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"MOVE_TO, 9, 4,\n" \
"R_V_LINE_TO, 12,\n" \
"V_LINE_TO, 4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kContentCopyIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseContentPasteIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 5, 21,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, -0.19, -1.41, -0.59,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 3, 19,\n" \
"V_LINE_TO, 5,\n" \
"R_CUBIC_TO, 0, -0.55, 0.2, -1.02, 0.59, -1.41,\n" \
"CUBIC_TO, 3.98, 3.2, 4.45, 3, 5, 3,\n" \
"R_H_LINE_TO, 4.18,\n" \
"R_ARC_TO, 2.72, 2.72, 0, 0, 1, 1.07, -1.44,\n" \
"ARC_TO, 2.98, 2.98, 0, 0, 1, 12, 1,\n" \
"R_CUBIC_TO, 0.67, 0, 1.26, 0.19, 1.79, 0.56,\n" \
"R_CUBIC_TO, 0.52, 0.38, 0.88, 0.86, 1.06, 1.44,\n" \
"H_LINE_TO, 19,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, 0.2, 1.41, 0.59,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.59, 0.86, 0.59, 1.41,\n" \
"R_V_LINE_TO, 14,\n" \
"R_CUBIC_TO, 0, 0.55, -0.19, 1.02, -0.59, 1.41,\n" \
"R_CUBIC_TO, -0.39, 0.39, -0.86, 0.59, -1.41, 0.59,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2,\n" \
"R_H_LINE_TO, 14,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 3,\n" \
"H_LINE_TO, 7,\n" \
"V_LINE_TO, 5,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7, -14,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, -0.1, 0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 13, 4,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 12, 3,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.71, 0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 11, 4,\n" \
"R_CUBIC_TO, 0, 0.28, 0.1, 0.52, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.43, 0.29, 0.71, 0.29,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 17,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3, 15.5,\n" \
"R_V_LINE_TO, -11,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4.5, 3,\n" \
"R_H_LINE_TO, 3.56,\n" \
"R_CUBIC_TO, 0.11, -0.43, 0.34, -0.79, 0.7, -1.07,\n" \
"ARC_TO, 1.91, 1.91, 0, 0, 1, 10, 1.5,\n" \
"R_CUBIC_TO, 0.47, 0, 0.89, 0.14, 1.24, 0.43,\n" \
"R_CUBIC_TO, 0.36, 0.28, 0.59, 0.64, 0.7, 1.07,\n" \
"H_LINE_TO, 15.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 15.5, 17,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 11,\n" \
"R_V_LINE_TO, -11,\n" \
"H_LINE_TO, 14,\n" \
"V_LINE_TO, 7,\n" \
"H_LINE_TO, 6,\n" \
"V_LINE_TO, 4.5,\n" \
"H_LINE_TO, 4.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.5, -11,\n" \
"R_CUBIC_TO, 0.21, 0, 0.39, -0.07, 0.54, -0.21,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, 0.22, -0.53,\n" \
"R_CUBIC_TO, 0, -0.21, -0.07, -0.39, -0.21, -0.53,\n" \
"ARC_TO, 0.72, 0.72, 0, 0, 0, 10, 3,\n" \
"R_CUBIC_TO, -0.21, 0, -0.39, 0.07, -0.53, 0.22,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, -0.22, 0.53,\n" \
"R_CUBIC_TO, 0, 0.21, 0.07, 0.39, 0.22, 0.54,\n" \
"R_CUBIC_TO, 0.14, 0.14, 0.32, 0.22, 0.53, 0.22,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kContentPasteIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseContentPasteOffIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 21, 18.15,\n" \
"R_LINE_TO, -2, -2,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, -6.15,\n" \
"R_LINE_TO, -5, -5,\n" \
"R_H_LINE_TO, 3.32,\n" \
"R_ARC_TO, 2.72, 2.72, 0, 0, 1, 1.07, -1.44,\n" \
"ARC_TO, 2.98, 2.98, 0, 0, 1, 12, 1,\n" \
"R_CUBIC_TO, 0.67, 0, 1.26, 0.19, 1.79, 0.56,\n" \
"R_CUBIC_TO, 0.52, 0.38, 0.88, 0.86, 1.06, 1.44,\n" \
"H_LINE_TO, 19,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, 0.2, 1.41, 0.59,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.59, 0.86, 0.59, 1.41,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 5,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, -0.1, 0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 13, 4,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 12, 3,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.71, 0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 11, 4,\n" \
"R_CUBIC_TO, 0, 0.28, 0.1, 0.52, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.43, 0.29, 0.71, 0.29,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.15, 14,\n" \
"LINE_TO, 5, 7.85,\n" \
"V_LINE_TO, 19,\n" \
"CLOSE,\n" \
"MOVE_TO, 5, 21,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, -0.19, -1.41, -0.59,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 3, 19,\n" \
"V_LINE_TO, 5.85,\n" \
"LINE_TO, 1.38, 4.23,\n" \
"LINE_TO, 2.8, 2.8,\n" \
"R_LINE_TO, 18.4, 18.4,\n" \
"R_LINE_TO, -1.43, 1.43,\n" \
"LINE_TO, 18.15, 21,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 17, 14.88,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 4.5,\n" \
"H_LINE_TO, 14,\n" \
"V_LINE_TO, 7,\n" \
"H_LINE_TO, 9.13,\n" \
"R_LINE_TO, -4, -4,\n" \
"R_H_LINE_TO, 2.94,\n" \
"R_CUBIC_TO, 0.15, -0.44, 0.39, -0.8, 0.72, -1.08,\n" \
"CUBIC_TO, 9.11, 1.64, 9.52, 1.5, 10, 1.5,\n" \
"R_CUBIC_TO, 0.48, 0, 0.9, 0.14, 1.23, 0.41,\n" \
"R_CUBIC_TO, 0.33, 0.27, 0.57, 0.64, 0.71, 1.09,\n" \
"H_LINE_TO, 15.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 4.5,\n" \
"R_CUBIC_TO, 0.21, 0, 0.39, -0.07, 0.54, -0.21,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, 0.22, -0.53,\n" \
"R_CUBIC_TO, 0, -0.21, -0.07, -0.39, -0.21, -0.53,\n" \
"ARC_TO, 0.72, 0.72, 0, 0, 0, 10, 3,\n" \
"R_CUBIC_TO, -0.21, 0, -0.39, 0.07, -0.53, 0.22,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, -0.22, 0.53,\n" \
"R_CUBIC_TO, 0, 0.21, 0.07, 0.39, 0.22, 0.54,\n" \
"R_CUBIC_TO, 0.14, 0.14, 0.32, 0.22, 0.53, 0.22,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.38, 11,\n" \
"LINE_TO, 4.5, 6.63,\n" \
"V_LINE_TO, 15.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5, 17,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3, 15.5,\n" \
"V_LINE_TO, 5.13,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"LINE_TO, 16, 18.13,\n" \
"LINE_TO, 14.88, 17,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kContentPasteOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCookieIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 31, 15,\n" \
"LINE_TO, 31, 14,\n" \
"CUBIC_TO, 31, 12.04, 30, 11, 28, 11,\n" \
"LINE_TO, 27, 11,\n" \
"CUBIC_TO, 26.45, 11, 26, 10.55, 26, 10,\n" \
"LINE_TO, 26, 6,\n" \
"CUBIC_TO, 26, 4.04, 24, 4, 24, 4,\n" \
"CUBIC_TO, 12.6, 4, 3.42, 13.54, 4.03, 25.08,\n" \
"CUBIC_TO, 4.56, 35.17, 12.83, 43.44, 22.92, 43.97,\n" \
"CUBIC_TO, 34.46, 44.58, 44, 35.4, 44, 24,\n" \
"LINE_TO, 44, 22,\n" \
"CUBIC_TO, 44, 20.9, 43.11, 20, 42, 20,\n" \
"LINE_TO, 38, 20,\n" \
"CUBIC_TO, 37.45, 20, 37, 19.55, 37, 19,\n" \
"LINE_TO, 37, 18,\n" \
"CUBIC_TO, 37, 16.13, 36.04, 15.04, 34, 15,\n" \
"LINE_TO, 31, 15,\n" \
"LINE_TO, 31, 15,\n" \
"CLOSE,\n" \
"MOVE_TO, 11.59, 24.94,\n" \
"CUBIC_TO, 9.47, 25.35, 7.66, 23.53, 8.06, 21.41,\n" \
"CUBIC_TO, 8.28, 20.24, 9.24, 19.28, 10.41, 19.06,\n" \
"CUBIC_TO, 12.53, 18.66, 14.35, 20.47, 13.94, 22.59,\n" \
"CUBIC_TO, 13.72, 23.76, 12.76, 24.72, 11.59, 24.94,\n" \
"LINE_TO, 11.59, 24.94,\n" \
"CLOSE,\n" \
"MOVE_TO, 15.06, 14.59,\n" \
"CUBIC_TO, 14.66, 12.47, 16.47, 10.66, 18.59, 11.06,\n" \
"CUBIC_TO, 19.76, 11.28, 20.72, 12.24, 20.94, 13.41,\n" \
"CUBIC_TO, 21.35, 15.53, 19.53, 17.35, 17.41, 16.94,\n" \
"CUBIC_TO, 16.24, 16.72, 15.28, 15.76, 15.06, 14.59,\n" \
"LINE_TO, 15.06, 14.59,\n" \
"CLOSE,\n" \
"MOVE_TO, 22.59, 38.94,\n" \
"CUBIC_TO, 20.47, 39.35, 18.66, 37.53, 19.06, 35.41,\n" \
"CUBIC_TO, 19.28, 34.24, 20.24, 33.28, 21.41, 33.06,\n" \
"CUBIC_TO, 23.53, 32.66, 25.35, 34.47, 24.94, 36.59,\n" \
"CUBIC_TO, 24.72, 37.76, 23.76, 38.72, 22.59, 38.94,\n" \
"LINE_TO, 22.59, 38.94,\n" \
"CLOSE,\n" \
"MOVE_TO, 23, 28,\n" \
"CUBIC_TO, 21.34, 28, 20, 26.66, 20, 25,\n" \
"CUBIC_TO, 20, 23.34, 21.34, 22, 23, 22,\n" \
"CUBIC_TO, 24.66, 22, 26, 23.34, 26, 25,\n" \
"CUBIC_TO, 26, 26.66, 24.66, 28, 23, 28,\n" \
"LINE_TO, 23, 28,\n" \
"CLOSE,\n" \
"MOVE_TO, 33, 32,\n" \
"CUBIC_TO, 31.34, 32, 30, 30.66, 30, 29,\n" \
"CUBIC_TO, 30, 27.34, 31.34, 26, 33, 26,\n" \
"CUBIC_TO, 34.66, 26, 36, 27.34, 36, 29,\n" \
"CUBIC_TO, 36, 30.66, 34.66, 32, 33, 32,\n" \
"LINE_TO, 33, 32,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCookieIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCookieChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 10.5, 10,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, -0.14, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.29, 0.44, -0.64, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.42, -0.14, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 10.5, 7,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, 0.14, -1.06, 0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 9, 8.5,\n" \
"R_CUBIC_TO, 0, 0.42, 0.14, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.64, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2, 5,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, -0.14, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.29, 0.44, -0.64, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.42, -0.14, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 8.5, 12,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, 0.14, -1.06, 0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 7, 13.5,\n" \
"R_CUBIC_TO, 0, 0.42, 0.14, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.64, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6.5, 1,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, -0.1, 0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 16, 15,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 15, 14,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.71, 0.29,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.29, 0.71,\n" \
"R_CUBIC_TO, 0, 0.28, 0.1, 0.52, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.43, 0.29, 0.71, 0.29,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3, 6.14,\n" \
"R_ARC_TO, 9.89, 9.89, 0, 0, 1, -3.95, -0.8,\n" \
"R_ARC_TO, 10.2, 10.2, 0, 0, 1, -3.22, -2.17,\n" \
"R_ARC_TO, 10.2, 10.2, 0, 0, 1, -2.17, -3.22,\n" \
"ARC_TO, 9.89, 9.89, 0, 0, 1, 1.86, 12,\n" \
"R_CUBIC_TO, 0, -1.48, 0.32, -2.89, 0.95, -4.25,\n" \
"R_ARC_TO, 10.3, 10.3, 0, 0, 1, 2.62, -3.48,\n" \
"R_CUBIC_TO, 1.11, -0.96, 2.42, -1.66, 3.93, -2.1,\n" \
"R_CUBIC_TO, 1.5, -0.43, 3.13, -0.46, 4.88, -0.08,\n" \
"R_CUBIC_TO, -0.16, 0.76, -0.12, 1.47, 0.11, 2.13,\n" \
"R_CUBIC_TO, 0.23, 0.66, 0.59, 1.21, 1.07, 1.65,\n" \
"R_CUBIC_TO, 0.49, 0.45, 1.07, 0.75, 1.76, 0.91,\n" \
"R_CUBIC_TO, 0.68, 0.16, 1.41, 0.13, 2.18, -0.12,\n" \
"R_CUBIC_TO, -0.48, 1.15, -0.38, 2.14, 0.28, 2.95,\n" \
"R_CUBIC_TO, 0.67, 0.82, 1.48, 1.23, 2.43, 1.24,\n" \
"R_CUBIC_TO, 0.16, 1.51, 0.01, 2.95, -0.45, 4.32,\n" \
"R_CUBIC_TO, -0.46, 1.36, -1.15, 2.56, -2.07, 3.59,\n" \
"R_ARC_TO, 10.28, 10.28, 0, 0, 1, -3.32, 2.45,\n" \
"R_CUBIC_TO, -1.3, 0.61, -2.71, 0.91, -4.23, 0.91,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.27,\n" \
"R_CUBIC_TO, 1.98, 0, 3.75, -0.68, 5.29, -2.03,\n" \
"R_CUBIC_TO, 1.54, -1.35, 2.39, -3.09, 2.56, -5.19,\n" \
"R_CUBIC_TO, -0.81, -0.37, -1.45, -0.87, -1.92, -1.5,\n" \
"R_ARC_TO, 5.38, 5.38, 0, 0, 1, -0.96, -2.09,\n" \
"R_ARC_TO, 5.84, 5.84, 0, 0, 1, -3.28, -1.65,\n" \
"R_ARC_TO, 5.97, 5.97, 0, 0, 1, -1.73, -3.27,\n" \
"R_CUBIC_TO, -1.26, -0.02, -2.38, 0.23, -3.35, 0.72,\n" \
"R_ARC_TO, 7.97, 7.97, 0, 0, 0, -2.45, 1.91,\n" \
"R_ARC_TO, 8.47, 8.47, 0, 0, 0, -1.5, 2.56,\n" \
"R_CUBIC_TO, -0.34, 0.93, -0.51, 1.82, -0.51, 2.68,\n" \
"R_CUBIC_TO, 0, 2.18, 0.77, 4.04, 2.3, 5.57,\n" \
"R_CUBIC_TO, 1.53, 1.53, 3.39, 2.3, 5.57, 2.3,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 8.75, 8.5,\n" \
"R_CUBIC_TO, 0.35, 0, 0.64, -0.12, 0.89, -0.36,\n" \
"R_CUBIC_TO, 0.24, -0.25, 0.36, -0.54, 0.36, -0.89,\n" \
"R_CUBIC_TO, 0, -0.35, -0.12, -0.64, -0.36, -0.89,\n" \
"ARC_TO, 1.22, 1.22, 0, 0, 0, 8.75, 6,\n" \
"R_CUBIC_TO, -0.35, 0, -0.64, 0.12, -0.89, 0.36,\n" \
"R_CUBIC_TO, -0.24, 0.25, -0.36, 0.54, -0.36, 0.89,\n" \
"R_CUBIC_TO, 0, 0.35, 0.12, 0.64, 0.36, 0.89,\n" \
"R_CUBIC_TO, 0.25, 0.24, 0.54, 0.36, 0.89, 0.36,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2, 4,\n" \
"R_CUBIC_TO, 0.35, 0, 0.64, -0.12, 0.89, -0.36,\n" \
"R_CUBIC_TO, 0.24, -0.25, 0.36, -0.54, 0.36, -0.89,\n" \
"R_CUBIC_TO, 0, -0.35, -0.12, -0.64, -0.36, -0.89,\n" \
"ARC_TO, 1.22, 1.22, 0, 0, 0, 6.75, 10,\n" \
"R_CUBIC_TO, -0.35, 0, -0.64, 0.12, -0.89, 0.36,\n" \
"R_CUBIC_TO, -0.24, 0.25, -0.36, 0.54, -0.36, 0.89,\n" \
"R_CUBIC_TO, 0, 0.35, 0.12, 0.64, 0.36, 0.89,\n" \
"R_CUBIC_TO, 0.25, 0.24, 0.54, 0.36, 0.89, 0.36,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.75, 0.75,\n" \
"R_CUBIC_TO, 0.21, 0, 0.39, -0.07, 0.54, -0.21,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, 0.22, -0.53,\n" \
"R_CUBIC_TO, 0, -0.21, -0.07, -0.39, -0.21, -0.53,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, -0.53, -0.22,\n" \
"R_CUBIC_TO, -0.21, 0, -0.39, 0.07, -0.53, 0.22,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, -0.22, 0.53,\n" \
"R_CUBIC_TO, 0, 0.21, 0.07, 0.39, 0.22, 0.54,\n" \
"R_CUBIC_TO, 0.14, 0.14, 0.32, 0.22, 0.53, 0.22,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 18.11,\n" \
"R_ARC_TO, 7.92, 7.92, 0, 0, 1, -3.15, -0.63,\n" \
"R_ARC_TO, 8.16, 8.16, 0, 0, 1, -4.33, -4.33,\n" \
"ARC_TO, 7.92, 7.92, 0, 0, 1, 1.89, 10,\n" \
"R_CUBIC_TO, 0, -1.35, 0.29, -2.59, 0.89, -3.72,\n" \
"R_CUBIC_TO, 0.59, -1.12, 1.36, -2.05, 2.31, -2.79,\n" \
"ARC_TO, 8.23, 8.23, 0, 0, 1, 8.27, 1.98,\n" \
"R_ARC_TO, 6.82, 6.82, 0, 0, 1, 3.53, 0.1,\n" \
"R_CUBIC_TO, -0.11, 0.62, -0.09, 1.19, 0.08, 1.7,\n" \
"R_CUBIC_TO, 0.16, 0.51, 0.44, 0.94, 0.82, 1.27,\n" \
"R_CUBIC_TO, 0.38, 0.33, 0.84, 0.57, 1.4, 0.7,\n" \
"R_CUBIC_TO, 0.56, 0.13, 1.15, 0.13, 1.8, -0.02,\n" \
"R_CUBIC_TO, -0.32, 0.79, -0.26, 1.55, 0.16, 2.27,\n" \
"R_CUBIC_TO, 0.43, 0.72, 1.1, 1.08, 2.01, 1.09,\n" \
"R_ARC_TO, 7.82, 7.82, 0, 0, 1, -0.36, 3.44,\n" \
"R_ARC_TO, 8.37, 8.37, 0, 0, 1, -1.66, 2.88,\n" \
"R_ARC_TO, 8.21, 8.21, 0, 0, 1, -2.66, 1.98,\n" \
"R_CUBIC_TO, -1.04, 0.49, -2.17, 0.74, -3.39, 0.74,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.73,\n" \
"R_CUBIC_TO, 1.7, 0, 3.15, -0.57, 4.36, -1.71,\n" \
"R_CUBIC_TO, 1.21, -1.14, 1.88, -2.55, 2.01, -4.24,\n" \
"R_ARC_TO, 3.8, 3.8, 0, 0, 1, -1.53, -1.16,\n" \
"R_ARC_TO, 3.86, 3.86, 0, 0, 1, -0.79, -1.75,\n" \
"R_ARC_TO, 4.86, 4.86, 0, 0, 1, -2.67, -1.31,\n" \
"R_ARC_TO, 4.42, 4.42, 0, 0, 1, -1.34, -2.6,\n" \
"R_ARC_TO, 5.58, 5.58, 0, 0, 0, -2.37, 0.41,\n" \
"R_CUBIC_TO, -0.77, 0.31, -1.46, 0.75, -2.06, 1.32,\n" \
"R_ARC_TO, 6.35, 6.35, 0, 0, 0, -1.45, 2.05,\n" \
"ARC_TO, 6.23, 6.23, 0, 0, 0, 3.61, 10,\n" \
"R_CUBIC_TO, 0, 1.77, 0.63, 3.28, 1.87, 4.52,\n" \
"R_CUBIC_TO, 1.24, 1.24, 2.75, 1.87, 4.52, 1.87,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 7, 6.8,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, 0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 8, 5.8,\n" \
"R_ARC_TO, 0.97, 0.97, 0, 0, 0, -0.29, -0.71,\n" \
"ARC_TO, 0.97, 0.97, 0, 0, 0, 7, 4.8,\n" \
"R_ARC_TO, 0.97, 0.97, 0, 0, 0, -0.71, 0.29,\n" \
"ARC_TO, 0.97, 0.97, 0, 0, 0, 6, 5.8,\n" \
"R_CUBIC_TO, 0, 0.28, 0.1, 0.51, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0.19, 0.2, 0.43, 0.29, 0.71, 0.29,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.4, 10,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, 0.71, -0.29,\n" \
"ARC_TO, 0.95, 0.95, 0, 0, 0, 6.4, 9,\n" \
"R_ARC_TO, 0.95, 0.95, 0, 0, 0, -0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 5.4, 8,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.71, 0.29,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.29, 0.71,\n" \
"R_CUBIC_TO, 0, 0.28, 0.1, 0.52, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0.2, 0.2, 0.43, 0.29, 0.71, 0.29,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.6, 0.6,\n" \
"R_ARC_TO, 0.6, 0.6, 0, 0, 0, 0.43, -0.17,\n" \
"R_ARC_TO, 0.58, 0.58, 0, 0, 0, 0.18, -0.43,\n" \
"R_ARC_TO, 0.6, 0.6, 0, 0, 0, -0.17, -0.43,\n" \
"R_ARC_TO, 0.58, 0.58, 0, 0, 0, -0.43, -0.18,\n" \
"R_ARC_TO, 0.6, 0.6, 0, 0, 0, -0.43, 0.17,\n" \
"R_ARC_TO, 0.58, 0.58, 0, 0, 0, -0.18, 0.43,\n" \
"R_CUBIC_TO, 0, 0.17, 0.06, 0.31, 0.17, 0.43,\n" \
"R_ARC_TO, 0.58, 0.58, 0, 0, 0, 0.43, 0.18,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 14.62,\n" \
"R_CUBIC_TO, -0.91, 0, -1.77, -0.17, -2.57, -0.51,\n" \
"R_ARC_TO, 6.61, 6.61, 0, 0, 1, -2.11, -1.42,\n" \
"R_ARC_TO, 6.61, 6.61, 0, 0, 1, -1.42, -2.11,\n" \
"ARC_TO, 6.46, 6.46, 0, 0, 1, 1.38, 8,\n" \
"R_CUBIC_TO, 0, -1.15, 0.25, -2.19, 0.75, -3.11,\n" \
"ARC_TO, 6.85, 6.85, 0, 0, 1, 4.08, 2.63,\n" \
"ARC_TO, 6.79, 6.79, 0, 0, 1, 6.73, 1.44,\n" \
"R_ARC_TO, 5.67, 5.67, 0, 0, 1, 2.86, 0.12,\n" \
"R_CUBIC_TO, -0.09, 0.51, -0.09, 0.97, 0.03, 1.38,\n" \
"R_CUBIC_TO, 0.11, 0.41, 0.32, 0.75, 0.61, 1.01,\n" \
"R_CUBIC_TO, 0.29, 0.26, 0.66, 0.45, 1.1, 0.55,\n" \
"R_CUBIC_TO, 0.44, 0.1, 0.93, 0.1, 1.47, -0.01,\n" \
"R_CUBIC_TO, -0.22, 0.64, -0.18, 1.24, 0.14, 1.82,\n" \
"R_CUBIC_TO, 0.32, 0.57, 0.86, 0.86, 1.64, 0.84,\n" \
"R_ARC_TO, 6.49, 6.49, 0, 0, 1, -0.26, 2.83,\n" \
"R_ARC_TO, 6.79, 6.79, 0, 0, 1, -1.34, 2.38,\n" \
"ARC_TO, 6.68, 6.68, 0, 0, 1, 10.79, 14,\n" \
"R_CUBIC_TO, -0.85, 0.41, -1.78, 0.62, -2.79, 0.62,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.63,\n" \
"R_CUBIC_TO, 1.31, 0, 2.44, -0.44, 3.37, -1.3,\n" \
"R_CUBIC_TO, 0.94, -0.87, 1.47, -1.95, 1.58, -3.25,\n" \
"R_ARC_TO, 3.13, 3.13, 0, 0, 1, -1.18, -0.92,\n" \
"R_ARC_TO, 3.05, 3.05, 0, 0, 1, -0.63, -1.37,\n" \
"ARC_TO, 4.05, 4.05, 0, 0, 1, 9.03, 5.09,\n" \
"ARC_TO, 3.55, 3.55, 0, 0, 1, 7.93, 3.04,\n" \
"R_ARC_TO, 4.47, 4.47, 0, 0, 0, -1.75, 0.32,\n" \
"R_CUBIC_TO, -0.58, 0.23, -1.11, 0.56, -1.58, 0.99,\n" \
"R_CUBIC_TO, -0.47, 0.43, -0.86, 0.95, -1.14, 1.57,\n" \
"R_CUBIC_TO, -0.29, 0.62, -0.43, 1.31, -0.43, 2.07,\n" \
"R_CUBIC_TO, 0, 1.38, 0.49, 2.56, 1.46, 3.53,\n" \
"R_CUBIC_TO, 0.97, 0.97, 2.15, 1.46, 3.53, 1.46,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCookieChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseCookieOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"R_MOVE_TO, 20.55, 17.47,\n" \
"R_LINE_TO, -1.66, -1.66,\n" \
"R_CUBIC_TO, 0.26, -0.47, 0.48, -0.98, 0.64, -1.51,\n" \
"R_CUBIC_TO, 0.16, -0.53, 0.27, -1.09, 0.32, -1.66,\n" \
"R_CUBIC_TO, -0.81, -0.37, -1.45, -0.87, -1.92, -1.5,\n" \
"R_ARC_TO, 5.38, 5.38, 0, 0, 1, -0.96, -2.09,\n" \
"R_ARC_TO, 5.84, 5.84, 0, 0, 1, -3.28, -1.65,\n" \
"R_ARC_TO, 5.97, 5.97, 0, 0, 1, -1.73, -3.27,\n" \
"R_ARC_TO, 6.8, 6.8, 0, 0, 0, -2.04, 0.24,\n" \
"R_ARC_TO, 7.57, 7.57, 0, 0, 0, -1.74, 0.73,\n" \
"R_LINE_TO, -1.64, -1.64,\n" \
"R_CUBIC_TO, 1.05, -0.7, 2.22, -1.18, 3.52, -1.44,\n" \
"R_CUBIC_TO, 1.3, -0.26, 2.7, -0.23, 4.18, 0.08,\n" \
"R_CUBIC_TO, -0.16, 0.76, -0.12, 1.47, 0.11, 2.13,\n" \
"R_CUBIC_TO, 0.23, 0.66, 0.59, 1.21, 1.07, 1.65,\n" \
"R_CUBIC_TO, 0.49, 0.45, 1.07, 0.75, 1.76, 0.91,\n" \
"R_CUBIC_TO, 0.68, 0.16, 1.41, 0.13, 2.18, -0.12,\n" \
"R_CUBIC_TO, -0.48, 1.15, -0.38, 2.14, 0.28, 2.95,\n" \
"R_CUBIC_TO, 0.67, 0.82, 1.48, 1.23, 2.43, 1.24,\n" \
"R_ARC_TO, 9.48, 9.48, 0, 0, 1, -0.21, 3.5,\n" \
"R_ARC_TO, 10.67, 10.67, 0, 0, 1, -1.32, 3.1,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.5, 15,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, -0.14, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 7, 13.5,\n" \
"R_CUBIC_TO, 0, -0.42, 0.14, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 8.5, 12,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, 0.14, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.64, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0, 0.42, -0.14, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 8.5, 15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12.07, 8.43,\n" \
"R_LINE_TO, -2.92, -2.91,\n" \
"R_CUBIC_TO, -0.82, 0.55, -1.7, 0.97, -2.64, 1.25,\n" \
"R_CUBIC_TO, -0.95, 0.29, -1.95, 0.44, -3.01, 0.44,\n" \
"R_CUBIC_TO, -1.41, 0, -2.74, -0.26, -3.98, -0.8,\n" \
"R_ARC_TO, 10.38, 10.38, 0, 0, 1, -3.24, -2.17,\n" \
"R_CUBIC_TO, -0.91, -0.92, -1.64, -2, -2.17, -3.24,\n" \
"R_CUBIC_TO, -0.54, -1.24, -0.8, -2.57, -0.8, -3.98,\n" \
"R_CUBIC_TO, 0, -1.06, 0.14, -2.06, 0.44, -3.01,\n" \
"R_ARC_TO, 9.96, 9.96, 0, 0, 1, 1.25, -2.64,\n" \
"LINE_TO, 0.65, 3.5,\n" \
"R_LINE_TO, 1.48, -1.48,\n" \
"LINE_TO, 22.06, 21.94,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 19.93,\n" \
"R_CUBIC_TO, 0.73, 0, 1.43, -0.1, 2.1, -0.29,\n" \
"R_ARC_TO, 8.27, 8.27, 0, 0, 0, 1.88, -0.8,\n" \
"LINE_TO, 5.16, 8.02,\n" \
"R_CUBIC_TO, -0.34, 0.59, -0.61, 1.21, -0.8, 1.88,\n" \
"ARC_TO, 7.57, 7.57, 0, 0, 0, 4.07, 12,\n" \
"R_CUBIC_TO, 0, 2.2, 0.77, 4.07, 2.32, 5.61,\n" \
"CUBIC_TO, 7.93, 19.16, 9.81, 19.93, 12, 19.93,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"R_MOVE_TO, 16.77, 14.47,\n" \
"R_LINE_TO, -1.28, -1.28,\n" \
"R_CUBIC_TO, 0.24, -0.41, 0.44, -0.84, 0.59, -1.3,\n" \
"R_ARC_TO, 6.24, 6.24, 0, 0, 0, 0.29, -1.45,\n" \
"R_ARC_TO, 3.8, 3.8, 0, 0, 1, -1.53, -1.16,\n" \
"R_ARC_TO, 3.86, 3.86, 0, 0, 1, -0.79, -1.75,\n" \
"R_ARC_TO, 4.86, 4.86, 0, 0, 1, -2.67, -1.31,\n" \
"R_ARC_TO, 4.42, 4.42, 0, 0, 1, -1.34, -2.6,\n" \
"R_ARC_TO, 5.69, 5.69, 0, 0, 0, -1.71, 0.18,\n" \
"R_ARC_TO, 5.89, 5.89, 0, 0, 0, -1.56, 0.67,\n" \
"LINE_TO, 5.49, 3.19,\n" \
"R_ARC_TO, 8, 8, 0, 0, 1, 3.02, -1.28,\n" \
"R_ARC_TO, 6.69, 6.69, 0, 0, 1, 3.29, 0.17,\n" \
"R_CUBIC_TO, -0.11, 0.62, -0.09, 1.19, 0.07, 1.7,\n" \
"R_CUBIC_TO, 0.16, 0.51, 0.44, 0.94, 0.82, 1.27,\n" \
"R_CUBIC_TO, 0.38, 0.33, 0.84, 0.57, 1.4, 0.7,\n" \
"R_CUBIC_TO, 0.56, 0.13, 1.15, 0.13, 1.8, -0.02,\n" \
"R_CUBIC_TO, -0.32, 0.79, -0.26, 1.55, 0.16, 2.27,\n" \
"R_CUBIC_TO, 0.43, 0.72, 1.1, 1.08, 2.01, 1.09,\n" \
"R_ARC_TO, 7.82, 7.82, 0, 0, 1, -0.18, 2.87,\n" \
"R_ARC_TO, 8.05, 8.05, 0, 0, 1, -1.12, 2.52,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.75, 12.5,\n" \
"R_CUBIC_TO, -0.35, 0, -0.64, -0.12, -0.89, -0.36,\n" \
"R_ARC_TO, 1.22, 1.22, 0, 0, 1, -0.36, -0.89,\n" \
"R_CUBIC_TO, 0, -0.35, 0.12, -0.64, 0.36, -0.89,\n" \
"R_CUBIC_TO, 0.25, -0.24, 0.54, -0.36, 0.89, -0.36,\n" \
"R_CUBIC_TO, 0.35, 0, 0.64, 0.12, 0.89, 0.36,\n" \
"R_CUBIC_TO, 0.24, 0.25, 0.36, 0.54, 0.36, 0.89,\n" \
"R_CUBIC_TO, 0, 0.35, -0.12, 0.64, -0.36, 0.89,\n" \
"R_CUBIC_TO, -0.25, 0.24, -0.54, 0.36, -0.89, 0.36,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 9.2, 5.61,\n" \
"R_LINE_TO, -1.35, -1.35,\n" \
"R_ARC_TO, 8.76, 8.76, 0, 0, 1, -2.2, 1.05,\n" \
"R_ARC_TO, 8.01, 8.01, 0, 0, 1, -2.4, 0.36,\n" \
"R_ARC_TO, 7.97, 7.97, 0, 0, 1, -3.18, -0.64,\n" \
"R_ARC_TO, 8.2, 8.2, 0, 0, 1, -2.6, -1.75,\n" \
"R_ARC_TO, 8.2, 8.2, 0, 0, 1, -1.75, -2.6,\n" \
"ARC_TO, 7.97, 7.97, 0, 0, 1, 1.83, 10,\n" \
"R_CUBIC_TO, 0, -0.83, 0.12, -1.62, 0.36, -2.4,\n" \
"R_ARC_TO, 8.76, 8.76, 0, 0, 1, 1.05, -2.2,\n" \
"R_LINE_TO, -1.26, -1.27,\n" \
"R_LINE_TO, 1.11, -1.11,\n" \
"R_LINE_TO, 13.98, 13.98,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5.95, -1.66,\n" \
"R_CUBIC_TO, 0.6, 0, 1.17, -0.09, 1.74, -0.26,\n" \
"R_ARC_TO, 8.67, 8.67, 0, 0, 0, 1.62, -0.69,\n" \
"LINE_TO, 4.51, 6.64,\n" \
"R_ARC_TO, 8.82, 8.82, 0, 0, 0, -0.69, 1.62,\n" \
"R_ARC_TO, 5.89, 5.89, 0, 0, 0, -0.26, 1.74,\n" \
"R_CUBIC_TO, 0, 1.79, 0.63, 3.31, 1.88, 4.56,\n" \
"R_CUBIC_TO, 1.25, 1.25, 2.78, 1.88, 4.56, 1.88,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"R_MOVE_TO, 13.57, 11.54,\n" \
"R_LINE_TO, -1.22, -1.21,\n" \
"R_ARC_TO, 4.42, 4.42, 0, 0, 0, 0.4, -0.9,\n" \
"R_CUBIC_TO, 0.1, -0.32, 0.17, -0.65, 0.2, -1,\n" \
"R_ARC_TO, 3.13, 3.13, 0, 0, 1, -1.18, -0.92,\n" \
"R_ARC_TO, 3.05, 3.05, 0, 0, 1, -0.63, -1.37,\n" \
"ARC_TO, 4.05, 4.05, 0, 0, 1, 9.03, 5.09,\n" \
"ARC_TO, 3.55, 3.55, 0, 0, 1, 7.93, 3.04,\n" \
"R_ARC_TO, 4.42, 4.42, 0, 0, 0, -2.28, 0.58,\n" \
"LINE_TO, 4.42, 2.4,\n" \
"ARC_TO, 6.57, 6.57, 0, 0, 1, 6.92, 1.39,\n" \
"R_ARC_TO, 5.42, 5.42, 0, 0, 1, 2.67, 0.17,\n" \
"R_CUBIC_TO, -0.09, 0.51, -0.09, 0.97, 0.03, 1.38,\n" \
"R_CUBIC_TO, 0.11, 0.41, 0.32, 0.75, 0.61, 1.01,\n" \
"R_CUBIC_TO, 0.29, 0.26, 0.66, 0.45, 1.1, 0.55,\n" \
"R_CUBIC_TO, 0.44, 0.1, 0.93, 0.1, 1.47, -0.01,\n" \
"R_CUBIC_TO, -0.22, 0.64, -0.18, 1.24, 0.14, 1.82,\n" \
"R_CUBIC_TO, 0.32, 0.57, 0.86, 0.85, 1.64, 0.84,\n" \
"R_ARC_TO, 6.59, 6.59, 0, 0, 1, -0.12, 2.34,\n" \
"R_ARC_TO, 6.64, 6.64, 0, 0, 1, -0.87, 2.06,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.4, 10,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 1, -0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 4.4, 9,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 5.4, 8,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.2, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_ARC_TO, 0.95, 0.95, 0, 0, 1, -0.29, 0.71,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 1, -0.71, 0.29,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7.45, 4.6,\n" \
"R_LINE_TO, -1.05, -1.04,\n" \
"R_ARC_TO, 7.28, 7.28, 0, 0, 1, -1.81, 0.86,\n" \
"R_CUBIC_TO, -0.64, 0.2, -1.3, 0.3, -1.98, 0.3,\n" \
"R_ARC_TO, 6.51, 6.51, 0, 0, 1, -2.62, -0.52,\n" \
"R_ARC_TO, 6.7, 6.7, 0, 0, 1, -2.14, -1.44,\n" \
"R_ARC_TO, 6.7, 6.7, 0, 0, 1, -1.44, -2.14,\n" \
"ARC_TO, 6.51, 6.51, 0, 0, 1, 1.28, 8,\n" \
"R_CUBIC_TO, 0, -0.68, 0.1, -1.34, 0.3, -1.98,\n" \
"R_ARC_TO, 7.28, 7.28, 0, 0, 1, 0.86, -1.81,\n" \
"R_LINE_TO, -0.89, -0.9,\n" \
"R_LINE_TO, 0.94, -0.95,\n" \
"R_LINE_TO, 11.3, 11.3,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 13.09,\n" \
"R_ARC_TO, 4.8, 4.8, 0, 0, 0, 1.36, -0.19,\n" \
"R_ARC_TO, 6.43, 6.43, 0, 0, 0, 1.27, -0.53,\n" \
"R_LINE_TO, -6.98, -6.99,\n" \
"R_ARC_TO, 6.5, 6.5, 0, 0, 0, -0.53, 1.26,\n" \
"ARC_TO, 4.84, 4.84, 0, 0, 0, 2.91, 8,\n" \
"R_CUBIC_TO, 0, 1.41, 0.5, 2.62, 1.48, 3.61,\n" \
"R_CUBIC_TO, 0.99, 0.99, 2.2, 1.48, 3.6, 1.48,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kCookieOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDangerousIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 6.18, 19.24,\n" \
"LINE_TO, 0.76, 13.8,\n" \
"V_LINE_TO, 6.18,\n" \
"LINE_TO, 6.18, 0.76,\n" \
"R_H_LINE_TO, 7.65,\n" \
"LINE_TO, 19.24, 6.18,\n" \
"R_V_LINE_TO, 7.65,\n" \
"LINE_TO, 13.8, 19.24,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.06, 14.32,\n" \
"LINE_TO, 10, 11.4,\n" \
"LINE_TO, 12.94, 14.32,\n" \
"LINE_TO, 14.32, 12.94,\n" \
"LINE_TO, 11.4, 10,\n" \
"LINE_TO, 14.32, 7.06,\n" \
"LINE_TO, 12.94, 5.68,\n" \
"LINE_TO, 10, 8.6,\n" \
"LINE_TO, 7.06, 5.68,\n" \
"LINE_TO, 5.68, 7.06,\n" \
"LINE_TO, 8.6, 10,\n" \
"LINE_TO, 5.68, 12.94,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kDangerousIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDangerousChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 7.11, 17,\n" \
"LINE_TO, 3, 12.88,\n" \
"LINE_TO, 3, 7.11,\n" \
"LINE_TO, 7.11, 3,\n" \
"LINE_TO, 12.89, 3,\n" \
"LINE_TO, 17, 7.11,\n" \
"LINE_TO, 17, 12.89,\n" \
"LINE_TO, 12.88, 17,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.77, 13.27,\n" \
"LINE_TO, 10, 11.06,\n" \
"LINE_TO, 12.23, 13.27,\n" \
"LINE_TO, 13.27, 12.23,\n" \
"LINE_TO, 11.06, 10,\n" \
"LINE_TO, 13.27, 7.77,\n" \
"LINE_TO, 12.23, 6.73,\n" \
"LINE_TO, 10, 8.94,\n" \
"LINE_TO, 7.77, 6.73,\n" \
"LINE_TO, 6.73, 7.77,\n" \
"LINE_TO, 8.94, 10,\n" \
"LINE_TO, 6.73, 12.23,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kDangerousChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDatabaseIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 12, 21,\n" \
"R_CUBIC_TO, -2.52, 0, -4.64, -0.39, -6.39, -1.16,\n" \
"CUBIC_TO, 3.87, 19.06, 3, 18.12, 3, 17,\n" \
"V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, -1.1, 0.88, -2.04, 2.64, -2.82,\n" \
"CUBIC_TO, 7.4, 3.39, 9.52, 3, 12, 3,\n" \
"R_CUBIC_TO, 2.48, 0, 4.61, 0.39, 6.36, 1.18,\n" \
"CUBIC_TO, 20.12, 4.96, 21, 5.9, 21, 7,\n" \
"R_V_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0, 1.12, -0.87, 2.06, -2.61, 2.84,\n" \
"CUBIC_TO, 16.65, 20.61, 14.52, 21, 12, 21,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -11.98,\n" \
"R_CUBIC_TO, 1.48, 0, 2.98, -0.21, 4.48, -0.64,\n" \
"R_CUBIC_TO, 1.5, -0.43, 2.34, -0.88, 2.52, -1.36,\n" \
"R_CUBIC_TO, -0.18, -0.48, -1.02, -0.94, -2.51, -1.37,\n" \
"ARC_TO, 16.01, 16.01, 0, 0, 0, 12, 5,\n" \
"R_CUBIC_TO, -1.52, 0, -3, 0.21, -4.46, 0.64,\n" \
"R_CUBIC_TO, -1.46, 0.43, -2.31, 0.89, -2.54, 1.39,\n" \
"R_CUBIC_TO, 0.23, 0.5, 1.08, 0.96, 2.54, 1.38,\n" \
"R_CUBIC_TO, 1.46, 0.42, 2.94, 0.63, 4.46, 0.63,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 14,\n" \
"R_CUBIC_TO, 0.7, 0, 1.38, -0.04, 2.02, -0.1,\n" \
"R_CUBIC_TO, 0.65, -0.07, 1.27, -0.16, 1.86, -0.28,\n" \
"R_ARC_TO, 13.4, 13.4, 0, 0, 0, 1.68, -0.46,\n" \
"ARC_TO, 9.95, 9.95, 0, 0, 0, 19, 12.52,\n" \
"R_V_LINE_TO, -3,\n" \
"R_ARC_TO, 9.95, 9.95, 0, 0, 1, -1.44, 0.63,\n" \
"R_CUBIC_TO, -0.52, 0.18, -1.08, 0.34, -1.67, 0.47,\n" \
"R_CUBIC_TO, -0.59, 0.13, -1.21, 0.22, -1.86, 0.28,\n" \
"CUBIC_TO, 13.38, 10.97, 12.7, 11, 12, 11,\n" \
"R_CUBIC_TO, -0.7, 0, -1.38, -0.04, -2.05, -0.1,\n" \
"R_ARC_TO, 17.58, 17.58, 0, 0, 1, -1.89, -0.28,\n" \
"R_ARC_TO, 13.21, 13.21, 0, 0, 1, -1.66, -0.46,\n" \
"ARC_TO, 9.08, 9.08, 0, 0, 1, 5, 9.52,\n" \
"R_V_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0.42, 0.24, 0.88, 0.44, 1.4, 0.63,\n" \
"R_CUBIC_TO, 0.52, 0.18, 1.08, 0.34, 1.67, 0.47,\n" \
"R_CUBIC_TO, 0.59, 0.13, 1.22, 0.22, 1.89, 0.28,\n" \
"R_CUBIC_TO, 0.67, 0.07, 1.35, 0.1, 2.05, 0.1,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 5,\n" \
"R_CUBIC_TO, 0.77, 0, 1.55, -0.06, 2.34, -0.18,\n" \
"R_CUBIC_TO, 0.79, -0.12, 1.52, -0.27, 2.19, -0.46,\n" \
"R_CUBIC_TO, 0.67, -0.19, 1.23, -0.41, 1.68, -0.65,\n" \
"R_CUBIC_TO, 0.45, -0.24, 0.72, -0.48, 0.8, -0.73,\n" \
"R_V_LINE_TO, -2.45,\n" \
"R_ARC_TO, 9.95, 9.95, 0, 0, 1, -1.44, 0.63,\n" \
"R_CUBIC_TO, -0.52, 0.18, -1.08, 0.34, -1.67, 0.47,\n" \
"R_CUBIC_TO, -0.59, 0.13, -1.21, 0.22, -1.86, 0.28,\n" \
"CUBIC_TO, 13.38, 15.97, 12.7, 16, 12, 16,\n" \
"R_CUBIC_TO, -0.7, 0, -1.38, -0.04, -2.05, -0.1,\n" \
"R_ARC_TO, 17.58, 17.58, 0, 0, 1, -1.89, -0.28,\n" \
"R_ARC_TO, 13.21, 13.21, 0, 0, 1, -1.66, -0.46,\n" \
"ARC_TO, 9.08, 9.08, 0, 0, 1, 5, 14.52,\n" \
"V_LINE_TO, 17,\n" \
"R_CUBIC_TO, 0.08, 0.25, 0.35, 0.49, 0.79, 0.73,\n" \
"R_CUBIC_TO, 0.44, 0.23, 1, 0.45, 1.66, 0.64,\n" \
"R_CUBIC_TO, 0.67, 0.19, 1.4, 0.34, 2.2, 0.46,\n" \
"R_CUBIC_TO, 0.8, 0.12, 1.59, 0.18, 2.35, 0.18,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kDatabaseIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDatabaseOffIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 3.57, 2.56,\n" \
"LINE_TO, 2.15, 3.98,\n" \
"R_LINE_TO, 18.59, 18.59,\n" \
"R_LINE_TO, 1.41, -1.41,\n" \
"LINE_TO, 3.57, 2.56,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"MOVE_TO, 5.31, 4.3,\n" \
"CUBIC_TO, 4.27, 4.96, 3.75, 5.71, 3.75, 6.56,\n" \
"R_V_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0, 1.12, 0.87, 2.07, 2.6, 2.85,\n" \
"R_CUBIC_TO, 1.75, 0.77, 3.88, 1.15, 6.4, 1.15,\n" \
"R_CUBIC_TO, 2.52, 0, 4.64, -0.38, 6.38, -1.15,\n" \
"R_CUBIC_TO, 0.32, -0.14, 0.6, -0.29, 0.86, -0.44,\n" \
"R_LINE_TO, -1.48, -1.48,\n" \
"R_CUBIC_TO, -0.36, 0.16, -0.77, 0.3, -1.23, 0.44,\n" \
"R_CUBIC_TO, -0.67, 0.18, -1.4, 0.33, -2.2, 0.45,\n" \
"R_ARC_TO, 15.76, 15.76, 0, 0, 1, -2.33, 0.17,\n" \
"R_CUBIC_TO, -0.77, 0, -1.55, -0.06, -2.35, -0.17,\n" \
"R_CUBIC_TO, -0.8, -0.12, -1.53, -0.27, -2.2, -0.45,\n" \
"R_CUBIC_TO, -0.67, -0.2, -1.23, -0.42, -1.67, -0.65,\n" \
"R_CUBIC_TO, -0.43, -0.23, -0.69, -0.47, -0.77, -0.72,\n" \
"R_V_LINE_TO, -2.47,\n" \
"R_CUBIC_TO, 0.42, 0.23, 0.88, 0.44, 1.4, 0.63,\n" \
"R_CUBIC_TO, 0.52, 0.18, 1.07, 0.34, 1.65, 0.48,\n" \
"R_ARC_TO, 20.81, 20.81, 0, 0, 0, 3.95, 0.37,\n" \
"R_ARC_TO, 19.84, 19.84, 0, 0, 0, 3.52, -0.31,\n" \
"LINE_TO, 14.5, 13.49,\n" \
"R_CUBIC_TO, -0.56, 0.05, -1.15, 0.07, -1.75, 0.07,\n" \
"R_ARC_TO, 20.81, 20.81, 0, 0, 1, -3.95, -0.37,\n" \
"R_ARC_TO, 14.79, 14.79, 0, 0, 1, -1.65, -0.47,\n" \
"R_ARC_TO, 9.07, 9.07, 0, 0, 1, -1.4, -0.62,\n" \
"R_V_LINE_TO, -3,\n" \
"R_CUBIC_TO, 0.42, 0.23, 0.88, 0.44, 1.4, 0.63,\n" \
"R_CUBIC_TO, 0.52, 0.18, 1.07, 0.34, 1.65, 0.48,\n" \
"R_ARC_TO, 20.81, 20.81, 0, 0, 0, 2.74, 0.34,\n" \
"LINE_TO, 9.21, 8.2,\n" \
"R_ARC_TO, 16.62, 16.62, 0, 0, 1, -0.94, -0.24,\n" \
"R_CUBIC_TO, -1.45, -0.42, -2.29, -0.87, -2.52, -1.37,\n" \
"R_CUBIC_TO, 0.14, -0.29, 0.48, -0.57, 1.02, -0.83,\n" \
"LINE_TO, 5.31, 4.3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.57, 0.75,\n" \
"LINE_TO, 7.24, 3.4,\n" \
"R_CUBIC_TO, 1.58, -0.56, 3.42, -0.83, 5.51, -0.83,\n" \
"R_CUBIC_TO, 2.48, 0, 4.6, 0.39, 6.35, 1.17,\n" \
"R_CUBIC_TO, 1.77, 0.78, 2.65, 1.73, 2.65, 2.83,\n" \
"R_V_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0, 0.37, -0.1, 0.72, -0.29, 1.06,\n" \
"R_LINE_TO, -1.71, -1.71,\n" \
"R_V_LINE_TO, -1.82,\n" \
"R_ARC_TO, 9.49, 9.49, 0, 0, 1, -1.26, 0.56,\n" \
"R_LINE_TO, -1.53, -1.54,\n" \
"R_CUBIC_TO, 0.48, -0.12, 0.93, -0.25, 1.35, -0.4,\n" \
"R_ARC_TO, 9.73, 9.73, 0, 0, 0, 1.45, -0.62,\n" \
"R_V_LINE_TO, -3,\n" \
"R_ARC_TO, 9.73, 9.73, 0, 0, 1, -1.45, 0.63,\n" \
"R_CUBIC_TO, -0.52, 0.18, -1.07, 0.34, -1.67, 0.48,\n" \
"R_ARC_TO, 19.68, 19.68, 0, 0, 1, -2.28, 0.31,\n" \
"R_LINE_TO, -1.92, -1.92,\n" \
"R_CUBIC_TO, 0.11, 0, 0.22, 0, 0.33, 0,\n" \
"R_CUBIC_TO, 1.48, 0, 2.98, -0.21, 4.47, -0.62,\n" \
"R_CUBIC_TO, 1.5, -0.43, 2.34, -0.89, 2.53, -1.37,\n" \
"R_CUBIC_TO, -0.18, -0.48, -1.02, -0.94, -2.52, -1.37,\n" \
"R_ARC_TO, 15.89, 15.89, 0, 0, 0, -4.47, -0.65,\n" \
"R_CUBIC_TO, -1.31, 0, -2.59, 0.16, -3.86, 0.48,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kDatabaseOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDescriptionIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 8, 16,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -4,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6, -10,\n" \
"H_LINE_TO, 6,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 16,\n" \
"R_CUBIC_TO, 0, 1.1, 0.89, 2, 1.99, 2,\n" \
"H_LINE_TO, 18,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 8,\n" \
"R_LINE_TO, -6, -6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, 18,\n" \
"H_LINE_TO, 6,\n" \
"V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 7,\n" \
"R_V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 11,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kDescriptionIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDevicesIcon) { \
              std::string s = \
"// Copyright 2019 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 1.91, 20.27,\n" \
"V_LINE_TO, 17,\n" \
"R_H_LINE_TO, 2,\n" \
"V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, -0.63, 0.22, -1.17, 0.66, -1.61,\n" \
"R_CUBIC_TO, 0.45, -0.44, 0.98, -0.66, 1.61, -0.66,\n" \
"H_LINE_TO, 21,\n" \
"V_LINE_TO, 6,\n" \
"H_LINE_TO, 6.18,\n" \
"R_V_LINE_TO, 11,\n" \
"R_H_LINE_TO, 5.73,\n" \
"R_V_LINE_TO, 3.27,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12.95, 0,\n" \
"R_CUBIC_TO, -0.3, 0, -0.55, -0.1, -0.75, -0.3,\n" \
"R_CUBIC_TO, -0.2, -0.2, -0.3, -0.45, -0.3, -0.75,\n" \
"V_LINE_TO, 8.95,\n" \
"R_CUBIC_TO, 0, -0.3, 0.1, -0.54, 0.3, -0.74,\n" \
"R_CUBIC_TO, 0.2, -0.2, 0.45, -0.3, 0.75, -0.3,\n" \
"R_H_LINE_TO, 6.18,\n" \
"R_CUBIC_TO, 0.3, 0, 0.54, 0.1, 0.74, 0.3,\n" \
"R_CUBIC_TO, 0.2, 0.2, 0.3, 0.45, 0.3, 0.74,\n" \
"R_V_LINE_TO, 10.27,\n" \
"R_CUBIC_TO, 0, 0.3, -0.1, 0.55, -0.3, 0.75,\n" \
"R_CUBIC_TO, -0.2, 0.2, -0.44, 0.3, -0.74, 0.3,\n" \
"CLOSE,\n" \
"MOVE_TO, 15.91, 17,\n" \
"H_LINE_TO, 20,\n" \
"R_V_LINE_TO, -7,\n" \
"R_H_LINE_TO, -4.09,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 1.93, 16.23,\n" \
"V_LINE_TO, 14,\n" \
"R_H_LINE_TO, 2,\n" \
"V_LINE_TO, 5.5,\n" \
"R_CUBIC_TO, 0, -0.48, 0.17, -0.89, 0.5, -1.22,\n" \
"R_CUBIC_TO, 0.34, -0.34, 0.75, -0.51, 1.23, -0.51,\n" \
"H_LINE_TO, 17,\n" \
"V_LINE_TO, 5.5,\n" \
"H_LINE_TO, 5.66,\n" \
"V_LINE_TO, 14,\n" \
"R_H_LINE_TO, 4.27,\n" \
"R_V_LINE_TO, 2.23,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 10.96, 0,\n" \
"R_CUBIC_TO, -0.3, 0, -0.54, -0.1, -0.74, -0.3,\n" \
"R_CUBIC_TO, -0.2, -0.2, -0.3, -0.45, -0.3, -0.74,\n" \
"V_LINE_TO, 7.96,\n" \
"R_CUBIC_TO, 0, -0.29, 0.1, -0.54, 0.3, -0.74,\n" \
"R_CUBIC_TO, 0.2, -0.2, 0.45, -0.3, 0.74, -0.3,\n" \
"R_H_LINE_TO, 4.15,\n" \
"R_CUBIC_TO, 0.29, 0, 0.54, 0.1, 0.74, 0.3,\n" \
"R_CUBIC_TO, 0.2, 0.2, 0.3, 0.45, 0.3, 0.74,\n" \
"R_V_LINE_TO, 7.23,\n" \
"R_CUBIC_TO, 0, 0.3, -0.1, 0.54, -0.3, 0.74,\n" \
"R_CUBIC_TO, -0.2, 0.2, -0.44, 0.3, -0.74, 0.3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.54, -2.23,\n" \
"H_LINE_TO, 16.5,\n" \
"V_LINE_TO, 8.5,\n" \
"R_H_LINE_TO, -3.07,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 1.46, 13.23,\n" \
"V_LINE_TO, 11.2,\n" \
"R_H_LINE_TO, 1.6,\n" \
"V_LINE_TO, 4.4,\n" \
"R_CUBIC_TO, 0, -0.45, 0.16, -0.84, 0.48, -1.16,\n" \
"R_ARC_TO, 1.58, 1.58, 0, 0, 1, 1.16, -0.48,\n" \
"R_H_LINE_TO, 8.91,\n" \
"R_V_LINE_TO, 1.63,\n" \
"R_H_LINE_TO, -8.91,\n" \
"V_LINE_TO, 11.2,\n" \
"H_LINE_TO, 7.86,\n" \
"R_V_LINE_TO, 2.04,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.73, 0,\n" \
"R_ARC_TO, 0.85, 0.85, 0, 0, 1, -0.62, -0.25,\n" \
"R_ARC_TO, 0.84, 0.84, 0, 0, 1, -0.25, -0.62,\n" \
"V_LINE_TO, 6.32,\n" \
"R_CUBIC_TO, 0, -0.25, 0.08, -0.45, 0.25, -0.62,\n" \
"R_ARC_TO, 0.86, 0.86, 0, 0, 1, 0.63, -0.25,\n" \
"R_H_LINE_TO, 3.49,\n" \
"R_CUBIC_TO, 0.25, 0, 0.45, 0.09, 0.62, 0.25,\n" \
"R_CUBIC_TO, 0.16, 0.17, 0.25, 0.37, 0.25, 0.62,\n" \
"R_V_LINE_TO, 6.04,\n" \
"R_ARC_TO, 0.86, 0.86, 0, 0, 1, -0.25, 0.62,\n" \
"R_ARC_TO, 0.84, 0.84, 0, 0, 1, -0.62, 0.25,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.48, -2.03,\n" \
"R_H_LINE_TO, 2.54,\n" \
"V_LINE_TO, 6.8,\n" \
"R_H_LINE_TO, -2.54,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kDevicesIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDevicesOffIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 7.63, 5.5,\n" \
"LINE_TO, 6.13, 4,\n" \
"H_LINE_TO, 17,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 18, 15.5,\n" \
"LINE_TO, 16.5, 14,\n" \
"V_LINE_TO, 8.5,\n" \
"R_H_LINE_TO, -3,\n" \
"R_V_LINE_TO, 2.88,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 7.75,\n" \
"R_CUBIC_TO, 0, -0.21, 0.07, -0.39, 0.22, -0.53,\n" \
"ARC_TO, 0.72, 0.72, 0, 0, 1, 12.75, 7,\n" \
"R_H_LINE_TO, 4.49,\n" \
"R_ARC_TO, 0.73, 0.73, 0, 0, 1, 0.54, 0.22,\n" \
"R_ARC_TO, 0.73, 0.73, 0, 0, 1, 0.22, 0.54,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2, 2.63,\n" \
"LINE_TO, 13.88, 16,\n" \
"H_LINE_TO, 12.75,\n" \
"R_CUBIC_TO, -0.21, 0, -0.39, -0.07, -0.53, -0.21,\n" \
"ARC_TO, 0.73, 0.73, 0, 0, 1, 12, 15.25,\n" \
"R_V_LINE_TO, -1.12,\n" \
"R_LINE_TO, -6.5, -6.5,\n" \
"V_LINE_TO, 14,\n" \
"H_LINE_TO, 10,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 2,\n" \
"V_LINE_TO, 6.13,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1, -5.44,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kDevicesOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDirectionsCarIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 6, 19,\n" \
"R_V_LINE_TO, 1,\n" \
"R_CUBIC_TO, 0, 0.28, -0.1, 0.52, -0.29, 0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 5, 21,\n" \
"H_LINE_TO, 4,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 1, -0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 3, 20,\n" \
"R_V_LINE_TO, -8,\n" \
"R_LINE_TO, 2.1, -6,\n" \
"R_CUBIC_TO, 0.1, -0.3, 0.28, -0.54, 0.54, -0.73,\n" \
"CUBIC_TO, 5.9, 5.09, 6.18, 5, 6.5, 5,\n" \
"R_H_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0.32, 0, 0.61, 0.09, 0.86, 0.27,\n" \
"R_CUBIC_TO, 0.26, 0.18, 0.44, 0.43, 0.54, 0.73,\n" \
"LINE_TO, 21, 12,\n" \
"R_V_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, 0.28, -0.1, 0.52, -0.29, 0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 20, 21,\n" \
"R_H_LINE_TO, -1,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 1, -0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 18, 20,\n" \
"R_V_LINE_TO, -1,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.2, -9,\n" \
"R_H_LINE_TO, 12.4,\n" \
"R_LINE_TO, -1.05, -3,\n" \
"H_LINE_TO, 6.85,\n" \
"CLOSE,\n" \
"MOVE_TO, 5, 12,\n" \
"R_V_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.5, 4,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, -0.14, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.29, 0.44, -0.64, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.42, -0.14, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 7.5, 13,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, 0.14, -1.06, 0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 6, 14.5,\n" \
"R_CUBIC_TO, 0, 0.42, 0.14, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.64, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 9, 0,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, -0.14, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.29, 0.44, -0.64, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.42, -0.14, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 16.5, 13,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, 0.14, -1.06, 0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 15, 14.5,\n" \
"R_CUBIC_TO, 0, 0.42, 0.14, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.64, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 5, 17,\n" \
"R_H_LINE_TO, 14,\n" \
"R_V_LINE_TO, -5,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kDirectionsCarIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDocumentScannerIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 4.67, 2,\n" \
"H_LINE_TO, 2.67,\n" \
"V_LINE_TO, 4,\n" \
"H_LINE_TO, 1.33,\n" \
"V_LINE_TO, 0.67,\n" \
"H_LINE_TO, 4.67,\n" \
"V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"MOVE_TO, 14.67, 4,\n" \
"V_LINE_TO, 0.67,\n" \
"H_LINE_TO, 11.33,\n" \
"V_LINE_TO, 2,\n" \
"H_LINE_TO, 13.33,\n" \
"V_LINE_TO, 4,\n" \
"H_LINE_TO, 14.67,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.67, 14,\n" \
"H_LINE_TO, 2.67,\n" \
"V_LINE_TO, 12,\n" \
"H_LINE_TO, 1.33,\n" \
"V_LINE_TO, 15.33,\n" \
"H_LINE_TO, 4.67,\n" \
"V_LINE_TO, 14,\n" \
"CLOSE,\n" \
"MOVE_TO, 13.33, 12,\n" \
"V_LINE_TO, 14,\n" \
"H_LINE_TO, 11.33,\n" \
"V_LINE_TO, 15.33,\n" \
"H_LINE_TO, 14.67,\n" \
"V_LINE_TO, 12,\n" \
"H_LINE_TO, 13.33,\n" \
"CLOSE,\n" \
"MOVE_TO, 11.33, 4,\n" \
"H_LINE_TO, 4.67,\n" \
"V_LINE_TO, 12,\n" \
"H_LINE_TO, 11.33,\n" \
"V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"MOVE_TO, 12.67, 12,\n" \
"CUBIC_TO, 12.67, 12.73, 12.07, 13.33, 11.33, 13.33,\n" \
"H_LINE_TO, 4.67,\n" \
"CUBIC_TO, 3.93, 13.33, 3.33, 12.73, 3.33, 12,\n" \
"V_LINE_TO, 4,\n" \
"CUBIC_TO, 3.33, 3.27, 3.93, 2.67, 4.67, 2.67,\n" \
"H_LINE_TO, 11.33,\n" \
"CUBIC_TO, 12.07, 2.67, 12.67, 3.27, 12.67, 4,\n" \
"V_LINE_TO, 12,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 5.33,\n" \
"H_LINE_TO, 6,\n" \
"V_LINE_TO, 6.67,\n" \
"H_LINE_TO, 10,\n" \
"V_LINE_TO, 5.33,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 7.33,\n" \
"H_LINE_TO, 6,\n" \
"V_LINE_TO, 8.67,\n" \
"H_LINE_TO, 10,\n" \
"V_LINE_TO, 7.33,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 9.33,\n" \
"H_LINE_TO, 6,\n" \
"V_LINE_TO, 10.67,\n" \
"H_LINE_TO, 10,\n" \
"V_LINE_TO, 9.33,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kDocumentScannerIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseDogfoodIcon) { \
              std::string s = \
"// Copyright 2020 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 9.75, 9.25,\n" \
"CUBIC_TO, 10.72, 9.25, 11.5, 8.47, 11.5, 7.5,\n" \
"CUBIC_TO, 11.5, 6.53, 10.72, 5.75, 9.75, 5.75,\n" \
"CUBIC_TO, 8.78, 5.75, 8, 6.53, 8, 7.5,\n" \
"CUBIC_TO, 8, 8.47, 8.78, 9.25, 9.75, 9.25,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.88, 12.25,\n" \
"CUBIC_TO, 7.85, 12.25, 8.63, 11.47, 8.63, 10.5,\n" \
"CUBIC_TO, 8.63, 9.53, 7.85, 8.75, 6.88, 8.75,\n" \
"CUBIC_TO, 5.91, 8.75, 5.13, 9.53, 5.13, 10.5,\n" \
"CUBIC_TO, 5.13, 11.47, 5.91, 12.25, 6.88, 12.25,\n" \
"CLOSE,\n" \
"MOVE_TO, 15.75, 7.5,\n" \
"CUBIC_TO, 15.75, 8.47, 14.97, 9.25, 14, 9.25,\n" \
"CUBIC_TO, 13.03, 9.25, 12.25, 8.47, 12.25, 7.5,\n" \
"CUBIC_TO, 12.25, 6.53, 13.03, 5.75, 14, 5.75,\n" \
"CUBIC_TO, 14.97, 5.75, 15.75, 6.53, 15.75, 7.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 17.12, 12.25,\n" \
"CUBIC_TO, 18.09, 12.25, 18.87, 11.47, 18.87, 10.5,\n" \
"CUBIC_TO, 18.87, 9.53, 18.09, 8.75, 17.12, 8.75,\n" \
"CUBIC_TO, 16.15, 8.75, 15.37, 9.53, 15.37, 10.5,\n" \
"CUBIC_TO, 15.37, 11.47, 16.15, 12.25, 17.12, 12.25,\n" \
"CLOSE,\n" \
"MOVE_TO, 12.63, 10.36,\n" \
"CUBIC_TO, 13.17, 10.54, 13.54, 10.9, 13.89, 11.3,\n" \
"CUBIC_TO, 14.07, 11.51, 14.24, 11.72, 14.42, 11.93,\n" \
"CUBIC_TO, 14.79, 12.38, 15.16, 12.83, 15.58, 13.25,\n" \
"CUBIC_TO, 15.67, 13.34, 15.76, 13.43, 15.85, 13.52,\n" \
"CUBIC_TO, 16.7, 14.38, 17.59, 15.27, 17.34, 16.54,\n" \
"CUBIC_TO, 17.18, 17.34, 16.65, 17.96, 15.81, 18.22,\n" \
"CUBIC_TO, 15.58, 18.29, 15.05, 18.22, 14.35, 18.13,\n" \
"CUBIC_TO, 13.62, 18.04, 12.72, 17.92, 11.81, 17.92,\n" \
"CUBIC_TO, 10.88, 17.92, 10, 18.04, 9.31, 18.13,\n" \
"CUBIC_TO, 8.68, 18.22, 8.19, 18.29, 7.97, 18.22,\n" \
"CUBIC_TO, 7.13, 17.96, 6.59, 17.34, 6.43, 16.54,\n" \
"CUBIC_TO, 6.18, 15.26, 7.07, 14.37, 7.92, 13.51,\n" \
"CUBIC_TO, 8.01, 13.42, 8.09, 13.34, 8.18, 13.25,\n" \
"CUBIC_TO, 8.56, 12.86, 8.91, 12.44, 9.25, 12.01,\n" \
"CUBIC_TO, 9.45, 11.77, 9.64, 11.53, 9.84, 11.3,\n" \
"CUBIC_TO, 10.19, 10.9, 10.52, 10.54, 11.06, 10.36,\n" \
"CUBIC_TO, 11.3, 10.29, 11.55, 10.25, 11.8, 10.25,\n" \
"CUBIC_TO, 11.8, 10.25, 12.39, 10.28, 12.63, 10.36,\n" \
"CLOSE\n" \
"\n" \
; \
              CheckThatParsedElementsMatch(s, kDogfoodIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseEditIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 48,\n" \
"MOVE_TO, 6, 34.51,\n" \
"LINE_TO, 6, 42.01,\n" \
"LINE_TO, 13.5, 42.01,\n" \
"LINE_TO, 35.62, 19.89,\n" \
"LINE_TO, 28.12, 12.39,\n" \
"LINE_TO, 6, 34.51,\n" \
"CLOSE,\n" \
"MOVE_TO, 41.42, 14.09,\n" \
"CUBIC_TO, 42.2, 13.31, 42.2, 12.05, 41.42, 11.27,\n" \
"LINE_TO, 36.74, 6.59,\n" \
"CUBIC_TO, 35.96, 5.81, 34.7, 5.81, 33.92, 6.59,\n" \
"LINE_TO, 30.26, 10.25,\n" \
"LINE_TO, 37.76, 17.75,\n" \
"LINE_TO, 41.42, 14.09,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kEditIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseEditChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 15.5,\n" \
"R_H_LINE_TO, 1.06,\n" \
"R_LINE_TO, 7.81, -7.81,\n" \
"R_LINE_TO, -1.06, -1.06,\n" \
"LINE_TO, 4.5, 14.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 3, 17,\n" \
"R_V_LINE_TO, -3.19,\n" \
"LINE_TO, 13.38, 3.44,\n" \
"R_CUBIC_TO, 0.15, -0.15, 0.32, -0.27, 0.5, -0.33,\n" \
"R_ARC_TO, 1.55, 1.55, 0, 0, 1, 1.13, 0,\n" \
"R_CUBIC_TO, 0.18, 0.07, 0.35, 0.18, 0.5, 0.33,\n" \
"LINE_TO, 16.56, 4.5,\n" \
"R_ARC_TO, 1.4, 1.4, 0, 0, 1, 0.33, 0.5,\n" \
"R_ARC_TO, 1.56, 1.56, 0, 0, 1, 0, 1.13,\n" \
"R_CUBIC_TO, -0.07, 0.18, -0.18, 0.35, -0.33, 0.5,\n" \
"LINE_TO, 6.19, 17,\n" \
"CLOSE,\n" \
"MOVE_TO, 15.5, 5.56,\n" \
"LINE_TO, 14.44, 4.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.66, 1.6,\n" \
"R_LINE_TO, -0.52, -0.54,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.65, 12.35,\n" \
"R_H_LINE_TO, 0.85,\n" \
"R_LINE_TO, 5.98, -5.98,\n" \
"R_LINE_TO, -0.85, -0.85,\n" \
"R_LINE_TO, -5.98, 5.99,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.38, 1.38,\n" \
"V_LINE_TO, 10.93,\n" \
"R_LINE_TO, 8.29, -8.3,\n" \
"R_CUBIC_TO, 0.13, -0.13, 0.27, -0.22, 0.42, -0.28,\n" \
"R_CUBIC_TO, 0.15, -0.06, 0.31, -0.09, 0.48, -0.09,\n" \
"R_CUBIC_TO, 0.16, 0, 0.32, 0.03, 0.47, 0.09,\n" \
"R_CUBIC_TO, 0.15, 0.06, 0.29, 0.15, 0.43, 0.28,\n" \
"LINE_TO, 13.37, 3.64,\n" \
"R_CUBIC_TO, 0.13, 0.13, 0.22, 0.27, 0.28, 0.43,\n" \
"R_CUBIC_TO, 0.06, 0.15, 0.09, 0.31, 0.09, 0.47,\n" \
"R_ARC_TO, 1.25, 1.25, 0, 0, 1, -0.37, 0.91,\n" \
"R_LINE_TO, -8.29, 8.29,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 10.05, -9.19,\n" \
"R_LINE_TO, -0.85, -0.86,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.26, 1.41,\n" \
"R_LINE_TO, -0.42, -0.43,\n" \
"R_LINE_TO, 0.85, 0.85,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kEditChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseEmailIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 20, 4,\n" \
"H_LINE_TO, 4,\n" \
"R_CUBIC_TO, -1.1, 0, -1.99, 0.9, -1.99, 2,\n" \
"LINE_TO, 2, 18,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 16,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 4,\n" \
"R_LINE_TO, -8, 5,\n" \
"R_LINE_TO, -8, -5,\n" \
"V_LINE_TO, 6,\n" \
"R_LINE_TO, 8, 5,\n" \
"R_LINE_TO, 8, -5,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kEmailIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseEmailOutlineIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 20, 4,\n" \
"H_LINE_TO, 4,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 12,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 16,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.8, 2,\n" \
"LINE_TO, 12, 10.8,\n" \
"LINE_TO, 4.8, 6,\n" \
"R_H_LINE_TO, 14.4,\n" \
"CLOSE,\n" \
"MOVE_TO, 4, 18,\n" \
"V_LINE_TO, 7.87,\n" \
"R_LINE_TO, 8, 5.33,\n" \
"R_LINE_TO, 8, -5.33,\n" \
"V_LINE_TO, 18,\n" \
"H_LINE_TO, 4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kEmailOutlineIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseErrorIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 8, 0.91,\n" \
"CUBIC_TO, 4.07, 0.91, 0.89, 4.1, 0.89, 8.03,\n" \
"CUBIC_TO, 0.89, 11.96, 4.07, 15.14, 8, 15.14,\n" \
"CUBIC_TO, 11.93, 15.14, 15.11, 11.96, 15.11, 8.03,\n" \
"CUBIC_TO, 15.11, 4.1, 11.93, 0.91, 8, 0.91,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.89, 11.55,\n" \
"LINE_TO, 7.11, 11.55,\n" \
"LINE_TO, 7.11, 9.78,\n" \
"LINE_TO, 8.89, 9.78,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.89, 8.89,\n" \
"LINE_TO, 7.11, 8.89,\n" \
"LINE_TO, 7.11, 4.45,\n" \
"LINE_TO, 8.89, 4.45,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.89, 8.89,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kErrorIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseErrorOutlineIcon) { \
              std::string s = \
"// Copyright 2020 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 11, 15,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 6,\n" \
"R_H_LINE_TO, -2,\n" \
"V_LINE_TO, 7,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.99, -5,\n" \
"CUBIC_TO, 6.47, 2, 2, 6.48, 2, 12,\n" \
"R_CUBIC_TO, 0, 5.52, 4.47, 10, 9.99, 10,\n" \
"CUBIC_TO, 17.52, 22, 22, 17.52, 22, 12,\n" \
"CUBIC_TO_SHORTHAND, 17.52, 2, 11.99, 2,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 20,\n" \
"R_CUBIC_TO, -4.42, 0, -8, -3.58, -8, -8,\n" \
"R_CUBIC_TO, 0, -4.42, 3.58, -8, 8, -8,\n" \
"R_CUBIC_TO, 4.42, 0, 8, 3.58, 8, 8,\n" \
"R_CUBIC_TO, 0, 4.42, -3.58, 8, -8, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kErrorOutlineIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseEthernetIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 17, 6,\n" \
"R_LINE_TO, -1.41, 1.41,\n" \
"LINE_TO, 20.17, 12,\n" \
"R_LINE_TO, -4.58, 4.59,\n" \
"LINE_TO, 17, 18,\n" \
"R_LINE_TO, 6, -6,\n" \
"R_LINE_TO, -6, -6,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.41, 7.41,\n" \
"LINE_TO, 7, 6,\n" \
"R_LINE_TO, -6, 6,\n" \
"R_LINE_TO, 6, 6,\n" \
"R_LINE_TO, 1.41, -1.41,\n" \
"LINE_TO, 3.83, 12,\n" \
"R_LINE_TO, 4.58, -4.59,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 13,\n" \
"R_CUBIC_TO, 0.55, 0, 1, -0.45, 1, -1,\n" \
"R_CUBIC_TO, 0, -0.55, -0.45, -1, -1, -1,\n" \
"R_CUBIC_TO, -0.55, 0, -1, 0.45, -1, 1,\n" \
"R_CUBIC_TO, 0, 0.55, 0.45, 1, 1, 1,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, 0,\n" \
"R_CUBIC_TO, 0.55, 0, 1, -0.45, 1, -1,\n" \
"R_CUBIC_TO, 0, -0.55, -0.45, -1, -1, -1,\n" \
"R_CUBIC_TO, -0.55, 0, -1, 0.45, -1, 1,\n" \
"R_CUBIC_TO, 0, 0.55, 0.45, 1, 1, 1,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, 0,\n" \
"R_CUBIC_TO, 0.55, 0, 1, -0.45, 1, -1,\n" \
"R_CUBIC_TO, 0, -0.55, -0.45, -1, -1, -1,\n" \
"R_CUBIC_TO, -0.55, 0, -1, 0.45, -1, 1,\n" \
"R_CUBIC_TO, 0, 0.55, 0.45, 1, 1, 1,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kEthernetIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseExpandMoreIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"R_MOVE_TO, 8, 10.74,\n" \
"R_LINE_TO, -4.3, -4.3,\n" \
"R_LINE_TO, 1.15, -1.15,\n" \
"LINE_TO, 8, 8.44,\n" \
"R_LINE_TO, 3.15, -3.15,\n" \
"R_LINE_TO, 1.15, 1.15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kExpandMoreIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseExtensionIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 41, 22,\n" \
"R_H_LINE_TO, -3,\n" \
"R_V_LINE_TO, -8,\n" \
"R_CUBIC_TO, 0, -2.21, -1.79, -4, -4, -4,\n" \
"R_H_LINE_TO, -8,\n" \
"V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, -2.76, -2.24, -5, -5, -5,\n" \
"R_CUBIC_TO, -2.76, 0, -5, 2.24, -5, 5,\n" \
"R_V_LINE_TO, 3,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, -2.21, 0, -3.98, 1.79, -3.98, 4,\n" \
"R_LINE_TO, -0.01, 7.6,\n" \
"H_LINE_TO, 7,\n" \
"R_CUBIC_TO, 2.98, 0, 5.4, 2.42, 5.4, 5.4,\n" \
"R_CUBIC_TO, 0, 2.98, -2.42, 5.4, -5.4, 5.4,\n" \
"H_LINE_TO, 4.01,\n" \
"LINE_TO, 4, 40,\n" \
"R_CUBIC_TO, 0, 2.21, 1.79, 4, 4, 4,\n" \
"R_H_LINE_TO, 7.6,\n" \
"R_V_LINE_TO, -3,\n" \
"R_CUBIC_TO, 0, -2.98, 2.42, -5.4, 5.4, -5.4,\n" \
"R_CUBIC_TO, 2.98, 0, 5.4, 2.42, 5.4, 5.4,\n" \
"R_V_LINE_TO, 3,\n" \
"H_LINE_TO, 34,\n" \
"R_CUBIC_TO, 2.21, 0, 4, -1.79, 4, -4,\n" \
"R_V_LINE_TO, -8,\n" \
"R_H_LINE_TO, 3,\n" \
"R_CUBIC_TO, 2.76, 0, 5, -2.24, 5, -5,\n" \
"R_CUBIC_TO, 0, -2.76, -2.24, -5, -5, -5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kExtensionIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseExtensionChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 17,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3, 15.5,\n" \
"V_LINE_TO, 12,\n" \
"R_ARC_TO, 2.12, 2.12, 0, 0, 0, 1.42, -0.61,\n" \
"ARC_TO, 1.88, 1.88, 0, 0, 0, 5, 10,\n" \
"R_ARC_TO, 1.88, 1.88, 0, 0, 0, -0.58, -1.39,\n" \
"ARC_TO, 2.12, 2.12, 0, 0, 0, 3, 8,\n" \
"V_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4.5, 3,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, -0.56, 0.19, -1.03, 0.58, -1.42,\n" \
"ARC_TO, 1.93, 1.93, 0, 0, 1, 10, 1,\n" \
"R_CUBIC_TO, 0.56, 0, 1.03, 0.19, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"R_H_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 8,\n" \
"R_ARC_TO, 1.94, 1.94, 0, 0, 1, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"R_CUBIC_TO, 0, 0.56, -0.19, 1.03, -0.58, 1.42,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 17, 12,\n" \
"R_V_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 15.5, 17,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 11,\n" \
"R_V_LINE_TO, -11,\n" \
"R_H_LINE_TO, -11,\n" \
"R_V_LINE_TO, 2.33,\n" \
"R_ARC_TO, 3.23, 3.23, 0, 0, 1, 1.47, 1.28,\n" \
"CUBIC_TO, 6.32, 8.69, 6.5, 9.32, 6.5, 10,\n" \
"R_CUBIC_TO, 0, 0.7, -0.18, 1.33, -0.53, 1.91,\n" \
"ARC_TO, 3.12, 3.12, 0, 0, 1, 4.5, 13.17,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.6, 13.73,\n" \
"R_CUBIC_TO, -0.37, 0, -0.69, -0.13, -0.95, -0.39,\n" \
"R_ARC_TO, 1.28, 1.28, 0, 0, 1, -0.39, -0.95,\n" \
"R_V_LINE_TO, -2.87,\n" \
"R_CUBIC_TO, 0.42, -0.02, 0.77, -0.18, 1.05, -0.47,\n" \
"ARC_TO, 1.46, 1.46, 0, 0, 0, 3.75, 8,\n" \
"R_CUBIC_TO, 0, -0.41, -0.14, -0.76, -0.43, -1.05,\n" \
"R_ARC_TO, 1.51, 1.51, 0, 0, 0, -1.05, -0.47,\n" \
"V_LINE_TO, 3.6,\n" \
"R_CUBIC_TO, 0, -0.37, 0.13, -0.69, 0.39, -0.95,\n" \
"R_CUBIC_TO, 0.26, -0.26, 0.57, -0.39, 0.95, -0.39,\n" \
"R_H_LINE_TO, 2.68,\n" \
"R_CUBIC_TO, 0.03, -0.46, 0.21, -0.84, 0.53, -1.16,\n" \
"R_ARC_TO, 1.63, 1.63, 0, 0, 1, 1.18, -0.47,\n" \
"R_CUBIC_TO, 0.46, 0, 0.86, 0.16, 1.18, 0.47,\n" \
"R_CUBIC_TO, 0.33, 0.31, 0.51, 0.7, 0.54, 1.16,\n" \
"R_H_LINE_TO, 2.68,\n" \
"R_CUBIC_TO, 0.37, 0, 0.69, 0.13, 0.95, 0.39,\n" \
"R_CUBIC_TO, 0.26, 0.26, 0.39, 0.57, 0.39, 0.95,\n" \
"R_V_LINE_TO, 2.68,\n" \
"R_CUBIC_TO, 0.46, 0.03, 0.84, 0.21, 1.16, 0.53,\n" \
"R_CUBIC_TO, 0.32, 0.33, 0.47, 0.72, 0.47, 1.18,\n" \
"R_CUBIC_TO, 0, 0.46, -0.16, 0.86, -0.47, 1.18,\n" \
"R_CUBIC_TO, -0.31, 0.33, -0.7, 0.51, -1.16, 0.54,\n" \
"R_V_LINE_TO, 2.68,\n" \
"R_CUBIC_TO, 0, 0.37, -0.13, 0.69, -0.39, 0.95,\n" \
"R_CUBIC_TO, -0.26, 0.26, -0.57, 0.39, -0.95, 0.39,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.05, -1.38,\n" \
"R_H_LINE_TO, 8.7,\n" \
"V_LINE_TO, 3.65,\n" \
"H_LINE_TO, 3.65,\n" \
"R_V_LINE_TO, 1.81,\n" \
"R_CUBIC_TO, 0.47, 0.25, 0.83, 0.61, 1.09, 1.07,\n" \
"R_CUBIC_TO, 0.26, 0.46, 0.39, 0.95, 0.39, 1.48,\n" \
"R_CUBIC_TO, 0, 0.54, -0.13, 1.05, -0.39, 1.5,\n" \
"R_ARC_TO, 2.58, 2.58, 0, 0, 1, -1.09, 1.04,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kExtensionChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseExtensionOffIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16, 18.13,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1, -3.25,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 4.5,\n" \
"H_LINE_TO, 6.63,\n" \
"LINE_TO, 5.13, 3,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, -0.56, 0.19, -1.03, 0.58, -1.42,\n" \
"ARC_TO, 1.93, 1.93, 0, 0, 1, 10, 1,\n" \
"R_CUBIC_TO, 0.56, 0, 1.03, 0.19, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"R_H_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 8,\n" \
"R_ARC_TO, 1.94, 1.94, 0, 0, 1, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"R_CUBIC_TO, 0, 0.56, -0.19, 1.03, -0.58, 1.42,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 17, 12,\n" \
"CLOSE,\n" \
"MOVE_TO, 11.54, 8.52,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.15, 2.15,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5, 17,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3, 15.5,\n" \
"V_LINE_TO, 12,\n" \
"R_ARC_TO, 2.12, 2.12, 0, 0, 0, 1.42, -0.61,\n" \
"ARC_TO, 1.88, 1.88, 0, 0, 0, 5, 10,\n" \
"R_ARC_TO, 1.88, 1.88, 0, 0, 0, -0.58, -1.39,\n" \
"ARC_TO, 2.12, 2.12, 0, 0, 0, 3, 8,\n" \
"V_LINE_TO, 4.52,\n" \
"R_CUBIC_TO, 0, -0.2, 0.05, -0.38, 0.14, -0.56,\n" \
"R_CUBIC_TO, 0.09, -0.17, 0.21, -0.32, 0.36, -0.46,\n" \
"R_LINE_TO, 1, 1,\n" \
"R_V_LINE_TO, 2.33,\n" \
"R_ARC_TO, 3.23, 3.23, 0, 0, 1, 1.47, 1.28,\n" \
"CUBIC_TO, 6.32, 8.69, 6.5, 9.32, 6.5, 10,\n" \
"R_CUBIC_TO, 0, 0.7, -0.18, 1.33, -0.53, 1.91,\n" \
"ARC_TO, 3.12, 3.12, 0, 0, 1, 4.5, 13.17,\n" \
"V_LINE_TO, 15.5,\n" \
"R_H_LINE_TO, 11,\n" \
"R_LINE_TO, 1, 1,\n" \
"R_CUBIC_TO, -0.15, 0.14, -0.32, 0.25, -0.49, 0.33,\n" \
"R_CUBIC_TO, -0.18, 0.09, -0.36, 0.14, -0.56, 0.17,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kExtensionOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseExtensionOnIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.5, 7,\n" \
"H_LINE_TO, 4.96,\n" \
"R_LINE_TO, 1.5, -1.5,\n" \
"H_LINE_TO, 15.5,\n" \
"R_V_LINE_TO, -11,\n" \
"R_H_LINE_TO, -11,\n" \
"R_V_LINE_TO, 2.33,\n" \
"R_CUBIC_TO, 0.48, 0.24, 0.9, 0.56, 1.24, 0.97,\n" \
"R_CUBIC_TO, 0.34, 0.41, 0.57, 0.87, 0.68, 1.39,\n" \
"LINE_TO, 4.86, 10.75,\n" \
"R_CUBIC_TO, 0.06, -0.11, 0.09, -0.23, 0.11, -0.35,\n" \
"R_CUBIC_TO, 0.02, -0.12, 0.03, -0.25, 0.03, -0.4,\n" \
"R_ARC_TO, 1.88, 1.88, 0, 0, 0, -0.58, -1.39,\n" \
"ARC_TO, 2.12, 2.12, 0, 0, 0, 3, 8,\n" \
"V_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4.5, 3,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, -0.56, 0.19, -1.03, 0.58, -1.42,\n" \
"ARC_TO, 1.93, 1.93, 0, 0, 1, 10, 1,\n" \
"R_CUBIC_TO, 0.56, 0, 1.03, 0.19, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"R_H_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 8,\n" \
"R_ARC_TO, 1.94, 1.94, 0, 0, 1, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"R_CUBIC_TO, 0, 0.56, -0.19, 1.03, -0.58, 1.42,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 17, 12,\n" \
"R_V_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 15.5, 17,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.83, 16,\n" \
"LINE_TO, 1, 13.19,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 1.77, 1.77,\n" \
"LINE_TO, 7.73, 10,\n" \
"R_LINE_TO, 1.04, 1.06,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kExtensionOnIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFamilyLinkIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"// A monochrome outline of the Family Link Kite icon, identical to the\n" \
"// \"cr20:kite\" used in WebUI.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"R_MOVE_TO, 4.63, 8,\n" \
"R_LINE_TO, 5.69, -6,\n" \
"LINE_TO, 16, 8,\n" \
"R_LINE_TO, -5.81, 8.87,\n" \
"R_CUBIC_TO, -0.08, 0.12, -0.18, 0.24, -0.28, 0.35,\n" \
"R_ARC_TO, 2.44, 2.44, 0, 0, 1, -1.78, 0.78,\n" \
"R_ARC_TO, 2.44, 2.44, 0, 0, 1, -1.78, -0.78,\n" \
"R_ARC_TO, 2.72, 2.72, 0, 0, 1, -0.74, -1.88,\n" \
"R_CUBIC_TO, 0, -0.36, -0.13, -0.69, -0.37, -0.94,\n" \
"R_ARC_TO, 1.21, 1.21, 0, 0, 0, -0.89, -0.39,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -1.33,\n" \
"R_H_LINE_TO, 0.33,\n" \
"R_CUBIC_TO, 0.68, 0, 1.31, 0.28, 1.78, 0.78,\n" \
"R_CUBIC_TO, 0.48, 0.5, 0.74, 1.17, 0.74, 1.88,\n" \
"R_CUBIC_TO, 0, 0.36, 0.13, 0.69, 0.37, 0.95,\n" \
"R_CUBIC_TO, 0.24, 0.26, 0.55, 0.39, 0.9, 0.39,\n" \
"R_CUBIC_TO, 0.34, 0, 0.65, -0.14, 0.9, -0.39,\n" \
"R_LINE_TO, 0.04, -0.04,\n" \
"R_CUBIC_TO, 0.04, -0.04, 0.07, -0.09, 0.11, -0.14,\n" \
"R_LINE_TO, 0.4, -0.6,\n" \
"LINE_TO, 4.63, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.69, 5.94,\n" \
"LINE_TO, 6.54, 8.17,\n" \
"R_LINE_TO, 3.78, -3.99,\n" \
"R_LINE_TO, 3.78, 3.99,\n" \
"R_LINE_TO, -3.78, 5.77,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFamilyLinkIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFeedbackIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 480, 600,\n" \
"R_QUADRATIC_TO, 17, 0, 28.5, -11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 520, 560,\n" \
"R_QUADRATIC_TO, 0, -17, -11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 520,\n" \
"R_QUADRATIC_TO, -17, 0, -28.5, 11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 440, 560,\n" \
"R_QUADRATIC_TO, 0, 17, 11.5, 28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 600,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -40, -160,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -240,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 240,\n" \
"CLOSE,\n" \
"MOVE_TO, 80, 880,\n" \
"R_V_LINE_TO, -720,\n" \
"R_QUADRATIC_TO, 0, -33, 23.5, -56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 160, 80,\n" \
"R_H_LINE_TO, 640,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 880, 160,\n" \
"R_V_LINE_TO, 480,\n" \
"R_QUADRATIC_TO, 0, 33, -23.5, 56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 800, 720,\n" \
"H_LINE_TO, 240,\n" \
"LINE_TO, 80, 880,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 126, -240,\n" \
"R_H_LINE_TO, 594,\n" \
"R_V_LINE_TO, -480,\n" \
"H_LINE_TO, 160,\n" \
"R_V_LINE_TO, 525,\n" \
"R_LINE_TO, 46, -45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -46, 0,\n" \
"R_V_LINE_TO, -480,\n" \
"R_V_LINE_TO, 480,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFeedbackIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFileDownloadIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 38, 18,\n" \
"R_H_LINE_TO, -8,\n" \
"V_LINE_TO, 6,\n" \
"H_LINE_TO, 18,\n" \
"R_V_LINE_TO, 12,\n" \
"R_H_LINE_TO, -8,\n" \
"R_LINE_TO, 14, 14,\n" \
"R_LINE_TO, 14, -14,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 36,\n" \
"R_V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 28,\n" \
"R_V_LINE_TO, -4,\n" \
"H_LINE_TO, 10,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFileDownloadIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFileDownloadChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 13,\n" \
"LINE_TO, 6, 9,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 2.19, 2.19,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 7.13,\n" \
"R_LINE_TO, 2.19, -2.19,\n" \
"LINE_TO, 14, 9,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4.51, 3,\n" \
"R_ARC_TO, 1.44, 1.44, 0, 0, 1, -1.05, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4, 14.5,\n" \
"V_LINE_TO, 13,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, 9,\n" \
"V_LINE_TO, 13,\n" \
"H_LINE_TO, 16,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 14.5, 16,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFileDownloadChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFileDownloadOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16, 18.13,\n" \
"LINE_TO, 13.88, 16,\n" \
"H_LINE_TO, 5.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4, 14.5,\n" \
"V_LINE_TO, 13,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, 6.88,\n" \
"R_LINE_TO, -1.94, -1.94,\n" \
"LINE_TO, 10, 13,\n" \
"LINE_TO, 6, 9,\n" \
"R_LINE_TO, 0.44, -0.44,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3.44, -7.69,\n" \
"LINE_TO, 11.5, 9.38,\n" \
"R_LINE_TO, 1.44, -1.44,\n" \
"LINE_TO, 14, 9,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.75, 8.63,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.25, 5.25,\n" \
"LINE_TO, 15.13, 13,\n" \
"H_LINE_TO, 16,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFileDownloadOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFilterIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 9.5, 17,\n" \
"V_LINE_TO, 12,\n" \
"H_LINE_TO, 11,\n" \
"R_V_LINE_TO, 1.75,\n" \
"R_H_LINE_TO, 6,\n" \
"R_V_LINE_TO, 1.5,\n" \
"H_LINE_TO, 11,\n" \
"V_LINE_TO, 17,\n" \
"H_LINE_TO, 9.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 3, 15.25,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 1.5,\n" \
"H_LINE_TO, 3,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 12.5,\n" \
"V_LINE_TO, 10.75,\n" \
"H_LINE_TO, 3,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 3,\n" \
"V_LINE_TO, 7.5,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 5,\n" \
"H_LINE_TO, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3, -1.75,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 1.5,\n" \
"H_LINE_TO, 9,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 8,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 1.75,\n" \
"H_LINE_TO, 17,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, -3.5,\n" \
"V_LINE_TO, 8,\n" \
"H_LINE_TO, 12,\n" \
"CLOSE,\n" \
"MOVE_TO, 3, 6.25,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 1.5,\n" \
"H_LINE_TO, 3,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFilterIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFingerprintIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 4,\n" \
"R_CUBIC_TO, 1.6, 0, 3.07, 0.34, 4.42, 1.02,\n" \
"R_CUBIC_TO, 1.35, 0.68, 2.35, 1.59, 3.02, 2.73,\n" \
"R_CUBIC_TO, 0.07, 0.11, 0.08, 0.23, 0.04, 0.38,\n" \
"R_ARC_TO, 0.55, 0.55, 0, 0, 1, -0.23, 0.31,\n" \
"R_ARC_TO, 0.46, 0.46, 0, 0, 1, -0.37, 0.04,\n" \
"R_ARC_TO, 0.56, 0.56, 0, 0, 1, -0.29, -0.23,\n" \
"R_CUBIC_TO, -0.58, -1, -1.47, -1.79, -2.66, -2.37,\n" \
"CUBIC_TO, 12.74, 5.29, 11.43, 5, 10, 5,\n" \
"R_CUBIC_TO, -1.43, 0, -2.74, 0.29, -3.93, 0.88,\n" \
"CUBIC_TO, 4.89, 6.46, 4, 7.25, 3.42, 8.25,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.67, 0.19,\n" \
"R_ARC_TO, 0.55, 0.55, 0, 0, 1, -0.23, -0.31,\n" \
"R_ARC_TO, 0.44, 0.44, 0, 0, 1, 0.04, -0.37,\n" \
"R_CUBIC_TO, 0.67, -1.14, 1.67, -2.05, 3.02, -2.73,\n" \
"CUBIC_TO, 6.93, 4.34, 8.4, 4, 10, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 2,\n" \
"R_CUBIC_TO, 1.79, 0, 3.32, 0.63, 4.58, 1.9,\n" \
"R_CUBIC_TO, 1.27, 1.26, 1.9, 2.8, 1.9, 4.61,\n" \
"R_CUBIC_TO, 0, 0.7, -0.24, 1.29, -0.72, 1.77,\n" \
"R_CUBIC_TO, -0.48, 0.49, -1.07, 0.73, -1.76, 0.73,\n" \
"R_ARC_TO, 2.4, 2.4, 0, 0, 1, -1.77, -0.73,\n" \
"R_ARC_TO, 2.4, 2.4, 0, 0, 1, -0.73, -1.77,\n" \
"R_CUBIC_TO, 0, -0.42, -0.14, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.42, 1.42, 0, 0, 0, 10.02, 11,\n" \
"R_CUBIC_TO, -0.4, 0, -0.75, 0.14, -1.05, 0.44,\n" \
"R_CUBIC_TO, -0.3, 0.29, -0.45, 0.64, -0.45, 1.06,\n" \
"R_CUBIC_TO, 0, 1.48, 0.42, 2.61, 1.24, 3.36,\n" \
"CUBIC_TO, 10.59, 16.62, 11.5, 17, 12.5, 17,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, 0.36, 0.14,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, 0.14, 0.36,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.14, 0.36,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.35, 0.14,\n" \
"R_CUBIC_TO, -1.22, 0, -2.36, -0.47, -3.41, -1.41,\n" \
"R_CUBIC_TO, -1.05, -0.94, -1.57, -2.3, -1.57, -4.09,\n" \
"R_CUBIC_TO, 0, -0.69, 0.24, -1.28, 0.72, -1.77,\n" \
"ARC_TO, 2.37, 2.37, 0, 0, 1, 10, 10,\n" \
"R_CUBIC_TO, 0.7, 0, 1.29, 0.24, 1.77, 0.73,\n" \
"R_CUBIC_TO, 0.49, 0.49, 0.73, 1.08, 0.73, 1.77,\n" \
"R_CUBIC_TO, 0, 0.42, 0.14, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.64, 0.44, 1.04, 0.44,\n" \
"R_CUBIC_TO, 0.4, 0, 0.75, -0.14, 1.05, -0.44,\n" \
"R_CUBIC_TO, 0.3, -0.29, 0.45, -0.64, 0.45, -1.06,\n" \
"R_CUBIC_TO, 0, -1.52, -0.54, -2.81, -1.61, -3.89,\n" \
"CUBIC_TO, 12.81, 7.54, 11.52, 7, 10, 7,\n" \
"R_CUBIC_TO, -1.52, 0, -2.8, 0.54, -3.87, 1.61,\n" \
"CUBIC_TO, 5.06, 9.67, 4.52, 10.97, 4.52, 12.5,\n" \
"R_CUBIC_TO, 0, 0.4, 0.04, 0.8, 0.13, 1.18,\n" \
"R_CUBIC_TO, 0.09, 0.38, 0.21, 0.75, 0.38, 1.12,\n" \
"R_CUBIC_TO, 0.07, 0.13, 0.08, 0.25, 0.03, 0.38,\n" \
"R_ARC_TO, 0.53, 0.53, 0, 0, 1, -0.26, 0.29,\n" \
"R_ARC_TO, 0.49, 0.49, 0, 0, 1, -0.39, 0.01,\n" \
"R_ARC_TO, 0.47, 0.47, 0, 0, 1, -0.28, -0.26,\n" \
"R_ARC_TO, 6.54, 6.54, 0, 0, 1, -0.44, -1.32,\n" \
"R_ARC_TO, 6.55, 6.55, 0, 0, 1, -0.14, -1.39,\n" \
"R_CUBIC_TO, 0, -1.8, 0.63, -3.34, 1.88, -4.61,\n" \
"CUBIC_TO, 6.68, 6.63, 8.21, 6, 10, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -4,\n" \
"R_CUBIC_TO, 0.96, 0, 1.88, 0.12, 2.77, 0.36,\n" \
"R_CUBIC_TO, 0.89, 0.24, 1.71, 0.57, 2.46, 1.02,\n" \
"R_CUBIC_TO, 0.13, 0.07, 0.2, 0.17, 0.23, 0.3,\n" \
"R_ARC_TO, 0.54, 0.54, 0, 0, 1, -0.06, 0.39,\n" \
"R_ARC_TO, 0.53, 0.53, 0, 0, 1, -0.3, 0.23,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.39, -0.04,\n" \
"R_ARC_TO, 8, 8, 0, 0, 0, -2.22, -0.91,\n" \
"ARC_TO, 10.21, 10.21, 0, 0, 0, 10, 3.04,\n" \
"R_CUBIC_TO, -0.85, 0, -1.68, 0.1, -2.49, 0.3,\n" \
"R_ARC_TO, 8, 8, 0, 0, 0, -2.22, 0.91,\n" \
"R_ARC_TO, 0.5, 0.5, 0, 0, 1, -0.69, -0.19,\n" \
"R_ARC_TO, 0.47, 0.47, 0, 0, 1, -0.06, -0.36,\n" \
"R_ARC_TO, 0.44, 0.44, 0, 0, 1, 0.23, -0.3,\n" \
"R_ARC_TO, 9.2, 9.2, 0, 0, 1, 2.46, -1.04,\n" \
"ARC_TO, 10.64, 10.64, 0, 0, 1, 10, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 6,\n" \
"R_CUBIC_TO, 1.06, 0, 1.99, 0.36, 2.8, 1.08,\n" \
"R_CUBIC_TO, 0.81, 0.72, 1.34, 1.66, 1.58, 2.81,\n" \
"R_CUBIC_TO, 0.03, 0.14, 0, 0.27, -0.07, 0.39,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.32, 0.22,\n" \
"R_ARC_TO, 0.47, 0.47, 0, 0, 1, -0.38, -0.07,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.21, -0.32,\n" \
"R_CUBIC_TO, -0.19, -0.92, -0.6, -1.66, -1.23, -2.24,\n" \
"CUBIC_TO, 11.54, 9.29, 10.82, 9, 10, 9,\n" \
"R_CUBIC_TO, -0.96, 0, -1.78, 0.39, -2.46, 1.17,\n" \
"R_CUBIC_TO, -0.68, 0.78, -1.02, 1.72, -1.02, 2.83,\n" \
"R_CUBIC_TO, 0, 0.75, 0.16, 1.48, 0.49, 2.19,\n" \
"R_CUBIC_TO, 0.32, 0.71, 0.76, 1.27, 1.3, 1.69,\n" \
"R_ARC_TO, 0.51, 0.51, 0, 0, 1, 0.2, 0.33,\n" \
"R_ARC_TO, 0.49, 0.49, 0, 0, 1, -0.09, 0.38,\n" \
"R_ARC_TO, 0.45, 0.45, 0, 0, 1, -0.34, 0.19,\n" \
"R_ARC_TO, 0.52, 0.52, 0, 0, 1, -0.37, -0.1,\n" \
"R_CUBIC_TO, -0.66, -0.53, -1.19, -1.22, -1.59, -2.06,\n" \
"ARC_TO, 6.08, 6.08, 0, 0, 1, 5.52, 13,\n" \
"R_CUBIC_TO, 0, -1.39, 0.43, -2.57, 1.3, -3.54,\n" \
"CUBIC_TO, 7.69, 8.48, 8.75, 8, 10, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 4,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, 0.36, 0.14,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, 0.14, 0.36,\n" \
"R_CUBIC_TO, 0, 1.04, 0.36, 1.83, 1.08, 2.38,\n" \
"R_ARC_TO, 3.9, 3.9, 0, 0, 0, 2.4, 0.81,\n" \
"R_CUBIC_TO, 0.15, 0, 0.3, -0.02, 0.45, -0.04,\n" \
"R_LINE_TO, 0.43, -0.08,\n" \
"R_ARC_TO, 0.52, 0.52, 0, 0, 1, 0.37, 0.06,\n" \
"R_CUBIC_TO, 0.13, 0.07, 0.2, 0.17, 0.23, 0.31,\n" \
"R_ARC_TO, 0.53, 0.53, 0, 0, 1, -0.06, 0.39,\n" \
"R_ARC_TO, 0.44, 0.44, 0, 0, 1, -0.31, 0.22,\n" \
"R_CUBIC_TO, -0.19, 0.04, -0.38, 0.07, -0.55, 0.1,\n" \
"R_CUBIC_TO, -0.17, 0.03, -0.36, 0.04, -0.55, 0.04,\n" \
"R_CUBIC_TO, -1.32, 0, -2.4, -0.4, -3.23, -1.21,\n" \
"R_CUBIC_TO, -0.83, -0.81, -1.25, -1.8, -1.25, -2.98,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, 0.14, -0.35,\n" \
"ARC_TO, 0.48, 0.48, 0, 0, 1, 10, 12,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFingerprintIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFingerprintOffIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 17.06, 18.48,\n" \
"R_LINE_TO, -7.42, -7.44,\n" \
"R_ARC_TO, 1.55, 1.55, 0, 0, 0, -0.81, 0.52,\n" \
"R_ARC_TO, 1.41, 1.41, 0, 0, 0, -0.33, 0.94,\n" \
"R_CUBIC_TO, 0, 1.48, 0.41, 2.61, 1.24, 3.36,\n" \
"R_CUBIC_TO, 0.83, 0.76, 1.74, 1.14, 2.74, 1.14,\n" \
"R_CUBIC_TO, 0.14, 0, 0.25, 0.05, 0.35, 0.14,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, 0.15, 0.36,\n" \
"R_CUBIC_TO, 0, 0.14, -0.05, 0.26, -0.15, 0.36,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.35, 0.14,\n" \
"R_CUBIC_TO, -1.22, 0, -2.36, -0.47, -3.41, -1.41,\n" \
"CUBIC_TO, 8.02, 15.66, 7.5, 14.29, 7.5, 12.5,\n" \
"R_CUBIC_TO, 0, -0.48, 0.13, -0.93, 0.39, -1.32,\n" \
"R_CUBIC_TO, 0.25, -0.39, 0.59, -0.7, 0.99, -0.91,\n" \
"R_LINE_TO, -0.69, -0.66,\n" \
"R_CUBIC_TO, -0.5, 0.36, -0.9, 0.84, -1.21, 1.43,\n" \
"R_CUBIC_TO, -0.31, 0.59, -0.46, 1.25, -0.46, 1.97,\n" \
"R_CUBIC_TO, 0, 0.75, 0.16, 1.48, 0.49, 2.19,\n" \
"R_CUBIC_TO, 0.32, 0.71, 0.76, 1.27, 1.3, 1.69,\n" \
"R_ARC_TO, 0.51, 0.51, 0, 0, 1, 0.2, 0.33,\n" \
"R_ARC_TO, 0.49, 0.49, 0, 0, 1, -0.09, 0.38,\n" \
"R_ARC_TO, 0.45, 0.45, 0, 0, 1, -0.34, 0.19,\n" \
"R_ARC_TO, 0.52, 0.52, 0, 0, 1, -0.37, -0.1,\n" \
"R_CUBIC_TO, -0.66, -0.53, -1.19, -1.22, -1.59, -2.06,\n" \
"ARC_TO, 6.08, 6.08, 0, 0, 1, 5.52, 13,\n" \
"R_CUBIC_TO, 0, -0.86, 0.18, -1.65, 0.53, -2.36,\n" \
"R_CUBIC_TO, 0.36, -0.71, 0.83, -1.3, 1.41, -1.74,\n" \
"R_LINE_TO, -0.75, -0.77,\n" \
"R_ARC_TO, 5.58, 5.58, 0, 0, 0, -1.59, 1.89,\n" \
"ARC_TO, 5.29, 5.29, 0, 0, 0, 4.52, 12.5,\n" \
"R_CUBIC_TO, 0, 0.4, 0.04, 0.8, 0.13, 1.18,\n" \
"R_CUBIC_TO, 0.09, 0.38, 0.21, 0.75, 0.38, 1.12,\n" \
"R_CUBIC_TO, 0.07, 0.13, 0.08, 0.25, 0.03, 0.38,\n" \
"R_ARC_TO, 0.53, 0.53, 0, 0, 1, -0.26, 0.29,\n" \
"R_ARC_TO, 0.49, 0.49, 0, 0, 1, -0.39, 0.01,\n" \
"R_ARC_TO, 0.47, 0.47, 0, 0, 1, -0.28, -0.26,\n" \
"R_ARC_TO, 6.54, 6.54, 0, 0, 1, -0.44, -1.32,\n" \
"R_ARC_TO, 6.55, 6.55, 0, 0, 1, -0.14, -1.39,\n" \
"R_CUBIC_TO, 0, -1.04, 0.22, -2, 0.66, -2.87,\n" \
"ARC_TO, 6.63, 6.63, 0, 0, 1, 6, 7.42,\n" \
"LINE_TO, 5.08, 6.5,\n" \
"R_CUBIC_TO, -0.35, 0.25, -0.66, 0.52, -0.94, 0.81,\n" \
"R_ARC_TO, 4.76, 4.76, 0, 0, 0, -0.72, 0.94,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.67, 0.19,\n" \
"R_ARC_TO, 0.56, 0.56, 0, 0, 1, -0.23, -0.3,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, 0.04, -0.39,\n" \
"R_ARC_TO, 7.06, 7.06, 0, 0, 1, 1.81, -1.96,\n" \
"LINE_TO, 1.5, 2.94,\n" \
"R_LINE_TO, 0.73, -0.71,\n" \
"R_LINE_TO, 15.54, 15.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 13.98, 12.5,\n" \
"R_ARC_TO, 0.47, 0.47, 0, 0, 1, -0.37, -0.07,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.21, -0.32,\n" \
"R_CUBIC_TO, -0.16, -0.84, -0.52, -1.53, -1.05, -2.08,\n" \
"R_CUBIC_TO, -0.53, -0.55, -1.16, -0.88, -1.89, -0.99,\n" \
"R_LINE_TO, -0.98, -0.98,\n" \
"R_CUBIC_TO, 0.1, -0.03, 0.18, -0.05, 0.26, -0.05,\n" \
"ARC_TO, 2.63, 2.63, 0, 0, 1, 10, 8,\n" \
"R_CUBIC_TO, 1.06, 0, 1.99, 0.36, 2.8, 1.08,\n" \
"R_CUBIC_TO, 0.81, 0.72, 1.34, 1.66, 1.58, 2.81,\n" \
"R_CUBIC_TO, 0.03, 0.14, 0, 0.27, -0.07, 0.39,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.32, 0.22,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 2,\n" \
"R_CUBIC_TO, 0.96, 0, 1.88, 0.12, 2.77, 0.36,\n" \
"R_CUBIC_TO, 0.89, 0.24, 1.71, 0.57, 2.46, 1.02,\n" \
"R_CUBIC_TO, 0.13, 0.07, 0.2, 0.17, 0.23, 0.3,\n" \
"R_ARC_TO, 0.54, 0.54, 0, 0, 1, -0.06, 0.39,\n" \
"R_ARC_TO, 0.53, 0.53, 0, 0, 1, -0.3, 0.23,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.39, -0.04,\n" \
"R_ARC_TO, 8.03, 8.03, 0, 0, 0, -2.21, -0.91,\n" \
"R_ARC_TO, 10.1, 10.1, 0, 0, 0, -2.48, -0.3,\n" \
"R_CUBIC_TO, -0.8, 0, -1.59, 0.09, -2.36, 0.27,\n" \
"R_CUBIC_TO, -0.77, 0.18, -1.48, 0.45, -2.14, 0.81,\n" \
"R_LINE_TO, -0.73, -0.75,\n" \
"R_ARC_TO, 9.62, 9.62, 0, 0, 1, 2.44, -1.02,\n" \
"ARC_TO, 10.56, 10.56, 0, 0, 1, 10, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 2,\n" \
"R_CUBIC_TO, 1.6, 0, 3.07, 0.34, 4.42, 1.02,\n" \
"R_CUBIC_TO, 1.35, 0.68, 2.35, 1.59, 3.02, 2.73,\n" \
"R_ARC_TO, 0.5, 0.5, 0, 0, 1, -0.19, 0.69,\n" \
"R_ARC_TO, 0.46, 0.46, 0, 0, 1, -0.37, 0.04,\n" \
"R_ARC_TO, 0.56, 0.56, 0, 0, 1, -0.29, -0.23,\n" \
"R_CUBIC_TO, -0.58, -1, -1.47, -1.79, -2.66, -2.37,\n" \
"CUBIC_TO, 12.74, 5.29, 11.43, 5, 10, 5,\n" \
"R_ARC_TO, 9.45, 9.45, 0, 0, 0, -3.06, 0.52,\n" \
"R_LINE_TO, -0.75, -0.75,\n" \
"ARC_TO, 9.73, 9.73, 0, 0, 1, 10, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 2,\n" \
"R_CUBIC_TO, 1.79, 0, 3.32, 0.63, 4.57, 1.9,\n" \
"R_CUBIC_TO, 1.25, 1.26, 1.88, 2.8, 1.88, 4.61,\n" \
"R_CUBIC_TO, 0, 0.36, -0.07, 0.69, -0.19, 0.99,\n" \
"R_CUBIC_TO, -0.13, 0.3, -0.32, 0.57, -0.56, 0.81,\n" \
"LINE_TO, 15, 13.58,\n" \
"R_CUBIC_TO, 0.15, -0.14, 0.27, -0.3, 0.36, -0.48,\n" \
"R_ARC_TO, 1.46, 1.46, 0, 0, 0, 0.13, -0.6,\n" \
"R_CUBIC_TO, 0, -1.52, -0.53, -2.81, -1.6, -3.89,\n" \
"CUBIC_TO, 12.81, 7.54, 11.52, 7, 10, 7,\n" \
"R_CUBIC_TO, -0.23, 0, -0.47, 0.02, -0.71, 0.04,\n" \
"R_ARC_TO, 3.64, 3.64, 0, 0, 0, -0.69, 0.14,\n" \
"R_LINE_TO, -0.79, -0.79,\n" \
"R_ARC_TO, 6.79, 6.79, 0, 0, 1, 1.06, -0.29,\n" \
"CUBIC_TO, 9.24, 6.04, 9.61, 6, 10, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.98, 10.67,\n" \
"R_CUBIC_TO, -1.22, 0, -2.28, -0.37, -3.16, -1.12,\n" \
"R_CUBIC_TO, -0.88, -0.75, -1.32, -1.78, -1.32, -3.09,\n" \
"R_CUBIC_TO, 0, -0.12, 0.05, -0.23, 0.14, -0.32,\n" \
"ARC_TO, 0.47, 0.47, 0, 0, 1, 9.98, 12,\n" \
"R_CUBIC_TO, 0.14, 0, 0.26, 0.04, 0.36, 0.11,\n" \
"R_ARC_TO, 0.36, 0.36, 0, 0, 1, 0.16, 0.31,\n" \
"R_CUBIC_TO, 0, 1.03, 0.34, 1.82, 1.03, 2.4,\n" \
"R_CUBIC_TO, 0.69, 0.57, 1.5, 0.86, 2.45, 0.86,\n" \
"R_CUBIC_TO, 0.14, 0, 0.28, -0.01, 0.41, -0.02,\n" \
"R_CUBIC_TO, 0.14, -0.01, 0.28, -0.04, 0.42, -0.06,\n" \
"R_ARC_TO, 0.35, 0.35, 0, 0, 1, 0.33, 0.09,\n" \
"R_CUBIC_TO, 0.1, 0.09, 0.16, 0.2, 0.19, 0.32,\n" \
"R_ARC_TO, 0.43, 0.43, 0, 0, 1, -0.06, 0.34,\n" \
"R_ARC_TO, 0.44, 0.44, 0, 0, 1, -0.29, 0.2,\n" \
"R_CUBIC_TO, -0.17, 0.04, -0.33, 0.07, -0.5, 0.09,\n" \
"R_ARC_TO, 4.35, 4.35, 0, 0, 1, -0.5, 0.03,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFingerprintOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFlightIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 7, 18,\n" \
"R_V_LINE_TO, -1,\n" \
"R_LINE_TO, 1.75, -1.31,\n" \
"R_V_LINE_TO, -4.62,\n" \
"LINE_TO, 2, 13,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_LINE_TO, 6.75, -3.85,\n" \
"V_LINE_TO, 3.25,\n" \
"R_CUBIC_TO, 0, -0.35, 0.12, -0.64, 0.36, -0.89,\n" \
"CUBIC_TO, 9.36, 2.12, 9.65, 2, 10, 2,\n" \
"R_CUBIC_TO, 0.35, 0, 0.64, 0.12, 0.89, 0.36,\n" \
"R_CUBIC_TO, 0.24, 0.25, 0.36, 0.54, 0.36, 0.89,\n" \
"R_V_LINE_TO, 4.39,\n" \
"LINE_TO, 18, 11.5,\n" \
"V_LINE_TO, 13,\n" \
"R_LINE_TO, -6.75, -1.94,\n" \
"R_V_LINE_TO, 4.63,\n" \
"LINE_TO, 13, 17,\n" \
"R_V_LINE_TO, 1,\n" \
"R_LINE_TO, -3, -1,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFlightIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFolderIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"MOVE_TO, 4.8, 5,\n" \
"CUBIC_TO, 3.26, 5, 2.01, 6.29, 2.01, 7.88,\n" \
"LINE_TO, 2, 25.13,\n" \
"CUBIC_TO, 2, 26.71, 3.26, 28, 4.8, 28,\n" \
"LINE_TO, 27.2, 28,\n" \
"CUBIC_TO, 28.74, 28, 30, 26.71, 30, 25.13,\n" \
"LINE_TO, 30, 10.75,\n" \
"CUBIC_TO, 30, 9.17, 28.54, 8, 27, 8,\n" \
"LINE_TO, 16, 8,\n" \
"LINE_TO, 13.2, 5,\n" \
"LINE_TO, 4.8, 5,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 2.5, 2,\n" \
"CUBIC_TO, 1.73, 2, 1.01, 3.15, 1.01, 3.94,\n" \
"LINE_TO, 1, 12.56,\n" \
"CUBIC_TO, 1, 13.35, 1.63, 14, 2.4, 14,\n" \
"LINE_TO, 13.6, 14,\n" \
"CUBIC_TO, 14.37, 14, 15, 13.35, 15, 12.56,\n" \
"LINE_TO, 15, 5.38,\n" \
"CUBIC_TO, 15, 4.58, 14.27, 4, 13.5, 4,\n" \
"LINE_TO, 9, 4,\n" \
"LINE_TO, 7, 2,\n" \
"LINE_TO, 2.5, 2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFolderIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFolderChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4, 20,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, -0.19, -1.41, -0.59,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 2, 18,\n" \
"V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, -0.55, 0.2, -1.02, 0.59, -1.41,\n" \
"CUBIC_TO, 2.98, 4.2, 3.45, 4, 4, 4,\n" \
"R_H_LINE_TO, 6,\n" \
"R_LINE_TO, 2, 2,\n" \
"R_H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, 0.2, 1.41, 0.59,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.59, 0.86, 0.59, 1.41,\n" \
"R_V_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0, 0.55, -0.19, 1.02, -0.59, 1.41,\n" \
"R_CUBIC_TO, -0.39, 0.39, -0.86, 0.59, -1.41, 0.59,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2,\n" \
"R_H_LINE_TO, 16,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, -8.82,\n" \
"R_LINE_TO, -2, -2,\n" \
"H_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.5, 16,\n" \
"R_CUBIC_TO, -0.4, 0, -0.75, -0.15, -1.05, -0.45,\n" \
"R_CUBIC_TO, -0.3, -0.3, -0.45, -0.65, -0.45, -1.05,\n" \
"R_V_LINE_TO, -9,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.45, -1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 3.5, 4,\n" \
"H_LINE_TO, 8,\n" \
"R_LINE_TO, 2, 2,\n" \
"R_H_LINE_TO, 6.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, 0.4, -0.15, 0.75, -0.44, 1.05,\n" \
"R_CUBIC_TO, -0.29, 0.3, -0.64, 0.45, -1.06, 0.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 13,\n" \
"R_V_LINE_TO, -7,\n" \
"H_LINE_TO, 9.38,\n" \
"R_LINE_TO, -2, -2,\n" \
"H_LINE_TO, 3.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"R_V_LINE_TO, -9,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFolderChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFolderManagedRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 17, 22,\n" \
"R_LINE_TO, -0.3, -1.5,\n" \
"R_ARC_TO, 7.56, 7.56, 0, 0, 1, -0.56, -0.26,\n" \
"R_ARC_TO, 4.18, 4.18, 0, 0, 1, -0.53, -0.34,\n" \
"R_LINE_TO, -1.45, 0.45,\n" \
"R_LINE_TO, -1, -1.7,\n" \
"R_LINE_TO, 1.15, -1,\n" \
"R_CUBIC_TO, -0.04, -0.2, -0.05, -0.41, -0.05, -0.65,\n" \
"R_CUBIC_TO, 0, -0.23, 0.02, -0.45, 0.05, -0.65,\n" \
"R_LINE_TO, -1.15, -1,\n" \
"R_LINE_TO, 1, -1.7,\n" \
"R_LINE_TO, 1.45, 0.45,\n" \
"R_CUBIC_TO, 0.18, -0.14, 0.36, -0.25, 0.54, -0.34,\n" \
"R_CUBIC_TO, 0.18, -0.09, 0.36, -0.18, 0.56, -0.26,\n" \
"LINE_TO, 17, 12,\n" \
"R_H_LINE_TO, 2,\n" \
"R_LINE_TO, 0.3, 1.5,\n" \
"R_CUBIC_TO, 0.2, 0.08, 0.39, 0.17, 0.56, 0.26,\n" \
"R_CUBIC_TO, 0.18, 0.09, 0.35, 0.2, 0.54, 0.34,\n" \
"R_LINE_TO, 1.45, -0.45,\n" \
"R_LINE_TO, 1, 1.7,\n" \
"R_LINE_TO, -1.15, 1,\n" \
"R_CUBIC_TO, 0.04, 0.2, 0.05, 0.41, 0.05, 0.65,\n" \
"R_CUBIC_TO, 0, 0.23, -0.02, 0.45, -0.05, 0.65,\n" \
"R_LINE_TO, 1.15, 1,\n" \
"R_LINE_TO, -1, 1.7,\n" \
"R_LINE_TO, -1.45, -0.45,\n" \
"R_CUBIC_TO, -0.18, 0.14, -0.36, 0.25, -0.53, 0.34,\n" \
"R_ARC_TO, 7.56, 7.56, 0, 0, 1, -0.56, 0.26,\n" \
"LINE_TO, 19, 22,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1, -3,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, -0.19, 1.41, -0.59,\n" \
"R_CUBIC_TO, 0.39, -0.39, 0.59, -0.86, 0.59, -1.41,\n" \
"R_CUBIC_TO, 0, -0.55, -0.19, -1.02, -0.59, -1.41,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 0, 18, 15,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, 0.2, -1.41, 0.59,\n" \
"R_CUBIC_TO, -0.39, 0.39, -0.59, 0.86, -0.59, 1.41,\n" \
"R_CUBIC_TO, 0, 0.55, 0.2, 1.02, 0.59, 1.41,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.86, 0.59, 1.41, 0.59,\n" \
"CLOSE,\n" \
"MOVE_TO, 4, 18,\n" \
"V_LINE_TO, 6,\n" \
"R_V_LINE_TO, 4.3,\n" \
"R_V_LINE_TO, -0.3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 2,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, -0.19, -1.41, -0.59,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 2, 18,\n" \
"V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, -0.55, 0.2, -1.02, 0.59, -1.41,\n" \
"CUBIC_TO, 2.98, 4.2, 3.45, 4, 4, 4,\n" \
"R_H_LINE_TO, 6,\n" \
"R_LINE_TO, 2, 2,\n" \
"R_H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, 0.2, 1.41, 0.59,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.59, 0.86, 0.59, 1.41,\n" \
"R_V_LINE_TO, 3.27,\n" \
"R_ARC_TO, 6.46, 6.46, 0, 0, 0, -0.95, -0.56,\n" \
"ARC_TO, 8.89, 8.89, 0, 0, 0, 20, 10.3,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, -8.82,\n" \
"R_LINE_TO, -2, -2,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 12,\n" \
"R_H_LINE_TO, 7.07,\n" \
"R_CUBIC_TO, 0.05, 0.35, 0.13, 0.69, 0.24, 1.02,\n" \
"R_CUBIC_TO, 0.11, 0.34, 0.24, 0.66, 0.39, 0.98,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 14.83, 18,\n" \
"R_LINE_TO, -0.27, -1.17,\n" \
"R_ARC_TO, 3.9, 3.9, 0, 0, 1, -0.55, -0.24,\n" \
"R_ARC_TO, 2.19, 2.19, 0, 0, 1, -0.49, -0.34,\n" \
"R_LINE_TO, -1.14, 0.36,\n" \
"R_LINE_TO, -0.67, -1.15,\n" \
"R_LINE_TO, 0.86, -0.83,\n" \
"R_ARC_TO, 2.73, 2.73, 0, 0, 1, -0.06, -0.61,\n" \
"R_CUBIC_TO, 0, -0.22, 0.02, -0.42, 0.06, -0.62,\n" \
"R_LINE_TO, -0.86, -0.81,\n" \
"R_LINE_TO, 0.67, -1.14,\n" \
"R_LINE_TO, 1.13, 0.33,\n" \
"R_CUBIC_TO, 0.15, -0.15, 0.32, -0.28, 0.5, -0.37,\n" \
"R_CUBIC_TO, 0.18, -0.1, 0.37, -0.17, 0.56, -0.23,\n" \
"LINE_TO, 14.83, 10,\n" \
"R_H_LINE_TO, 1.34,\n" \
"R_LINE_TO, 0.27, 1.17,\n" \
"R_CUBIC_TO, 0.2, 0.07, 0.39, 0.15, 0.57, 0.24,\n" \
"R_CUBIC_TO, 0.19, 0.09, 0.35, 0.21, 0.49, 0.36,\n" \
"R_LINE_TO, 1.13, -0.31,\n" \
"R_LINE_TO, 0.67, 1.15,\n" \
"R_LINE_TO, -0.84, 0.79,\n" \
"R_CUBIC_TO, 0.04, 0.19, 0.06, 0.39, 0.06, 0.61,\n" \
"R_CUBIC_TO, 0, 0.22, -0.02, 0.43, -0.06, 0.62,\n" \
"R_LINE_TO, 0.86, 0.81,\n" \
"R_LINE_TO, -0.67, 1.14,\n" \
"R_LINE_TO, -1.14, -0.33,\n" \
"R_CUBIC_TO, -0.15, 0.14, -0.32, 0.25, -0.5, 0.34,\n" \
"R_ARC_TO, 4.5, 4.5, 0, 0, 1, -0.56, 0.24,\n" \
"LINE_TO, 16.17, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.67, -2.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, -0.15, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.3, 0.44, -0.65, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.41, -0.15, -0.77, -0.44, -1.06,\n" \
"R_ARC_TO, 1.45, 1.45, 0, 0, 0, -1.06, -0.44,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, 0.15, -1.06, 0.44,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 0, 14, 14,\n" \
"R_CUBIC_TO, 0, 0.41, 0.15, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.3, 0.29, 0.65, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.5, 14.5,\n" \
"R_V_LINE_TO, -9,\n" \
"R_V_LINE_TO, 3.11,\n" \
"V_LINE_TO, 8.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 1.5,\n" \
"R_CUBIC_TO, -0.4, 0, -0.75, -0.15, -1.05, -0.45,\n" \
"R_CUBIC_TO, -0.3, -0.3, -0.45, -0.65, -0.45, -1.05,\n" \
"R_V_LINE_TO, -9,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.45, -1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 3.5, 4,\n" \
"H_LINE_TO, 8,\n" \
"R_LINE_TO, 2, 2,\n" \
"R_H_LINE_TO, 6.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 1.65,\n" \
"R_ARC_TO, 5.86, 5.86, 0, 0, 0, -0.72, -0.32,\n" \
"R_ARC_TO, 5.07, 5.07, 0, 0, 0, -0.78, -0.22,\n" \
"V_LINE_TO, 7.5,\n" \
"H_LINE_TO, 9.38,\n" \
"R_LINE_TO, -2, -2,\n" \
"H_LINE_TO, 3.5,\n" \
"R_V_LINE_TO, 9,\n" \
"R_H_LINE_TO, 6.52,\n" \
"R_CUBIC_TO, 0.03, 0.26, 0.07, 0.52, 0.14, 0.77,\n" \
"R_CUBIC_TO, 0.06, 0.25, 0.14, 0.5, 0.24, 0.73,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFolderManagedRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFolderOpenIcon) { \
              std::string s = \
"// Copyright 2019 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 40, 12,\n" \
"H_LINE_TO, 24,\n" \
"R_LINE_TO, -4, -4,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, -2.21, 0, -3.98, 1.79, -3.98, 4,\n" \
"LINE_TO, 4, 36,\n" \
"R_CUBIC_TO, 0, 2.21, 1.79, 4, 4, 4,\n" \
"R_H_LINE_TO, 32,\n" \
"R_CUBIC_TO, 2.21, 0, 4, -1.79, 4, -4,\n" \
"V_LINE_TO, 16,\n" \
"R_CUBIC_TO, 0, -2.21, -1.79, -4, -4, -4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 24,\n" \
"H_LINE_TO, 8,\n" \
"V_LINE_TO, 16,\n" \
"R_H_LINE_TO, 32,\n" \
"R_V_LINE_TO, 20,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFolderOpenIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFontDownloadIcon) { \
              std::string s = \
"// Copyright 2020 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 20, 2,\n" \
"H_LINE_TO, 4,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 16,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 16,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 18,\n" \
"H_LINE_TO, 4,\n" \
"V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 16,\n" \
"R_V_LINE_TO, 16,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.69, 6,\n" \
"R_H_LINE_TO, 2.6,\n" \
"R_LINE_TO, 4.51, 12,\n" \
"R_H_LINE_TO, -2.5,\n" \
"R_LINE_TO, -1.01, -2.87,\n" \
"H_LINE_TO, 9.7,\n" \
"LINE_TO, 8.7, 18,\n" \
"H_LINE_TO, 6.2,\n" \
"R_LINE_TO, 4.49, -12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.87, 7.06,\n" \
"R_LINE_TO, -1.06, -3.02,\n" \
"R_LINE_TO, -0.43, -1.44,\n" \
"R_H_LINE_TO, -0.13,\n" \
"R_LINE_TO, -0.44, 1.44,\n" \
"R_LINE_TO, -1.07, 3.02,\n" \
"R_H_LINE_TO, 3.13,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFontDownloadIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFontDownloadChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 5.36, 15,\n" \
"H_LINE_TO, 7.09,\n" \
"LINE_TO, 8, 12.44,\n" \
"R_H_LINE_TO, 4.02,\n" \
"LINE_TO, 12.92, 15,\n" \
"R_H_LINE_TO, 1.72,\n" \
"R_LINE_TO, -3.75, -10,\n" \
"R_H_LINE_TO, -1.79,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.5, 11,\n" \
"R_LINE_TO, 1.46, -4.14,\n" \
"R_H_LINE_TO, 0.06,\n" \
"LINE_TO, 11.48, 11,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5, 7,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 2, 16.5,\n" \
"R_V_LINE_TO, -13,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 2,\n" \
"R_H_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 13,\n" \
"R_V_LINE_TO, -13,\n" \
"R_H_LINE_TO, -13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -13,\n" \
"R_V_LINE_TO, 13,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFontDownloadChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseFontDownloadOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16.73, 18.83,\n" \
"LINE_TO, 15.88, 18,\n" \
"H_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 2, 16.5,\n" \
"V_LINE_TO, 4.13,\n" \
"R_LINE_TO, -0.83, -0.85,\n" \
"LINE_TO, 2.23, 2.23,\n" \
"R_LINE_TO, 15.54, 15.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.5, 16.5,\n" \
"R_H_LINE_TO, 10.88,\n" \
"LINE_TO, 3.5, 5.63,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 14.5, -0.62,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 3.5,\n" \
"H_LINE_TO, 5.63,\n" \
"LINE_TO, 4.13, 2,\n" \
"H_LINE_TO, 16.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4.83, -4.83,\n" \
"LINE_TO, 10.61, 8.48,\n" \
"R_LINE_TO, -0.56, -1.62,\n" \
"R_H_LINE_TO, -0.09,\n" \
"R_LINE_TO, -0.25, 0.73,\n" \
"R_LINE_TO, -1.14, -1.14,\n" \
"LINE_TO, 9.11, 5,\n" \
"R_H_LINE_TO, 1.79,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4.27, 0.04,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.71, -2.58,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.19, 6.27,\n" \
"R_LINE_TO, -1.06, -3.04,\n" \
"R_LINE_TO, 2.71, 2.71,\n" \
"R_LINE_TO, 0.13, 0.33,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.33, 15,\n" \
"LINE_TO, 7.98, 7.98,\n" \
"R_LINE_TO, 1.19, 1.19,\n" \
"LINE_TO, 8.52, 11,\n" \
"R_H_LINE_TO, 2.48,\n" \
"R_LINE_TO, 1.43, 1.44,\n" \
"H_LINE_TO, 8.06,\n" \
"LINE_TO, 7.17, 15,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kFontDownloadOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseForward10Icon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"MOVE_TO, 360, 640,\n" \
"R_V_LINE_TO, -180,\n" \
"R_H_LINE_TO, -60,\n" \
"R_V_LINE_TO, -60,\n" \
"R_H_LINE_TO, 120,\n" \
"R_V_LINE_TO, 240,\n" \
"R_H_LINE_TO, -60,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 140, 0,\n" \
"R_QUADRATIC_TO, -17, 0, -28.5, -11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 460, 600,\n" \
"R_V_LINE_TO, -160,\n" \
"R_QUADRATIC_TO, 0, -17, 11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 500, 400,\n" \
"R_H_LINE_TO, 80,\n" \
"R_QUADRATIC_TO, 17, 0, 28.5, 11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 620, 440,\n" \
"R_V_LINE_TO, 160,\n" \
"R_QUADRATIC_TO, 0, 17, -11.5, 28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 580, 640,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 20, -60,\n" \
"R_H_LINE_TO, 40,\n" \
"R_V_LINE_TO, -120,\n" \
"R_H_LINE_TO, -40,\n" \
"R_V_LINE_TO, 120,\n" \
"CLOSE,\n" \
"MOVE_TO, 480, 880,\n" \
"R_QUADRATIC_TO, -75, 0, -140.5, -28.5,\n" \
"R_QUADRATIC_TO, -65.5, -28.5, -114, -77,\n" \
"R_QUADRATIC_TO, -48.5, -48.5, -77, -114,\n" \
"QUADRATIC_TO_SHORTHAND, 120, 520,\n" \
"R_QUADRATIC_TO, 0, -75, 28.5, -140.5,\n" \
"R_QUADRATIC_TO, 28.5, -65.5, 77, -114,\n" \
"R_QUADRATIC_TO, 48.5, -48.5, 114, -77,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 160,\n" \
"R_H_LINE_TO, 6,\n" \
"R_LINE_TO, -62, -62,\n" \
"R_LINE_TO, 56, -58,\n" \
"R_LINE_TO, 160, 160,\n" \
"R_LINE_TO, -160, 160,\n" \
"R_LINE_TO, -56, -58,\n" \
"R_LINE_TO, 62, -62,\n" \
"R_H_LINE_TO, -6,\n" \
"R_QUADRATIC_TO, -117, 0, -198.5, 81.5,\n" \
"QUADRATIC_TO_SHORTHAND, 200, 520,\n" \
"R_QUADRATIC_TO, 0, 117, 81.5, 198.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 800,\n" \
"R_QUADRATIC_TO, 117, 0, 198.5, -81.5,\n" \
"QUADRATIC_TO_SHORTHAND, 760, 520,\n" \
"R_H_LINE_TO, 80,\n" \
"R_QUADRATIC_TO, 0, 75, -28.5, 140.5,\n" \
"R_QUADRATIC_TO, -28.5, 65.5, -77, 114,\n" \
"R_QUADRATIC_TO, -48.5, 48.5, -114, 77,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 880,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kForward10Icon); \
              } \
 \
TEST_F(VectorIconsTest, ParseForwardArrowIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"MOVE_TO, 27.78, 14.03,\n" \
"CUBIC_TO, 28.47, 14.15, 29, 14.77, 29, 15.5,\n" \
"CUBIC_TO, 29, 16.25, 28.47, 16.86, 27.76, 16.98,\n" \
"LINE_TO, 27.98, 17.19,\n" \
"LINE_TO, 17.64, 27.51,\n" \
"CUBIC_TO, 17.35, 27.81, 16.95, 28, 16.5, 28,\n" \
"CUBIC_TO, 15.67, 28, 15, 27.33, 15, 26.5,\n" \
"CUBIC_TO, 15, 26.05, 15.2, 25.65, 15.51, 25.38,\n" \
"LINE_TO, 15.45, 25.32,\n" \
"LINE_TO, 23.78, 17,\n" \
"LINE_TO, 4.49, 17,\n" \
"CUBIC_TO, 3.67, 17, 3, 16.33, 3, 15.5,\n" \
"CUBIC_TO, 3, 14.67, 3.67, 14, 4.49, 14,\n" \
"LINE_TO, 23.8, 14,\n" \
"LINE_TO, 15.45, 5.66,\n" \
"CUBIC_TO, 15.19, 5.34, 15, 4.94, 15, 4.5,\n" \
"CUBIC_TO, 15, 3.68, 15.67, 3, 16.5, 3,\n" \
"CUBIC_TO, 16.94, 3, 17.34, 3.2, 17.61, 3.5,\n" \
"LINE_TO, 17.64, 3.47,\n" \
"LINE_TO, 28, 13.81,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 7, 13.51,\n" \
"LINE_TO, 11.51, 9,\n" \
"LINE_TO, 2, 9,\n" \
"CUBIC_TO, 1.45, 9, 1, 8.55, 1, 8,\n" \
"CUBIC_TO, 1, 7.45, 1.45, 7, 2, 7,\n" \
"LINE_TO, 11.51, 7,\n" \
"LINE_TO, 7, 2.49,\n" \
"CUBIC_TO, 7.03, 2.28, 7, 2.14, 7, 2,\n" \
"CUBIC_TO, 7, 1.45, 7.45, 1, 8, 1,\n" \
"CUBIC_TO, 8.14, 1, 8.28, 1.03, 8.41, 1.09,\n" \
"LINE_TO, 8.49, 1,\n" \
"LINE_TO, 15, 7.51,\n" \
"LINE_TO, 14.91, 7.59,\n" \
"CUBIC_TO, 14.97, 7.72, 15, 7.85, 15, 8,\n" \
"CUBIC_TO, 15, 8.15, 14.97, 8.28, 14.91, 8.41,\n" \
"LINE_TO, 15, 8.49,\n" \
"LINE_TO, 8.49, 15,\n" \
"CUBIC_TO, 8.28, 14.97, 8.14, 15, 8, 15,\n" \
"CUBIC_TO, 7.45, 15, 7, 14.55, 7, 14,\n" \
"CUBIC_TO, 7, 13.86, 7.03, 13.72, 7.09, 13.59,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kForwardArrowIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseForwardArrowChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 13.13, 10.75,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 9.13,\n" \
"LINE_TO, 8.94, 5.06,\n" \
"LINE_TO, 10, 4,\n" \
"R_LINE_TO, 6, 6,\n" \
"R_LINE_TO, -6, 6,\n" \
"R_LINE_TO, -1.06, -1.06,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kForwardArrowChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseGlobeIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 8, 14.54,\n" \
"R_CUBIC_TO, -0.9, 0, -1.75, -0.17, -2.54, -0.51,\n" \
"R_ARC_TO, 6.52, 6.52, 0, 0, 1, -2.08, -1.4,\n" \
"R_ARC_TO, 6.52, 6.52, 0, 0, 1, -1.4, -2.08,\n" \
"R_ARC_TO, 6.33, 6.33, 0, 0, 1, -0.51, -2.55,\n" \
"R_CUBIC_TO, 0, -0.91, 0.17, -1.75, 0.51, -2.54,\n" \
"ARC_TO, 6.53, 6.53, 0, 0, 1, 3.38, 3.38,\n" \
"ARC_TO, 6.52, 6.52, 0, 0, 1, 5.46, 1.98,\n" \
"R_ARC_TO, 6.33, 6.33, 0, 0, 1, 2.55, -0.51,\n" \
"R_CUBIC_TO, 0.91, 0, 1.75, 0.17, 2.54, 0.51,\n" \
"R_CUBIC_TO, 0.79, 0.34, 1.48, 0.8, 2.07, 1.4,\n" \
"R_ARC_TO, 6.5, 6.5, 0, 0, 1, 1.4, 2.08,\n" \
"R_ARC_TO, 6.29, 6.29, 0, 0, 1, 0.51, 2.54,\n" \
"R_CUBIC_TO, 0, 0.9, -0.17, 1.75, -0.51, 2.54,\n" \
"R_ARC_TO, 6.52, 6.52, 0, 0, 1, -1.4, 2.08,\n" \
"R_ARC_TO, 6.5, 6.5, 0, 0, 1, -2.08, 1.4,\n" \
"R_ARC_TO, 6.29, 6.29, 0, 0, 1, -2.54, 0.51,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 13.15,\n" \
"R_CUBIC_TO, 1.3, 0, 2.42, -0.41, 3.37, -1.25,\n" \
"R_ARC_TO, 5.06, 5.06, 0, 0, 0, 1.72, -3.1,\n" \
"R_CUBIC_TO, 0, 0.02, 0, 0.04, 0, 0.06,\n" \
"R_V_LINE_TO, 0.05,\n" \
"R_ARC_TO, 1.72, 1.72, 0, 0, 1, -0.34, 0.09,\n" \
"R_CUBIC_TO, -0.11, 0.02, -0.23, 0.03, -0.36, 0.03,\n" \
"R_H_LINE_TO, -2.37,\n" \
"R_ARC_TO, 1.2, 1.2, 0, 0, 1, -0.88, -0.37,\n" \
"R_ARC_TO, 1.21, 1.21, 0, 0, 1, -0.36, -0.88,\n" \
"R_V_LINE_TO, -0.82,\n" \
"H_LINE_TO, 5.87,\n" \
"V_LINE_TO, 5.34,\n" \
"R_CUBIC_TO, 0, -0.34, 0.13, -0.64, 0.37, -0.88,\n" \
"R_CUBIC_TO, 0.25, -0.25, 0.54, -0.37, 0.88, -0.37,\n" \
"R_H_LINE_TO, 0.82,\n" \
"R_V_LINE_TO, -0.4,\n" \
"R_CUBIC_TO, 0, -0.17, 0.03, -0.33, 0.09, -0.48,\n" \
"R_ARC_TO, 0.89, 0.89, 0, 0, 1, 0.24, -0.35,\n" \
"R_ARC_TO, 0.76, 0.76, 0, 0, 0, -0.16, -0.01,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, -1.43, 0, -2.65, 0.5, -3.65, 1.5,\n" \
"CUBIC_TO_SHORTHAND, 2.85, 6.57, 2.85, 8,\n" \
"R_CUBIC_TO, 0, 0.01, 0, 0.03, 0, 0.04,\n" \
"R_V_LINE_TO, 0.04,\n" \
"R_H_LINE_TO, 3.36,\n" \
"R_CUBIC_TO, 0.7, 0, 1.29, 0.24, 1.77, 0.73,\n" \
"R_CUBIC_TO, 0.48, 0.48, 0.73, 1.07, 0.73, 1.77,\n" \
"R_V_LINE_TO, 0.82,\n" \
"H_LINE_TO, 6.23,\n" \
"R_V_LINE_TO, 1.45,\n" \
"R_CUBIC_TO, 0.27, 0.11, 0.56, 0.18, 0.85, 0.23,\n" \
"R_CUBIC_TO, 0.29, 0.05, 0.6, 0.07, 0.91, 0.07,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kGlobeIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseGoogleColorIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"PATH_COLOR_ARGB, 0xFF, 0x42, 0x85, 0xF4,\n" \
"MOVE_TO, 43.5f, 20,\n" \
"H_LINE_TO, 24,\n" \
"R_V_LINE_TO, 8,\n" \
"R_H_LINE_TO, 11.6f,\n" \
"R_CUBIC_TO, -0.5f, 2.5f, -2.2f, 4.7f, -4.3f, 6.1f,\n" \
"R_LINE_TO, 6.6f, 5,\n" \
"R_CUBIC_TO, 3.9f, -3.5f, 6.1f, -8.6f, 6.1f, -14.7f,\n" \
"CUBIC_TO, 44, 23, 43.7f, 21.3f, 43.5f, 20,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"PATH_COLOR_ARGB, 0xFF, 0x0F, 0x9D, 0x58,\n" \
"MOVE_TO, 24.4f, 36.1f,\n" \
"R_CUBIC_TO, -5.3f, 0, -9.8f, -3.5f, -11.4f, -8.2f,\n" \
"R_H_LINE_TO, 0,\n" \
"R_LINE_TO, -1.5f, 1.1f,\n" \
"R_LINE_TO, -5.3f, 4,\n" \
"R_V_LINE_TO, 0,\n" \
"R_CUBIC_TO, 3.4f, 6.5f, 10.2f, 11, 18.2f, 11,\n" \
"R_CUBIC_TO, 5.5f, 0, 10.1f, -1.8f, 13.5f, -4.8f,\n" \
"R_LINE_TO, -6.6f, -5,\n" \
"CUBIC_TO, 29.5f, 35.3f, 27.2f, 36.1f, 24.4f, 36.1f,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"PATH_COLOR_ARGB, 0xFF, 0xFF, 0xCD, 0x40,\n" \
"MOVE_TO, 13, 27.8f,\n" \
"LINE_TO, 13, 27.8f,\n" \
"R_CUBIC_TO, -0.4f, -1.2f, -0.6f, -2.5f, -0.6f, -3.8f,\n" \
"R_CUBIC_TO, 0, -1.3f, 0.2f, -2.6f, 0.6f, -3.8f,\n" \
"LINE_TO, 6.2f, 15,\n" \
"CUBIC_TO, 4.8f, 17.7f, 4, 20.8f, 4, 24,\n" \
"R_CUBIC_TO, 0, 3.2f, 0.8f, 6.3f, 2.2f, 9,\n" \
"R_LINE_TO, 5.3f, -4,\n" \
"LINE_TO, 13, 27.8f,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"PATH_COLOR_ARGB, 0xFF, 0xDB, 0x44, 0x37,\n" \
"MOVE_TO, 24.4f, 4,\n" \
"R_CUBIC_TO, -8, 0, -14.9f, 4.5f, -18.2f, 11,\n" \
"R_LINE_TO, 6.8f, 5.2f,\n" \
"R_CUBIC_TO, 1.6f, -4.7f, 6.1f, -8.2f, 11.4f, -8.2f,\n" \
"R_CUBIC_TO, 3, 0, 5.7f, 1, 7.8f, 3,\n" \
"R_LINE_TO, 5.8f, -5.7f,\n" \
"CUBIC_TO, 34.5f, 6, 29.9f, 4, 24.4f, 4,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"PATH_COLOR_ARGB, 0xFF, 0x42, 0x85, 0xF4,\n" \
"MOVE_TO, 21.73f, 10,\n" \
"LINE_TO, 12, 10,\n" \
"LINE_TO, 12, 14,\n" \
"LINE_TO, 17.82f, 14,\n" \
"CUBIC_TO, 17.58f, 15.25f, 16.73f, 16.35f, 15.65f, 17.06f,\n" \
"LINE_TO, 15.65f, 19.58f,\n" \
"LINE_TO, 18.96f, 19.58f,\n" \
"CUBIC_TO, 20.89f, 17.84f, 22, 15.27f, 22, 12.23f,\n" \
"CUBIC_TO, 22, 11.52f, 21.85f, 10.65f, 21.73f, 10,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"PATH_COLOR_ARGB, 0xFF, 0x0F, 0x9D, 0x58,\n" \
"MOVE_TO, 12.2f, 22,\n" \
"CUBIC_TO, 14.96f, 22, 17.27f, 21.11f, 18.96f, 19.58f,\n" \
"LINE_TO, 15.65f, 17.06f,\n" \
"CUBIC_TO, 14.74f, 17.66f, 13.58f, 18.03f, 12.2f, 18.03f,\n" \
"CUBIC_TO, 9.55f, 18.03f, 7.3f, 16.27f, 6.49f, 13.91f,\n" \
"LINE_TO, 3.09f, 13.91f,\n" \
"LINE_TO, 3.09f, 16.49f,\n" \
"CUBIC_TO, 4.77f, 19.75f, 8.22f, 22, 12.2f, 22,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"PATH_COLOR_ARGB, 0xFF, 0xFF, 0xCD, 0x40,\n" \
"MOVE_TO, 6.49f, 13.9f,\n" \
"CUBIC_TO, 6.29f, 13.3f, 6.17f, 12.66f, 6.17f, 12,\n" \
"CUBIC_TO, 6.17f, 11.34f, 6.29f, 10.7f, 6.49f, 10.1f,\n" \
"LINE_TO, 6.49f, 7.52f,\n" \
"LINE_TO, 3.09f, 7.52f,\n" \
"CUBIC_TO, 2.4f, 8.86f, 2, 10.38f, 2, 12,\n" \
"CUBIC_TO, 2, 13.62f, 2.4f, 15.14f, 3.09f, 16.48f,\n" \
"LINE_TO, 5.74f, 14.46f,\n" \
"LINE_TO, 6.49f, 13.9f,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"PATH_COLOR_ARGB, 0xFF, 0xDB, 0x44, 0x37,\n" \
"MOVE_TO, 12.2f, 5.98f,\n" \
"CUBIC_TO, 13.71f, 5.98f, 15.04f, 6.49f, 16.11f, 7.47f,\n" \
"LINE_TO, 19.03f, 4.61f,\n" \
"CUBIC_TO, 17.26f, 2.99f, 14.96f, 2, 12.2f, 2,\n" \
"CUBIC_TO, 8.22f, 2, 4.77f, 4.25f, 3.09f, 7.52f,\n" \
"LINE_TO, 6.49f, 10.1f,\n" \
"CUBIC_TO, 7.3f, 7.74f, 9.55f, 5.98f, 12.2f, 5.98f,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kGoogleColorIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseGppMaybeIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 12, 4.24,\n" \
"R_LINE_TO, 6, 3,\n" \
"R_V_LINE_TO, 4.1,\n" \
"R_CUBIC_TO, 0, 3.9, -2.55, 7.5, -6, 8.59,\n" \
"R_CUBIC_TO, -3.45, -1.09, -6, -4.7, -6, -8.59,\n" \
"R_V_LINE_TO, -4.1,\n" \
"R_LINE_TO, 6, -3,\n" \
"MOVE_TO, 12, 2,\n" \
"LINE_TO, 4, 6,\n" \
"R_V_LINE_TO, 5.33,\n" \
"R_CUBIC_TO, 0, 4.93, 3.41, 9.55, 8, 10.67,\n" \
"R_CUBIC_TO, 4.59, -1.12, 8, -5.73, 8, -10.67,\n" \
"V_LINE_TO, 6,\n" \
"R_LINE_TO, -8, -4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1, 13,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2, -7,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 2,\n" \
"V_LINE_TO, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kGppMaybeIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseHandGestureIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 22, 5.02,\n" \
"R_CUBIC_TO, 0, -0.85, -0.29, -1.57, -0.87, -2.15,\n" \
"R_CUBIC_TO, -0.58, -0.58, -1.3, -0.87, -2.15, -0.87,\n" \
"V_LINE_TO, 0.5,\n" \
"R_CUBIC_TO, 1.25, 0, 2.32, 0.44, 3.2, 1.32,\n" \
"R_CUBIC_TO, 0.88, 0.88, 1.32, 1.95, 1.32, 3.2,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 23,\n" \
"R_CUBIC_TO, -1.38, 0, -2.56, -0.49, -3.54, -1.46,\n" \
"CUBIC_TO, 1.49, 20.56, 1, 19.38, 1, 18,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0, 0.97, 0.34, 1.79, 1.02, 2.48,\n" \
"ARC_TO, 3.38, 3.38, 0, 0, 0, 6, 21.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.05, 0,\n" \
"R_CUBIC_TO, -0.5, 0, -0.97, -0.11, -1.4, -0.34,\n" \
"R_ARC_TO, 2.92, 2.92, 0, 0, 1, -1.07, -0.96,\n" \
"LINE_TO, 1.2, 12.38,\n" \
"R_LINE_TO, 0.6, -0.57,\n" \
"R_CUBIC_TO, 0.32, -0.32, 0.69, -0.5, 1.13, -0.55,\n" \
"R_CUBIC_TO, 0.43, -0.05, 0.82, 0.05, 1.18, 0.3,\n" \
"LINE_TO, 7, 13.58,\n" \
"V_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 8, 3,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_V_LINE_TO, 13.43,\n" \
"R_LINE_TO, -3.7, -2.6,\n" \
"R_LINE_TO, 3.93, 5.73,\n" \
"R_ARC_TO, 0.99, 0.99, 0, 0, 0, 0.82, 0.45,\n" \
"LINE_TO, 17, 21,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, -0.19, 1.41, -0.59,\n" \
"R_CUBIC_TO, 0.39, -0.39, 0.59, -0.86, 0.59, -1.41,\n" \
"V_LINE_TO, 5,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 20, 4,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_V_LINE_TO, 14,\n" \
"R_CUBIC_TO, 0, 1.1, -0.39, 2.04, -1.18, 2.82,\n" \
"CUBIC_TO, 19.04, 22.61, 18.1, 23, 17, 23,\n" \
"CLOSE,\n" \
"MOVE_TO, 11, 12,\n" \
"V_LINE_TO, 2,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 12, 1,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_V_LINE_TO, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, 0,\n" \
"V_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 16, 2,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_V_LINE_TO, 9,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.85, 4.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kHandGestureIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseHandGestureOffIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 21, 18.15,\n" \
"R_LINE_TO, -2, -2,\n" \
"V_LINE_TO, 5,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 20, 4,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -12, -12,\n" \
"R_LINE_TO, -2, -2,\n" \
"V_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 8, 3,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, 4,\n" \
"R_LINE_TO, -2, -2,\n" \
"V_LINE_TO, 2,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 12, 1,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, 1.88,\n" \
"R_H_LINE_TO, -2,\n" \
"V_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 16, 2,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.93, 8.73,\n" \
"LINE_TO, 9, 11.82,\n" \
"R_V_LINE_TO, 6.3,\n" \
"R_LINE_TO, -3.95, -1.98,\n" \
"R_LINE_TO, 4.43, 4.4,\n" \
"R_CUBIC_TO, 0.13, 0.12, 0.27, 0.22, 0.4, 0.31,\n" \
"R_CUBIC_TO, 0.13, 0.09, 0.28, 0.14, 0.45, 0.14,\n" \
"H_LINE_TO, 17,\n" \
"R_CUBIC_TO, 0.17, 0, 0.33, -0.02, 0.49, -0.06,\n" \
"R_CUBIC_TO, 0.16, -0.04, 0.31, -0.1, 0.44, -0.19,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.32, 23,\n" \
"R_CUBIC_TO, -0.4, 0, -0.78, -0.07, -1.15, -0.23,\n" \
"R_ARC_TO, 2.91, 2.91, 0, 0, 1, -0.98, -0.65,\n" \
"LINE_TO, 0.98, 14.9,\n" \
"R_LINE_TO, 1.35, -1.07,\n" \
"R_CUBIC_TO, 0.3, -0.23, 0.64, -0.37, 1.03, -0.41,\n" \
"R_ARC_TO, 1.86, 1.86, 0, 0, 1, 1.1, 0.21,\n" \
"LINE_TO, 7, 14.88,\n" \
"R_V_LINE_TO, -5.05,\n" \
"LINE_TO, 0.68, 3.5,\n" \
"R_LINE_TO, 1.43, -1.43,\n" \
"R_LINE_TO, 19.8, 19.8,\n" \
"R_LINE_TO, -1.42, 1.43,\n" \
"R_LINE_TO, -1.1, -1.1,\n" \
"R_CUBIC_TO, -0.33, 0.25, -0.7, 0.45, -1.1, 0.59,\n" \
"R_CUBIC_TO, -0.4, 0.14, -0.82, 0.21, -1.27, 0.21,\n" \
"CLOSE,\n" \
"MOVE_TO, 15, 12.02,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7, -7,\n" \
"R_CUBIC_TO, 0, -0.85, -0.29, -1.57, -0.87, -2.15,\n" \
"R_CUBIC_TO, -0.58, -0.58, -1.3, -0.87, -2.15, -0.87,\n" \
"V_LINE_TO, 0.5,\n" \
"R_CUBIC_TO, 1.25, 0, 2.32, 0.44, 3.2, 1.32,\n" \
"R_CUBIC_TO, 0.88, 0.88, 1.32, 1.95, 1.32, 3.2,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 23,\n" \
"R_CUBIC_TO, -1.38, 0, -2.56, -0.49, -3.54, -1.46,\n" \
"CUBIC_TO, 1.49, 20.56, 1, 19.38, 1, 18,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0, 0.97, 0.34, 1.79, 1.02, 2.48,\n" \
"ARC_TO, 3.38, 3.38, 0, 0, 0, 6, 21.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7.6, -6.6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kHandGestureOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseHeadsetIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 16, 15,\n" \
"R_CUBIC_TO, 0, 0.55, -0.45, 1, -1, 1,\n" \
"R_H_LINE_TO, -1,\n" \
"R_V_LINE_TO, -3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 16,\n" \
"H_LINE_TO, 5,\n" \
"R_CUBIC_TO, -0.55, 0, -1, -0.45, -1, -1,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 3,\n" \
"CLOSE,\n" \
"MOVE_TO, 2, 9,\n" \
"R_V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, 1.66, 1.34, 3, 3, 3,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, -7,\n" \
"H_LINE_TO, 4,\n" \
"V_LINE_TO, 9,\n" \
"R_CUBIC_TO, 0, -3.87, 2.13, -5, 6, -5,\n" \
"R_CUBIC_TO, 3.87, 0, 6, 1.13, 6, 5,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, 7,\n" \
"R_H_LINE_TO, 3,\n" \
"R_CUBIC_TO, 1.66, 0, 3, -1.34, 3, -3,\n" \
"V_LINE_TO, 9,\n" \
"R_CUBIC_TO, 0, -4.97, -3.03, -7, -8, -7,\n" \
"R_CUBIC_TO, -4.97, 0, -8, 2.03, -8, 7,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kHeadsetIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseHelpIcon) { \
              std::string s = \
"// Copyright 2020 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 12, 2,\n" \
"CUBIC_TO, 6.48, 2, 2, 6.48, 2, 12,\n" \
"R_CUBIC_TO, 0, 5.52, 4.48, 10, 10, 10,\n" \
"R_CUBIC_TO, 5.52, 0, 10, -4.48, 10, -10,\n" \
"CUBIC_TO_SHORTHAND, 17.52, 2, 12, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1, 17,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.07, -7.75,\n" \
"R_LINE_TO, -0.9, 0.92,\n" \
"CUBIC_TO, 13.45, 12.9, 13, 13.5, 13, 15,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, -0.5,\n" \
"R_CUBIC_TO, 0, -1.1, 0.45, -2.1, 1.17, -2.83,\n" \
"R_LINE_TO, 1.24, -1.26,\n" \
"R_CUBIC_TO, 0.37, -0.36, 0.59, -0.86, 0.59, -1.41,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, -2.21, 1.79, -4, 4, -4,\n" \
"R_CUBIC_TO, 2.21, 0, 4, 1.79, 4, 4,\n" \
"R_CUBIC_TO, 0, 0.88, -0.36, 1.68, -0.93, 2.25,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kHelpIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseHelpOutlineIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 11, 18,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1, -16,\n" \
"CUBIC_TO, 6.48, 2, 2, 6.48, 2, 12,\n" \
"R_CUBIC_TO, 0, 5.52, 4.48, 10, 10, 10,\n" \
"R_CUBIC_TO, 5.52, 0, 10, -4.48, 10, -10,\n" \
"CUBIC_TO_SHORTHAND, 17.52, 2, 12, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 18,\n" \
"R_CUBIC_TO, -4.41, 0, -8, -3.59, -8, -8,\n" \
"R_CUBIC_TO, 0, -4.41, 3.59, -8, 8, -8,\n" \
"R_CUBIC_TO, 4.41, 0, 8, 3.59, 8, 8,\n" \
"R_CUBIC_TO, 0, 4.41, -3.59, 8, -8, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -14,\n" \
"R_CUBIC_TO, -2.21, 0, -4, 1.79, -4, 4,\n" \
"R_H_LINE_TO, 2,\n" \
"R_CUBIC_TO, 0, -1.1, 0.9, -2, 2, -2,\n" \
"R_CUBIC_TO, 1.1, 0, 2, 0.9, 2, 2,\n" \
"R_CUBIC_TO, 0, 2, -3, 1.75, -3, 5,\n" \
"R_H_LINE_TO, 2,\n" \
"R_CUBIC_TO, 0, -2.25, 3, -2.5, 3, -5,\n" \
"R_CUBIC_TO, 0, -2.21, -1.79, -4, -4, -4,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 10, 2,\n" \
"R_CUBIC_TO, 4.42, 0, 8, 3.58, 8, 8,\n" \
"R_CUBIC_TO, 0, 4.42, -3.58, 8, -8, 8,\n" \
"R_CUBIC_TO, -4.42, 0, -8, -3.58, -8, -8,\n" \
"R_CUBIC_TO, 0, -4.42, 3.58, -8, 8, -8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 14,\n" \
"R_CUBIC_TO, 3.31, 0, 6, -2.69, 6, -6,\n" \
"R_CUBIC_TO, 0, -3.31, -2.69, -6, -6, -6,\n" \
"R_CUBIC_TO, -3.31, 0, -6, 2.69, -6, 6,\n" \
"R_CUBIC_TO, 0, 3.31, 2.69, 6, 6, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.74, -1.4,\n" \
"R_V_LINE_TO, -1.39,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 1.39,\n" \
"H_LINE_TO, 9.26,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 5.6,\n" \
"R_CUBIC_TO, 1.66, 0, 3, 1.25, 3, 2.8,\n" \
"R_CUBIC_TO, 0, 1.75, -2.24, 1.88, -2.24, 3.46,\n" \
"H_LINE_TO, 9.26,\n" \
"CUBIC_TO, 9.26, 9.58, 11.5, 9.8, 11.5, 8.4,\n" \
"R_CUBIC_TO, 0, -0.77, -0.67, -1.4, -1.5, -1.4,\n" \
"R_CUBIC_TO, -0.83, 0, -1.5, 0.63, -1.5, 1.4,\n" \
"H_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, -1.55, 1.34, -2.8, 3, -2.8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kHelpOutlineIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseHistoryIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 10.81, 17.5,\n" \
"R_QUADRATIC_TO, -1.44, 0, -2.76, -0.53,\n" \
"R_QUADRATIC_TO, -1.32, -0.53, -2.41, -1.64,\n" \
"R_LINE_TO, 1.23, -1.23,\n" \
"R_QUADRATIC_TO, 0.81, 0.79, 1.87, 1.22,\n" \
"R_QUADRATIC_TO, 1.05, 0.43, 2.11, 0.43,\n" \
"R_QUADRATIC_TO, 2.4, 0, 4.06, -1.68,\n" \
"R_QUADRATIC_TO, 1.67, -1.68, 1.67, -4.07,\n" \
"R_QUADRATIC_TO, 0, -2.4, -1.67, -4.07,\n" \
"QUADRATIC_TO, 13.25, 4.25, 10.85, 4.25,\n" \
"R_QUADRATIC_TO, -2.37, 0, -4.04, 1.66,\n" \
"R_QUADRATIC_TO, -1.67, 1.66, -1.67, 4.28,\n" \
"R_LINE_TO, 1.54, -1.54,\n" \
"R_LINE_TO, 1.21, 1.21,\n" \
"LINE_TO, 4.25, 13.5,\n" \
"LINE_TO, 0.63, 9.88,\n" \
"R_LINE_TO, 1.23, -1.23,\n" \
"R_LINE_TO, 1.54, 1.54,\n" \
"R_QUADRATIC_TO, 0, -1.63, 0.58, -3.03,\n" \
"R_QUADRATIC_TO, 0.58, -1.41, 1.6, -2.44,\n" \
"R_QUADRATIC_TO, 1.02, -1.03, 2.38, -1.62,\n" \
"QUADRATIC_TO, 9.31, 2.5, 10.85, 2.5,\n" \
"R_QUADRATIC_TO, 1.54, 0, 2.9, 0.59,\n" \
"R_QUADRATIC_TO, 1.35, 0.59, 2.38, 1.63,\n" \
"R_QUADRATIC_TO, 1.02, 1.03, 1.61, 2.38,\n" \
"R_QUADRATIC_TO, 0.59, 1.35, 0.59, 2.9,\n" \
"R_QUADRATIC_TO, 0, 1.52, -0.6, 2.89,\n" \
"R_QUADRATIC_TO, -0.6, 1.36, -1.62, 2.4,\n" \
"R_QUADRATIC_TO, -1.02, 1.04, -2.39, 1.63,\n" \
"R_QUADRATIC_TO, -1.36, 0.59, -2.91, 0.59,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.31, -3.98,\n" \
"R_LINE_TO, -3.15, -3.19,\n" \
"R_V_LINE_TO, -4.5,\n" \
"R_H_LINE_TO, 1.75,\n" \
"R_V_LINE_TO, 3.81,\n" \
"R_LINE_TO, 2.63, 2.65,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kHistoryIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseHistoryChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 17,\n" \
"R_CUBIC_TO, -1.94, 0, -3.6, -0.68, -4.96, -2.04,\n" \
"CUBIC_TO, 3.68, 13.6, 3, 11.95, 3, 10,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0, 1.52, 0.54, 2.81, 1.61, 3.89,\n" \
"CUBIC_TO, 7.19, 14.96, 8.48, 15.5, 10, 15.5,\n" \
"R_CUBIC_TO, 1.52, 0, 2.81, -0.54, 3.89, -1.61,\n" \
"CUBIC_TO, 14.96, 12.81, 15.5, 11.52, 15.5, 10,\n" \
"R_CUBIC_TO, 0, -1.52, -0.54, -2.81, -1.61, -3.89,\n" \
"CUBIC_TO, 12.81, 5.04, 11.52, 4.5, 10, 4.5,\n" \
"R_CUBIC_TO, -0.86, 0, -1.66, 0.18, -2.39, 0.54,\n" \
"ARC_TO, 5.33, 5.33, 0, 0, 0, 5.77, 6.5,\n" \
"H_LINE_TO, 8,\n" \
"V_LINE_TO, 8,\n" \
"H_LINE_TO, 3,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 2.71,\n" \
"ARC_TO, 6.94, 6.94, 0, 0, 1, 6.89, 3.73,\n" \
"CUBIC_TO, 7.84, 3.24, 8.88, 3, 10, 3,\n" \
"R_CUBIC_TO, 0.97, 0, 1.88, 0.18, 2.73, 0.56,\n" \
"R_CUBIC_TO, 0.85, 0.37, 1.59, 0.87, 2.22, 1.5,\n" \
"R_ARC_TO, 7.05, 7.05, 0, 0, 1, 1.5, 2.22,\n" \
"R_CUBIC_TO, 0.37, 0.85, 0.56, 1.76, 0.56, 2.73,\n" \
"R_CUBIC_TO, 0, 0.97, -0.18, 1.88, -0.56, 2.73,\n" \
"R_ARC_TO, 7.05, 7.05, 0, 0, 1, -1.5, 2.22,\n" \
"R_ARC_TO, 7.05, 7.05, 0, 0, 1, -2.22, 1.5,\n" \
"ARC_TO, 6.73, 6.73, 0, 0, 1, 10, 17,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.08, -4.17,\n" \
"LINE_TO, 9.25, 10,\n" \
"V_LINE_TO, 6,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 3.38,\n" \
"R_LINE_TO, 2.4, 2.4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kHistoryChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseHomeIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 5, 15.83f,\n" \
"LINE_TO, 7.5f, 15.83f,\n" \
"LINE_TO, 7.5f, 10.83f,\n" \
"LINE_TO, 12.5f, 10.83f,\n" \
"LINE_TO, 12.5f, 15.83f,\n" \
"LINE_TO, 15, 15.83f,\n" \
"LINE_TO, 15, 8.33f,\n" \
"LINE_TO, 10, 4.58f,\n" \
"LINE_TO, 5, 8.33f,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.33f, 17.5f,\n" \
"LINE_TO, 3.33f, 7.5f,\n" \
"LINE_TO, 10, 2.5f,\n" \
"LINE_TO, 16.67f, 7.5f,\n" \
"LINE_TO, 16.67f, 17.5f,\n" \
"LINE_TO, 10.83f, 17.5f,\n" \
"LINE_TO, 10.83f, 12.5f,\n" \
"LINE_TO, 9.17f, 12.5f,\n" \
"LINE_TO, 9.17f, 17.5f,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 10.21f,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 10.21f,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kHomeIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseHttpsValidIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 6.07, 22.18,\n" \
"R_ARC_TO, 2.19, 2.19, 0, 0, 1, -1.6, -0.67,\n" \
"R_ARC_TO, 2.19, 2.19, 0, 0, 1, -0.67, -1.61,\n" \
"R_V_LINE_TO, -9.85,\n" \
"R_CUBIC_TO, 0, -0.62, 0.22, -1.16, 0.67, -1.61,\n" \
"R_ARC_TO, 2.19, 2.19, 0, 0, 1, 1.61, -0.67,\n" \
"R_H_LINE_TO, 0.84,\n" \
"V_LINE_TO, 5.95,\n" \
"R_CUBIC_TO, 0, -1.42, 0.5, -2.62, 1.49, -3.62,\n" \
"CUBIC_TO, 9.39, 1.34, 10.59, 0.84, 12, 0.84,\n" \
"R_CUBIC_TO, 1.41, 0, 2.61, 0.5, 3.6, 1.49,\n" \
"R_CUBIC_TO, 0.99, 1, 1.49, 2.2, 1.49, 3.62,\n" \
"R_V_LINE_TO, 1.82,\n" \
"R_H_LINE_TO, 0.84,\n" \
"R_CUBIC_TO, 0.63, 0, 1.16, 0.22, 1.61, 0.67,\n" \
"R_CUBIC_TO, 0.45, 0.45, 0.67, 0.99, 0.67, 1.61,\n" \
"R_V_LINE_TO, 9.86,\n" \
"R_ARC_TO, 2.19, 2.19, 0, 0, 1, -0.67, 1.61,\n" \
"R_ARC_TO, 2.19, 2.19, 0, 0, 1, -1.6, 0.67,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.27,\n" \
"R_H_LINE_TO, 11.86,\n" \
"R_V_LINE_TO, -9.85,\n" \
"H_LINE_TO, 6.07,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 16.98,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, -0.19, 1.41, -0.58,\n" \
"R_CUBIC_TO, 0.39, -0.39, 0.59, -0.86, 0.59, -1.41,\n" \
"R_CUBIC_TO, 0, -0.55, -0.19, -1.02, -0.59, -1.41,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 0, 12, 12.98,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, 0.2, -1.41, 0.59,\n" \
"R_CUBIC_TO, -0.39, 0.39, -0.59, 0.86, -0.59, 1.41,\n" \
"R_CUBIC_TO, 0, 0.55, 0.2, 1.02, 0.59, 1.42,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.86, 0.59, 1.41, 0.59,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.18, 7.78,\n" \
"R_H_LINE_TO, 5.63,\n" \
"V_LINE_TO, 5.95,\n" \
"R_CUBIC_TO, 0, -0.79, -0.27, -1.46, -0.82, -2.01,\n" \
"ARC_TO, 2.7, 2.7, 0, 0, 0, 12, 3.12,\n" \
"R_CUBIC_TO, -0.78, 0, -1.45, 0.27, -2, 0.82,\n" \
"R_ARC_TO, 2.74, 2.74, 0, 0, 0, -0.82, 2.01,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.07, 19.91,\n" \
"R_V_LINE_TO, -9.85,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 5.56, 18.16,\n" \
"R_CUBIC_TO, -0.48, 0, -0.88, -0.17, -1.22, -0.51,\n" \
"R_ARC_TO, 1.68, 1.68, 0, 0, 1, -0.5, -1.22,\n" \
"V_LINE_TO, 8.54,\n" \
"R_CUBIC_TO, 0, -0.47, 0.17, -0.88, 0.51, -1.22,\n" \
"R_CUBIC_TO, 0.34, -0.34, 0.75, -0.51, 1.22, -0.51,\n" \
"R_H_LINE_TO, 0.37,\n" \
"V_LINE_TO, 4.96,\n" \
"R_CUBIC_TO, 0, -1.13, 0.4, -2.1, 1.19, -2.89,\n" \
"CUBIC_TO, 7.91, 1.27, 8.88, 0.87, 10, 0.87,\n" \
"R_CUBIC_TO, 1.13, 0, 2.09, 0.4, 2.88, 1.2,\n" \
"R_CUBIC_TO, 0.79, 0.8, 1.19, 1.76, 1.19, 2.9,\n" \
"R_V_LINE_TO, 1.86,\n" \
"R_H_LINE_TO, 0.37,\n" \
"R_CUBIC_TO, 0.47, 0, 0.88, 0.17, 1.22, 0.51,\n" \
"R_CUBIC_TO, 0.34, 0.34, 0.51, 0.75, 0.51, 1.22,\n" \
"R_V_LINE_TO, 7.88,\n" \
"R_CUBIC_TO, 0, 0.48, -0.17, 0.88, -0.51, 1.22,\n" \
"R_ARC_TO, 1.67, 1.67, 0, 0, 1, -1.22, 0.51,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.73,\n" \
"R_H_LINE_TO, 8.88,\n" \
"V_LINE_TO, 8.54,\n" \
"H_LINE_TO, 5.56,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.45, -2.44,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, -0.15, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.29, 0.44, -0.65, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.41, -0.15, -0.76, -0.44, -1.06,\n" \
"R_ARC_TO, 1.47, 1.47, 0, 0, 0, -1.06, -0.44,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, 0.15, -1.06, 0.44,\n" \
"R_ARC_TO, 1.45, 1.45, 0, 0, 0, -0.44, 1.06,\n" \
"R_CUBIC_TO, 0, 0.41, 0.15, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.3, 0.29, 0.65, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.66, 6.82,\n" \
"R_H_LINE_TO, 4.69,\n" \
"V_LINE_TO, 4.96,\n" \
"R_CUBIC_TO, 0, -0.66, -0.23, -1.22, -0.68, -1.67,\n" \
"R_CUBIC_TO, -0.45, -0.46, -1.01, -0.69, -1.66, -0.69,\n" \
"R_CUBIC_TO, -0.66, 0, -1.21, 0.23, -1.66, 0.69,\n" \
"R_ARC_TO, 2.29, 2.29, 0, 0, 0, -0.68, 1.67,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.1, 9.61,\n" \
"V_LINE_TO, 8.54,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 4.51, 14.69,\n" \
"R_CUBIC_TO, -0.45, 0, -0.83, -0.16, -1.15, -0.48,\n" \
"R_ARC_TO, 1.58, 1.58, 0, 0, 1, -0.48, -1.15,\n" \
"V_LINE_TO, 6.89,\n" \
"R_CUBIC_TO, 0, -0.45, 0.16, -0.84, 0.48, -1.16,\n" \
"R_CUBIC_TO, 0.32, -0.32, 0.7, -0.48, 1.15, -0.48,\n" \
"R_H_LINE_TO, 0.15,\n" \
"V_LINE_TO, 3.93,\n" \
"R_CUBIC_TO, 0, -0.94, 0.32, -1.73, 0.97, -2.39,\n" \
"R_ARC_TO, 3.22, 3.22, 0, 0, 1, 2.37, -0.98,\n" \
"R_CUBIC_TO, 0.93, 0, 1.72, 0.33, 2.37, 0.98,\n" \
"R_CUBIC_TO, 0.65, 0.66, 0.97, 1.45, 0.97, 2.39,\n" \
"V_LINE_TO, 5.25,\n" \
"R_H_LINE_TO, 0.15,\n" \
"R_CUBIC_TO, 0.45, 0, 0.83, 0.16, 1.15, 0.48,\n" \
"R_CUBIC_TO, 0.32, 0.32, 0.48, 0.71, 0.48, 1.16,\n" \
"R_V_LINE_TO, 6.17,\n" \
"R_CUBIC_TO, 0, 0.45, -0.16, 0.83, -0.48, 1.15,\n" \
"R_CUBIC_TO, -0.32, 0.32, -0.7, 0.48, -1.16, 0.48,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.63,\n" \
"R_H_LINE_TO, 6.98,\n" \
"V_LINE_TO, 6.89,\n" \
"H_LINE_TO, 4.51,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.49, -1.89,\n" \
"R_CUBIC_TO, 0.33, 0, 0.61, -0.12, 0.85, -0.36,\n" \
"R_CUBIC_TO, 0.23, -0.23, 0.35, -0.51, 0.35, -0.85,\n" \
"R_CUBIC_TO, 0, -0.33, -0.12, -0.61, -0.35, -0.85,\n" \
"R_ARC_TO, 1.17, 1.17, 0, 0, 0, -0.85, -0.35,\n" \
"R_CUBIC_TO, -0.33, 0, -0.61, 0.12, -0.85, 0.35,\n" \
"R_CUBIC_TO, -0.23, 0.23, -0.35, 0.52, -0.35, 0.85,\n" \
"R_CUBIC_TO, 0, 0.33, 0.12, 0.61, 0.35, 0.84,\n" \
"R_CUBIC_TO, 0.24, 0.24, 0.52, 0.35, 0.85, 0.35,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.29, 5.25,\n" \
"R_H_LINE_TO, 3.41,\n" \
"V_LINE_TO, 3.93,\n" \
"R_CUBIC_TO, 0, -0.48, -0.16, -0.89, -0.49, -1.23,\n" \
"ARC_TO, 1.62, 1.62, 0, 0, 0, 8, 2.19,\n" \
"R_CUBIC_TO, -0.48, 0, -0.89, 0.17, -1.22, 0.51,\n" \
"R_ARC_TO, 1.7, 1.7, 0, 0, 0, -0.49, 1.23,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.78, 7.81,\n" \
"V_LINE_TO, 6.89,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kHttpsValidIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseIdCardIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 14, 13,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, -5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"R_H_LINE_TO, 5,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, -5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -9, 6,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, -0.55,\n" \
"R_CUBIC_TO, 0, -0.75, -0.37, -1.34, -1.1, -1.79,\n" \
"CUBIC_TO, 11.17, 13.22, 10.2, 13, 9, 13,\n" \
"R_CUBIC_TO, -1.2, 0, -2.17, 0.22, -2.9, 0.66,\n" \
"CUBIC_TO, 5.37, 14.11, 5, 14.7, 5, 15.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, -4,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, -0.19, 1.41, -0.59,\n" \
"R_CUBIC_TO, 0.39, -0.39, 0.59, -0.86, 0.59, -1.41,\n" \
"R_CUBIC_TO, 0, -0.55, -0.19, -1.02, -0.59, -1.41,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 0, 9, 8,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, 0.2, -1.41, 0.59,\n" \
"CUBIC_TO, 7.2, 8.98, 7, 9.45, 7, 10,\n" \
"R_CUBIC_TO, 0, 0.55, 0.2, 1.02, 0.59, 1.41,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.86, 0.59, 1.41, 0.59,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5, 8,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, -0.19, -1.41, -0.59,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 2, 18,\n" \
"V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, -0.55, 0.2, -1.02, 0.59, -1.41,\n" \
"CUBIC_TO, 2.98, 4.2, 3.45, 4, 4, 4,\n" \
"R_H_LINE_TO, 16,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, 0.2, 1.41, 0.59,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.59, 0.86, 0.59, 1.41,\n" \
"R_V_LINE_TO, 12,\n" \
"R_CUBIC_TO, 0, 0.55, -0.19, 1.02, -0.59, 1.41,\n" \
"R_CUBIC_TO, -0.39, 0.39, -0.86, 0.59, -1.41, 0.59,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2,\n" \
"R_H_LINE_TO, 16,\n" \
"V_LINE_TO, 6,\n" \
"H_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kIdCardIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseIframeIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 10.09, 10.91,\n" \
"R_H_LINE_TO, 3.82,\n" \
"V_LINE_TO, 9.59,\n" \
"R_H_LINE_TO, -3.82,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.18, 1.18,\n" \
"V_LINE_TO, 8.41,\n" \
"R_H_LINE_TO, 6.18,\n" \
"R_V_LINE_TO, 3.68,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.56, 16.17,\n" \
"R_CUBIC_TO, -0.48, 0, -0.89, -0.17, -1.22, -0.5,\n" \
"R_ARC_TO, 1.67, 1.67, 0, 0, 1, -0.5, -1.23,\n" \
"V_LINE_TO, 5.56,\n" \
"R_CUBIC_TO, 0, -0.48, 0.17, -0.89, 0.5, -1.22,\n" \
"R_CUBIC_TO, 0.34, -0.34, 0.75, -0.5, 1.22, -0.5,\n" \
"H_LINE_TO, 16.44,\n" \
"R_CUBIC_TO, 0.48, 0, 0.89, 0.17, 1.22, 0.5,\n" \
"R_CUBIC_TO, 0.34, 0.34, 0.5, 0.75, 0.5, 1.23,\n" \
"R_V_LINE_TO, 8.88,\n" \
"R_CUBIC_TO, 0, 0.48, -0.17, 0.89, -0.5, 1.22,\n" \
"R_CUBIC_TO, -0.34, 0.34, -0.75, 0.5, -1.22, 0.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.73,\n" \
"H_LINE_TO, 16.44,\n" \
"V_LINE_TO, 7.17,\n" \
"H_LINE_TO, 3.56,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kIframeIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseIframeOffIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 16.81, 18.94,\n" \
"R_LINE_TO, -2.79, -2.77,\n" \
"H_LINE_TO, 3.56,\n" \
"R_CUBIC_TO, -0.48, 0, -0.89, -0.17, -1.22, -0.5,\n" \
"R_ARC_TO, 1.67, 1.67, 0, 0, 1, -0.5, -1.23,\n" \
"V_LINE_TO, 5.56,\n" \
"R_CUBIC_TO, 0, -0.45, 0.16, -0.84, 0.48, -1.17,\n" \
"R_CUBIC_TO, 0.32, -0.33, 0.71, -0.51, 1.18, -0.54,\n" \
"R_H_LINE_TO, 0.23,\n" \
"V_LINE_TO, 5.84,\n" \
"LINE_TO, 1.14, 3.27,\n" \
"R_LINE_TO, 1.11, -1.09,\n" \
"R_LINE_TO, 15.65, 15.64,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.56, 14.44,\n" \
"R_H_LINE_TO, 8.73,\n" \
"LINE_TO, 5.02, 7.17,\n" \
"H_LINE_TO, 3.56,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 14.23, 1.05,\n" \
"R_LINE_TO, -1.35, -1.35,\n" \
"V_LINE_TO, 7.17,\n" \
"H_LINE_TO, 9.47,\n" \
"LINE_TO, 6.14, 3.83,\n" \
"H_LINE_TO, 16.44,\n" \
"R_CUBIC_TO, 0.48, 0, 0.89, 0.17, 1.22, 0.5,\n" \
"R_CUBIC_TO, 0.34, 0.34, 0.5, 0.75, 0.5, 1.22,\n" \
"R_V_LINE_TO, 8.88,\n" \
"R_CUBIC_TO, 0, 0.2, -0.04, 0.4, -0.1, 0.59,\n" \
"R_CUBIC_TO, -0.07, 0.19, -0.16, 0.34, -0.27, 0.46,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.7, -2.7,\n" \
"R_LINE_TO, -1.18, -1.18,\n" \
"V_LINE_TO, 9.59,\n" \
"R_H_LINE_TO, -2.01,\n" \
"R_LINE_TO, -1.18, -1.18,\n" \
"R_H_LINE_TO, 4.38,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kIframeOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseImageSearchIcon) { \
              std::string s = \
"// Copyright 2022 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 3.6, 14.4,\n" \
"CUBIC_TO, 3.27, 14.4, 2.98, 14.28, 2.75, 14.05,\n" \
"CUBIC_TO, 2.52, 13.82, 2.4, 13.54, 2.4, 13.2,\n" \
"LINE_TO, 2.4, 4.4,\n" \
"CUBIC_TO, 2.4, 4.07, 2.52, 3.79, 2.75, 3.55,\n" \
"CUBIC_TO, 2.98, 3.32, 3.27, 3.2, 3.6, 3.2,\n" \
"LINE_TO, 6.8, 3.2,\n" \
"LINE_TO, 6.8, 4.4,\n" \
"LINE_TO, 3.6, 4.4,\n" \
"LINE_TO, 3.6, 13.2,\n" \
"LINE_TO, 12.4, 13.2,\n" \
"LINE_TO, 12.4, 9.35,\n" \
"LINE_TO, 13.6, 10.55,\n" \
"LINE_TO, 13.6, 13.2,\n" \
"CUBIC_TO, 13.6, 13.54, 13.48, 13.82, 13.25, 14.05,\n" \
"CUBIC_TO, 13.02, 14.28, 12.73, 14.4, 12.4, 14.4,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.4, 12,\n" \
"LINE_TO, 6.2, 9.6,\n" \
"LINE_TO, 7.4, 11.2,\n" \
"LINE_TO, 9.2, 8.8,\n" \
"LINE_TO, 11.6, 12,\n" \
"CLOSE,\n" \
"MOVE_TO, 14.35, 9.6,\n" \
"LINE_TO, 12.3, 7.55,\n" \
"CUBIC_TO, 12.08, 7.7, 11.84, 7.8, 11.59, 7.88,\n" \
"CUBIC_TO, 11.34, 7.96, 11.08, 8, 10.8, 8,\n" \
"CUBIC_TO, 10.02, 8, 9.36, 7.73, 8.82, 7.18,\n" \
"CUBIC_TO, 8.27, 6.64, 8, 5.98, 8, 5.2,\n" \
"CUBIC_TO, 8, 4.42, 8.27, 3.76, 8.82, 3.22,\n" \
"CUBIC_TO, 9.36, 2.67, 10.02, 2.4, 10.8, 2.4,\n" \
"CUBIC_TO, 11.58, 2.4, 12.24, 2.67, 12.78, 3.22,\n" \
"CUBIC_TO, 13.33, 3.76, 13.6, 4.42, 13.6, 5.2,\n" \
"CUBIC_TO, 13.6, 5.48, 13.56, 5.74, 13.48, 5.99,\n" \
"CUBIC_TO, 13.41, 6.24, 13.29, 6.48, 13.15, 6.7,\n" \
"LINE_TO, 15.2, 8.75,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.8, 6.8,\n" \
"CUBIC_TO, 11.25, 6.8, 11.62, 6.64, 11.93, 6.33,\n" \
"CUBIC_TO, 12.25, 6.02, 12.4, 5.64, 12.4, 5.2,\n" \
"CUBIC_TO, 12.4, 4.75, 12.25, 4.38, 11.93, 4.07,\n" \
"CUBIC_TO, 11.62, 3.75, 11.25, 3.6, 10.8, 3.6,\n" \
"CUBIC_TO, 10.36, 3.6, 9.98, 3.75, 9.67, 4.07,\n" \
"CUBIC_TO, 9.36, 4.38, 9.2, 4.75, 9.2, 5.2,\n" \
"CUBIC_TO, 9.2, 5.64, 9.36, 6.02, 9.67, 6.33,\n" \
"CUBIC_TO, 9.98, 6.64, 10.36, 6.8, 10.8, 6.8,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.8, 6.8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kImageSearchIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseInfoOutlineIcon) { \
              std::string s = \
"// Copyright 2016 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"MOVE_TO, 16, 2,\n" \
"CUBIC_TO, 8.27, 2, 2, 8.27, 2, 16,\n" \
"CUBIC_TO, 2, 23.73, 8.27, 30, 16, 30,\n" \
"CUBIC_TO, 23.73, 30, 30, 23.73, 30, 16,\n" \
"CUBIC_TO, 30, 8.27, 23.73, 2, 16, 2,\n" \
"CLOSE,\n" \
"MOVE_TO, 16, 27.2,\n" \
"CUBIC_TO, 9.83, 27.2, 4.8, 22.17, 4.8, 16,\n" \
"CUBIC_TO, 4.8, 9.83, 9.83, 4.8, 16, 4.8,\n" \
"CUBIC_TO, 22.17, 4.8, 27.2, 9.83, 27.2, 16,\n" \
"CUBIC_TO, 27.2, 22.17, 22.17, 27.2, 16, 27.2,\n" \
"CLOSE,\n" \
"MOVE_TO, 14.6, 23,\n" \
"LINE_TO, 17.4, 23,\n" \
"LINE_TO, 17.4, 14.6,\n" \
"LINE_TO, 14.6, 14.6,\n" \
"LINE_TO, 14.6, 23,\n" \
"CLOSE,\n" \
"MOVE_TO, 14.6, 11.8,\n" \
"LINE_TO, 17.4, 11.8,\n" \
"LINE_TO, 17.4, 9,\n" \
"LINE_TO, 14.6, 9,\n" \
"LINE_TO, 14.6, 11.8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kInfoOutlineIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseInfoRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 7.34, 11.26,\n" \
"R_H_LINE_TO, 1.32,\n" \
"V_LINE_TO, 7.2,\n" \
"H_LINE_TO, 7.34,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.66, -5.18,\n" \
"R_ARC_TO, 0.66, 0.66, 0, 0, 0, 0.48, -0.19,\n" \
"R_ARC_TO, 0.66, 0.66, 0, 0, 0, 0.2, -0.48,\n" \
"R_ARC_TO, 0.67, 0.67, 0, 0, 0, -0.2, -0.48,\n" \
"R_ARC_TO, 0.64, 0.64, 0, 0, 0, -0.48, -0.2,\n" \
"R_ARC_TO, 0.66, 0.66, 0, 0, 0, -0.48, 0.2,\n" \
"R_ARC_TO, 0.64, 0.64, 0, 0, 0, -0.2, 0.48,\n" \
"R_CUBIC_TO, 0, 0.2, 0.07, 0.36, 0.2, 0.49,\n" \
"R_ARC_TO, 0.64, 0.64, 0, 0, 0, 0.48, 0.2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.01, 8.46,\n" \
"R_CUBIC_TO, -0.9, 0, -1.75, -0.17, -2.54, -0.51,\n" \
"R_ARC_TO, 6.52, 6.52, 0, 0, 1, -2.08, -1.4,\n" \
"R_ARC_TO, 6.52, 6.52, 0, 0, 1, -1.4, -2.08,\n" \
"R_ARC_TO, 6.33, 6.33, 0, 0, 1, -0.51, -2.55,\n" \
"R_CUBIC_TO, 0, -0.91, 0.17, -1.75, 0.51, -2.54,\n" \
"ARC_TO, 6.53, 6.53, 0, 0, 1, 3.38, 3.38,\n" \
"ARC_TO, 6.52, 6.52, 0, 0, 1, 5.46, 1.98,\n" \
"R_ARC_TO, 6.33, 6.33, 0, 0, 1, 2.55, -0.51,\n" \
"R_CUBIC_TO, 0.91, 0, 1.75, 0.17, 2.54, 0.51,\n" \
"R_CUBIC_TO, 0.79, 0.34, 1.48, 0.8, 2.07, 1.4,\n" \
"R_ARC_TO, 6.5, 6.5, 0, 0, 1, 1.4, 2.08,\n" \
"R_ARC_TO, 6.29, 6.29, 0, 0, 1, 0.51, 2.54,\n" \
"R_CUBIC_TO, 0, 0.9, -0.17, 1.75, -0.51, 2.54,\n" \
"R_ARC_TO, 6.52, 6.52, 0, 0, 1, -1.4, 2.08,\n" \
"R_ARC_TO, 6.5, 6.5, 0, 0, 1, -2.08, 1.4,\n" \
"R_ARC_TO, 6.29, 6.29, 0, 0, 1, -2.54, 0.51,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 13.15,\n" \
"R_CUBIC_TO, 1.43, 0, 2.65, -0.5, 3.65, -1.5,\n" \
"R_CUBIC_TO, 1, -1, 1.5, -2.22, 1.5, -3.65,\n" \
"R_CUBIC_TO, 0, -1.43, -0.5, -2.65, -1.5, -3.65,\n" \
"CUBIC_TO_SHORTHAND, 9.43, 2.85, 8, 2.85,\n" \
"R_CUBIC_TO, -1.43, 0, -2.65, 0.5, -3.65, 1.5,\n" \
"CUBIC_TO_SHORTHAND, 2.85, 6.57, 2.85, 8,\n" \
"R_CUBIC_TO, 0, 1.43, 0.5, 2.65, 1.5, 3.65,\n" \
"R_CUBIC_TO, 1, 1, 2.22, 1.5, 3.65, 1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kInfoRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseInsertDriveFileOutlineIcon) { \
              std::string s = \
"// Copyright 2019 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 14, 2,\n" \
"H_LINE_TO, 6,\n" \
"R_CUBIC_TO, -1.1, 0, -1.99, 0.9, -1.99, 2,\n" \
"LINE_TO, 4, 20,\n" \
"R_CUBIC_TO, 0, 1.1, 0.89, 2, 1.99, 2,\n" \
"H_LINE_TO, 18,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 8,\n" \
"R_LINE_TO, -6, -6,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 20,\n" \
"V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 7,\n" \
"R_V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 11,\n" \
"H_LINE_TO, 6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kInsertDriveFileOutlineIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseInstallDesktopIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 6.66f, 17.53f,\n" \
"R_V_LINE_TO, -1.67f,\n" \
"H_LINE_TO, 3.32f,\n" \
"R_CUBIC_TO, -0.46f, 0, -0.85f, -0.17f, -1.18f, -0.49f,\n" \
"R_CUBIC_TO, -0.33f, -0.32f, -0.49f, -0.72f, -0.49f, -1.18f,\n" \
"V_LINE_TO, 4.17f,\n" \
"R_CUBIC_TO, 0, -0.46f, 0.17f, -0.85f, 0.49f, -1.18f,\n" \
"R_CUBIC_TO, 0.32f, -0.33f, 0.72f, -0.49f, 1.18f, -0.49f,\n" \
"H_LINE_TO, 10,\n" \
"R_V_LINE_TO, 1.67f,\n" \
"H_LINE_TO, 3.32f,\n" \
"R_V_LINE_TO, 10.02f,\n" \
"R_H_LINE_TO, 13.36f,\n" \
"R_V_LINE_TO, -2.51f,\n" \
"R_H_LINE_TO, 1.67f,\n" \
"R_V_LINE_TO, 2.51f,\n" \
"R_CUBIC_TO, 0, 0.46f, -0.17f, 0.85f, -0.49f, 1.18f,\n" \
"R_CUBIC_TO, -0.32f, 0.33f, -0.72f, 0.49f, -1.18f, 0.49f,\n" \
"R_H_LINE_TO, -3.34f,\n" \
"R_V_LINE_TO, 1.67f,\n" \
"H_LINE_TO, 6.66f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7.51f, -5.84f,\n" \
"LINE_TO, 9.99f, 7.51f,\n" \
"R_LINE_TO, 1.17f, -1.17f,\n" \
"R_LINE_TO, 2.17f, 2.15f,\n" \
"V_LINE_TO, 2.5f,\n" \
"H_LINE_TO, 15,\n" \
"R_V_LINE_TO, 6,\n" \
"R_LINE_TO, 2.17f, -2.15f,\n" \
"R_LINE_TO, 1.17f, 1.17f,\n" \
"R_LINE_TO, -4.18f, 4.18f,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kInstallDesktopIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseInstallDesktopOffIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 14.45f, 11.9f,\n" \
"R_LINE_TO, 4.05f, -4.05f,\n" \
"R_LINE_TO, -1.17f, -1.17f,\n" \
"R_LINE_TO, -2.16f, 2.15f,\n" \
"V_LINE_TO, 2.85f,\n" \
"H_LINE_TO, 13.5f,\n" \
"R_V_LINE_TO, 5.98f,\n" \
"R_LINE_TO, -2.17f, -2.15f,\n" \
"R_LINE_TO, -1.05f, 1.05f,\n" \
"R_LINE_TO, 4.17f, 4.17f,\n" \
"CLOSE,\n" \
"MOVE_TO, 16.83f, 14.28f,\n" \
"R_LINE_TO, 1.28f, 1.28f,\n" \
"R_CUBIC_TO, 0.25f, -0.3f, 0.38f, -0.65f, 0.38f, -1.05f,\n" \
"R_V_LINE_TO, -2.5f,\n" \
"R_H_LINE_TO, -1.67f,\n" \
"R_V_LINE_TO, 2.27f,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.07f, 4.51f,\n" \
"R_H_LINE_TO, 3.1f,\n" \
"V_LINE_TO, 2.85f,\n" \
"H_LINE_TO, 5.4f,\n" \
"R_LINE_TO, 1.67f, 1.66f,\n" \
"CLOSE,\n" \
"MOVE_TO, 2.18f, 2.09f,\n" \
"LINE_TO, 1, 3.27f,\n" \
"R_LINE_TO, 0.81f, 0.81f,\n" \
"R_CUBIC_TO, -0.03f, 0.13f, -0.06f, 0.27f, -0.06f, 0.43f,\n" \
"R_V_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0, 0.46f, 0.17f, 0.85f, 0.49f, 1.17f,\n" \
"R_CUBIC_TO, 0.32f, 0.32f, 0.72f, 0.49f, 1.17f, 0.49f,\n" \
"R_H_LINE_TO, 3.33f,\n" \
"R_V_LINE_TO, 1.67f,\n" \
"R_H_LINE_TO, 6.67f,\n" \
"R_V_LINE_TO, -1.67f,\n" \
"R_H_LINE_TO, 0.49f,\n" \
"R_LINE_TO, 0.24f, 0.24f,\n" \
"R_LINE_TO, 2.18f, 2.18f,\n" \
"R_LINE_TO, 1.19f, -1.18f,\n" \
"LINE_TO, 2.18f, 2.09f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.24f, 12.42f,\n" \
"V_LINE_TO, 5.69f,\n" \
"R_LINE_TO, 2.75f, 2.75f,\n" \
"R_LINE_TO, 4.17f, 4.17f,\n" \
"R_LINE_TO, 1.91f, 1.91f,\n" \
"H_LINE_TO, 3.42f,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kInstallDesktopOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseKeyboardIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 20, 7,\n" \
"R_V_LINE_TO, 10,\n" \
"H_LINE_TO, 4,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, 16,\n" \
"R_MOVE_TO, 0, -2,\n" \
"H_LINE_TO, 4,\n" \
"R_CUBIC_TO, -1.1, 0, -1.99, 0.9, -1.99, 2,\n" \
"LINE_TO, 2, 17,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 16,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -9, 3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 8,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3, 0,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3, 6,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6, -3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3, 3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kKeyboardIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseKeyboardLockIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 160, 680,\n" \
"R_V_LINE_TO, -400,\n" \
"R_V_LINE_TO, 400,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 40, -160,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 120, 0,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 120, 0,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 120, 0,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"MOVE_TO, 200, 400,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 120, 0,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 120, 0,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 120, 0,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 120, 0,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 97,\n" \
"R_QUADRATIC_TO, 18, -8, 38.5, -12.5,\n" \
"QUADRATIC_TO_SHORTHAND, 760, 480,\n" \
"R_V_LINE_TO, -40,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 57,\n" \
"CLOSE,\n" \
"MOVE_TO, 320, 640,\n" \
"R_H_LINE_TO, 244,\n" \
"R_QUADRATIC_TO, 5, -23, 14, -43,\n" \
"R_QUADRATIC_TO, 9, -20, 22, -37,\n" \
"H_LINE_TO, 320,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"MOVE_TO, 160, 760,\n" \
"R_QUADRATIC_TO, -33, 0, -56.5, -23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 80, 680,\n" \
"R_V_LINE_TO, -400,\n" \
"R_QUADRATIC_TO, 0, -33, 23.5, -56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 160, 200,\n" \
"R_H_LINE_TO, 640,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 880, 280,\n" \
"R_V_LINE_TO, 240,\n" \
"R_QUADRATIC_TO, -17, -13, -37, -22,\n" \
"R_QUADRATIC_TO, -20, -9, -43, -14,\n" \
"R_V_LINE_TO, -204,\n" \
"H_LINE_TO, 160,\n" \
"R_V_LINE_TO, 400,\n" \
"R_H_LINE_TO, 400,\n" \
"R_V_LINE_TO, 80,\n" \
"H_LINE_TO, 160,\n" \
"CLOSE,\n" \
"MOVE_TO, 674, 880,\n" \
"R_QUADRATIC_TO, -14, 0, -24, -10,\n" \
"R_QUADRATIC_TO, -10, -10, -10, -24,\n" \
"R_V_LINE_TO, -132,\n" \
"R_QUADRATIC_TO, 0, -14, 10, -24,\n" \
"R_QUADRATIC_TO, 10, -10, 24, -10,\n" \
"R_H_LINE_TO, 6,\n" \
"R_V_LINE_TO, -40,\n" \
"R_QUADRATIC_TO, 0, -33, 23.5, -56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 760, 560,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 840, 640,\n" \
"R_V_LINE_TO, 40,\n" \
"R_H_LINE_TO, 6,\n" \
"R_QUADRATIC_TO, 14, 0, 24, 10,\n" \
"R_QUADRATIC_TO, 10, 10, 10, 24,\n" \
"R_V_LINE_TO, 132,\n" \
"R_QUADRATIC_TO, 0, 14, -10, 24,\n" \
"R_QUADRATIC_TO, -10, 10, -24, 10,\n" \
"H_LINE_TO, 674,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 46, -200,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -40,\n" \
"R_QUADRATIC_TO, 0, -17, -11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 760, 600,\n" \
"R_QUADRATIC_TO, -17, 0, -28.5, 11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 720, 640,\n" \
"R_V_LINE_TO, 40,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kKeyboardLockIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseKeyboardLockOffIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 360, 480,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 167, 160,\n" \
"H_LINE_TO, 320,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, 207,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"MOVE_TO, 200, 520,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, 80,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 120, 0,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, 80,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE,\n" \
"MOVE_TO, 200, 400,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, 80,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 360, 0,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 120, 0,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 97,\n" \
"R_QUADRATIC_TO, 18, -8, 38.5, -12.5,\n" \
"QUADRATIC_TO_SHORTHAND, 760, 480,\n" \
"R_V_LINE_TO, -40,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 57,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -40, 23,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 6,\n" \
"R_LINE_TO, 74, 74,\n" \
"R_H_LINE_TO, 6,\n" \
"CLOSE,\n" \
"MOVE_TO, 520, 400,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, 6,\n" \
"R_LINE_TO, 74, 74,\n" \
"R_H_LINE_TO, 6,\n" \
"CLOSE,\n" \
"MOVE_TO, 819, 932,\n" \
"LINE_TO, 28, 140,\n" \
"R_LINE_TO, 56, -56,\n" \
"LINE_TO, 876, 876,\n" \
"R_LINE_TO, -57, 56,\n" \
"CLOSE,\n" \
"MOVE_TO, 516, 401,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 364, 119,\n" \
"R_QUADRATIC_TO, -17, -13, -37, -22,\n" \
"R_QUADRATIC_TO, -20, -9, -43, -14,\n" \
"R_V_LINE_TO, -204,\n" \
"H_LINE_TO, 394,\n" \
"R_LINE_TO, -80, -80,\n" \
"R_H_LINE_TO, 486,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 880, 280,\n" \
"R_V_LINE_TO, 240,\n" \
"CLOSE,\n" \
"MOVE_TO, 160, 760,\n" \
"R_QUADRATIC_TO, -33, 0, -56.5, -23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 80, 680,\n" \
"R_V_LINE_TO, -400,\n" \
"R_QUADRATIC_TO, 0, -33, 23.5, -56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 160, 200,\n" \
"R_H_LINE_TO, 40,\n" \
"R_LINE_TO, 80, 80,\n" \
"H_LINE_TO, 160,\n" \
"R_V_LINE_TO, 400,\n" \
"R_H_LINE_TO, 400,\n" \
"R_V_LINE_TO, 80,\n" \
"H_LINE_TO, 160,\n" \
"CLOSE,\n" \
"MOVE_TO, 674, 880,\n" \
"R_QUADRATIC_TO, -14, 0, -24, -10,\n" \
"R_QUADRATIC_TO, -10, -10, -10, -24,\n" \
"R_V_LINE_TO, -132,\n" \
"R_QUADRATIC_TO, 0, -14, 10, -24,\n" \
"R_QUADRATIC_TO, 10, -10, 24, -10,\n" \
"R_H_LINE_TO, 6,\n" \
"R_LINE_TO, 160, 160,\n" \
"R_V_LINE_TO, 40,\n" \
"H_LINE_TO, 674,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 206, -114,\n" \
"R_LINE_TO, -80, -80,\n" \
"R_V_LINE_TO, -46,\n" \
"R_QUADRATIC_TO, 0, -17, -11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 760, 600,\n" \
"R_QUADRATIC_TO, -9, 0, -17.5, 4,\n" \
"QUADRATIC_TO_SHORTHAND, 729, 615,\n" \
"R_LINE_TO, -28, -29,\n" \
"R_QUADRATIC_TO, 11, -12, 26.5, -19,\n" \
"R_QUADRATIC_TO, 15.5, -7, 32.5, -7,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 840, 640,\n" \
"R_V_LINE_TO, 40,\n" \
"R_H_LINE_TO, 6,\n" \
"R_QUADRATIC_TO, 14, 0, 24, 10,\n" \
"R_QUADRATIC_TO, 10, 10, 10, 24,\n" \
"R_V_LINE_TO, 52,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kKeyboardLockOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLaunchIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 18,\n" \
"FLIPS_IN_RTL,\n" \
"MOVE_TO, 13.5, 13.5,\n" \
"R_H_LINE_TO, -9,\n" \
"R_V_LINE_TO, -9,\n" \
"H_LINE_TO, 8,\n" \
"V_LINE_TO, 3,\n" \
"H_LINE_TO, 4.33,\n" \
"CUBIC_TO, 3.59, 3, 3, 3.6, 3, 4.33,\n" \
"R_V_LINE_TO, 9.33,\n" \
"CUBIC_TO, 3, 14.4, 3.59, 15, 4.33, 15,\n" \
"R_H_LINE_TO, 9.33,\n" \
"CUBIC_TO, 14.4, 15, 15, 14.4, 15, 13.67,\n" \
"V_LINE_TO, 10,\n" \
"R_H_LINE_TO, -1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 3,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, 2.5,\n" \
"R_LINE_TO, -6, 6,\n" \
"R_LINE_TO, 1, 1,\n" \
"R_LINE_TO, 6, -6,\n" \
"V_LINE_TO, 8,\n" \
"H_LINE_TO, 15,\n" \
"V_LINE_TO, 3,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLaunchIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLaunchChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"FLIPS_IN_RTL,\n" \
"MOVE_TO, 4.5, 17,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3, 15.5,\n" \
"R_V_LINE_TO, -11,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4.5, 3,\n" \
"H_LINE_TO, 10,\n" \
"R_V_LINE_TO, 1.5,\n" \
"H_LINE_TO, 4.5,\n" \
"R_V_LINE_TO, 11,\n" \
"R_H_LINE_TO, 11,\n" \
"V_LINE_TO, 10,\n" \
"H_LINE_TO, 17,\n" \
"R_V_LINE_TO, 5.5,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 15.5, 17,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.56, -4,\n" \
"LINE_TO, 7, 11.94,\n" \
"LINE_TO, 14.44, 4.5,\n" \
"H_LINE_TO, 12,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 5,\n" \
"R_H_LINE_TO, -1.5,\n" \
"V_LINE_TO, 5.56,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLaunchChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLaunchOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16, 18.13,\n" \
"LINE_TO, 14.88, 17,\n" \
"H_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3, 15.5,\n" \
"V_LINE_TO, 5.13,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5, 15.5,\n" \
"R_H_LINE_TO, 8.88,\n" \
"LINE_TO, 9.48, 11.58,\n" \
"LINE_TO, 8.06, 13,\n" \
"LINE_TO, 7, 11.94,\n" \
"R_LINE_TO, 1.42, -1.42,\n" \
"LINE_TO, 4.5, 6.63,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.13, -11,\n" \
"LINE_TO, 5.13, 3,\n" \
"H_LINE_TO, 10,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.96, 4.98,\n" \
"LINE_TO, 10.52, 8.42,\n" \
"LINE_TO, 14.44, 4.5,\n" \
"H_LINE_TO, 12,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 5,\n" \
"R_H_LINE_TO, -1.5,\n" \
"V_LINE_TO, 5.56,\n" \
"CLOSE,\n" \
"MOVE_TO, 17, 14.88,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 10,\n" \
"H_LINE_TO, 17,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLaunchOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLightbulbOutlineIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 9, 21,\n" \
"R_CUBIC_TO, 0, 0.55, 0.45, 1, 1, 1,\n" \
"R_H_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0.55, 0, 1, -0.45, 1, -1,\n" \
"R_V_LINE_TO, -1,\n" \
"H_LINE_TO, 9,\n" \
"R_V_LINE_TO, 1,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3, -19,\n" \
"CUBIC_TO, 8.14, 2, 5, 5.14, 5, 9,\n" \
"R_CUBIC_TO, 0, 2.38, 1.19, 4.47, 3, 5.74,\n" \
"V_LINE_TO, 17,\n" \
"R_CUBIC_TO, 0, 0.55, 0.45, 1, 1, 1,\n" \
"R_H_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0.55, 0, 1, -0.45, 1, -1,\n" \
"R_V_LINE_TO, -2.26,\n" \
"R_CUBIC_TO, 1.81, -1.27, 3, -3.36, 3, -5.74,\n" \
"R_CUBIC_TO, 0, -3.86, -3.14, -7, -7, -7,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.85, 11.1,\n" \
"R_LINE_TO, -0.85, 0.6,\n" \
"V_LINE_TO, 16,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -2.3,\n" \
"R_LINE_TO, -0.85, -0.6,\n" \
"ARC_TO, 5, 5, 0, 0, 1, 7, 9,\n" \
"R_CUBIC_TO, 0, -2.76, 2.24, -5, 5, -5,\n" \
"R_CUBIC_TO, 2.76, 0, 5, 2.24, 5, 5,\n" \
"R_CUBIC_TO, 0, 1.63, -0.8, 3.16, -2.15, 4.1,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLightbulbOutlineIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLightbulbOutlineChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 18,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 8.5, 16.5,\n" \
"R_H_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, 0.42, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 10, 18,\n" \
"CLOSE,\n" \
"MOVE_TO, 7, 15.5,\n" \
"V_LINE_TO, 14,\n" \
"R_H_LINE_TO, 6,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.69, 13,\n" \
"ARC_TO, 5.34, 5.34, 0, 0, 1, 4.7, 10.86,\n" \
"ARC_TO, 6.12, 6.12, 0, 0, 1, 4, 8,\n" \
"R_CUBIC_TO, 0, -1.67, 0.58, -3.08, 1.75, -4.25,\n" \
"CUBIC_TO, 6.92, 2.58, 8.33, 2, 10, 2,\n" \
"R_CUBIC_TO, 1.67, 0, 3.08, 0.58, 4.25, 1.75,\n" \
"CUBIC_TO, 15.42, 4.92, 16, 6.33, 16, 8,\n" \
"R_ARC_TO, 6.12, 6.12, 0, 0, 1, -0.7, 2.86,\n" \
"ARC_TO, 5.34, 5.34, 0, 0, 1, 13.31, 13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.48, -1.5,\n" \
"R_H_LINE_TO, 5.66,\n" \
"R_ARC_TO, 4.52, 4.52, 0, 0, 0, 1.23, -1.56,\n" \
"R_CUBIC_TO, 0.29, -0.61, 0.44, -1.26, 0.44, -1.94,\n" \
"R_CUBIC_TO, 0, -1.25, -0.44, -2.32, -1.31, -3.19,\n" \
"R_CUBIC_TO, -0.87, -0.87, -1.93, -1.31, -3.19, -1.31,\n" \
"R_CUBIC_TO, -1.25, 0, -2.32, 0.44, -3.19, 1.31,\n" \
"CUBIC_TO, 5.94, 5.68, 5.5, 6.75, 5.5, 8,\n" \
"R_CUBIC_TO, 0, 0.68, 0.14, 1.33, 0.44, 1.94,\n" \
"R_ARC_TO, 4.52, 4.52, 0, 0, 0, 1.23, 1.56,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.83, 0,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLightbulbOutlineChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLinkIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 14, 8,\n" \
"H_LINE_TO, 12,\n" \
"V_LINE_TO, 6,\n" \
"H_LINE_TO, 14,\n" \
"CUBIC_TO, 16.21, 6, 18, 7.79, 18, 10,\n" \
"CUBIC_TO, 18, 12.21, 16.21, 14, 14, 14,\n" \
"H_LINE_TO, 12,\n" \
"V_LINE_TO, 12,\n" \
"H_LINE_TO, 14,\n" \
"CUBIC_TO, 15.1, 12, 16, 11.1, 16, 10,\n" \
"CUBIC_TO, 16, 8.9, 15.1, 8, 14, 8,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 6,\n" \
"V_LINE_TO, 8,\n" \
"H_LINE_TO, 6,\n" \
"CUBIC_TO, 4.9, 8, 4, 8.9, 4, 10,\n" \
"CUBIC_TO, 4, 11.1, 4.9, 12, 6, 12,\n" \
"H_LINE_TO, 8,\n" \
"V_LINE_TO, 14,\n" \
"H_LINE_TO, 6,\n" \
"CUBIC_TO, 3.79, 14, 2, 12.21, 2, 10,\n" \
"CUBIC_TO, 2, 7.79, 3.79, 6, 6, 6,\n" \
"H_LINE_TO, 8,\n" \
"CLOSE,\n" \
"MOVE_TO, 13, 9,\n" \
"H_LINE_TO, 7,\n" \
"V_LINE_TO, 11,\n" \
"H_LINE_TO, 13,\n" \
"V_LINE_TO, 9,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLinkIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLiveCaptionOnIcon) { \
              std::string s = \
"// Copyright 2020 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 2, 5.5,\n" \
"ARC_TO, 1.5, 1.5, 0, 0, 1, 3.5, 4,\n" \
"R_H_LINE_TO, 13,\n" \
"ARC_TO, 1.5, 1.5, 0, 0, 1, 18, 5.5,\n" \
"R_V_LINE_TO, 9,\n" \
"R_ARC_TO, 1.5, 1.5, 0, 0, 1, -1.5, 1.5,\n" \
"R_H_LINE_TO, -13,\n" \
"ARC_TO, 1.5, 1.5, 0, 0, 1, 2, 14.5,\n" \
"R_V_LINE_TO, -9,\n" \
"CLOSE,\n" \
"MOVE_TO, 4, 6,\n" \
"R_H_LINE_TO, 12,\n" \
"R_V_LINE_TO, 8,\n" \
"H_LINE_TO, 4,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1, 2,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 5,\n" \
"V_LINE_TO, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 3,\n" \
"R_H_LINE_TO, 7,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 5,\n" \
"R_V_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 10, 0,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5, -3,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -5,\n" \
"V_LINE_TO, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLiveCaptionOnIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLocationOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 14.06, 11.92,\n" \
"R_LINE_TO, -1.08, -1.09,\n" \
"R_ARC_TO, 7.11, 7.11, 0, 0, 0, 0.74, -1.59,\n" \
"ARC_TO, 5.59, 5.59, 0, 0, 0, 14, 7.5,\n" \
"R_CUBIC_TO, 0, -1.1, -0.39, -2.05, -1.17, -2.83,\n" \
"CUBIC_TO, 12.05, 3.89, 11.11, 3.5, 10, 3.5,\n" \
"R_CUBIC_TO, -0.58, 0, -1.13, 0.12, -1.64, 0.36,\n" \
"ARC_TO, 4.34, 4.34, 0, 0, 0, 7, 4.88,\n" \
"LINE_TO, 5.94, 3.81,\n" \
"ARC_TO, 5.46, 5.46, 0, 0, 1, 7.78, 2.47,\n" \
"ARC_TO, 5.4, 5.4, 0, 0, 1, 10, 2,\n" \
"R_CUBIC_TO, 1.53, 0, 2.83, 0.54, 3.9, 1.6,\n" \
"CUBIC_TO, 14.97, 4.67, 15.5, 5.97, 15.5, 7.5,\n" \
"R_CUBIC_TO, 0, 0.8, -0.13, 1.56, -0.39, 2.31,\n" \
"R_ARC_TO, 9.12, 9.12, 0, 0, 1, -1.05, 2.11,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.68, -2.67,\n" \
"R_CUBIC_TO, 0.19, -0.19, 0.34, -0.42, 0.45, -0.67,\n" \
"R_CUBIC_TO, 0.11, -0.25, 0.17, -0.51, 0.17, -0.79,\n" \
"R_ARC_TO, 1.93, 1.93, 0, 0, 0, -0.58, -1.42,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 0, 10, 5.79,\n" \
"R_ARC_TO, 2.22, 2.22, 0, 0, 0, -0.8, 0.14,\n" \
"R_ARC_TO, 1.53, 1.53, 0, 0, 0, -0.66, 0.48,\n" \
"CLOSE,\n" \
"MOVE_TO, 16, 18.13,\n" \
"R_LINE_TO, -3.69, -3.71,\n" \
"R_CUBIC_TO, -0.23, 0.38, -0.47, 0.76, -0.7, 1.16,\n" \
"R_CUBIC_TO, -0.23, 0.4, -0.44, 0.81, -0.63, 1.24,\n" \
"R_ARC_TO, 9.4, 9.4, 0, 0, 1, -0.37, 0.8,\n" \
"ARC_TO, 0.65, 0.65, 0, 0, 1, 10, 18,\n" \
"R_ARC_TO, 0.65, 0.65, 0, 0, 1, -0.61, -0.39,\n" \
"R_ARC_TO, 9.4, 9.4, 0, 0, 1, -0.37, -0.8,\n" \
"R_ARC_TO, 11.49, 11.49, 0, 0, 0, -1.13, -2.08,\n" \
"R_ARC_TO, 44.89, 44.89, 0, 0, 0, -1.39, -1.89,\n" \
"R_ARC_TO, 15.05, 15.05, 0, 0, 1, -1.41, -2.54,\n" \
"R_ARC_TO, 6.97, 6.97, 0, 0, 1, -0.57, -3.22,\n" \
"R_CUBIC_TO, 0, -0.13, 0.02, -0.27, 0.05, -0.4,\n" \
"LINE_TO, 1.88, 3.98,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -6, -2.73,\n" \
"R_CUBIC_TO, 0.17, -0.36, 0.36, -0.71, 0.58, -1.04,\n" \
"R_LINE_TO, 0.67, -1,\n" \
"R_LINE_TO, -5.21, -5.21,\n" \
"R_CUBIC_TO, 0.08, 0.78, 0.32, 1.5, 0.71, 2.17,\n" \
"R_CUBIC_TO, 0.39, 0.67, 0.81, 1.32, 1.25, 1.96,\n" \
"R_CUBIC_TO, 0.38, 0.5, 0.73, 1, 1.07, 1.51,\n" \
"R_CUBIC_TO, 0.34, 0.51, 0.65, 1.05, 0.93, 1.61,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.67, 11,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.38, -3.08,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLocationOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLocationOnIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 24, 4,\n" \
"R_CUBIC_TO, -7.73, 0, -14, 6.27, -14, 14,\n" \
"R_CUBIC_TO, 0, 10.5, 14, 26, 14, 26,\n" \
"R_CUBIC_TO, 0, 0, 14, -15.5, 14, -26,\n" \
"R_CUBIC_TO, 0, -7.73, -6.27, -14, -14, -14,\n" \
"CLOSE,\n" \
"CIRCLE, 24, 18, 5\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 8, 1,\n" \
"CUBIC_TO, 5.24, 1, 3, 3.19, 3, 5.9,\n" \
"CUBIC_TO, 3, 9.57, 8, 15, 8, 15,\n" \
"CUBIC_TO, 8, 15, 13, 9.57, 13, 5.9,\n" \
"CUBIC_TO, 13, 3.19, 10.76, 1, 8, 1,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 8,\n" \
"CUBIC_TO, 6.9, 8, 6, 7.1, 6, 6,\n" \
"CUBIC_TO, 6, 4.9, 6.9, 4, 8, 4,\n" \
"CUBIC_TO, 9.1, 4, 10, 4.9, 10, 6,\n" \
"CUBIC_TO, 10, 7.1, 9.1, 8, 8, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLocationOnIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLocationOnChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 18,\n" \
"R_ARC_TO, 0.78, 0.78, 0, 0, 1, -0.49, -0.16,\n" \
"R_ARC_TO, 0.82, 0.82, 0, 0, 1, -0.29, -0.41,\n" \
"R_ARC_TO, 14.33, 14.33, 0, 0, 0, -0.95, -2.09,\n" \
"R_CUBIC_TO, -0.37, -0.66, -0.9, -1.43, -1.59, -2.32,\n" \
"R_CUBIC_TO, -0.68, -0.87, -1.23, -1.71, -1.66, -2.52,\n" \
"R_CUBIC_TO, -0.42, -0.8, -0.64, -1.77, -0.64, -2.9,\n" \
"R_CUBIC_TO, 0, -1.55, 0.55, -2.87, 1.63, -3.96,\n" \
"CUBIC_TO, 7.11, 2.55, 8.44, 2, 10, 2,\n" \
"R_CUBIC_TO, 1.56, 0, 2.89, 0.54, 3.97, 1.63,\n" \
"R_CUBIC_TO, 1.09, 1.09, 1.63, 2.42, 1.63, 3.98,\n" \
"R_CUBIC_TO, 0, 1.22, -0.23, 2.22, -0.7, 3.02,\n" \
"R_ARC_TO, 23.89, 23.89, 0, 0, 1, -1.59, 2.38,\n" \
"R_CUBIC_TO, -0.72, 0.95, -1.26, 1.74, -1.64, 2.39,\n" \
"R_ARC_TO, 11.54, 11.54, 0, 0, 0, -0.91, 2.03,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 1, -0.3, 0.41,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 1, -0.46, 0.16,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 15.36,\n" \
"R_CUBIC_TO, 0.23, -0.47, 0.5, -0.94, 0.8, -1.4,\n" \
"R_CUBIC_TO, 0.3, -0.46, 0.74, -1.08, 1.33, -1.85,\n" \
"R_CUBIC_TO, 0.58, -0.75, 1.06, -1.45, 1.43, -2.09,\n" \
"R_CUBIC_TO, 0.37, -0.64, 0.55, -1.45, 0.55, -2.41,\n" \
"R_CUBIC_TO, 0, -1.12, -0.4, -2.09, -1.21, -2.9,\n" \
"CUBIC_TO, 12.09, 3.9, 11.12, 3.5, 9.99, 3.5,\n" \
"R_ARC_TO, 3.93, 3.93, 0, 0, 0, -2.89, 1.21,\n" \
"R_CUBIC_TO, -0.8, 0.81, -1.2, 1.77, -1.2, 2.9,\n" \
"R_CUBIC_TO, 0, 0.96, 0.19, 1.77, 0.56, 2.41,\n" \
"R_CUBIC_TO, 0.38, 0.64, 0.85, 1.34, 1.42, 2.09,\n" \
"R_CUBIC_TO, 0.58, 0.77, 1.02, 1.39, 1.32, 1.85,\n" \
"R_CUBIC_TO, 0.3, 0.46, 0.57, 0.93, 0.8, 1.4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -5.75,\n" \
"R_CUBIC_TO, 0.56, 0, 1.03, -0.19, 1.42, -0.58,\n" \
"ARC_TO, 1.93, 1.93, 0, 0, 0, 12, 7.61,\n" \
"R_CUBIC_TO, 0, -0.56, -0.19, -1.03, -0.58, -1.42,\n" \
"R_ARC_TO, 1.92, 1.92, 0, 0, 0, -1.42, -0.58,\n" \
"R_CUBIC_TO, -0.56, 0, -1.03, 0.19, -1.42, 0.58,\n" \
"ARC_TO, 1.92, 1.92, 0, 0, 0, 8, 7.6,\n" \
"R_CUBIC_TO, 0, 0.56, 0.19, 1.03, 0.58, 1.42,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.86, 0.59, 1.42, 0.59,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 7.58,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 8, 14.52,\n" \
"R_ARC_TO, 0.74, 0.74, 0, 0, 1, -0.46, -0.15,\n" \
"R_ARC_TO, 0.86, 0.86, 0, 0, 1, -0.29, -0.39,\n" \
"R_CUBIC_TO, -0.25, -0.64, -0.52, -1.21, -0.8, -1.72,\n" \
"R_CUBIC_TO, -0.28, -0.51, -0.69, -1.09, -1.2, -1.76,\n" \
"R_CUBIC_TO, -0.55, -0.71, -1, -1.39, -1.35, -2.05,\n" \
"R_CUBIC_TO, -0.34, -0.66, -0.51, -1.44, -0.51, -2.37,\n" \
"R_CUBIC_TO, 0, -1.28, 0.45, -2.37, 1.34, -3.27,\n" \
"CUBIC_TO, 5.62, 1.91, 6.72, 1.47, 8, 1.47,\n" \
"R_CUBIC_TO, 1.29, 0, 2.38, 0.45, 3.27, 1.35,\n" \
"R_CUBIC_TO, 0.9, 0.9, 1.34, 1.99, 1.34, 3.28,\n" \
"R_CUBIC_TO, 0, 1, -0.19, 1.82, -0.57, 2.47,\n" \
"R_ARC_TO, 19.3, 19.3, 0, 0, 1, -1.3, 1.94,\n" \
"R_CUBIC_TO, -0.53, 0.69, -0.94, 1.29, -1.23, 1.79,\n" \
"R_ARC_TO, 11.26, 11.26, 0, 0, 0, -0.78, 1.69,\n" \
"R_ARC_TO, 1.02, 1.02, 0, 0, 1, -0.3, 0.4,\n" \
"R_ARC_TO, 0.7, 0.7, 0, 0, 1, -0.44, 0.15,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 12.21,\n" \
"R_CUBIC_TO, 0.19, -0.38, 0.4, -0.75, 0.63, -1.12,\n" \
"R_CUBIC_TO, 0.24, -0.37, 0.58, -0.85, 1.03, -1.44,\n" \
"R_CUBIC_TO, 0.47, -0.59, 0.84, -1.15, 1.14, -1.66,\n" \
"R_CUBIC_TO, 0.29, -0.51, 0.44, -1.14, 0.44, -1.91,\n" \
"R_CUBIC_TO, 0, -0.89, -0.32, -1.64, -0.95, -2.28,\n" \
"R_ARC_TO, 3.11, 3.11, 0, 0, 0, -2.29, -0.95,\n" \
"R_CUBIC_TO, -0.89, 0, -1.65, 0.32, -2.28, 0.95,\n" \
"R_ARC_TO, 3.13, 3.13, 0, 0, 0, -0.94, 2.28,\n" \
"R_CUBIC_TO, 0, 0.77, 0.15, 1.4, 0.45, 1.91,\n" \
"R_CUBIC_TO, 0.3, 0.51, 0.68, 1.07, 1.13, 1.66,\n" \
"R_CUBIC_TO, 0.45, 0.59, 0.79, 1.07, 1.03, 1.44,\n" \
"R_CUBIC_TO, 0.24, 0.37, 0.45, 0.74, 0.63, 1.12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -4.51,\n" \
"R_CUBIC_TO, 0.45, 0, 0.83, -0.16, 1.15, -0.46,\n" \
"R_CUBIC_TO, 0.31, -0.31, 0.47, -0.69, 0.47, -1.14,\n" \
"R_CUBIC_TO, 0, -0.45, -0.16, -0.83, -0.47, -1.14,\n" \
"R_ARC_TO, 1.54, 1.54, 0, 0, 0, -1.14, -0.47,\n" \
"R_CUBIC_TO, -0.45, 0, -0.83, 0.15, -1.14, 0.47,\n" \
"R_ARC_TO, 1.55, 1.55, 0, 0, 0, -0.47, 1.14,\n" \
"R_CUBIC_TO, 0, 0.45, 0.16, 0.83, 0.47, 1.15,\n" \
"R_CUBIC_TO, 0.31, 0.31, 0.69, 0.47, 1.14, 0.47,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 6.07,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLocationOnChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLockIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 36, 16,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, -4,\n" \
"R_CUBIC_TO, 0, -5.52, -4.48, -10, -10, -10,\n" \
"CUBIC_TO, 18.48, 2, 14, 6.48, 14, 12,\n" \
"R_V_LINE_TO, 4,\n" \
"R_H_LINE_TO, -2,\n" \
"R_CUBIC_TO, -2.21, 0, -4, 1.79, -4, 4,\n" \
"R_V_LINE_TO, 20,\n" \
"R_CUBIC_TO, 0, 2.21, 1.79, 4, 4, 4,\n" \
"R_H_LINE_TO, 24,\n" \
"R_CUBIC_TO, 2.21, 0, 4, -1.79, 4, -4,\n" \
"V_LINE_TO, 20,\n" \
"R_CUBIC_TO, 0, -2.21, -1.79, -4, -4, -4,\n" \
"CLOSE,\n" \
"MOVE_TO, 24, 34,\n" \
"R_CUBIC_TO, -2.21, 0, -4, -1.79, -4, -4,\n" \
"R_CUBIC_TO, 0, -2.21, 1.79, -4, 4, -4,\n" \
"R_CUBIC_TO, 2.21, 0, 4, 1.79, 4, 4,\n" \
"R_CUBIC_TO, 0, 2.21, -1.79, 4, -4, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6.2, -18,\n" \
"H_LINE_TO, 17.8,\n" \
"R_V_LINE_TO, -4,\n" \
"R_CUBIC_TO, 0, -3.42, 2.78, -6.2, 6.2, -6.2,\n" \
"R_CUBIC_TO, 3.42, 0, 6.2, 2.78, 6.2, 6.2,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLockIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseLoyaltyIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 13, 17.5,\n" \
"R_LINE_TO, 3.5, -3.5,\n" \
"R_CUBIC_TO, 0.18, -0.18, 0.33, -0.4, 0.44, -0.65,\n" \
"R_CUBIC_TO, 0.11, -0.25, 0.16, -0.52, 0.16, -0.8,\n" \
"R_ARC_TO, 1.98, 1.98, 0, 0, 0, -0.6, -1.45,\n" \
"R_ARC_TO, 1.96, 1.96, 0, 0, 0, -1.45, -0.6,\n" \
"R_CUBIC_TO, -0.32, 0, -0.63, 0.09, -0.94, 0.27,\n" \
"R_CUBIC_TO, -0.31, 0.18, -0.68, 0.49, -1.11, 0.93,\n" \
"R_CUBIC_TO, -0.5, -0.46, -0.89, -0.78, -1.18, -0.95,\n" \
"R_ARC_TO, 1.69, 1.69, 0, 0, 0, -0.87, -0.25,\n" \
"R_CUBIC_TO, -0.57, 0, -1.05, 0.2, -1.45, 0.6,\n" \
"R_ARC_TO, 1.98, 1.98, 0, 0, 0, -0.6, 1.45,\n" \
"R_CUBIC_TO, 0, 0.28, 0.06, 0.55, 0.17, 0.8,\n" \
"R_CUBIC_TO, 0.11, 0.25, 0.25, 0.47, 0.44, 0.65,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.4, -3.25,\n" \
"R_LINE_TO, -7.15, 7.15,\n" \
"R_ARC_TO, 1.98, 1.98, 0, 0, 1, -0.68, 0.45,\n" \
"R_CUBIC_TO, -0.25, 0.1, -0.5, 0.15, -0.75, 0.15,\n" \
"R_CUBIC_TO, -0.25, 0, -0.5, -0.05, -0.75, -0.15,\n" \
"R_ARC_TO, 2.02, 2.02, 0, 0, 1, -0.68, -0.45,\n" \
"R_LINE_TO, -8.82, -8.82,\n" \
"ARC_TO, 1.96, 1.96, 0, 0, 1, 2, 11.18,\n" \
"V_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0, -0.55, 0.2, -1.02, 0.59, -1.41,\n" \
"CUBIC_TO, 2.98, 2.2, 3.45, 2, 4, 2,\n" \
"R_H_LINE_TO, 7.18,\n" \
"R_CUBIC_TO, 0.27, 0, 0.52, 0.06, 0.77, 0.16,\n" \
"R_CUBIC_TO, 0.25, 0.11, 0.47, 0.25, 0.65, 0.44,\n" \
"R_LINE_TO, 8.8, 8.82,\n" \
"R_CUBIC_TO, 0.2, 0.2, 0.35, 0.43, 0.44, 0.68,\n" \
"R_CUBIC_TO, 0.09, 0.25, 0.14, 0.5, 0.14, 0.75,\n" \
"R_CUBIC_TO, 0, 0.25, -0.05, 0.5, -0.14, 0.73,\n" \
"R_CUBIC_TO, -0.09, 0.24, -0.23, 0.47, -0.44, 0.66,\n" \
"CLOSE,\n" \
"MOVE_TO, 12.82, 20,\n" \
"R_LINE_TO, 7.15, -7.15,\n" \
"LINE_TO, 11.15, 4,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 7.15,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.5, 8,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, -0.14, 1.06, -0.44,\n" \
"CUBIC_TO, 7.85, 7.27, 8, 6.92, 8, 6.5,\n" \
"R_CUBIC_TO, 0, -0.42, -0.14, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 6.5, 5,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, 0.14, -1.06, 0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 5, 6.5,\n" \
"R_CUBIC_TO, 0, 0.42, 0.14, 0.77, 0.44, 1.06,\n" \
"CUBIC_TO, 5.73, 7.85, 6.08, 8, 6.5, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.5, 4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kLoyaltyIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMagicButtonIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 32, 44.35,\n" \
"LINE_TO, 26.41, 38.76,\n" \
"LINE_TO, 32, 33.18,\n" \
"LINE_TO, 37.59, 38.76,\n" \
"CLOSE,\n" \
"MOVE_TO, 15.29, 37.6,\n" \
"LINE_TO, 3.7, 26,\n" \
"LINE_TO, 15.29, 14.4,\n" \
"LINE_TO, 26.88, 26,\n" \
"CLOSE,\n" \
"MOVE_TO, 35, 23.48,\n" \
"CUBIC_TO, 35, 20.56, 33.98, 18.09, 31.95, 16.05,\n" \
"CUBIC_TO, 29.91, 14.02, 27.44, 13, 24.52, 13,\n" \
"CUBIC_TO, 27.44, 13, 29.91, 11.98, 31.95, 9.95,\n" \
"CUBIC_TO, 33.98, 7.91, 35, 5.43, 35, 2.51,\n" \
"CUBIC_TO, 35, 5.43, 36.02, 7.91, 38.05, 9.95,\n" \
"CUBIC_TO, 40.09, 11.98, 42.57, 13, 45.49, 13,\n" \
"CUBIC_TO, 42.57, 13, 40.09, 14.02, 38.05, 16.05,\n" \
"CUBIC_TO, 36.02, 18.09, 35, 20.56, 35, 23.48,\n" \
"CLOSE,\n" \
"MOVE_TO, 35, 23.48,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMagicButtonIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaNextTrackIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 6, 18,\n" \
"R_LINE_TO, 8.5, -6,\n" \
"LINE_TO, 6, 6,\n" \
"R_V_LINE_TO, 12,\n" \
"CLOSE,\n" \
"MOVE_TO, 16, 6,\n" \
"R_V_LINE_TO, 12,\n" \
"R_H_LINE_TO, 2,\n" \
"V_LINE_TO, 6,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaNextTrackIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaPreviousTrackIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 6, 6,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 12,\n" \
"H_LINE_TO, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.5, 6,\n" \
"R_LINE_TO, 8.5, 6,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaPreviousTrackIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaRouterActiveChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 15, 13,\n" \
"R_H_LINE_TO, -3.98,\n" \
"R_ARC_TO, 6.57, 6.57, 0, 0, 0, -0.3, -0.78,\n" \
"R_CUBIC_TO, -0.12, -0.26, -0.23, -0.5, -0.34, -0.72,\n" \
"H_LINE_TO, 13.5,\n" \
"R_V_LINE_TO, -3,\n" \
"H_LINE_TO, 7.83,\n" \
"R_ARC_TO, 9.7, 9.7, 0, 0, 0, -1.31, -0.85,\n" \
"ARC_TO, 9.63, 9.63, 0, 0, 0, 5.06, 7,\n" \
"H_LINE_TO, 15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5, -3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -8, 6,\n" \
"R_V_LINE_TO, -2,\n" \
"R_CUBIC_TO, 0.56, 0, 1.03, 0.2, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.5, 0,\n" \
"R_CUBIC_TO, 0, -0.97, -0.34, -1.8, -1.02, -2.48,\n" \
"ARC_TO, 3.38, 3.38, 0, 0, 0, 2, 12.5,\n" \
"V_LINE_TO, 11,\n" \
"R_CUBIC_TO, 1.39, 0, 2.57, 0.48, 3.54, 1.46,\n" \
"CUBIC_TO, 6.52, 13.43, 7, 14.61, 7, 16,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3, 0,\n" \
"R_ARC_TO, 6.3, 6.3, 0, 0, 0, -0.51, -2.53,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 0, -1.39, -2.07,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 0, -2.07, -1.39,\n" \
"ARC_TO, 6.3, 6.3, 0, 0, 0, 2, 9.5,\n" \
"V_LINE_TO, 8,\n" \
"R_CUBIC_TO, 1.11, 0, 2.15, 0.21, 3.12, 0.63,\n" \
"ARC_TO, 8.09, 8.09, 0, 0, 1, 7.66, 10.34,\n" \
"R_ARC_TO, 8.09, 8.09, 0, 0, 1, 1.72, 2.54,\n" \
"CUBIC_TO, 9.79, 13.85, 10, 14.89, 10, 16,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8, 0,\n" \
"R_H_LINE_TO, -5,\n" \
"R_CUBIC_TO, 0, -0.25, -0.01, -0.5, -0.03, -0.75,\n" \
"R_CUBIC_TO, -0.02, -0.25, -0.05, -0.5, -0.09, -0.75,\n" \
"H_LINE_TO, 16.5,\n" \
"R_V_LINE_TO, -9,\n" \
"R_H_LINE_TO, -13,\n" \
"R_V_LINE_TO, 1.13,\n" \
"R_CUBIC_TO, -0.25, -0.04, -0.5, -0.07, -0.75, -0.09,\n" \
"R_CUBIC_TO, -0.25, -0.02, -0.5, -0.03, -0.75, -0.03,\n" \
"R_V_LINE_TO, -1,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 4,\n" \
"R_H_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 9.01,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.76, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 16,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaRouterActiveChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaRouterErrorIcon) { \
              std::string s = \
"// Copyright 2016 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 14.55, 1.5,\n" \
"H_LINE_TO, 1.45,\n" \
"CUBIC_TO, 0.65, 1.5, 0, 2.15, 0, 2.94,\n" \
"R_V_LINE_TO, 2.18,\n" \
"R_LINE_TO, 1.5, 0,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 13,\n" \
"R_V_LINE_TO, 10,\n" \
"R_H_LINE_TO, -5,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, 5.05,\n" \
"R_CUBIC_TO, 0.8, 0, 1.45, -0.65, 1.45, -1.44,\n" \
"V_LINE_TO, 2.94,\n" \
"CUBIC_TO, 16, 2.15, 15.35, 1.5, 14.55, 1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 0, 12.37,\n" \
"R_V_LINE_TO, 2.13,\n" \
"R_H_LINE_TO, 2.12,\n" \
"CUBIC_TO, 2.12, 13.3, 1.21, 12.37, 0, 12.37,\n" \
"CLOSE,\n" \
"MOVE_TO, 0, 9.37,\n" \
"R_LINE_TO, 0, 1.5,\n" \
"R_CUBIC_TO, 2.01, 0, 3.62, 1.63, 3.62, 3.63,\n" \
"R_H_LINE_TO, 1.5,\n" \
"CUBIC_TO, 5.12, 11.71, 2.81, 9.37, 0, 9.37,\n" \
"CLOSE,\n" \
"MOVE_TO, 0, 6.56,\n" \
"V_LINE_TO, 8,\n" \
"R_CUBIC_TO, 3.61, 0, 6.5, 2.91, 6.5, 6.5,\n" \
"H_LINE_TO, 8,\n" \
"CUBIC_TO, 8, 10.11, 4.41, 6.56, 0, 6.56,\n" \
"CLOSE,\n" \
"MOVE_TO, 12.94, 6,\n" \
"R_LINE_TO, -1.69, 1.69,\n" \
"LINE_TO, 9.56, 6,\n" \
"LINE_TO, 8.5, 7.06,\n" \
"R_LINE_TO, 1.69, 1.69,\n" \
"R_LINE_TO, -1.69, 1.69,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"R_LINE_TO, 1.69, -1.69,\n" \
"R_LINE_TO, 1.69, 1.69,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, -1.69, -1.69,\n" \
"R_LINE_TO, 1.69, -1.69,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaRouterErrorIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaRouterIdleIcon) { \
              std::string s = \
"// Copyright 2016 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 14.55, 1.5,\n" \
"H_LINE_TO, 1.45,\n" \
"CUBIC_TO, 0.65, 1.5, 0, 2.15, 0, 2.94,\n" \
"R_V_LINE_TO, 2.18,\n" \
"R_H_LINE_TO, 1.5,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 13,\n" \
"R_V_LINE_TO, 10,\n" \
"R_H_LINE_TO, -5,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, 5.05,\n" \
"R_CUBIC_TO, 0.8, 0, 1.45, -0.65, 1.45, -1.44,\n" \
"V_LINE_TO, 2.94,\n" \
"R_CUBIC_TO, 0, -0.79, -0.65, -1.44, -1.45, -1.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 0, 12.37,\n" \
"R_V_LINE_TO, 2.13,\n" \
"R_H_LINE_TO, 2.12,\n" \
"R_CUBIC_TO, 0, -1.2, -0.91, -2.13, -2.12, -2.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 2.01, 0, 3.62, 1.63, 3.62, 3.63,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0, -2.79, -2.31, -5.13, -5.12, -5.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2.81,\n" \
"V_LINE_TO, 8,\n" \
"R_CUBIC_TO, 3.61, 0, 6.5, 2.91, 6.5, 6.5,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, -4.39, -3.59, -7.94, -8, -7.94,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaRouterIdleIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaRouterIdleChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6.5, 6,\n" \
"R_H_LINE_TO, -5,\n" \
"R_CUBIC_TO, 0, -0.25, -0.01, -0.5, -0.03, -0.75,\n" \
"R_CUBIC_TO, -0.02, -0.25, -0.05, -0.5, -0.09, -0.75,\n" \
"H_LINE_TO, 16.5,\n" \
"R_V_LINE_TO, -9,\n" \
"R_H_LINE_TO, -13,\n" \
"R_V_LINE_TO, 1.13,\n" \
"R_CUBIC_TO, -0.25, -0.04, -0.5, -0.07, -0.75, -0.09,\n" \
"R_CUBIC_TO, -0.25, -0.02, -0.5, -0.03, -0.75, -0.03,\n" \
"R_V_LINE_TO, -1,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 4,\n" \
"R_H_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 9.01,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.76, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 16,\n" \
"CLOSE,\n" \
"MOVE_TO, 2, 16,\n" \
"R_V_LINE_TO, -2,\n" \
"R_CUBIC_TO, 0.56, 0, 1.03, 0.2, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.5, 0,\n" \
"R_CUBIC_TO, 0, -0.97, -0.34, -1.8, -1.02, -2.48,\n" \
"ARC_TO, 3.38, 3.38, 0, 0, 0, 2, 12.5,\n" \
"V_LINE_TO, 11,\n" \
"R_CUBIC_TO, 1.39, 0, 2.57, 0.48, 3.54, 1.46,\n" \
"CUBIC_TO, 6.52, 13.43, 7, 14.61, 7, 16,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3, 0,\n" \
"R_ARC_TO, 6.3, 6.3, 0, 0, 0, -0.51, -2.53,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 0, -1.39, -2.07,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 0, -2.07, -1.39,\n" \
"ARC_TO, 6.3, 6.3, 0, 0, 0, 2, 9.5,\n" \
"V_LINE_TO, 8,\n" \
"R_CUBIC_TO, 1.11, 0, 2.15, 0.21, 3.12, 0.63,\n" \
"ARC_TO, 8.09, 8.09, 0, 0, 1, 7.66, 10.34,\n" \
"R_ARC_TO, 8.09, 8.09, 0, 0, 1, 1.72, 2.54,\n" \
"CUBIC_TO, 9.79, 13.85, 10, 14.89, 10, 16,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaRouterIdleChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaRouterPausedIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 15.13, 7.19,\n" \
"R_H_LINE_TO, 1.08,\n" \
"V_LINE_TO, 3.61,\n" \
"R_H_LINE_TO, -1.08,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2, 0,\n" \
"R_H_LINE_TO, 1.08,\n" \
"V_LINE_TO, 3.61,\n" \
"R_H_LINE_TO, -1.08,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -8.17, 4.09,\n" \
"R_CUBIC_TO, 0.58, 0, 1.07, 0.2, 1.47, 0.61,\n" \
"R_CUBIC_TO, 0.4, 0.41, 0.61, 0.9, 0.61, 1.47,\n" \
"H_LINE_TO, 1.83,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3.07,\n" \
"R_CUBIC_TO, 1.43, 0, 2.65, 0.5, 3.65, 1.5,\n" \
"R_CUBIC_TO, 1, 1, 1.5, 2.22, 1.5, 3.65,\n" \
"H_LINE_TO, 5.41,\n" \
"R_CUBIC_TO, 0, -0.99, -0.35, -1.84, -1.05, -2.53,\n" \
"R_ARC_TO, 3.46, 3.46, 0, 0, 0, -2.53, -1.04,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3.07,\n" \
"R_CUBIC_TO, 1.15, 0, 2.21, 0.22, 3.21, 0.64,\n" \
"R_ARC_TO, 8.32, 8.32, 0, 0, 1, 2.61, 1.77,\n" \
"R_ARC_TO, 8.23, 8.23, 0, 0, 1, 1.77, 2.61,\n" \
"R_CUBIC_TO, 0.43, 1, 0.64, 2.07, 0.64, 3.21,\n" \
"H_LINE_TO, 8.49,\n" \
"R_CUBIC_TO, 0, -0.92, -0.17, -1.78, -0.52, -2.59,\n" \
"R_ARC_TO, 6.8, 6.8, 0, 0, 0, -1.43, -2.12,\n" \
"R_ARC_TO, 6.8, 6.8, 0, 0, 0, -2.12, -1.43,\n" \
"R_ARC_TO, 6.45, 6.45, 0, 0, 0, -2.59, -0.52,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 14.61, 8.23,\n" \
"R_H_LINE_TO, -4.88,\n" \
"R_ARC_TO, 9.06, 9.06, 0, 0, 0, -0.16, -1.73,\n" \
"R_H_LINE_TO, 5.05,\n" \
"R_V_LINE_TO, -3.87,\n" \
"R_CUBIC_TO, 0.32, -0.1, 0.62, -0.22, 0.91, -0.37,\n" \
"R_CUBIC_TO, 0.29, -0.15, 0.56, -0.34, 0.82, -0.56,\n" \
"R_V_LINE_TO, 4.79,\n" \
"R_CUBIC_TO, 0, 0.48, -0.17, 0.89, -0.5, 1.23,\n" \
"R_CUBIC_TO, -0.34, 0.34, -0.75, 0.5, -1.22, 0.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 1.83, 6.44,\n" \
"V_LINE_TO, 5.56,\n" \
"R_CUBIC_TO, 0, -0.48, 0.17, -0.89, 0.5, -1.22,\n" \
"R_ARC_TO, 1.67, 1.67, 0, 0, 1, 1.22, -0.5,\n" \
"R_H_LINE_TO, 5.84,\n" \
"R_ARC_TO, 4.7, 4.7, 0, 0, 0, -0.18, 0.84,\n" \
"R_CUBIC_TO, -0.04, 0.29, -0.05, 0.59, -0.04, 0.88,\n" \
"H_LINE_TO, 3.56,\n" \
"R_V_LINE_TO, 1.05,\n" \
"R_ARC_TO, 9.19, 9.19, 0, 0, 0, -0.86, -0.12,\n" \
"R_ARC_TO, 9.18, 9.18, 0, 0, 0, -0.86, -0.04,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12.83, 2.96,\n" \
"R_CUBIC_TO, -1.1, 0, -2.05, -0.39, -2.83, -1.17,\n" \
"R_CUBIC_TO, -0.78, -0.78, -1.17, -1.73, -1.17, -2.83,\n" \
"R_CUBIC_TO, 0, -1.11, 0.39, -2.05, 1.17, -2.83,\n" \
"R_CUBIC_TO, 0.78, -0.78, 1.73, -1.17, 2.84, -1.17,\n" \
"R_CUBIC_TO, 1.11, 0, 2.05, 0.39, 2.83, 1.17,\n" \
"R_CUBIC_TO, 0.78, 0.78, 1.16, 1.73, 1.16, 2.84,\n" \
"R_CUBIC_TO, 0, 1.11, -0.39, 2.05, -1.17, 2.83,\n" \
"R_CUBIC_TO, -0.78, 0.78, -1.72, 1.16, -2.83, 1.16,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaRouterPausedIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaRouterWarningChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -8, 4,\n" \
"R_CUBIC_TO, 0.56, 0, 1.03, 0.2, 1.42, 0.58,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.58, 0.86, 0.58, 1.42,\n" \
"H_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"R_CUBIC_TO, 1.39, 0, 2.57, 0.48, 3.54, 1.46,\n" \
"CUBIC_TO, 6.52, 13.43, 7, 14.61, 7, 16,\n" \
"H_LINE_TO, 5.5,\n" \
"R_CUBIC_TO, 0, -0.97, -0.34, -1.8, -1.02, -2.48,\n" \
"ARC_TO, 3.38, 3.38, 0, 0, 0, 2, 12.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"R_CUBIC_TO, 1.11, 0, 2.15, 0.21, 3.12, 0.63,\n" \
"ARC_TO, 8.09, 8.09, 0, 0, 1, 7.66, 10.34,\n" \
"R_ARC_TO, 8.09, 8.09, 0, 0, 1, 1.72, 2.54,\n" \
"CUBIC_TO, 9.79, 13.85, 10, 14.89, 10, 16,\n" \
"H_LINE_TO, 8.5,\n" \
"R_ARC_TO, 6.3, 6.3, 0, 0, 0, -0.51, -2.53,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 0, -1.39, -2.07,\n" \
"R_ARC_TO, 6.56, 6.56, 0, 0, 0, -2.07, -1.39,\n" \
"ARC_TO, 6.3, 6.3, 0, 0, 0, 2, 9.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 14.5, 8,\n" \
"R_H_LINE_TO, -5,\n" \
"R_CUBIC_TO, 0, -0.25, -0.01, -0.5, -0.03, -0.75,\n" \
"R_CUBIC_TO, -0.02, -0.25, -0.05, -0.5, -0.09, -0.75,\n" \
"H_LINE_TO, 16.5,\n" \
"R_V_LINE_TO, -3.87,\n" \
"R_CUBIC_TO, 0.28, -0.1, 0.54, -0.23, 0.79, -0.37,\n" \
"ARC_TO, 4.05, 4.05, 0, 0, 0, 18, 9.75,\n" \
"R_V_LINE_TO, 4.75,\n" \
"R_CUBIC_TO, 0, 0.42, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 16,\n" \
"CLOSE,\n" \
"MOVE_TO, 2, 6.5,\n" \
"R_V_LINE_TO, -1,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 4,\n" \
"R_H_LINE_TO, 5.71,\n" \
"R_ARC_TO, 5.64, 5.64, 0, 0, 0, -0.16, 0.73,\n" \
"CUBIC_TO, 9.02, 4.98, 9, 5.24, 9, 5.5,\n" \
"H_LINE_TO, 3.5,\n" \
"R_V_LINE_TO, 1.13,\n" \
"R_CUBIC_TO, -0.25, -0.04, -0.5, -0.07, -0.75, -0.09,\n" \
"R_CUBIC_TO, -0.25, -0.02, -0.5, -0.03, -0.75, -0.03,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12.5, 3,\n" \
"R_CUBIC_TO, -1.11, 0, -2.05, -0.39, -2.83, -1.17,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 1, 10.5, 5.5,\n" \
"R_CUBIC_TO, 0, -1.11, 0.39, -2.05, 1.17, -2.83,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 1, 14.5, 1.5,\n" \
"R_CUBIC_TO, 1.11, 0, 2.05, 0.39, 2.83, 1.17,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 1, 18.5, 5.5,\n" \
"R_CUBIC_TO, 0, 1.11, -0.39, 2.05, -1.17, 2.83,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 1, 14.5, 9.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2,\n" \
"R_CUBIC_TO, 0.13, 0, 0.25, -0.05, 0.35, -0.15,\n" \
"ARC_TO, 0.49, 0.49, 0, 0, 0, 15, 7,\n" \
"R_CUBIC_TO, 0, -0.13, -0.05, -0.25, -0.15, -0.35,\n" \
"ARC_TO, 0.49, 0.49, 0, 0, 0, 14.5, 6.5,\n" \
"R_CUBIC_TO, -0.13, 0, -0.25, 0.05, -0.35, 0.15,\n" \
"ARC_TO, 0.49, 0.49, 0, 0, 0, 14, 7,\n" \
"R_CUBIC_TO, 0, 0.13, 0.05, 0.25, 0.15, 0.35,\n" \
"R_ARC_TO, 0.49, 0.49, 0, 0, 0, 0.35, 0.15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.5, -2,\n" \
"R_H_LINE_TO, 1,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, -1,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaRouterWarningChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaSeekBackwardIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 11, 18,\n" \
"V_LINE_TO, 6,\n" \
"R_LINE_TO, -8.5, 6,\n" \
"R_LINE_TO, 8.5, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.5, -6,\n" \
"R_LINE_TO, 8.5, 6,\n" \
"V_LINE_TO, 6,\n" \
"R_LINE_TO, -8.5, 6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaSeekBackwardIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMediaSeekForwardIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 4, 18,\n" \
"R_LINE_TO, 8.5, -6,\n" \
"LINE_TO, 4, 6,\n" \
"R_V_LINE_TO, 12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 9, -12,\n" \
"R_V_LINE_TO, 12,\n" \
"R_LINE_TO, 8.5, -6,\n" \
"LINE_TO, 13, 6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMediaSeekForwardIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMicIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 24, 30,\n" \
"R_CUBIC_TO, 3.31, 0, 5.98, -2.69, 5.98, -6,\n" \
"LINE_TO, 30, 12,\n" \
"R_CUBIC_TO, 0, -3.32, -2.68, -6, -6, -6,\n" \
"R_CUBIC_TO, -3.31, 0, -6, 2.68, -6, 6,\n" \
"R_V_LINE_TO, 12,\n" \
"R_CUBIC_TO, 0, 3.31, 2.69, 6, 6, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 10.6, -6,\n" \
"R_CUBIC_TO, 0, 6, -5.07, 10.2, -10.6, 10.2,\n" \
"R_CUBIC_TO, -5.52, 0, -10.6, -4.2, -10.6, -10.2,\n" \
"H_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0, 6.83, 5.44, 12.47, 12, 13.44,\n" \
"V_LINE_TO, 44,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -6.56,\n" \
"R_CUBIC_TO, 6.56, -0.97, 12, -6.61, 12, -13.44,\n" \
"R_H_LINE_TO, -3.4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMicIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMicChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 12,\n" \
"R_ARC_TO, 2.4, 2.4, 0, 0, 1, -1.77, -0.73,\n" \
"ARC_TO, 2.4, 2.4, 0, 0, 1, 7.5, 9.5,\n" \
"R_V_LINE_TO, -5,\n" \
"R_CUBIC_TO, 0, -0.69, 0.24, -1.28, 0.73, -1.77,\n" \
"ARC_TO, 2.4, 2.4, 0, 0, 1, 10, 2,\n" \
"R_CUBIC_TO, 0.7, 0, 1.29, 0.24, 1.77, 0.73,\n" \
"R_CUBIC_TO, 0.49, 0.49, 0.73, 1.08, 0.73, 1.77,\n" \
"R_V_LINE_TO, 5,\n" \
"R_CUBIC_TO, 0, 0.7, -0.24, 1.29, -0.73, 1.77,\n" \
"ARC_TO, 2.4, 2.4, 0, 0, 1, 10, 12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.75, 10,\n" \
"R_V_LINE_TO, -2.06,\n" \
"R_CUBIC_TO, -1.37, -0.18, -2.51, -0.78, -3.4, -1.81,\n" \
"CUBIC_TO, 4.95, 12.1, 4.5, 10.89, 4.5, 9.5,\n" \
"H_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, 1.11, 0.39, 2.05, 1.17, 2.83,\n" \
"R_CUBIC_TO, 0.78, 0.78, 1.73, 1.17, 2.83, 1.17,\n" \
"R_CUBIC_TO, 1.11, 0, 2.05, -0.39, 2.83, -1.17,\n" \
"CUBIC_TO, 13.61, 11.55, 14, 10.61, 14, 9.5,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0, 1.39, -0.45, 2.6, -1.35, 3.63,\n" \
"R_CUBIC_TO, -0.89, 1.03, -2.03, 1.63, -3.4, 1.81,\n" \
"V_LINE_TO, 17,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.75, -6.5,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, -0.1, 0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 11, 9.5,\n" \
"R_V_LINE_TO, -5,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.29, -0.71,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.71, -0.29,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.71, 0.29,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.29, 0.71,\n" \
"R_V_LINE_TO, 5,\n" \
"R_CUBIC_TO, 0, 0.28, 0.1, 0.52, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.43, 0.29, 0.71, 0.29,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMicChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMicOffIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 19, 11,\n" \
"R_H_LINE_TO, -2,\n" \
"R_CUBIC_TO, 0, 0.91, -0.25, 1.76, -0.68, 2.49,\n" \
"R_LINE_TO, 1.45, 1.45,\n" \
"CUBIC_TO, 18.54, 13.82, 19, 12.47, 19, 11,\n" \
"CLOSE,\n" \
"MOVE_TO, 2.81, 2.81,\n" \
"LINE_TO, 1.39, 4.22,\n" \
"R_LINE_TO, 11.66, 11.66,\n" \
"CUBIC_TO, 12.71, 15.96, 12.36, 16, 12, 16,\n" \
"R_CUBIC_TO, -2.76, 0, -5, -2.24, -5, -5,\n" \
"H_LINE_TO, 5,\n" \
"R_CUBIC_TO, 0, 3.53, 2.61, 6.43, 6, 6.92,\n" \
"V_LINE_TO, 21,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -3.08,\n" \
"R_CUBIC_TO, 0.57, -0.08, 1.12, -0.24, 1.64, -0.45,\n" \
"R_LINE_TO, 5.14, 5.14,\n" \
"R_LINE_TO, 1.41, -1.41,\n" \
"LINE_TO, 2.81, 2.81,\n" \
"CLOSE,\n" \
"MOVE_TO, 15, 11,\n" \
"V_LINE_TO, 5,\n" \
"R_CUBIC_TO, 0, -1.66, -1.34, -3, -3, -3,\n" \
"CUBIC_TO_SHORTHAND, 9, 3.34, 9, 5,\n" \
"R_V_LINE_TO, 1.17,\n" \
"R_LINE_TO, 5.81, 5.81,\n" \
"CUBIC_TO, 14.92, 11.67, 15, 11.35, 15, 11,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMicOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMicOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 14.61, 12.48,\n" \
"LINE_TO, 13.5, 11.38,\n" \
"R_CUBIC_TO, 0.15, -0.29, 0.27, -0.59, 0.36, -0.9,\n" \
"R_CUBIC_TO, 0.09, -0.31, 0.13, -0.64, 0.13, -0.97,\n" \
"H_LINE_TO, 15.5,\n" \
"R_CUBIC_TO, 0, 0.53, -0.08, 1.04, -0.23, 1.55,\n" \
"R_CUBIC_TO, -0.15, 0.51, -0.37, 0.98, -0.66, 1.43,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.94, 7.86,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.44, 2.4,\n" \
"LINE_TO, 11, 8.88,\n" \
"V_LINE_TO, 4.5,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.29, -0.71,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.71, -0.29,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.71, 0.29,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 0, -0.29, 0.71,\n" \
"R_V_LINE_TO, 2.38,\n" \
"LINE_TO, 7.5, 5.37,\n" \
"V_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, 0, -0.69, 0.24, -1.28, 0.73, -1.77,\n" \
"ARC_TO, 2.4, 2.4, 0, 0, 1, 10, 2,\n" \
"R_CUBIC_TO, 0.7, 0, 1.29, 0.24, 1.77, 0.73,\n" \
"R_CUBIC_TO, 0.49, 0.49, 0.73, 1.08, 0.73, 1.77,\n" \
"R_V_LINE_TO, 5,\n" \
"R_ARC_TO, 2.43, 2.43, 0, 0, 1, -0.12, 0.75,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.25, 17,\n" \
"R_V_LINE_TO, -2.06,\n" \
"R_CUBIC_TO, -1.37, -0.17, -2.51, -0.76, -3.41, -1.8,\n" \
"CUBIC_TO, 4.95, 12.1, 4.5, 10.89, 4.5, 9.5,\n" \
"H_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, 1.11, 0.39, 2.05, 1.17, 2.83,\n" \
"CUBIC_TO, 7.95, 13.11, 8.89, 13.5, 10, 13.5,\n" \
"R_CUBIC_TO, 0.47, 0, 0.92, -0.09, 1.34, -0.27,\n" \
"R_ARC_TO, 4.73, 4.73, 0, 0, 0, 1.18, -0.73,\n" \
"R_LINE_TO, 1.09, 1.08,\n" \
"R_ARC_TO, 6.51, 6.51, 0, 0, 1, -1.33, 0.89,\n" \
"R_ARC_TO, 4.8, 4.8, 0, 0, 1, -1.52, 0.47,\n" \
"V_LINE_TO, 17,\n" \
"CLOSE,\n" \
"MOVE_TO, 16, 18.13,\n" \
"LINE_TO, 1.88, 3.98,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.15,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMicOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMidiIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 19, 3,\n" \
"H_LINE_TO, 5,\n" \
"CUBIC_TO, 3.9, 3, 3, 3.9, 3, 5,\n" \
"R_V_LINE_TO, 14,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 14,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 5,\n" \
"CUBIC_TO, 21, 3.9, 20.1, 3, 19, 3,\n" \
"CLOSE,\n" \
"MOVE_TO, 14, 14.5,\n" \
"R_H_LINE_TO, 0.25,\n" \
"V_LINE_TO, 19,\n" \
"R_H_LINE_TO, -4.5,\n" \
"R_V_LINE_TO, -4.5,\n" \
"H_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0.55, 0, 1, -0.45, 1, -1,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 8.5,\n" \
"CUBIC_TO, 13, 14.05, 13.45, 14.5, 14, 14.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 5, 5,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 8.5,\n" \
"R_CUBIC_TO, 0, 0.55, 0.45, 1, 1, 1,\n" \
"R_H_LINE_TO, 0.25,\n" \
"V_LINE_TO, 19,\n" \
"H_LINE_TO, 5,\n" \
"V_LINE_TO, 5,\n" \
"CLOSE,\n" \
"MOVE_TO, 19, 19,\n" \
"R_H_LINE_TO, -3.25,\n" \
"R_V_LINE_TO, -4.5,\n" \
"H_LINE_TO, 16,\n" \
"R_CUBIC_TO, 0.55, 0, 1, -0.45, 1, -1,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 2,\n" \
"V_LINE_TO, 19,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMidiIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMidiChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 17,\n" \
"R_CUBIC_TO, -0.4, 0, -0.75, -0.15, -1.05, -0.45,\n" \
"R_CUBIC_TO, -0.3, -0.3, -0.45, -0.65, -0.45, -1.05,\n" \
"R_V_LINE_TO, -11,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.45, -1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 4.5, 3,\n" \
"R_H_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0, 0.4, -0.15, 0.75, -0.44, 1.05,\n" \
"R_CUBIC_TO, -0.29, 0.3, -0.64, 0.45, -1.06, 0.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"H_LINE_TO, 7,\n" \
"V_LINE_TO, 12,\n" \
"R_H_LINE_TO, -0.5,\n" \
"R_ARC_TO, 0.47, 0.47, 0, 0, 1, -0.35, -0.14,\n" \
"ARC_TO, 0.47, 0.47, 0, 0, 1, 6, 11.5,\n" \
"R_V_LINE_TO, -7,\n" \
"H_LINE_TO, 4.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.5, 0,\n" \
"R_H_LINE_TO, 2.5,\n" \
"R_V_LINE_TO, -11,\n" \
"H_LINE_TO, 14,\n" \
"R_V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, 0.14, -0.05, 0.26, -0.14, 0.36,\n" \
"ARC_TO, 0.47, 0.47, 0, 0, 1, 13.5, 12,\n" \
"H_LINE_TO, 13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5, 0,\n" \
"R_H_LINE_TO, 4,\n" \
"V_LINE_TO, 12,\n" \
"R_H_LINE_TO, -0.5,\n" \
"R_ARC_TO, 0.47, 0.47, 0, 0, 1, -0.35, -0.14,\n" \
"ARC_TO, 0.47, 0.47, 0, 0, 1, 11, 11.5,\n" \
"R_V_LINE_TO, -7,\n" \
"H_LINE_TO, 9,\n" \
"R_V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, 0.14, -0.05, 0.26, -0.14, 0.36,\n" \
"ARC_TO, 0.47, 0.47, 0, 0, 1, 8.5, 12,\n" \
"H_LINE_TO, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMidiChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseMidiOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16, 18.13,\n" \
"LINE_TO, 14.88, 17,\n" \
"H_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3, 15.5,\n" \
"V_LINE_TO, 5.13,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1, -3.25,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 4.5,\n" \
"H_LINE_TO, 14,\n" \
"R_V_LINE_TO, 7,\n" \
"R_ARC_TO, 0.43, 0.43, 0, 0, 1, -0.1, 0.27,\n" \
"LINE_TO, 11, 8.88,\n" \
"V_LINE_TO, 4.5,\n" \
"H_LINE_TO, 9,\n" \
"R_V_LINE_TO, 2.38,\n" \
"LINE_TO, 5.13, 3,\n" \
"H_LINE_TO, 15.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5, 15.5,\n" \
"H_LINE_TO, 7,\n" \
"V_LINE_TO, 12,\n" \
"R_H_LINE_TO, -0.5,\n" \
"R_ARC_TO, 0.47, 0.47, 0, 0, 1, -0.35, -0.14,\n" \
"ARC_TO, 0.47, 0.47, 0, 0, 1, 6, 11.5,\n" \
"V_LINE_TO, 8.13,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.5, 0,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -1.37,\n" \
"R_LINE_TO, -3, -3,\n" \
"R_V_LINE_TO, 0.38,\n" \
"R_ARC_TO, 0.48, 0.48, 0, 0, 1, -0.14, 0.36,\n" \
"ARC_TO, 0.48, 0.48, 0, 0, 1, 8.5, 12,\n" \
"H_LINE_TO, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5, -0.04,\n" \
"R_H_LINE_TO, 0.19,\n" \
"LINE_TO, 13, 15.27,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kMidiOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseNoEncryptionIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 12.93, 11.06,\n" \
"LINE_TO, 11.55, 9.67,\n" \
"V_LINE_TO, 6.84,\n" \
"H_LINE_TO, 8.72,\n" \
"LINE_TO, 7.34, 5.45,\n" \
"R_H_LINE_TO, 2.54,\n" \
"V_LINE_TO, 3.97,\n" \
"R_CUBIC_TO, 0, -0.52, -0.18, -0.97, -0.54, -1.34,\n" \
"R_ARC_TO, 1.81, 1.81, 0, 0, 0, -1.34, -0.55,\n" \
"R_ARC_TO, 1.82, 1.82, 0, 0, 0, -1.34, 0.55,\n" \
"R_ARC_TO, 1.81, 1.81, 0, 0, 0, -0.55, 1.33,\n" \
"R_V_LINE_TO, 0.27,\n" \
"LINE_TO, 4.92, 3.04,\n" \
"ARC_TO, 3.15, 3.15, 0, 0, 1, 8, 0.7,\n" \
"R_CUBIC_TO, 0.91, 0, 1.68, 0.32, 2.31, 0.95,\n" \
"R_CUBIC_TO, 0.63, 0.63, 0.95, 1.4, 0.95, 2.31,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, 0.29,\n" \
"R_CUBIC_TO, 0.38, 0, 0.71, 0.13, 0.98, 0.41,\n" \
"R_CUBIC_TO, 0.27, 0.27, 0.4, 0.59, 0.4, 0.98,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.53, 4.09,\n" \
"R_LINE_TO, -0.98, -0.99,\n" \
"R_ARC_TO, 1.49, 1.49, 0, 0, 1, -0.4, 0.25,\n" \
"R_ARC_TO, 1.23, 1.23, 0, 0, 1, -0.53, 0.11,\n" \
"H_LINE_TO, 4.44,\n" \
"R_ARC_TO, 1.35, 1.35, 0, 0, 1, -0.98, -0.4,\n" \
"R_ARC_TO, 1.26, 1.26, 0, 0, 1, -0.39, -0.98,\n" \
"V_LINE_TO, 6.84,\n" \
"R_CUBIC_TO, 0, -0.3, 0.08, -0.56, 0.24, -0.79,\n" \
"R_CUBIC_TO, 0.16, -0.23, 0.36, -0.39, 0.59, -0.48,\n" \
"LINE_TO, 0.93, 2.61,\n" \
"R_LINE_TO, 0.89, -0.89,\n" \
"R_LINE_TO, 12.53, 12.55,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.01, -2.01,\n" \
"R_LINE_TO, -2.49, -2.48,\n" \
"R_ARC_TO, 1.05, 1.05, 0, 0, 1, -0.41, 0.39,\n" \
"R_ARC_TO, 1.18, 1.18, 0, 0, 1, -1.4, -0.21,\n" \
"R_ARC_TO, 1.16, 1.16, 0, 0, 1, -0.35, -0.85,\n" \
"R_CUBIC_TO, 0, -0.19, 0.04, -0.37, 0.13, -0.54,\n" \
"R_CUBIC_TO, 0.08, -0.17, 0.22, -0.32, 0.39, -0.44,\n" \
"LINE_TO, 5.16, 6.84,\n" \
"H_LINE_TO, 4.45,\n" \
"R_V_LINE_TO, 6.31,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.14, 9.82,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.94, -1.62,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kNoEncryptionIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseNotSecureWarningIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"MOVE_TO, 4, 27,\n" \
"R_H_LINE_TO, 25,\n" \
"LINE_TO, 16.5, 5,\n" \
"LINE_TO, 4, 27,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 14, -3,\n" \
"R_H_LINE_TO, -3,\n" \
"R_V_LINE_TO, -3,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, 3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -5,\n" \
"R_H_LINE_TO, -3,\n" \
"R_V_LINE_TO, -6,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, 6,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 0.5, 14,\n" \
"R_H_LINE_TO, 15,\n" \
"LINE_TO, 8, 1,\n" \
"LINE_TO, 0.5, 14,\n" \
"CLOSE,\n" \
"MOVE_TO, 9, 12,\n" \
"H_LINE_TO, 7,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"H_LINE_TO, 7,\n" \
"V_LINE_TO, 6,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 3,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kNotSecureWarningIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseNotSecureWarningChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 1, 17,\n" \
"R_LINE_TO, 9, -15,\n" \
"R_LINE_TO, 9, 15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.65, -1.5,\n" \
"R_H_LINE_TO, 12.71,\n" \
"LINE_TO, 10, 4.92,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6.35, -1,\n" \
"R_CUBIC_TO, 0.21, 0, 0.39, -0.07, 0.54, -0.21,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, 0.22, -0.53,\n" \
"R_CUBIC_TO, 0, -0.21, -0.07, -0.39, -0.21, -0.53,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, -0.53, -0.22,\n" \
"R_CUBIC_TO, -0.21, 0, -0.39, 0.07, -0.53, 0.22,\n" \
"R_ARC_TO, 0.72, 0.72, 0, 0, 0, -0.22, 0.53,\n" \
"R_CUBIC_TO, 0, 0.21, 0.07, 0.39, 0.22, 0.54,\n" \
"R_CUBIC_TO, 0.14, 0.14, 0.32, 0.22, 0.53, 0.22,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.25, 12,\n" \
"R_H_LINE_TO, 1.5,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, -1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.75, -1.79,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 0.68, 13.67,\n" \
"LINE_TO, 8, 1.47,\n" \
"R_LINE_TO, 7.32, 12.2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.33, -1.32,\n" \
"R_H_LINE_TO, 9.97,\n" \
"LINE_TO, 8, 4.05,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.98, -0.75,\n" \
"R_CUBIC_TO, 0.17, 0, 0.32, -0.06, 0.44, -0.18,\n" \
"R_ARC_TO, 0.6, 0.6, 0, 0, 0, 0.18, -0.44,\n" \
"R_ARC_TO, 0.62, 0.62, 0, 0, 0, -0.62, -0.62,\n" \
"R_ARC_TO, 0.6, 0.6, 0, 0, 0, -0.44, 0.18,\n" \
"R_ARC_TO, 0.58, 0.58, 0, 0, 0, -0.18, 0.43,\n" \
"R_ARC_TO, 0.62, 0.62, 0, 0, 0, 0.62, 0.63,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.6, -1.99,\n" \
"R_H_LINE_TO, 1.2,\n" \
"V_LINE_TO, 6.45,\n" \
"H_LINE_TO, 7.4,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 8.2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kNotSecureWarningChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseNotSecureWarningOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 14.48,\n" \
"R_ARC_TO, 0.76, 0.76, 0, 0, 1, -0.55, -0.22,\n" \
"R_ARC_TO, 0.74, 0.74, 0, 0, 1, -0.23, -0.55,\n" \
"R_CUBIC_TO, 0, -0.21, 0.07, -0.4, 0.22, -0.55,\n" \
"R_ARC_TO, 0.73, 0.73, 0, 0, 1, 0.55, -0.23,\n" \
"R_CUBIC_TO, 0.22, 0, 0.4, 0.08, 0.55, 0.22,\n" \
"R_ARC_TO, 0.73, 0.73, 0, 0, 1, 0.23, 0.55,\n" \
"R_ARC_TO, 0.76, 0.76, 0, 0, 1, -0.22, 0.55,\n" \
"R_ARC_TO, 0.73, 0.73, 0, 0, 1, -0.55, 0.23,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.25, 12,\n" \
"V_LINE_TO, 9.25,\n" \
"R_LINE_TO, 1.5, 1.5,\n" \
"V_LINE_TO, 12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 9.78, 4.91,\n" \
"R_LINE_TO, -4.8, -4.8,\n" \
"LINE_TO, 10, 5.06,\n" \
"LINE_TO, 8.95, 6.82,\n" \
"LINE_TO, 7.73, 5.6,\n" \
"LINE_TO, 10, 1.83,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.77, 15.43,\n" \
"R_H_LINE_TO, 9.43,\n" \
"LINE_TO, 7.3, 9.54,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 13.31, 3.88,\n" \
"R_LINE_TO, -2.22, -2.22,\n" \
"R_H_LINE_TO, -14,\n" \
"LINE_TO, 6.11, 8.34,\n" \
"LINE_TO, 0.7, 2.93,\n" \
"R_LINE_TO, 1.11, -1.11,\n" \
"R_LINE_TO, 16.38, 16.38,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -6.82, -6.82,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.34, -3.03,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kNotSecureWarningOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseNotUploadedIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 16.5, 18.83,\n" \
"LINE_TO, 14.29, 16.67,\n" \
"H_LINE_TO, 5.42,\n" \
"R_CUBIC_TO, -1.28, 0, -2.36, -0.44, -3.25, -1.33,\n" \
"CUBIC_TO, 1.28, 14.44, 0.83, 13.36, 0.83, 12.08,\n" \
"R_CUBIC_TO, 0, -1.07, 0.33, -2.02, 0.98, -2.85,\n" \
"R_CUBIC_TO, 0.67, -0.83, 1.52, -1.37, 2.56, -1.6,\n" \
"R_CUBIC_TO, 0.04, -0.11, 0.08, -0.21, 0.13, -0.31,\n" \
"R_CUBIC_TO, 0.04, -0.11, 0.08, -0.23, 0.13, -0.35,\n" \
"LINE_TO, 1.17, 3.5,\n" \
"R_LINE_TO, 1.17, -1.17,\n" \
"R_LINE_TO, 15.33, 15.33,\n" \
"R_LINE_TO, -1.17, 1.17,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.42, 15,\n" \
"R_H_LINE_TO, 7.21,\n" \
"LINE_TO, 5.92, 8.29,\n" \
"R_ARC_TO, 5.52, 5.52, 0, 0, 0, -0.06, 0.44,\n" \
"ARC_TO, 4.4, 4.4, 0, 0, 0, 5.83, 9.17,\n" \
"R_H_LINE_TO, -0.42,\n" \
"R_CUBIC_TO, -0.81, 0, -1.49, 0.28, -2.06, 0.85,\n" \
"ARC_TO, 2.81, 2.81, 0, 0, 0, 2.5, 12.08,\n" \
"R_CUBIC_TO, 0, 0.81, 0.28, 1.49, 0.85, 2.06,\n" \
"ARC_TO, 2.81, 2.81, 0, 0, 0, 5.42, 15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12.58, 0.63,\n" \
"R_LINE_TO, -1.21, -1.17,\n" \
"R_CUBIC_TO, 0.24, -0.19, 0.41, -0.42, 0.52, -0.67,\n" \
"R_CUBIC_TO, 0.13, -0.26, 0.19, -0.56, 0.19, -0.87,\n" \
"R_CUBIC_TO, 0, -0.58, -0.2, -1.08, -0.6, -1.48,\n" \
"CUBIC_TO, 16.49, 11.04, 16, 10.83, 15.42, 10.83,\n" \
"H_LINE_TO, 14.17,\n" \
"R_V_LINE_TO, -1.67,\n" \
"R_CUBIC_TO, 0, -1.15, -0.41, -2.13, -1.23, -2.94,\n" \
"CUBIC_TO, 12.13, 5.41, 11.15, 5, 10, 5,\n" \
"R_CUBIC_TO, -0.37, 0, -0.74, 0.05, -1.08, 0.15,\n" \
"R_ARC_TO, 3.48, 3.48, 0, 0, 0, -1, 0.42,\n" \
"R_LINE_TO, -1.21, -1.21,\n" \
"R_CUBIC_TO, 0.49, -0.33, 1, -0.58, 1.54, -0.75,\n" \
"ARC_TO, 5.63, 5.63, 0, 0, 1, 10, 3.33,\n" \
"R_CUBIC_TO, 1.63, 0, 3, 0.57, 4.13, 1.71,\n" \
"CUBIC_TO, 15.26, 6.17, 15.83, 7.54, 15.83, 9.17,\n" \
"R_CUBIC_TO, 0.96, 0.11, 1.75, 0.53, 2.38, 1.25,\n" \
"R_CUBIC_TO, 0.64, 0.71, 0.96, 1.54, 0.96, 2.5,\n" \
"R_ARC_TO, 3.73, 3.73, 0, 0, 1, -0.31, 1.52,\n" \
"ARC_TO, 3.66, 3.66, 0, 0, 1, 18, 15.63,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kNotUploadedIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseNotificationWarningIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 96,\n" \
"MOVE_TO, 48, 8,\n" \
"R_CUBIC_TO, 22.08, 0, 40, 17.92, 40, 40,\n" \
"CUBIC_TO_SHORTHAND, 70.08, 88, 48, 88,\n" \
"CUBIC_TO_SHORTHAND, 8, 70.08, 8, 48,\n" \
"CUBIC_TO_SHORTHAND, 25.92, 8, 48, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4, 20,\n" \
"R_V_LINE_TO, 24,\n" \
"R_H_LINE_TO, 8,\n" \
"V_LINE_TO, 28,\n" \
"R_H_LINE_TO, -8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 32,\n" \
"R_V_LINE_TO, 8,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, -8,\n" \
"R_H_LINE_TO, -8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kNotificationWarningIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseNotificationsIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 24, 44,\n" \
"R_CUBIC_TO, 2.21, 0, 4, -1.49, 4, -4,\n" \
"R_H_LINE_TO, -8,\n" \
"R_CUBIC_TO, 0, 2.21, 1.79, 4, 4, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12, -12,\n" \
"V_LINE_TO, 22,\n" \
"R_CUBIC_TO, 0, -6.15, -3.27, -11.28, -9, -12.64,\n" \
"V_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, -1.66, -1.34, -3, -3, -3,\n" \
"R_CUBIC_TO, -1.66, 0, -3, 1.34, -3, 3,\n" \
"R_V_LINE_TO, 1.36,\n" \
"R_CUBIC_TO, -5.73, 1.36, -9, 6.49, -9, 12.64,\n" \
"R_V_LINE_TO, 10,\n" \
"R_LINE_TO, -4, 4,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, 32,\n" \
"R_V_LINE_TO, -2,\n" \
"R_LINE_TO, -4, -4\n" \
; \
              CheckThatParsedElementsMatch(s, kNotificationsIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseNotificationsChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4, 15.5,\n" \
"V_LINE_TO, 14,\n" \
"R_H_LINE_TO, 1,\n" \
"V_LINE_TO, 9,\n" \
"R_CUBIC_TO, 0, -1.21, 0.37, -2.27, 1.11, -3.19,\n" \
"CUBIC_TO, 6.86, 4.9, 7.82, 4.33, 9, 4.11,\n" \
"V_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 10, 2,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.2, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_V_LINE_TO, 1.11,\n" \
"R_CUBIC_TO, 1.18, 0.22, 2.14, 0.79, 2.89, 1.71,\n" \
"CUBIC_TO, 14.63, 6.73, 15, 7.79, 15, 9,\n" \
"R_V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 1,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6, -5.75,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 18,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 8.5, 16.5,\n" \
"R_H_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, 0.42, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 10, 18,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.5, 14,\n" \
"R_H_LINE_TO, 7,\n" \
"V_LINE_TO, 9,\n" \
"R_CUBIC_TO, 0, -0.97, -0.34, -1.8, -1.02, -2.48,\n" \
"R_CUBIC_TO, -0.68, -0.68, -1.51, -1.02, -2.48, -1.02,\n" \
"R_CUBIC_TO, -0.97, 0, -1.8, 0.34, -2.48, 1.02,\n" \
"CUBIC_TO, 6.84, 7.2, 6.5, 8.03, 6.5, 9,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kNotificationsChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseNotificationsOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4, 15.5,\n" \
"V_LINE_TO, 14,\n" \
"R_H_LINE_TO, 1,\n" \
"V_LINE_TO, 9,\n" \
"R_CUBIC_TO, 0, -0.57, 0.09, -1.11, 0.28, -1.64,\n" \
"R_CUBIC_TO, 0.19, -0.52, 0.44, -1.01, 0.76, -1.47,\n" \
"LINE_TO, 7.15, 7,\n" \
"R_ARC_TO, 3.46, 3.46, 0, 0, 0, -0.48, 0.95,\n" \
"ARC_TO, 3.32, 3.32, 0, 0, 0, 6.5, 9,\n" \
"R_V_LINE_TO, 5,\n" \
"H_LINE_TO, 12,\n" \
"LINE_TO, 1.88, 3.88,\n" \
"R_LINE_TO, 1.06, -1.08,\n" \
"R_LINE_TO, 14.12, 14.15,\n" \
"LINE_TO, 16, 18,\n" \
"R_LINE_TO, -2.5, -2.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 11, -2.75,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 9,\n" \
"R_CUBIC_TO, 0, -0.97, -0.34, -1.8, -1.02, -2.48,\n" \
"R_CUBIC_TO, -0.68, -0.68, -1.51, -1.02, -2.48, -1.02,\n" \
"R_CUBIC_TO, -0.32, 0, -0.62, 0.04, -0.91, 0.13,\n" \
"R_ARC_TO, 4.05, 4.05, 0, 0, 0, -0.86, 0.36,\n" \
"LINE_TO, 7.15, 4.89,\n" \
"R_CUBIC_TO, 0.28, -0.2, 0.57, -0.37, 0.89, -0.49,\n" \
"ARC_TO, 5.53, 5.53, 0, 0, 1, 9, 4.11,\n" \
"V_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 10, 2,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.2, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_V_LINE_TO, 1.11,\n" \
"R_CUBIC_TO, 1.18, 0.22, 2.14, 0.79, 2.89, 1.71,\n" \
"CUBIC_TO, 14.63, 6.73, 15, 7.79, 15, 9,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5.75, -1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 18,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 8.5, 16.5,\n" \
"R_H_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, 0.42, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 10, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.88, -9.39,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kNotificationsOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePageInfoContentPasteIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 18,\n" \
"MOVE_TO, 6.71, 3.25,\n" \
"R_ARC_TO, 2.5, 2.5, 0, 0, 1, 4.58, 0,\n" \
"H_LINE_TO, 14,\n" \
"R_ARC_TO, 1, 1, 0, 0, 1, 1, 1,\n" \
"R_V_LINE_TO, 11,\n" \
"R_ARC_TO, 1, 1, 0, 0, 1, -1, 1,\n" \
"H_LINE_TO, 4,\n" \
"R_ARC_TO, 1, 1, 0, 0, 1, -1, -1,\n" \
"R_V_LINE_TO, -11,\n" \
"R_ARC_TO, 1, 1, 0, 0, 1, 1, -1,\n" \
"CLOSE,\n" \
"MOVE_TO, 9, 5.25,\n" \
"R_ARC_TO, 1, 1, 0, 1, 0, 0, -2,\n" \
"R_ARC_TO, 1, 1, 0, 0, 0, 0, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5, -0.5,\n" \
"R_V_LINE_TO, 10,\n" \
"R_H_LINE_TO, 10,\n" \
"R_V_LINE_TO, -10,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_ARC_TO, 1, 1, 0, 0, 1, -1, 1,\n" \
"H_LINE_TO, 7,\n" \
"R_ARC_TO, 1, 1, 0, 0, 1, -1, -1,\n" \
"R_V_LINE_TO, -1.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPageInfoContentPasteIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePasskeyIcon) { \
              std::string s = \
"// Copyright 2022 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 11, 12,\n" \
"R_QUADRATIC_TO, -1.65, 0, -2.81, -1.17,\n" \
"QUADRATIC_TO, 7.03, 9.65, 7.03, 8,\n" \
"R_QUADRATIC_TO, 0, -1.65, 1.16, -2.82,\n" \
"QUADRATIC_TO, 9.35, 4, 11, 4,\n" \
"R_QUADRATIC_TO, 1.65, 0, 2.83, 1.18,\n" \
"QUADRATIC_TO, 15, 6.35, 15, 8,\n" \
"R_QUADRATIC_TO, 0, 1.65, -1.17, 2.83,\n" \
"QUADRATIC_TO, 12.65, 12, 11, 12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7.25, 9,\n" \
"LINE_TO, 17, 19.5,\n" \
"R_V_LINE_TO, -3.67,\n" \
"R_QUADRATIC_TO, -0.87, -0.32, -1.44, -1.09,\n" \
"QUADRATIC_TO, 15, 13.98, 15, 13,\n" \
"R_QUADRATIC_TO, 0, -1.25, 0.88, -2.12,\n" \
"QUADRATIC_TO_SHORTHAND, 18, 10,\n" \
"R_QUADRATIC_TO, 1.25, 0, 2.13, 0.88,\n" \
"QUADRATIC_TO_SHORTHAND, 21, 13,\n" \
"R_QUADRATIC_TO, 0, 0.98, -0.56, 1.74,\n" \
"R_QUADRATIC_TO, -0.56, 0.76, -1.44, 1.09,\n" \
"V_LINE_TO, 16,\n" \
"R_LINE_TO, 1, 1,\n" \
"R_LINE_TO, -1, 1,\n" \
"R_LINE_TO, 1, 1,\n" \
"CLOSE,\n" \
"MOVE_TO, 3, 20,\n" \
"R_V_LINE_TO, -2.77,\n" \
"R_QUADRATIC_TO, 0, -0.85, 0.43, -1.57,\n" \
"R_QUADRATIC_TO, 0.43, -0.72, 1.18, -1.1,\n" \
"R_QUADRATIC_TO, 1.27, -0.65, 2.89, -1.1,\n" \
"QUADRATIC_TO, 9.1, 13, 11, 13,\n" \
"R_QUADRATIC_TO, 0.53, 0, 1.02, 0.04,\n" \
"R_QUADRATIC_TO, 0.5, 0.04, 1, 0.11,\n" \
"R_QUADRATIC_TO, 0.05, 1.13, 0.54, 2.15,\n" \
"R_QUADRATIC_TO, 0.49, 1.02, 1.44, 1.68,\n" \
"V_LINE_TO, 20,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 15, -5.5,\n" \
"R_QUADRATIC_TO, 0.63, 0, 1.06, -0.44,\n" \
"R_QUADRATIC_TO, 0.44, -0.44, 0.44, -1.06,\n" \
"R_QUADRATIC_TO, 0, -0.62, -0.44, -1.06,\n" \
"QUADRATIC_TO, 18.63, 11.5, 18, 11.5,\n" \
"R_QUADRATIC_TO, -0.63, 0, -1.06, 0.44,\n" \
"QUADRATIC_TO, 16.5, 12.38, 16.5, 13,\n" \
"R_QUADRATIC_TO, 0, 0.62, 0.44, 1.06,\n" \
"R_QUADRATIC_TO, 0.44, 0.44, 1.06, 0.44,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPasskeyIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePasswordManagerIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 6, 12,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, -0.19, 1.41, -0.59,\n" \
"CUBIC_TO, 7.8, 11.02, 8, 10.55, 8, 10,\n" \
"R_CUBIC_TO, 0, -0.55, -0.19, -1.02, -0.59, -1.41,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 0, 6, 8,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, 0.2, -1.41, 0.59,\n" \
"CUBIC_TO, 4.2, 8.98, 4, 9.45, 4, 10,\n" \
"R_CUBIC_TO, 0, 0.55, 0.2, 1.02, 0.59, 1.41,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.86, 0.59, 1.41, 0.59,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 3,\n" \
"R_CUBIC_TO, -1.39, 0, -2.57, -0.48, -3.54, -1.46,\n" \
"CUBIC_TO, 1.48, 12.57, 1, 11.39, 1, 10,\n" \
"R_CUBIC_TO, 0, -1.39, 0.48, -2.57, 1.46, -3.54,\n" \
"CUBIC_TO, 3.43, 5.48, 4.61, 5, 6, 5,\n" \
"R_CUBIC_TO, 1, 0, 1.91, 0.27, 2.73, 0.81,\n" \
"ARC_TO, 5.09, 5.09, 0, 0, 1, 10.58, 8,\n" \
"H_LINE_TO, 17.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 17.5, 15,\n" \
"R_H_LINE_TO, -3,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 13, 13.5,\n" \
"V_LINE_TO, 12,\n" \
"R_H_LINE_TO, -2.42,\n" \
"R_ARC_TO, 5.09, 5.09, 0, 0, 1, -1.85, 2.19,\n" \
"ARC_TO, 4.84, 4.84, 0, 0, 1, 6, 15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.46, -4.5,\n" \
"H_LINE_TO, 14.5,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 1,\n" \
"V_LINE_TO, 12,\n" \
"R_CUBIC_TO, 0, -0.14, 0.05, -0.27, 0.14, -0.36,\n" \
"R_ARC_TO, 0.49, 0.49, 0, 0, 1, 0.36, -0.14,\n" \
"R_CUBIC_TO, 0.14, 0, 0.27, 0.05, 0.36, 0.14,\n" \
"R_CUBIC_TO, 0.09, 0.09, 0.14, 0.22, 0.14, 0.36,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, 1,\n" \
"R_V_LINE_TO, -4,\n" \
"H_LINE_TO, 9.46,\n" \
"R_ARC_TO, 3.27, 3.27, 0, 0, 0, -1.16, -2.16,\n" \
"ARC_TO, 3.45, 3.45, 0, 0, 0, 6, 6.5,\n" \
"R_CUBIC_TO, -0.97, 0, -1.8, 0.34, -2.48, 1.02,\n" \
"CUBIC_TO, 2.84, 8.2, 2.5, 9.03, 2.5, 10,\n" \
"R_CUBIC_TO, 0, 0.97, 0.34, 1.8, 1.02, 2.48,\n" \
"R_CUBIC_TO, 0.68, 0.68, 1.51, 1.02, 2.48, 1.02,\n" \
"R_CUBIC_TO, 0.87, 0, 1.64, -0.28, 2.3, -0.84,\n" \
"ARC_TO, 3.27, 3.27, 0, 0, 0, 9.46, 10.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPasswordManagerIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePasswordManagerOffIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 16, 18.13,\n" \
"R_LINE_TO, -5.64, -5.64,\n" \
"R_ARC_TO, 4.96, 4.96, 0, 0, 1, -1.83, 1.84,\n" \
"ARC_TO, 4.96, 4.96, 0, 0, 1, 6, 15,\n" \
"R_CUBIC_TO, -1.39, 0, -2.57, -0.48, -3.54, -1.46,\n" \
"CUBIC_TO, 1.48, 12.57, 1, 11.39, 1, 10,\n" \
"R_CUBIC_TO, 0, -0.9, 0.22, -1.74, 0.67, -2.52,\n" \
"ARC_TO, 4.81, 4.81, 0, 0, 1, 3.52, 5.65,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 13.5,\n" \
"R_CUBIC_TO, 0.71, 0, 1.36, -0.19, 1.94, -0.57,\n" \
"R_ARC_TO, 3.53, 3.53, 0, 0, 0, 1.31, -1.57,\n" \
"LINE_TO, 4.64, 6.77,\n" \
"R_ARC_TO, 3.39, 3.39, 0, 0, 0, -1.57, 1.29,\n" \
"R_ARC_TO, 3.46, 3.46, 0, 0, 0, -0.57, 1.95,\n" \
"R_CUBIC_TO, 0, 0.96, 0.34, 1.78, 1.02, 2.47,\n" \
"ARC_TO, 3.38, 3.38, 0, 0, 0, 6, 13.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.63, -4,\n" \
"CLOSE,\n" \
"MOVE_TO, 19, 15,\n" \
"R_H_LINE_TO, -1.87,\n" \
"LINE_TO, 15, 12.88,\n" \
"R_V_LINE_TO, -0.62,\n" \
"R_CUBIC_TO, 0, -0.21, 0.07, -0.39, 0.22, -0.53,\n" \
"R_ARC_TO, 0.71, 0.71, 0, 0, 1, 0.53, -0.21,\n" \
"R_CUBIC_TO, 0.21, 0, 0.39, 0.07, 0.54, 0.22,\n" \
"R_ARC_TO, 0.73, 0.73, 0, 0, 1, 0.22, 0.54,\n" \
"R_V_LINE_TO, 1.25,\n" \
"R_H_LINE_TO, 1,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, -5.87,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"H_LINE_TO, 19,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 12,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, -0.19, -1.41, -0.59,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 4, 10,\n" \
"R_CUBIC_TO, 0, -0.55, 0.2, -1.02, 0.59, -1.41,\n" \
"CUBIC_TO, 4.98, 8.2, 5.45, 8, 6, 8,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, 0.2, 1.41, 0.59,\n" \
"CUBIC_TO, 7.8, 8.98, 8, 9.45, 8, 10,\n" \
"R_CUBIC_TO, 0, 0.55, -0.19, 1.02, -0.59, 1.41,\n" \
"CUBIC_TO, 7.02, 11.8, 6.55, 12, 6, 12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.87,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPasswordManagerOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePauseIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 12, 38,\n" \
"R_H_LINE_TO, 8,\n" \
"V_LINE_TO, 10,\n" \
"R_H_LINE_TO, -8,\n" \
"R_V_LINE_TO, 28,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 16, -28,\n" \
"R_V_LINE_TO, 28,\n" \
"R_H_LINE_TO, 8,\n" \
"V_LINE_TO, 10,\n" \
"R_H_LINE_TO, -8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPauseIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePauseChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 11, 16,\n" \
"V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -7, 0,\n" \
"V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.5, -1.5,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -9,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -7, 0,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -9,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -9,\n" \
"R_V_LINE_TO, 9,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7, 0,\n" \
"R_V_LINE_TO, 9,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPauseChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePersonCheckIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 14.48f, 10.5f,\n" \
"LINE_TO, 12, 8.02f,\n" \
"LINE_TO, 13.06f, 6.96f,\n" \
"LINE_TO, 14.48f, 8.38f,\n" \
"LINE_TO, 17.67f, 5.21f,\n" \
"LINE_TO, 18.73f, 6.25f,\n" \
"LINE_TO, 14.48f, 10.5f,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 10,\n" \
"CUBIC_TO, 7.17f, 10, 6.46f, 9.71f, 5.88f, 9.13f,\n" \
"CUBIC_TO, 5.29f, 8.54f, 5, 7.83f, 5, 7,\n" \
"CUBIC_TO, 5, 6.17f, 5.29f, 5.46f, 5.88f, 4.88f,\n" \
"CUBIC_TO, 6.46f, 4.29f, 7.17f, 4, 8, 4,\n" \
"CUBIC_TO, 8.83f, 4, 9.54f, 4.29f, 10.13f, 4.88f,\n" \
"CUBIC_TO, 10.71f, 5.46f, 11, 6.17f, 11, 7,\n" \
"CUBIC_TO, 11, 7.83f, 10.71f, 8.54f, 10.13f, 9.13f,\n" \
"CUBIC_TO, 9.54f, 9.71f, 8.83f, 10, 8, 10,\n" \
"CLOSE,\n" \
"MOVE_TO, 2, 16,\n" \
"V_LINE_TO, 14.08f,\n" \
"CUBIC_TO, 2, 13.72f, 2.08f, 13.4f, 2.25f, 13.1f,\n" \
"CUBIC_TO, 2.43f, 12.8f, 2.67f, 12.56f, 2.98f, 12.38f,\n" \
"CUBIC_TO, 3.74f, 11.93f, 4.55f, 11.59f, 5.4f, 11.35f,\n" \
"CUBIC_TO, 6.24f, 11.12f, 7.11f, 11, 8, 11,\n" \
"CUBIC_TO, 8.89f, 11, 9.76f, 11.12f, 10.6f, 11.35f,\n" \
"CUBIC_TO, 11.45f, 11.59f, 12.26f, 11.93f, 13.02f, 12.38f,\n" \
"CUBIC_TO, 13.33f, 12.56f, 13.56f, 12.8f, 13.73f, 13.1f,\n" \
"CUBIC_TO, 13.91f, 13.4f, 14, 13.72f, 14, 14.08f,\n" \
"V_LINE_TO, 16,\n" \
"H_LINE_TO, 2,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.5f, 14.5f,\n" \
"H_LINE_TO, 12.5f,\n" \
"V_LINE_TO, 14.08f,\n" \
"CUBIC_TO, 12.5f, 14, 12.48f, 13.92f, 12.44f, 13.85f,\n" \
"CUBIC_TO, 12.4f, 13.77f, 12.34f, 13.71f, 12.27f, 13.67f,\n" \
"CUBIC_TO, 11.62f, 13.29f, 10.93f, 13.01f, 10.21f, 12.81f,\n" \
"CUBIC_TO, 9.49f, 12.6f, 8.75f, 12.5f, 8, 12.5f,\n" \
"CUBIC_TO, 7.25f, 12.5f, 6.51f, 12.6f, 5.79f, 12.81f,\n" \
"CUBIC_TO, 5.07f, 13.01f, 4.38f, 13.29f, 3.73f, 13.67f,\n" \
"CUBIC_TO, 3.66f, 13.71f, 3.6f, 13.77f, 3.56f, 13.85f,\n" \
"CUBIC_TO, 3.52f, 13.92f, 3.5f, 14, 3.5f, 14.08f,\n" \
"V_LINE_TO, 14.5f,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 8.5f,\n" \
"CUBIC_TO, 8.42f, 8.5f, 8.77f, 8.35f, 9.06f, 8.06f,\n" \
"CUBIC_TO, 9.35f, 7.77f, 9.5f, 7.42f, 9.5f, 7,\n" \
"CUBIC_TO, 9.5f, 6.58f, 9.35f, 6.23f, 9.06f, 5.94f,\n" \
"CUBIC_TO, 8.77f, 5.65f, 8.42f, 5.5f, 8, 5.5f,\n" \
"CUBIC_TO, 7.58f, 5.5f, 7.23f, 5.65f, 6.94f, 5.94f,\n" \
"CUBIC_TO, 6.65f, 6.23f, 6.5f, 6.58f, 6.5f, 7,\n" \
"CUBIC_TO, 6.5f, 7.42f, 6.65f, 7.77f, 6.94f, 8.06f,\n" \
"CUBIC_TO, 7.23f, 8.35f, 7.58f, 8.5f, 8, 8.5f,\n" \
"CLOSE,\n" \
"NEW_PATH\n" \
; \
              CheckThatParsedElementsMatch(s, kPersonCheckIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePhotoIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 18,\n" \
"MOVE_TO, 15, 13.67,\n" \
"V_LINE_TO, 4.33,\n" \
"CUBIC_TO, 15, 3.6, 14.4, 3, 13.67, 3,\n" \
"H_LINE_TO, 4.33,\n" \
"CUBIC_TO, 3.6, 3, 3, 3.6, 3, 4.33,\n" \
"R_V_LINE_TO, 9.33,\n" \
"CUBIC_TO, 3, 14.4, 3.6, 15, 4.33, 15,\n" \
"R_H_LINE_TO, 9.33,\n" \
"CUBIC_TO, 14.4, 15, 15, 14.4, 15, 13.67,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.67, 10,\n" \
"R_LINE_TO, 1.67, 2.01,\n" \
"LINE_TO, 10.67, 9,\n" \
"R_LINE_TO, 3, 4,\n" \
"H_LINE_TO, 4.33,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPhotoIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePhotoChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 17,\n" \
"R_ARC_TO, 1.42, 1.42, 0, 0, 1, -1.06, -0.45,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 3, 15.5,\n" \
"R_V_LINE_TO, -11,\n" \
"R_CUBIC_TO, 0, -0.4, 0.15, -0.75, 0.44, -1.05,\n" \
"CUBIC_TO, 3.73, 3.15, 4.09, 3, 4.5, 3,\n" \
"R_H_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.45,\n" \
"R_CUBIC_TO, 0.29, 0.3, 0.44, 0.65, 0.44, 1.05,\n" \
"R_V_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0, 0.4, -0.15, 0.75, -0.44, 1.05,\n" \
"R_CUBIC_TO, -0.29, 0.3, -0.64, 0.45, -1.06, 0.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 11,\n" \
"R_V_LINE_TO, -11,\n" \
"R_H_LINE_TO, -11,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1, -1.5,\n" \
"R_H_LINE_TO, 9,\n" \
"R_LINE_TO, -3, -4,\n" \
"R_LINE_TO, -2.25, 3,\n" \
"R_LINE_TO, -1.5, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1, 1.5,\n" \
"R_V_LINE_TO, -11,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPhotoChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePhotoOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 17, 14.88,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 4.5,\n" \
"H_LINE_TO, 6.63,\n" \
"LINE_TO, 5.13, 3,\n" \
"H_LINE_TO, 15.5,\n" \
"R_CUBIC_TO, 0.4, 0, 0.75, 0.15, 1.05, 0.45,\n" \
"R_CUBIC_TO, 0.3, 0.3, 0.45, 0.65, 0.45, 1.05,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1, 3.25,\n" \
"LINE_TO, 14.88, 17,\n" \
"H_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, -0.4, 0, -0.75, -0.15, -1.05, -0.45,\n" \
"R_CUBIC_TO, -0.3, -0.3, -0.45, -0.65, -0.45, -1.05,\n" \
"V_LINE_TO, 5.13,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.5, 14,\n" \
"R_LINE_TO, 2.25, -3,\n" \
"R_LINE_TO, 1.5, 2,\n" \
"R_LINE_TO, 0.71, -0.94,\n" \
"LINE_TO, 4.5, 6.63,\n" \
"V_LINE_TO, 15.5,\n" \
"R_H_LINE_TO, 8.88,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 11, 9,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.12, 2.13,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPhotoOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePictureInPictureIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 4, 20,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, -0.19, -1.41, -0.59,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 2, 18,\n" \
"V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, -0.55, 0.2, -1.02, 0.59, -1.41,\n" \
"CUBIC_TO, 2.98, 4.2, 3.45, 4, 4, 4,\n" \
"R_H_LINE_TO, 16,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, 0.2, 1.41, 0.59,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.59, 0.86, 0.59, 1.41,\n" \
"R_V_LINE_TO, 12,\n" \
"R_CUBIC_TO, 0, 0.55, -0.19, 1.02, -0.59, 1.41,\n" \
"R_CUBIC_TO, -0.39, 0.39, -0.86, 0.59, -1.41, 0.59,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2,\n" \
"R_H_LINE_TO, 16,\n" \
"V_LINE_TO, 6,\n" \
"H_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7, -5,\n" \
"R_H_LINE_TO, 8,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, -8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2, -2,\n" \
"V_LINE_TO, 9,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPictureInPictureIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePictureInPictureAltIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"MOVE_TO, 160, 800,\n" \
"R_QUADRATIC_TO, -33, 0, -56.5, -23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 80, 720,\n" \
"R_V_LINE_TO, -480,\n" \
"R_QUADRATIC_TO, 0, -33, 23.5, -56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 160, 160,\n" \
"R_H_LINE_TO, 640,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 880, 240,\n" \
"R_V_LINE_TO, 480,\n" \
"R_QUADRATIC_TO, 0, 33, -23.5, 56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 800, 800,\n" \
"H_LINE_TO, 160,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -80,\n" \
"R_H_LINE_TO, 640,\n" \
"R_V_LINE_TO, -480,\n" \
"H_LINE_TO, 160,\n" \
"R_V_LINE_TO, 480,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"R_V_LINE_TO, -480,\n" \
"R_V_LINE_TO, 480,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 280, -40,\n" \
"R_H_LINE_TO, 320,\n" \
"R_V_LINE_TO, -240,\n" \
"H_LINE_TO, 440,\n" \
"R_V_LINE_TO, 240,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 80, -80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, 160,\n" \
"R_V_LINE_TO, 80,\n" \
"H_LINE_TO, 520,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPictureInPictureAltIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePipExitIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 160, 800,\n" \
"R_QUADRATIC_TO, -33, 0, -56.5, -23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 80, 720,\n" \
"R_V_LINE_TO, -280,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, 280,\n" \
"R_H_LINE_TO, 640,\n" \
"R_V_LINE_TO, -480,\n" \
"H_LINE_TO, 440,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, 360,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 880, 240,\n" \
"R_V_LINE_TO, 480,\n" \
"R_QUADRATIC_TO, 0, 33, -23.5, 56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 800, 800,\n" \
"H_LINE_TO, 160,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 523, -140,\n" \
"R_LINE_TO, 57, -57,\n" \
"R_LINE_TO, -124, -123,\n" \
"R_H_LINE_TO, 104,\n" \
"R_V_LINE_TO, -80,\n" \
"H_LINE_TO, 480,\n" \
"R_V_LINE_TO, 240,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -103,\n" \
"R_LINE_TO, 123, 123,\n" \
"CLOSE,\n" \
"MOVE_TO, 80, 360,\n" \
"R_V_LINE_TO, -200,\n" \
"R_H_LINE_TO, 280,\n" \
"R_V_LINE_TO, 200,\n" \
"H_LINE_TO, 80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 400, 120,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPipExitIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePlayArrowIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"FLIPS_IN_RTL,\n" \
"MOVE_TO, 16, 10,\n" \
"R_V_LINE_TO, 28,\n" \
"R_LINE_TO, 22, -14,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPlayArrowIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePlayArrowChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 7, 15.5,\n" \
"R_V_LINE_TO, -11,\n" \
"R_LINE_TO, 8.5, 5.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.52, -5.52,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.02, 2.77,\n" \
"LINE_TO, 12.75, 10,\n" \
"LINE_TO, 8.5, 7.25,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPlayArrowChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePointerLockIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 480, 440,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 440,\n" \
"R_QUADRATIC_TO, -116, 0, -198, -82,\n" \
"R_QUADRATIC_TO, -82, -82, -82, -198,\n" \
"R_V_LINE_TO, -240,\n" \
"R_QUADRATIC_TO, 0, -116, 82, -198,\n" \
"R_QUADRATIC_TO, 82, -82, 198, -82,\n" \
"R_QUADRATIC_TO, 116, 0, 198, 82,\n" \
"R_QUADRATIC_TO, 82, 82, 82, 198,\n" \
"R_V_LINE_TO, 124,\n" \
"R_QUADRATIC_TO, -20, -4, -40, -4,\n" \
"R_QUADRATIC_TO, -20, 0, -40, 4,\n" \
"R_V_LINE_TO, -44,\n" \
"H_LINE_TO, 280,\n" \
"R_V_LINE_TO, 160,\n" \
"R_QUADRATIC_TO, 0, 83, 58.5, 141.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 800,\n" \
"R_QUADRATIC_TO, 11, 0, 20.5, -1,\n" \
"R_QUADRATIC_TO, 9.5, -1, 19.5, -3,\n" \
"R_V_LINE_TO, 81,\n" \
"R_QUADRATIC_TO, -10, 2, -19.5, 2.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 880,\n" \
"CLOSE,\n" \
"MOVE_TO, 280, 360,\n" \
"R_H_LINE_TO, 160,\n" \
"R_V_LINE_TO, -196,\n" \
"R_QUADRATIC_TO, -69, 14, -114.5, 69,\n" \
"QUADRATIC_TO_SHORTHAND, 280, 360,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 240, 0,\n" \
"R_H_LINE_TO, 160,\n" \
"R_QUADRATIC_TO, 0, -72, -45.5, -127,\n" \
"QUADRATIC_TO_SHORTHAND, 520, 164,\n" \
"R_V_LINE_TO, 196,\n" \
"CLOSE,\n" \
"MOVE_TO, 634, 880,\n" \
"R_QUADRATIC_TO, -14, 0, -24, -10,\n" \
"R_QUADRATIC_TO, -10, -10, -10, -24,\n" \
"R_V_LINE_TO, -132,\n" \
"R_QUADRATIC_TO, 0, -14, 10, -24,\n" \
"R_QUADRATIC_TO, 10, -10, 24, -10,\n" \
"R_H_LINE_TO, 6,\n" \
"R_V_LINE_TO, -40,\n" \
"R_QUADRATIC_TO, 0, -33, 23.5, -56.5,\n" \
"QUADRATIC_TO_SHORTHAND, 720, 560,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 800, 640,\n" \
"R_V_LINE_TO, 40,\n" \
"R_H_LINE_TO, 6,\n" \
"R_QUADRATIC_TO, 14, 0, 24, 10,\n" \
"R_QUADRATIC_TO, 10, 10, 10, 24,\n" \
"R_V_LINE_TO, 132,\n" \
"R_QUADRATIC_TO, 0, 14, -10, 24,\n" \
"R_QUADRATIC_TO, -10, 10, -24, 10,\n" \
"H_LINE_TO, 634,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 46, -200,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -40,\n" \
"R_QUADRATIC_TO, 0, -17, -11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 720, 600,\n" \
"R_QUADRATIC_TO, -17, 0, -28.5, 11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 680, 640,\n" \
"R_V_LINE_TO, 40,\n" \
"CLOSE,\n" \
"MOVE_TO, 480, 440,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 40, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -80, 0,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPointerLockIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePointerLockOffIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 440, 328,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 111, 112,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -131, 93,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 20, -93,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 80, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -40, -280,\n" \
"R_QUADRATIC_TO, 116, 0, 198, 82,\n" \
"R_QUADRATIC_TO, 82, 82, 82, 198,\n" \
"R_V_LINE_TO, 124,\n" \
"R_QUADRATIC_TO, -20, -4, -40, -4,\n" \
"R_QUADRATIC_TO, -20, 0, -40, 4,\n" \
"R_V_LINE_TO, -44,\n" \
"H_LINE_TO, 551,\n" \
"LINE_TO, 440, 328,\n" \
"R_V_LINE_TO, -164,\n" \
"R_QUADRATIC_TO, -31, 6, -57.5, 21,\n" \
"QUADRATIC_TO_SHORTHAND, 335, 222,\n" \
"R_LINE_TO, -56, -57,\n" \
"R_QUADRATIC_TO, 38, -39, 90, -62,\n" \
"R_QUADRATIC_TO, 52, -23, 111, -23,\n" \
"CLOSE,\n" \
"MOVE_TO, 231, 232,\n" \
"R_LINE_TO, 209, 208,\n" \
"H_LINE_TO, 280,\n" \
"R_V_LINE_TO, 160,\n" \
"R_QUADRATIC_TO, 0, 83, 58.5, 141.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 800,\n" \
"R_QUADRATIC_TO, 11, 0, 20.5, -1,\n" \
"R_QUADRATIC_TO, 9.5, -1, 19.5, -3,\n" \
"R_V_LINE_TO, 81,\n" \
"R_QUADRATIC_TO, -10, 2, -19.5, 2.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 880,\n" \
"R_QUADRATIC_TO, -116, 0, -198, -82,\n" \
"R_QUADRATIC_TO, -82, -82, -82, -198,\n" \
"R_V_LINE_TO, -240,\n" \
"R_QUADRATIC_TO, 0, -35, 8, -67,\n" \
"R_QUADRATIC_TO, 8, -32, 23, -61,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 289, 128,\n" \
"R_H_LINE_TO, 160,\n" \
"R_QUADRATIC_TO, 0, -72, -45.5, -127,\n" \
"QUADRATIC_TO_SHORTHAND, 520, 164,\n" \
"R_V_LINE_TO, 196,\n" \
"CLOSE,\n" \
"MOVE_TO, 819, 932,\n" \
"LINE_TO, 28, 140,\n" \
"R_LINE_TO, 56, -56,\n" \
"LINE_TO, 876, 876,\n" \
"R_LINE_TO, -57, 56,\n" \
"CLOSE,\n" \
"MOVE_TO, 634, 880,\n" \
"R_QUADRATIC_TO, -14, 0, -24, -10,\n" \
"R_QUADRATIC_TO, -10, -10, -10, -24,\n" \
"R_V_LINE_TO, -126,\n" \
"R_QUADRATIC_TO, 0, -17, 11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 640, 680,\n" \
"R_H_LINE_TO, 40,\n" \
"R_LINE_TO, 160, 160,\n" \
"R_V_LINE_TO, 6,\n" \
"R_QUADRATIC_TO, 0, 14, -10, 24,\n" \
"R_QUADRATIC_TO, -10, 10, -24, 10,\n" \
"H_LINE_TO, 634,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 166, -188,\n" \
"R_LINE_TO, -40, -40,\n" \
"R_V_LINE_TO, -12,\n" \
"R_QUADRATIC_TO, 0, -17, -11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 720, 600,\n" \
"R_H_LINE_TO, -5,\n" \
"R_QUADRATIC_TO, -3, 0, -5, 1,\n" \
"R_LINE_TO, -30, -30,\n" \
"R_QUADRATIC_TO, 9, -5, 19, -8,\n" \
"R_QUADRATIC_TO, 10, -3, 21, -3,\n" \
"R_QUADRATIC_TO, 33, 0, 56.5, 23.5,\n" \
"QUADRATIC_TO_SHORTHAND, 800, 640,\n" \
"R_V_LINE_TO, 52,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPointerLockOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParsePrinterIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"MOVE_TO, 640, 320,\n" \
"R_V_LINE_TO, -120,\n" \
"H_LINE_TO, 320,\n" \
"R_V_LINE_TO, 120,\n" \
"R_H_LINE_TO, -80,\n" \
"R_V_LINE_TO, -200,\n" \
"R_H_LINE_TO, 480,\n" \
"R_V_LINE_TO, 200,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -480, 80,\n" \
"R_H_LINE_TO, 640,\n" \
"R_H_LINE_TO, -640,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 560, 100,\n" \
"R_QUADRATIC_TO, 17, 0, 28.5, -11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 760, 460,\n" \
"R_QUADRATIC_TO, 0, -17, -11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 720, 420,\n" \
"R_QUADRATIC_TO, -17, 0, -28.5, 11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 680, 460,\n" \
"R_QUADRATIC_TO, 0, 17, 11.5, 28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 720, 500,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -80, 260,\n" \
"R_V_LINE_TO, -160,\n" \
"H_LINE_TO, 320,\n" \
"R_V_LINE_TO, 160,\n" \
"R_H_LINE_TO, 320,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 80, 80,\n" \
"H_LINE_TO, 240,\n" \
"R_V_LINE_TO, -160,\n" \
"H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -240,\n" \
"R_QUADRATIC_TO, 0, -51, 35, -85.5,\n" \
"R_QUADRATIC_TO, 35, -34.5, 85, -34.5,\n" \
"R_H_LINE_TO, 560,\n" \
"R_QUADRATIC_TO, 51, 0, 85.5, 34.5,\n" \
"QUADRATIC_TO_SHORTHAND, 880, 440,\n" \
"R_V_LINE_TO, 240,\n" \
"H_LINE_TO, 720,\n" \
"R_V_LINE_TO, 160,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 80, -240,\n" \
"R_V_LINE_TO, -160,\n" \
"R_QUADRATIC_TO, 0, -17, -11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 760, 400,\n" \
"H_LINE_TO, 200,\n" \
"R_QUADRATIC_TO, -17, 0, -28.5, 11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 160, 440,\n" \
"R_V_LINE_TO, 160,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, -80,\n" \
"R_H_LINE_TO, 480,\n" \
"R_V_LINE_TO, 80,\n" \
"R_H_LINE_TO, 80,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kPrinterIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseProtectedContentIcon) { \
              std::string s = \
"// Copyright 2019 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 18,\n" \
"MOVE_TO, 7.54, 11.99,\n" \
"R_LINE_TO, -0.81, -0.81,\n" \
"R_V_LINE_TO, 0,\n" \
"LINE_TO, 5.11, 9.56,\n" \
"R_LINE_TO, 0.71, -0.71,\n" \
"R_LINE_TO, 1.62, 1.62,\n" \
"R_LINE_TO, 4.24, -4.24,\n" \
"R_LINE_TO, 0.81, 0.81,\n" \
"R_LINE_TO, -4.95, 4.95,\n" \
"CLOSE,\n" \
"MOVE_TO, 15.86, 3,\n" \
"H_LINE_TO, 2.14,\n" \
"CUBIC_TO, 1.34, 3, 1, 3.32, 1, 4,\n" \
"R_V_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0, 0.67, 0.34, 1, 1.14, 1,\n" \
"H_LINE_TO, 5.57,\n" \
"R_V_LINE_TO, 1,\n" \
"R_H_LINE_TO, 6.86,\n" \
"R_V_LINE_TO, -1,\n" \
"R_H_LINE_TO, 3.43,\n" \
"R_CUBIC_TO, 0.8, 0, 1.14, -0.33, 1.14, -1,\n" \
"V_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0, -0.68, -0.34, -1, -1.14, -1,\n" \
"CLOSE,\n" \
"MOVE_TO, 15, 13,\n" \
"H_LINE_TO, 3,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 12,\n" \
"R_V_LINE_TO, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kProtectedContentIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseProtocolHandlerIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 43.44, 22.66,\n" \
"LINE_TO, 30.15, 8.59,\n" \
"CUBIC_TO, 29.42, 7.82, 28.14, 7.81, 27.39, 8.57,\n" \
"LINE_TO, 24.05, 12.01,\n" \
"LINE_TO, 20.82, 8.59,\n" \
"CUBIC_TO, 20.08, 7.81, 18.79, 7.81, 18.06, 8.57,\n" \
"LINE_TO, 4.58, 22.44,\n" \
"CUBIC_TO, 4.21, 22.82, 4, 23.33, 4, 23.87,\n" \
"CUBIC_TO, 4, 24.42, 4.2, 24.93, 4.56, 25.32,\n" \
"LINE_TO, 17.85, 39.39,\n" \
"CUBIC_TO, 18.22, 39.78, 18.71, 40, 19.24, 40,\n" \
"CUBIC_TO, 19.75, 40, 20.24, 39.79, 20.61, 39.41,\n" \
"LINE_TO, 23.95, 35.97,\n" \
"LINE_TO, 27.18, 39.39,\n" \
"CUBIC_TO, 27.55, 39.78, 28.04, 40, 28.57, 40,\n" \
"CUBIC_TO, 29.09, 40, 29.58, 39.79, 29.94, 39.41,\n" \
"LINE_TO, 43.42, 25.54,\n" \
"CUBIC_TO, 44.18, 24.76, 44.19, 23.46, 43.44, 22.66,\n" \
"LINE_TO, 43.44, 22.66,\n" \
"LINE_TO, 43.44, 22.66,\n" \
"CLOSE,\n" \
"MOVE_TO, 13.89, 24.93,\n" \
"LINE_TO, 21.21, 32.5,\n" \
"LINE_TO, 19.26, 34.46,\n" \
"LINE_TO, 8.71, 23.54,\n" \
"LINE_TO, 19.41, 12.79,\n" \
"LINE_TO, 21.27, 14.71,\n" \
"LINE_TO, 13.92, 22.11,\n" \
"CUBIC_TO, 13.54, 22.49, 13.34, 22.98, 13.33, 23.52,\n" \
"CUBIC_TO, 13.33, 24.05, 13.53, 24.55, 13.89, 24.93,\n" \
"LINE_TO, 13.89, 24.93,\n" \
"LINE_TO, 13.89, 24.93,\n" \
"CLOSE,\n" \
"MOVE_TO, 28.59, 34.46,\n" \
"LINE_TO, 26.73, 32.54,\n" \
"LINE_TO, 34.09, 25.14,\n" \
"CUBIC_TO, 34.85, 24.37, 34.86, 23.1, 34.11, 22.32,\n" \
"LINE_TO, 26.79, 14.75,\n" \
"LINE_TO, 28.74, 12.79,\n" \
"LINE_TO, 39.29, 23.71,\n" \
"LINE_TO, 28.59, 34.46,\n" \
"LINE_TO, 28.59, 34.46,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kProtocolHandlerIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseProtocolHandlerChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 7.88, 16.4,\n" \
"R_ARC_TO, 0.97, 0.97, 0, 0, 1, -0.39, -0.08,\n" \
"R_ARC_TO, 1.18, 1.18, 0, 0, 1, -0.35, -0.25,\n" \
"LINE_TO, 2.25, 10.67,\n" \
"R_ARC_TO, 1.08, 1.08, 0, 0, 1, -0.19, -0.32,\n" \
"R_ARC_TO, 1, 1, 0, 0, 1, 0.19, -1.01,\n" \
"R_LINE_TO, 4.88, -5.39,\n" \
"R_CUBIC_TO, 0.11, -0.11, 0.23, -0.19, 0.36, -0.25,\n" \
"R_ARC_TO, 0.97, 0.97, 0, 0, 1, 0.78, 0,\n" \
"R_CUBIC_TO, 0.12, 0.06, 0.23, 0.14, 0.34, 0.25,\n" \
"LINE_TO, 10, 5.48,\n" \
"R_LINE_TO, 1.4, -1.54,\n" \
"R_ARC_TO, 1.3, 1.3, 0, 0, 1, 0.34, -0.25,\n" \
"R_ARC_TO, 0.97, 0.97, 0, 0, 1, 0.78, 0,\n" \
"R_CUBIC_TO, 0.13, 0.06, 0.25, 0.14, 0.36, 0.25,\n" \
"R_LINE_TO, 4.88, 5.4,\n" \
"R_CUBIC_TO, 0.08, 0.1, 0.14, 0.21, 0.19, 0.32,\n" \
"R_CUBIC_TO, 0.04, 0.12, 0.06, 0.24, 0.06, 0.34,\n" \
"R_ARC_TO, 1.08, 1.08, 0, 0, 1, -0.25, 0.67,\n" \
"R_LINE_TO, -4.87, 5.4,\n" \
"R_CUBIC_TO, -0.11, 0.11, -0.23, 0.2, -0.35, 0.25,\n" \
"R_ARC_TO, 0.97, 0.97, 0, 0, 1, -0.78, 0,\n" \
"R_ARC_TO, 1.3, 1.3, 0, 0, 1, -0.34, -0.25,\n" \
"LINE_TO, 10, 14.52,\n" \
"R_LINE_TO, -1.39, 1.54,\n" \
"R_ARC_TO, 1.3, 1.3, 0, 0, 1, -0.34, 0.25,\n" \
"R_ARC_TO, 0.92, 0.92, 0, 0, 1, -0.39, 0.08,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.75,\n" \
"LINE_TO, 9, 13.42,\n" \
"R_LINE_TO, -2.5, -2.75,\n" \
"ARC_TO, 0.99, 0.99, 0, 0, 1, 6.25, 10,\n" \
"R_CUBIC_TO, 0, -0.25, 0.08, -0.47, 0.25, -0.67,\n" \
"LINE_TO, 9, 6.58,\n" \
"LINE_TO, 7.88, 5.36,\n" \
"LINE_TO, 3.67, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.25, 0,\n" \
"LINE_TO, 16.33, 10,\n" \
"R_LINE_TO, -4.21, -4.64,\n" \
"LINE_TO, 11, 6.58,\n" \
"R_LINE_TO, 2.5, 2.75,\n" \
"R_CUBIC_TO, 0.17, 0.2, 0.25, 0.42, 0.25, 0.67,\n" \
"R_CUBIC_TO, 0, 0.25, -0.08, 0.47, -0.25, 0.67,\n" \
"R_LINE_TO, -2.5, 2.75,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kProtocolHandlerChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseProtocolHandlerOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16, 18.13,\n" \
"LINE_TO, 13.38, 15.5,\n" \
"R_LINE_TO, -0.5, 0.56,\n" \
"R_ARC_TO, 0.97, 0.97, 0, 0, 1, -0.33, 0.25,\n" \
"R_ARC_TO, 1.04, 1.04, 0, 0, 1, -0.42, 0.08,\n" \
"R_CUBIC_TO, -0.15, 0, -0.29, -0.03, -0.42, -0.08,\n" \
"R_ARC_TO, 0.81, 0.81, 0, 0, 1, -0.31, -0.25,\n" \
"LINE_TO, 10, 14.52,\n" \
"R_LINE_TO, -1.39, 1.54,\n" \
"R_ARC_TO, 0.98, 0.98, 0, 0, 1, -0.33, 0.25,\n" \
"R_ARC_TO, 1.04, 1.04, 0, 0, 1, -0.81, 0,\n" \
"R_ARC_TO, 0.97, 0.97, 0, 0, 1, -0.33, -0.25,\n" \
"LINE_TO, 2.25, 10.67,\n" \
"R_ARC_TO, 0.84, 0.84, 0, 0, 1, -0.19, -0.3,\n" \
"ARC_TO, 1.05, 1.05, 0, 0, 1, 2, 10,\n" \
"R_ARC_TO, 0.99, 0.99, 0, 0, 1, 0.25, -0.67,\n" \
"LINE_TO, 4.61, 6.73,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3.69, -3.69,\n" \
"R_LINE_TO, -1.17, -1.19,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4.44, 0.21,\n" \
"LINE_TO, 9, 13.42,\n" \
"R_LINE_TO, -2.5, -2.75,\n" \
"ARC_TO, 0.99, 0.99, 0, 0, 1, 6.25, 10,\n" \
"R_CUBIC_TO, 0, -0.25, 0.08, -0.47, 0.25, -0.67,\n" \
"R_LINE_TO, 0.33, -0.37,\n" \
"R_LINE_TO, -1.16, -1.16,\n" \
"R_LINE_TO, -2, 2.21,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7.52, -1.37,\n" \
"R_LINE_TO, -1.06, -1.06,\n" \
"R_LINE_TO, 2, -2.21,\n" \
"R_LINE_TO, -4.35, -4.5,\n" \
"LINE_TO, 11, 6.58,\n" \
"R_LINE_TO, 2.5, 2.75,\n" \
"R_CUBIC_TO, 0.17, 0.2, 0.25, 0.42, 0.25, 0.67,\n" \
"R_CUBIC_TO, 0, 0.25, -0.08, 0.47, -0.25, 0.67,\n" \
"R_LINE_TO, -0.33, 0.38,\n" \
"LINE_TO, 6.63, 4.5,\n" \
"R_LINE_TO, 0.5, -0.56,\n" \
"R_CUBIC_TO, 0.11, -0.11, 0.23, -0.19, 0.34, -0.25,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 1, 0.41, -0.08,\n" \
"R_CUBIC_TO, 0.14, 0, 0.27, 0.03, 0.4, 0.08,\n" \
"R_CUBIC_TO, 0.13, 0.05, 0.24, 0.14, 0.34, 0.25,\n" \
"LINE_TO, 10, 5.48,\n" \
"R_LINE_TO, 1.4, -1.54,\n" \
"R_ARC_TO, 0.98, 0.98, 0, 0, 1, 0.34, -0.25,\n" \
"R_ARC_TO, 1.04, 1.04, 0, 0, 1, 0.81, 0,\n" \
"R_CUBIC_TO, 0.13, 0.06, 0.23, 0.14, 0.33, 0.25,\n" \
"R_LINE_TO, 4.88, 5.4,\n" \
"ARC_TO, 0.99, 0.99, 0, 0, 1, 18, 10,\n" \
"R_ARC_TO, 0.99, 0.99, 0, 0, 1, -0.25, 0.67,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kProtocolHandlerOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseQwacStatusIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 9.97f, 0,\n" \
"R_ARC_TO, 4.87f, 4.87f, 0, 0, 0, -4.86f, 4.87f,\n" \
"V_LINE_TO, 7.56f,\n" \
"ARC_TO, 2.07f, 2.07f, 0, 0, 0, 3.2f, 9.63f,\n" \
"R_V_LINE_TO, 8.3f,\n" \
"CUBIC_TO, 3.2f, 19.07f, 4.13f, 20, 5.27f, 20,\n" \
"R_H_LINE_TO, 9.46f,\n" \
"R_CUBIC_TO, 1.15f, 0, 2.07f, -0.93f, 2.07f, -2.08f,\n" \
"V_LINE_TO, 9.63f,\n" \
"R_CUBIC_TO, 0, -1.11f, -0.86f, -2.01f, -1.95f, -2.07f,\n" \
"R_CUBIC_TO, -0.01f, -0.8f, -0.02f, -1.73f, -0.02f, -2.68f,\n" \
"ARC_TO, 4.87f, 4.87f, 0, 0, 0, 9.97f, 0,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 2.08f,\n" \
"R_ARC_TO, 2.8f, 2.8f, 0, 0, 1, 2.79f, 2.8f,\n" \
"R_CUBIC_TO, 0, 0.95f, 0.01f, 1.87f, 0.02f, 2.68f,\n" \
"H_LINE_TO, 7.18f,\n" \
"R_V_LINE_TO, -2.68f,\n" \
"R_ARC_TO, 2.8f, 2.8f, 0, 0, 1, 2.79f, -2.8f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.03f, 6.45f,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.34f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.37f, 0.32f,\n" \
"R_CUBIC_TO, 0.2f, 0.07f, 0.37f, 0.18f, 0.56f, 0.56f,\n" \
"R_ARC_TO, 17.78f, 17.78f, 0, 0, 0, -1.68f, 1.75f,\n" \
"R_LINE_TO, 0.09f, 0.26f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.1f, -0.08f,\n" \
"R_CUBIC_TO, -0.26f, 0.32f, -0.52f, 0.65f, -0.78f, 1,\n" \
"R_ARC_TO, 50.19f, 50.19f, 0, 0, 0, -2.76f, 4.07f,\n" \
"R_CUBIC_TO, -0.72f, -0.92f, -1.94f, -2.23f, -3.64f, -3.93f,\n" \
"R_LINE_TO, 0.85f, -0.83f,\n" \
"LINE_TO, 9.77f, 13.87f,\n" \
"R_ARC_TO, 85.52f, 85.52f, 0, 0, 1, 3.44f, -3.27f,\n" \
"R_CUBIC_TO, 1.07f, -0.95f, 1.79f, -1.53f, 2.17f, -1.76f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.88f, 0.35f,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.35f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4.98f, 0,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.35f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.77f, 10.97f,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.35f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.29f, 0.4f,\n" \
"R_LINE_TO, -0.03f, 0.04f,\n" \
"R_H_LINE_TO, 0.02f,\n" \
"R_LINE_TO, 0.01f, -0.04f,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.4f, 13.13f,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.35f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 9.21f, 0,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.35f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -8.71f, 2.37f,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.35f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.3f, 0,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.35f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.7f, 1.58f,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.35f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4.98f, 0.17f,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.35f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.49f, 0.58f,\n" \
"R_LINE_TO, 0.14f, 0.44f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, -0.35f, 0.28f,\n" \
"R_LINE_TO, 0.13f, 0.44f,\n" \
"R_LINE_TO, -0.36f, -0.26f,\n" \
"R_LINE_TO, -0.36f, 0.26f,\n" \
"R_LINE_TO, 0.13f, -0.44f,\n" \
"R_LINE_TO, -0.34f, -0.28f,\n" \
"R_H_LINE_TO, 0.43f,\n" \
"R_LINE_TO, 0.14f, -0.44f,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kQwacStatusIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseRadioButtonCheckedIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 10, 14,\n" \
"R_CUBIC_TO, 1.11, 0, 2.05, -0.39, 2.83, -1.17,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 0, 14, 10,\n" \
"R_CUBIC_TO, 0, -1.11, -0.39, -2.05, -1.17, -2.83,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 0, 10, 6,\n" \
"R_CUBIC_TO, -1.11, 0, -2.05, 0.39, -2.83, 1.17,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 0, 6, 10,\n" \
"R_CUBIC_TO, 0, 1.11, 0.39, 2.05, 1.17, 2.83,\n" \
"ARC_TO, 3.87, 3.87, 0, 0, 0, 10, 14,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 4.17,\n" \
"R_ARC_TO, 7.99, 7.99, 0, 0, 1, -3.18, -0.64,\n" \
"R_ARC_TO, 8.22, 8.22, 0, 0, 1, -4.36, -4.36,\n" \
"R_ARC_TO, 7.99, 7.99, 0, 0, 1, -0.64, -3.18,\n" \
"R_CUBIC_TO, 0, -1.13, 0.21, -2.19, 0.64, -3.18,\n" \
"ARC_TO, 8.22, 8.22, 0, 0, 1, 6.82, 2.47,\n" \
"R_ARC_TO, 7.99, 7.99, 0, 0, 1, 3.18, -0.64,\n" \
"R_CUBIC_TO, 1.13, 0, 2.19, 0.21, 3.18, 0.64,\n" \
"R_ARC_TO, 8.22, 8.22, 0, 0, 1, 2.59, 1.75,\n" \
"R_ARC_TO, 8.2, 8.2, 0, 0, 1, 1.75, 2.6,\n" \
"R_CUBIC_TO, 0.43, 0.99, 0.64, 2.05, 0.64, 3.18,\n" \
"R_CUBIC_TO, 0, 1.13, -0.21, 2.19, -0.64, 3.18,\n" \
"R_ARC_TO, 8.22, 8.22, 0, 0, 1, -4.35, 4.36,\n" \
"R_ARC_TO, 7.94, 7.94, 0, 0, 1, -3.18, 0.64,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 16.44,\n" \
"R_CUBIC_TO, 1.79, 0, 3.31, -0.62, 4.56, -1.88,\n" \
"R_CUBIC_TO, 1.25, -1.25, 1.88, -2.77, 1.88, -4.56,\n" \
"R_CUBIC_TO, 0, -1.79, -0.62, -3.31, -1.88, -4.56,\n" \
"CUBIC_TO, 13.31, 4.18, 11.79, 3.56, 10, 3.56,\n" \
"R_CUBIC_TO, -1.79, 0, -3.31, 0.63, -4.56, 1.88,\n" \
"CUBIC_TO, 4.18, 6.69, 3.56, 8.21, 3.56, 10,\n" \
"R_CUBIC_TO, 0, 1.79, 0.63, 3.31, 1.88, 4.56,\n" \
"R_CUBIC_TO, 1.25, 1.25, 2.77, 1.88, 4.56, 1.88,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kRadioButtonCheckedIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseReceiptLongIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 6, 22,\n" \
"R_ARC_TO, 2.9, 2.9, 0, 0, 1, -2.12, -0.87,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 1, 3, 19,\n" \
"R_V_LINE_TO, -3,\n" \
"R_H_LINE_TO, 3,\n" \
"V_LINE_TO, 2,\n" \
"R_LINE_TO, 1.5, 1.5,\n" \
"LINE_TO, 9, 2,\n" \
"R_LINE_TO, 1.5, 1.5,\n" \
"LINE_TO, 12, 2,\n" \
"R_LINE_TO, 1.5, 1.5,\n" \
"LINE_TO, 15, 2,\n" \
"R_LINE_TO, 1.5, 1.5,\n" \
"LINE_TO, 18, 2,\n" \
"R_LINE_TO, 1.5, 1.5,\n" \
"LINE_TO, 21, 2,\n" \
"R_V_LINE_TO, 17,\n" \
"R_CUBIC_TO, 0, 0.83, -0.29, 1.54, -0.87, 2.13,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 1, 18, 22,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12, -2,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, -0.1, 0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 0, 19, 19,\n" \
"V_LINE_TO, 5,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 11,\n" \
"R_H_LINE_TO, 9,\n" \
"R_V_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, 0.28, 0.1, 0.52, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.43, 0.29, 0.71, 0.29,\n" \
"CLOSE,\n" \
"MOVE_TO, 9, 9,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, 6,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 3,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 6,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8, -3,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 1, -0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 16, 8,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 17, 7,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0, 0.28, -0.1, 0.52, -0.29, 0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 17, 9,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 3,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 1, -0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 16, 11,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 17, 10,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0, 0.28, -0.1, 0.52, -0.29, 0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 17, 12,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 20,\n" \
"R_H_LINE_TO, 9,\n" \
"R_V_LINE_TO, -2,\n" \
"H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 1,\n" \
"R_CUBIC_TO, 0, 0.28, 0.1, 0.52, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.43, 0.29, 0.71, 0.29,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1, 0,\n" \
"R_V_LINE_TO, -2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kReceiptLongIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseReloadIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"MOVE_TO, 25.1, 20.15,\n" \
"LINE_TO, 25.08, 20.14,\n" \
"CUBIC_TO, 23.51, 23.59, 20.04, 26, 16, 26,\n" \
"CUBIC_TO, 10.48, 26, 6, 21.52, 6, 16,\n" \
"CUBIC_TO, 6, 10.48, 10.48, 6, 16, 6,\n" \
"CUBIC_TO, 19.02, 6, 21.72, 7.34, 23.55, 9.45,\n" \
"LINE_TO, 23.55, 9.45,\n" \
"LINE_TO, 19, 14,\n" \
"LINE_TO, 25.8, 14,\n" \
"LINE_TO, 28.83, 14,\n" \
"LINE_TO, 30, 14,\n" \
"LINE_TO, 30, 3,\n" \
"LINE_TO, 25.67, 7.33,\n" \
"CUBIC_TO, 23.3, 4.67, 19.85, 3, 16, 3,\n" \
"CUBIC_TO, 8.82, 3, 3, 8.82, 3, 16,\n" \
"CUBIC_TO, 3, 23.18, 8.82, 29, 16, 29,\n" \
"CUBIC_TO, 21.27, 29, 25.8, 25.86, 27.84, 21.34,\n" \
"CUBIC_TO, 27.96, 21.13, 28.03, 20.88, 28.03, 20.61,\n" \
"CUBIC_TO, 28.03, 19.78, 27.36, 19.11, 26.53, 19.11,\n" \
"CUBIC_TO, 25.87, 19.11, 25.3, 19.55, 25.1, 20.15,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 15, 1,\n" \
"LINE_TO, 15, 7,\n" \
"LINE_TO, 9, 7,\n" \
"LINE_TO, 11.62, 4.37,\n" \
"CUBIC_TO, 10.67, 3.43, 9.44, 3, 8, 3,\n" \
"CUBIC_TO, 5.09, 3, 3, 5.1, 3, 8,\n" \
"CUBIC_TO, 3, 10.9, 5.09, 13, 8, 13,\n" \
"CUBIC_TO, 10.3, 13, 12.08, 11.81, 12.75, 9.83,\n" \
"LINE_TO, 12.77, 9.79,\n" \
"CUBIC_TO, 12.93, 9.44, 13.47, 9.18, 13.76, 9.18,\n" \
"CUBIC_TO, 14.15, 9.18, 14.79, 9.41, 14.79, 10.01,\n" \
"CUBIC_TO, 14.79, 10.14, 14.71, 10.46, 14.66, 10.57,\n" \
"CUBIC_TO, 13.66, 13.13, 10.98, 15, 8.03, 15,\n" \
"CUBIC_TO, 4.15, 15, 1, 11.87, 1, 8,\n" \
"CUBIC_TO, 1, 4.13, 4.15, 1, 8.03, 1,\n" \
"CUBIC_TO, 9.96, 1, 11.7, 1.77, 12.97, 3.03,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kReloadIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseReloadChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 16,\n" \
"R_CUBIC_TO, -1.67, 0, -3.08, -0.58, -4.25, -1.75,\n" \
"CUBIC_TO, 4.58, 13.08, 4, 11.67, 4, 10,\n" \
"R_CUBIC_TO, 0, -1.67, 0.58, -3.08, 1.75, -4.25,\n" \
"CUBIC_TO, 6.92, 4.58, 8.33, 4, 10, 4,\n" \
"R_CUBIC_TO, 0.9, 0, 1.74, 0.19, 2.51, 0.56,\n" \
"R_CUBIC_TO, 0.77, 0.38, 1.43, 0.88, 1.99, 1.5,\n" \
"V_LINE_TO, 4,\n" \
"H_LINE_TO, 16,\n" \
"R_V_LINE_TO, 5,\n" \
"R_H_LINE_TO, -5,\n" \
"V_LINE_TO, 7.5,\n" \
"R_H_LINE_TO, 2.73,\n" \
"R_ARC_TO, 4.44, 4.44, 0, 0, 0, -1.58, -1.46,\n" \
"ARC_TO, 4.33, 4.33, 0, 0, 0, 10, 5.5,\n" \
"R_CUBIC_TO, -1.25, 0, -2.31, 0.44, -3.19, 1.31,\n" \
"CUBIC_TO_SHORTHAND, 5.5, 8.75, 5.5, 10,\n" \
"R_CUBIC_TO, 0, 1.25, 0.44, 2.31, 1.31, 3.19,\n" \
"CUBIC_TO, 7.69, 14.06, 8.75, 14.5, 10, 14.5,\n" \
"R_CUBIC_TO, 1.17, 0, 2.17, -0.39, 3, -1.16,\n" \
"R_ARC_TO, 4.43, 4.43, 0, 0, 0, 1.44, -2.84,\n" \
"R_H_LINE_TO, 1.54,\n" \
"R_CUBIC_TO, -0.12, 1.56, -0.76, 2.86, -1.9, 3.92,\n" \
"CUBIC_TO, 12.95, 15.47, 11.58, 16, 10, 16,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kReloadChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseReplayIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 12, 5,\n" \
"V_LINE_TO, 1,\n" \
"LINE_TO, 7, 6,\n" \
"R_LINE_TO, 5, 5,\n" \
"V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 3.31, 0, 6, 2.69, 6, 6,\n" \
"R_CUBIC_TO, 0, 3.31, -2.69, 6, -6, 6,\n" \
"R_CUBIC_TO, -3.31, 0, -6, -2.69, -6, -6,\n" \
"H_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0, 4.42, 3.58, 8, 8, 8,\n" \
"R_CUBIC_TO, 4.42, 0, 8, -3.58, 8, -8,\n" \
"R_CUBIC_TO, 0, -4.42, -3.58, -8, -8, -8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kReplayIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseReplay10Icon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"MOVE_TO, 480, 880,\n" \
"R_QUADRATIC_TO, -75, 0, -140.5, -28.5,\n" \
"R_QUADRATIC_TO, -65.5, -28.5, -114, -77,\n" \
"R_QUADRATIC_TO, -48.5, -48.5, -77, -114,\n" \
"QUADRATIC_TO_SHORTHAND, 120, 520,\n" \
"R_H_LINE_TO, 80,\n" \
"R_QUADRATIC_TO, 0, 117, 81.5, 198.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 800,\n" \
"R_QUADRATIC_TO, 117, 0, 198.5, -81.5,\n" \
"QUADRATIC_TO_SHORTHAND, 760, 520,\n" \
"R_QUADRATIC_TO, 0, -117, -81.5, -198.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 240,\n" \
"R_H_LINE_TO, -6,\n" \
"R_LINE_TO, 62, 62,\n" \
"R_LINE_TO, -56, 58,\n" \
"R_LINE_TO, -160, -160,\n" \
"R_LINE_TO, 160, -160,\n" \
"R_LINE_TO, 56, 58,\n" \
"R_LINE_TO, -62, 62,\n" \
"R_H_LINE_TO, 6,\n" \
"R_QUADRATIC_TO, 75, 0, 140.5, 28.5,\n" \
"R_QUADRATIC_TO, 65.5, 28.5, 114, 77,\n" \
"R_QUADRATIC_TO, 48.5, 48.5, 77, 114,\n" \
"QUADRATIC_TO_SHORTHAND, 840, 520,\n" \
"R_QUADRATIC_TO, 0, 75, -28.5, 140.5,\n" \
"R_QUADRATIC_TO, -28.5, 65.5, -77, 114,\n" \
"R_QUADRATIC_TO, -48.5, 48.5, -114, 77,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 880,\n" \
"CLOSE,\n" \
"MOVE_TO, 360, 640,\n" \
"R_V_LINE_TO, -180,\n" \
"R_H_LINE_TO, -60,\n" \
"R_V_LINE_TO, -60,\n" \
"R_H_LINE_TO, 120,\n" \
"R_V_LINE_TO, 240,\n" \
"R_H_LINE_TO, -60,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 140, 0,\n" \
"R_QUADRATIC_TO, -17, 0, -28.5, -11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 460, 600,\n" \
"R_V_LINE_TO, -160,\n" \
"R_QUADRATIC_TO, 0, -17, 11.5, -28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 500, 400,\n" \
"R_H_LINE_TO, 80,\n" \
"R_QUADRATIC_TO, 17, 0, 28.5, 11.5,\n" \
"QUADRATIC_TO_SHORTHAND, 620, 440,\n" \
"R_V_LINE_TO, 160,\n" \
"R_QUADRATIC_TO, 0, 17, -11.5, 28.5,\n" \
"QUADRATIC_TO_SHORTHAND, 580, 640,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 20, -60,\n" \
"R_H_LINE_TO, 40,\n" \
"R_V_LINE_TO, -120,\n" \
"R_H_LINE_TO, -40,\n" \
"R_V_LINE_TO, 120,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kReplay10Icon); \
              } \
 \
TEST_F(VectorIconsTest, ParseRouterIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5f, 17,\n" \
"R_CUBIC_TO, -0.41f, 0, -0.77f, -0.15f, -1.06f, -0.44f,\n" \
"ARC_TO, 1.45f, 1.45f, 0, 0, 1, 3, 15.5f,\n" \
"R_V_LINE_TO, -3,\n" \
"R_CUBIC_TO, 0, -0.41f, 0.15f, -0.77f, 0.44f, -1.06f,\n" \
"ARC_TO, 1.44f, 1.44f, 0, 0, 1, 4.5f, 11,\n" \
"R_H_LINE_TO, 8,\n" \
"V_LINE_TO, 8,\n" \
"H_LINE_TO, 14,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 1.5f,\n" \
"R_CUBIC_TO, 0.43f, 0, 0.79f, 0.14f, 1.09f, 0.44f,\n" \
"R_CUBIC_TO, 0.3f, 0.29f, 0.44f, 0.64f, 0.41f, 1.06f,\n" \
"R_V_LINE_TO, 3,\n" \
"R_ARC_TO, 1.64f, 1.64f, 0, 0, 1, -0.47f, 1.05f,\n" \
"R_CUBIC_TO, -0.28f, 0.3f, -0.63f, 0.45f, -1.03f, 0.45f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5f,\n" \
"R_H_LINE_TO, 11,\n" \
"R_V_LINE_TO, -3,\n" \
"R_H_LINE_TO, -11,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.75f, -0.75f,\n" \
"R_CUBIC_TO, 0.21f, 0, 0.39f, -0.07f, 0.54f, -0.21f,\n" \
"ARC_TO, 0.72f, 0.72f, 0, 0, 0, 7, 14,\n" \
"R_CUBIC_TO, 0, -0.21f, -0.07f, -0.39f, -0.21f, -0.53f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 0, -0.53f, -0.22f,\n" \
"R_CUBIC_TO, -0.21f, 0, -0.39f, 0.07f, -0.53f, 0.22f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 0, -0.22f, 0.53f,\n" \
"R_CUBIC_TO, 0, 0.21f, 0.07f, 0.39f, 0.22f, 0.54f,\n" \
"R_CUBIC_TO, 0.14f, 0.14f, 0.32f, 0.22f, 0.53f, 0.22f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.75f, 0,\n" \
"R_CUBIC_TO, 0.21f, 0, 0.39f, -0.07f, 0.54f, -0.21f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 0, 0.22f, -0.53f,\n" \
"R_CUBIC_TO, 0, -0.21f, -0.07f, -0.39f, -0.21f, -0.53f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 0, -0.53f, -0.22f,\n" \
"R_CUBIC_TO, -0.21f, 0, -0.39f, 0.07f, -0.53f, 0.22f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 0, -0.22f, 0.53f,\n" \
"R_CUBIC_TO, 0, 0.21f, 0.07f, 0.39f, 0.22f, 0.54f,\n" \
"R_CUBIC_TO, 0.14f, 0.14f, 0.32f, 0.22f, 0.53f, 0.22f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.75f, 0,\n" \
"R_CUBIC_TO, 0.21f, 0, 0.39f, -0.07f, 0.54f, -0.21f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 0, 0.22f, -0.53f,\n" \
"R_CUBIC_TO, 0, -0.21f, -0.07f, -0.39f, -0.21f, -0.53f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 0, -0.53f, -0.22f,\n" \
"R_CUBIC_TO, -0.21f, 0, -0.39f, 0.07f, -0.53f, 0.22f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 0, -0.22f, 0.53f,\n" \
"R_CUBIC_TO, 0, 0.21f, 0.07f, 0.39f, 0.22f, 0.54f,\n" \
"R_CUBIC_TO, 0.14f, 0.14f, 0.32f, 0.22f, 0.53f, 0.22f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.36f, -7.89f,\n" \
"R_LINE_TO, -1.06f, -1.06f,\n" \
"R_CUBIC_TO, 0.29f, -0.28f, 0.63f, -0.48f, 1.02f, -0.61f,\n" \
"R_ARC_TO, 3.87f, 3.87f, 0, 0, 1, 2.37f, 0,\n" \
"R_CUBIC_TO, 0.39f, 0.13f, 0.73f, 0.33f, 1.02f, 0.61f,\n" \
"R_LINE_TO, -1.06f, 1.06f,\n" \
"R_ARC_TO, 1.04f, 1.04f, 0, 0, 0, -0.53f, -0.29f,\n" \
"R_ARC_TO, 2.92f, 2.92f, 0, 0, 0, -0.61f, -0.06f,\n" \
"R_CUBIC_TO, -0.21f, 0, -0.41f, 0.02f, -0.61f, 0.06f,\n" \
"R_CUBIC_TO, -0.2f, 0.04f, -0.38f, 0.14f, -0.53f, 0.29f,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 4.75f,\n" \
"LINE_TO, 8.92f, 3.67f,\n" \
"R_ARC_TO, 5.05f, 5.05f, 0, 0, 1, 1.99f, -1.27f,\n" \
"ARC_TO, 7.13f, 7.13f, 0, 0, 1, 13.25f, 2,\n" \
"R_CUBIC_TO, 0.81f, 0, 1.59f, 0.13f, 2.34f, 0.4f,\n" \
"R_ARC_TO, 5.05f, 5.05f, 0, 0, 1, 1.99f, 1.27f,\n" \
"LINE_TO, 16.5f, 4.75f,\n" \
"R_ARC_TO, 4.01f, 4.01f, 0, 0, 0, -1.49f, -0.95f,\n" \
"R_ARC_TO, 5.25f, 5.25f, 0, 0, 0, -1.76f, -0.3f,\n" \
"R_CUBIC_TO, -0.61f, 0, -1.2f, 0.1f, -1.76f, 0.3f,\n" \
"ARC_TO, 4.01f, 4.01f, 0, 0, 0, 10, 4.75f,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5f, 15.5f,\n" \
"R_V_LINE_TO, -3,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kRouterIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseRouterOffIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 12.11f, 6.86f,\n" \
"R_LINE_TO, -1.06f, -1.06f,\n" \
"R_CUBIC_TO, 0.29f, -0.28f, 0.63f, -0.48f, 1.02f, -0.61f,\n" \
"R_ARC_TO, 3.87f, 3.87f, 0, 0, 1, 2.37f, 0,\n" \
"R_CUBIC_TO, 0.39f, 0.13f, 0.73f, 0.33f, 1.02f, 0.61f,\n" \
"R_LINE_TO, -1.06f, 1.06f,\n" \
"R_ARC_TO, 1.04f, 1.04f, 0, 0, 0, -0.53f, -0.29f,\n" \
"R_ARC_TO, 2.92f, 2.92f, 0, 0, 0, -0.61f, -0.06f,\n" \
"R_CUBIC_TO, -0.21f, 0, -0.41f, 0.02f, -0.61f, 0.06f,\n" \
"R_CUBIC_TO, -0.2f, 0.04f, -0.38f, 0.14f, -0.53f, 0.29f,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 4.75f,\n" \
"LINE_TO, 8.92f, 3.67f,\n" \
"R_ARC_TO, 5.05f, 5.05f, 0, 0, 1, 1.99f, -1.27f,\n" \
"ARC_TO, 7.13f, 7.13f, 0, 0, 1, 13.25f, 2,\n" \
"R_CUBIC_TO, 0.81f, 0, 1.59f, 0.13f, 2.34f, 0.4f,\n" \
"R_ARC_TO, 5.05f, 5.05f, 0, 0, 1, 1.99f, 1.27f,\n" \
"LINE_TO, 16.5f, 4.75f,\n" \
"R_ARC_TO, 4.01f, 4.01f, 0, 0, 0, -1.49f, -0.95f,\n" \
"R_ARC_TO, 5.25f, 5.25f, 0, 0, 0, -1.76f, -0.3f,\n" \
"R_CUBIC_TO, -0.61f, 0, -1.2f, 0.1f, -1.76f, 0.3f,\n" \
"ARC_TO, 4.01f, 4.01f, 0, 0, 0, 10, 4.75f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7, 10.13f,\n" \
"R_LINE_TO, -4.5f, -4.51f,\n" \
"V_LINE_TO, 8,\n" \
"H_LINE_TO, 14,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 1.5f,\n" \
"R_CUBIC_TO, 0.43f, 0, 0.79f, 0.14f, 1.09f, 0.44f,\n" \
"R_CUBIC_TO, 0.3f, 0.29f, 0.44f, 0.64f, 0.41f, 1.06f,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5f, 15.5f,\n" \
"R_H_LINE_TO, 8.88f,\n" \
"R_LINE_TO, -3, -3,\n" \
"H_LINE_TO, 4.5f,\n" \
"CLOSE,\n" \
"MOVE_TO, 16, 18.13f,\n" \
"LINE_TO, 14.88f, 17,\n" \
"H_LINE_TO, 4.5f,\n" \
"R_CUBIC_TO, -0.41f, 0, -0.77f, -0.15f, -1.06f, -0.44f,\n" \
"ARC_TO, 1.45f, 1.45f, 0, 0, 1, 3, 15.5f,\n" \
"R_V_LINE_TO, -3,\n" \
"R_CUBIC_TO, 0, -0.41f, 0.15f, -0.77f, 0.44f, -1.06f,\n" \
"ARC_TO, 1.44f, 1.44f, 0, 0, 1, 4.5f, 11,\n" \
"R_H_LINE_TO, 4.38f,\n" \
"R_LINE_TO, -7, -7,\n" \
"R_LINE_TO, 1.06f, -1.06f,\n" \
"R_LINE_TO, 14.12f, 14.13f,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.25f, 14.75f,\n" \
"R_CUBIC_TO, -0.21f, 0, -0.39f, -0.07f, -0.53f, -0.21f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 1, -0.22f, -0.53f,\n" \
"R_CUBIC_TO, 0, -0.21f, 0.07f, -0.39f, 0.22f, -0.53f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 1, 0.53f, -0.22f,\n" \
"R_CUBIC_TO, 0.21f, 0, 0.39f, 0.07f, 0.54f, 0.22f,\n" \
"R_CUBIC_TO, 0.14f, 0.14f, 0.22f, 0.32f, 0.22f, 0.53f,\n" \
"R_CUBIC_TO, 0, 0.21f, -0.07f, 0.39f, -0.21f, 0.54f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 1, -0.53f, 0.22f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.75f, 0,\n" \
"R_CUBIC_TO, -0.21f, 0, -0.39f, -0.07f, -0.53f, -0.21f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 1, -0.22f, -0.53f,\n" \
"R_CUBIC_TO, 0, -0.21f, 0.07f, -0.39f, 0.22f, -0.53f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 1, 0.53f, -0.22f,\n" \
"R_CUBIC_TO, 0.21f, 0, 0.39f, 0.07f, 0.54f, 0.22f,\n" \
"R_CUBIC_TO, 0.14f, 0.14f, 0.22f, 0.32f, 0.22f, 0.53f,\n" \
"R_CUBIC_TO, 0, 0.21f, -0.07f, 0.39f, -0.21f, 0.54f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 1, -0.53f, 0.22f,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.75f, 0,\n" \
"R_CUBIC_TO, -0.21f, 0, -0.39f, -0.07f, -0.53f, -0.21f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 1, -0.22f, -0.53f,\n" \
"R_CUBIC_TO, 0, -0.21f, 0.07f, -0.39f, 0.22f, -0.53f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 1, 0.53f, -0.22f,\n" \
"R_CUBIC_TO, 0.21f, 0, 0.39f, 0.07f, 0.54f, 0.22f,\n" \
"R_CUBIC_TO, 0.14f, 0.14f, 0.22f, 0.32f, 0.22f, 0.53f,\n" \
"R_CUBIC_TO, 0, 0.21f, -0.07f, 0.39f, -0.21f, 0.54f,\n" \
"R_ARC_TO, 0.72f, 0.72f, 0, 0, 1, -0.53f, 0.22f,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5f, 15.5f,\n" \
"R_V_LINE_TO, -3,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kRouterOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSaveCloudIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 5.02, 16.17,\n" \
"CUBIC_TO, 3.85, 16.17, 2.87, 15.76, 2.06, 14.96,\n" \
"CUBIC_TO, 1.26, 14.14, 0.85, 13.15, 0.85, 12,\n" \
"CUBIC_TO, 0.85, 10.9, 1.21, 9.96, 1.92, 9.17,\n" \
"CUBIC_TO, 2.64, 8.36, 3.54, 7.92, 4.63, 7.85,\n" \
"CUBIC_TO, 4.97, 6.66, 5.64, 5.69, 6.63, 4.96,\n" \
"CUBIC_TO, 7.63, 4.21, 8.75, 3.83, 10, 3.83,\n" \
"CUBIC_TO, 11.5, 3.83, 12.81, 4.31, 13.92, 5.27,\n" \
"CUBIC_TO, 15.03, 6.22, 15.6, 7.41, 15.65, 8.85,\n" \
"CUBIC_TO, 16.63, 8.9, 17.46, 9.27, 18.13, 9.98,\n" \
"CUBIC_TO, 18.81, 10.67, 19.15, 11.51, 19.15, 12.5,\n" \
"CUBIC_TO, 19.15, 13.51, 18.78, 14.38, 18.06, 15.1,\n" \
"CUBIC_TO, 17.35, 15.81, 16.49, 16.17, 15.48, 16.17,\n" \
"H_LINE_TO, 10.94,\n" \
"CUBIC_TO, 10.47, 16.17, 10.06, 16, 9.71, 15.67,\n" \
"CUBIC_TO, 9.38, 15.32, 9.21, 14.91, 9.21, 14.44,\n" \
"V_LINE_TO, 11.13,\n" \
"LINE_TO, 7.98, 12.35,\n" \
"LINE_TO, 6.88, 11.23,\n" \
"LINE_TO, 10, 8.1,\n" \
"LINE_TO, 13.13, 11.23,\n" \
"LINE_TO, 12.02, 12.35,\n" \
"LINE_TO, 10.79, 11.13,\n" \
"V_LINE_TO, 14.44,\n" \
"H_LINE_TO, 15.48,\n" \
"CUBIC_TO, 16.02, 14.44, 16.48, 14.25, 16.85, 13.88,\n" \
"CUBIC_TO, 17.23, 13.5, 17.42, 13.04, 17.42, 12.5,\n" \
"CUBIC_TO, 17.42, 11.96, 17.23, 11.5, 16.85, 11.13,\n" \
"CUBIC_TO, 16.48, 10.75, 16.02, 10.56, 15.48, 10.56,\n" \
"H_LINE_TO, 13.92,\n" \
"V_LINE_TO, 8.98,\n" \
"CUBIC_TO, 13.92, 7.99, 13.53, 7.18, 12.75, 6.54,\n" \
"CUBIC_TO, 11.97, 5.89, 11.05, 5.56, 9.98, 5.56,\n" \
"CUBIC_TO, 8.92, 5.56, 8.03, 5.96, 7.29, 6.75,\n" \
"CUBIC_TO, 6.57, 7.53, 6.13, 8.47, 5.98, 9.56,\n" \
"H_LINE_TO, 5.02,\n" \
"CUBIC_TO, 4.34, 9.56, 3.76, 9.81, 3.29, 10.29,\n" \
"CUBIC_TO, 2.82, 10.76, 2.58, 11.33, 2.58, 12,\n" \
"CUBIC_TO, 2.58, 12.68, 2.82, 13.26, 3.29, 13.73,\n" \
"CUBIC_TO, 3.76, 14.2, 4.34, 14.44, 5.02, 14.44,\n" \
"H_LINE_TO, 7.71,\n" \
"V_LINE_TO, 16.17,\n" \
"H_LINE_TO, 5.02,\n" \
"CLOSE,\n" \
"NEW_PATH\n" \
; \
              CheckThatParsedElementsMatch(s, kSaveCloudIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseScienceIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 5, 21,\n" \
"R_CUBIC_TO, -0.85, 0, -1.45, -0.38, -1.81, -1.14,\n" \
"R_CUBIC_TO, -0.36, -0.76, -0.27, -1.46, 0.26, -2.11,\n" \
"LINE_TO, 9, 11,\n" \
"V_LINE_TO, 5,\n" \
"H_LINE_TO, 8,\n" \
"R_ARC_TO, 0.96, 0.96, 0, 0, 1, -0.71, -0.29,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 7, 4,\n" \
"R_CUBIC_TO, 0, -0.28, 0.1, -0.52, 0.29, -0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 8, 3,\n" \
"R_H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0.28, 0, 0.52, 0.1, 0.71, 0.29,\n" \
"R_CUBIC_TO, 0.19, 0.19, 0.29, 0.43, 0.29, 0.71,\n" \
"R_CUBIC_TO, 0, 0.28, -0.1, 0.52, -0.29, 0.71,\n" \
"ARC_TO, 0.96, 0.96, 0, 0, 1, 16, 5,\n" \
"R_H_LINE_TO, -1,\n" \
"R_V_LINE_TO, 6,\n" \
"R_LINE_TO, 5.55, 6.75,\n" \
"R_CUBIC_TO, 0.53, 0.65, 0.62, 1.36, 0.26, 2.11,\n" \
"CUBIC_TO, 20.45, 20.62, 19.85, 21, 19, 21,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -2,\n" \
"R_H_LINE_TO, 14,\n" \
"R_LINE_TO, -6, -7.3,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 6.7,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7, -7,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 17,\n" \
"R_CUBIC_TO, -0.64, 0, -1.1, -0.28, -1.37, -0.84,\n" \
"R_CUBIC_TO, -0.28, -0.56, -0.21, -1.09, 0.19, -1.58,\n" \
"LINE_TO, 7.5, 9.5,\n" \
"R_V_LINE_TO, -5,\n" \
"R_H_LINE_TO, -0.75,\n" \
"R_CUBIC_TO, -0.21, 0, -0.39, -0.07, -0.53, -0.21,\n" \
"ARC_TO, 0.72, 0.72, 0, 0, 1, 6, 3.75,\n" \
"R_CUBIC_TO, 0, -0.21, 0.07, -0.39, 0.22, -0.53,\n" \
"ARC_TO, 0.73, 0.73, 0, 0, 1, 6.75, 3,\n" \
"R_H_LINE_TO, 6.5,\n" \
"R_CUBIC_TO, 0.21, 0, 0.39, 0.07, 0.54, 0.22,\n" \
"R_CUBIC_TO, 0.14, 0.14, 0.22, 0.32, 0.22, 0.53,\n" \
"R_CUBIC_TO, 0, 0.21, -0.07, 0.39, -0.21, 0.54,\n" \
"R_ARC_TO, 0.73, 0.73, 0, 0, 1, -0.53, 0.22,\n" \
"R_H_LINE_TO, -0.75,\n" \
"R_V_LINE_TO, 5,\n" \
"R_LINE_TO, 4.19, 5.08,\n" \
"R_CUBIC_TO, 0.39, 0.49, 0.45, 1.01, 0.17, 1.57,\n" \
"R_CUBIC_TO, -0.28, 0.56, -0.73, 0.84, -1.35, 0.84,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 11,\n" \
"LINE_TO, 11, 10,\n" \
"V_LINE_TO, 4.5,\n" \
"H_LINE_TO, 9,\n" \
"V_LINE_TO, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.52, -5.5,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.6, 13.73,\n" \
"R_CUBIC_TO, -0.57, 0, -0.98, -0.25, -1.22, -0.75,\n" \
"R_CUBIC_TO, -0.25, -0.5, -0.19, -0.97, 0.17, -1.41,\n" \
"R_LINE_TO, 3.32, -4.03,\n" \
"V_LINE_TO, 3.69,\n" \
"R_H_LINE_TO, -0.47,\n" \
"R_ARC_TO, 0.68, 0.68, 0, 0, 1, -0.49, -0.2,\n" \
"R_ARC_TO, 0.67, 0.67, 0, 0, 1, -0.2, -0.49,\n" \
"R_ARC_TO, 0.67, 0.67, 0, 0, 1, 0.2, -0.49,\n" \
"R_ARC_TO, 0.66, 0.66, 0, 0, 1, 0.49, -0.2,\n" \
"R_H_LINE_TO, 5.23,\n" \
"R_CUBIC_TO, 0.19, 0, 0.36, 0.07, 0.49, 0.2,\n" \
"R_ARC_TO, 0.66, 0.66, 0, 0, 1, 0.2, 0.49,\n" \
"R_ARC_TO, 0.67, 0.67, 0, 0, 1, -0.2, 0.49,\n" \
"R_ARC_TO, 0.66, 0.66, 0, 0, 1, -0.49, 0.2,\n" \
"R_H_LINE_TO, -0.48,\n" \
"R_V_LINE_TO, 3.86,\n" \
"R_LINE_TO, 3.32, 4.03,\n" \
"R_CUBIC_TO, 0.35, 0.43, 0.4, 0.9, 0.15, 1.4,\n" \
"R_CUBIC_TO, -0.25, 0.5, -0.65, 0.75, -1.21, 0.75,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.1, -1.38,\n" \
"R_H_LINE_TO, 8.59,\n" \
"LINE_TO, 8.75, 8.02,\n" \
"V_LINE_TO, 3.69,\n" \
"H_LINE_TO, 7.25,\n" \
"V_LINE_TO, 8.02,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.02, 8.02,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kScienceIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseScreenRecordIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.95f, 17.95f,\n" \
"CUBIC_TO, 3.33f, 17.12f, 2.86f, 16.2f, 2.51f, 15.19f,\n" \
"CUBIC_TO, 2.17f, 14.18f, 2, 13.12f, 2, 12,\n" \
"CUBIC_TO, 2, 10.88f, 2.17f, 9.82f, 2.5f, 8.82f,\n" \
"CUBIC_TO, 2.83f, 7.82f, 3.31f, 6.91f, 3.93f, 6.07f,\n" \
"LINE_TO, 5.38f, 7.5f,\n" \
"CUBIC_TO, 4.94f, 8.13f, 4.61f, 8.83f, 4.36f, 9.59f,\n" \
"CUBIC_TO, 4.12f, 10.35f, 4, 11.15f, 4, 12,\n" \
"CUBIC_TO, 4, 12.85f, 4.12f, 13.66f, 4.36f, 14.43f,\n" \
"CUBIC_TO, 4.61f, 15.19f, 4.94f, 15.89f, 5.38f, 16.52f,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 22,\n" \
"CUBIC_TO, 10.88f, 22, 9.82f, 21.83f, 8.82f, 21.5f,\n" \
"CUBIC_TO, 7.82f, 21.17f, 6.91f, 20.69f, 6.07f, 20.07f,\n" \
"LINE_TO, 7.5f, 18.63f,\n" \
"CUBIC_TO, 8.13f, 19.06f, 8.83f, 19.39f, 9.59f, 19.64f,\n" \
"CUBIC_TO, 10.35f, 19.88f, 11.15f, 20, 12, 20,\n" \
"CUBIC_TO, 12.85f, 20, 13.65f, 19.88f, 14.41f, 19.64f,\n" \
"CUBIC_TO, 15.17f, 19.39f, 15.87f, 19.06f, 16.5f, 18.63f,\n" \
"LINE_TO, 17.93f, 20.07f,\n" \
"CUBIC_TO, 17.09f, 20.69f, 16.18f, 21.17f, 15.18f, 21.5f,\n" \
"CUBIC_TO, 14.18f, 21.83f, 13.12f, 22, 12, 22,\n" \
"CLOSE,\n" \
"MOVE_TO, 20.05f, 17.95f,\n" \
"LINE_TO, 18.63f, 16.52f,\n" \
"CUBIC_TO, 19.06f, 15.89f, 19.39f, 15.19f, 19.64f, 14.43f,\n" \
"CUBIC_TO, 19.88f, 13.66f, 20, 12.85f, 20, 12,\n" \
"CUBIC_TO, 20, 11.15f, 19.88f, 10.35f, 19.64f, 9.59f,\n" \
"CUBIC_TO, 19.39f, 8.83f, 19.06f, 8.13f, 18.63f, 7.5f,\n" \
"LINE_TO, 20.07f, 6.07f,\n" \
"CUBIC_TO, 20.69f, 6.91f, 21.17f, 7.82f, 21.5f, 8.82f,\n" \
"CUBIC_TO, 21.83f, 9.82f, 22, 10.88f, 22, 12,\n" \
"CUBIC_TO, 22, 13.12f, 21.83f, 14.18f, 21.49f, 15.19f,\n" \
"CUBIC_TO, 21.14f, 16.2f, 20.67f, 17.12f, 20.05f, 17.95f,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.48f, 5.38f,\n" \
"LINE_TO, 6.05f, 3.95f,\n" \
"CUBIC_TO, 6.88f, 3.33f, 7.8f, 2.86f, 8.81f, 2.51f,\n" \
"CUBIC_TO, 9.82f, 2.17f, 10.88f, 2, 12, 2,\n" \
"CUBIC_TO, 13.13f, 2, 14.2f, 2.17f, 15.2f, 2.51f,\n" \
"CUBIC_TO, 16.2f, 2.86f, 17.12f, 3.33f, 17.95f, 3.95f,\n" \
"LINE_TO, 16.52f, 5.38f,\n" \
"CUBIC_TO, 15.89f, 4.94f, 15.19f, 4.61f, 14.43f, 4.36f,\n" \
"CUBIC_TO, 13.66f, 4.12f, 12.85f, 4, 12, 4,\n" \
"CUBIC_TO, 11.15f, 4, 10.34f, 4.12f, 9.57f, 4.36f,\n" \
"CUBIC_TO, 8.81f, 4.61f, 8.11f, 4.94f, 7.48f, 5.38f,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 17,\n" \
"CUBIC_TO, 10.62f, 17, 9.44f, 16.51f, 8.46f, 15.54f,\n" \
"CUBIC_TO, 7.49f, 14.56f, 7, 13.38f, 7, 12,\n" \
"CUBIC_TO, 7, 10.62f, 7.49f, 9.44f, 8.46f, 8.46f,\n" \
"CUBIC_TO, 9.44f, 7.49f, 10.62f, 7, 12, 7,\n" \
"CUBIC_TO, 13.38f, 7, 14.56f, 7.49f, 15.54f, 8.46f,\n" \
"CUBIC_TO, 16.51f, 9.44f, 17, 10.62f, 17, 12,\n" \
"CUBIC_TO, 17, 13.38f, 16.51f, 14.56f, 15.54f, 15.54f,\n" \
"CUBIC_TO, 14.56f, 16.51f, 13.38f, 17, 12, 17,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kScreenRecordIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseScreenShareIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 5.57, 9.23,\n" \
"R_H_LINE_TO, 1.27,\n" \
"V_LINE_TO, 8.2,\n" \
"R_ARC_TO, 0.75, 0.75, 0, 0, 1, 0.77, -0.77,\n" \
"H_LINE_TO, 8.77,\n" \
"V_LINE_TO, 8.87,\n" \
"R_LINE_TO, 2.07, -2.07,\n" \
"LINE_TO, 8.77, 4.73,\n" \
"R_V_LINE_TO, 1.44,\n" \
"H_LINE_TO, 7.2,\n" \
"R_CUBIC_TO, -0.45, 0, -0.84, 0.16, -1.16, 0.48,\n" \
"R_CUBIC_TO, -0.32, 0.32, -0.48, 0.7, -0.48, 1.16,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.72, 2.5,\n" \
"R_CUBIC_TO, -0.38, 0, -0.71, -0.13, -0.98, -0.4,\n" \
"R_ARC_TO, 1.34, 1.34, 0, 0, 1, -0.4, -0.98,\n" \
"V_LINE_TO, 3.65,\n" \
"R_CUBIC_TO, 0, -0.38, 0.13, -0.71, 0.4, -0.98,\n" \
"R_CUBIC_TO, 0.27, -0.27, 0.6, -0.4, 0.98, -0.4,\n" \
"R_H_LINE_TO, 10.31,\n" \
"R_CUBIC_TO, 0.38, 0, 0.71, 0.13, 0.98, 0.4,\n" \
"R_CUBIC_TO, 0.27, 0.27, 0.4, 0.6, 0.4, 0.98,\n" \
"R_V_LINE_TO, 6.7,\n" \
"R_CUBIC_TO, 0, 0.38, -0.13, 0.71, -0.4, 0.98,\n" \
"R_CUBIC_TO, -0.27, 0.27, -0.6, 0.4, -0.98, 0.4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.38,\n" \
"R_H_LINE_TO, 10.3,\n" \
"V_LINE_TO, 3.65,\n" \
"H_LINE_TO, 2.85,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"V_LINE_TO, 3.65,\n" \
"CLOSE,\n" \
"MOVE_TO, 0.68, 13.8,\n" \
"R_V_LINE_TO, -1.26,\n" \
"R_H_LINE_TO, 14.65,\n" \
"R_V_LINE_TO, 1.26,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kScreenShareIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseScreenShareOldIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 40, 36,\n" \
"R_CUBIC_TO, 2.2, 0, 3.98, -1.8, 3.98, -4,\n" \
"LINE_TO, 44, 12,\n" \
"R_CUBIC_TO, 0, -2.22, -1.8, -4, -4, -4,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, -2.22, 0, -4, 1.78, -4, 4,\n" \
"R_V_LINE_TO, 20,\n" \
"R_CUBIC_TO, 0, 2.2, 1.78, 4, 4, 4,\n" \
"H_LINE_TO, 0,\n" \
"R_V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 48,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, -8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -14, -7.06,\n" \
"R_V_LINE_TO, -4.38,\n" \
"R_CUBIC_TO, -5.56, 0, -9.22, 1.7, -12, 5.44,\n" \
"R_CUBIC_TO, 1.12, -5.34, 4.22, -10.66, 12, -11.74,\n" \
"V_LINE_TO, 14,\n" \
"R_LINE_TO, 8, 7.46,\n" \
"R_LINE_TO, -8, 7.48,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kScreenShareOldIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSearchIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 32,\n" \
"MOVE_TO, 20.29, 19.45,\n" \
"LINE_TO, 19.93, 19.09,\n" \
"CUBIC_TO, 21.23, 17.59, 22.13, 15.52, 22.13, 12.96,\n" \
"CUBIC_TO, 22.13, 8.23, 17.85, 4, 13.07, 4,\n" \
"CUBIC_TO, 8.28, 4, 4, 8.23, 4, 12.96,\n" \
"CUBIC_TO, 4, 17.69, 9.18, 21.92, 13.26, 21.92,\n" \
"CUBIC_TO, 15.66, 21.92, 17.51, 21.26, 18.98, 20.02,\n" \
"LINE_TO, 19.35, 20.38,\n" \
"LINE_TO, 19.35, 21.42,\n" \
"LINE_TO, 26.01, 28,\n" \
"LINE_TO, 28, 26.04,\n" \
"LINE_TO, 21.35, 19.45,\n" \
"LINE_TO, 20.29, 19.45,\n" \
"CLOSE,\n" \
"MOVE_TO, 13.07, 19.36,\n" \
"CUBIC_TO, 9.49, 19.36, 6.59, 16.5, 6.59, 12.96,\n" \
"CUBIC_TO, 6.59, 9.43, 9.49, 6.56, 13.07, 6.56,\n" \
"CUBIC_TO, 16.64, 6.56, 19.54, 9.43, 19.54, 12.96,\n" \
"CUBIC_TO, 19.54, 16.5, 16.64, 19.36, 13.07, 19.36,\n" \
"LINE_TO, 13.07, 19.36,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSearchIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSearchChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 15.94, 17,\n" \
"R_LINE_TO, -4.98, -4.98,\n" \
"R_ARC_TO, 4.99, 4.99, 0, 0, 1, -1.37, 0.72,\n" \
"ARC_TO, 4.71, 4.71, 0, 0, 1, 8, 13,\n" \
"R_CUBIC_TO, -1.39, 0, -2.57, -0.48, -3.54, -1.46,\n" \
"CUBIC_TO, 3.48, 10.57, 3, 9.39, 3, 8,\n" \
"R_CUBIC_TO, 0, -1.39, 0.48, -2.57, 1.46, -3.54,\n" \
"CUBIC_TO, 5.43, 3.48, 6.61, 3, 8, 3,\n" \
"R_CUBIC_TO, 1.39, 0, 2.57, 0.48, 3.54, 1.46,\n" \
"CUBIC_TO, 12.52, 5.43, 13, 6.61, 13, 8,\n" \
"R_CUBIC_TO, 0, 0.56, -0.09, 1.09, -0.26, 1.59,\n" \
"R_CUBIC_TO, -0.17, 0.5, -0.41, 0.96, -0.72, 1.37,\n" \
"R_LINE_TO, 4.98, 4.98,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 11.5,\n" \
"R_CUBIC_TO, 0.97, 0, 1.8, -0.34, 2.48, -1.02,\n" \
"R_CUBIC_TO, 0.68, -0.68, 1.02, -1.51, 1.02, -2.48,\n" \
"R_CUBIC_TO, 0, -0.97, -0.34, -1.8, -1.02, -2.48,\n" \
"CUBIC_TO, 9.8, 4.84, 8.97, 4.5, 8, 4.5,\n" \
"R_CUBIC_TO, -0.97, 0, -1.8, 0.34, -2.48, 1.02,\n" \
"CUBIC_TO, 4.84, 6.2, 4.5, 7.03, 4.5, 8,\n" \
"R_CUBIC_TO, 0, 0.97, 0.34, 1.8, 1.02, 2.48,\n" \
"R_CUBIC_TO, 0.68, 0.68, 1.51, 1.02, 2.48, 1.02,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSearchChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSelectWindowIcon) { \
              std::string s = \
"// Copyright 2020 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 21, 1,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 6,\n" \
"H_LINE_TO, 3,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 13,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"R_V_LINE_TO, -6,\n" \
"R_H_LINE_TO, 3,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5, 20,\n" \
"H_LINE_TO, 3,\n" \
"R_V_LINE_TO, -8,\n" \
"R_H_LINE_TO, 13,\n" \
"R_V_LINE_TO, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5, -8,\n" \
"R_H_LINE_TO, -3,\n" \
"R_V_LINE_TO, -2,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"H_LINE_TO, 8,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 13,\n" \
"R_V_LINE_TO, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSelectWindowIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSelectWindowChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.5, 18,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 2, 16.5,\n" \
"R_V_LINE_TO, -7,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 8,\n" \
"H_LINE_TO, 5,\n" \
"V_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 6.5, 2,\n" \
"R_H_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 12,\n" \
"H_LINE_TO, 15,\n" \
"R_V_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 13.5, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 10,\n" \
"V_LINE_TO, 11,\n" \
"R_H_LINE_TO, -10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 11.5, -6,\n" \
"R_H_LINE_TO, 1.5,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, -10,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSelectWindowChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSelectWindowOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16.73, 18.83,\n" \
"LINE_TO, 8.9, 11,\n" \
"H_LINE_TO, 3.5,\n" \
"R_V_LINE_TO, 5.5,\n" \
"R_H_LINE_TO, 10,\n" \
"R_V_LINE_TO, -3,\n" \
"R_LINE_TO, 1.5, 1.5,\n" \
"R_V_LINE_TO, 1.51,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.76, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 13.5, 18,\n" \
"R_H_LINE_TO, -10,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 2, 16.5,\n" \
"V_LINE_TO, 9.5,\n" \
"R_CUBIC_TO, 0, -0.43, 0.14, -0.79, 0.44, -1.09,\n" \
"ARC_TO, 1.31, 1.31, 0, 0, 1, 3.5, 8,\n" \
"H_LINE_TO, 5,\n" \
"R_V_LINE_TO, -0.87,\n" \
"LINE_TO, 1.17, 3.27,\n" \
"LINE_TO, 2.23, 2.23,\n" \
"R_LINE_TO, 15.54, 15.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 15, 12.88,\n" \
"LINE_TO, 10.13, 8,\n" \
"H_LINE_TO, 13.5,\n" \
"R_CUBIC_TO, 0.43, 0, 0.79, 0.14, 1.07, 0.43,\n" \
"R_CUBIC_TO, 0.28, 0.28, 0.43, 0.64, 0.43, 1.07,\n" \
"R_V_LINE_TO, 1,\n" \
"R_H_LINE_TO, 1.5,\n" \
"V_LINE_TO, 5,\n" \
"H_LINE_TO, 7.13,\n" \
"R_LINE_TO, -2.02, -2.02,\n" \
"R_CUBIC_TO, 0.1, -0.31, 0.27, -0.55, 0.53, -0.72,\n" \
"CUBIC_TO, 5.9, 2.09, 6.18, 2, 6.5, 2,\n" \
"R_H_LINE_TO, 10,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 12,\n" \
"H_LINE_TO, 15,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSelectWindowOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSendIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 3, 3,\n" \
"LINE_TO, 18, 10,\n" \
"LINE_TO, 3, 17,\n" \
"V_LINE_TO, 3,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5, 5.36,\n" \
"LINE_TO, 14.45, 10,\n" \
"LINE_TO, 4.5, 14.64,\n" \
"V_LINE_TO, 12,\n" \
"LINE_TO, 9, 10,\n" \
"LINE_TO, 4.5, 8,\n" \
"V_LINE_TO, 5.36,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSendIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSensorsIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 18,\n" \
"MOVE_TO, 9, 7.6,\n" \
"CUBIC_TO, 8.2, 7.6, 7.6, 8.2, 7.6, 9,\n" \
"R_CUBIC_TO, 0, 0.8, 0.6, 1.4, 1.4, 1.4,\n" \
"R_CUBIC_TO, 0.8, 0, 1.4, -0.6, 1.4, -1.4,\n" \
"CUBIC_TO_SHORTHAND, 9.8, 7.6, 9, 7.6,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.7, 5.1,\n" \
"CUBIC_TO, 5.4, 5.9, 4.5, 7.3, 4.5, 9,\n" \
"R_CUBIC_TO, 0, 1.7, 0.9, 3.2, 2.2, 4,\n" \
"R_LINE_TO, 0.7, -1.3,\n" \
"CUBIC_TO, 6.6, 11.1, 6, 10.1, 6, 9,\n" \
"R_CUBIC_TO, 0, -1.1, 0.6, -2.1, 1.5, -2.6,\n" \
"LINE_TO, 6.7, 5.1,\n" \
"CLOSE,\n" \
"MOVE_TO, 13.5, 9,\n" \
"R_CUBIC_TO, 0, -1.7, -0.9, -3.1, -2.2, -3.9,\n" \
"R_LINE_TO, -0.8, 1.3,\n" \
"CUBIC_TO, 11.4, 6.9, 12, 7.9, 12, 9,\n" \
"R_CUBIC_TO, 0, 1.1, -0.6, 2.1, -1.5, 2.6,\n" \
"R_LINE_TO, 0.8, 1.3,\n" \
"CUBIC_TO, 12.6, 12.2, 13.5, 10.7, 13.5, 9,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.2, 2.4,\n" \
"CUBIC_TO, 3, 3.8, 1.5, 6.2, 1.5, 9,\n" \
"R_CUBIC_TO, 0, 2.8, 1.5, 5.3, 3.7, 6.6,\n" \
"LINE_TO, 6, 14.2,\n" \
"R_CUBIC_TO, -1.8, -1.1, -3, -3, -3, -5.3,\n" \
"R_CUBIC_TO, 0, -2.2, 1.2, -4.2, 3, -5.2,\n" \
"LINE_TO, 5.2, 2.4,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 3.7,\n" \
"R_CUBIC_TO, 1.8, 1.1, 3, 3, 3, 5.2,\n" \
"R_CUBIC_TO, 0, 2.2, -1.2, 4.2, -3, 5.3,\n" \
"R_LINE_TO, 0.8, 1.3,\n" \
"R_CUBIC_TO, 2.2, -1.3, 3.8, -3.8, 3.8, -6.6,\n" \
"R_CUBIC_TO, 0, -2.8, -1.5, -5.2, -3.7, -6.6,\n" \
"LINE_TO, 12, 3.7,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSensorsIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSensorsChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.36, 15.65,\n" \
"R_ARC_TO, 7.89, 7.89, 0, 0, 1, -1.73, -2.53,\n" \
"ARC_TO, 7.72, 7.72, 0, 0, 1, 2, 10.01,\n" \
"R_CUBIC_TO, 0, -1.12, 0.21, -2.16, 0.63, -3.12,\n" \
"R_ARC_TO, 8.37, 8.37, 0, 0, 1, 1.71, -2.56,\n" \
"R_LINE_TO, 1.06, 1.06,\n" \
"R_ARC_TO, 6.81, 6.81, 0, 0, 0, -1.38, 2.08,\n" \
"R_ARC_TO, 6.28, 6.28, 0, 0, 0, -0.51, 2.52,\n" \
"R_ARC_TO, 6.33, 6.33, 0, 0, 0, 1.92, 4.59,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.1, -2.1,\n" \
"R_ARC_TO, 5.5, 5.5, 0, 0, 1, -1.06, -1.59,\n" \
"ARC_TO, 4.85, 4.85, 0, 0, 1, 5, 10,\n" \
"R_CUBIC_TO, 0, -0.7, 0.13, -1.36, 0.4, -1.96,\n" \
"R_ARC_TO, 5.48, 5.48, 0, 0, 1, 1.06, -1.59,\n" \
"LINE_TO, 7.52, 7.52,\n" \
"R_ARC_TO, 3.59, 3.59, 0, 0, 0, -0.75, 1.12,\n" \
"R_ARC_TO, 3.4, 3.4, 0, 0, 0, -0.27, 1.36,\n" \
"R_CUBIC_TO, 0, 0.48, 0.09, 0.94, 0.27, 1.36,\n" \
"R_CUBIC_TO, 0.18, 0.42, 0.43, 0.8, 0.75, 1.12,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.54, -1.79,\n" \
"R_ARC_TO, 1.67, 1.67, 0, 0, 1, -1.24, -0.52,\n" \
"R_ARC_TO, 1.7, 1.7, 0, 0, 1, -0.51, -1.24,\n" \
"R_CUBIC_TO, 0, -0.48, 0.17, -0.89, 0.52, -1.23,\n" \
"R_CUBIC_TO, 0.34, -0.34, 0.76, -0.51, 1.24, -0.51,\n" \
"R_CUBIC_TO, 0.48, 0, 0.89, 0.17, 1.23, 0.52,\n" \
"R_CUBIC_TO, 0.34, 0.34, 0.51, 0.76, 0.51, 1.24,\n" \
"R_CUBIC_TO, 0, 0.48, -0.17, 0.89, -0.52, 1.23,\n" \
"R_CUBIC_TO, -0.34, 0.34, -0.76, 0.51, -1.24, 0.51,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.52, 1.77,\n" \
"R_LINE_TO, -1.04, -1.04,\n" \
"R_CUBIC_TO, 0.32, -0.32, 0.57, -0.69, 0.75, -1.12,\n" \
"R_ARC_TO, 3.4, 3.4, 0, 0, 0, 0.27, -1.36,\n" \
"R_CUBIC_TO, 0, -0.48, -0.09, -0.94, -0.27, -1.36,\n" \
"R_ARC_TO, 3.57, 3.57, 0, 0, 0, -0.75, -1.12,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_CUBIC_TO, 0.44, 0.46, 0.8, 0.99, 1.06, 1.59,\n" \
"ARC_TO, 4.89, 4.89, 0, 0, 1, 15, 10.01,\n" \
"R_ARC_TO, 4.83, 4.83, 0, 0, 1, -1.48, 3.51,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.13, 2.13,\n" \
"R_LINE_TO, -1.06, -1.06,\n" \
"R_ARC_TO, 6.33, 6.33, 0, 0, 0, 1.41, -2.06,\n" \
"R_CUBIC_TO, 0.34, -0.78, 0.51, -1.62, 0.51, -2.52,\n" \
"R_CUBIC_TO, 0, -0.9, -0.17, -1.75, -0.51, -2.54,\n" \
"R_ARC_TO, 6.84, 6.84, 0, 0, 0, -1.38, -2.07,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_ARC_TO, 8.37, 8.37, 0, 0, 1, 1.71, 2.56,\n" \
"R_ARC_TO, 7.76, 7.76, 0, 0, 1, 0.63, 3.12,\n" \
"R_CUBIC_TO, 0, 1.11, -0.21, 2.14, -0.62, 3.11,\n" \
"R_ARC_TO, 7.89, 7.89, 0, 0, 1, -1.73, 2.53,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSensorsChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSensorsOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16, 18.13,\n" \
"LINE_TO, 6.71, 8.83,\n" \
"R_ARC_TO, 3.53, 3.53, 0, 0, 0, -0.16, 0.56,\n" \
"R_CUBIC_TO, -0.04, 0.2, -0.05, 0.4, -0.05, 0.61,\n" \
"R_CUBIC_TO, 0, 0.48, 0.09, 0.94, 0.27, 1.36,\n" \
"R_CUBIC_TO, 0.18, 0.43, 0.43, 0.8, 0.75, 1.12,\n" \
"R_LINE_TO, -1.04, 1.04,\n" \
"ARC_TO, 4.86, 4.86, 0, 0, 1, 5, 10,\n" \
"R_CUBIC_TO, 0, -0.42, 0.05, -0.82, 0.16, -1.2,\n" \
"R_ARC_TO, 5.72, 5.72, 0, 0, 1, 0.43, -1.09,\n" \
"LINE_TO, 4.48, 6.58,\n" \
"R_CUBIC_TO, -0.31, 0.5, -0.55, 1.04, -0.72, 1.62,\n" \
"ARC_TO, 6.12, 6.12, 0, 0, 0, 3.5, 10,\n" \
"R_ARC_TO, 6.36, 6.36, 0, 0, 0, 1.92, 4.58,\n" \
"R_LINE_TO, -1.06, 1.06,\n" \
"R_ARC_TO, 7.89, 7.89, 0, 0, 1, -1.73, -2.53,\n" \
"ARC_TO, 7.75, 7.75, 0, 0, 1, 2, 10,\n" \
"R_ARC_TO, 7.92, 7.92, 0, 0, 1, 1.38, -4.5,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.63, -3.62,\n" \
"R_LINE_TO, -1.1, -1.08,\n" \
"R_CUBIC_TO, 0.31, -0.5, 0.55, -1.04, 0.72, -1.62,\n" \
"ARC_TO, 6.12, 6.12, 0, 0, 0, 16.5, 10,\n" \
"R_ARC_TO, 6.36, 6.36, 0, 0, 0, -1.92, -4.58,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_ARC_TO, 7.89, 7.89, 0, 0, 1, 1.73, 2.53,\n" \
"ARC_TO, 7.75, 7.75, 0, 0, 1, 18, 10,\n" \
"R_CUBIC_TO, 0, 0.83, -0.12, 1.63, -0.35, 2.4,\n" \
"R_ARC_TO, 7.51, 7.51, 0, 0, 1, -1.02, 2.11,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.19, -2.19,\n" \
"R_LINE_TO, -1.14, -1.14,\n" \
"R_CUBIC_TO, 0.07, -0.19, 0.12, -0.39, 0.16, -0.57,\n" \
"R_CUBIC_TO, 0.04, -0.19, 0.05, -0.39, 0.05, -0.59,\n" \
"R_CUBIC_TO, 0, -0.48, -0.09, -0.94, -0.27, -1.36,\n" \
"R_ARC_TO, 3.6, 3.6, 0, 0, 0, -0.75, -1.12,\n" \
"R_LINE_TO, 1.04, -1.04,\n" \
"ARC_TO, 4.86, 4.86, 0, 0, 1, 15, 10,\n" \
"R_CUBIC_TO, 0, 0.42, -0.05, 0.82, -0.14, 1.21,\n" \
"R_CUBIC_TO, -0.09, 0.39, -0.23, 0.76, -0.42, 1.11,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSensorsOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSerialPortIcon) { \
              std::string s = \
"// Copyright 2019 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 22, 9,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, -2,\n" \
"V_LINE_TO, 5,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"H_LINE_TO, 4,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 14,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 14,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, -2,\n" \
"V_LINE_TO, 9,\n" \
"R_H_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4, 10,\n" \
"H_LINE_TO, 4,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, 14,\n" \
"R_V_LINE_TO, 14,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 13,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 4,\n" \
"H_LINE_TO, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6, -6,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, -4,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 7,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 5,\n" \
"H_LINE_TO, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6, 4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 6,\n" \
"R_H_LINE_TO, -4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSerialPortIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSerialPortChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.5, 17,\n" \
"R_CUBIC_TO, -0.4, 0, -0.75, -0.15, -1.05, -0.45,\n" \
"R_CUBIC_TO, -0.3, -0.3, -0.45, -0.65, -0.45, -1.05,\n" \
"R_V_LINE_TO, -11,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.45, -1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 3.5, 3,\n" \
"R_H_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0.4, 0, 0.75, 0.15, 1.05, 0.44,\n" \
"R_CUBIC_TO, 0.3, 0.29, 0.45, 0.65, 0.45, 1.06,\n" \
"V_LINE_TO, 6,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 1.75,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 1.75,\n" \
"R_H_LINE_TO, 2,\n" \
"V_LINE_TO, 14,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 1.49,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.45, 1.06,\n" \
"R_CUBIC_TO, -0.3, 0.3, -0.65, 0.45, -1.05, 0.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 11,\n" \
"R_V_LINE_TO, -11,\n" \
"R_H_LINE_TO, -11,\n" \
"CLOSE,\n" \
"MOVE_TO, 5, 14,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -3,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5, -6,\n" \
"R_H_LINE_TO, 3,\n" \
"V_LINE_TO, 6,\n" \
"R_H_LINE_TO, -3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5, 2,\n" \
"R_H_LINE_TO, 4,\n" \
"V_LINE_TO, 6,\n" \
"H_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5, 4,\n" \
"R_H_LINE_TO, 3,\n" \
"V_LINE_TO, 9,\n" \
"R_H_LINE_TO, -3,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.5, 4.5,\n" \
"R_V_LINE_TO, 11,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSerialPortChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSerialPortOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16.73, 18.83,\n" \
"LINE_TO, 1.17, 3.27,\n" \
"LINE_TO, 2.23, 2.23,\n" \
"R_LINE_TO, 15.54, 15.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.13, 3,\n" \
"R_H_LINE_TO, 9.37,\n" \
"R_CUBIC_TO, 0.42, 0, 0.78, 0.15, 1.07, 0.45,\n" \
"R_CUBIC_TO, 0.29, 0.3, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 6,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 1.75,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, -2,\n" \
"R_V_LINE_TO, 1.75,\n" \
"R_H_LINE_TO, 2,\n" \
"V_LINE_TO, 14,\n" \
"R_H_LINE_TO, -1.87,\n" \
"LINE_TO, 14.5, 12.38,\n" \
"V_LINE_TO, 4.5,\n" \
"H_LINE_TO, 6.63,\n" \
"CLOSE,\n" \
"MOVE_TO, 9, 6.88,\n" \
"LINE_TO, 8.13, 6,\n" \
"H_LINE_TO, 9,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.13, 8,\n" \
"LINE_TO, 10, 7.88,\n" \
"V_LINE_TO, 6,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"MOVE_TO, 13, 10.88,\n" \
"LINE_TO, 11.13, 9,\n" \
"H_LINE_TO, 13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4, 0.25,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.69, -2.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 5, 14,\n" \
"R_V_LINE_TO, -3,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 3,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.13, 3.13,\n" \
"LINE_TO, 4.5, 4.5,\n" \
"R_H_LINE_TO, -1,\n" \
"R_V_LINE_TO, 11,\n" \
"R_H_LINE_TO, 11.02,\n" \
"R_V_LINE_TO, -0.98,\n" \
"LINE_TO, 16, 16,\n" \
"R_ARC_TO, 1.81, 1.81, 0, 0, 1, -0.61, 0.7,\n" \
"R_CUBIC_TO, -0.28, 0.2, -0.58, 0.3, -0.89, 0.3,\n" \
"R_H_LINE_TO, -11,\n" \
"R_CUBIC_TO, -0.4, 0, -0.75, -0.15, -1.05, -0.45,\n" \
"R_CUBIC_TO, -0.3, -0.3, -0.45, -0.65, -0.45, -1.05,\n" \
"R_V_LINE_TO, -11,\n" \
"R_CUBIC_TO, 0, -0.32, 0.11, -0.59, 0.34, -0.84,\n" \
"R_CUBIC_TO, 0.23, -0.25, 0.49, -0.43, 0.78, -0.53,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 10,\n" \
"R_LINE_TO, 3.33, 3.33,\n" \
"V_LINE_TO, 14,\n" \
"H_LINE_TO, 10,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 6,\n" \
"R_LINE_TO, 3.29, 3.29,\n" \
"V_LINE_TO, 10,\n" \
"H_LINE_TO, 5,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSerialPortOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSettingsIcon) { \
              std::string s = \
"// Copyright 2016 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 38.86, 25.95,\n" \
"R_CUBIC_TO, 0.08, -0.64, 0.14, -1.29, 0.14, -1.95,\n" \
"R_CUBIC_TO, 0, -0.66, -0.06, -1.31, -0.14, -1.95,\n" \
"R_LINE_TO, 4.23, -3.31,\n" \
"R_CUBIC_TO, 0.38, -0.3, 0.49, -0.84, 0.24, -1.28,\n" \
"R_LINE_TO, -4, -6.93,\n" \
"R_CUBIC_TO, -0.25, -0.43, -0.77, -0.61, -1.22, -0.43,\n" \
"R_LINE_TO, -4.98, 2.01,\n" \
"R_CUBIC_TO, -1.03, -0.79, -2.16, -1.46, -3.38, -1.97,\n" \
"LINE_TO, 29, 4.84,\n" \
"R_CUBIC_TO, -0.09, -0.47, -0.5, -0.84, -1, -0.84,\n" \
"R_H_LINE_TO, -8,\n" \
"R_CUBIC_TO, -0.5, 0, -0.91, 0.37, -0.99, 0.84,\n" \
"R_LINE_TO, -0.75, 5.3,\n" \
"R_CUBIC_TO, -1.22, 0.51, -2.35, 1.17, -3.38, 1.97,\n" \
"LINE_TO, 9.9, 10.1,\n" \
"R_CUBIC_TO, -0.45, -0.17, -0.97, 0, -1.22, 0.43,\n" \
"R_LINE_TO, -4, 6.93,\n" \
"R_CUBIC_TO, -0.25, 0.43, -0.14, 0.97, 0.24, 1.28,\n" \
"R_LINE_TO, 4.22, 3.31,\n" \
"CUBIC_TO, 9.06, 22.69, 9, 23.34, 9, 24,\n" \
"R_CUBIC_TO, 0, 0.66, 0.06, 1.31, 0.14, 1.95,\n" \
"R_LINE_TO, -4.22, 3.31,\n" \
"R_CUBIC_TO, -0.38, 0.3, -0.49, 0.84, -0.24, 1.28,\n" \
"R_LINE_TO, 4, 6.93,\n" \
"R_CUBIC_TO, 0.25, 0.43, 0.77, 0.61, 1.22, 0.43,\n" \
"R_LINE_TO, 4.98, -2.01,\n" \
"R_CUBIC_TO, 1.03, 0.79, 2.16, 1.46, 3.38, 1.97,\n" \
"R_LINE_TO, 0.75, 5.3,\n" \
"R_CUBIC_TO, 0.08, 0.47, 0.49, 0.84, 0.99, 0.84,\n" \
"R_H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0.5, 0, 0.91, -0.37, 0.99, -0.84,\n" \
"R_LINE_TO, 0.75, -5.3,\n" \
"R_CUBIC_TO, 1.22, -0.51, 2.35, -1.17, 3.38, -1.97,\n" \
"R_LINE_TO, 4.98, 2.01,\n" \
"R_CUBIC_TO, 0.45, 0.17, 0.97, 0, 1.22, -0.43,\n" \
"R_LINE_TO, 4, -6.93,\n" \
"R_CUBIC_TO, 0.25, -0.43, 0.14, -0.97, -0.24, -1.28,\n" \
"R_LINE_TO, -4.22, -3.31,\n" \
"CLOSE,\n" \
"MOVE_TO, 24, 31,\n" \
"R_CUBIC_TO, -3.87, 0, -7, -3.13, -7, -7,\n" \
"R_CUBIC_TO, 0, -3.87, 3.13, -7, 7, -7,\n" \
"R_CUBIC_TO, 3.87, 0, 7, 3.13, 7, 7,\n" \
"R_CUBIC_TO, 0, 3.87, -3.13, 7, -7, 7,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 12.58, 8.59,\n" \
"CUBIC_TO, 12.61, 8.4, 12.63, 8.2, 12.63, 8,\n" \
"CUBIC_TO, 12.63, 7.8, 12.61, 7.6, 12.58, 7.41,\n" \
"LINE_TO, 13.89, 6.42,\n" \
"CUBIC_TO, 14, 6.33, 14.03, 6.17, 13.96, 6.04,\n" \
"LINE_TO, 12.73, 3.96,\n" \
"CUBIC_TO, 12.65, 3.83, 12.49, 3.78, 12.35, 3.83,\n" \
"LINE_TO, 10.81, 4.43,\n" \
"CUBIC_TO, 10.49, 4.19, 10.15, 3.99, 9.77, 3.84,\n" \
"LINE_TO, 9.54, 2.25,\n" \
"CUBIC_TO, 9.52, 2.11, 9.39, 2, 9.24, 2,\n" \
"LINE_TO, 6.77, 2,\n" \
"CUBIC_TO, 6.61, 2, 6.48, 2.11, 6.47, 2.25,\n" \
"LINE_TO, 6.23, 3.84,\n" \
"CUBIC_TO, 5.85, 3.99, 5.51, 4.2, 5.19, 4.43,\n" \
"LINE_TO, 3.65, 3.83,\n" \
"CUBIC_TO, 3.51, 3.78, 3.35, 3.83, 3.28, 3.96,\n" \
"LINE_TO, 2.04, 6.04,\n" \
"CUBIC_TO, 1.96, 6.17, 2, 6.33, 2.12, 6.42,\n" \
"LINE_TO, 3.42, 7.41,\n" \
"CUBIC_TO, 3.39, 7.6, 3.38, 7.8, 3.38, 8,\n" \
"CUBIC_TO, 3.38, 8.2, 3.39, 8.4, 3.42, 8.59,\n" \
"LINE_TO, 2.12, 9.58,\n" \
"CUBIC_TO, 2, 9.67, 1.97, 9.83, 2.04, 9.96,\n" \
"LINE_TO, 3.28, 12.04,\n" \
"CUBIC_TO, 3.35, 12.17, 3.52, 12.22, 3.65, 12.17,\n" \
"LINE_TO, 5.19, 11.57,\n" \
"CUBIC_TO, 5.51, 11.81, 5.85, 12.01, 6.23, 12.16,\n" \
"LINE_TO, 6.47, 13.75,\n" \
"CUBIC_TO, 6.48, 13.89, 6.61, 14, 6.77, 14,\n" \
"LINE_TO, 9.24, 14,\n" \
"CUBIC_TO, 9.39, 14, 9.52, 13.89, 9.54, 13.75,\n" \
"LINE_TO, 9.77, 12.16,\n" \
"CUBIC_TO, 10.15, 12.01, 10.49, 11.8, 10.81, 11.57,\n" \
"LINE_TO, 12.35, 12.17,\n" \
"CUBIC_TO, 12.49, 12.22, 12.65, 12.17, 12.73, 12.04,\n" \
"LINE_TO, 13.96, 9.96,\n" \
"CUBIC_TO, 14.03, 9.83, 14, 9.67, 13.89, 9.58,\n" \
"LINE_TO, 12.58, 8.59,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 10,\n" \
"CUBIC_TO, 6.9, 10, 6, 9.1, 6, 8,\n" \
"CUBIC_TO, 6, 6.9, 6.9, 6, 8, 6,\n" \
"CUBIC_TO, 9.1, 6, 10, 6.9, 10, 8,\n" \
"CUBIC_TO, 10, 9.1, 9.1, 10, 8, 10,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSettingsIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSettingsChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 8.4, 18,\n" \
"R_LINE_TO, -0.46, -2.37,\n" \
"R_ARC_TO, 7.16, 7.16, 0, 0, 1, -0.93, -0.44,\n" \
"R_CUBIC_TO, -0.3, -0.17, -0.58, -0.36, -0.84, -0.58,\n" \
"R_LINE_TO, -2.29, 0.77,\n" \
"R_LINE_TO, -1.6, -2.77,\n" \
"R_LINE_TO, 1.81, -1.58,\n" \
"R_ARC_TO, 6.99, 6.99, 0, 0, 1, -0.06, -0.5,\n" \
"R_ARC_TO, 7.33, 7.33, 0, 0, 1, 0, -1.04,\n" \
"R_ARC_TO, 6.99, 6.99, 0, 0, 1, 0.06, -0.5,\n" \
"LINE_TO, 2.27, 7.4,\n" \
"R_LINE_TO, 1.61, -2.77,\n" \
"R_LINE_TO, 2.29, 0.77,\n" \
"R_CUBIC_TO, 0.26, -0.22, 0.54, -0.41, 0.84, -0.58,\n" \
"R_CUBIC_TO, 0.3, -0.17, 0.61, -0.31, 0.93, -0.44,\n" \
"LINE_TO, 8.4, 2,\n" \
"R_H_LINE_TO, 3.21,\n" \
"R_LINE_TO, 0.46, 2.38,\n" \
"R_CUBIC_TO, 0.32, 0.13, 0.63, 0.27, 0.93, 0.44,\n" \
"R_CUBIC_TO, 0.3, 0.17, 0.58, 0.36, 0.84, 0.58,\n" \
"R_LINE_TO, 2.29, -0.77,\n" \
"R_LINE_TO, 1.61, 2.77,\n" \
"R_LINE_TO, -1.81, 1.59,\n" \
"R_CUBIC_TO, 0.03, 0.17, 0.05, 0.33, 0.06, 0.5,\n" \
"R_ARC_TO, 7.33, 7.33, 0, 0, 1, 0, 1.04,\n" \
"R_ARC_TO, 6.99, 6.99, 0, 0, 1, -0.06, 0.5,\n" \
"R_LINE_TO, 1.81, 1.59,\n" \
"R_LINE_TO, -1.6, 2.77,\n" \
"R_LINE_TO, -2.29, -0.77,\n" \
"R_ARC_TO, 5.23, 5.23, 0, 0, 1, -0.84, 0.58,\n" \
"R_ARC_TO, 7.16, 7.16, 0, 0, 1, -0.92, 0.44,\n" \
"LINE_TO, 11.6, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.23, -1.5,\n" \
"R_H_LINE_TO, 0.75,\n" \
"R_LINE_TO, 0.4, -2.06,\n" \
"R_ARC_TO, 4.59, 4.59, 0, 0, 0, 1.48, -0.54,\n" \
"R_ARC_TO, 4.31, 4.31, 0, 0, 0, 1.19, -1,\n" \
"R_LINE_TO, 2, 0.67,\n" \
"R_LINE_TO, 0.37, -0.63,\n" \
"R_LINE_TO, -1.58, -1.39,\n" \
"R_CUBIC_TO, 0.08, -0.24, 0.15, -0.48, 0.2, -0.74,\n" \
"R_CUBIC_TO, 0.05, -0.25, 0.07, -0.52, 0.07, -0.8,\n" \
"R_CUBIC_TO, 0, -0.28, -0.02, -0.55, -0.07, -0.8,\n" \
"R_ARC_TO, 5.05, 5.05, 0, 0, 0, -0.2, -0.74,\n" \
"R_LINE_TO, 1.58, -1.39,\n" \
"R_LINE_TO, -0.37, -0.62,\n" \
"R_LINE_TO, -2, 0.67,\n" \
"R_ARC_TO, 4.31, 4.31, 0, 0, 0, -1.19, -1,\n" \
"R_ARC_TO, 4.59, 4.59, 0, 0, 0, -1.48, -0.54,\n" \
"LINE_TO, 10.38, 3.5,\n" \
"R_H_LINE_TO, -0.75,\n" \
"LINE_TO, 9.23, 5.56,\n" \
"R_ARC_TO, 4.59, 4.59, 0, 0, 0, -1.48, 0.54,\n" \
"R_ARC_TO, 4.31, 4.31, 0, 0, 0, -1.19, 1,\n" \
"R_LINE_TO, -2, -0.67,\n" \
"R_LINE_TO, -0.37, 0.63,\n" \
"LINE_TO, 5.77, 8.46,\n" \
"R_ARC_TO, 5.05, 5.05, 0, 0, 0, -0.2, 0.74,\n" \
"R_CUBIC_TO, -0.05, 0.25, -0.07, 0.52, -0.07, 0.8,\n" \
"R_CUBIC_TO, 0, 0.28, 0.02, 0.55, 0.07, 0.8,\n" \
"R_CUBIC_TO, 0.05, 0.26, 0.11, 0.51, 0.2, 0.74,\n" \
"R_LINE_TO, -1.58, 1.4,\n" \
"R_LINE_TO, 0.38, 0.62,\n" \
"R_LINE_TO, 2, -0.67,\n" \
"R_CUBIC_TO, 0.33, 0.4, 0.73, 0.74, 1.19, 1,\n" \
"R_ARC_TO, 4.59, 4.59, 0, 0, 0, 1.48, 0.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 13,\n" \
"R_CUBIC_TO, 0.83, 0, 1.54, -0.29, 2.13, -0.87,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 0, 13, 10,\n" \
"R_CUBIC_TO, 0, -0.83, -0.29, -1.54, -0.87, -2.12,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 0, 10, 7,\n" \
"R_CUBIC_TO, -0.83, 0, -1.54, 0.29, -2.12, 0.88,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 0, 7, 10,\n" \
"R_CUBIC_TO, 0, 0.83, 0.29, 1.54, 0.88, 2.13,\n" \
"ARC_TO, 2.9, 2.9, 0, 0, 0, 10, 13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 6.6, 14.54,\n" \
"R_LINE_TO, -0.37, -1.94,\n" \
"R_CUBIC_TO, -0.23, -0.09, -0.46, -0.2, -0.68, -0.32,\n" \
"R_ARC_TO, 3.78, 3.78, 0, 0, 1, -0.62, -0.43,\n" \
"R_LINE_TO, -1.87, 0.64,\n" \
"R_LINE_TO, -1.4, -2.43,\n" \
"R_LINE_TO, 1.48, -1.29,\n" \
"R_ARC_TO, 6.3, 6.3, 0, 0, 1, -0.05, -0.37,\n" \
"ARC_TO, 4.13, 4.13, 0, 0, 1, 3.06, 8,\n" \
"R_CUBIC_TO, 0, -0.13, 0.01, -0.26, 0.02, -0.39,\n" \
"R_CUBIC_TO, 0.01, -0.12, 0.03, -0.24, 0.05, -0.37,\n" \
"LINE_TO, 1.64, 5.96,\n" \
"R_LINE_TO, 1.4, -2.43,\n" \
"R_LINE_TO, 1.88, 0.63,\n" \
"R_CUBIC_TO, 0.2, -0.16, 0.4, -0.3, 0.62, -0.43,\n" \
"R_CUBIC_TO, 0.22, -0.12, 0.45, -0.23, 0.68, -0.32,\n" \
"R_LINE_TO, 0.38, -1.95,\n" \
"R_H_LINE_TO, 2.8,\n" \
"R_LINE_TO, 0.38, 1.95,\n" \
"R_CUBIC_TO, 0.24, 0.09, 0.46, 0.2, 0.68, 0.32,\n" \
"R_CUBIC_TO, 0.22, 0.12, 0.43, 0.26, 0.63, 0.43,\n" \
"R_LINE_TO, 1.88, -0.63,\n" \
"R_LINE_TO, 1.4, 2.43,\n" \
"R_LINE_TO, -1.49, 1.29,\n" \
"R_ARC_TO, 4.15, 4.15, 0, 0, 1, 0.06, 0.75,\n" \
"R_ARC_TO, 4.15, 4.15, 0, 0, 1, -0.06, 0.75,\n" \
"R_LINE_TO, 1.49, 1.29,\n" \
"R_LINE_TO, -1.41, 2.43,\n" \
"R_LINE_TO, -1.88, -0.64,\n" \
"R_ARC_TO, 4.07, 4.07, 0, 0, 1, -0.62, 0.43,\n" \
"R_ARC_TO, 5.2, 5.2, 0, 0, 1, -0.68, 0.32,\n" \
"R_LINE_TO, -0.37, 1.95,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.14, -1.38,\n" \
"R_H_LINE_TO, 0.51,\n" \
"R_LINE_TO, 0.32, -1.64,\n" \
"R_ARC_TO, 3.73, 3.73, 0, 0, 0, 1.2, -0.44,\n" \
"R_ARC_TO, 3.28, 3.28, 0, 0, 0, 0.96, -0.82,\n" \
"R_LINE_TO, 1.59, 0.53,\n" \
"R_LINE_TO, 0.25, -0.43,\n" \
"R_LINE_TO, -1.25, -1.11,\n" \
"R_CUBIC_TO, 0.07, -0.19, 0.12, -0.39, 0.16, -0.6,\n" \
"R_CUBIC_TO, 0.04, -0.21, 0.06, -0.42, 0.06, -0.65,\n" \
"R_CUBIC_TO, 0, -0.23, -0.02, -0.44, -0.06, -0.65,\n" \
"R_CUBIC_TO, -0.04, -0.21, -0.09, -0.41, -0.16, -0.61,\n" \
"R_LINE_TO, 1.26, -1.11,\n" \
"R_LINE_TO, -0.26, -0.42,\n" \
"R_LINE_TO, -1.59, 0.53,\n" \
"R_ARC_TO, 3.4, 3.4, 0, 0, 0, -0.96, -0.82,\n" \
"R_ARC_TO, 3.72, 3.72, 0, 0, 0, -1.2, -0.44,\n" \
"R_LINE_TO, -0.32, -1.64,\n" \
"R_H_LINE_TO, -0.52,\n" \
"LINE_TO, 7.43, 4.48,\n" \
"R_ARC_TO, 3.76, 3.76, 0, 0, 0, -1.21, 0.44,\n" \
"R_ARC_TO, 3.42, 3.42, 0, 0, 0, -0.96, 0.82,\n" \
"R_LINE_TO, -1.59, -0.53,\n" \
"R_LINE_TO, -0.26, 0.43,\n" \
"R_LINE_TO, 1.25, 1.11,\n" \
"R_ARC_TO, 4.45, 4.45, 0, 0, 0, -0.16, 0.61,\n" \
"R_CUBIC_TO, -0.04, 0.21, -0.06, 0.43, -0.06, 0.65,\n" \
"R_CUBIC_TO, 0, 0.23, 0.02, 0.44, 0.06, 0.65,\n" \
"R_CUBIC_TO, 0.04, 0.21, 0.09, 0.41, 0.16, 0.61,\n" \
"R_LINE_TO, -1.25, 1.1,\n" \
"R_LINE_TO, 0.26, 0.43,\n" \
"R_LINE_TO, 1.59, -0.53,\n" \
"R_CUBIC_TO, 0.27, 0.33, 0.59, 0.6, 0.97, 0.81,\n" \
"R_CUBIC_TO, 0.38, 0.22, 0.78, 0.36, 1.2, 0.44,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.25, -2.75,\n" \
"R_ARC_TO, 2.3, 2.3, 0, 0, 0, 1.7, -0.7,\n" \
"ARC_TO, 2.3, 2.3, 0, 0, 0, 10.39, 8,\n" \
"R_ARC_TO, 2.3, 2.3, 0, 0, 0, -0.7, -1.7,\n" \
"R_ARC_TO, 2.3, 2.3, 0, 0, 0, -1.7, -0.7,\n" \
"R_CUBIC_TO, -0.66, 0, -1.23, 0.23, -1.7, 0.7,\n" \
"ARC_TO, 2.3, 2.3, 0, 0, 0, 5.59, 8,\n" \
"R_ARC_TO, 2.3, 2.3, 0, 0, 0, 0.7, 1.7,\n" \
"R_CUBIC_TO, 0.47, 0.47, 1.03, 0.7, 1.7, 0.7,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSettingsChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSettingsOutlineIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 11.48, 18,\n" \
"H_LINE_TO, 8.52,\n" \
"R_CUBIC_TO, -0.59, 0, -1.09, -0.42, -1.16, -0.99,\n" \
"R_LINE_TO, -0.22, -1.48,\n" \
"R_CUBIC_TO, -0.22, -0.11, -0.42, -0.23, -0.63, -0.36,\n" \
"R_LINE_TO, -1.44, 0.56,\n" \
"R_CUBIC_TO, -0.56, 0.2, -1.18, -0.02, -1.45, -0.51,\n" \
"R_LINE_TO, -1.46, -2.48,\n" \
"R_CUBIC_TO, -0.28, -0.52, -0.16, -1.12, 0.29, -1.47,\n" \
"R_LINE_TO, 1.22, -0.93,\n" \
"R_CUBIC_TO, -0.01, -0.12, -0.02, -0.23, -0.02, -0.36,\n" \
"R_CUBIC_TO, 0, -0.12, 0.01, -0.24, 0.02, -0.36,\n" \
"R_LINE_TO, -1.22, -0.93,\n" \
"R_CUBIC_TO, -0.47, -0.35, -0.59, -0.98, -0.3, -1.47,\n" \
"R_LINE_TO, 1.48, -2.49,\n" \
"R_CUBIC_TO, 0.27, -0.48, 0.89, -0.7, 1.43, -0.49,\n" \
"R_LINE_TO, 1.45, 0.57,\n" \
"R_CUBIC_TO, 0.21, -0.13, 0.42, -0.25, 0.62, -0.36,\n" \
"R_LINE_TO, 0.22, -1.49,\n" \
"CUBIC_TO, 7.43, 2.43, 7.93, 2, 8.51, 2,\n" \
"R_H_LINE_TO, 2.96,\n" \
"R_CUBIC_TO, 0.59, 0, 1.09, 0.42, 1.16, 0.99,\n" \
"R_LINE_TO, 0.22, 1.48,\n" \
"R_CUBIC_TO, 0.22, 0.11, 0.42, 0.23, 0.63, 0.36,\n" \
"R_LINE_TO, 1.44, -0.56,\n" \
"R_CUBIC_TO, 0.57, -0.2, 1.18, 0.02, 1.46, 0.51,\n" \
"R_LINE_TO, 1.47, 2.48,\n" \
"R_CUBIC_TO, 0.29, 0.52, 0.16, 1.12, -0.29, 1.47,\n" \
"R_LINE_TO, -1.22, 0.93,\n" \
"R_CUBIC_TO, 0.01, 0.12, 0.02, 0.23, 0.02, 0.36,\n" \
"R_CUBIC_TO, 0, 0.12, -0.01, 0.24, -0.02, 0.36,\n" \
"R_LINE_TO, 1.22, 0.93,\n" \
"R_CUBIC_TO, 0.45, 0.35, 0.58, 0.96, 0.3, 1.45,\n" \
"R_LINE_TO, -1.49, 2.51,\n" \
"R_CUBIC_TO, -0.27, 0.48, -0.89, 0.7, -1.44, 0.49,\n" \
"R_LINE_TO, -1.44, -0.56,\n" \
"R_CUBIC_TO, -0.21, 0.13, -0.42, 0.25, -0.62, 0.36,\n" \
"R_LINE_TO, -0.22, 1.49,\n" \
"CUBIC_TO, 12.57, 17.58, 12.07, 18, 11.48, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.53, -2,\n" \
"R_H_LINE_TO, 2.11,\n" \
"R_LINE_TO, 0.28, -1.85,\n" \
"R_LINE_TO, 0.41, -0.16,\n" \
"R_CUBIC_TO, 0.34, -0.13, 0.67, -0.32, 1.03, -0.57,\n" \
"R_LINE_TO, 0.34, -0.25,\n" \
"R_LINE_TO, 1.82, 0.7,\n" \
"LINE_TO, 16, 12.12,\n" \
"R_LINE_TO, -1.55, -1.15,\n" \
"R_LINE_TO, 0.05, -0.41,\n" \
"R_CUBIC_TO, 0.02, -0.19, 0.05, -0.37, 0.05, -0.57,\n" \
"R_CUBIC_TO, 0, -0.2, -0.02, -0.39, -0.05, -0.57,\n" \
"R_LINE_TO, -0.05, -0.41,\n" \
"LINE_TO, 16, 7.88,\n" \
"R_LINE_TO, -1.06, -1.75,\n" \
"R_LINE_TO, -1.83, 0.7,\n" \
"R_LINE_TO, -0.34, -0.25,\n" \
"R_CUBIC_TO, -0.32, -0.23, -0.67, -0.42, -1.02, -0.56,\n" \
"R_LINE_TO, -0.4, -0.16,\n" \
"LINE_TO, 11.06, 4,\n" \
"H_LINE_TO, 8.95,\n" \
"R_LINE_TO, -0.28, 1.85,\n" \
"R_LINE_TO, -0.41, 0.15,\n" \
"R_CUBIC_TO, -0.34, 0.14, -0.67, 0.32, -1.03, 0.57,\n" \
"R_LINE_TO, -0.34, 0.24,\n" \
"R_LINE_TO, -1.82, -0.69,\n" \
"LINE_TO, 4, 7.87,\n" \
"R_LINE_TO, 1.55, 1.15,\n" \
"R_LINE_TO, -0.05, 0.41,\n" \
"R_CUBIC_TO, -0.02, 0.19, -0.05, 0.39, -0.05, 0.57,\n" \
"R_CUBIC_TO, 0, 0.19, 0.02, 0.39, 0.05, 0.57,\n" \
"R_LINE_TO, 0.05, 0.41,\n" \
"LINE_TO, 4, 12.12,\n" \
"R_LINE_TO, 1.06, 1.75,\n" \
"R_LINE_TO, 1.83, -0.7,\n" \
"R_LINE_TO, 0.34, 0.25,\n" \
"R_CUBIC_TO, 0.33, 0.24, 0.66, 0.42, 1.02, 0.56,\n" \
"R_LINE_TO, 0.41, 0.16,\n" \
"LINE_TO, 8.95, 16,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 12.5,\n" \
"R_CUBIC_TO, 1.38, 0, 2.5, -1.12, 2.5, -2.5,\n" \
"R_CUBIC_TO, 0, -1.38, -1.12, -2.5, -2.5, -2.5,\n" \
"R_CUBIC_TO, -1.38, 0, -2.5, 1.12, -2.5, 2.5,\n" \
"R_CUBIC_TO, 0, 1.38, 1.12, 2.5, 2.5, 2.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSettingsOutlineIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseShoppingBagIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 18, 6,\n" \
"R_H_LINE_TO, -2,\n" \
"R_CUBIC_TO, 0, -2.21, -1.79, -4, -4, -4,\n" \
"CUBIC_TO_SHORTHAND, 8, 3.79, 8, 6,\n" \
"H_LINE_TO, 6,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 12,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 12,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -6, -2,\n" \
"R_CUBIC_TO, 1.1, 0, 2, 0.9, 2, 2,\n" \
"R_H_LINE_TO, -4,\n" \
"R_CUBIC_TO, 0, -1.1, 0.9, -2, 2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 6, 16,\n" \
"H_LINE_TO, 6,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_CUBIC_TO, 0, 0.55, 0.45, 1, 1, 1,\n" \
"R_CUBIC_TO, 0.55, 0, 1, -0.45, 1, -1,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 2,\n" \
"R_CUBIC_TO, 0, 0.55, 0.45, 1, 1, 1,\n" \
"R_CUBIC_TO, 0.55, 0, 1, -0.45, 1, -1,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 12,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kShoppingBagIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseShoppingBagRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 5.49, 18,\n" \
"R_ARC_TO, 1.44, 1.44, 0, 0, 1, -1.05, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4, 16.5,\n" \
"V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 5.5, 5.5,\n" \
"H_LINE_TO, 7,\n" \
"R_V_LINE_TO, -0.33,\n" \
"R_CUBIC_TO, 0, -0.84, 0.28, -1.57, 0.84, -2.21,\n" \
"CUBIC_TO, 8.41, 2.32, 9.13, 2, 10, 2,\n" \
"R_CUBIC_TO, 0.83, 0, 1.54, 0.29, 2.13, 0.88,\n" \
"CUBIC_TO, 12.71, 3.46, 13, 4.17, 13, 5,\n" \
"R_V_LINE_TO, 0.5,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 9.5,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 14.5, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.01, -1.5,\n" \
"R_H_LINE_TO, 9,\n" \
"V_LINE_TO, 7,\n" \
"H_LINE_TO, 13,\n" \
"R_V_LINE_TO, 1.25,\n" \
"R_CUBIC_TO, 0, 0.21, -0.07, 0.39, -0.21, 0.54,\n" \
"R_ARC_TO, 0.71, 0.71, 0, 0, 1, -0.53, 0.22,\n" \
"R_CUBIC_TO, -0.21, 0, -0.39, -0.07, -0.53, -0.21,\n" \
"R_ARC_TO, 0.73, 0.73, 0, 0, 1, -0.22, -0.53,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, -3,\n" \
"R_V_LINE_TO, 1.25,\n" \
"R_CUBIC_TO, 0, 0.21, -0.07, 0.39, -0.21, 0.54,\n" \
"ARC_TO, 0.71, 0.71, 0, 0, 1, 7.75, 9,\n" \
"R_CUBIC_TO, -0.21, 0, -0.39, -0.07, -0.53, -0.21,\n" \
"ARC_TO, 0.73, 0.73, 0, 0, 1, 7, 8.25,\n" \
"V_LINE_TO, 7,\n" \
"H_LINE_TO, 5.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3, -11,\n" \
"R_H_LINE_TO, 3,\n" \
"V_LINE_TO, 5,\n" \
"R_CUBIC_TO, 0, -0.41, -0.15, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.46, 1.46, 0, 0, 0, 10, 3.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, 0.15, -1.06, 0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 0, 8.5, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3, 11,\n" \
"V_LINE_TO, 7,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kShoppingBagRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseShoppingmodeIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10.42, 17.56,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 9.36, 18,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, -0.14, -1.06, -0.44,\n" \
"R_LINE_TO, -5.85, -5.85,\n" \
"ARC_TO, 1.42, 1.42, 0, 0, 1, 2, 10.65,\n" \
"R_CUBIC_TO, 0, -0.42, 0.14, -0.77, 0.44, -1.07,\n" \
"LINE_TO, 9.59, 2.43,\n" \
"R_CUBIC_TO, 0.15, -0.15, 0.31, -0.26, 0.49, -0.32,\n" \
"ARC_TO, 1.52, 1.52, 0, 0, 1, 10.65, 2,\n" \
"H_LINE_TO, 16.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 5.88,\n" \
"R_ARC_TO, 1.45, 1.45, 0, 0, 1, -0.44, 1.04,\n" \
"CLOSE,\n" \
"MOVE_TO, 14.5, 6.75,\n" \
"R_CUBIC_TO, 0.35, 0, 0.64, -0.12, 0.89, -0.36,\n" \
"R_CUBIC_TO, 0.24, -0.25, 0.36, -0.54, 0.36, -0.89,\n" \
"R_CUBIC_TO, 0, -0.35, -0.12, -0.64, -0.36, -0.89,\n" \
"R_ARC_TO, 1.22, 1.22, 0, 0, 0, -0.89, -0.36,\n" \
"R_CUBIC_TO, -0.35, 0, -0.64, 0.12, -0.89, 0.36,\n" \
"R_CUBIC_TO, -0.24, 0.25, -0.36, 0.54, -0.36, 0.89,\n" \
"R_CUBIC_TO, 0, 0.35, 0.12, 0.64, 0.36, 0.89,\n" \
"R_CUBIC_TO, 0.25, 0.24, 0.54, 0.36, 0.89, 0.36,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.36, 16.5,\n" \
"LINE_TO, 16.5, 9.38,\n" \
"V_LINE_TO, 3.5,\n" \
"R_H_LINE_TO, -5.85,\n" \
"LINE_TO, 3.5, 10.65,\n" \
"CLOSE,\n" \
"MOVE_TO, 16.48, 3.58,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kShoppingmodeIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSkipNextIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"MOVE_TO, 660, 720,\n" \
"R_V_LINE_TO, -480,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, 480,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -440, 0,\n" \
"R_V_LINE_TO, -480,\n" \
"R_LINE_TO, 360, 240,\n" \
"R_LINE_TO, -360, 240,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 80, -240,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 90,\n" \
"R_LINE_TO, 136, -90,\n" \
"R_LINE_TO, -136, -90,\n" \
"R_V_LINE_TO, 180,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSkipNextIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSkipPreviousIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"MOVE_TO, 220, 720,\n" \
"R_V_LINE_TO, -480,\n" \
"R_H_LINE_TO, 80,\n" \
"R_V_LINE_TO, 480,\n" \
"R_H_LINE_TO, -80,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 520, 0,\n" \
"LINE_TO, 380, 480,\n" \
"R_LINE_TO, 360, -240,\n" \
"R_V_LINE_TO, 480,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -80, -240,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 90,\n" \
"R_V_LINE_TO, -180,\n" \
"R_LINE_TO, -136, 90,\n" \
"R_LINE_TO, 136, 90,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSkipPreviousIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSmartCardReaderIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 2, 18,\n" \
"R_V_LINE_TO, -2.5,\n" \
"R_CUBIC_TO, 0, -0.42, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 14,\n" \
"R_H_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.5, -1.5,\n" \
"R_H_LINE_TO, 13,\n" \
"R_V_LINE_TO, -1,\n" \
"R_H_LINE_TO, -13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1, -3.5,\n" \
"V_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0, -0.46, 0.14, -0.82, 0.41, -1.09,\n" \
"CUBIC_TO, 5.18, 2.14, 5.54, 2, 6, 2,\n" \
"R_H_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0.46, 0, 0.82, 0.14, 1.09, 0.41,\n" \
"R_CUBIC_TO, 0.27, 0.27, 0.41, 0.63, 0.41, 1.09,\n" \
"V_LINE_TO, 13,\n" \
"H_LINE_TO, 14,\n" \
"V_LINE_TO, 3.5,\n" \
"H_LINE_TO, 6,\n" \
"V_LINE_TO, 13,\n" \
"CLOSE,\n" \
"MOVE_TO, 7, 11.5,\n" \
"R_H_LINE_TO, 0.43,\n" \
"R_CUBIC_TO, 0.55, 0, 0.99, -0.27, 1.32, -0.82,\n" \
"R_CUBIC_TO, 0.33, -0.55, 0.5, -1.28, 0.5, -2.18,\n" \
"R_CUBIC_TO, 0, -0.9, -0.16, -1.62, -0.5, -2.17,\n" \
"R_CUBIC_TO, -0.33, -0.55, -0.77, -0.82, -1.32, -0.82,\n" \
"H_LINE_TO, 7,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.5, -1.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, -0.15, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.3, 0.44, -0.65, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.41, -0.15, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 0, 11.5, 7,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, 0.15, -1.06, 0.44,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 0, 10, 8.5,\n" \
"R_CUBIC_TO, 0, 0.41, 0.15, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.3, 0.29, 0.65, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 16.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8.25,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSmartCardReaderIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSmartCardReaderOffIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 14, 1.17,\n" \
"R_CUBIC_TO, 0.46, 0, 0.82, 0.13, 1.09, 0.41,\n" \
"R_CUBIC_TO, 0.27, 0.27, 0.41, 0.63, 0.41, 1.09,\n" \
"R_V_LINE_TO, 9.51,\n" \
"R_H_LINE_TO, -0.39,\n" \
"LINE_TO, 14, 11.06,\n" \
"V_LINE_TO, 2.67,\n" \
"H_LINE_TO, 5.65,\n" \
"LINE_TO, 4.75, 1.77,\n" \
"R_CUBIC_TO, 0.13, -0.19, 0.3, -0.34, 0.51, -0.45,\n" \
"R_CUBIC_TO, 0.22, -0.1, 0.46, -0.16, 0.74, -0.16,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.98, 7.91,\n" \
"R_LINE_TO, -1.91, -1.94,\n" \
"R_ARC_TO, 1.56, 1.56, 0, 0, 1, 0.54, -0.7,\n" \
"R_ARC_TO, 1.38, 1.38, 0, 0, 1, 0.86, -0.28,\n" \
"R_CUBIC_TO, 0.41, 0, 0.76, 0.14, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0, 0.32, -0.09, 0.61, -0.28, 0.87,\n" \
"R_ARC_TO, 1.57, 1.57, 0, 0, 1, -0.7, 0.55,\n" \
"CLOSE,\n" \
"MOVE_TO, 7, 10.67,\n" \
"R_V_LINE_TO, -4.5,\n" \
"R_LINE_TO, 2.21, 2.21,\n" \
"R_CUBIC_TO, -0.08, 0.71, -0.28, 1.27, -0.59, 1.68,\n" \
"R_CUBIC_TO, -0.31, 0.41, -0.71, 0.62, -1.18, 0.62,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3.5, 5,\n" \
"R_H_LINE_TO, 10.88,\n" \
"R_LINE_TO, -1, -1,\n" \
"H_LINE_TO, 3.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 13.23, 2.35,\n" \
"R_LINE_TO, -0.85, -0.85,\n" \
"H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -2.5,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"R_ARC_TO, 1.44, 1.44, 0, 0, 1, 1.06, -0.44,\n" \
"R_H_LINE_TO, 8.38,\n" \
"LINE_TO, 6, 7.29,\n" \
"R_V_LINE_TO, 4.88,\n" \
"H_LINE_TO, 4.5,\n" \
"V_LINE_TO, 5.79,\n" \
"LINE_TO, 1.15, 2.44,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 15.59, 15.58,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -7.79, -2.35,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.9, -8.81,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.44, 9.73,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSmartCardReaderOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSmsIcon) { \
              std::string s = \
"// Copyright 2021 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 20, 2,\n" \
"H_LINE_TO, 4,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 18,\n" \
"R_LINE_TO, 4, -4,\n" \
"R_H_LINE_TO, 14,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 14,\n" \
"H_LINE_TO, 6,\n" \
"R_LINE_TO, -2, 2,\n" \
"V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 16,\n" \
"R_V_LINE_TO, 12,\n" \
"CLOSE,\n" \
"MOVE_TO, 7, 9,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 7,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, 0,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, 0,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, -2,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSmsIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseStarIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 13,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 3.39, 10.41,\n" \
"R_LINE_TO, 0.83, -3.49,\n" \
"R_LINE_TO, -2.76, -2.35,\n" \
"R_LINE_TO, 3.62, -0.31,\n" \
"R_LINE_TO, 1.42, -3.3,\n" \
"R_LINE_TO, 1.42, 3.31,\n" \
"R_LINE_TO, 3.62, 0.3,\n" \
"R_LINE_TO, -2.76, 2.35,\n" \
"R_LINE_TO, 0.83, 3.5,\n" \
"LINE_TO, 6.5, 8.55,\n" \
"R_LINE_TO, -3.11, 1.86,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kStarIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseStarHalfIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 13,\n" \
"PATH_COLOR_ARGB, 0xFF, 0xDA, 0xDC, 0xE0,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 3.39, 10.41,\n" \
"R_LINE_TO, 0.83, -3.49,\n" \
"R_LINE_TO, -2.76, -2.35,\n" \
"R_LINE_TO, 3.62, -0.31,\n" \
"R_LINE_TO, 1.42, -3.3,\n" \
"R_LINE_TO, 1.42, 3.31,\n" \
"R_LINE_TO, 3.62, 0.3,\n" \
"R_LINE_TO, -2.76, 2.35,\n" \
"R_LINE_TO, 0.83, 3.5,\n" \
"LINE_TO, 6.5, 8.55,\n" \
"R_LINE_TO, -3.11, 1.86,\n" \
"CLOSE,\n" \
"NEW_PATH,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 3.39, 10.41,\n" \
"R_LINE_TO, 0.83, -3.5,\n" \
"R_LINE_TO, -2.76, -2.35,\n" \
"R_LINE_TO, 3.62, -0.31,\n" \
"LINE_TO, 6.5, 0.95,\n" \
"R_V_LINE_TO, 7.61,\n" \
"R_LINE_TO, -3.11, 1.86,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kStarHalfIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseStickyNote2Icon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 15.5,\n" \
"R_H_LINE_TO, 7,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, -7,\n" \
"R_H_LINE_TO, -11,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 1.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3, 15.5,\n" \
"R_V_LINE_TO, -11,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4.5, 3,\n" \
"R_H_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 12,\n" \
"R_LINE_TO, -5, 5,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 11.5,\n" \
"V_LINE_TO, 10,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.5, 7,\n" \
"R_V_LINE_TO, -11,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kStickyNote2Icon); \
              } \
 \
TEST_F(VectorIconsTest, ParseStopCircleIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 320, 640,\n" \
"R_H_LINE_TO, 320,\n" \
"R_V_LINE_TO, -320,\n" \
"H_LINE_TO, 320,\n" \
"R_V_LINE_TO, 320,\n" \
"CLOSE,\n" \
"MOVE_TO, 480, 880,\n" \
"R_QUADRATIC_TO, -83, 0, -156, -31.5,\n" \
"QUADRATIC_TO_SHORTHAND, 197, 763,\n" \
"R_QUADRATIC_TO, -54, -54, -85.5, -127,\n" \
"QUADRATIC_TO_SHORTHAND, 80, 480,\n" \
"R_QUADRATIC_TO, 0, -83, 31.5, -156,\n" \
"QUADRATIC_TO_SHORTHAND, 197, 197,\n" \
"R_QUADRATIC_TO, 54, -54, 127, -85.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 80,\n" \
"R_QUADRATIC_TO, 83, 0, 156, 31.5,\n" \
"QUADRATIC_TO_SHORTHAND, 763, 197,\n" \
"R_QUADRATIC_TO, 54, 54, 85.5, 127,\n" \
"QUADRATIC_TO_SHORTHAND, 880, 480,\n" \
"R_QUADRATIC_TO, 0, 83, -31.5, 156,\n" \
"QUADRATIC_TO_SHORTHAND, 763, 763,\n" \
"R_QUADRATIC_TO, -54, 54, -127, 85.5,\n" \
"QUADRATIC_TO_SHORTHAND, 480, 880,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -80,\n" \
"R_QUADRATIC_TO, 134, 0, 227, -93,\n" \
"R_QUADRATIC_TO, 93, -93, 93, -227,\n" \
"R_QUADRATIC_TO, 0, -134, -93, -227,\n" \
"R_QUADRATIC_TO, -93, -93, -227, -93,\n" \
"R_QUADRATIC_TO, -134, 0, -227, 93,\n" \
"R_QUADRATIC_TO, -93, 93, -93, 227,\n" \
"R_QUADRATIC_TO, 0, 134, 93, 227,\n" \
"R_QUADRATIC_TO, 93, 93, 227, 93,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -320,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kStopCircleIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseStorageAccessIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"// Based on vr180_create2d, 400 font-weight.\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 6.5, 14.69,\n" \
"R_ARC_TO, 6.46, 6.46, 0, 0, 1, -3.26, -2.36,\n" \
"ARC_TO, 6.34, 6.34, 0, 0, 1, 2, 8.5,\n" \
"R_CUBIC_TO, 0, -1.8, 0.63, -3.34, 1.9, -4.61,\n" \
"CUBIC_TO, 5.16, 2.63, 6.7, 2, 8.5, 2,\n" \
"R_ARC_TO, 6.34, 6.34, 0, 0, 1, 3.82, 1.24,\n" \
"ARC_TO, 6.46, 6.46, 0, 0, 1, 14.69, 6.5,\n" \
"R_H_LINE_TO, -1.61,\n" \
"R_ARC_TO, 4.92, 4.92, 0, 0, 0, -1.84, -2.19,\n" \
"ARC_TO, 4.87, 4.87, 0, 0, 0, 8.5, 3.5,\n" \
"R_CUBIC_TO, -1.39, 0, -2.57, 0.48, -3.54, 1.46,\n" \
"CUBIC_TO, 3.98, 5.93, 3.5, 7.11, 3.5, 8.5,\n" \
"R_CUBIC_TO, 0, 1, 0.27, 1.91, 0.81, 2.73,\n" \
"ARC_TO, 4.97, 4.97, 0, 0, 0, 6.5, 13.06,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.5, 18,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 8, 16.5,\n" \
"R_V_LINE_TO, -7,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 9.5, 8,\n" \
"R_H_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 7,\n" \
"R_V_LINE_TO, -7,\n" \
"R_H_LINE_TO, -7,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.5, -1,\n" \
"R_H_LINE_TO, 6,\n" \
"LINE_TO, 14.11, 13,\n" \
"R_LINE_TO, -1.5, 1.98,\n" \
"LINE_TO, 11.5, 13.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3, -2.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.29, 8.27,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kStorageAccessIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseStorageAccessOffIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"// Based on vr180_create2d_off, 400 font-weight.\n" \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 18, 15.88,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 9.5,\n" \
"R_H_LINE_TO, -4.87,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"H_LINE_TO, 16.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"CLOSE,\n" \
"MOVE_TO, 13.08, 6.5,\n" \
"R_ARC_TO, 4.92, 4.92, 0, 0, 0, -1.84, -2.19,\n" \
"ARC_TO, 4.87, 4.87, 0, 0, 0, 8.5, 3.5,\n" \
"R_CUBIC_TO, -0.4, 0, -0.79, 0.05, -1.18, 0.14,\n" \
"R_ARC_TO, 5.1, 5.1, 0, 0, 0, -1.12, 0.44,\n" \
"LINE_TO, 5.08, 2.98,\n" \
"R_ARC_TO, 6.9, 6.9, 0, 0, 1, 1.65, -0.73,\n" \
"ARC_TO, 6.26, 6.26, 0, 0, 1, 8.5, 2,\n" \
"R_ARC_TO, 6.34, 6.34, 0, 0, 1, 3.82, 1.24,\n" \
"ARC_TO, 6.46, 6.46, 0, 0, 1, 14.69, 6.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.14, 7.56,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.79, 4.79,\n" \
"LINE_TO, 15.88, 18,\n" \
"H_LINE_TO, 9.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 8, 16.5,\n" \
"R_V_LINE_TO, -6.37,\n" \
"LINE_TO, 4.06, 6.19,\n" \
"ARC_TO, 4.97, 4.97, 0, 0, 0, 3.5, 8.5,\n" \
"R_CUBIC_TO, 0, 1, 0.27, 1.91, 0.81, 2.73,\n" \
"ARC_TO, 4.97, 4.97, 0, 0, 0, 6.5, 13.06,\n" \
"R_V_LINE_TO, 1.63,\n" \
"R_ARC_TO, 6.46, 6.46, 0, 0, 1, -3.26, -2.36,\n" \
"ARC_TO, 6.34, 6.34, 0, 0, 1, 2, 8.5,\n" \
"R_CUBIC_TO, 0, -0.63, 0.08, -1.23, 0.25, -1.8,\n" \
"R_ARC_TO, 6.7, 6.7, 0, 0, 1, 0.71, -1.61,\n" \
"LINE_TO, 1.17, 3.29,\n" \
"LINE_TO, 2.23, 2.23,\n" \
"R_LINE_TO, 15.56, 15.56,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.04, 6.54,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.6, 0.89,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 15.5,\n" \
"R_LINE_TO, 1.48, -2,\n" \
"R_LINE_TO, 1.27, 1.29,\n" \
"R_LINE_TO, 0.88, -1.17,\n" \
"LINE_TO, 15.5, 15.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.5, 1,\n" \
"R_H_LINE_TO, 4.88,\n" \
"LINE_TO, 9.5, 11.63,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.56, -4.56,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kStorageAccessOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseStorefrontIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 14,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 12.6, 5.6,\n" \
"R_V_LINE_TO, 4.8,\n" \
"R_CUBIC_TO, 0, 0.32, -0.12, 0.61, -0.35, 0.85,\n" \
"R_CUBIC_TO, -0.23, 0.23, -0.52, 0.35, -0.85, 0.35,\n" \
"H_LINE_TO, 2.6,\n" \
"R_CUBIC_TO, -0.33, 0, -0.62, -0.12, -0.85, -0.35,\n" \
"R_ARC_TO, 1.19, 1.19, 0, 0, 1, -0.35, -0.85,\n" \
"V_LINE_TO, 5.58,\n" \
"CUBIC_TO, 1.03, 5.33, 0.8, 5, 0.7, 4.6,\n" \
"R_CUBIC_TO, -0.09, -0.4, -0.08, -0.82, 0.02, -1.27,\n" \
"LINE_TO, 1.48, 1.2,\n" \
"R_ARC_TO, 0.91, 0.91, 0, 0, 1, 0.38, -0.6,\n" \
"R_CUBIC_TO, 0.2, -0.13, 0.44, -0.2, 0.73, -0.2,\n" \
"R_H_LINE_TO, 8.8,\n" \
"R_CUBIC_TO, 0.29, 0, 0.53, 0.06, 0.72, 0.18,\n" \
"R_CUBIC_TO, 0.2, 0.12, 0.33, 0.33, 0.4, 0.62,\n" \
"R_LINE_TO, 0.77, 2.13,\n" \
"R_CUBIC_TO, 0.1, 0.45, 0.1, 0.87, 0, 1.27,\n" \
"R_CUBIC_TO, -0.1, 0.4, -0.33, 0.73, -0.68, 1,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4.13, -0.8,\n" \
"R_ARC_TO, 0.79, 0.79, 0, 0, 0, 0.58, -0.23,\n" \
"ARC_TO, 0.65, 0.65, 0, 0, 0, 9.27, 4,\n" \
"R_LINE_TO, -0.4, -2.4,\n" \
"R_H_LINE_TO, -1.2,\n" \
"V_LINE_TO, 4,\n" \
"R_CUBIC_TO, 0, 0.22, 0.08, 0.41, 0.23, 0.57,\n" \
"R_ARC_TO, 0.77, 0.77, 0, 0, 0, 0.57, 0.23,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.93, 0,\n" \
"R_ARC_TO, 0.77, 0.77, 0, 0, 0, 0.57, -0.23,\n" \
"ARC_TO, 0.77, 0.77, 0, 0, 0, 6.33, 4,\n" \
"V_LINE_TO, 1.6,\n" \
"R_H_LINE_TO, -1.2,\n" \
"R_LINE_TO, -0.4, 2.4,\n" \
"R_ARC_TO, 0.67, 0.67, 0, 0, 0, 0.2, 0.57,\n" \
"R_ARC_TO, 0.85, 0.85, 0, 0, 0, 0.6, 0.23,\n" \
"CLOSE,\n" \
"MOVE_TO, 2.6, 4.8,\n" \
"R_CUBIC_TO, 0.2, 0, 0.37, -0.06, 0.52, -0.18,\n" \
"R_ARC_TO, 0.68, 0.68, 0, 0, 0, 0.27, -0.47,\n" \
"LINE_TO, 3.8, 1.6,\n" \
"H_LINE_TO, 2.6,\n" \
"R_LINE_TO, -0.75, 2.13,\n" \
"R_ARC_TO, 0.79, 0.79, 0, 0, 0, 0.1, 0.73,\n" \
"R_ARC_TO, 0.75, 0.75, 0, 0, 0, 0.65, 0.33,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.8, 0,\n" \
"R_ARC_TO, 0.75, 0.75, 0, 0, 0, 0.65, -0.33,\n" \
"R_ARC_TO, 0.77, 0.77, 0, 0, 0, 0.12, -0.73,\n" \
"LINE_TO, 11.4, 1.6,\n" \
"R_H_LINE_TO, -1.2,\n" \
"R_LINE_TO, 0.42, 2.55,\n" \
"R_ARC_TO, 0.68, 0.68, 0, 0, 0, 0.27, 0.47,\n" \
"R_ARC_TO, 0.82, 0.82, 0, 0, 0, 0.52, 0.18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -8.8, 5.6,\n" \
"R_H_LINE_TO, 8.8,\n" \
"V_LINE_TO, 6,\n" \
"R_CUBIC_TO, -0.28, 0, -0.54, -0.05, -0.8, -0.15,\n" \
"R_ARC_TO, 2.02, 2.02, 0, 0, 1, -0.67, -0.5,\n" \
"R_CUBIC_TO, -0.2, 0.22, -0.43, 0.39, -0.68, 0.5,\n" \
"R_CUBIC_TO, -0.24, 0.1, -0.51, 0.15, -0.78, 0.15,\n" \
"R_ARC_TO, 1.87, 1.87, 0, 0, 1, -0.78, -0.17,\n" \
"ARC_TO, 2.33, 2.33, 0, 0, 1, 7, 5.35,\n" \
"R_ARC_TO, 1.9, 1.9, 0, 0, 1, -0.67, 0.48,\n" \
"R_ARC_TO, 1.99, 1.99, 0, 0, 1, -0.8, 0.17,\n" \
"R_CUBIC_TO, -0.28, 0, -0.54, -0.04, -0.78, -0.13,\n" \
"R_ARC_TO, 1.83, 1.83, 0, 0, 1, -0.68, -0.52,\n" \
"R_ARC_TO, 2.28, 2.28, 0, 0, 1, -0.73, 0.52,\n" \
"ARC_TO, 2.04, 2.04, 0, 0, 1, 2.6, 6,\n" \
"R_V_LINE_TO, 4.4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.8, 0,\n" \
"H_LINE_TO, 2.6,\n" \
"R_CUBIC_TO, 0.03, 0, 0.03, 0, 0, 0,\n" \
"R_CUBIC_TO, -0.02, 0, -0.02, 0, 0, 0,\n" \
"H_LINE_TO, 11.4,\n" \
"R_H_LINE_TO, -0.02,\n" \
"R_H_LINE_TO, 0.02,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kStorefrontIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSubmenuArrowIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FLIPS_IN_RTL,\n" \
"MOVE_TO, 3, 16,\n" \
"R_LINE_TO, 11, -8,\n" \
"LINE_TO, 3, 0,\n" \
"R_V_LINE_TO, 16,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 8,\n" \
"FLIPS_IN_RTL,\n" \
"MOVE_TO, 2, 8,\n" \
"R_LINE_TO, 5, -4,\n" \
"R_LINE_TO, -5, -4,\n" \
"R_V_LINE_TO, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSubmenuArrowIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSubmenuArrowChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FLIPS_IN_RTL,\n" \
"R_MOVE_TO, 6.4, 12.13,\n" \
"R_LINE_TO, -0.98, -0.98,\n" \
"LINE_TO, 8.57, 8,\n" \
"LINE_TO, 5.42, 4.85,\n" \
"R_LINE_TO, 0.98, -0.98,\n" \
"LINE_TO, 10.53, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSubmenuArrowChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSyncIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 8, 3.27,\n" \
"V_LINE_TO, 1.5,\n" \
"LINE_TO, 5.5, 3.86,\n" \
"LINE_TO, 8, 6.23,\n" \
"V_LINE_TO, 4.46,\n" \
"R_CUBIC_TO, 2.07, 0, 3.75, 1.59, 3.75, 3.55,\n" \
"R_CUBIC_TO, 0, 0.6, -0.16, 1.16, -0.44, 1.66,\n" \
"R_LINE_TO, 0.91, 0.86,\n" \
"CUBIC_TO, 12.71, 9.79, 13, 8.93, 13, 8,\n" \
"R_CUBIC_TO, 0, -2.61, -2.24, -4.73, -5, -4.73,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 8.27,\n" \
"R_CUBIC_TO, -2.07, 0, -3.75, -1.59, -3.75, -3.54,\n" \
"R_CUBIC_TO, 0, -0.6, 0.16, -1.16, 0.44, -1.65,\n" \
"R_LINE_TO, -0.91, -0.86,\n" \
"CUBIC_TO, 3.29, 6.21, 3, 7.07, 3, 8,\n" \
"R_CUBIC_TO, 0, 2.61, 2.24, 4.73, 5, 4.73,\n" \
"V_LINE_TO, 14.5,\n" \
"R_LINE_TO, 2.5, -2.36,\n" \
"LINE_TO, 8, 9.77,\n" \
"R_V_LINE_TO, 1.77,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSyncIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSyncChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 16,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 1.54,\n" \
"R_ARC_TO, 6.27, 6.27, 0, 0, 1, -1.49, -1.99,\n" \
"ARC_TO, 5.78, 5.78, 0, 0, 1, 4, 10,\n" \
"R_CUBIC_TO, 0, -1.4, 0.42, -2.64, 1.27, -3.7,\n" \
"ARC_TO, 5.92, 5.92, 0, 0, 1, 8.5, 4.21,\n" \
"V_LINE_TO, 5.77,\n" \
"R_CUBIC_TO, -0.87, 0.32, -1.59, 0.86, -2.16, 1.62,\n" \
"CUBIC_TO, 5.78, 8.14, 5.5, 9.02, 5.5, 10,\n" \
"R_CUBIC_TO, 0, 0.67, 0.14, 1.29, 0.41, 1.86,\n" \
"ARC_TO, 4.5, 4.5, 0, 0, 0, 7, 13.31,\n" \
"V_LINE_TO, 12,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7, -0.21,\n" \
"V_LINE_TO, 14.23,\n" \
"R_ARC_TO, 4.61, 4.61, 0, 0, 0, 2.16, -1.62,\n" \
"R_CUBIC_TO, 0.56, -0.75, 0.84, -1.63, 0.84, -2.61,\n" \
"R_CUBIC_TO, 0, -0.67, -0.14, -1.28, -0.41, -1.85,\n" \
"ARC_TO, 4.5, 4.5, 0, 0, 0, 13, 6.69,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, -1.5,\n" \
"V_LINE_TO, 4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, -1.54,\n" \
"R_ARC_TO, 6.27, 6.27, 0, 0, 1, 1.49, 1.99,\n" \
"R_CUBIC_TO, 0.37, 0.77, 0.55, 1.61, 0.55, 2.51,\n" \
"R_CUBIC_TO, 0, 1.4, -0.42, 2.64, -1.27, 3.7,\n" \
"R_ARC_TO, 5.92, 5.92, 0, 0, 1, -3.23, 2.09,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 3.62, 12.99,\n" \
"R_V_LINE_TO, -1.34,\n" \
"R_H_LINE_TO, 0.93,\n" \
"R_LINE_TO, 0.12, 0.09,\n" \
"R_ARC_TO, 5.92, 5.92, 0, 0, 1, -1.23, -1.68,\n" \
"ARC_TO, 4.62, 4.62, 0, 0, 1, 2.96, 8,\n" \
"R_CUBIC_TO, 0, -1.22, 0.39, -2.29, 1.17, -3.2,\n" \
"R_ARC_TO, 5.07, 5.07, 0, 0, 1, 2.91, -1.73,\n" \
"R_V_LINE_TO, 1.68,\n" \
"R_CUBIC_TO, -0.71, 0.2, -1.29, 0.59, -1.75, 1.19,\n" \
"ARC_TO, 3.28, 3.28, 0, 0, 0, 4.6, 8,\n" \
"R_CUBIC_TO, 0, 0.5, 0.11, 0.96, 0.31, 1.38,\n" \
"R_CUBIC_TO, 0.21, 0.43, 0.48, 0.8, 0.82, 1.11,\n" \
"R_LINE_TO, -0.07, -0.07,\n" \
"R_V_LINE_TO, -0.81,\n" \
"R_H_LINE_TO, 1.34,\n" \
"R_V_LINE_TO, 3.39,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.35, -0.06,\n" \
"R_V_LINE_TO, -1.68,\n" \
"R_CUBIC_TO, 0.71, -0.2, 1.29, -0.59, 1.75, -1.19,\n" \
"ARC_TO, 3.28, 3.28, 0, 0, 0, 11.4, 8,\n" \
"R_CUBIC_TO, 0, -0.5, -0.1, -0.96, -0.31, -1.38,\n" \
"R_ARC_TO, 3.57, 3.57, 0, 0, 0, -0.82, -1.1,\n" \
"R_LINE_TO, 0.07, 0.07,\n" \
"R_V_LINE_TO, 0.81,\n" \
"R_H_LINE_TO, -1.34,\n" \
"V_LINE_TO, 3.01,\n" \
"R_H_LINE_TO, 3.38,\n" \
"R_V_LINE_TO, 1.34,\n" \
"R_H_LINE_TO, -0.93,\n" \
"R_LINE_TO, -0.12, -0.09,\n" \
"R_CUBIC_TO, 0.52, 0.47, 0.93, 1.03, 1.24, 1.67,\n" \
"R_CUBIC_TO, 0.31, 0.64, 0.46, 1.33, 0.46, 2.07,\n" \
"R_CUBIC_TO, 0, 1.22, -0.39, 2.29, -1.16, 3.2,\n" \
"R_ARC_TO, 5.07, 5.07, 0, 0, 1, -2.91, 1.73,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSyncChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSyncOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 16, 17.06,\n" \
"R_LINE_TO, -2.98, -2.98,\n" \
"R_CUBIC_TO, -0.23, 0.14, -0.48, 0.27, -0.73, 0.39,\n" \
"R_CUBIC_TO, -0.25, 0.12, -0.52, 0.21, -0.79, 0.28,\n" \
"R_V_LINE_TO, -1.58,\n" \
"R_CUBIC_TO, 0.07, -0.03, 0.14, -0.06, 0.21, -0.09,\n" \
"R_CUBIC_TO, 0.07, -0.04, 0.14, -0.07, 0.21, -0.09,\n" \
"LINE_TO, 5.96, 7,\n" \
"R_ARC_TO, 5.21, 5.21, 0, 0, 0, -0.33, 0.93,\n" \
"ARC_TO, 4.15, 4.15, 0, 0, 0, 5.5, 8.94,\n" \
"R_CUBIC_TO, 0, 0.67, 0.14, 1.29, 0.41, 1.86,\n" \
"ARC_TO, 4.5, 4.5, 0, 0, 0, 7, 12.25,\n" \
"R_V_LINE_TO, -1.31,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 4,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 1.54,\n" \
"R_ARC_TO, 6.27, 6.27, 0, 0, 1, -1.49, -1.99,\n" \
"ARC_TO, 5.78, 5.78, 0, 0, 1, 4, 8.94,\n" \
"R_CUBIC_TO, 0, -0.56, 0.07, -1.09, 0.22, -1.61,\n" \
"R_CUBIC_TO, 0.14, -0.51, 0.35, -0.99, 0.61, -1.44,\n" \
"LINE_TO, 1.88, 2.92,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"LINE_TO, 17.06, 16,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.83, -5.08,\n" \
"R_LINE_TO, -1.12, -1.12,\n" \
"R_CUBIC_TO, 0.14, -0.29, 0.25, -0.6, 0.33, -0.92,\n" \
"R_ARC_TO, 4.33, 4.33, 0, 0, 0, -0.28, -2.86,\n" \
"ARC_TO, 4.43, 4.43, 0, 0, 0, 13, 5.61,\n" \
"R_V_LINE_TO, 1.33,\n" \
"R_H_LINE_TO, -1.5,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, -1.54,\n" \
"R_ARC_TO, 6.27, 6.27, 0, 0, 1, 1.49, 1.99,\n" \
"R_CUBIC_TO, 0.37, 0.77, 0.55, 1.61, 0.55, 2.51,\n" \
"R_CUBIC_TO, 0, 0.56, -0.07, 1.09, -0.22, 1.61,\n" \
"R_ARC_TO, 6.11, 6.11, 0, 0, 1, -0.61, 1.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.08, 4.89,\n" \
"LINE_TO, 6.98, 3.77,\n" \
"R_CUBIC_TO, 0.24, -0.14, 0.48, -0.26, 0.73, -0.37,\n" \
"R_CUBIC_TO, 0.25, -0.11, 0.52, -0.2, 0.79, -0.27,\n" \
"R_V_LINE_TO, 1.58,\n" \
"R_CUBIC_TO, -0.07, 0.03, -0.14, 0.06, -0.21, 0.09,\n" \
"R_CUBIC_TO, -0.07, 0.04, -0.14, 0.07, -0.21, 0.09,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 12.77, 13.69,\n" \
"R_LINE_TO, -2.34, -2.34,\n" \
"R_CUBIC_TO, -0.21, 0.13, -0.43, 0.24, -0.65, 0.34,\n" \
"R_CUBIC_TO, -0.23, 0.1, -0.46, 0.18, -0.71, 0.24,\n" \
"R_V_LINE_TO, -1.45,\n" \
"R_CUBIC_TO, 0.06, -0.02, 0.11, -0.05, 0.17, -0.07,\n" \
"R_CUBIC_TO, 0.06, -0.03, 0.11, -0.05, 0.17, -0.07,\n" \
"LINE_TO, 4.79, 5.69,\n" \
"R_ARC_TO, 3.69, 3.69, 0, 0, 0, -0.24, 0.7,\n" \
"R_CUBIC_TO, -0.06, 0.24, -0.09, 0.5, -0.09, 0.76,\n" \
"R_CUBIC_TO, 0, 0.52, 0.11, 1, 0.32, 1.44,\n" \
"R_CUBIC_TO, 0.21, 0.45, 0.5, 0.82, 0.85, 1.14,\n" \
"R_LINE_TO, -0.03, -0.03,\n" \
"R_V_LINE_TO, -0.95,\n" \
"R_H_LINE_TO, 1.26,\n" \
"R_V_LINE_TO, 3.28,\n" \
"R_H_LINE_TO, -3.28,\n" \
"V_LINE_TO, 10.77,\n" \
"R_H_LINE_TO, 1.11,\n" \
"R_LINE_TO, 0.05, 0.04,\n" \
"R_ARC_TO, 5.13, 5.13, 0, 0, 1, -1.21, -1.62,\n" \
"R_ARC_TO, 4.72, 4.72, 0, 0, 1, -0.45, -2.04,\n" \
"R_CUBIC_TO, 0, -0.45, 0.06, -0.89, 0.18, -1.31,\n" \
"R_CUBIC_TO, 0.12, -0.42, 0.29, -0.81, 0.5, -1.17,\n" \
"LINE_TO, 1.47, 2.38,\n" \
"R_LINE_TO, 0.89, -0.89,\n" \
"R_LINE_TO, 11.3, 11.32,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.58, -4.07,\n" \
"R_LINE_TO, -1.04, -1.04,\n" \
"R_ARC_TO, 3.39, 3.39, 0, 0, 0, 0.25, -0.68,\n" \
"R_ARC_TO, 3.37, 3.37, 0, 0, 0, -0.23, -2.2,\n" \
"R_ARC_TO, 3.53, 3.53, 0, 0, 0, -0.85, -1.15,\n" \
"R_LINE_TO, 0.03, 0.03,\n" \
"R_V_LINE_TO, 0.97,\n" \
"H_LINE_TO, 9.09,\n" \
"V_LINE_TO, 2.27,\n" \
"R_H_LINE_TO, 3.28,\n" \
"V_LINE_TO, 3.53,\n" \
"R_H_LINE_TO, -1.1,\n" \
"R_LINE_TO, -0.05, -0.04,\n" \
"R_CUBIC_TO, 0.51, 0.46, 0.91, 1, 1.21, 1.63,\n" \
"R_CUBIC_TO, 0.3, 0.63, 0.45, 1.31, 0.45, 2.04,\n" \
"R_CUBIC_TO, 0, 0.45, -0.06, 0.89, -0.18, 1.31,\n" \
"R_ARC_TO, 4.65, 4.65, 0, 0, 1, -0.5, 1.17,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.54, 3.97,\n" \
"R_LINE_TO, -1.02, -1.03,\n" \
"R_CUBIC_TO, 0.21, -0.12, 0.42, -0.23, 0.65, -0.33,\n" \
"R_CUBIC_TO, 0.22, -0.1, 0.46, -0.17, 0.7, -0.23,\n" \
"R_V_LINE_TO, 1.45,\n" \
"R_CUBIC_TO, -0.05, 0.02, -0.11, 0.04, -0.17, 0.07,\n" \
"R_ARC_TO, 1.48, 1.48, 0, 0, 1, -0.16, 0.07,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSyncOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSyncProblemChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 2.75, 12.88,\n" \
"V_LINE_TO, 11.62,\n" \
"R_H_LINE_TO, 1.11,\n" \
"R_LINE_TO, 0.05, 0.04,\n" \
"R_ARC_TO, 5.3, 5.3, 0, 0, 1, -1.21, -1.63,\n" \
"R_CUBIC_TO, -0.3, -0.62, -0.45, -1.3, -0.45, -2.03,\n" \
"R_CUBIC_TO, 0, -1.16, 0.36, -2.18, 1.07, -3.06,\n" \
"R_ARC_TO, 4.88, 4.88, 0, 0, 1, 2.72, -1.7,\n" \
"R_V_LINE_TO, 1.43,\n" \
"R_ARC_TO, 3.46, 3.46, 0, 0, 0, -1.73, 1.25,\n" \
"ARC_TO, 3.36, 3.36, 0, 0, 0, 3.63, 8,\n" \
"R_CUBIC_TO, 0, 0.52, 0.11, 1, 0.32, 1.44,\n" \
"R_CUBIC_TO, 0.21, 0.45, 0.5, 0.83, 0.85, 1.14,\n" \
"R_LINE_TO, -0.03, -0.03,\n" \
"V_LINE_TO, 9.6,\n" \
"R_H_LINE_TO, 1.26,\n" \
"R_V_LINE_TO, 3.28,\n" \
"CLOSE,\n" \
"MOVE_TO, 8, 11.25,\n" \
"R_ARC_TO, 0.62, 0.62, 0, 0, 1, -0.46, -0.19,\n" \
"R_ARC_TO, 0.61, 0.61, 0, 0, 1, -0.19, -0.46,\n" \
"ARC_TO, 0.64, 0.64, 0, 0, 1, 8, 9.95,\n" \
"R_CUBIC_TO, 0.18, 0, 0.33, 0.06, 0.46, 0.19,\n" \
"R_ARC_TO, 0.61, 0.61, 0, 0, 1, 0.19, 0.46,\n" \
"R_ARC_TO, 0.64, 0.64, 0, 0, 1, -0.65, 0.65,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.63, -2.41,\n" \
"V_LINE_TO, 4.74,\n" \
"R_H_LINE_TO, 1.27,\n" \
"R_V_LINE_TO, 4.09,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.59, 3.93,\n" \
"R_V_LINE_TO, -1.43,\n" \
"R_ARC_TO, 3.46, 3.46, 0, 0, 0, 1.73, -1.25,\n" \
"R_CUBIC_TO, 0.45, -0.6, 0.68, -1.29, 0.68, -2.07,\n" \
"R_CUBIC_TO, 0, -0.52, -0.1, -1, -0.32, -1.44,\n" \
"R_ARC_TO, 3.54, 3.54, 0, 0, 0, -0.85, -1.14,\n" \
"R_LINE_TO, 0.03, 0.03,\n" \
"R_V_LINE_TO, 0.95,\n" \
"H_LINE_TO, 9.97,\n" \
"V_LINE_TO, 3.12,\n" \
"R_H_LINE_TO, 3.28,\n" \
"V_LINE_TO, 4.38,\n" \
"R_H_LINE_TO, -1.1,\n" \
"R_LINE_TO, -0.05, -0.04,\n" \
"R_CUBIC_TO, 0.51, 0.45, 0.91, 1, 1.21, 1.63,\n" \
"R_CUBIC_TO, 0.3, 0.63, 0.45, 1.31, 0.45, 2.04,\n" \
"R_CUBIC_TO, 0, 1.16, -0.35, 2.18, -1.07, 3.06,\n" \
"R_ARC_TO, 4.88, 4.88, 0, 0, 1, -2.72, 1.7,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSyncProblemChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSyncSavedLocallyIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 9.11, 11.63,\n" \
"R_LINE_TO, 4.6, -4.61,\n" \
"R_LINE_TO, -1.14, -1.12,\n" \
"R_LINE_TO, -3.46, 3.46,\n" \
"R_LINE_TO, -1.69, -1.69,\n" \
"LINE_TO, 6.27, 8.79,\n" \
"CLOSE,\n" \
"MOVE_TO, 1, 17,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 18,\n" \
"V_LINE_TO, 17,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.5, -2.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 2, 13,\n" \
"V_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 3,\n" \
"R_H_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"R_ARC_TO, 1.44, 1.44, 0, 0, 1, -1.06, 0.44,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 13,\n" \
"V_LINE_TO, 4.5,\n" \
"R_H_LINE_TO, -13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"V_LINE_TO, 4.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSyncSavedLocallyIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseSyncSavedLocallyOffIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16.61f, 14.5f,\n" \
"LINE_TO, 15.13f, 13,\n" \
"LINE_TO, 16.5f, 13,\n" \
"LINE_TO, 16.5f, 4.5f,\n" \
"LINE_TO, 6.63f, 4.5f,\n" \
"LINE_TO, 5.13f, 3,\n" \
"LINE_TO, 16.5f, 3,\n" \
"CUBIC_TO, 16.91f, 3, 17.27f, 3.15f, 17.56f, 3.44f,\n" \
"CUBIC_TO, 17.85f, 3.73f, 18, 4.09f, 18, 4.5f,\n" \
"LINE_TO, 18, 13,\n" \
"CUBIC_TO, 18, 13.38f, 17.87f, 13.72f, 17.61f, 14.01f,\n" \
"CUBIC_TO, 17.34f, 14.29f, 17.01f, 14.46f, 16.61f, 14.5f,\n" \
"CLOSE,\n" \
"MOVE_TO, 16.73f, 18.83f,\n" \
"LINE_TO, 14.88f, 17,\n" \
"LINE_TO, 1, 17,\n" \
"LINE_TO, 1, 15.5f,\n" \
"LINE_TO, 13.38f, 15.5f,\n" \
"LINE_TO, 12.38f, 14.5f,\n" \
"LINE_TO, 3.5f, 14.5f,\n" \
"CUBIC_TO, 3.09f, 14.5f, 2.73f, 14.35f, 2.44f, 14.06f,\n" \
"CUBIC_TO, 2.15f, 13.77f, 2, 13.41f, 2, 13,\n" \
"LINE_TO, 2, 4.5f,\n" \
"CUBIC_TO, 2, 4.45f, 2, 4.39f, 2.01f, 4.33f,\n" \
"CUBIC_TO, 2.02f, 4.28f, 2.03f, 4.22f, 2.04f, 4.17f,\n" \
"LINE_TO, 1.17f, 3.29f,\n" \
"LINE_TO, 2.23f, 2.23f,\n" \
"LINE_TO, 17.77f, 17.79f,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.88f, 13,\n" \
"LINE_TO, 3.5f, 5.63f,\n" \
"LINE_TO, 3.5f, 13,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.88f, 8.75f,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.19f, 9.31f,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.29f, 8.17f,\n" \
"LINE_TO, 12.56f, 5.89f,\n" \
"LINE_TO, 13.71f, 7.02f,\n" \
"LINE_TO, 11.44f, 9.29f,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kSyncSavedLocallyOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseTabSearchIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4, 20,\n" \
"CUBIC_TO, 3.45f, 20, 2.98f, 19.8f, 2.59f, 19.41f,\n" \
"CUBIC_TO, 2.2f, 19.02f, 2, 18.55f, 2, 18,\n" \
"LINE_TO, 2, 6,\n" \
"CUBIC_TO, 2, 5.45f, 2.2f, 4.98f, 2.59f, 4.59f,\n" \
"CUBIC_TO, 2.98f, 4.2f, 3.45f, 4, 4, 4,\n" \
"LINE_TO, 20, 4,\n" \
"CUBIC_TO, 20.55f, 4, 21.02f, 4.2f, 21.41f, 4.59f,\n" \
"CUBIC_TO, 21.8f, 4.98f, 22, 5.45f, 22, 6,\n" \
"LINE_TO, 22, 10,\n" \
"LINE_TO, 13, 10,\n" \
"LINE_TO, 13, 6,\n" \
"LINE_TO, 4, 6,\n" \
"LINE_TO, 4, 18,\n" \
"LINE_TO, 10, 18,\n" \
"LINE_TO, 10, 20,\n" \
"CLOSE,\n" \
"MOVE_TO, 4, 18,\n" \
"LINE_TO, 4, 9,\n" \
"LINE_TO, 4, 10,\n" \
"LINE_TO, 4, 6,\n" \
"CLOSE,\n" \
"MOVE_TO, 21.6f, 22,\n" \
"LINE_TO, 18.9f, 19.3f,\n" \
"CUBIC_TO, 18.54f, 19.52f, 18.15f, 19.69f, 17.75f, 19.81f,\n" \
"CUBIC_TO, 17.35f, 19.94f, 16.93f, 20, 16.5f, 20,\n" \
"CUBIC_TO, 15.25f, 20, 14.19f, 19.56f, 13.31f, 18.69f,\n" \
"CUBIC_TO, 12.44f, 17.81f, 12, 16.75f, 12, 15.5f,\n" \
"CUBIC_TO, 12, 14.25f, 12.44f, 13.19f, 13.31f, 12.31f,\n" \
"CUBIC_TO, 14.19f, 11.44f, 15.25f, 11, 16.5f, 11,\n" \
"CUBIC_TO, 17.75f, 11, 18.81f, 11.44f, 19.69f, 12.31f,\n" \
"CUBIC_TO, 20.56f, 13.19f, 21, 14.25f, 21, 15.5f,\n" \
"CUBIC_TO, 21, 15.93f, 20.94f, 16.35f, 20.81f, 16.75f,\n" \
"CUBIC_TO, 20.69f, 17.15f, 20.52f, 17.54f, 20.3f, 17.9f,\n" \
"LINE_TO, 23, 20.6f,\n" \
"CLOSE,\n" \
"MOVE_TO, 16.5f, 18,\n" \
"CUBIC_TO, 17.2f, 18, 17.79f, 17.76f, 18.27f, 17.27f,\n" \
"CUBIC_TO, 18.76f, 16.79f, 19, 16.2f, 19, 15.5f,\n" \
"CUBIC_TO, 19, 14.8f, 18.76f, 14.21f, 18.27f, 13.73f,\n" \
"CUBIC_TO, 17.79f, 13.24f, 17.2f, 13, 16.5f, 13,\n" \
"CUBIC_TO, 15.8f, 13, 15.21f, 13.24f, 14.73f, 13.73f,\n" \
"CUBIC_TO, 14.24f, 14.21f, 14, 14.8f, 14, 15.5f,\n" \
"CUBIC_TO, 14, 16.2f, 14.24f, 16.79f, 14.73f, 17.27f,\n" \
"CUBIC_TO, 15.21f, 17.76f, 15.8f, 18, 16.5f, 18,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kTabSearchIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseTenancyIcon) { \
              std::string s = \
"// Copyright 2022 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 3.5, 17,\n" \
"R_QUADRATIC_TO, -1.04, 0, -1.77, -0.73,\n" \
"QUADRATIC_TO, 1, 15.54, 1, 14.5,\n" \
"R_QUADRATIC_TO, 0, -1.04, 0.73, -1.77,\n" \
"QUADRATIC_TO, 2.46, 12, 3.5, 12,\n" \
"R_QUADRATIC_TO, 0.29, 0, 0.57, 0.06,\n" \
"R_QUADRATIC_TO, 0.28, 0.06, 0.53, 0.19,\n" \
"R_LINE_TO, 3.13, -4.29,\n" \
"R_QUADRATIC_TO, -0.35, -0.4, -0.54, -0.9,\n" \
"QUADRATIC_TO, 7, 6.56, 7, 6,\n" \
"R_QUADRATIC_TO, 0, -1.25, 0.88, -2.12,\n" \
"QUADRATIC_TO_SHORTHAND, 10, 3,\n" \
"R_QUADRATIC_TO, 1.25, 0, 2.13, 0.88,\n" \
"QUADRATIC_TO_SHORTHAND, 13, 6,\n" \
"R_QUADRATIC_TO, 0, 0.56, -0.19, 1.06,\n" \
"R_QUADRATIC_TO, -0.19, 0.5, -0.54, 0.9,\n" \
"R_LINE_TO, 3.13, 4.29,\n" \
"R_QUADRATIC_TO, 0.25, -0.12, 0.53, -0.19,\n" \
"R_QUADRATIC_TO, 0.28, -0.06, 0.57, -0.06,\n" \
"R_QUADRATIC_TO, 1.04, 0, 1.77, 0.73,\n" \
"R_QUADRATIC_TO, 0.73, 0.73, 0.73, 1.77,\n" \
"R_QUADRATIC_TO, 0, 1.04, -0.73, 1.77,\n" \
"QUADRATIC_TO, 17.54, 17, 16.5, 17,\n" \
"R_QUADRATIC_TO, -1.04, 0, -1.77, -0.73,\n" \
"QUADRATIC_TO, 14, 15.54, 14, 14.5,\n" \
"R_QUADRATIC_TO, 0, -0.33, 0.08, -0.62,\n" \
"R_QUADRATIC_TO, 0.08, -0.29, 0.23, -0.56,\n" \
"R_LINE_TO, -3.27, -4.5,\n" \
"R_QUADRATIC_TO, -0.08, 0.04, -0.15, 0.05,\n" \
"R_QUADRATIC_TO, -0.06, 0.01, -0.15, 0.03,\n" \
"R_V_LINE_TO, 3.23,\n" \
"R_QUADRATIC_TO, 0.77, 0.23, 1.26, 0.88,\n" \
"R_QUADRATIC_TO, 0.49, 0.65, 0.49, 1.5,\n" \
"R_QUADRATIC_TO, 0, 1.04, -0.73, 1.77,\n" \
"QUADRATIC_TO, 11.04, 17, 10, 17,\n" \
"R_QUADRATIC_TO, -1.04, 0, -1.77, -0.73,\n" \
"QUADRATIC_TO, 7.5, 15.54, 7.5, 14.5,\n" \
"R_QUADRATIC_TO, 0, -0.85, 0.49, -1.5,\n" \
"R_QUADRATIC_TO, 0.49, -0.65, 1.26, -0.87,\n" \
"V_LINE_TO, 8.9,\n" \
"R_QUADRATIC_TO, -0.08, -0.02, -0.16, -0.03,\n" \
"R_QUADRATIC_TO, -0.07, -0.01, -0.14, -0.05,\n" \
"R_LINE_TO, -3.25, 4.5,\n" \
"R_QUADRATIC_TO, 0.15, 0.27, 0.22, 0.56,\n" \
"R_QUADRATIC_TO, 0.07, 0.29, 0.07, 0.63,\n" \
"R_QUADRATIC_TO, 0, 1.04, -0.73, 1.77,\n" \
"QUADRATIC_TO, 4.54, 17, 3.5, 17,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kTenancyIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseThumbDownIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 6, 3,\n" \
"R_H_LINE_TO, 11,\n" \
"R_V_LINE_TO, 13,\n" \
"R_LINE_TO, -7, 7,\n" \
"R_LINE_TO, -1.25, -1.25,\n" \
"R_ARC_TO, 1.45, 1.45, 0, 0, 1, -0.3, -0.47,\n" \
"R_CUBIC_TO, -0.07, -0.2, -0.1, -0.39, -0.1, -0.57,\n" \
"R_V_LINE_TO, -0.35,\n" \
"LINE_TO, 9.45, 16,\n" \
"H_LINE_TO, 3,\n" \
"R_CUBIC_TO, -0.53, 0, -1, -0.2, -1.4, -0.6,\n" \
"R_CUBIC_TO, -0.4, -0.4, -0.6, -0.87, -0.6, -1.4,\n" \
"R_V_LINE_TO, -2,\n" \
"R_CUBIC_TO, 0, -0.12, 0.02, -0.24, 0.05, -0.37,\n" \
"R_CUBIC_TO, 0.03, -0.13, 0.07, -0.26, 0.1, -0.37,\n" \
"R_LINE_TO, 3, -7.05,\n" \
"R_CUBIC_TO, 0.15, -0.33, 0.4, -0.62, 0.75, -0.85,\n" \
"CUBIC_TO, 5.25, 3.12, 5.62, 3, 6, 3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 9, 2,\n" \
"H_LINE_TO, 6,\n" \
"R_LINE_TO, -3, 7,\n" \
"R_V_LINE_TO, 2,\n" \
"R_H_LINE_TO, 9,\n" \
"R_LINE_TO, -1.35, 5.5,\n" \
"LINE_TO, 15, 15.15,\n" \
"V_LINE_TO, 5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 10.15,\n" \
"V_LINE_TO, 5,\n" \
"R_V_LINE_TO, 10.15,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2, 0.85,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 3,\n" \
"V_LINE_TO, 5,\n" \
"R_H_LINE_TO, -3,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 5,\n" \
"R_V_LINE_TO, 13,\n" \
"R_H_LINE_TO, -5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kThumbDownIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseThumbDownFilledIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 6, 3,\n" \
"R_H_LINE_TO, 10,\n" \
"R_V_LINE_TO, 13,\n" \
"R_LINE_TO, -7, 7,\n" \
"R_LINE_TO, -1.25, -1.25,\n" \
"R_ARC_TO, 1.34, 1.34, 0, 0, 1, -0.29, -0.48,\n" \
"R_ARC_TO, 1.66, 1.66, 0, 0, 1, -0.11, -0.57,\n" \
"R_V_LINE_TO, -0.35,\n" \
"LINE_TO, 8.45, 16,\n" \
"H_LINE_TO, 3,\n" \
"R_CUBIC_TO, -0.53, 0, -1, -0.2, -1.4, -0.6,\n" \
"CUBIC_TO, 1.2, 15, 1, 14.54, 1, 14,\n" \
"R_V_LINE_TO, -2,\n" \
"R_CUBIC_TO, 0, -0.12, 0.01, -0.24, 0.04, -0.37,\n" \
"R_CUBIC_TO, 0.02, -0.13, 0.06, -0.26, 0.11, -0.37,\n" \
"R_LINE_TO, 3, -7.05,\n" \
"R_CUBIC_TO, 0.15, -0.33, 0.4, -0.62, 0.75, -0.85,\n" \
"ARC_TO, 1.96, 1.96, 0, 0, 1, 6, 3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 12, 13,\n" \
"V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kThumbDownFilledIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseThumbUpIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 18, 21,\n" \
"H_LINE_TO, 7,\n" \
"V_LINE_TO, 8,\n" \
"R_LINE_TO, 7, -7,\n" \
"R_LINE_TO, 1.25, 1.25,\n" \
"R_CUBIC_TO, 0.12, 0.12, 0.21, 0.28, 0.28, 0.48,\n" \
"R_CUBIC_TO, 0.08, 0.2, 0.13, 0.39, 0.13, 0.57,\n" \
"R_V_LINE_TO, 0.35,\n" \
"LINE_TO, 14.55, 8,\n" \
"H_LINE_TO, 21,\n" \
"R_CUBIC_TO, 0.53, 0, 1, 0.2, 1.4, 0.6,\n" \
"R_CUBIC_TO, 0.4, 0.4, 0.6, 0.87, 0.6, 1.4,\n" \
"R_V_LINE_TO, 2,\n" \
"R_CUBIC_TO, 0, 0.12, -0.02, 0.24, -0.05, 0.38,\n" \
"R_CUBIC_TO, -0.03, 0.14, -0.07, 0.26, -0.1, 0.38,\n" \
"R_LINE_TO, -3, 7.05,\n" \
"R_CUBIC_TO, -0.15, 0.33, -0.4, 0.62, -0.75, 0.85,\n" \
"R_CUBIC_TO, -0.35, 0.23, -0.72, 0.35, -1.1, 0.35,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -9, -2,\n" \
"R_H_LINE_TO, 9,\n" \
"R_LINE_TO, 3, -7,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, -9,\n" \
"R_LINE_TO, 1.35, -5.5,\n" \
"LINE_TO, 9, 8.85,\n" \
"V_LINE_TO, 19,\n" \
"CLOSE,\n" \
"MOVE_TO, 9, 8.85,\n" \
"V_LINE_TO, 19,\n" \
"V_LINE_TO, 8.85,\n" \
"CLOSE,\n" \
"MOVE_TO, 7, 8,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 9,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, 2,\n" \
"H_LINE_TO, 2,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, 5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kThumbUpIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseThumbUpFilledIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 18, 21,\n" \
"H_LINE_TO, 8,\n" \
"V_LINE_TO, 8,\n" \
"R_LINE_TO, 7, -7,\n" \
"R_LINE_TO, 1.25, 1.25,\n" \
"R_CUBIC_TO, 0.12, 0.12, 0.21, 0.27, 0.29, 0.48,\n" \
"R_CUBIC_TO, 0.07, 0.2, 0.11, 0.39, 0.11, 0.57,\n" \
"R_V_LINE_TO, 0.35,\n" \
"LINE_TO, 15.55, 8,\n" \
"H_LINE_TO, 21,\n" \
"R_CUBIC_TO, 0.54, 0, 1, 0.2, 1.4, 0.6,\n" \
"CUBIC_TO, 22.8, 9, 23, 9.47, 23, 10,\n" \
"R_V_LINE_TO, 2,\n" \
"R_CUBIC_TO, 0, 0.12, -0.01, 0.24, -0.04, 0.38,\n" \
"R_ARC_TO, 1.9, 1.9, 0, 0, 1, -0.11, 0.38,\n" \
"R_LINE_TO, -3, 7.05,\n" \
"R_ARC_TO, 2.04, 2.04, 0, 0, 1, -0.75, 0.85,\n" \
"ARC_TO, 1.96, 1.96, 0, 0, 1, 18, 21,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 8,\n" \
"R_V_LINE_TO, 13,\n" \
"H_LINE_TO, 2,\n" \
"V_LINE_TO, 8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kThumbUpFilledIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseTouchpadMouseIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 16.5, 20,\n" \
"CUBIC_TO, 17.46, 20, 18.29, 19.66, 18.98, 18.98,\n" \
"CUBIC_TO, 19.66, 18.29, 20, 17.46, 20, 16.5,\n" \
"LINE_TO, 20, 15,\n" \
"LINE_TO, 13, 15,\n" \
"LINE_TO, 13, 16.5,\n" \
"CUBIC_TO, 13, 17.46, 13.34, 18.29, 14.02, 18.98,\n" \
"CUBIC_TO, 14.71, 19.66, 15.54, 20, 16.5, 20,\n" \
"CLOSE,\n" \
"MOVE_TO, 13.02, 13,\n" \
"LINE_TO, 15.5, 13,\n" \
"LINE_TO, 15.5, 10.15,\n" \
"CUBIC_TO, 14.87, 10.33, 14.32, 10.68, 13.86, 11.2,\n" \
"CUBIC_TO, 13.4, 11.71, 13.13, 12.32, 13.02, 13,\n" \
"CLOSE,\n" \
"MOVE_TO, 17.5, 13,\n" \
"LINE_TO, 19.98, 13,\n" \
"CUBIC_TO, 19.88, 12.32, 19.6, 11.71, 19.14, 11.2,\n" \
"CUBIC_TO, 18.68, 10.68, 18.13, 10.33, 17.5, 10.15,\n" \
"CLOSE,\n" \
"MOVE_TO, 16.5, 22,\n" \
"CUBIC_TO, 14.96, 22, 13.67, 21.46, 12.6, 20.4,\n" \
"CUBIC_TO, 11.54, 19.33, 11, 18.04, 11, 16.5,\n" \
"LINE_TO, 11, 13.5,\n" \
"CUBIC_TO, 11, 11.96, 11.54, 10.67, 12.6, 9.6,\n" \
"CUBIC_TO, 13.67, 8.54, 14.96, 8, 16.5, 8,\n" \
"CUBIC_TO, 18.04, 8, 19.33, 8.54, 20.4, 9.6,\n" \
"CUBIC_TO, 21.46, 10.67, 22, 11.96, 22, 13.5,\n" \
"LINE_TO, 22, 16.5,\n" \
"CUBIC_TO, 22, 18.04, 21.46, 19.33, 20.4, 20.4,\n" \
"CUBIC_TO, 19.33, 21.46, 18.04, 22, 16.5, 22,\n" \
"CLOSE,\n" \
"MOVE_TO, 4, 18,\n" \
"LINE_TO, 4, 6,\n" \
"CLOSE,\n" \
"MOVE_TO, 4, 20,\n" \
"CUBIC_TO, 3.45, 20, 2.98, 19.8, 2.59, 19.41,\n" \
"CUBIC_TO, 2.2, 19.02, 2, 18.55, 2, 18,\n" \
"LINE_TO, 2, 6,\n" \
"CUBIC_TO, 2, 5.45, 2.2, 4.98, 2.59, 4.59,\n" \
"CUBIC_TO, 2.98, 4.2, 3.45, 4, 4, 4,\n" \
"LINE_TO, 20, 4,\n" \
"CUBIC_TO, 20.55, 4, 21.02, 4.2, 21.41, 4.59,\n" \
"CUBIC_TO, 21.8, 4.98, 22, 5.45, 22, 6,\n" \
"LINE_TO, 22, 8.4,\n" \
"CUBIC_TO, 21.71, 8.08, 21.41, 7.8, 21.07, 7.54,\n" \
"CUBIC_TO, 20.74, 7.28, 20.38, 7.05, 20, 6.85,\n" \
"LINE_TO, 20, 6,\n" \
"LINE_TO, 4, 6,\n" \
"LINE_TO, 4, 18,\n" \
"LINE_TO, 9.15, 18,\n" \
"CUBIC_TO, 9.23, 18.35, 9.33, 18.69, 9.44, 19.02,\n" \
"CUBIC_TO, 9.55, 19.36, 9.68, 19.68, 9.85, 20,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kTouchpadMouseIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseTouchpadMouseOffIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16.5, 15.02,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 3.98, 8.28,\n" \
"LINE_TO, 0.68, 3.5,\n" \
"R_LINE_TO, 1.43, -1.43,\n" \
"R_LINE_TO, 19.8, 19.8,\n" \
"CLOSE,\n" \
"MOVE_TO, 16.5, 15.02,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 6.98,\n" \
"R_CUBIC_TO, -1.53, 0, -2.83, -0.53, -3.9, -1.6,\n" \
"CUBIC_TO, 11.54, 19.33, 11, 18.04, 11, 16.5,\n" \
"R_V_LINE_TO, -3,\n" \
"R_CUBIC_TO, 0, -0.38, 0.04, -0.75, 0.11, -1.09,\n" \
"R_CUBIC_TO, 0.08, -0.34, 0.18, -0.68, 0.31, -1.02,\n" \
"LINE_TO, 15.02, 15,\n" \
"H_LINE_TO, 13,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0, 0.97, 0.34, 1.79, 1.02, 2.48,\n" \
"ARC_TO, 3.38, 3.38, 0, 0, 0, 16.5, 20,\n" \
"R_ARC_TO, 3.4, 3.4, 0, 0, 0, 1.36, -0.27,\n" \
"R_ARC_TO, 3.31, 3.31, 0, 0, 0, 1.11, -0.78,\n" \
"R_LINE_TO, 1.42, 1.43,\n" \
"R_CUBIC_TO, -0.5, 0.5, -1.08, 0.9, -1.75, 1.19,\n" \
"ARC_TO, 5.27, 5.27, 0, 0, 1, 16.5, 22,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.05, -3.32,\n" \
"R_LINE_TO, -1.6, -1.6,\n" \
"R_CUBIC_TO, 0.02, -0.1, 0.03, -0.19, 0.04, -0.28,\n" \
"R_CUBIC_TO, 0.01, -0.09, 0.01, -0.19, 0.01, -0.29,\n" \
"V_LINE_TO, 15,\n" \
"R_H_LINE_TO, -2.12,\n" \
"LINE_TO, 15.5, 12.63,\n" \
"R_V_LINE_TO, -2.48,\n" \
"R_ARC_TO, 3.52, 3.52, 0, 0, 0, -0.81, 0.36,\n" \
"R_ARC_TO, 3.38, 3.38, 0, 0, 0, -0.71, 0.59,\n" \
"LINE_TO, 12.55, 9.68,\n" \
"R_ARC_TO, 5.6, 5.6, 0, 0, 1, 1.76, -1.23,\n" \
"CUBIC_TO, 14.99, 8.15, 15.72, 8, 16.5, 8,\n" \
"R_CUBIC_TO, 1.54, 0, 2.83, 0.54, 3.9, 1.6,\n" \
"CUBIC_TO, 21.47, 10.67, 22, 11.97, 22, 13.5,\n" \
"R_V_LINE_TO, 3,\n" \
"R_CUBIC_TO, 0, 0.38, -0.04, 0.76, -0.12, 1.13,\n" \
"R_ARC_TO, 6.99, 6.99, 0, 0, 1, -0.32, 1.05,\n" \
"CLOSE,\n" \
"MOVE_TO, 17.5, 13,\n" \
"R_H_LINE_TO, 2.48,\n" \
"R_ARC_TO, 3.39, 3.39, 0, 0, 0, -0.84, -1.8,\n" \
"R_ARC_TO, 3.44, 3.44, 0, 0, 0, -1.64, -1.05,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.93, 9.9,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.22, -1.62,\n" \
"CLOSE,\n" \
"MOVE_TO, 4, 20,\n" \
"R_CUBIC_TO, -0.55, 0, -1.02, -0.19, -1.41, -0.59,\n" \
"ARC_TO, 1.94, 1.94, 0, 0, 1, 2, 18,\n" \
"V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, -0.55, 0.2, -1.02, 0.59, -1.41,\n" \
"CUBIC_TO, 2.98, 4.2, 3.45, 4, 4, 4,\n" \
"R_H_LINE_TO, 0.02,\n" \
"R_LINE_TO, 2, 2,\n" \
"H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 12,\n" \
"R_H_LINE_TO, 5.15,\n" \
"R_CUBIC_TO, 0.09, 0.35, 0.18, 0.69, 0.29, 1.02,\n" \
"R_CUBIC_TO, 0.11, 0.34, 0.25, 0.66, 0.41, 0.98,\n" \
"CLOSE,\n" \
"MOVE_TO, 22, 8.4,\n" \
"R_ARC_TO, 6.88, 6.88, 0, 0, 0, -0.93, -0.86,\n" \
"ARC_TO, 6.79, 6.79, 0, 0, 0, 20, 6.85,\n" \
"V_LINE_TO, 6,\n" \
"H_LINE_TO, 8.88,\n" \
"R_LINE_TO, -2, -2,\n" \
"H_LINE_TO, 20,\n" \
"R_CUBIC_TO, 0.55, 0, 1.02, 0.2, 1.41, 0.59,\n" \
"R_CUBIC_TO, 0.39, 0.39, 0.59, 0.86, 0.59, 1.41,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kTouchpadMouseOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseTranslateIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"R_MOVE_TO, 10, 18.16,\n" \
"R_LINE_TO, -1.02, -3.05,\n" \
"H_LINE_TO, 3.38,\n" \
"R_ARC_TO, 1.59, 1.59, 0, 0, 1, -1.08, -0.46,\n" \
"R_ARC_TO, 1.46, 1.46, 0, 0, 1, -0.45, -1.07,\n" \
"V_LINE_TO, 3.38,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.45, -1.07,\n" \
"R_CUBIC_TO, 0.3, -0.3, 0.66, -0.46, 1.08, -0.46,\n" \
"R_H_LINE_TO, 4.59,\n" \
"R_LINE_TO, 1.01, 3.05,\n" \
"R_H_LINE_TO, 7.64,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.07, 0.46,\n" \
"R_CUBIC_TO, 0.31, 0.3, 0.46, 0.66, 0.46, 1.07,\n" \
"R_V_LINE_TO, 10.2,\n" \
"R_CUBIC_TO, 0, 0.43, -0.15, 0.79, -0.46, 1.08,\n" \
"R_CUBIC_TO, -0.3, 0.3, -0.66, 0.45, -1.07, 0.45,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.99, 12.04,\n" \
"R_CUBIC_TO, 0.81, 0, 1.5, -0.26, 2.07, -0.79,\n" \
"R_CUBIC_TO, 0.57, -0.53, 0.86, -1.27, 0.86, -2.21,\n" \
"R_CUBIC_TO, 0, -0.1, 0, -0.2, -0.01, -0.29,\n" \
"R_ARC_TO, 2.07, 2.07, 0, 0, 0, -0.05, -0.32,\n" \
"H_LINE_TO, 5.99,\n" \
"R_V_LINE_TO, 1.04,\n" \
"R_H_LINE_TO, 1.78,\n" \
"R_CUBIC_TO, -0.05, 0.47, -0.24, 0.83, -0.58, 1.09,\n" \
"R_CUBIC_TO, -0.34, 0.26, -0.74, 0.39, -1.21, 0.39,\n" \
"R_CUBIC_TO, -0.54, 0, -1, -0.19, -1.37, -0.57,\n" \
"R_ARC_TO, 1.94, 1.94, 0, 0, 1, -0.56, -1.4,\n" \
"R_CUBIC_TO, 0, -0.54, 0.19, -1, 0.56, -1.39,\n" \
"R_CUBIC_TO, 0.38, -0.39, 0.83, -0.59, 1.37, -0.59,\n" \
"R_CUBIC_TO, 0.23, 0, 0.45, 0.04, 0.66, 0.13,\n" \
"R_ARC_TO, 2, 2, 0, 0, 1, 0.57, 0.36,\n" \
"R_LINE_TO, 0.81, -0.79,\n" \
"R_ARC_TO, 2.75, 2.75, 0, 0, 0, -0.96, -0.59,\n" \
"R_ARC_TO, 3.26, 3.26, 0, 0, 0, -1.09, -0.19,\n" \
"R_CUBIC_TO, -0.85, 0, -1.58, 0.3, -2.17, 0.89,\n" \
"R_ARC_TO, 2.95, 2.95, 0, 0, 0, -0.89, 2.17,\n" \
"R_CUBIC_TO, 0, 0.84, 0.3, 1.57, 0.89, 2.16,\n" \
"R_CUBIC_TO, 0.59, 0.6, 1.32, 0.9, 2.17, 0.9,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.39, 0.72,\n" \
"R_LINE_TO, 0.41, -0.4,\n" \
"R_ARC_TO, 5.65, 5.65, 0, 0, 1, -0.53, -0.66,\n" \
"R_ARC_TO, 7.64, 7.64, 0, 0, 1, -0.44, -0.73,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1, -1.04,\n" \
"R_CUBIC_TO, 0.29, -0.34, 0.55, -0.69, 0.79, -1.07,\n" \
"R_CUBIC_TO, 0.24, -0.37, 0.42, -0.79, 0.53, -1.24,\n" \
"R_H_LINE_TO, -3.37,\n" \
"R_LINE_TO, 0.29, 0.89,\n" \
"R_LINE_TO, 0.83, -0.02,\n" \
"R_CUBIC_TO, 0.13, 0.27, 0.26, 0.53, 0.41, 0.77,\n" \
"R_CUBIC_TO, 0.15, 0.24, 0.32, 0.46, 0.52, 0.68,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.36, 5.43,\n" \
"R_H_LINE_TO, 5.61,\n" \
"R_ARC_TO, 0.52, 0.52, 0, 0, 0, 0.37, -0.14,\n" \
"R_ARC_TO, 0.5, 0.5, 0, 0, 0, 0.15, -0.37,\n" \
"R_V_LINE_TO, -10.2,\n" \
"R_ARC_TO, 0.5, 0.5, 0, 0, 0, -0.52, -0.51,\n" \
"H_LINE_TO, 9.24,\n" \
"R_LINE_TO, 0.81, 2.62,\n" \
"R_H_LINE_TO, 1.7,\n" \
"R_V_LINE_TO, -0.87,\n" \
"R_H_LINE_TO, 0.85,\n" \
"R_V_LINE_TO, 0.87,\n" \
"R_H_LINE_TO, 3.07,\n" \
"R_V_LINE_TO, 0.87,\n" \
"H_LINE_TO, 14.59,\n" \
"R_ARC_TO, 6.11, 6.11, 0, 0, 1, -0.64, 1.58,\n" \
"R_CUBIC_TO, -0.29, 0.49, -0.61, 0.95, -0.98, 1.37,\n" \
"R_LINE_TO, 2.29, 2.25,\n" \
"R_LINE_TO, -0.61, 0.61,\n" \
"R_LINE_TO, -2.25, -2.25,\n" \
"R_LINE_TO, -0.74, 0.73,\n" \
"R_LINE_TO, 0.76, 2.41,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kTranslateIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseUndoIcon) { \
              std::string s = \
"// Copyright 2023 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 960,\n" \
"MOVE_TO, 229.85, 812.57,\n" \
"R_V_LINE_TO, -71.25,\n" \
"R_H_LINE_TO, 348.33,\n" \
"R_CUBIC_TO, 56.25, 0, 104.66, -18.41, 145.24, -55.23,\n" \
"R_CUBIC_TO, 40.58, -36.82, 60.87, -82.54, 60.87, -137.17,\n" \
"R_CUBIC_TO, 0, -54.63, -20.29, -100.35, -60.87, -137.17,\n" \
"R_CUBIC_TO, -40.58, -36.82, -88.99, -55.22, -145.24, -55.22,\n" \
"H_LINE_TO, 222.62,\n" \
"R_LINE_TO, 137.4, 135.39,\n" \
"R_LINE_TO, -50.62, 49.88,\n" \
"LINE_TO, 85.21, 320.9,\n" \
"LINE_TO, 309.4, 100,\n" \
"R_LINE_TO, 50.62, 49.88,\n" \
"R_LINE_TO, -137.4, 135.39,\n" \
"R_H_LINE_TO, 354.36,\n" \
"R_CUBIC_TO, 76.33, 0, 142.03, 25.33, 197.07, 76.01,\n" \
"R_CUBIC_TO, 55.04, 50.67, 82.56, 113.22, 82.56, 187.64,\n" \
"R_CUBIC_TO, 0, 74.43, -27.52, 136.97, -82.56, 187.64,\n" \
"R_CUBIC_TO, -55.04, 50.67, -120.73, 76.01, -197.06, 76.01,\n" \
"H_LINE_TO, 229.85,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kUndoIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseUsbIcon) { \
              std::string s = \
"// Copyright 2018 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 30, 14,\n" \
"R_V_LINE_TO, 8,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 4,\n" \
"R_H_LINE_TO, -6,\n" \
"V_LINE_TO, 10,\n" \
"R_H_LINE_TO, 4,\n" \
"R_LINE_TO, -6, -8,\n" \
"R_LINE_TO, -6, 8,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 16,\n" \
"R_H_LINE_TO, -6,\n" \
"R_V_LINE_TO, -4.14,\n" \
"R_CUBIC_TO, 1.41, -0.73, 2.4, -2.16, 2.4, -3.86,\n" \
"R_CUBIC_TO, 0, -2.43, -1.97, -4.4, -4.4, -4.4,\n" \
"R_CUBIC_TO, -2.43, 0, -4.4, 1.97, -4.4, 4.4,\n" \
"R_CUBIC_TO, 0, 1.7, 0.99, 3.13, 2.4, 3.86,\n" \
"V_LINE_TO, 26,\n" \
"R_CUBIC_TO, 0, 2.21, 1.79, 4, 4, 4,\n" \
"R_H_LINE_TO, 6,\n" \
"R_V_LINE_TO, 6.1,\n" \
"R_CUBIC_TO, -1.42, 0.73, -2.4, 2.19, -2.4, 3.9,\n" \
"R_CUBIC_TO, 0, 2.43, 1.97, 4.4, 4.4, 4.4,\n" \
"R_CUBIC_TO, 2.43, 0, 4.4, -1.97, 4.4, -4.4,\n" \
"R_CUBIC_TO, 0, -1.71, -0.98, -3.17, -2.4, -3.9,\n" \
"V_LINE_TO, 30,\n" \
"R_H_LINE_TO, 6,\n" \
"R_CUBIC_TO, 2.21, 0, 4, -1.79, 4, -4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, -8,\n" \
"R_H_LINE_TO, -8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kUsbIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseUsbChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 10, 18,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 8.5, 16.5,\n" \
"R_CUBIC_TO, 0, -0.27, 0.07, -0.52, 0.2, -0.75,\n" \
"R_CUBIC_TO, 0.13, -0.23, 0.32, -0.41, 0.55, -0.54,\n" \
"V_LINE_TO, 13,\n" \
"R_H_LINE_TO, -2.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.14, -1.06, -0.44,\n" \
"R_ARC_TO, 1.44, 1.44, 0, 0, 1, -0.44, -1.05,\n" \
"V_LINE_TO, 9.79,\n" \
"R_ARC_TO, 1.36, 1.36, 0, 0, 1, -0.55, -0.53,\n" \
"R_CUBIC_TO, -0.13, -0.23, -0.2, -0.48, -0.2, -0.76,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.46, 1.46, 0, 0, 1, 6, 7,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0, 0.28, -0.07, 0.53, -0.2, 0.76,\n" \
"R_ARC_TO, 1.36, 1.36, 0, 0, 1, -0.55, 0.53,\n" \
"V_LINE_TO, 11.5,\n" \
"R_H_LINE_TO, 2.5,\n" \
"V_LINE_TO, 5,\n" \
"H_LINE_TO, 7.5,\n" \
"LINE_TO, 10, 2,\n" \
"R_LINE_TO, 2.5, 3,\n" \
"R_H_LINE_TO, -1.75,\n" \
"R_V_LINE_TO, 6.5,\n" \
"R_H_LINE_TO, 2.5,\n" \
"V_LINE_TO, 10,\n" \
"R_H_LINE_TO, -0.75,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, -0.75,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 13.25, 13,\n" \
"R_H_LINE_TO, -2.5,\n" \
"R_V_LINE_TO, 2.21,\n" \
"R_CUBIC_TO, 0.24, 0.13, 0.42, 0.31, 0.56, 0.53,\n" \
"R_CUBIC_TO, 0.13, 0.23, 0.2, 0.48, 0.2, 0.76,\n" \
"R_CUBIC_TO, 0, 0.42, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 10, 18,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kUsbChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseUsbOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16, 18.13,\n" \
"LINE_TO, 10.87, 13,\n" \
"R_H_LINE_TO, -0.12,\n" \
"R_V_LINE_TO, 2.21,\n" \
"R_CUBIC_TO, 0.24, 0.13, 0.42, 0.3, 0.56, 0.53,\n" \
"R_CUBIC_TO, 0.13, 0.23, 0.2, 0.49, 0.2, 0.76,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.46, 1.46, 0, 0, 1, 10, 18,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 8.5, 16.5,\n" \
"R_CUBIC_TO, 0, -0.28, 0.07, -0.53, 0.2, -0.75,\n" \
"R_ARC_TO, 1.53, 1.53, 0, 0, 1, 0.55, -0.54,\n" \
"V_LINE_TO, 13,\n" \
"R_H_LINE_TO, -2.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.14, -1.06, -0.44,\n" \
"R_ARC_TO, 1.44, 1.44, 0, 0, 1, -0.44, -1.05,\n" \
"V_LINE_TO, 9.79,\n" \
"R_ARC_TO, 1.22, 1.22, 0, 0, 1, -0.55, -0.53,\n" \
"R_ARC_TO, 1.54, 1.54, 0, 0, 1, -0.2, -0.77,\n" \
"R_CUBIC_TO, 0, -0.23, 0.06, -0.46, 0.18, -0.68,\n" \
"R_CUBIC_TO, 0.12, -0.21, 0.28, -0.4, 0.47, -0.55,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.52, -5.77,\n" \
"R_LINE_TO, -1.23, -1.23,\n" \
"V_LINE_TO, 10,\n" \
"R_H_LINE_TO, -0.75,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, -0.75,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_CUBIC_TO, 0, 0.15, -0.02, 0.31, -0.06, 0.45,\n" \
"R_ARC_TO, 1.19, 1.19, 0, 0, 1, -0.21, 0.41,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.75, 11.5,\n" \
"R_H_LINE_TO, 2.5,\n" \
"R_V_LINE_TO, -0.12,\n" \
"R_LINE_TO, -2.02, -2.02,\n" \
"R_ARC_TO, 1.31, 1.31, 0, 0, 1, -0.22, 0.24,\n" \
"R_ARC_TO, 6.41, 6.41, 0, 0, 1, -0.26, 0.2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, -2.87,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 5,\n" \
"H_LINE_TO, 7.5,\n" \
"LINE_TO, 10, 2,\n" \
"R_LINE_TO, 2.5, 3,\n" \
"R_H_LINE_TO, -1.75,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kUsbOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVideoLibraryIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 9.65, 11.75,\n" \
"LINE_TO, 14.55, 8.6,\n" \
"LINE_TO, 9.65, 5.45,\n" \
"V_LINE_TO, 11.75,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.2, 14.2,\n" \
"CUBIC_TO, 6.82, 14.2, 6.49, 14.06, 6.21, 13.79,\n" \
"CUBIC_TO, 5.94, 13.51, 5.8, 13.19, 5.8, 12.8,\n" \
"V_LINE_TO, 4.4,\n" \
"CUBIC_TO, 5.8, 4.01, 5.94, 3.69, 6.21, 3.41,\n" \
"CUBIC_TO, 6.49, 3.14, 6.82, 3, 7.2, 3,\n" \
"H_LINE_TO, 15.6,\n" \
"CUBIC_TO, 15.99, 3, 16.31, 3.14, 16.59, 3.41,\n" \
"CUBIC_TO, 16.86, 3.69, 17, 4.01, 17, 4.4,\n" \
"V_LINE_TO, 12.8,\n" \
"CUBIC_TO, 17, 13.19, 16.86, 13.51, 16.59, 13.79,\n" \
"CUBIC_TO, 16.31, 14.06, 15.99, 14.2, 15.6, 14.2,\n" \
"H_LINE_TO, 7.2,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.2, 12.8,\n" \
"H_LINE_TO, 15.6,\n" \
"V_LINE_TO, 4.4,\n" \
"H_LINE_TO, 7.2,\n" \
"V_LINE_TO, 12.8,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.4, 17,\n" \
"CUBIC_TO, 4.01, 17, 3.69, 16.86, 3.41, 16.59,\n" \
"CUBIC_TO, 3.14, 16.31, 3, 15.99, 3, 15.6,\n" \
"V_LINE_TO, 5.8,\n" \
"H_LINE_TO, 4.4,\n" \
"V_LINE_TO, 15.6,\n" \
"H_LINE_TO, 14.2,\n" \
"V_LINE_TO, 17,\n" \
"H_LINE_TO, 4.4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVideoLibraryIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVideocamIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 34, 21,\n" \
"R_V_LINE_TO, -7,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"H_LINE_TO, 8,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 20,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 24,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"R_V_LINE_TO, -7,\n" \
"R_LINE_TO, 8, 8,\n" \
"V_LINE_TO, 13,\n" \
"R_LINE_TO, -8, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVideocamIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVideocamChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 16,\n" \
"R_CUBIC_TO, -0.4, 0, -0.75, -0.15, -1.05, -0.45,\n" \
"R_CUBIC_TO, -0.3, -0.3, -0.45, -0.65, -0.45, -1.05,\n" \
"R_V_LINE_TO, -9,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.45, -1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 4.5, 4,\n" \
"R_H_LINE_TO, 9,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 9,\n" \
"R_LINE_TO, 3, -3,\n" \
"R_V_LINE_TO, 8,\n" \
"R_LINE_TO, -3, -3,\n" \
"R_V_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0, 0.4, -0.15, 0.75, -0.44, 1.05,\n" \
"R_CUBIC_TO, -0.29, 0.3, -0.64, 0.45, -1.06, 0.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 9,\n" \
"R_V_LINE_TO, -9,\n" \
"R_H_LINE_TO, -9,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0,\n" \
"R_V_LINE_TO, -9,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVideocamChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVideocamOffIcon) { \
              std::string s = \
"// Copyright 2020 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 21, 18,\n" \
"LINE_TO, 9, 6,\n" \
"H_LINE_TO, 16,\n" \
"CUBIC_TO, 16.55, 6, 17, 6.45, 17, 7,\n" \
"V_LINE_TO, 10.5,\n" \
"LINE_TO, 21, 6.5,\n" \
"V_LINE_TO, 18,\n" \
"CLOSE,\n" \
"MOVE_TO, 21.8, 21.6,\n" \
"LINE_TO, 17, 16.8,\n" \
"LINE_TO, 6.2, 6,\n" \
"LINE_TO, 2.6, 2.4,\n" \
"LINE_TO, 1.2, 3.8,\n" \
"LINE_TO, 3.53, 6.13,\n" \
"CUBIC_TO, 3.22, 6.3, 3, 6.62, 3, 7,\n" \
"V_LINE_TO, 17,\n" \
"CUBIC_TO, 3, 17.55, 3.45, 18, 4, 18,\n" \
"H_LINE_TO, 15.4,\n" \
"LINE_TO, 20.4, 23,\n" \
"LINE_TO, 21.8, 21.6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVideocamOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVideocamOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 18, 13.98,\n" \
"R_LINE_TO, -3, -3,\n" \
"R_V_LINE_TO, 1.88,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 5.5,\n" \
"H_LINE_TO, 7.63,\n" \
"LINE_TO, 6.13, 4,\n" \
"H_LINE_TO, 13.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 3.48,\n" \
"R_LINE_TO, 3, -3,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.27, 4.83,\n" \
"LINE_TO, 1.17, 3.27,\n" \
"LINE_TO, 2.23, 2.21,\n" \
"LINE_TO, 17.79, 17.75,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.56, 8.42,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.46, 10.52,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.04, 4.04,\n" \
"LINE_TO, 5.5, 5.48,\n" \
"R_H_LINE_TO, -1,\n" \
"R_V_LINE_TO, 9.02,\n" \
"R_H_LINE_TO, 9,\n" \
"R_V_LINE_TO, -1.04,\n" \
"R_LINE_TO, 1.5, 1.5,\n" \
"R_CUBIC_TO, -0.11, 0.28, -0.28, 0.52, -0.52, 0.73,\n" \
"R_CUBIC_TO, -0.24, 0.21, -0.56, 0.31, -0.98, 0.31,\n" \
"H_LINE_TO, 4.5,\n" \
"R_ARC_TO, 1.45, 1.45, 0, 0, 1, -1.05, -0.44,\n" \
"R_CUBIC_TO, -0.3, -0.29, -0.45, -0.65, -0.45, -1.06,\n" \
"V_LINE_TO, 5.48,\n" \
"R_CUBIC_TO, 0, -0.35, 0.1, -0.66, 0.3, -0.91,\n" \
"R_CUBIC_TO, 0.2, -0.25, 0.45, -0.42, 0.74, -0.52,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVideocamOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVideogameAssetIcon) { \
              std::string s = \
"// Copyright 2019 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"// Generated from iron-icons hardware:videogame-asset using Skiafy.\n" \
"// iron-icons: https://github.com/PolymerElements/iron-icons\n" \
"// Skiafy: http://evanstade.github.io/skiafy/\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 21, 6,\n" \
"H_LINE_TO, 3,\n" \
"R_CUBIC_TO, -1.1, 0, -2, 0.9, -2, 2,\n" \
"R_V_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, 1.1, 0.9, 2, 2, 2,\n" \
"R_H_LINE_TO, 18,\n" \
"R_CUBIC_TO, 1.1, 0, 2, -0.9, 2, -2,\n" \
"V_LINE_TO, 8,\n" \
"R_CUBIC_TO, 0, -1.1, -0.9, -2, -2, -2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -10, 7,\n" \
"H_LINE_TO, 8,\n" \
"R_V_LINE_TO, 3,\n" \
"H_LINE_TO, 6,\n" \
"R_V_LINE_TO, -3,\n" \
"H_LINE_TO, 3,\n" \
"R_V_LINE_TO, -2,\n" \
"R_H_LINE_TO, 3,\n" \
"V_LINE_TO, 8,\n" \
"R_H_LINE_TO, 2,\n" \
"R_V_LINE_TO, 3,\n" \
"R_H_LINE_TO, 3,\n" \
"R_V_LINE_TO, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.5, 2,\n" \
"R_CUBIC_TO, -0.83, 0, -1.5, -0.67, -1.5, -1.5,\n" \
"R_CUBIC_TO, 0, -0.83, 0.67, -1.5, 1.5, -1.5,\n" \
"R_CUBIC_TO, 0.83, 0, 1.5, 0.67, 1.5, 1.5,\n" \
"R_CUBIC_TO, 0, 0.83, -0.67, 1.5, -1.5, 1.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, -3,\n" \
"R_CUBIC_TO, -0.83, 0, -1.5, -0.67, -1.5, -1.5,\n" \
"CUBIC_TO_SHORTHAND, 18.67, 9, 19.5, 9,\n" \
"R_CUBIC_TO, 0.83, 0, 1.5, 0.67, 1.5, 1.5,\n" \
"R_CUBIC_TO, 0, 0.83, -0.67, 1.5, -1.5, 1.5,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVideogameAssetIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVideogameAssetChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.5, 15,\n" \
"R_ARC_TO, 1.42, 1.42, 0, 0, 1, -1.06, -0.45,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 2, 13.5,\n" \
"R_V_LINE_TO, -7,\n" \
"R_CUBIC_TO, 0, -0.4, 0.15, -0.75, 0.44, -1.05,\n" \
"CUBIC_TO, 2.73, 5.15, 3.09, 5, 3.5, 5,\n" \
"R_H_LINE_TO, 13,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.45,\n" \
"R_CUBIC_TO, 0.29, 0.3, 0.44, 0.65, 0.44, 1.05,\n" \
"R_V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, 0.4, -0.15, 0.75, -0.44, 1.05,\n" \
"R_CUBIC_TO, -0.29, 0.3, -0.64, 0.45, -1.06, 0.45,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.5,\n" \
"R_H_LINE_TO, 13,\n" \
"R_V_LINE_TO, -7,\n" \
"R_H_LINE_TO, -13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.54, -1,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, -1.75,\n" \
"R_H_LINE_TO, 1.75,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, -1.75,\n" \
"V_LINE_TO, 7.5,\n" \
"R_H_LINE_TO, -1.5,\n" \
"R_V_LINE_TO, 1.75,\n" \
"R_H_LINE_TO, -1.75,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, 1.75,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 12.25,\n" \
"R_CUBIC_TO, 0.35, 0, 0.64, -0.12, 0.89, -0.36,\n" \
"R_CUBIC_TO, 0.24, -0.25, 0.36, -0.54, 0.36, -0.89,\n" \
"R_CUBIC_TO, 0, -0.35, -0.12, -0.64, -0.36, -0.89,\n" \
"ARC_TO, 1.22, 1.22, 0, 0, 0, 12, 9.75,\n" \
"R_CUBIC_TO, -0.35, 0, -0.64, 0.12, -0.89, 0.36,\n" \
"R_CUBIC_TO, -0.24, 0.25, -0.36, 0.54, -0.36, 0.89,\n" \
"R_CUBIC_TO, 0, 0.35, 0.12, 0.64, 0.36, 0.89,\n" \
"R_CUBIC_TO, 0.25, 0.24, 0.54, 0.36, 0.89, 0.36,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.5, -2,\n" \
"R_CUBIC_TO, 0.35, 0, 0.64, -0.12, 0.89, -0.36,\n" \
"R_CUBIC_TO, 0.24, -0.25, 0.36, -0.54, 0.36, -0.89,\n" \
"R_CUBIC_TO, 0, -0.35, -0.12, -0.64, -0.36, -0.89,\n" \
"R_ARC_TO, 1.22, 1.22, 0, 0, 0, -0.89, -0.36,\n" \
"R_CUBIC_TO, -0.35, 0, -0.64, 0.12, -0.89, 0.36,\n" \
"R_CUBIC_TO, -0.24, 0.25, -0.36, 0.54, -0.36, 0.89,\n" \
"R_CUBIC_TO, 0, 0.35, 0.12, 0.64, 0.36, 0.89,\n" \
"R_CUBIC_TO, 0.25, 0.24, 0.54, 0.36, 0.89, 0.36,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -11, 3.25,\n" \
"R_V_LINE_TO, -7,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVideogameAssetChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVideogameAssetOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 14.5, 10.25,\n" \
"R_CUBIC_TO, -0.35, 0, -0.64, -0.12, -0.89, -0.36,\n" \
"ARC_TO, 1.22, 1.22, 0, 0, 1, 13.25, 9,\n" \
"R_CUBIC_TO, 0, -0.35, 0.12, -0.64, 0.36, -0.89,\n" \
"R_CUBIC_TO, 0.25, -0.24, 0.54, -0.36, 0.89, -0.36,\n" \
"R_CUBIC_TO, 0.35, 0, 0.64, 0.12, 0.89, 0.36,\n" \
"R_CUBIC_TO, 0.24, 0.25, 0.36, 0.54, 0.36, 0.89,\n" \
"R_CUBIC_TO, 0, 0.35, -0.12, 0.64, -0.36, 0.89,\n" \
"R_CUBIC_TO, -0.25, 0.24, -0.54, 0.36, -0.89, 0.36,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -6.64, -0.17,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 12.5,\n" \
"R_V_LINE_TO, -1.75,\n" \
"H_LINE_TO, 4.25,\n" \
"R_V_LINE_TO, -1.5,\n" \
"H_LINE_TO, 6,\n" \
"V_LINE_TO, 7.5,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 1.75,\n" \
"R_H_LINE_TO, 1.75,\n" \
"R_V_LINE_TO, 1.5,\n" \
"H_LINE_TO, 7.5,\n" \
"R_V_LINE_TO, 1.75,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.5, 15,\n" \
"R_CUBIC_TO, -0.4, 0, -0.75, -0.15, -1.05, -0.45,\n" \
"R_CUBIC_TO, -0.3, -0.3, -0.45, -0.65, -0.45, -1.05,\n" \
"R_V_LINE_TO, -7,\n" \
"R_CUBIC_TO, 0, -0.4, 0.14, -0.75, 0.42, -1.05,\n" \
"R_CUBIC_TO, 0.28, -0.3, 0.62, -0.45, 1.01, -0.45,\n" \
"H_LINE_TO, 5,\n" \
"R_LINE_TO, 1.5, 1.5,\n" \
"R_H_LINE_TO, -3,\n" \
"R_V_LINE_TO, 7,\n" \
"R_H_LINE_TO, 7.88,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"LINE_TO, 16, 18.13,\n" \
"LINE_TO, 12.88, 15,\n" \
"CLOSE,\n" \
"MOVE_TO, 18, 13.5,\n" \
"R_CUBIC_TO, 0, 0.32, -0.08, 0.61, -0.25, 0.86,\n" \
"R_CUBIC_TO, -0.17, 0.25, -0.41, 0.43, -0.73, 0.54,\n" \
"R_LINE_TO, -0.52, -0.52,\n" \
"V_LINE_TO, 6.5,\n" \
"H_LINE_TO, 8.63,\n" \
"LINE_TO, 7.13, 5,\n" \
"H_LINE_TO, 16.5,\n" \
"R_CUBIC_TO, 0.4, 0, 0.75, 0.15, 1.05, 0.45,\n" \
"R_CUBIC_TO, 0.3, 0.3, 0.45, 0.65, 0.45, 1.05,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -5.44, -3.06,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVideogameAssetOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseViewInArChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 9.25, 16.42,\n" \
"R_LINE_TO, -4.5, -2.56,\n" \
"R_ARC_TO, 1.69, 1.69, 0, 0, 1, -0.55, -0.57,\n" \
"R_ARC_TO, 1.43, 1.43, 0, 0, 1, -0.2, -0.75,\n" \
"V_LINE_TO, 7.44,\n" \
"R_CUBIC_TO, 0, -0.27, 0.07, -0.52, 0.2, -0.75,\n" \
"R_ARC_TO, 1.51, 1.51, 0, 0, 1, 0.56, -0.54,\n" \
"R_LINE_TO, 4.5, -2.58,\n" \
"R_ARC_TO, 1.45, 1.45, 0, 0, 1, 0.75, -0.21,\n" \
"R_CUBIC_TO, 0.27, 0, 0.52, 0.07, 0.75, 0.21,\n" \
"R_LINE_TO, 4.5, 2.58,\n" \
"R_CUBIC_TO, 0.24, 0.14, 0.42, 0.32, 0.56, 0.54,\n" \
"R_CUBIC_TO, 0.13, 0.23, 0.2, 0.48, 0.2, 0.75,\n" \
"R_V_LINE_TO, 5.13,\n" \
"R_ARC_TO, 1.4, 1.4, 0, 0, 1, -0.2, 0.73,\n" \
"R_ARC_TO, 1.71, 1.71, 0, 0, 1, -0.55, 0.56,\n" \
"R_LINE_TO, -4.5, 2.56,\n" \
"R_ARC_TO, 1.47, 1.47, 0, 0, 1, -1.5, 0,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -1.71,\n" \
"R_V_LINE_TO, -4.29,\n" \
"LINE_TO, 5.5, 8.17,\n" \
"R_V_LINE_TO, 4.39,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.5, 0,\n" \
"R_LINE_TO, 3.75, -2.14,\n" \
"V_LINE_TO, 8.19,\n" \
"R_LINE_TO, -3.75, 2.25,\n" \
"CLOSE,\n" \
"MOVE_TO, 2, 6,\n" \
"V_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, 0, -0.42, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 3.5, 2,\n" \
"H_LINE_TO, 6,\n" \
"R_V_LINE_TO, 1.5,\n" \
"H_LINE_TO, 3.5,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4, 12,\n" \
"H_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 2, 16.5,\n" \
"V_LINE_TO, 14,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 2.5,\n" \
"H_LINE_TO, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8, 0,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 2.5,\n" \
"V_LINE_TO, 14,\n" \
"H_LINE_TO, 18,\n" \
"R_V_LINE_TO, 2.5,\n" \
"R_CUBIC_TO, 0, 0.42, -0.15, 0.77, -0.44, 1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 16.5, 18,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.5, -12,\n" \
"V_LINE_TO, 3.5,\n" \
"H_LINE_TO, 14,\n" \
"V_LINE_TO, 2,\n" \
"R_H_LINE_TO, 2.5,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 9.13,\n" \
"R_LINE_TO, 3.63, -2.17,\n" \
"LINE_TO, 10, 4.88,\n" \
"LINE_TO, 6.38, 6.96,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 0.83,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -0.83,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0.75, 1.31,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.5, -0.02,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kViewInArChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseViewInArOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 11.33, 9.21,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.65, 3.52,\n" \
"LINE_TO, 6.98, 4.86,\n" \
"R_LINE_TO, 2.27, -1.29,\n" \
"R_ARC_TO, 1.45, 1.45, 0, 0, 1, 0.75, -0.21,\n" \
"R_CUBIC_TO, 0.27, 0, 0.52, 0.07, 0.75, 0.21,\n" \
"R_LINE_TO, 4.5, 2.58,\n" \
"ARC_TO, 1.48, 1.48, 0, 0, 1, 16, 7.44,\n" \
"R_V_LINE_TO, 5.13,\n" \
"R_CUBIC_TO, 0, 0.18, -0.03, 0.34, -0.1, 0.5,\n" \
"R_CUBIC_TO, -0.06, 0.16, -0.15, 0.31, -0.26, 0.45,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.25, 16.42,\n" \
"R_LINE_TO, -4.5, -2.56,\n" \
"R_ARC_TO, 1.68, 1.68, 0, 0, 1, -0.55, -0.56,\n" \
"R_ARC_TO, 1.44, 1.44, 0, 0, 1, -0.2, -0.75,\n" \
"V_LINE_TO, 7.44,\n" \
"R_ARC_TO, 1.49, 1.49, 0, 0, 1, 0.75, -1.29,\n" \
"R_LINE_TO, 0.9, -0.5,\n" \
"R_LINE_TO, 8.71, 8.71,\n" \
"R_LINE_TO, -3.6, 2.06,\n" \
"R_ARC_TO, 1.47, 1.47, 0, 0, 1, -1.5, 0,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.54, -7.75,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.54, 1.75,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2.63, -0.67,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.12, 1,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.11, 5.98,\n" \
"R_LINE_TO, 2.69, 2.69,\n" \
"R_LINE_TO, 2.83, -1.71,\n" \
"LINE_TO, 10, 4.88,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 1.15, 8.73,\n" \
"R_V_LINE_TO, -4.29,\n" \
"LINE_TO, 5.5, 8.17,\n" \
"R_V_LINE_TO, 4.39,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.25, -2.33,\n" \
"V_LINE_TO, 8.19,\n" \
"LINE_TO, 11.88, 9.75,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3.75, 2.33,\n" \
"R_LINE_TO, 2.52, -1.44,\n" \
"R_LINE_TO, -2.52, -2.52,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 5.98, 4.13,\n" \
"LINE_TO, 15.9, 18,\n" \
"H_LINE_TO, 14,\n" \
"R_V_LINE_TO, -1.5,\n" \
"R_H_LINE_TO, 0.56,\n" \
"R_V_LINE_TO, 0.17,\n" \
"LINE_TO, 3.21, 5.31,\n" \
"H_LINE_TO, 3.5,\n" \
"V_LINE_TO, 6,\n" \
"H_LINE_TO, 2,\n" \
"V_LINE_TO, 4.11,\n" \
"R_LINE_TO, -0.83, -0.83,\n" \
"LINE_TO, 2.23, 2.23,\n" \
"R_LINE_TO, 15.54, 15.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 18, 14,\n" \
"R_V_LINE_TO, 1.88,\n" \
"R_LINE_TO, -1.5, -1.5,\n" \
"V_LINE_TO, 14,\n" \
"CLOSE,\n" \
"MOVE_TO, 6, 2,\n" \
"R_V_LINE_TO, 1.5,\n" \
"R_H_LINE_TO, -0.37,\n" \
"LINE_TO, 4.13, 2,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 16,\n" \
"H_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, -0.42, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 2, 16.5,\n" \
"V_LINE_TO, 14,\n" \
"R_H_LINE_TO, 1.5,\n" \
"R_V_LINE_TO, 2.5,\n" \
"H_LINE_TO, 6,\n" \
"CLOSE,\n" \
"MOVE_TO, 16.5, 6,\n" \
"V_LINE_TO, 3.5,\n" \
"H_LINE_TO, 14,\n" \
"V_LINE_TO, 2,\n" \
"R_H_LINE_TO, 2.5,\n" \
"R_CUBIC_TO, 0.42, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"V_LINE_TO, 6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kViewInArOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVisibilityIcon) { \
              std::string s = \
"// Copyright 2020 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 12, 7,\n" \
"R_CUBIC_TO, -2.48, 0, -4.5, 2.02, -4.5, 4.5,\n" \
"CUBIC_TO_SHORTHAND, 9.52, 16, 12, 16,\n" \
"R_CUBIC_TO, 2.48, 0, 4.5, -2.02, 4.5, -4.5,\n" \
"CUBIC_TO_SHORTHAND, 14.48, 7, 12, 7,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 7.2,\n" \
"R_CUBIC_TO, -1.49, 0, -2.7, -1.21, -2.7, -2.7,\n" \
"R_CUBIC_TO, 0, -1.49, 1.21, -2.7, 2.7, -2.7,\n" \
"R_CUBIC_TO, 1.49, 0, 2.7, 1.21, 2.7, 2.7,\n" \
"R_CUBIC_TO, 0, 1.49, -1.21, 2.7, -2.7, 2.7,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 4,\n" \
"CUBIC_TO, 7, 4, 2.73, 7.11, 1, 11.5,\n" \
"CUBIC_TO, 2.73, 15.89, 7, 19, 12, 19,\n" \
"R_CUBIC_TO, 5, 0, 9.27, -3.11, 11, -7.5,\n" \
"CUBIC_TO, 21.27, 7.11, 17, 4, 12, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, 13,\n" \
"R_CUBIC_TO, -3.79, 0, -7.17, -2.13, -8.82, -5.5,\n" \
"CUBIC_TO, 4.83, 8.13, 8.21, 6, 12, 6,\n" \
"R_CUBIC_TO, 3.79, 0, 7.17, 2.13, 8.82, 5.5,\n" \
"CUBIC_TO, 19.17, 14.87, 15.79, 17, 12, 17,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVisibilityIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVisibilityOffIcon) { \
              std::string s = \
"// Copyright 2020 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 10.58, 7.25,\n" \
"R_LINE_TO, 1.56, 1.56,\n" \
"R_CUBIC_TO, 1.38, 0.07, 2.47, 1.17, 2.54, 2.54,\n" \
"R_LINE_TO, 1.56, 1.56,\n" \
"R_CUBIC_TO, 0.16, -0.44, 0.26, -0.91, 0.26, -1.41,\n" \
"CUBIC_TO, 16.5, 9.02, 14.48, 7, 12, 7,\n" \
"R_CUBIC_TO, -0.5, 0, -0.97, 0.1, -1.42, 0.25,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 6,\n" \
"R_CUBIC_TO, 3.79, 0, 7.17, 2.13, 8.82, 5.5,\n" \
"R_CUBIC_TO, -0.64, 1.32, -1.56, 2.44, -2.66, 3.33,\n" \
"R_LINE_TO, 1.42, 1.42,\n" \
"R_CUBIC_TO, 1.51, -1.26, 2.7, -2.89, 3.43, -4.74,\n" \
"CUBIC_TO, 21.27, 7.11, 17, 4, 12, 4,\n" \
"R_CUBIC_TO, -1.4, 0, -2.73, 0.25, -3.98, 0.7,\n" \
"R_LINE_TO, 1.61, 1.6,\n" \
"CUBIC_TO, 10.4, 6.12, 11.19, 6, 12, 6,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 4.43, 9.93,\n" \
"R_LINE_TO, -1.25, -1.25,\n" \
"R_LINE_TO, -1.27, -1.27,\n" \
"R_LINE_TO, -3.82, -3.82,\n" \
"R_LINE_TO, -1.27, -1.27,\n" \
"R_LINE_TO, -1.25, -1.25,\n" \
"R_LINE_TO, -1.48, -1.48,\n" \
"R_LINE_TO, -2.78, -2.78,\n" \
"R_LINE_TO, -1.42, 1.41,\n" \
"R_LINE_TO, 2.53, 2.53,\n" \
"CUBIC_TO, 2.92, 8.02, 1.73, 9.64, 1, 11.5,\n" \
"CUBIC_TO, 2.73, 15.89, 7, 19, 12, 19,\n" \
"R_CUBIC_TO, 1.4, 0, 2.73, -0.25, 3.98, -0.7,\n" \
"R_LINE_TO, 4.3, 4.3,\n" \
"R_LINE_TO, 1.41, -1.41,\n" \
"R_LINE_TO, -3.78, -3.78,\n" \
"R_LINE_TO, -1.48, -1.48,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -4.57, -1.74,\n" \
"R_CUBIC_TO, -1.38, -0.07, -2.47, -1.17, -2.54, -2.54,\n" \
"R_LINE_TO, 2.54, 2.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 12, 17,\n" \
"R_CUBIC_TO, -3.79, 0, -7.17, -2.13, -8.82, -5.5,\n" \
"R_CUBIC_TO, 0.64, -1.32, 1.56, -2.44, 2.66, -3.33,\n" \
"R_LINE_TO, 1.91, 1.91,\n" \
"R_CUBIC_TO, -0.15, 0.45, -0.25, 0.92, -0.25, 1.42,\n" \
"R_CUBIC_TO, 0, 2.48, 2.02, 4.5, 4.5, 4.5,\n" \
"R_CUBIC_TO, 0.5, 0, 0.97, -0.1, 1.42, -0.25,\n" \
"R_LINE_TO, 0.95, 0.95,\n" \
"R_CUBIC_TO, -0.77, 0.18, -1.56, 0.3, -2.37, 0.3,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVisibilityOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVolumeOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 16, 18.13,\n" \
"R_LINE_TO, -2.23, -2.23,\n" \
"R_ARC_TO, 6.93, 6.93, 0, 0, 1, -1.09, 0.55,\n" \
"R_CUBIC_TO, -0.42, 0.17, -0.82, 0.3, -1.18, 0.38,\n" \
"R_V_LINE_TO, -1.54,\n" \
"R_CUBIC_TO, 0.17, -0.07, 0.36, -0.15, 0.59, -0.24,\n" \
"R_CUBIC_TO, 0.23, -0.09, 0.42, -0.18, 0.57, -0.26,\n" \
"LINE_TO, 10, 12.13,\n" \
"R_V_LINE_TO, 3.9,\n" \
"R_LINE_TO, -4, -4,\n" \
"H_LINE_TO, 3,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 2.88,\n" \
"R_LINE_TO, -4, -4.02,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.12, -4.36,\n" \
"R_LINE_TO, -1.08, -1.08,\n" \
"R_CUBIC_TO, 0.22, -0.4, 0.4, -0.83, 0.52, -1.28,\n" \
"R_CUBIC_TO, 0.12, -0.45, 0.19, -0.91, 0.19, -1.39,\n" \
"R_ARC_TO, 5.27, 5.27, 0, 0, 0, -1.11, -3.3,\n" \
"ARC_TO, 5.64, 5.64, 0, 0, 0, 11.5, 4.73,\n" \
"V_LINE_TO, 3.19,\n" \
"R_CUBIC_TO, 1.63, 0.32, 2.95, 1.13, 3.97, 2.42,\n" \
"CUBIC_TO, 16.49, 6.9, 17, 8.37, 17, 10.02,\n" \
"R_CUBIC_TO, 0, 0.67, -0.1, 1.32, -0.29, 1.95,\n" \
"R_ARC_TO, 7.31, 7.31, 0, 0, 1, -0.83, 1.8,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.58, -2.58,\n" \
"LINE_TO, 11.5, 9.4,\n" \
"R_V_LINE_TO, -2.54,\n" \
"R_ARC_TO, 3.21, 3.21, 0, 0, 1, 1.47, 1.28,\n" \
"R_ARC_TO, 3.51, 3.51, 0, 0, 1, 0.53, 1.88,\n" \
"R_CUBIC_TO, 0, 0.2, -0.02, 0.39, -0.04, 0.59,\n" \
"R_ARC_TO, 2.37, 2.37, 0, 0, 1, -0.16, 0.58,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 7.89,\n" \
"LINE_TO, 8.06, 5.96,\n" \
"LINE_TO, 10, 4.02,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.5, 4.5,\n" \
"R_V_LINE_TO, -1.77,\n" \
"LINE_TO, 7, 9.13,\n" \
"R_LINE_TO, -0.37, 0.4,\n" \
"H_LINE_TO, 4.5,\n" \
"R_V_LINE_TO, 1,\n" \
"R_H_LINE_TO, 2.13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.75, -2.52,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"R_MOVE_TO, 12.84, 14.63,\n" \
"R_LINE_TO, -1.79, -1.8,\n" \
"R_ARC_TO, 5.1, 5.1, 0, 0, 1, -0.84, 0.41,\n" \
"R_CUBIC_TO, -0.33, 0.13, -0.63, 0.23, -0.91, 0.29,\n" \
"R_V_LINE_TO, -1.3,\n" \
"R_CUBIC_TO, 0.11, -0.05, 0.25, -0.11, 0.41, -0.17,\n" \
"R_CUBIC_TO, 0.16, -0.07, 0.3, -0.12, 0.41, -0.18,\n" \
"LINE_TO, 8.1, 9.86,\n" \
"R_V_LINE_TO, 3.28,\n" \
"LINE_TO, 4.71, 9.75,\n" \
"H_LINE_TO, 2.23,\n" \
"V_LINE_TO, 6.28,\n" \
"H_LINE_TO, 4.54,\n" \
"LINE_TO, 1.36, 3.07,\n" \
"R_LINE_TO, 0.89, -0.89,\n" \
"R_LINE_TO, 11.48, 11.56,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -3.57,\n" \
"R_LINE_TO, -0.91, -0.92,\n" \
"R_CUBIC_TO, 0.18, -0.32, 0.32, -0.66, 0.42, -1.02,\n" \
"R_CUBIC_TO, 0.1, -0.36, 0.15, -0.73, 0.15, -1.1,\n" \
"R_ARC_TO, 4.2, 4.2, 0, 0, 0, -0.89, -2.63,\n" \
"R_ARC_TO, 4.57, 4.57, 0, 0, 0, -2.3, -1.6,\n" \
"V_LINE_TO, 2.48,\n" \
"R_ARC_TO, 5.31, 5.31, 0, 0, 1, 3.22, 1.96,\n" \
"R_ARC_TO, 5.6, 5.6, 0, 0, 1, 1.24, 3.58,\n" \
"R_CUBIC_TO, 0, 0.55, -0.08, 1.07, -0.24, 1.58,\n" \
"R_CUBIC_TO, -0.16, 0.51, -0.39, 1, -0.68, 1.46,\n" \
"CLOSE,\n" \
"MOVE_TO, 10.78, 8.98,\n" \
"R_LINE_TO, -1.48, -1.49,\n" \
"V_LINE_TO, 5.43,\n" \
"R_ARC_TO, 2.82, 2.82, 0, 0, 1, 1.66, 2.58,\n" \
"R_CUBIC_TO, 0, 0.16, -0.02, 0.33, -0.04, 0.49,\n" \
"R_ARC_TO, 1.79, 1.79, 0, 0, 1, -0.14, 0.48,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -2.68, -2.7,\n" \
"LINE_TO, 6.42, 4.58,\n" \
"R_LINE_TO, 1.69, -1.68,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.72, 9.8,\n" \
"V_LINE_TO, 8.46,\n" \
"LINE_TO, 5.61, 7.35,\n" \
"R_LINE_TO, -0.3, 0.32,\n" \
"H_LINE_TO, 3.62,\n" \
"R_V_LINE_TO, 0.7,\n" \
"R_H_LINE_TO, 1.67,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -0.55, -1.9,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVolumeOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVolumeUpIcon) { \
              std::string s = \
"// Copyright 2017 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"MOVE_TO, 2, 5.95,\n" \
"R_V_LINE_TO, 4.1,\n" \
"R_H_LINE_TO, 2.67,\n" \
"LINE_TO, 8, 13.47,\n" \
"V_LINE_TO, 2.53,\n" \
"R_LINE_TO, -3.33, 3.42,\n" \
"H_LINE_TO, 2,\n" \
"CLOSE,\n" \
"MOVE_TO, 11, 8,\n" \
"R_CUBIC_TO, 0, -1.21, -0.68, -2.25, -1.67, -2.76,\n" \
"R_V_LINE_TO, 5.51,\n" \
"CUBIC_TO, 10.32, 10.25, 11, 9.21, 11, 8,\n" \
"CLOSE,\n" \
"MOVE_TO, 9.33, 2,\n" \
"R_V_LINE_TO, 1.41,\n" \
"R_CUBIC_TO, 1.93, 0.59, 3.33, 2.42, 3.33, 4.59,\n" \
"R_CUBIC_TO, 0, 2.17, -1.41, 4, -3.33, 4.59,\n" \
"V_LINE_TO, 14,\n" \
"CUBIC_TO, 12.01, 13.38, 14, 10.93, 14, 8,\n" \
"R_CUBIC_TO, 0, -2.93, -1.99, -5.38, -4.67, -6,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVolumeUpIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVolumeUpChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 11.5, 16.83,\n" \
"V_LINE_TO, 15.27,\n" \
"R_ARC_TO, 5.18, 5.18, 0, 0, 0, 2.9, -1.94,\n" \
"R_CUBIC_TO, 0.74, -0.98, 1.11, -2.08, 1.11, -3.32,\n" \
"R_CUBIC_TO, 0, -1.24, -0.37, -2.34, -1.11, -3.3,\n" \
"ARC_TO, 5.67, 5.67, 0, 0, 0, 11.5, 4.71,\n" \
"V_LINE_TO, 3.15,\n" \
"R_CUBIC_TO, 1.61, 0.35, 2.93, 1.16, 3.96, 2.44,\n" \
"CUBIC_TO, 16.48, 6.86, 17, 8.33, 17, 9.98,\n" \
"R_CUBIC_TO, 0, 1.65, -0.51, 3.12, -1.53, 4.41,\n" \
"R_CUBIC_TO, -1.02, 1.29, -2.34, 2.1, -3.97, 2.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 3, 11.98,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 3,\n" \
"R_LINE_TO, 4, -4,\n" \
"R_V_LINE_TO, 12,\n" \
"R_LINE_TO, -4, -4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.5, 1.15,\n" \
"V_LINE_TO, 6.83,\n" \
"R_CUBIC_TO, 0.63, 0.28, 1.12, 0.71, 1.47, 1.28,\n" \
"R_CUBIC_TO, 0.36, 0.57, 0.53, 1.19, 0.53, 1.88,\n" \
"R_CUBIC_TO, 0, 0.68, -0.18, 1.29, -0.53, 1.86,\n" \
"R_ARC_TO, 3.3, 3.3, 0, 0, 1, -1.47, 1.28,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -3, -5.52,\n" \
"LINE_TO, 6.63, 9.48,\n" \
"H_LINE_TO, 4.5,\n" \
"R_V_LINE_TO, 1,\n" \
"R_H_LINE_TO, 2.13,\n" \
"LINE_TO, 8.5, 12.36,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.61, 9.96,\n" \
"CLOSE\n" \
"\n" \
"CANVAS_DIMENSIONS, 16,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 9.31, 13.54,\n" \
"R_V_LINE_TO, -1.32,\n" \
"R_ARC_TO, 4.19, 4.19, 0, 0, 0, 2.31, -1.56,\n" \
"R_ARC_TO, 4.28, 4.28, 0, 0, 0, 0.89, -2.65,\n" \
"R_ARC_TO, 4.2, 4.2, 0, 0, 0, -0.89, -2.64,\n" \
"R_ARC_TO, 4.58, 4.58, 0, 0, 0, -2.3, -1.6,\n" \
"V_LINE_TO, 2.45,\n" \
"R_CUBIC_TO, 1.3, 0.28, 2.38, 0.94, 3.21, 1.97,\n" \
"R_CUBIC_TO, 0.84, 1.03, 1.25, 2.22, 1.25, 3.56,\n" \
"R_ARC_TO, 5.57, 5.57, 0, 0, 1, -1.24, 3.57,\n" \
"R_CUBIC_TO, -0.83, 1.04, -1.9, 1.7, -3.21, 1.98,\n" \
"CLOSE,\n" \
"MOVE_TO, 2.24, 9.72,\n" \
"V_LINE_TO, 6.25,\n" \
"R_H_LINE_TO, 2.48,\n" \
"R_LINE_TO, 3.39, -3.39,\n" \
"R_V_LINE_TO, 10.25,\n" \
"LINE_TO, 4.72, 9.72,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7.07, 0.83,\n" \
"V_LINE_TO, 5.42,\n" \
"R_CUBIC_TO, 0.51, 0.23, 0.91, 0.58, 1.21, 1.04,\n" \
"R_CUBIC_TO, 0.3, 0.46, 0.45, 0.97, 0.45, 1.54,\n" \
"R_CUBIC_TO, 0, 0.55, -0.15, 1.06, -0.45, 1.52,\n" \
"R_CUBIC_TO, -0.3, 0.46, -0.7, 0.81, -1.21, 1.04,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.72, 6.2,\n" \
"R_LINE_TO, -1.43, 1.43,\n" \
"H_LINE_TO, 3.62,\n" \
"R_V_LINE_TO, 0.7,\n" \
"H_LINE_TO, 5.29,\n" \
"LINE_TO, 6.72, 9.77,\n" \
"CLOSE,\n" \
"MOVE_TO, 5.25, 7.97,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVolumeUpChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVrHeadsetIcon) { \
              std::string s = \
"// Copyright 2019 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 24,\n" \
"MOVE_TO, 20.91, 6.68,\n" \
"ARC_TO, 2.54, 2.54, 0, 0, 0, 19.16, 6,\n" \
"H_LINE_TO, 4.9,\n" \
"R_CUBIC_TO, -0.66, 0, -1.28, 0.24, -1.75, 0.68,\n" \
"R_ARC_TO, 2.23, 2.23, 0, 0, 0, -0.72, 1.64,\n" \
"R_V_LINE_TO, 7.37,\n" \
"R_CUBIC_TO, 0, 0.62, 0.26, 1.2, 0.72, 1.64,\n" \
"ARC_TO, 2.54, 2.54, 0, 0, 0, 4.9, 18,\n" \
"R_H_LINE_TO, 3.42,\n" \
"R_CUBIC_TO, 0.45, 0, 0.88, -0.11, 1.27, -0.33,\n" \
"R_CUBIC_TO, 0.38, -0.22, 0.7, -0.52, 0.91, -0.89,\n" \
"R_LINE_TO, 0.97, -1.68,\n" \
"R_ARC_TO, 0.57, 0.57, 0, 0, 1, 0.16, -0.74,\n" \
"R_ARC_TO, 0.67, 0.67, 0, 0, 1, 0.81, 0,\n" \
"R_CUBIC_TO, 0.24, 0.18, 0.3, 0.49, 0.16, 0.74,\n" \
"R_LINE_TO, 0.97, 1.68,\n" \
"R_CUBIC_TO, 0.21, 0.37, 0.52, 0.68, 0.91, 0.89,\n" \
"R_CUBIC_TO, 0.38, 0.22, 0.82, 0.33, 1.27, 0.33,\n" \
"R_H_LINE_TO, 3.42,\n" \
"R_CUBIC_TO, 0.66, 0, 1.28, -0.24, 1.75, -0.68,\n" \
"R_CUBIC_TO, 0.47, -0.44, 0.72, -1.02, 0.72, -1.64,\n" \
"R_V_LINE_TO, -7.37,\n" \
"R_CUBIC_TO, 0, -0.62, -0.26, -1.2, -0.72, -1.64,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.83, 13.8,\n" \
"R_CUBIC_TO, -1.33, 0, -2.4, -1.08, -2.4, -2.4,\n" \
"R_CUBIC_TO, 0, -1.32, 1.08, -2.4, 2.4, -2.4,\n" \
"R_CUBIC_TO, 1.32, 0, 2.4, 1.08, 2.4, 2.4,\n" \
"R_CUBIC_TO, 0, 1.32, -1.07, 2.4, -2.4, 2.4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.4, 0,\n" \
"R_CUBIC_TO, -1.33, 0, -2.4, -1.08, -2.4, -2.4,\n" \
"R_CUBIC_TO, 0, -1.32, 1.08, -2.4, 2.4, -2.4,\n" \
"R_CUBIC_TO, 1.32, 0, 2.4, 1.08, 2.4, 2.4,\n" \
"R_CUBIC_TO, 0, 1.32, -1.07, 2.4, -2.4, 2.4,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVrHeadsetIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVrHeadsetChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 4.5, 13,\n" \
"R_H_LINE_TO, 3.19,\n" \
"R_LINE_TO, 0.62, -1.22,\n" \
"R_ARC_TO, 1.95, 1.95, 0, 0, 1, 0.7, -0.75,\n" \
"ARC_TO, 1.82, 1.82, 0, 0, 1, 10, 10.75,\n" \
"R_ARC_TO, 1.88, 1.88, 0, 0, 1, 1.69, 1.02,\n" \
"R_LINE_TO, 0.62, 1.23,\n" \
"H_LINE_TO, 15.5,\n" \
"V_LINE_TO, 7,\n" \
"R_H_LINE_TO, -11,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 2, -1.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, -0.15, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.3, 0.44, -0.65, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.41, -0.15, -0.77, -0.44, -1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 0, 6.5, 8.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, 0.15, -1.06, 0.44,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 0, 5, 10,\n" \
"R_CUBIC_TO, 0, 0.41, 0.15, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.3, 0.29, 0.65, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 3, 13,\n" \
"V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4.5, 5.5,\n" \
"R_H_LINE_TO, 11,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 6,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"R_ARC_TO, 1.44, 1.44, 0, 0, 1, -1.06, 0.44,\n" \
"R_H_LINE_TO, -3.19,\n" \
"R_CUBIC_TO, -0.28, 0, -0.54, -0.07, -0.78, -0.22,\n" \
"R_ARC_TO, 1.42, 1.42, 0, 0, 1, -0.55, -0.59,\n" \
"R_LINE_TO, -0.62, -1.21,\n" \
"R_ARC_TO, 0.4, 0.4, 0, 0, 0, -0.15, -0.17,\n" \
"R_ARC_TO, 0.36, 0.36, 0, 0, 0, -0.41, 0,\n" \
"R_ARC_TO, 0.34, 0.34, 0, 0, 0, -0.15, 0.17,\n" \
"R_LINE_TO, -0.62, 1.21,\n" \
"R_CUBIC_TO, -0.12, 0.25, -0.31, 0.45, -0.55, 0.59,\n" \
"R_ARC_TO, 1.5, 1.5, 0, 0, 1, -0.78, 0.22,\n" \
"H_LINE_TO, 4.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 3, 13,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 10.5, -1.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, -0.15, 1.06, -0.44,\n" \
"R_CUBIC_TO, 0.29, -0.3, 0.44, -0.65, 0.44, -1.06,\n" \
"R_CUBIC_TO, 0, -0.41, -0.15, -0.77, -0.44, -1.06,\n" \
"R_ARC_TO, 1.45, 1.45, 0, 0, 0, -1.06, -0.44,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, 0.15, -1.06, 0.44,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 0, 12, 10,\n" \
"R_CUBIC_TO, 0, 0.41, 0.15, 0.77, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0.3, 0.29, 0.65, 0.44, 1.06, 0.44,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 10,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVrHeadsetChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseVrHeadsetOffChromeRefreshIcon) { \
              std::string s = \
"// Copyright 2024 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16, 18.13,\n" \
"LINE_TO, 1.88, 4,\n" \
"R_LINE_TO, 1.06, -1.06,\n" \
"R_LINE_TO, 14.12, 14.13,\n" \
"CLOSE,\n" \
"MOVE_TO, 7.88, 10,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 8.69, 4.06,\n" \
"LINE_TO, 15.5, 13,\n" \
"V_LINE_TO, 7,\n" \
"H_LINE_TO, 9.11,\n" \
"R_LINE_TO, -1.48, -1.5,\n" \
"H_LINE_TO, 15.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 5.99,\n" \
"R_CUBIC_TO, 0, 0.22, -0.04, 0.41, -0.11, 0.59,\n" \
"R_CUBIC_TO, -0.08, 0.18, -0.18, 0.34, -0.32, 0.48,\n" \
"CLOSE,\n" \
"MOVE_TO, 6.5, 11.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 5, 10,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 6.5, 8.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.3, 0.44, 0.65, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0, 0.41, -0.15, 0.77, -0.44, 1.06,\n" \
"R_ARC_TO, 1.45, 1.45, 0, 0, 1, -1.06, 0.44,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 7.05, 0,\n" \
"LINE_TO, 12, 9.94,\n" \
"R_CUBIC_TO, 0.03, -0.4, 0.18, -0.74, 0.47, -1.02,\n" \
"ARC_TO, 1.43, 1.43, 0, 0, 1, 13.5, 8.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_CUBIC_TO, 0, 0.4, -0.14, 0.75, -0.43, 1.04,\n" \
"R_CUBIC_TO, -0.28, 0.29, -0.63, 0.44, -1.03, 0.46,\n" \
"CLOSE,\n" \
"R_MOVE_TO, -1.48, -1.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 4.5, 14.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 3, 13,\n" \
"V_LINE_TO, 7,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.77, 0.44, -1.06,\n" \
"ARC_TO, 1.44, 1.44, 0, 0, 1, 4.5, 5.5,\n" \
"R_H_LINE_TO, 1,\n" \
"LINE_TO, 7, 7,\n" \
"H_LINE_TO, 4.5,\n" \
"R_V_LINE_TO, 6,\n" \
"R_H_LINE_TO, 3.19,\n" \
"R_LINE_TO, 0.62, -1.22,\n" \
"R_ARC_TO, 1.95, 1.95, 0, 0, 1, 0.7, -0.75,\n" \
"ARC_TO, 1.82, 1.82, 0, 0, 1, 10, 10.75,\n" \
"R_H_LINE_TO, 0.75,\n" \
"R_LINE_TO, 3.75, 3.75,\n" \
"R_H_LINE_TO, -2.19,\n" \
"R_CUBIC_TO, -0.28, 0, -0.54, -0.07, -0.78, -0.22,\n" \
"R_ARC_TO, 1.42, 1.42, 0, 0, 1, -0.55, -0.59,\n" \
"R_LINE_TO, -0.62, -1.21,\n" \
"R_ARC_TO, 0.4, 0.4, 0, 0, 0, -0.15, -0.17,\n" \
"R_ARC_TO, 0.36, 0.36, 0, 0, 0, -0.41, 0,\n" \
"R_ARC_TO, 0.34, 0.34, 0, 0, 0, -0.15, 0.17,\n" \
"R_LINE_TO, -0.62, 1.21,\n" \
"R_CUBIC_TO, -0.12, 0.25, -0.31, 0.45, -0.55, 0.59,\n" \
"R_ARC_TO, 1.5, 1.5, 0, 0, 1, -0.78, 0.22,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kVrHeadsetOffChromeRefreshIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseWarningIcon) { \
              std::string s = \
"// Copyright 2015 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"MOVE_TO, 2, 42,\n" \
"R_H_LINE_TO, 44,\n" \
"LINE_TO, 24, 4,\n" \
"LINE_TO, 2, 42,\n" \
"CLOSE,\n" \
"// Divergence from ic_warning: white fill on the ! portion.\n" \
"NEW_PATH,\n" \
"PATH_COLOR_ARGB, 0xFF, 0xFF, 0xFF, 0xFF,\n" \
"MOVE_TO, 26, 36,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -4,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 4,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 0, -8,\n" \
"R_H_LINE_TO, -4,\n" \
"R_V_LINE_TO, -8,\n" \
"R_H_LINE_TO, 4,\n" \
"R_V_LINE_TO, 8,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kWarningIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseWarningOutlineIcon) { \
              std::string s = \
"// Copyright 2022 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"MOVE_TO, 9.13, 2.5,\n" \
"CUBIC_TO, 9.52, 1.83, 10.48, 1.83, 10.87, 2.5,\n" \
"LINE_TO, 18.87, 16.5,\n" \
"CUBIC_TO, 19.25, 17.17, 18.77, 18, 18, 18,\n" \
"H_LINE_TO, 2,\n" \
"CUBIC_TO, 1.23, 18, 0.75, 17.17, 1.13, 16.5,\n" \
"LINE_TO, 9.13, 2.5,\n" \
"CLOSE,\n" \
"MOVE_TO, 10, 5.02,\n" \
"LINE_TO, 3.72, 16,\n" \
"H_LINE_TO, 16.28,\n" \
"LINE_TO, 10, 5.02,\n" \
"CLOSE,\n" \
"MOVE_TO, 11, 12,\n" \
"V_LINE_TO, 8,\n" \
"H_LINE_TO, 9,\n" \
"V_LINE_TO, 12,\n" \
"H_LINE_TO, 11,\n" \
"CLOSE,\n" \
"MOVE_TO, 11, 13,\n" \
"V_LINE_TO, 15,\n" \
"H_LINE_TO, 9,\n" \
"V_LINE_TO, 13,\n" \
"H_LINE_TO, 11,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kWarningOutlineIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseWebAssetOffIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 16.73, 18.83,\n" \
"LINE_TO, 13.88, 16,\n" \
"H_LINE_TO, 3.5,\n" \
"R_CUBIC_TO, -0.41, 0, -0.77, -0.15, -1.06, -0.44,\n" \
"ARC_TO, 1.45, 1.45, 0, 0, 1, 2, 14.5,\n" \
"V_LINE_TO, 5.49,\n" \
"R_CUBIC_TO, 0, -0.41, 0.15, -0.76, 0.46, -1.05,\n" \
"ARC_TO, 1.55, 1.55, 0, 0, 1, 3.56, 4,\n" \
"R_H_LINE_TO, 0.23,\n" \
"R_V_LINE_TO, 1.9,\n" \
"LINE_TO, 1.17, 3.27,\n" \
"LINE_TO, 2.23, 2.23,\n" \
"R_LINE_TO, 15.54, 15.54,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.5, 14.5,\n" \
"R_H_LINE_TO, 8.88,\n" \
"LINE_TO, 4.88, 7,\n" \
"H_LINE_TO, 3.5,\n" \
"CLOSE,\n" \
"R_MOVE_TO, 14.13, 1,\n" \
"LINE_TO, 16.5, 14.38,\n" \
"V_LINE_TO, 7,\n" \
"H_LINE_TO, 9.13,\n" \
"R_LINE_TO, -3, -3,\n" \
"H_LINE_TO, 16.5,\n" \
"R_CUBIC_TO, 0.41, 0, 0.77, 0.15, 1.06, 0.44,\n" \
"R_CUBIC_TO, 0.29, 0.29, 0.44, 0.65, 0.44, 1.06,\n" \
"R_V_LINE_TO, 9,\n" \
"R_CUBIC_TO, 0, 0.19, -0.04, 0.38, -0.1, 0.56,\n" \
"R_CUBIC_TO, -0.07, 0.18, -0.16, 0.33, -0.27, 0.44,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kWebAssetOffIcon); \
              } \
 \
TEST_F(VectorIconsTest, ParseWorkIcon) { \
              std::string s = \
"// Copyright 2025 The Chromium Authors\n" \
"// Use of this source code is governed by a BSD-style license that can be\n" \
"// found in the LICENSE file.\n" \
"\n" \
"CANVAS_DIMENSIONS, 20,\n" \
"FILL_RULE_NONZERO,\n" \
"MOVE_TO, 3.33f, 17.5f,\n" \
"CUBIC_TO, 2.88f, 17.5f, 2.48f, 17.34f, 2.16f, 17.01f,\n" \
"CUBIC_TO, 1.83f, 16.68f, 1.67f, 16.29f, 1.67f, 15.83f,\n" \
"LINE_TO, 1.67f, 6.67f,\n" \
"CUBIC_TO, 1.67f, 6.21f, 1.83f, 5.82f, 2.16f, 5.49f,\n" \
"CUBIC_TO, 2.48f, 5.16f, 2.88f, 5, 3.33f, 5,\n" \
"LINE_TO, 6.67f, 5,\n" \
"LINE_TO, 6.67f, 3.33f,\n" \
"CUBIC_TO, 6.67f, 2.88f, 6.83f, 2.48f, 7.16f, 2.16f,\n" \
"CUBIC_TO, 7.48f, 1.83f, 7.88f, 1.67f, 8.33f, 1.67f,\n" \
"LINE_TO, 11.67f, 1.67f,\n" \
"CUBIC_TO, 12.13f, 1.67f, 12.52f, 1.83f, 12.84f, 2.16f,\n" \
"CUBIC_TO, 13.17f, 2.48f, 13.33f, 2.88f, 13.33f, 3.33f,\n" \
"LINE_TO, 13.33f, 5,\n" \
"LINE_TO, 16.67f, 5,\n" \
"CUBIC_TO, 17.13f, 5, 17.52f, 5.16f, 17.84f, 5.49f,\n" \
"CUBIC_TO, 18.17f, 5.82f, 18.33f, 6.21f, 18.33f, 6.67f,\n" \
"LINE_TO, 18.33f, 15.83f,\n" \
"CUBIC_TO, 18.33f, 16.29f, 18.17f, 16.68f, 17.84f, 17.01f,\n" \
"CUBIC_TO, 17.52f, 17.34f, 17.13f, 17.5f, 16.67f, 17.5f,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.33f, 15.83f,\n" \
"LINE_TO, 16.67f, 15.83f,\n" \
"LINE_TO, 16.67f, 6.67f,\n" \
"LINE_TO, 3.33f, 6.67f,\n" \
"CLOSE,\n" \
"MOVE_TO, 8.33f, 5,\n" \
"LINE_TO, 11.67f, 5,\n" \
"LINE_TO, 11.67f, 3.33f,\n" \
"LINE_TO, 8.33f, 3.33f,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.33f, 15.83f,\n" \
"LINE_TO, 3.33f, 6.67f,\n" \
"CLOSE,\n" \
"MOVE_TO, 3.33f, 15.83f,\n" \
"CLOSE\n" \
; \
              CheckThatParsedElementsMatch(s, kWorkIcon); \
              } \
 \

#else
#error This file should not be included more than once.
#endif  // COMPONENTS_VECTOR_ICONS_VECTOR_ICONS_UNITTEST_H_

