import type { AvatarIcon } from 'chrome://resources/cr_elements/cr_profile_avatar_selector/cr_profile_avatar_selector.js';
/**
 * This is the data structure sent back and forth between C++ and JS.
 */
export interface ProfileState {
    profilePath: string;
    localProfileName: string;
    isSyncing: boolean;
    needsSignin: boolean;
    gaiaName: string;
    userName: string;
    avatarIcon: string;
    profileCardButtonLabel: string;
    avatarBadge: string;
    hasEnterpriseLabel: boolean;
}
/**
 * This is the data structure sent back and forth between C++ and JS.
 * `colorId` has the following special values:
 *   - `-1` for the default theme.
 *   - `0` for a manually picked color theme.
 */
export interface AutogeneratedThemeColorInfo {
    colorId: number;
    color: number;
    themeFrameColor: string;
    themeShapeColor: string;
    themeFrameTextColor: string;
    themeGenericAvatar: string;
}
/**
 * This is the data structure sent back and forth between C++ and JS.
 * `colorId` has the following special values:
 *   - `-1` for the default theme..
 *   - `0` for a manually picked color theme
 * `color` is defined only for manually picked themes.
 */
export interface UserThemeChoice {
    colorId: number;
    color?: number;
}
export interface ManageProfilesBrowserProxy {
    /**
     * Initializes profile picker main view.
     */
    initializeMainView(): void;
    /**
     * Launches picked profile and closes the profile picker.
     */
    launchSelectedProfile(profilePath: string): void;
    /**
     * Opens profile on manage profile settings sub page and closes the
     * profile picker.
     */
    openManageProfileSettingsSubPage(profilePath: string): void;
    /** Launches Guest profile. */
    launchGuestProfile(): void;
    /**
     * Inform native the user's choice on whether to show the profile picker
     * on startup or not.
     */
    askOnStartupChanged(shouldShow: boolean): void;
    /**
     * Retrieves suggested theme for the new profile.
     * @return A promise firing with the suggested theme info, once it has been
     *     retrieved.
     */
    getNewProfileSuggestedThemeInfo(): Promise<AutogeneratedThemeColorInfo>;
    /**
     * Retrieves all relevant theme information for the particular theme.
     * @param theme A theme which info needs to be retrieved.
     * @return A promise firing with the theme info, once it has been retrieved.
     */
    getProfileThemeInfo(theme: UserThemeChoice): Promise<AutogeneratedThemeColorInfo>;
    /**
     * Retrieves profile statistics to be shown in the remove profile warning.
     */
    getProfileStatistics(profilePath: string): void;
    /**
     * Stops showing the profile statistics and removes the related keep alive,
     * unloading the profile for which the statistics are currently being shown if
     * it has no more keep alives.
     */
    closeProfileStatistics(): void;
    /**
     * Removes profile.
     */
    removeProfile(profilePath: string): void;
    /**
     * Starts a signin flow to get an account that will be added to a new profile.
     */
    selectNewAccount(profileColor: number | null): void;
    /**
     * Retrieves custom avatar list for the select avatar dialog.
     */
    getAvailableIcons(): Promise<AvatarIcon[]>;
    /**
     * Creates local profile and opens a profile customization modal dialog on a
     * browser window.
     * TODO(crbug.com/40209493): Add createShortcut parameter.
     */
    continueWithoutAccount(profileColor: number): void;
    /**
     * Sets the local profile name.
     */
    setProfileName(profilePath: string, profileName: string): void;
    /** Records impression of a sign-in promo to metrics. */
    recordSignInPromoImpression(): void;
    /**
     * Gets the `ProfileState` for a profile.
     */
    getProfileState(profilePath: string): Promise<ProfileState>;
    /**
     * Switches to an already existing profile at `profile_path`.
     */
    confirmProfileSwitch(profilePath: string): void;
    /**
     * Cancels the profile switch which aborts the sign-in profile creation
     * flow.
     */
    cancelProfileSwitch(): void;
    /**
     * Sends the profile order changes
     * @param fromIndex the initial index of the tile that was dragged.
     * @param toIndex the index to which the profile has been moved/dropped.
     * All other profiles between `fromIndex` and `toIndex` +/-1 should be shifted
     * by +/-1 depending on the change direction.
     */
    updateProfileOrder(fromIndex: number, toIndex: number): void;
    /**
     * Loads the last used profile; opens/uses a browser and open the "Sign in to
     * Chrome" Help center page. Does not close the Picker.
     */
    onLearnMoreClicked(): void;
}
/** @implements {ManageProfilesBrowserProxy} */
export declare class ManageProfilesBrowserProxyImpl {
    initializeMainView(): void;
    launchSelectedProfile(profilePath: string): void;
    openManageProfileSettingsSubPage(profilePath: string): void;
    launchGuestProfile(): void;
    askOnStartupChanged(shouldShow: boolean): void;
    getNewProfileSuggestedThemeInfo(): Promise<any>;
    getProfileThemeInfo(theme: UserThemeChoice): Promise<any>;
    removeProfile(profilePath: string): void;
    getProfileStatistics(profilePath: string): void;
    closeProfileStatistics(): void;
    selectNewAccount(profileColor: number | null): void;
    getAvailableIcons(): Promise<any>;
    continueWithoutAccount(profileColor: number): void;
    setProfileName(profilePath: string, profileName: string): void;
    recordSignInPromoImpression(): void;
    getProfileState(profileSwitchPath: string): Promise<any>;
    confirmProfileSwitch(profilePath: string): void;
    cancelProfileSwitch(): void;
    updateProfileOrder(fromIndex: number, toIndex: number): void;
    onLearnMoreClicked(): void;
    static getInstance(): ManageProfilesBrowserProxy;
    static setInstance(obj: ManageProfilesBrowserProxy): void;
}
