import type { ContentSettingsTypes } from '../site_settings/constants.js';
/**
 * Constants used in safety hub C++ to JS communication.
 * Their values need be kept in sync with their counterparts in
 * chrome/browser/ui/webui/settings/safety_hub_handler.h and
 * chrome/browser/ui/webui/settings/safety_hub_handler.cc
 */
export declare enum SafetyHubEvent {
    UNUSED_PERMISSIONS_MAYBE_CHANGED = "unused-permission-review-list-maybe-changed",
    NOTIFICATION_PERMISSIONS_MAYBE_CHANGED = "notification-permission-review-list-maybe-changed",
    EXTENSIONS_CHANGED = "extensions-review-list-maybe-changed"
}
export interface NotificationPermission {
    origin: string;
    notificationInfoString: string;
}
export declare enum PermissionsRevocationType {
    UNUSED_PERMISSIONS = 0,
    ABUSIVE_NOTIFICATION_PERMISSIONS = 1,
    DISRUPTIVE_NOTIFICATION_PERMISSIONS = 2,
    UNUSED_PERMISSIONS_AND_ABUSIVE_NOTIFICATIONS = 3,
    UNUSED_PERMISSIONS_AND_DISRUPTIVE_NOTIFICATIONS = 4
}
export interface UnusedSitePermissions {
    origin: string;
    permissions: ContentSettingsTypes[];
    expiration: string;
    revocationType: PermissionsRevocationType;
}
export interface CardInfo {
    header: string;
    subheader: string;
    state: CardState;
}
/**
 * A Safety Hub card has 4 different states as represented below. Depending on
 * the card state, the card will be updated.
 * Should be kept in sync with the corresponding enum in
 * chrome/browser/ui/safety_hub/safety_hub_constants.h.
 */
export declare enum CardState {
    WARNING = 0,
    WEAK = 1,
    INFO = 2,
    SAFE = 3
}
export interface EntryPointInfo {
    hasRecommendations: boolean;
    header: string;
    subheader: string;
}
export interface SafetyHubBrowserProxy {
    /**
     * Mark revoked permissions of unused sites as reviewed by the user so they
     * will not be shown again.
     */
    acknowledgeRevokedUnusedSitePermissionsList(): void;
    /**
     * Allow permissions again for an unused site where permissions were
     * auto-revoked. The origin will not appear again for the user to review and
     * permissions will not be auto-revoked for this origin in the future.
     */
    allowPermissionsAgainForUnusedSite(origin: string): void;
    /**
     * Gets the unused origins along with the permissions they have been granted.
     */
    getRevokedUnusedSitePermissionsList(): Promise<UnusedSitePermissions[]>;
    /**
     * Reverse the changes made by |acknowledgeRevokedUnusedSitePermissionsList|.
     * The list of sites will be presented again to the user for review.
     */
    undoAcknowledgeRevokedUnusedSitePermissionsList(unusedSitePermissionsList: UnusedSitePermissions[]): void;
    /**
     * Reverse the changes made by |allowPermissionsAgainForUnusedSite|. This will
     * revoke the origin's permissions, re-enable auto-revocation for this origin,
     * and the entry will be visible again in the UI.
     */
    undoAllowPermissionsAgainForUnusedSite(unusedSitePermissions: UnusedSitePermissions): void;
    /** Gets the site list that send a lot of notifications. */
    getNotificationPermissionReview(): Promise<NotificationPermission[]>;
    /** Blocks the notification permission for all origins in the list. */
    blockNotificationPermissionForOrigins(origins: string[]): void;
    /** Allows the notification permission for all origins in the list */
    allowNotificationPermissionForOrigins(origins: string[]): void;
    /** Adds the origins to blocklist for the notification permissions feature. */
    ignoreNotificationPermissionForOrigins(origins: string[]): void;
    /**
     * Removes the origins from the blocklist for the notification permissions
     * feature.
     */
    undoIgnoreNotificationPermissionForOrigins(origins: string[]): void;
    /** Resets the notification permission for the origins. */
    resetNotificationPermissionForOrigins(origin: string[]): void;
    /**
     * When Safety Hub is visited, the active three-dot menu notification is
     * dismissed, if there is any.
     */
    dismissActiveMenuNotification(): void;
    /** Gets data for the password top card. */
    getPasswordCardData(): Promise<CardInfo>;
    /** Gets data for the Safe Browsing top card. */
    getSafeBrowsingCardData(): Promise<CardInfo>;
    /** Gets data for the version top card. */
    getVersionCardData(): Promise<CardInfo>;
    /** Get the number of extensions that should be reviewed by the user. */
    getNumberOfExtensionsThatNeedReview(): Promise<number>;
    /** Get the subheader for Safety Hub entry point in settings. */
    getSafetyHubEntryPointData(): Promise<EntryPointInfo>;
    recordSafetyHubPageVisit(): void;
    recordSafetyHubInteraction(): void;
}
export declare class SafetyHubBrowserProxyImpl implements SafetyHubBrowserProxy {
    acknowledgeRevokedUnusedSitePermissionsList(): void;
    allowPermissionsAgainForUnusedSite(origin: string): void;
    getRevokedUnusedSitePermissionsList(): Promise<any>;
    undoAcknowledgeRevokedUnusedSitePermissionsList(unusedSitePermissionsList: UnusedSitePermissions[]): void;
    undoAllowPermissionsAgainForUnusedSite(unusedSitePermissions: UnusedSitePermissions): void;
    getNotificationPermissionReview(): Promise<any>;
    blockNotificationPermissionForOrigins(origins: string[]): void;
    allowNotificationPermissionForOrigins(origins: string[]): void;
    ignoreNotificationPermissionForOrigins(origins: string[]): void;
    undoIgnoreNotificationPermissionForOrigins(origins: string[]): void;
    resetNotificationPermissionForOrigins(origins: string[]): void;
    dismissActiveMenuNotification(): void;
    getPasswordCardData(): Promise<any>;
    getSafeBrowsingCardData(): Promise<any>;
    getVersionCardData(): Promise<any>;
    getNumberOfExtensionsThatNeedReview(): Promise<any>;
    getSafetyHubEntryPointData(): Promise<any>;
    recordSafetyHubPageVisit(): Promise<any>;
    recordSafetyHubInteraction(): Promise<any>;
    static getInstance(): SafetyHubBrowserProxy;
    static setInstance(obj: SafetyHubBrowserProxy): void;
}
