// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'credit-card-list-entry' is a credit card row to be shown in
 * the settings page.
 */
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_shared_vars.css.js';
import '../settings_shared.css.js';
import './passwords_shared.css.js';
import './screen_reader_only.css.js';
import { I18nMixin } from '//resources/cr_elements/i18n_mixin.js';
import { assert, assertNotReached } from 'chrome://resources/js/assert.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { loadTimeData } from '../i18n_setup.js';
import { CardBenefitsUserAction, MetricsBrowserProxyImpl } from '../metrics_browser_proxy.js';
import { getTemplate } from './credit_card_list_entry.html.js';
const SettingsCreditCardListEntryElementBase = I18nMixin(PolymerElement);
export class SettingsCreditCardListEntryElement extends SettingsCreditCardListEntryElementBase {
    static get is() {
        return 'settings-credit-card-list-entry';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /** A saved credit card. */
            creditCard: Object,
            showNewFopDisplayEnabled_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('enableNewFopDisplay');
                },
                readOnly: true,
            },
        };
    }
    get dotsMenu() {
        return this.shadowRoot.getElementById('creditCardMenu');
    }
    /**
     * Opens the credit card action menu.
     */
    onDotsMenuClick_() {
        this.dispatchEvent(new CustomEvent('dots-card-menu-click', {
            bubbles: true,
            composed: true,
            detail: {
                creditCard: this.creditCard,
                anchorElement: this.shadowRoot.querySelector('#creditCardMenu'),
            },
        }));
    }
    onRemoteEditClick_() {
        this.dispatchEvent(new CustomEvent('remote-card-menu-click', {
            bubbles: true,
            composed: true,
            detail: {
                creditCard: this.creditCard,
                anchorElement: this.shadowRoot.querySelector('#creditCardMenu'),
            },
        }));
    }
    onSummarySublabelTermsLinkClick_() {
        // Log the metric for user clicking on the card benefits terms hyperlink.
        MetricsBrowserProxyImpl.getInstance().recordAction(CardBenefitsUserAction.CARD_BENEFITS_TERMS_LINK_CLICKED);
    }
    getCardNumberDescription_(creditCard) {
        const cardNumber = creditCard.cardNumber;
        if (cardNumber) {
            const lastFourDigits = cardNumber.substring(Math.max(0, cardNumber.length - 4));
            if (lastFourDigits) {
                const network = creditCard.network || this.i18n('genericCreditCard');
                return this.i18n('creditCardDescription', network, lastFourDigits);
            }
        }
        return undefined;
    }
    /**
     * @returns the title for the More Actions button corresponding to the card
     *     which is described by the nickname or the network name and last 4
     *     digits or name. If a card has CVC saved, there will be additional
     *     description to notify of the same.
     */
    moreActionsTitle_() {
        const cardDescription = this.creditCard.nickname ||
            this.getCardNumberDescription_(this.creditCard) ||
            this.creditCard.name;
        return this.i18n(this.creditCard.cvc ? 'moreActionsForCreditCardWithCvc' :
            'moreActionsForCreditCard', cardDescription);
    }
    /**
     * Returns true if the new FOP display should be shown.
     */
    shouldShowNewFopDisplay_() {
        return this.showNewFopDisplayEnabled_;
    }
    /**
     * The card has a product description or a nickname.
     */
    hasCardIdentifier_() {
        return (this.creditCard.metadata.summarySublabel || '').length > 0;
    }
    /**
     * The 3-dot menu should be shown if the card is not a masked server card or
     * if the card is eligible for virtual card enrollment.
     */
    showDots_() {
        return !!(this.creditCard.metadata.isLocal ||
            this.isVirtualCardEnrollmentEligible_());
    }
    isVirtualCardEnrollmentEligible_() {
        return this.creditCard.metadata.isVirtualCardEnrollmentEligible;
    }
    isVirtualCardEnrolled_() {
        return this.creditCard.metadata.isVirtualCardEnrolled;
    }
    getCardIdentifierAriaLabel_() {
        return this.creditCard.metadata.summaryLabel || '';
    }
    getSummaryAriaLabel_() {
        const cardNumberDescription = this.getCardNumberDescription_(this.creditCard);
        if (cardNumberDescription) {
            return this.i18n('creditCardA11yLabeled', cardNumberDescription);
        }
        return this.creditCard.metadata.summaryLabel;
    }
    /**
     * Returns an aria label for the benefits terms link such as "See terms for
     * Amex ending in 0001". If no card description is available, then the
     * default text such as "See terms here" is returned.
     */
    getBenefitsTermsAriaLabel_() {
        const cardNumberDescription = this.getCardNumberDescription_(this.creditCard);
        if (cardNumberDescription) {
            return this.i18n('benefitsTermsAriaLabel', cardNumberDescription);
        }
        return this.i18n('benefitsTermsTagForCreditCardListEntry');
    }
    getCardExpiryDate_() {
        assert(this.creditCard.expirationMonth);
        assert(this.creditCard.expirationYear);
        // Truncate the year down to two digits (eg. 2023 to 23).
        return this.creditCard.expirationMonth + '/' +
            this.creditCard.expirationYear.toString().substring(2);
    }
    getCardSublabelType() {
        if (this.isVirtualCardEnrolled_()) {
            if (this.isCardCvcAvailable_()) {
                return this.isCardBenefitsProductUrlAvailable_() ?
                    7 /* CardSummarySublabelType.VIRTUAL_CARD_WITH_CVC_AND_BENEFITS_TAG */ :
                    6 /* CardSummarySublabelType.VIRTUAL_CARD_WITH_CVC_TAG */;
            }
            return this.isCardBenefitsProductUrlAvailable_() ?
                5 /* CardSummarySublabelType.VIRTUAL_CARD_WITH_BENEFITS_TAG */ :
                4 /* CardSummarySublabelType.VIRTUAL_CARD */;
        }
        if (this.isCardCvcAvailable_()) {
            return this.isCardBenefitsProductUrlAvailable_() ?
                3 /* CardSummarySublabelType.EXPIRATION_DATE_WITH_CVC_AND_BENEFITS_TAG */ :
                2 /* CardSummarySublabelType.EXPIRATION_DATE_WITH_CVC_TAG */;
        }
        return this.isCardBenefitsProductUrlAvailable_() ?
            1 /* CardSummarySublabelType.EXPIRATION_DATE_WITH_BENEFITS_TAG */ :
            0 /* CardSummarySublabelType.EXPIRATION_DATE */;
    }
    /**
     * Returns expiration date.
     */
    getExpirationlabel_() {
        return ' · ' + this.getCardExpiryDate_();
    }
    /**
     * Returns one of the following sublabels, based on the card's status:
     *   Virtual card enrollment tag
     *   'CVC saved' tag
     *   Benefit tag (Place the benefit tag last because it includes a link to
     *                product terms.)
     * e.g., one of the following:
     *   CVC saved
     *   Card benefits available (terms apply)
     *   CVC saved | Card benefits available (terms apply)
     *   Virtual card turned on
     *   Virtual card turned on | CVC saved
     *   Virtual card turned on | Card benefits available (terms apply)
     *   Virtual card turned on | CVC saved | Card benefits available (terms
     *     apply)
     */
    getSummarySublabel_() {
        const separator = ' | ';
        let summarySublabel = this.isVirtualCardEnrolled_() ?
            this.i18n('virtualCardTurnedOn') :
            (this.shouldShowNewFopDisplay_() ? '' : this.getCardExpiryDate_());
        if (this.isCardCvcAvailable_()) {
            if (summarySublabel.length > 0) {
                summarySublabel += separator;
            }
            summarySublabel += this.i18n('cvcTagForCreditCardListEntry');
        }
        return summarySublabel;
    }
    hasSummaryAndBenefitSublabel_() {
        return this.getSummarySublabel_().length > 0 &&
            this.isCardBenefitsProductUrlAvailable_();
    }
    getSummaryAriaSublabel_() {
        switch (this.getCardSublabelType()) {
            case 7 /* CardSummarySublabelType.VIRTUAL_CARD_WITH_CVC_AND_BENEFITS_TAG */:
            case 5 /* CardSummarySublabelType.VIRTUAL_CARD_WITH_BENEFITS_TAG */:
            case 6 /* CardSummarySublabelType.VIRTUAL_CARD_WITH_CVC_TAG */:
            case 4 /* CardSummarySublabelType.VIRTUAL_CARD */:
                return this.shouldShowNewFopDisplay_() ?
                    this.i18n('creditCardExpDateA11yLabeled', this.getCardExpiryDate_()) +
                        this.getSummarySublabel_() :
                    this.getSummarySublabel_();
            case 3 /* CardSummarySublabelType.EXPIRATION_DATE_WITH_CVC_AND_BENEFITS_TAG */:
            case 1 /* CardSummarySublabelType.EXPIRATION_DATE_WITH_BENEFITS_TAG */:
            case 2 /* CardSummarySublabelType.EXPIRATION_DATE_WITH_CVC_TAG */:
            case 0 /* CardSummarySublabelType.EXPIRATION_DATE */:
                return this.i18n('creditCardExpDateA11yLabeled', this.shouldShowNewFopDisplay_() ? this.getCardExpiryDate_() :
                    this.getSummarySublabel_());
            default:
                assertNotReached();
        }
    }
    shouldShowVirtualCardSecondarySublabel_() {
        return this.creditCard.metadata.summarySublabel.trim() !== '' ||
            this.isVirtualCardEnrolled_() ||
            this.isVirtualCardEnrollmentEligible_();
    }
    shouldShowPaymentsIndicator_() {
        return !this.creditCard.metadata.isLocal;
    }
    isCardCvcAvailable_() {
        return loadTimeData.getBoolean('cvcStorageAvailable') &&
            !!this.creditCard.cvc;
    }
    isCardBenefitsProductUrlAvailable_() {
        return !!this.creditCard.productTermsUrl;
    }
    getCardBenefitsProductUrl_() {
        return this.creditCard.productTermsUrl || '';
    }
    /**
     * When the provided `imageSrc` points toward a processor's default card art,
     * this function returns a string that will scale the image based on the
     * user's screen resolution, otherwise it will return the unmodified
     * `imageSrc`.
     */
    getCardImage_(imageSrc) {
        return imageSrc.startsWith('chrome://theme') ?
            this.getScaledSrcSet_(imageSrc) :
            imageSrc;
    }
    /**
     * This function returns a string that can be used in a srcset to scale
     * the provided `url` based on the user's screen resolution.
     */
    getScaledSrcSet_(url) {
        return `${url} 1x, ${url}@2x 2x`;
    }
}
customElements.define(SettingsCreditCardListEntryElement.is, SettingsCreditCardListEntryElement);
