// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_components/managed_dialog/managed_dialog.js';
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_link_row/cr_link_row.js';
import 'chrome://resources/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/cr_elements/md_select.css.js';
import '../controls/controlled_radio_button.js';
import '/shared/settings/controls/extension_controlled_indicator.js';
import '../controls/settings_radio_group.js';
import '../controls/settings_toggle_button.js';
import '../settings_page/settings_section.js';
import '../settings_shared.css.js';
import '../settings_vars.css.js';
import './home_url_input.js';
import '../controls/settings_dropdown_menu.js';
import { PrefsMixin } from '/shared/settings/prefs/prefs_mixin.js';
import { CustomizeColorSchemeModeBrowserProxy } from 'chrome://resources/cr_components/customize_color_scheme_mode/browser_proxy.js';
import { ColorSchemeMode } from 'chrome://resources/cr_components/customize_color_scheme_mode/customize_color_scheme_mode.mojom-webui.js';
import { I18nMixin } from 'chrome://resources/cr_elements/i18n_mixin.js';
import { assert } from 'chrome://resources/js/assert.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { loadTimeData } from '../i18n_setup.js';
import { pageVisibility } from '../page_visibility.js';
import { RelaunchMixin, RestartType } from '../relaunch_mixin.js';
import { routes } from '../route.js';
import { Router } from '../router.js';
import { SettingsViewMixin } from '../settings_page/settings_view_mixin.js';
import { AppearanceBrowserProxyImpl } from './appearance_browser_proxy.js';
import { getTemplate } from './appearance_page.html.js';
/**
 * This is the absolute difference maintained between standard and
 * fixed-width font sizes. http://crbug.com/91922.
 */
const SIZE_DIFFERENCE_FIXED_STANDARD = 3;
/**
 * ID for autogenerated themes. Should match
 * |ThemeService::kAutogeneratedThemeID|.
 */
const AUTOGENERATED_THEME_ID = 'autogenerated_theme_id';
/**
 * ID for user color themes. Should match
 * |ThemeService::kUserColorThemeID|.
 */
const USER_COLOR_THEME_ID = 'user_color_theme_id';
// This must be kept in sync with the SystemTheme enum in
// ui/color/system_theme.h.
export var SystemTheme;
(function (SystemTheme) {
    // Either classic or web theme.
    SystemTheme[SystemTheme["DEFAULT"] = 0] = "DEFAULT";
    // 
})(SystemTheme || (SystemTheme = {}));
const SettingsAppearancePageElementBase = SettingsViewMixin(RelaunchMixin(I18nMixin(PrefsMixin(PolymerElement))));
export class SettingsAppearancePageElement extends SettingsAppearancePageElementBase {
    static get is() {
        return 'settings-appearance-page';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * Dictionary defining page visibility.
             */
            pageVisibility_: {
                type: Object,
                value: () => pageVisibility?.appearance,
            },
            defaultZoom_: Number,
            isWallpaperPolicyControlled_: { type: Boolean, value: true },
            colorSchemeModeOptions_: {
                readOnly: true,
                type: Array,
                value() {
                    return [
                        {
                            value: ColorSchemeMode.kLight,
                            name: loadTimeData.getString('lightMode'),
                        },
                        {
                            value: ColorSchemeMode.kDark,
                            name: loadTimeData.getString('darkMode'),
                        },
                        {
                            value: ColorSchemeMode.kSystem,
                            name: loadTimeData.getString('systemMode'),
                        },
                    ];
                },
            },
            selectedColorSchemeMode_: Number,
            /**
             * List of options for the font size drop-down menu.
             */
            fontSizeOptions_: {
                readOnly: true,
                type: Array,
                value() {
                    return [
                        { value: 9, name: loadTimeData.getString('verySmall') },
                        { value: 12, name: loadTimeData.getString('small') },
                        { value: 16, name: loadTimeData.getString('medium') },
                        { value: 20, name: loadTimeData.getString('large') },
                        { value: 24, name: loadTimeData.getString('veryLarge') },
                    ];
                },
            },
            /**
             * Predefined zoom factors to be used when zooming in/out. These are in
             * ascending order. Values are displayed in the page zoom drop-down menu
             * as percentages.
             */
            pageZoomLevels_: Array,
            themeSublabel_: String,
            themeUrl_: String,
            systemTheme_: {
                type: Object,
                value: SystemTheme.DEFAULT,
            },
            isForcedTheme_: {
                type: Boolean,
                computed: 'computeIsForcedTheme_(' +
                    'prefs.autogenerated.theme.policy.color.controlledBy)',
            },
            // 
            showHoverCardImagesOption_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('showHoverCardImagesOption');
                },
            },
            showManagedThemeDialog_: Boolean,
            sidePanelOptions_: {
                readOnly: true,
                type: Array,
                value() {
                    return [
                        {
                            value: 'true',
                            name: loadTimeData.getString('uiFeatureAlignRight'),
                        },
                        {
                            value: 'false',
                            name: loadTimeData.getString('uiFeatureAlignLeft'),
                        },
                    ];
                },
            },
            showTabSearchPositionSettings_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('showTabSearchPositionSettings');
                },
            },
            showTabSearchPositionRestartButton_: {
                type: Boolean,
                value: false,
            },
            showSplitViewDragAndDropSetting_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('showSplitViewDragAndDropSetting');
                },
            },
            showResetPinnedActionsButton_: {
                type: Boolean,
                value: false,
            },
            tabSearchOptions_: {
                readOnly: true,
                type: Array,
                value() {
                    return [
                        {
                            value: 'true',
                            name: loadTimeData.getString('uiFeatureAlignRight'),
                        },
                        {
                            value: 'false',
                            name: loadTimeData.getString('uiFeatureAlignLeft'),
                        },
                    ];
                },
            },
        };
    }
    static get observers() {
        return [
            'defaultFontSizeChanged_(prefs.webkit.webprefs.default_font_size.value)',
            'themeChanged_(' +
                'prefs.extensions.theme.id.value, systemTheme_, isForcedTheme_)',
            'updateShowTabSearchRestartButton_(' +
                'prefs.tab_search.is_right_aligned.value)',
            // 
            'toolbarPinningStateChanged_(prefs.toolbar.pinned_actions.value,' +
                'prefs.browser.show_home_button.value,' +
                'prefs.browser.show_forward_button.value)',
        ];
    }
    appearanceBrowserProxy_ = AppearanceBrowserProxyImpl.getInstance();
    colorSchemeModeHandler_ = CustomizeColorSchemeModeBrowserProxy.getInstance().handler;
    colorSchemeModeCallbackRouter_ = CustomizeColorSchemeModeBrowserProxy.getInstance().callbackRouter;
    setColorSchemeModeListenerId_ = null;
    ready() {
        super.ready();
        this.$.defaultFontSize.menuOptions = this.fontSizeOptions_;
        // TODO(dschuyler): Look into adding a listener for the
        // default zoom percent.
        this.appearanceBrowserProxy_.getDefaultZoom().then(zoom => {
            this.defaultZoom_ = zoom;
        });
        this.pageZoomLevels_ =
            JSON.parse(loadTimeData.getString('presetZoomFactors'));
        this.setColorSchemeModeListenerId_ =
            this.colorSchemeModeCallbackRouter_.setColorSchemeMode.addListener((colorSchemeMode) => {
                this.selectedColorSchemeMode_ =
                    this.colorSchemeModeOptions_
                        .find(mode => colorSchemeMode === mode.value)
                        ?.value;
            });
        this.colorSchemeModeHandler_.initializeColorSchemeMode();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        assert(this.setColorSchemeModeListenerId_);
        this.colorSchemeModeCallbackRouter_.removeListener(this.setColorSchemeModeListenerId_);
    }
    /** @return A zoom easier read by users. */
    formatZoom_(zoom) {
        return Math.round(zoom * 100);
    }
    /**
     * @param showHomepage Whether to show home page.
     * @param isNtp Whether to use the NTP as the home page.
     * @param homepageValue If not using NTP, use this URL.
     */
    getShowHomeSubLabel_(showHomepage, isNtp, homepageValue) {
        if (!showHomepage) {
            return this.i18n('homeButtonDisabled');
        }
        if (isNtp) {
            return this.i18n('homePageNtp');
        }
        return homepageValue || this.i18n('customWebAddress');
    }
    onCustomizeFontsClick_() {
        Router.getInstance().navigateTo(routes.FONTS);
    }
    onDisableExtension_() {
        this.dispatchEvent(new CustomEvent('refresh-pref', { bubbles: true, composed: true, detail: 'homepage' }));
    }
    /**
     * @param value The changed font size slider value.
     */
    defaultFontSizeChanged_(value) {
        // This pref is handled separately in some extensions, but here it is tied
        // to default_font_size (to simplify the UI).
        this.set('prefs.webkit.webprefs.default_fixed_font_size.value', value - SIZE_DIFFERENCE_FIXED_STANDARD);
    }
    onThemeClick_() {
        this.appearanceBrowserProxy_.openCustomizeChrome();
    }
    onCustomizeToolbarClick_() {
        this.appearanceBrowserProxy_.openCustomizeChromeToolbarSection();
    }
    onUseDefaultClick_() {
        if (this.isForcedTheme_) {
            this.showManagedThemeDialog_ = true;
            return;
        }
        this.appearanceBrowserProxy_.useDefaultTheme();
    }
    onResetPinnedToolbarActionsClick_() {
        this.appearanceBrowserProxy_.resetPinnedToolbarActions();
    }
    // 
    themeChanged_(themeId) {
        if (this.prefs === undefined || this.systemTheme_ === undefined) {
            return;
        }
        if (themeId.length > 0 && themeId !== AUTOGENERATED_THEME_ID &&
            themeId !== USER_COLOR_THEME_ID && !this.isForcedTheme_) {
            assert(this.systemTheme_ === SystemTheme.DEFAULT);
            this.appearanceBrowserProxy_.getThemeInfo(themeId).then(info => {
                this.themeSublabel_ = info.name;
            });
            this.themeUrl_ = 'https://chrome.google.com/webstore/detail/' + themeId;
            return;
        }
        this.themeUrl_ = '';
        if (themeId === AUTOGENERATED_THEME_ID || themeId === USER_COLOR_THEME_ID ||
            this.isForcedTheme_) {
            this.themeSublabel_ = this.i18n('chromeColors');
            return;
        }
        // 
        // 
        this.themeSublabel_ = '';
        // 
    }
    /** @return Whether applied theme is set by policy. */
    computeIsForcedTheme_() {
        return !!this.getPref('autogenerated.theme.policy.color').controlledBy;
    }
    async toolbarPinningStateChanged_() {
        this.showResetPinnedActionsButton_ =
            !(await this.appearanceBrowserProxy_.pinnedToolbarActionsAreDefault());
    }
    isSelectedColorSchemeMode_(colorSchemeMode) {
        return colorSchemeMode === this.selectedColorSchemeMode_;
    }
    onColorSchemeModeChange_() {
        this.colorSchemeModeHandler_.setColorSchemeMode(parseInt(this.$.colorSchemeModeSelect.value, 10));
    }
    onZoomLevelChange_() {
        chrome.settingsPrivate.setDefaultZoom(parseFloat(this.$.zoomLevel.value));
    }
    /** @see blink::ZoomValuesEqual(). */
    zoomValuesEqual_(zoom1, zoom2) {
        return Math.abs(zoom1 - zoom2) <= 0.001;
    }
    showHr_(previousIsVisible, nextIsVisible) {
        return previousIsVisible && nextIsVisible;
    }
    onHoverCardImagesToggleChange_(event) {
        const enabled = event.target.checked;
        this.appearanceBrowserProxy_.recordHoverCardImagesEnabledChanged(enabled);
    }
    onManagedDialogClosed_() {
        this.showManagedThemeDialog_ = false;
    }
    onTabSearchPositionRestartClick_(e) {
        // Prevent event from bubbling up to the toggle button.
        e.stopPropagation();
        this.performRestart(RestartType.RESTART);
    }
    updateShowTabSearchRestartButton_(newValue) {
        this.showTabSearchPositionRestartButton_ = newValue !==
            loadTimeData.getBoolean('tabSearchIsRightAlignedAtStartup');
    }
    // SettingsViewMixin
    getFocusConfig() {
        const map = new Map();
        if (routes.FONTS) {
            map.set(routes.FONTS.path, '#customize-fonts-subpage-trigger');
        }
        return map;
    }
    // SettingsViewMixin implementation.
    getAssociatedControlFor(childViewId) {
        assert(childViewId === 'fonts');
        const control = this.shadowRoot.querySelector('#customize-fonts-subpage-trigger');
        assert(control);
        return control;
    }
}
customElements.define(SettingsAppearancePageElement.is, SettingsAppearancePageElement);
