import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { css, html, CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { mojo } from 'chrome://resources/mojo/mojo/public/js/bindings.js';
import { String16Spec } from 'chrome://resources/mojo/mojo/public/mojom/base/string16.mojom-webui.js';
import { TimeTicksSpec, JSTimeSpec, TimeDeltaSpec } from 'chrome://resources/mojo/mojo/public/mojom/base/time.mojom-webui.js';
import { UrlSpec } from 'chrome://resources/mojo/url/mojom/url.mojom-webui.js';
import 'chrome://omnibox-popup.top-chrome/strings.m.js';
import { PageCallbackRouter as PageCallbackRouter$1, PageHandler as PageHandler$1 } from 'chrome://resources/cr_components/color_change_listener/color_change_listener.mojom-webui.js';

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getBaseFaviconUrl() {
    const faviconUrl = new URL('chrome://favicon2/');
    faviconUrl.searchParams.set('size', '16');
    faviconUrl.searchParams.set('scaleFactor', 'SCALEFACTORx');
    return faviconUrl;
}
function getDefaultFaviconUrlParams() {
    return {
        isSyncedUrlForHistoryUi: false,
        remoteIconUrlForUma: '',
        size: 16,
        forceLightMode: false,
        fallbackToHost: true,
        forceEmptyDefaultFavicon: false,
        scaleFactor: '',
    };
}
/**
 * Creates a favicon request URL based on the given parameters.
 *
 * @param url URL of the original page
 * @param optionalParams Options object that specifies additional parameters to
 *     configure the favicon request URL that's constructed by this function.
 *
 * @return URL for the favicon request.
 */
function getFaviconUrl(url, optionalParams) {
    const params = Object.assign(getDefaultFaviconUrlParams(), optionalParams);
    // Note: URL param keys used below must match those in the description of
    // chrome://favicon2 format in components/favicon_base/favicon_url_parser.h.
    const faviconUrl = getBaseFaviconUrl();
    faviconUrl.searchParams.set('pageUrl', url);
    faviconUrl.searchParams.set('size', params.size.toString());
    // TODO(dbeam): use the presence of 'allowGoogleServerFallback' to
    // indicate true, otherwise false.
    const fallback = params.isSyncedUrlForHistoryUi ? '1' : '0';
    faviconUrl.searchParams.set('allowGoogleServerFallback', fallback);
    if (params.isSyncedUrlForHistoryUi) {
        faviconUrl.searchParams.set('iconUrl', params.remoteIconUrlForUma);
    }
    if (params.forceLightMode) {
        faviconUrl.searchParams.set('forceLightMode', 'true');
    }
    if (!params.fallbackToHost) {
        faviconUrl.searchParams.set('fallbackToHost', '0');
    }
    if (params.forceEmptyDefaultFavicon) {
        faviconUrl.searchParams.set('forceEmptyDefaultFavicon', '1');
    }
    if (params.scaleFactor) {
        faviconUrl.searchParams.set('scaleFactor', params.scaleFactor);
    }
    return faviconUrl.toString();
}

let instance$f = null;
function getCss$b() {
    return instance$f || (instance$f = [...[], css `[hidden],:host([hidden]){display:none !important}`]);
}

let instance$e = null;
function getCss$a() {
    return instance$e || (instance$e = [...[getCss$b()], css `:host{--cr-searchbox-icon-border-radius:8px;align-items:center;display:flex;flex-shrink:0;justify-content:center;width:var(--cr-searchbox-icon-container-size,32px)}:host(:not([is-lens-searchbox_])){--cr-searchbox-icon-border-radius:4px}#container{align-items:center;aspect-ratio:1/1;border-radius:var(--cr-searchbox-icon-border-radius);display:flex;justify-content:center;overflow:hidden;position:relative;width:100%}:host([has-image_]:not([in-searchbox]):not([is-weather-answer])) #container{background-color:var(--color-searchbox-results-icon-container-background,var(--container-bg-color))}:host([has-icon-container-background]:not([in-searchbox])) #container{background-color:var(--color-searchbox-answer-icon-background)}:host([is-weather-answer]:not([in-searchbox])) #container{background-color:var(--color-searchbox-results-background)}#image{display:none;height:100%;object-fit:contain;width:100%}:host([has-image_]:not([in-searchbox])) #image{display:initial}:host([is-answer]) #image{max-height:24px;max-width:24px}#icon{height:24px;width:24px}#faviconImageContainer{width:24px;height:24px;display:flex;justify-content:center;align-items:center}#faviconImage{height:16px;width:16px}#icon{-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-searchbox-results-search-icon-size,16px);background-color:var(--color-searchbox-search-icon-background)}:host([in-searchbox][is-lens-searchbox_]) #icon{background-color:var(--color-searchbox-google-g-background);height:var(--cr-searchbox-icon-size-in-searchbox);width:var(--cr-searchbox-icon-size-in-searchbox)}@media (forced-colors:active){:host([in-searchbox][is-lens-searchbox_]) #icon{background-color:ButtonText}}:host([in-searchbox][favicon-image_*='//resources/cr_components/omnibox/icons/google_g.svg']) #faviconImage{width:24px;height:24px}:host([in-searchbox]) #icon{-webkit-mask-size:var(--cr-searchbox-icon-size-in-searchbox)}:host([in-searchbox]) #faviconImage{width:var(--cr-searchbox-icon-size-in-searchbox);height:var(--cr-searchbox-icon-size-in-searchbox)}:host([has-icon-container-background]:not([in-searchbox])) #icon{background-color:var(--color-searchbox-answer-icon-foreground)}:host([has-icon-container-background][is-starter-pack]:not([in-searchbox])) #icon,:host([has-icon-container-background][is-featured-enterprise-search]:not([in-searchbox])) #icon{background-color:var(--color-searchbox-results-starter-pack-icon,var(--color-searchbox-answer-icon-foreground))}#iconImg{height:var(--cr-searchbox-results-search-icon-size,16px);width:var(--cr-searchbox-results-search-icon-size,16px)}:host([in-searchbox]) #iconImg{height:var(--cr-searchbox-icon-size-in-searchbox);width:var(--cr-searchbox-icon-size-in-searchbox)}:host([has-image_]:not([in-searchbox])) #icon,:host([has-image_]:not([in-searchbox])) #iconImg,:host([has-image_]:not([in-searchbox])) #faviconImageContainer{display:none}:host(:not([in-searchbox])[is-lens-searchbox_]) #container{background-color:var(--color-searchbox-results-icon-container-background);border-radius:4000px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$5() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="container"
    style="--container-bg-color:${this.getContainerBgColor_()};">
  <img id="image" src="${this.imageSrc_}" ?hidden="${!this.showImage_}"
      @load="${this.onImageLoad_}" @error="${this.onImageError_}">

  <div ?hidden="${this.showIconImg_}">
    <div id="icon" style="-webkit-mask-image: ${this.maskImage};"
        ?hidden="${this.showFaviconImage_}">
    </div>
    <div id="faviconImageContainer"
        ?hidden="${!this.showFaviconImage_}">
      <img id="faviconImage" src="${this.faviconImage_}"
          srcset="${this.faviconImageSrcSet_}"
          @load="${this.onFaviconLoad_}"
          @error="${this.onFaviconError_}">
    </div>
  </div>

  <img id="iconImg" src="${this.iconSrc_}" ?hidden="${!this.showIconImg_}"
      @load="${this.onIconLoad_}">
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CALCULATOR = 'search-calculator-answer';
const DOCUMENT_MATCH_TYPE = 'document';
const FEATURED_ENTERPRISE_SEARCH = 'featured-enterprise-search';
const HISTORY_CLUSTER_MATCH_TYPE = 'history-cluster';
const PEDAL = 'pedal';
const STARTER_PACK = 'starter-pack';
// The LHS icon. Used on autocomplete matches as well as the searchbox input to
// render icons, favicons, and entity images.
class SearchboxIconElement extends CrLitElement {
    static get is() {
        return 'cr-searchbox-icon';
    }
    static get styles() {
        return getCss$a();
    }
    render() {
        return getHtml$5.bind(this)();
    }
    static get properties() {
        return {
            //========================================================================
            // Public properties
            //========================================================================
            /**
             * The default icon to show when no match is selected and/or for
             * non-navigation matches. Only set in the context of the searchbox input.
             */
            defaultIcon: { type: String },
            /**  Whether icon should have a background. */
            hasIconContainerBackground: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether icon is in searchbox or not. Used to prevent
             * the match icon of rich suggestions from showing in the context of the
             * searchbox input.
             */
            inSearchbox: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether icon belongs to an answer or not. Used to prevent
             * the match image from taking size of container.
             */
            isAnswer: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether icon belongs to a starter pack match.
             */
            isStarterPack: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether icon belongs to a featured enterprise search match.
             */
            isFeaturedEnterpriseSearch: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether suggestion answer is of answer type weather. Weather answers
             * don't have the same background as other suggestion answers.
             */
            isWeatherAnswer: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether suggestion is an enterprise search aggregator people
             * suggestion. Enterprise search aggregator people suggestions should not
             * use a set background color when image is missing, unlike other rich
             * suggestion answers.
             */
            isEnterpriseSearchAggregatorPeopleType: {
                type: Boolean,
                reflect: true,
            },
            /** Used as a mask image on #icon if `faviconImage_` is empty. */
            maskImage: {
                type: String,
                reflect: true,
            },
            match: { type: Object },
            //========================================================================
            // Private properties
            //========================================================================
            /** Used as the image src for the #faviconImage if non-empty. */
            faviconImage_: {
                type: String,
                reflect: true,
            },
            /**
             * Used as the image srcset for the #faviconImage if non-empty.
             */
            faviconImageSrcSet_: { state: true, type: String },
            /**
             * Whether the match features an image (as opposed to an icon or favicon).
             */
            hasImage_: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether to use the favicon image instead of the default vector icon
             * for the suggestion.
             */
            showFaviconImage_: { state: true, type: Boolean },
            /**
             * Flag indicating whether or not a favicon is loading.
             */
            faviconLoading_: { state: true, type: Boolean },
            /**
             * Flag indicating whether or not a favicon was successfully loaded.
             * This is used to force the WebUI popup to make use of the default vector
             * icon when the favicon image is unavailable.
             */
            faviconError_: { state: true, type: Boolean },
            /** Used as the image src for the #iconImg if non-empty. */
            iconSrc_: { state: true, type: String },
            /**
             * Flag indicating whether or not an icon image is loading. This is used
             * to show a default icon while the image is loading.
             */
            iconLoading_: { state: true, type: Boolean },
            /**
             * Whether to use the icon image instead of the default icon for the
             * suggestion.
             */
            showIconImg_: { state: true, type: Boolean },
            showImage_: { state: true, type: Boolean },
            imageSrc_: { state: true, type: String },
            /**
             * Flag indicating whether or not an image is loading. This is used to
             * show a placeholder color while the image is loading.
             */
            imageLoading_: { state: true, type: Boolean },
            /**
             * Flag indicating whether or not an image was successfully loaded. This
             * is used to suppress the default "broken image" icon as needed.
             */
            imageError_: {
                state: true,
                type: Boolean,
            },
            isTopChromeSearchbox_: { state: true, type: Boolean },
            isLensSearchbox_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #defaultIcon_accessor_storage = '';
    get defaultIcon() { return this.#defaultIcon_accessor_storage; }
    set defaultIcon(value) { this.#defaultIcon_accessor_storage = value; }
    #hasIconContainerBackground_accessor_storage = false;
    get hasIconContainerBackground() { return this.#hasIconContainerBackground_accessor_storage; }
    set hasIconContainerBackground(value) { this.#hasIconContainerBackground_accessor_storage = value; }
    #inSearchbox_accessor_storage = false;
    get inSearchbox() { return this.#inSearchbox_accessor_storage; }
    set inSearchbox(value) { this.#inSearchbox_accessor_storage = value; }
    #isAnswer_accessor_storage = false;
    get isAnswer() { return this.#isAnswer_accessor_storage; }
    set isAnswer(value) { this.#isAnswer_accessor_storage = value; }
    #isStarterPack_accessor_storage = false;
    get isStarterPack() { return this.#isStarterPack_accessor_storage; }
    set isStarterPack(value) { this.#isStarterPack_accessor_storage = value; }
    #isFeaturedEnterpriseSearch_accessor_storage = false;
    get isFeaturedEnterpriseSearch() { return this.#isFeaturedEnterpriseSearch_accessor_storage; }
    set isFeaturedEnterpriseSearch(value) { this.#isFeaturedEnterpriseSearch_accessor_storage = value; }
    #isWeatherAnswer_accessor_storage = false;
    get isWeatherAnswer() { return this.#isWeatherAnswer_accessor_storage; }
    set isWeatherAnswer(value) { this.#isWeatherAnswer_accessor_storage = value; }
    #isEnterpriseSearchAggregatorPeopleType_accessor_storage = false;
    get isEnterpriseSearchAggregatorPeopleType() { return this.#isEnterpriseSearchAggregatorPeopleType_accessor_storage; }
    set isEnterpriseSearchAggregatorPeopleType(value) { this.#isEnterpriseSearchAggregatorPeopleType_accessor_storage = value; }
    #maskImage_accessor_storage = '';
    get maskImage() { return this.#maskImage_accessor_storage; }
    set maskImage(value) { this.#maskImage_accessor_storage = value; }
    #match_accessor_storage = null;
    get match() { return this.#match_accessor_storage; }
    set match(value) { this.#match_accessor_storage = value; }
    #faviconImage__accessor_storage = '';
    get faviconImage_() { return this.#faviconImage__accessor_storage; }
    set faviconImage_(value) { this.#faviconImage__accessor_storage = value; }
    #faviconImageSrcSet__accessor_storage = '';
    get faviconImageSrcSet_() { return this.#faviconImageSrcSet__accessor_storage; }
    set faviconImageSrcSet_(value) { this.#faviconImageSrcSet__accessor_storage = value; }
    #hasImage__accessor_storage = false;
    get hasImage_() { return this.#hasImage__accessor_storage; }
    set hasImage_(value) { this.#hasImage__accessor_storage = value; }
    #showFaviconImage__accessor_storage = false;
    get showFaviconImage_() { return this.#showFaviconImage__accessor_storage; }
    set showFaviconImage_(value) { this.#showFaviconImage__accessor_storage = value; }
    #faviconLoading__accessor_storage = false;
    get faviconLoading_() { return this.#faviconLoading__accessor_storage; }
    set faviconLoading_(value) { this.#faviconLoading__accessor_storage = value; }
    #faviconError__accessor_storage = false;
    get faviconError_() { return this.#faviconError__accessor_storage; }
    set faviconError_(value) { this.#faviconError__accessor_storage = value; }
    #iconSrc__accessor_storage = '';
    get iconSrc_() { return this.#iconSrc__accessor_storage; }
    set iconSrc_(value) { this.#iconSrc__accessor_storage = value; }
    #iconLoading__accessor_storage = false;
    get iconLoading_() { return this.#iconLoading__accessor_storage; }
    set iconLoading_(value) { this.#iconLoading__accessor_storage = value; }
    #showIconImg__accessor_storage = false;
    get showIconImg_() { return this.#showIconImg__accessor_storage; }
    set showIconImg_(value) { this.#showIconImg__accessor_storage = value; }
    #showImage__accessor_storage = false;
    get showImage_() { return this.#showImage__accessor_storage; }
    set showImage_(value) { this.#showImage__accessor_storage = value; }
    #imageSrc__accessor_storage = '';
    get imageSrc_() { return this.#imageSrc__accessor_storage; }
    set imageSrc_(value) { this.#imageSrc__accessor_storage = value; }
    #imageLoading__accessor_storage = false;
    get imageLoading_() { return this.#imageLoading__accessor_storage; }
    set imageLoading_(value) { this.#imageLoading__accessor_storage = value; }
    #imageError__accessor_storage = false;
    get imageError_() { return this.#imageError__accessor_storage; }
    set imageError_(value) { this.#imageError__accessor_storage = value; }
    #isTopChromeSearchbox__accessor_storage = loadTimeData.getBoolean('isTopChromeSearchbox');
    get isTopChromeSearchbox_() { return this.#isTopChromeSearchbox__accessor_storage; }
    set isTopChromeSearchbox_(value) { this.#isTopChromeSearchbox__accessor_storage = value; }
    #isLensSearchbox__accessor_storage = loadTimeData.getBoolean('isLensSearchbox');
    get isLensSearchbox_() { return this.#isLensSearchbox__accessor_storage; }
    set isLensSearchbox_(value) { this.#isLensSearchbox__accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('match')) {
            this.iconSrc_ = this.computeIconSrc_();
            this.imageSrc_ = this.computeImageSrc_();
            this.isAnswer = this.computeIsAnswer_();
            this.isEnterpriseSearchAggregatorPeopleType =
                this.computeIsEnterpriseSearchAggregatorPeopleType_();
            this.isStarterPack = this.computeIsStarterPack_();
            this.isFeaturedEnterpriseSearch =
                this.computeIsFeaturedEnterpriseSearch();
            this.isWeatherAnswer = this.computeIsWeatherAnswer_();
            this.hasImage_ = this.computeHasImage_();
            this.maskImage = this.computeMaskImage_();
        }
        if (changedProperties.has('match') ||
            changedProperties.has('isWeatherAnswer')) {
            this.hasIconContainerBackground =
                this.computeHasIconContainerBackground_();
        }
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('match') ||
            changedProperties.has('defaultIcon') ||
            changedPrivateProperties.has('isTopChromeSearchbox_')) {
            this.faviconImage_ = this.computeFaviconImage_();
        }
        if (changedProperties.has('match') ||
            changedPrivateProperties.has('faviconImage_') ||
            changedPrivateProperties.has('isTopChromeSearchbox_')) {
            this.faviconImageSrcSet_ = this.computeFaviconImageSrcSet_();
        }
        if (changedPrivateProperties.has('faviconImage_')) {
            // If `faviconImage_` changes to a new truthy value, a new favicon is
            // being loaded.
            this.faviconLoading_ = !!this.faviconImage_;
            this.faviconError_ = false;
        }
        if (changedProperties.has('match') ||
            changedPrivateProperties.has('isLensSearchbox_') ||
            changedPrivateProperties.has('faviconImage_') ||
            changedPrivateProperties.has('faviconLoading_') ||
            changedPrivateProperties.has('faviconError_')) {
            this.showFaviconImage_ = this.computeShowFaviconImage_();
        }
        if (changedPrivateProperties.has('iconSrc_')) {
            // If iconSrc_ changes to a new truthy value, a new icon is being loaded.
            this.iconLoading_ = !!this.iconSrc_;
        }
        if (changedPrivateProperties.has('imageSrc_')) {
            // If imageSrc_ changes to a new truthy value, a new image is being
            // loaded.
            this.imageLoading_ = !!this.imageSrc_;
            this.imageError_ = false;
        }
        if (changedPrivateProperties.has('imageSrc_') ||
            changedPrivateProperties.has('imageError_')) {
            this.showImage_ = this.computeShowImage_();
        }
        if (changedProperties.has('match') ||
            changedPrivateProperties.has('isLensSearchbox_') ||
            changedPrivateProperties.has('iconLoading_')) {
            this.showIconImg_ = this.computeShowIconImg_();
        }
    }
    //============================================================================
    // Helpers
    //============================================================================
    computeFaviconUrl_(scaleFactor) {
        if (!this.match?.destinationUrl.url) {
            return '';
        }
        return getFaviconUrl(
        /* url= */ this.match.destinationUrl.url, {
            forceLightMode: !this.isTopChromeSearchbox_,
            forceEmptyDefaultFavicon: true,
            scaleFactor: `${scaleFactor}x`,
        });
    }
    computeFaviconImageSrcSet_() {
        if (!this.faviconImage_.startsWith('chrome://favicon2/')) {
            return '';
        }
        return [
            `${this.computeFaviconUrl_(/* scaleFactor= */ 1)} 1x`,
            `${this.computeFaviconUrl_(/* scaleFactor= */ 2)} 2x`,
        ].join(', ');
    }
    computeFaviconImage_() {
        if (this.match && !this.match.isSearchType) {
            if (this.match.type === DOCUMENT_MATCH_TYPE ||
                this.match.type === PEDAL ||
                this.match.isEnterpriseSearchAggregatorPeopleType) {
                return this.match.iconPath;
            }
            // Featured enterprise search suggestions have the icon set match.iconUrl.
            if (this.match.type !== HISTORY_CLUSTER_MATCH_TYPE &&
                this.match.type !== FEATURED_ENTERPRISE_SEARCH) {
                return this.computeFaviconUrl_(/* scaleFactor= */ 1);
            }
        }
        if (this.defaultIcon ===
            '//resources/cr_components/searchbox/icons/google_g.svg' ||
            this.defaultIcon ===
                '//resources/cr_components/searchbox/icons/google_g_gradient.svg') {
            // The google_g.svg is a fully colored icon, so it needs to be displayed
            // as a favicon image as mask images will mask the colors.
            return this.defaultIcon;
        }
        return '';
    }
    computeIsAnswer_() {
        return !!this.match && !!this.match.answer;
    }
    computeIsWeatherAnswer_() {
        return this.match?.isWeatherAnswerSuggestion || false;
    }
    computeHasImage_() {
        return !!this.match && !!this.match.imageUrl;
    }
    computeIsEnterpriseSearchAggregatorPeopleType_() {
        return this.match?.isEnterpriseSearchAggregatorPeopleType || false;
    }
    computeShowIconImg_() {
        // Lens searchbox should not use icon URL.
        return !this.isLensSearchbox_ && !!this.match && !!this.match.iconUrl.url &&
            !this.iconLoading_;
    }
    computeMaskImage_() {
        // Lens searchboxes should always have the Google G in the searchbox.
        if (this.isLensSearchbox_ && this.inSearchbox) {
            return `url(${this.defaultIcon})`;
        }
        // Enterprise search aggregator people and starter pack/featured enterprise
        // search suggestions should show icon even in searchbox.
        if (this.match &&
            (!this.match.isRichSuggestion || this.match.type === STARTER_PACK ||
                this.match.type === FEATURED_ENTERPRISE_SEARCH ||
                this.match.isEnterpriseSearchAggregatorPeopleType ||
                !this.inSearchbox)) {
            return `url(${this.match.iconPath})`;
        }
        else {
            return `url(${this.defaultIcon})`;
        }
    }
    // Controls whether the favicon image should be used instead of the mask
    // image.
    computeShowFaviconImage_() {
        if (!this.faviconImage_) {
            return false;
        }
        // If the favicon resource is still loading or there was an error, then
        // fall back to rendering the default vector icon (generic globe icon).
        if (this.faviconLoading_ || this.faviconError_) {
            return false;
        }
        // Navigation suggestions should always use the background image, except for
        // Lens searchboxes and pedal/starter pack suggestions, which prefer to use
        // the default icon in the mask image.
        if (!this.isLensSearchbox_ && this.match && !this.match.isSearchType &&
            this.match.type !== STARTER_PACK && this.match.type !== PEDAL) {
            return true;
        }
        // The following icons should not use the GM3 foreground color.
        // TODO(niharm): Refactor logic in C++ and send via mojom in
        // "chrome/browser/ui/webui/searchbox/searchbox_handler.cc".
        const themedIcons = [
            'calendar',
            'drive_docs',
            'drive_folder',
            'drive_form',
            'drive_image',
            'drive_logo',
            'drive_pdf',
            'drive_sheets',
            'drive_slides',
            'drive_video',
            'google_agentspace_logo',
            'google_agentspace_logo_25',
            'google_g',
            'google_g_gradient',
            'note',
            'sites',
        ];
        for (const icon of themedIcons) {
            if (this.faviconImage_ ===
                '//resources/cr_components/searchbox/icons/' + icon + '.svg') {
                return true;
            }
        }
        return false;
    }
    computeSrc_(url) {
        if (!url) {
            return '';
        }
        if (url.startsWith('data:image/')) {
            // Zero-prefix matches come with the data URI content in |url|.
            return url;
        }
        return `//image?staticEncode=true&encodeType=webp&url=${url}`;
    }
    computeIconSrc_() {
        return this.computeSrc_(this.match?.iconUrl?.url);
    }
    computeShowImage_() {
        return !!this.imageSrc_ && !this.imageError_;
    }
    computeImageSrc_() {
        return this.computeSrc_(this.match?.imageUrl);
    }
    getContainerBgColor_() {
        // If the match has an image dominant color, show that color in place of the
        // image until it loads. This helps the image appear to load more smoothly.
        return ((this.imageLoading_ || this.imageError_) &&
            this.match?.imageDominantColor) ?
            // .25 opacity matching c/b/u/views/omnibox/omnibox_match_cell_view.cc.
            (this.match.imageDominantColor ?
                `${this.match.imageDominantColor}40` :
                'var(--cr-searchbox-match-icon-container-background-fallback)') :
            'transparent';
    }
    onFaviconLoad_() {
        this.faviconLoading_ = false;
        this.faviconError_ = false;
    }
    onFaviconError_() {
        this.faviconLoading_ = false;
        this.faviconError_ = true;
    }
    onIconLoad_() {
        this.iconLoading_ = false;
    }
    onImageLoad_() {
        this.imageLoading_ = false;
        this.imageError_ = false;
    }
    onImageError_() {
        this.imageLoading_ = false;
        this.imageError_ = true;
    }
    // All pedals, starter pack/featured enterprise search suggestions, and AiS
    // except weather should have a colored background container that matches the
    // current theme.
    // TODO(niharm): Refactor logic in C++ and send via mojom in
    // "chrome/browser/ui/webui/searchbox/searchbox_handler.cc".
    computeHasIconContainerBackground_() {
        if (this.match) {
            return this.match.type === PEDAL ||
                this.match.type === HISTORY_CLUSTER_MATCH_TYPE ||
                this.match.type === CALCULATOR || this.match.type === STARTER_PACK ||
                this.match.type === FEATURED_ENTERPRISE_SEARCH ||
                (!!this.match.answer && !this.isWeatherAnswer);
        }
        return false;
    }
    computeIsStarterPack_() {
        return this.match?.type === STARTER_PACK;
    }
    computeIsFeaturedEnterpriseSearch() {
        return this.match?.type === FEATURED_ENTERPRISE_SEARCH;
    }
}
customElements.define(SearchboxIconElement.is, SearchboxIconElement);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Verify |value| is truthy.
 * @param value A value to check for truthiness. Note that this
 *     may be used to test whether |value| is defined or not, and we don't want
 *     to force a cast to boolean.
 */
function assert(value, message) {
    if (value) {
        return;
    }
    throw new Error('Assertion failed' + (message ? `: ${message}` : ''));
}
/**
 * Call this from places in the code that should never be reached.
 *
 * For example, handling all the values of enum with a switch() like this:
 *
 *   function getValueFromEnum(enum) {
 *     switch (enum) {
 *       case ENUM_FIRST_OF_TWO:
 *         return first
 *       case ENUM_LAST_OF_TWO:
 *         return last;
 *     }
 *     assertNotReached();
 *   }
 *
 * This code should only be hit in the case of serious programmer error or
 * unexpected input.
 */
function assertNotReached(message = 'Unreachable code hit') {
    assert(false, message);
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Make a string safe for Polymer bindings that are inner-h-t-m-l or other
 * innerHTML use.
 * @param rawString The unsanitized string
 * @param opts Optional additional allowed tags and attributes.
 */
function sanitizeInnerHtmlInternal(rawString, opts) {
    opts = opts || {};
    const html = parseHtmlSubset(`<b>${rawString}</b>`, opts.tags, opts.attrs)
        .firstElementChild;
    return html.innerHTML;
}
// 
let sanitizedPolicy = null;
/**
 * Same as |sanitizeInnerHtmlInternal|, but it passes through sanitizedPolicy
 * to create a TrustedHTML.
 */
function sanitizeInnerHtml(rawString, opts) {
    assert(window.trustedTypes);
    if (sanitizedPolicy === null) {
        // Initialize |sanitizedPolicy| lazily.
        sanitizedPolicy = window.trustedTypes.createPolicy('sanitize-inner-html', {
            createHTML: sanitizeInnerHtmlInternal,
            createScript: () => assertNotReached(),
            createScriptURL: () => assertNotReached(),
        });
    }
    return sanitizedPolicy.createHTML(rawString, opts);
}
const allowAttribute = (_node, _value) => true;
/** Allow-list of attributes in parseHtmlSubset. */
const allowedAttributes = new Map([
    [
        'href',
        (node, value) => {
            // Only allow a[href] starting with chrome:// or https:// or equaling
            // to #.
            return node.tagName === 'A' &&
                (value.startsWith('chrome://') || value.startsWith('https://') ||
                    value === '#');
        },
    ],
    [
        'target',
        (node, value) => {
            // Only allow a[target='_blank'].
            // TODO(dbeam): are there valid use cases for target !== '_blank'?
            return node.tagName === 'A' && value === '_blank';
        },
    ],
]);
/** Allow-list of optional attributes in parseHtmlSubset. */
const allowedOptionalAttributes = new Map([
    ['class', allowAttribute],
    ['id', allowAttribute],
    ['is', (_node, value) => value === 'action-link' || value === ''],
    ['role', (_node, value) => value === 'link'],
    [
        'src',
        (node, value) => {
            // Only allow img[src] starting with chrome://
            return node.tagName === 'IMG' &&
                value.startsWith('chrome://');
        },
    ],
    ['tabindex', allowAttribute],
    ['aria-description', allowAttribute],
    ['aria-hidden', allowAttribute],
    ['aria-label', allowAttribute],
    ['aria-labelledby', allowAttribute],
]);
/** Allow-list of tag names in parseHtmlSubset. */
const allowedTags = new Set(['A', 'B', 'I', 'BR', 'DIV', 'EM', 'KBD', 'P', 'PRE', 'SPAN', 'STRONG']);
/** Allow-list of optional tag names in parseHtmlSubset. */
const allowedOptionalTags = new Set(['IMG', 'LI', 'UL']);
/**
 * This policy maps a given string to a `TrustedHTML` object
 * without performing any validation. Callsites must ensure
 * that the resulting object will only be used in inert
 * documents. Initialized lazily.
 */
let unsanitizedPolicy;
/**
 * @param optTags an Array to merge.
 * @return Set of allowed tags.
 */
function mergeTags(optTags) {
    const clone = new Set(allowedTags);
    optTags.forEach(str => {
        const tag = str.toUpperCase();
        if (allowedOptionalTags.has(tag)) {
            clone.add(tag);
        }
    });
    return clone;
}
/**
 * @param optAttrs an Array to merge.
 * @return Map of allowed attributes.
 */
function mergeAttrs(optAttrs) {
    const clone = new Map(allowedAttributes);
    optAttrs.forEach(key => {
        if (allowedOptionalAttributes.has(key)) {
            clone.set(key, allowedOptionalAttributes.get(key));
        }
    });
    return clone;
}
function walk(n, f) {
    f(n);
    for (let i = 0; i < n.childNodes.length; i++) {
        walk(n.childNodes[i], f);
    }
}
function assertElement(tags, node) {
    if (!tags.has(node.tagName)) {
        throw Error(node.tagName + ' is not supported');
    }
}
function assertAttribute(attrs, attrNode, node) {
    const n = attrNode.nodeName;
    const v = attrNode.nodeValue || '';
    if (!attrs.has(n) || !attrs.get(n)(node, v)) {
        throw Error(node.tagName + '[' + n + '="' + v +
            '"] is not supported');
    }
}
/**
 * Parses a very small subset of HTML. This ensures that insecure HTML /
 * javascript cannot be injected into WebUI.
 * @param s The string to parse.
 * @param extraTags Optional extra allowed tags.
 * @param extraAttrs
 *     Optional extra allowed attributes (all tags are run through these).
 * @throws an Error in case of non supported markup.
 * @return A document fragment containing the DOM tree.
 */
function parseHtmlSubset(s, extraTags, extraAttrs) {
    const tags = extraTags ? mergeTags(extraTags) : allowedTags;
    const attrs = extraAttrs ? mergeAttrs(extraAttrs) : allowedAttributes;
    const doc = document.implementation.createHTMLDocument('');
    const r = doc.createRange();
    r.selectNode(doc.body);
    if (window.trustedTypes) {
        if (!unsanitizedPolicy) {
            unsanitizedPolicy =
                window.trustedTypes.createPolicy('parse-html-subset', {
                    createHTML: (untrustedHTML) => untrustedHTML,
                    createScript: () => assertNotReached(),
                    createScriptURL: () => assertNotReached(),
                });
        }
        s = unsanitizedPolicy.createHTML(s);
    }
    // This does not execute any scripts because the document has no view.
    const df = r.createContextualFragment(s);
    walk(df, function (node) {
        switch (node.nodeType) {
            case Node.ELEMENT_NODE:
                assertElement(tags, node);
                const nodeAttrs = node.attributes;
                for (let i = 0; i < nodeAttrs.length; ++i) {
                    assertAttribute(attrs, nodeAttrs[i], node);
                }
                break;
            case Node.COMMENT_NODE:
            case Node.DOCUMENT_FRAGMENT_NODE:
            case Node.TEXT_NODE:
                break;
            default:
                throw Error('Node type ' + node.nodeType + ' is not supported');
        }
    });
    return df;
}

const sheet = new CSSStyleSheet();
sheet.replaceSync(`html{--google-blue-50-rgb:232,240,254;--google-blue-50:rgb(var(--google-blue-50-rgb));--google-blue-100-rgb:210,227,252;--google-blue-100:rgb(var(--google-blue-100-rgb));--google-blue-200-rgb:174,203,250;--google-blue-200:rgb(var(--google-blue-200-rgb));--google-blue-300-rgb:138,180,248;--google-blue-300:rgb(var(--google-blue-300-rgb));--google-blue-400-rgb:102,157,246;--google-blue-400:rgb(var(--google-blue-400-rgb));--google-blue-500-rgb:66,133,244;--google-blue-500:rgb(var(--google-blue-500-rgb));--google-blue-600-rgb:26,115,232;--google-blue-600:rgb(var(--google-blue-600-rgb));--google-blue-700-rgb:25,103,210;--google-blue-700:rgb(var(--google-blue-700-rgb));--google-blue-800-rgb:24,90,188;--google-blue-800:rgb(var(--google-blue-800-rgb));--google-blue-900-rgb:23,78,166;--google-blue-900:rgb(var(--google-blue-900-rgb));--google-green-50-rgb:230,244,234;--google-green-50:rgb(var(--google-green-50-rgb));--google-green-200-rgb:168,218,181;--google-green-200:rgb(var(--google-green-200-rgb));--google-green-300-rgb:129,201,149;--google-green-300:rgb(var(--google-green-300-rgb));--google-green-400-rgb:91,185,116;--google-green-400:rgb(var(--google-green-400-rgb));--google-green-500-rgb:52,168,83;--google-green-500:rgb(var(--google-green-500-rgb));--google-green-600-rgb:30,142,62;--google-green-600:rgb(var(--google-green-600-rgb));--google-green-700-rgb:24,128,56;--google-green-700:rgb(var(--google-green-700-rgb));--google-green-800-rgb:19,115,51;--google-green-800:rgb(var(--google-green-800-rgb));--google-green-900-rgb:13,101,45;--google-green-900:rgb(var(--google-green-900-rgb));--google-grey-50-rgb:248,249,250;--google-grey-50:rgb(var(--google-grey-50-rgb));--google-grey-100-rgb:241,243,244;--google-grey-100:rgb(var(--google-grey-100-rgb));--google-grey-200-rgb:232,234,237;--google-grey-200:rgb(var(--google-grey-200-rgb));--google-grey-300-rgb:218,220,224;--google-grey-300:rgb(var(--google-grey-300-rgb));--google-grey-400-rgb:189,193,198;--google-grey-400:rgb(var(--google-grey-400-rgb));--google-grey-500-rgb:154,160,166;--google-grey-500:rgb(var(--google-grey-500-rgb));--google-grey-600-rgb:128,134,139;--google-grey-600:rgb(var(--google-grey-600-rgb));--google-grey-700-rgb:95,99,104;--google-grey-700:rgb(var(--google-grey-700-rgb));--google-grey-800-rgb:60,64,67;--google-grey-800:rgb(var(--google-grey-800-rgb));--google-grey-900-rgb:32,33,36;--google-grey-900:rgb(var(--google-grey-900-rgb));--google-grey-900-white-4-percent:#292a2d;--google-purple-200-rgb:215,174,251;--google-purple-200:rgb(var(--google-purple-200-rgb));--google-purple-900-rgb:104,29,168;--google-purple-900:rgb(var(--google-purple-900-rgb));--google-red-100-rgb:244,199,195;--google-red-100:rgb(var(--google-red-100-rgb));--google-red-300-rgb:242,139,130;--google-red-300:rgb(var(--google-red-300-rgb));--google-red-500-rgb:234,67,53;--google-red-500:rgb(var(--google-red-500-rgb));--google-red-600-rgb:217,48,37;--google-red-600:rgb(var(--google-red-600-rgb));--google-red-700-rgb:197,57,41;--google-red-700:rgb(var(--google-red-700-rgb));--google-yellow-50-rgb:254,247,224;--google-yellow-50:rgb(var(--google-yellow-50-rgb));--google-yellow-100-rgb:254,239,195;--google-yellow-100:rgb(var(--google-yellow-100-rgb));--google-yellow-200-rgb:253,226,147;--google-yellow-200:rgb(var(--google-yellow-200-rgb));--google-yellow-300-rgb:253,214,51;--google-yellow-300:rgb(var(--google-yellow-300-rgb));--google-yellow-400-rgb:252,201,52;--google-yellow-400:rgb(var(--google-yellow-400-rgb));--google-yellow-500-rgb:251,188,4;--google-yellow-500:rgb(var(--google-yellow-500-rgb));--google-yellow-700-rgb:240,147,0;--google-yellow-700:rgb(var(--google-yellow-700-rgb));--cr-card-background-color:white;--cr-shadow-key-color_:color-mix(in srgb,var(--cr-shadow-color) 30%,transparent);--cr-shadow-ambient-color_:color-mix(in srgb,var(--cr-shadow-color) 15%,transparent);--cr-elevation-1:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 1px 3px 1px;--cr-elevation-2:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 2px 6px 2px;--cr-elevation-3:var(--cr-shadow-key-color_) 0 1px 3px 0,var(--cr-shadow-ambient-color_) 0 4px 8px 3px;--cr-elevation-4:var(--cr-shadow-key-color_) 0 2px 3px 0,var(--cr-shadow-ambient-color_) 0 6px 10px 4px;--cr-elevation-5:var(--cr-shadow-key-color_) 0 4px 4px 0,var(--cr-shadow-ambient-color_) 0 8px 12px 6px;--cr-card-shadow:var(--cr-elevation-2);--cr-focused-item-color:var(--google-grey-300);--cr-form-field-label-color:var(--google-grey-700);--cr-hairline-rgb:0,0,0;--cr-iph-anchor-highlight-color:rgba(var(--google-blue-600-rgb),0.1);--cr-menu-background-color:white;--cr-menu-background-focus-color:var(--google-grey-400);--cr-menu-shadow:var(--cr-elevation-2);--cr-separator-color:rgba(0,0,0,.06);--cr-title-text-color:rgb(90,90,90);--cr-scrollable-border-color:var(--google-grey-300)}@media (prefers-color-scheme:dark){html{--cr-card-background-color:var(--google-grey-900-white-4-percent);--cr-focused-item-color:var(--google-grey-800);--cr-form-field-label-color:var(--dark-secondary-color);--cr-hairline-rgb:255,255,255;--cr-iph-anchor-highlight-color:rgba(var(--google-grey-100-rgb),0.1);--cr-menu-background-color:var(--google-grey-900);--cr-menu-background-focus-color:var(--google-grey-700);--cr-menu-background-sheen:rgba(255,255,255,.06);--cr-menu-shadow:rgba(0,0,0,.3) 0 1px 2px 0,rgba(0,0,0,.15) 0 3px 6px 2px;--cr-separator-color:rgba(255,255,255,.1);--cr-title-text-color:var(--cr-primary-text-color);--cr-scrollable-border-color:var(--google-grey-700)}}@media (forced-colors:active){html{--cr-focus-outline-hcm:2px solid transparent;--cr-border-hcm:2px solid transparent}}html{--cr-button-edge-spacing:12px;--cr-controlled-by-spacing:24px;--cr-default-input-max-width:264px;--cr-icon-ripple-size:36px;--cr-icon-ripple-padding:8px;--cr-icon-size:20px;--cr-icon-button-margin-start:16px;--cr-icon-ripple-margin:calc(var(--cr-icon-ripple-padding) * -1);--cr-section-min-height:48px;--cr-section-two-line-min-height:64px;--cr-section-padding:20px;--cr-section-vertical-padding:12px;--cr-section-indent-width:40px;--cr-section-indent-padding:calc(var(--cr-section-padding) + var(--cr-section-indent-width));--cr-section-vertical-margin:21px;--cr-centered-card-max-width:680px;--cr-centered-card-width-percentage:0.96;--cr-hairline:1px solid rgba(var(--cr-hairline-rgb),.14);--cr-separator-height:1px;--cr-separator-line:var(--cr-separator-height) solid var(--cr-separator-color);--cr-toolbar-overlay-animation-duration:150ms;--cr-toolbar-height:56px;--cr-container-shadow-height:6px;--cr-container-shadow-margin:calc(-1 * var(--cr-container-shadow-height));--cr-container-shadow-max-opacity:1;--cr-card-border-radius:8px;--cr-disabled-opacity:.38;--cr-form-field-bottom-spacing:16px;--cr-form-field-label-font-size:.625rem;--cr-form-field-label-height:1em;--cr-form-field-label-line-height:1}html{--cr-fallback-color-outline:rgb(116,119,117);--cr-fallback-color-primary:rgb(11,87,208);--cr-fallback-color-on-primary:rgb(255,255,255);--cr-fallback-color-primary-container:rgb(211,227,253);--cr-fallback-color-on-primary-container:rgb(4,30,73);--cr-fallback-color-secondary-container:rgb(194,231,255);--cr-fallback-color-on-secondary-container:rgb(0,29,53);--cr-fallback-color-neutral-container:rgb(242,242,242);--cr-fallback-color-neutral-outline:rgb(199,199,199);--cr-fallback-color-surface:rgb(255,255,255);--cr-fallback-color-surface1:rgb(248,250,253);--cr-fallback-color-surface2:rgb(243,246,252);--cr-fallback-color-surface3:rgb(239,243,250);--cr-fallback-color-on-surface-rgb:31,31,31;--cr-fallback-color-on-surface:rgb(var(--cr-fallback-color-on-surface-rgb));--cr-fallback-color-surface-variant:rgb(225,227,225);--cr-fallback-color-on-surface-variant:rgb(138,141,140);--cr-fallback-color-on-surface-subtle:rgb(71,71,71);--cr-fallback-color-inverse-primary:rgb(168,199,250);--cr-fallback-color-inverse-surface:rgb(48,48,48);--cr-fallback-color-inverse-on-surface:rgb(242,242,242);--cr-fallback-color-tonal-container:rgb(211,227,253);--cr-fallback-color-on-tonal-container:rgb(4,30,73);--cr-fallback-color-tonal-outline:rgb(168,199,250);--cr-fallback-color-error:rgb(179,38,30);--cr-fallback-color-divider:rgb(211,227,253);--cr-fallback-color-state-hover-on-prominent_:rgba(253,252,251,.1);--cr-fallback-color-state-on-subtle-rgb_:31,31,31;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.06);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.08);--cr-fallback-color-state-ripple-primary-rgb_:124,172,248;--cr-fallback-color-state-ripple-primary_:rgba(var(--cr-fallback-color-state-ripple-primary-rgb_),0.32);--cr-fallback-color-base-container:rgb(236,239,247);--cr-fallback-color-disabled-background:rgba(var(--cr-fallback-color-on-surface-rgb),.12);--cr-fallback-color-disabled-foreground:rgba(var(--cr-fallback-color-on-surface-rgb),var(--cr-disabled-opacity));--cr-hover-background-color:var(--color-sys-state-hover,rgba(var(--cr-fallback-color-on-surface-rgb),.08));--cr-hover-on-prominent-background-color:var(--color-sys-state-hover-on-prominent,var(--cr-fallback-color-state-hover-on-prominent_));--cr-hover-on-subtle-background-color:var(--color-sys-state-hover-on-subtle,var(--cr-fallback-color-state-hover-on-subtle_));--cr-active-background-color:var(--color-sys-state-pressed,rgba(var(--cr-fallback-color-on-surface-rgb),.12));--cr-active-on-primary-background-color:var(--color-sys-state-ripple-primary,var(--cr-fallback-color-state-ripple-primary_));--cr-active-neutral-on-subtle-background-color:var(--color-sys-state-ripple-neutral-on-subtle,var(--cr-fallback-color-state-ripple-neutral-on-subtle_));--cr-focus-outline-color:var(--color-sys-state-focus-ring,var(--owl-control-accent-color,var(--cr-fallback-color-primary)));--cr-focus-outline-inverse-color:var(--color-sys-state-focus-ring-inverse,var(--cr-fallback-color-inverse-primary));--cr-primary-text-color:var(--color-primary-foreground,var(--cr-fallback-color-on-surface));--cr-secondary-text-color:var(--color-secondary-foreground,var(--cr-fallback-color-on-surface-variant));--cr-link-color:var(--color-link-foreground-default,var(--cr-fallback-color-primary));--cr-button-height:36px;--cr-shadow-color:var(--color-sys-shadow,rgb(0,0,0));--cr-checked-color:var(--color-checkbox-foreground-checked,var(--owl-control-accent-color,var(--cr-fallback-color-primary)))}@media (prefers-color-scheme:dark){html{--cr-fallback-color-outline:rgb(142,145,143);--cr-fallback-color-primary:rgb(168,199,250);--cr-fallback-color-on-primary:rgb(6,46,111);--cr-fallback-color-primary-container:rgb(8,66,160);--cr-fallback-color-on-primary-container:rgb(211,227,253);--cr-fallback-color-secondary-container:rgb(0,74,119);--cr-fallback-color-on-secondary-container:rgb(194,231,255);--cr-fallback-color-neutral-container:rgb(40,40,40);--cr-fallback-color-neutral-outline:rgb(117,117,117);--cr-fallback-color-surface:rgb(31,31,31);--cr-fallback-color-surface1:rgb(39,40,42);--cr-fallback-color-surface2:rgb(45,47,49);--cr-fallback-color-surface3:rgb(51,52,56);--cr-fallback-color-on-surface-rgb:227,227,227;--cr-fallback-color-surface-variant:rgb(68,71,70);--cr-fallback-color-on-surface-variant:rgb(158,161,160);--cr-fallback-color-on-surface-subtle:rgb(199,199,199);--cr-fallback-color-inverse-primary:rgb(11,87,208);--cr-fallback-color-inverse-surface:rgb(227,227,227);--cr-fallback-color-inverse-on-surface:rgb(31,31,31);--cr-fallback-color-tonal-container:rgb(0,74,119);--cr-fallback-color-on-tonal-container:rgb(194,231,255);--cr-fallback-color-tonal-outline:rgb(4,125,183);--cr-fallback-color-error:rgb(242,184,181);--cr-fallback-color-divider:rgb(94,94,94);--cr-fallback-color-state-hover-on-prominent_:rgba(31,31,31,.06);--cr-fallback-color-state-on-subtle-rgb_:253,252,251;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.10);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.16);--cr-fallback-color-state-ripple-primary-rgb_:76,141,246;--cr-fallback-color-base-container:rgba(40,40,40,1)}}@media (forced-colors:active){html{--cr-fallback-color-disabled-background:Canvas;--cr-fallback-color-disabled-foreground:GrayText}}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet];

let instance$d = null;
function getCss$9() {
    return instance$d || (instance$d = [...[], css `.icon-arrow-back{--cr-icon-image:url(//resources/images/icon_arrow_back.svg)}.icon-arrow-dropdown{--cr-icon-image:url(//resources/images/icon_arrow_dropdown.svg)}.icon-arrow-drop-down-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_down_cr23.svg)}.icon-arrow-drop-up-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_up_cr23.svg)}.icon-arrow-upward{--cr-icon-image:url(//resources/images/icon_arrow_upward.svg)}.icon-cancel{--cr-icon-image:url(//resources/images/icon_cancel.svg)}.icon-clear{--cr-icon-image:url(//resources/images/icon_clear.svg)}.icon-copy-content{--cr-icon-image:url(//resources/images/icon_copy_content.svg)}.icon-delete-gray{--cr-icon-image:url(//resources/images/icon_delete_gray.svg)}.icon-edit{--cr-icon-image:url(//resources/images/icon_edit.svg)}.icon-file{--cr-icon-image:url(//resources/images/icon_filetype_generic.svg)}.icon-folder-open{--cr-icon-image:url(//resources/images/icon_folder_open.svg)}.icon-picture-delete{--cr-icon-image:url(//resources/images/icon_picture_delete.svg)}.icon-expand-less{--cr-icon-image:url(//resources/images/icon_expand_less.svg)}.icon-expand-more{--cr-icon-image:url(//resources/images/icon_expand_more.svg)}.icon-external{--cr-icon-image:url(//resources/images/open_in_new.svg)}.icon-more-vert{--cr-icon-image:url(//resources/images/icon_more_vert.svg)}.icon-refresh{--cr-icon-image:url(//resources/images/icon_refresh.svg)}.icon-search{--cr-icon-image:url(//resources/images/icon_search.svg)}.icon-settings{--cr-icon-image:url(//resources/images/icon_settings.svg)}.icon-visibility{--cr-icon-image:url(//resources/images/icon_visibility.svg)}.icon-visibility-off{--cr-icon-image:url(//resources/images/icon_visibility_off.svg)}.subpage-arrow{--cr-icon-image:url(//resources/images/arrow_right.svg)}.cr-icon{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-size);background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-700)));flex-shrink:0;height:var(--cr-icon-ripple-size);margin-inline-end:var(--cr-icon-ripple-margin);margin-inline-start:var(--cr-icon-button-margin-start);user-select:none;width:var(--cr-icon-ripple-size)}:host-context([dir=rtl]) .cr-icon{transform:scaleX(-1)}.cr-icon.no-overlap{margin-inline-end:0;margin-inline-start:0}@media (prefers-color-scheme:dark){.cr-icon{background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-500)))}}`]);
}

let instance$c = null;
function getCss$8() {
    return instance$c || (instance$c = [...[getCss$b(), getCss$9()], css `[actionable]{cursor:pointer}.hr{border-top:var(--cr-separator-line)}iron-list.cr-separators>*:not([first]){border-top:var(--cr-separator-line)}[scrollable]{border-color:transparent;border-style:solid;border-width:1px 0;overflow-y:auto}[scrollable].is-scrolled{border-top-color:var(--cr-scrollable-border-color)}[scrollable].can-scroll:not(.scrolled-to-bottom){border-bottom-color:var(--cr-scrollable-border-color)}[scrollable] iron-list>:not(.no-outline):focus-visible,[selectable]:focus-visible,[selectable]>:focus-visible{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}.scroll-container{display:flex;flex-direction:column;min-height:1px}[selectable]>*{cursor:pointer}.cr-centered-card-container{box-sizing:border-box;display:block;height:inherit;margin:0 auto;max-width:var(--cr-centered-card-max-width);min-width:550px;position:relative;width:calc(100% * var(--cr-centered-card-width-percentage))}.cr-container-shadow{height:var(--cr-container-shadow-height);left:0;margin:0 0 var(--cr-container-shadow-margin);opacity:0;pointer-events:none;position:relative;right:0;top:0;transition:opacity 500ms;z-index:1}#cr-container-shadow-bottom{margin-bottom:0;margin-top:var(--cr-container-shadow-margin);transform:scaleY(-1)}#cr-container-shadow-top:has(+#container.can-scroll:not(.scrolled-to-top)),#container.can-scroll:not(.scrolled-to-bottom)+#cr-container-shadow-bottom,#cr-container-shadow-bottom.force-shadow,#cr-container-shadow-top.force-shadow{opacity:var(--cr-container-shadow-max-opacity)}.cr-row{align-items:center;border-top:var(--cr-separator-line);display:flex;min-height:var(--cr-section-min-height);padding:0 var(--cr-section-padding)}.cr-row.first,.cr-row.continuation{border-top:none}.cr-row-gap{padding-inline-start:16px}.cr-button-gap{margin-inline-start:8px}paper-tooltip::part(tooltip),cr-tooltip::part(tooltip){border-radius:var(--paper-tooltip-border-radius,2px);font-size:92.31%;font-weight:500;max-width:330px;min-width:var(--paper-tooltip-min-width,200px);padding:var(--paper-tooltip-padding,10px 8px)}.cr-padded-text{padding-block-end:var(--cr-section-vertical-padding);padding-block-start:var(--cr-section-vertical-padding)}.cr-title-text{color:var(--cr-title-text-color);font-size:107.6923%;font-weight:500}.cr-secondary-text{color:var(--cr-secondary-text-color);font-weight:400}.cr-form-field-label{color:var(--cr-form-field-label-color);display:block;font-size:var(--cr-form-field-label-font-size);font-weight:500;letter-spacing:.4px;line-height:var(--cr-form-field-label-line-height);margin-bottom:8px}.cr-vertical-tab{align-items:center;display:flex}.cr-vertical-tab::before{border-radius:0 3px 3px 0;content:'';display:block;flex-shrink:0;height:var(--cr-vertical-tab-height,100%);width:4px}.cr-vertical-tab.selected::before{background:var(--cr-vertical-tab-selected-color,var(--cr-checked-color))}:host-context([dir=rtl]) .cr-vertical-tab::before{transform:scaleX(-1)}.iph-anchor-highlight{background-color:var(--cr-iph-anchor-highlight-color)}`]);
}

let instance$b = null;
function getCss$7() {
    return instance$b || (instance$b = [...[getCss$8()], css `:host{border:solid 1px var(--color-searchbox-results-action-chip);border-radius:8px;display:flex;height:var(--cr-searchbox-results-action-chip-height,28px);min-width:0;outline:none;padding-inline-end:8px;padding-inline-start:8px;position:relative}:host(:hover){background-color:var(--color-searchbox-results-button-hover)}:host(:focus),:host(.selected){margin:2px;box-shadow:none}:host(.selected:hover){background-color:var(--color-searchbox-results-button-selected-hover)}:host(:active) #overlay{background-color:var(--color-omnibox-results-button-ink-drop-selected-row-hovered)}:host(.selected:active) #overlay{background-color:var(--color-omnibox-results-button-ink-drop-selected-row-selected)}#overlay{--overlay-inset:calc(var(--border-width) * -1);border-radius:inherit;display:inherit;position:absolute;top:var(--overlay-inset);left:var(--overlay-inset);right:var(--overlay-inset);bottom:var(--overlay-inset)}.contents{align-items:center;display:flex;min-width:0}#action-icon{flex-shrink:0;-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:15px;background-color:var(--color-searchbox-results-action-chip-icon);background-position:center center;background-repeat:no-repeat;height:16px;width:16px}:host-context(:is(:focus,[selected])) #action-icon{background-color:var(--color-searchbox-results-action-chip-icon-selected,var(--color-searchbox-results-action-chip-icon))}#text{overflow:hidden;padding-inline-start:8px;text-overflow:ellipsis;white-space:nowrap}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$4() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="overlay"></div>
<div class="contents" title="${this.suggestionContents}">
  <div id="action-icon" style="${this.iconStyle_}"></div>
  <div id="text" .innerHTML="${this.hintHtml_}"></div>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Displays a UI chip for an `AutocompleteMatch`. E.g. for keywords ('Search
// YouTube') or actions ('Clear browsing data').
class SearchboxActionElement extends CrLitElement {
    static get is() {
        return 'cr-searchbox-action';
    }
    static get styles() {
        return getCss$7();
    }
    render() {
        return getHtml$4.bind(this)();
    }
    static get properties() {
        return {
            hint: { type: String },
            hintHtml_: {
                state: true,
                type: String,
            },
            suggestionContents: { type: String },
            iconPath: { type: String },
            iconStyle_: {
                state: true,
                type: String,
            },
            ariaLabel: { type: String },
            // Index of the action in the autocomplete result. Used to inform handler
            // of action that was selected.
            actionIndex: { type: Number },
        };
    }
    #hint_accessor_storage = '';
    get hint() { return this.#hint_accessor_storage; }
    set hint(value) { this.#hint_accessor_storage = value; }
    #hintHtml__accessor_storage = window.trustedTypes.emptyHTML;
    get hintHtml_() { return this.#hintHtml__accessor_storage; }
    set hintHtml_(value) { this.#hintHtml__accessor_storage = value; }
    #suggestionContents_accessor_storage = '';
    get suggestionContents() { return this.#suggestionContents_accessor_storage; }
    set suggestionContents(value) { this.#suggestionContents_accessor_storage = value; }
    #iconPath_accessor_storage = '';
    get iconPath() { return this.#iconPath_accessor_storage; }
    set iconPath(value) { this.#iconPath_accessor_storage = value; }
    #iconStyle__accessor_storage = '';
    get iconStyle_() { return this.#iconStyle__accessor_storage; }
    set iconStyle_(value) { this.#iconStyle__accessor_storage = value; }
    #ariaLabel_accessor_storage = '';
    get ariaLabel() { return this.#ariaLabel_accessor_storage; }
    set ariaLabel(value) { this.#ariaLabel_accessor_storage = value; }
    #actionIndex_accessor_storage = -1;
    get actionIndex() { return this.#actionIndex_accessor_storage; }
    set actionIndex(value) { this.#actionIndex_accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('click', (event) => this.onActionClick_(event));
        this.addEventListener('keydown', (event) => this.onActionKeyDown_(event));
        this.addEventListener('mousedown', (event) => this.onActionMouseDown_(event));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('hint')) {
            this.hintHtml_ = this.computeHintHtml_();
        }
        if (changedProperties.has('iconPath')) {
            this.iconStyle_ = this.computeActionIconStyle_();
        }
    }
    onActionClick_(e) {
        this.fire('execute-action', {
            event: e,
            actionIndex: this.actionIndex,
        });
        e.preventDefault(); // Prevents default browser action (navigation).
        e.stopPropagation(); // Prevents <iron-selector> from selecting the match.
    }
    onActionKeyDown_(e) {
        if (e.key && (e.key === 'Enter' || e.key === ' ')) {
            this.onActionClick_(e);
        }
    }
    onActionMouseDown_(e) {
        e.preventDefault(); // Prevents default browser action (focus).
    }
    //============================================================================
    // Helpers
    //============================================================================
    computeHintHtml_() {
        if (this.hint) {
            return sanitizeInnerHtml(this.hint);
        }
        return window.trustedTypes.emptyHTML;
    }
    computeActionIconStyle_() {
        // If this is a custom icon, shouldn't follow the standard theming given to
        // all other action icons.
        if (this.iconPath.startsWith('data:image/')) {
            return `background-image: url(${this.iconPath})`;
        }
        return `-webkit-mask-image: url(${this.iconPath})`;
    }
}
customElements.define(SearchboxActionElement.is, SearchboxActionElement);

let instance$a = null;
function getCss$6() {
    return instance$a || (instance$a = [...[getCss$b()], css `:host{align-items:center;display:inline-flex;justify-content:center;position:relative;vertical-align:middle;fill:var(--iron-icon-fill-color,currentcolor);stroke:var(--iron-icon-stroke-color,none);width:var(--iron-icon-width,24px);height:var(--iron-icon-height,24px)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let iconsetMap = null;
class IconsetMap extends EventTarget {
    iconsets_ = new Map();
    static getInstance() {
        return iconsetMap || (iconsetMap = new IconsetMap());
    }
    static resetInstanceForTesting(instance) {
        iconsetMap = instance;
    }
    get(id) {
        return this.iconsets_.get(id) || null;
    }
    set(id, iconset) {
        assert(!this.iconsets_.has(id), `Tried to add a second iconset with id '${id}'`);
        this.iconsets_.set(id, iconset);
        this.dispatchEvent(new CustomEvent('cr-iconset-added', { detail: id }));
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrIconElement extends CrLitElement {
    static get is() {
        return 'cr-icon';
    }
    static get styles() {
        return getCss$6();
    }
    static get properties() {
        return {
            /**
             * The name of the icon to use. The name should be of the form:
             * `iconset_name:icon_name`.
             */
            icon: { type: String },
        };
    }
    #icon_accessor_storage = '';
    get icon() { return this.#icon_accessor_storage; }
    set icon(value) { this.#icon_accessor_storage = value; }
    iconsetName_ = '';
    iconName_ = '';
    iconset_ = null;
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('icon')) {
            const [iconsetName, iconName] = this.icon.split(':');
            this.iconName_ = iconName || '';
            this.iconsetName_ = iconsetName || '';
            this.updateIcon_();
        }
    }
    updateIcon_() {
        if (this.iconName_ === '' && this.iconset_) {
            this.iconset_.removeIcon(this);
        }
        else if (this.iconsetName_) {
            const iconsetMap = IconsetMap.getInstance();
            this.iconset_ = iconsetMap.get(this.iconsetName_);
            assert(this.iconset_, `Could not find iconset for: '${this.iconsetName_}:${this.iconName_}'`);
            this.iconset_.applyIcon(this, this.iconName_);
        }
    }
}
customElements.define(CrIconElement.is, CrIconElement);

// Copyright 2011 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview EventTracker is a simple class that manages the addition and
 * removal of DOM event listeners. In particular, it keeps track of all
 * listeners that have been added and makes it easy to remove some or all of
 * them without requiring all the information again. This is particularly handy
 * when the listener is a generated function such as a lambda or the result of
 * calling Function.bind.
 */
class EventTracker {
    listeners_ = [];
    /**
     * Add an event listener - replacement for EventTarget.addEventListener.
     * @param target The DOM target to add a listener to.
     * @param eventType The type of event to subscribe to.
     * @param listener The listener to add.
     * @param capture Whether to invoke during the capture phase. Defaults to
     *     false.
     */
    add(target, eventType, listener, capture = false) {
        const h = {
            target: target,
            eventType: eventType,
            listener: listener,
            capture: capture,
        };
        this.listeners_.push(h);
        target.addEventListener(eventType, listener, capture);
    }
    /**
     * Remove any specified event listeners added with this EventTracker.
     * @param target The DOM target to remove a listener from.
     * @param eventType The type of event to remove.
     */
    remove(target, eventType) {
        this.listeners_ = this.listeners_.filter(listener => {
            if (listener.target === target &&
                (!eventType || (listener.eventType === eventType))) {
                EventTracker.removeEventListener(listener);
                return false;
            }
            return true;
        });
    }
    /** Remove all event listeners added with this EventTracker. */
    removeAll() {
        this.listeners_.forEach(listener => EventTracker.removeEventListener(listener));
        this.listeners_ = [];
    }
    /**
     * Remove a single event listener given it's tracking entry. It's up to the
     * caller to ensure the entry is removed from listeners_.
     * @param entry The entry describing the listener to
     * remove.
     */
    static removeEventListener(entry) {
        entry.target.removeEventListener(entry.eventType, entry.listener, entry.capture);
    }
}

let instance$9 = null;
function getCss$5() {
    return instance$9 || (instance$9 = [...[], css `:host{bottom:0;display:block;left:0;overflow:hidden;pointer-events:none;position:absolute;right:0;top:0;transform:translate3d(0,0,0)}.ripple{background-color:currentcolor;left:0;opacity:var(--paper-ripple-opacity,0.25);pointer-events:none;position:absolute;will-change:height,transform,width}.ripple,:host(.circle){border-radius:50%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrRippleElement extends CrLitElement {
    static get is() {
        return 'cr-ripple';
    }
    static get styles() {
        return getCss$5();
    }
    static get properties() {
        return {
            holdDown: { type: Boolean },
            recenters: { type: Boolean },
            noink: { type: Boolean },
        };
    }
    #holdDown_accessor_storage = false;
    get holdDown() { return this.#holdDown_accessor_storage; }
    set holdDown(value) { this.#holdDown_accessor_storage = value; }
    #recenters_accessor_storage = false;
    get recenters() { return this.#recenters_accessor_storage; }
    set recenters(value) { this.#recenters_accessor_storage = value; }
    #noink_accessor_storage = false;
    get noink() { return this.#noink_accessor_storage; }
    set noink(value) { this.#noink_accessor_storage = value; }
    ripples_ = [];
    eventTracker_ = new EventTracker();
    connectedCallback() {
        super.connectedCallback();
        assert(this.parentNode);
        const keyEventTarget = this.parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ?
            this.parentNode.host :
            this.parentElement;
        this.eventTracker_.add(keyEventTarget, 'pointerdown', (e) => this.uiDownAction(e));
        this.eventTracker_.add(keyEventTarget, 'pointerup', () => this.uiUpAction());
        // 'pointerup' does not fire if the pointer is moved outside the bounds of
        // `keyEventTarget` before releasing, so also listen for `pointerout`.
        this.eventTracker_.add(keyEventTarget, 'pointerout', () => this.uiUpAction());
        this.eventTracker_.add(keyEventTarget, 'keydown', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === 'Enter') {
                this.onEnterKeydown_();
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeydown_();
            }
        });
        this.eventTracker_.add(keyEventTarget, 'keyup', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeyup_();
            }
        });
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('holdDown')) {
            this.holdDownChanged_(this.holdDown, changedProperties.get('holdDown'));
        }
    }
    uiDownAction(e) {
        if (e !== undefined && e.button !== 0) {
            // Ignore secondary mouse button clicks.
            return;
        }
        if (!this.noink) {
            this.downAction_(e);
        }
    }
    downAction_(e) {
        if (this.ripples_.length && this.holdDown) {
            return;
        }
        this.showRipple_(e);
    }
    clear() {
        this.hideRipple_();
        this.holdDown = false;
    }
    showAndHoldDown() {
        this.ripples_.forEach(ripple => {
            ripple.remove();
        });
        this.ripples_ = [];
        this.holdDown = true;
    }
    showRipple_(e) {
        // OWL: We don't want ripples.
        return;
    }
    uiUpAction() {
        if (!this.noink) {
            this.upAction_();
        }
    }
    upAction_() {
        if (!this.holdDown) {
            this.hideRipple_();
        }
    }
    hideRipple_() {
        if (this.ripples_.length === 0) {
            return;
        }
        this.ripples_.forEach(function (ripple) {
            const opacity = ripple.computedStyleMap().get('opacity');
            if (opacity === null) {
                ripple.remove();
                return;
            }
            const animation = ripple.animate({
                opacity: [opacity.value, 0],
            }, {
                duration: 150,
                fill: 'forwards',
            });
            animation.finished.then(() => {
                ripple.remove();
            });
        });
        this.ripples_ = [];
    }
    onEnterKeydown_() {
        this.uiDownAction();
        window.setTimeout(() => {
            this.uiUpAction();
        }, 1);
    }
    onSpaceKeydown_() {
        this.uiDownAction();
    }
    onSpaceKeyup_() {
        this.uiUpAction();
    }
    holdDownChanged_(newHoldDown, oldHoldDown) {
        if (oldHoldDown === undefined) {
            return;
        }
        if (newHoldDown) {
            this.downAction_();
        }
        else {
            this.upAction_();
        }
    }
}
customElements.define(CrRippleElement.is, CrRippleElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrRippleMixin = (superClass) => {
    class CrRippleMixin extends superClass {
        static get properties() {
            return {
                /**
                 * If true, the element will not produce a ripple effect when
                 * interacted with via the pointer.
                 */
                noink: { type: Boolean },
            };
        }
        #noink_accessor_storage = false;
        get noink() { return this.#noink_accessor_storage; }
        set noink(value) { this.#noink_accessor_storage = value; }
        rippleContainer = null;
        ripple_ = null;
        updated(changedProperties) {
            super.updated(changedProperties);
            if (changedProperties.has('noink') && this.hasRipple()) {
                assert(this.ripple_);
                this.ripple_.noink = this.noink;
            }
        }
        ensureRippleOnPointerdown() {
            // 'capture: true' is necessary so that the cr-ripple is created early
            // enough so that it also receives the 'pointerdown' event. Otherwise
            // the ripple is created, but not shown on the 1st click.
            this.addEventListener('pointerdown', () => this.ensureRipple(), { capture: true });
        }
        /**
         * Ensures this element contains a ripple effect. For startup efficiency
         * the ripple effect is dynamically added on demand when needed.
         */
        ensureRipple() {
            if (this.hasRipple()) {
                return;
            }
            this.ripple_ = this.createRipple();
            this.ripple_.noink = this.noink;
            const rippleContainer = this.rippleContainer || this.shadowRoot;
            assert(rippleContainer);
            rippleContainer.appendChild(this.ripple_);
        }
        /**
         * Returns the `<cr-ripple>` element used by this element to create
         * ripple effects. The element's ripple is created on demand, when
         * necessary, and calling this method will force the
         * ripple to be created.
         */
        getRipple() {
            this.ensureRipple();
            assert(this.ripple_);
            return this.ripple_;
        }
        /**
         * Returns true if this element currently contains a ripple effect.
         */
        hasRipple() {
            return Boolean(this.ripple_);
        }
        /**
         * Create the element's ripple effect via creating a `<cr-ripple
         * id="ink">` instance. Override this method to customize the ripple
         * element.
         */
        createRipple() {
            const ripple = document.createElement('cr-ripple');
            ripple.id = 'ink';
            return ripple;
        }
    }
    return CrRippleMixin;
};

let instance$8 = null;
function getCss$4() {
    return instance$8 || (instance$8 = [...[], css `:host{--cr-icon-button-fill-color:currentColor;--cr-icon-button-icon-start-offset:0;--cr-icon-button-icon-size:20px;--cr-icon-button-size:32px;--cr-icon-button-height:var(--cr-icon-button-size);--cr-icon-button-transition:150ms ease-in-out;--cr-icon-button-width:var(--cr-icon-button-size);-webkit-tap-highlight-color:transparent;border-radius:50%;color:var(--cr-icon-button-stroke-color,var(--cr-icon-button-fill-color));cursor:pointer;display:inline-flex;flex-shrink:0;height:var(--cr-icon-button-height);margin-inline-end:var(--cr-icon-button-margin-end,var(--cr-icon-ripple-margin));margin-inline-start:var(--cr-icon-button-margin-start);outline:none;overflow:hidden;position:relative;user-select:none;vertical-align:middle;width:var(--cr-icon-button-width)}:host(:hover){background-color:var(--cr-icon-button-hover-background-color,var(--cr-hover-background-color))}:host(:focus-visible:focus){box-shadow:inset 0 0 0 2px var(--cr-icon-button-focus-outline-color,var(--cr-focus-outline-color))}@media (forced-colors:active){:host(:focus-visible:focus){outline:var(--cr-focus-outline-hcm)}}#ink{--paper-ripple-opacity:1;color:var(--cr-icon-button-active-background-color,var(--cr-active-background-color))}:host([disabled]){cursor:initial;opacity:var(--cr-disabled-opacity);pointer-events:none}:host(.no-overlap){--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0}:host-context([dir=rtl]):host(:not([suppress-rtl-flip]):not([multiple-icons_])){transform:scaleX(-1)}:host-context([dir=rtl]):host(:not([suppress-rtl-flip])[multiple-icons_]) cr-icon{transform:scaleX(-1)}:host(:not([iron-icon])) #maskedImage{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-button-icon-size);-webkit-transform:var(--cr-icon-image-transform,none);background-color:var(--cr-icon-button-fill-color);height:100%;transition:background-color var(--cr-icon-button-transition);width:100%}@media (forced-colors:active){:host(:not([iron-icon])) #maskedImage{background-color:ButtonText}}#icon{align-items:center;border-radius:4px;display:flex;height:100%;justify-content:center;padding-inline-start:var(--cr-icon-button-icon-start-offset);position:relative;width:100%}cr-icon{--iron-icon-fill-color:var(--cr-icon-button-fill-color);--iron-icon-stroke-color:var(--cr-icon-button-stroke-color,none);--iron-icon-height:var(--cr-icon-button-icon-size);--iron-icon-width:var(--cr-icon-button-icon-size);transition:fill var(--cr-icon-button-transition),stroke var(--cr-icon-button-transition)}@media (prefers-color-scheme:dark){:host{--cr-icon-button-fill-color:var(--google-grey-500)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$3() {
    return html `
<div id="icon">
  <div id="maskedImage"></div>
</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-icon-button' is a button which displays an icon with a
 * ripple. It can be interacted with like a normal button using click as well as
 * space and enter to effectively click the button and fire a 'click' event.
 *
 * There are two sources to icons:
 * Option 1: CSS classes defined in cr_icons.css.
 * Option 2: SVG icons defined in a cr-iconset or iron-iconset-svg,
 *     with the name passed to cr-icon-button via the |ironIcon| property.
 *
 * Example of using CSS classes:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/cr_icons.css.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button class="icon-class-name"></cr-icon-button>`;
 * }
 *
 * When an icon is specified using a class, the expectation is the
 * class will set an image to the --cr-icon-image variable.
 *
 * Example of using a cr-iconset to supply an icon via the iron-icon parameter:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/icons.html.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button iron-icon="cr:icon-key"></cr-icon-button>`;
 * }
 *
 * The color of the icon can be overridden using CSS variables. When using
 * the ironIcon property to populate cr-icon-button's internal <cr-icon>, the
 * following CSS variables for fill and stroke can be overridden for cr-icon:
 * --iron-icon-button-fill-color
 * --iron-icon-button-stroke-color
 *
 * When not using the ironIcon property, cr-icon-button will not create a
 * <cr-icon>, so the cr-icon related CSS variables above are ignored.
 *
 * When using the ironIcon property, more than one icon can be specified by
 * setting the |ironIcon| property to a comma-delimited list of keys.
 */
const CrIconbuttonElementBase = CrRippleMixin(CrLitElement);
class CrIconButtonElement extends CrIconbuttonElementBase {
    static get is() {
        return 'cr-icon-button';
    }
    static get styles() {
        return getCss$4();
    }
    render() {
        return getHtml$3.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            ironIcon: {
                type: String,
                reflect: true,
            },
            suppressRtlFlip: {
                type: Boolean,
                value: false,
                reflect: true,
            },
            multipleIcons_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #ironIcon_accessor_storage;
    get ironIcon() { return this.#ironIcon_accessor_storage; }
    set ironIcon(value) { this.#ironIcon_accessor_storage = value; }
    #multipleIcons__accessor_storage = false;
    get multipleIcons_() { return this.#multipleIcons__accessor_storage; }
    set multipleIcons_(value) { this.#multipleIcons__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('ironIcon')) {
            const icons = (this.ironIcon || '').split(',');
            this.multipleIcons_ = icons.length > 1;
        }
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
        if (changedProperties.has('ironIcon')) {
            this.onIronIconChanged_();
        }
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onIronIconChanged_() {
        this.shadowRoot.querySelectorAll('cr-icon').forEach(el => el.remove());
        if (!this.ironIcon) {
            return;
        }
        const icons = (this.ironIcon || '').split(',');
        icons.forEach(async (icon) => {
            const crIcon = document.createElement('cr-icon');
            crIcon.icon = icon;
            this.$.icon.appendChild(crIcon);
            await crIcon.updateComplete;
            crIcon.shadowRoot.querySelectorAll('svg, img')
                .forEach(child => child.setAttribute('role', 'none'));
        });
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.click();
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key === ' ' || e.key === 'Enter') {
            e.preventDefault();
            e.stopPropagation();
        }
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
        }
    }
}
customElements.define(CrIconButtonElement.is, CrIconButtonElement);

// components/omnibox/browser/omnibox.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const NavigationPredictorSpec = { $: mojo.internal.Enum() };
var NavigationPredictor;
(function (NavigationPredictor) {
    NavigationPredictor[NavigationPredictor["MIN_VALUE"] = 1] = "MIN_VALUE";
    NavigationPredictor[NavigationPredictor["MAX_VALUE"] = 3] = "MAX_VALUE";
    NavigationPredictor[NavigationPredictor["kMouseDown"] = 1] = "kMouseDown";
    NavigationPredictor[NavigationPredictor["kUpOrDownArrowButton"] = 2] = "kUpOrDownArrowButton";
    NavigationPredictor[NavigationPredictor["kTouchDown"] = 3] = "kTouchDown";
})(NavigationPredictor || (NavigationPredictor = {}));
({ $: mojo.internal.Enum() });
var PageClassification;
(function (PageClassification) {
    PageClassification[PageClassification["MIN_VALUE"] = 0] = "MIN_VALUE";
    PageClassification[PageClassification["MAX_VALUE"] = 28] = "MAX_VALUE";
    PageClassification[PageClassification["INVALID_SPEC"] = 0] = "INVALID_SPEC";
    PageClassification[PageClassification["NTP"] = 1] = "NTP";
    PageClassification[PageClassification["BLANK"] = 2] = "BLANK";
    PageClassification[PageClassification["HOME_PAGE"] = 3] = "HOME_PAGE";
    PageClassification[PageClassification["OTHER"] = 4] = "OTHER";
    PageClassification[PageClassification["SEARCH_RESULT_PAGE_DOING_SEARCH_TERM_REPLACEMENT"] = 6] = "SEARCH_RESULT_PAGE_DOING_SEARCH_TERM_REPLACEMENT";
    PageClassification[PageClassification["INSTANT_NTP_WITH_OMNIBOX_AS_STARTING_FOCUS"] = 7] = "INSTANT_NTP_WITH_OMNIBOX_AS_STARTING_FOCUS";
    PageClassification[PageClassification["SEARCH_RESULT_PAGE_NO_SEARCH_TERM_REPLACEMENT"] = 9] = "SEARCH_RESULT_PAGE_NO_SEARCH_TERM_REPLACEMENT";
    PageClassification[PageClassification["APP_HOME"] = 10] = "APP_HOME";
    PageClassification[PageClassification["APP_SEARCH"] = 11] = "APP_SEARCH";
    PageClassification[PageClassification["APP_MAPS"] = 12] = "APP_MAPS";
    PageClassification[PageClassification["SEARCH_BUTTON_AS_STARTING_FOCUS"] = 13] = "SEARCH_BUTTON_AS_STARTING_FOCUS";
    PageClassification[PageClassification["CHROMEOS_APP_LIST"] = 14] = "CHROMEOS_APP_LIST";
    PageClassification[PageClassification["NTP_REALBOX"] = 15] = "NTP_REALBOX";
    PageClassification[PageClassification["ANDROID_SEARCH_WIDGET"] = 16] = "ANDROID_SEARCH_WIDGET";
    PageClassification[PageClassification["ANDROID_SHORTCUTS_WIDGET"] = 19] = "ANDROID_SHORTCUTS_WIDGET";
    PageClassification[PageClassification["NTP_ZPS_PREFETCH"] = 20] = "NTP_ZPS_PREFETCH";
    PageClassification[PageClassification["JOURNEYS"] = 21] = "JOURNEYS";
    PageClassification[PageClassification["SRP_ZPS_PREFETCH"] = 22] = "SRP_ZPS_PREFETCH";
    PageClassification[PageClassification["OTHER_ZPS_PREFETCH"] = 23] = "OTHER_ZPS_PREFETCH";
    PageClassification[PageClassification["CONTEXTUAL_SEARCHBOX"] = 24] = "CONTEXTUAL_SEARCHBOX";
    PageClassification[PageClassification["SEARCH_SIDE_PANEL_SEARCHBOX"] = 25] = "SEARCH_SIDE_PANEL_SEARCHBOX";
    PageClassification[PageClassification["LENS_SIDE_PANEL_SEARCHBOX"] = 26] = "LENS_SIDE_PANEL_SEARCHBOX";
    PageClassification[PageClassification["SEARCH_RESULT_PAGE_ON_CCT"] = 27] = "SEARCH_RESULT_PAGE_ON_CCT";
    PageClassification[PageClassification["OTHER_ON_CCT"] = 28] = "OTHER_ON_CCT";
})(PageClassification || (PageClassification = {}));

// mojo/public/mojom/base/big_buffer.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const BigBufferSharedMemoryRegionSpec = { $: {} };
const BigBufferSpec = { $: {} };
mojo.internal.Struct(BigBufferSharedMemoryRegionSpec.$, 'BigBufferSharedMemoryRegion', [
    mojo.internal.StructField('bufferHandle', 0, 0, mojo.internal.Handle, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('size', 4, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Union(BigBufferSpec.$, 'BigBuffer', {
    'bytes': {
        'ordinal': 0,
        'type': mojo.internal.Array(mojo.internal.Uint8, false),
    },
    'sharedMemory': {
        'ordinal': 1,
        'type': BigBufferSharedMemoryRegionSpec.$,
    },
    'invalidBuffer': {
        'ordinal': 2,
        'type': mojo.internal.Bool,
    },
});
var BigBufferFieldTags;
(function (BigBufferFieldTags) {
    BigBufferFieldTags[BigBufferFieldTags["BYTES"] = 0] = "BYTES";
    BigBufferFieldTags[BigBufferFieldTags["SHARED_MEMORY"] = 1] = "SHARED_MEMORY";
    BigBufferFieldTags[BigBufferFieldTags["INVALID_BUFFER"] = 2] = "INVALID_BUFFER";
})(BigBufferFieldTags || (BigBufferFieldTags = {}));

// mojo/public/mojom/base/unguessable_token.mojom-converters.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class UnguessableTokenDataView {
    decoder_;
    version_;
    fieldSpecs_;
    constructor(decoder, version, fieldSpecs) {
        this.decoder_ = decoder;
        this.version_ = version;
        this.fieldSpecs_ = fieldSpecs;
    }
    get high() {
        const field = this.fieldSpecs_[0];
        return mojo.internal.decodeStructField(this.decoder_, field, this.version_);
    }
    get low() {
        const field = this.fieldSpecs_[1];
        return mojo.internal.decodeStructField(this.decoder_, field, this.version_);
    }
}

const HEX_BASE = 16;
// Token component is 64 bits, hex is 4 bits. This gives us a necessary
// string length.
const TOKEN_COMPONENT_STR_LENGTH = 64 / 4;
class UnguessableTokenConverter {
    // Field accessors for encoding a typemap to a mojo type.
    validate(token) {
        if (token.length !== (2 * TOKEN_COMPONENT_STR_LENGTH)) {
            throw new Error('token is malformed: ' + token);
        }
        if (token !== token.toUpperCase()) {
            throw new Error('token is not uppercase: ' + token);
        }
    }
    high(token) {
        this.validate(token);
        return BigInt(`0x${token.slice(0, TOKEN_COMPONENT_STR_LENGTH)}`);
    }
    low(token) {
        this.validate(token);
        return BigInt(`0x${token.slice(TOKEN_COMPONENT_STR_LENGTH)}`);
    }
    // Converts a mojo type to the typemap.
    convert(view) {
        return (view.high.toString(HEX_BASE).padStart(TOKEN_COMPONENT_STR_LENGTH, '0') +
            view.low.toString(HEX_BASE).padStart(TOKEN_COMPONENT_STR_LENGTH, '0'))
            .toUpperCase();
    }
}

// mojo/public/mojom/base/unguessable_token.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const UnguessableTokenSpec = { $: {} };
const converterForUnguessableToken = new UnguessableTokenConverter();
mojo.internal.TypemappedStruct(UnguessableTokenSpec.$, 'UnguessableToken', UnguessableTokenDataView, converterForUnguessableToken, [
    mojo.internal.StructField('high', 0, 0, mojo.internal.Uint64, BigInt(0), false /* nullable */, 0, undefined, ((value) => converterForUnguessableToken.high(value))),
    mojo.internal.StructField('low', 8, 0, mojo.internal.Uint64, BigInt(0), false /* nullable */, 0, undefined, ((value) => converterForUnguessableToken.low(value))),
], [[0, 24],]);

// components/omnibox/composebox/composebox_query.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const FileUploadStatusSpec = { $: mojo.internal.Enum() };
var FileUploadStatus;
(function (FileUploadStatus) {
    FileUploadStatus[FileUploadStatus["MIN_VALUE"] = 0] = "MIN_VALUE";
    FileUploadStatus[FileUploadStatus["MAX_VALUE"] = 7] = "MAX_VALUE";
    FileUploadStatus[FileUploadStatus["kNotUploaded"] = 0] = "kNotUploaded";
    FileUploadStatus[FileUploadStatus["kProcessing"] = 1] = "kProcessing";
    FileUploadStatus[FileUploadStatus["kValidationFailed"] = 2] = "kValidationFailed";
    FileUploadStatus[FileUploadStatus["kUploadStarted"] = 3] = "kUploadStarted";
    FileUploadStatus[FileUploadStatus["kUploadSuccessful"] = 4] = "kUploadSuccessful";
    FileUploadStatus[FileUploadStatus["kUploadFailed"] = 5] = "kUploadFailed";
    FileUploadStatus[FileUploadStatus["kUploadExpired"] = 6] = "kUploadExpired";
    FileUploadStatus[FileUploadStatus["kProcessingSuggestSignalsReady"] = 7] = "kProcessingSuggestSignalsReady";
})(FileUploadStatus || (FileUploadStatus = {}));
const FileUploadErrorTypeSpec = { $: mojo.internal.Enum() };
var FileUploadErrorType;
(function (FileUploadErrorType) {
    FileUploadErrorType[FileUploadErrorType["MIN_VALUE"] = 0] = "MIN_VALUE";
    FileUploadErrorType[FileUploadErrorType["MAX_VALUE"] = 6] = "MAX_VALUE";
    FileUploadErrorType[FileUploadErrorType["kUnknown"] = 0] = "kUnknown";
    FileUploadErrorType[FileUploadErrorType["kBrowserProcessingError"] = 1] = "kBrowserProcessingError";
    FileUploadErrorType[FileUploadErrorType["kNetworkError"] = 2] = "kNetworkError";
    FileUploadErrorType[FileUploadErrorType["kServerError"] = 3] = "kServerError";
    FileUploadErrorType[FileUploadErrorType["kServerSizeLimitExceeded"] = 4] = "kServerSizeLimitExceeded";
    FileUploadErrorType[FileUploadErrorType["kAborted"] = 5] = "kAborted";
    FileUploadErrorType[FileUploadErrorType["kImageProcessingError"] = 6] = "kImageProcessingError";
})(FileUploadErrorType || (FileUploadErrorType = {}));

// components/omnibox/browser/searchbox.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const SideTypeSpec = { $: mojo.internal.Enum() };
var SideType;
(function (SideType) {
    SideType[SideType["MIN_VALUE"] = 0] = "MIN_VALUE";
    SideType[SideType["MAX_VALUE"] = 1] = "MAX_VALUE";
    SideType[SideType["kDefaultPrimary"] = 0] = "kDefaultPrimary";
    SideType[SideType["kSecondary"] = 1] = "kSecondary";
})(SideType || (SideType = {}));
const RenderTypeSpec = { $: mojo.internal.Enum() };
var RenderType;
(function (RenderType) {
    RenderType[RenderType["MIN_VALUE"] = 0] = "MIN_VALUE";
    RenderType[RenderType["MAX_VALUE"] = 2] = "MAX_VALUE";
    RenderType[RenderType["kDefaultVertical"] = 0] = "kDefaultVertical";
    RenderType[RenderType["kHorizontal"] = 1] = "kHorizontal";
    RenderType[RenderType["kGrid"] = 2] = "kGrid";
})(RenderType || (RenderType = {}));
const SelectionLineStateSpec = { $: mojo.internal.Enum() };
var SelectionLineState;
(function (SelectionLineState) {
    SelectionLineState[SelectionLineState["MIN_VALUE"] = 1] = "MIN_VALUE";
    SelectionLineState[SelectionLineState["MAX_VALUE"] = 4] = "MAX_VALUE";
    SelectionLineState[SelectionLineState["kNormal"] = 1] = "kNormal";
    SelectionLineState[SelectionLineState["kKeywordMode"] = 2] = "kKeywordMode";
    SelectionLineState[SelectionLineState["kFocusedButtonAction"] = 3] = "kFocusedButtonAction";
    SelectionLineState[SelectionLineState["kFocusedButtonRemoveSuggestion"] = 4] = "kFocusedButtonRemoveSuggestion";
})(SelectionLineState || (SelectionLineState = {}));
class PageHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'searchbox.mojom.PageHandler', scope);
    }
}
class PageHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    setPage(page) {
        this.proxy.sendMessage(0, PageHandler_SetPage_ParamsSpec.$, null, [
            page
        ], false);
    }
    onFocusChanged(focused) {
        this.proxy.sendMessage(1, PageHandler_OnFocusChanged_ParamsSpec.$, null, [
            focused
        ], false);
    }
    queryAutocomplete(input, preventInlineAutocomplete) {
        this.proxy.sendMessage(2, PageHandler_QueryAutocomplete_ParamsSpec.$, null, [
            input,
            preventInlineAutocomplete
        ], false);
    }
    stopAutocomplete(clearResult) {
        this.proxy.sendMessage(3, PageHandler_StopAutocomplete_ParamsSpec.$, null, [
            clearResult
        ], false);
    }
    openAutocompleteMatch(line, url, areMatchesShowing, mouseButton, altKey, ctrlKey, metaKey, shiftKey) {
        this.proxy.sendMessage(4, PageHandler_OpenAutocompleteMatch_ParamsSpec.$, null, [
            line,
            url,
            areMatchesShowing,
            mouseButton,
            altKey,
            ctrlKey,
            metaKey,
            shiftKey
        ], false);
    }
    onNavigationLikely(line, url, navigationPredictor) {
        this.proxy.sendMessage(5, PageHandler_OnNavigationLikely_ParamsSpec.$, null, [
            line,
            url,
            navigationPredictor
        ], false);
    }
    deleteAutocompleteMatch(line, url) {
        this.proxy.sendMessage(6, PageHandler_DeleteAutocompleteMatch_ParamsSpec.$, null, [
            line,
            url
        ], false);
    }
    activateKeyword(line, url, matchSelectionTimestamp, isMouseEvent) {
        this.proxy.sendMessage(7, PageHandler_ActivateKeyword_ParamsSpec.$, null, [
            line,
            url,
            matchSelectionTimestamp,
            isMouseEvent
        ], false);
    }
    executeAction(line, actionIndex, url, matchSelectionTimestamp, mouseButton, altKey, ctrlKey, metaKey, shiftKey) {
        this.proxy.sendMessage(8, PageHandler_ExecuteAction_ParamsSpec.$, null, [
            line,
            actionIndex,
            url,
            matchSelectionTimestamp,
            mouseButton,
            altKey,
            ctrlKey,
            metaKey,
            shiftKey
        ], false);
    }
    onThumbnailRemoved() {
        this.proxy.sendMessage(9, PageHandler_OnThumbnailRemoved_ParamsSpec.$, null, [], false);
    }
    getPlaceholderConfig() {
        return this.proxy.sendMessage(10, PageHandler_GetPlaceholderConfig_ParamsSpec.$, PageHandler_GetPlaceholderConfig_ResponseParamsSpec.$, [], false);
    }
    getRecentTabs() {
        return this.proxy.sendMessage(11, PageHandler_GetRecentTabs_ParamsSpec.$, PageHandler_GetRecentTabs_ResponseParamsSpec.$, [], false);
    }
    getTabPreview(tabId) {
        return this.proxy.sendMessage(12, PageHandler_GetTabPreview_ParamsSpec.$, PageHandler_GetTabPreview_ResponseParamsSpec.$, [
            tabId
        ], false);
    }
    notifySessionStarted() {
        this.proxy.sendMessage(13, PageHandler_NotifySessionStarted_ParamsSpec.$, null, [], false);
    }
    notifySessionAbandoned() {
        this.proxy.sendMessage(14, PageHandler_NotifySessionAbandoned_ParamsSpec.$, null, [], false);
    }
    addFileContext(fileInfo, fileBytes) {
        return this.proxy.sendMessage(15, PageHandler_AddFileContext_ParamsSpec.$, PageHandler_AddFileContext_ResponseParamsSpec.$, [
            fileInfo,
            fileBytes
        ], false);
    }
    addTabContext(tabId) {
        return this.proxy.sendMessage(16, PageHandler_AddTabContext_ParamsSpec.$, PageHandler_AddTabContext_ResponseParamsSpec.$, [
            tabId
        ], false);
    }
    deleteContext(token) {
        this.proxy.sendMessage(17, PageHandler_DeleteContext_ParamsSpec.$, null, [
            token
        ], false);
    }
    clearFiles() {
        this.proxy.sendMessage(18, PageHandler_ClearFiles_ParamsSpec.$, null, [], false);
    }
    submitQuery(queryText, mouseButton, altKey, ctrlKey, metaKey, shiftKey) {
        this.proxy.sendMessage(19, PageHandler_SubmitQuery_ParamsSpec.$, null, [
            queryText,
            mouseButton,
            altKey,
            ctrlKey,
            metaKey,
            shiftKey
        ], false);
    }
}
class PageHandler {
    static get $interfaceName() {
        return "searchbox.mojom.PageHandler";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageHandlerRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
class PagePendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'searchbox.mojom.Page', scope);
    }
}
class PageRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PagePendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    autocompleteResultChanged(result) {
        this.proxy.sendMessage(0, Page_AutocompleteResultChanged_ParamsSpec.$, null, [
            result
        ], false);
    }
    updateSelection(oldSelection, selection) {
        this.proxy.sendMessage(1, Page_UpdateSelection_ParamsSpec.$, null, [
            oldSelection,
            selection
        ], false);
    }
    setInputText(input) {
        this.proxy.sendMessage(2, Page_SetInputText_ParamsSpec.$, null, [
            input
        ], false);
    }
    setThumbnail(thumbnailUrl, isDeletable) {
        this.proxy.sendMessage(3, Page_SetThumbnail_ParamsSpec.$, null, [
            thumbnailUrl,
            isDeletable
        ], false);
    }
    onContextualInputStatusChanged(token, status, errorType) {
        this.proxy.sendMessage(4, Page_OnContextualInputStatusChanged_ParamsSpec.$, null, [
            token,
            status,
            errorType
        ], false);
    }
    onTabStripChanged() {
        this.proxy.sendMessage(5, Page_OnTabStripChanged_ParamsSpec.$, null, [], false);
    }
}
/**
 * An object which receives request messages for the Page
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
class PageCallbackRouter {
    helper_internal_;
    $;
    router_;
    autocompleteResultChanged;
    updateSelection;
    setInputText;
    setThumbnail;
    onContextualInputStatusChanged;
    onTabStripChanged;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(PageRemote);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.autocompleteResultChanged =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, Page_AutocompleteResultChanged_ParamsSpec.$, null, this.autocompleteResultChanged.createReceiverHandler(false /* expectsResponse */), false);
        this.updateSelection =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(1, Page_UpdateSelection_ParamsSpec.$, null, this.updateSelection.createReceiverHandler(false /* expectsResponse */), false);
        this.setInputText =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(2, Page_SetInputText_ParamsSpec.$, null, this.setInputText.createReceiverHandler(false /* expectsResponse */), false);
        this.setThumbnail =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(3, Page_SetThumbnail_ParamsSpec.$, null, this.setThumbnail.createReceiverHandler(false /* expectsResponse */), false);
        this.onContextualInputStatusChanged =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(4, Page_OnContextualInputStatusChanged_ParamsSpec.$, null, this.onContextualInputStatusChanged.createReceiverHandler(false /* expectsResponse */), false);
        this.onTabStripChanged =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(5, Page_OnTabStripChanged_ParamsSpec.$, null, this.onTabStripChanged.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
}
const ACMatchClassificationSpec = { $: {} };
const ActionSpec = { $: {} };
const SuggestionAnswerSpec = { $: {} };
const AutocompleteMatchSpec = { $: {} };
const TabInfoSpec = { $: {} };
const SelectedFileInfoSpec = { $: {} };
const SuggestionGroupSpec = { $: {} };
const AutocompleteResultSpec = { $: {} };
const OmniboxPopupSelectionSpec = { $: {} };
const PlaceholderConfigSpec = { $: {} };
const PageHandler_SetPage_ParamsSpec = { $: {} };
const PageHandler_OnFocusChanged_ParamsSpec = { $: {} };
const PageHandler_QueryAutocomplete_ParamsSpec = { $: {} };
const PageHandler_StopAutocomplete_ParamsSpec = { $: {} };
const PageHandler_OpenAutocompleteMatch_ParamsSpec = { $: {} };
const PageHandler_OnNavigationLikely_ParamsSpec = { $: {} };
const PageHandler_DeleteAutocompleteMatch_ParamsSpec = { $: {} };
const PageHandler_ActivateKeyword_ParamsSpec = { $: {} };
const PageHandler_ExecuteAction_ParamsSpec = { $: {} };
const PageHandler_OnThumbnailRemoved_ParamsSpec = { $: {} };
const PageHandler_GetPlaceholderConfig_ParamsSpec = { $: {} };
const PageHandler_GetPlaceholderConfig_ResponseParamsSpec = { $: {} };
const PageHandler_GetRecentTabs_ParamsSpec = { $: {} };
const PageHandler_GetRecentTabs_ResponseParamsSpec = { $: {} };
const PageHandler_GetTabPreview_ParamsSpec = { $: {} };
const PageHandler_GetTabPreview_ResponseParamsSpec = { $: {} };
const PageHandler_NotifySessionStarted_ParamsSpec = { $: {} };
const PageHandler_NotifySessionAbandoned_ParamsSpec = { $: {} };
const PageHandler_AddFileContext_ParamsSpec = { $: {} };
const PageHandler_AddFileContext_ResponseParamsSpec = { $: {} };
const PageHandler_AddTabContext_ParamsSpec = { $: {} };
const PageHandler_AddTabContext_ResponseParamsSpec = { $: {} };
const PageHandler_DeleteContext_ParamsSpec = { $: {} };
const PageHandler_ClearFiles_ParamsSpec = { $: {} };
const PageHandler_SubmitQuery_ParamsSpec = { $: {} };
const Page_AutocompleteResultChanged_ParamsSpec = { $: {} };
const Page_UpdateSelection_ParamsSpec = { $: {} };
const Page_SetInputText_ParamsSpec = { $: {} };
const Page_SetThumbnail_ParamsSpec = { $: {} };
const Page_OnContextualInputStatusChanged_ParamsSpec = { $: {} };
const Page_OnTabStripChanged_ParamsSpec = { $: {} };
mojo.internal.Struct(ACMatchClassificationSpec.$, 'ACMatchClassification', [
    mojo.internal.StructField('offset', 0, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('style', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(ActionSpec.$, 'Action', [
    mojo.internal.StructField('hint', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('suggestionContents', 8, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('iconPath', 16, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('a11yLabel', 24, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 40],]);
mojo.internal.Struct(SuggestionAnswerSpec.$, 'SuggestionAnswer', [
    mojo.internal.StructField('firstLine', 0, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('secondLine', 8, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(AutocompleteMatchSpec.$, 'AutocompleteMatch', [
    mojo.internal.StructField('a11yLabel', 0, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('allowedToBeDefaultMatch', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('actions', 16, 0, mojo.internal.Array(ActionSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('answer', 24, 0, SuggestionAnswerSpec.$, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('contents', 32, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('contentsClass', 40, 0, mojo.internal.Array(ACMatchClassificationSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('description', 48, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('descriptionClass', 56, 0, mojo.internal.Array(ACMatchClassificationSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('destinationUrl', 64, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('inlineAutocompletion', 72, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('fillIntoEdit', 80, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('is_weather_answer_suggestion_$flag', 8, 1, mojo.internal.Bool, false, false /* nullable */, 0, {
        isPrimary: true,
        linkedValueFieldName: "is_weather_answer_suggestion_$value",
        originalFieldName: "isWeatherAnswerSuggestion",
    }, undefined),
    mojo.internal.StructField('is_weather_answer_suggestion_$value', 8, 2, mojo.internal.Bool, false, false /* nullable */, 0, {
        isPrimary: false,
        originalFieldName: "isWeatherAnswerSuggestion",
    }, undefined),
    mojo.internal.StructField('iconPath', 88, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('iconUrl', 96, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('imageDominantColor', 104, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('imageUrl', 112, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isNoncannedAimSuggestion', 8, 3, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isRichSuggestion', 8, 4, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isSearchType', 8, 5, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isEnterpriseSearchAggregatorPeopleType', 8, 6, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('type', 120, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('removeButtonA11yLabel', 128, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('swapContentsAndDescription', 8, 7, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('suggestionGroupId', 12, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('supportsDeletion', 9, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('tailSuggestCommonPrefix', 136, 0, String16Spec.$, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('hasInstantKeyword', 9, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('keywordChipHint', 144, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('keywordChipA11y', 152, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 168],]);
mojo.internal.Struct(TabInfoSpec.$, 'TabInfo', [
    mojo.internal.StructField('tabId', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('title', 8, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('url', 16, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('lastActive', 24, 0, TimeTicksSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 40],]);
mojo.internal.Struct(SelectedFileInfoSpec.$, 'SelectedFileInfo', [
    mojo.internal.StructField('fileName', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('mimeType', 8, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('selectionTime', 16, 0, JSTimeSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(SuggestionGroupSpec.$, 'SuggestionGroup', [
    mojo.internal.StructField('header', 0, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('renderType', 8, 0, RenderTypeSpec.$, RenderType.kDefaultVertical, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('sideType', 12, 0, SideTypeSpec.$, SideType.kDefaultPrimary, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(AutocompleteResultSpec.$, 'AutocompleteResult', [
    mojo.internal.StructField('input', 0, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('suggestionGroupsMap', 8, 0, mojo.internal.Map(mojo.internal.Int32, SuggestionGroupSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('matches', 16, 0, mojo.internal.Array(AutocompleteMatchSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('smartComposeInlineHint', 24, 0, String16Spec.$, null, true /* nullable */, 0, undefined, undefined),
], [[0, 40],]);
mojo.internal.Struct(OmniboxPopupSelectionSpec.$, 'OmniboxPopupSelection', [
    mojo.internal.StructField('line', 0, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('state', 4, 0, SelectionLineStateSpec.$, 1, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('actionIndex', 1, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PlaceholderConfigSpec.$, 'PlaceholderConfig', [
    mojo.internal.StructField('texts', 0, 0, mojo.internal.Array(String16Spec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('changeTextAnimationInterval', 8, 0, TimeDeltaSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('fadeTextAnimationDuration', 16, 0, TimeDeltaSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(PageHandler_SetPage_ParamsSpec.$, 'PageHandler_SetPage_Params', [
    mojo.internal.StructField('page', 0, 0, mojo.internal.InterfaceProxy(PageRemote), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_OnFocusChanged_ParamsSpec.$, 'PageHandler_OnFocusChanged_Params', [
    mojo.internal.StructField('focused', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_QueryAutocomplete_ParamsSpec.$, 'PageHandler_QueryAutocomplete_Params', [
    mojo.internal.StructField('input', 0, 0, String16Spec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('preventInlineAutocomplete', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_StopAutocomplete_ParamsSpec.$, 'PageHandler_StopAutocomplete_Params', [
    mojo.internal.StructField('clearResult', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_OpenAutocompleteMatch_ParamsSpec.$, 'PageHandler_OpenAutocompleteMatch_Params', [
    mojo.internal.StructField('line', 0, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('url', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('areMatchesShowing', 1, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('mouseButton', 2, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('altKey', 1, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('ctrlKey', 1, 2, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('metaKey', 1, 3, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('shiftKey', 1, 4, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_OnNavigationLikely_ParamsSpec.$, 'PageHandler_OnNavigationLikely_Params', [
    mojo.internal.StructField('line', 0, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('url', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('navigationPredictor', 4, 0, NavigationPredictorSpec.$, 1, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_DeleteAutocompleteMatch_ParamsSpec.$, 'PageHandler_DeleteAutocompleteMatch_Params', [
    mojo.internal.StructField('line', 0, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('url', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_ActivateKeyword_ParamsSpec.$, 'PageHandler_ActivateKeyword_Params', [
    mojo.internal.StructField('line', 0, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('url', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('matchSelectionTimestamp', 16, 0, TimeTicksSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isMouseEvent', 1, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(PageHandler_ExecuteAction_ParamsSpec.$, 'PageHandler_ExecuteAction_Params', [
    mojo.internal.StructField('line', 0, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('actionIndex', 1, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('url', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('matchSelectionTimestamp', 16, 0, TimeTicksSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('mouseButton', 2, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('altKey', 3, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('ctrlKey', 3, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('metaKey', 3, 2, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('shiftKey', 3, 3, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(PageHandler_OnThumbnailRemoved_ParamsSpec.$, 'PageHandler_OnThumbnailRemoved_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_GetPlaceholderConfig_ParamsSpec.$, 'PageHandler_GetPlaceholderConfig_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_GetPlaceholderConfig_ResponseParamsSpec.$, 'PageHandler_GetPlaceholderConfig_ResponseParams', [
    mojo.internal.StructField('config', 0, 0, PlaceholderConfigSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_GetRecentTabs_ParamsSpec.$, 'PageHandler_GetRecentTabs_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_GetRecentTabs_ResponseParamsSpec.$, 'PageHandler_GetRecentTabs_ResponseParams', [
    mojo.internal.StructField('tabs', 0, 0, mojo.internal.Array(TabInfoSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_GetTabPreview_ParamsSpec.$, 'PageHandler_GetTabPreview_Params', [
    mojo.internal.StructField('tabId', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_GetTabPreview_ResponseParamsSpec.$, 'PageHandler_GetTabPreview_ResponseParams', [
    mojo.internal.StructField('previewDataUrl', 0, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_NotifySessionStarted_ParamsSpec.$, 'PageHandler_NotifySessionStarted_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_NotifySessionAbandoned_ParamsSpec.$, 'PageHandler_NotifySessionAbandoned_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_AddFileContext_ParamsSpec.$, 'PageHandler_AddFileContext_Params', [
    mojo.internal.StructField('fileInfo', 0, 0, SelectedFileInfoSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('fileBytes', 8, 0, BigBufferSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(PageHandler_AddFileContext_ResponseParamsSpec.$, 'PageHandler_AddFileContext_ResponseParams', [
    mojo.internal.StructField('token', 0, 0, UnguessableTokenSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_AddTabContext_ParamsSpec.$, 'PageHandler_AddTabContext_Params', [
    mojo.internal.StructField('tabId', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_AddTabContext_ResponseParamsSpec.$, 'PageHandler_AddTabContext_ResponseParams', [
    mojo.internal.StructField('token', 0, 0, UnguessableTokenSpec.$, null, true /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_DeleteContext_ParamsSpec.$, 'PageHandler_DeleteContext_Params', [
    mojo.internal.StructField('token', 0, 0, UnguessableTokenSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_ClearFiles_ParamsSpec.$, 'PageHandler_ClearFiles_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_SubmitQuery_ParamsSpec.$, 'PageHandler_SubmitQuery_Params', [
    mojo.internal.StructField('queryText', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('mouseButton', 8, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('altKey', 9, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('ctrlKey', 9, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('metaKey', 9, 2, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('shiftKey', 9, 3, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Page_AutocompleteResultChanged_ParamsSpec.$, 'Page_AutocompleteResultChanged_Params', [
    mojo.internal.StructField('result', 0, 0, AutocompleteResultSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_UpdateSelection_ParamsSpec.$, 'Page_UpdateSelection_Params', [
    mojo.internal.StructField('oldSelection', 0, 0, OmniboxPopupSelectionSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('selection', 8, 0, OmniboxPopupSelectionSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Page_SetInputText_ParamsSpec.$, 'Page_SetInputText_Params', [
    mojo.internal.StructField('input', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_SetThumbnail_ParamsSpec.$, 'Page_SetThumbnail_Params', [
    mojo.internal.StructField('thumbnailUrl', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isDeletable', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Page_OnContextualInputStatusChanged_ParamsSpec.$, 'Page_OnContextualInputStatusChanged_Params', [
    mojo.internal.StructField('token', 0, 0, UnguessableTokenSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('status', 8, 0, FileUploadStatusSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('error_type_$flag', 12, 0, mojo.internal.Bool, false, false /* nullable */, 0, {
        isPrimary: true,
        linkedValueFieldName: "error_type_$value",
        originalFieldName: "errorType",
    }, undefined),
    mojo.internal.StructField('error_type_$value', 16, 0, FileUploadErrorTypeSpec.$, 0, false /* nullable */, 0, {
        isPrimary: false,
        originalFieldName: "errorType",
    }, undefined),
], [[0, 32],]);
mojo.internal.Struct(Page_OnTabStripChanged_ParamsSpec.$, 'Page_OnTabStripChanged_Params', [], [[0, 8],]);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function createAutocompleteMatch() {
    return {
        a11yLabel: { data: [] },
        actions: [],
        allowedToBeDefaultMatch: false,
        isSearchType: false,
        isEnterpriseSearchAggregatorPeopleType: false,
        swapContentsAndDescription: false,
        supportsDeletion: false,
        suggestionGroupId: -1,
        contents: { data: [] },
        contentsClass: [{ offset: 0, style: 0 }],
        description: { data: [] },
        descriptionClass: [{ offset: 0, style: 0 }],
        destinationUrl: { url: '' },
        inlineAutocompletion: { data: [] },
        fillIntoEdit: { data: [] },
        iconPath: '',
        iconUrl: { url: '' },
        imageDominantColor: '',
        imageUrl: '',
        isNoncannedAimSuggestion: false,
        removeButtonA11yLabel: { data: [] },
        type: '',
        isRichSuggestion: false,
        isWeatherAnswerSuggestion: null,
        answer: null,
        tailSuggestCommonPrefix: null,
        hasInstantKeyword: false,
        keywordChipHint: '',
        keywordChipA11y: '',
    };
}
class SearchboxBrowserProxy {
    static getInstance() {
        return instance$7 || (instance$7 = new SearchboxBrowserProxy());
    }
    static setInstance(newInstance) {
        instance$7 = newInstance;
    }
    handler;
    callbackRouter;
    constructor() {
        this.handler = PageHandler.getRemote();
        this.callbackRouter = new PageCallbackRouter();
        this.handler.setPage(this.callbackRouter.$.bindNewPipeAndPassRemote());
    }
}
let instance$7 = null;

let instance$6 = null;
function getCss$3() {
    return instance$6 || (instance$6 = [...[], css `.action-icon{--cr-icon-button-active-background-color:var(--color-new-tab-page-active-background);--cr-icon-button-fill-color:var(--color-searchbox-results-icon);--cr-icon-button-focus-outline-color:var(--color-searchbox-results-icon-focused-outline);--cr-icon-button-hover-background-color:var(--color-searchbox-results-button-hover);--cr-icon-button-icon-size:16px;--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0;--cr-icon-button-size:24px}`]);
}

let instance$5 = null;
function getCss$2() {
    return instance$5 || (instance$5 = [...[getCss$b(), getCss$9(), getCss$3()], css `:host{display:block;outline:none}#actions-focus-border{overflow:hidden}#actions-focus-border:focus-within,#actions-focus-border:focus-within:has(#action:active),#actions-focus-border:has(#action.selected),#actions-focus-border:has(#keyword.selected){outline:2px solid var(--color-searchbox-results-action-chip-focus-outline);border-radius:10px;margin-inline-start:-2px}#actions-focus-border:has(#action:active){outline:none}.container{align-items:center;cursor:default;display:flex;overflow:hidden;padding-bottom:var(--cr-searchbox-match-padding,6px);padding-inline-end:16px;padding-inline-start:var(--cr-searchbox-match-padding-inline-start,12px);padding-top:var(--cr-searchbox-match-padding,6px);position:relative}.container+.container{flex-direction:row;margin-inline-start:40px;padding-top:0;padding-bottom:12px}:host([has-action]) .container{height:38px;padding-top:3px;padding-bottom:3px}:host([is-top-chrome-searchbox_]:is([has-action],[has-keyword])) .container{height:40px;padding-top:0;padding-bottom:0}:host(:not([is-lens-searchbox_])) .container:not(.actions){margin-inline-end:16px;border-start-end-radius:24px;border-end-end-radius:24px}:host-context([has-secondary-side]):host-context([can-show-secondary-side]) .container:not(.actions){margin-inline-end:0px}.container:not(.actions):hover{background-color:var(--color-searchbox-results-background-hovered)}:host(:is(:focus-visible,[selected])) .container:not(.actions){background-color:var(--color-searchbox-results-background-selected,var(--color-searchbox-results-background-hovered))}:host([enable-csb-motion-tweaks_][is-lens-searchbox_]) .container{height:48px;padding-bottom:0;padding-top:0}.actions.container{align-self:center;flex-grow:1;flex-shrink:0;padding-bottom:0;padding-inline-end:0px;padding-inline-start:0px;padding-top:0;display:none}:host-context(.vertical) .actions.container{display:flex}:host([has-action]) .actions.container{padding-inline-end:8px;padding-inline-start:8px}#contents,#description{overflow:hidden;text-overflow:ellipsis}#ellipsis{inset-inline-end:0;position:absolute}:host([show-thumbnail]) #ellipsis{position:relative}#focus-indicator{--searchbox-match-focus-indicator-width_:7px;background-color:var(--color-searchbox-results-focus-indicator);border-radius:3px;display:none;height:100%;inset-inline-start:round(up,calc(-1 * var(--searchbox-match-focus-indicator-width_) / 2),1px);position:absolute;width:var(--searchbox-match-focus-indicator-width_)}:host-context(.vertical):host(:is(:focus-visible,[selected]:not(:focus-within))) #focus-indicator:not(.selected-within){display:block}:host-context(cr-searchbox-match:-webkit-any(:focus-within,[selected])) #icon{--color-searchbox-search-icon-background:var(--color-searchbox-results-dim-selected)}#prefix{opacity:0}#separator{white-space:pre}#tail-suggest-prefix{position:relative}#text-container{align-items:center;display:flex;flex-grow:0;overflow:hidden;padding-inline-end:8px;padding-inline-start:var(--cr-searchbox-match-text-padding-inline-start,8px);white-space:nowrap}#suggestion{display:inherit;overflow:inherit;flex-direction:inherit;max-width:100%}:host([is-top-chrome-searchbox_]) #text-container #suggestion{display:inline-block;min-width:0;flex:1 1 auto;white-space:nowrap;text-overflow:ellipsis}:host([is-lens-searchbox_]) #text-container{display:-webkit-box;line-clamp:2;-webkit-line-clamp:2;-webkit-box-orient:vertical;white-space:normal}:host([has-action]) #text-container{padding-inline-end:4px}:host([is-rich-suggestion]) #text-container{align-items:flex-start;flex-direction:column}:host([is-rich-suggestion]) #separator{display:none}:host([is-rich-suggestion]) #contents,:host([is-rich-suggestion]) #description{width:100%}:host([is-rich-suggestion]) #description{font-size:.875em}.match{font-weight:var(--cr-searchbox-match-font-weight,600)}:host(:not([is-top-chrome-searchbox_])) #contents span:not(.match),#ellipsis{color:var(--color-searchbox-results-typed-prefix,--color-searchbox-results-foreground)}:host-context([has-empty-input]) #contents span,:host-context([has-empty-input]) #ellipsis{color:var(--color-searchbox-results-foreground)}#description,.dim{color:var(--color-searchbox-results-foreground-dimmed)}:host-context(cr-searchbox-match:-webkit-any(:focus-within,[selected])):host([is-entity-suggestion]) #description,:host-context(cr-searchbox-match:-webkit-any(:focus-within,[selected])) .dim{color:var(--color-searchbox-results-dim-selected)}#description:has(.url),.url{color:var(--color-searchbox-results-url)}:host-context(cr-searchbox-match:-webkit-any(:focus-within,[selected])) .url{color:var(--color-searchbox-results-url-selected)}#remove{display:none;margin-inline-end:1px}:host-context(cr-searchbox-match:-webkit-any(:focus-within,[selected])) #remove{--cr-icon-button-fill-color:var(--color-searchbox-results-icon-selected)}:host-context(cr-searchbox-match:-webkit-any(:focus-within,[selected])) #remove:hover{--cr-icon-button-hover-background-color:var(--color-searchbox-results-button-selected-hover)}:host-context(.vertical) .container:hover #remove,:host-context(cr-searchbox-match:-webkit-any(:focus-within,[selected])):host-context(.vertical) #remove{display:inline-flex}.selected:not(#action):not(#keyword){box-shadow:inset 0 0 0 2px var(--color-searchbox-results-icon-focused-outline)}:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]),:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]) .container{border-radius:16px}:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]) .container{box-sizing:border-box;flex-direction:column;margin-inline-end:0;padding:6px;padding-block-end:16px;width:102px;height:auto}:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]) .focus-indicator{display:none}:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]) #icon{--cr-searchbox-icon-border-radius:12px;--color-searchbox-results-icon-container-background:transparent;height:90px;margin-block-end:8px;width:90px}:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]) #text-container{padding:0;white-space:normal;width:100%}:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]) #contents,:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]) #description{-webkit-box-orient:vertical;-webkit-line-clamp:2;display:-webkit-box;font-weight:400;overflow:hidden}:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]) #contents{font-size:13px;line-height:20px;margin-block-end:4px}:host-context(.secondary-side):host-context(.horizontal):host([is-entity-suggestion][has-image]) #description{font-size:12px;line-height:16px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$2() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div class="container" aria-hidden="true">
  <div id="focus-indicator" class="${this.getFocusIndicatorCssClass_()}"></div>
  <cr-searchbox-icon id="icon" .match="${this.match}"></cr-searchbox-icon>
  <div id="text-container">
    <span id="tail-suggest-prefix" ?hidden="${!this.tailSuggestPrefix_}">
      <span id="prefix">${this.tailSuggestPrefix_}</span>
      <!-- This is equivalent to AutocompleteMatch::kEllipsis which is
           prepended to the match content in other surfaces-->
      <span id="ellipsis">...&nbsp</span>
    </span>
    <!-- When a thumbnail is in the searchbox all results should have an
         ellipsis prepended to the suggestion. -->
    <span id="ellipsis" ?hidden="${!this.showEllipsis}">...&nbsp</span>
    <span id="suggestion">
      <span id="contents" .innerHTML="${this.contentsHtml_}"></span>
      <span id="separator" class="dim">${this.separatorText_}</span>
      <span id="description" .innerHTML="${this.descriptionHtml_}"></span>
    </span>
  </div>
  <div aria-hidden="true">
    ${this.match.keywordChipHint ? html `
      <div id="actions-focus-border">
        <cr-searchbox-action id="keyword"
            class="${this.getKeywordCssClass_()}"
            hint="${this.match.keywordChipHint}"
            icon-path="//resources/images/icon_search.svg"
            aria-label="${this.match.keywordChipA11y}"
            @execute-action="${this.onActivateKeyword_}"
            tabindex="1">
        </cr-searchbox-action>
      </div>
    ` : ''}
  </div>
  <div id="actions-container" class="actions container" aria-hidden="true">
    ${this.match.actions.map((item, index) => html `
      <div id="actions-focus-border">
        <cr-searchbox-action id="action"
            class="${this.getActionCssClass_(index)}"
            hint="${item.hint}"
            suggestion-contents="${item.suggestionContents}"
            icon-path="${item.iconPath}"
            aria-label="${item.a11yLabel}"
            action-index="${index}"
            @execute-action="${this.onExecuteAction_}" tabindex="2">
        </cr-searchbox-action>
      </div>
    `)}
  </div>
  <cr-icon-button id="remove"
      class="action-icon icon-clear ${this.getRemoveCssClass_()}"
      tabindex="3"
      aria-label="${this.removeButtonAriaLabel_}"
      title="${this.removeButtonTitle_}"
      ?hidden="${!this.match.supportsDeletion}"
      @click="${this.onRemoveButtonClick_}"
      @mousedown="${this.onRemoveButtonMouseDown_}">
  </cr-icon-button>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** Converts a String16 to a JavaScript String. */
function decodeString16(str) {
    return str ? str.data.map(ch => String.fromCodePoint(ch)).join('') : '';
}
/**
 * Converts a time ticks in milliseconds to TimeTicks.
 * @param timeTicks time ticks in milliseconds
 */
function mojoTimeTicks(timeTicks) {
    return { internalValue: BigInt(Math.floor(timeTicks * 1000)) };
}
/** Converts a side type to a string to be used in CSS. */
function sideTypeToClass(sideType) {
    switch (sideType) {
        case SideType.kDefaultPrimary:
            return 'primary-side';
        case SideType.kSecondary:
            return 'secondary-side';
        default:
            assertNotReached('Unexpected side type');
    }
}
/** Converts a render type to a string to be used in CSS. */
function renderTypeToClass(renderType) {
    switch (renderType) {
        case RenderType.kDefaultVertical:
            return 'vertical';
        case RenderType.kHorizontal:
            return 'horizontal';
        case RenderType.kGrid:
            return 'grid';
        default:
            assertNotReached('Unexpected render type');
    }
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// clang-format off
/**
 * Bitmap used to decode the value of ACMatchClassification style
 * field.
 * See components/omnibox/browser/autocomplete_match.h.
 */
var AcMatchClassificationStyle;
(function (AcMatchClassificationStyle) {
    AcMatchClassificationStyle[AcMatchClassificationStyle["NONE"] = 0] = "NONE";
    AcMatchClassificationStyle[AcMatchClassificationStyle["URL"] = 1] = "URL";
    AcMatchClassificationStyle[AcMatchClassificationStyle["MATCH"] = 2] = "MATCH";
    AcMatchClassificationStyle[AcMatchClassificationStyle["DIM"] = 4] = "DIM";
})(AcMatchClassificationStyle || (AcMatchClassificationStyle = {}));
// clang-format on
const ENTITY_MATCH_TYPE = 'search-suggest-entity';
// Displays an autocomplete match.
class SearchboxMatchElement extends CrLitElement {
    static get is() {
        return 'cr-searchbox-match';
    }
    static get styles() {
        return getCss$2();
    }
    render() {
        return getHtml$2.bind(this)();
    }
    static get properties() {
        return {
            //========================================================================
            // Public properties
            //========================================================================
            /** Element's 'aria-label' attribute. */
            ariaLabel: { type: String },
            hasAction: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether the match features an image (as opposed to an icon or favicon).
             */
            hasImage: {
                type: Boolean,
                reflect: true,
            },
            hasKeyword: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether the match is an entity suggestion (with or without an image).
             */
            isEntitySuggestion: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether the match should be rendered in a two-row layout. Currently
             * limited to matches that feature an image, calculator, and answers.
             */
            isRichSuggestion: {
                type: Boolean,
                reflect: true,
            },
            match: { type: Object },
            selection: { type: Object },
            /**
             * Index of the match in the autocomplete result. Used to inform embedder
             * of events such as deletion, click, etc.
             */
            matchIndex: { type: Number },
            showThumbnail: {
                type: Boolean,
                reflect: true,
            },
            showEllipsis: { type: Boolean },
            sideType: { type: Number },
            //========================================================================
            // Private properties
            //========================================================================
            isTopChromeSearchbox_: {
                type: Boolean,
                reflect: true,
            },
            isLensSearchbox_: {
                type: Boolean,
                reflect: true,
            },
            forceHideEllipsis_: { type: Boolean },
            /** Rendered match contents based on autocomplete provided styling. */
            contentsHtml_: { type: String },
            /** Rendered match description based on autocomplete provided styling. */
            descriptionHtml_: { type: String },
            enableCsbMotionTweaks_: {
                type: Boolean,
                reflect: true,
            },
            /** Remove button's 'aria-label' attribute. */
            removeButtonAriaLabel_: { type: String },
            removeButtonTitle_: { type: String },
            /** Used to separate the contents from the description. */
            separatorText_: { type: String },
            /** Rendered tail suggest common prefix. */
            tailSuggestPrefix_: { type: String },
        };
    }
    #ariaLabel_accessor_storage = '';
    get ariaLabel() { return this.#ariaLabel_accessor_storage; }
    set ariaLabel(value) { this.#ariaLabel_accessor_storage = value; }
    #hasAction_accessor_storage = false;
    get hasAction() { return this.#hasAction_accessor_storage; }
    set hasAction(value) { this.#hasAction_accessor_storage = value; }
    #hasImage_accessor_storage = false;
    get hasImage() { return this.#hasImage_accessor_storage; }
    set hasImage(value) { this.#hasImage_accessor_storage = value; }
    #hasKeyword_accessor_storage = false;
    get hasKeyword() { return this.#hasKeyword_accessor_storage; }
    set hasKeyword(value) { this.#hasKeyword_accessor_storage = value; }
    #isEntitySuggestion_accessor_storage = false;
    get isEntitySuggestion() { return this.#isEntitySuggestion_accessor_storage; }
    set isEntitySuggestion(value) { this.#isEntitySuggestion_accessor_storage = value; }
    #isRichSuggestion_accessor_storage = false;
    get isRichSuggestion() { return this.#isRichSuggestion_accessor_storage; }
    set isRichSuggestion(value) { this.#isRichSuggestion_accessor_storage = value; }
    #match_accessor_storage = createAutocompleteMatch();
    get match() { return this.#match_accessor_storage; }
    set match(value) { this.#match_accessor_storage = value; }
    #selection_accessor_storage = {
        line: -1,
        state: SelectionLineState.kNormal,
        actionIndex: 0,
    };
    get selection() { return this.#selection_accessor_storage; }
    set selection(value) { this.#selection_accessor_storage = value; }
    #matchIndex_accessor_storage = -1;
    get matchIndex() { return this.#matchIndex_accessor_storage; }
    set matchIndex(value) { this.#matchIndex_accessor_storage = value; }
    #sideType_accessor_storage = SideType.kDefaultPrimary;
    get sideType() { return this.#sideType_accessor_storage; }
    set sideType(value) { this.#sideType_accessor_storage = value; }
    #showThumbnail_accessor_storage = false;
    get showThumbnail() { return this.#showThumbnail_accessor_storage; }
    set showThumbnail(value) { this.#showThumbnail_accessor_storage = value; }
    #showEllipsis_accessor_storage = false;
    get showEllipsis() { return this.#showEllipsis_accessor_storage; }
    set showEllipsis(value) { this.#showEllipsis_accessor_storage = value; }
    #isTopChromeSearchbox__accessor_storage = loadTimeData.getBoolean('isTopChromeSearchbox');
    get isTopChromeSearchbox_() { return this.#isTopChromeSearchbox__accessor_storage; }
    set isTopChromeSearchbox_(value) { this.#isTopChromeSearchbox__accessor_storage = value; }
    #isLensSearchbox__accessor_storage = loadTimeData.getBoolean('isLensSearchbox');
    get isLensSearchbox_() { return this.#isLensSearchbox__accessor_storage; }
    set isLensSearchbox_(value) { this.#isLensSearchbox__accessor_storage = value; }
    #forceHideEllipsis__accessor_storage = loadTimeData.getBoolean('forceHideEllipsis');
    get forceHideEllipsis_() { return this.#forceHideEllipsis__accessor_storage; }
    set forceHideEllipsis_(value) { this.#forceHideEllipsis__accessor_storage = value; }
    #contentsHtml__accessor_storage = window.trustedTypes.emptyHTML;
    get contentsHtml_() { return this.#contentsHtml__accessor_storage; }
    set contentsHtml_(value) { this.#contentsHtml__accessor_storage = value; }
    #descriptionHtml__accessor_storage = window.trustedTypes.emptyHTML;
    get descriptionHtml_() { return this.#descriptionHtml__accessor_storage; }
    set descriptionHtml_(value) { this.#descriptionHtml__accessor_storage = value; }
    #enableCsbMotionTweaks__accessor_storage = loadTimeData.getBoolean('enableCsbMotionTweaks');
    get enableCsbMotionTweaks_() { return this.#enableCsbMotionTweaks__accessor_storage; }
    set enableCsbMotionTweaks_(value) { this.#enableCsbMotionTweaks__accessor_storage = value; }
    #removeButtonAriaLabel__accessor_storage = '';
    get removeButtonAriaLabel_() { return this.#removeButtonAriaLabel__accessor_storage; }
    set removeButtonAriaLabel_(value) { this.#removeButtonAriaLabel__accessor_storage = value; }
    #removeButtonTitle__accessor_storage = loadTimeData.getString('removeSuggestion');
    get removeButtonTitle_() { return this.#removeButtonTitle__accessor_storage; }
    set removeButtonTitle_(value) { this.#removeButtonTitle__accessor_storage = value; }
    #separatorText__accessor_storage = '';
    get separatorText_() { return this.#separatorText__accessor_storage; }
    set separatorText_(value) { this.#separatorText__accessor_storage = value; }
    #tailSuggestPrefix__accessor_storage = '';
    get tailSuggestPrefix_() { return this.#tailSuggestPrefix__accessor_storage; }
    set tailSuggestPrefix_(value) { this.#tailSuggestPrefix__accessor_storage = value; }
    pageHandler_;
    constructor() {
        super();
        this.pageHandler_ = SearchboxBrowserProxy.getInstance().handler;
    }
    firstUpdated() {
        this.addEventListener('click', (event) => this.onMatchClick_(event));
        this.addEventListener('focusin', () => this.onMatchFocusin_());
        this.addEventListener('mousedown', () => this.onMatchMouseDown_());
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('match')) {
            this.ariaLabel = this.computeAriaLabel_();
            this.contentsHtml_ = this.computeContentsHtml_();
            this.descriptionHtml_ = this.computeDescriptionHtml_();
            this.hasAction = this.computeHasAction_();
            this.hasKeyword = this.computeHasKeyword_();
            this.hasImage = this.computeHasImage_();
            this.isEntitySuggestion = this.computeIsEntitySuggestion_();
            this.isRichSuggestion = this.computeIsRichSuggestion_();
            this.removeButtonAriaLabel_ = this.computeRemoveButtonAriaLabel_();
            this.separatorText_ = this.computeSeparatorText_();
            this.tailSuggestPrefix_ = this.computeTailSuggestPrefix_();
        }
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('showThumbnail') ||
            changedPrivateProperties.has('isLensSearchbox_') ||
            changedPrivateProperties.has('forceHideEllipsis_')) {
            this.showEllipsis = this.computeShowEllipsis_();
        }
    }
    //============================================================================
    // Event handlers
    //============================================================================
    onActivateKeyword_(e) {
        // Keyboard activation isn't possible because when the keyword chip is
        // focused, focus is redirected to the omnibox view.
        const event = e.detail.event;
        this.pageHandler_.activateKeyword(this.matchIndex, this.match.destinationUrl, mojoTimeTicks(Date.now()), 
        // Distinguish mouse and touch or pen events for logging purposes.
        event.pointerType === 'mouse');
    }
    /**
     * containing index of the action that was removed as well as modifier key
     * presses.
     */
    onExecuteAction_(e) {
        const event = e.detail.event;
        this.pageHandler_.executeAction(this.matchIndex, e.detail.actionIndex, this.match.destinationUrl, mojoTimeTicks(Date.now()), event.button || 0, event.altKey, event.ctrlKey, event.metaKey, event.shiftKey);
    }
    onMatchClick_(e) {
        if (e.button > 1) {
            // Only handle main (generally left) and middle button presses.
            return;
        }
        e.preventDefault(); // Prevents default browser action (navigation).
        e.stopPropagation(); // Prevents <iron-selector> from selecting the match.
        this.pageHandler_.openAutocompleteMatch(this.matchIndex, this.match.destinationUrl, 
        /* are_matches_showing */ true, e.button || 0, e.altKey, e.ctrlKey, e.metaKey, e.shiftKey);
        this.fire('match-click');
    }
    onMatchFocusin_() {
        this.fire('match-focusin', this.matchIndex);
    }
    onMatchMouseDown_() {
        this.pageHandler_.onNavigationLikely(this.matchIndex, this.match.destinationUrl, NavigationPredictor.kMouseDown);
    }
    onRemoveButtonClick_(e) {
        if (e.button !== 0) {
            // Only handle main (generally left) button presses.
            return;
        }
        e.preventDefault(); // Prevents default browser action (navigation).
        e.stopPropagation(); // Prevents <iron-selector> from selecting the match.
        this.pageHandler_.deleteAutocompleteMatch(this.matchIndex, this.match.destinationUrl);
    }
    onRemoveButtonMouseDown_(e) {
        e.preventDefault(); // Prevents default browser action (focus).
    }
    //============================================================================
    // Helpers
    //============================================================================
    computeAriaLabel_() {
        if (!this.match) {
            return '';
        }
        return decodeString16(this.match.a11yLabel);
    }
    sanitizeInnerHtml_(html) {
        return sanitizeInnerHtml(html, { attrs: ['class'] });
    }
    computeContentsHtml_() {
        if (!this.match) {
            return window.trustedTypes.emptyHTML;
        }
        // `match.answer.firstLine` is generated by appending an optional additional
        // text from the answer's first line to `match.contents`, making the latter
        // a prefix of the former. Thus `match.answer.firstLine` can be rendered
        // using the markup in `match.contentsClass` which contains positions in
        // `match.contents` and the markup to be applied to those positions.
        // See //chrome/browser/ui/webui/searchbox/searchbox_handler.cc
        return this.sanitizeInnerHtml_(this.renderTextWithClassifications_(this.getMatchContents_(), this.getMatchContentsClassifications_())
            .innerHTML);
    }
    computeDescriptionHtml_() {
        if (!this.match) {
            return window.trustedTypes.emptyHTML;
        }
        const match = this.match;
        if (match.answer) {
            return this.sanitizeInnerHtml_(this.getMatchDescription_());
        }
        return this.sanitizeInnerHtml_(this.renderTextWithClassifications_(this.getMatchDescription_(), this.getMatchDescriptionClassifications_())
            .innerHTML);
    }
    computeHasAction_() {
        return this.match?.actions?.length > 0;
    }
    computeHasKeyword_() {
        return this.match && !!this.match.keywordChipHint;
    }
    computeHasImage_() {
        return this.match && !!this.match.imageUrl;
    }
    computeIsEntitySuggestion_() {
        return this.match && this.match.type === ENTITY_MATCH_TYPE;
    }
    computeIsRichSuggestion_() {
        // When the searchbox is embedded in the top-chrome (i.e. Omnibox), all
        // suggestions should be rendered using a one-line layout.
        return !this.isTopChromeSearchbox_ && this.match &&
            this.match.isRichSuggestion;
    }
    computeRemoveButtonAriaLabel_() {
        if (!this.match) {
            return '';
        }
        return decodeString16(this.match.removeButtonA11yLabel);
    }
    computeSeparatorText_() {
        return this.getMatchDescription_() ?
            loadTimeData.getString('searchboxSeparator') :
            '';
    }
    computeTailSuggestPrefix_() {
        if (!this.match || !this.match.tailSuggestCommonPrefix) {
            return '';
        }
        const prefix = decodeString16(this.match.tailSuggestCommonPrefix);
        // Replace last space with non breaking space since spans collapse
        // trailing white spaces and the prefix always ends with a white space.
        if (prefix.slice(-1) === ' ') {
            return prefix.slice(0, -1) + '\u00A0';
        }
        return prefix;
    }
    computeShowEllipsis_() {
        if (this.isLensSearchbox_ && this.forceHideEllipsis_) {
            return false;
        }
        return this.showThumbnail;
    }
    /**
     * Decodes the AcMatchClassificationStyle entries encoded in the given
     * ACMatchClassification style field, maps each entry to a CSS
     * class and returns them.
     */
    convertClassificationStyleToCssClasses_(style) {
        const classes = [];
        if (style & AcMatchClassificationStyle.DIM) {
            classes.push('dim');
        }
        if (style & AcMatchClassificationStyle.MATCH) {
            classes.push('match');
        }
        if (style & AcMatchClassificationStyle.URL) {
            classes.push('url');
        }
        return classes;
    }
    createSpanWithClasses_(text, classes) {
        const span = document.createElement('span');
        if (classes.length) {
            span.classList.add(...classes);
        }
        span.textContent = text;
        return span;
    }
    /**
     * Renders |text| based on the given ACMatchClassification(s)
     * Each classification contains an 'offset' and an encoded list of styles for
     * styling a substring starting with the 'offset' and ending with the next.
     * @return A <span> with <span> children for each styled substring.
     */
    renderTextWithClassifications_(text, classifications) {
        return classifications
            .map(({ offset, style }, index) => {
            const next = classifications[index + 1] || { offset: text.length };
            const subText = text.substring(offset, next.offset);
            const classes = this.convertClassificationStyleToCssClasses_(style);
            return this.createSpanWithClasses_(subText, classes);
        })
            .reduce((container, currentElement) => {
            container.appendChild(currentElement);
            return container;
        }, document.createElement('span'));
    }
    getMatchContents_() {
        if (!this.match) {
            return '';
        }
        const match = this.match;
        const matchContents = match.answer ? match.answer.firstLine : match.contents;
        const matchDescription = match.answer ? match.answer.secondLine : match.description;
        return decodeString16(match.swapContentsAndDescription ? matchDescription : matchContents);
    }
    getMatchDescription_() {
        if (!this.match) {
            return '';
        }
        const match = this.match;
        const matchContents = match.answer ? match.answer.firstLine : match.contents;
        const matchDescription = match.answer ? match.answer.secondLine : match.description;
        return decodeString16(match.swapContentsAndDescription ? matchContents : matchDescription);
    }
    getMatchContentsClassifications_() {
        if (!this.match) {
            return [];
        }
        const match = this.match;
        return match.swapContentsAndDescription ? match.descriptionClass :
            match.contentsClass;
    }
    getMatchDescriptionClassifications_() {
        if (!this.match) {
            return [];
        }
        const match = this.match;
        return match.swapContentsAndDescription ? match.contentsClass :
            match.descriptionClass;
    }
    getFocusIndicatorCssClass_() {
        return this.selection.line === this.matchIndex &&
            this.selection.state !== SelectionLineState.kNormal &&
            !this.match.hasInstantKeyword ?
            'selected-within' :
            '';
    }
    getKeywordCssClass_() {
        return this.selection.line === this.matchIndex &&
            this.selection.state === SelectionLineState.kKeywordMode ?
            'selected' :
            '';
    }
    getActionCssClass_(actionIndex) {
        return this.selection.line === this.matchIndex &&
            this.selection.state === SelectionLineState.kFocusedButtonAction &&
            this.selection.actionIndex === actionIndex ?
            'selected' :
            '';
    }
    getRemoveCssClass_() {
        return this.selection.line === this.matchIndex &&
            this.selection.state ===
                SelectionLineState.kFocusedButtonRemoveSuggestion ?
            'selected' :
            '';
    }
}
customElements.define(SearchboxMatchElement.is, SearchboxMatchElement);

// ui/webui/resources/js/metrics_reporter/metrics_reporter.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PageMetricsHostPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'metrics_reporter.mojom.PageMetricsHost', scope);
    }
}
class PageMetricsHostRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageMetricsHostPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    onPageRemoteCreated(page) {
        this.proxy.sendMessage(0, PageMetricsHost_OnPageRemoteCreated_ParamsSpec.$, null, [
            page
        ], false);
    }
    onGetMark(name) {
        return this.proxy.sendMessage(1, PageMetricsHost_OnGetMark_ParamsSpec.$, PageMetricsHost_OnGetMark_ResponseParamsSpec.$, [
            name
        ], false);
    }
    onClearMark(name) {
        this.proxy.sendMessage(2, PageMetricsHost_OnClearMark_ParamsSpec.$, null, [
            name
        ], false);
    }
    onUmaReportTime(name, time) {
        this.proxy.sendMessage(3, PageMetricsHost_OnUmaReportTime_ParamsSpec.$, null, [
            name,
            time
        ], false);
    }
}
class PageMetricsHost {
    static get $interfaceName() {
        return "metrics_reporter.mojom.PageMetricsHost";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageMetricsHostRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
class PageMetricsPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'metrics_reporter.mojom.PageMetrics', scope);
    }
}
class PageMetricsRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageMetricsPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    onGetMark(name) {
        return this.proxy.sendMessage(0, PageMetrics_OnGetMark_ParamsSpec.$, PageMetrics_OnGetMark_ResponseParamsSpec.$, [
            name
        ], false);
    }
    onClearMark(name) {
        this.proxy.sendMessage(1, PageMetrics_OnClearMark_ParamsSpec.$, null, [
            name
        ], false);
    }
}
/**
 * An object which receives request messages for the PageMetrics
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
class PageMetricsCallbackRouter {
    helper_internal_;
    $;
    router_;
    onGetMark;
    onClearMark;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(PageMetricsRemote);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.onGetMark =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, PageMetrics_OnGetMark_ParamsSpec.$, PageMetrics_OnGetMark_ResponseParamsSpec.$, this.onGetMark.createReceiverHandler(true /* expectsResponse */), false);
        this.onClearMark =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(1, PageMetrics_OnClearMark_ParamsSpec.$, null, this.onClearMark.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
}
const PageMetricsHost_OnPageRemoteCreated_ParamsSpec = { $: {} };
const PageMetricsHost_OnGetMark_ParamsSpec = { $: {} };
const PageMetricsHost_OnGetMark_ResponseParamsSpec = { $: {} };
const PageMetricsHost_OnClearMark_ParamsSpec = { $: {} };
const PageMetricsHost_OnUmaReportTime_ParamsSpec = { $: {} };
const PageMetrics_OnGetMark_ParamsSpec = { $: {} };
const PageMetrics_OnGetMark_ResponseParamsSpec = { $: {} };
const PageMetrics_OnClearMark_ParamsSpec = { $: {} };
mojo.internal.Struct(PageMetricsHost_OnPageRemoteCreated_ParamsSpec.$, 'PageMetricsHost_OnPageRemoteCreated_Params', [
    mojo.internal.StructField('page', 0, 0, mojo.internal.InterfaceProxy(PageMetricsRemote), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageMetricsHost_OnGetMark_ParamsSpec.$, 'PageMetricsHost_OnGetMark_Params', [
    mojo.internal.StructField('name', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageMetricsHost_OnGetMark_ResponseParamsSpec.$, 'PageMetricsHost_OnGetMark_ResponseParams', [
    mojo.internal.StructField('markedTime', 0, 0, TimeDeltaSpec.$, null, true /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageMetricsHost_OnClearMark_ParamsSpec.$, 'PageMetricsHost_OnClearMark_Params', [
    mojo.internal.StructField('name', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageMetricsHost_OnUmaReportTime_ParamsSpec.$, 'PageMetricsHost_OnUmaReportTime_Params', [
    mojo.internal.StructField('name', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('time', 8, 0, TimeDeltaSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageMetrics_OnGetMark_ParamsSpec.$, 'PageMetrics_OnGetMark_Params', [
    mojo.internal.StructField('name', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageMetrics_OnGetMark_ResponseParamsSpec.$, 'PageMetrics_OnGetMark_ResponseParams', [
    mojo.internal.StructField('markedTime', 0, 0, TimeDeltaSpec.$, null, true /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageMetrics_OnClearMark_ParamsSpec.$, 'PageMetrics_OnClearMark_Params', [
    mojo.internal.StructField('name', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class BrowserProxyImpl {
    callbackRouter;
    host;
    constructor() {
        this.callbackRouter = new PageMetricsCallbackRouter();
        this.host = PageMetricsHost.getRemote();
        this.host.onPageRemoteCreated(this.callbackRouter.$.bindNewPipeAndPassRemote());
    }
    getMark(name) {
        return this.host.onGetMark(name);
    }
    clearMark(name) {
        this.host.onClearMark(name);
    }
    umaReportTime(name, time) {
        this.host.onUmaReportTime(name, time);
    }
    now() {
        return chrome.timeTicks.nowInMicroseconds();
    }
    getCallbackRouter() {
        return this.callbackRouter;
    }
    static getInstance() {
        return instance$4 || (instance$4 = new BrowserProxyImpl());
    }
    static setInstance(obj) {
        instance$4 = obj;
    }
}
let instance$4 = null;

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function timeFromMojo(delta) {
    return delta.microseconds;
}
function timeToMojo(mark) {
    return { microseconds: mark };
}
class MetricsReporterImpl {
    marks_ = new Map();
    browserProxy_ = BrowserProxyImpl.getInstance();
    constructor() {
        const callbackRouter = this.browserProxy_.getCallbackRouter();
        callbackRouter.onGetMark.addListener((name) => ({
            markedTime: this.marks_.has(name) ? timeToMojo(this.marks_.get(name)) : null,
        }));
        callbackRouter.onClearMark.addListener((name) => this.marks_.delete(name));
    }
    static getInstance() {
        return instance$3 || (instance$3 = new MetricsReporterImpl());
    }
    static setInstanceForTest(newInstance) {
        instance$3 = newInstance;
    }
    mark(name) {
        this.marks_.set(name, this.browserProxy_.now());
    }
    async measure(startMark, endMark) {
        let endTime;
        if (endMark) {
            const entry = this.marks_.get(endMark);
            assert(entry, `Mark "${endMark}" does not exist locally.`);
            endTime = entry;
        }
        else {
            endTime = this.browserProxy_.now();
        }
        let startTime;
        if (this.marks_.has(startMark)) {
            startTime = this.marks_.get(startMark);
        }
        else {
            const remoteStartTime = await this.browserProxy_.getMark(startMark);
            assert(remoteStartTime.markedTime, `Mark "${startMark}" does not exist locally or remotely.`);
            startTime = timeFromMojo(remoteStartTime.markedTime);
        }
        return endTime - startTime;
    }
    async hasMark(name) {
        if (this.marks_.has(name)) {
            return true;
        }
        const remoteMark = await this.browserProxy_.getMark(name);
        return remoteMark !== null && remoteMark.markedTime !== null;
    }
    hasLocalMark(name) {
        return this.marks_.has(name);
    }
    clearMark(name) {
        this.marks_.delete(name);
        this.browserProxy_.clearMark(name);
    }
    umaReportTime(histogram, time) {
        this.browserProxy_.umaReportTime(histogram, timeToMojo(time));
    }
}
let instance$3 = null;

let instance$2 = null;
function getCss$1() {
    return instance$2 || (instance$2 = [...[getCss$9(), getCss$3()], css `:host{user-select:none}#content{background-color:var(--color-searchbox-results-background);border-radius:calc(0.5 * var(--cr-searchbox-height));box-shadow:var(--cr-searchbox-shadow);display:flex;gap:16px;margin-bottom:var(--cr-searchbox-results-margin-bottom,8px);overflow:hidden;padding-bottom:8px;padding-top:var(--cr-searchbox-height)}@media (forced-colors:active){#content{border:1px solid ActiveBorder}}.matches{display:contents}cr-searchbox-match{color:var(--color-searchbox-results-foreground)}cr-searchbox-match:-webkit-any(:focus-within,[selected]){color:var(--color-searchbox-results-foreground-selected)}.header{align-items:center;box-sizing:border-box;display:flex;font-size:inherit;font-weight:inherit;height:44px;margin-block-end:0;margin-block-start:0;outline:none;padding-bottom:6px;padding-inline-end:16px;padding-inline-start:12px;padding-top:6px}.header .text{color:var(--color-searchbox-results-foreground-dimmed);font-size:.875em;font-weight:500;overflow:hidden;padding-inline-end:1px;text-overflow:ellipsis;white-space:nowrap}@media (forced-colors:active){cr-searchbox-match:-webkit-any(:hover,:focus-within,[selected]){background-color:Highlight}}.primary-side{flex:1;min-width:0}:host-context([is-lens-searchbox_]) .primary-side::before{content:'';position:relative;height:1px;background-color:var(--color-searchbox-dropdown-divider);top:0;width:calc(var(--cr-searchbox-width) - 24px);display:block;inset-inline-start:12px;margin-block-end:4px}.secondary-side{display:var(--cr-searchbox-secondary-side-display,none);min-width:0;padding-block-end:8px;padding-inline-end:16px;width:314px}.secondary-side .header{padding-inline-end:0;padding-inline-start:0}.secondary-side .matches{display:block}.secondary-side .matches.horizontal{display:flex;gap:4px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$1() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="content" part="dropdown-content">
  ${this.sideTypes_().map(sideType => html `
    <div class="${this.sideTypeClass_(sideType)}">
      ${this.groupIdsForSideType_(sideType).map(groupId => html `
        ${this.hasHeaderForGroup_(groupId) ? html `
          <!-- Header cannot be tabbed into but gets focus when clicked. This
              stops the dropdown from losing focus and closing as a result. -->
          <h3 class="header" data-id="${groupId}" tabindex="-1"
              @mousedown="${this.onHeaderMousedown_}" aria-hidden="true">
            <span class="text">${this.headerForGroup_(groupId)}</span>
          </h3>
        ` : ''}
        <div class="matches ${this.renderTypeClassForGroup_(groupId)}">
          ${this.matchesForGroup_(groupId).map(match => html `
            <cr-searchbox-match tabindex="0" role="option"
                .match="${match}" match-index="${this.matchIndex_(match)}"
                side-type="${sideType}"
                ?selected="${this.isSelected_(match)}"
                ?show-thumbnail="${this.showThumbnail}">
            </cr-searchbox-match>
          `)}
        </div>
      `)}
    </div>
  `)}
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// The '%' operator in JS returns negative numbers. This workaround avoids that.
const remainder = (lhs, rhs) => ((lhs % rhs) + rhs) % rhs;
// A dropdown element that contains autocomplete matches. Provides an API for
// the embedder (i.e., <cr-searchbox>) to change the selection.
class SearchboxDropdownElement extends CrLitElement {
    static get is() {
        return 'cr-searchbox-dropdown';
    }
    static get styles() {
        return getCss$1();
    }
    render() {
        return getHtml$1.bind(this)();
    }
    static get properties() {
        return {
            //========================================================================
            // Public properties
            //========================================================================
            /**
             * Whether the secondary side can be shown based on the feature state and
             * the width available to the dropdown.
             */
            canShowSecondarySide: { type: Boolean },
            /**
             * Whether the secondary side was at any point available to be shown.
             */
            hadSecondarySide: {
                type: Boolean,
                notify: true,
            },
            /*
             * Whether the secondary side is currently available to be shown.
             */
            hasSecondarySide: {
                type: Boolean,
                notify: true,
                reflect: true,
            },
            hasEmptyInput: {
                type: Boolean,
                reflect: true,
            },
            result: { type: Object },
            /** Index of the selected match. */
            selectedMatchIndex: {
                type: Number,
                notify: true,
            },
            showThumbnail: { type: Boolean },
            //========================================================================
            // Private properties
            //========================================================================
            /**
             * Computed value for whether or not the dropdown should show the
             * secondary side. This depends on whether the parent has set
             * `canShowSecondarySide` to true and whether there are visible primary
             * matches.
             */
            showSecondarySide_: { type: Boolean },
        };
    }
    #canShowSecondarySide_accessor_storage = false;
    get canShowSecondarySide() { return this.#canShowSecondarySide_accessor_storage; }
    set canShowSecondarySide(value) { this.#canShowSecondarySide_accessor_storage = value; }
    #hadSecondarySide_accessor_storage = false;
    get hadSecondarySide() { return this.#hadSecondarySide_accessor_storage; }
    set hadSecondarySide(value) { this.#hadSecondarySide_accessor_storage = value; }
    #hasSecondarySide_accessor_storage = false;
    get hasSecondarySide() { return this.#hasSecondarySide_accessor_storage; }
    set hasSecondarySide(value) { this.#hasSecondarySide_accessor_storage = value; }
    #hasEmptyInput_accessor_storage = false;
    get hasEmptyInput() { return this.#hasEmptyInput_accessor_storage; }
    set hasEmptyInput(value) { this.#hasEmptyInput_accessor_storage = value; }
    #result_accessor_storage = null;
    get result() { return this.#result_accessor_storage; }
    set result(value) { this.#result_accessor_storage = value; }
    #selectedMatchIndex_accessor_storage = -1;
    get selectedMatchIndex() { return this.#selectedMatchIndex_accessor_storage; }
    set selectedMatchIndex(value) { this.#selectedMatchIndex_accessor_storage = value; }
    #showThumbnail_accessor_storage = false;
    get showThumbnail() { return this.#showThumbnail_accessor_storage; }
    set showThumbnail(value) { this.#showThumbnail_accessor_storage = value; }
    #showSecondarySide__accessor_storage = false;
    get showSecondarySide_() { return this.#showSecondarySide__accessor_storage; }
    set showSecondarySide_(value) { this.#showSecondarySide__accessor_storage = value; }
    /** The list of selectable match elements. */
    selectableMatchElements_ = [];
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('result')) {
            this.hasSecondarySide = this.computeHasSecondarySide_();
            this.hasEmptyInput = this.computeHasEmptyInput_();
        }
        if (changedProperties.has('result') ||
            changedProperties.has('canShowSecondarySide')) {
            this.showSecondarySide_ = this.computeShowSecondarySide_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        this.onResultRepaint_();
        // Update the list of selectable match elements.
        this.selectableMatchElements_ =
            [...this.shadowRoot.querySelectorAll('cr-searchbox-match')];
    }
    //============================================================================
    // Public methods
    //============================================================================
    /** Filters out secondary matches, if any, unless they can be shown. */
    get selectableMatchElements() {
        return this.selectableMatchElements_.filter(matchEl => matchEl.sideType === SideType.kDefaultPrimary ||
            this.showSecondarySide_);
    }
    /** Unselects the currently selected match, if any. */
    unselect() {
        this.selectedMatchIndex = -1;
    }
    /** Focuses the selected match, if any. */
    focusSelected() {
        this.selectableMatchElements[this.selectedMatchIndex]?.focus();
    }
    /** Selects the first match. */
    selectFirst() {
        this.selectedMatchIndex = 0;
        return this.updateComplete;
    }
    /** Selects the match at the given index. */
    selectIndex(index) {
        this.selectedMatchIndex = index;
        return this.updateComplete;
    }
    updateSelection(oldSelection, selection) {
        // If the updated selection is a new match, remove any remaining selection
        // on the previously selected match.
        if (oldSelection.line !== selection.line) {
            const oldMatch = this.selectableMatchElements[this.selectedMatchIndex];
            if (oldMatch) {
                oldMatch.selection = selection;
            }
        }
        this.selectIndex(selection.line);
        const newMatch = this.selectableMatchElements[this.selectedMatchIndex];
        if (newMatch) {
            newMatch.selection = selection;
        }
    }
    /**
     * Selects the previous match with respect to the currently selected one.
     * Selects the last match if the first one or no match is currently selected.
     */
    selectPrevious() {
        // The value of -1 for |this.selectedMatchIndex| indicates no selection.
        // Therefore subtract one from the maximum of its value and 0.
        const previous = Math.max(this.selectedMatchIndex, 0) - 1;
        this.selectedMatchIndex =
            remainder(previous, this.selectableMatchElements.length);
        return this.updateComplete;
    }
    /** Selects the last match. */
    selectLast() {
        this.selectedMatchIndex = this.selectableMatchElements.length - 1;
        return this.updateComplete;
    }
    /**
     * Selects the next match with respect to the currently selected one.
     * Selects the first match if the last one or no match is currently selected.
     */
    selectNext() {
        const next = this.selectedMatchIndex + 1;
        this.selectedMatchIndex =
            remainder(next, this.selectableMatchElements.length);
        return this.updateComplete;
    }
    //============================================================================
    // Event handlers
    //============================================================================
    onHeaderMousedown_(e) {
        e.preventDefault(); // Prevents default browser action (focus).
    }
    onResultRepaint_() {
        if (!loadTimeData.getBoolean('reportMetrics')) {
            return;
        }
        const metricsReporter = MetricsReporterImpl.getInstance();
        metricsReporter.measure('CharTyped')
            .then(duration => {
            metricsReporter.umaReportTime(loadTimeData.getString('charTypedToPaintMetricName'), duration);
        })
            .then(() => {
            metricsReporter.clearMark('CharTyped');
        })
            .catch(() => { }); // Fail silently if 'CharTyped' is not marked.
        metricsReporter.measure('ResultChanged')
            .then(duration => {
            metricsReporter.umaReportTime(loadTimeData.getString('resultChangedToPaintMetricName'), duration);
        })
            .then(() => {
            metricsReporter.clearMark('ResultChanged');
        })
            .catch(() => { }); // Fail silently if 'ResultChanged' is not marked.
    }
    //============================================================================
    // Helpers
    //============================================================================
    sideTypeClass_(side) {
        return sideTypeToClass(side);
    }
    renderTypeClassForGroup_(groupId) {
        return renderTypeToClass(this.result?.suggestionGroupsMap[groupId]?.renderType ??
            RenderType.kDefaultVertical);
    }
    computeHasSecondarySide_() {
        const hasSecondarySide = !!this.groupIdsForSideType_(SideType.kSecondary).length;
        if (!this.hadSecondarySide) {
            this.hadSecondarySide = hasSecondarySide;
        }
        return hasSecondarySide;
    }
    computeHasEmptyInput_() {
        return !!this.result && decodeString16(this.result.input) === '';
    }
    isSelected_(match) {
        return this.matchIndex_(match) === this.selectedMatchIndex;
    }
    /**
     * @returns The unique suggestion group IDs that belong to the given side type
     *     while preserving the order in which they appear in the list of matches.
     */
    groupIdsForSideType_(side) {
        return [...new Set(this.result?.matches.map(match => match.suggestionGroupId)
                .filter(groupId => this.sideTypeForGroup_(groupId) === side))];
    }
    /**
     * @returns Whether the given suggestion group ID has a header.
     */
    hasHeaderForGroup_(groupId) {
        return !!this.headerForGroup_(groupId);
    }
    /**
     * @returns The header for the given suggestion group ID, if any.
     */
    headerForGroup_(groupId) {
        return this.result?.suggestionGroupsMap[groupId] ?
            decodeString16(this.result.suggestionGroupsMap[groupId].header) :
            '';
    }
    /**
     * @returns Index of the match in the autocomplete result. Passed to the match
     *     so it knows its position in the list of matches.
     */
    matchIndex_(match) {
        return this.result?.matches.indexOf(match) ?? -1;
    }
    /**
     * @returns The list of visible matches that belong to the given suggestion
     *     group ID.
     */
    matchesForGroup_(groupId) {
        return (this.result?.matches ?? [])
            .filter(match => match.suggestionGroupId === groupId);
    }
    /**
     * @returns The list of side types to show.
     */
    sideTypes_() {
        return this.showSecondarySide_ ?
            [SideType.kDefaultPrimary, SideType.kSecondary] :
            [SideType.kDefaultPrimary];
    }
    /**
     * @returns The side type for the given suggestion group ID.
     */
    sideTypeForGroup_(groupId) {
        return this.result?.suggestionGroupsMap[groupId]?.sideType ??
            SideType.kDefaultPrimary;
    }
    computeShowSecondarySide_() {
        if (!this.canShowSecondarySide) {
            // Parent prohibits showing secondary side.
            return false;
        }
        // Only show secondary side if there are primary matches visible.
        const primaryGroupIds = this.groupIdsForSideType_(SideType.kDefaultPrimary);
        return primaryGroupIds.some((groupId) => {
            return this.matchesForGroup_(groupId).length > 0;
        });
    }
}
customElements.define(SearchboxDropdownElement.is, SearchboxDropdownElement);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This file provides a singleton class that exposes the Mojo
 * handler interface used for one way communication between the JS and the
 * browser.
 * TODO(tluk): Convert this into typescript once all dependencies have been
 * fully migrated.
 */
let instance$1 = null;
class BrowserProxy {
    callbackRouter;
    constructor() {
        this.callbackRouter = new PageCallbackRouter$1();
        const pageHandlerRemote = PageHandler$1.getRemote();
        pageHandlerRemote.setPage(this.callbackRouter.$.bindNewPipeAndPassRemote());
    }
    static getInstance() {
        return instance$1 || (instance$1 = new BrowserProxy());
    }
    static setInstance(newInstance) {
        instance$1 = newInstance;
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This file holds the functions that allow WebUI to update its
 * colors CSS stylesheet when a ColorProvider change in the browser is detected.
 */
/**
 * The CSS selector used to get the <link> node with the colors.css stylesheet.
 * The wildcard is needed since the URL ends with a timestamp.
 */
const COLORS_CSS_SELECTOR = 'link[href*=\'//theme/colors.css\']';
let documentInstance = null;
// 
class ColorChangeUpdater {
    listenerId_ = null;
    root_;
    // 
    constructor(root) {
        assert(documentInstance === null || root !== document);
        this.root_ = root;
    }
    /**
     * Starts listening for ColorProvider changes from the browser and updates the
     * `root_` whenever changes occur.
     */
    start() {
        if (this.listenerId_ !== null) {
            return;
        }
        this.listenerId_ = BrowserProxy.getInstance()
            .callbackRouter.onColorProviderChanged.addListener(this.onColorProviderChanged.bind(this));
    }
    // TODO(dpapad): Figure out how to properly trigger
    // `callbackRouter.onColorProviderChanged` listeners from tests and make this
    // method private.
    async onColorProviderChanged() {
        await this.refreshColorsCss();
        // 
    }
    /**
     * Forces `root_` to refresh its colors.css stylesheet. This is used to
     * fetch an updated stylesheet when the ColorProvider associated with the
     * WebUI has changed.
     * @return A promise which resolves to true once the new colors are loaded and
     *     installed into the DOM. In the case of an error returns false. When a
     *     new colors.css is loaded, this will always freshly query the existing
     *     colors.css, allowing multiple calls to successfully remove existing,
     *     outdated CSS.
     */
    async refreshColorsCss() {
        const colorCssNode = this.root_.querySelector(COLORS_CSS_SELECTOR);
        if (!colorCssNode) {
            return false;
        }
        const href = colorCssNode.getAttribute('href');
        if (!href) {
            return false;
        }
        const hrefURL = new URL(href, location.href);
        const params = new URLSearchParams(hrefURL.search);
        params.set('version', new Date().getTime().toString());
        const newHref = `${hrefURL.origin}${hrefURL.pathname}?${params.toString()}`;
        // A flickering effect may take place when setting the href property of
        // the existing color css node with a new value. In order to avoid
        // flickering, we create a new link element and once it is loaded we
        // remove the old one. See crbug.com/1365320 for additional details.
        const newColorsCssLink = document.createElement('link');
        newColorsCssLink.setAttribute('href', newHref);
        newColorsCssLink.rel = 'stylesheet';
        newColorsCssLink.type = 'text/css';
        const newColorsLoaded = new Promise(resolve => {
            newColorsCssLink.onload = resolve;
        });
        if (this.root_ === document) {
            document.getElementsByTagName('body')[0].appendChild(newColorsCssLink);
        }
        else {
            this.root_.appendChild(newColorsCssLink);
        }
        await newColorsLoaded;
        const oldColorCssNode = document.querySelector(COLORS_CSS_SELECTOR);
        if (oldColorCssNode) {
            oldColorCssNode.remove();
        }
        return true;
    }
    static forDocument() {
        return documentInstance ||
            (documentInstance = new ColorChangeUpdater(document));
    }
}

let instance = null;
function getCss() {
    return instance || (instance = [...[], css `:host{--cr-searchbox-results-search-icon-size:16px;--cr-searchbox-icon-container-size:28px;--cr-searchbox-match-icon-container-background-fallback:var(--color-omnibox-results-background-hovered);--cr-searchbox-match-padding:6px;--cr-searchbox-match-padding-inline-start:16px;--cr-searchbox-match-text-padding-inline-start:12px;--cr-searchbox-min-width:75%;--cr-searchbox-results-action-chip-height:26px;--cr-searchbox-results-margin-bottom:0;--cr-searchbox-width:100%;--cr-searchbox-match-font-weight:700;font-size:14.3px;--color-searchbox-answer-icon-background:var(--color-omnibox-answer-icon-g-m3-background);--color-searchbox-answer-icon-foreground:var(--color-omnibox-answer-icon-g-m3-foreground);--color-searchbox-results-action-chip-focus-outline:var(--color-omnibox-results-focus-indicator);--color-searchbox-results-action-chip:var(--color-omnibox-results-button-border);--color-searchbox-results-action-chip-icon:var(--color-omnibox-results-button-icon);--color-searchbox-results-action-chip-icon-selected:var(--color-omnibox-results-button-icon-selected);--color-searchbox-results-background:var(--color-omnibox-results-background);--color-searchbox-results-background-hovered:var(--color-omnibox-results-background-hovered);--color-searchbox-results-background-selected:var(--color-omnibox-results-background-selected);--color-searchbox-results-button-hover:var(--color-omnibox-results-button-ink-drop-row-hovered);--color-searchbox-results-button-selected-hover:var(--color-omnibox-results-button-ink-drop-row-selected);--color-searchbox-results-dim-selected:var(--color-omnibox-results-text-dimmed-selected);--color-searchbox-results-focus-indicator:var(--color-omnibox-results-focus-indicator);--color-searchbox-results-foreground:var(--color-omnibox-text);--color-searchbox-results-foreground-dimmed:var(--color-omnibox-results-text-dimmed);--color-searchbox-results-foreground-selected:var(--color-omnibox-results-text-selected);--color-searchbox-results-icon:var(--color-omnibox-results-icon);--color-searchbox-results-icon-container-background:unset;--color-searchbox-results-icon-focused-outline:var(--color-omnibox-results-focus-indicator);--color-searchbox-results-icon-selected:var(--color-omnibox-results-icon-selected);--color-searchbox-results-starter-pack-icon:var(--color-omnibox-results-starter-pack-icon);--color-searchbox-results-typed-prefix:unset;--color-searchbox-results-url-selected:var(--color-omnibox-results-url-selected);--color-searchbox-results-url:var(--color-omnibox-results-url);--color-searchbox-search-icon-background:var(--color-omnibox-results-icon)}:host([can-show-secondary-side][has-secondary-side]){--cr-searchbox-secondary-side-display:block}:host([is-debug]){display:block;width:50%}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-searchbox-dropdown id="matches" .result="${this.result_}"
    ?can-show-secondary-side="${this.canShowSecondarySide}"
    ?has-secondary-side="${this.hasSecondarySide}"
    @has-secondary-side-changed="${this.onHasSecondarySideChanged_}"
    @dom-change="${this.onResultRepaint_}">
</cr-searchbox-dropdown>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// 675px ~= 449px (--cr-realbox-primary-side-min-width) * 1.5 + some margin.
const canShowSecondarySideMediaQueryList = window.matchMedia('(min-width: 675px)');
// Displays the autocomplete matches in the autocomplete result.
class OmniboxPopupAppElement extends CrLitElement {
    static get is() {
        return 'omnibox-popup-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Whether the secondary side can be shown based on the feature state and
             * the width available to the dropdown.
             */
            canShowSecondarySide: {
                type: Boolean,
                reflect: true,
            },
            /*
             * Whether the secondary side is currently available to be shown.
             */
            hasSecondarySide: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether the app is in debug mode.
             */
            isDebug: {
                type: Boolean,
                reflect: true,
            },
            result_: { type: Object },
        };
    }
    #canShowSecondarySide_accessor_storage = canShowSecondarySideMediaQueryList.matches;
    get canShowSecondarySide() { return this.#canShowSecondarySide_accessor_storage; }
    set canShowSecondarySide(value) { this.#canShowSecondarySide_accessor_storage = value; }
    #hasSecondarySide_accessor_storage = false;
    get hasSecondarySide() { return this.#hasSecondarySide_accessor_storage; }
    set hasSecondarySide(value) { this.#hasSecondarySide_accessor_storage = value; }
    #isDebug_accessor_storage = false;
    get isDebug() { return this.#isDebug_accessor_storage; }
    set isDebug(value) { this.#isDebug_accessor_storage = value; }
    #result__accessor_storage = null;
    get result_() { return this.#result__accessor_storage; }
    set result_(value) { this.#result__accessor_storage = value; }
    callbackRouter_;
    autocompleteResultChangedListenerId_ = null;
    selectionChangedListenerId_ = null;
    eventTracker_ = new EventTracker();
    constructor() {
        super();
        this.callbackRouter_ = SearchboxBrowserProxy.getInstance().callbackRouter;
        this.isDebug = new URLSearchParams(window.location.search).has('debug');
        ColorChangeUpdater.forDocument().start();
    }
    connectedCallback() {
        super.connectedCallback();
        this.autocompleteResultChangedListenerId_ =
            this.callbackRouter_.autocompleteResultChanged.addListener(this.onAutocompleteResultChanged_.bind(this));
        this.selectionChangedListenerId_ =
            this.callbackRouter_.updateSelection.addListener(this.onUpdateSelection_.bind(this));
        canShowSecondarySideMediaQueryList.addEventListener('change', this.onCanShowSecondarySideChanged_.bind(this));
        if (!this.isDebug) {
            this.eventTracker_.add(document.documentElement, 'contextmenu', (e) => {
                e.preventDefault();
            });
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
        assert(this.autocompleteResultChangedListenerId_);
        this.callbackRouter_.removeListener(this.autocompleteResultChangedListenerId_);
        assert(this.selectionChangedListenerId_);
        this.callbackRouter_.removeListener(this.selectionChangedListenerId_);
        canShowSecondarySideMediaQueryList.removeEventListener('change', this.onCanShowSecondarySideChanged_.bind(this));
    }
    onCanShowSecondarySideChanged_(e) {
        this.canShowSecondarySide = e.matches;
    }
    onAutocompleteResultChanged_(result) {
        // Skip empty results. Otherwise, blurring/closing the omnibox would clear
        // the results in the debug UI.
        if (this.isDebug && !result.matches.length) {
            return;
        }
        this.result_ = result;
        if (result.matches[0]?.allowedToBeDefaultMatch) {
            this.$.matches.selectFirst();
        }
        else if (this.$.matches.selectedMatchIndex >= result.matches.length) {
            this.$.matches.unselect();
        }
    }
    onResultRepaint_() {
        const metricsReporter = MetricsReporterImpl.getInstance();
        metricsReporter.measure('ResultChanged')
            .then(duration => metricsReporter.umaReportTime('WebUIOmnibox.ResultChangedToRepaintLatency.ToPaint', duration))
            .then(() => metricsReporter.clearMark('ResultChanged'))
            // Ignore silently if mark 'ResultChanged' is missing.
            .catch(() => { });
    }
    onUpdateSelection_(oldSelection, selection) {
        this.$.matches.updateSelection(oldSelection, selection);
    }
    onHasSecondarySideChanged_(e) {
        this.hasSecondarySide = e.detail.value;
    }
}
customElements.define(OmniboxPopupAppElement.is, OmniboxPopupAppElement);
//# sourceMappingURL=omnibox_popup.rollup.js.map
