// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { assert } from '//resources/js/assert.js';
import { BrowserProxyImpl } from './browser_proxy.js';
/**
 * @fileoverview A browser proxy for receiving the viewport screenshot from the
 * browser.
 */
let instance = null;
export class ScreenshotBitmapBrowserProxyImpl {
    screenshot;
    screenshotListenerId;
    callbacks = [];
    constructor() {
        this.screenshotListenerId =
            BrowserProxyImpl.getInstance()
                .callbackRouter.screenshotDataReceived.addListener(this.screenshotDataReceived.bind(this));
    }
    static getInstance() {
        return instance || (instance = new ScreenshotBitmapBrowserProxyImpl());
    }
    static setInstance(obj) {
        instance = obj;
    }
    fetchScreenshot(callback) {
        // Store the callback for calling if the screenshot updates or when the
        // screenshot is ready if the below check fails.
        this.callbacks.push(callback);
        if (this.screenshot) {
            // We need to make a new bitmap because each canvas takes ownership of the
            // bitmap, so it cannot be drawn to multiple HTMLCanvasElement.
            createImageBitmap(this.screenshot).then((bitmap) => {
                callback(bitmap);
            });
            return;
        }
    }
    async screenshotDataReceived(screenshotData) {
        const data = screenshotData.pixelData;
        // TODO(b/334185985): This occurs when the browser failed to allocate the
        // memory for the pixels. Handle this case.
        if (data.invalidBuffer) {
            return;
        }
        // Pull the pixel data into a Uint8ClampedArray.
        let pixelData;
        if (Array.isArray(data.bytes)) {
            pixelData = new Uint8ClampedArray(data.bytes);
        }
        else {
            // If the buffer is not invalid or an array, it must be shared memory.
            assert(data.sharedMemory);
            const sharedMemory = data.sharedMemory;
            const { buffer, result } = sharedMemory.bufferHandle.mapBuffer(0, sharedMemory.size);
            assert(result === Mojo.RESULT_OK);
            pixelData = new Uint8ClampedArray(buffer);
        }
        const imageWidth = screenshotData.imageInfo.width;
        const imageHeight = screenshotData.imageInfo.height;
        // Put our screenshot into ImageData object so it can be rendered in a
        // Canvas.
        const imageData = new ImageData(pixelData, imageWidth, imageHeight);
        const imageBitmap = await createImageBitmap(imageData);
        // Cache the bitmap for future requests
        this.screenshot = imageBitmap;
        // Send the screenshot to all the callbacks.
        for (const callback of this.callbacks) {
            // We need to make a new bitmap because each canvas takes ownership of the
            // bitmap, so it cannot be drawn to multiple HTMLCanvasElement.
            createImageBitmap(this.screenshot).then((bitmap) => {
                callback(bitmap);
            });
        }
    }
}
