// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { GLIF_HEX_COLORS } from './color_utils.js';
import { getCss } from './overlay_border_glow.css.js';
import { getHtml } from './overlay_border_glow.html.js';
/*
 * Element responsible for rendering the border glow as a replacement for the
 * shimmer.
 */
export class OverlayBorderGlowElement extends CrLitElement {
    static get is() {
        return 'overlay-border-glow';
    }
    static get properties() {
        return {
            isFadingOut: {
                type: Boolean,
                reflect: true,
            },
            isFadingIn: {
                type: Boolean,
                reflect: true,
            },
            selectionOverlayRect: { type: DOMRect },
        };
    }
    #isFadingOut_accessor_storage = false;
    // Whether the border glow is fading out.
    get isFadingOut() { return this.#isFadingOut_accessor_storage; }
    set isFadingOut(value) { this.#isFadingOut_accessor_storage = value; }
    #isFadingIn_accessor_storage = false;
    // Whether the border glow is fading in.
    get isFadingIn() { return this.#isFadingIn_accessor_storage; }
    set isFadingIn(value) { this.#isFadingIn_accessor_storage = value; }
    #selectionOverlayRect_accessor_storage = new DOMRect(0, 0, 0, 0);
    // The bounding box of the selection overlay.
    get selectionOverlayRect() { return this.#selectionOverlayRect_accessor_storage; }
    set selectionOverlayRect(value) { this.#selectionOverlayRect_accessor_storage = value; }
    static get styles() {
        return getCss();
    }
    getGradientColorStyles() {
        const styles = [
            `--gradient-blue: ${GLIF_HEX_COLORS.blue}`,
            `--gradient-red: ${GLIF_HEX_COLORS.red}`,
            `--gradient-yellow: ${GLIF_HEX_COLORS.yellow}`,
            `--gradient-green: ${GLIF_HEX_COLORS.green}`,
        ];
        return styles.join(';');
    }
    getBoundsStyles() {
        /* Height and width must be larger than the diagonal of the viewport,
        in order to prevent gaps at the corners while rotating. */
        const longestSide = Math.max(this.selectionOverlayRect.width, this.selectionOverlayRect.height);
        return `width: ${longestSide * 1.5}px; height: ${longestSide * 1.5}px`;
    }
    handleGestureStart() {
        this.isFadingOut = true;
    }
    handlePostSelectionUpdated() {
        this.isFadingOut = true;
    }
    handleClearSelection() {
        this.isFadingOut = false;
        this.isFadingIn = true;
    }
    render() {
        return getHtml.bind(this)();
    }
}
customElements.define(OverlayBorderGlowElement.is, OverlayBorderGlowElement);
// Register the custom property for the gradient mask opacity middle value.
// Custom properties are ignored by the browser in shadow DOMs, so need to
// register them globally here. Additionally, the property can only by
// registered once per document, so this must be done in the main window, rather
// than in the class itself.
window.CSS.registerProperty({
    name: '--gradient-mask-opacity-middle-val',
    syntax: '<number>',
    inherits: false,
    initialValue: '0',
});
