// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var ResponseStopCause;
(function (ResponseStopCause) {
    /** User cancelled response. */
    ResponseStopCause[ResponseStopCause["USER"] = 0] = "USER";
    /** System cancelled response for another reason. */
    ResponseStopCause[ResponseStopCause["OTHER"] = 1] = "OTHER";
})(ResponseStopCause || (ResponseStopCause = {}));
/**
 * A panel can be in one of these three states.
 */
var PanelStateKind;
(function (PanelStateKind) {
    /** Not shown. This is the initial state. */
    PanelStateKind[PanelStateKind["HIDDEN"] = 0] = "HIDDEN";
    /** @deprecated Use DETACHED instead. */
    PanelStateKind[PanelStateKind["FLOATING"] = 1] = "FLOATING";
    /** A floating window detached from any Chrome window. */
    PanelStateKind[PanelStateKind["DETACHED"] = 1] = "DETACHED";
    /** @deprecated Use ATTACHED instead.*/
    PanelStateKind[PanelStateKind["DOCKED"] = 2] = "DOCKED";
    /** Attached to a Chrome window. */
    PanelStateKind[PanelStateKind["ATTACHED"] = 2] = "ATTACHED";
})(PanelStateKind || (PanelStateKind = {}));
/** Reason why the web client could not initialize. */
var WebClientInitializeErrorReason;
(function (WebClientInitializeErrorReason) {
    /**
     * Unknown reason. The user can manually retry loading, which reloads the
     * entire webview.
     */
    WebClientInitializeErrorReason[WebClientInitializeErrorReason["UNKNOWN"] = 0] = "UNKNOWN";
    /** This list will be expanded later. */
})(WebClientInitializeErrorReason || (WebClientInitializeErrorReason = {}));
/** Reason for failure while acting in the focused tab. */
var ActInFocusedTabErrorReason;
(function (ActInFocusedTabErrorReason) {
    ActInFocusedTabErrorReason[ActInFocusedTabErrorReason["UNKNOWN"] = 0] = "UNKNOWN";
    /** Context could not be gathered after acting. */
    ActInFocusedTabErrorReason[ActInFocusedTabErrorReason["GET_CONTEXT_FAILED"] = 1] = "GET_CONTEXT_FAILED";
    /** The action proto is invalid. */
    ActInFocusedTabErrorReason[ActInFocusedTabErrorReason["INVALID_ACTION_PROTO"] = 2] = "INVALID_ACTION_PROTO";
    /** Action target is not found. */
    ActInFocusedTabErrorReason[ActInFocusedTabErrorReason["TARGET_NOT_FOUND"] = 3] = "TARGET_NOT_FOUND";
    /** Failed to start a new task. */
    ActInFocusedTabErrorReason[ActInFocusedTabErrorReason["FAILED_TO_START_TASK"] = 4] = "FAILED_TO_START_TASK";
})(ActInFocusedTabErrorReason || (ActInFocusedTabErrorReason = {}));
/**
 * Top-level views of the glic web client.
 */
var ClientView;
(function (ClientView) {
    ClientView["ACTUATION"] = "actuation";
    ClientView["CONVERSATION"] = "conversation";
})(ClientView || (ClientView = {}));
/**
 * Describes how long the user grants the actor with the permission to actuate.
 * Used when the actor is to actuate with sensitive data, such as entering
 * payment information or login credentials.
 */
var UserGrantedPermissionDuration;
(function (UserGrantedPermissionDuration) {
    // The user only grants a one-time permission. The user will be asked again.
    // This is the default behavior.
    UserGrantedPermissionDuration[UserGrantedPermissionDuration["ONE_TIME"] = 0] = "ONE_TIME";
    // The user grants a permission to always allow the actor to actuate with
    // sensitive data. The persistence of this permission is defined differently
    // for different features.
    UserGrantedPermissionDuration[UserGrantedPermissionDuration["ALWAYS_ALLOW"] = 1] = "ALWAYS_ALLOW";
})(UserGrantedPermissionDuration || (UserGrantedPermissionDuration = {}));
///////////////////////////////////////////////////////////////////////////////
/// BEGIN_GENERATED - DO NOT MODIFY BELOW
// This block is generated by
// chrome/browser/resources/glic/glic_api_impl/generate.py
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// The type of user input reaction.
var MetricUserInputReactionType;
(function (MetricUserInputReactionType) {
    // An unknown reaction type.
    MetricUserInputReactionType[MetricUserInputReactionType["UNKNOWN"] = 0] = "UNKNOWN";
    // A canned reaction which can be presented without communication with the
    // server.
    MetricUserInputReactionType[MetricUserInputReactionType["CANNED"] = 1] = "CANNED";
    // A reaction which requires some generic modeling to produce.
    MetricUserInputReactionType[MetricUserInputReactionType["MODEL"] = 2] = "MODEL";
})(MetricUserInputReactionType || (MetricUserInputReactionType = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Reason for failure while acting.
var PerformActionsErrorReason;
(function (PerformActionsErrorReason) {
    PerformActionsErrorReason[PerformActionsErrorReason["UNKNOWN"] = 0] = "UNKNOWN";
    // The serialized actions proto failed to parse.
    PerformActionsErrorReason[PerformActionsErrorReason["INVALID_ACTION_PROTO"] = 1] = "INVALID_ACTION_PROTO";
})(PerformActionsErrorReason || (PerformActionsErrorReason = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Reason for failure when creating an actor task.
var CreateTaskErrorReason;
(function (CreateTaskErrorReason) {
    CreateTaskErrorReason[CreateTaskErrorReason["UNKNOWN"] = 0] = "UNKNOWN";
    // The host does not support the actor task system.
    CreateTaskErrorReason[CreateTaskErrorReason["TASK_SYSTEM_UNAVAILABLE"] = 1] = "TASK_SYSTEM_UNAVAILABLE";
})(CreateTaskErrorReason || (CreateTaskErrorReason = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// The state of an actor task.
var ActorTaskState;
(function (ActorTaskState) {
    ActorTaskState[ActorTaskState["UNKNOWN"] = 0] = "UNKNOWN";
    // The actor task is idle and waiting for the next action instruction.
    ActorTaskState[ActorTaskState["IDLE"] = 1] = "IDLE";
    // The actor task is performing an action.
    ActorTaskState[ActorTaskState["ACTING"] = 2] = "ACTING";
    // The actor task is paused and waiting to be resumed or stopped.
    ActorTaskState[ActorTaskState["PAUSED"] = 3] = "PAUSED";
    // The actor task is stopped and going away.
    ActorTaskState[ActorTaskState["STOPPED"] = 4] = "STOPPED";
})(ActorTaskState || (ActorTaskState = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// The reason/source of why an actor task was paused.
var ActorTaskPauseReason;
(function (ActorTaskPauseReason) {
    // Actor task was paused by the model.
    ActorTaskPauseReason[ActorTaskPauseReason["PAUSED_BY_MODEL"] = 0] = "PAUSED_BY_MODEL";
    // Actor task was puased by the user.
    ActorTaskPauseReason[ActorTaskPauseReason["PAUSED_BY_USER"] = 1] = "PAUSED_BY_USER";
})(ActorTaskPauseReason || (ActorTaskPauseReason = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// The reason/source of why an actor task was stopped.
var ActorTaskStopReason;
(function (ActorTaskStopReason) {
    // Actor task is complete.
    ActorTaskStopReason[ActorTaskStopReason["TASK_COMPLETE"] = 0] = "TASK_COMPLETE";
    // Actor task was stopped by the user.
    ActorTaskStopReason[ActorTaskStopReason["STOPPED_BY_USER"] = 1] = "STOPPED_BY_USER";
})(ActorTaskStopReason || (ActorTaskStopReason = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Reason why capturing desktop screenshot failed. NOTE: This may be extended in
// the future so avoid using complete switches on the currently used enum
// values.
var CaptureScreenshotErrorReason;
(function (CaptureScreenshotErrorReason) {
    // Screen capture or frame encoding failure.
    CaptureScreenshotErrorReason[CaptureScreenshotErrorReason["UNKNOWN"] = 0] = "UNKNOWN";
    // Screen capture requested but already in progress of serving another
    // request.
    CaptureScreenshotErrorReason[CaptureScreenshotErrorReason["SCREEN_CAPTURE_REQUEST_THROTTLED"] = 1] = "SCREEN_CAPTURE_REQUEST_THROTTLED";
    // User declined screen capture dialog before taking a screenshot.
    CaptureScreenshotErrorReason[CaptureScreenshotErrorReason["USER_CANCELLED_SCREEN_PICKER_DIALOG"] = 2] = "USER_CANCELLED_SCREEN_PICKER_DIALOG";
})(CaptureScreenshotErrorReason || (CaptureScreenshotErrorReason = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Reason why scrollTo() failed.
var ScrollToErrorReason;
(function (ScrollToErrorReason) {
    // Invalid params were provided to scrollTo(), or the browser doesn't support
    // scrollTo() yet.
    ScrollToErrorReason[ScrollToErrorReason["NOT_SUPPORTED"] = 0] = "NOT_SUPPORTED";
    // scrollTo() was called again before this call finished processing.
    ScrollToErrorReason[ScrollToErrorReason["NEWER_SCROLL_TO_CALL"] = 1] = "NEWER_SCROLL_TO_CALL";
    // There is no tab currently in focus.
    ScrollToErrorReason[ScrollToErrorReason["NO_FOCUSED_TAB"] = 2] = "NO_FOCUSED_TAB";
    // The input selector did not match any content in the document or a given
    // range.
    ScrollToErrorReason[ScrollToErrorReason["NO_MATCH_FOUND"] = 3] = "NO_MATCH_FOUND";
    // The currently focused tab changed or navigated while processing the
    // scrollTo() call.
    ScrollToErrorReason[ScrollToErrorReason["FOCUSED_TAB_CHANGED_OR_NAVIGATED"] = 4] = "FOCUSED_TAB_CHANGED_OR_NAVIGATED";
    // The document_id or url provided doesn't match the active document in the
    // primary main frame of the currently focused tab. The document may have been
    // navigated away, may not currently be in focus, or may not be in a primary
    // main frame (we don't currently support iframes).
    ScrollToErrorReason[ScrollToErrorReason["NO_MATCHING_DOCUMENT"] = 5] = "NO_MATCHING_DOCUMENT";
    // The search range starting from DOMNodeId did not result in a valid range.
    ScrollToErrorReason[ScrollToErrorReason["SEARCH_RANGE_INVALID"] = 6] = "SEARCH_RANGE_INVALID";
    // Page context access is disabled.
    ScrollToErrorReason[ScrollToErrorReason["TAB_CONTEXT_PERMISSION_DISABLED"] = 7] = "TAB_CONTEXT_PERMISSION_DISABLED";
    // The web client requested to drop the highlight via
    // `dropScrollToHighlight()`.
    ScrollToErrorReason[ScrollToErrorReason["DROPPED_BY_WEB_CLIENT"] = 8] = "DROPPED_BY_WEB_CLIENT";
})(ScrollToErrorReason || (ScrollToErrorReason = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Reason for failure when switching a conversation.
var SwitchConversationErrorReason;
(function (SwitchConversationErrorReason) {
    SwitchConversationErrorReason[SwitchConversationErrorReason["UNKNOWN"] = 0] = "UNKNOWN";
})(SwitchConversationErrorReason || (SwitchConversationErrorReason = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Reason for failure when registering a conversation.
var RegisterConversationErrorReason;
(function (RegisterConversationErrorReason) {
    RegisterConversationErrorReason[RegisterConversationErrorReason["UNKNOWN"] = 0] = "UNKNOWN";
    // The instance already has a conversation ID.
    RegisterConversationErrorReason[RegisterConversationErrorReason["INSTANCE_ALREADY_HAS_CONVERSATION_ID"] = 1] = "INSTANCE_ALREADY_HAS_CONVERSATION_ID";
})(RegisterConversationErrorReason || (RegisterConversationErrorReason = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Entry points that can trigger the opening of the panel.
var InvocationSource;
(function (InvocationSource) {
    // Button in the OS.
    InvocationSource[InvocationSource["OS_BUTTON"] = 0] = "OS_BUTTON";
    // Menu from button in the OS.
    InvocationSource[InvocationSource["OS_BUTTON_MENU"] = 1] = "OS_BUTTON_MENU";
    // OS-level hotkey.
    InvocationSource[InvocationSource["OS_HOTKEY"] = 2] = "OS_HOTKEY";
    // Button in top-chrome.
    InvocationSource[InvocationSource["TOP_CHROME_BUTTON"] = 3] = "TOP_CHROME_BUTTON";
    // First run experience.
    InvocationSource[InvocationSource["FRE"] = 4] = "FRE";
    // From the profile picker.
    InvocationSource[InvocationSource["PROFILE_PICKER"] = 5] = "PROFILE_PICKER";
    // From tab strip nudge.
    InvocationSource[InvocationSource["NUDGE"] = 6] = "NUDGE";
    // From 3-dot menu.
    InvocationSource[InvocationSource["THREE_DOTS_MENU"] = 7] = "THREE_DOTS_MENU";
    // An unsupported/unknown source.
    InvocationSource[InvocationSource["UNSUPPORTED"] = 8] = "UNSUPPORTED";
    // From the What's New page.
    InvocationSource[InvocationSource["WHATS_NEW"] = 9] = "WHATS_NEW";
    // User clicked the sign-in button and signed in.
    InvocationSource[InvocationSource["AFTER_SIGN_IN"] = 10] = "AFTER_SIGN_IN";
    // User shared a tab (e.g. via its context menu).
    InvocationSource[InvocationSource["SHARED_TAB"] = 11] = "SHARED_TAB";
    // From the actor task icon.
    InvocationSource[InvocationSource["ACTOR_TASK_ICON"] = 12] = "ACTOR_TASK_ICON";
    // User shared an image via the context menu.
    InvocationSource[InvocationSource["SHARED_IMAGE"] = 13] = "SHARED_IMAGE";
})(InvocationSource || (InvocationSource = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Web client's operation modes.
var WebClientMode;
(function (WebClientMode) {
    // Text operation mode.
    WebClientMode[WebClientMode["TEXT"] = 0] = "TEXT";
    // Audio operation mode.
    WebClientMode[WebClientMode["AUDIO"] = 1] = "AUDIO";
})(WebClientMode || (WebClientMode = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Web client's operation model.
var WebClientModel;
(function (WebClientModel) {
    // Default model.
    WebClientModel[WebClientModel["DEFAULT"] = 0] = "DEFAULT";
    // Actor operation mode.
    WebClientModel[WebClientModel["ACTOR"] = 1] = "ACTOR";
})(WebClientModel || (WebClientModel = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Fields of interest from the Glic settings page.
var SettingsPageField;
(function (SettingsPageField) {
    // The OS hotkey configuration field.
    SettingsPageField[SettingsPageField["OS_HOTKEY"] = 1] = "OS_HOTKEY";
    // The OS entrypoint enabling field.
    SettingsPageField[SettingsPageField["OS_ENTRYPOINT_TOGGLE"] = 2] = "OS_ENTRYPOINT_TOGGLE";
})(SettingsPageField || (SettingsPageField = {}));
///////////////////////////////////////////////
// WARNING - GENERATED FROM MOJOM, DO NOT EDIT.
// Describes the capability of the glic host.
var HostCapability;
(function (HostCapability) {
    // Glic host supports scrollTo() with PDF documents.
    HostCapability[HostCapability["SCROLL_TO_PDF"] = 0] = "SCROLL_TO_PDF";
    // Glic host will reset panel size and location on open.
    HostCapability[HostCapability["RESET_SIZE_AND_LOCATION_ON_OPEN"] = 1] = "RESET_SIZE_AND_LOCATION_ON_OPEN";
    // The glic host's getModelQualityClientId() is enabled and can be called
    // safely.
    HostCapability[HostCapability["GET_MODEL_QUALITY_CLIENT_ID"] = 2] = "GET_MODEL_QUALITY_CLIENT_ID";
    // Glic is in multi-instance mode.
    HostCapability[HostCapability["MULTI_INSTANCE"] = 3] = "MULTI_INSTANCE";
})(HostCapability || (HostCapability = {}));
/// END_GENERATED - DO NOT MODIFY ABOVE
///////////////////////////////////////////////////////////////////////////////

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class ObservableSubscription {
    observer;
    onUnsubscribe;
    constructor(observer, onUnsubscribe) {
        this.observer = observer;
        this.onUnsubscribe = onUnsubscribe;
    }
    unsubscribe() {
        this.onUnsubscribe(this);
    }
}
/**
 * A base class for observables that handles subscribers.
 */
class ObservableBase {
    subscribers = new Set();
    state_ = 'active';
    errorValue;
    next(value) {
        switch (this.state_) {
            case 'active':
                break;
            case 'complete':
            case 'error':
                throw new Error('Observable is not active');
        }
        this.subscribers.forEach((sub) => {
            // Ignore if removed since forEach was called.
            if (this.subscribers.has(sub)) {
                try {
                    sub.observer.next?.(value);
                }
                catch (e) {
                    console.warn(e);
                }
            }
        });
    }
    error(e) {
        switch (this.state_) {
            case 'active':
                this.state_ = 'error';
                this.errorValue = e;
                break;
            case 'complete':
            case 'error':
                throw new Error('Observable is not active');
        }
        let loggedWarning = false;
        const hadSubscribers = this.hasActiveSubscription();
        this.subscribers.forEach((sub) => {
            // Ignore if removed since forEach was called.
            if (this.subscribers.has(sub)) {
                if (sub.observer.error) {
                    try {
                        sub.observer.error(e);
                    }
                    catch (e) {
                        console.warn(e);
                    }
                }
                else {
                    if (!loggedWarning) {
                        console.warn('unhandled error: ', e);
                        loggedWarning = true;
                    }
                }
            }
        });
        this.subscribers.clear();
        if (hadSubscribers) {
            this.activeSubscriptionChanged(false);
        }
    }
    complete() {
        switch (this.state_) {
            case 'active':
                this.state_ = 'complete';
                break;
            case 'complete':
            case 'error':
                throw new Error('Observable is not active');
        }
        const hadSubscribers = this.hasActiveSubscription();
        this.subscribers.forEach((sub) => {
            // Ignore if removed since forEach was called.
            if (this.subscribers.has(sub)) {
                try {
                    sub.observer.complete?.();
                }
                catch (e) {
                    console.warn(e);
                }
            }
        });
        this.subscribers.clear();
        if (hadSubscribers) {
            this.activeSubscriptionChanged(false);
        }
    }
    isStopped() {
        return this.state_ !== 'active';
    }
    subscribe(changeOrObserver) {
        if (typeof changeOrObserver === 'function') {
            return this.subscribeObserver({
                next: (value) => {
                    changeOrObserver(value);
                },
            });
        }
        else {
            return this.subscribeObserver(changeOrObserver);
        }
    }
    /**
     * Subscribe to changes with an Observer.
     * This API was added in later, and provided as a separate function for Glic
     * API discovery purposes.
     */
    subscribeObserver(observer) {
        switch (this.state_) {
            case 'active':
                break;
            case 'complete':
                observer.complete?.();
                return { unsubscribe: () => { } };
            case 'error':
                observer.error?.(this.errorValue);
                return { unsubscribe: () => { } };
        }
        const newSub = new ObservableSubscription(observer, this.onUnsubscribe.bind(this));
        if (this.subscribers.size === 0) {
            this.activeSubscriptionChanged(true);
        }
        this.subscribers.add(newSub);
        this.subscriberAdded(newSub);
        return newSub;
    }
    onUnsubscribe(sub) {
        if (!this.subscribers) {
            return;
        }
        if (this.subscribers.size === 0) {
            return;
        }
        this.subscribers.delete(sub);
        if (this.subscribers.size === 0) {
            this.activeSubscriptionChanged(false);
        }
    }
    subscriberAdded(_sub) { }
    activeSubscriptionChanged(_hasActiveSubscription) { }
    hasActiveSubscription() {
        return this.subscribers.size > 0;
    }
}
/**
 * A simple observable with no memory of previous values.
 */
class Subject extends ObservableBase {
    next(value) {
        super.next(value);
    }
}
/**
 * A observable value that can change over time. If value is initialized, sends
 * it to new subscribers upon subscribe().
 */
class ObservableValue extends Subject {
    isSet;
    value;
    hasActiveSubscriptionCallback;
    constructor(isSet, value, hasActiveSubscriptionCallback) {
        super();
        this.isSet = isSet;
        this.value = value;
        this.hasActiveSubscriptionCallback = hasActiveSubscriptionCallback;
    }
    /**
     * Create an ObservableValue which has an initial value. Optionally a
     * `hasActiveSubscriptionCallback` can be added which will be called with
     * `true` when this observable has its first subscriber and `false` when there
     * are no longer any subscribers to this observable.
     */
    static withValue(value, hasActiveSubscriptionCallback) {
        return new ObservableValue(true, value, hasActiveSubscriptionCallback);
    }
    /**
     * Create an ObservableValue which has no initial value. Subscribers will not
     * be called until after assignAndSignal() is called the first time.
     * Optionally a `hasActiveSubscriptionCallback` can be added which will be
     * called with `true` when this observable has its first subscriber and
     * false` when there are no longer any subscribers to this observable.
     */
    static withNoValue(hasActiveSubscriptionCallback) {
        return new ObservableValue(false, undefined, hasActiveSubscriptionCallback);
    }
    /**
     * Assigns a new value to the ObservableValue and signals all subscribers.
     * Does nothing if the value is unchanged.
     */
    assignAndSignal(v, force = false) {
        if (this.isStopped()) {
            throw new Error('ObservableValue is not active');
        }
        const send = !this.isSet || this.value !== v || force;
        this.isSet = true;
        this.value = v;
        if (!send) {
            return;
        }
        super.next(v);
    }
    /** Returns the current value, or undefined if not initialized. */
    getCurrentValue() {
        return this.value;
    }
    /**
     * Asynchronously waits until the ObservableValue's current value satisfies a
     * given criteria.
     */
    async waitUntil(criteria) {
        const { promise, resolve, reject } = Promise.withResolvers();
        const sub = this.subscribe({
            next(newValue) {
                if (criteria(newValue)) {
                    resolve(newValue);
                }
            },
            error: reject,
            complete() {
                reject(new Error('Observable completed'));
            },
        });
        let resultValue;
        try {
            resultValue = await promise;
        }
        finally {
            sub.unsubscribe();
        }
        return resultValue;
    }
    subscriberAdded(sub) {
        if (this.isSet) {
            sub.observer.next?.(this.value);
        }
    }
    activeSubscriptionChanged(hasActiveSubscription) {
        super.activeSubscriptionChanged(hasActiveSubscription);
        this.hasActiveSubscriptionCallback?.(hasActiveSubscription);
    }
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* eslint-disable-next-line @typescript-eslint/naming-convention */
// Helper function to shallow-copy an object and replace some properties.
// Useful to convert from these private types to public types. This will fail to
// compile if a property is missed.
function replaceProperties(original, replacements) {
    return Object.assign(Object.assign({}, original), replacements);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// LINT.IfChange(ApiRequestType)
// New values here must be added to histograms.xml and to enums.xml.
(() => {
    const result = {
        WebClientCreated: 1,
        WebClientInitialized: 2,
        CreateTab: 3,
        OpenGlicSettingsPage: 4,
        ClosePanel: 5,
        ClosePanelAndShutdown: 6,
        ShowProfilePicker: 7,
        GetModelQualityClientId: 8,
        GetContextFromFocusedTab: 9,
        GetContextFromTab: 10,
        GetContextForActorFromTab: 11,
        SetMaximumNumberOfPinnedTabs: 12,
        StopActorTask: 13,
        PauseActorTask: 14,
        ResumeActorTask: 15,
        CaptureScreenshot: 16,
        ResizeWindow: 17,
        EnableDragResize: 18,
        SetWindowDraggableAreas: 19,
        SetMinimumWidgetSize: 20,
        SetMicrophonePermissionState: 21,
        SetLocationPermissionState: 22,
        SetTabContextPermissionState: 23,
        SetContextAccessIndicator: 24,
        GetUserProfileInfo: 25,
        RefreshSignInCookies: 26,
        AttachPanel: 27,
        DetachPanel: 28,
        SetAudioDucking: 29,
        LogBeginAsyncEvent: 30,
        LogEndAsyncEvent: 31,
        LogInstantEvent: 32,
        JournalClear: 33,
        JournalSnapshot: 34,
        JournalStart: 35,
        JournalStop: 36,
        JournalRecordFeedback: 37,
        OnUserInputSubmitted: 38,
        OnResponseRated: 39,
        OnResponseStarted: 40,
        OnResponseStopped: 41,
        OnSessionTerminated: 42,
        OnTurnCompleted: 43,
        OnModelChanged: 44,
        ScrollTo: 45,
        SetSyntheticExperimentState: 46,
        OpenOsPermissionSettingsMenu: 47,
        GetOsMicrophonePermissionStatus: 48,
        PinTabs: 49,
        UnpinTabs: 50,
        UnpinAllTabs: 51,
        SubscribeToPinCandidates: 52,
        UnsubscribeFromPinCandidates: 53,
        GetZeroStateSuggestionsForFocusedTab: 54,
        GetZeroStateSuggestionsAndSubscribe: 55,
        SetClosedCaptioningSetting: 56,
        DropScrollToHighlight: 57,
        MaybeRefreshUserStatus: 58,
        OnClosedCaptionsShown: 59,
        CreateTask: 60,
        PerformActions: 61,
        OnViewChanged: 62,
        SubscribeToPageMetadata: 63,
        SwitchConversation: 64,
        RegisterConversation: 65,
        OnReaction: 66,
        OnContextUploadCompleted: 67,
        OnContextUploadStarted: 68,
    };
    return { ...result, MAX_VALUE: Math.max(...Object.values(result)) };
})();
var ImageAlphaType;
(function (ImageAlphaType) {
    // RGB values are unmodified.
    ImageAlphaType[ImageAlphaType["UNPREMUL"] = 0] = "UNPREMUL";
    // RGB values have been premultiplied by alpha.
    ImageAlphaType[ImageAlphaType["PREMUL"] = 1] = "PREMUL";
})(ImageAlphaType || (ImageAlphaType = {}));
// Chromium currently only uses a single color type for BitmapN32.
var ImageColorType;
(function (ImageColorType) {
    ImageColorType[ImageColorType["BGRA"] = 0] = "BGRA";
})(ImageColorType || (ImageColorType = {}));
/** Reasons why the credential selection dialog request failed. */
var SelectCredentialDialogErrorReason;
(function (SelectCredentialDialogErrorReason) {
    // The hosting WebUI received the request, but the web client has not
    // subscribed to the request yet. We couldn't show the dialog in this case.
    SelectCredentialDialogErrorReason[SelectCredentialDialogErrorReason["DIALOG_PROMISE_NO_SUBSCRIBER"] = 0] = "DIALOG_PROMISE_NO_SUBSCRIBER";
})(SelectCredentialDialogErrorReason || (SelectCredentialDialogErrorReason = {}));
var UserConfirmationDialogErrorReason;
(function (UserConfirmationDialogErrorReason) {
    // The hosting WebUI received the request, but the web client has not
    // subscribed to the request yet. We couldn't show the dialog in this case.
    UserConfirmationDialogErrorReason[UserConfirmationDialogErrorReason["DIALOG_PROMISE_NO_SUBSCRIBER"] = 0] = "DIALOG_PROMISE_NO_SUBSCRIBER";
    // The task requested a new user confirmation dialog before the current
    // one completed.
    UserConfirmationDialogErrorReason[UserConfirmationDialogErrorReason["PREEMPTED_BY_NEW_REQUEST"] = 1] = "PREEMPTED_BY_NEW_REQUEST";
})(UserConfirmationDialogErrorReason || (UserConfirmationDialogErrorReason = {}));
// Constructs an exception from a TransferableException.
function exceptionFromTransferable(e) {
    // Error types are serializable, but they do not serialize all members.
    // If exceptionReason is provided, we use it to reconstruct a
    // ErrorWithReason by just setting additional fields after
    // serialization.
    if (e.exceptionReason !== undefined) {
        const withReason = e.exception;
        withReason.reason = e.exceptionReason.reason;
        withReason.reasonType = e.exceptionReason.reasonType;
    }
    return e.exception;
}
// Transform an Error into a TransferableException.
function newTransferableException(e) {
    let exceptionReason = undefined;
    const maybeWithReason = e;
    if (maybeWithReason.reasonType !== undefined &&
        maybeWithReason.reason !== undefined) {
        exceptionReason = {
            reason: maybeWithReason.reason,
            reasonType: maybeWithReason.reasonType,
        };
    }
    return { exception: e, exceptionReason };
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function newSenderId() {
    const array = new Uint8Array(8);
    crypto.getRandomValues(array);
    return Array.from(array).map((n) => n.toString(16)).join('');
}
class ResponseExtras {
    transfers = [];
    // Add objects to transfer when sending the response over postMessage.
    addTransfer(...transfers) {
        this.transfers.push(...transfers);
    }
}
class MessageLogger {
    prefix;
    loggingEnabled = false;
    loggingPrefix;
    constructor(senderId, prefix) {
        this.prefix = prefix;
        this.loggingPrefix = `${prefix}(${senderId.substring(0, 6)})`;
    }
    setLoggingEnabled(v) {
        this.loggingEnabled = v;
    }
    shouldLogMessage(requestType) {
        return this.loggingEnabled &&
            requestType !== 'glicWebClientCheckResponsive';
    }
    maybeLogMessage(requestType, message, payload) {
        if (!this.shouldLogMessage(requestType)) {
            return;
        }
        console.info(`${this.loggingPrefix} [${requestType}] ${message}: ${toDebugJson(payload)}`, payload);
    }
}
// Sends requests over postMessage. Ideally this type would be parameterized by
// only one of HostRequestTypes or WebClientRequestTypes, but typescript
// cannot represent this. Instead, this class can send messages of any type.
class PostMessageRequestSender extends MessageLogger {
    messageSender;
    remoteOrigin;
    senderId;
    requestId = 1;
    responseHandlers = new Map();
    onDestroy;
    constructor(messageSender, remoteOrigin, senderId, logPrefix) {
        super(senderId, logPrefix);
        this.messageSender = messageSender;
        this.remoteOrigin = remoteOrigin;
        this.senderId = senderId;
        const handler = this.onMessage.bind(this);
        window.addEventListener('message', handler);
        this.onDestroy = () => {
            window.removeEventListener('message', handler);
        };
    }
    destroy() {
        this.onDestroy();
    }
    // Handles responses from the host.
    onMessage(event) {
        // Ignore all messages that don't look like responses.
        if (event.origin !== this.remoteOrigin ||
            event.data.senderId !== this.senderId ||
            event.data.type === undefined || event.data.responseId === undefined) {
            return;
        }
        const response = event.data;
        const handler = this.responseHandlers.get(response.responseId);
        if (!handler) {
            // No handler for this request.
            return;
        }
        this.responseHandlers.delete(response.responseId);
        handler(response);
    }
    // Sends a request to the host, and returns a promise that resolves with its
    // response.
    requestWithResponse(requestType, request, transfer = []) {
        const { promise, resolve, reject } = Promise.withResolvers();
        const requestId = this.requestId++;
        this.responseHandlers.set(requestId, (response) => {
            if (response.exception !== undefined) {
                this.maybeLogMessage(requestType, 'received with exception', response.exception);
                reject(exceptionFromTransferable(response.exception));
            }
            else {
                this.maybeLogMessage(requestType, 'received', response.responsePayload);
                resolve(response.responsePayload);
            }
        });
        this.maybeLogMessage(requestType, 'sending', request);
        const message = {
            senderId: this.senderId,
            glicRequest: true,
            requestId,
            type: requestType,
            requestPayload: request,
        };
        this.messageSender.postMessage(message, this.remoteOrigin, transfer);
        return promise;
    }
    // Sends a request to the host, and does not track the response.
    requestNoResponse(requestType, request, transfer = []) {
        const message = {
            senderId: this.senderId,
            glicRequest: true,
            requestId: undefined,
            type: requestType,
            requestPayload: request,
        };
        this.maybeLogMessage(requestType, 'sending', request);
        this.messageSender.postMessage(message, this.remoteOrigin, transfer);
    }
}
// Receives requests over postMessage and forward them to a
// `PostMessageRequestHandler`.
class PostMessageRequestReceiver extends MessageLogger {
    embeddedOrigin;
    postMessageSender;
    handler;
    onDestroy;
    constructor(embeddedOrigin, senderId, postMessageSender, handler, logPrefix) {
        super(senderId, logPrefix);
        this.embeddedOrigin = embeddedOrigin;
        this.postMessageSender = postMessageSender;
        this.handler = handler;
        const handlerFunction = this.onMessage.bind(this);
        window.addEventListener('message', handlerFunction);
        this.onDestroy = () => {
            window.removeEventListener('message', handlerFunction);
        };
    }
    destroy() {
        this.onDestroy();
    }
    async onMessage(event) {
        // This receives all messages to the window, so ignore them if they don't
        // look compatible.
        if (event.origin !== this.embeddedOrigin || !event.source ||
            !event.data.glicRequest) {
            return;
        }
        const requestMessage = event.data;
        const { requestId, type, requestPayload, senderId } = requestMessage;
        let response;
        let exception;
        const extras = new ResponseExtras();
        this.handler.onRequestReceived(type);
        this.maybeLogMessage(type, 'processing request', requestPayload);
        try {
            response =
                await this.handler.handleRawRequest(type, requestPayload, extras);
        }
        catch (error) {
            this.handler.onRequestHandlerException(type);
            console.warn('Unexpected error', error);
            if (error instanceof Error) {
                exception = newTransferableException(error);
            }
            else {
                exception =
                    newTransferableException(new Error(`Unexpected error: ${error}`));
            }
        }
        if (!exception) {
            this.handler.onRequestCompleted(type);
        }
        // If the message contains no `requestId`, a response is not requested.
        if (!requestId) {
            return;
        }
        this.maybeLogMessage(type, 'sending response', response?.payload);
        const responseMessage = {
            type,
            responseId: requestId,
            responsePayload: response?.payload,
            senderId,
        };
        if (exception) {
            responseMessage.exception = exception;
        }
        this.postMessageSender.postMessage(responseMessage, this.embeddedOrigin, extras.transfers);
    }
}
// Converts a value to JSON for debug logging.
function toDebugJson(v) {
    return JSON.stringify(v, (_key, value) => {
        // stringify throws on bigint, so convert it.
        if (typeof value === 'bigint') {
            return value.toString();
        }
        return value;
    });
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Web client side of the Glic API.
// Communicates with the Chrome-WebUI-side in glic_api_host.ts
class GlicHostRegistryImpl {
    windowProxy;
    host;
    constructor(windowProxy) {
        this.windowProxy = windowProxy;
    }
    async registerWebClient(webClient) {
        this.host = new GlicBrowserHostImpl(webClient, this.windowProxy);
        await this.host.webClientCreated();
        let success = false;
        let exception;
        try {
            await webClient.initialize(this.host);
            success = true;
        }
        catch (e) {
            console.warn(e);
            if (e instanceof Error) {
                exception = newTransferableException(e);
            }
        }
        if (this.host) {
            this.host.webClientInitialized(success, exception);
        }
    }
}
class WebClientMessageHandler {
    webClient;
    host;
    cachedPinnedTabs = undefined;
    constructor(webClient, host) {
        this.webClient = webClient;
        this.host = host;
    }
    async glicWebClientNotifyPanelWillOpen(payload) {
        let openPanelInfo;
        try {
            const mergedArgument = Object.assign({}, payload.panelOpeningData, payload.panelOpeningData.panelState);
            const result = await this.webClient.notifyPanelWillOpen?.(mergedArgument);
            if (result) {
                openPanelInfo = result;
            }
        }
        catch (e) {
            console.warn(e);
        }
        return { openPanelInfo };
    }
    async glicWebClientNotifyPanelWasClosed() {
        try {
            await this.webClient.notifyPanelWasClosed?.();
        }
        catch (e) {
            console.warn(e);
        }
    }
    glicWebClientPanelStateChanged(payload) {
        this.host.getPanelState?.().assignAndSignal(payload.panelState);
    }
    glicWebClientRequestViewChange(payload) {
        this.host.viewChangeRequestsSubject.next(payload.request);
    }
    glicWebClientZeroStateSuggestionsChanged(payload) {
        this.host.currentZeroStateObserver?.assignAndSignal(payload.suggestions);
    }
    glicWebClientCanAttachStateChanged(payload) {
        this.host.canAttachPanelValue.assignAndSignal(payload.canAttach);
    }
    glicWebClientNotifyMicrophonePermissionStateChanged(payload) {
        this.host.getMicrophonePermissionState().assignAndSignal(payload.enabled);
    }
    glicWebClientNotifyLocationPermissionStateChanged(payload) {
        this.host.getLocationPermissionState().assignAndSignal(payload.enabled);
    }
    glicWebClientNotifyTabContextPermissionStateChanged(payload) {
        this.host.getTabContextPermissionState().assignAndSignal(payload.enabled);
    }
    glicWebClientNotifyDefaultTabContextPermissionStateChanged(payload) {
        this.host.defaultTabContextPermission.assignAndSignal(payload.enabled);
    }
    glicWebClientNotifyOsLocationPermissionStateChanged(payload) {
        this.host.getOsLocationPermissionState().assignAndSignal(payload.enabled);
    }
    glicWebClientNotifyClosedCaptioningSettingChanged(payload) {
        this.host.closedCaptioningState.assignAndSignal(payload.enabled);
    }
    glicWebClientNotifyFocusedTabChanged(payload) {
        const focusedTabData = convertFocusedTabDataFromPrivate(payload.focusedTabDataPrivate);
        this.host.getFocusedTabStateV2().assignAndSignal(focusedTabData);
    }
    glicWebClientNotifyPanelActiveChanged(payload) {
        this.host.panelActiveValue.assignAndSignal(payload.panelActive);
    }
    async glicWebClientCheckResponsive() {
        return this.webClient.checkResponsive?.();
    }
    glicWebClientNotifyManualResizeChanged(payload) {
        this.host.isManuallyResizing().assignAndSignal(payload.resizing);
    }
    glicWebClientBrowserIsOpenChanged(payload) {
        this.host.isBrowserOpenValue.assignAndSignal(payload.browserIsOpen);
    }
    glicWebClientNotifyOsHotkeyStateChanged(payload) {
        this.host.getOsHotkeyState().assignAndSignal(payload);
    }
    glicWebClientPinCandidatesChanged(payload) {
        this.host.pinCandidates?.processUpdate(payload.candidates, payload.observationId);
    }
    glicWebClientNotifyPinnedTabsChanged(payload) {
        this.cachedPinnedTabs =
            payload.tabData.map((x) => convertTabDataFromPrivate(x));
        this.host.pinnedTabs?.assignAndSignal(this.cachedPinnedTabs);
    }
    glicWebClientNotifyPinnedTabDataChanged(payload) {
        if (!this.cachedPinnedTabs) {
            return;
        }
        const tabData = convertTabDataFromPrivate(payload.tabData);
        this.cachedPinnedTabs = this.cachedPinnedTabs.map((cachedTab) => {
            if (cachedTab.tabId === tabData.tabId) {
                return tabData;
            }
            return cachedTab;
        });
        this.host.pinnedTabs.assignAndSignal(this.cachedPinnedTabs);
    }
    glicWebClientNotifyActorTaskStateChanged(payload) {
        this.host.setActorTaskState(payload.taskId, payload.state);
    }
    glicWebClientPageMetadataChanged(payload) {
        const observable = this.host.pageMetadataObservers.get(payload.tabId);
        if (!observable) {
            return;
        }
        if (payload.pageMetadata) {
            observable.assignAndSignal(payload.pageMetadata);
        }
        else {
            if (!observable.isStopped()) {
                observable.complete();
            }
            this.host.pageMetadataObservers.delete(payload.tabId);
        }
    }
    glicWebClientNotifyActiveBrowserChanged(payload) {
        this.host.activeBrowserInfo.assignAndSignal(payload.activeBrowserInfo);
    }
    async glicWebClientRequestToShowDialog(payload) {
        const request = payload.request;
        return new Promise(resolve => {
            if (!this.host.selectCredentialDialogRequestSubject
                .hasActiveSubscription()) {
                // Since there is no subscriber, respond to the browser immediately as
                // if no credential is selected.
                window.console.warn('GlicWebClient: no subscriber for selectCredentialDialogRequest()!');
                resolve({
                    response: {
                        taskId: request.taskId,
                        errorReason: SelectCredentialDialogErrorReason.DIALOG_PROMISE_NO_SUBSCRIBER,
                    },
                });
                return;
            }
            const iconsGetter = new Map();
            for (const [id, image] of payload.request.icons.entries()) {
                let promise;
                iconsGetter.set(id, () => {
                    if (!promise) {
                        promise = rgbaImageToBlob(image);
                    }
                    return promise;
                });
            }
            const credentials = request.credentials.map((credential) => {
                const getIcon = iconsGetter.get(credential.sourceSiteOrApp);
                if (getIcon) {
                    return {
                        ...credential,
                        getIcon,
                    };
                }
                return credential;
            });
            const requestWithCallback = {
                ...request,
                credentials,
                onDialogClosed: resolve,
            };
            this.host.selectCredentialDialogRequestSubject.next(requestWithCallback);
        });
    }
    glicWebClientRequestToShowConfirmationDialog(payload) {
        return new Promise(resolve => {
            if (!this.host.userConfirmationDialogRequestSubject
                .hasActiveSubscription()) {
                // Since there is no subscriber, respond to the browser immediately as
                // if the user denied the request.
                window.console.warn('GlicWebClient: no subscriber for ' +
                    'userConfirmationDialogRequest()!');
                resolve({
                    response: {
                        permissionGranted: false,
                        errorReason: UserConfirmationDialogErrorReason.DIALOG_PROMISE_NO_SUBSCRIBER,
                    },
                });
                return;
            }
            const requestWithCallback = {
                ...payload.request,
                onDialogClosed: resolve,
            };
            this.host.userConfirmationDialogRequestSubject.next(requestWithCallback);
        });
    }
    glicWebClientNotifyAdditionalContext(payload) {
        const context = payload.context;
        const parts = context.parts.map(p => {
            const annotatedPageData = p.annotatedPageData &&
                convertAnnotatedPageDataFromPrivate(p.annotatedPageData);
            const pdf = p.pdf && convertPdfDocumentDataFromPrivate(p.pdf);
            const data = p.data && new Blob([p.data.data], { type: p.data.mimeType });
            return {
                ...p,
                data,
                annotatedPageData,
                pdf,
            };
        });
        this.host.additionalContextSubject.next({
            name: context.name,
            tabId: context.tabId,
            origin: context.origin,
            frameUrl: context.frameUrl,
            parts,
        });
    }
}
class GlicBrowserHostImpl {
    webClient;
    hostId = newSenderId();
    sender;
    receiver;
    handlerFunctionNames = new Set();
    webClientMessageHandler;
    chromeVersion;
    panelState = ObservableValue.withNoValue();
    canAttachPanelValue = ObservableValue.withNoValue();
    focusedTabStateV2 = ObservableValue.withNoValue();
    activeBrowserInfo = ObservableValue.withNoValue();
    permissionStateMicrophone = ObservableValue.withNoValue();
    permissionStateLocation = ObservableValue.withNoValue();
    permissionStateTabContext = ObservableValue.withNoValue();
    defaultTabContextPermission = ObservableValue.withNoValue();
    permissionStateOsLocation = ObservableValue.withNoValue();
    closedCaptioningState = ObservableValue.withNoValue();
    osHotkeyState = ObservableValue.withNoValue();
    panelActiveValue = ObservableValue.withNoValue();
    isBrowserOpenValue = ObservableValue.withNoValue();
    journalHost;
    metrics;
    manuallyResizing = ObservableValue.withValue(false);
    pinnedTabs = ObservableValue.withNoValue();
    pinCandidates;
    // Makes IDs that are unique within the scope of this class.
    idGenerator = new IdGenerator();
    currentZeroStateSuggestionOptions = {
        isFirstRun: false,
        supportedTools: [],
    };
    currentZeroStateObserver = ObservableValue.withNoValue();
    hostCapabilities = new Set();
    actorTaskState = new Map();
    viewChangeRequestsSubject = new Subject();
    additionalContextSubject = new Subject();
    pageMetadataObservers = new Map();
    selectCredentialDialogRequestSubject = new Subject();
    userConfirmationDialogRequestSubject = new Subject();
    constructor(webClient, windowProxy) {
        this.webClient = webClient;
        // TODO(harringtond): Ideally, we could ensure we only process requests from
        // the single senderId used by the web client. This would avoid accidental
        // processing of requests from a previous client. This risk is very minimal,
        // as it would require reloading the webview page and initializing a new
        // web client very quickly, and in normal operation, the webview does not
        // reload after successful load.
        this.sender = new PostMessageRequestSender(windowProxy, 'chrome://glic', this.hostId, 'glic_api_client');
        this.receiver = new PostMessageRequestReceiver('chrome://glic', this.hostId, windowProxy, this, 'glic_api_client');
        this.webClientMessageHandler =
            new WebClientMessageHandler(this.webClient, this);
        this.journalHost = new GlicBrowserHostJournalImpl(this.sender);
        this.metrics = new GlicBrowserHostMetricsImpl(this.sender);
        for (const name of Object.getOwnPropertyNames(WebClientMessageHandler.prototype)) {
            if (name !== 'constructor') {
                this.handlerFunctionNames.add(name);
            }
        }
    }
    destroy() {
        this.receiver.destroy();
    }
    async webClientCreated() {
        const response = await this.sender.requestWithResponse('glicBrowserWebClientCreated', undefined);
        const state = response.initialState;
        this.receiver.setLoggingEnabled(state.loggingEnabled);
        this.sender.setLoggingEnabled(state.loggingEnabled);
        this.panelState.assignAndSignal(state.panelState);
        const focusedTabData = convertFocusedTabDataFromPrivate(state.focusedTabData);
        this.focusedTabStateV2.assignAndSignal(focusedTabData);
        this.permissionStateMicrophone.assignAndSignal(state.microphonePermissionEnabled);
        this.permissionStateLocation.assignAndSignal(state.locationPermissionEnabled);
        this.permissionStateTabContext.assignAndSignal(state.tabContextPermissionEnabled);
        this.defaultTabContextPermission.assignAndSignal(state.defaultTabContextSettingEnabled);
        this.permissionStateOsLocation.assignAndSignal(state.osLocationPermissionEnabled);
        this.canAttachPanelValue.assignAndSignal(state.canAttach);
        this.chromeVersion = state.chromeVersion;
        this.panelActiveValue.assignAndSignal(state.panelIsActive);
        this.isBrowserOpenValue.assignAndSignal(state.browserIsOpen);
        this.osHotkeyState.assignAndSignal({ hotkey: state.hotkey });
        this.closedCaptioningState.assignAndSignal(state.closedCaptioningSettingEnabled);
        this.activeBrowserInfo.assignAndSignal(state.activeBrowserInfo);
        for (const capability of state.hostCapabilities) {
            this.hostCapabilities.add(capability);
        }
        // Set the method to undefined since it's gated behind a mojo
        // RuntimeFeature. Calling a such a method when the feature is disabled
        // results in a mojo pipe closure.
        if (!this.hostCapabilities.has(HostCapability.GET_MODEL_QUALITY_CLIENT_ID)) {
            // MOJO_RUNTIME_FEATURE_GATED GetModelQualityClientId
            this.getModelQualityClientId = undefined;
        }
        if (!state.enableScrollTo) {
            this.scrollTo = undefined;
            this.dropScrollToHighlight = undefined;
        }
        if (!state.enableActInFocusedTab) {
            this.createTask = undefined;
            this.performActions = undefined;
            this.stopActorTask = undefined;
            this.pauseActorTask = undefined;
            this.resumeActorTask = undefined;
        }
        if (state.alwaysDetachedMode) {
            this.attachPanel = undefined;
            this.detachPanel = undefined;
            this.canAttachPanel = undefined;
            this.getPanelState = undefined;
        }
        if (!state.enableZeroStateSuggestions) {
            this.getZeroStateSuggestionsForFocusedTab = undefined;
            // MOJO_RUNTIME_FEATURE_GATED GetZeroStateSuggestionsAndSubscribe
            this.getZeroStateSuggestions = undefined;
        }
        if (!state.enableDefaultTabContextSettingFeature) {
            this.getDefaultTabContextPermissionState = undefined;
        }
        if (!state.enableClosedCaptioningFeature) {
            this.getClosedCaptioningSetting = undefined;
            this.setClosedCaptioningSetting = undefined;
            this.metrics.onClosedCaptionsShown = undefined;
        }
        if (!state.enableMaybeRefreshUserStatus) {
            this.maybeRefreshUserStatus = undefined;
        }
        if (!state.enableMultiTab) {
            // MOJO_RUNTIME_FEATURE_GATED GetContextFromTab
            this.getContextFromTab = undefined;
            this.getPinnedTabs = undefined;
            // MOJO_RUNTIME_FEATURE_GATED SubscribeToPinCandidates
            this.getPinCandidates = undefined;
            // MOJO_RUNTIME_FEATURE_GATED PinTabs
            this.pinTabs = undefined;
            // MOJO_RUNTIME_FEATURE_GATED SetMaximumNumberOfPinnedTabs
            this.setMaximumNumberOfPinnedTabs = undefined;
            // MOJO_RUNTIME_FEATURE_GATED UnpinTabs
            this.unpinTabs = undefined;
            // MOJO_RUNTIME_FEATURE_GATED UnpinAllTabs
            this.unpinAllTabs = undefined;
        }
        if (!state.enableGetContextActor) {
            // MOJO_RUNTIME_FEATURE_GATED GetContextForActorFromTab
            this.getContextForActorFromTab = undefined;
        }
        if (!state.enableGetPageMetadata) {
            this.getPageMetadata = undefined;
        }
    }
    webClientInitialized(success, exception) {
        this.sender.requestNoResponse('glicBrowserWebClientInitialized', { success, exception });
    }
    async handleRawRequest(type, payload, extras) {
        if (!this.handlerFunctionNames.has(type)) {
            return;
        }
        const handlerFunction = this.webClientMessageHandler[type];
        const response = await handlerFunction.call(this.webClientMessageHandler, payload, extras);
        if (!response) {
            return;
        }
        return { payload: response };
    }
    setActorTaskState(taskId, state) {
        this.getActorTaskState(taskId).assignAndSignal(state);
        if (state === ActorTaskState.STOPPED) {
            this.actorTaskState.delete(taskId);
        }
    }
    onRequestReceived(_type) { }
    onRequestHandlerException(_type) { }
    onRequestCompleted(_type) { }
    // GlicBrowserHost implementation.
    getChromeVersion() {
        return Promise.resolve(this.chromeVersion);
    }
    async createTab(url, options) {
        const result = await this.sender.requestWithResponse('glicBrowserCreateTab', {
            url,
            options,
        });
        if (!result.tabData) {
            throw new Error('createTab: failed');
        }
        return convertTabDataFromPrivate(result.tabData);
    }
    openGlicSettingsPage(options) {
        this.sender.requestNoResponse('glicBrowserOpenGlicSettingsPage', { options });
    }
    closePanel() {
        return this.sender.requestWithResponse('glicBrowserClosePanel', undefined);
    }
    closePanelAndShutdown() {
        this.sender.requestNoResponse('glicBrowserClosePanelAndShutdown', undefined);
    }
    attachPanel() {
        this.sender.requestNoResponse('glicBrowserAttachPanel', undefined);
    }
    detachPanel() {
        this.sender.requestNoResponse('glicBrowserDetachPanel', undefined);
    }
    showProfilePicker() {
        this.sender.requestNoResponse('glicBrowserShowProfilePicker', undefined);
    }
    async getModelQualityClientId() {
        const result = await this.sender.requestWithResponse('glicBrowserGetModelQualityClientId', undefined);
        return result.modelQualityClientId;
    }
    async switchConversation(info) {
        if (info && !info.conversationId) {
            throw new Error('conversationId cannot be empty.');
        }
        await this.sender.requestWithResponse('glicBrowserSwitchConversation', { info });
    }
    async registerConversation(info) {
        await this.sender.requestWithResponse('glicBrowserRegisterConversation', { info });
    }
    async getContextFromFocusedTab(options) {
        const context = await this.sender.requestWithResponse('glicBrowserGetContextFromFocusedTab', { options });
        return convertTabContextResultFromPrivate(context.tabContextResult);
    }
    async setMaximumNumberOfPinnedTabs(requestedMax) {
        const result = await this.sender.requestWithResponse('glicBrowserSetMaximumNumberOfPinnedTabs', { requestedMax });
        return result.effectiveMax;
    }
    async getContextFromTab(tabId, options) {
        const result = await this.sender.requestWithResponse('glicBrowserGetContextFromTab', { tabId, options });
        return convertTabContextResultFromPrivate(result.tabContextResult);
    }
    async getContextForActorFromTab(tabId, options) {
        const result = await this.sender.requestWithResponse('glicBrowserGetContextForActorFromTab', { tabId, options });
        return convertTabContextResultFromPrivate(result.tabContextResult);
    }
    async createTask(taskOptions) {
        const result = await this.sender.requestWithResponse('glicBrowserCreateTask', { taskOptions });
        return result.taskId;
    }
    async performActions(actions) {
        const result = await this.sender.requestWithResponse('glicBrowserPerformActions', { actions });
        return result.actionsResult;
    }
    stopActorTask(taskId, stopReason) {
        this.sender.requestNoResponse('glicBrowserStopActorTask', {
            taskId: taskId ?? 0,
            stopReason: stopReason ?? ActorTaskStopReason.TASK_COMPLETE,
        });
    }
    pauseActorTask(taskId, pauseReason) {
        this.sender.requestNoResponse('glicBrowserPauseActorTask', {
            taskId,
            pauseReason: pauseReason ?? ActorTaskPauseReason.PAUSED_BY_MODEL,
        });
    }
    async resumeActorTask(taskId, tabContextOptions) {
        const response = await this.sender.requestWithResponse('glicBrowserResumeActorTask', { taskId, tabContextOptions });
        return convertTabContextResultFromPrivate(response.tabContextResult);
    }
    getActorTaskState(taskId) {
        const stateObs = this.actorTaskState.get(taskId);
        if (stateObs) {
            return stateObs;
        }
        // TODO(mcnee): The client could pass an id that will never have
        // state updates (e.g. the task already finished and we cleared the old
        // observable in setActorTaskState). Consider removing these cases from the
        // map when all subscribers are removed.
        const newObs = ObservableValue.withNoValue();
        this.actorTaskState.set(taskId, newObs);
        return newObs;
    }
    async resizeWindow(width, height, options) {
        return this.sender.requestWithResponse('glicBrowserResizeWindow', { size: { width, height }, options });
    }
    enableDragResize(enabled) {
        return this.sender.requestWithResponse('glicBrowserEnableDragResize', { enabled });
    }
    async captureScreenshot() {
        const screenshotResult = await this.sender.requestWithResponse('glicBrowserCaptureScreenshot', undefined);
        return screenshotResult.screenshot;
    }
    setWindowDraggableAreas(areas) {
        return this.sender.requestWithResponse('glicBrowserSetWindowDraggableAreas', { areas });
    }
    setMinimumWidgetSize(width, height) {
        return this.sender.requestWithResponse('glicBrowserSetMinimumWidgetSize', { size: { width, height } });
    }
    getPanelState() {
        return this.panelState;
    }
    panelActive() {
        return this.panelActiveValue;
    }
    canAttachPanel() {
        return this.canAttachPanelValue;
    }
    isBrowserOpen() {
        return this.isBrowserOpenValue;
    }
    activeBrowser() {
        return this.activeBrowserInfo;
    }
    getFocusedTabStateV2() {
        return this.focusedTabStateV2;
    }
    getMicrophonePermissionState() {
        return this.permissionStateMicrophone;
    }
    getLocationPermissionState() {
        return this.permissionStateLocation;
    }
    getTabContextPermissionState() {
        return this.permissionStateTabContext;
    }
    getDefaultTabContextPermissionState() {
        return this.defaultTabContextPermission;
    }
    getOsLocationPermissionState() {
        return this.permissionStateOsLocation;
    }
    getClosedCaptioningSetting() {
        return this.closedCaptioningState;
    }
    setMicrophonePermissionState(enabled) {
        return this.sender.requestWithResponse('glicBrowserSetMicrophonePermissionState', { enabled });
    }
    setLocationPermissionState(enabled) {
        return this.sender.requestWithResponse('glicBrowserSetLocationPermissionState', { enabled });
    }
    setTabContextPermissionState(enabled) {
        return this.sender.requestWithResponse('glicBrowserSetTabContextPermissionState', { enabled });
    }
    setClosedCaptioningSetting(enabled) {
        return this.sender.requestWithResponse('glicBrowserSetClosedCaptioningSetting', { enabled });
    }
    setContextAccessIndicator(show) {
        this.sender.requestWithResponse('glicBrowserSetContextAccessIndicator', { show });
    }
    async getUserProfileInfo() {
        const { profileInfo } = await this.sender.requestWithResponse('glicBrowserGetUserProfileInfo', undefined);
        if (!profileInfo) {
            throw new Error('getUserProfileInfo failed');
        }
        const { avatarIcon } = profileInfo;
        return replaceProperties(profileInfo, { avatarIcon: async () => avatarIcon && rgbaImageToBlob(avatarIcon) });
    }
    async refreshSignInCookies() {
        const result = await this.sender.requestWithResponse('glicBrowserRefreshSignInCookies', undefined);
        if (!result.success) {
            throw Error('refreshSignInCookies failed');
        }
    }
    setAudioDucking(enabled) {
        this.sender.requestNoResponse('glicBrowserSetAudioDucking', { enabled });
    }
    getJournalHost() {
        return this.journalHost;
    }
    getMetrics() {
        return this.metrics;
    }
    scrollTo(params) {
        return this.sender.requestWithResponse('glicBrowserScrollTo', { params });
    }
    setSyntheticExperimentState(trialName, groupName) {
        this.sender.requestNoResponse('glicBrowserSetSyntheticExperimentState', { trialName, groupName });
    }
    openOsPermissionSettingsMenu(permission) {
        this.sender.requestNoResponse('glicBrowserOpenOsPermissionSettingsMenu', { permission });
    }
    async getOsMicrophonePermissionStatus() {
        return (await this.sender.requestWithResponse('glicBrowserGetOsMicrophonePermissionStatus', undefined))
            .enabled;
    }
    isManuallyResizing() {
        return this.manuallyResizing;
    }
    getOsHotkeyState() {
        return this.osHotkeyState;
    }
    getPinnedTabs() {
        return this.pinnedTabs;
    }
    async pinTabs(tabIds) {
        return (await this.sender.requestWithResponse('glicBrowserPinTabs', { tabIds }))
            .pinnedAll;
    }
    async unpinTabs(tabIds) {
        return (await this.sender.requestWithResponse('glicBrowserUnpinTabs', { tabIds }))
            .unpinnedAll;
    }
    unpinAllTabs() {
        this.sender.requestNoResponse('glicBrowserUnpinAllTabs', undefined);
    }
    getPinCandidates(options) {
        this.pinCandidates?.setObsolete();
        return this.pinCandidates = new PinCandidatesObservable(this.idGenerator.next(), this.sender, options);
    }
    async getZeroStateSuggestionsForFocusedTab(isFirstRun) {
        const zeroStateResult = await this.sender.requestWithResponse('glicBrowserGetZeroStateSuggestionsForFocusedTab', { isFirstRun });
        if (!zeroStateResult.suggestions) {
            return {
                suggestions: [],
                tabId: '',
                url: '',
            };
        }
        return zeroStateResult.suggestions;
    }
    async zeroStateActiveSubscriptionStateChanged(options, hasActiveSubscription) {
        if (options !== this.currentZeroStateSuggestionOptions) {
            // Dont send out of date updates.
            return;
        }
        const zeroStateResult = await this.sender.requestWithResponse('glicBrowserGetZeroStateSuggestionsAndSubscribe', {
            hasActiveSubscription: hasActiveSubscription,
            options: options,
        });
        if (zeroStateResult.suggestions) {
            this.currentZeroStateObserver?.assignAndSignal(zeroStateResult.suggestions);
        }
    }
    getZeroStateSuggestions(options) {
        options = options ?? {
            isFirstRun: false,
            supportedTools: [],
        };
        this.currentZeroStateSuggestionOptions = options;
        this.currentZeroStateObserver =
            ObservableValue.withNoValue(this.zeroStateActiveSubscriptionStateChanged.bind(this, options));
        return this.currentZeroStateObserver;
    }
    dropScrollToHighlight() {
        this.sender.requestNoResponse('glicBrowserDropScrollToHighlight', undefined);
    }
    maybeRefreshUserStatus() {
        this.sender.requestNoResponse('glicBrowserMaybeRefreshUserStatus', undefined);
    }
    getAdditionalContext() {
        return this.additionalContextSubject;
    }
    getHostCapabilities() {
        return this.hostCapabilities;
    }
    getViewChangeRequests() {
        return this.viewChangeRequestsSubject;
    }
    onViewChanged(notification) {
        this.sender.requestNoResponse('glicBrowserOnViewChanged', { notification });
    }
    getPageMetadata(tabId, names) {
        if (this.pageMetadataObservers.has(tabId)) {
            // Currently, we assume that names do not change and keep only
            // one observer per tabId.
            return this.pageMetadataObservers.get(tabId);
        }
        if (names.length === 0) {
            throw Error('names must not be empty');
        }
        const observableValue = ObservableValue.withNoValue(async (isActive) => {
            // If the client subscribes to an Observable with an invalid tabId,
            // it will emit nothing, even if the tab later becomes valid.
            const { success } = await this.sender.requestWithResponse('glicBrowserSubscribeToPageMetadata', { tabId, names: isActive ? names : [] });
            if (!success) {
                if (!observableValue.isStopped()) {
                    observableValue.complete();
                }
                this.pageMetadataObservers.delete(tabId);
            }
        });
        this.pageMetadataObservers.set(tabId, observableValue);
        return observableValue;
    }
    selectCredentialDialogRequestHandler() {
        return this.selectCredentialDialogRequestSubject;
    }
    selectUserConfirmationDialogRequestHandler() {
        return this.userConfirmationDialogRequestSubject;
    }
}
class GlicBrowserHostJournalImpl {
    sender;
    constructor(sender) {
        this.sender = sender;
    }
    beginAsyncEvent(asyncEventId, taskId, event, details) {
        this.sender.requestNoResponse('glicBrowserLogBeginAsyncEvent', { asyncEventId, taskId, event, details });
    }
    clear() {
        this.sender.requestNoResponse('glicBrowserJournalClear', undefined);
    }
    endAsyncEvent(asyncEventId, details) {
        this.sender.requestNoResponse('glicBrowserLogEndAsyncEvent', { asyncEventId, details });
    }
    instantEvent(taskId, event, details) {
        this.sender.requestNoResponse('glicBrowserLogInstantEvent', { taskId, event, details });
    }
    async snapshot(clear) {
        const snapshotResult = await this.sender.requestWithResponse('glicBrowserJournalSnapshot', { clear });
        return snapshotResult.journal;
    }
    start(maxBytes, captureScreenshots) {
        this.sender.requestNoResponse('glicBrowserJournalStart', { maxBytes, captureScreenshots });
    }
    stop() {
        this.sender.requestNoResponse('glicBrowserJournalStop', undefined);
    }
    recordFeedback(positive, reason) {
        this.sender.requestNoResponse('glicBrowserJournalRecordFeedback', { positive, reason });
    }
}
class GlicBrowserHostMetricsImpl {
    sender;
    constructor(sender) {
        this.sender = sender;
    }
    onUserInputSubmitted(mode) {
        this.sender.requestNoResponse('glicBrowserOnUserInputSubmitted', { mode });
    }
    onReaction(reactionType) {
        this.sender.requestNoResponse('glicBrowserOnReaction', { reactionType });
    }
    onContextUploadStarted() {
        this.sender.requestNoResponse('glicBrowserOnContextUploadStarted', undefined);
    }
    onContextUploadCompleted() {
        this.sender.requestNoResponse('glicBrowserOnContextUploadCompleted', undefined);
    }
    onResponseStarted() {
        this.sender.requestNoResponse('glicBrowserOnResponseStarted', undefined);
    }
    onResponseStopped(details) {
        this.sender.requestNoResponse('glicBrowserOnResponseStopped', { details });
    }
    onSessionTerminated() {
        this.sender.requestNoResponse('glicBrowserOnSessionTerminated', undefined);
    }
    onResponseRated(positive) {
        this.sender.requestNoResponse('glicBrowserOnResponseRated', { positive });
    }
    onClosedCaptionsShown() {
        this.sender.requestNoResponse('glicBrowserOnClosedCaptionsShown', undefined);
    }
    onTurnCompleted(model, duration) {
        this.sender.requestNoResponse('glicBrowserOnTurnCompleted', { model, duration });
    }
    onModelChanged(model) {
        this.sender.requestNoResponse('glicBrowserOnModelChanged', { model });
    }
}
class IdGenerator {
    nextId = 1;
    next() {
        return this.nextId++;
    }
}
class PinCandidatesObservable extends ObservableValue {
    observationId;
    sender;
    options;
    isObsolete = false;
    constructor(observationId, sender, options) {
        super(false);
        this.observationId = observationId;
        this.sender = sender;
        this.options = options;
    }
    activeSubscriptionChanged(hasActiveSubscription) {
        super.activeSubscriptionChanged(hasActiveSubscription);
        if (this.isObsolete) {
            console.warn(`getPinCandidates() observable is in use while obsolete.`);
            return;
        }
        if (hasActiveSubscription) {
            this.sender.requestNoResponse('glicBrowserSubscribeToPinCandidates', { options: this.options, observationId: this.observationId });
        }
        else {
            this.sender.requestNoResponse('glicBrowserUnsubscribeFromPinCandidates', { observationId: this.observationId });
        }
    }
    processUpdate(candidates, observationId) {
        if (this.observationId !== observationId) {
            return;
        }
        this.assignAndSignal(candidates.map(c => ({ tabData: convertTabDataFromPrivate(c.tabData) })));
    }
    // Mark this observable as obsolete. It should not be used any further.
    // Only one PinCandidatesObservable is active at one time.
    setObsolete() {
        if (this.hasActiveSubscription()) {
            console.warn(`getPinCandidates() observable was made obsolete with subscribers.`);
        }
        this.isObsolete = true;
    }
}
// Converts an RgbaImage into a Blob through the canvas API. Output is a PNG.
async function rgbaImageToBlob(image) {
    const canvas = document.createElement('canvas');
    canvas.width = image.width;
    canvas.height = image.height;
    const ctx = canvas.getContext('2d');
    if (!ctx) {
        throw Error('getContext error');
    }
    if (image.colorType !== ImageColorType.BGRA) {
        throw Error('unsupported colorType');
    }
    // Note that for either alphaType, we swap bytes from BGRA to RGBA order.
    const pixelData = new Uint8ClampedArray(image.dataRGBA);
    if (image.alphaType === ImageAlphaType.PREMUL) {
        for (let i = 0; i + 3 < pixelData.length; i += 4) {
            const alphaInt = pixelData[i + 3];
            if (alphaInt === 0) {
                // Don't divide by zero. In this case, RGB should already be zero, so
                // there's no purpose in swapping bytes.
                continue;
            }
            const alpha = alphaInt / 255.0;
            const [B, G, R] = [pixelData[i], pixelData[i + 1], pixelData[i + 2]];
            pixelData[i] = R / alpha;
            pixelData[i + 1] = G / alpha;
            pixelData[i + 2] = B / alpha;
        }
    }
    else {
        for (let i = 0; i + 3 < pixelData.length; i += 4) {
            const [B, R] = [pixelData[i], pixelData[i + 2]];
            pixelData[i] = R;
            pixelData[i + 2] = B;
        }
    }
    ctx.putImageData(new ImageData(pixelData, image.width, image.height), 0, 0);
    return new Promise((resolve) => {
        canvas.toBlob((result) => {
            if (!result) {
                throw Error('toBlob failed');
            }
            resolve(result);
        });
    });
}
function convertTabDataFromPrivate(data) {
    if (!data) {
        return undefined;
    }
    let faviconResult;
    const dataFavicon = data.favicon;
    async function getFavicon() {
        if (dataFavicon && !faviconResult) {
            faviconResult = rgbaImageToBlob(dataFavicon);
            return faviconResult;
        }
        return faviconResult;
    }
    const favicon = dataFavicon && getFavicon;
    return replaceProperties(data, { favicon });
}
function convertFocusedTabDataFromPrivate(data) {
    const result = {};
    if (data.hasFocus) {
        result.hasFocus = replaceProperties(data.hasFocus, {
            tabData: convertTabDataFromPrivate(data.hasFocus.tabData),
        });
    }
    if (data.hasNoFocus) {
        result.hasNoFocus = replaceProperties(data.hasNoFocus, {
            tabFocusCandidateData: convertTabDataFromPrivate(data.hasNoFocus.tabFocusCandidateData),
        });
    }
    return result;
}
function streamFromBuffer(buffer) {
    return new ReadableStream({
        start(controller) {
            controller.enqueue(buffer);
            controller.close();
        },
    });
}
function convertPdfDocumentDataFromPrivate(data) {
    const pdfData = data.pdfData && streamFromBuffer(new Uint8Array(data.pdfData));
    return replaceProperties(data, { pdfData });
}
function convertAnnotatedPageDataFromPrivate(data) {
    const annotatedPageContent = data.annotatedPageContent &&
        streamFromBuffer(new Uint8Array(data.annotatedPageContent));
    return replaceProperties(data, { annotatedPageContent });
}
function convertTabContextResultFromPrivate(data) {
    const tabData = convertTabDataFromPrivate(data.tabData);
    const pdfDocumentData = data.pdfDocumentData &&
        convertPdfDocumentDataFromPrivate(data.pdfDocumentData);
    const annotatedPageData = data.annotatedPageData &&
        convertAnnotatedPageDataFromPrivate(data.annotatedPageData);
    return replaceProperties(data, { tabData, pdfDocumentData, annotatedPageData });
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/*
This is bundled into a js file, and sent to the web client. It should be
directly executable in a <script> element, and therefore should not have any
exports.
*/
window.internalAutoGlicBoot = (windowProxy) => new GlicHostRegistryImpl(windowProxy);
//# sourceMappingURL=glic_api_injected_client.rollup.js.map
