// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '/strings.m.js';
import 'chrome://resources/cr_elements/cr_url_list_item/cr_url_list_item.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_action_menu/cr_action_menu.js';
import 'chrome://resources/cr_elements/cr_lazy_render/cr_lazy_render_lit.js';
import 'chrome://resources/cr_elements/icons.html.js';
import 'chrome://resources/cr_elements/cr_input/cr_input.js';
import 'chrome://resources/cr_elements/cr_checkbox/cr_checkbox.js';
import { ShowSetDisposition } from '//resources/cr_components/commerce/product_specifications.mojom-webui.js';
import { ShoppingServiceBrowserProxyImpl } from '//resources/cr_components/commerce/shopping_service_browser_proxy.js';
import { assert } from '//resources/js/assert.js';
import { PluralStringProxyImpl } from '//resources/js/plural_string_proxy.js';
import { ProductSpecificationsBrowserProxyImpl } from 'chrome://resources/cr_components/commerce/product_specifications_browser_proxy.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './comparison_table_list_item.css.js';
import { getHtml } from './comparison_table_list_item.html.js';
import { $$ } from './utils.js';
export class ComparisonTableListItemElement extends CrLitElement {
    static get is() {
        return 'comparison-table-list-item';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            hasCheckbox: { type: Boolean },
            imageUrl_: { type: Object },
            isMenuOpen_: { type: Boolean },
            isRenaming_: { type: Boolean },
            name: { type: String },
            numItemsString_: { type: String },
            tableUrl_: { type: Object },
            urls: { type: Array },
            uuid: { type: Object },
        };
    }
    #hasCheckbox_accessor_storage = false;
    get hasCheckbox() { return this.#hasCheckbox_accessor_storage; }
    set hasCheckbox(value) { this.#hasCheckbox_accessor_storage = value; }
    #name_accessor_storage = '';
    get name() { return this.#name_accessor_storage; }
    set name(value) { this.#name_accessor_storage = value; }
    #urls_accessor_storage = [];
    get urls() { return this.#urls_accessor_storage; }
    set urls(value) { this.#urls_accessor_storage = value; }
    #uuid_accessor_storage = { value: '' };
    get uuid() { return this.#uuid_accessor_storage; }
    set uuid(value) { this.#uuid_accessor_storage = value; }
    #imageUrl__accessor_storage = null;
    get imageUrl_() { return this.#imageUrl__accessor_storage; }
    set imageUrl_(value) { this.#imageUrl__accessor_storage = value; }
    #isMenuOpen__accessor_storage = false;
    get isMenuOpen_() { return this.#isMenuOpen__accessor_storage; }
    set isMenuOpen_(value) { this.#isMenuOpen__accessor_storage = value; }
    #isRenaming__accessor_storage = false;
    get isRenaming_() { return this.#isRenaming__accessor_storage; }
    set isRenaming_(value) { this.#isRenaming__accessor_storage = value; }
    #numItemsString__accessor_storage = '';
    get numItemsString_() { return this.#numItemsString__accessor_storage; }
    set numItemsString_(value) { this.#numItemsString__accessor_storage = value; }
    #tableUrl__accessor_storage = { url: '' };
    get tableUrl_() { return this.#tableUrl__accessor_storage; }
    set tableUrl_(value) { this.#tableUrl__accessor_storage = value; }
    pluralStringProxy_ = PluralStringProxyImpl.getInstance();
    productSpecificationsProxy_ = ProductSpecificationsBrowserProxyImpl.getInstance();
    shoppingApi_ = ShoppingServiceBrowserProxyImpl.getInstance();
    async connectedCallback() {
        super.connectedCallback();
        this.addEventListener('close', () => {
            this.isMenuOpen_ = false;
        });
        const { url } = await this.productSpecificationsProxy_.getComparisonTableUrlForUuid(this.uuid);
        this.tableUrl_ = url;
        this.updateImage_();
        this.updateNumItemsString_();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('urls')) {
            this.updateImage_();
            this.updateNumItemsString_();
        }
    }
    getTitle_() {
        return loadTimeData.getStringF('tableListItemTitle', this.name);
    }
    getFaviconUrl_() {
        // Display the favicon for the first product if no product images are
        // available. If there are no URLs, display the Compare favicon.
        if (this.urls.length > 0 && this.urls[0]) {
            return this.urls[0].url;
        }
        return this.tableUrl_.url;
    }
    async updateNumItemsString_() {
        this.numItemsString_ = await this.pluralStringProxy_.getPluralString('numItems', this.urls.length);
        this.fire('num-items-updated-for-testing');
    }
    async updateImage_() {
        // Hide the currently displayed image.
        this.imageUrl_ = null;
        // Find the first product with an image to use as the item's image.
        let imageUrl = null;
        for (let i = 0; i < this.urls.length; i++) {
            const url = this.urls[i];
            assert(url);
            const { productInfo } = await this.shoppingApi_.getProductInfoForUrl(url);
            if (productInfo.imageUrl.url) {
                imageUrl = productInfo.imageUrl;
                break;
            }
        }
        this.imageUrl_ = imageUrl;
        this.fire('image-updated-for-testing');
    }
    onClick_(event) {
        // Treat the item click as a checkbox click if it has checkbox.
        if (this.hasCheckbox) {
            event.preventDefault();
            event.stopPropagation();
            this.checkbox_.checked = !this.checkbox_.checked;
            return;
        }
        if (!this.isRenaming_) {
            this.fire('item-click', {
                uuid: this.uuid,
            });
        }
    }
    onContextMenu_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.isMenuOpen_ = true;
        this.$.menu.get().showAtPosition({
            top: event.clientY,
            left: event.clientX,
        });
    }
    onShowContextMenuClick_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.isMenuOpen_ = true;
        this.$.menu.get().showAt(this.trailingIconButton_);
    }
    onOpenInNewTabClick_() {
        this.$.menu.get().close();
        this.productSpecificationsProxy_.showProductSpecificationsSetForUuid(this.uuid, true);
    }
    onOpenInNewWindowClick_() {
        this.$.menu.get().close();
        this.productSpecificationsProxy_.showProductSpecificationsSetsForUuids([this.uuid], ShowSetDisposition.kInNewWindow);
    }
    async onRenameClick_() {
        this.$.menu.get().close();
        this.isRenaming_ = true;
        // Focus the input once it is rendered.
        await this.updateComplete;
        this.input_.focus();
    }
    onDeleteClick_() {
        this.$.menu.get().close();
        this.fire('delete-table', {
            uuid: this.uuid,
        });
    }
    onRenameInputBlur_() {
        if (this.input_.value.length !== 0) {
            this.fire('rename-table', {
                uuid: this.uuid,
                name: this.input_.value,
            });
        }
        this.isRenaming_ = false;
    }
    onRenameInputKeyDown_(event) {
        if (event.key === 'Enter') {
            event.stopPropagation();
            this.input_.blur();
        }
    }
    onCheckboxChange_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.fire('checkbox-change', {
            uuid: this.uuid,
            checked: event.target.checked,
        });
    }
    get trailingIconButton_() {
        const trailingIconButton = $$(this, '#trailingIconButton');
        assert(trailingIconButton);
        return trailingIconButton;
    }
    get input_() {
        const input = $$(this, '#renameInput');
        assert(input);
        return input;
    }
    get checkbox_() {
        const checkbox = $$(this, '#checkbox');
        assert(checkbox);
        return checkbox;
    }
}
customElements.define(ComparisonTableListItemElement.is, ComparisonTableListItemElement);
