// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './context_menu_entrypoint.js';
import './file_carousel.js';
import './recent_tab_chip.js';
import './icons.html.js';
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { FileUploadErrorType, FileUploadStatus } from './composebox_query.mojom-webui.js';
import { getCss } from './contextual_entrypoint_and_carousel.css.js';
import { getHtml } from './contextual_entrypoint_and_carousel.html.js';
export var ComposeboxMode;
(function (ComposeboxMode) {
    ComposeboxMode["DEFAULT"] = "";
    ComposeboxMode["DEEP_SEARCH"] = "deep-search";
    ComposeboxMode["CREATE_IMAGE"] = "create-image";
})(ComposeboxMode || (ComposeboxMode = {}));
const FILE_VALIDATION_ERRORS_MAP = new Map([
    [
        FileUploadErrorType.kImageProcessingError,
        'composeboxFileUploadImageProcessingError',
    ],
    [
        FileUploadErrorType.kUnknown,
        'composeboxFileUploadValidationFailed',
    ],
]);
export class ContextualEntrypointAndCarouselElement extends I18nMixinLit(CrLitElement) {
    static get is() {
        return 'contextual-entrypoint-and-carousel';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            // =========================================================================
            // Public properties
            // =========================================================================
            showDropdown: { type: Boolean },
            realboxLayoutMode: { type: String },
            tabSuggestions: { type: Array },
            entrypointName: { type: String },
            parentFocused: { type: Boolean },
            // =========================================================================
            // Protected properties
            // =========================================================================
            attachmentFileTypes_: { type: String },
            contextMenuEnabled_: { type: Boolean },
            files_: { type: Object },
            addedTabsIds_: { type: Object },
            imageFileTypes_: { type: String },
            inputsDisabled_: {
                reflect: true,
                type: Boolean,
            },
            recentTabChipDisabled_: {
                reflect: true,
                type: Boolean,
            },
            composeboxShowPdfUpload_: {
                reflect: true,
                type: Boolean,
            },
            showContextMenuDescription_: { type: Boolean },
            showFileCarousel_: {
                reflect: true,
                type: Boolean,
            },
            showRecentTabChip_: { type: Boolean },
            inDeepSearchMode_: {
                reflect: true,
                type: Boolean,
            },
            inCreateImageMode_: {
                reflect: true,
                type: Boolean,
            },
            recentTabInContext_: { type: Boolean },
        };
    }
    #showDropdown_accessor_storage = false;
    get showDropdown() { return this.#showDropdown_accessor_storage; }
    set showDropdown(value) { this.#showDropdown_accessor_storage = value; }
    #realboxLayoutMode_accessor_storage = '';
    get realboxLayoutMode() { return this.#realboxLayoutMode_accessor_storage; }
    set realboxLayoutMode(value) { this.#realboxLayoutMode_accessor_storage = value; }
    #entrypointName_accessor_storage = '';
    get entrypointName() { return this.#entrypointName_accessor_storage; }
    set entrypointName(value) { this.#entrypointName_accessor_storage = value; }
    #tabSuggestions_accessor_storage = [];
    get tabSuggestions() { return this.#tabSuggestions_accessor_storage; }
    set tabSuggestions(value) { this.#tabSuggestions_accessor_storage = value; }
    #parentFocused_accessor_storage = false;
    get parentFocused() { return this.#parentFocused_accessor_storage; }
    set parentFocused(value) { this.#parentFocused_accessor_storage = value; }
    #attachmentFileTypes__accessor_storage = loadTimeData.getString('composeboxAttachmentFileTypes');
    get attachmentFileTypes_() { return this.#attachmentFileTypes__accessor_storage; }
    set attachmentFileTypes_(value) { this.#attachmentFileTypes__accessor_storage = value; }
    #contextMenuEnabled__accessor_storage = loadTimeData.getBoolean('composeboxShowContextMenu');
    get contextMenuEnabled_() { return this.#contextMenuEnabled__accessor_storage; }
    set contextMenuEnabled_(value) { this.#contextMenuEnabled__accessor_storage = value; }
    #files__accessor_storage = new Map();
    get files_() { return this.#files__accessor_storage; }
    set files_(value) { this.#files__accessor_storage = value; }
    #addedTabsIds__accessor_storage = new Set();
    get addedTabsIds_() { return this.#addedTabsIds__accessor_storage; }
    set addedTabsIds_(value) { this.#addedTabsIds__accessor_storage = value; }
    #imageFileTypes__accessor_storage = loadTimeData.getString('composeboxImageFileTypes');
    get imageFileTypes_() { return this.#imageFileTypes__accessor_storage; }
    set imageFileTypes_(value) { this.#imageFileTypes__accessor_storage = value; }
    #inputsDisabled__accessor_storage = false;
    get inputsDisabled_() { return this.#inputsDisabled__accessor_storage; }
    set inputsDisabled_(value) { this.#inputsDisabled__accessor_storage = value; }
    #recentTabChipDisabled__accessor_storage = false;
    get recentTabChipDisabled_() { return this.#recentTabChipDisabled__accessor_storage; }
    set recentTabChipDisabled_(value) { this.#recentTabChipDisabled__accessor_storage = value; }
    #composeboxShowPdfUpload__accessor_storage = loadTimeData.getBoolean('composeboxShowPdfUpload');
    get composeboxShowPdfUpload_() { return this.#composeboxShowPdfUpload__accessor_storage; }
    set composeboxShowPdfUpload_(value) { this.#composeboxShowPdfUpload__accessor_storage = value; }
    #showContextMenuDescription__accessor_storage = loadTimeData.getBoolean('composeboxShowContextMenuDescription');
    get showContextMenuDescription_() { return this.#showContextMenuDescription__accessor_storage; }
    set showContextMenuDescription_(value) { this.#showContextMenuDescription__accessor_storage = value; }
    #showRecentTabChip__accessor_storage = loadTimeData.getBoolean('composeboxShowRecentTabChip');
    get showRecentTabChip_() { return this.#showRecentTabChip__accessor_storage; }
    set showRecentTabChip_(value) { this.#showRecentTabChip__accessor_storage = value; }
    #showFileCarousel__accessor_storage = false;
    get showFileCarousel_() { return this.#showFileCarousel__accessor_storage; }
    set showFileCarousel_(value) { this.#showFileCarousel__accessor_storage = value; }
    #inDeepSearchMode__accessor_storage = false;
    get inDeepSearchMode_() { return this.#inDeepSearchMode__accessor_storage; }
    set inDeepSearchMode_(value) { this.#inDeepSearchMode__accessor_storage = value; }
    #inCreateImageMode__accessor_storage = false;
    get inCreateImageMode_() { return this.#inCreateImageMode__accessor_storage; }
    set inCreateImageMode_(value) { this.#inCreateImageMode__accessor_storage = value; }
    #recentTabInContext__accessor_storage = false;
    get recentTabInContext_() { return this.#recentTabInContext__accessor_storage; }
    set recentTabInContext_(value) { this.#recentTabInContext__accessor_storage = value; }
    hasTabSuggestions_() {
        return this.tabSuggestions?.length > 0;
    }
    get inToolMode_() {
        return this.inDeepSearchMode_ || this.inCreateImageMode_;
    }
    get shouldShowRecentTabChip_() {
        return this.parentFocused && this.showRecentTabChip_ &&
            this.hasTabSuggestions_() && !this.recentTabInContext_ &&
            !this.inToolMode_;
    }
    maxFileCount_ = loadTimeData.getInteger('composeboxFileMaxCount');
    maxFileSize_ = loadTimeData.getInteger('composeboxFileMaxSize');
    createImageModeEnabled_ = loadTimeData.getBoolean('composeboxShowCreateImageButton');
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('files_') ||
            changedPrivateProperties.has(`inCreateImageMode_`)) {
            // If only 1 image is uploaded and the create image tool is enabled, we
            // don't want to disable the context menu entrypoint because the user
            // should still be able to use the tool within the context menu.
            const isCreateImageToolAvailableWithImages = this.createImageModeEnabled_ &&
                this.hasImageFiles() && this.files_.size === 1;
            // `inputsDisabled_` decides whether or not the context menu entrypoint is
            // shown to the user. Only set `inputsDisabled_` to true if
            // 1. The max number of files is reached, and the create image tool button
            //    is not available.
            // 2. The user has an image uploaded and is in create image mode.
            this.inputsDisabled_ =
                (this.files_.size >= this.maxFileCount_ &&
                    !isCreateImageToolAvailableWithImages) ||
                    (this.hasImageFiles() && this.inCreateImageMode_);
            this.showFileCarousel_ = this.files_.size > 0;
            this.recentTabChipDisabled_ = this.files_.size >= this.maxFileCount_;
            this.fire('on-context-files-changed', { files: this.files_.size });
        }
        if (changedPrivateProperties.has('files_') ||
            changedProperties.has('tabSuggestions')) {
            this.recentTabInContext_ = this.computeRecentTabInContext_();
        }
    }
    computeRecentTabInContext_() {
        const recentTab = this.tabSuggestions?.[0];
        if (!recentTab) {
            return false;
        }
        return this.addedTabsIds_.has(recentTab.tabId);
    }
    setContextFiles(files) {
        for (const file of files) {
            if (file.type === 'tab') {
                this.addTabContext_(new CustomEvent('addTabContext', {
                    detail: {
                        id: file.tabId,
                        title: file.name,
                        url: file.url,
                    },
                }));
            }
            else {
                this.addFileContext_([file.file], file.objectUrl !== null);
            }
        }
    }
    setInitialMode(mode) {
        switch (mode) {
            case ComposeboxMode.DEEP_SEARCH:
                this.onDeepSearchClick_();
                break;
            case ComposeboxMode.CREATE_IMAGE:
                this.onCreateImageClick_();
                break;
        }
    }
    updateFileStatus(token, status, errorType) {
        let errorMessage = null;
        let file = this.files_.get(token);
        if (file) {
            if ([FileUploadStatus.kValidationFailed,
                FileUploadStatus.kUploadFailed,
                FileUploadStatus.kUploadExpired].includes(status)) {
                this.files_.delete(token);
                if (file.tabId) {
                    this.addedTabsIds_ = new Set([...this.addedTabsIds_].filter((id) => id !== file.tabId));
                }
                switch (status) {
                    case FileUploadStatus.kValidationFailed:
                        errorMessage = this.i18n(FILE_VALIDATION_ERRORS_MAP.get(errorType) ??
                            'composeboxFileUploadValidationFailed');
                        break;
                    case FileUploadStatus.kUploadFailed:
                        errorMessage = this.i18n('composeboxFileUploadFailed');
                        break;
                    case FileUploadStatus.kUploadExpired:
                        errorMessage = this.i18n('composeboxFileUploadExpired');
                        break;
                    default:
                        break;
                }
            }
            else {
                file = { ...file, status: status };
                this.files_.set(token, file);
            }
            this.files_ = new Map([...this.files_]);
        }
        return { file, errorMessage };
    }
    resetContextFiles() {
        this.files_ = new Map();
        this.addedTabsIds_ = new Set();
    }
    resetModes() {
        if (this.inDeepSearchMode_) {
            this.inDeepSearchMode_ = false;
            this.inputsDisabled_ = false;
            this.fire('set-deep-search-mode', { inDeepSearchMode: this.inDeepSearchMode_ });
            this.showContextMenuDescription_ = true;
        }
        else if (this.inCreateImageMode_) {
            this.inCreateImageMode_ = false;
            this.fire('set-create-image-mode', {
                inCreateImageMode: this.inCreateImageMode_,
                imagePresent: this.hasImageFiles(),
            });
            this.showContextMenuDescription_ = true;
        }
    }
    hasImageFiles() {
        if (this.files_) {
            for (const file of this.files_.values()) {
                if (file.type.includes('image')) {
                    return true;
                }
            }
        }
        return false;
    }
    onDeleteFile_(e) {
        if (!e.detail.uuid || !this.files_.has(e.detail.uuid)) {
            return;
        }
        const file = this.files_.get(e.detail.uuid);
        if (file?.tabId) {
            this.addedTabsIds_ = new Set([...this.addedTabsIds_].filter((id) => id !== file.tabId));
        }
        this.files_ = new Map([...this.files_.entries()].filter(([uuid, _]) => uuid !== e.detail.uuid));
        this.fire('delete-context', { uuid: e.detail.uuid });
    }
    onFileChange_(e) {
        const input = e.target;
        const files = input.files;
        // Multiple is set to false in the input so only one file is expected.
        if (!files || files.length === 0 ||
            this.files_.size >= this.maxFileCount_) {
            this.recordFileValidationMetric_(1 /* ComposeboxFileValidationError.TOO_MANY_FILES */);
            return;
        }
        const filesToUpload = [];
        for (const file of files) {
            if (file.size === 0 || file.size > this.maxFileSize_) {
                const fileIsEmpty = file.size === 0;
                input.value = '';
                fileIsEmpty ? this.recordFileValidationMetric_(2 /* ComposeboxFileValidationError.FILE_EMPTY */) :
                    this.recordFileValidationMetric_(3 /* ComposeboxFileValidationError.FILE_SIZE_TOO_LARGE */);
                this.fire('on-file-validation-error', {
                    errorMessage: fileIsEmpty ?
                        this.i18n('composeboxFileUploadInvalidEmptySize') :
                        this.i18n('composeboxFileUploadInvalidTooLarge'),
                });
                return;
            }
            if (!file.type.includes('pdf') && !file.type.includes('image')) {
                return;
            }
            filesToUpload.push(file);
        }
        this.addFileContext_(filesToUpload, input === this.$.imageInput);
        input.value = '';
    }
    addFileContext_(filesToUpload, isImage) {
        this.fire('add-file-context', {
            files: filesToUpload,
            isImage: isImage,
            onContextAdded: (files) => {
                this.files_ = new Map([...this.files_.entries(), ...files.entries()]);
                this.recordFileValidationMetric_(0 /* ComposeboxFileValidationError.NONE */);
            },
        });
    }
    addTabContext_(e) {
        e.stopPropagation();
        this.fire('add-tab-context', {
            id: e.detail.id,
            title: e.detail.title,
            url: e.detail.url,
            onContextAdded: (file) => {
                this.files_ = new Map([...this.files_.entries(), [file.uuid, file]]);
                this.addedTabsIds_ = new Set([...this.addedTabsIds_, e.detail.id]);
            },
        });
    }
    openImageUpload_() {
        this.$.imageInput.click();
    }
    openFileUpload_() {
        this.$.fileInput.click();
    }
    onDeepSearchClick_() {
        if (this.entrypointName !== 'Realbox') {
            this.showContextMenuDescription_ = !this.showContextMenuDescription_;
            this.inputsDisabled_ = !this.inputsDisabled_;
            this.inDeepSearchMode_ = !this.inDeepSearchMode_;
        }
        this.fire('set-deep-search-mode', { inDeepSearchMode: this.inDeepSearchMode_ });
    }
    onCreateImageClick_() {
        if (this.entrypointName !== 'Realbox') {
            this.showContextMenuDescription_ = !this.showContextMenuDescription_;
            this.inCreateImageMode_ = !this.inCreateImageMode_;
            if (this.hasImageFiles()) {
                this.inputsDisabled_ = !this.inputsDisabled_;
            }
        }
        this.fire('set-create-image-mode', {
            inCreateImageMode: this.inCreateImageMode_,
            imagePresent: this.hasImageFiles(),
        });
    }
    recordFileValidationMetric_(enumValue) {
        chrome.metricsPrivate.recordEnumerationValue('NewTabPage.Composebox.File.WebUI.UploadAttemptFailure', enumValue, 3 /* ComposeboxFileValidationError.MAX_VALUE */ + 1);
    }
}
customElements.define(ContextualEntrypointAndCarouselElement.is, ContextualEntrypointAndCarouselElement);
