// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { mojoString16ToString } from '//resources/js/mojo_type_util.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './composebox_match.css.js';
import { getHtml } from './composebox_match.html.js';
import { ComposeboxProxyImpl, createAutocompleteMatch } from './composebox_proxy.js';
// Displays an autocomplete match
export class ComposeboxMatchElement extends CrLitElement {
    static get is() {
        return 'ntp-composebox-match';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            //========================================================================
            // Public properties
            //========================================================================
            match: { type: Object },
            /**
             * Index of the match in the autocomplete result. Used to inform embedder
             * of events such as deletion, click, etc.
             */
            matchIndex: { type: Number },
            removeButtonTitle_: { type: String },
        };
    }
    #match_accessor_storage = createAutocompleteMatch();
    get match() { return this.#match_accessor_storage; }
    set match(value) { this.#match_accessor_storage = value; }
    #matchIndex_accessor_storage = -1;
    get matchIndex() { return this.#matchIndex_accessor_storage; }
    set matchIndex(value) { this.#matchIndex_accessor_storage = value; }
    searchboxHandler_;
    #removeButtonTitle__accessor_storage = loadTimeData.getString('removeSuggestion');
    get removeButtonTitle_() { return this.#removeButtonTitle__accessor_storage; }
    set removeButtonTitle_(value) { this.#removeButtonTitle__accessor_storage = value; }
    constructor() {
        super();
        this.searchboxHandler_ = ComposeboxProxyImpl.getInstance().searchboxHandler;
    }
    connectedCallback() {
        super.connectedCallback();
        this.addEventListener('click', (event) => this.onMatchClick_(event));
        this.addEventListener('focusin', () => this.onMatchFocusin_());
    }
    computeContents_() {
        return mojoString16ToString(this.match.contents);
    }
    computeRemoveButtonAriaLabel_() {
        return mojoString16ToString(this.match.removeButtonA11yLabel);
    }
    iconPath_() {
        return this.match.iconPath || '';
    }
    onMatchFocusin_() {
        this.fire('match-focusin', {
            index: this.matchIndex,
        });
    }
    onMatchClick_(e) {
        if (e.button > 1) {
            // Only handle main (generally left) and middle button presses.
            return;
        }
        e.preventDefault(); // Prevents default browser action (navigation).
        this.searchboxHandler_.openAutocompleteMatch(this.matchIndex, this.match.destinationUrl, 
        /* are_matches_showing */ true, e.button || 0, e.altKey, e.ctrlKey, e.metaKey, e.shiftKey);
        this.fire('match-click');
    }
    onRemoveButtonClick_(e) {
        if (e.button !== 0) {
            // Only handle main (generally left) button presses.
            return;
        }
        e.preventDefault(); // Prevents default browser action (navigation).
        e.stopPropagation(); // Prevents <iron-selector> from selecting the match.
        this.searchboxHandler_.deleteAutocompleteMatch(this.matchIndex, this.match.destinationUrl);
    }
    onRemoveButtonMouseDown_(e) {
        e.preventDefault(); // Prevents default browser action (focus).
    }
}
customElements.define(ComposeboxMatchElement.is, ComposeboxMatchElement);
