// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_button/cr_button.js';
import '//resources/cr_elements/cr_toast/cr_toast.js';
import '//resources/cr_elements/cr_collapse/cr_collapse.js';
import '//resources/cr_elements/cr_expand_button/cr_expand_button.js';
import '//resources/cr_elements/cr_slider/cr_slider.js';
import { CrRouter } from '//resources/js/cr_router.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { TraceConfig, TraceConfig_BufferConfig_FillPolicy } from './perfetto_config.js';
// 
import { getCss } from './trace_recorder.css.js';
import { getHtml } from './trace_recorder.html.js';
import { downloadTraceData } from './trace_util.js';
import { TracesBrowserProxy } from './traces_browser_proxy.js';
var TracingState;
(function (TracingState) {
    TracingState["IDLE"] = "Idle";
    TracingState["STARTING"] = "Starting";
    TracingState["RECORDING"] = "Recording";
    TracingState["STOPPING"] = "Stopping";
})(TracingState || (TracingState = {}));
// 
export class TraceRecorderElement extends CrLitElement {
    static get is() {
        return 'trace-recorder';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            toastMessage: { type: String },
            bufferSizeMb: { type: Number },
            bufferFillPolicy: { type: Object },
            tracingState: { type: String },
            trackEventCategories: { type: Array },
            trackEventTags: { type: Array },
            privacyFilterEnabled_: { type: Boolean },
            enabledCategories: { type: Object },
            enabledTags: { type: Object },
            // 
            disabledTags: { type: Object },
            buffersExpanded_: { type: Boolean },
            categoriesExpanded_: { type: Boolean },
            tagsExpanded_: { type: Boolean },
            bufferUsage: { type: Number },
            hadDataLoss: { type: Boolean },
        };
    }
    browserProxy_ = TracesBrowserProxy.getInstance();
    // Bound method for router events
    boundLoadConfigFromUrl_ = this.loadConfigFromUrl_.bind(this);
    // Bound method for onTraceComplete listener
    boundOnTraceComplete_ = this.onTraceComplete_.bind(this);
    // Bound method for buffer usage polling
    boundPollBufferUsage_ = this.pollBufferUsage_.bind(this);
    // Property to store the listener ID for onTraceComplete
    onTraceCompleteListenerId_ = null;
    // ID for the polling interval
    bufferPollIntervalId_ = null;
    encodedConfigString = '';
    #toastMessage_accessor_storage = '';
    get toastMessage() { return this.#toastMessage_accessor_storage; }
    set toastMessage(value) { this.#toastMessage_accessor_storage = value; }
    #tracingState_accessor_storage = TracingState.IDLE;
    // Initialize the tracing state to IDLE.
    get tracingState() { return this.#tracingState_accessor_storage; }
    set tracingState(value) { this.#tracingState_accessor_storage = value; }
    #trackEventCategories_accessor_storage = [];
    get trackEventCategories() { return this.#trackEventCategories_accessor_storage; }
    set trackEventCategories(value) { this.#trackEventCategories_accessor_storage = value; }
    #trackEventTags_accessor_storage = [];
    get trackEventTags() { return this.#trackEventTags_accessor_storage; }
    set trackEventTags(value) { this.#trackEventTags_accessor_storage = value; }
    #privacyFilterEnabled__accessor_storage = false;
    get privacyFilterEnabled_() { return this.#privacyFilterEnabled__accessor_storage; }
    set privacyFilterEnabled_(value) { this.#privacyFilterEnabled__accessor_storage = value; }
    #bufferSizeMb_accessor_storage = 200;
    get bufferSizeMb() { return this.#bufferSizeMb_accessor_storage; }
    set bufferSizeMb(value) { this.#bufferSizeMb_accessor_storage = value; }
    #bufferFillPolicy_accessor_storage = TraceConfig_BufferConfig_FillPolicy.RING_BUFFER;
    get bufferFillPolicy() { return this.#bufferFillPolicy_accessor_storage; }
    set bufferFillPolicy(value) { this.#bufferFillPolicy_accessor_storage = value; }
    traceConfig;
    trackEventConfig;
    #enabledCategories_accessor_storage = new Set();
    get enabledCategories() { return this.#enabledCategories_accessor_storage; }
    set enabledCategories(value) { this.#enabledCategories_accessor_storage = value; }
    #enabledTags_accessor_storage = new Set();
    get enabledTags() { return this.#enabledTags_accessor_storage; }
    set enabledTags(value) { this.#enabledTags_accessor_storage = value; }
    #disabledTags_accessor_storage = new Set();
    get disabledTags() { return this.#disabledTags_accessor_storage; }
    set disabledTags(value) { this.#disabledTags_accessor_storage = value; }
    #buffersExpanded__accessor_storage = false;
    // 
    get buffersExpanded_() { return this.#buffersExpanded__accessor_storage; }
    set buffersExpanded_(value) { this.#buffersExpanded__accessor_storage = value; }
    #categoriesExpanded__accessor_storage = false;
    get categoriesExpanded_() { return this.#categoriesExpanded__accessor_storage; }
    set categoriesExpanded_(value) { this.#categoriesExpanded__accessor_storage = value; }
    #tagsExpanded__accessor_storage = false;
    get tagsExpanded_() { return this.#tagsExpanded__accessor_storage; }
    set tagsExpanded_(value) { this.#tagsExpanded__accessor_storage = value; }
    #bufferUsage_accessor_storage = 0;
    get bufferUsage() { return this.#bufferUsage_accessor_storage; }
    set bufferUsage(value) { this.#bufferUsage_accessor_storage = value; }
    #hadDataLoss_accessor_storage = false;
    get hadDataLoss() { return this.#hadDataLoss_accessor_storage; }
    set hadDataLoss(value) { this.#hadDataLoss_accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.loadConfigFromUrl_();
        this.loadEventCategories_();
        CrRouter.getInstance().addEventListener('cr-router-path-changed', this.boundLoadConfigFromUrl_);
        this.onTraceCompleteListenerId_ =
            this.browserProxy_.callbackRouter.onTraceComplete.addListener(this.boundOnTraceComplete_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        CrRouter.getInstance().removeEventListener('cr-router-path-changed', this.boundLoadConfigFromUrl_);
        if (this.onTraceCompleteListenerId_ !== null) {
            this.browserProxy_.callbackRouter.removeListener(this.onTraceCompleteListenerId_);
            this.onTraceCompleteListenerId_ = null;
        }
    }
    get isStartTracingEnabled() {
        return this.tracingState === TracingState.IDLE && !!this.traceConfig;
    }
    get isRecording() {
        return this.tracingState === TracingState.RECORDING;
    }
    get fillPolicyEnum() {
        return TraceConfig_BufferConfig_FillPolicy;
    }
    get statusClass() {
        switch (this.tracingState) {
            case TracingState.IDLE:
                return 'status-idle';
            case TracingState.STARTING:
                return 'status-starting';
            case TracingState.RECORDING:
                return 'status-tracing';
            case TracingState.STOPPING:
                return 'status-stopping';
            default:
                return '';
        }
    }
    async pollBufferUsage_() {
        const { success, percentFull, dataLoss } = await this.browserProxy_.handler.getBufferUsage();
        if (success) {
            this.bufferUsage = percentFull;
            this.hadDataLoss = dataLoss;
        }
    }
    async startTracing_() {
        const bigBufferConfig = this.serializeTraceConfigToBigBuffer_();
        if (!bigBufferConfig) {
            return;
        }
        // Set state to RECORDING immediately to disable start button.
        this.tracingState = TracingState.STARTING;
        const { success } = await this.browserProxy_.handler.startTraceSession(bigBufferConfig, this.privacyFilterEnabled_);
        if (!success) {
            this.showToast_('Failed to start tracing.');
            // Revert to IDLE if starting failed.
            this.tracingState = TracingState.IDLE;
        }
        else {
            this.tracingState = TracingState.RECORDING;
            this.bufferPollIntervalId_ =
                window.setInterval(this.boundPollBufferUsage_, 1000);
        }
    }
    async stopTracing_() {
        if (this.bufferPollIntervalId_ !== null) {
            window.clearInterval(this.bufferPollIntervalId_);
            this.bufferPollIntervalId_ = null;
        }
        // Set state to STOPPING to indicate an ongoing operation.
        this.tracingState = TracingState.STOPPING;
        const { success } = await this.browserProxy_.handler.stopTraceSession();
        if (!success) {
            this.showToast_('Failed to stop tracing.');
        }
    }
    async cloneTraceSession_() {
        const { trace, uuid } = await this.browserProxy_.handler.cloneTraceSession();
        this.downloadData_(trace, uuid);
    }
    privacyFilterDidChange_(event) {
        if (this.privacyFilterEnabled_ === event.detail) {
            return;
        }
        this.privacyFilterEnabled_ = event.detail;
    }
    onBuffersExpandedChanged_(e) {
        this.buffersExpanded_ = e.detail.value;
    }
    onCategoriesExpandedChanged_(e) {
        this.categoriesExpanded_ = e.detail.value;
    }
    onTagsExpandedChanged_(e) {
        this.tagsExpanded_ = e.detail.value;
    }
    isCategoryEnabled(category) {
        if (this.enabledCategories.has(category.name)) {
            return true;
        }
        return this.isCategoryForced(category);
    }
    canonicalCategoryName(category) {
        const disabledPrefix = 'disabled-by-default-';
        if (category.name.startsWith(disabledPrefix)) {
            const mainPart = category.name.substring(disabledPrefix.length);
            return `${mainPart} (disabled-by-default)`;
        }
        return category.name;
    }
    isCategoryForced(category) {
        for (const tag of category.tags) {
            if (this.disabledTags.has(tag)) {
                return false;
            }
            if (this.enabledTags.has(tag)) {
                return true;
            }
        }
        return false;
    }
    isTagEnabled(tagName) {
        return this.enabledTags.has(tagName);
    }
    isTagDisabled(tagName) {
        return this.disabledTags.has(tagName);
    }
    onBufferSizeChanged_(e) {
        const slider = e.target;
        this.bufferSizeMb = Math.floor(slider.value);
        this.updateBufferConfigField_('sizeKb', this.bufferSizeMb * 1024);
    }
    onBufferFillPolicyChanged_(e) {
        const selectElement = e.target;
        const policyValue = Number(selectElement.value);
        this.bufferFillPolicy = policyValue;
        this.updateBufferConfigField_('fillPolicy', policyValue);
    }
    updateBufferConfigField_(field, value) {
        if (!this.traceConfig?.buffers?.[0]) {
            return;
        }
        this.traceConfig.buffers[0][field] = value;
        this.updateUrlFromConfig_();
    }
    onCategoryChange_(event, categoryName) {
        if (!this.trackEventConfig) {
            return;
        }
        const isChecked = event.target.checked;
        if (isChecked) {
            this.enabledCategories.add(categoryName);
        }
        else {
            this.enabledCategories.delete(categoryName);
        }
        this.trackEventConfig.enabledCategories = [...this.enabledCategories];
        // Reset property to force UI update.
        this.enabledCategories = new Set(this.trackEventConfig.enabledCategories);
        this.updateUrlFromConfig_();
    }
    // 
    onTagsChange_(event, tagName, enabled) {
        if (!this.trackEventConfig) {
            return;
        }
        const isChecked = event.target.checked;
        const primarySet = enabled ? this.enabledTags : this.disabledTags;
        const secondarySet = enabled ? this.disabledTags : this.enabledTags;
        if (isChecked) {
            primarySet.add(tagName);
            secondarySet.delete(tagName);
        }
        else {
            primarySet.delete(tagName);
        }
        this.trackEventConfig.enabledTags = [...this.enabledTags];
        this.trackEventConfig.disabledTags = [...this.disabledTags];
        // Reset properties to force UI update.
        this.enabledTags = new Set(this.trackEventConfig.enabledTags);
        this.disabledTags = new Set(this.trackEventConfig.disabledTags);
        this.updateUrlFromConfig_();
    }
    downloadData_(traceData, uuid) {
        if (!traceData || !uuid) {
            this.showToast_('Failed to download trace or no trace data.');
            return;
        }
        try {
            downloadTraceData(traceData, uuid);
        }
        catch (e) {
            this.showToast_(`Error downloading trace: ${e}`);
        }
    }
    async loadEventCategories_() {
        let { categories } = await this.browserProxy_.handler.getTrackEventCategories();
        // Filter category groups.
        categories = categories.filter(category => !category.isGroup);
        // Create a map to get unique categories by name, keeping the last one
        // found.
        categories = Array.from(categories
            .reduce((map, category) => {
            map.set(category.name, category);
            return map;
        }, new Map())
            .values());
        // Sort the unique categories and assign them.
        const disabledPrefix = 'disabled-by-default-';
        this.trackEventCategories = categories.sort((a, b) => a.name.replace(disabledPrefix, '')
            .localeCompare(b.name.replace(disabledPrefix, '')));
        // Extract unique tags using flatMap and a Set.
        this.trackEventTags =
            [...new Set(categories.map(category => category.tags).flat())];
        // 
    }
    // Decodes a Base64 string into a Uint8Array.
    base64ToUint8Array_(base64String) {
        const binaryString = atob(base64String);
        const len = binaryString.length;
        const bytes = new Uint8Array(len);
        for (let i = 0; i < len; i++) {
            bytes[i] = binaryString.charCodeAt(i);
        }
        return bytes;
    }
    // Encode from Uint8Array into a Base64 string.
    uint8ArrayToBase64_(bytes) {
        let binary = '';
        for (const byte of bytes) {
            binary += String.fromCharCode(byte);
        }
        return btoa(binary);
    }
    serializeTraceConfigToBigBuffer_() {
        if (!this.traceConfig) {
            return;
        }
        let bigBuffer = undefined;
        try {
            const serializedConfig = TraceConfig.encode(this.traceConfig).finish();
            bigBuffer = {
                bytes: Array.from(serializedConfig),
            };
            return bigBuffer;
        }
        catch (error) {
            this.showToast_(`Error encoding: ${error}`);
        }
        return bigBuffer;
    }
    onTraceComplete_(trace, uuid) {
        if (this.bufferPollIntervalId_ !== null) {
            window.clearInterval(this.bufferPollIntervalId_);
            this.bufferPollIntervalId_ = null;
        }
        this.bufferUsage = 0;
        this.hadDataLoss = false;
        this.downloadData_(trace, uuid);
        // Crucially, only set to IDLE here after the trace has been
        // processed/handled.
        this.tracingState = TracingState.IDLE;
    }
    showToast_(message) {
        this.toastMessage = message;
        this.$.toast?.show();
    }
    loadConfigFromUrl_() {
        const params = new URLSearchParams(document.location.search);
        const host = params.get('trace_config');
        const newConfig = host ?? '';
        if (this.encodedConfigString === newConfig && newConfig !== '') {
            return;
        }
        this.encodedConfigString = newConfig;
        const serializedConfig = this.base64ToUint8Array_(newConfig);
        if (serializedConfig.length === 0) {
            this.initializeDefaultConfig_();
        }
        else {
            try {
                this.traceConfig = TraceConfig.decode(serializedConfig);
                // Fallback for the buffer config if the loaded one is missing it.
                if (!this.traceConfig.buffers ||
                    this.traceConfig.buffers.length === 0) {
                    this.traceConfig.buffers = this.createDefaultBufferConfig_();
                }
                const trackEventDataSource = this.traceConfig.dataSources?.find(ds => ds.config?.trackEventConfig !== undefined);
                if (trackEventDataSource) {
                    this.trackEventConfig = trackEventDataSource.config?.trackEventConfig;
                }
                else {
                    this.trackEventConfig = this.createDefaultTrackEventConfig_();
                    this.addDataSourceConfig_({
                        name: 'track_event',
                        targetBuffer: 0,
                        trackEventConfig: this.trackEventConfig,
                    });
                }
                // 
            }
            catch (e) {
                this.showToast_(`Could not parse trace config: ${e}`);
                this.initializeDefaultConfig_();
            }
        }
        // Centralized calls to sync the UI and update the URL.
        this.updatePropertiesFromConfig_();
    }
    updateUrlFromConfig_() {
        if (!this.traceConfig) {
            return;
        }
        try {
            // Encode the modified config object back to a Uint8Array
            const writer = TraceConfig.encode(this.traceConfig);
            // Convert to Base64 and update the URL
            const newEncodedConfigString = this.uint8ArrayToBase64_(writer.finish());
            if (this.encodedConfigString === newEncodedConfigString) {
                return;
            }
            const newUrl = new URL(window.location.href);
            newUrl.searchParams.set('trace_config', newEncodedConfigString);
            // Update URL without reloading the page
            history.replaceState({}, '', newUrl.toString());
        }
        catch (e) {
            this.showToast_(`Could not update trace config: ${e}`);
        }
    }
    initializeDefaultConfig_() {
        this.trackEventConfig = this.createDefaultTrackEventConfig_();
        this.traceConfig = {
            buffers: this.createDefaultBufferConfig_(),
            dataSources: [
                // DataSource for track events
                {
                    config: {
                        name: 'track_event',
                        targetBuffer: 0,
                        trackEventConfig: this.trackEventConfig,
                    },
                },
                // DataSource for org.chromium.trace_metadata2
                {
                    config: {
                        name: 'org.chromium.trace_metadata2',
                        targetBuffer: 1,
                    },
                },
            ],
        };
    }
    createDefaultBufferConfig_() {
        return [
            {
                sizeKb: 200 * 1024,
                fillPolicy: TraceConfig_BufferConfig_FillPolicy.RING_BUFFER,
            },
            {
                sizeKb: 256,
                fillPolicy: TraceConfig_BufferConfig_FillPolicy.DISCARD,
            },
        ];
    }
    createDefaultTrackEventConfig_() {
        return {
            enabledCategories: [],
            disabledCategories: ['*'],
            enabledTags: [],
            disabledTags: ['slow', 'debug', 'sensitive'],
        };
    }
    addDataSourceConfig_(config) {
        if (!this.traceConfig) {
            return;
        }
        if (!this.traceConfig.dataSources) {
            this.traceConfig.dataSources = [];
        }
        this.traceConfig.dataSources.push({ config: config });
    }
    updatePropertiesFromConfig_() {
        const mainBuffer = this.traceConfig?.buffers?.[0];
        if (mainBuffer) {
            this.bufferSizeMb = Math.floor((mainBuffer.sizeKb ?? 0) / 1024);
            this.bufferFillPolicy = mainBuffer.fillPolicy ??
                TraceConfig_BufferConfig_FillPolicy.RING_BUFFER;
        }
        // Sync the tag and category UI state.
        this.enabledCategories = new Set(this.trackEventConfig?.enabledCategories);
        this.enabledTags = new Set(this.trackEventConfig?.enabledTags);
        this.disabledTags = new Set(this.trackEventConfig?.disabledTags);
        // 
    }
}
customElements.define(TraceRecorderElement.is, TraceRecorderElement);
