// Copyright 2008 Google Inc.  All rights reserved.
// support library is itself covered by the above license.
/**
 * Read a 64 bit varint as two JS numbers.
 *
 * Returns tuple:
 * [0]: low bits
 * [1]: high bits
 *
 * Copyright 2008 Google Inc.  All rights reserved.
 *
 * See https://github.com/protocolbuffers/protobuf/blob/8a71927d74a4ce34efe2d8769fda198f52d20d12/js/experimental/runtime/kernel/buffer_decoder.js#L175
 */
function varint64read(){let lowBits=0;let highBits=0;for(let shift=0;shift<28;shift+=7){let b=this.buf[this.pos++];lowBits|=(b&127)<<shift;if((b&128)==0){this.assertBounds();return[lowBits,highBits]}}let middleByte=this.buf[this.pos++];lowBits|=(middleByte&15)<<28;highBits=(middleByte&112)>>4;if((middleByte&128)==0){this.assertBounds();return[lowBits,highBits]}for(let shift=3;shift<=31;shift+=7){let b=this.buf[this.pos++];highBits|=(b&127)<<shift;if((b&128)==0){this.assertBounds();return[lowBits,highBits]}}throw new Error("invalid varint")}
/**
 * Write a 64 bit varint, given as two JS numbers, to the given bytes array.
 *
 * Copyright 2008 Google Inc.  All rights reserved.
 *
 * See https://github.com/protocolbuffers/protobuf/blob/8a71927d74a4ce34efe2d8769fda198f52d20d12/js/experimental/runtime/kernel/writer.js#L344
 */function varint64write(lo,hi,bytes){for(let i=0;i<28;i=i+7){const shift=lo>>>i;const hasNext=!(shift>>>7==0&&hi==0);const byte=(hasNext?shift|128:shift)&255;bytes.push(byte);if(!hasNext){return}}const splitBits=lo>>>28&15|(hi&7)<<4;const hasMoreBits=!(hi>>3==0);bytes.push((hasMoreBits?splitBits|128:splitBits)&255);if(!hasMoreBits){return}for(let i=3;i<31;i=i+7){const shift=hi>>>i;const hasNext=!(shift>>>7==0);const byte=(hasNext?shift|128:shift)&255;bytes.push(byte);if(!hasNext){return}}bytes.push(hi>>>31&1)}const TWO_PWR_32_DBL=4294967296;
/**
 * Parse decimal string of 64 bit integer value as two JS numbers.
 *
 * Copyright 2008 Google Inc.  All rights reserved.
 *
 * See https://github.com/protocolbuffers/protobuf-javascript/blob/a428c58273abad07c66071d9753bc4d1289de426/experimental/runtime/int64.js#L10
 */function int64FromString(dec){const minus=dec[0]==="-";if(minus){dec=dec.slice(1)}const base=1e6;let lowBits=0;let highBits=0;function add1e6digit(begin,end){const digit1e6=Number(dec.slice(begin,end));highBits*=base;lowBits=lowBits*base+digit1e6;if(lowBits>=TWO_PWR_32_DBL){highBits=highBits+(lowBits/TWO_PWR_32_DBL|0);lowBits=lowBits%TWO_PWR_32_DBL}}add1e6digit(-24,-18);add1e6digit(-18,-12);add1e6digit(-12,-6);add1e6digit(-6);return minus?negate(lowBits,highBits):newBits(lowBits,highBits)}
/**
 * Losslessly converts a 64-bit signed integer in 32:32 split representation
 * into a decimal string.
 *
 * Copyright 2008 Google Inc.  All rights reserved.
 *
 * See https://github.com/protocolbuffers/protobuf-javascript/blob/a428c58273abad07c66071d9753bc4d1289de426/experimental/runtime/int64.js#L10
 */function int64ToString(lo,hi){let bits=newBits(lo,hi);const negative=bits.hi&2147483648;if(negative){bits=negate(bits.lo,bits.hi)}const result=uInt64ToString(bits.lo,bits.hi);return negative?"-"+result:result}
/**
 * Losslessly converts a 64-bit unsigned integer in 32:32 split representation
 * into a decimal string.
 *
 * Copyright 2008 Google Inc.  All rights reserved.
 *
 * See https://github.com/protocolbuffers/protobuf-javascript/blob/a428c58273abad07c66071d9753bc4d1289de426/experimental/runtime/int64.js#L10
 */function uInt64ToString(lo,hi){({lo:lo,hi:hi}=toUnsigned(lo,hi));if(hi<=2097151){return String(TWO_PWR_32_DBL*hi+lo)}const low=lo&16777215;const mid=(lo>>>24|hi<<8)&16777215;const high=hi>>16&65535;let digitA=low+mid*6777216+high*6710656;let digitB=mid+high*8147497;let digitC=high*2;const base=1e7;if(digitA>=base){digitB+=Math.floor(digitA/base);digitA%=base}if(digitB>=base){digitC+=Math.floor(digitB/base);digitB%=base}return digitC.toString()+decimalFrom1e7WithLeadingZeros(digitB)+decimalFrom1e7WithLeadingZeros(digitA)}function toUnsigned(lo,hi){return{lo:lo>>>0,hi:hi>>>0}}function newBits(lo,hi){return{lo:lo|0,hi:hi|0}}function negate(lowBits,highBits){highBits=~highBits;if(lowBits){lowBits=~lowBits+1}else{highBits+=1}return newBits(lowBits,highBits)}const decimalFrom1e7WithLeadingZeros=digit1e7=>{const partial=String(digit1e7);return"0000000".slice(partial.length)+partial};
/**
 * Write a 32 bit varint, signed or unsigned. Same as `varint64write(0, value, bytes)`
 *
 * Copyright 2008 Google Inc.  All rights reserved.
 *
 * See https://github.com/protocolbuffers/protobuf/blob/1b18833f4f2a2f681f4e4a25cdf3b0a43115ec26/js/binary/encoder.js#L144
 */function varint32write(value,bytes){if(value>=0){while(value>127){bytes.push(value&127|128);value=value>>>7}bytes.push(value)}else{for(let i=0;i<9;i++){bytes.push(value&127|128);value=value>>7}bytes.push(1)}}function varint32read(){let b=this.buf[this.pos++];let result=b&127;if((b&128)==0){this.assertBounds();return result}b=this.buf[this.pos++];result|=(b&127)<<7;if((b&128)==0){this.assertBounds();return result}b=this.buf[this.pos++];result|=(b&127)<<14;if((b&128)==0){this.assertBounds();return result}b=this.buf[this.pos++];result|=(b&127)<<21;if((b&128)==0){this.assertBounds();return result}b=this.buf[this.pos++];result|=(b&15)<<28;for(let readBytes=5;(b&128)!==0&&readBytes<10;readBytes++)b=this.buf[this.pos++];if((b&128)!=0)throw new Error("invalid varint");this.assertBounds();return result>>>0}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
const protoInt64=makeInt64Support();function makeInt64Support(){const dv=new DataView(new ArrayBuffer(8));const ok=typeof BigInt==="function"&&typeof dv.getBigInt64==="function"&&typeof dv.getBigUint64==="function"&&typeof dv.setBigInt64==="function"&&typeof dv.setBigUint64==="function"&&(typeof process!="object"||typeof process.env!="object"||process.env.BUF_BIGINT_DISABLE!=="1");if(ok){const MIN=BigInt("-9223372036854775808"),MAX=BigInt("9223372036854775807"),UMIN=BigInt("0"),UMAX=BigInt("18446744073709551615");return{zero:BigInt(0),supported:true,parse(value){const bi=typeof value=="bigint"?value:BigInt(value);if(bi>MAX||bi<MIN){throw new Error(`invalid int64: ${value}`)}return bi},uParse(value){const bi=typeof value=="bigint"?value:BigInt(value);if(bi>UMAX||bi<UMIN){throw new Error(`invalid uint64: ${value}`)}return bi},enc(value){dv.setBigInt64(0,this.parse(value),true);return{lo:dv.getInt32(0,true),hi:dv.getInt32(4,true)}},uEnc(value){dv.setBigInt64(0,this.uParse(value),true);return{lo:dv.getInt32(0,true),hi:dv.getInt32(4,true)}},dec(lo,hi){dv.setInt32(0,lo,true);dv.setInt32(4,hi,true);return dv.getBigInt64(0,true)},uDec(lo,hi){dv.setInt32(0,lo,true);dv.setInt32(4,hi,true);return dv.getBigUint64(0,true)}}}return{zero:"0",supported:false,parse(value){if(typeof value!="string"){value=value.toString()}assertInt64String(value);return value},uParse(value){if(typeof value!="string"){value=value.toString()}assertUInt64String(value);return value},enc(value){if(typeof value!="string"){value=value.toString()}assertInt64String(value);return int64FromString(value)},uEnc(value){if(typeof value!="string"){value=value.toString()}assertUInt64String(value);return int64FromString(value)},dec(lo,hi){return int64ToString(lo,hi)},uDec(lo,hi){return uInt64ToString(lo,hi)}}}function assertInt64String(value){if(!/^-?[0-9]+$/.test(value)){throw new Error("invalid int64: "+value)}}function assertUInt64String(value){if(!/^[0-9]+$/.test(value)){throw new Error("invalid uint64: "+value)}}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
const symbol=Symbol.for("@bufbuild/protobuf/text-encoding");function configureTextEncoding(textEncoding){globalThis[symbol]=textEncoding}function getTextEncoding(){if(globalThis[symbol]==undefined){const te=new globalThis.TextEncoder;const td=new globalThis.TextDecoder;globalThis[symbol]={encodeUtf8(text){return te.encode(text)},decodeUtf8(bytes){return td.decode(bytes)},checkUtf8(text){try{encodeURIComponent(text);return true}catch(e){return false}}}}return globalThis[symbol]}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
var WireType;(function(WireType){WireType[WireType["Varint"]=0]="Varint";WireType[WireType["Bit64"]=1]="Bit64";WireType[WireType["LengthDelimited"]=2]="LengthDelimited";WireType[WireType["StartGroup"]=3]="StartGroup";WireType[WireType["EndGroup"]=4]="EndGroup";WireType[WireType["Bit32"]=5]="Bit32"})(WireType||(WireType={}));const FLOAT32_MAX=34028234663852886e22;const FLOAT32_MIN=-34028234663852886e22;const UINT32_MAX=4294967295;const INT32_MAX=2147483647;const INT32_MIN=-2147483648;class BinaryWriter{constructor(encodeUtf8=getTextEncoding().encodeUtf8){this.encodeUtf8=encodeUtf8;this.stack=[];this.chunks=[];this.buf=[]}finish(){if(this.buf.length){this.chunks.push(new Uint8Array(this.buf));this.buf=[]}let len=0;for(let i=0;i<this.chunks.length;i++)len+=this.chunks[i].length;let bytes=new Uint8Array(len);let offset=0;for(let i=0;i<this.chunks.length;i++){bytes.set(this.chunks[i],offset);offset+=this.chunks[i].length}this.chunks=[];return bytes}fork(){this.stack.push({chunks:this.chunks,buf:this.buf});this.chunks=[];this.buf=[];return this}join(){let chunk=this.finish();let prev=this.stack.pop();if(!prev)throw new Error("invalid state, fork stack empty");this.chunks=prev.chunks;this.buf=prev.buf;this.uint32(chunk.byteLength);return this.raw(chunk)}tag(fieldNo,type){return this.uint32((fieldNo<<3|type)>>>0)}raw(chunk){if(this.buf.length){this.chunks.push(new Uint8Array(this.buf));this.buf=[]}this.chunks.push(chunk);return this}uint32(value){assertUInt32(value);while(value>127){this.buf.push(value&127|128);value=value>>>7}this.buf.push(value);return this}int32(value){assertInt32(value);varint32write(value,this.buf);return this}bool(value){this.buf.push(value?1:0);return this}bytes(value){this.uint32(value.byteLength);return this.raw(value)}string(value){let chunk=this.encodeUtf8(value);this.uint32(chunk.byteLength);return this.raw(chunk)}float(value){assertFloat32(value);let chunk=new Uint8Array(4);new DataView(chunk.buffer).setFloat32(0,value,true);return this.raw(chunk)}double(value){let chunk=new Uint8Array(8);new DataView(chunk.buffer).setFloat64(0,value,true);return this.raw(chunk)}fixed32(value){assertUInt32(value);let chunk=new Uint8Array(4);new DataView(chunk.buffer).setUint32(0,value,true);return this.raw(chunk)}sfixed32(value){assertInt32(value);let chunk=new Uint8Array(4);new DataView(chunk.buffer).setInt32(0,value,true);return this.raw(chunk)}sint32(value){assertInt32(value);value=(value<<1^value>>31)>>>0;varint32write(value,this.buf);return this}sfixed64(value){let chunk=new Uint8Array(8),view=new DataView(chunk.buffer),tc=protoInt64.enc(value);view.setInt32(0,tc.lo,true);view.setInt32(4,tc.hi,true);return this.raw(chunk)}fixed64(value){let chunk=new Uint8Array(8),view=new DataView(chunk.buffer),tc=protoInt64.uEnc(value);view.setInt32(0,tc.lo,true);view.setInt32(4,tc.hi,true);return this.raw(chunk)}int64(value){let tc=protoInt64.enc(value);varint64write(tc.lo,tc.hi,this.buf);return this}sint64(value){let tc=protoInt64.enc(value),sign=tc.hi>>31,lo=tc.lo<<1^sign,hi=(tc.hi<<1|tc.lo>>>31)^sign;varint64write(lo,hi,this.buf);return this}uint64(value){let tc=protoInt64.uEnc(value);varint64write(tc.lo,tc.hi,this.buf);return this}}class BinaryReader{constructor(buf,decodeUtf8=getTextEncoding().decodeUtf8){this.decodeUtf8=decodeUtf8;this.varint64=varint64read;this.uint32=varint32read;this.buf=buf;this.len=buf.length;this.pos=0;this.view=new DataView(buf.buffer,buf.byteOffset,buf.byteLength)}tag(){let tag=this.uint32(),fieldNo=tag>>>3,wireType=tag&7;if(fieldNo<=0||wireType<0||wireType>5)throw new Error("illegal tag: field no "+fieldNo+" wire type "+wireType);return[fieldNo,wireType]}skip(wireType,fieldNo){let start=this.pos;switch(wireType){case WireType.Varint:while(this.buf[this.pos++]&128){}break;case WireType.Bit64:this.pos+=4;case WireType.Bit32:this.pos+=4;break;case WireType.LengthDelimited:let len=this.uint32();this.pos+=len;break;case WireType.StartGroup:for(;;){const[fn,wt]=this.tag();if(wt===WireType.EndGroup){if(fieldNo!==undefined&&fn!==fieldNo){throw new Error("invalid end group tag")}break}this.skip(wt,fn)}break;default:throw new Error("cant skip wire type "+wireType)}this.assertBounds();return this.buf.subarray(start,this.pos)}assertBounds(){if(this.pos>this.len)throw new RangeError("premature EOF")}int32(){return this.uint32()|0}sint32(){let zze=this.uint32();return zze>>>1^-(zze&1)}int64(){return protoInt64.dec(...this.varint64())}uint64(){return protoInt64.uDec(...this.varint64())}sint64(){let[lo,hi]=this.varint64();let s=-(lo&1);lo=(lo>>>1|(hi&1)<<31)^s;hi=hi>>>1^s;return protoInt64.dec(lo,hi)}bool(){let[lo,hi]=this.varint64();return lo!==0||hi!==0}fixed32(){return this.view.getUint32((this.pos+=4)-4,true)}sfixed32(){return this.view.getInt32((this.pos+=4)-4,true)}fixed64(){return protoInt64.uDec(this.sfixed32(),this.sfixed32())}sfixed64(){return protoInt64.dec(this.sfixed32(),this.sfixed32())}float(){return this.view.getFloat32((this.pos+=4)-4,true)}double(){return this.view.getFloat64((this.pos+=8)-8,true)}bytes(){let len=this.uint32(),start=this.pos;this.pos+=len;this.assertBounds();return this.buf.subarray(start,start+len)}string(){return this.decodeUtf8(this.bytes())}}function assertInt32(arg){if(typeof arg=="string"){arg=Number(arg)}else if(typeof arg!="number"){throw new Error("invalid int32: "+typeof arg)}if(!Number.isInteger(arg)||arg>INT32_MAX||arg<INT32_MIN)throw new Error("invalid int32: "+arg)}function assertUInt32(arg){if(typeof arg=="string"){arg=Number(arg)}else if(typeof arg!="number"){throw new Error("invalid uint32: "+typeof arg)}if(!Number.isInteger(arg)||arg>UINT32_MAX||arg<0)throw new Error("invalid uint32: "+arg)}function assertFloat32(arg){if(typeof arg=="string"){const o=arg;arg=Number(arg);if(isNaN(arg)&&o!=="NaN"){throw new Error("invalid float32: "+o)}}else if(typeof arg!="number"){throw new Error("invalid float32: "+typeof arg)}if(Number.isFinite(arg)&&(arg>FLOAT32_MAX||arg<FLOAT32_MIN))throw new Error("invalid float32: "+arg)}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
function base64Decode(base64Str){const table=getDecodeTable();let es=base64Str.length*3/4;if(base64Str[base64Str.length-2]=="=")es-=2;else if(base64Str[base64Str.length-1]=="=")es-=1;let bytes=new Uint8Array(es),bytePos=0,groupPos=0,b,p=0;for(let i=0;i<base64Str.length;i++){b=table[base64Str.charCodeAt(i)];if(b===undefined){switch(base64Str[i]){case"=":groupPos=0;case"\n":case"\r":case"\t":case" ":continue;default:throw Error("invalid base64 string")}}switch(groupPos){case 0:p=b;groupPos=1;break;case 1:bytes[bytePos++]=p<<2|(b&48)>>4;p=b;groupPos=2;break;case 2:bytes[bytePos++]=(p&15)<<4|(b&60)>>2;p=b;groupPos=3;break;case 3:bytes[bytePos++]=(p&3)<<6|b;groupPos=0;break}}if(groupPos==1)throw Error("invalid base64 string");return bytes.subarray(0,bytePos)}function base64Encode(bytes,encoding="std"){const table=getEncodeTable(encoding);const pad=encoding=="std";let base64="",groupPos=0,b,p=0;for(let i=0;i<bytes.length;i++){b=bytes[i];switch(groupPos){case 0:base64+=table[b>>2];p=(b&3)<<4;groupPos=1;break;case 1:base64+=table[p|b>>4];p=(b&15)<<2;groupPos=2;break;case 2:base64+=table[p|b>>6];base64+=table[b&63];groupPos=0;break}}if(groupPos){base64+=table[p];if(pad){base64+="=";if(groupPos==1)base64+="="}}return base64}let encodeTableStd;let encodeTableUrl;let decodeTable;function getEncodeTable(encoding){if(!encodeTableStd){encodeTableStd="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/".split("");encodeTableUrl=encodeTableStd.slice(0,-2).concat("-","_")}return encoding=="url"?encodeTableUrl:encodeTableStd}function getDecodeTable(){if(!decodeTable){decodeTable=[];const encodeTable=getEncodeTable("std");for(let i=0;i<encodeTable.length;i++)decodeTable[encodeTable[i].charCodeAt(0)]=i;decodeTable["-".charCodeAt(0)]=encodeTable.indexOf("+");decodeTable["_".charCodeAt(0)]=encodeTable.indexOf("/")}return decodeTable}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
var ScalarType;(function(ScalarType){ScalarType[ScalarType["DOUBLE"]=1]="DOUBLE";ScalarType[ScalarType["FLOAT"]=2]="FLOAT";ScalarType[ScalarType["INT64"]=3]="INT64";ScalarType[ScalarType["UINT64"]=4]="UINT64";ScalarType[ScalarType["INT32"]=5]="INT32";ScalarType[ScalarType["FIXED64"]=6]="FIXED64";ScalarType[ScalarType["FIXED32"]=7]="FIXED32";ScalarType[ScalarType["BOOL"]=8]="BOOL";ScalarType[ScalarType["STRING"]=9]="STRING";ScalarType[ScalarType["BYTES"]=12]="BYTES";ScalarType[ScalarType["UINT32"]=13]="UINT32";ScalarType[ScalarType["SFIXED32"]=15]="SFIXED32";ScalarType[ScalarType["SFIXED64"]=16]="SFIXED64";ScalarType[ScalarType["SINT32"]=17]="SINT32";ScalarType[ScalarType["SINT64"]=18]="SINT64"})(ScalarType||(ScalarType={}));
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
function parseTextFormatEnumValue(descEnum,value){const enumValue=descEnum.values.find((v=>v.name===value));if(!enumValue){throw new Error(`cannot parse ${descEnum} default value: ${value}`)}return enumValue.number}function parseTextFormatScalarValue(type,value){switch(type){case ScalarType.STRING:return value;case ScalarType.BYTES:{const u=unescapeBytesDefaultValue(value);if(u===false){throw new Error(`cannot parse ${ScalarType[type]} default value: ${value}`)}return u}case ScalarType.INT64:case ScalarType.SFIXED64:case ScalarType.SINT64:return protoInt64.parse(value);case ScalarType.UINT64:case ScalarType.FIXED64:return protoInt64.uParse(value);case ScalarType.DOUBLE:case ScalarType.FLOAT:switch(value){case"inf":return Number.POSITIVE_INFINITY;case"-inf":return Number.NEGATIVE_INFINITY;case"nan":return Number.NaN;default:return parseFloat(value)}case ScalarType.BOOL:return value==="true";case ScalarType.INT32:case ScalarType.UINT32:case ScalarType.SINT32:case ScalarType.FIXED32:case ScalarType.SFIXED32:return parseInt(value,10)}}function unescapeBytesDefaultValue(str){const b=[];const input={tail:str,c:"",next(){if(this.tail.length==0){return false}this.c=this.tail[0];this.tail=this.tail.substring(1);return true},take(n){if(this.tail.length>=n){const r=this.tail.substring(0,n);this.tail=this.tail.substring(n);return r}return false}};while(input.next()){switch(input.c){case"\\":if(input.next()){switch(input.c){case"\\":b.push(input.c.charCodeAt(0));break;case"b":b.push(8);break;case"f":b.push(12);break;case"n":b.push(10);break;case"r":b.push(13);break;case"t":b.push(9);break;case"v":b.push(11);break;case"0":case"1":case"2":case"3":case"4":case"5":case"6":case"7":{const s=input.c;const t=input.take(2);if(t===false){return false}const n=parseInt(s+t,8);if(isNaN(n)){return false}b.push(n);break}case"x":{const s=input.c;const t=input.take(2);if(t===false){return false}const n=parseInt(s+t,16);if(isNaN(n)){return false}b.push(n);break}case"u":{const s=input.c;const t=input.take(4);if(t===false){return false}const n=parseInt(s+t,16);if(isNaN(n)){return false}const chunk=new Uint8Array(4);const view=new DataView(chunk.buffer);view.setInt32(0,n,true);b.push(chunk[0],chunk[1],chunk[2],chunk[3]);break}case"U":{const s=input.c;const t=input.take(8);if(t===false){return false}const tc=protoInt64.uEnc(s+t);const chunk=new Uint8Array(8);const view=new DataView(chunk.buffer);view.setInt32(0,tc.lo,true);view.setInt32(4,tc.hi,true);b.push(chunk[0],chunk[1],chunk[2],chunk[3],chunk[4],chunk[5],chunk[6],chunk[7]);break}}}break;default:b.push(input.c.charCodeAt(0))}}return new Uint8Array(b)}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
function isMessage(arg,schema){const isMessage=arg!==null&&typeof arg=="object"&&"$typeName"in arg&&typeof arg.$typeName=="string";if(!isMessage){return false}if(schema===undefined){return true}return schema.typeName===arg.$typeName}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
class FieldError extends Error{constructor(fieldOrOneof,message,name="FieldValueInvalidError"){super(message);this.name=name;this.field=()=>fieldOrOneof}}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
function scalarZeroValue(type,longAsString){switch(type){case ScalarType.STRING:return"";case ScalarType.BOOL:return false;default:return 0;case ScalarType.DOUBLE:case ScalarType.FLOAT:return 0;case ScalarType.INT64:case ScalarType.UINT64:case ScalarType.SFIXED64:case ScalarType.FIXED64:case ScalarType.SINT64:return longAsString?"0":protoInt64.zero;case ScalarType.BYTES:return new Uint8Array(0)}}function isScalarZeroValue(type,value){switch(type){case ScalarType.BOOL:return value===false;case ScalarType.STRING:return value==="";case ScalarType.BYTES:return value instanceof Uint8Array&&!value.byteLength;default:return value==0}}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
const IMPLICIT$1=2;const unsafeLocal=Symbol.for("reflect unsafe local");function unsafeOneofCase(target,oneof){const c=target[oneof.localName].case;if(c===undefined){return c}return oneof.fields.find((f=>f.localName===c))}function unsafeIsSet(target,field){const name=field.localName;if(field.oneof){return target[field.oneof.localName].case===name}if(field.presence!=IMPLICIT$1){return target[name]!==undefined&&Object.prototype.hasOwnProperty.call(target,name)}switch(field.fieldKind){case"list":return target[name].length>0;case"map":return Object.keys(target[name]).length>0;case"scalar":return!isScalarZeroValue(field.scalar,target[name]);case"enum":return target[name]!==field.enum.values[0].number}throw new Error("message field with implicit presence")}function unsafeGet(target,field){if(field.oneof){const oneof=target[field.oneof.localName];if(oneof.case===field.localName){return oneof.value}return undefined}return target[field.localName]}function unsafeSet(target,field,value){if(field.oneof){target[field.oneof.localName]={case:field.localName,value:value}}else{target[field.localName]=value}}function unsafeClear(target,field){const name=field.localName;if(field.oneof){const oneofLocalName=field.oneof.localName;if(target[oneofLocalName].case===name){target[oneofLocalName]={case:undefined}}}else if(field.presence!=IMPLICIT$1){delete target[name]}else{switch(field.fieldKind){case"map":target[name]={};break;case"list":target[name]=[];break;case"enum":target[name]=field.enum.values[0].number;break;case"scalar":target[name]=scalarZeroValue(field.scalar,field.longAsString);break}}}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
function isObject(arg){return arg!==null&&typeof arg=="object"&&!Array.isArray(arg)}function isReflectList(arg,field){var _a,_b,_c,_d;if(isObject(arg)&&unsafeLocal in arg&&"add"in arg&&"field"in arg&&typeof arg.field=="function"){if(field!==undefined){const a=field,b=arg.field();return a.listKind==b.listKind&&a.scalar===b.scalar&&((_a=a.message)===null||_a===void 0?void 0:_a.typeName)===((_b=b.message)===null||_b===void 0?void 0:_b.typeName)&&((_c=a.enum)===null||_c===void 0?void 0:_c.typeName)===((_d=b.enum)===null||_d===void 0?void 0:_d.typeName)}return true}return false}function isReflectMap(arg,field){var _a,_b,_c,_d;if(isObject(arg)&&unsafeLocal in arg&&"has"in arg&&"field"in arg&&typeof arg.field=="function"){if(field!==undefined){const a=field,b=arg.field();return a.mapKey===b.mapKey&&a.mapKind==b.mapKind&&a.scalar===b.scalar&&((_a=a.message)===null||_a===void 0?void 0:_a.typeName)===((_b=b.message)===null||_b===void 0?void 0:_b.typeName)&&((_c=a.enum)===null||_c===void 0?void 0:_c.typeName)===((_d=b.enum)===null||_d===void 0?void 0:_d.typeName)}return true}return false}function isReflectMessage(arg,messageDesc){return isObject(arg)&&unsafeLocal in arg&&"desc"in arg&&isObject(arg.desc)&&arg.desc.kind==="message"&&(messageDesc===undefined||arg.desc.typeName==messageDesc.typeName)}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
function checkField(field,value){const check=field.fieldKind=="list"?isReflectList(value,field):field.fieldKind=="map"?isReflectMap(value,field):checkSingular(field,value);if(check===true){return undefined}let reason;switch(field.fieldKind){case"list":reason=`expected ${formatReflectList(field)}, got ${formatVal(value)}`;break;case"map":reason=`expected ${formatReflectMap(field)}, got ${formatVal(value)}`;break;default:{reason=reasonSingular(field,value,check)}}return new FieldError(field,reason)}function checkListItem(field,index,value){const check=checkSingular(field,value);if(check!==true){return new FieldError(field,`list item #${index+1}: ${reasonSingular(field,value,check)}`)}return undefined}function checkMapEntry(field,key,value){const checkKey=checkScalarValue(key,field.mapKey);if(checkKey!==true){return new FieldError(field,`invalid map key: ${reasonSingular({scalar:field.mapKey},key,checkKey)}`)}const checkVal=checkSingular(field,value);if(checkVal!==true){return new FieldError(field,`map entry ${formatVal(key)}: ${reasonSingular(field,value,checkVal)}`)}return undefined}function checkSingular(field,value){if(field.scalar!==undefined){return checkScalarValue(value,field.scalar)}if(field.enum!==undefined){if(field.enum.open){return Number.isInteger(value)}return field.enum.values.some((v=>v.number===value))}return isReflectMessage(value,field.message)}function checkScalarValue(value,scalar){switch(scalar){case ScalarType.DOUBLE:return typeof value=="number";case ScalarType.FLOAT:if(typeof value!="number"){return false}if(Number.isNaN(value)||!Number.isFinite(value)){return true}if(value>FLOAT32_MAX||value<FLOAT32_MIN){return`${value.toFixed()} out of range`}return true;case ScalarType.INT32:case ScalarType.SFIXED32:case ScalarType.SINT32:if(typeof value!=="number"||!Number.isInteger(value)){return false}if(value>INT32_MAX||value<INT32_MIN){return`${value.toFixed()} out of range`}return true;case ScalarType.FIXED32:case ScalarType.UINT32:if(typeof value!=="number"||!Number.isInteger(value)){return false}if(value>UINT32_MAX||value<0){return`${value.toFixed()} out of range`}return true;case ScalarType.BOOL:return typeof value=="boolean";case ScalarType.STRING:if(typeof value!="string"){return false}return getTextEncoding().checkUtf8(value)||"invalid UTF8";case ScalarType.BYTES:return value instanceof Uint8Array;case ScalarType.INT64:case ScalarType.SFIXED64:case ScalarType.SINT64:if(typeof value=="bigint"||typeof value=="number"||typeof value=="string"&&value.length>0){try{protoInt64.parse(value);return true}catch(e){return`${value} out of range`}}return false;case ScalarType.FIXED64:case ScalarType.UINT64:if(typeof value=="bigint"||typeof value=="number"||typeof value=="string"&&value.length>0){try{protoInt64.uParse(value);return true}catch(e){return`${value} out of range`}}return false}}function reasonSingular(field,val,details){details=typeof details=="string"?`: ${details}`:`, got ${formatVal(val)}`;if(field.scalar!==undefined){return`expected ${scalarTypeDescription(field.scalar)}`+details}else if(field.enum!==undefined){return`expected ${field.enum.toString()}`+details}return`expected ${formatReflectMessage(field.message)}`+details}function formatVal(val){switch(typeof val){case"object":if(val===null){return"null"}if(val instanceof Uint8Array){return`Uint8Array(${val.length})`}if(Array.isArray(val)){return`Array(${val.length})`}if(isReflectList(val)){return formatReflectList(val.field())}if(isReflectMap(val)){return formatReflectMap(val.field())}if(isReflectMessage(val)){return formatReflectMessage(val.desc)}if(isMessage(val)){return`message ${val.$typeName}`}return"object";case"string":return val.length>30?"string":`"${val.split('"').join('\\"')}"`;case"boolean":return String(val);case"number":return String(val);case"bigint":return String(val)+"n";default:return typeof val}}function formatReflectMessage(desc){return`ReflectMessage (${desc.typeName})`}function formatReflectList(field){switch(field.listKind){case"message":return`ReflectList (${field.message.toString()})`;case"enum":return`ReflectList (${field.enum.toString()})`;case"scalar":return`ReflectList (${ScalarType[field.scalar]})`}}function formatReflectMap(field){switch(field.mapKind){case"message":return`ReflectMap (${ScalarType[field.mapKey]}, ${field.message.toString()})`;case"enum":return`ReflectMap (${ScalarType[field.mapKey]}, ${field.enum.toString()})`;case"scalar":return`ReflectMap (${ScalarType[field.mapKey]}, ${ScalarType[field.scalar]})`}}function scalarTypeDescription(scalar){switch(scalar){case ScalarType.STRING:return"string";case ScalarType.BOOL:return"boolean";case ScalarType.INT64:case ScalarType.SINT64:case ScalarType.SFIXED64:return"bigint (int64)";case ScalarType.UINT64:case ScalarType.FIXED64:return"bigint (uint64)";case ScalarType.BYTES:return"Uint8Array";case ScalarType.DOUBLE:return"number (float64)";case ScalarType.FLOAT:return"number (float32)";case ScalarType.FIXED32:case ScalarType.UINT32:return"number (uint32)";case ScalarType.INT32:case ScalarType.SFIXED32:case ScalarType.SINT32:return"number (int32)"}}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
function isWrapper(arg){return isWrapperTypeName(arg.$typeName)}function isWrapperDesc(messageDesc){const f=messageDesc.fields[0];return isWrapperTypeName(messageDesc.typeName)&&f!==undefined&&f.fieldKind=="scalar"&&f.name=="value"&&f.number==1}function isWrapperTypeName(name){return name.startsWith("google.protobuf.")&&["DoubleValue","FloatValue","Int64Value","UInt64Value","Int32Value","UInt32Value","BoolValue","StringValue","BytesValue"].includes(name.substring(16))}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
const EDITION_PROTO3=999;const EDITION_PROTO2=998;const IMPLICIT=2;function create(schema,init){if(isMessage(init,schema)){return init}const message=createZeroMessage(schema);return message}const tokenZeroMessageField=Symbol();const messagePrototypes=new WeakMap;function createZeroMessage(desc){let msg;if(!needsPrototypeChain(desc)){msg={$typeName:desc.typeName};for(const member of desc.members){if(member.kind=="oneof"||member.presence==IMPLICIT){msg[member.localName]=createZeroField(member)}}}else{const cached=messagePrototypes.get(desc);let prototype;let members;if(cached){({prototype:prototype,members:members}=cached)}else{prototype={};members=new Set;for(const member of desc.members){if(member.kind=="oneof"){continue}if(member.fieldKind!="scalar"&&member.fieldKind!="enum"){continue}if(member.presence==IMPLICIT){continue}members.add(member);prototype[member.localName]=createZeroField(member)}messagePrototypes.set(desc,{prototype:prototype,members:members})}msg=Object.create(prototype);msg.$typeName=desc.typeName;for(const member of desc.members){if(members.has(member)){continue}if(member.kind=="field"){if(member.fieldKind=="message"){continue}if(member.fieldKind=="scalar"||member.fieldKind=="enum"){if(member.presence!=IMPLICIT){continue}}}msg[member.localName]=createZeroField(member)}}return msg}function needsPrototypeChain(desc){switch(desc.file.edition){case EDITION_PROTO3:return false;case EDITION_PROTO2:return true;default:return desc.fields.some((f=>f.presence!=IMPLICIT&&f.fieldKind!="message"&&!f.oneof))}}function createZeroField(field){if(field.kind=="oneof"){return{case:undefined}}if(field.fieldKind=="list"){return[]}if(field.fieldKind=="map"){return{}}if(field.fieldKind=="message"){return tokenZeroMessageField}const defaultValue=field.getDefaultValue();if(defaultValue!==undefined){return field.fieldKind=="scalar"&&field.longAsString?defaultValue.toString():defaultValue}return field.fieldKind=="scalar"?scalarZeroValue(field.scalar,field.longAsString):field.enum.values[0].number}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
function reflect(messageDesc,message,check=true){return new ReflectMessageImpl(messageDesc,message,check)}class ReflectMessageImpl{get sortedFields(){var _a;return(_a=this._sortedFields)!==null&&_a!==void 0?_a:this._sortedFields=this.desc.fields.concat().sort(((a,b)=>a.number-b.number))}constructor(messageDesc,message,check=true){this.lists=new Map;this.maps=new Map;this.check=check;this.desc=messageDesc;this.message=this[unsafeLocal]=message!==null&&message!==void 0?message:create(messageDesc);this.fields=messageDesc.fields;this.oneofs=messageDesc.oneofs;this.members=messageDesc.members}findNumber(number){if(!this._fieldsByNumber){this._fieldsByNumber=new Map(this.desc.fields.map((f=>[f.number,f])))}return this._fieldsByNumber.get(number)}oneofCase(oneof){assertOwn(this.message,oneof);return unsafeOneofCase(this.message,oneof)}isSet(field){assertOwn(this.message,field);return unsafeIsSet(this.message,field)}clear(field){assertOwn(this.message,field);unsafeClear(this.message,field)}get(field){assertOwn(this.message,field);const value=unsafeGet(this.message,field);switch(field.fieldKind){case"list":let list=this.lists.get(field);if(!list||list[unsafeLocal]!==value){this.lists.set(field,list=new ReflectListImpl(field,value,this.check))}return list;case"map":let map=this.maps.get(field);if(!map||map[unsafeLocal]!==value){this.maps.set(field,map=new ReflectMapImpl(field,value,this.check))}return map;case"message":return messageToReflect(field,value,this.check);case"scalar":return value===undefined?scalarZeroValue(field.scalar,false):longToReflect(field,value);case"enum":return value!==null&&value!==void 0?value:field.enum.values[0].number}}set(field,value){assertOwn(this.message,field);if(this.check){const err=checkField(field,value);if(err){throw err}}let local;if(field.fieldKind=="message"){local=messageToLocal(field,value)}else if(isReflectMap(value)||isReflectList(value)){local=value[unsafeLocal]}else{local=longToLocal(field,value)}unsafeSet(this.message,field,local)}getUnknown(){return this.message.$unknown}setUnknown(value){this.message.$unknown=value}}function assertOwn(owner,member){if(member.parent.typeName!==owner.$typeName){throw new FieldError(member,`cannot use ${member.toString()} with message ${owner.$typeName}`,"ForeignFieldError")}}class ReflectListImpl{field(){return this._field}get size(){return this._arr.length}constructor(field,unsafeInput,check){this._field=field;this._arr=this[unsafeLocal]=unsafeInput;this.check=check}get(index){const item=this._arr[index];return item===undefined?undefined:listItemToReflect(this._field,item,this.check)}set(index,item){if(index<0||index>=this._arr.length){throw new FieldError(this._field,`list item #${index+1}: out of range`)}if(this.check){const err=checkListItem(this._field,index,item);if(err){throw err}}this._arr[index]=listItemToLocal(this._field,item)}add(item){if(this.check){const err=checkListItem(this._field,this._arr.length,item);if(err){throw err}}this._arr.push(listItemToLocal(this._field,item));return undefined}clear(){this._arr.splice(0,this._arr.length)}[Symbol.iterator](){return this.values()}keys(){return this._arr.keys()}*values(){for(const item of this._arr){yield listItemToReflect(this._field,item,this.check)}}*entries(){for(let i=0;i<this._arr.length;i++){yield[i,listItemToReflect(this._field,this._arr[i],this.check)]}}}class ReflectMapImpl{constructor(field,unsafeInput,check=true){this.obj=this[unsafeLocal]=unsafeInput!==null&&unsafeInput!==void 0?unsafeInput:{};this.check=check;this._field=field}field(){return this._field}set(key,value){if(this.check){const err=checkMapEntry(this._field,key,value);if(err){throw err}}this.obj[mapKeyToLocal(key)]=mapValueToLocal(this._field,value);return this}delete(key){const k=mapKeyToLocal(key);const has=Object.prototype.hasOwnProperty.call(this.obj,k);if(has){delete this.obj[k]}return has}clear(){for(const key of Object.keys(this.obj)){delete this.obj[key]}}get(key){let val=this.obj[mapKeyToLocal(key)];if(val!==undefined){val=mapValueToReflect(this._field,val,this.check)}return val}has(key){return Object.prototype.hasOwnProperty.call(this.obj,mapKeyToLocal(key))}*keys(){for(const objKey of Object.keys(this.obj)){yield mapKeyToReflect(objKey,this._field.mapKey)}}*entries(){for(const objEntry of Object.entries(this.obj)){yield[mapKeyToReflect(objEntry[0],this._field.mapKey),mapValueToReflect(this._field,objEntry[1],this.check)]}}[Symbol.iterator](){return this.entries()}get size(){return Object.keys(this.obj).length}*values(){for(const val of Object.values(this.obj)){yield mapValueToReflect(this._field,val,this.check)}}forEach(callbackfn,thisArg){for(const mapEntry of this.entries()){callbackfn.call(thisArg,mapEntry[1],mapEntry[0],this)}}}function messageToLocal(field,value){if(!isReflectMessage(value)){return value}if(isWrapper(value.message)&&!field.oneof&&field.fieldKind=="message"){return value.message.value}if(value.desc.typeName=="google.protobuf.Struct"&&field.parent.typeName!="google.protobuf.Value"){return wktStructToLocal(value.message)}return value.message}function messageToReflect(field,value,check){if(value!==undefined){if(isWrapperDesc(field.message)&&!field.oneof&&field.fieldKind=="message"){value={$typeName:field.message.typeName,value:longToReflect(field.message.fields[0],value)}}else if(field.message.typeName=="google.protobuf.Struct"&&field.parent.typeName!="google.protobuf.Value"&&isObject(value)){value=wktStructToReflect(value)}}return new ReflectMessageImpl(field.message,value,check)}function listItemToLocal(field,value){if(field.listKind=="message"){return messageToLocal(field,value)}return longToLocal(field,value)}function listItemToReflect(field,value,check){if(field.listKind=="message"){return messageToReflect(field,value,check)}return longToReflect(field,value)}function mapValueToLocal(field,value){if(field.mapKind=="message"){return messageToLocal(field,value)}return longToLocal(field,value)}function mapValueToReflect(field,value,check){if(field.mapKind=="message"){return messageToReflect(field,value,check)}return value}function mapKeyToLocal(key){return typeof key=="string"||typeof key=="number"?key:String(key)}function mapKeyToReflect(key,type){switch(type){case ScalarType.STRING:return key;case ScalarType.INT32:case ScalarType.FIXED32:case ScalarType.UINT32:case ScalarType.SFIXED32:case ScalarType.SINT32:{const n=Number.parseInt(key);if(Number.isFinite(n)){return n}break}case ScalarType.BOOL:switch(key){case"true":return true;case"false":return false}break;case ScalarType.UINT64:case ScalarType.FIXED64:try{return protoInt64.uParse(key)}catch(_a){}break;default:try{return protoInt64.parse(key)}catch(_b){}break}return key}function longToReflect(field,value){switch(field.scalar){case ScalarType.INT64:case ScalarType.SFIXED64:case ScalarType.SINT64:if("longAsString"in field&&field.longAsString&&typeof value=="string"){value=protoInt64.parse(value)}break;case ScalarType.FIXED64:case ScalarType.UINT64:if("longAsString"in field&&field.longAsString&&typeof value=="string"){value=protoInt64.uParse(value)}break}return value}function longToLocal(field,value){switch(field.scalar){case ScalarType.INT64:case ScalarType.SFIXED64:case ScalarType.SINT64:if("longAsString"in field&&field.longAsString){value=String(value)}else if(typeof value=="string"||typeof value=="number"){value=protoInt64.parse(value)}break;case ScalarType.FIXED64:case ScalarType.UINT64:if("longAsString"in field&&field.longAsString){value=String(value)}else if(typeof value=="string"||typeof value=="number"){value=protoInt64.uParse(value)}break}return value}function wktStructToReflect(json){const struct={$typeName:"google.protobuf.Struct",fields:{}};if(isObject(json)){for(const[k,v]of Object.entries(json)){struct.fields[k]=wktValueToReflect(v)}}return struct}function wktStructToLocal(val){const json={};for(const[k,v]of Object.entries(val.fields)){json[k]=wktValueToLocal(v)}return json}function wktValueToLocal(val){switch(val.kind.case){case"structValue":return wktStructToLocal(val.kind.value);case"listValue":return val.kind.value.values.map(wktValueToLocal);case"nullValue":case undefined:return null;default:return val.kind.value}}function wktValueToReflect(json){const value={$typeName:"google.protobuf.Value",kind:{case:undefined}};switch(typeof json){case"number":value.kind={case:"numberValue",value:json};break;case"string":value.kind={case:"stringValue",value:json};break;case"boolean":value.kind={case:"boolValue",value:json};break;case"object":if(json===null){const nullValue=0;value.kind={case:"nullValue",value:nullValue}}else if(Array.isArray(json)){const listValue={$typeName:"google.protobuf.ListValue",values:[]};if(Array.isArray(json)){for(const e of json){listValue.values.push(wktValueToReflect(e))}}value.kind={case:"listValue",value:listValue}}else{value.kind={case:"structValue",value:wktStructToReflect(json)}}break}return value}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
const LEGACY_REQUIRED=3;const writeDefaults={writeUnknownFields:true};function makeWriteOptions(options){return options?Object.assign(Object.assign({},writeDefaults),options):writeDefaults}function toBinary(schema,message,options){return writeFields(new BinaryWriter,makeWriteOptions(options),reflect(schema,message)).finish()}function writeFields(writer,opts,msg){var _a;for(const f of msg.sortedFields){if(!msg.isSet(f)){if(f.presence==LEGACY_REQUIRED){throw new Error(`cannot encode field ${msg.desc.typeName}.${f.name} to binary: required field not set`)}continue}writeField(writer,opts,msg,f)}if(opts.writeUnknownFields){for(const{no:no,wireType:wireType,data:data}of(_a=msg.getUnknown())!==null&&_a!==void 0?_a:[]){writer.tag(no,wireType).raw(data)}}return writer}function writeField(writer,opts,msg,field){var _a;switch(field.fieldKind){case"scalar":case"enum":writeScalar(writer,msg.desc.typeName,field.name,(_a=field.scalar)!==null&&_a!==void 0?_a:ScalarType.INT32,field.number,msg.get(field));break;case"list":writeListField(writer,opts,field,msg.get(field));break;case"message":writeMessageField(writer,opts,field,msg.get(field));break;case"map":for(const[key,val]of msg.get(field)){writeMapEntry(writer,opts,field,key,val)}break}}function writeScalar(writer,msgName,fieldName,scalarType,fieldNo,value){writeScalarValue(writer.tag(fieldNo,writeTypeOfScalar(scalarType)),msgName,fieldName,scalarType,value)}function writeMessageField(writer,opts,field,message){if(field.delimitedEncoding){writeFields(writer.tag(field.number,WireType.StartGroup),opts,message).tag(field.number,WireType.EndGroup)}else{writeFields(writer.tag(field.number,WireType.LengthDelimited).fork(),opts,message).join()}}function writeListField(writer,opts,field,list){var _a;if(field.listKind=="message"){for(const item of list){writeMessageField(writer,opts,field,item)}return}const scalarType=(_a=field.scalar)!==null&&_a!==void 0?_a:ScalarType.INT32;if(field.packed){if(!list.size){return}writer.tag(field.number,WireType.LengthDelimited).fork();for(const item of list){writeScalarValue(writer,field.parent.typeName,field.name,scalarType,item)}writer.join();return}for(const item of list){writeScalar(writer,field.parent.typeName,field.name,scalarType,field.number,item)}}function writeMapEntry(writer,opts,field,key,value){var _a;writer.tag(field.number,WireType.LengthDelimited).fork();writeScalar(writer,field.parent.typeName,field.name,field.mapKey,1,key);switch(field.mapKind){case"scalar":case"enum":writeScalar(writer,field.parent.typeName,field.name,(_a=field.scalar)!==null&&_a!==void 0?_a:ScalarType.INT32,2,value);break;case"message":writeFields(writer.tag(2,WireType.LengthDelimited).fork(),opts,value).join();break}writer.join()}function writeScalarValue(writer,msgName,fieldName,type,value){try{switch(type){case ScalarType.STRING:writer.string(value);break;case ScalarType.BOOL:writer.bool(value);break;case ScalarType.DOUBLE:writer.double(value);break;case ScalarType.FLOAT:writer.float(value);break;case ScalarType.INT32:writer.int32(value);break;case ScalarType.INT64:writer.int64(value);break;case ScalarType.UINT64:writer.uint64(value);break;case ScalarType.FIXED64:writer.fixed64(value);break;case ScalarType.BYTES:writer.bytes(value);break;case ScalarType.FIXED32:writer.fixed32(value);break;case ScalarType.SFIXED32:writer.sfixed32(value);break;case ScalarType.SFIXED64:writer.sfixed64(value);break;case ScalarType.SINT64:writer.sint64(value);break;case ScalarType.UINT32:writer.uint32(value);break;case ScalarType.SINT32:writer.sint32(value);break}}catch(e){if(e instanceof Error){throw new Error(`cannot encode field ${msgName}.${fieldName} to binary: ${e.message}`)}throw e}}function writeTypeOfScalar(type){switch(type){case ScalarType.BYTES:case ScalarType.STRING:return WireType.LengthDelimited;case ScalarType.DOUBLE:case ScalarType.FIXED64:case ScalarType.SFIXED64:return WireType.Bit64;case ScalarType.FIXED32:case ScalarType.SFIXED32:case ScalarType.FLOAT:return WireType.Bit32;default:return WireType.Varint}}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
const readDefaults={readUnknownFields:true};function makeReadOptions(options){return options?Object.assign(Object.assign({},readDefaults),options):readDefaults}function fromBinary(schema,bytes,options){const msg=reflect(schema,undefined,false);readMessage(msg,new BinaryReader(bytes),makeReadOptions(options),false,bytes.byteLength);return msg.message}function readMessage(message,reader,options,delimited,lengthOrDelimitedFieldNo){var _a;const end=delimited?reader.len:reader.pos+lengthOrDelimitedFieldNo;let fieldNo,wireType;const unknownFields=(_a=message.getUnknown())!==null&&_a!==void 0?_a:[];while(reader.pos<end){[fieldNo,wireType]=reader.tag();if(delimited&&wireType==WireType.EndGroup){break}const field=message.findNumber(fieldNo);if(!field){const data=reader.skip(wireType,fieldNo);if(options.readUnknownFields){unknownFields.push({no:fieldNo,wireType:wireType,data:data})}continue}readField(message,reader,field,wireType,options)}if(delimited){if(wireType!=WireType.EndGroup||fieldNo!==lengthOrDelimitedFieldNo){throw new Error(`invalid end group tag`)}}if(unknownFields.length>0){message.setUnknown(unknownFields)}}function readField(message,reader,field,wireType,options){switch(field.fieldKind){case"scalar":message.set(field,readScalar(reader,field.scalar));break;case"enum":message.set(field,readScalar(reader,ScalarType.INT32));break;case"message":message.set(field,readMessageField(reader,options,field,message.get(field)));break;case"list":readListField(reader,wireType,message.get(field),options);break;case"map":readMapEntry(reader,message.get(field),options);break}}function readMapEntry(reader,map,options){const field=map.field();let key,val;const end=reader.pos+reader.uint32();while(reader.pos<end){const[fieldNo]=reader.tag();switch(fieldNo){case 1:key=readScalar(reader,field.mapKey);break;case 2:switch(field.mapKind){case"scalar":val=readScalar(reader,field.scalar);break;case"enum":val=reader.int32();break;case"message":val=readMessageField(reader,options,field);break}break}}if(key===undefined){key=scalarZeroValue(field.mapKey,false)}if(val===undefined){switch(field.mapKind){case"scalar":val=scalarZeroValue(field.scalar,false);break;case"enum":val=field.enum.values[0].number;break;case"message":val=reflect(field.message,undefined,false);break}}map.set(key,val)}function readListField(reader,wireType,list,options){var _a;const field=list.field();if(field.listKind==="message"){list.add(readMessageField(reader,options,field));return}const scalarType=(_a=field.scalar)!==null&&_a!==void 0?_a:ScalarType.INT32;const packed=wireType==WireType.LengthDelimited&&scalarType!=ScalarType.STRING&&scalarType!=ScalarType.BYTES;if(!packed){list.add(readScalar(reader,scalarType));return}const e=reader.uint32()+reader.pos;while(reader.pos<e){list.add(readScalar(reader,scalarType))}}function readMessageField(reader,options,field,mergeMessage){const delimited=field.delimitedEncoding;const message=mergeMessage!==null&&mergeMessage!==void 0?mergeMessage:reflect(field.message,undefined,false);readMessage(message,reader,options,delimited,delimited?field.number:reader.uint32());return message}function readScalar(reader,type){switch(type){case ScalarType.STRING:return reader.string();case ScalarType.BOOL:return reader.bool();case ScalarType.DOUBLE:return reader.double();case ScalarType.FLOAT:return reader.float();case ScalarType.INT32:return reader.int32();case ScalarType.INT64:return reader.int64();case ScalarType.UINT64:return reader.uint64();case ScalarType.FIXED64:return reader.fixed64();case ScalarType.BYTES:return reader.bytes();case ScalarType.FIXED32:return reader.fixed32();case ScalarType.SFIXED32:return reader.sfixed32();case ScalarType.SFIXED64:return reader.sfixed64();case ScalarType.SINT64:return reader.sint64();case ScalarType.UINT32:return reader.uint32();case ScalarType.SINT32:return reader.sint32()}}
// Copyright 2021-2025 Buf Technologies, Inc.
//      http://www.apache.org/licenses/LICENSE-2.0
var __asyncValues=undefined&&undefined.__asyncValues||function(o){if(!Symbol.asyncIterator)throw new TypeError("Symbol.asyncIterator is not defined.");var m=o[Symbol.asyncIterator],i;return m?m.call(o):(o=typeof __values==="function"?__values(o):o[Symbol.iterator](),i={},verb("next"),verb("throw"),verb("return"),i[Symbol.asyncIterator]=function(){return this},i);function verb(n){i[n]=o[n]&&function(v){return new Promise((function(resolve,reject){v=o[n](v),settle(resolve,reject,v.done,v.value)}))}}function settle(resolve,reject,d,v){Promise.resolve(v).then((function(v){resolve({value:v,done:d})}),reject)}};var __await=undefined&&undefined.__await||function(v){return this instanceof __await?(this.v=v,this):new __await(v)};var __asyncGenerator=undefined&&undefined.__asyncGenerator||function(thisArg,_arguments,generator){if(!Symbol.asyncIterator)throw new TypeError("Symbol.asyncIterator is not defined.");var g=generator.apply(thisArg,_arguments||[]),i,q=[];return i=Object.create((typeof AsyncIterator==="function"?AsyncIterator:Object).prototype),verb("next"),verb("throw"),verb("return",awaitReturn),i[Symbol.asyncIterator]=function(){return this},i;function awaitReturn(f){return function(v){return Promise.resolve(v).then(f,reject)}}function verb(n,f){if(g[n]){i[n]=function(v){return new Promise((function(a,b){q.push([n,v,a,b])>1||resume(n,v)}))};if(f)i[n]=f(i[n])}}function resume(n,v){try{step(g[n](v))}catch(e){settle(q[0][3],e)}}function step(r){r.value instanceof __await?Promise.resolve(r.value.v).then(fulfill,reject):settle(q[0][2],r)}function fulfill(value){resume("next",value)}function reject(value){resume("throw",value)}function settle(f,v){if(f(v),q.shift(),q.length)resume(q[0][0],q[0][1])}};function sizeDelimitedEncode(messageDesc,message,options){const writer=new BinaryWriter;writer.bytes(toBinary(messageDesc,message,options));return writer.finish()}function sizeDelimitedDecodeStream(messageDesc,iterable,options){return __asyncGenerator(this,arguments,(function*sizeDelimitedDecodeStream_1(){var _a,e_1,_b,_c;function append(buffer,chunk){const n=new Uint8Array(buffer.byteLength+chunk.byteLength);n.set(buffer);n.set(chunk,buffer.length);return n}let buffer=new Uint8Array(0);try{for(var _d=true,iterable_1=__asyncValues(iterable),iterable_1_1;iterable_1_1=yield __await(iterable_1.next()),_a=iterable_1_1.done,!_a;_d=true){_c=iterable_1_1.value;_d=false;const chunk=_c;buffer=append(buffer,chunk);for(;;){const size=sizeDelimitedPeek(buffer);if(size.eof){break}if(size.offset+size.size>buffer.byteLength){break}yield yield __await(fromBinary(messageDesc,buffer.subarray(size.offset,size.offset+size.size),options));buffer=buffer.subarray(size.offset+size.size)}}}catch(e_1_1){e_1={error:e_1_1}}finally{try{if(!_d&&!_a&&(_b=iterable_1.return))yield __await(_b.call(iterable_1))}finally{if(e_1)throw e_1.error}}if(buffer.byteLength>0){throw new Error("incomplete data")}}))}function sizeDelimitedPeek(data){const sizeEof={eof:true,size:null,offset:null};for(let i=0;i<10;i++){if(i>data.byteLength){return sizeEof}if((data[i]&128)==0){const reader=new BinaryReader(data);let size;try{size=reader.uint32()}catch(e){if(e instanceof RangeError){return sizeEof}throw e}return{eof:false,size:size,offset:reader.pos}}}throw new Error("invalid varint")}export{BinaryReader,BinaryWriter,FLOAT32_MAX,FLOAT32_MIN,INT32_MAX,INT32_MIN,UINT32_MAX,WireType,base64Decode,base64Encode,configureTextEncoding,getTextEncoding,parseTextFormatEnumValue,parseTextFormatScalarValue,sizeDelimitedDecodeStream,sizeDelimitedEncode,sizeDelimitedPeek};