// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_input/cr_input.js';
import 'chrome://resources/cr_elements/cr_page_selector/cr_page_selector.js';
import '/strings.m.js';
import './auto_tab_groups_new_badge.js';
import './auto_tab_groups_results_actions.js';
import '../tab_search_item.js';
import { getInstance as getAnnouncerInstance } from 'chrome://resources/cr_elements/cr_a11y_announcer/cr_a11y_announcer.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { normalizeURL, TabData, TabItemType } from '../tab_data.js';
import { getCss } from './auto_tab_groups_group.css.js';
import { getHtml } from './auto_tab_groups_group.html.js';
function getEventTargetIndex(e) {
    return Number(e.currentTarget.dataset['index']);
}
export class AutoTabGroupsGroupElement extends CrLitElement {
    static get is() {
        return 'auto-tab-groups-group';
    }
    static get properties() {
        return {
            tabs: { type: Array },
            firstNewTabIndex: { type: Number },
            name: { type: String },
            organizationId: { type: Number },
            showReject: {
                type: Boolean,
                reflect: true,
            },
            lastFocusedIndex_: { type: Number },
            showInput_: { type: Boolean },
            tabDatas_: { type: Array },
            changedName_: { type: Boolean },
        };
    }
    #tabs_accessor_storage = [];
    get tabs() { return this.#tabs_accessor_storage; }
    set tabs(value) { this.#tabs_accessor_storage = value; }
    #firstNewTabIndex_accessor_storage = 0;
    get firstNewTabIndex() { return this.#firstNewTabIndex_accessor_storage; }
    set firstNewTabIndex(value) { this.#firstNewTabIndex_accessor_storage = value; }
    #name_accessor_storage = '';
    get name() { return this.#name_accessor_storage; }
    set name(value) { this.#name_accessor_storage = value; }
    #organizationId_accessor_storage = -1;
    get organizationId() { return this.#organizationId_accessor_storage; }
    set organizationId(value) { this.#organizationId_accessor_storage = value; }
    #showReject_accessor_storage = false;
    get showReject() { return this.#showReject_accessor_storage; }
    set showReject(value) { this.#showReject_accessor_storage = value; }
    #lastFocusedIndex__accessor_storage = 0;
    get lastFocusedIndex_() { return this.#lastFocusedIndex__accessor_storage; }
    set lastFocusedIndex_(value) { this.#lastFocusedIndex__accessor_storage = value; }
    #showInput__accessor_storage = false;
    get showInput_() { return this.#showInput__accessor_storage; }
    set showInput_(value) { this.#showInput__accessor_storage = value; }
    #tabDatas__accessor_storage = [];
    get tabDatas_() { return this.#tabDatas__accessor_storage; }
    set tabDatas_(value) { this.#tabDatas__accessor_storage = value; }
    #changedName__accessor_storage = false;
    get changedName_() { return this.#changedName__accessor_storage; }
    set changedName_(value) { this.#changedName__accessor_storage = value; }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    connectedCallback() {
        super.connectedCallback();
        this.showInput_ = false;
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('tabs')) {
            if (this.lastFocusedIndex_ > this.tabs.length - 1) {
                this.lastFocusedIndex_ = 0;
            }
            this.tabDatas_ = this.computeTabDatas_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('showInput_')) {
            this.focusInput();
        }
    }
    focusInput() {
        const input = this.getInput_();
        if (input) {
            input.focus();
        }
    }
    getInput_() {
        if (!this.showInput_) {
            return null;
        }
        return this.shadowRoot.querySelector('#input');
    }
    computeTabDatas_() {
        return this.tabs.map(tab => new TabData(tab, TabItemType.OPEN_TAB, new URL(normalizeURL(tab.url.url)).hostname));
    }
    getTabIndex_(index) {
        return index === this.lastFocusedIndex_ ? 0 : -1;
    }
    getInputAriaLabel_() {
        return loadTimeData.getStringF('inputAriaLabel', this.name);
    }
    getEditButtonAriaLabel_() {
        return loadTimeData.getStringF('editAriaLabel', this.name);
    }
    getRejectButtonAriaLabel_() {
        return loadTimeData.getStringF('rejectAriaLabel', this.name);
    }
    showNewTabSectionHeader_(index) {
        return this.firstNewTabIndex > 0 && this.firstNewTabIndex === index;
    }
    onInputFocus_() {
        const input = this.getInput_();
        if (input) {
            input.select();
        }
    }
    onInputBlur_() {
        if (this.getInput_()) {
            this.showInput_ = false;
        }
        this.maybeRenameGroup_();
    }
    maybeRenameGroup_() {
        if (this.changedName_) {
            this.fire('name-change', { organizationId: this.organizationId, name: this.name });
            this.changedName_ = false;
        }
    }
    onInputKeyDown_(event) {
        if (event.key === 'Enter') {
            event.stopPropagation();
            this.showInput_ = false;
        }
    }
    onListKeyDown_(event) {
        const selector = this.$.selector;
        if (selector.selected === undefined) {
            return;
        }
        let handled = false;
        if (event.shiftKey && event.key === 'Tab') {
            // Explicitly focus the element prior to the list in focus order and
            // override the default behavior, which would be to focus the row that
            // the currently focused close button is in.
            this.shadowRoot.querySelector(`#rejectButton`).focus();
            handled = true;
        }
        else if (!event.shiftKey) {
            if (event.key === 'ArrowUp') {
                selector.selectPrevious();
                handled = true;
            }
            else if (event.key === 'ArrowDown') {
                selector.selectNext();
                handled = true;
            }
        }
        if (handled) {
            event.stopPropagation();
            event.preventDefault();
        }
    }
    onSelectedChanged_() {
        if (this.$.selector.selectedItem) {
            const selectedItem = this.$.selector.selectedItem;
            const selectedItemCloseButton = selectedItem.shadowRoot.querySelector(`cr-icon-button`);
            selectedItemCloseButton.focus();
            this.lastFocusedIndex_ =
                Number.parseInt(selectedItem.dataset['index'], 10);
        }
    }
    onTabRemove_(e) {
        const index = getEventTargetIndex(e);
        const tab = this.tabs[index];
        this.fire('remove-tab', { organizationId: this.organizationId, tab });
        getAnnouncerInstance().announce(loadTimeData.getString('a11yTabExcludedFromGroup'));
    }
    onTabFocus_(e) {
        // Ensure that when a TabSearchItemElement receives focus, it becomes the
        // selected item in the list.
        const index = getEventTargetIndex(e);
        this.$.selector.selected = index;
    }
    onTabBlur_(_e) {
        // Ensure the selector deselects its current selection on blur. If
        // selection should move to another element in the list, this will be done
        // in onTabFocus_.
        this.$.selector.select(-1);
    }
    getCloseButtonAriaLabel_(tabData) {
        return loadTimeData.getStringF('tabOrganizationCloseTabAriaLabel', tabData.tab.title);
    }
    onEditClick_() {
        this.showInput_ = true;
    }
    onRejectGroupClick_(event) {
        event.stopPropagation();
        event.preventDefault();
        this.dispatchEvent(new CustomEvent('reject-click', {
            bubbles: true,
            composed: true,
            detail: { organizationId: this.organizationId },
        }));
    }
    onCreateGroupClick_(event) {
        event.stopPropagation();
        event.preventDefault();
        this.dispatchEvent(new CustomEvent('create-group-click', {
            bubbles: true,
            composed: true,
            detail: {
                organizationId: this.organizationId,
                tabs: this.tabs,
            },
        }));
    }
    onNameChanged_(e) {
        if (this.name !== e.detail.value) {
            this.name = e.detail.value;
            this.changedName_ = true;
        }
    }
}
customElements.define(AutoTabGroupsGroupElement.is, AutoTabGroupsGroupElement);
