// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

/** @fileoverview Handles interprocess communication for the privacy page. */

// clang-format off
import {sendWithPromise} from 'chrome://resources/js/cr.js';
// clang-format on

export interface MetricsReporting {
  enabled: boolean;
  managed: boolean;
}

export interface ResolverOption {
  name: string;
  value: string;
  policy: string;
}

/**
 * Contains the possible string values for the secure DNS mode. This must be
 * kept in sync with the mode names in chrome/browser/net/secure_dns_config.h.
 */
export enum SecureDnsMode {
  OFF = 'off',
  AUTOMATIC = 'automatic',
  SECURE = 'secure',
}

/**
 * Contains the possible management modes. This should be kept in sync with
 * the management modes in chrome/browser/net/secure_dns_config.h.
 */
export enum SecureDnsUiManagementMode {
  NO_OVERRIDE = 0,
  DISABLED_MANAGED = 1,
  DISABLED_PARENTAL_CONTROLS = 2,
}

export interface SecureDnsSetting {
  mode: SecureDnsMode;
  config: string;
  managementMode: SecureDnsUiManagementMode;
  // 
}

export interface PrivacyPageBrowserProxy {
  // 

  setBlockAutoplayEnabled(enabled: boolean): void;
  getSecureDnsResolverList(): Promise<ResolverOption[]>;
  getSecureDnsSetting(): Promise<SecureDnsSetting>;

  /**
   * @return true if the config string is syntactically valid.
   */
  isValidConfig(entry: string): Promise<boolean>;

  /**
   * @return True if a test query succeeded in the specified DoH
   *     configuration or the probe was cancelled.
   */
  probeConfig(entry: string): Promise<boolean>;
}

export class PrivacyPageBrowserProxyImpl implements PrivacyPageBrowserProxy {
  // 

  setBlockAutoplayEnabled(enabled: boolean) {
    chrome.send('setBlockAutoplayEnabled', [enabled]);
  }

  getSecureDnsResolverList() {
    return sendWithPromise('getSecureDnsResolverList');
  }

  getSecureDnsSetting() {
    return sendWithPromise('getSecureDnsSetting');
  }

  isValidConfig(entry: string): Promise<boolean> {
    return sendWithPromise('isValidConfig', entry);
  }

  probeConfig(entry: string): Promise<boolean> {
    return sendWithPromise('probeConfig', entry);
  }

  static getInstance(): PrivacyPageBrowserProxy {
    return instance || (instance = new PrivacyPageBrowserProxyImpl());
  }

  static setInstance(obj: PrivacyPageBrowserProxy) {
    instance = obj;
  }
}

let instance: PrivacyPageBrowserProxy|null = null;
