// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'payments-list' is a list of saved payment methods (credit
 * cards etc.) to be shown in the settings page.
 */
import 'chrome://resources/cr_elements/cr_shared_vars.css.js';
import '../settings_shared.css.js';
import './credit_card_list_entry.js';
import './iban_list_entry.js';
import './pay_over_time_issuer_list_entry.js';
import './passwords_shared.css.js';
import { focusWithoutInk } from 'chrome://resources/js/focus_without_ink.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { loadTimeData } from '../i18n_setup.js';
import { getTemplate } from './payments_list.html.js';
export class SettingsPaymentsListElement extends PolymerElement {
    static get is() {
        return 'settings-payments-list';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            /**
             * An array of all saved credit cards.
             */
            creditCards: Array,
            /**
             * An array of all saved IBANs.
             */
            ibans: Array,
            /**
             * An array of all saved Pay Over Time issuers.
             */
            payOverTimeIssuers: Array,
            /**
             * True if displaying IBANs in settings is enabled.
             */
            enableIbans_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('showIbansSettings');
                },
            },
            /**
             * True if displaying Pay Over Time in settings is enabled.
             */
            enablePayOverTime_: {
                type: Boolean,
                value() {
                    return loadTimeData.getBoolean('shouldShowPayOverTimeSettings');
                },
            },
            /**
             * True iff any payment methods will be shown.
             */
            showAnyPaymentMethods_: {
                type: Boolean,
                value: false,
                computed: 'computeShowAnyPaymentMethods_(' +
                    'creditCards, ibans, enableIbans_, payOverTimeIssuers, enablePayOverTime_)',
            },
        };
    }
    /**
     * Focuses the next most appropriate element after removing a specific
     * credit card. Returns `false` if it could not find such an element,
     * in this case the focus is supposed to be handled by someone else.
     */
    updateFocusBeforeCreditCardRemoval(cardIndex) {
        // The focused element is to be reset only if the last element is deleted,
        // when the number of "dom-repeat" nodes changes and the focus get lost.
        if (cardIndex === this.creditCards.length - 1) {
            return this.updateFocusBeforeRemoval_(this.getCreditCardId_(cardIndex));
        }
        else {
            return true;
        }
    }
    /**
     * Focuses the next most appropriate element after removing a specific
     * iban. Returns `false` if it could not find such an element,
     * in this case the focus is supposed to be handled by someone else.
     */
    updateFocusBeforeIbanRemoval(ibanIndex) {
        // The focused element is to be reset only if the last element is deleted,
        // when the number of "dom-repeat" nodes changes and the focus get lost.
        if (ibanIndex === this.ibans.length - 1) {
            return this.updateFocusBeforeRemoval_(this.getIbanId_(ibanIndex));
        }
        else {
            return true;
        }
    }
    /**
     * Handles focus resetting across all payment method lists. Returns `false`
     * only when the last payment method is removed, in other cases sets the focus
     * to either the next or previous payment method.
     */
    updateFocusBeforeRemoval_(id) {
        const paymentMethods = this.shadowRoot.querySelectorAll('.payment-method');
        if (paymentMethods.length <= 1) {
            return false;
        }
        const index = [...paymentMethods].findIndex((element) => element.id === id);
        const isLastItem = index === paymentMethods.length - 1;
        const indexToFocus = index + (isLastItem ? -1 : +1);
        const menu = paymentMethods[indexToFocus].dotsMenu;
        if (menu) {
            focusWithoutInk(menu);
            return true;
        }
        return false;
    }
    getCreditCardId_(index) {
        return `card-${index}`;
    }
    getIbanId_(index) {
        return `iban-${index}`;
    }
    /**
     * @return Whether the list exists and has items.
     */
    hasSome_(list) {
        return !!(list && list.length);
    }
    /**
     * @return true iff there are credit cards to be shown.
     */
    showCreditCards_() {
        return this.hasSome_(this.creditCards);
    }
    /**
     * @return true iff there are IBANs to be shown.
     */
    showIbans_() {
        return this.enableIbans_ && this.hasSome_(this.ibans);
    }
    /**
     * @return true iff there are Pay Over Time issuers to be shown.
     */
    showPayOverTimeIssuers_() {
        return this.enablePayOverTime_ && this.hasSome_(this.payOverTimeIssuers);
    }
    /**
     * @return true iff any payment methods will be shown.
     */
    computeShowAnyPaymentMethods_() {
        return this.showCreditCards_() || this.showIbans_() ||
            this.showPayOverTimeIssuers_();
    }
}
customElements.define(SettingsPaymentsListElement.is, SettingsPaymentsListElement);
