/**
 * @fileoverview 'settings-credit-card-edit-dialog' is the dialog that allows
 * editing or creating a credit card entry.
 */
import '/shared/settings/prefs/prefs.js';
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_dialog/cr_dialog.js';
import 'chrome://resources/cr_elements/cr_input/cr_input.js';
import 'chrome://resources/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/cr_elements/cr_shared_vars.css.js';
import 'chrome://resources/cr_elements/md_select.css.js';
import '../settings_shared.css.js';
import '../settings_vars.css.js';
import type { CrButtonElement } from 'chrome://resources/cr_elements/cr_button/cr_button.js';
import type { CrDialogElement } from 'chrome://resources/cr_elements/cr_dialog/cr_dialog.js';
import type { CrInputElement } from 'chrome://resources/cr_elements/cr_input/cr_input.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
/**
 * Enum of possible states for the credit card number. A card number is valid
 * if it is of a supported length and passes a Luhn check. Otherwise, it is
 * invalid and we may show an error to the user in cases where we are certain
 * they have entered an invalid card (i.e. vs still typing).
 */
declare enum CardNumberValidationState {
    VALID = "valid",
    INVALID_NO_ERROR = "invalid-no-error",
    INVALID_WITH_ERROR = "invalid-with-error"
}
declare global {
    interface HTMLElementEventMap {
        'save-credit-card': CustomEvent<chrome.autofillPrivate.CreditCardEntry>;
    }
}
export interface SettingsCreditCardEditDialogElement {
    $: {
        cancelButton: CrButtonElement;
        cvcInput: CrInputElement;
        dialog: CrDialogElement;
        expiredError: HTMLElement;
        month: HTMLSelectElement;
        nameInput: CrInputElement;
        nicknameInput: CrInputElement;
        numberInput: CrInputElement;
        saveButton: CrButtonElement;
        year: HTMLSelectElement;
    };
}
declare const SettingsCreditCardEditDialogElementBase: typeof PolymerElement & (new (...args: any[]) => import("chrome://resources/cr_elements/i18n_mixin.js").I18nMixinInterface);
export declare class SettingsCreditCardEditDialogElement extends SettingsCreditCardEditDialogElementBase {
    static get is(): string;
    static get template(): HTMLTemplateElement;
    static get properties(): {
        /**
         * User preferences state.
         */
        prefs: ObjectConstructor;
        /**
         * The underlying credit card object for the dialog. After initialization
         * of the dialog, this object is only modified once the 'Save' button is
         * clicked.
         */
        creditCard: ObjectConstructor;
        /**
         * The actual title that's used for this dialog. Will be context sensitive
         * based on if |creditCard| is being created or edited.
         */
        title_: StringConstructor;
        /**
         * The list of months to show in the dropdown.
         */
        monthList_: {
            type: ArrayConstructor;
            value: string[];
        };
        /** The list of years to show in the dropdown. */
        yearList_: ArrayConstructor;
        /**
         * Backing data for inputs in the dialog, each bound to the corresponding
         * HTML elements.
         *
         * Note that rawCardNumber_ is unsanitized; code should instead use
         * `sanitizedCardNumber_`.
         */
        name_: StringConstructor;
        rawCardNumber_: StringConstructor;
        cvc_: StringConstructor;
        nickname_: StringConstructor;
        expirationYear_: StringConstructor;
        expirationMonth_: StringConstructor;
        /**
         * A sanitized version of `rawCardNumber_` that strips out commonly used
         * separators and trims whitespace.
         */
        sanitizedCardNumber_: {
            type: StringConstructor;
            computed: string;
            observer: string;
        };
        /** Whether the current nickname input is invalid. */
        nicknameInvalid_: {
            type: BooleanConstructor;
            value: boolean;
        };
        /** Whether the current card number field is invalid. */
        cardNumberValidationState_: {
            type: typeof CardNumberValidationState;
            value: boolean;
        };
        /**
         * Computed property that tracks if the entered credit card is expired -
         * that is, if its expiration month and year are in the past.
         */
        expired_: {
            type: BooleanConstructor;
            computed: string;
            reflectToAttribute: boolean;
            observer: string;
        };
        /**
         * Checks if CVC storage is available based on the feature flag.
         */
        cvcStorageAvailable_: {
            type: BooleanConstructor;
            value(): boolean;
        };
    };
    prefs: {
        [key: string]: any;
    };
    creditCard: chrome.autofillPrivate.CreditCardEntry;
    private title_;
    private monthList_;
    private yearList_;
    private name_?;
    private rawCardNumber_;
    private cvc_?;
    private nickname_?;
    private expirationYear_?;
    private expirationMonth_?;
    private sanitizedCardNumber_;
    private nicknameInvalid_;
    private cardNumberValidationState_;
    private expired_;
    private cvcStorageAvailable_;
    connectedCallback(): void;
    /** Closes the dialog. */
    close(): void;
    /**
     * Handler for tapping the 'cancel' button. Should just dismiss the dialog.
     */
    private onCancelButtonClick_;
    /**
     * Handler for tapping the save button.
     */
    private onSaveButtonClick_;
    private onSanitizedCardNumberChanged_;
    private onNumberInputBlurred_;
    private showErrorForCardNumber_;
    private onMonthChange_;
    private onYearChange_;
    /**
     * Handles a11y error announcement the same way as in cr-input.
     */
    private onExpiredChanged_;
    /**
     * @return 'true' or 'false' for the aria-invalid attribute
     *     of expiration selectors.
     */
    private getExpirationAriaInvalid_;
    private checkIfCvcStorageIsAvailable_;
    private getCvcImageSource_;
    private getCvcImageTooltip_;
    /**
     * Validate no digits are used in nickname. Display error message and disable
     * the save button when invalid.
     */
    private validateNickname_;
    /**
     * @param  nickname of the card, undefined when not set.
     * @return nickname character length.
     */
    private computeNicknameCharCount_;
    private saveEnabled_;
    /**
     * @return True iff the provided expiration date is passed.
     */
    private computeExpired_;
    /**
     * Trim credit card's name, cardNumber and nickname if exist.
     */
    private trimCreditCard_;
    private isCardAmex_;
    /**
     * Sanitize the raw card number entered by the user, trimming whitespace and
     * removing commonly used separators.
     */
    private sanitizeCardNumber_;
    /**
     * Compute whether or not the provided card number is valid, i.e. that it is a
     * number and passes a Luhn check. If the card number isn't complete yet, it
     * is still considered invalid but no error will be shown.
     */
    private computeCardNumberValidationState_;
    /**
     * Validates if a given card number passes a Luhn check.
     *
     * http://en.wikipedia.org/wiki/Luhn_algorithm
     */
    private passesLuhnCheck_;
}
declare global {
    interface HTMLElementTagNameMap {
        'settings-credit-card-edit-dialog': SettingsCreditCardEditDialogElement;
    }
}
export {};
