import { css, CrLitElement, html, nothing, render } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { sendWithPromise, addWebUiListener, removeWebUiListener } from 'chrome://resources/js/cr.js';
import 'chrome://print/strings.m.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { pdfCreateOutOfProcessPlugin } from 'chrome://print/pdf/pdf_scripting_api.js';

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The class name to set on the document element.
 */
const CLASS_NAME = 'focus-outline-visible';
const docsToManager = new Map();
/**
 * This class sets a CSS class name on the HTML element of |doc| when the user
 * presses a key. It removes the class name when the user clicks anywhere.
 *
 * This allows you to write CSS like this:
 *
 * html.focus-outline-visible my-element:focus {
 *   outline: 5px auto -webkit-focus-ring-color;
 * }
 *
 * And the outline will only be shown if the user uses the keyboard to get to
 * it.
 *
 */
class FocusOutlineManager {
    // Whether focus change is triggered by a keyboard event.
    focusByKeyboard_ = true;
    classList_;
    /**
     * @param doc The document to attach the focus outline manager to.
     */
    constructor(doc) {
        this.classList_ = doc.documentElement.classList;
        doc.addEventListener('keydown', (e) => this.onEvent_(true, e), true);
        doc.addEventListener('mousedown', (e) => this.onEvent_(false, e), true);
        this.updateVisibility();
    }
    onEvent_(focusByKeyboard, e) {
        if (this.focusByKeyboard_ === focusByKeyboard) {
            return;
        }
        if (e instanceof KeyboardEvent && e.repeat) {
            // A repeated keydown should not trigger the focus state. For example,
            // there is a repeated ALT keydown if ALT+CLICK is used to open the
            // context menu and ALT is not released.
            return;
        }
        this.focusByKeyboard_ = focusByKeyboard;
        this.updateVisibility();
    }
    updateVisibility() {
        this.visible = this.focusByKeyboard_;
    }
    /**
     * Whether the focus outline should be visible.
     */
    set visible(visible) {
        this.classList_.toggle(CLASS_NAME, visible);
    }
    get visible() {
        return this.classList_.contains(CLASS_NAME);
    }
    /**
     * Gets a per document singleton focus outline manager.
     * @param doc The document to get the |FocusOutlineManager| for.
     * @return The per document singleton focus outline manager.
     */
    static forDocument(doc) {
        let manager = docsToManager.get(doc);
        if (!manager) {
            manager = new FocusOutlineManager(doc);
            docsToManager.set(doc, manager);
        }
        return manager;
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Verify |value| is truthy.
 * @param value A value to check for truthiness. Note that this
 *     may be used to test whether |value| is defined or not, and we don't want
 *     to force a cast to boolean.
 */
function assert(value, message) {
    if (value) {
        return;
    }
    throw new Error('Assertion failed' + (message ? `: ${message}` : ''));
}
/**
 * Call this from places in the code that should never be reached.
 *
 * For example, handling all the values of enum with a switch() like this:
 *
 *   function getValueFromEnum(enum) {
 *     switch (enum) {
 *       case ENUM_FIRST_OF_TWO:
 *         return first
 *       case ENUM_LAST_OF_TWO:
 *         return last;
 *     }
 *     assertNotReached();
 *   }
 *
 * This code should only be hit in the case of serious programmer error or
 * unexpected input.
 */
function assertNotReached(message = 'Unreachable code hit') {
    assert(false, message);
}

// Copyright 2011 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview EventTracker is a simple class that manages the addition and
 * removal of DOM event listeners. In particular, it keeps track of all
 * listeners that have been added and makes it easy to remove some or all of
 * them without requiring all the information again. This is particularly handy
 * when the listener is a generated function such as a lambda or the result of
 * calling Function.bind.
 */
class EventTracker {
    listeners_ = [];
    /**
     * Add an event listener - replacement for EventTarget.addEventListener.
     * @param target The DOM target to add a listener to.
     * @param eventType The type of event to subscribe to.
     * @param listener The listener to add.
     * @param capture Whether to invoke during the capture phase. Defaults to
     *     false.
     */
    add(target, eventType, listener, capture = false) {
        const h = {
            target: target,
            eventType: eventType,
            listener: listener,
            capture: capture,
        };
        this.listeners_.push(h);
        target.addEventListener(eventType, listener, capture);
    }
    /**
     * Remove any specified event listeners added with this EventTracker.
     * @param target The DOM target to remove a listener from.
     * @param eventType The type of event to remove.
     */
    remove(target, eventType) {
        this.listeners_ = this.listeners_.filter(listener => {
            if (listener.target === target &&
                (!eventType || (listener.eventType === eventType))) {
                EventTracker.removeEventListener(listener);
                return false;
            }
            return true;
        });
    }
    /** Remove all event listeners added with this EventTracker. */
    removeAll() {
        this.listeners_.forEach(listener => EventTracker.removeEventListener(listener));
        this.listeners_ = [];
    }
    /**
     * Remove a single event listener given it's tracking entry. It's up to the
     * caller to ensure the entry is removed from listeners_.
     * @param entry The entry describing the listener to
     * remove.
     */
    static removeEventListener(entry) {
        entry.target.removeEventListener(entry.eventType, entry.listener, entry.capture);
    }
}

let instance$P = null;
function getCss$L() {
    return instance$P || (instance$P = [...[], css `:host{bottom:0;display:block;left:0;overflow:hidden;pointer-events:none;position:absolute;right:0;top:0;transform:translate3d(0,0,0)}.ripple{background-color:currentcolor;left:0;opacity:var(--paper-ripple-opacity,0.25);pointer-events:none;position:absolute;will-change:height,transform,width}.ripple,:host(.circle){border-radius:50%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrRippleElement extends CrLitElement {
    static get is() {
        return 'cr-ripple';
    }
    static get styles() {
        return getCss$L();
    }
    static get properties() {
        return {
            holdDown: { type: Boolean },
            recenters: { type: Boolean },
            noink: { type: Boolean },
        };
    }
    #holdDown_accessor_storage = false;
    get holdDown() { return this.#holdDown_accessor_storage; }
    set holdDown(value) { this.#holdDown_accessor_storage = value; }
    #recenters_accessor_storage = false;
    get recenters() { return this.#recenters_accessor_storage; }
    set recenters(value) { this.#recenters_accessor_storage = value; }
    #noink_accessor_storage = false;
    get noink() { return this.#noink_accessor_storage; }
    set noink(value) { this.#noink_accessor_storage = value; }
    ripples_ = [];
    eventTracker_ = new EventTracker();
    connectedCallback() {
        super.connectedCallback();
        assert(this.parentNode);
        const keyEventTarget = this.parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ?
            this.parentNode.host :
            this.parentElement;
        this.eventTracker_.add(keyEventTarget, 'pointerdown', (e) => this.uiDownAction(e));
        this.eventTracker_.add(keyEventTarget, 'pointerup', () => this.uiUpAction());
        // 'pointerup' does not fire if the pointer is moved outside the bounds of
        // `keyEventTarget` before releasing, so also listen for `pointerout`.
        this.eventTracker_.add(keyEventTarget, 'pointerout', () => this.uiUpAction());
        this.eventTracker_.add(keyEventTarget, 'keydown', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === 'Enter') {
                this.onEnterKeydown_();
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeydown_();
            }
        });
        this.eventTracker_.add(keyEventTarget, 'keyup', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeyup_();
            }
        });
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('holdDown')) {
            this.holdDownChanged_(this.holdDown, changedProperties.get('holdDown'));
        }
    }
    uiDownAction(e) {
        if (e !== undefined && e.button !== 0) {
            // Ignore secondary mouse button clicks.
            return;
        }
        if (!this.noink) {
            this.downAction_(e);
        }
    }
    downAction_(e) {
        if (this.ripples_.length && this.holdDown) {
            return;
        }
        this.showRipple_(e);
    }
    clear() {
        this.hideRipple_();
        this.holdDown = false;
    }
    showAndHoldDown() {
        this.ripples_.forEach(ripple => {
            ripple.remove();
        });
        this.ripples_ = [];
        this.holdDown = true;
    }
    showRipple_(e) {
        // OWL: We don't want ripples.
        return;
    }
    uiUpAction() {
        if (!this.noink) {
            this.upAction_();
        }
    }
    upAction_() {
        if (!this.holdDown) {
            this.hideRipple_();
        }
    }
    hideRipple_() {
        if (this.ripples_.length === 0) {
            return;
        }
        this.ripples_.forEach(function (ripple) {
            const opacity = ripple.computedStyleMap().get('opacity');
            if (opacity === null) {
                ripple.remove();
                return;
            }
            const animation = ripple.animate({
                opacity: [opacity.value, 0],
            }, {
                duration: 150,
                fill: 'forwards',
            });
            animation.finished.then(() => {
                ripple.remove();
            });
        });
        this.ripples_ = [];
    }
    onEnterKeydown_() {
        this.uiDownAction();
        window.setTimeout(() => {
            this.uiUpAction();
        }, 1);
    }
    onSpaceKeydown_() {
        this.uiDownAction();
    }
    onSpaceKeyup_() {
        this.uiUpAction();
    }
    holdDownChanged_(newHoldDown, oldHoldDown) {
        if (oldHoldDown === undefined) {
            return;
        }
        if (newHoldDown) {
            this.downAction_();
        }
        else {
            this.upAction_();
        }
    }
}
customElements.define(CrRippleElement.is, CrRippleElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrRippleMixin = (superClass) => {
    class CrRippleMixin extends superClass {
        static get properties() {
            return {
                /**
                 * If true, the element will not produce a ripple effect when
                 * interacted with via the pointer.
                 */
                noink: { type: Boolean },
            };
        }
        #noink_accessor_storage = false;
        get noink() { return this.#noink_accessor_storage; }
        set noink(value) { this.#noink_accessor_storage = value; }
        rippleContainer = null;
        ripple_ = null;
        updated(changedProperties) {
            super.updated(changedProperties);
            if (changedProperties.has('noink') && this.hasRipple()) {
                assert(this.ripple_);
                this.ripple_.noink = this.noink;
            }
        }
        ensureRippleOnPointerdown() {
            // 'capture: true' is necessary so that the cr-ripple is created early
            // enough so that it also receives the 'pointerdown' event. Otherwise
            // the ripple is created, but not shown on the 1st click.
            this.addEventListener('pointerdown', () => this.ensureRipple(), { capture: true });
        }
        /**
         * Ensures this element contains a ripple effect. For startup efficiency
         * the ripple effect is dynamically added on demand when needed.
         */
        ensureRipple() {
            if (this.hasRipple()) {
                return;
            }
            this.ripple_ = this.createRipple();
            this.ripple_.noink = this.noink;
            const rippleContainer = this.rippleContainer || this.shadowRoot;
            assert(rippleContainer);
            rippleContainer.appendChild(this.ripple_);
        }
        /**
         * Returns the `<cr-ripple>` element used by this element to create
         * ripple effects. The element's ripple is created on demand, when
         * necessary, and calling this method will force the
         * ripple to be created.
         */
        getRipple() {
            this.ensureRipple();
            assert(this.ripple_);
            return this.ripple_;
        }
        /**
         * Returns true if this element currently contains a ripple effect.
         */
        hasRipple() {
            return Boolean(this.ripple_);
        }
        /**
         * Create the element's ripple effect via creating a `<cr-ripple
         * id="ink">` instance. Override this method to customize the ripple
         * element.
         */
        createRipple() {
            const ripple = document.createElement('cr-ripple');
            ripple.id = 'ink';
            return ripple;
        }
    }
    return CrRippleMixin;
};

let instance$O = null;
function getCss$K() {
    return instance$O || (instance$O = [...[], css `[hidden],:host([hidden]){display:none !important}`]);
}

const sheet$1 = new CSSStyleSheet();
sheet$1.replaceSync(`html{--google-blue-50-rgb:232,240,254;--google-blue-50:rgb(var(--google-blue-50-rgb));--google-blue-100-rgb:210,227,252;--google-blue-100:rgb(var(--google-blue-100-rgb));--google-blue-200-rgb:174,203,250;--google-blue-200:rgb(var(--google-blue-200-rgb));--google-blue-300-rgb:138,180,248;--google-blue-300:rgb(var(--google-blue-300-rgb));--google-blue-400-rgb:102,157,246;--google-blue-400:rgb(var(--google-blue-400-rgb));--google-blue-500-rgb:66,133,244;--google-blue-500:rgb(var(--google-blue-500-rgb));--google-blue-600-rgb:26,115,232;--google-blue-600:rgb(var(--google-blue-600-rgb));--google-blue-700-rgb:25,103,210;--google-blue-700:rgb(var(--google-blue-700-rgb));--google-blue-800-rgb:24,90,188;--google-blue-800:rgb(var(--google-blue-800-rgb));--google-blue-900-rgb:23,78,166;--google-blue-900:rgb(var(--google-blue-900-rgb));--google-green-50-rgb:230,244,234;--google-green-50:rgb(var(--google-green-50-rgb));--google-green-200-rgb:168,218,181;--google-green-200:rgb(var(--google-green-200-rgb));--google-green-300-rgb:129,201,149;--google-green-300:rgb(var(--google-green-300-rgb));--google-green-400-rgb:91,185,116;--google-green-400:rgb(var(--google-green-400-rgb));--google-green-500-rgb:52,168,83;--google-green-500:rgb(var(--google-green-500-rgb));--google-green-600-rgb:30,142,62;--google-green-600:rgb(var(--google-green-600-rgb));--google-green-700-rgb:24,128,56;--google-green-700:rgb(var(--google-green-700-rgb));--google-green-800-rgb:19,115,51;--google-green-800:rgb(var(--google-green-800-rgb));--google-green-900-rgb:13,101,45;--google-green-900:rgb(var(--google-green-900-rgb));--google-grey-50-rgb:248,249,250;--google-grey-50:rgb(var(--google-grey-50-rgb));--google-grey-100-rgb:241,243,244;--google-grey-100:rgb(var(--google-grey-100-rgb));--google-grey-200-rgb:232,234,237;--google-grey-200:rgb(var(--google-grey-200-rgb));--google-grey-300-rgb:218,220,224;--google-grey-300:rgb(var(--google-grey-300-rgb));--google-grey-400-rgb:189,193,198;--google-grey-400:rgb(var(--google-grey-400-rgb));--google-grey-500-rgb:154,160,166;--google-grey-500:rgb(var(--google-grey-500-rgb));--google-grey-600-rgb:128,134,139;--google-grey-600:rgb(var(--google-grey-600-rgb));--google-grey-700-rgb:95,99,104;--google-grey-700:rgb(var(--google-grey-700-rgb));--google-grey-800-rgb:60,64,67;--google-grey-800:rgb(var(--google-grey-800-rgb));--google-grey-900-rgb:32,33,36;--google-grey-900:rgb(var(--google-grey-900-rgb));--google-grey-900-white-4-percent:#292a2d;--google-purple-200-rgb:215,174,251;--google-purple-200:rgb(var(--google-purple-200-rgb));--google-purple-900-rgb:104,29,168;--google-purple-900:rgb(var(--google-purple-900-rgb));--google-red-100-rgb:244,199,195;--google-red-100:rgb(var(--google-red-100-rgb));--google-red-300-rgb:242,139,130;--google-red-300:rgb(var(--google-red-300-rgb));--google-red-500-rgb:234,67,53;--google-red-500:rgb(var(--google-red-500-rgb));--google-red-600-rgb:217,48,37;--google-red-600:rgb(var(--google-red-600-rgb));--google-red-700-rgb:197,57,41;--google-red-700:rgb(var(--google-red-700-rgb));--google-yellow-50-rgb:254,247,224;--google-yellow-50:rgb(var(--google-yellow-50-rgb));--google-yellow-100-rgb:254,239,195;--google-yellow-100:rgb(var(--google-yellow-100-rgb));--google-yellow-200-rgb:253,226,147;--google-yellow-200:rgb(var(--google-yellow-200-rgb));--google-yellow-300-rgb:253,214,51;--google-yellow-300:rgb(var(--google-yellow-300-rgb));--google-yellow-400-rgb:252,201,52;--google-yellow-400:rgb(var(--google-yellow-400-rgb));--google-yellow-500-rgb:251,188,4;--google-yellow-500:rgb(var(--google-yellow-500-rgb));--google-yellow-700-rgb:240,147,0;--google-yellow-700:rgb(var(--google-yellow-700-rgb));--cr-card-background-color:white;--cr-shadow-key-color_:color-mix(in srgb,var(--cr-shadow-color) 30%,transparent);--cr-shadow-ambient-color_:color-mix(in srgb,var(--cr-shadow-color) 15%,transparent);--cr-elevation-1:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 1px 3px 1px;--cr-elevation-2:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 2px 6px 2px;--cr-elevation-3:var(--cr-shadow-key-color_) 0 1px 3px 0,var(--cr-shadow-ambient-color_) 0 4px 8px 3px;--cr-elevation-4:var(--cr-shadow-key-color_) 0 2px 3px 0,var(--cr-shadow-ambient-color_) 0 6px 10px 4px;--cr-elevation-5:var(--cr-shadow-key-color_) 0 4px 4px 0,var(--cr-shadow-ambient-color_) 0 8px 12px 6px;--cr-card-shadow:var(--cr-elevation-2);--cr-focused-item-color:var(--google-grey-300);--cr-form-field-label-color:var(--google-grey-700);--cr-hairline-rgb:0,0,0;--cr-iph-anchor-highlight-color:rgba(var(--google-blue-600-rgb),0.1);--cr-menu-background-color:white;--cr-menu-background-focus-color:var(--google-grey-400);--cr-menu-shadow:var(--cr-elevation-2);--cr-separator-color:rgba(0,0,0,.06);--cr-title-text-color:rgb(90,90,90);--cr-scrollable-border-color:var(--google-grey-300)}@media (prefers-color-scheme:dark){html{--cr-card-background-color:var(--google-grey-900-white-4-percent);--cr-focused-item-color:var(--google-grey-800);--cr-form-field-label-color:var(--dark-secondary-color);--cr-hairline-rgb:255,255,255;--cr-iph-anchor-highlight-color:rgba(var(--google-grey-100-rgb),0.1);--cr-menu-background-color:var(--google-grey-900);--cr-menu-background-focus-color:var(--google-grey-700);--cr-menu-background-sheen:rgba(255,255,255,.06);--cr-menu-shadow:rgba(0,0,0,.3) 0 1px 2px 0,rgba(0,0,0,.15) 0 3px 6px 2px;--cr-separator-color:rgba(255,255,255,.1);--cr-title-text-color:var(--cr-primary-text-color);--cr-scrollable-border-color:var(--google-grey-700)}}@media (forced-colors:active){html{--cr-focus-outline-hcm:2px solid transparent;--cr-border-hcm:2px solid transparent}}html{--cr-button-edge-spacing:12px;--cr-controlled-by-spacing:24px;--cr-default-input-max-width:264px;--cr-icon-ripple-size:36px;--cr-icon-ripple-padding:8px;--cr-icon-size:20px;--cr-icon-button-margin-start:16px;--cr-icon-ripple-margin:calc(var(--cr-icon-ripple-padding) * -1);--cr-section-min-height:48px;--cr-section-two-line-min-height:64px;--cr-section-padding:20px;--cr-section-vertical-padding:12px;--cr-section-indent-width:40px;--cr-section-indent-padding:calc(var(--cr-section-padding) + var(--cr-section-indent-width));--cr-section-vertical-margin:21px;--cr-centered-card-max-width:680px;--cr-centered-card-width-percentage:0.96;--cr-hairline:1px solid rgba(var(--cr-hairline-rgb),.14);--cr-separator-height:1px;--cr-separator-line:var(--cr-separator-height) solid var(--cr-separator-color);--cr-toolbar-overlay-animation-duration:150ms;--cr-toolbar-height:56px;--cr-container-shadow-height:6px;--cr-container-shadow-margin:calc(-1 * var(--cr-container-shadow-height));--cr-container-shadow-max-opacity:1;--cr-card-border-radius:8px;--cr-disabled-opacity:.38;--cr-form-field-bottom-spacing:16px;--cr-form-field-label-font-size:.625rem;--cr-form-field-label-height:1em;--cr-form-field-label-line-height:1}html{--cr-fallback-color-outline:rgb(116,119,117);--cr-fallback-color-primary:rgb(11,87,208);--cr-fallback-color-on-primary:rgb(255,255,255);--cr-fallback-color-primary-container:rgb(211,227,253);--cr-fallback-color-on-primary-container:rgb(4,30,73);--cr-fallback-color-secondary-container:rgb(194,231,255);--cr-fallback-color-on-secondary-container:rgb(0,29,53);--cr-fallback-color-neutral-container:rgb(242,242,242);--cr-fallback-color-neutral-outline:rgb(199,199,199);--cr-fallback-color-surface:rgb(255,255,255);--cr-fallback-color-surface1:rgb(248,250,253);--cr-fallback-color-surface2:rgb(243,246,252);--cr-fallback-color-surface3:rgb(239,243,250);--cr-fallback-color-on-surface-rgb:31,31,31;--cr-fallback-color-on-surface:rgb(var(--cr-fallback-color-on-surface-rgb));--cr-fallback-color-surface-variant:rgb(225,227,225);--cr-fallback-color-on-surface-variant:rgb(138,141,140);--cr-fallback-color-on-surface-subtle:rgb(71,71,71);--cr-fallback-color-inverse-primary:rgb(168,199,250);--cr-fallback-color-inverse-surface:rgb(48,48,48);--cr-fallback-color-inverse-on-surface:rgb(242,242,242);--cr-fallback-color-tonal-container:rgb(211,227,253);--cr-fallback-color-on-tonal-container:rgb(4,30,73);--cr-fallback-color-tonal-outline:rgb(168,199,250);--cr-fallback-color-error:rgb(179,38,30);--cr-fallback-color-divider:rgb(211,227,253);--cr-fallback-color-state-hover-on-prominent_:rgba(253,252,251,.1);--cr-fallback-color-state-on-subtle-rgb_:31,31,31;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.06);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.08);--cr-fallback-color-state-ripple-primary-rgb_:124,172,248;--cr-fallback-color-state-ripple-primary_:rgba(var(--cr-fallback-color-state-ripple-primary-rgb_),0.32);--cr-fallback-color-base-container:rgb(236,239,247);--cr-fallback-color-disabled-background:rgba(var(--cr-fallback-color-on-surface-rgb),.12);--cr-fallback-color-disabled-foreground:rgba(var(--cr-fallback-color-on-surface-rgb),var(--cr-disabled-opacity));--cr-hover-background-color:var(--color-sys-state-hover,rgba(var(--cr-fallback-color-on-surface-rgb),.08));--cr-hover-on-prominent-background-color:var(--color-sys-state-hover-on-prominent,var(--cr-fallback-color-state-hover-on-prominent_));--cr-hover-on-subtle-background-color:var(--color-sys-state-hover-on-subtle,var(--cr-fallback-color-state-hover-on-subtle_));--cr-active-background-color:var(--color-sys-state-pressed,rgba(var(--cr-fallback-color-on-surface-rgb),.12));--cr-active-on-primary-background-color:var(--color-sys-state-ripple-primary,var(--cr-fallback-color-state-ripple-primary_));--cr-active-neutral-on-subtle-background-color:var(--color-sys-state-ripple-neutral-on-subtle,var(--cr-fallback-color-state-ripple-neutral-on-subtle_));--cr-focus-outline-color:var(--color-sys-state-focus-ring,var(--owl-control-accent-color,var(--cr-fallback-color-primary)));--cr-focus-outline-inverse-color:var(--color-sys-state-focus-ring-inverse,var(--cr-fallback-color-inverse-primary));--cr-primary-text-color:var(--color-primary-foreground,var(--cr-fallback-color-on-surface));--cr-secondary-text-color:var(--color-secondary-foreground,var(--cr-fallback-color-on-surface-variant));--cr-link-color:var(--color-link-foreground-default,var(--cr-fallback-color-primary));--cr-button-height:36px;--cr-shadow-color:var(--color-sys-shadow,rgb(0,0,0));--cr-checked-color:var(--color-checkbox-foreground-checked,var(--owl-control-accent-color,var(--cr-fallback-color-primary)))}@media (prefers-color-scheme:dark){html{--cr-fallback-color-outline:rgb(142,145,143);--cr-fallback-color-primary:rgb(168,199,250);--cr-fallback-color-on-primary:rgb(6,46,111);--cr-fallback-color-primary-container:rgb(8,66,160);--cr-fallback-color-on-primary-container:rgb(211,227,253);--cr-fallback-color-secondary-container:rgb(0,74,119);--cr-fallback-color-on-secondary-container:rgb(194,231,255);--cr-fallback-color-neutral-container:rgb(40,40,40);--cr-fallback-color-neutral-outline:rgb(117,117,117);--cr-fallback-color-surface:rgb(31,31,31);--cr-fallback-color-surface1:rgb(39,40,42);--cr-fallback-color-surface2:rgb(45,47,49);--cr-fallback-color-surface3:rgb(51,52,56);--cr-fallback-color-on-surface-rgb:227,227,227;--cr-fallback-color-surface-variant:rgb(68,71,70);--cr-fallback-color-on-surface-variant:rgb(158,161,160);--cr-fallback-color-on-surface-subtle:rgb(199,199,199);--cr-fallback-color-inverse-primary:rgb(11,87,208);--cr-fallback-color-inverse-surface:rgb(227,227,227);--cr-fallback-color-inverse-on-surface:rgb(31,31,31);--cr-fallback-color-tonal-container:rgb(0,74,119);--cr-fallback-color-on-tonal-container:rgb(194,231,255);--cr-fallback-color-tonal-outline:rgb(4,125,183);--cr-fallback-color-error:rgb(242,184,181);--cr-fallback-color-divider:rgb(94,94,94);--cr-fallback-color-state-hover-on-prominent_:rgba(31,31,31,.06);--cr-fallback-color-state-on-subtle-rgb_:253,252,251;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.10);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.16);--cr-fallback-color-state-ripple-primary-rgb_:76,141,246;--cr-fallback-color-base-container:rgba(40,40,40,1)}}@media (forced-colors:active){html{--cr-fallback-color-disabled-background:Canvas;--cr-fallback-color-disabled-foreground:GrayText}}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet$1];

let instance$N = null;
function getCss$J() {
    return instance$N || (instance$N = [...[getCss$K()], css `:host{--cr-button-background-color:transparent;--cr-button-border-color:var(--color-button-border,var(--cr-fallback-color-tonal-outline));--cr-button-text-color:var(--color-button-foreground,var(--cr-fallback-color-primary));--cr-button-ripple-opacity:1;--cr-button-ripple-color:var(--cr-active-background-color);--cr-button-disabled-background-color:transparent;--cr-button-disabled-border-color:var(--color-button-border-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground))}:host(.action-button){--cr-button-background-color:var(--color-button-background-prominent,var(--cr-fallback-color-primary));--cr-button-text-color:var(--color-button-foreground-prominent,var(--cr-fallback-color-on-primary));--cr-button-ripple-color:var(--cr-active-on-primary-background-color);--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-prominent-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host(.tonal-button),:host(.floating-button){--cr-button-background-color:var(--color-button-background-tonal,var(--cr-fallback-color-secondary-container));--cr-button-text-color:var(--color-button-foreground-tonal,var(--cr-fallback-color-on-tonal-container));--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-tonal-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host{flex-shrink:0;display:inline-flex;align-items:center;justify-content:center;box-sizing:border-box;min-width:5.14em;height:var(--cr-button-height);padding:8px 16px;outline-width:0;overflow:hidden;position:relative;cursor:pointer;user-select:none;-webkit-tap-highlight-color:transparent;border:var(--cr-button-border,1px solid var(--cr-button-border-color));border-radius:8px;background:var(--cr-button-background-color);color:var(--cr-button-text-color);font-weight:500;line-height:20px;isolation:isolate}@media (forced-colors:active){:host{forced-color-adjust:none}}:host(.floating-button){border-radius:8px;height:40px;transition:box-shadow 80ms linear}:host(.floating-button:hover){box-shadow:var(--cr-elevation-3)}:host([has-prefix-icon_]),:host([has-suffix-icon_]){--iron-icon-height:20px;--iron-icon-width:20px;--icon-block-padding-large:16px;--icon-block-padding-small:12px;gap:8px;padding-block-end:8px;padding-block-start:8px}:host([has-prefix-icon_]){padding-inline-end:var(--icon-block-padding-large);padding-inline-start:var(--icon-block-padding-small)}:host([has-suffix-icon_]){padding-inline-end:var(--icon-block-padding-small);padding-inline-start:var(--icon-block-padding-large)}#background{border-radius:inherit;inset:0;pointer-events:none;position:absolute}#content{display:inline}#hoverBackground{content:'';display:none;inset:0;pointer-events:none;position:absolute;z-index:1}:host(:hover) #hoverBackground{background:var(--cr-hover-background-color);display:block}:host(.action-button:hover) #hoverBackground{background:var(--cr-hover-on-prominent-background-color)}:host([disabled]){background:var(--cr-button-disabled-background-color);border:var(--cr-button-disabled-border,1px solid var(--cr-button-disabled-border-color));color:var(--cr-button-disabled-text-color);cursor:auto;pointer-events:none}:host(.cancel-button){margin-inline-end:8px}:host(.action-button),:host(.cancel-button){line-height:154%}#ink{color:var(--cr-button-ripple-color);--paper-ripple-opacity:var(--cr-button-ripple-opacity)}#background{z-index:0}#hoverBackground,cr-ripple{z-index:1}#content,::slotted(*){z-index:2}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$D() {
    return html `
<div id="background"></div>
<slot id="prefixIcon" name="prefix-icon"
    @slotchange="${this.onPrefixIconSlotChanged_}">
</slot>
<span id="content"><slot></slot></span>
<slot id="suffixIcon" name="suffix-icon"
    @slotchange="${this.onSuffixIconSlotChanged_}">
</slot>
<div id="hoverBackground" part="hoverBackground"></div>`;
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-button' is a button which displays slotted elements. It can
 * be interacted with like a normal button using click as well as space and
 * enter to effectively click the button and fire a 'click' event. It can also
 * style an icon inside of the button with the [has-icon] attribute.
 */
const CrButtonElementBase = CrRippleMixin(CrLitElement);
class CrButtonElement extends CrButtonElementBase {
    static get is() {
        return 'cr-button';
    }
    static get styles() {
        return getCss$J();
    }
    render() {
        return getHtml$D.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            hasPrefixIcon_: {
                type: Boolean,
                reflect: true,
            },
            hasSuffixIcon_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #hasPrefixIcon__accessor_storage = false;
    get hasPrefixIcon_() { return this.#hasPrefixIcon__accessor_storage; }
    set hasPrefixIcon_(value) { this.#hasPrefixIcon__accessor_storage = value; }
    #hasSuffixIcon__accessor_storage = false;
    get hasSuffixIcon_() { return this.#hasSuffixIcon__accessor_storage; }
    set hasSuffixIcon_(value) { this.#hasSuffixIcon__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    timeoutIds_ = new Set();
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        // Must be added in constructor so that stopImmediatePropagation() works as
        // expected.
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
        FocusOutlineManager.forDocument(document);
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.timeoutIds_.forEach(clearTimeout);
        this.timeoutIds_.clear();
    }
    setTimeout_(fn, delay) {
        if (!this.isConnected) {
            return;
        }
        const id = setTimeout(() => {
            this.timeoutIds_.delete(id);
            fn();
        }, delay);
        this.timeoutIds_.add(id);
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
        // If a keyup event is never fired (e.g. after keydown the focus is moved to
        // another element), we need to clear the ripple here. 100ms delay was
        // chosen manually as a good time period for the ripple to be visible.
        this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onPrefixIconSlotChanged_() {
        this.hasPrefixIcon_ = this.$.prefixIcon.assignedElements().length > 0;
    }
    onSuffixIconSlotChanged_() {
        this.hasSuffixIcon_ = this.$.suffixIcon.assignedElements().length > 0;
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        this.getRipple().uiDownAction();
        if (e.key === 'Enter') {
            this.click();
            // Delay was chosen manually as a good time period for the ripple to be
            // visible.
            this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
            this.getRipple().uiUpAction();
        }
    }
}
customElements.define(CrButtonElement.is, CrButtonElement);

let instance$M = null;
function getCss$I() {
    return instance$M || (instance$M = [...[], css `:host{-webkit-tap-highlight-color:transparent;align-items:center;cursor:pointer;display:flex;outline:none;user-select:none;--cr-checkbox-border-size:2px;--cr-checkbox-size:16px;--cr-checkbox-ripple-size:32px;--cr-checkbox-ripple-offset:50%;--cr-checkbox-checked-box-color:var(--owl-control-accent-color,var(--cr-checked-color));--cr-checkbox-ripple-checked-color:var(--cr-active-background-color);--cr-checkbox-ripple-opacity:1;--cr-checkbox-mark-color:var(--color-checkbox-check,var(--cr-fallback-color-on-primary));--cr-checkbox-ripple-unchecked-color:var(--cr-active-background-color);--cr-checkbox-unchecked-box-color:var(--color-checkbox-foreground-unchecked,var(--cr-fallback-color-outline));--cr-checkbox-checked-ripple-opacity:.2;--cr-checkbox-unchecked-ripple-opacity:.15}@media (prefers-color-scheme:dark){:host{--cr-checkbox-checked-ripple-opacity:.4;--cr-checkbox-unchecked-ripple-opacity:.4}}:host([disabled]){cursor:initial;opacity:1;pointer-events:none;--cr-checkbox-checked-box-color:var(--color-checkbox-container-disabled,var(--cr-fallback-color-disabled-background));--cr-checkbox-unchecked-box-color:var(--color-checkbox-outline-disabled,var(--cr-fallback-color-disabled-background));--cr-checkbox-mark-color:var(--color-checkbox-check-disabled,var(--cr-fallback-color-disabled-foreground))}#checkbox{background:none;border:var(--cr-checkbox-border-size) solid var(--cr-checkbox-unchecked-box-color);border-radius:2px;box-sizing:border-box;cursor:pointer;display:block;flex-shrink:0;height:var(--cr-checkbox-size);isolation:isolate;margin:0;outline:none;padding:0;position:relative;transform:none;width:var(--cr-checkbox-size)}:host([disabled][checked]) #checkbox{border-color:transparent}#hover-layer{display:none}#checkbox:hover #hover-layer{background-color:var(--cr-hover-background-color);border-radius:50%;display:block;height:32px;left:50%;overflow:hidden;pointer-events:none;position:absolute;top:50%;transform:translate(-50%,-50%);width:32px}@media (forced-colors:active){:host(:focus) #checkbox{outline:var(--cr-focus-outline-hcm)}}#checkbox:focus-visible{outline:var(--cr-checkbox-focus-outline,2px solid var(--cr-focus-outline-color));outline-offset:2px}#checkmark{display:block;forced-color-adjust:auto;position:relative;transform:scale(0);z-index:1}#checkmark path{fill:var(--cr-checkbox-mark-color)}:host([checked]) #checkmark{transform:scale(1);transition:transform 140ms ease-out}:host([checked]) #checkbox{background:var(--cr-checkbox-checked-box-background-color,var(--cr-checkbox-checked-box-color));border-color:var(--cr-checkbox-checked-box-color)}#ink{--paper-ripple-opacity:var(--cr-checkbox-ripple-opacity,var(--cr-checkbox-unchecked-ripple-opacity));color:var(--cr-checkbox-ripple-unchecked-color);height:var(--cr-checkbox-ripple-size);left:var(--cr-checkbox-ripple-offset);outline:var(--cr-checkbox-ripple-ring,none);pointer-events:none;top:var(--cr-checkbox-ripple-offset);transform:translate(-50%,-50%);transition:color linear 80ms;width:var(--cr-checkbox-ripple-size)}:host([checked]) #ink{--paper-ripple-opacity:var(--cr-checkbox-ripple-opacity,var(--cr-checkbox-checked-ripple-opacity));color:var(--cr-checkbox-ripple-checked-color)}:host-context([dir=rtl]) #ink{left:auto;right:var(--cr-checkbox-ripple-offset);transform:translate(50%,-50%)}#labelContainer{color:var(--cr-checkbox-label-color,var(--cr-primary-text-color));padding-inline-start:var(--cr-checkbox-label-padding-start,20px);white-space:normal}:host(.label-first) #labelContainer{order:-1;padding-inline-end:var(--cr-checkbox-label-padding-end,20px);padding-inline-start:0}:host(.no-label) #labelContainer{display:none}#ariaDescription{height:0;overflow:hidden;width:0}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$C() {
    return html `
<div id="checkbox" tabindex="${this.tabIndex}" role="checkbox"
    @keydown="${this.onKeyDown_}" @keyup="${this.onKeyUp_}"

    aria-disabled="${this.getAriaDisabled_()}"
    aria-checked="${this.getAriaChecked_()}"
    aria-label="${this.ariaLabelOverride || nothing}"
    aria-labelledby="${this.ariaLabelOverride ? nothing : 'labelContainer'}"
    aria-describedby="ariaDescription">
  <!-- Inline SVG paints faster than loading it from a separate file. -->
  <svg id="checkmark" width="12" height="12" viewBox="0 0 12 12"
      fill="none" xmlns="http://www.w3.org/2000/svg">
    <path fill-rule="evenodd" clip-rule="evenodd"
        d="m10.192 2.121-6.01 6.01-2.121-2.12L1 7.07l2.121 2.121.707.707.354.354 7.071-7.071-1.06-1.06Z">
  </svg>
  <div id="hover-layer"></div>
</div>
<div id="labelContainer" part="label-container">
  <slot></slot>
</div>
<div id="ariaDescription" aria-hidden="true">${this.ariaDescription}</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-checkbox' is a component similar to native checkbox. It
 * fires a 'change' event *only* when its state changes as a result of a user
 * interaction. By default it assumes there will be child(ren) passed in to be
 * used as labels. If no label will be provided, a .no-label class should be
 * added to hide the spacing between the checkbox and the label container.
 *
 * If a label is provided, it will be shown by default after the checkbox. A
 * .label-first CSS class can be added to show the label before the checkbox.
 *
 * List of customizable styles:
 *  --cr-checkbox-border-size
 *  --cr-checkbox-checked-box-background-color
 *  --cr-checkbox-checked-box-color
 *  --cr-checkbox-label-color
 *  --cr-checkbox-label-padding-start
 *  --cr-checkbox-mark-color
 *  --cr-checkbox-ripple-checked-color
 *  --cr-checkbox-ripple-size
 *  --cr-checkbox-ripple-unchecked-color
 *  --cr-checkbox-size
 *  --cr-checkbox-unchecked-box-color
 */
const CrCheckboxElementBase = CrRippleMixin(CrLitElement);
class CrCheckboxElement extends CrCheckboxElementBase {
    static get is() {
        return 'cr-checkbox';
    }
    static get styles() {
        return getCss$I();
    }
    render() {
        return getHtml$C.bind(this)();
    }
    static get properties() {
        return {
            checked: {
                type: Boolean,
                reflect: true,
                notify: true,
            },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            ariaDescription: { type: String },
            ariaLabelOverride: { type: String },
            tabIndex: { type: Number },
        };
    }
    #checked_accessor_storage = false;
    get checked() { return this.#checked_accessor_storage; }
    set checked(value) { this.#checked_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #ariaDescription_accessor_storage = null;
    get ariaDescription() { return this.#ariaDescription_accessor_storage; }
    set ariaDescription(value) { this.#ariaDescription_accessor_storage = value; }
    #ariaLabelOverride_accessor_storage;
    get ariaLabelOverride() { return this.#ariaLabelOverride_accessor_storage; }
    set ariaLabelOverride(value) { this.#ariaLabelOverride_accessor_storage = value; }
    #tabIndex_accessor_storage = 0;
    get tabIndex() { return this.#tabIndex_accessor_storage; }
    set tabIndex(value) { this.#tabIndex_accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('pointerup', this.hideRipple_.bind(this));
        this.$.labelContainer.addEventListener('pointerdown', this.showRipple_.bind(this));
        this.$.labelContainer.addEventListener('pointerleave', this.hideRipple_.bind(this));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('disabled')) {
            const previousTabIndex = changedProperties.get('disabled');
            // During initialization, don't alter tabIndex if not disabled. During
            // subsequent 'disabled' changes, always update tabIndex.
            if (previousTabIndex !== undefined || this.disabled) {
                this.tabIndex = this.disabled ? -1 : 0;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('tabIndex')) {
            // :host shouldn't have a tabindex because it's set on #checkbox.
            this.removeAttribute('tabindex');
        }
    }
    focus() {
        this.$.checkbox.focus();
    }
    getFocusableElement() {
        return this.$.checkbox;
    }
    getAriaDisabled_() {
        return this.disabled ? 'true' : 'false';
    }
    getAriaChecked_() {
        return this.checked ? 'true' : 'false';
    }
    showRipple_() {
        if (this.noink) {
            return;
        }
        this.getRipple().showAndHoldDown();
    }
    hideRipple_() {
        this.getRipple().clear();
    }
    // 
    async onClick_(e) {
        if (this.disabled || e.target.tagName === 'A') {
            return;
        }
        // Prevent |click| event from bubbling. It can cause parents of this
        // elements to erroneously re-toggle this control.
        e.stopPropagation();
        e.preventDefault();
        this.checked = !this.checked;
        await this.updateComplete;
        this.fire('change', this.checked);
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.click();
        }
    }
    onKeyUp_(e) {
        if (e.key === ' ' || e.key === 'Enter') {
            e.preventDefault();
            e.stopPropagation();
        }
        if (e.key === ' ') {
            this.click();
        }
    }
    // Overridden from CrRippleMixin
    createRipple() {
        this.rippleContainer = this.$.checkbox;
        const ripple = super.createRipple();
        ripple.setAttribute('recenters', '');
        ripple.classList.add('circle');
        return ripple;
    }
}
customElements.define(CrCheckboxElement.is, CrCheckboxElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let iconsetMap = null;
class IconsetMap extends EventTarget {
    iconsets_ = new Map();
    static getInstance() {
        return iconsetMap || (iconsetMap = new IconsetMap());
    }
    static resetInstanceForTesting(instance) {
        iconsetMap = instance;
    }
    get(id) {
        return this.iconsets_.get(id) || null;
    }
    set(id, iconset) {
        assert(!this.iconsets_.has(id), `Tried to add a second iconset with id '${id}'`);
        this.iconsets_.set(id, iconset);
        this.dispatchEvent(new CustomEvent('cr-iconset-added', { detail: id }));
    }
}

let instance$L = null;
function getCss$H() {
    return instance$L || (instance$L = [...[getCss$K()], css `:host{align-items:center;display:inline-flex;justify-content:center;position:relative;vertical-align:middle;fill:var(--iron-icon-fill-color,currentcolor);stroke:var(--iron-icon-stroke-color,none);width:var(--iron-icon-width,24px);height:var(--iron-icon-height,24px)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrIconElement extends CrLitElement {
    static get is() {
        return 'cr-icon';
    }
    static get styles() {
        return getCss$H();
    }
    static get properties() {
        return {
            /**
             * The name of the icon to use. The name should be of the form:
             * `iconset_name:icon_name`.
             */
            icon: { type: String },
        };
    }
    #icon_accessor_storage = '';
    get icon() { return this.#icon_accessor_storage; }
    set icon(value) { this.#icon_accessor_storage = value; }
    iconsetName_ = '';
    iconName_ = '';
    iconset_ = null;
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('icon')) {
            const [iconsetName, iconName] = this.icon.split(':');
            this.iconName_ = iconName || '';
            this.iconsetName_ = iconsetName || '';
            this.updateIcon_();
        }
    }
    updateIcon_() {
        if (this.iconName_ === '' && this.iconset_) {
            this.iconset_.removeIcon(this);
        }
        else if (this.iconsetName_) {
            const iconsetMap = IconsetMap.getInstance();
            this.iconset_ = iconsetMap.get(this.iconsetName_);
            assert(this.iconset_, `Could not find iconset for: '${this.iconsetName_}:${this.iconName_}'`);
            this.iconset_.applyIcon(this, this.iconName_);
        }
    }
}
customElements.define(CrIconElement.is, CrIconElement);

let instance$K = null;
function getCss$G() {
    return instance$K || (instance$K = [...[], css `:host{--cr-icon-button-fill-color:currentColor;--cr-icon-button-icon-start-offset:0;--cr-icon-button-icon-size:20px;--cr-icon-button-size:32px;--cr-icon-button-height:var(--cr-icon-button-size);--cr-icon-button-transition:150ms ease-in-out;--cr-icon-button-width:var(--cr-icon-button-size);-webkit-tap-highlight-color:transparent;border-radius:50%;color:var(--cr-icon-button-stroke-color,var(--cr-icon-button-fill-color));cursor:pointer;display:inline-flex;flex-shrink:0;height:var(--cr-icon-button-height);margin-inline-end:var(--cr-icon-button-margin-end,var(--cr-icon-ripple-margin));margin-inline-start:var(--cr-icon-button-margin-start);outline:none;overflow:hidden;position:relative;user-select:none;vertical-align:middle;width:var(--cr-icon-button-width)}:host(:hover){background-color:var(--cr-icon-button-hover-background-color,var(--cr-hover-background-color))}:host(:focus-visible:focus){box-shadow:inset 0 0 0 2px var(--cr-icon-button-focus-outline-color,var(--cr-focus-outline-color))}@media (forced-colors:active){:host(:focus-visible:focus){outline:var(--cr-focus-outline-hcm)}}#ink{--paper-ripple-opacity:1;color:var(--cr-icon-button-active-background-color,var(--cr-active-background-color))}:host([disabled]){cursor:initial;opacity:var(--cr-disabled-opacity);pointer-events:none}:host(.no-overlap){--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0}:host-context([dir=rtl]):host(:not([suppress-rtl-flip]):not([multiple-icons_])){transform:scaleX(-1)}:host-context([dir=rtl]):host(:not([suppress-rtl-flip])[multiple-icons_]) cr-icon{transform:scaleX(-1)}:host(:not([iron-icon])) #maskedImage{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-button-icon-size);-webkit-transform:var(--cr-icon-image-transform,none);background-color:var(--cr-icon-button-fill-color);height:100%;transition:background-color var(--cr-icon-button-transition);width:100%}@media (forced-colors:active){:host(:not([iron-icon])) #maskedImage{background-color:ButtonText}}#icon{align-items:center;border-radius:4px;display:flex;height:100%;justify-content:center;padding-inline-start:var(--cr-icon-button-icon-start-offset);position:relative;width:100%}cr-icon{--iron-icon-fill-color:var(--cr-icon-button-fill-color);--iron-icon-stroke-color:var(--cr-icon-button-stroke-color,none);--iron-icon-height:var(--cr-icon-button-icon-size);--iron-icon-width:var(--cr-icon-button-icon-size);transition:fill var(--cr-icon-button-transition),stroke var(--cr-icon-button-transition)}@media (prefers-color-scheme:dark){:host{--cr-icon-button-fill-color:var(--google-grey-500)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$B() {
    return html `
<div id="icon">
  <div id="maskedImage"></div>
</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-icon-button' is a button which displays an icon with a
 * ripple. It can be interacted with like a normal button using click as well as
 * space and enter to effectively click the button and fire a 'click' event.
 *
 * There are two sources to icons:
 * Option 1: CSS classes defined in cr_icons.css.
 * Option 2: SVG icons defined in a cr-iconset or iron-iconset-svg,
 *     with the name passed to cr-icon-button via the |ironIcon| property.
 *
 * Example of using CSS classes:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/cr_icons.css.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button class="icon-class-name"></cr-icon-button>`;
 * }
 *
 * When an icon is specified using a class, the expectation is the
 * class will set an image to the --cr-icon-image variable.
 *
 * Example of using a cr-iconset to supply an icon via the iron-icon parameter:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/icons.html.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button iron-icon="cr:icon-key"></cr-icon-button>`;
 * }
 *
 * The color of the icon can be overridden using CSS variables. When using
 * the ironIcon property to populate cr-icon-button's internal <cr-icon>, the
 * following CSS variables for fill and stroke can be overridden for cr-icon:
 * --iron-icon-button-fill-color
 * --iron-icon-button-stroke-color
 *
 * When not using the ironIcon property, cr-icon-button will not create a
 * <cr-icon>, so the cr-icon related CSS variables above are ignored.
 *
 * When using the ironIcon property, more than one icon can be specified by
 * setting the |ironIcon| property to a comma-delimited list of keys.
 */
const CrIconbuttonElementBase = CrRippleMixin(CrLitElement);
class CrIconButtonElement extends CrIconbuttonElementBase {
    static get is() {
        return 'cr-icon-button';
    }
    static get styles() {
        return getCss$G();
    }
    render() {
        return getHtml$B.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            ironIcon: {
                type: String,
                reflect: true,
            },
            suppressRtlFlip: {
                type: Boolean,
                value: false,
                reflect: true,
            },
            multipleIcons_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #ironIcon_accessor_storage;
    get ironIcon() { return this.#ironIcon_accessor_storage; }
    set ironIcon(value) { this.#ironIcon_accessor_storage = value; }
    #multipleIcons__accessor_storage = false;
    get multipleIcons_() { return this.#multipleIcons__accessor_storage; }
    set multipleIcons_(value) { this.#multipleIcons__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('ironIcon')) {
            const icons = (this.ironIcon || '').split(',');
            this.multipleIcons_ = icons.length > 1;
        }
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
        if (changedProperties.has('ironIcon')) {
            this.onIronIconChanged_();
        }
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onIronIconChanged_() {
        this.shadowRoot.querySelectorAll('cr-icon').forEach(el => el.remove());
        if (!this.ironIcon) {
            return;
        }
        const icons = (this.ironIcon || '').split(',');
        icons.forEach(async (icon) => {
            const crIcon = document.createElement('cr-icon');
            crIcon.icon = icon;
            this.$.icon.appendChild(crIcon);
            await crIcon.updateComplete;
            crIcon.shadowRoot.querySelectorAll('svg, img')
                .forEach(child => child.setAttribute('role', 'none'));
        });
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.click();
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key === ' ' || e.key === 'Enter') {
            e.preventDefault();
            e.stopPropagation();
        }
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
        }
    }
}
customElements.define(CrIconButtonElement.is, CrIconButtonElement);

let instance$J = null;
function getCss$F() {
    return instance$J || (instance$J = [...[], css `.icon-arrow-back{--cr-icon-image:url(//resources/images/icon_arrow_back.svg)}.icon-arrow-dropdown{--cr-icon-image:url(//resources/images/icon_arrow_dropdown.svg)}.icon-arrow-drop-down-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_down_cr23.svg)}.icon-arrow-drop-up-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_up_cr23.svg)}.icon-arrow-upward{--cr-icon-image:url(//resources/images/icon_arrow_upward.svg)}.icon-cancel{--cr-icon-image:url(//resources/images/icon_cancel.svg)}.icon-clear{--cr-icon-image:url(//resources/images/icon_clear.svg)}.icon-copy-content{--cr-icon-image:url(//resources/images/icon_copy_content.svg)}.icon-delete-gray{--cr-icon-image:url(//resources/images/icon_delete_gray.svg)}.icon-edit{--cr-icon-image:url(//resources/images/icon_edit.svg)}.icon-file{--cr-icon-image:url(//resources/images/icon_filetype_generic.svg)}.icon-folder-open{--cr-icon-image:url(//resources/images/icon_folder_open.svg)}.icon-picture-delete{--cr-icon-image:url(//resources/images/icon_picture_delete.svg)}.icon-expand-less{--cr-icon-image:url(//resources/images/icon_expand_less.svg)}.icon-expand-more{--cr-icon-image:url(//resources/images/icon_expand_more.svg)}.icon-external{--cr-icon-image:url(//resources/images/open_in_new.svg)}.icon-more-vert{--cr-icon-image:url(//resources/images/icon_more_vert.svg)}.icon-refresh{--cr-icon-image:url(//resources/images/icon_refresh.svg)}.icon-search{--cr-icon-image:url(//resources/images/icon_search.svg)}.icon-settings{--cr-icon-image:url(//resources/images/icon_settings.svg)}.icon-visibility{--cr-icon-image:url(//resources/images/icon_visibility.svg)}.icon-visibility-off{--cr-icon-image:url(//resources/images/icon_visibility_off.svg)}.subpage-arrow{--cr-icon-image:url(//resources/images/arrow_right.svg)}.cr-icon{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-size);background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-700)));flex-shrink:0;height:var(--cr-icon-ripple-size);margin-inline-end:var(--cr-icon-ripple-margin);margin-inline-start:var(--cr-icon-button-margin-start);user-select:none;width:var(--cr-icon-ripple-size)}:host-context([dir=rtl]) .cr-icon{transform:scaleX(-1)}.cr-icon.no-overlap{margin-inline-end:0;margin-inline-start:0}@media (prefers-color-scheme:dark){.cr-icon{background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-500)))}}`]);
}

let instance$I = null;
function getCss$E() {
    return instance$I || (instance$I = [...[getCss$K(), getCss$F()], css `[actionable]{cursor:pointer}.hr{border-top:var(--cr-separator-line)}iron-list.cr-separators>*:not([first]){border-top:var(--cr-separator-line)}[scrollable]{border-color:transparent;border-style:solid;border-width:1px 0;overflow-y:auto}[scrollable].is-scrolled{border-top-color:var(--cr-scrollable-border-color)}[scrollable].can-scroll:not(.scrolled-to-bottom){border-bottom-color:var(--cr-scrollable-border-color)}[scrollable] iron-list>:not(.no-outline):focus-visible,[selectable]:focus-visible,[selectable]>:focus-visible{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}.scroll-container{display:flex;flex-direction:column;min-height:1px}[selectable]>*{cursor:pointer}.cr-centered-card-container{box-sizing:border-box;display:block;height:inherit;margin:0 auto;max-width:var(--cr-centered-card-max-width);min-width:550px;position:relative;width:calc(100% * var(--cr-centered-card-width-percentage))}.cr-container-shadow{height:var(--cr-container-shadow-height);left:0;margin:0 0 var(--cr-container-shadow-margin);opacity:0;pointer-events:none;position:relative;right:0;top:0;transition:opacity 500ms;z-index:1}#cr-container-shadow-bottom{margin-bottom:0;margin-top:var(--cr-container-shadow-margin);transform:scaleY(-1)}#cr-container-shadow-top:has(+#container.can-scroll:not(.scrolled-to-top)),#container.can-scroll:not(.scrolled-to-bottom)+#cr-container-shadow-bottom,#cr-container-shadow-bottom.force-shadow,#cr-container-shadow-top.force-shadow{opacity:var(--cr-container-shadow-max-opacity)}.cr-row{align-items:center;border-top:var(--cr-separator-line);display:flex;min-height:var(--cr-section-min-height);padding:0 var(--cr-section-padding)}.cr-row.first,.cr-row.continuation{border-top:none}.cr-row-gap{padding-inline-start:16px}.cr-button-gap{margin-inline-start:8px}paper-tooltip::part(tooltip),cr-tooltip::part(tooltip){border-radius:var(--paper-tooltip-border-radius,2px);font-size:92.31%;font-weight:500;max-width:330px;min-width:var(--paper-tooltip-min-width,200px);padding:var(--paper-tooltip-padding,10px 8px)}.cr-padded-text{padding-block-end:var(--cr-section-vertical-padding);padding-block-start:var(--cr-section-vertical-padding)}.cr-title-text{color:var(--cr-title-text-color);font-size:107.6923%;font-weight:500}.cr-secondary-text{color:var(--cr-secondary-text-color);font-weight:400}.cr-form-field-label{color:var(--cr-form-field-label-color);display:block;font-size:var(--cr-form-field-label-font-size);font-weight:500;letter-spacing:.4px;line-height:var(--cr-form-field-label-line-height);margin-bottom:8px}.cr-vertical-tab{align-items:center;display:flex}.cr-vertical-tab::before{border-radius:0 3px 3px 0;content:'';display:block;flex-shrink:0;height:var(--cr-vertical-tab-height,100%);width:4px}.cr-vertical-tab.selected::before{background:var(--cr-vertical-tab-selected-color,var(--cr-checked-color))}:host-context([dir=rtl]) .cr-vertical-tab::before{transform:scaleX(-1)}.iph-anchor-highlight{background-color:var(--cr-iph-anchor-highlight-color)}`]);
}

let instance$H = null;
function getCss$D() {
    return instance$H || (instance$H = [...[], css `:host{--cr-input-background-color:rgba(255,255,255,1.0);--cr-input-border-bottom:0px;--cr-input-border-radius:8px;--cr-input-color:var(--cr-primary-text-color);--cr-input-error-color:var(--color-textfield-filled-error,var(--cr-fallback-color-error));--cr-input-focus-color:var(--color-textfield-filled-underline-focused,var(--cr-fallback-color-primary));--cr-input-hover-background-color:var(--cr-hover-background-color);--cr-input-label-color:var(--color-textfield-foreground-label,var(--cr-fallback-color-on-surface-subtle));--cr-input-padding-bottom:10px;--cr-input-padding-end:10px;--cr-input-padding-start:10px;--cr-input-padding-top:10px;--cr-input-placeholder-color:var(--color-textfield-foreground-placeholder,var(--cr-fallback-on-surface-subtle));display:block;isolation:isolate;outline:none}:host([readonly]){--cr-input-border-radius:8px}#label{color:var(--cr-input-label-color);font-size:11px;line-height:16px}:host([focused_]:not([readonly]):not([invalid])) #label{color:var(--cr-input-focus-label-color,var(--cr-input-label-color))}#input-container{border-radius:8px;overflow:hidden;position:relative;width:var(--cr-input-width,100%)}:host([focused_]) #input-container{outline:var(--cr-input-focus-outline,none)}#inner-input-container{background-color:var(--cr-input-background-color);box-sizing:border-box;padding:0}#inner-input-content ::slotted(*){--cr-icon-button-fill-color:var(--color-textfield-foreground-icon,var(--cr-fallback-color-on-surface-subtle));--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px;--cr-icon-button-margin-start:0;--cr-icon-color:var(--color-textfield-foreground-icon,var(--cr-fallback-color-on-surface-subtle))}#inner-input-content ::slotted([slot='inline-prefix']){--cr-icon-button-margin-start:-8px}#inner-input-content ::slotted([slot='inline-suffix']){--cr-icon-button-margin-end:-4px}:host([invalid]) #inner-input-content ::slotted(*){--cr-icon-color:var(--cr-input-error-color);--cr-icon-button-fill-color:var(--cr-input-error-color)}#hover-layer{display:none;inset:0;pointer-events:none;position:absolute;z-index:0}:host(:not([readonly]):not([disabled])) #input-container:hover #hover-layer{display:block}#input{-webkit-appearance:none;background-color:transparent;border:none;box-sizing:border-box;caret-color:var(--cr-input-focus-color);color:var(--cr-input-color);font-family:inherit;font-size:var(--cr-input-font-size,12px);font-weight:inherit;line-height:16px;min-height:var(--cr-input-min-height,auto);outline:none;padding:0;text-align:inherit;text-overflow:ellipsis;width:100%}#inner-input-content{padding-bottom:var(--cr-input-padding-bottom);padding-inline-end:var(--cr-input-padding-end);padding-inline-start:var(--cr-input-padding-start);padding-top:var(--cr-input-padding-top)}#underline{border-bottom:0;border-radius:8px;bottom:0;box-sizing:border-box;display:var(--cr-input-underline-display);height:var(--cr-input-underline-height,0);left:0;margin:auto;opacity:0;position:absolute;right:0;transition:opacity 120ms ease-out,width 0s linear 180ms;width:0}:host([invalid]) #underline,:host([force-underline]) #underline,:host([focused_]) #underline{opacity:1;transition:opacity 120ms ease-in,width 180ms ease-out;width:100%}#underline-base{display:none}:host([readonly]) #underline{display:none}:host(:not([readonly])) #underline-base{border-bottom:0;bottom:0;display:block;left:0;position:absolute;right:0}:host([disabled]){color:var(--color-textfield-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-input-border-bottom:1px solid currentColor;--cr-input-placeholder-color:currentColor;--cr-input-color:currentColor;--cr-input-background-color:var(--color-textfield-background-disabled,var(--cr-fallback-color-disabled-background))}:host([disabled]) #inner-input-content ::slotted(*){--cr-icon-color:currentColor;--cr-icon-button-fill-color:currentColor}:host(.stroked){--cr-input-background-color:transparent;--cr-input-border:1px solid var(--color-side-panel-textfield-border,var(--cr-fallback-color-neutral-outline));--cr-input-border-bottom:none;--cr-input-border-radius:8px;--cr-input-padding-bottom:9px;--cr-input-padding-end:9px;--cr-input-padding-start:9px;--cr-input-padding-top:9px;--cr-input-underline-display:none;--cr-input-min-height:36px;line-height:16px}:host(.stroked[focused_]){--cr-input-border:2px solid var(--cr-focus-outline-color);--cr-input-padding-bottom:8px;--cr-input-padding-end:8px;--cr-input-padding-start:8px;--cr-input-padding-top:8px}:host(.stroked[invalid]){--cr-input-border:1px solid var(--cr-input-error-color)}:host(.stroked[focused_][invalid]){--cr-input-border:2px solid var(--cr-input-error-color)}@media (prefers-color-scheme:dark){:host{--cr-input-background-color:rgba(33,33,33,1.0)}}`]);
}

let instance$G = null;
function getCss$C() {
    return instance$G || (instance$G = [...[getCss$K(), getCss$D(), getCss$E()], css `:host([disabled]) :-webkit-any(#label,#error,#input-container){opacity:var(--cr-disabled-opacity);pointer-events:none}:host([disabled]) :is(#label,#error,#input-container){opacity:1}:host ::slotted(cr-button[slot=suffix]){margin-inline-start:var(--cr-button-edge-spacing) !important}:host([invalid]) #label{color:var(--cr-input-error-color)}#input{border-bottom:none;letter-spacing:var(--cr-input-letter-spacing)}#input-container{border-radius:8px;border:var(--owl-border-override,1px solid rgba(0,0,0,0.1))}#input::placeholder{color:var(--cr-input-placeholder-color,var(--cr-secondary-text-color));letter-spacing:var(--cr-input-placeholder-letter-spacing)}:host([invalid]) #input{caret-color:var(--cr-input-error-color)}:host([readonly]) #input{opacity:var(--cr-input-readonly-opacity,0.6)}:host([invalid]) #underline{border-color:var(--cr-input-error-color)}#error{color:var(--cr-input-error-color);display:var(--cr-input-error-display,block);font-size:11px;min-height:var(--cr-form-field-label-height);line-height:16px;margin:4px 10px;visibility:hidden;white-space:var(--cr-input-error-white-space);height:auto;overflow:hidden;text-overflow:ellipsis}:host([invalid]) #error{visibility:visible}#row-container,#inner-input-content{align-items:center;display:flex;justify-content:space-between;position:relative}#inner-input-content{gap:4px;height:16px;z-index:1}#input[type='search']::-webkit-search-cancel-button{display:none}:host-context([dir=rtl]) #input[type=url]{text-align:right}#input[type=url]{direction:ltr}@media (prefers-color-scheme:dark){#input-container{border:var(--owl-border-override,1px solid rgba(255,255,255,0.8))}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$A() {
    return html `
<div id="label" class="cr-form-field-label" ?hidden="${!this.label}"
    aria-hidden="true">
  ${this.label}
</div>
<div id="row-container" part="row-container">
  <div id="input-container">
    <div id="inner-input-container">
      <div id="hover-layer"></div>
      <div id="inner-input-content">
        <slot name="inline-prefix"></slot>
        <input id="input" ?disabled="${this.disabled}"
            ?autofocus="${this.autofocus}"
            .value="${this.internalValue_}" tabindex="${this.inputTabindex}"
            .type="${this.type}"
            ?readonly="${this.readonly}" maxlength="${this.maxlength}"
            pattern="${this.pattern || nothing}" ?required="${this.required}"
            minlength="${this.minlength}" inputmode="${this.inputmode}"
            aria-description="${this.ariaDescription || nothing}"
            aria-errormessage="${this.getAriaErrorMessage_() || nothing}"
            aria-label="${this.getAriaLabel_()}"
            aria-invalid="${this.getAriaInvalid_()}"
            .max="${this.max || nothing}" .min="${this.min || nothing}"
            @focus="${this.onInputFocus_}"
            @blur="${this.onInputBlur_}" @change="${this.onInputChange_}"
            @input="${this.onInput_}"
            part="input"
            autocomplete="off">
        <slot name="inline-suffix"></slot>
      </div>
    </div>
    <div id="underline-base"></div>
    <div id="underline"></div>
  </div>
  <slot name="suffix"></slot>
</div>
<div id="error" role="${this.getErrorRole_() || nothing}"
    aria-live="assertive">${this.getErrorMessage_()}</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Input types supported by cr-input.
 */
const SUPPORTED_INPUT_TYPES = new Set([
    'number',
    'password',
    'search',
    'text',
    'url',
]);
class CrInputElement extends CrLitElement {
    static get is() {
        return 'cr-input';
    }
    static get styles() {
        return getCss$C();
    }
    render() {
        return getHtml$A.bind(this)();
    }
    static get properties() {
        return {
            ariaDescription: { type: String },
            ariaLabel: { type: String },
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            autoValidate: { type: Boolean },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            errorMessage: { type: String },
            errorRole_: { type: String },
            /**
             * This is strictly used internally for styling, do not attempt to use
             * this to set focus.
             */
            focused_: {
                type: Boolean,
                reflect: true,
            },
            invalid: {
                type: Boolean,
                notify: true,
                reflect: true,
            },
            max: {
                type: Number,
                reflect: true,
            },
            min: {
                type: Number,
                reflect: true,
            },
            maxlength: {
                type: Number,
                reflect: true,
            },
            minlength: {
                type: Number,
                reflect: true,
            },
            pattern: {
                type: String,
                reflect: true,
            },
            inputmode: { type: String },
            label: { type: String },
            placeholder: { type: String },
            readonly: {
                type: Boolean,
                reflect: true,
            },
            required: {
                type: Boolean,
                reflect: true,
            },
            inputTabindex: { type: Number },
            type: { type: String },
            value: {
                type: String,
                notify: true,
            },
            internalValue_: {
                type: String,
                state: true,
            },
        };
    }
    #ariaDescription_accessor_storage = null;
    get ariaDescription() { return this.#ariaDescription_accessor_storage; }
    set ariaDescription(value) { this.#ariaDescription_accessor_storage = value; }
    #ariaLabel_accessor_storage = '';
    get ariaLabel() { return this.#ariaLabel_accessor_storage; }
    set ariaLabel(value) { this.#ariaLabel_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #autoValidate_accessor_storage = false;
    get autoValidate() { return this.#autoValidate_accessor_storage; }
    set autoValidate(value) { this.#autoValidate_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #errorMessage_accessor_storage = '';
    get errorMessage() { return this.#errorMessage_accessor_storage; }
    set errorMessage(value) { this.#errorMessage_accessor_storage = value; }
    #inputmode_accessor_storage;
    get inputmode() { return this.#inputmode_accessor_storage; }
    set inputmode(value) { this.#inputmode_accessor_storage = value; }
    #inputTabindex_accessor_storage = 0;
    get inputTabindex() { return this.#inputTabindex_accessor_storage; }
    set inputTabindex(value) { this.#inputTabindex_accessor_storage = value; }
    #invalid_accessor_storage = false;
    get invalid() { return this.#invalid_accessor_storage; }
    set invalid(value) { this.#invalid_accessor_storage = value; }
    #label_accessor_storage = '';
    get label() { return this.#label_accessor_storage; }
    set label(value) { this.#label_accessor_storage = value; }
    #max_accessor_storage;
    get max() { return this.#max_accessor_storage; }
    set max(value) { this.#max_accessor_storage = value; }
    #min_accessor_storage;
    get min() { return this.#min_accessor_storage; }
    set min(value) { this.#min_accessor_storage = value; }
    #maxlength_accessor_storage;
    get maxlength() { return this.#maxlength_accessor_storage; }
    set maxlength(value) { this.#maxlength_accessor_storage = value; }
    #minlength_accessor_storage;
    get minlength() { return this.#minlength_accessor_storage; }
    set minlength(value) { this.#minlength_accessor_storage = value; }
    #pattern_accessor_storage;
    get pattern() { return this.#pattern_accessor_storage; }
    set pattern(value) { this.#pattern_accessor_storage = value; }
    #placeholder_accessor_storage = null;
    get placeholder() { return this.#placeholder_accessor_storage; }
    set placeholder(value) { this.#placeholder_accessor_storage = value; }
    #readonly_accessor_storage = false;
    get readonly() { return this.#readonly_accessor_storage; }
    set readonly(value) { this.#readonly_accessor_storage = value; }
    #required_accessor_storage = false;
    get required() { return this.#required_accessor_storage; }
    set required(value) { this.#required_accessor_storage = value; }
    #type_accessor_storage = 'text';
    get type() { return this.#type_accessor_storage; }
    set type(value) { this.#type_accessor_storage = value; }
    #value_accessor_storage = '';
    get value() { return this.#value_accessor_storage; }
    set value(value) { this.#value_accessor_storage = value; }
    #internalValue__accessor_storage = '';
    get internalValue_() { return this.#internalValue__accessor_storage; }
    set internalValue_(value) { this.#internalValue__accessor_storage = value; }
    #focused__accessor_storage = false;
    get focused_() { return this.#focused__accessor_storage; }
    set focused_(value) { this.#focused__accessor_storage = value; }
    firstUpdated() {
        // Use inputTabindex instead.
        assert(!this.hasAttribute('tabindex'));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('value')) {
            // Don't allow null or undefined as these will render in the input.
            // cr-input cannot use Lit's "nothing" in the HTML template; this breaks
            // the underlying native input's auto validation if |required| is set.
            this.internalValue_ =
                (this.value === undefined || this.value === null) ? '' : this.value;
        }
        if (changedProperties.has('inputTabindex')) {
            // CrInput only supports 0 or -1 values for the input's tabindex to allow
            // having the input in tab order or not. Values greater than 0 will not
            // work as the shadow root encapsulates tabindices.
            assert(this.inputTabindex === 0 || this.inputTabindex === -1);
        }
        if (changedProperties.has('type')) {
            // Check that the 'type' is one of the supported types.
            assert(SUPPORTED_INPUT_TYPES.has(this.type));
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('value')) {
            const previous = changedProperties.get('value');
            if ((!!this.value || !!previous) && this.autoValidate) {
                this.invalid = !this.inputElement.checkValidity();
            }
        }
        if (changedProperties.has('placeholder')) {
            if (this.placeholder === null || this.placeholder === undefined) {
                this.inputElement.removeAttribute('placeholder');
            }
            else {
                this.inputElement.setAttribute('placeholder', this.placeholder);
            }
        }
    }
    get inputElement() {
        return this.$.input;
    }
    focus() {
        this.focusInput();
    }
    /**
     * Focuses the input element.
     * TODO(crbug.com/40593040): Replace this with focus() after resolving the text
     * selection issue described in onFocus_().
     * @return Whether the <input> element was focused.
     */
    focusInput() {
        if (this.shadowRoot.activeElement === this.inputElement) {
            return false;
        }
        this.inputElement.focus();
        return true;
    }
    /**
     * 'change' event fires when <input> value changes and user presses 'Enter'.
     * This function helps propagate it to host since change events don't
     * propagate across Shadow DOM boundary by default.
     */
    async onInputChange_(e) {
        // Ensure that |value| has been updated before re-firing 'change'.
        await this.updateComplete;
        this.fire('change', { sourceEvent: e });
    }
    onInput_(e) {
        this.internalValue_ = e.target.value;
        this.value = this.internalValue_;
    }
    onInputFocus_() {
        this.focused_ = true;
    }
    onInputBlur_() {
        this.focused_ = false;
    }
    getAriaLabel_() {
        return this.ariaLabel || this.label || this.placeholder;
    }
    getAriaInvalid_() {
        return this.invalid ? 'true' : 'false';
    }
    getErrorMessage_() {
        return this.invalid ? this.errorMessage : '';
    }
    getErrorRole_() {
        // On VoiceOver role="alert" is not consistently announced when its
        // content changes. Adding and removing the |role| attribute every time
        // there is an error, triggers VoiceOver to consistently announce.
        return this.invalid ? 'alert' : '';
    }
    getAriaErrorMessage_() {
        return this.invalid ? 'error' : '';
    }
    /**
     * Selects the text within the input. If no parameters are passed, it will
     * select the entire string. Either no params or both params should be passed.
     * Publicly, this function should be used instead of inputElement.select() or
     * manipulating inputElement.selectionStart/selectionEnd because the order of
     * execution between focus() and select() is sensitive.
     */
    select(start, end) {
        this.inputElement.focus();
        if (start !== undefined && end !== undefined) {
            this.inputElement.setSelectionRange(start, end);
        }
        else {
            // Can't just pass one param.
            assert(start === undefined && end === undefined);
            this.inputElement.select();
        }
    }
    // Note: In order to preserve it as a synchronous API, validate() forces 2
    // rendering updates to cr-input. This allows this function to be used to
    // synchronously determine the validity of a <cr-input>, however, as a result
    // of these 2 forced updates it may result in slower performance. validate()
    // should not be called internally from within cr_input.ts, and should only
    // be called where necessary from clients.
    validate() {
        // Ensure that any changes to |value| have propagated to the native <input>.
        this.performUpdate();
        this.invalid = !this.inputElement.checkValidity();
        // Perform update again to ensure change propagates via 2 way binding to
        // Polymer parent before returning.
        this.performUpdate();
        return !this.invalid;
    }
}
customElements.define(CrInputElement.is, CrInputElement);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A helper object used to get a pluralized string.
 */
// clang-format off
class PluralStringProxyImpl {
    getPluralString(messageName, itemCount) {
        return sendWithPromise('getPluralString', messageName, itemCount);
    }
    getPluralStringTupleWithComma(messageName1, itemCount1, messageName2, itemCount2) {
        return sendWithPromise('getPluralStringTupleWithComma', messageName1, itemCount1, messageName2, itemCount2);
    }
    getPluralStringTupleWithPeriods(messageName1, itemCount1, messageName2, itemCount2) {
        return sendWithPromise('getPluralStringTupleWithPeriods', messageName1, itemCount1, messageName2, itemCount2);
    }
    static getInstance() {
        return instance$F || (instance$F = new PluralStringProxyImpl());
    }
    static setInstance(obj) {
        instance$F = obj;
    }
}
let instance$F = null;

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return Whether the passed tagged template literal is a valid array.
 */
function isValidArray(arr) {
    if (arr instanceof Array && Object.isFrozen(arr)) {
        return true;
    }
    return false;
}
/**
 * Checks if the passed tagged template literal only contains static string.
 * And return the string in the literal if so.
 * Throws an Error if the passed argument is not supported literals.
 */
function getStaticString(literal) {
    const isStaticString = isValidArray(literal) && !!literal.raw &&
        isValidArray(literal.raw) && literal.length === literal.raw.length &&
        literal.length === 1;
    assert(isStaticString, 'static_types.js only allows static strings');
    return literal.join('');
}
function createTypes(_ignore, literal) {
    return getStaticString(literal);
}
/**
 * Rules used to enforce static literal checks.
 */
const rules = {
    createHTML: createTypes,
    createScript: createTypes,
    createScriptURL: createTypes,
};
/**
 * This policy returns Trusted Types if the passed literal is static.
 */
let staticPolicy;
if (window.trustedTypes) {
    staticPolicy = window.trustedTypes.createPolicy('static-types', rules);
}
else {
    staticPolicy = rules;
}
/**
 * Returns TrustedHTML if the passed literal is static.
 */
function getTrustedHTML(literal) {
    return staticPolicy.createHTML('', literal);
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Same as cloud_devices::printer::TypedValueVendorCapability::ValueType.
 */
var VendorCapabilityValueType;
(function (VendorCapabilityValueType) {
    VendorCapabilityValueType["BOOLEAN"] = "BOOLEAN";
    VendorCapabilityValueType["FLOAT"] = "FLOAT";
    VendorCapabilityValueType["INTEGER"] = "INTEGER";
    VendorCapabilityValueType["STRING"] = "STRING";
})(VendorCapabilityValueType || (VendorCapabilityValueType = {}));
/**
 * Values matching the types of duplex in a CDD.
 */
var DuplexType;
(function (DuplexType) {
    DuplexType["NO_DUPLEX"] = "NO_DUPLEX";
    DuplexType["LONG_EDGE"] = "LONG_EDGE";
    DuplexType["SHORT_EDGE"] = "SHORT_EDGE";
})(DuplexType || (DuplexType = {}));

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Enumeration of the origin types for destinations.
 */
var DestinationOrigin;
(function (DestinationOrigin) {
    DestinationOrigin["LOCAL"] = "local";
    // Note: Cookies, device and privet are deprecated, but used to filter any
    // legacy entries in the recent destinations, since we can't guarantee all
    // such recent printers have been overridden.
    DestinationOrigin["COOKIES"] = "cookies";
    DestinationOrigin["PRIVET"] = "privet";
    DestinationOrigin["EXTENSION"] = "extension";
    DestinationOrigin["CROS"] = "chrome_os";
})(DestinationOrigin || (DestinationOrigin = {}));
/**
 * Printer types for capabilities and printer list requests.
 * Must match PrinterType in printing/mojom/print.mojom
 */
var PrinterType;
(function (PrinterType) {
    PrinterType[PrinterType["EXTENSION_PRINTER"] = 0] = "EXTENSION_PRINTER";
    PrinterType[PrinterType["PDF_PRINTER"] = 1] = "PDF_PRINTER";
    PrinterType[PrinterType["LOCAL_PRINTER"] = 2] = "LOCAL_PRINTER";
})(PrinterType || (PrinterType = {}));
/**
 * Enumeration of color modes used by Chromium.
 */
var ColorMode;
(function (ColorMode) {
    ColorMode[ColorMode["GRAY"] = 1] = "GRAY";
    ColorMode[ColorMode["COLOR"] = 2] = "COLOR";
})(ColorMode || (ColorMode = {}));
function isPdfPrinter(id) {
    return id === GooglePromotedDestinationId.SAVE_AS_PDF;
}
/**
 * Creates a |RecentDestination| to represent |destination| in the app
 * state.
 */
function makeRecentDestination(destination) {
    return {
        id: destination.id,
        origin: destination.origin,
        capabilities: destination.capabilities,
        displayName: destination.displayName || '',
        extensionId: destination.extensionId || '',
        extensionName: destination.extensionName || '',
        icon: destination.icon || '',
    };
}
/**
 * @return key that maps to a destination with the selected |id| and |origin|.
 */
function createDestinationKey(id, origin) {
    return `${id}/${origin}/`;
}
/**
 * @return A key that maps to a destination with parameters matching
 *     |recentDestination|.
 */
function createRecentDestinationKey(recentDestination) {
    return createDestinationKey(recentDestination.id, recentDestination.origin);
}
/**
 * List of capability types considered color.
 */
const COLOR_TYPES = ['STANDARD_COLOR', 'CUSTOM_COLOR'];
/**
 * List of capability types considered monochrome.
 */
const MONOCHROME_TYPES = ['STANDARD_MONOCHROME', 'CUSTOM_MONOCHROME'];
/**
 * Print destination data object.
 */
class Destination {
    /**
     * ID of the destination.
     */
    id_;
    /**
     * Origin of the destination.
     */
    origin_;
    /**
     * Display name of the destination.
     */
    displayName_;
    /**
     * Print capabilities of the destination.
     */
    capabilities_ = null;
    /**
     * Whether the destination is an enterprise policy controlled printer.
     */
    isEnterprisePrinter_;
    /**
     * Destination location.
     */
    location_ = '';
    /**
     * Printer description.
     */
    description_;
    /**
     * Extension ID for extension managed printers.
     */
    extensionId_;
    /**
     * Extension name for extension managed printers.
     */
    extensionName_;
    type_;
    constructor(id, origin, displayName, params) {
        this.id_ = id;
        this.origin_ = origin;
        this.displayName_ = displayName || '';
        this.isEnterprisePrinter_ = (params && params.isEnterprisePrinter) || false;
        this.description_ = (params && params.description) || '';
        this.extensionId_ = (params && params.extensionId) || '';
        this.extensionName_ = (params && params.extensionName) || '';
        this.location_ = (params && params.location) || '';
        this.type_ = this.computeType_(id, origin);
    }
    computeType_(id, origin) {
        if (isPdfPrinter(id)) {
            return PrinterType.PDF_PRINTER;
        }
        return origin === DestinationOrigin.EXTENSION ?
            PrinterType.EXTENSION_PRINTER :
            PrinterType.LOCAL_PRINTER;
    }
    get type() {
        return this.type_;
    }
    get id() {
        return this.id_;
    }
    get origin() {
        return this.origin_;
    }
    get displayName() {
        return this.displayName_;
    }
    /**
     * @return Whether the destination is an extension managed printer.
     */
    get isExtension() {
        return this.origin_ === DestinationOrigin.EXTENSION;
    }
    /**
     * @return Most relevant string to help user to identify this
     *     destination.
     */
    get hint() {
        return this.location_ || this.extensionName || this.description_;
    }
    /**
     * @return Extension ID associated with the destination. Non-empty
     *     only for extension managed printers.
     */
    get extensionId() {
        return this.extensionId_;
    }
    /**
     * @return Extension name associated with the destination.
     *     Non-empty only for extension managed printers.
     */
    get extensionName() {
        return this.extensionName_;
    }
    /** @return Print capabilities of the destination. */
    get capabilities() {
        return this.capabilities_;
    }
    set capabilities(capabilities) {
        if (capabilities) {
            this.capabilities_ = capabilities;
        }
    }
    /** @return Path to the SVG for the destination's icon. */
    get icon() {
        if (this.id_ === GooglePromotedDestinationId.SAVE_AS_PDF) {
            return 'cr:insert-drive-file';
        }
        if (this.isEnterprisePrinter) {
            return 'print-preview:business';
        }
        return 'print-preview:print';
    }
    /**
     * @return Properties (besides display name) to match search queries against.
     */
    get extraPropertiesToMatch() {
        return [this.location_, this.description_];
    }
    /**
     * Matches a query against the destination.
     * @param query Query to match against the destination.
     * @return Whether the query matches this destination.
     */
    matches(query) {
        return !!this.displayName_.match(query) ||
            !!this.extensionName_.match(query) || !!this.location_.match(query) ||
            !!this.description_.match(query);
    }
    /**
     * Whether the printer is enterprise policy controlled printer.
     */
    get isEnterprisePrinter() {
        return this.isEnterprisePrinter_;
    }
    copiesCapability_() {
        return this.capabilities && this.capabilities.printer &&
            this.capabilities.printer.copies ?
            this.capabilities.printer.copies :
            null;
    }
    colorCapability_() {
        return this.capabilities && this.capabilities.printer &&
            this.capabilities.printer.color ?
            this.capabilities.printer.color :
            null;
    }
    /** @return Whether the printer supports copies. */
    get hasCopiesCapability() {
        const capability = this.copiesCapability_();
        if (!capability) {
            return false;
        }
        return capability.max ? capability.max > 1 : true;
    }
    /**
     * @return Whether the printer supports both black and white and
     *     color printing.
     */
    get hasColorCapability() {
        const capability = this.colorCapability_();
        if (!capability || !capability.option) {
            return false;
        }
        let hasColor = false;
        let hasMonochrome = false;
        capability.option.forEach(option => {
            assert(option.type);
            hasColor = hasColor || COLOR_TYPES.includes(option.type);
            hasMonochrome = hasMonochrome || MONOCHROME_TYPES.includes(option.type);
        });
        return hasColor && hasMonochrome;
    }
    /**
     * @param isColor Whether to use a color printing mode.
     * @return Native color model of the destination.
     */
    getNativeColorModel(isColor) {
        // For printers without capability, native color model is ignored.
        const capability = this.colorCapability_();
        if (!capability || !capability.option) {
            return isColor ? ColorMode.COLOR : ColorMode.GRAY;
        }
        const selected = this.getColor(isColor);
        const mode = parseInt(selected ? selected.vendor_id : '', 10);
        if (isNaN(mode)) {
            return isColor ? ColorMode.COLOR : ColorMode.GRAY;
        }
        return mode;
    }
    /**
     * @return The default color option for the destination.
     */
    get defaultColorOption() {
        const capability = this.colorCapability_();
        if (!capability || !capability.option) {
            return null;
        }
        const defaultOptions = capability.option.filter(option => {
            return option.is_default;
        });
        return defaultOptions.length > 0 ? defaultOptions[0] : null;
    }
    /**
     * @return Color option value of the destination with the given binary color
     * value. Returns null if the destination doesn't support such color value.
     */
    getColor(isColor) {
        const typesToLookFor = isColor ? COLOR_TYPES : MONOCHROME_TYPES;
        const capability = this.colorCapability_();
        if (!capability || !capability.option) {
            return null;
        }
        for (let i = 0; i < typesToLookFor.length; i++) {
            const matchingOptions = capability.option.filter(option => {
                return option.type === typesToLookFor[i];
            });
            if (matchingOptions.length > 0) {
                return matchingOptions[0];
            }
        }
        return null;
    }
    /**
     * @return Media size value of the destination with the given width and height
     * values. Returns undefined if there is no such media size value.
     */
    getMediaSize(width, height) {
        return this.capabilities?.printer.media_size?.option.find(o => {
            return o.width_microns === width && o.height_microns === height;
        });
    }
    /**
     * @return DPI (Dots per Inch) value of the destination with the given
     * horizontal and vertical resolutions. Returns undefined if there is no such
     * DPI value.
     */
    getDpi(horizontal, vertical) {
        return this.capabilities?.printer.dpi?.option.find(o => {
            return o.horizontal_dpi === horizontal && o.vertical_dpi === vertical;
        });
    }
    /**
     * @return Returns true if the current printing destination supports the given
     * duplex value. Returns false in all other cases.
     */
    supportsDuplex(duplex) {
        const availableDuplexOptions = this.capabilities?.printer.duplex?.option;
        if (!availableDuplexOptions) {
            // There are no duplex capabilities reported by the printer.
            return false;
        }
        return availableDuplexOptions.some(o => {
            return o.type === duplex;
        });
    }
    /** @return A unique identifier for this destination. */
    get key() {
        return `${this.id_}/${this.origin_}/`;
    }
}
/**
 * Enumeration of Google-promoted destination IDs.
 * @enum {string}
 */
var GooglePromotedDestinationId;
(function (GooglePromotedDestinationId) {
    GooglePromotedDestinationId["SAVE_AS_PDF"] = "Save as PDF";
})(GooglePromotedDestinationId || (GooglePromotedDestinationId = {}));
/* Unique identifier for the Save as PDF destination */
const PDF_DESTINATION_KEY = `${GooglePromotedDestinationId.SAVE_AS_PDF}/${DestinationOrigin.LOCAL}/`;

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Enumeration of background graphics printing mode restrictions used by
 * Chromium.
 * This has to coincide with |printing::BackgroundGraphicsModeRestriction| as
 * defined in printing/backend/printing_restrictions.h
 */
var BackgroundGraphicsModeRestriction;
(function (BackgroundGraphicsModeRestriction) {
    BackgroundGraphicsModeRestriction[BackgroundGraphicsModeRestriction["UNSET"] = 0] = "UNSET";
    BackgroundGraphicsModeRestriction[BackgroundGraphicsModeRestriction["ENABLED"] = 1] = "ENABLED";
    BackgroundGraphicsModeRestriction[BackgroundGraphicsModeRestriction["DISABLED"] = 2] = "DISABLED";
})(BackgroundGraphicsModeRestriction || (BackgroundGraphicsModeRestriction = {}));
/**
 * Enumeration of color mode restrictions used by Chromium.
 * This has to coincide with |printing::ColorModeRestriction| as defined in
 * printing/backend/printing_restrictions.h
 */
var ColorModeRestriction;
(function (ColorModeRestriction) {
    ColorModeRestriction[ColorModeRestriction["UNSET"] = 0] = "UNSET";
    ColorModeRestriction[ColorModeRestriction["MONOCHROME"] = 1] = "MONOCHROME";
    ColorModeRestriction[ColorModeRestriction["COLOR"] = 2] = "COLOR";
})(ColorModeRestriction || (ColorModeRestriction = {}));
/**
 * Enumeration of duplex mode restrictions used by Chromium.
 * This has to coincide with |printing::DuplexModeRestriction| as defined in
 * printing/backend/printing_restrictions.h
 */
var DuplexModeRestriction;
(function (DuplexModeRestriction) {
    DuplexModeRestriction[DuplexModeRestriction["UNSET"] = 0] = "UNSET";
    DuplexModeRestriction[DuplexModeRestriction["SIMPLEX"] = 1] = "SIMPLEX";
    DuplexModeRestriction[DuplexModeRestriction["LONG_EDGE"] = 2] = "LONG_EDGE";
    DuplexModeRestriction[DuplexModeRestriction["SHORT_EDGE"] = 4] = "SHORT_EDGE";
    DuplexModeRestriction[DuplexModeRestriction["DUPLEX"] = 6] = "DUPLEX";
})(DuplexModeRestriction || (DuplexModeRestriction = {}));
class NativeLayerImpl {
    getInitialSettings() {
        return sendWithPromise('getInitialSettings');
    }
    getPrinters(type) {
        return sendWithPromise('getPrinters', type);
    }
    getPrinterCapabilities(destinationId, type) {
        return sendWithPromise('getPrinterCapabilities', destinationId, type);
    }
    getPreview(printTicket) {
        return sendWithPromise('getPreview', printTicket);
    }
    managePrinters() {
        chrome.send('managePrinters');
    }
    doPrint(printTicket) {
        return sendWithPromise('doPrint', printTicket);
    }
    cancelPendingPrintRequest() {
        chrome.send('cancelPendingPrintRequest');
    }
    saveAppState(appStateStr) {
        chrome.send('saveAppState', [appStateStr]);
    }
    // 
    showSystemDialog() {
        chrome.send('showSystemDialog');
    }
    // 
    dialogClose(isCancel) {
        if (isCancel) {
            chrome.send('closePrintPreviewDialog');
        }
        chrome.send('dialogClose');
    }
    hidePreview() {
        chrome.send('hidePreview');
    }
    recordInHistogram(histogram, bucket, maxBucket) {
        chrome.send('metricsHandler:recordInHistogram', [histogram, bucket, maxBucket]);
    }
    recordBooleanHistogram(histogram, value) {
        chrome.send('metricsHandler:recordBooleanHistogram', [histogram, value]);
    }
    static getInstance() {
        return instance$E || (instance$E = new NativeLayerImpl());
    }
    static setInstance(obj) {
        instance$E = obj;
    }
}
let instance$E = null;

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class DestinationMatch {
    idRegExp_;
    displayNameRegExp_;
    /**
     * A set of key parameters describing a destination used to determine
     * if two destinations are the same.
     * @param idRegExp Match destination's id.
     * @param displayNameRegExp Match destination's displayName.
     */
    constructor(idRegExp, displayNameRegExp) {
        this.idRegExp_ = idRegExp;
        this.displayNameRegExp_ = displayNameRegExp;
    }
    match(destination) {
        if (this.idRegExp_ && !this.idRegExp_.test(destination.id)) {
            return false;
        }
        if (this.displayNameRegExp_ &&
            !this.displayNameRegExp_.test(destination.displayName)) {
            return false;
        }
        if (destination.type === PrinterType.PDF_PRINTER) {
            return false;
        }
        return true;
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* @fileoverview Utilities for determining the current platform. */
/** Whether we are using a Mac or not. */
const isMac = /Mac/.test(navigator.platform);
/** Whether this is on the Windows platform or not. */
const isWindows = /Win/.test(navigator.platform);

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @param type The type of printer to parse.
 * @param printer Information about the printer.
 *       Type expected depends on |type|:
 *       For LOCAL_PRINTER => LocalDestinationInfo
 *       For EXTENSION_PRINTER => ExtensionDestinationInfo
 * @return Destination, or null if an invalid value is provided for |type|.
 */
function parseDestination(type, printer) {
    if (type === PrinterType.LOCAL_PRINTER || type === PrinterType.PDF_PRINTER) {
        return parseLocalDestination(printer);
    }
    if (type === PrinterType.EXTENSION_PRINTER) {
        return parseExtensionDestination(printer);
    }
    assertNotReached('Unknown printer type ' + type);
}
/**
 * @param destinationInfo Information describing a local print destination.
 * @return Parsed local print destination.
 */
function parseLocalDestination(destinationInfo) {
    const options = {
        description: destinationInfo.printerDescription,
        isEnterprisePrinter: destinationInfo.cupsEnterprisePrinter,
        location: '',
    };
    const locationOptions = new Set(['location', 'printer-location']);
    if (destinationInfo.printerOptions) {
        // The only printer option currently used by Print Preview's UI is location.
        for (const printerOption of Object.keys(destinationInfo.printerOptions)) {
            if (locationOptions.has(printerOption)) {
                options.location = destinationInfo.printerOptions[printerOption] || '';
            }
        }
    }
    return new Destination(destinationInfo.deviceName, DestinationOrigin.LOCAL, destinationInfo.printerName, options);
}
/**
 * Parses an extension destination from an extension supplied printer
 * description.
 */
function parseExtensionDestination(destinationInfo) {
    return new Destination(destinationInfo.id, DestinationOrigin.EXTENSION, destinationInfo.name, {
        description: destinationInfo.description || '',
        extensionId: destinationInfo.extensionId,
        extensionName: destinationInfo.extensionName || '',
    });
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Printer search statuses used by the destination store.
 */
var DestinationStorePrinterSearchStatus;
(function (DestinationStorePrinterSearchStatus) {
    DestinationStorePrinterSearchStatus["START"] = "start";
    DestinationStorePrinterSearchStatus["SEARCHING"] = "searching";
    DestinationStorePrinterSearchStatus["DONE"] = "done";
})(DestinationStorePrinterSearchStatus || (DestinationStorePrinterSearchStatus = {}));
/**
 * Enumeration of possible destination errors.
 */
var DestinationErrorType;
(function (DestinationErrorType) {
    DestinationErrorType[DestinationErrorType["INVALID"] = 0] = "INVALID";
    DestinationErrorType[DestinationErrorType["NO_DESTINATIONS"] = 1] = "NO_DESTINATIONS";
})(DestinationErrorType || (DestinationErrorType = {}));
/**
 * Localizes printer capabilities.
 * @param capabilities Printer capabilities to localize.
 * @return Localized capabilities.
 */
function localizeCapabilities(capabilities) {
    if (!capabilities.printer) {
        return capabilities;
    }
    const mediaSize = capabilities.printer.media_size;
    if (!mediaSize) {
        return capabilities;
    }
    for (let i = 0, media; (media = mediaSize.option[i]); i++) {
        // No need to patch capabilities with localized names provided.
        if (!media.custom_display_name_localized) {
            media.custom_display_name = media.custom_display_name ||
                MEDIA_DISPLAY_NAMES_[media.name] || media.name;
        }
    }
    return capabilities;
}
/**
 * Compare two media sizes by their names.
 * @return 1 if a > b, -1 if a < b, or 0 if a === b.
 */
function compareMediaNames(a, b) {
    const nameA = a.custom_display_name_localized || a.custom_display_name || '';
    const nameB = b.custom_display_name_localized || b.custom_display_name || '';
    return nameA === nameB ? 0 : (nameA > nameB ? 1 : -1);
}
/**
 * Sort printer media sizes.
 */
function sortMediaSizes(capabilities) {
    if (!capabilities.printer) {
        return capabilities;
    }
    const mediaSize = capabilities.printer.media_size;
    if (!mediaSize) {
        return capabilities;
    }
    // For the standard sizes, separate into categories, as seen in the Cloud
    // Print CDD guide:
    // - North American
    // - Chinese
    // - ISO
    // - Japanese
    // - Other metric
    // Otherwise, assume they are custom sizes.
    const categoryStandardNA = [];
    const categoryStandardCN = [];
    const categoryStandardISO = [];
    const categoryStandardJP = [];
    const categoryStandardMisc = [];
    const categoryCustom = [];
    for (let i = 0, media; (media = mediaSize.option[i]); i++) {
        const name = media.name || 'CUSTOM';
        let category;
        if (name.startsWith('NA_')) {
            category = categoryStandardNA;
        }
        else if (name.startsWith('PRC_') || name.startsWith('ROC_') ||
            name === 'OM_DAI_PA_KAI' || name === 'OM_JUURO_KU_KAI' ||
            name === 'OM_PA_KAI') {
            category = categoryStandardCN;
        }
        else if (name.startsWith('ISO_')) {
            category = categoryStandardISO;
        }
        else if (name.startsWith('JIS_') || name.startsWith('JPN_')) {
            category = categoryStandardJP;
        }
        else if (name.startsWith('OM_')) {
            category = categoryStandardMisc;
        }
        else {
            assert(name === 'CUSTOM', 'Unknown media size. Assuming custom');
            category = categoryCustom;
        }
        category.push(media);
    }
    // For each category, sort by name.
    categoryStandardNA.sort(compareMediaNames);
    categoryStandardCN.sort(compareMediaNames);
    categoryStandardISO.sort(compareMediaNames);
    categoryStandardJP.sort(compareMediaNames);
    categoryStandardMisc.sort(compareMediaNames);
    categoryCustom.sort(compareMediaNames);
    // Then put it all back together.
    mediaSize.option = categoryStandardNA;
    mediaSize.option.push(...categoryStandardCN, ...categoryStandardISO, ...categoryStandardJP, ...categoryStandardMisc, ...categoryCustom);
    return capabilities;
}
/**
 * Event types dispatched by the destination store.
 */
var DestinationStoreEventType;
(function (DestinationStoreEventType) {
    DestinationStoreEventType["DESTINATION_SEARCH_DONE"] = "DestinationStore.DESTINATION_SEARCH_DONE";
    DestinationStoreEventType["DESTINATION_SELECT"] = "DestinationStore.DESTINATION_SELECT";
    DestinationStoreEventType["DESTINATIONS_INSERTED"] = "DestinationStore.DESTINATIONS_INSERTED";
    DestinationStoreEventType["ERROR"] = "DestinationStore.ERROR";
    DestinationStoreEventType["SELECTED_DESTINATION_CAPABILITIES_READY"] = "DestinationStore.SELECTED_DESTINATION_CAPABILITIES_READY";
})(DestinationStoreEventType || (DestinationStoreEventType = {}));
class DestinationStore extends EventTarget {
    /**
     * Whether the destination store will auto select the destination that
     * matches this set of parameters.
     */
    autoSelectMatchingDestination_ = null;
    /**
     * Cache used for constant lookup of destinations by key.
     */
    destinationMap_ = new Map();
    /**
     * Internal backing store for the data store.
     */
    destinations_ = [];
    /**
     * Whether a search for destinations is in progress for each type of
     * printer.
     */
    destinationSearchStatus_;
    initialDestinationSelected_ = false;
    /**
     * Used to fetch local print destinations.
     */
    nativeLayer_ = NativeLayerImpl.getInstance();
    /**
     * Whether PDF printer is enabled. It's disabled, for example, in App
     * Kiosk mode or when PDF printing is disallowed by policy.
     */
    pdfPrinterEnabled_ = false;
    recentDestinationKeys_ = [];
    /**
     * Currently selected destination.
     */
    selectedDestination_ = null;
    /**
     * Key of the system default destination.
     */
    systemDefaultDestinationKey_ = '';
    /**
     * Event tracker used to track event listeners of the destination store.
     */
    tracker_ = new EventTracker();
    typesToSearch_ = new Set();
    /**
     * Whether to default to the system default printer instead of the most
     * recent destination.
     */
    useSystemDefaultAsDefault_;
    /**
     * A data store that stores destinations and dispatches events when the
     * data store changes.
     * @param addListenerCallback Function to call to add Web UI listeners in
     *     DestinationStore constructor.
     */
    constructor(addListenerCallback) {
        super();
        this.destinationSearchStatus_ = new Map([
            [
                PrinterType.EXTENSION_PRINTER,
                DestinationStorePrinterSearchStatus.START,
            ],
            [PrinterType.LOCAL_PRINTER, DestinationStorePrinterSearchStatus.START],
        ]);
        this.useSystemDefaultAsDefault_ =
            loadTimeData.getBoolean('useSystemDefaultPrinter');
        addListenerCallback('printers-added', (type, printers) => this.onPrintersAdded_(type, printers));
    }
    /**
     * @return List of destinations
     */
    destinations() {
        return this.destinations_.slice();
    }
    /**
     * @return Whether a search for print destinations is in progress.
     */
    get isPrintDestinationSearchInProgress() {
        return Array.from(this.destinationSearchStatus_.values())
            .some(el => el === DestinationStorePrinterSearchStatus.SEARCHING);
    }
    /**
     * @return The currently selected destination or null if none is selected.
     */
    get selectedDestination() {
        return this.selectedDestination_;
    }
    getPrinterTypeForRecentDestination_(destination) {
        if (isPdfPrinter(destination.id)) {
            return PrinterType.PDF_PRINTER;
        }
        if (destination.origin === DestinationOrigin.LOCAL ||
            destination.origin === DestinationOrigin.CROS) {
            return PrinterType.LOCAL_PRINTER;
        }
        assert(destination.origin === DestinationOrigin.EXTENSION);
        return PrinterType.EXTENSION_PRINTER;
    }
    /**
     * Initializes the destination store. Sets the initially selected
     * destination. If any inserted destinations match this ID, that destination
     * will be automatically selected.
     * @param pdfPrinterDisabled Whether the PDF print destination is
     *     disabled in print preview.
     * @param saveToDriveDisabled Whether the 'Save to Google Drive' destination
     *     is disabled in print preview. Only used on Chrome OS.
     * @param systemDefaultDestinationId ID of the system default
     *     destination.
     * @param serializedDefaultDestinationSelectionRulesStr Serialized
     *     default destination selection rules.
     * @param recentDestinations The recent print destinations.
     */
    init(pdfPrinterDisabled, systemDefaultDestinationId, serializedDefaultDestinationSelectionRulesStr, recentDestinations) {
        if (systemDefaultDestinationId) {
            const systemDefaultVirtual = isPdfPrinter(systemDefaultDestinationId);
            const systemDefaultType = systemDefaultVirtual ?
                PrinterType.PDF_PRINTER :
                PrinterType.LOCAL_PRINTER;
            const systemDefaultOrigin = DestinationOrigin.LOCAL;
            this.systemDefaultDestinationKey_ =
                createDestinationKey(systemDefaultDestinationId, systemDefaultOrigin);
            this.typesToSearch_.add(systemDefaultType);
        }
        this.recentDestinationKeys_ = recentDestinations.map(destination => createRecentDestinationKey(destination));
        for (const recent of recentDestinations) {
            this.typesToSearch_.add(this.getPrinterTypeForRecentDestination_(recent));
        }
        this.autoSelectMatchingDestination_ = this.convertToDestinationMatch_(serializedDefaultDestinationSelectionRulesStr);
        if (this.autoSelectMatchingDestination_) {
            this.typesToSearch_.add(PrinterType.EXTENSION_PRINTER);
            this.typesToSearch_.add(PrinterType.LOCAL_PRINTER);
        }
        this.pdfPrinterEnabled_ = !pdfPrinterDisabled;
        this.createLocalPdfPrintDestination_();
        // Nothing recent, no system default ==> try to get a fallback printer as
        // destinationsInserted_ may never be called.
        if (this.typesToSearch_.size === 0) {
            this.tryToSelectInitialDestination_();
            return;
        }
        for (const printerType of this.typesToSearch_) {
            this.startLoadDestinations_(printerType);
        }
        // Start a 10s timeout so that we never hang forever.
        window.setTimeout(() => {
            this.tryToSelectInitialDestination_(true);
        }, 10000);
    }
    /**
     * @param timeoutExpired Whether the select timeout is expired.
     *     Defaults to false.
     */
    tryToSelectInitialDestination_(timeoutExpired = false) {
        if (this.initialDestinationSelected_) {
            return;
        }
        const success = this.selectInitialDestination_(timeoutExpired);
        if (!success && !this.isPrintDestinationSearchInProgress &&
            this.typesToSearch_.size === 0) {
            // No destinations
            this.dispatchEvent(new CustomEvent(DestinationStoreEventType.ERROR, { detail: DestinationErrorType.NO_DESTINATIONS }));
        }
        this.initialDestinationSelected_ = success;
    }
    selectDefaultDestination() {
        if (this.tryToSelectDestinationByKey_(this.systemDefaultDestinationKey_)) {
            return;
        }
        this.selectFinalFallbackDestination_();
    }
    /**
     * Called when destinations are added to the store when the initial
     * destination has not yet been set. Selects the initial destination based on
     * relevant policies, recent printers, and system default.
     * @param timeoutExpired Whether the initial timeout has expired.
     * @return Whether an initial destination was successfully selected.
     */
    selectInitialDestination_(timeoutExpired) {
        const searchInProgress = this.typesToSearch_.size !== 0 && !timeoutExpired;
        // System default printer policy takes priority.
        if (this.useSystemDefaultAsDefault_) {
            if (this.tryToSelectDestinationByKey_(this.systemDefaultDestinationKey_)) {
                return true;
            }
            // If search is still in progress, wait. The printer might come in a later
            // batch of destinations.
            if (searchInProgress) {
                return false;
            }
        }
        // Check recent destinations. If all the printers have loaded, check for all
        // of them. Otherwise, just look at the most recent.
        for (const key of this.recentDestinationKeys_) {
            if (this.tryToSelectDestinationByKey_(key)) {
                return true;
            }
            else if (searchInProgress) {
                return false;
            }
        }
        // Try the default destination rules, if they exist.
        if (this.autoSelectMatchingDestination_) {
            for (const destination of this.destinations_) {
                if (this.autoSelectMatchingDestination_.match(destination)) {
                    this.selectDestination(destination);
                    return true;
                }
            }
            // If search is still in progress, wait for other possible matching
            // printers.
            if (searchInProgress) {
                return false;
            }
        }
        // If there either aren't any recent printers or rules, or destinations are
        // all loaded and none could be found, try the system default.
        if (this.tryToSelectDestinationByKey_(this.systemDefaultDestinationKey_)) {
            return true;
        }
        if (searchInProgress) {
            return false;
        }
        // Everything's loaded, but we couldn't find either the system default, a
        // match for the selection rules, or a recent printer. Fallback to Save
        // as PDF, or the first printer to load (if in kiosk mode).
        if (this.selectFinalFallbackDestination_()) {
            return true;
        }
        return false;
    }
    /**
     * @param key The destination key to try to select.
     * @return Whether the destination was found and selected.
     */
    tryToSelectDestinationByKey_(key) {
        const candidate = this.destinationMap_.get(key);
        if (candidate) {
            this.selectDestination(candidate);
            return true;
        }
        return false;
    }
    /** Removes all events being tracked from the tracker. */
    resetTracker() {
        this.tracker_.removeAll();
    }
    /**
     * @return Creates rules matching previously selected destination.
     */
    convertToDestinationMatch_(serializedDefaultDestinationSelectionRulesStr) {
        let matchRules = null;
        try {
            if (serializedDefaultDestinationSelectionRulesStr) {
                matchRules = JSON.parse(serializedDefaultDestinationSelectionRulesStr);
            }
        }
        catch (e) {
            console.warn('Failed to parse defaultDestinationSelectionRules: ' + e);
        }
        if (!matchRules) {
            return null;
        }
        const isLocal = !matchRules.kind || matchRules.kind === 'local';
        if (!isLocal) {
            console.warn('Unsupported type: "' + matchRules.kind + '"');
            return null;
        }
        let idRegExp = null;
        try {
            if (matchRules.idPattern) {
                idRegExp = new RegExp(matchRules.idPattern || '.*');
            }
        }
        catch (e) {
            console.warn('Failed to parse regexp for "id": ' + e);
        }
        let displayNameRegExp = null;
        try {
            if (matchRules.namePattern) {
                displayNameRegExp = new RegExp(matchRules.namePattern || '.*');
            }
        }
        catch (e) {
            console.warn('Failed to parse regexp for "name": ' + e);
        }
        return new DestinationMatch(idRegExp, displayNameRegExp);
    }
    /**
     * This function is only invoked when the user selects a new destination via
     * the UI. Programmatic selection of a destination should not use this
     * function.
     * @param Key identifying the destination to select
     */
    selectDestinationByKey(key) {
        const success = this.tryToSelectDestinationByKey_(key);
        assert(success);
    }
    /**
     * @param destination Destination to select.
     * @param refreshDestination Set to true to allow the currently selected
     *          destination to be re-selected.
     */
    selectDestination(destination) {
        if (destination === this.selectedDestination_) {
            return;
        }
        if (destination === null) {
            this.selectedDestination_ = null;
            this.dispatchEvent(new CustomEvent(DestinationStoreEventType.DESTINATION_SELECT));
            return;
        }
        // Update and persist selected destination.
        this.selectedDestination_ = destination;
        // Notify about selected destination change.
        this.dispatchEvent(new CustomEvent(DestinationStoreEventType.DESTINATION_SELECT));
        // Request destination capabilities from backend, since they are not
        // known yet.
        if (destination.capabilities === null) {
            this.nativeLayer_.getPrinterCapabilities(destination.id, destination.type)
                .then((caps) => this.onCapabilitiesSet_(destination.origin, destination.id, caps), () => this.onGetCapabilitiesFail_(destination.origin, destination.id));
        }
        else {
            this.sendSelectedDestinationUpdateEvent_();
        }
    }
    /**
     * Selects the Save as PDF fallback if it is available. If not, selects the
     * first destination if it exists.
     * @return Whether a final destination could be found.
     */
    selectFinalFallbackDestination_() {
        // Save as PDF should always exist if it is enabled.
        if (this.pdfPrinterEnabled_) {
            const destination = this.destinationMap_.get(PDF_DESTINATION_KEY);
            assert(destination);
            this.selectDestination(destination);
            return true;
        }
        // Try selecting the first destination if there is at least one
        // destination already loaded.
        if (this.destinations_.length > 0) {
            this.selectDestination(this.destinations_[0]);
            return true;
        }
        // Trigger a load of all destination types, to try to select the first one.
        this.startLoadAllDestinations();
        return false;
    }
    /**
     * Initiates loading of destinations.
     * @param type The type of destinations to load.
     */
    startLoadDestinations_(type) {
        if (this.destinationSearchStatus_.get(type) ===
            DestinationStorePrinterSearchStatus.DONE) {
            return;
        }
        this.destinationSearchStatus_.set(type, DestinationStorePrinterSearchStatus.SEARCHING);
        this.nativeLayer_.getPrinters(type).then(() => this.onDestinationSearchDone_(type));
    }
    /** Initiates loading of all known destination types. */
    startLoadAllDestinations() {
        // Printer types that need to be retrieved from the handler.
        const types = [
            PrinterType.EXTENSION_PRINTER,
            PrinterType.LOCAL_PRINTER,
        ];
        for (const printerType of types) {
            this.startLoadDestinations_(printerType);
        }
    }
    /**
     * @return The destination matching the key, if it exists.
     */
    getDestinationByKey(key) {
        return this.destinationMap_.get(key);
    }
    /**
     * Inserts {@code destination} to the data store and dispatches a
     * DESTINATIONS_INSERTED event.
     */
    insertDestination_(destination) {
        if (this.insertIntoStore_(destination)) {
            this.destinationsInserted_();
        }
    }
    /**
     * Inserts multiple {@code destinations} to the data store and dispatches
     * single DESTINATIONS_INSERTED event.
     */
    insertDestinations_(destinations) {
        let inserted = false;
        destinations.forEach(destination => {
            if (destination) {
                inserted = this.insertIntoStore_(destination) || inserted;
            }
        });
        if (inserted) {
            this.destinationsInserted_();
        }
    }
    /**
     * Dispatches DESTINATIONS_INSERTED event. In auto select mode, tries to
     * update selected destination to match
     * {@code autoSelectMatchingDestination_}.
     */
    destinationsInserted_() {
        this.dispatchEvent(new CustomEvent(DestinationStoreEventType.DESTINATIONS_INSERTED));
        this.tryToSelectInitialDestination_();
    }
    /**
     * Sends SELECTED_DESTINATION_CAPABILITIES_READY event if the destination
     * is supported, or ERROR otherwise of with error type UNSUPPORTED.
     */
    sendSelectedDestinationUpdateEvent_() {
        this.dispatchEvent(new CustomEvent(DestinationStoreEventType.SELECTED_DESTINATION_CAPABILITIES_READY));
    }
    /**
     * Updates an existing print destination with capabilities and display name
     * information. If the destination doesn't already exist, it will be added.
     */
    updateDestination_(destination) {
        assert(destination.constructor !== Array, 'Single printer expected');
        assert(destination.capabilities);
        destination.capabilities = localizeCapabilities(destination.capabilities);
        if (destination.type !== PrinterType.LOCAL_PRINTER) {
            destination.capabilities = sortMediaSizes(destination.capabilities);
        }
        const existingDestination = this.destinationMap_.get(destination.key);
        if (existingDestination !== undefined) {
            existingDestination.capabilities = destination.capabilities;
        }
        else {
            this.insertDestination_(destination);
        }
        if (this.selectedDestination_ &&
            (existingDestination === this.selectedDestination_ ||
                destination === this.selectedDestination_)) {
            this.sendSelectedDestinationUpdateEvent_();
        }
    }
    /**
     * Inserts a destination into the store without dispatching any events.
     * @return Whether the inserted destination was not already in the store.
     */
    insertIntoStore_(destination) {
        const key = destination.key;
        const existingDestination = this.destinationMap_.get(key);
        if (existingDestination === undefined) {
            this.destinations_.push(destination);
            this.destinationMap_.set(key, destination);
            return true;
        }
        return false;
    }
    /**
     * Creates a local PDF print destination.
     */
    createLocalPdfPrintDestination_() {
        if (this.pdfPrinterEnabled_) {
            this.insertDestination_(new Destination(GooglePromotedDestinationId.SAVE_AS_PDF, DestinationOrigin.LOCAL, loadTimeData.getString('printToPDF')));
        }
        if (this.typesToSearch_.has(PrinterType.PDF_PRINTER)) {
            this.typesToSearch_.delete(PrinterType.PDF_PRINTER);
        }
    }
    /**
     * Called when destination search is complete for some type of printer.
     * @param type The type of printers that are done being retrieved.
     */
    onDestinationSearchDone_(type) {
        this.destinationSearchStatus_.set(type, DestinationStorePrinterSearchStatus.DONE);
        this.dispatchEvent(new CustomEvent(DestinationStoreEventType.DESTINATION_SEARCH_DONE));
        if (this.typesToSearch_.has(type)) {
            this.typesToSearch_.delete(type);
            this.tryToSelectInitialDestination_();
        }
        else if (this.typesToSearch_.size === 0) {
            this.tryToSelectInitialDestination_();
        }
    }
    /**
     * Called when the native layer retrieves the capabilities for the selected
     * local destination. Updates the destination with new capabilities if the
     * destination already exists, otherwise it creates a new destination and
     * then updates its capabilities.
     * @param origin The origin of the print destination.
     * @param id The id of the print destination.
     * @param settingsInfo Contains the capabilities of the print destination,
     *     and information about the destination except in the case of extension
     *     printers.
     */
    onCapabilitiesSet_(origin, id, settingsInfo) {
        let dest = null;
        const key = createDestinationKey(id, origin);
        dest = this.destinationMap_.get(key);
        if (!dest) {
            // Ignore unrecognized extension printers
            if (!settingsInfo.printer) {
                assert(origin === DestinationOrigin.EXTENSION);
                return;
            }
            assert(settingsInfo.printer);
            // PDF, CROS, and LOCAL printers all get parsed the same way.
            const typeToParse = origin === DestinationOrigin.EXTENSION ?
                PrinterType.EXTENSION_PRINTER :
                PrinterType.LOCAL_PRINTER;
            dest = parseDestination(typeToParse, settingsInfo.printer);
        }
        if (dest) {
            if (dest.type !== PrinterType.EXTENSION_PRINTER && dest.capabilities) {
                // If capabilities are already set for this destination ignore new
                // results. This prevents custom margins from being cleared as long
                // as the user does not change to a new non-recent destination.
                return;
            }
            dest.capabilities = settingsInfo.capabilities;
            this.updateDestination_(dest);
        }
    }
    /**
     * Called when a request to get a local destination's print capabilities
     * fails. If the destination is the initial destination, auto-select another
     * destination instead.
     * @param _origin The origin type of the failed destination.
     * @param destinationId The destination ID that failed.
     */
    onGetCapabilitiesFail_(_origin, destinationId) {
        console.warn('Failed to get print capabilities for printer ' + destinationId);
        if (this.selectedDestination_ &&
            this.selectedDestination_.id === destinationId) {
            this.dispatchEvent(new CustomEvent(DestinationStoreEventType.ERROR, { detail: DestinationErrorType.INVALID }));
        }
    }
    /**
     * Called when a printer or printers are detected after sending getPrinters
     * from the native layer.
     * @param type The type of printer(s) added.
     * @param printers Information about the printers that have been retrieved.
     */
    onPrintersAdded_(type, printers) {
        this.insertDestinations_(printers.map((printer) => parseDestination(type, printer)));
    }
}
/**
 * Human readable names for media sizes in the cloud print CDD.
 * https://developers.google.com/cloud-print/docs/cdd
 */
const MEDIA_DISPLAY_NAMES_ = {
    'ISO_2A0': '2A0',
    'ISO_A0': 'A0',
    'ISO_A0X3': 'A0x3',
    'ISO_A1': 'A1',
    'ISO_A10': 'A10',
    'ISO_A1X3': 'A1x3',
    'ISO_A1X4': 'A1x4',
    'ISO_A2': 'A2',
    'ISO_A2X3': 'A2x3',
    'ISO_A2X4': 'A2x4',
    'ISO_A2X5': 'A2x5',
    'ISO_A3': 'A3',
    'ISO_A3X3': 'A3x3',
    'ISO_A3X4': 'A3x4',
    'ISO_A3X5': 'A3x5',
    'ISO_A3X6': 'A3x6',
    'ISO_A3X7': 'A3x7',
    'ISO_A3_EXTRA': 'A3 Extra',
    'ISO_A4': 'A4',
    'ISO_A4X3': 'A4x3',
    'ISO_A4X4': 'A4x4',
    'ISO_A4X5': 'A4x5',
    'ISO_A4X6': 'A4x6',
    'ISO_A4X7': 'A4x7',
    'ISO_A4X8': 'A4x8',
    'ISO_A4X9': 'A4x9',
    'ISO_A4_EXTRA': 'A4 Extra',
    'ISO_A4_TAB': 'A4 Tab',
    'ISO_A5': 'A5',
    'ISO_A5_EXTRA': 'A5 Extra',
    'ISO_A6': 'A6',
    'ISO_A7': 'A7',
    'ISO_A8': 'A8',
    'ISO_A9': 'A9',
    'ISO_B0': 'B0',
    'ISO_B1': 'B1',
    'ISO_B10': 'B10',
    'ISO_B2': 'B2',
    'ISO_B3': 'B3',
    'ISO_B4': 'B4',
    'ISO_B5': 'B5',
    'ISO_B5_EXTRA': 'B5 Extra',
    'ISO_B6': 'B6',
    'ISO_B6C4': 'B6C4',
    'ISO_B7': 'B7',
    'ISO_B8': 'B8',
    'ISO_B9': 'B9',
    'ISO_C0': 'C0',
    'ISO_C1': 'C1',
    'ISO_C10': 'C10',
    'ISO_C2': 'C2',
    'ISO_C3': 'C3',
    'ISO_C4': 'C4',
    'ISO_C5': 'C5',
    'ISO_C6': 'C6',
    'ISO_C6C5': 'C6C5',
    'ISO_C7': 'C7',
    'ISO_C7C6': 'C7C6',
    'ISO_C8': 'C8',
    'ISO_C9': 'C9',
    'ISO_DL': 'Envelope DL',
    'ISO_RA0': 'RA0',
    'ISO_RA1': 'RA1',
    'ISO_RA2': 'RA2',
    'ISO_SRA0': 'SRA0',
    'ISO_SRA1': 'SRA1',
    'ISO_SRA2': 'SRA2',
    'JIS_B0': 'B0 (JIS)',
    'JIS_B1': 'B1 (JIS)',
    'JIS_B10': 'B10 (JIS)',
    'JIS_B2': 'B2 (JIS)',
    'JIS_B3': 'B3 (JIS)',
    'JIS_B4': 'B4 (JIS)',
    'JIS_B5': 'B5 (JIS)',
    'JIS_B6': 'B6 (JIS)',
    'JIS_B7': 'B7 (JIS)',
    'JIS_B8': 'B8 (JIS)',
    'JIS_B9': 'B9 (JIS)',
    'JIS_EXEC': 'Executive (JIS)',
    'JPN_CHOU2': 'Choukei 2',
    'JPN_CHOU3': 'Choukei 3',
    'JPN_CHOU4': 'Choukei 4',
    'JPN_HAGAKI': 'Hagaki',
    'JPN_KAHU': 'Kahu Envelope',
    'JPN_KAKU2': 'Kaku 2',
    'JPN_OUFUKU': 'Oufuku Hagaki',
    'JPN_YOU4': 'You 4',
    'NA_10X11': '10x11',
    'NA_10X13': '10x13',
    'NA_10X14': '10x14',
    'NA_10X15': '10x15',
    'NA_11X12': '11x12',
    'NA_11X15': '11x15',
    'NA_12X19': '12x19',
    'NA_5X7': '5x7',
    'NA_6X9': '6x9',
    'NA_7X9': '7x9',
    'NA_9X11': '9x11',
    'NA_A2': 'A2',
    'NA_ARCH_A': 'Arch A',
    'NA_ARCH_B': 'Arch B',
    'NA_ARCH_C': 'Arch C',
    'NA_ARCH_D': 'Arch D',
    'NA_ARCH_E': 'Arch E',
    'NA_ASME_F': 'ASME F',
    'NA_B_PLUS': 'B-plus',
    'NA_C': 'C',
    'NA_C5': 'C5',
    'NA_D': 'D',
    'NA_E': 'E',
    'NA_EDP': 'EDP',
    'NA_EUR_EDP': 'European EDP',
    'NA_EXECUTIVE': 'Executive',
    'NA_F': 'F',
    'NA_FANFOLD_EUR': 'FanFold European',
    'NA_FANFOLD_US': 'FanFold US',
    'NA_FOOLSCAP': 'FanFold German Legal',
    'NA_GOVT_LEGAL': '8x13',
    'NA_GOVT_LETTER': '8x10',
    'NA_INDEX_3X5': 'Index 3x5',
    'NA_INDEX_4X6': 'Index 4x6',
    'NA_INDEX_4X6_EXT': 'Index 4x6 ext',
    'NA_INDEX_5X8': '5x8',
    'NA_INVOICE': 'Invoice',
    'NA_LEDGER': 'Tabloid', // Ledger in portrait is called Tabloid.
    'NA_LEGAL': 'Legal',
    'NA_LEGAL_EXTRA': 'Legal extra',
    'NA_LETTER': 'Letter',
    'NA_LETTER_EXTRA': 'Letter extra',
    'NA_LETTER_PLUS': 'Letter plus',
    'NA_MONARCH': 'Monarch',
    'NA_NUMBER_10': 'Envelope #10',
    'NA_NUMBER_11': 'Envelope #11',
    'NA_NUMBER_12': 'Envelope #12',
    'NA_NUMBER_14': 'Envelope #14',
    'NA_NUMBER_9': 'Envelope #9',
    'NA_PERSONAL': 'Personal',
    'NA_QUARTO': 'Quarto',
    'NA_SUPER_A': 'Super A',
    'NA_SUPER_B': 'Super B',
    'NA_WIDE_FORMAT': 'Wide format',
    'OM_DAI_PA_KAI': 'Dai-pa-kai',
    'OM_FOLIO': 'Folio',
    'OM_FOLIO_SP': 'Folio SP',
    'OM_INVITE': 'Invite Envelope',
    'OM_ITALIAN': 'Italian Envelope',
    'OM_JUURO_KU_KAI': 'Juuro-ku-kai',
    'OM_LARGE_PHOTO': 'Large photo',
    'OM_OFICIO': 'Oficio',
    'OM_PA_KAI': 'Pa-kai',
    'OM_POSTFIX': 'Postfix Envelope',
    'OM_SMALL_PHOTO': 'Small photo',
    'PRC_1': 'prc1 Envelope',
    'PRC_10': 'prc10 Envelope',
    'PRC_16K': 'prc 16k',
    'PRC_2': 'prc2 Envelope',
    'PRC_3': 'prc3 Envelope',
    'PRC_32K': 'prc 32k',
    'PRC_4': 'prc4 Envelope',
    'PRC_5': 'prc5 Envelope',
    'PRC_6': 'prc6 Envelope',
    'PRC_7': 'prc7 Envelope',
    'PRC_8': 'prc8 Envelope',
    'ROC_16K': 'ROC 16K',
    'ROC_8K': 'ROC 8k',
};

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const WebUiListenerMixinLit = (superClass) => {
    class WebUiListenerMixinLit extends superClass {
        /**
         * Holds WebUI listeners that need to be removed when this element is
         * destroyed.
         */
        webUiListeners_ = [];
        /**
         * Adds a WebUI listener and registers it for automatic removal when
         * this element is detached. Note: Do not use this method if you intend
         * to remove this listener manually (use addWebUiListener directly
         * instead).
         *
         * @param eventName The event to listen to.
         * @param callback The callback run when the event is fired.
         */
        addWebUiListener(eventName, callback) {
            this.webUiListeners_.push(addWebUiListener(eventName, callback));
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            while (this.webUiListeners_.length > 0) {
                removeWebUiListener(this.webUiListeners_.pop());
            }
        }
    }
    return WebUiListenerMixinLit;
};

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class Coordinate2d {
    x_;
    y_;
    /**
     * Immutable two dimensional point in space. The units of the dimensions are
     * undefined.
     * @param x X-dimension of the point.
     * @param y Y-dimension of the point.
     */
    constructor(x, y) {
        this.x_ = x;
        this.y_ = y;
    }
    get x() {
        return this.x_;
    }
    get y() {
        return this.y_;
    }
    equals(other) {
        return other !== null && this.x_ === other.x_ && this.y_ === other.y_;
    }
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Enumeration of the orientations of margins.
 */
var CustomMarginsOrientation;
(function (CustomMarginsOrientation) {
    CustomMarginsOrientation["TOP"] = "top";
    CustomMarginsOrientation["RIGHT"] = "right";
    CustomMarginsOrientation["BOTTOM"] = "bottom";
    CustomMarginsOrientation["LEFT"] = "left";
})(CustomMarginsOrientation || (CustomMarginsOrientation = {}));
/**
 * Must be kept in sync with the C++ MarginType enum in
 * printing/print_job_constants.h.
 */
var MarginsType;
(function (MarginsType) {
    MarginsType[MarginsType["DEFAULT"] = 0] = "DEFAULT";
    MarginsType[MarginsType["NO_MARGINS"] = 1] = "NO_MARGINS";
    MarginsType[MarginsType["MINIMUM"] = 2] = "MINIMUM";
    MarginsType[MarginsType["CUSTOM"] = 3] = "CUSTOM";
})(MarginsType || (MarginsType = {}));
class Margins {
    /**
     * Backing store for the margin values in points. The numbers are stored as
     * integer values, because that is what the C++ `printing::PageMargins` class
     * expects.
     */
    value_ = { top: 0, bottom: 0, left: 0, right: 0 };
    /**
     * Creates a Margins object that holds four margin values in points.
     */
    constructor(top, right, bottom, left) {
        this.value_ = {
            top: Math.round(top),
            right: Math.round(right),
            bottom: Math.round(bottom),
            left: Math.round(left),
        };
    }
    /**
     * @param orientation Specifies the margin value to get.
     * @return Value of the margin of the given orientation.
     */
    get(orientation) {
        return this.value_[orientation];
    }
    /**
     * @param orientation Specifies the margin to set.
     * @param value Updated value of the margin in points to modify.
     * @return A new copy of |this| with the modification made to the specified
     *     margin.
     */
    set(orientation, value) {
        const newValue = this.clone_();
        newValue[orientation] = value;
        return new Margins(newValue[CustomMarginsOrientation.TOP], newValue[CustomMarginsOrientation.RIGHT], newValue[CustomMarginsOrientation.BOTTOM], newValue[CustomMarginsOrientation.LEFT]);
    }
    /**
     * @param other The other margins object to compare against.
     * @return Whether this margins object is equal to another.
     */
    equals(other) {
        if (other === null) {
            return false;
        }
        for (const key in this.value_) {
            const orientation = key;
            if (this.value_[orientation] !== other.value_[orientation]) {
                return false;
            }
        }
        return true;
    }
    /** @return A serialized representation of the margins. */
    serialize() {
        return this.clone_();
    }
    clone_() {
        const clone = { top: 0, bottom: 0, left: 0, right: 0 };
        for (const o in this.value_) {
            const orientation = o;
            clone[orientation] = this.value_[orientation];
        }
        return clone;
    }
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PrintableArea {
    origin_;
    size_;
    /**
     * Object describing the printable area of a page in the document.
     * @param origin Top left corner of the printable area of the document.
     * @param size Size of the printable area of the document.
     */
    constructor(origin, size) {
        this.origin_ = origin;
        this.size_ = size;
    }
    get origin() {
        return this.origin_;
    }
    get size() {
        return this.size_;
    }
    equals(other) {
        return other !== null && this.origin_.equals(other.origin_) &&
            this.size_.equals(other.size_);
    }
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class Size {
    width_;
    height_;
    /**
     * Immutable two-dimensional size.
     */
    constructor(width, height) {
        this.width_ = width;
        this.height_ = height;
    }
    get width() {
        return this.width_;
    }
    get height() {
        return this.height_;
    }
    equals(other) {
        return other !== null && this.width_ === other.width_ &&
            this.height_ === other.height_;
    }
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function createDocumentSettings() {
    return {
        allPagesHaveCustomSize: false,
        allPagesHaveCustomOrientation: false,
        hasSelection: false,
        isModifiable: true,
        isScalingDisabled: false,
        fitToPageScaling: 100,
        pageCount: 0,
        title: '',
    };
}
const PrintPreviewDocumentInfoElementBase = WebUiListenerMixinLit(CrLitElement);
class PrintPreviewDocumentInfoElement extends PrintPreviewDocumentInfoElementBase {
    static get is() {
        return 'print-preview-document-info';
    }
    static get properties() {
        return {
            documentSettings: {
                type: Object,
                notify: true,
            },
            margins: {
                type: Object,
                notify: true,
            },
            /**
             * Size of the pages of the document in points. Actual page-related
             * information won't be set until preview generation occurs, so use
             * a default value until then.
             */
            pageSize: {
                type: Object,
                notify: true,
            },
            /**
             * Printable area of the document in points.
             */
            printableArea: {
                type: Object,
                notify: true,
            },
        };
    }
    #documentSettings_accessor_storage = createDocumentSettings();
    get documentSettings() { return this.#documentSettings_accessor_storage; }
    set documentSettings(value) { this.#documentSettings_accessor_storage = value; }
    inFlightRequestId = -1;
    #margins_accessor_storage = null;
    get margins() { return this.#margins_accessor_storage; }
    set margins(value) { this.#margins_accessor_storage = value; }
    #pageSize_accessor_storage = new Size(612, 792);
    get pageSize() { return this.#pageSize_accessor_storage; }
    set pageSize(value) { this.#pageSize_accessor_storage = value; }
    #printableArea_accessor_storage = new PrintableArea(new Coordinate2d(0, 0), new Size(612, 792));
    get printableArea() { return this.#printableArea_accessor_storage; }
    set printableArea(value) { this.#printableArea_accessor_storage = value; }
    isInitialized_ = false;
    connectedCallback() {
        super.connectedCallback();
        this.addWebUiListener('page-count-ready', (pageCount, previewResponseId, scaling) => this.onPageCountReady_(pageCount, previewResponseId, scaling));
        this.addWebUiListener('page-layout-ready', (pageLayout, allPagesHaveCustomSize, allPagesHaveCustomOrientation) => this.onPageLayoutReady_(pageLayout, allPagesHaveCustomSize, allPagesHaveCustomOrientation));
    }
    // Whenever documentSettings needs to be modified need to use
    // cloneAndModify_(), which implements the immutable data pattern, because
    // modifying in-place will not trigger Lit observers.
    cloneAndModify_(callback) {
        const clone = structuredClone(this.documentSettings);
        callback(clone);
        this.documentSettings = clone;
    }
    /**
     * Initializes the state of the data model.
     */
    init(isModifiable, title, hasSelection) {
        this.isInitialized_ = true;
        this.cloneAndModify_(documentSettings => {
            documentSettings.isModifiable = isModifiable;
            documentSettings.title = title;
            documentSettings.hasSelection = hasSelection;
        });
    }
    /**
     * Updates whether scaling is disabled for the document.
     */
    updateIsScalingDisabled(isScalingDisabled) {
        if (this.isInitialized_) {
            this.cloneAndModify_(documentSettings => {
                documentSettings.isScalingDisabled = isScalingDisabled;
            });
        }
    }
    /**
     * Called when the page layout of the document is ready. Always occurs
     * as a result of a preview request.
     * @param pageLayout Layout information about the document.
     * @param allPagesHaveCustomSize Whether this document has a custom page size
     *     or style to use for all pages.
     * @param allPagesHaveCustomOrientation Whether this document has a custom
     *     page orientation to use for all pages.
     */
    onPageLayoutReady_(pageLayout, allPagesHaveCustomSize, allPagesHaveCustomOrientation) {
        const origin = new Coordinate2d(pageLayout.printableAreaX, pageLayout.printableAreaY);
        const size = new Size(pageLayout.printableAreaWidth, pageLayout.printableAreaHeight);
        const pageSize = new Size(Math.round(pageLayout.contentWidth + pageLayout.marginLeft +
            pageLayout.marginRight), Math.round(pageLayout.contentHeight + pageLayout.marginTop +
            pageLayout.marginBottom));
        // Note that `Margins` stores rounded margin values, which is not
        // appropriate for use with `pageSize` above, as that could cause rounding
        // errors.
        const margins = new Margins(pageLayout.marginTop, pageLayout.marginRight, pageLayout.marginBottom, pageLayout.marginLeft);
        if (this.isInitialized_) {
            this.printableArea = new PrintableArea(origin, size);
            this.pageSize = pageSize;
            this.cloneAndModify_(documentSettings => {
                documentSettings.allPagesHaveCustomSize = allPagesHaveCustomSize;
                documentSettings.allPagesHaveCustomOrientation =
                    allPagesHaveCustomOrientation;
            });
            this.margins = margins;
        }
    }
    /**
     * Called when the document page count is received from the native layer.
     * Always occurs as a result of a preview request.
     * @param pageCount The document's page count.
     * @param previewResponseId The request ID for this page count event.
     * @param fitToPageScaling The scaling required to fit the document to page.
     */
    onPageCountReady_(pageCount, previewResponseId, fitToPageScaling) {
        if (this.inFlightRequestId !== previewResponseId || !this.isInitialized_) {
            return;
        }
        this.cloneAndModify_(documentSettings => {
            documentSettings.pageCount = pageCount;
            documentSettings.fitToPageScaling = fitToPageScaling;
        });
    }
}
customElements.define(PrintPreviewDocumentInfoElement.is, PrintPreviewDocumentInfoElement);

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Enumeration of measurement unit types.
 */
var MeasurementSystemUnitType;
(function (MeasurementSystemUnitType) {
    MeasurementSystemUnitType[MeasurementSystemUnitType["METRIC"] = 0] = "METRIC";
    MeasurementSystemUnitType[MeasurementSystemUnitType["IMPERIAL"] = 1] = "IMPERIAL";
})(MeasurementSystemUnitType || (MeasurementSystemUnitType = {}));
class MeasurementSystem {
    /**
     * The thousands delimiter to use when displaying numbers.
     */
    thousandsDelimiter_;
    /**
     * The decimal delimiter to use when displaying numbers.
     */
    decimalDelimiter_;
    /**
     * The measurement system preferences based on the unit type.
     */
    measurementSystemPrefs_;
    /**
     * Measurement system of the print preview. Used to parse and serialize
     * point measurements into the system's local units (e.g. millimeters,
     * inches).
     * @param thousandsDelimiter Delimiter between thousands digits.
     * @param decimalDelimiter Delimiter between integers and decimals.
     * @param unitType Measurement unit type of the system.
     */
    constructor(thousandsDelimiter, decimalDelimiter, unitType) {
        this.thousandsDelimiter_ = thousandsDelimiter || ',';
        this.decimalDelimiter_ = decimalDelimiter || '.';
        assert(measurementSystemPrefs.has(unitType));
        this.measurementSystemPrefs_ = measurementSystemPrefs.get(unitType);
    }
    get unitSymbol() {
        return this.measurementSystemPrefs_.unitSymbol;
    }
    get thousandsDelimiter() {
        return this.thousandsDelimiter_;
    }
    get decimalDelimiter() {
        return this.decimalDelimiter_;
    }
    /**
     * Rounds a value in the local system's units to the appropriate precision.
     */
    roundValue(value) {
        const precision = this.measurementSystemPrefs_.precision;
        const roundedValue = Math.round(value / precision) * precision;
        // Truncate
        return +roundedValue.toFixed(this.measurementSystemPrefs_.decimalPlaces);
    }
    /**
     * @param pts Value in points to convert to local units.
     * @return Value in local units.
     */
    convertFromPoints(pts) {
        return pts / this.measurementSystemPrefs_.ptsPerUnit;
    }
    /**
     * @param localUnits Value in local units to convert to points.
     * @return Value in points.
     */
    convertToPoints(localUnits) {
        return localUnits * this.measurementSystemPrefs_.ptsPerUnit;
    }
}
/**
 * Maximum resolution and number of decimal places for local unit values.
 */
const measurementSystemPrefs = new Map([
    [
        MeasurementSystemUnitType.METRIC,
        {
            precision: 0.5,
            decimalPlaces: 1,
            ptsPerUnit: 72.0 / 25.4,
            unitSymbol: 'mm',
        },
    ],
    [
        MeasurementSystemUnitType.IMPERIAL,
        { precision: 0.01, decimalPlaces: 2, ptsPerUnit: 72.0, unitSymbol: '"' },
    ],
]);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview PromiseResolver is a helper class that allows creating a
 * Promise that will be fulfilled (resolved or rejected) some time later.
 *
 * Example:
 *  const resolver = new PromiseResolver();
 *  resolver.promise.then(function(result) {
 *    console.log('resolved with', result);
 *  });
 *  ...
 *  ...
 *  resolver.resolve({hello: 'world'});
 */
class PromiseResolver {
    resolve_ = () => { };
    reject_ = () => { };
    isFulfilled_ = false;
    promise_;
    constructor() {
        this.promise_ = new Promise((resolve, reject) => {
            this.resolve_ = (resolution) => {
                resolve(resolution);
                this.isFulfilled_ = true;
            };
            this.reject_ = (reason) => {
                reject(reason);
                this.isFulfilled_ = true;
            };
        });
    }
    /** Whether this resolver has been resolved or rejected. */
    get isFulfilled() {
        return this.isFulfilled_;
    }
    get promise() {
        return this.promise_;
    }
    get resolve() {
        return this.resolve_;
    }
    get reject() {
        return this.reject_;
    }
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Non-Polymer subproperty observation implementation, to facilitate Lit
// migrations for WebUIs that heavily depend on Polymer's subproperty
// observation.
//
// High level design:
//  - buildProxy(): Helper function to wrap the object to be observed with an ES
//    Proxy. The function calls itself recursively (lazily) to create as many
//    Proxy instances as needed to intercept all sub-objects/arrays. 'callback'
//    is invoked whenever any path changes. This function is the primitive upon
//    the rest of this approach is based on.
//  - ObserverTree: A tree data structure of ObserverNode instances, to keep
//    track of observers that are registered for each node of the original
//    object.
//  - Observable: The class to be used by client code to observe an object,
//    leverages buildProxy and ObserverTree internally.
//
//  The format of change notifications is following Polymer observers
//  notifications (with extensive test coverage to ensure compatibility), to
//  make it easier to migrate Polymer observers to Observable observers.
//
//  Note: Array push() and splice() notifications are not implemented yet, will
//  be implemented if/when the need arises.
function buildProxy(obj, callback, path, proxyCache) {
    function getPath(prop) {
        return path.slice(1).concat(prop).join('.');
    }
    return new Proxy(obj, {
        get(target, prop) {
            const value = target[prop];
            if (value && typeof value === 'object' &&
                ['Array', 'Object'].includes(value.constructor.name)) {
                let proxy = proxyCache.get(value) || null;
                if (proxy === null) {
                    proxy = buildProxy(value, callback, path.concat(prop), proxyCache);
                    proxyCache.set(value, proxy);
                }
                return proxy;
            }
            return value;
        },
        set(target, prop, value) {
            const previousValue = target[prop];
            if (previousValue === value) {
                return true;
            }
            target[prop] = value;
            callback(value, previousValue, getPath(prop));
            return true;
        },
    });
}
function getValueAtPath(pathParts, obj) {
    let result = obj;
    let counter = pathParts.length;
    while (counter > 1) {
        const current = pathParts[pathParts.length - counter--];
        result = result[current];
    }
    return result[pathParts.at(-1)];
}
function setValueAtPath(pathParts, obj, value) {
    let parent = obj;
    let counter = pathParts.length;
    while (counter > 1) {
        const current = pathParts[pathParts.length - counter--];
        parent = parent[current];
    }
    parent[pathParts.at(-1)] = value;
}
// A tree data structure to keep track of observers for a nested data object. It
// replicates the behavior of Polymer observers.
class ObserverTree {
    root = { key: '' };
    // Internal book keeping to make it easy to remove observers using an ID.
    nextObserverId_ = -1;
    observers_ = new Map();
    getNode(path, create = false) {
        const pathParts = path.split('.');
        let node = this.root;
        while (pathParts.length > 0 && node !== null) {
            const currentPart = pathParts.shift();
            if (create && !node.children) {
                node.children = new Map();
            }
            let child = node.children?.get(currentPart) || null;
            if (create && child === null) {
                child = { parent: node, key: currentPart };
                node.children.set(child.key, child);
            }
            node = child;
        }
        return node;
    }
    // Traverses all nodes between the root of the tree and the node corresponding
    // to 'path' (including that node as well).
    traversePath(path, callback) {
        const pathParts = path.split('.');
        const traversedParts = [this.root.key];
        let node = this.root;
        callback(node, false, traversedParts);
        while (pathParts.length > 0 && node !== null) {
            const currentPart = pathParts.shift();
            traversedParts.push(currentPart);
            node = node.children?.get(currentPart) || null;
            if (node !== null) {
                callback(node, pathParts.length === 0, traversedParts);
            }
        }
    }
    // Traverses all nodes from the given root node and below (including the root
    // itself) and invokes the callback on each node visited. Besides the
    // current node, it also passes the path from the original input `node` to the
    // current node.
    traverseTree(node, callback) {
        function visitNode(node, relativePath) {
            callback(node, relativePath);
            if (node.children) {
                for (const child of node.children.values()) {
                    visitNode(child, relativePath.concat(child.key));
                }
            }
        }
        visitNode(node, []);
    }
    addObserver(path, callback) {
        let effectivePath = path;
        // Observers ending with '.*' receive notifications for any change
        // happening under the corresponding node.
        const isWildcard = path.endsWith('.*');
        if (isWildcard) {
            effectivePath = path.slice(0, -2);
        }
        const node = this.getNode(effectivePath, /*create=*/ true);
        if (!node.observers) {
            node.observers = new Set();
        }
        // Add observer to the ObserverNode.
        const id = ++this.nextObserverId_;
        node.observers.add({ id, isWildcard, callback });
        // Add entry in `observers_` to be used in removeObserver.
        this.observers_.set(id, node);
        return id;
    }
    removeObserver(id) {
        const node = this.observers_.get(id) || null;
        if (!node) {
            return false;
        }
        assert(node.observers);
        const observerEntry = Array.from(node.observers).find(node => node.id === id) || null;
        assert(observerEntry);
        this.observers_.delete(id);
        const deleted = node.observers.delete(observerEntry);
        assert(deleted);
        return true;
    }
    removeAllObservers() {
        for (const id of this.observers_.keys()) {
            this.removeObserver(id);
        }
    }
}
class Observable {
    proxyCache_ = new WeakMap();
    proxy_;
    target_;
    observerTree_ = new ObserverTree();
    constructor(target) {
        this.target_ = target;
        this.proxy_ =
            buildProxy(target, this.onChange_.bind(this), [''], this.proxyCache_);
    }
    getProxy() {
        return this.proxy_;
    }
    // Returns the original raw object. Useful when it needs to be serialized.
    getTarget() {
        return this.target_;
    }
    onChange_(newValue, previousValue, path) {
        let lastNode = null;
        this.observerTree_.traversePath(path, (node, isLast, pathParts) => {
            if (isLast) {
                // Remember the last ObserverNode on 'path' for later.
                lastNode = node;
                return;
            }
            if (node.observers) {
                const base = getValueAtPath(pathParts.slice(1), this.proxy_);
                for (const { isWildcard, callback } of node.observers) {
                    // Notify '.*' observers between the root and the modified node.
                    // For wildcard observers above the changed node, report the
                    // changed path and new values verbatim.
                    if (isWildcard) {
                        callback({ path, value: newValue, base });
                    }
                }
            }
        });
        if (lastNode === null) {
            // No observers exist. Nothing to do.
            return;
        }
        // Notify observers directly on the modified node or anywhere below it.
        this.observerTree_.traverseTree(lastNode, (node, relativePath) => {
            if (!node.observers) {
                return;
            }
            let observerNewValue = newValue;
            let observerPreviousValue = previousValue;
            // Calculate the `newValue` and `previousValue` from each observer's
            // point of view.
            if (node !== lastNode) {
                observerNewValue = getValueAtPath(relativePath, newValue);
                observerPreviousValue = getValueAtPath(relativePath, previousValue);
            }
            const observedPath = [path, ...relativePath].join('.');
            for (const observer of node.observers) {
                if (observer.isWildcard) {
                    if (node !== lastNode) {
                        // For wildcard observers below the changed node, report the
                        // observed path as 'path' and the relative new value as
                        // 'value' and 'base'. This is to maintain parity with Polymer,
                        // even though it is a bit odd.
                        observer.callback({
                            path: observedPath,
                            value: observerNewValue,
                            base: observerNewValue,
                        });
                    }
                    else {
                        // For wildcard observers at the changed node, report the
                        // changed path as 'path' and the new value verbatim as
                        // 'value'.
                        observer.callback({ path, value: newValue, base: newValue });
                    }
                    continue;
                }
                // For non-wildcard observers below or at the changed node, report
                // the observed path as 'path' and the relative new value as
                // 'value'.
                observer.callback(observerNewValue, observerPreviousValue, observedPath);
            }
        });
    }
    addObserver(path, callback) {
        return this.observerTree_.addObserver(path, callback);
    }
    removeObserver(id) {
        return this.observerTree_.removeObserver(id);
    }
    removeAllObservers() {
        return this.observerTree_.removeAllObservers();
    }
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Must be kept in sync with the C++ ScalingType enum in
 * printing/print_job_constants.h.
 */
var ScalingType;
(function (ScalingType) {
    ScalingType[ScalingType["DEFAULT"] = 0] = "DEFAULT";
    ScalingType[ScalingType["FIT_TO_PAGE"] = 1] = "FIT_TO_PAGE";
    ScalingType[ScalingType["FIT_TO_PAPER"] = 2] = "FIT_TO_PAPER";
    ScalingType[ScalingType["CUSTOM"] = 3] = "CUSTOM";
})(ScalingType || (ScalingType = {}));

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Constant values matching printing::DuplexMode enum.
 */
var DuplexMode;
(function (DuplexMode) {
    DuplexMode[DuplexMode["SIMPLEX"] = 0] = "SIMPLEX";
    DuplexMode[DuplexMode["LONG_EDGE"] = 1] = "LONG_EDGE";
    DuplexMode[DuplexMode["SHORT_EDGE"] = 2] = "SHORT_EDGE";
    DuplexMode[DuplexMode["UNKNOWN_DUPLEX_MODE"] = -1] = "UNKNOWN_DUPLEX_MODE";
})(DuplexMode || (DuplexMode = {}));
let instance$D = null;
let whenReadyResolver = new PromiseResolver();
function getInstance() {
    assert(instance$D);
    return instance$D;
}
function whenReady() {
    return whenReadyResolver.promise;
}
/**
 * Sticky setting names in alphabetical order.
 */
const STICKY_SETTING_NAMES = [
    'recentDestinations',
    'collate',
    'color',
    'cssBackground',
    'customMargins',
    'dpi',
    'duplex',
    'duplexShortEdge',
    'headerFooter',
    'layout',
    'margins',
    'mediaSize',
    'scaling',
    'scalingType',
    'scalingTypePdf',
    'vendorItems',
];
/**
 * Minimum height of page in microns to allow headers and footers. Should
 * match the value for min_size_printer_units in printing/print_settings.cc
 * so that we do not request header/footer for margins that will be zero.
 */
const MINIMUM_HEIGHT_MICRONS = 25400;
function createSettings() {
    return {
        pages: {
            value: [1],
            unavailableValue: [],
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: '',
            updatesPreview: false,
        },
        copies: {
            value: 1,
            unavailableValue: 1,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: '',
            updatesPreview: false,
        },
        collate: {
            value: true,
            unavailableValue: false,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'isCollateEnabled',
            updatesPreview: false,
        },
        layout: {
            value: false, /* portrait */
            unavailableValue: false,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'isLandscapeEnabled',
            updatesPreview: true,
        },
        color: {
            value: true, /* color */
            unavailableValue: false,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'isColorEnabled',
            updatesPreview: true,
        },
        mediaSize: {
            value: {},
            unavailableValue: {
                width_microns: 215900,
                height_microns: 279400,
                imageable_area_left_microns: 0,
                imageable_area_bottom_microns: 0,
                imageable_area_right_microns: 215900,
                imageable_area_top_microns: 279400,
            },
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'mediaSize',
            updatesPreview: true,
        },
        margins: {
            value: MarginsType.DEFAULT,
            unavailableValue: MarginsType.DEFAULT,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'marginsType',
            updatesPreview: true,
        },
        customMargins: {
            value: {},
            unavailableValue: {},
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'customMargins',
            updatesPreview: true,
        },
        dpi: {
            value: {},
            unavailableValue: {},
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'dpi',
            updatesPreview: false,
        },
        scaling: {
            value: '100',
            unavailableValue: '100',
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'scaling',
            updatesPreview: true,
        },
        scalingType: {
            value: ScalingType.DEFAULT,
            unavailableValue: ScalingType.DEFAULT,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'scalingType',
            updatesPreview: true,
        },
        scalingTypePdf: {
            value: ScalingType.DEFAULT,
            unavailableValue: ScalingType.DEFAULT,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'scalingTypePdf',
            updatesPreview: true,
        },
        duplex: {
            value: true,
            unavailableValue: false,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'isDuplexEnabled',
            updatesPreview: false,
        },
        duplexShortEdge: {
            value: false,
            unavailableValue: false,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'isDuplexShortEdge',
            updatesPreview: false,
        },
        cssBackground: {
            value: false,
            unavailableValue: false,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'isCssBackgroundEnabled',
            updatesPreview: true,
        },
        selectionOnly: {
            value: false,
            unavailableValue: false,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: '',
            updatesPreview: true,
        },
        headerFooter: {
            value: true,
            unavailableValue: false,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'isHeaderFooterEnabled',
            updatesPreview: true,
        },
        rasterize: {
            value: false,
            unavailableValue: false,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: '',
            updatesPreview: true,
        },
        vendorItems: {
            value: {},
            unavailableValue: {},
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'vendorOptions',
            updatesPreview: false,
        },
        pagesPerSheet: {
            value: 1,
            unavailableValue: 1,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: '',
            updatesPreview: true,
        },
        // This does not represent a real setting value, and is used only to
        // expose the availability of the other options settings section.
        otherOptions: {
            value: null,
            unavailableValue: null,
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: '',
            updatesPreview: false,
        },
        // This does not represent a real settings value, but is used to
        // propagate the correctly formatted ranges for print tickets.
        ranges: {
            value: [],
            unavailableValue: [],
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: '',
            updatesPreview: true,
        },
        recentDestinations: {
            value: [],
            unavailableValue: [],
            valid: true,
            available: true,
            setByGlobalPolicy: false,
            setFromUi: false,
            key: 'recentDestinations',
            updatesPreview: false,
        },
    };
}
class PrintPreviewModelElement extends CrLitElement {
    static get is() {
        return 'print-preview-model';
    }
    static get properties() {
        return {
            settingsManaged: {
                type: Boolean,
                notify: true,
            },
            destination: { type: Object },
            documentSettings: { type: Object },
            margins: { type: Object },
            pageSize: { type: Object },
        };
    }
    #settingsManaged_accessor_storage = false;
    get settingsManaged() { return this.#settingsManaged_accessor_storage; }
    set settingsManaged(value) { this.#settingsManaged_accessor_storage = value; }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #documentSettings_accessor_storage = createDocumentSettings();
    get documentSettings() { return this.#documentSettings_accessor_storage; }
    set documentSettings(value) { this.#documentSettings_accessor_storage = value; }
    #margins_accessor_storage = null;
    get margins() { return this.#margins_accessor_storage; }
    set margins(value) { this.#margins_accessor_storage = value; }
    #pageSize_accessor_storage = new Size(612, 792);
    get pageSize() { return this.#pageSize_accessor_storage; }
    set pageSize(value) { this.#pageSize_accessor_storage = value; }
    observable;
    initialized_ = false;
    stickySettings_ = null;
    policySettings_ = null;
    lastDestinationCapabilities_ = null;
    /**
     * Object containing current settings of Print Preview.
     * Initialize all settings to available so that more settings always stays
     * in a collapsed state during startup, when document information and
     * printer capabilities may arrive at slightly different times.
     */
    settings_;
    constructor() {
        super();
        this.observable = new Observable(createSettings());
        this.settings_ = this.observable.getProxy();
    }
    connectedCallback() {
        super.connectedCallback();
        assert(!instance$D);
        instance$D = this;
        whenReadyResolver.resolve();
        this.observable.addObserver('margins.value', this.updateHeaderFooterAvailable_.bind(this));
        this.observable.addObserver('mediaSize.value', this.updateHeaderFooterAvailable_.bind(this));
        this.updateHeaderFooterAvailable_();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        instance$D = null;
        whenReadyResolver = new PromiseResolver();
        this.observable.removeAllObservers();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('destination')) {
            this.updateSettingsFromDestination();
        }
        if (changedProperties.has('documentSettings')) {
            this.updateSettingsAvailabilityFromDocumentSettings_();
        }
        if (changedProperties.has('margins')) {
            this.updateHeaderFooterAvailable_();
        }
    }
    // Returns a direct reference to the non-proxied Settings object. The returned
    // object should never be mutated manually by callers, since such mutation
    // will not generate any Observable notifications.
    getSetting(settingName) {
        const setting = this.observable.getTarget()[settingName];
        assert(setting, 'Setting is missing: ' + settingName);
        return setting;
    }
    /**
     * @param settingName Name of the setting to get the value for.
     * @return The value of the setting, accounting for availability.
     */
    getSettingValue(settingName) {
        const setting = this.getSetting(settingName);
        return setting.available ? setting.value : setting.unavailableValue;
    }
    /**
     * Updates settings.settingPath to |value|. Fires a preview-setting-changed
     * event if the modification results in a change to the value returned by
     * getSettingValue().
     */
    setSettingPath_(settingPath, value) {
        const parts = settingPath.split('.');
        assert(parts.length >= 2);
        const settingName = parts[0];
        const setting = this.getSetting(settingName);
        const oldValue = this.getSettingValue(settingName);
        setValueAtPath(parts, this.settings_, value);
        const newValue = this.getSettingValue(settingName);
        if (newValue !== oldValue && setting.updatesPreview) {
            this.fire('preview-setting-changed');
        }
    }
    /**
     * Sets settings.settingName.value to |value|, unless updating the setting is
     * disallowed by enterprise policy. Fires preview-setting-changed and
     * sticky-setting-changed events if the update impacts the preview or requires
     * an update to sticky settings. Used for setting settings from UI elements.
     * @param settingName Name of the setting to set
     * @param value The value to set the setting to.
     * @param noSticky Whether to avoid stickying the setting. Defaults to false.
     */
    setSetting(settingName, value, noSticky) {
        const setting = this.getSetting(settingName);
        if (setting.setByGlobalPolicy) {
            return;
        }
        const fireStickyEvent = !noSticky && setting.value !== value && setting.key;
        this.setSettingPath_(`${settingName}.value`, value);
        if (!noSticky) {
            this.setSettingPath_(`${settingName}.setFromUi`, true);
        }
        if (fireStickyEvent && this.initialized_) {
            this.fire('sticky-setting-changed', this.getStickySettings_());
        }
    }
    /**
     * Sets the validity of |settingName| to |valid|. If the validity is changed,
     * fires a setting-valid-changed event.
     * @param settingName Name of the setting to set
     * @param valid Whether the setting value is currently valid.
     */
    setSettingValid(settingName, valid) {
        const setting = this.getSetting(settingName);
        // Should not set the setting to invalid if it is not available, as there
        // is no way for the user to change the value in this case.
        if (!valid) {
            assert(setting.available, 'Setting is not available: ' + settingName);
        }
        const shouldFireEvent = valid !== setting.valid;
        this.settings_[settingName].valid = valid;
        if (shouldFireEvent) {
            this.fire('setting-valid-changed', valid);
        }
    }
    /**
     * Sets the availability of |settingName| to |available|, exposed only for
     * testing purposes, where `destination` and `documentSettings` are not
     * always set and therefore availability is not automatically inferred.
     */
    setSettingAvailableForTesting(settingName, available) {
        this.settings_[settingName].available = available;
    }
    /**
     * Sets the setByGlobalPolicy of |settingName| to |setByGlobalPolicy|, exposed
     * only for testing purposes, where `policySettings_` aren't always set.
     */
    setSettingSetByGlobalPolicyForTesting(settingName, setByGlobalPolicy) {
        this.settings_[settingName].setByGlobalPolicy = setByGlobalPolicy;
    }
    /**
     * Updates the availability of the settings sections and values of various
     * settings based on the destination capabilities.
     */
    updateSettingsFromDestination() {
        if (!this.destination) {
            return;
        }
        if (this.destination.capabilities === this.lastDestinationCapabilities_) {
            return;
        }
        this.lastDestinationCapabilities_ = this.destination.capabilities;
        this.updateSettingsAvailabilityFromDestination_();
        if (!this.destination.capabilities?.printer) {
            return;
        }
        this.updateSettingsValues_();
        this.applyPersistentCddDefaults_();
    }
    updateSettingsAvailabilityFromDestination_() {
        assert(this.destination);
        const caps = this.destination.capabilities ?
            this.destination.capabilities.printer :
            null;
        this.setSettingPath_('copies.available', this.destination.hasCopiesCapability);
        this.setSettingPath_('collate.available', !!caps && !!caps.collate);
        // TODO(crbug.com/374066702): "color.available" is set to false if the
        // per-printer job options policy allows to use only a single color (even if
        // the destination supports both b&w and color printing). This hides the
        // color setting dropdown instead of disabling it. Figure out if that's
        // desirable behaviour.
        this.setSettingPath_('color.available', this.destination.hasColorCapability);
        const capsHasDuplex = !!caps && !!caps.duplex && !!caps.duplex.option;
        const capsHasLongEdge = capsHasDuplex &&
            caps.duplex.option.some(o => o.type === DuplexType.LONG_EDGE);
        const capsHasShortEdge = capsHasDuplex &&
            caps.duplex.option.some(o => o.type === DuplexType.SHORT_EDGE);
        this.setSettingPath_('duplexShortEdge.available', capsHasLongEdge && capsHasShortEdge);
        this.setSettingPath_('duplex.available', (capsHasLongEdge || capsHasShortEdge) &&
            caps.duplex.option.some(o => o.type === DuplexType.NO_DUPLEX));
        this.setSettingPath_('vendorItems.available', !!caps && !!caps.vendor_capability);
        this.updateSettingsAvailabilityFromDestinationAndDocumentSettings_();
    }
    updateSettingsAvailabilityFromDestinationAndDocumentSettings_() {
        if (!this.documentSettings || !this.destination) {
            return;
        }
        const isSaveAsPDF = this.destination.type === PrinterType.PDF_PRINTER;
        const knownSizeToSaveAsPdf = isSaveAsPDF &&
            (!this.documentSettings.isModifiable ||
                this.documentSettings.allPagesHaveCustomSize);
        const scalingAvailable = !knownSizeToSaveAsPdf;
        this.setSettingPath_('scaling.available', scalingAvailable);
        this.setSettingPath_('scalingType.available', scalingAvailable && this.documentSettings.isModifiable);
        this.setSettingPath_('scalingTypePdf.available', scalingAvailable && !this.documentSettings.isModifiable);
        const caps = this.destination.capabilities?.printer || null;
        this.setSettingPath_('mediaSize.available', !!caps && !!caps.media_size && !knownSizeToSaveAsPdf);
        this.setSettingPath_('dpi.available', !!caps && !!caps.dpi && !!caps.dpi.option &&
            caps.dpi.option.length > 1);
        this.setSettingPath_('layout.available', this.isLayoutAvailable_(caps));
    }
    updateSettingsAvailabilityFromDocumentSettings_() {
        this.setSettingPath_('margins.available', this.documentSettings.isModifiable);
        this.setSettingPath_('customMargins.available', this.documentSettings.isModifiable);
        this.setSettingPath_('cssBackground.available', this.documentSettings.isModifiable);
        this.setSettingPath_('selectionOnly.available', this.documentSettings.isModifiable &&
            this.documentSettings.hasSelection);
        this.setSettingPath_('headerFooter.available', this.isHeaderFooterAvailable_());
        this.setSettingPath_('rasterize.available', this.isRasterizeAvailable_());
        this.setSettingPath_('otherOptions.available', this.settings_.cssBackground.available ||
            this.settings_.selectionOnly.available ||
            this.settings_.headerFooter.available ||
            this.settings_.rasterize.available);
        this.updateSettingsAvailabilityFromDestinationAndDocumentSettings_();
    }
    updateHeaderFooterAvailable_() {
        if (this.documentSettings === undefined) {
            return;
        }
        this.setSettingPath_('headerFooter.available', this.isHeaderFooterAvailable_());
    }
    /**
     * @return Whether the header/footer setting should be available.
     */
    isHeaderFooterAvailable_() {
        // Always unavailable for PDFs.
        if (!this.documentSettings.isModifiable) {
            return false;
        }
        // Always unavailable for small paper sizes.
        const microns = this.getSettingValue('layout') ?
            this.getSettingValue('mediaSize').width_microns :
            this.getSettingValue('mediaSize').height_microns;
        if (microns < MINIMUM_HEIGHT_MICRONS) {
            return false;
        }
        // Otherwise, availability depends on the margins.
        const marginsType = this.getSettingValue('margins');
        if (marginsType === MarginsType.NO_MARGINS) {
            return false;
        }
        if (marginsType === MarginsType.MINIMUM) {
            return true;
        }
        return !this.margins ||
            this.margins.get(CustomMarginsOrientation.TOP) > 0 ||
            this.margins.get(CustomMarginsOrientation.BOTTOM) > 0;
    }
    // 
    updateRasterizeAvailable_() {
        // Need document settings to know if source is PDF.
        if (this.documentSettings === undefined) {
            return;
        }
        this.setSettingPath_('rasterize.available', this.isRasterizeAvailable_());
    }
    // 
    /**
     * @return Whether the rasterization setting should be available.
     */
    isRasterizeAvailable_() {
        // Only a possibility for PDFs.  Always available for PDFs on Linux and
        // ChromeOS.  crbug.com/675798
        let available = !!this.documentSettings && !this.documentSettings.isModifiable;
        // 
        // Availability on Windows or macOS depends upon policy.
        if (!available || !this.policySettings_) {
            return false;
        }
        const policy = this.policySettings_['printPdfAsImageAvailability'];
        available = policy !== undefined && policy.value;
        // 
        return available;
    }
    isLayoutAvailable_(caps) {
        if (!caps || !caps.page_orientation || !caps.page_orientation.option ||
            !this.documentSettings.isModifiable ||
            this.documentSettings.allPagesHaveCustomOrientation) {
            return false;
        }
        let hasAutoOrPortraitOption = false;
        let hasLandscapeOption = false;
        caps.page_orientation.option.forEach(option => {
            hasAutoOrPortraitOption = hasAutoOrPortraitOption ||
                option.type === 'AUTO' || option.type === 'PORTRAIT';
            hasLandscapeOption = hasLandscapeOption || option.type === 'LANDSCAPE';
        });
        return hasLandscapeOption && hasAutoOrPortraitOption;
    }
    updateSettingsValues_() {
        assert(this.destination);
        const caps = this.destination.capabilities ?
            this.destination.capabilities.printer :
            null;
        if (!caps) {
            return;
        }
        if (this.settings_.mediaSize.available) {
            const defaultOption = caps.media_size.option.find(o => !!o.is_default) ||
                caps.media_size.option[0];
            this.setSetting('mediaSize', defaultOption, true);
        }
        if (this.settings_.dpi.available) {
            const defaultOption = caps.dpi.option.find(o => !!o.is_default) || caps.dpi.option[0];
            let matchingOption = null;
            if (this.settings_.dpi.setFromUi) {
                const currentDpi = this.getSettingValue('dpi');
                matchingOption = this.destination.getDpi(currentDpi.horizontal_dpi, currentDpi.vertical_dpi);
            }
            this.setSetting('dpi', matchingOption || defaultOption, true);
        }
        else if (caps.dpi && caps.dpi.option && caps.dpi.option.length > 0) {
            const unavailableValue = caps.dpi.option.find(o => !!o.is_default) || caps.dpi.option[0];
            this.setSettingPath_('dpi.unavailableValue', unavailableValue);
        }
        if (!this.settings_.color.setFromUi && this.settings_.color.available) {
            const defaultOption = this.destination.defaultColorOption;
            if (defaultOption) {
                this.setSetting('color', !['STANDARD_MONOCHROME', 'CUSTOM_MONOCHROME'].includes(defaultOption.type), true);
            }
        }
        else if (!this.settings_.color.available && caps.color && caps.color.option &&
            caps.color.option.length > 0) {
            this.setSettingPath_('color.unavailableValue', !['STANDARD_MONOCHROME', 'CUSTOM_MONOCHROME'].includes(caps.color.option[0].type));
        }
        else if (!this.settings_.color.available) {
            // if no color capability is reported, assume black and white.
            this.setSettingPath_('color.unavailableValue', false);
        }
        if (!this.settings_.duplex.setFromUi && this.settings_.duplex.available) {
            const defaultOption = caps.duplex.option.find(o => !!o.is_default);
            if (defaultOption !== undefined) {
                const defaultOptionIsDuplex = defaultOption.type === DuplexType.SHORT_EDGE ||
                    defaultOption.type === DuplexType.LONG_EDGE;
                this.setSetting('duplex', defaultOptionIsDuplex, true);
                if (defaultOptionIsDuplex) {
                    this.setSetting('duplexShortEdge', defaultOption.type === DuplexType.SHORT_EDGE, true);
                }
                if (!this.settings_.duplexShortEdge.available) {
                    // Duplex is available, so must have only one two sided printing
                    // option. Set duplexShortEdge's unavailable value based on the
                    // printer.
                    this.setSettingPath_('duplexShortEdge.unavailableValue', caps.duplex.option.some(o => o.type === DuplexType.SHORT_EDGE));
                }
            }
        }
        else if (!this.settings_.duplex.available && caps && caps.duplex &&
            caps.duplex.option) {
            // In this case, there must only be one option.
            const hasLongEdge = caps.duplex.option.some(o => o.type === DuplexType.LONG_EDGE);
            const hasShortEdge = caps.duplex.option.some(o => o.type === DuplexType.SHORT_EDGE);
            // If the only option available is long edge, the value should always be
            // true.
            this.setSettingPath_('duplex.unavailableValue', hasLongEdge || hasShortEdge);
            this.setSettingPath_('duplexShortEdge.unavailableValue', hasShortEdge);
        }
        else if (!this.settings_.duplex.available) {
            // If no duplex capability is reported, assume false.
            this.setSettingPath_('duplex.unavailableValue', false);
            this.setSettingPath_('duplexShortEdge.unavailableValue', false);
        }
        if (this.settings_.vendorItems.available) {
            const vendorSettings = {};
            for (const item of caps.vendor_capability) {
                let defaultValue = null;
                if (item.type === 'SELECT' && item.select_cap &&
                    item.select_cap.option) {
                    const defaultOption = item.select_cap.option.find(o => !!o.is_default);
                    defaultValue = defaultOption ? defaultOption.value : null;
                }
                else if (item.type === 'RANGE') {
                    if (item.range_cap) {
                        defaultValue = item.range_cap.default || null;
                    }
                }
                else if (item.type === 'TYPED_VALUE') {
                    if (item.typed_value_cap) {
                        defaultValue = item.typed_value_cap.default || null;
                    }
                }
                if (defaultValue !== null) {
                    vendorSettings[item.id] = defaultValue;
                }
            }
            this.setSetting('vendorItems', vendorSettings, true);
        }
    }
    /**
     * Caches the sticky settings and sets up the recent destinations. Sticky
     * settings will be applied when destinaton capabilities have been retrieved.
     */
    setStickySettings(savedSettingsStr) {
        assert(!this.stickySettings_);
        if (!savedSettingsStr) {
            return;
        }
        let savedSettings;
        try {
            savedSettings = JSON.parse(savedSettingsStr);
        }
        catch (e) {
            console.warn('Unable to parse state ' + e);
            return; // use default values rather than updating.
        }
        if (savedSettings.version !== 2) {
            return;
        }
        if (savedSettings.marginsType === MarginsType.CUSTOM) {
            let valid = false;
            if (savedSettings.customMargins) {
                // Per crbug.com/40067498 and crbug.com/348806045, the C++ side expects
                // the custom margins values to be integers, so round them here.
                savedSettings.customMargins.marginTop =
                    Math.round(savedSettings.customMargins.marginTop);
                savedSettings.customMargins.marginRight =
                    Math.round(savedSettings.customMargins.marginRight);
                savedSettings.customMargins.marginBottom =
                    Math.round(savedSettings.customMargins.marginBottom);
                savedSettings.customMargins.marginLeft =
                    Math.round(savedSettings.customMargins.marginLeft);
                // The above fix is still insufficient to make the C++ side happy, so
                // do an additional sanity check here. See crbug.com/379053829.
                const isValidCustomMarginValue = (value) => value >= 0;
                valid =
                    isValidCustomMarginValue(savedSettings.customMargins.marginTop) &&
                        isValidCustomMarginValue(savedSettings.customMargins.marginRight) &&
                        isValidCustomMarginValue(savedSettings.customMargins.marginBottom) &&
                        isValidCustomMarginValue(savedSettings.customMargins.marginLeft);
            }
            if (!valid) {
                // If the sanity check above fails, then fall back to the default
                // margins type, so the C++ side does not encounter conflicting data.
                savedSettings.marginsType = MarginsType.DEFAULT;
                delete savedSettings.customMargins;
            }
        }
        let recentDestinations = savedSettings.recentDestinations || [];
        if (!Array.isArray(recentDestinations)) {
            recentDestinations = [recentDestinations];
        }
        // Remove unsupported privet and cloud printers from the sticky settings,
        // to free up these spots for supported printers.
        const unsupportedOrigins = [
            DestinationOrigin.COOKIES,
            DestinationOrigin.PRIVET,
        ];
        recentDestinations = recentDestinations.filter((d) => {
            return !unsupportedOrigins.includes(d.origin);
        });
        // Initialize recent destinations early so that the destination store can
        // start trying to fetch them.
        this.setSetting('recentDestinations', recentDestinations);
        savedSettings.recentDestinations = recentDestinations;
        this.stickySettings_ = savedSettings;
    }
    /**
     * Helper function for configurePolicySetting_(). Sets value and managed flag
     * for given setting.
     * @param settingName Name of the setting being applied.
     * @param value Value of the setting provided via policy.
     * @param managed Flag showing whether value of setting is managed.
     * @param applyOnDestinationUpdate Flag showing whether policy
     *     should be applied on every destination update.
     */
    setPolicySetting_(settingName, value, managed, applyOnDestinationUpdate) {
        if (!this.policySettings_) {
            this.policySettings_ = {};
        }
        this.policySettings_[settingName] = {
            value: value,
            managed: managed,
            applyOnDestinationUpdate: applyOnDestinationUpdate,
        };
    }
    /**
     * Helper function for setPolicySettings(). Calculates value and managed flag
     * of the setting according to allowed and default modes.
     */
    configurePolicySetting_(settingName, allowedMode, defaultMode) {
        switch (settingName) {
            case 'headerFooter': {
                const value = allowedMode !== undefined ? allowedMode : defaultMode;
                if (value !== undefined) {
                    this.setPolicySetting_(settingName, value, allowedMode !== undefined, 
                    /*applyOnDestinationUpdate=*/ false);
                }
                break;
            }
            case 'cssBackground': {
                const value = allowedMode ? allowedMode : defaultMode;
                if (value !== undefined) {
                    this.setPolicySetting_(settingName, value === BackgroundGraphicsModeRestriction.ENABLED, !!allowedMode, /*applyOnDestinationUpdate=*/ false);
                }
                break;
            }
            case 'mediaSize': {
                if (defaultMode !== undefined) {
                    this.setPolicySetting_(settingName, defaultMode, /*managed=*/ false, 
                    /*applyOnDestinationUpdate=*/ true);
                }
                break;
            }
            // 
            case 'printPdfAsImageAvailability': {
                const value = allowedMode !== undefined ? allowedMode : defaultMode;
                if (value !== undefined) {
                    this.setPolicySetting_(settingName, value, /*managed=*/ false, 
                    /*applyOnDestinationUpdate=*/ false);
                }
                break;
            }
            // 
            case 'printPdfAsImage': {
                if (defaultMode !== undefined) {
                    this.setPolicySetting_(settingName, defaultMode, /*managed=*/ false, 
                    /*applyOnDestinationUpdate=*/ false);
                }
                break;
            }
        }
    }
    /**
     * Sets settings in accordance to policies from native code, and prevents
     * those settings from being changed via other means.
     */
    setPolicySettings(policies) {
        if (policies === undefined) {
            return;
        }
        const policiesObject = policies;
        ['headerFooter', 'cssBackground', 'mediaSize'].forEach(settingName => {
            if (!policiesObject[settingName]) {
                return;
            }
            const defaultMode = policiesObject[settingName].defaultMode;
            const allowedMode = policiesObject[settingName].allowedMode;
            this.configurePolicySetting_(settingName, allowedMode, defaultMode);
        });
        // 
        if (policies['printPdfAsImageAvailability']) {
            const allowedMode = policies['printPdfAsImageAvailability'].allowedMode;
            this.configurePolicySetting_('printPdfAsImageAvailability', allowedMode, /*defaultMode=*/ false);
        }
        // 
        if (policies['printPdfAsImage']) {
            const defaultMode = policies['printPdfAsImage'].defaultMode;
            this.configurePolicySetting_('printPdfAsImage', /*allowedMode=*/ undefined, defaultMode);
        }
    }
    applyStickySettings() {
        if (this.stickySettings_) {
            STICKY_SETTING_NAMES.forEach(settingName => {
                const stickySettingsKey = this.getSetting(settingName).key;
                const value = this.stickySettings_[stickySettingsKey];
                if (value !== undefined) {
                    this.setSetting(settingName, value);
                }
                else {
                    this.applyScalingStickySettings_(settingName);
                }
            });
        }
        this.applyPersistentCddDefaults_();
        this.applyPolicySettings_();
        this.initialized_ = true;
        this.updateManaged_();
        this.stickySettings_ = null;
        this.fire('sticky-setting-changed', this.getStickySettings_());
    }
    /**
     * Helper function for applyStickySettings(). Checks if the setting
     * is a scaling setting and applies by applying the old types
     * that rely on 'fitToPage' and 'customScaling'.
     * @param settingName Name of the setting being applied.
     */
    applyScalingStickySettings_(settingName) {
        // TODO(dhoss): Remove checks for 'customScaling' and 'fitToPage'
        if (settingName === 'scalingType' &&
            'customScaling' in this.stickySettings_) {
            const isCustom = this.stickySettings_['customScaling'];
            const scalingType = isCustom ? ScalingType.CUSTOM : ScalingType.DEFAULT;
            this.setSetting(settingName, scalingType);
        }
        else if (settingName === 'scalingTypePdf') {
            if ('isFitToPageEnabled' in this.stickySettings_) {
                const isFitToPage = this.stickySettings_['isFitToPageEnabled'];
                const scalingTypePdf = isFitToPage ?
                    ScalingType.FIT_TO_PAGE :
                    this.getSetting('scalingType').value;
                this.setSetting(settingName, scalingTypePdf);
            }
            else if (this.getSetting('scalingType').value === ScalingType.CUSTOM) {
                // In the event that 'isFitToPageEnabled' was not in the sticky
                // settings, and 'scalingType' has been set to custom, we want
                // 'scalingTypePdf' to match.
                this.setSetting(settingName, ScalingType.CUSTOM);
            }
        }
    }
    applyPolicySettings_() {
        if (this.policySettings_) {
            for (const [settingName, policy] of Object.entries(this.policySettings_)) {
                const policyEntry = policy;
                // 
                if (settingName === 'printPdfAsImageAvailability') {
                    this.updateRasterizeAvailable_();
                    if (this.settings_.rasterize.available) {
                        // If rasterize is available then otherOptions must be available.
                        this.setSettingPath_('otherOptions.available', true);
                    }
                    continue;
                }
                // 
                if (settingName === 'printPdfAsImage') {
                    if (policyEntry.value) {
                        this.setSetting('rasterize', policyEntry.value, true);
                    }
                    continue;
                }
                if (policyEntry.value !== undefined &&
                    !policyEntry.applyOnDestinationUpdate) {
                    const settingKey = settingName;
                    this.setSetting(settingKey, policyEntry.value, true);
                    if (policyEntry.managed) {
                        this.settings_[settingKey].setByGlobalPolicy = true;
                    }
                }
            }
        }
    }
    /**
     * If the setting has a default value specified in the CDD capabilities and
     * the attribute `reset_to_default` is true, this method will return the
     * default value for the setting; otherwise it will return null.
     */
    getResetValue_(capability) {
        if (!capability.reset_to_default) {
            return null;
        }
        const cddDefault = capability.option.find(o => !!o.is_default);
        if (!cddDefault) {
            return null;
        }
        return cddDefault;
    }
    /**
     * For PrinterProvider printers, it's possible to specify for a setting to
     * always reset to the default value using the `reset_to_default` attribute.
     * If `reset_to_default` is true and a default value for the
     * setting is specified, this method will reset the setting
     * value to the default value.
     */
    applyPersistentCddDefaults_() {
        if (!this.destination || !this.destination.isExtension) {
            return;
        }
        const caps = this.destination && this.destination.capabilities ?
            this.destination.capabilities.printer :
            null;
        if (!caps) {
            return;
        }
        if (this.settings_.mediaSize.available) {
            const cddDefault = this.getResetValue_(caps['media_size']);
            if (cddDefault) {
                this.setSettingPath_('mediaSize.value', cddDefault);
            }
        }
        if (this.settings_.color.available) {
            const cddDefault = this.getResetValue_(caps['color']);
            if (cddDefault) {
                this.setSettingPath_('color.value', !['STANDARD_MONOCHROME', 'CUSTOM_MONOCHROME'].includes(cddDefault.type));
            }
        }
        if (this.settings_.duplex.available) {
            const cddDefault = this.getResetValue_(caps['duplex']);
            if (cddDefault) {
                this.setSettingPath_('duplex.value', cddDefault.type === DuplexType.LONG_EDGE ||
                    cddDefault.type === DuplexType.SHORT_EDGE);
                if (!this.settings_.duplexShortEdge.available) {
                    this.setSettingPath_('duplexShortEdge.value', cddDefault.type === DuplexType.SHORT_EDGE);
                }
            }
        }
        if (this.settings_.dpi.available) {
            const cddDefault = this.getResetValue_(caps['dpi']);
            if (cddDefault) {
                this.setSettingPath_('dpi.value', cddDefault);
            }
        }
    }
    /**
     * Re-applies policies after the destination changes. Necessary for policies
     * that apply to settings where available options are based on the current
     * print destination.
     */
    applyPoliciesOnDestinationUpdate() {
        if (!this.policySettings_ || !this.policySettings_['mediaSize'] ||
            !this.policySettings_['mediaSize'].value ||
            !this.settings_.mediaSize.available) {
            return;
        }
        assert(this.destination);
        const mediaSizePolicy = this.policySettings_['mediaSize'].value;
        const matchingOption = this.destination.getMediaSize(mediaSizePolicy.width, mediaSizePolicy.height);
        if (matchingOption !== undefined) {
            this.setSettingPath_('mediaSize.value', matchingOption);
        }
    }
    updateManaged_() {
        const managedSettings = ['cssBackground', 'headerFooter'];
        this.settingsManaged = managedSettings.some(settingName => {
            const setting = this.getSetting(settingName);
            return setting.available && setting.setByGlobalPolicy;
        });
    }
    initialized() {
        return this.initialized_;
    }
    getStickySettings_() {
        const serialization = {};
        serialization['version'] = 2;
        STICKY_SETTING_NAMES.forEach(settingName => {
            const setting = this.getSetting(settingName);
            if (setting.setFromUi) {
                serialization[setting.key] = setting.value;
            }
        });
        return JSON.stringify(serialization);
    }
    getDuplexMode_() {
        if (!this.getSettingValue('duplex')) {
            return DuplexMode.SIMPLEX;
        }
        return this.getSettingValue('duplexShortEdge') ? DuplexMode.SHORT_EDGE :
            DuplexMode.LONG_EDGE;
    }
    getCddDuplexType_() {
        if (!this.getSettingValue('duplex')) {
            return DuplexType.NO_DUPLEX;
        }
        return this.getSettingValue('duplexShortEdge') ? DuplexType.SHORT_EDGE :
            DuplexType.LONG_EDGE;
    }
    /**
     * Creates a string that represents a print ticket.
     * @param destination Destination to print to.
     * @param openPdfInPreview Whether this print request is to open
     *     the PDF in Preview app (Mac only).
     * @param showSystemDialog Whether this print request is to show
     *     the system dialog.
     * @return Serialized print ticket.
     */
    createPrintTicket(destination, openPdfInPreview, showSystemDialog) {
        const dpi = this.getSettingValue('dpi');
        const scalingSettingKey = this.getSetting('scalingTypePdf').available ?
            'scalingTypePdf' :
            'scalingType';
        const ticket = {
            mediaSize: this.getSettingValue('mediaSize'),
            pageCount: this.getSettingValue('pages').length,
            landscape: this.getSettingValue('layout'),
            color: destination.getNativeColorModel(this.getSettingValue('color')),
            headerFooterEnabled: false, // only used in print preview
            marginsType: this.getSettingValue('margins'),
            duplex: this.getDuplexMode_(),
            copies: this.getSettingValue('copies'),
            collate: this.getSettingValue('collate'),
            shouldPrintBackgrounds: this.getSettingValue('cssBackground'),
            shouldPrintSelectionOnly: false, // only used in print preview
            previewModifiable: this.documentSettings.isModifiable,
            printerType: destination.type,
            rasterizePDF: this.getSettingValue('rasterize'),
            scaleFactor: this.getSettingValue(scalingSettingKey) === ScalingType.CUSTOM ?
                parseInt(this.getSettingValue('scaling'), 10) :
                100,
            scalingType: this.getSettingValue(scalingSettingKey),
            pagesPerSheet: this.getSettingValue('pagesPerSheet'),
            dpiHorizontal: (dpi && 'horizontal_dpi' in dpi) ? dpi.horizontal_dpi : 0,
            dpiVertical: (dpi && 'vertical_dpi' in dpi) ? dpi.vertical_dpi : 0,
            dpiDefault: (dpi && 'is_default' in dpi) ? dpi.is_default : false,
            deviceName: destination.id,
            pageWidth: this.pageSize.width,
            pageHeight: this.pageSize.height,
            showSystemDialog: showSystemDialog,
        };
        if (openPdfInPreview) {
            ticket['openPDFInPreview'] = openPdfInPreview;
        }
        if (this.getSettingValue('margins') === MarginsType.CUSTOM) {
            ticket['marginsCustom'] = this.getSettingValue('customMargins');
        }
        if (destination.isExtension) {
            // TODO(rbpotter): Get local and PDF printers to use the same ticket and
            // send only this ticket instead of nesting it in a larger ticket.
            ticket['ticket'] = this.createCloudJobTicket(destination);
            ticket['capabilities'] = JSON.stringify(destination.capabilities);
        }
        return JSON.stringify(ticket);
    }
    /**
     * Creates an object that represents a Google Cloud Print print ticket.
     * @param destination Destination to print to.
     * @return Google Cloud Print print ticket.
     */
    createCloudJobTicket(destination) {
        assert(destination.isExtension, 'Trying to create a Google Cloud Print print ticket for a local ' +
            ' non-extension destination');
        assert(destination.capabilities, 'Trying to create a Google Cloud Print print ticket for a ' +
            'destination with no print capabilities');
        // Create CJT (Cloud Job Ticket)
        const cjt = { version: '1.0', print: {} };
        if (this.settings_.collate.available) {
            cjt.print.collate = { collate: this.settings_.collate.value };
        }
        if (this.settings_.color.available) {
            const selectedOption = destination.getColor(this.settings_.color.value);
            if (!selectedOption) {
                console.warn('Could not find correct color option');
            }
            else {
                cjt.print.color = { type: selectedOption.type };
                if (selectedOption.hasOwnProperty('vendor_id')) {
                    cjt.print.color.vendor_id = selectedOption.vendor_id;
                }
            }
        }
        else {
            // Always try setting the color in the print ticket, otherwise a
            // reasonable reader of the ticket will have to do more work, or process
            // the ticket sub-optimally, in order to safely handle the lack of a
            // color ticket item.
            const defaultOption = destination.defaultColorOption;
            if (defaultOption) {
                cjt.print.color = { type: defaultOption.type };
                if (defaultOption.hasOwnProperty('vendor_id')) {
                    cjt.print.color.vendor_id = defaultOption.vendor_id;
                }
            }
        }
        if (this.settings_.copies.available) {
            cjt.print.copies = { copies: this.getSettingValue('copies') };
        }
        if (this.settings_.duplex.available) {
            cjt.print.duplex = {
                type: this.getCddDuplexType_(),
            };
        }
        if (this.settings_.mediaSize.available) {
            const mediaValue = this.settings_.mediaSize.value;
            cjt.print.media_size = {
                width_microns: mediaValue.width_microns,
                height_microns: mediaValue.height_microns,
                is_continuous_feed: mediaValue.is_continuous_feed,
                vendor_id: mediaValue.vendor_id,
            };
        }
        if (!this.settings_.layout.available) {
            // In this case "orientation" option is hidden from user, so user can't
            // adjust it for page content, see Landscape.isCapabilityAvailable().
            // We can improve results if we set AUTO here.
            const capability = destination.capabilities.printer ?
                destination.capabilities.printer.page_orientation :
                null;
            if (capability && capability.option &&
                capability.option.some(option => option.type === 'AUTO')) {
                cjt.print.page_orientation = { type: 'AUTO' };
            }
        }
        else {
            cjt.print.page_orientation = {
                type: this.settings_.layout.value ? 'LANDSCAPE' : 'PORTRAIT',
            };
        }
        if (this.settings_.dpi.available) {
            const dpiValue = this.settings_.dpi.value;
            cjt.print.dpi = {
                horizontal_dpi: dpiValue.horizontal_dpi,
                vertical_dpi: dpiValue.vertical_dpi,
                vendor_id: dpiValue.vendor_id,
            };
        }
        if (this.settings_.vendorItems.available) {
            const items = this.settings_.vendorItems.value;
            cjt.print.vendor_ticket_item = [];
            for (const itemId in items) {
                if (items.hasOwnProperty(itemId)) {
                    cjt.print.vendor_ticket_item.push({ id: itemId, value: items[itemId] });
                }
            }
        }
        return JSON.stringify(cjt);
    }
}
customElements.define(PrintPreviewModelElement.is, PrintPreviewModelElement);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var State;
(function (State) {
    State[State["NOT_READY"] = 0] = "NOT_READY";
    State[State["READY"] = 1] = "READY";
    State[State["PRINT_PENDING"] = 2] = "PRINT_PENDING";
    State[State["HIDDEN"] = 3] = "HIDDEN";
    State[State["PRINTING"] = 4] = "PRINTING";
    State[State["SYSTEM_DIALOG"] = 5] = "SYSTEM_DIALOG";
    State[State["ERROR"] = 6] = "ERROR";
    State[State["FATAL_ERROR"] = 7] = "FATAL_ERROR";
    State[State["CLOSING"] = 8] = "CLOSING";
})(State || (State = {}));
/**
 * These values are persisted to logs. New entries should replace MAX_BUCKET but
 * existing entries should not be renumbered and numeric values should never be
 * reused.
 */
var Error$1;
(function (Error) {
    Error[Error["NONE"] = 0] = "NONE";
    Error[Error["INVALID_TICKET"] = 1] = "INVALID_TICKET";
    Error[Error["INVALID_PRINTER"] = 2] = "INVALID_PRINTER";
    Error[Error["NO_DESTINATIONS"] = 3] = "NO_DESTINATIONS";
    Error[Error["PREVIEW_FAILED"] = 4] = "PREVIEW_FAILED";
    Error[Error["PRINT_FAILED"] = 5] = "PRINT_FAILED";
    Error[Error["MAX_BUCKET"] = 6] = "MAX_BUCKET";
})(Error$1 || (Error$1 = {}));
class PrintPreviewStateElement extends CrLitElement {
    static get is() {
        return 'print-preview-state';
    }
    static get properties() {
        return {
            error: {
                type: Number,
                notify: true,
            },
        };
    }
    state_ = State.NOT_READY;
    #error_accessor_storage = Error$1.NONE;
    get error() { return this.#error_accessor_storage; }
    set error(value) { this.#error_accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.sendStateChanged_();
    }
    sendStateChanged_() {
        this.dispatchEvent(new CustomEvent('state-changed', { detail: { value: this.state_ } }));
    }
    transitTo(newState) {
        switch (newState) {
            case (State.NOT_READY):
                assert(this.state_ === State.NOT_READY || this.state_ === State.READY ||
                    this.state_ === State.ERROR);
                break;
            case (State.READY):
                assert(this.state_ === State.ERROR || this.state_ === State.NOT_READY ||
                    this.state_ === State.PRINTING);
                break;
            case (State.PRINT_PENDING):
                assert(this.state_ === State.READY);
                break;
            case (State.HIDDEN):
                assert(this.state_ === State.PRINT_PENDING);
                break;
            case (State.PRINTING):
                assert(this.state_ === State.READY || this.state_ === State.HIDDEN ||
                    this.state_ === State.PRINT_PENDING);
                break;
            case (State.SYSTEM_DIALOG):
                assert(this.state_ !== State.HIDDEN && this.state_ !== State.PRINTING &&
                    this.state_ !== State.CLOSING);
                break;
            case (State.ERROR):
                assert(this.state_ === State.ERROR || this.state_ === State.NOT_READY ||
                    this.state_ === State.READY);
                break;
            case (State.CLOSING):
                assert(this.state_ !== State.HIDDEN);
                break;
        }
        const oldState = this.state_;
        this.state_ = newState;
        if (oldState !== newState) {
            // Fire a manual 'state-changed' event to ensure that all states changes
            // are reported, even if a state is changed twice in the same cycle, which
            // wouldn't be the case if CrLitElement's 'notify: true' was used.
            this.sendStateChanged_();
        }
        if (newState !== State.ERROR && newState !== State.FATAL_ERROR) {
            this.error = Error$1.NONE;
        }
    }
}
customElements.define(PrintPreviewStateElement.is, PrintPreviewStateElement);

const prefersDark = window.matchMedia('(prefers-color-scheme: dark)');
const DarkModeMixin = (superClass) => {
    class DarkModeMixin extends superClass {
        static get properties() {
            return {
                /** Whether or not the OS is in dark mode. */
                inDarkMode: { type: Boolean },
            };
        }
        boundOnChange_ = null;
        #inDarkMode_accessor_storage = prefersDark.matches;
        get inDarkMode() { return this.#inDarkMode_accessor_storage; }
        set inDarkMode(value) { this.#inDarkMode_accessor_storage = value; }
        connectedCallback() {
            super.connectedCallback();
            if (!this.boundOnChange_) {
                this.boundOnChange_ = () => this.onChange_();
            }
            prefersDark.addListener(this.boundOnChange_);
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            prefersDark.removeListener(this.boundOnChange_);
            this.boundOnChange_ = null;
        }
        onChange_() {
            this.inDarkMode = prefersDark.matches;
        }
    }
    return DarkModeMixin;
};
function inDarkMode() {
    return prefersDark.matches;
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Returns true if the contents of the two page ranges are equal.
 */
function areRangesEqual(array1, array2) {
    if (array1.length !== array2.length) {
        return false;
    }
    for (let i = 0; i < array1.length; i++) {
        if (array1[i].from !== array2[i].from ||
            array1[i].to !== array2[i].to) {
            return false;
        }
    }
    return true;
}
/**
 * @param localizedStrings An array of strings with corresponding locales.
 * @param locale Locale to look the string up for.
 * @return A string for the requested {@code locale}. An empty string
 *     if there's no string for the specified locale found.
 */
function getStringForLocale(localizedStrings, locale) {
    locale = locale.toLowerCase();
    for (const localizedString of localizedStrings) {
        if (localizedString.locale.toLowerCase() === locale) {
            return localizedString.value;
        }
    }
    return '';
}
/**
 * @param localizedStrings An array of strings with corresponding locales.
 * @return A string for the current locale. An empty string if there's
 *     no string for the current locale found.
 */
function getStringForCurrentLocale(localizedStrings) {
    // First try to find an exact match and then look for the language only.
    return getStringForLocale(localizedStrings, navigator.language) ||
        getStringForLocale(localizedStrings, navigator.language.split('-')[0]);
}
/**
 * Returns background images (icon and dropdown arrow) for use in a md-select.
 * @param iconset The iconset the icon is in.
 * @param iconName The icon name
 * @param el The element that contains the select.
 * @return String containing inlined SVG of the icon and
 *     url(path_to_arrow) separated by a comma.
 */
function getSelectDropdownBackground(iconset, iconName, el) {
    const serializer = new XMLSerializer();
    const iconElement = iconset.createIcon(iconName);
    assert(iconElement);
    const dark = inDarkMode();
    const fillColor = getComputedStyle(el).getPropertyValue(dark ? '--google-grey-500' : '--google-grey-600');
    iconElement.style.fill = fillColor;
    const serializedIcon = serializer.serializeToString(iconElement);
    const uri = encodeURIComponent(serializedIcon);
    const arrowDownPath = dark ? 'chrome://resources/images/dark/arrow_down.svg' :
        'chrome://resources/images/arrow_down.svg';
    return `url("data:image/svg+xml;charset=utf-8,${uri}"),` +
        `url("${arrowDownPath}")`;
}

let instance$C = null;
function getCss$B() {
    return instance$C || (instance$C = [...[], css `.cr-scrollable{anchor-name:--cr-scrollable;anchor-scope:--cr-scrollable;container-type:scroll-state;overflow:auto}.cr-scrollable-top,.cr-scrollable-top-shadow,.cr-scrollable-bottom{display:none;position:fixed;position-anchor:--cr-scrollable;left:anchor(left);width:anchor-size(width);pointer-events:none;&:where(.force-on){display:block}}.cr-scrollable-top{top:anchor(top);border-top:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:top){display:block}}.cr-scrollable-bottom{bottom:anchor(bottom);border-bottom:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:bottom){display:block}}.cr-scrollable-top-shadow{box-shadow:inset 0 5px 6px -3px rgba(0,0,0,.4);display:block;height:8px;opacity:0;top:anchor(top);transition:opacity 500ms;z-index:1;&:where(.force-on){opacity:1}@container scroll-state(scrollable:top){opacity:1}}`]);
}

let instance$B = null;
function getCss$A() {
    return instance$B || (instance$B = [...[getCss$K(), getCss$F(), getCss$B()], css `dialog{background-color:var(--cr-dialog-background-color,white);border:0;border-radius:var(--cr-dialog-border-radius,8px);bottom:50%;box-shadow:0 0 16px rgba(0,0,0,0.12),0 16px 16px rgba(0,0,0,0.24);color:inherit;line-height:20px;max-height:initial;max-width:initial;overflow-y:hidden;padding:0;position:absolute;top:50%;width:var(--cr-dialog-width,512px)}@media (prefers-color-scheme:dark){dialog{background-color:var(--cr-dialog-background-color,var(--google-grey-900));background-image:linear-gradient(rgba(255,255,255,.04),rgba(255,255,255,.04))}}@media (forced-colors:active){dialog{border:var(--cr-border-hcm)}}dialog[open] #content-wrapper{display:flex;flex-direction:column;max-height:100vh;overflow:auto}.top-container,:host ::slotted([slot=button-container]),:host ::slotted([slot=footer]){flex-shrink:0}dialog::backdrop{background-color:rgba(0,0,0,0.6);bottom:0;left:0;position:fixed;right:0;top:0}:host ::slotted([slot=body]){color:var(--cr-secondary-text-color);padding:0 var(--cr-dialog-body-padding-horizontal,20px)}:host ::slotted([slot=title]){color:var(--cr-primary-text-color);flex:1;font-family:var(--cr-dialog-font-family,inherit);font-size:var(--cr-dialog-title-font-size,calc(15 / 13 * 100%));line-height:1;padding-bottom:var(--cr-dialog-title-slot-padding-bottom,16px);padding-inline-end:var(--cr-dialog-title-slot-padding-end,20px);padding-inline-start:var(--cr-dialog-title-slot-padding-start,20px);padding-top:var(--cr-dialog-title-slot-padding-top,20px)}:host ::slotted([slot=button-container]){display:flex;justify-content:flex-end;padding-bottom:var(--cr-dialog-button-container-padding-bottom,16px);padding-inline-end:var(--cr-dialog-button-container-padding-horizontal,16px);padding-inline-start:var(--cr-dialog-button-container-padding-horizontal,16px);padding-top:var(--cr-dialog-button-container-padding-top,16px)}:host ::slotted([slot=footer]){border-bottom-left-radius:inherit;border-bottom-right-radius:inherit;border-top:1px solid #dbdbdb;margin:0;padding:16px 20px}:host([hide-backdrop]) dialog::backdrop{opacity:0}@media (prefers-color-scheme:dark){:host ::slotted([slot=footer]){border-top-color:var(--cr-separator-color)}}.body-container{box-sizing:border-box;display:flex;flex-direction:column;min-height:1.375rem;overflow:auto}.top-container{align-items:flex-start;display:flex;min-height:var(--cr-dialog-top-container-min-height,31px)}.title-container{display:flex;flex:1;font-size:inherit;font-weight:inherit;margin:0;outline:none}#close{align-self:flex-start;margin-inline-end:4px;margin-top:4px}@container style(--cr-dialog-body-border-top){.cr-scrollable-top{display:block;border-top:var(--cr-dialog-body-border-top)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$z() {
    // clang-format off
    return html `
<dialog id="dialog" @close="${this.onNativeDialogClose_}"
    @cancel="${this.onNativeDialogCancel_}" part="dialog"
    aria-labelledby="title"
    aria-description="${this.ariaDescriptionText || nothing}">
<!-- This wrapper is necessary, such that the "pulse" animation is not
    erroneously played when the user clicks on the outer-most scrollbar. -->
  <div id="content-wrapper" part="wrapper">
    <div class="top-container">
      <h2 id="title" class="title-container" tabindex="-1">
        <slot name="title"></slot>
      </h2>
      ${this.showCloseButton ? html `
        <cr-icon-button id="close" class="icon-clear"
            aria-label="${this.closeText || nothing}"
            title="${this.closeText || nothing}"
            @click="${this.cancel}" @keypress="${this.onCloseKeypress_}">
        </cr-icon-button>
       ` : ''}
    </div>
    <slot name="header"></slot>
    <div class="body-container cr-scrollable" id="container"
        part="body-container">
      <div class="cr-scrollable-top"></div>
      <slot name="body"></slot>
      <div class="cr-scrollable-bottom"></div>
    </div>
    <slot name="button-container"></slot>
    <slot name="footer"></slot>
  </div>
</dialog>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-dialog' is a component for showing a modal dialog. If the
 * dialog is closed via close(), a 'close' event is fired. If the dialog is
 * canceled via cancel(), a 'cancel' event is fired followed by a 'close' event.
 *
 * Additionally clients can get a reference to the internal native <dialog> via
 * calling getNative() and inspecting the |returnValue| property inside
 * the 'close' event listener to determine whether it was canceled or just
 * closed, where a truthy value means success, and a falsy value means it was
 * canceled.
 *
 * Note that <cr-dialog> wrapper itself always has 0x0 dimensions, and
 * specifying width/height on <cr-dialog> directly will have no effect on the
 * internal native <dialog>. Instead use cr-dialog::part(dialog) to specify
 * width/height (as well as other available mixins to style other parts of the
 * dialog contents).
 */
class CrDialogElement extends CrLitElement {
    static get is() {
        return 'cr-dialog';
    }
    static get styles() {
        return getCss$A();
    }
    render() {
        return getHtml$z.bind(this)();
    }
    static get properties() {
        return {
            open: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Alt-text for the dialog close button.
             */
            closeText: { type: String },
            /**
             * True if the dialog should remain open on 'popstate' events. This is
             * used for navigable dialogs that have their separate navigation handling
             * code.
             */
            ignorePopstate: { type: Boolean },
            /**
             * True if the dialog should ignore 'Enter' keypresses.
             */
            ignoreEnterKey: { type: Boolean },
            /**
             * True if the dialog should consume 'keydown' events. If ignoreEnterKey
             * is true, 'Enter' key won't be consumed.
             */
            consumeKeydownEvent: { type: Boolean },
            /**
             * True if the dialog should not be able to be cancelled, which will
             * prevent 'Escape' key presses from closing the dialog.
             */
            noCancel: { type: Boolean },
            // True if dialog should show the 'X' close button.
            showCloseButton: { type: Boolean },
            showOnAttach: { type: Boolean },
            /**
             * Text for the aria description.
             */
            ariaDescriptionText: { type: String },
        };
    }
    #closeText_accessor_storage;
    get closeText() { return this.#closeText_accessor_storage; }
    set closeText(value) { this.#closeText_accessor_storage = value; }
    #consumeKeydownEvent_accessor_storage = false;
    get consumeKeydownEvent() { return this.#consumeKeydownEvent_accessor_storage; }
    set consumeKeydownEvent(value) { this.#consumeKeydownEvent_accessor_storage = value; }
    #ignoreEnterKey_accessor_storage = false;
    get ignoreEnterKey() { return this.#ignoreEnterKey_accessor_storage; }
    set ignoreEnterKey(value) { this.#ignoreEnterKey_accessor_storage = value; }
    #ignorePopstate_accessor_storage = false;
    get ignorePopstate() { return this.#ignorePopstate_accessor_storage; }
    set ignorePopstate(value) { this.#ignorePopstate_accessor_storage = value; }
    #noCancel_accessor_storage = false;
    get noCancel() { return this.#noCancel_accessor_storage; }
    set noCancel(value) { this.#noCancel_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #showCloseButton_accessor_storage = false;
    get showCloseButton() { return this.#showCloseButton_accessor_storage; }
    set showCloseButton(value) { this.#showCloseButton_accessor_storage = value; }
    #showOnAttach_accessor_storage = false;
    get showOnAttach() { return this.#showOnAttach_accessor_storage; }
    set showOnAttach(value) { this.#showOnAttach_accessor_storage = value; }
    #ariaDescriptionText_accessor_storage;
    get ariaDescriptionText() { return this.#ariaDescriptionText_accessor_storage; }
    set ariaDescriptionText(value) { this.#ariaDescriptionText_accessor_storage = value; }
    mutationObserver_ = null;
    boundKeydown_ = null;
    firstUpdated() {
        // If the active history entry changes (i.e. user clicks back button),
        // all open dialogs should be cancelled.
        window.addEventListener('popstate', () => {
            if (!this.ignorePopstate && this.$.dialog.open) {
                this.cancel();
            }
        });
        if (!this.ignoreEnterKey) {
            this.addEventListener('keypress', this.onKeypress_.bind(this));
        }
        this.addEventListener('pointerdown', e => this.onPointerdown_(e));
    }
    connectedCallback() {
        super.connectedCallback();
        const mutationObserverCallback = () => {
            if (this.$.dialog.open) {
                this.addKeydownListener_();
            }
            else {
                this.removeKeydownListener_();
            }
        };
        this.mutationObserver_ = new MutationObserver(mutationObserverCallback);
        this.mutationObserver_.observe(this.$.dialog, {
            attributes: true,
            attributeFilter: ['open'],
        });
        // In some cases dialog already has the 'open' attribute by this point.
        mutationObserverCallback();
        if (this.showOnAttach) {
            this.showModal();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeKeydownListener_();
        if (this.mutationObserver_) {
            this.mutationObserver_.disconnect();
            this.mutationObserver_ = null;
        }
    }
    addKeydownListener_() {
        if (!this.consumeKeydownEvent) {
            return;
        }
        this.boundKeydown_ = this.boundKeydown_ || this.onKeydown_.bind(this);
        this.addEventListener('keydown', this.boundKeydown_);
        // Sometimes <body> is key event's target and in that case the event
        // will bypass cr-dialog. We should consume those events too in order to
        // behave modally. This prevents accidentally triggering keyboard commands.
        document.body.addEventListener('keydown', this.boundKeydown_);
    }
    removeKeydownListener_() {
        if (!this.boundKeydown_) {
            return;
        }
        this.removeEventListener('keydown', this.boundKeydown_);
        document.body.removeEventListener('keydown', this.boundKeydown_);
        this.boundKeydown_ = null;
    }
    async showModal() {
        if (this.showOnAttach) {
            const element = this.querySelector('[autofocus]');
            if (element && element instanceof CrLitElement && !element.shadowRoot) {
                // Force initial render, so that any inner elements with [autofocus] are
                // picked up by the browser.
                element.ensureInitialRender();
            }
        }
        this.$.dialog.showModal();
        assert(this.$.dialog.open);
        this.open = true;
        await this.updateComplete;
        this.fire('cr-dialog-open');
    }
    cancel() {
        this.fire('cancel');
        this.$.dialog.close();
        assert(!this.$.dialog.open);
        this.open = false;
    }
    close() {
        this.$.dialog.close('success');
        assert(!this.$.dialog.open);
        this.open = false;
    }
    /**
     * Set the title of the dialog for a11y reader.
     * @param title Title of the dialog.
     */
    setTitleAriaLabel(title) {
        this.$.dialog.removeAttribute('aria-labelledby');
        this.$.dialog.setAttribute('aria-label', title);
    }
    onCloseKeypress_(e) {
        // Because the dialog may have a default Enter key handler, prevent
        // keypress events from bubbling up from this element.
        e.stopPropagation();
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    async onNativeDialogCancel_(e) {
        // Ignore any 'cancel' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        if (this.noCancel) {
            e.preventDefault();
            return;
        }
        // When the dialog is dismissed using the 'Esc' key, need to manually update
        // the |open| property (since close() is not called).
        this.open = false;
        await this.updateComplete;
        // Catch and re-fire the native 'cancel' event such that it bubbles across
        // Shadow DOM v1.
        this.fire('cancel');
    }
    /**
     * Expose the inner native <dialog> for some rare cases where it needs to be
     * directly accessed (for example to programmatically setheight/width, which
     * would not work on the wrapper).
     */
    getNative() {
        return this.$.dialog;
    }
    onKeypress_(e) {
        if (e.key !== 'Enter') {
            return;
        }
        // Accept Enter keys from either the dialog itself, or a child cr-input,
        // considering that the event may have been retargeted, for example if the
        // cr-input is nested inside another element. Also exclude inputs of type
        // 'search', since hitting 'Enter' on a search field most likely intends to
        // trigger searching.
        const accept = e.target === this ||
            e.composedPath().some(el => el.tagName === 'CR-INPUT' &&
                el.type !== 'search');
        if (!accept) {
            return;
        }
        const actionButton = this.querySelector('.action-button:not([disabled]):not([hidden])');
        if (actionButton) {
            actionButton.click();
            e.preventDefault();
        }
    }
    onKeydown_(e) {
        assert(this.consumeKeydownEvent);
        if (!this.getNative().open) {
            return;
        }
        if (this.ignoreEnterKey && e.key === 'Enter') {
            return;
        }
        // Stop propagation to behave modally.
        e.stopPropagation();
    }
    onPointerdown_(e) {
        // Only show pulse animation if user left-clicked outside of the dialog
        // contents.
        if (e.button !== 0 ||
            e.composedPath()[0].tagName !== 'DIALOG') {
            return;
        }
        this.$.dialog.animate([
            { transform: 'scale(1)', offset: 0 },
            { transform: 'scale(1.02)', offset: 0.4 },
            { transform: 'scale(1.02)', offset: 0.6 },
            { transform: 'scale(1)', offset: 1 },
        ], {
            duration: 180,
            easing: 'ease-in-out',
            iterations: 1,
        });
        // Prevent any text from being selected within the dialog when clicking in
        // the backdrop area.
        e.preventDefault();
    }
    focus() {
        const titleContainer = this.shadowRoot.querySelector('.title-container');
        assert(titleContainer);
        titleContainer.focus();
    }
}
customElements.define(CrDialogElement.is, CrDialogElement);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const WRAPPER_CSS_CLASS = 'search-highlight-wrapper';
const ORIGINAL_CONTENT_CSS_CLASS = 'search-highlight-original-content';
const HIT_CSS_CLASS = 'search-highlight-hit';
const SEARCH_BUBBLE_CSS_CLASS = 'search-bubble';
/**
 * Replaces the the highlight wrappers given in |wrappers| with the original
 * search nodes.
 */
function removeHighlights(wrappers) {
    for (const wrapper of wrappers) {
        // If wrapper is already removed, do nothing.
        if (!wrapper.parentElement) {
            continue;
        }
        const originalContent = wrapper.querySelector(`.${ORIGINAL_CONTENT_CSS_CLASS}`);
        assert(originalContent);
        const textNode = originalContent.firstChild;
        assert(textNode);
        wrapper.parentElement.replaceChild(textNode, wrapper);
    }
}
/**
 * Applies the highlight UI (yellow rectangle) around all matches in |node|.
 * @param node The text node to be highlighted. |node| ends up
 *     being hidden.
 * @return The new highlight wrapper.
 */
function highlight(node, ranges) {
    assert(ranges.length > 0);
    const wrapper = document.createElement('span');
    wrapper.classList.add(WRAPPER_CSS_CLASS);
    // Use existing node as placeholder to determine where to insert the
    // replacement content.
    assert(node.parentNode);
    node.parentNode.replaceChild(wrapper, node);
    // Keep the existing node around for when the highlights are removed. The
    // existing text node might be involved in data-binding and therefore should
    // not be discarded.
    const span = document.createElement('span');
    span.classList.add(ORIGINAL_CONTENT_CSS_CLASS);
    span.style.display = 'none';
    span.appendChild(node);
    wrapper.appendChild(span);
    const text = node.textContent;
    const tokens = [];
    for (let i = 0; i < ranges.length; ++i) {
        const range = ranges[i];
        const prev = ranges[i - 1] || { start: 0, length: 0 };
        const start = prev.start + prev.length;
        const length = range.start - start;
        tokens.push(text.substr(start, length));
        tokens.push(text.substr(range.start, range.length));
    }
    const last = ranges.slice(-1)[0];
    tokens.push(text.substr(last.start + last.length));
    for (let i = 0; i < tokens.length; ++i) {
        if (i % 2 === 0) {
            wrapper.appendChild(document.createTextNode(tokens[i]));
        }
        else {
            const hitSpan = document.createElement('span');
            hitSpan.classList.add(HIT_CSS_CLASS);
            // Defaults to the color associated with --paper-yellow-500.
            hitSpan.style.backgroundColor =
                'var(--search-highlight-hit-background-color, #ffeb3b)';
            // Defaults to the color associated with --google-grey-900.
            hitSpan.style.color = 'var(--search-highlight-hit-color, #202124)';
            hitSpan.textContent = tokens[i];
            wrapper.appendChild(hitSpan);
        }
    }
    return wrapper;
}
/**
 * Creates an empty search bubble (styled HTML element without text).
 * |node| should already be visible or the bubble will render incorrectly.
 * @param node The node to be highlighted.
 * @param horizontallyCenter Whether or not to horizontally center
 *     the shown search bubble (if any) based on |node|'s left and width.
 * @return The search bubble that was added, or null if no new
 *     bubble was added.
 */
function createEmptySearchBubble(node, horizontallyCenter) {
    let anchor = node;
    if (node.nodeName === 'SELECT') {
        anchor = node.parentNode;
    }
    if (anchor instanceof ShadowRoot) {
        anchor = anchor.host.parentNode;
    }
    let searchBubble = anchor
        .querySelector(`.${SEARCH_BUBBLE_CSS_CLASS}`);
    // If the node has already been highlighted, there is no need to do
    // anything.
    if (searchBubble) {
        return searchBubble;
    }
    searchBubble = document.createElement('div');
    searchBubble.classList.add(SEARCH_BUBBLE_CSS_CLASS);
    const innards = document.createElement('div');
    innards.classList.add('search-bubble-innards');
    innards.textContent = '\u00a0'; // Non-breaking space for offsetHeight.
    searchBubble.appendChild(innards);
    anchor.appendChild(searchBubble);
    const updatePosition = function () {
        const nodeEl = node;
        assert(searchBubble);
        assert(typeof nodeEl.offsetTop === 'number');
        searchBubble.style.top = nodeEl.offsetTop +
            (innards.classList.contains('above') ? -searchBubble.offsetHeight :
                nodeEl.offsetHeight) +
            'px';
    };
    updatePosition();
    searchBubble.addEventListener('mouseover', function () {
        innards.classList.toggle('above');
        updatePosition();
    });
    // TODO(crbug.com/41096577): create a way to programmatically update these
    // bubbles (i.e. call updatePosition()) when outer scope knows they need to
    // be repositioned.
    return searchBubble;
}
function stripDiacritics(text) {
    return text.normalize('NFD').replace(/[\u0300-\u036f]/g, '');
}

let instance$A = null;
function getCss$z() {
    return instance$A || (instance$A = [...[], css `.md-select{--md-arrow-width:7px;--md-select-bg-color:transparent;--md-select-option-bg-color:white;--md-select-side-padding:10px;--md-select-text-color:inherit;-webkit-appearance:none;background:url(//resources/images/arrow_down.svg) calc(100% - var(--md-select-side-padding)) center no-repeat;background-color:var(--md-select-bg-color);background-size:var(--md-arrow-width);border:solid 1px var(--color-combobox-container-outline,var(--cr-fallback-color-neutral-outline));border-radius:8px;box-sizing:border-box;color:var(--md-select-text-color);cursor:pointer;font-family:inherit;font-size:12px;height:36px;max-width:100%;outline:none;padding-block-end:0;padding-block-start:0;padding-inline-end:calc(var(--md-select-side-padding) + var(--md-arrow-width) + 3px);padding-inline-start:var(--md-select-side-padding);width:var(--md-select-width,200px)}@media (prefers-color-scheme:dark){.md-select{--md-select-option-bg-color:var(--google-grey-900-white-4-percent);background-image:url(//resources/images/dark/arrow_down.svg)}}.md-select:hover{background-color:var(--color-comboxbox-ink-drop-hovered,var(--cr-hover-on-subtle-background-color))}.md-select :-webkit-any(option,optgroup){background-color:var(--md-select-option-bg-color)}.md-select[disabled]{background-color:var(--color-combobox-background-disabled,var(--cr-fallback-color-disabled-background));border-color:transparent;color:var(--color-textfield-foreground-disabled,var(--cr-fallback-color-disabled-foreground));opacity:1;pointer-events:none}.md-select:focus{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-1px}:host-context([dir=rtl]) .md-select{background-position-x:var(--md-select-side-padding)}`]);
}

let instance$z = null;
function getCss$y() {
    return instance$z || (instance$z = [...[], css `.search-bubble{--search-bubble-color:#ffeb3b;position:absolute;z-index:1}.search-bubble-innards{align-items:center;background-color:var(--search-bubble-color);border-radius:2px;color:var(--google-grey-900);max-width:100px;min-width:64px;overflow:hidden;padding:4px 10px;text-align:center;text-overflow:ellipsis;white-space:nowrap}.search-bubble-innards::after{background-color:var(--search-bubble-color);content:'';height:10px;left:calc(50% - 5px);position:absolute;top:-5px;transform:rotate(-45deg);width:10px;z-index:-1}.search-bubble-innards.above::after{bottom:-5px;top:auto;transform:rotate(-135deg)}`]);
}

const sheet = new CSSStyleSheet();
sheet.replaceSync(`html{--print-preview-row-height:38px;--print-preview-sidebar-width:384px;--print-preview-title-width:120px;--print-preview-sidebar-margin:24px;--print-preview-dropdown-width:calc(var(--print-preview-sidebar-width) - var(--print-preview-title-width) - 3 * var(--print-preview-sidebar-margin));--print-preview-settings-border:1px solid var(--google-grey-200);--print-preview-dialog-margin:34px;--cr-form-field-label-height:initial;--cr-form-field-label-line-height:.75rem;--destination-item-height:32px;--preview-area-background-color:var(--google-grey-300);--iron-icon-fill-color:var(--google-grey-700);--iron-icon-height:var(--cr-icon-size);--iron-icon-width:var(--cr-icon-size);--search-icon-size:32px;--throbber-size:16px}@media (prefers-color-scheme:dark){html{--preview-area-background-color:var(--google-grey-700);--print-preview-settings-border:var(--cr-separator-line);--iron-icon-fill-color:var(--google-grey-500)}}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet];

let instance$y = null;
function getCss$x() {
    return instance$y || (instance$y = [...[getCss$E()], css `select.md-select{margin-bottom:2px;margin-top:2px;min-height:32px;padding-inline-end:32px;user-select:none;--md-select-width:calc(100% - 2 * var(--print-preview-sidebar-margin))}.checkbox cr-checkbox{min-height:var(--print-preview-row-height)}.checkbox cr-checkbox::part(label-container){overflow:hidden;padding-inline-start:16px}cr-input{line-height:20px}cr-input::part(row-container){min-height:var(--print-preview-row-height)}print-preview-settings-section [slot=controls]>*{margin:0 var(--print-preview-sidebar-margin)}cr-dialog::part(wrapper){max-height:calc(100vh - 68px);max-width:100%;width:calc(100vw - 68px)}cr-dialog [slot=body]{box-sizing:border-box}#dialog div[slot='title']{padding-bottom:8px}#dialog div[slot='button-container']{align-items:center;box-shadow:0 -1px 1px 0 var(--cr-separator-color);min-height:64px;padding-bottom:0;padding-top:0}`]);
}

let instance$x = null;
function getCss$w() {
    return instance$x || (instance$x = [...[getCss$x(), getCss$z(), getCss$y(), getCss$K()], css `:host{display:flex;min-height:54px;position:relative}:host>*{overflow:hidden;text-overflow:ellipsis;vertical-align:middle}.label,.value{align-self:center;color:var(--cr-primary-text-color);overflow:hidden}.label{flex:1;min-width:96px;opacity:0.87}.value{flex:0;min-width:239px;text-overflow:ellipsis;white-space:nowrap}.value>*{display:flex;margin-inline-start:10px}cr-input{width:239px;--cr-input-error-display:none}select.md-select{font-size:1em}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$y() {
    // clang-format off
    return html `<!--_html_template_start_-->
<label class="label searchable">${this.getDisplayName_(this.capability)}</label>
<div class="value">
  ${this.isCapabilityTypeSelect_() ? html `
    <div>
      <select class="md-select" @change="${this.onUserInput_}">
        ${this.capability.select_cap.option.map(item => html `
          <option class="searchable" value="${item.value}"
              ?selected="${this.isOptionSelected_(item)}">
             ${this.getDisplayName_(item)}
          </option>
        `)}
      </select>
    </div>
  ` : ''}
  <span ?hidden="${!this.isCapabilityTypeInput_()}">
    <cr-input type="text" @input="${this.onUserInput_}" spellcheck="false"
        placeholder="${this.getCapabilityPlaceholder_()}">
    </cr-input>
  </span>
  <span ?hidden="${!this.isCapabilityTypeCheckbox_()}">
    <cr-checkbox @change="${this.onCheckboxInput_}"
        ?checked="${this.isChecked_()}">
    </cr-checkbox>
  </span>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @param element The element to update. Element should have a shadow root.
 * @param query The current search query
 * @param bubbles A map of bubbles created / results found so far.
 * @return The highlight wrappers that were created.
 */
function updateHighlights(element, query, bubbles) {
    const highlights = [];
    if (!query) {
        return highlights;
    }
    assert(query.global);
    element.shadowRoot.querySelectorAll('.searchable').forEach(childElement => {
        childElement.childNodes.forEach(node => {
            if (node.nodeType !== Node.TEXT_NODE) {
                return;
            }
            const textContent = node.nodeValue;
            if (textContent.trim().length === 0) {
                return;
            }
            const strippedText = stripDiacritics(textContent);
            const ranges = [];
            for (let match; match = query.exec(strippedText);) {
                ranges.push({ start: match.index, length: match[0].length });
            }
            if (ranges.length > 0) {
                // Don't highlight <select> nodes, yellow rectangles can't be
                // displayed within an <option>.
                if (node.parentNode.nodeName === 'OPTION') {
                    // The bubble should be parented by the select node's parent.
                    // Note: The bubble's ::after element, a yellow arrow, will not
                    // appear correctly in print preview without SPv175 enabled. See
                    // https://crbug.com/817058.
                    // TODO(crbug.com/40666299): turn on horizontallyCenter when we fix
                    // incorrect positioning caused by scrollbar width changing after
                    // search finishes.
                    assert(node.parentNode);
                    assert(node.parentNode.parentNode);
                    const bubble = createEmptySearchBubble(node.parentNode.parentNode);
                    const numHits = ranges.length + (bubbles.get(bubble) || 0);
                    bubbles.set(bubble, numHits);
                    const msgName = numHits === 1 ? 'searchResultBubbleText' :
                        'searchResultsBubbleText';
                    bubble.firstChild.textContent =
                        loadTimeData.getStringF(msgName, numHits);
                }
                else {
                    highlights.push(highlight(node, ranges));
                }
            }
        });
    });
    return highlights;
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const SettingsMixin = (superClass) => {
    class SettingsMixin extends superClass {
        observers_ = [];
        model_ = null;
        connectedCallback() {
            super.connectedCallback();
            // Cache this reference, so that the same one can be used in
            // disconnectedCallback(), othehrwise if `model_` has already been removed
            // from the DOM, getInstance() will throw an error.
            this.model_ = getInstance();
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            assert(this.model_);
            if (this.model_.isConnected) {
                // Only remove observers if the PrintPreviewModelElement original
                // singleton instance is still connected to the DOM. Otherwise all
                // observers have already been remomved in PrintPreviewModelElement's
                // disconnectedCallback.
                for (const id of this.observers_) {
                    const removed = this.model_.observable.removeObserver(id);
                    assert(removed);
                }
            }
            this.model_ = null;
            this.observers_ = [];
        }
        addSettingObserver(path, callback) {
            const id = getInstance().observable.addObserver(path, callback);
            this.observers_.push(id);
        }
        getSetting(settingName) {
            return getInstance().getSetting(settingName);
        }
        getSettingValue(settingName) {
            return getInstance().getSettingValue(settingName);
        }
        setSetting(settingName, value, noSticky) {
            getInstance().setSetting(settingName, value, noSticky);
        }
        setSettingValid(settingName, valid) {
            getInstance().setSettingValid(settingName, valid);
        }
    }
    return SettingsMixin;
};

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewAdvancedSettingsItemElementBase = SettingsMixin(CrLitElement);
class PrintPreviewAdvancedSettingsItemElement extends PrintPreviewAdvancedSettingsItemElementBase {
    static get is() {
        return 'print-preview-advanced-settings-item';
    }
    static get styles() {
        return getCss$w();
    }
    render() {
        return getHtml$y.bind(this)();
    }
    static get properties() {
        return {
            capability: { type: Object },
            currentValue_: { type: String },
        };
    }
    #capability_accessor_storage = { id: '', type: '' };
    get capability() { return this.#capability_accessor_storage; }
    set capability(value) { this.#capability_accessor_storage = value; }
    #currentValue__accessor_storage = '';
    get currentValue_() { return this.#currentValue__accessor_storage; }
    set currentValue_(value) { this.#currentValue__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('vendorItems.value', () => this.updateFromSettings_());
        this.updateFromSettings_();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('capability')) {
            this.updateFromSettings_();
        }
    }
    updateFromSettings_() {
        const settings = this.getSetting('vendorItems').value;
        // The settings may not have a property with the id if they were populated
        // from sticky settings from a different destination or if the
        // destination's capabilities changed since the sticky settings were
        // generated.
        if (!settings.hasOwnProperty(this.capability.id)) {
            return;
        }
        const value = settings[this.capability.id];
        if (this.isCapabilityTypeSelect_()) {
            // Ignore a value that can't be selected.
            if (this.hasOptionWithValue_(value)) {
                this.currentValue_ = value;
            }
        }
        else {
            this.currentValue_ = value;
            this.shadowRoot.querySelector('cr-input').value = this.currentValue_;
        }
    }
    /**
     * @return The display name for the setting.
     */
    getDisplayName_(item) {
        let displayName = item.display_name;
        if (!displayName && item.display_name_localized) {
            displayName = getStringForCurrentLocale(item.display_name_localized);
        }
        return displayName || '';
    }
    /**
     * @return Whether the capability represented by this item is of type select.
     */
    isCapabilityTypeSelect_() {
        return this.capability.type === 'SELECT';
    }
    /**
     * @return Whether the capability represented by this item is of type
     *     checkbox.
     */
    isCapabilityTypeCheckbox_() {
        return this.capability.type === 'TYPED_VALUE' &&
            this.capability.typed_value_cap.value_type === 'BOOLEAN';
    }
    /**
     * @return Whether the capability represented by this item is of type input.
     */
    isCapabilityTypeInput_() {
        return !this.isCapabilityTypeSelect_() && !this.isCapabilityTypeCheckbox_();
    }
    /**
     * @return Whether the checkbox setting is checked.
     */
    isChecked_() {
        return this.currentValue_ === 'true';
    }
    /**
     * @param option The option for a select capability.
     * @return Whether the option is selected.
     */
    isOptionSelected_(option) {
        return this.currentValue_ === undefined ?
            !!option.is_default :
            option.value === this.currentValue_;
    }
    /**
     * @return The placeholder value for the capability's text input.
     */
    getCapabilityPlaceholder_() {
        if (this.capability.type === 'TYPED_VALUE' &&
            this.capability.typed_value_cap &&
            this.capability.typed_value_cap.default !== undefined) {
            return this.capability.typed_value_cap.default.toString() || '';
        }
        if (this.capability.type === 'RANGE' && this.capability.range_cap &&
            this.capability.range_cap.default !== undefined) {
            return this.capability.range_cap.default.toString() || '';
        }
        return '';
    }
    hasOptionWithValue_(value) {
        return !!this.capability.select_cap &&
            !!this.capability.select_cap.option &&
            this.capability.select_cap.option.some(option => option.value === value);
    }
    /**
     * @param query The current search query.
     * @return Whether the item has a match for the query.
     */
    hasMatch(query) {
        if (!query) {
            return true;
        }
        const strippedCapabilityName = stripDiacritics(this.getDisplayName_(this.capability));
        if (strippedCapabilityName.match(query)) {
            return true;
        }
        if (!this.isCapabilityTypeSelect_()) {
            return false;
        }
        for (const option of this.capability.select_cap.option) {
            const strippedOptionName = stripDiacritics(this.getDisplayName_(option));
            if (strippedOptionName.match(query)) {
                return true;
            }
        }
        return false;
    }
    onUserInput_(e) {
        this.currentValue_ = e.target.value;
    }
    onCheckboxInput_(e) {
        this.currentValue_ =
            e.target.checked ? 'true' : 'false';
    }
    /**
     * @return The current value of the setting, or the empty string if it is not
     *     set.
     */
    getCurrentValue() {
        return this.currentValue_ || '';
    }
    /**
     * Only used in tests.
     * @param value A value to set the setting to.
     */
    setCurrentValueForTest(value) {
        this.currentValue_ = value;
    }
    /**
     * @return The highlight wrappers and that were created.
     */
    updateHighlighting(query, bubbles) {
        return updateHighlights(this, query, bubbles);
    }
}
customElements.define(PrintPreviewAdvancedSettingsItemElement.is, PrintPreviewAdvancedSettingsItemElement);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Helper functions for implementing an incremental search field. See
 * <settings-subpage-search> for a simple implementation.
 */
const CrSearchFieldMixinLit = (superClass) => {
    class CrSearchFieldMixinLit extends superClass {
        static get properties() {
            return {
                // Prompt text to display in the search field.
                label: {
                    type: String,
                },
                // Tooltip to display on the clear search button.
                clearLabel: {
                    type: String,
                },
                hasSearchText: {
                    type: Boolean,
                    reflect: true,
                },
            };
        }
        #label_accessor_storage = '';
        get label() { return this.#label_accessor_storage; }
        set label(value) { this.#label_accessor_storage = value; }
        #clearLabel_accessor_storage = '';
        get clearLabel() { return this.#clearLabel_accessor_storage; }
        set clearLabel(value) { this.#clearLabel_accessor_storage = value; }
        #hasSearchText_accessor_storage = false;
        get hasSearchText() { return this.#hasSearchText_accessor_storage; }
        set hasSearchText(value) { this.#hasSearchText_accessor_storage = value; }
        effectiveValue_ = '';
        searchDelayTimer_ = -1;
        /**
         * @return The input field element the behavior should use.
         */
        getSearchInput() {
            assertNotReached();
        }
        /**
         * @return The value of the search field.
         */
        getValue() {
            return this.getSearchInput().value;
        }
        /**
         * Sets the value of the search field.
         * @param noEvent Whether to prevent a 'search-changed' event
         *     firing for this change.
         */
        setValue(value, noEvent) {
            const updated = this.updateEffectiveValue_(value);
            this.getSearchInput().value = this.effectiveValue_;
            if (!updated) {
                // If the input is only whitespace and value is empty,
                // |hasSearchText| needs to be updated.
                if (value === '' && this.hasSearchText) {
                    this.hasSearchText = false;
                }
                return;
            }
            this.onSearchTermInput();
            if (!noEvent) {
                this.fire('search-changed', this.effectiveValue_);
            }
        }
        scheduleSearch_() {
            if (this.searchDelayTimer_ >= 0) {
                clearTimeout(this.searchDelayTimer_);
            }
            // Dispatch 'search' event after:
            //    0ms if the value is empty
            //  500ms if the value length is 1
            //  400ms if the value length is 2
            //  300ms if the value length is 3
            //  200ms if the value length is 4 or greater.
            // The logic here was copied from WebKit's native 'search' event.
            const length = this.getValue().length;
            const timeoutMs = length > 0 ? (500 - 100 * (Math.min(length, 4) - 1)) : 0;
            this.searchDelayTimer_ = setTimeout(() => {
                this.getSearchInput().dispatchEvent(new CustomEvent('search', { composed: true, detail: this.getValue() }));
                this.searchDelayTimer_ = -1;
            }, timeoutMs);
        }
        onSearchTermSearch() {
            this.onValueChanged_(this.getValue(), false);
        }
        /**
         * Update the state of the search field whenever the underlying input
         * value changes. Unlike onsearch or onkeypress, this is reliably called
         * immediately after any change, whether the result of user input or JS
         * modification.
         */
        onSearchTermInput() {
            this.hasSearchText = this.getSearchInput().value !== '';
            this.scheduleSearch_();
        }
        /**
         * Updates the internal state of the search field based on a change that
         * has already happened.
         * @param noEvent Whether to prevent a 'search-changed' event
         *     firing for this change.
         */
        onValueChanged_(newValue, noEvent) {
            const updated = this.updateEffectiveValue_(newValue);
            if (updated && !noEvent) {
                this.fire('search-changed', this.effectiveValue_);
            }
        }
        /**
         * Trim leading whitespace and replace consecutive whitespace with
         * single space. This will prevent empty string searches and searches
         * for effectively the same query.
         */
        updateEffectiveValue_(value) {
            const effectiveValue = value.replace(/\s+/g, ' ').replace(/^\s/, '');
            if (effectiveValue === this.effectiveValue_) {
                return false;
            }
            this.effectiveValue_ = effectiveValue;
            return true;
        }
    }
    return CrSearchFieldMixinLit;
};

let instance$w = null;
function getCss$v() {
    return instance$w || (instance$w = [...[getCss$x(), getCss$D(), getCss$F()], css `:host{display:flex;--cr-input-error-display:none}cr-input::part(row-container){min-height:32px}#icon,#clearSearch{margin-inline-end:0;margin-inline-start:0}#icon{height:var(--search-icon-size);width:var(--search-icon-size)}#clearSearch{--clear-icon-size:28px;--cr-icon-button-size:var(--clear-icon-size);--cr-icon-button-icon-size:20px;height:var(--clear-icon-size);position:absolute;right:0;width:var(--clear-icon-size);z-index:1}:host-context([dir=rtl]) #clearSearch{left:0;right:auto}:host([has-search-text]) cr-input{--cr-input-padding-end:24px}.search-box-input{width:100%}.search-box-input::-webkit-search-cancel-button{-webkit-appearance:none}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$x() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-input type="search" id="searchInput" class="search-box-input"
    @search="${this.onSearchTermSearch}" @input="${this.onSearchTermInput}"
    aria-label="${this.label}" placeholder="${this.label}"
    autofocus="${this.autofocus}" spellcheck="false">
  <div slot="inline-prefix" id="icon" class="cr-icon icon-search" alt=""></div>
  <cr-icon-button id="clearSearch" class="icon-cancel"
      ?hidden="${!this.hasSearchText}" slot="suffix"
      @click="${this.onClearClick_}" title="$i18n{clearSearch}">
  </cr-icon-button>
</cr-input>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const SANITIZE_REGEX = /[-[\]{}()*+?.,\\^$|#\s]/g;
const PrintPreviewSearchBoxElementBase = CrSearchFieldMixinLit(WebUiListenerMixinLit(CrLitElement));
class PrintPreviewSearchBoxElement extends PrintPreviewSearchBoxElementBase {
    static get is() {
        return 'print-preview-search-box';
    }
    static get styles() {
        return getCss$v();
    }
    render() {
        return getHtml$x.bind(this)();
    }
    static get properties() {
        return {
            autofocus: { type: Boolean },
            searchQuery: {
                type: Object,
                notify: true,
            },
        };
    }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #searchQuery_accessor_storage = null;
    get searchQuery() { return this.#searchQuery_accessor_storage; }
    set searchQuery(value) { this.#searchQuery_accessor_storage = value; }
    lastQuery_ = '';
    firstUpdated() {
        this.addEventListener('search-changed', e => this.onSearchChanged_(e));
    }
    getSearchInput() {
        return this.$.searchInput;
    }
    focus() {
        this.$.searchInput.focus();
    }
    onSearchChanged_(e) {
        const strippedQuery = stripDiacritics(e.detail.trim());
        const safeQuery = strippedQuery.replace(SANITIZE_REGEX, '\\$&');
        if (safeQuery === this.lastQuery_) {
            return;
        }
        this.lastQuery_ = safeQuery;
        this.searchQuery =
            safeQuery.length > 0 ? new RegExp(`(${safeQuery})`, 'ig') : null;
    }
    onClearClick_() {
        this.setValue('');
        this.$.searchInput.focus();
    }
}
customElements.define(PrintPreviewSearchBoxElement.is, PrintPreviewSearchBoxElement);

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Make a string safe for Polymer bindings that are inner-h-t-m-l or other
 * innerHTML use.
 * @param rawString The unsanitized string
 * @param opts Optional additional allowed tags and attributes.
 */
function sanitizeInnerHtmlInternal(rawString, opts) {
    opts = opts || {};
    const html = parseHtmlSubset(`<b>${rawString}</b>`, opts.tags, opts.attrs)
        .firstElementChild;
    return html.innerHTML;
}
// 
let sanitizedPolicy = null;
/**
 * Same as |sanitizeInnerHtmlInternal|, but it passes through sanitizedPolicy
 * to create a TrustedHTML.
 */
function sanitizeInnerHtml(rawString, opts) {
    assert(window.trustedTypes);
    if (sanitizedPolicy === null) {
        // Initialize |sanitizedPolicy| lazily.
        sanitizedPolicy = window.trustedTypes.createPolicy('sanitize-inner-html', {
            createHTML: sanitizeInnerHtmlInternal,
            createScript: () => assertNotReached(),
            createScriptURL: () => assertNotReached(),
        });
    }
    return sanitizedPolicy.createHTML(rawString, opts);
}
const allowAttribute = (_node, _value) => true;
/** Allow-list of attributes in parseHtmlSubset. */
const allowedAttributes = new Map([
    [
        'href',
        (node, value) => {
            // Only allow a[href] starting with chrome:// or https:// or equaling
            // to #.
            return node.tagName === 'A' &&
                (value.startsWith('chrome://') || value.startsWith('https://') ||
                    value === '#');
        },
    ],
    [
        'target',
        (node, value) => {
            // Only allow a[target='_blank'].
            // TODO(dbeam): are there valid use cases for target !== '_blank'?
            return node.tagName === 'A' && value === '_blank';
        },
    ],
]);
/** Allow-list of optional attributes in parseHtmlSubset. */
const allowedOptionalAttributes = new Map([
    ['class', allowAttribute],
    ['id', allowAttribute],
    ['is', (_node, value) => value === 'action-link' || value === ''],
    ['role', (_node, value) => value === 'link'],
    [
        'src',
        (node, value) => {
            // Only allow img[src] starting with chrome://
            return node.tagName === 'IMG' &&
                value.startsWith('chrome://');
        },
    ],
    ['tabindex', allowAttribute],
    ['aria-description', allowAttribute],
    ['aria-hidden', allowAttribute],
    ['aria-label', allowAttribute],
    ['aria-labelledby', allowAttribute],
]);
/** Allow-list of tag names in parseHtmlSubset. */
const allowedTags = new Set(['A', 'B', 'I', 'BR', 'DIV', 'EM', 'KBD', 'P', 'PRE', 'SPAN', 'STRONG']);
/** Allow-list of optional tag names in parseHtmlSubset. */
const allowedOptionalTags = new Set(['IMG', 'LI', 'UL']);
/**
 * This policy maps a given string to a `TrustedHTML` object
 * without performing any validation. Callsites must ensure
 * that the resulting object will only be used in inert
 * documents. Initialized lazily.
 */
let unsanitizedPolicy;
/**
 * @param optTags an Array to merge.
 * @return Set of allowed tags.
 */
function mergeTags(optTags) {
    const clone = new Set(allowedTags);
    optTags.forEach(str => {
        const tag = str.toUpperCase();
        if (allowedOptionalTags.has(tag)) {
            clone.add(tag);
        }
    });
    return clone;
}
/**
 * @param optAttrs an Array to merge.
 * @return Map of allowed attributes.
 */
function mergeAttrs(optAttrs) {
    const clone = new Map(allowedAttributes);
    optAttrs.forEach(key => {
        if (allowedOptionalAttributes.has(key)) {
            clone.set(key, allowedOptionalAttributes.get(key));
        }
    });
    return clone;
}
function walk(n, f) {
    f(n);
    for (let i = 0; i < n.childNodes.length; i++) {
        walk(n.childNodes[i], f);
    }
}
function assertElement(tags, node) {
    if (!tags.has(node.tagName)) {
        throw Error(node.tagName + ' is not supported');
    }
}
function assertAttribute(attrs, attrNode, node) {
    const n = attrNode.nodeName;
    const v = attrNode.nodeValue || '';
    if (!attrs.has(n) || !attrs.get(n)(node, v)) {
        throw Error(node.tagName + '[' + n + '="' + v +
            '"] is not supported');
    }
}
/**
 * Parses a very small subset of HTML. This ensures that insecure HTML /
 * javascript cannot be injected into WebUI.
 * @param s The string to parse.
 * @param extraTags Optional extra allowed tags.
 * @param extraAttrs
 *     Optional extra allowed attributes (all tags are run through these).
 * @throws an Error in case of non supported markup.
 * @return A document fragment containing the DOM tree.
 */
function parseHtmlSubset(s, extraTags, extraAttrs) {
    const tags = extraTags ? mergeTags(extraTags) : allowedTags;
    const attrs = extraAttrs ? mergeAttrs(extraAttrs) : allowedAttributes;
    const doc = document.implementation.createHTMLDocument('');
    const r = doc.createRange();
    r.selectNode(doc.body);
    if (window.trustedTypes) {
        if (!unsanitizedPolicy) {
            unsanitizedPolicy =
                window.trustedTypes.createPolicy('parse-html-subset', {
                    createHTML: (untrustedHTML) => untrustedHTML,
                    createScript: () => assertNotReached(),
                    createScriptURL: () => assertNotReached(),
                });
        }
        s = unsanitizedPolicy.createHTML(s);
    }
    // This does not execute any scripts because the document has no view.
    const df = r.createContextualFragment(s);
    walk(df, function (node) {
        switch (node.nodeType) {
            case Node.ELEMENT_NODE:
                assertElement(tags, node);
                const nodeAttrs = node.attributes;
                for (let i = 0; i < nodeAttrs.length; ++i) {
                    assertAttribute(attrs, nodeAttrs[i], node);
                }
                break;
            case Node.COMMENT_NODE:
            case Node.DOCUMENT_FRAGMENT_NODE:
            case Node.TEXT_NODE:
                break;
            default:
                throw Error('Node type ' + node.nodeType + ' is not supported');
        }
    });
    return df;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'I18nMixinLit' is a Mixin offering loading of internationalization
 * strings. Typically it is used as [[i18n('someString')]] computed bindings or
 * for this.i18n('foo'). It is not needed for HTML $i18n{otherString}, which is
 * handled by a C++ templatizer.
 */
const I18nMixinLit = (superClass) => {
    class I18nMixinLit extends superClass {
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, substituted string.
         */
        i18nRaw_(id, ...varArgs) {
            return varArgs.length === 0 ? loadTimeData.getString(id) :
                loadTimeData.getStringF(id, ...varArgs);
        }
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values. Also sanitizes the output to filter out dangerous HTML/JS.
         * Use with Lit bindings that are *not* innerHTML.
         * NOTE: This is not related to $i18n{foo} in HTML, see file overview.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18n(id, ...varArgs) {
            const rawString = this.i18nRaw_(id, ...varArgs);
            return parseHtmlSubset(`<b>${rawString}</b>`).firstChild.textContent;
        }
        /**
         * Similar to 'i18n', returns a translated, sanitized, substituted
         * string. It receives the string ID and a dictionary containing the
         * substitutions as well as optional additional allowed tags and
         * attributes. Use with Lit bindings that are innerHTML.
         * @param id The ID of the string to translate.
         */
        i18nAdvanced(id, opts) {
            opts = opts || {};
            const rawString = this.i18nRaw_(id, ...(opts.substitutions || []));
            return sanitizeInnerHtml(rawString, opts);
        }
        /**
         * Similar to 'i18n', with an unused |locale| parameter used to trigger
         * updates when the locale changes.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18nDynamic(_locale, id, ...varArgs) {
            return this.i18n(id, ...varArgs);
        }
        /**
         * Similar to 'i18nDynamic', but varArgs values are interpreted as keys
         * in loadTimeData. This allows generation of strings that take other
         * localized strings as parameters.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9
         *     in the string. Values are interpreted as strings IDs if found in
         * the list of localized strings.
         * @return A translated, sanitized, substituted string.
         */
        i18nRecursive(locale, id, ...varArgs) {
            let args = varArgs;
            if (args.length > 0) {
                // Try to replace IDs with localized values.
                args = args.map(str => {
                    return this.i18nExists(str) ? loadTimeData.getString(str) : str;
                });
            }
            return this.i18nDynamic(locale, id, ...args);
        }
        /**
         * Returns true if a translation exists for |id|.
         */
        i18nExists(id) {
            return loadTimeData.valueExists(id);
        }
    }
    return I18nMixinLit;
};

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Enumeration of buckets that a user can enter while using the destination
 * search widget.
 */
var DestinationSearchBucket;
(function (DestinationSearchBucket) {
    // Used when the print destination search widget is shown.
    DestinationSearchBucket[DestinationSearchBucket["DESTINATION_SHOWN"] = 0] = "DESTINATION_SHOWN";
    // Used when the user selects a print destination.
    DestinationSearchBucket[DestinationSearchBucket["DESTINATION_CLOSED_CHANGED"] = 1] = "DESTINATION_CLOSED_CHANGED";
    // Used when the print destination search widget is closed without selecting
    // a print destination.
    DestinationSearchBucket[DestinationSearchBucket["DESTINATION_CLOSED_UNCHANGED"] = 2] = "DESTINATION_CLOSED_UNCHANGED";
    // Note: values 3-13 are intentionally unset as these correspond to
    // deprecated values in histograms/enums.xml. These enums are append-only.
    // User clicked on Manage button
    DestinationSearchBucket[DestinationSearchBucket["MANAGE_BUTTON_CLICKED"] = 14] = "MANAGE_BUTTON_CLICKED";
    // Max value.
    DestinationSearchBucket[DestinationSearchBucket["DESTINATION_SEARCH_MAX_BUCKET"] = 15] = "DESTINATION_SEARCH_MAX_BUCKET";
})(DestinationSearchBucket || (DestinationSearchBucket = {}));
/**
 * Print settings UI usage metrics buckets.
 */
var PrintSettingsUiBucket;
(function (PrintSettingsUiBucket) {
    // Advanced settings dialog is shown.
    PrintSettingsUiBucket[PrintSettingsUiBucket["ADVANCED_SETTINGS_DIALOG_SHOWN"] = 0] = "ADVANCED_SETTINGS_DIALOG_SHOWN";
    // Advanced settings dialog is closed without saving a selection.
    PrintSettingsUiBucket[PrintSettingsUiBucket["ADVANCED_SETTINGS_DIALOG_CANCELED"] = 1] = "ADVANCED_SETTINGS_DIALOG_CANCELED";
    // 'More/less settings' expanded.
    PrintSettingsUiBucket[PrintSettingsUiBucket["MORE_SETTINGS_CLICKED"] = 2] = "MORE_SETTINGS_CLICKED";
    // 'More/less settings' collapsed.
    PrintSettingsUiBucket[PrintSettingsUiBucket["LESS_SETTINGS_CLICKED"] = 3] = "LESS_SETTINGS_CLICKED";
    // User printed with extra settings expanded.
    PrintSettingsUiBucket[PrintSettingsUiBucket["PRINT_WITH_SETTINGS_EXPANDED"] = 4] = "PRINT_WITH_SETTINGS_EXPANDED";
    // User printed with extra settings collapsed.
    PrintSettingsUiBucket[PrintSettingsUiBucket["PRINT_WITH_SETTINGS_COLLAPSED"] = 5] = "PRINT_WITH_SETTINGS_COLLAPSED";
    // Max value.
    PrintSettingsUiBucket[PrintSettingsUiBucket["PRINT_SETTINGS_UI_MAX_BUCKET"] = 6] = "PRINT_SETTINGS_UI_MAX_BUCKET";
})(PrintSettingsUiBucket || (PrintSettingsUiBucket = {}));
/* A context for recording a value in a specific UMA histogram. */
class MetricsContext {
    histogram_;
    maxBucket_;
    nativeLayer_ = NativeLayerImpl.getInstance();
    /**
     * @param histogram The name of the histogram to be recorded in.
     * @param maxBucket The max value for the last histogram bucket.
     */
    constructor(histogram, maxBucket) {
        this.histogram_ = histogram;
        this.maxBucket_ = maxBucket;
    }
    /**
     * Record a histogram value in UMA. If specified value is larger than the
     * max bucket value, record the value in the largest bucket
     * @param bucket Value to record.
     */
    record(bucket) {
        this.nativeLayer_.recordInHistogram(this.histogram_, (bucket > this.maxBucket_) ? this.maxBucket_ : bucket, this.maxBucket_);
    }
    /**
     * Print settings UI specific usage statistics context
     */
    static printSettingsUi() {
        return new MetricsContext('PrintPreview.PrintSettingsUi', PrintSettingsUiBucket.PRINT_SETTINGS_UI_MAX_BUCKET);
    }
}

let instance$v = null;
function getCss$u() {
    return instance$v || (instance$v = [...[getCss$x(), getCss$K()], css `#dialog::part(dialog){height:-webkit-fit-content;max-height:calc(100vh - 2 * var(--print-preview-dialog-margin));max-width:calc(100vw - 2 * var(--print-preview-dialog-margin))}#dialog::part(wrapper){height:calc(100vh - 2 * var(--print-preview-dialog-margin))}#dialog::part(body-container){flex:1}print-preview-search-box{margin-bottom:8px;margin-top:16px}cr-dialog [slot=body]{display:flex;flex-direction:column;height:100%}#itemList{flex:1;overflow-x:hidden;overflow-y:overlay}#itemList.searching{padding-bottom:20px;padding-top:20px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$w() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-dialog id="dialog" @close="${this.onCloseOrCancel_}">
  <div slot="title">
    ${this.i18n('advancedSettingsDialogTitle', this.destination?.displayName || '')}
  </div>
  <div slot="body">
    <print-preview-search-box id="searchBox"
        ?hidden="${!this.hasMultipleItems_()}"
        label="$i18n{advancedSettingsSearchBoxPlaceholder}"
        .search-query="${this.searchQuery_}"
        @search-query-changed="${this.onSearchQueryChanged_}" autofocus>
    </print-preview-search-box>
    <div id="itemList" class="${this.isSearching_()}">
      ${this.getVendorCapabilities_().map(item => html `
        <print-preview-advanced-settings-item .capability="${item}">
        </print-preview-advanced-settings-item>
      `)}
    </div>
    <div class="no-settings-match-hint" ?hidden="${!this.shouldShowHint_()}">
      $i18n{noAdvancedSettingsMatchSearchHint}
    </div>
  </div>
  <div slot="button-container">
    <cr-button class="cancel-button" @click="${this.onCancelButtonClick_}">
      $i18n{cancel}
    </cr-button>
    <cr-button class="action-button" @click="${this.onApplyButtonClick_}">
      $i18n{advancedSettingsDialogConfirm}
    </cr-button>
  </div>
</cr-dialog>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewAdvancedSettingsDialogElementBase = I18nMixinLit(SettingsMixin(CrLitElement));
class PrintPreviewAdvancedSettingsDialogElement extends PrintPreviewAdvancedSettingsDialogElementBase {
    static get is() {
        return 'print-preview-advanced-settings-dialog';
    }
    static get styles() {
        return getCss$u();
    }
    render() {
        return getHtml$w.bind(this)();
    }
    static get properties() {
        return {
            destination: { type: Object },
            searchQuery_: { type: Object },
            hasMatching_: {
                type: Boolean,
                notify: true,
            },
        };
    }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #searchQuery__accessor_storage = null;
    get searchQuery_() { return this.#searchQuery__accessor_storage; }
    set searchQuery_(value) { this.#searchQuery__accessor_storage = value; }
    #hasMatching__accessor_storage = false;
    get hasMatching_() { return this.#hasMatching__accessor_storage; }
    set hasMatching_(value) { this.#hasMatching__accessor_storage = value; }
    highlights_ = [];
    bubbles_ = new Map();
    metrics_ = MetricsContext.printSettingsUi();
    updated(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('searchQuery_')) {
            // Note: computeHasMatching_() updates the DOM in addition to calculating
            // the hasMatching_ value, so needs to be done in updated().
            this.hasMatching_ = this.computeHasMatching_();
        }
    }
    firstUpdated() {
        this.addEventListener('keydown', e => this.onKeydown_(e));
    }
    connectedCallback() {
        super.connectedCallback();
        this.metrics_.record(PrintSettingsUiBucket.ADVANCED_SETTINGS_DIALOG_SHOWN);
        this.$.dialog.showModal();
    }
    onKeydown_(e) {
        e.stopPropagation();
        const searchInput = this.$.searchBox.getSearchInput();
        const eventInSearchBox = e.composedPath().includes(searchInput);
        if (e.key === 'Escape' &&
            (!eventInSearchBox || !searchInput.value.trim())) {
            this.$.dialog.cancel();
            e.preventDefault();
            return;
        }
        if (e.key === 'Enter' && !eventInSearchBox) {
            const activeElementTag = e.composedPath()[0].tagName;
            if (['CR-BUTTON', 'SELECT'].includes(activeElementTag)) {
                return;
            }
            this.onApplyButtonClick_();
            e.preventDefault();
            return;
        }
    }
    /**
     * @return Whether there is more than one vendor item to display.
     */
    hasMultipleItems_() {
        if (!this.destination) {
            return false;
        }
        return this.destination.capabilities.printer.vendor_capability.length > 1;
    }
    /**
     * @return Whether there is a setting matching the query.
     */
    computeHasMatching_() {
        if (!this.shadowRoot) {
            return true;
        }
        removeHighlights(this.highlights_);
        this.bubbles_.forEach((_number, bubble) => bubble.remove());
        this.highlights_ = [];
        this.bubbles_.clear();
        const listItems = this.shadowRoot.querySelectorAll('print-preview-advanced-settings-item');
        let hasMatch = false;
        listItems.forEach(item => {
            const matches = item.hasMatch(this.searchQuery_);
            item.hidden = !matches;
            hasMatch = hasMatch || matches;
            this.highlights_.push(...item.updateHighlighting(this.searchQuery_, this.bubbles_));
        });
        return hasMatch;
    }
    /**
     * @return Whether the no matching settings hint should be shown.
     */
    shouldShowHint_() {
        return !!this.searchQuery_ && !this.hasMatching_;
    }
    onCloseOrCancel_() {
        if (this.searchQuery_) {
            this.$.searchBox.setValue('');
        }
        if (this.$.dialog.getNative().returnValue === 'success') {
            this.metrics_.record(PrintSettingsUiBucket.ADVANCED_SETTINGS_DIALOG_CANCELED);
        }
    }
    onCancelButtonClick_() {
        this.$.dialog.cancel();
    }
    onApplyButtonClick_() {
        const settingsValues = {};
        const itemList = this.shadowRoot.querySelectorAll('print-preview-advanced-settings-item');
        itemList.forEach(item => {
            settingsValues[item.capability.id] = item.getCurrentValue();
        });
        this.setSetting('vendorItems', settingsValues);
        this.$.dialog.close();
    }
    close() {
        this.$.dialog.close();
    }
    isSearching_() {
        return this.searchQuery_ ? 'searching' : '';
    }
    getVendorCapabilities_() {
        return this.destination?.capabilities?.printer.vendor_capability || [];
    }
    onSearchQueryChanged_(e) {
        this.searchQuery_ = e.detail.value;
    }
}
customElements.define(PrintPreviewAdvancedSettingsDialogElement.is, PrintPreviewAdvancedSettingsDialogElement);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return The currently focused element (including elements that are
 *     behind a shadow root), or null if nothing is focused.
 */
function getDeepActiveElement() {
    let a = document.activeElement;
    while (a && a.shadowRoot && a.shadowRoot.activeElement) {
        a = a.shadowRoot.activeElement;
    }
    return a;
}
/**
 * @return Whether a modifier key was down when processing |e|.
 */
function hasKeyModifiers(e) {
    return !!(e.altKey || e.ctrlKey || e.metaKey || e.shiftKey);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const InputMixin = (superClass) => {
    class InputMixin extends superClass {
        static get properties() {
            return {
                lastValue_: { type: String },
            };
        }
        #lastValue__accessor_storage = '';
        get lastValue_() { return this.#lastValue__accessor_storage; }
        set lastValue_(value) { this.#lastValue__accessor_storage = value; }
        /** Timeout used to delay processing of the input, in ms. */
        timeout_ = null;
        connectedCallback() {
            super.connectedCallback();
            this.getInput().addEventListener('input', () => this.resetTimeout_());
            this.getInput().addEventListener('keydown', (e) => this.onKeyDown_(e));
        }
        getInput() {
            assertNotReached();
        }
        /**
         * @return The delay to use for the timeout, in ms. Elements using
         *     this behavior must set this delay as data-timeout-delay on the
         *     input element returned by getInput().
         */
        getTimeoutDelayMs_() {
            const delay = parseInt(this.getInput().dataset['timeoutDelay'], 10);
            assert(!Number.isNaN(delay));
            return delay;
        }
        /**
         * Called when a key is pressed on the input.
         */
        onKeyDown_(event) {
            if (event.key !== 'Enter' && event.key !== 'Tab') {
                return;
            }
            this.resetAndUpdate();
        }
        /**
         * Called when a input event occurs on the textfield. Starts an input
         * timeout.
         */
        resetTimeout_() {
            if (this.timeout_) {
                clearTimeout(this.timeout_);
            }
            this.timeout_ =
                setTimeout(() => this.onTimeout_(), this.getTimeoutDelayMs_());
        }
        /**
         * Called after a timeout after user input into the textfield.
         */
        onTimeout_() {
            this.timeout_ = null;
            const value = this.getInput().value || '';
            if (this.lastValue_ !== value) {
                this.lastValue_ = value;
                this.dispatchEvent(new CustomEvent('input-change', { bubbles: true, composed: true, detail: value }));
            }
        }
        resetString() {
            this.lastValue_ = null;
        }
        resetAndUpdate() {
            if (this.timeout_) {
                clearTimeout(this.timeout_);
            }
            this.onTimeout_();
        }
    }
    return InputMixin;
};

let instance$u = null;
function getCss$t() {
    return instance$u || (instance$u = [...[getCss$D()], css `:host{display:block;position:absolute;transition:opacity 150ms linear}:host([invisible]){opacity:0}:host([disabled]),:host([invisible]){pointer-events:none}:host([side=top]) #lineContainer,:host([side=bottom]) #lineContainer{cursor:ns-resize;padding:8px 0;width:100%}:host([side=left]) #lineContainer,:host([side=right]) #lineContainer{cursor:ew-resize;height:100%;padding:0 8px}#line{border:1px dashed var(--google-blue-500)}@media (prefers-color-scheme:dark){#line{border-color:var(--google-blue-300)}}:host([side=top]) #line,:host([side=bottom]) #line{width:100%}:host([side=left]) #line,:host([side=right]) #line{height:100%}#row-container{border-radius:4px;font-size:0.8rem;min-height:25px;overflow:hidden;padding:1px;position:absolute}@media (prefers-color-scheme:light){#row-container{--cr-input-background-color:var(--cr-primary-text-color);--cr-input-color:white;background-color:var(--cr-primary-text-color);color:white}}@media (prefers-color-scheme:dark){#row-container{--cr-input-background-color:rgb(27,28,30);--cr-input-color:var(--cr-primary-text-color);background-color:rgb(27,28,30);color:var(--cr-primary-text-color)}}:host([side=top]) #row-container{left:50%;top:9px}:host([side=right]) #row-container{right:9px;top:50%}:host([side=bottom]) #row-container{bottom:9px;right:50%}:host([side=left]) #row-container{bottom:50%;left:9px}:host([disabled]) #row-container{opacity:var(--cr-disabled-opacity)}:host([invalid]) #input{caret-color:var(--cr-input-error-color)}:host([invalid]) #underline{border-color:var(--cr-input-error-color)}#row-container,#input-container{align-items:center;display:flex}#input-container{position:relative}#input{padding-inline-end:0;text-align:end;width:44px}#unit{padding-inline-end:8px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$v() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="lineContainer">
  <div id="line"></div>
</div>
<div id="row-container">
  <div id="input-container">
    <input id="input" ?disabled="${this.disabled}"
        aria-label="${this.i18n(this.side)}"
        aria-hidden="${this.getAriaHidden_()}"
        @focus="${this.onFocus_}" @blur="${this.onBlur_}"
        data-timeout-delay="1000">
    <span id="unit">${this.measurementSystem?.unitSymbol || ''}</span>
  </div>
  <div id="underline"></div>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Radius of the margin control in pixels. Padding of control + 1 for border.
 */
const RADIUS_PX = 9;
const PrintPreviewMarginControlElementBase = I18nMixinLit(WebUiListenerMixinLit(InputMixin(CrLitElement)));
class PrintPreviewMarginControlElement extends PrintPreviewMarginControlElementBase {
    static get is() {
        return 'print-preview-margin-control';
    }
    static get styles() {
        return getCss$t();
    }
    render() {
        return getHtml$v.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            side: {
                type: String,
                reflect: true,
            },
            invalid: {
                type: Boolean,
                reflect: true,
            },
            invisible: {
                type: Boolean,
                reflect: true,
            },
            measurementSystem: { type: Object },
            focused_: {
                type: Boolean,
                reflect: true,
            },
            positionInPts_: { type: Number },
            scaleTransform: { type: Number },
            translateTransform: { type: Object },
            pageSize: { type: Object },
            clipSize: { type: Object },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #side_accessor_storage = CustomMarginsOrientation.TOP;
    get side() { return this.#side_accessor_storage; }
    set side(value) { this.#side_accessor_storage = value; }
    #invalid_accessor_storage = false;
    get invalid() { return this.#invalid_accessor_storage; }
    set invalid(value) { this.#invalid_accessor_storage = value; }
    #invisible_accessor_storage = false;
    get invisible() { return this.#invisible_accessor_storage; }
    set invisible(value) { this.#invisible_accessor_storage = value; }
    #measurementSystem_accessor_storage = null;
    get measurementSystem() { return this.#measurementSystem_accessor_storage; }
    set measurementSystem(value) { this.#measurementSystem_accessor_storage = value; }
    #scaleTransform_accessor_storage = 1;
    get scaleTransform() { return this.#scaleTransform_accessor_storage; }
    set scaleTransform(value) { this.#scaleTransform_accessor_storage = value; }
    #translateTransform_accessor_storage = new Coordinate2d(0, 0);
    get translateTransform() { return this.#translateTransform_accessor_storage; }
    set translateTransform(value) { this.#translateTransform_accessor_storage = value; }
    #pageSize_accessor_storage = new Size(612, 792);
    get pageSize() { return this.#pageSize_accessor_storage; }
    set pageSize(value) { this.#pageSize_accessor_storage = value; }
    #clipSize_accessor_storage = null;
    get clipSize() { return this.#clipSize_accessor_storage; }
    set clipSize(value) { this.#clipSize_accessor_storage = value; }
    #focused__accessor_storage = false;
    get focused_() { return this.#focused__accessor_storage; }
    set focused_(value) { this.#focused__accessor_storage = value; }
    #positionInPts__accessor_storage = 0;
    get positionInPts_() { return this.#positionInPts__accessor_storage; }
    set positionInPts_(value) { this.#positionInPts__accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('disabled')) {
            if (this.disabled) {
                this.focused_ = false;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('clipSize') ||
            changedProperties.has('invisible')) {
            this.onClipSizeChange_();
        }
        if (changedPrivateProperties.has('positionInPts_') ||
            changedProperties.has('scaleTransform') ||
            changedProperties.has('translateTransform') ||
            changedProperties.has('pageSize') || changedProperties.has('side')) {
            this.updatePosition_();
        }
    }
    firstUpdated() {
        this.addEventListener('input-change', e => this.onInputChange_(e));
    }
    /** @return The input element for InputBehavior. */
    getInput() {
        return this.$.input;
    }
    /**
     * @param valueInPts New value of the margin control's textbox in pts.
     */
    setTextboxValue(valueInPts) {
        const textbox = this.$.input;
        const pts = textbox.value ? this.parseValueToPts_(textbox.value) : null;
        if (pts !== null && valueInPts === Math.round(pts)) {
            // If the textbox's value represents the same value in pts as the new one,
            // don't reset. This allows the "undo" command to work as expected, see
            // https://crbug.com/452844.
            return;
        }
        textbox.value = this.serializeValueFromPts_(valueInPts);
        this.resetString();
    }
    /** @return The current position of the margin control. */
    getPositionInPts() {
        return this.positionInPts_;
    }
    /** @param position The new position for the margin control. */
    setPositionInPts(position) {
        this.positionInPts_ = position;
    }
    /**
     * @return 'true' or 'false', indicating whether the input should be
     *     aria-hidden.
     */
    getAriaHidden_() {
        return this.invisible.toString();
    }
    /**
     * Converts a value in pixels to points.
     * @param pixels Pixel value to convert.
     * @return Given value expressed in points.
     */
    convertPixelsToPts(pixels) {
        let pts;
        const Orientation = CustomMarginsOrientation;
        if (this.side === Orientation.TOP) {
            pts = pixels - this.translateTransform.y + RADIUS_PX;
            pts /= this.scaleTransform;
        }
        else if (this.side === Orientation.RIGHT) {
            pts = pixels - this.translateTransform.x + RADIUS_PX;
            pts /= this.scaleTransform;
            pts = this.pageSize.width - pts;
        }
        else if (this.side === Orientation.BOTTOM) {
            pts = pixels - this.translateTransform.y + RADIUS_PX;
            pts /= this.scaleTransform;
            pts = this.pageSize.height - pts;
        }
        else {
            assert(this.side === Orientation.LEFT);
            pts = pixels - this.translateTransform.x + RADIUS_PX;
            pts /= this.scaleTransform;
        }
        return pts;
    }
    /**
     * @param event A pointerdown event triggered by this element.
     * @return Whether the margin should start being dragged.
     */
    shouldDrag(event) {
        return !this.disabled && event.button === 0 &&
            (event.composedPath()[0] === this.$.lineContainer ||
                event.composedPath()[0] === this.$.line);
    }
    /**
     * @param value Value to parse to points. E.g. '3.40' or '200'.
     * @return Value in points represented by the input value.
     */
    parseValueToPts_(value) {
        value = value.trim();
        if (value.length === 0) {
            return null;
        }
        assert(this.measurementSystem);
        const decimal = this.measurementSystem.decimalDelimiter;
        const thousands = this.measurementSystem.thousandsDelimiter;
        const whole = `(?:0|[1-9]\\d*|[1-9]\\d{0,2}(?:[${thousands}]\\d{3})*)`;
        const fractional = `(?:[${decimal}]\\d+)`;
        const wholeDecimal = `(?:${whole}[${decimal}])`;
        const validationRegex = new RegExp(`^-?(?:${whole}${fractional}?|${fractional}|${wholeDecimal})$`);
        if (validationRegex.test(value)) {
            // Removing thousands delimiters and replacing the decimal delimiter with
            // the dot symbol in order to use parseFloat() properly.
            value = value.replace(new RegExp(`\\${thousands}`, 'g'), '')
                .replace(decimal, '.');
            return this.measurementSystem.convertToPoints(parseFloat(value));
        }
        return null;
    }
    /**
     * @param value Value in points to serialize.
     * @return String representation of the value in the system's local units.
     */
    serializeValueFromPts_(value) {
        assert(this.measurementSystem);
        value = this.measurementSystem.convertFromPoints(value);
        value = this.measurementSystem.roundValue(value);
        // Convert the dot symbol to the decimal delimiter for the locale.
        return value.toString().replace('.', this.measurementSystem.decimalDelimiter);
    }
    fire_(eventName, detail) {
        this.dispatchEvent(new CustomEvent(eventName, { bubbles: true, composed: true, detail }));
    }
    /**
     * @param e Contains the new value of the input.
     */
    onInputChange_(e) {
        if (e.detail === '') {
            return;
        }
        const value = this.parseValueToPts_(e.detail);
        if (value === null) {
            this.invalid = true;
            return;
        }
        this.fire_('text-change', value);
    }
    onBlur_() {
        this.focused_ = false;
        this.resetAndUpdate();
        this.fire_('text-blur', this.invalid || !this.$.input.value);
    }
    onFocus_() {
        this.focused_ = true;
        this.fire_('text-focus');
    }
    updatePosition_() {
        if (!this.translateTransform || !this.scaleTransform ||
            !this.measurementSystem) {
            return;
        }
        const Orientation = CustomMarginsOrientation;
        let x = this.translateTransform.x;
        let y = this.translateTransform.y;
        let width = null;
        let height = null;
        if (this.side === Orientation.TOP) {
            y = this.scaleTransform * this.positionInPts_ +
                this.translateTransform.y - RADIUS_PX;
            width = this.scaleTransform * this.pageSize.width;
        }
        else if (this.side === Orientation.RIGHT) {
            x = this.scaleTransform * (this.pageSize.width - this.positionInPts_) +
                this.translateTransform.x - RADIUS_PX;
            height = this.scaleTransform * this.pageSize.height;
        }
        else if (this.side === Orientation.BOTTOM) {
            y = this.scaleTransform * (this.pageSize.height - this.positionInPts_) +
                this.translateTransform.y - RADIUS_PX;
            width = this.scaleTransform * this.pageSize.width;
        }
        else {
            x = this.scaleTransform * this.positionInPts_ +
                this.translateTransform.x - RADIUS_PX;
            height = this.scaleTransform * this.pageSize.height;
        }
        window.requestAnimationFrame(() => {
            this.style.left = Math.round(x) + 'px';
            this.style.top = Math.round(y) + 'px';
            if (width !== null) {
                this.style.width = Math.round(width) + 'px';
            }
            if (height !== null) {
                this.style.height = Math.round(height) + 'px';
            }
        });
        this.onClipSizeChange_();
    }
    onClipSizeChange_() {
        if (!this.clipSize) {
            return;
        }
        window.requestAnimationFrame(() => {
            const offsetLeft = this.offsetLeft;
            const offsetTop = this.offsetTop;
            this.style.clip = 'rect(' + (-offsetTop) + 'px, ' +
                (this.clipSize.width - offsetLeft) + 'px, ' +
                (this.clipSize.height - offsetTop) + 'px, ' + (-offsetLeft) + 'px)';
        });
    }
}
customElements.define(PrintPreviewMarginControlElement.is, PrintPreviewMarginControlElement);

let instance$t = null;
function getCss$s() {
    return instance$t || (instance$t = [...[], css `:host{display:block;left:0;position:absolute;top:0}:host([dragging_=dragging-vertical]){cursor:ns-resize}:host([dragging_=dragging-horizontal]){cursor:ew-resize}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$u() {
    // clang-format off
    return html `<!--_html_template_start_-->
${this.marginSides_.map(item => html `
  <print-preview-margin-control side="${item}" ?invisible="${this.invisible_}"
      ?disabled="${this.controlsDisabled_()}"
      .translateTransform="${this.translateTransform_}"
      .clipSize="${this.clipSize_}"
      .measurementSystem="${this.measurementSystem}"
      .scaleTransform="${this.scaleTransform_}"
      .pageSize="${this.pageSize}"
      @pointerdown="${this.onPointerDown_}"
      @text-change="${this.onTextChange_}" @text-blur="${this.onTextBlur_}"
      @text-focus="${this.onTextFocus_}"
      @transition-end="${this.onTransitionEnd_}">
  </print-preview-margin-control>
`)}
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const MARGIN_KEY_MAP = new Map([
    [CustomMarginsOrientation.TOP, 'marginTop'],
    [CustomMarginsOrientation.RIGHT, 'marginRight'],
    [CustomMarginsOrientation.BOTTOM, 'marginBottom'],
    [CustomMarginsOrientation.LEFT, 'marginLeft'],
]);
const MINIMUM_DISTANCE = 72; // 1 inch
const PrintPreviewMarginControlContainerElementBase = SettingsMixin(CrLitElement);
class PrintPreviewMarginControlContainerElement extends PrintPreviewMarginControlContainerElementBase {
    static get is() {
        return 'print-preview-margin-control-container';
    }
    static get styles() {
        return getCss$s();
    }
    render() {
        return getHtml$u.bind(this)();
    }
    static get properties() {
        return {
            pageSize: { type: Object },
            documentMargins: { type: Object },
            previewLoaded: { type: Boolean },
            measurementSystem: { type: Object },
            state: { type: Number },
            scaleTransform_: { type: Number },
            translateTransform_: { type: Object },
            clipSize_: { type: Object },
            available_: { type: Boolean },
            invisible_: { type: Boolean },
            marginSides_: { type: Array },
            /**
             * String attribute used to set cursor appearance. Possible values:
             * empty (''): No margin control is currently being dragged.
             * 'dragging-horizontal': The left or right control is being dragged.
             * 'dragging-vertical': The top or bottom control is being dragged.
             */
            dragging_: {
                type: String,
                reflect: true,
            },
        };
    }
    #pageSize_accessor_storage = new Size(612, 792);
    get pageSize() { return this.#pageSize_accessor_storage; }
    set pageSize(value) { this.#pageSize_accessor_storage = value; }
    #documentMargins_accessor_storage = null;
    get documentMargins() { return this.#documentMargins_accessor_storage; }
    set documentMargins(value) { this.#documentMargins_accessor_storage = value; }
    #previewLoaded_accessor_storage = false;
    get previewLoaded() { return this.#previewLoaded_accessor_storage; }
    set previewLoaded(value) { this.#previewLoaded_accessor_storage = value; }
    #measurementSystem_accessor_storage = null;
    get measurementSystem() { return this.#measurementSystem_accessor_storage; }
    set measurementSystem(value) { this.#measurementSystem_accessor_storage = value; }
    #state_accessor_storage = State.NOT_READY;
    get state() { return this.#state_accessor_storage; }
    set state(value) { this.#state_accessor_storage = value; }
    #available__accessor_storage = false;
    get available_() { return this.#available__accessor_storage; }
    set available_(value) { this.#available__accessor_storage = value; }
    #invisible__accessor_storage = true;
    get invisible_() { return this.#invisible__accessor_storage; }
    set invisible_(value) { this.#invisible__accessor_storage = value; }
    #clipSize__accessor_storage = new Size(0, 0);
    get clipSize_() { return this.#clipSize__accessor_storage; }
    set clipSize_(value) { this.#clipSize__accessor_storage = value; }
    #scaleTransform__accessor_storage = 0;
    get scaleTransform_() { return this.#scaleTransform__accessor_storage; }
    set scaleTransform_(value) { this.#scaleTransform__accessor_storage = value; }
    #translateTransform__accessor_storage = new Coordinate2d(0, 0);
    get translateTransform_() { return this.#translateTransform__accessor_storage; }
    set translateTransform_(value) { this.#translateTransform__accessor_storage = value; }
    #dragging__accessor_storage = '';
    get dragging_() { return this.#dragging__accessor_storage; }
    set dragging_(value) { this.#dragging__accessor_storage = value; }
    #marginSides__accessor_storage = [
        CustomMarginsOrientation.TOP,
        CustomMarginsOrientation.RIGHT,
        CustomMarginsOrientation.BOTTOM,
        CustomMarginsOrientation.LEFT,
    ];
    get marginSides_() { return this.#marginSides__accessor_storage; }
    set marginSides_(value) { this.#marginSides__accessor_storage = value; }
    pointerStartPositionInPixels_ = new Coordinate2d(0, 0);
    marginStartPositionInPixels_ = null;
    resetMargins_ = null;
    eventTracker_ = new EventTracker();
    textboxFocused_ = false;
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('customMargins.value', this.onMarginSettingsChange_.bind(this));
        this.onMarginSettingsChange_();
        this.addSettingObserver('mediaSize.value', this.onMediaSizeOrLayoutChange_.bind(this));
        this.addSettingObserver('layout.value', this.onMediaSizeOrLayoutChange_.bind(this));
        this.addSettingObserver('margins.value', () => {
            this.available_ = this.computeAvailable_();
        });
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('state')) {
            this.onStateChanged_();
        }
        if (changedProperties.has('previewLoaded')) {
            this.available_ = this.computeAvailable_();
        }
        if (changedPrivateProperties.has('available_')) {
            this.onAvailableChange_();
        }
    }
    computeAvailable_() {
        return this.previewLoaded &&
            (this.getSettingValue('margins') ===
                MarginsType.CUSTOM) &&
            !!this.pageSize;
    }
    onAvailableChange_() {
        if (this.available_ && this.resetMargins_) {
            // Set the custom margins values to the current document margins if the
            // custom margins were reset.
            const newMargins = {};
            assert(this.documentMargins);
            for (const side of Object.values(CustomMarginsOrientation)) {
                const key = MARGIN_KEY_MAP.get(side);
                newMargins[key] = this.documentMargins.get(side);
            }
            this.setSetting('customMargins', newMargins);
            this.resetMargins_ = false;
        }
        this.invisible_ = !this.available_;
    }
    onMarginSettingsChange_() {
        const margins = this.getSettingValue('customMargins');
        if (!margins || margins.marginTop === undefined) {
            // This may be called when print preview model initially sets the
            // settings. It sets custom margins empty by default.
            return;
        }
        this.shadowRoot.querySelectorAll('print-preview-margin-control')
            .forEach(control => {
            const key = MARGIN_KEY_MAP.get(control.side);
            const newValue = margins[key] || 0;
            control.setPositionInPts(newValue);
            control.setTextboxValue(newValue);
        });
    }
    onMediaSizeOrLayoutChange_() {
        // Reset the custom margins when the paper size changes. Don't do this if
        // it is the first preview.
        if (this.resetMargins_ === null) {
            return;
        }
        this.resetMargins_ = true;
        // Reset custom margins so that the sticky value is not restored for the new
        // paper size.
        this.setSetting('customMargins', {});
    }
    onStateChanged_() {
        if (this.state === State.READY && this.resetMargins_ === null) {
            // Don't reset margins if there are sticky values. Otherwise, set them
            // to the document margins when the user selects custom margins.
            const margins = this.getSettingValue('customMargins');
            this.resetMargins_ = !margins || margins.marginTop === undefined;
        }
    }
    /**
     * @return Whether the controls should be disabled.
     */
    controlsDisabled_() {
        return this.state !== State.READY || this.invisible_;
    }
    /**
     * @param orientation Orientation value to test.
     * @return Whether the given orientation is TOP or BOTTOM.
     */
    isTopOrBottom_(orientation) {
        return orientation === CustomMarginsOrientation.TOP ||
            orientation === CustomMarginsOrientation.BOTTOM;
    }
    /**
     * @param control Control being repositioned.
     * @param posInPixels Desired position, in pixels.
     * @return The new position for the control, in pts. Returns the
     *     position for the dimension that the control operates in, i.e.
     *     x direction for the left/right controls, y direction otherwise.
     */
    posInPixelsToPts_(control, posInPixels) {
        const side = control.side;
        return this.clipAndRoundValue_(side, control.convertPixelsToPts(this.isTopOrBottom_(side) ? posInPixels.y : posInPixels.x));
    }
    /**
     * Moves the position of the given control to the desired position in pts
     * within some constraint minimum and maximum.
     * @param control Control to move.
     * @param posInPts Desired position to move to, in pts. Position is
     *     1 dimensional and represents position in the x direction if control
     * is for the left or right margin, and the y direction otherwise.
     */
    moveControlWithConstraints_(control, posInPts) {
        control.setPositionInPts(posInPts);
        control.setTextboxValue(posInPts);
    }
    /**
     * Translates the position of the margin control relative to the pointer
     * position in pixels.
     * @param pointerPosition New position of the pointer.
     * @return New position of the margin control.
     */
    translatePointerToPositionInPixels(pointerPosition) {
        return new Coordinate2d(pointerPosition.x - this.pointerStartPositionInPixels_.x +
            this.marginStartPositionInPixels_.x, pointerPosition.y - this.pointerStartPositionInPixels_.y +
            this.marginStartPositionInPixels_.y);
    }
    /**
     * Called when the pointer moves in the custom margins component. Moves the
     * dragged margin control.
     * @param event Contains the position of the pointer.
     */
    onPointerMove_(event) {
        const control = event.target;
        const posInPts = this.posInPixelsToPts_(control, this.translatePointerToPositionInPixels(new Coordinate2d(event.x, event.y)));
        this.moveControlWithConstraints_(control, posInPts);
    }
    /**
     * Called when the pointer is released in the custom margins component.
     * Releases the dragged margin control.
     * @param event Contains the position of the pointer.
     */
    onPointerUp_(event) {
        const control = event.target;
        this.dragging_ = '';
        const posInPixels = this.translatePointerToPositionInPixels(new Coordinate2d(event.x, event.y));
        const posInPts = this.posInPixelsToPts_(control, posInPixels);
        this.moveControlWithConstraints_(control, posInPts);
        this.setMargin_(control.side, posInPts);
        this.eventTracker_.remove(control, 'pointercancel');
        this.eventTracker_.remove(control, 'pointerup');
        this.eventTracker_.remove(control, 'pointermove');
        this.fireDragChanged_(false);
    }
    /**
     * @param invisible Whether the margin controls should be invisible.
     */
    setInvisible(invisible) {
        // Ignore changes if the margin controls are not available.
        if (!this.available_) {
            return;
        }
        // Do not set the controls invisible if the user is dragging or focusing
        // the textbox for one of them.
        if (invisible && (this.dragging_ !== '' || this.textboxFocused_)) {
            return;
        }
        this.invisible_ = invisible;
    }
    /**
     * @param e Contains information about what control fired the event.
     */
    onTextFocus_(e) {
        this.textboxFocused_ = true;
        const control = e.target;
        const x = control.offsetLeft;
        const y = control.offsetTop;
        const isTopOrBottom = this.isTopOrBottom_(control.side);
        const position = {};
        // Extra padding, in px, to ensure the full textbox will be visible and
        // not just a portion of it. Can't be less than half the width or height
        // of the clip area for the computations below to work.
        const padding = Math.min(Math.min(this.clipSize_.width / 2, this.clipSize_.height / 2), 50);
        // Note: clipSize_ gives the current visible area of the margin control
        // container. The offsets of the controls are relative to the origin of
        // this visible area.
        if (isTopOrBottom) {
            // For top and bottom controls, the horizontal position of the box is
            // around halfway across the control's width.
            position.x = Math.min(x + control.offsetWidth / 2 - padding, 0);
            position.x = Math.max(x + control.offsetWidth / 2 + padding - this.clipSize_.width, position.x);
            // For top and bottom controls, the vertical position of the box is
            // nearly the same as the vertical position of the control.
            position.y = Math.min(y - padding, 0);
            position.y = Math.max(y - this.clipSize_.height + padding, position.y);
        }
        else {
            // For left and right controls, the horizontal position of the box is
            // nearly the same as the horizontal position of the control.
            position.x = Math.min(x - padding, 0);
            position.x = Math.max(x - this.clipSize_.width + padding, position.x);
            // For top and bottom controls, the vertical position of the box is
            // around halfway up the control's height.
            position.y = Math.min(y + control.offsetHeight / 2 - padding, 0);
            position.y = Math.max(y + control.offsetHeight / 2 + padding - this.clipSize_.height, position.y);
        }
        this.dispatchEvent(new CustomEvent('text-focus-position', { bubbles: true, composed: true, detail: position }));
    }
    /**
     * @param marginSide The margin side. Must be a CustomMarginsOrientation.
     * @param marginValue New value for the margin in points.
     */
    setMargin_(marginSide, marginValue) {
        const oldMargins = this.getSettingValue('customMargins');
        const key = MARGIN_KEY_MAP.get(marginSide);
        if (oldMargins[key] === marginValue) {
            return;
        }
        const newMargins = Object.assign({}, oldMargins);
        newMargins[key] = marginValue;
        this.setSetting('customMargins', newMargins);
    }
    /**
     * @param marginSide The margin side.
     * @param value The new margin value in points.
     * @return The clipped margin value in points.
     */
    clipAndRoundValue_(marginSide, value) {
        if (value < 0) {
            return 0;
        }
        const Orientation = CustomMarginsOrientation;
        let limit = 0;
        const margins = this.getSettingValue('customMargins');
        if (marginSide === Orientation.TOP) {
            limit = this.pageSize.height - margins.marginBottom - MINIMUM_DISTANCE;
        }
        else if (marginSide === Orientation.RIGHT) {
            limit = this.pageSize.width - margins.marginLeft - MINIMUM_DISTANCE;
        }
        else if (marginSide === Orientation.BOTTOM) {
            limit = this.pageSize.height - margins.marginTop - MINIMUM_DISTANCE;
        }
        else {
            assert(marginSide === Orientation.LEFT);
            limit = this.pageSize.width - margins.marginRight - MINIMUM_DISTANCE;
        }
        return Math.round(Math.min(value, limit));
    }
    /**
     * @param e Event containing the new textbox value.
     */
    onTextChange_(e) {
        const control = e.target;
        control.invalid = false;
        const clippedValue = this.clipAndRoundValue_(control.side, e.detail);
        control.setPositionInPts(clippedValue);
        this.setMargin_(control.side, clippedValue);
    }
    /**
     * @param e Event fired when a control's text field is blurred. Contains
     *     information about whether the control is in an invalid state.
     */
    onTextBlur_(e) {
        const control = e.target;
        control.setTextboxValue(control.getPositionInPts());
        if (e.detail /* detail is true if the control is in an invalid state */) {
            control.invalid = false;
        }
        this.textboxFocused_ = false;
    }
    /**
     * @param e Fired when pointerdown occurs on a margin control.
     */
    onPointerDown_(e) {
        const control = e.target;
        if (!control.shouldDrag(e)) {
            return;
        }
        this.pointerStartPositionInPixels_ = new Coordinate2d(e.x, e.y);
        this.marginStartPositionInPixels_ =
            new Coordinate2d(control.offsetLeft, control.offsetTop);
        this.dragging_ = this.isTopOrBottom_(control.side) ? 'dragging-vertical' :
            'dragging-horizontal';
        this.eventTracker_.add(control, 'pointercancel', (e) => this.onPointerUp_(e));
        this.eventTracker_.add(control, 'pointerup', (e) => this.onPointerUp_(e));
        this.eventTracker_.add(control, 'pointermove', (e) => this.onPointerMove_(e));
        control.setPointerCapture(e.pointerId);
        this.fireDragChanged_(true);
    }
    /**
     * @param dragChanged
     */
    fireDragChanged_(dragChanged) {
        this.dispatchEvent(new CustomEvent('margin-drag-changed', { bubbles: true, composed: true, detail: dragChanged }));
    }
    /**
     * Set display:none after the opacity transition for the controls is done.
     */
    onTransitionEnd_() {
        if (this.invisible_) {
            this.style.display = 'none';
        }
    }
    /**
     * Updates the translation transformation that translates pixel values in
     * the space of the HTML DOM.
     * @param translateTransform Updated value of the translation transformation.
     */
    updateTranslationTransform(translateTransform) {
        if (!translateTransform.equals(this.translateTransform_)) {
            this.translateTransform_ = translateTransform;
        }
    }
    /**
     * Updates the scaling transform that scales pixels values to point values.
     * @param scaleTransform Updated value of the scale transform.
     */
    updateScaleTransform(scaleTransform) {
        if (scaleTransform !== this.scaleTransform_) {
            this.scaleTransform_ = scaleTransform;
        }
    }
    /**
     * Clips margin controls to the given clip size in pixels.
     * @param clipSize Size to clip the margin controls to.
     */
    updateClippingMask(clipSize) {
        this.clipSize_ = clipSize;
    }
}
customElements.define(PrintPreviewMarginControlContainerElement.is, PrintPreviewMarginControlContainerElement);

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PluginProxyImpl {
    plugin_ = null;
    pluginReady() {
        return !!this.plugin_;
    }
    createPlugin(previewUid, index) {
        assert(!this.plugin_);
        const srcUrl = this.getPreviewUrl_(previewUid, index);
        this.plugin_ = pdfCreateOutOfProcessPlugin(srcUrl, 'chrome://print/pdf/index_print.html');
        this.plugin_.classList.add('preview-area-plugin');
        // NOTE: The plugin's 'id' field must be set to 'pdf-viewer' since
        // chrome/renderer/printing/print_render_frame_helper.cc actually
        // references it.
        this.plugin_.setAttribute('id', 'pdf-viewer');
        return this.plugin_;
    }
    /**
     * Get the URL for the plugin.
     * @param previewUid Unique identifier of preview.
     * @param index Page index for plugin.
     */
    getPreviewUrl_(previewUid, index) {
        return `chrome-untrusted://print/${previewUid}/${index}/print.pdf`;
    }
    resetPrintPreviewMode(previewUid, index, color, pages, modifiable) {
        this.plugin_.resetPrintPreviewMode(this.getPreviewUrl_(previewUid, index), color, pages, modifiable);
    }
    scrollPosition(scrollX, scrollY) {
        this.plugin_.scrollPosition(scrollX, scrollY);
    }
    sendKeyEvent(e) {
        this.plugin_.sendKeyEvent(e);
    }
    hideToolbar() {
        this.plugin_.hideToolbar();
    }
    setPointerEvents(eventsEnabled) {
        this.plugin_.style.pointerEvents = eventsEnabled ? 'auto' : 'none';
    }
    loadPreviewPage(previewUid, pageIndex, index) {
        this.plugin_.loadPreviewPage(this.getPreviewUrl_(previewUid, pageIndex), index);
    }
    setKeyEventCallback(keyEventCallback) {
        this.plugin_.setKeyEventCallback(keyEventCallback);
    }
    setLoadCompleteCallback(loadCompleteCallback) {
        this.plugin_.setLoadCompleteCallback(loadCompleteCallback);
    }
    setViewportChangedCallback(viewportChangedCallback) {
        this.plugin_.setViewportChangedCallback(viewportChangedCallback);
    }
    darkModeChanged(darkMode) {
        this.plugin_.darkModeChanged(darkMode);
    }
    static setInstance(obj) {
        instance$s = obj;
    }
    static getInstance() {
        return instance$s || (instance$s = new PluginProxyImpl());
    }
}
let instance$s = null;

let instance$r = null;
function getCss$r() {
    return instance$r || (instance$r = [...[getCss$K()], css `@keyframes dancing-dots-jump{0%{top:0}55%{top:0}60%{top:-10px}80%{top:3px}90%{top:-2px}95%{top:1px}100%{top:0}}span.jumping-dots>span{animation:dancing-dots-jump 1800ms infinite;padding:1px;position:relative}span.jumping-dots>span:nth-child(2){animation-delay:100ms}span.jumping-dots>span:nth-child(3){animation-delay:300ms}:host{display:block;height:100%;overflow:hidden;position:relative;user-select:none}.preview-area-plugin-wrapper{height:100%}.preview-area-plugin{border:none;cursor:inherit;height:100%;opacity:1;transition:opacity 200ms linear;transition-delay:100ms;width:100%}.preview-area-overlay-layer{background:var(--preview-area-background-color);display:flex;flex-direction:column;height:100%;justify-content:center;margin:0;opacity:1;position:absolute;transition:opacity 200ms linear;transition-delay:350ms;user-select:none;width:100%;z-index:1}.preview-area-overlay-layer.invisible{opacity:0;pointer-events:none;transition:opacity 100ms linear}.preview-area-message{color:var(--cr-primary-text-color);line-height:20px;margin:0 10px;position:relative;text-align:center}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$t() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div class="preview-area-overlay-layer ${this.getInvisible_()}"
    aria-hidden="${this.isInDisplayPreviewState_()}">
  <div class="preview-area-message">
    <div>
      <span .innerHTML="${this.currentMessage_()}"></span>
      <span class="preview-area-loading-message-jumping-dots
          ${this.getJumpingDots_()}" ?hidden="${!this.isPreviewLoading_()}">
        <span>.</span><span>.</span><span>.</span>
      </span>
    </div>
  </div>
</div>
<div class="preview-area-plugin-wrapper"></div>
<print-preview-margin-control-container id="marginControlContainer"
    .pageSize="${this.pageSize}" .documentMargins="${this.margins}"
    .measurementSystem="${this.measurementSystem}" state="${this.state}"
    ?preview-loaded="${this.previewLoaded()}"
    @text-focus-position="${this.onTextFocusPosition_}"
    @margin-drag-changed="${this.onMarginDragChanged_}">
</print-preview-margin-control-container>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var PreviewAreaState;
(function (PreviewAreaState) {
    PreviewAreaState["LOADING"] = "loading";
    PreviewAreaState["DISPLAY_PREVIEW"] = "display-preview";
    PreviewAreaState["OPEN_IN_PREVIEW_LOADING"] = "open-in-preview-loading";
    PreviewAreaState["OPEN_IN_PREVIEW_LOADED"] = "open-in-preview-loaded";
    PreviewAreaState["ERROR"] = "error";
})(PreviewAreaState || (PreviewAreaState = {}));
const PrintPreviewPreviewAreaElementBase = WebUiListenerMixinLit(I18nMixinLit(SettingsMixin(DarkModeMixin(CrLitElement))));
class PrintPreviewPreviewAreaElement extends PrintPreviewPreviewAreaElementBase {
    static get is() {
        return 'print-preview-preview-area';
    }
    static get styles() {
        return getCss$r();
    }
    render() {
        return getHtml$t.bind(this)();
    }
    static get properties() {
        return {
            destination: { type: Object },
            documentModifiable: { type: Boolean },
            error: {
                type: Number,
                notify: true,
            },
            margins: { type: Object },
            measurementSystem: { type: Object },
            pageSize: { type: Object },
            previewState: {
                type: String,
                notify: true,
            },
            state: { type: Number },
            pluginLoadComplete_: { type: Boolean },
            documentReady_: { type: Boolean },
        };
    }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #documentModifiable_accessor_storage = false;
    get documentModifiable() { return this.#documentModifiable_accessor_storage; }
    set documentModifiable(value) { this.#documentModifiable_accessor_storage = value; }
    #error_accessor_storage = null;
    get error() { return this.#error_accessor_storage; }
    set error(value) { this.#error_accessor_storage = value; }
    #margins_accessor_storage = null;
    get margins() { return this.#margins_accessor_storage; }
    set margins(value) { this.#margins_accessor_storage = value; }
    #measurementSystem_accessor_storage = null;
    get measurementSystem() { return this.#measurementSystem_accessor_storage; }
    set measurementSystem(value) { this.#measurementSystem_accessor_storage = value; }
    #pageSize_accessor_storage = new Size(612, 792);
    get pageSize() { return this.#pageSize_accessor_storage; }
    set pageSize(value) { this.#pageSize_accessor_storage = value; }
    #previewState_accessor_storage = PreviewAreaState.LOADING;
    get previewState() { return this.#previewState_accessor_storage; }
    set previewState(value) { this.#previewState_accessor_storage = value; }
    #state_accessor_storage = State.NOT_READY;
    get state() { return this.#state_accessor_storage; }
    set state(value) { this.#state_accessor_storage = value; }
    #pluginLoadComplete__accessor_storage = false;
    get pluginLoadComplete_() { return this.#pluginLoadComplete__accessor_storage; }
    set pluginLoadComplete_(value) { this.#pluginLoadComplete__accessor_storage = value; }
    #documentReady__accessor_storage = false;
    get documentReady_() { return this.#documentReady__accessor_storage; }
    set documentReady_(value) { this.#documentReady__accessor_storage = value; }
    nativeLayer_ = null;
    lastTicket_ = null;
    inFlightRequestId_ = -1;
    pluginProxy_ = PluginProxyImpl.getInstance();
    keyEventCallback_ = null;
    connectedCallback() {
        super.connectedCallback();
        this.nativeLayer_ = NativeLayerImpl.getInstance();
        this.addWebUiListener('page-preview-ready', this.onPagePreviewReady_.bind(this));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('state') || changedProperties.has('error')) {
            this.onStateOrErrorChange_();
        }
        if (changedPrivateProperties.has('documentReady_') ||
            changedPrivateProperties.has('pluginLoadComplete_')) {
            this.pluginOrDocumentStatusChanged_();
        }
    }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        this.addEventListener('pointerover', this.onPointerOver_.bind(this));
        this.addEventListener('pointerout', this.onPointerOut_.bind(this));
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('inDarkMode')) {
            this.onDarkModeChanged_();
        }
    }
    getLastTicketForTest() {
        return this.lastTicket_;
    }
    previewLoaded() {
        return this.documentReady_ && this.pluginLoadComplete_;
    }
    /**
     * Called when the pointer moves onto the component. Shows the margin
     * controls if custom margins are being used.
     */
    onPointerOver_() {
        this.$.marginControlContainer.setInvisible(false);
    }
    /**
     * Called when the pointer moves off of the component. Hides the margin
     * controls if they are visible.
     */
    onPointerOut_() {
        this.$.marginControlContainer.setInvisible(true);
    }
    pluginOrDocumentStatusChanged_() {
        if (!this.pluginLoadComplete_ || !this.documentReady_ ||
            this.previewState === PreviewAreaState.ERROR) {
            return;
        }
        this.previewState =
            this.previewState === PreviewAreaState.OPEN_IN_PREVIEW_LOADING ?
                PreviewAreaState.OPEN_IN_PREVIEW_LOADED :
                PreviewAreaState.DISPLAY_PREVIEW;
    }
    /**
     * @return 'invisible' if overlay is invisible, '' otherwise.
     */
    getInvisible_() {
        return this.isInDisplayPreviewState_() ? 'invisible' : '';
    }
    /**
     * @return Whether the preview area is in DISPLAY_PREVIEW state.
     */
    isInDisplayPreviewState_() {
        return this.previewState === PreviewAreaState.DISPLAY_PREVIEW;
    }
    /**
     * @return Whether the preview is currently loading.
     */
    isPreviewLoading_() {
        return this.previewState === PreviewAreaState.LOADING;
    }
    /**
     * @return 'jumping-dots' to enable animation, '' otherwise.
     */
    getJumpingDots_() {
        return this.isPreviewLoading_() ? 'jumping-dots' : '';
    }
    /**
     * @return The current preview area message to display.
     */
    currentMessage_() {
        switch (this.previewState) {
            case PreviewAreaState.LOADING:
                return this.i18nAdvanced('loading');
            case PreviewAreaState.DISPLAY_PREVIEW:
                return window.trustedTypes.emptyHTML;
            // 
            case PreviewAreaState.OPEN_IN_PREVIEW_LOADING:
            case PreviewAreaState.OPEN_IN_PREVIEW_LOADED:
                return this.i18nAdvanced('openingPDFInPreview');
            // 
            case PreviewAreaState.ERROR:
                // The preview area is responsible for displaying all errors except
                // print failed.
                return this.getErrorMessage_();
            default:
                return window.trustedTypes.emptyHTML;
        }
    }
    /**
     * @param forceUpdate Whether to force the preview area to update
     *     regardless of whether the print ticket has changed.
     */
    startPreview(forceUpdate) {
        if (!this.hasTicketChanged_() && !forceUpdate &&
            this.previewState !== PreviewAreaState.ERROR) {
            return;
        }
        this.previewState = PreviewAreaState.LOADING;
        this.documentReady_ = false;
        this.getPreview_().then(previewUid => {
            if (!this.documentModifiable) {
                this.onPreviewStart_(previewUid, -1);
            }
            this.documentReady_ = true;
        }, type => {
            if (type === 'SETTINGS_INVALID') {
                this.error = Error$1.INVALID_PRINTER;
                this.previewState = PreviewAreaState.ERROR;
            }
            else if (type !== 'CANCELLED') {
                console.warn('Preview failed in getPreview(): ' + type);
                this.error = Error$1.PREVIEW_FAILED;
                this.previewState = PreviewAreaState.ERROR;
            }
        });
    }
    // 
    /** Set the preview state to display the "opening in preview" message. */
    setOpeningPdfInPreview() {
        this.previewState = this.previewState === PreviewAreaState.LOADING ?
            PreviewAreaState.OPEN_IN_PREVIEW_LOADING :
            PreviewAreaState.OPEN_IN_PREVIEW_LOADED;
    }
    // 
    /**
     * @param previewUid The unique identifier of the preview.
     * @param index The index of the page to preview.
     */
    onPreviewStart_(previewUid, index) {
        if (!this.pluginProxy_.pluginReady()) {
            const plugin = this.pluginProxy_.createPlugin(previewUid, index);
            this.pluginProxy_.setKeyEventCallback(this.keyEventCallback_);
            this.shadowRoot.querySelector('.preview-area-plugin-wrapper').appendChild(plugin);
            this.pluginProxy_.setLoadCompleteCallback(this.onPluginLoadComplete_.bind(this));
            this.pluginProxy_.setViewportChangedCallback(this.onPreviewVisualStateChange_.bind(this));
        }
        this.pluginLoadComplete_ = false;
        if (this.inDarkMode) {
            this.pluginProxy_.darkModeChanged(true);
        }
        this.pluginProxy_.resetPrintPreviewMode(previewUid, index, !this.getSettingValue('color'), this.getSettingValue('pages'), this.documentModifiable);
    }
    /**
     * Called when the plugin loads the preview completely.
     * @param success Whether the plugin load succeeded or not.
     */
    onPluginLoadComplete_(success) {
        if (success) {
            this.pluginLoadComplete_ = true;
        }
        else {
            console.warn('Preview failed in onPluginLoadComplete_()');
            this.error = Error$1.PREVIEW_FAILED;
            this.previewState = PreviewAreaState.ERROR;
        }
    }
    /**
     * Called when the preview plugin's visual state has changed. This is a
     * consequence of scrolling or zooming the plugin. Updates the custom
     * margins component if shown.
     * @param pageX The horizontal offset for the page corner in pixels.
     * @param pageY The vertical offset for the page corner in pixels.
     * @param pageWidth The page width in pixels.
     * @param viewportWidth The viewport width in pixels.
     * @param viewportHeight The viewport height in pixels.
     */
    onPreviewVisualStateChange_(pageX, pageY, pageWidth, viewportWidth, viewportHeight) {
        // Ensure the PDF viewer isn't tabbable if the window is small enough that
        // the zoom toolbar isn't displayed.
        const tabindex = viewportWidth < 300 || viewportHeight < 200 ? '-1' : '0';
        this.shadowRoot.querySelector('.preview-area-plugin').setAttribute('tabindex', tabindex);
        this.$.marginControlContainer.updateTranslationTransform(new Coordinate2d(pageX, pageY));
        this.$.marginControlContainer.updateScaleTransform(pageWidth / this.pageSize.width);
        this.$.marginControlContainer.updateClippingMask(new Size(viewportWidth, viewportHeight));
        // Align the margin control container with the preview content area.
        // The offset may be caused by the scrollbar on the left in the preview
        // area in right-to-left direction.
        const previewDocument = this.shadowRoot
            .querySelector('.preview-area-plugin').contentDocument;
        if (previewDocument && previewDocument.documentElement) {
            this.$.marginControlContainer.style.left =
                previewDocument.documentElement.offsetLeft + 'px';
        }
    }
    /**
     * Called when a page's preview has been generated.
     * @param pageIndex The index of the page whose preview is ready.
     * @param previewUid The unique ID of the print preview UI.
     * @param previewResponseId The preview request ID that this page
     *     preview is a response to.
     */
    onPagePreviewReady_(pageIndex, previewUid, previewResponseId) {
        if (this.inFlightRequestId_ !== previewResponseId) {
            return;
        }
        const pageNumber = pageIndex + 1;
        let index = this.getSettingValue('pages').indexOf(pageNumber);
        // When pagesPerSheet > 1, the backend will always return page indices 0 to
        // N-1, where N is the total page count of the N-upped document.
        const pagesPerSheet = this.getSettingValue('pagesPerSheet');
        if (pagesPerSheet > 1) {
            index = pageIndex;
        }
        if (index === 0) {
            this.onPreviewStart_(previewUid, pageIndex);
        }
        if (index !== -1) {
            this.pluginProxy_.loadPreviewPage(previewUid, pageIndex, index);
        }
    }
    onDarkModeChanged_() {
        if (this.pluginProxy_.pluginReady()) {
            this.pluginProxy_.darkModeChanged(this.inDarkMode);
        }
        if (this.previewState === PreviewAreaState.DISPLAY_PREVIEW) {
            this.startPreview(true);
        }
    }
    /**
     * Processes a keyboard event that could possibly be used to change state of
     * the preview plugin.
     * @param e Keyboard event to process.
     */
    handleDirectionalKeyEvent(e) {
        // Make sure the PDF plugin is there.
        // We only care about: PageUp, PageDown, Left, Up, Right, Down.
        // If the user is holding a modifier key, ignore.
        if (!this.pluginProxy_.pluginReady() ||
            !['PageUp', 'PageDown', 'ArrowLeft', 'ArrowRight', 'ArrowUp',
                'ArrowDown']
                .includes(e.key) ||
            hasKeyModifiers(e)) {
            return;
        }
        // Don't handle the key event for these elements.
        const tagName = e.composedPath()[0].tagName;
        if (['INPUT', 'SELECT', 'EMBED'].includes(tagName)) {
            return;
        }
        // For the most part, if any div of header was the last clicked element,
        // then the active element is the body. Starting with the last clicked
        // element, and work up the DOM tree to see if any element has a
        // scrollbar. If there exists a scrollbar, do not handle the key event
        // here.
        const isEventHorizontal = ['ArrowLeft', 'ArrowRight'].includes(e.key);
        for (let i = 0; i < e.composedPath().length; i++) {
            const element = e.composedPath()[i];
            if (element.scrollHeight > element.clientHeight && !isEventHorizontal ||
                element.scrollWidth > element.clientWidth && isEventHorizontal) {
                return;
            }
        }
        // No scroll bar anywhere, or the active element is something else, like a
        // button. Note: buttons have a bigger scrollHeight than clientHeight.
        this.pluginProxy_.sendKeyEvent(e);
        e.preventDefault();
    }
    /**
     * Sends a message to the plugin to hide the toolbars after a delay.
     */
    hideToolbar() {
        if (!this.pluginProxy_.pluginReady()) {
            return;
        }
        this.pluginProxy_.hideToolbar();
    }
    /**
     * Set a callback that gets called when a key event is received that
     * originates in the plugin.
     * @param callback The callback to be called with a key event.
     */
    setPluginKeyEventCallback(callback) {
        this.keyEventCallback_ = callback;
    }
    /**
     * Called when dragging margins starts or stops.
     */
    onMarginDragChanged_(e) {
        if (!this.pluginProxy_.pluginReady()) {
            return;
        }
        // When hovering over the plugin (which may be in a separate iframe)
        // pointer events will be sent to the frame. When dragging the margins,
        // we don't want this to happen as it can cause the margin to stop
        // being draggable.
        this.pluginProxy_.setPointerEvents(!e.detail);
    }
    /**
     * @param e Contains information about where the plugin should scroll to.
     */
    onTextFocusPosition_(e) {
        // TODO(tkent): This is a workaround of a preview-area scrolling
        // issue. Blink scrolls preview-area on focus, but we don't want it.  We
        // should adjust scroll position of PDF preview and positions of
        // MarginContgrols here, or restructure the HTML so that the PDF review
        // and MarginControls are on the single scrollable container.
        // crbug.com/601341
        this.scrollTop = 0;
        this.scrollLeft = 0;
        const position = e.detail;
        if (position.x === 0 && position.y === 0) {
            return;
        }
        this.pluginProxy_.scrollPosition(position.x, position.y);
    }
    /**
     * @return Whether margin settings are valid for the print ticket.
     */
    marginsValid_() {
        const type = this.getSettingValue('margins');
        if (!Object.values(MarginsType).includes(type)) {
            // Unrecognized margins type.
            return false;
        }
        if (type !== MarginsType.CUSTOM) {
            return true;
        }
        const customMargins = this.getSettingValue('customMargins');
        return customMargins.marginTop !== undefined &&
            customMargins.marginLeft !== undefined &&
            customMargins.marginBottom !== undefined &&
            customMargins.marginRight !== undefined;
    }
    hasTicketChanged_() {
        if (!this.marginsValid_()) {
            return false;
        }
        if (!this.lastTicket_) {
            return true;
        }
        assert(this.destination);
        const lastTicket = this.lastTicket_;
        // Margins
        const newMarginsType = this.getSettingValue('margins');
        if (newMarginsType !== lastTicket.marginsType &&
            newMarginsType !== MarginsType.CUSTOM) {
            return true;
        }
        if (newMarginsType === MarginsType.CUSTOM) {
            const customMargins = this.getSettingValue('customMargins');
            // Change in custom margins values.
            if (!!lastTicket.marginsCustom &&
                (lastTicket.marginsCustom.marginTop !== customMargins.marginTop ||
                    lastTicket.marginsCustom.marginLeft !== customMargins.marginLeft ||
                    lastTicket.marginsCustom.marginRight !== customMargins.marginRight ||
                    lastTicket.marginsCustom.marginBottom !==
                        customMargins.marginBottom)) {
                return true;
            }
            // Changed to custom margins from a different margins type.
            if (!this.margins) {
                return false;
            }
            const customMarginsChanged = Object.values(CustomMarginsOrientation).some(side => {
                return this.margins.get(side) !==
                    customMargins[MARGIN_KEY_MAP.get(side)];
            });
            if (customMarginsChanged) {
                return true;
            }
        }
        // Simple settings: ranges, layout, header/footer, pages per sheet, fit to
        // page, css background, selection only, rasterize, scaling, dpi
        if (!areRangesEqual(this.getSettingValue('ranges'), lastTicket.pageRange) ||
            this.getSettingValue('layout') !== lastTicket.landscape ||
            this.getColorForTicket_() !== lastTicket.color ||
            this.getSettingValue('headerFooter') !==
                lastTicket.headerFooterEnabled ||
            this.getSettingValue('cssBackground') !==
                lastTicket.shouldPrintBackgrounds ||
            this.getSettingValue('selectionOnly') !==
                lastTicket.shouldPrintSelectionOnly ||
            this.getSettingValue('rasterize') !== lastTicket.rasterizePDF ||
            this.isScalingChanged_(lastTicket)) {
            return true;
        }
        // Pages per sheet. If margins are non-default, wait for the return to
        // default margins to trigger a request.
        if (this.getSettingValue('pagesPerSheet') !== lastTicket.pagesPerSheet &&
            this.getSettingValue('margins') === MarginsType.DEFAULT) {
            return true;
        }
        // Media size
        const newValue = this.getSettingValue('mediaSize');
        if (newValue.height_microns !== lastTicket.mediaSize.height_microns ||
            newValue.width_microns !== lastTicket.mediaSize.width_microns ||
            newValue.imageable_area_left_microns !==
                lastTicket.mediaSize.imageable_area_left_microns ||
            newValue.imageable_area_bottom_microns !==
                lastTicket.mediaSize.imageable_area_bottom_microns ||
            newValue.imageable_area_right_microns !==
                lastTicket.mediaSize.imageable_area_right_microns ||
            newValue.imageable_area_top_microns !==
                lastTicket.mediaSize.imageable_area_top_microns ||
            (this.destination.id !== lastTicket.deviceName &&
                this.getSettingValue('margins') === MarginsType.MINIMUM)) {
            return true;
        }
        // Destination
        if (this.destination.type !== lastTicket.printerType) {
            return true;
        }
        return false;
    }
    /** @return Native color model of the destination. */
    getColorForTicket_() {
        assert(this.destination);
        return this.destination.getNativeColorModel(this.getSettingValue('color'));
    }
    /** @return Scale factor for print ticket. */
    getScaleFactorForTicket_() {
        return this.getSettingValue(this.getScalingSettingKey_()) ===
            ScalingType.CUSTOM ?
            parseInt(this.getSettingValue('scaling'), 10) :
            100;
    }
    /** @return Appropriate key for the scaling type setting. */
    getScalingSettingKey_() {
        return this.getSetting('scalingTypePdf').available ? 'scalingTypePdf' :
            'scalingType';
    }
    /**
     * @param lastTicket Last print ticket.
     * @return Whether new scaling settings update the previewed
     *     document.
     */
    isScalingChanged_(lastTicket) {
        // Preview always updates if the scale factor is changed.
        if (this.getScaleFactorForTicket_() !== lastTicket.scaleFactor) {
            return true;
        }
        // If both scale factors and type match, no scaling change happened.
        const scalingType = this.getSettingValue(this.getScalingSettingKey_());
        if (scalingType === lastTicket.scalingType) {
            return false;
        }
        // Scaling doesn't always change because of a scalingType change. Changing
        // between custom scaling with a scale factor of 100 and default scaling
        // makes no difference.
        const defaultToCustom = scalingType === ScalingType.DEFAULT &&
            lastTicket.scalingType === ScalingType.CUSTOM;
        const customToDefault = scalingType === ScalingType.CUSTOM &&
            lastTicket.scalingType === ScalingType.DEFAULT;
        return !defaultToCustom && !customToDefault;
    }
    /**
     * @param dpiField The field in dpi to retrieve.
     * @return Field value.
     */
    getDpiForTicket_(dpiField) {
        const dpi = this.getSettingValue('dpi');
        const value = (dpi && dpiField in dpi) ? dpi[dpiField] : 0;
        return value;
    }
    /**
     * Requests a preview from the native layer.
     * @return Promise that resolves when the preview has been
     *     generated.
     */
    getPreview_() {
        assert(this.destination);
        this.inFlightRequestId_++;
        const ticket = {
            pageRange: this.getSettingValue('ranges'),
            mediaSize: this.getSettingValue('mediaSize'),
            landscape: this.getSettingValue('layout'),
            color: this.getColorForTicket_(),
            headerFooterEnabled: this.getSettingValue('headerFooter'),
            marginsType: this.getSettingValue('margins'),
            pagesPerSheet: this.getSettingValue('pagesPerSheet'),
            isFirstRequest: this.inFlightRequestId_ === 0,
            requestID: this.inFlightRequestId_,
            previewModifiable: this.documentModifiable,
            scaleFactor: this.getScaleFactorForTicket_(),
            scalingType: this.getSettingValue(this.getScalingSettingKey_()),
            shouldPrintBackgrounds: this.getSettingValue('cssBackground'),
            shouldPrintSelectionOnly: this.getSettingValue('selectionOnly'),
            // NOTE: Even though the remaining fields don't directly relate to the
            // preview, they still need to be included.
            // e.g. printing::PrintSettingsFromJobSettings() still checks for them.
            collate: true,
            copies: 1,
            deviceName: this.destination.id,
            dpiHorizontal: this.getDpiForTicket_('horizontal_dpi'),
            dpiVertical: this.getDpiForTicket_('vertical_dpi'),
            duplex: this.getSettingValue('duplex') ? DuplexMode.LONG_EDGE :
                DuplexMode.SIMPLEX,
            printerType: this.destination.type,
            rasterizePDF: this.getSettingValue('rasterize'),
        };
        if (this.getSettingValue('margins') === MarginsType.CUSTOM) {
            ticket.marginsCustom = this.getSettingValue('customMargins');
        }
        this.lastTicket_ = ticket;
        this.dispatchEvent(new CustomEvent('preview-start', { bubbles: true, composed: true, detail: this.inFlightRequestId_ }));
        return this.nativeLayer_.getPreview(JSON.stringify(ticket));
    }
    onStateOrErrorChange_() {
        if ((this.state === State.ERROR || this.state === State.FATAL_ERROR) &&
            this.getErrorMessage_().toString() !== '') {
            this.previewState = PreviewAreaState.ERROR;
        }
    }
    /** @return The error message to display in the preview area. */
    getErrorMessage_() {
        switch (this.error) {
            case Error$1.INVALID_PRINTER:
                return this.i18nAdvanced('invalidPrinterSettings', {
                    substitutions: [],
                    tags: ['BR'],
                });
            case Error$1.PREVIEW_FAILED:
                return this.i18nAdvanced('previewFailed');
            default:
                return window.trustedTypes.emptyHTML;
        }
    }
}
customElements.define(PrintPreviewPreviewAreaElement.is, PrintPreviewPreviewAreaElement);

let instance$q = null;
function getCss$q() {
    return instance$q || (instance$q = [...[], css `:host{--collapse-duration:var(--iron-collapse-transition-duration,300ms);display:block;transition:max-height var(--collapse-duration) ease-out;overflow:visible}:host([no-animation]){transition:none}:host(.collapse-closed){display:none}:host(:not(.collapse-opened)){overflow:hidden}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$s() {
    return html `<slot></slot>`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrCollapseElement extends CrLitElement {
    static get is() {
        return 'cr-collapse';
    }
    static get styles() {
        return getCss$q();
    }
    render() {
        return getHtml$s.bind(this)();
    }
    static get properties() {
        return {
            opened: {
                type: Boolean,
                notify: true,
            },
            noAnimation: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #opened_accessor_storage = false;
    get opened() { return this.#opened_accessor_storage; }
    set opened(value) { this.#opened_accessor_storage = value; }
    #noAnimation_accessor_storage = false;
    get noAnimation() { return this.#noAnimation_accessor_storage; }
    set noAnimation(value) { this.#noAnimation_accessor_storage = value; }
    toggle() {
        this.opened = !this.opened;
    }
    show() {
        this.opened = true;
    }
    hide() {
        this.opened = false;
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'group');
        }
        this.setAttribute('aria-hidden', 'true');
        this.addEventListener('transitionend', (e) => this.onTransitionEnd_(e));
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (!changedProperties.has('opened')) {
            return;
        }
        this.setAttribute('aria-hidden', this.opened ? 'false' : 'true');
        this.classList.toggle('collapse-closed', false);
        this.classList.toggle('collapse-opened', false);
        this.updateHeight_(this.opened, changedProperties.get('opened'));
        // Focus the current collapse.
        if (this.opened) {
            this.focus();
        }
    }
    updateHeight_(opening, lastOpened) {
        const finalMaxHeight = opening ? '' : '0px';
        const animationStartSize = `${this.getBoundingClientRect().height}px`;
        const animationEndSize = opening ? `${this.scrollHeight}px` : '0px';
        const willAnimate = lastOpened !== undefined && !this.noAnimation &&
            this.style.maxHeight !== finalMaxHeight &&
            animationStartSize !== animationEndSize;
        if (willAnimate && !opening) {
            // Force layout to ensure transition will go. Set maxHeight to a px
            // value and scrollTop to itself.
            this.style.maxHeight = animationStartSize;
            this.scrollTop = this.scrollTop;
        }
        // Set the final size.
        this.style.maxHeight = animationEndSize;
        // If it won't animate, set correct classes. Otherwise these are set in
        // onTransitionEnd_().
        if (!willAnimate) {
            this.updateStyles_();
        }
    }
    onTransitionEnd_(e) {
        if (e.composedPath()[0] === this) {
            this.updateStyles_();
        }
    }
    updateStyles_() {
        this.style.maxHeight = this.opened ? '' : '0px';
        this.classList.toggle('collapse-closed', !this.opened);
        this.classList.toggle('collapse-opened', this.opened);
    }
}
customElements.define(CrCollapseElement.is, CrCollapseElement);

let instance$p = null;
function getCss$p() {
    return instance$p || (instance$p = [...[getCss$x()], css `:host{display:flex;padding-inline-start:var(--print-preview-sidebar-margin)}::slotted([slot=controls]),::slotted([slot=title]){display:flex;flex-direction:column;justify-content:center;min-height:var(--print-preview-row-height);word-break:break-word}::slotted([slot=controls]){flex:1;overflow:hidden}::slotted([slot=title]){color:var(--cr-primary-text-color);flex:none;font-size:1em;line-height:calc(20/13 * 1em);width:var(--print-preview-title-width)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$r() {
    // clang-format off
    return html `<!--_html_template_start_-->
<slot name="title"></slot>
<slot name="controls"></slot>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PrintPreviewSettingsSectionElement extends CrLitElement {
    static get is() {
        return 'print-preview-settings-section';
    }
    static get styles() {
        return getCss$p();
    }
    render() {
        return getHtml$r.bind(this)();
    }
}
customElements.define(PrintPreviewSettingsSectionElement.is, PrintPreviewSettingsSectionElement);

let instance$o = null;
function getCss$o() {
    return instance$o || (instance$o = [...[getCss$x()], css `cr-button{height:fit-content;min-height:32px;text-align:center;width:calc(100% - 2 * var(--print-preview-sidebar-margin))}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$q() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span slot="title"></span>
  <div slot="controls">
    <cr-button id="button" ?disabled="${this.disabled}"
        @click="${this.onButtonClick_}">
      $i18n{newShowAdvancedOptions}
    </cr-button>
  </div>
</print-preview-settings-section>
${this.showAdvancedDialog_ ? html `
  <print-preview-advanced-settings-dialog
      .destination="${this.destination}" @close="${this.onDialogClose_}">
  </print-preview-advanced-settings-dialog>
` : ''}
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PrintPreviewAdvancedOptionsSettingsElement extends CrLitElement {
    static get is() {
        return 'print-preview-advanced-options-settings';
    }
    static get styles() {
        return getCss$o();
    }
    render() {
        return getHtml$q.bind(this)();
    }
    static get properties() {
        return {
            disabled: { type: Boolean },
            destination: { type: Object },
            showAdvancedDialog_: { type: Boolean },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #showAdvancedDialog__accessor_storage = false;
    get showAdvancedDialog_() { return this.#showAdvancedDialog__accessor_storage; }
    set showAdvancedDialog_(value) { this.#showAdvancedDialog__accessor_storage = value; }
    onButtonClick_() {
        this.showAdvancedDialog_ = true;
    }
    onDialogClose_() {
        this.showAdvancedDialog_ = false;
        this.$.button.focus();
    }
}
customElements.define(PrintPreviewAdvancedOptionsSettingsElement.is, PrintPreviewAdvancedOptionsSettingsElement);

let instance$n = null;
function getCss$n() {
    return instance$n || (instance$n = [...[], css `:host{display:flex;flex-direction:row;justify-content:flex-end;padding-bottom:var(--print-preview-sidebar-margin);padding-inline-end:var(--print-preview-sidebar-margin);padding-top:16px}@media (prefers-color-scheme:light){:host{background-color:white}}:host cr-button:not(:last-child){margin-inline-end:8px}:host cr-button:last-child{margin-inline-end:0}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$p() {
    // clang-format off
    return html `<!--_html_template_start_-->

  <cr-button class="cancel-button" @click="${this.onCancelClick_}">
    $i18n{cancel}
  </cr-button>

  <cr-button class="action-button" @click="${this.onPrintClick_}"
      ?disabled="${!this.printButtonEnabled_}">
    ${this.printButtonLabel_}
  </cr-button>

<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PrintPreviewButtonStripElement extends CrLitElement {
    static get is() {
        return 'print-preview-button-strip';
    }
    static get styles() {
        return getCss$n();
    }
    render() {
        return getHtml$p.bind(this)();
    }
    static get properties() {
        return {
            destination: { type: Object },
            firstLoad: { type: Boolean },
            state: { type: Number },
            printButtonEnabled_: { type: Boolean },
            printButtonLabel_: { type: String },
        };
    }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #firstLoad_accessor_storage = false;
    get firstLoad() { return this.#firstLoad_accessor_storage; }
    set firstLoad(value) { this.#firstLoad_accessor_storage = value; }
    #state_accessor_storage = State.NOT_READY;
    get state() { return this.#state_accessor_storage; }
    set state(value) { this.#state_accessor_storage = value; }
    #printButtonEnabled__accessor_storage = false;
    get printButtonEnabled_() { return this.#printButtonEnabled__accessor_storage; }
    set printButtonEnabled_(value) { this.#printButtonEnabled__accessor_storage = value; }
    #printButtonLabel__accessor_storage = loadTimeData.getString('printButton');
    get printButtonLabel_() { return this.#printButtonLabel__accessor_storage; }
    set printButtonLabel_(value) { this.#printButtonLabel__accessor_storage = value; }
    lastState_ = State.NOT_READY;
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('destination')) {
            this.printButtonLabel_ =
                loadTimeData.getString(this.isPdf_() ? 'saveButton' : 'printButton');
        }
        if (changedProperties.has('state')) {
            this.updatePrintButtonEnabled_();
        }
    }
    onPrintClick_() {
        this.fire('print-requested');
    }
    onCancelClick_() {
        this.fire('cancel-requested');
    }
    isPdf_() {
        return !!this.destination &&
            this.destination.type === PrinterType.PDF_PRINTER;
    }
    updatePrintButtonEnabled_() {
        switch (this.state) {
            case (State.PRINTING):
                this.printButtonEnabled_ = false;
                break;
            case (State.READY):
                this.printButtonEnabled_ = true;
                if (this.firstLoad || this.lastState_ === State.PRINTING) {
                    this.shadowRoot
                        .querySelector('cr-button.action-button').focus();
                    this.fire('print-button-focused');
                }
                break;
            default:
                this.printButtonEnabled_ = false;
                break;
        }
        this.lastState_ = this.state;
    }
}
customElements.define(PrintPreviewButtonStripElement.is, PrintPreviewButtonStripElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$o() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span id="color-label" slot="title">$i18n{optionColor}</span>
  <div slot="controls">
    <select class="md-select" aria-labelledby="color-label"
        ?disabled="${this.computeDisabled_()}" .value="${this.selectedValue}"
        @change="${this.onSelectChange}">
      <option value="bw" selected>$i18n{optionBw}</option>
      <option value="color">$i18n{optionColor}</option>
    </select>
  </div>
</print-preview-settings-section>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class Debouncer {
    duration_ = 0;
    timeoutId_ = -1;
    constructor(duration) {
        this.duration_ = duration;
    }
    clear() {
        if (this.timeoutId_ !== -1) {
            clearTimeout(this.timeoutId_);
            this.timeoutId_ = -1;
        }
    }
    call(fn) {
        this.clear();
        this.timeoutId_ = setTimeout(() => {
            this.timeoutId_ = -1;
            fn();
        }, this.duration_);
    }
}

const SelectMixin = (superClass) => {
    class SelectMixin extends superClass {
        static get properties() {
            return {
                selectedValue: { type: String },
            };
        }
        #selectedValue_accessor_storage = '';
        get selectedValue() { return this.#selectedValue_accessor_storage; }
        set selectedValue(value) { this.#selectedValue_accessor_storage = value; }
        debouncer_ = new Debouncer(100);
        onSelectChange(e) {
            const newValue = e.target.value;
            this.debouncer_.call(() => this.callProcessSelectChange_(newValue));
        }
        callProcessSelectChange_(newValue) {
            if (!this.isConnected || newValue === this.selectedValue) {
                return;
            }
            this.selectedValue = newValue;
            this.onProcessSelectChange(newValue);
            // For testing only
            this.dispatchEvent(new CustomEvent('process-select-change', { bubbles: true, composed: true }));
        }
        onProcessSelectChange(_value) { }
    }
    return SelectMixin;
};

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewColorSettingsElementBase = SettingsMixin(SelectMixin(CrLitElement));
class PrintPreviewColorSettingsElement extends PrintPreviewColorSettingsElementBase {
    static get is() {
        return 'print-preview-color-settings';
    }
    static get styles() {
        return [
            getCss$x(),
            getCss$z(),
        ];
    }
    render() {
        return getHtml$o.bind(this)();
    }
    static get properties() {
        return {
            disabled: { type: Boolean },
            managed_: { type: Boolean },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #managed__accessor_storage = false;
    get managed_() { return this.#managed__accessor_storage; }
    set managed_(value) { this.#managed__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('color.value', this.onColorSettingChange_.bind(this));
        this.onColorSettingChange_(this.getSettingValue('color'));
        this.addSettingObserver('color.setByGlobalPolicy', (value) => {
            this.managed_ = value;
        });
        this.managed_ = this.getSetting('color').setByGlobalPolicy;
    }
    onColorSettingChange_(newValue) {
        this.selectedValue = newValue ? 'color' : 'bw';
    }
    /**
     * @return Whether drop-down should be disabled.
     */
    computeDisabled_() {
        return this.disabled || this.managed_;
    }
    /** @param value The new select value. */
    onProcessSelectChange(value) {
        this.setSetting('color', value === 'color');
    }
}
customElements.define(PrintPreviewColorSettingsElement.is, PrintPreviewColorSettingsElement);

let instance$m = null;
function getCss$m() {
    return instance$m || (instance$m = [...[getCss$x()], css `:host{--cr-input-width:calc(4em + 16px)}#sectionTitle{align-self:baseline}cr-input{margin-inline-end:16px;min-height:var(--print-preview-row-height);overflow:hidden}.input-wrapper{align-items:center;display:flex}:host([input-valid]) cr-input{--cr-input-error-display:none}span[slot=suffix]{max-width:calc(100% - 5em);min-height:var(--print-preview-row-height);overflow:hidden;word-wrap:break-word}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$n() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span slot="title" id="sectionTitle">${this.inputLabel}</span>
  <div slot="controls" id="controls">
    <span class="input-wrapper">
      <cr-input id="userValue" type="number" class="stroked"
          max="${this.maxValue}" min="${this.minValue}" data-timeout-delay="250"
          ?disabled="${this.getDisabled_()}" @keydown="${this.onKeydown_}"
          @blur="${this.onBlur_}" aria-label="${this.inputAriaLabel}"
          error-message="${this.errorMessage_}" auto-validate>
        <span slot="suffix">
          <slot name="opt-inside-content"></slot>
        </span>
      </cr-input>
    </span>
  </div>
</print-preview-settings-section>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewNumberSettingsSectionElementBase = WebUiListenerMixinLit(InputMixin(CrLitElement));
class PrintPreviewNumberSettingsSectionElement extends PrintPreviewNumberSettingsSectionElementBase {
    static get is() {
        return 'print-preview-number-settings-section';
    }
    static get styles() {
        return getCss$m();
    }
    render() {
        return getHtml$n.bind(this)();
    }
    static get properties() {
        return {
            inputValid: {
                type: Boolean,
                notify: true,
                reflect: true,
            },
            currentValue: {
                type: String,
                notify: true,
            },
            defaultValue: { type: String },
            maxValue: { type: Number },
            minValue: { type: Number },
            inputLabel: { type: String },
            inputAriaLabel: { type: String },
            hintMessage: { type: String },
            disabled: { type: Boolean },
            errorMessage_: { type: String },
        };
    }
    #currentValue_accessor_storage = '';
    get currentValue() { return this.#currentValue_accessor_storage; }
    set currentValue(value) { this.#currentValue_accessor_storage = value; }
    #defaultValue_accessor_storage = '';
    get defaultValue() { return this.#defaultValue_accessor_storage; }
    set defaultValue(value) { this.#defaultValue_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #hintMessage_accessor_storage = '';
    get hintMessage() { return this.#hintMessage_accessor_storage; }
    set hintMessage(value) { this.#hintMessage_accessor_storage = value; }
    #inputAriaLabel_accessor_storage = '';
    get inputAriaLabel() { return this.#inputAriaLabel_accessor_storage; }
    set inputAriaLabel(value) { this.#inputAriaLabel_accessor_storage = value; }
    #inputLabel_accessor_storage = '';
    get inputLabel() { return this.#inputLabel_accessor_storage; }
    set inputLabel(value) { this.#inputLabel_accessor_storage = value; }
    #inputValid_accessor_storage = true;
    get inputValid() { return this.#inputValid_accessor_storage; }
    set inputValid(value) { this.#inputValid_accessor_storage = value; }
    #minValue_accessor_storage;
    get minValue() { return this.#minValue_accessor_storage; }
    set minValue(value) { this.#minValue_accessor_storage = value; }
    #maxValue_accessor_storage;
    get maxValue() { return this.#maxValue_accessor_storage; }
    set maxValue(value) { this.#maxValue_accessor_storage = value; }
    #errorMessage__accessor_storage = '';
    get errorMessage_() { return this.#errorMessage__accessor_storage; }
    set errorMessage_(value) { this.#errorMessage__accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('input-change', e => this.onInputChangeEvent_(e));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('hintMessage') ||
            changedProperties.has('inputValid')) {
            this.errorMessage_ = this.inputValid ? '' : this.hintMessage;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('currentValue')) {
            this.onCurrentValueChanged_();
        }
    }
    /** @return The cr-input field element for InputBehavior. */
    getInput() {
        return this.$.userValue;
    }
    /**
     * @param e Contains the new input value.
     */
    onInputChangeEvent_(e) {
        if (e.detail === '') {
            // Set current value first in this case, because if the input was
            // previously invalid, it will become valid in the line below but
            // we do not want to set the setting to the invalid value.
            this.currentValue = '';
        }
        this.inputValid = this.$.userValue.validate();
        this.currentValue = e.detail;
    }
    /**
     * @return Whether the input should be disabled.
     */
    getDisabled_() {
        return this.disabled && this.inputValid;
    }
    onKeydown_(e) {
        if (['.', 'e', 'E', '-', '+'].includes(e.key)) {
            e.preventDefault();
            return;
        }
        if (e.key === 'Enter') {
            this.onBlur_();
        }
    }
    onBlur_() {
        if (this.currentValue === '') {
            this.currentValue = this.defaultValue;
            this.inputValid = this.$.userValue.validate();
        }
        if (this.$.userValue.value === '') {
            this.$.userValue.value = this.defaultValue;
        }
    }
    onCurrentValueChanged_() {
        if (this.currentValue !== this.$.userValue.value) {
            this.$.userValue.value = this.currentValue;
            this.inputValid = this.$.userValue.validate();
        }
        this.resetString();
    }
}
customElements.define(PrintPreviewNumberSettingsSectionElement.is, PrintPreviewNumberSettingsSectionElement);

let instance$l = null;
function getCss$l() {
    return instance$l || (instance$l = [...[getCss$x()], css `#collate{margin-inline-start:16px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$m() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-number-settings-section max-value="${this.copiesMax_}"
    min-value="1" default-value="1" input-label="$i18n{copiesLabel}"
    input-aria-label="$i18n{copiesLabel}"
    ?disabled="${this.disabled}" current-value="${this.currentValue_}"
    @current-value-changed="${this.onCurrentValueChanged_}"
    ?input-valid="${this.inputValid_}"
    @input-valid-changed="${this.onInputValidChanged_}"
    hint-message="${this.getHintMessage_()}">
  <div slot="opt-inside-content" class="checkbox" aria-live="polite"
      ?hidden="${this.collateHidden_()}">
    <cr-checkbox id="collate" @change="${this.onCollateChange_}"
        ?disabled="${this.disabled}" aria-labelledby="copies-collate-label">
      <span id="copies-collate-label">$i18n{optionCollate}</span>
    </cr-checkbox>
  </div>
</print-preview-number-settings-section>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Maximum number of copies supported by the printer if not explicitly
 * specified.
 */
const DEFAULT_MAX_COPIES = 999;
const PrintPreviewCopiesSettingsElementBase = SettingsMixin(CrLitElement);
class PrintPreviewCopiesSettingsElement extends PrintPreviewCopiesSettingsElementBase {
    static get is() {
        return 'print-preview-copies-settings';
    }
    static get styles() {
        return getCss$l();
    }
    render() {
        return getHtml$m.bind(this)();
    }
    static get properties() {
        return {
            capability: { type: Object },
            copiesMax_: { type: Number },
            currentValue_: { type: String },
            inputValid_: { type: Boolean },
            disabled: { type: Boolean },
            collateAvailable_: { type: Boolean },
        };
    }
    #capability_accessor_storage = null;
    get capability() { return this.#capability_accessor_storage; }
    set capability(value) { this.#capability_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #copiesMax__accessor_storage = DEFAULT_MAX_COPIES;
    get copiesMax_() { return this.#copiesMax__accessor_storage; }
    set copiesMax_(value) { this.#copiesMax__accessor_storage = value; }
    #currentValue__accessor_storage = '';
    get currentValue_() { return this.#currentValue__accessor_storage; }
    set currentValue_(value) { this.#currentValue__accessor_storage = value; }
    #inputValid__accessor_storage = false;
    get inputValid_() { return this.#inputValid__accessor_storage; }
    set inputValid_(value) { this.#inputValid__accessor_storage = value; }
    #collateAvailable__accessor_storage = false;
    get collateAvailable_() { return this.#collateAvailable__accessor_storage; }
    set collateAvailable_(value) { this.#collateAvailable__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('copies.value', () => this.onSettingsChanged_());
        this.addSettingObserver('collate.*', () => this.onSettingsChanged_());
        this.onSettingsChanged_();
        this.addSettingObserver('collate.available', (value) => {
            this.collateAvailable_ = value;
        });
        this.collateAvailable_ = this.getSetting('collate').available;
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('capability')) {
            this.copiesMax_ = this.computeCopiesMax_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('currentValue_')) {
            this.onInputChanged_();
        }
    }
    /**
     * @return The maximum number of copies this printer supports.
     */
    computeCopiesMax_() {
        return (this.capability && this.capability.max) ? this.capability.max :
            DEFAULT_MAX_COPIES;
    }
    /**
     * @return The message to show as hint.
     */
    getHintMessage_() {
        return loadTimeData.getStringF('copiesInstruction', this.copiesMax_);
    }
    /**
     * Updates the input string when the setting has been initialized.
     */
    onSettingsChanged_() {
        const copies = this.getSetting('copies');
        if (this.inputValid_) {
            this.currentValue_ = copies.value.toString();
        }
        const collate = this.getSetting('collate');
        this.$.collate.checked = collate.value;
    }
    /**
     * Updates model.copies and model.copiesInvalid based on the validity
     * and current value of the copies input.
     */
    onInputChanged_() {
        if (this.currentValue_ !== '' &&
            this.currentValue_ !== this.getSettingValue('copies').toString()) {
            this.setSetting('copies', this.inputValid_ ? parseInt(this.currentValue_, 10) : 1);
        }
        this.setSettingValid('copies', this.inputValid_);
    }
    /**
     * @return Whether collate checkbox should be hidden.
     */
    collateHidden_() {
        return !this.collateAvailable_ || !this.inputValid_ ||
            this.currentValue_ === '' || parseInt(this.currentValue_, 10) === 1;
    }
    onCollateChange_() {
        this.setSetting('collate', this.$.collate.checked);
    }
    onCurrentValueChanged_(e) {
        this.currentValue_ = e.detail.value;
    }
    onInputValidChanged_(e) {
        this.inputValid_ = e.detail.value;
    }
}
customElements.define(PrintPreviewCopiesSettingsElement.is, PrintPreviewCopiesSettingsElement);

let instance$k = null;
function getCss$k() {
    return instance$k || (instance$k = [...[getCss$x(), getCss$z()], css `select.md-select{margin:2px;--md-select-width:calc(100% - 4px)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$l() {
    // clang-format off
    return html `<!--_html_template_start_-->
<select class="md-select" ?disabled="${this.disabled}"
    aria-label="${this.ariaLabel}" .value="${this.selectedValue}"
    @change="${this.onSelectChange}">
  ${this.capability ? html `
    ${this.capability.option.map(item => html `
      <option ?selected="${this.isSelected_(item)}"
          value="${this.getValue_(item)}">
        ${this.getDisplayName_(item)}
      </option>
    `)}
  ` : ''}
</select>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewSettingsSelectElementBase = SettingsMixin(SelectMixin(CrLitElement));
class PrintPreviewSettingsSelectElement extends PrintPreviewSettingsSelectElementBase {
    static get is() {
        return 'print-preview-settings-select';
    }
    static get styles() {
        return getCss$k();
    }
    render() {
        return getHtml$l.bind(this)();
    }
    static get properties() {
        return {
            ariaLabel: { type: String },
            capability: { type: Object },
            settingName: { type: String },
            disabled: { type: Boolean },
        };
    }
    #ariaLabel_accessor_storage = '';
    get ariaLabel() { return this.#ariaLabel_accessor_storage; }
    set ariaLabel(value) { this.#ariaLabel_accessor_storage = value; }
    #capability_accessor_storage = null;
    get capability() { return this.#capability_accessor_storage; }
    set capability(value) { this.#capability_accessor_storage = value; }
    #settingName_accessor_storage = 'dpi';
    get settingName() { return this.#settingName_accessor_storage; }
    set settingName(value) { this.#settingName_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    /**
     * @param option Option to check.
     * @return Whether the option is selected.
     */
    isSelected_(option) {
        return this.getValue_(option) === this.selectedValue ||
            (!!option.is_default && this.selectedValue === '');
    }
    selectValue(value) {
        this.selectedValue = value;
    }
    /**
     * @param option Option to get the value for.
     * @return Value for the option.
     */
    getValue_(option) {
        return JSON.stringify(option);
    }
    /**
     * @param option Option to get the display name for.
     * @return Display name for the option.
     */
    getDisplayName_(option) {
        let displayName = option.custom_display_name;
        if (!displayName && option.custom_display_name_localized) {
            displayName =
                getStringForCurrentLocale(option.custom_display_name_localized);
        }
        return displayName || option.name || '';
    }
    onProcessSelectChange(value) {
        let newValue = null;
        try {
            newValue = JSON.parse(value);
        }
        catch (e) {
            assertNotReached();
        }
        if (value !== JSON.stringify(this.getSettingValue(this.settingName))) {
            this.setSetting(this.settingName, newValue);
        }
    }
}
customElements.define(PrintPreviewSettingsSelectElement.is, PrintPreviewSettingsSelectElement);

let instance$j = null;
function getCss$j() {
    return instance$j || (instance$j = [...[getCss$x()], css `:host print-preview-settings-select{margin:0 calc(var(--print-preview-sidebar-margin) - 2px)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$k() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span id="dpi-label" slot="title">$i18n{dpiLabel}</span>
  <div slot="controls">
    <print-preview-settings-select aria-label="$i18n{dpiLabel}"
        .capability="${this.capabilityWithLabels_}" setting-name="dpi"
        ?disabled="${this.disabled}">
    </print-preview-settings-select>
  </div>
</print-preview-settings-section>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewDpiSettingsElementBase = SettingsMixin(CrLitElement);
class PrintPreviewDpiSettingsElement extends PrintPreviewDpiSettingsElementBase {
    static get is() {
        return 'print-preview-dpi-settings';
    }
    static get styles() {
        return getCss$j();
    }
    render() {
        return getHtml$k.bind(this)();
    }
    static get properties() {
        return {
            capability: { type: Object },
            disabled: { type: Boolean },
            capabilityWithLabels_: { type: Object },
        };
    }
    #capability_accessor_storage = null;
    get capability() { return this.#capability_accessor_storage; }
    set capability(value) { this.#capability_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #capabilityWithLabels__accessor_storage = null;
    get capabilityWithLabels_() { return this.#capabilityWithLabels__accessor_storage; }
    set capabilityWithLabels_(value) { this.#capabilityWithLabels__accessor_storage = value; }
    lastSelectedValue_ = null;
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('dpi.*', () => this.onDpiSettingChange_());
        this.onDpiSettingChange_();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('capability')) {
            this.capabilityWithLabels_ = this.computeCapabilityWithLabels_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('capabilityWithLabels_')) {
            this.onDpiSettingChange_();
        }
    }
    /**
     * Adds default labels for each option.
     */
    computeCapabilityWithLabels_() {
        if (!this.capability) {
            return null;
        }
        const result = structuredClone(this.capability);
        for (const dpiOption of result.option) {
            const hDpi = dpiOption.horizontal_dpi || 0;
            const vDpi = dpiOption.vertical_dpi || 0;
            if (hDpi > 0 && vDpi > 0 && hDpi !== vDpi) {
                dpiOption.name = loadTimeData.getStringF('nonIsotropicDpiItemLabel', hDpi.toLocaleString(), vDpi.toLocaleString());
            }
            else {
                dpiOption.name = loadTimeData.getStringF('dpiItemLabel', (hDpi || vDpi).toLocaleString());
            }
        }
        return result;
    }
    onDpiSettingChange_() {
        if (this.capabilityWithLabels_ === null) {
            return;
        }
        const dpiValue = this.getSettingValue('dpi');
        for (const option of this.capabilityWithLabels_.option) {
            const dpiOption = option;
            if (dpiValue.horizontal_dpi === dpiOption.horizontal_dpi &&
                dpiValue.vertical_dpi === dpiOption.vertical_dpi &&
                dpiValue.vendor_id === dpiOption.vendor_id) {
                this.shadowRoot.querySelector('print-preview-settings-select')
                    .selectValue(JSON.stringify(option));
                this.lastSelectedValue_ = dpiValue;
                return;
            }
        }
        // If the sticky settings are not compatible with the initially selected
        // printer, reset this setting to the printer default. Only do this when
        // the setting changes, as occurs for sticky settings, and not for a printer
        // change which can also trigger this observer. The model is responsible for
        // setting a compatible media size value after printer changes.
        if (dpiValue !== this.lastSelectedValue_) {
            const defaultOption = this.capabilityWithLabels_.option.find(o => !!o.is_default) ||
                this.capabilityWithLabels_.option[0];
            this.setSetting('dpi', defaultOption, /*noSticky=*/ true);
        }
    }
}
customElements.define(PrintPreviewDpiSettingsElement.is, PrintPreviewDpiSettingsElement);

let instance$i = null;
function getCss$i() {
    return instance$i || (instance$i = [...[], css `:host{display:none}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$j() {
    return html `
<svg id="baseSvg" xmlns="http://www.w3.org/2000/svg"
     viewBox="0 0 ${this.size} ${this.size}"
     preserveAspectRatio="xMidYMid meet" focusable="false"
     style="pointer-events: none; display: block; width: 100%; height: 100%;">
 </svg>
<slot></slot>
`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const APPLIED_ICON_CLASS = 'cr-iconset-svg-icon_';
class CrIconsetElement extends CrLitElement {
    static get is() {
        return 'cr-iconset';
    }
    static get styles() {
        return getCss$i();
    }
    render() {
        return getHtml$j.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The name of the iconset.
             */
            name: { type: String },
            /**
             * The size of an individual icon. Note that icons must be square.
             */
            size: { type: Number },
        };
    }
    #name_accessor_storage = '';
    get name() { return this.#name_accessor_storage; }
    set name(value) { this.#name_accessor_storage = value; }
    #size_accessor_storage = 24;
    get size() { return this.#size_accessor_storage; }
    set size(value) { this.#size_accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('name')) {
            assert(changedProperties.get('name') === undefined);
            IconsetMap.getInstance().set(this.name, this);
        }
    }
    /**
     * Applies an icon to the given element.
     *
     * An svg icon is prepended to the element's shadowRoot, which should always
     * exist.
     * @param element Element to which the icon is applied.
     * @param iconName Name of the icon to apply.
     * @return The svg element which renders the icon.
     */
    applyIcon(element, iconName) {
        // Remove old svg element
        this.removeIcon(element);
        // install new svg element
        const svg = this.cloneIcon_(iconName);
        if (svg) {
            // Add special class so we can identify it in remove.
            svg.classList.add(APPLIED_ICON_CLASS);
            // insert svg element into shadow root
            element.shadowRoot.insertBefore(svg, element.shadowRoot.childNodes[0]);
            return svg;
        }
        return null;
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or null if there is no matching element.
     * @param iconName Name of the icon to apply.
     */
    createIcon(iconName) {
        return this.cloneIcon_(iconName);
    }
    /**
     * Remove an icon from the given element by undoing the changes effected
     * by `applyIcon`.
     */
    removeIcon(element) {
        // Remove old svg element
        const oldSvg = element.shadowRoot.querySelector(`.${APPLIED_ICON_CLASS}`);
        if (oldSvg) {
            oldSvg.remove();
        }
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or `undefined` if there is no matching element.
     *
     * Returns an installable clone of the SVG element matching `id` or null if
     * no such element exists.
     */
    cloneIcon_(id) {
        const sourceSvg = this.querySelector(`g[id="${id}"]`);
        if (!sourceSvg) {
            return null;
        }
        const svgClone = this.$.baseSvg.cloneNode(true);
        const content = sourceSvg.cloneNode(true);
        content.removeAttribute('id');
        const contentViewBox = content.getAttribute('viewBox');
        if (contentViewBox) {
            svgClone.setAttribute('viewBox', contentViewBox);
        }
        svgClone.appendChild(content);
        return svgClone;
    }
}
customElements.define(CrIconsetElement.is, CrIconsetElement);

const div$1 = document.createElement('div');
div$1.innerHTML = getTrustedHTML `<cr-iconset name="print-preview" size="24">
  <svg>
    <defs>
      <!-- Custom SVGs (namratakannan) -->
      <g id="short-edge" viewBox="0 0 26 20" fill-rule="evenodd">
        <path d="M2 14v2h2v-2H2zm12 4v2h2v-2h-2zm4 0v2h2v-2h-2zm3.556-18H4.444C3.1 0 2 1.35 2 3v6h2V2h18v7h2V3c0-1.65-1.1-3-2.444-3zM24 18h-2v2c1.1 0 2-.9 2-2zM0 10v2h26v-2H0zm6 8v2h2v-2H6zm16-4v2h2v-2h-2zm-12 4v2h2v-2h-2zm-8 0c0 1.1.9 2 2 2v-2H2z">
        <path d="M29-6v32H-3V-6z">
      </g>
      <g id="long-edge" viewBox="0 0 23 22" fill-rule="evenodd">
        <path d="M17 20h2v-2h-2v2zm4-12h2V6h-2v2zM0 4v14c0 1.1 1.35 2 3 2h6v-2H2V4h7V2H3c-1.65 0-3 .9-3 2zm21-2v2h2c0-1.1-.9-2-2-2zM10 22h2V0h-2v22zm11-6h2v-2h-2v2zM17 4h2V2h-2v2zm-4 16h2v-2h-2v2zm0-16h2V2h-2v2zm8 8h2v-2h-2v2zm0 8c1.1 0 2-.9 2-2h-2v2z">
        <path d="M-5-5h32v32H-5z">
      </g>

      <!--
      These icons are copied from Polymer's iron-icons and kept in sorted order.
      See http://goo.gl/Y1OdAq for instructions on adding additional icons.
      -->
      <g id="business"><path d="M12 7V3H2v18h20V7H12zM6 19H4v-2h2v2zm0-4H4v-2h2v2zm0-4H4V9h2v2zm0-4H4V5h2v2zm4 12H8v-2h2v2zm0-4H8v-2h2v2zm0-4H8V9h2v2zm0-4H8V5h2v2zm10 12h-8v-2h2v-2h-2v-2h2v-2h-2V9h8v10zm-2-8h-2v2h2v-2zm0 4h-2v2h2v-2z"></path></g>
      <g id="print"><path d="M19 8H5c-1.66 0-3 1.34-3 3v6h4v4h12v-4h4v-6c0-1.66-1.34-3-3-3zm-3 11H8v-5h8v5zm3-7c-.55 0-1-.45-1-1s.45-1 1-1 1 .45 1 1-.45 1-1 1zm-1-9H6v4h12V3z"></path></g>
    </defs>
  </svg>
</cr-iconset>
`;
const iconsets$1 = div$1.querySelectorAll('cr-iconset');
for (const iconset of iconsets$1) {
    document.head.appendChild(iconset);
}

let instance$h = null;
function getCss$h() {
    return instance$h || (instance$h = [...[getCss$x(), getCss$K(), getCss$z()], css `:host{--duplex-icon-side-padding:8px;--duplex-icon-size:16px}.md-select{background-position:var(--duplex-icon-side-padding) center,calc(100% - var(--md-select-side-padding)) center;background-size:var(--duplex-icon-size),var(--md-arrow-width);padding-inline-start:32px}:host-context([dir=rtl]) .md-select{background-position-x:calc(100% - var(--duplex-icon-side-padding)),var(--md-select-side-padding)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$i() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <div slot="title">
    <label id="label">$i18n{optionTwoSided}</label>
  </div>
  <div slot="controls" class="checkbox">
    <cr-checkbox id="duplex" aria-labelledby="label"
        ?disabled="${this.getDisabled_(this.duplexManaged_)}"
        @change="${this.onCheckboxChange_}">
      $i18n{printOnBothSidesLabel}
    </cr-checkbox>
  </div>
</print-preview-settings-section>
<cr-collapse ?opened="${this.collapseOpened_}">
  <print-preview-settings-section>
    <div slot="title"></div>
    <div slot="controls">
      <select class="md-select" aria-labelledby="duplex"
          .style="background-image: ${this.backgroundImages_};"
          ?disabled="${this.getDisabled_(this.duplexShortEdgeManaged_)}"
          .value="${this.selectedValue}" @change="${this.onSelectChange}">
        <option value="${DuplexMode.LONG_EDGE}">
          $i18n{optionLongEdge}
        </option>
        <option value="${DuplexMode.SHORT_EDGE}">
          $i18n{optionShortEdge}
        </option>
      </select>
    </div>
  </print-preview-settings-section>
</cr-collapse><!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewDuplexSettingsElementBase = SettingsMixin(SelectMixin(CrLitElement));
class PrintPreviewDuplexSettingsElement extends PrintPreviewDuplexSettingsElementBase {
    static get is() {
        return 'print-preview-duplex-settings';
    }
    static get styles() {
        return getCss$h();
    }
    render() {
        return getHtml$i.bind(this)();
    }
    static get properties() {
        return {
            dark: { type: Boolean },
            disabled: { type: Boolean },
            duplexManaged_: { type: Boolean },
            duplexShortEdgeManaged_: { type: Boolean },
            collapseOpened_: { type: Boolean },
            backgroundImages_: { type: String },
        };
    }
    #dark_accessor_storage = false;
    get dark() { return this.#dark_accessor_storage; }
    set dark(value) { this.#dark_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #duplexManaged__accessor_storage = false;
    get duplexManaged_() { return this.#duplexManaged__accessor_storage; }
    set duplexManaged_(value) { this.#duplexManaged__accessor_storage = value; }
    #duplexShortEdgeManaged__accessor_storage = false;
    get duplexShortEdgeManaged_() { return this.#duplexShortEdgeManaged__accessor_storage; }
    set duplexShortEdgeManaged_(value) { this.#duplexShortEdgeManaged__accessor_storage = value; }
    #collapseOpened__accessor_storage = false;
    get collapseOpened_() { return this.#collapseOpened__accessor_storage; }
    set collapseOpened_(value) { this.#collapseOpened__accessor_storage = value; }
    #backgroundImages__accessor_storage = '';
    // An inline svg corresponding to |icon| and the image for the dropdown arrow.
    get backgroundImages_() { return this.#backgroundImages__accessor_storage; }
    set backgroundImages_(value) { this.#backgroundImages__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('duplex.*', this.onDuplexSettingChange_.bind(this));
        this.onDuplexSettingChange_();
        this.addSettingObserver('duplexShortEdge.*', this.onDuplexTypeChange_.bind(this));
        this.onDuplexTypeChange_();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('dark')) {
            this.updateBackgroundImages_();
        }
    }
    onDuplexSettingChange_() {
        this.$.duplex.checked = this.getSettingValue('duplex');
        this.duplexManaged_ = this.getSetting('duplex').setByGlobalPolicy;
        this.updateCollapseOpened_();
    }
    onDuplexTypeChange_() {
        this.selectedValue = this.getSettingValue('duplexShortEdge') ?
            DuplexMode.SHORT_EDGE.toString() :
            DuplexMode.LONG_EDGE.toString();
        this.duplexShortEdgeManaged_ =
            this.getSetting('duplexShortEdge').setByGlobalPolicy;
        this.updateCollapseOpened_();
        this.updateBackgroundImages_();
    }
    onCheckboxChange_() {
        this.setSetting('duplex', this.$.duplex.checked);
    }
    onProcessSelectChange(value) {
        this.setSetting('duplexShortEdge', value === DuplexMode.SHORT_EDGE.toString());
    }
    updateCollapseOpened_() {
        this.collapseOpened_ = this.getSetting('duplexShortEdge').available &&
            this.getSettingValue('duplex');
    }
    /**
     * @param managed Whether the setting is managed by policy.
     * @return Whether the controls should be disabled.
     */
    getDisabled_(managed) {
        return managed || this.disabled;
    }
    updateBackgroundImages_() {
        const icon = this.getSettingValue('duplexShortEdge') ? 'short-edge' : 'long-edge';
        const iconset = IconsetMap.getInstance().get('print-preview');
        assert(iconset);
        this.backgroundImages_ = getSelectDropdownBackground(iconset, icon, this);
    }
}
customElements.define(PrintPreviewDuplexSettingsElement.is, PrintPreviewDuplexSettingsElement);

let instance$g = null;
function getCss$g() {
    return instance$g || (instance$g = [...[], css `:host{align-items:center;display:flex;justify-content:space-between;padding:var(--print-preview-sidebar-margin) var(--print-preview-sidebar-margin) 12px}@media (prefers-color-scheme:light){:host{background-color:white}}#headerContainer{align-items:center;display:flex}.summary,.title{color:var(--cr-primary-text-color);line-height:calc(20/13 * 1em);margin:0}.title{font-size:calc(16/13 * 1em);font-weight:400}.summary{font-weight:500}cr-icon{align-self:center;fill:var(--cr-secondary-text-color);height:16px;margin-inline-start:8px;width:16px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$h() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="headerContainer">
  <h1 class="title">$i18n{title}</h1>
  <cr-icon ?hidden="${!this.managed}" icon="print-preview:business"
       alt="" title="$i18n{managedSettings}">
  </cr-icon>
</div>
<span class="summary">${this.summary_}</span>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewHeaderElementBase = SettingsMixin(CrLitElement);
class PrintPreviewHeaderElement extends PrintPreviewHeaderElementBase {
    static get is() {
        return 'print-preview-header';
    }
    static get styles() {
        return getCss$g();
    }
    render() {
        return getHtml$h.bind(this)();
    }
    static get properties() {
        return {
            destination: { type: Object },
            error: { type: Number },
            state: { type: Number },
            managed: { type: Boolean },
            sheetCount_: { type: Number },
            summary_: { type: String },
        };
    }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #error_accessor_storage = null;
    get error() { return this.#error_accessor_storage; }
    set error(value) { this.#error_accessor_storage = value; }
    #state_accessor_storage = State.NOT_READY;
    get state() { return this.#state_accessor_storage; }
    set state(value) { this.#state_accessor_storage = value; }
    #managed_accessor_storage = false;
    get managed() { return this.#managed_accessor_storage; }
    set managed(value) { this.#managed_accessor_storage = value; }
    #sheetCount__accessor_storage = 0;
    get sheetCount_() { return this.#sheetCount__accessor_storage; }
    set sheetCount_(value) { this.#sheetCount__accessor_storage = value; }
    #summary__accessor_storage = null;
    get summary_() { return this.#summary__accessor_storage; }
    set summary_(value) { this.#summary__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('pages.*', this.updateSheetCount_.bind(this));
        this.addSettingObserver('duplex.*', this.updateSheetCount_.bind(this));
        this.addSettingObserver('copies.*', this.updateSheetCount_.bind(this));
        this.updateSheetCount_();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('sheetCount_') ||
            changedProperties.has('state') ||
            changedProperties.has('destination')) {
            this.updateSummary_();
        }
    }
    updateSheetCount_() {
        let sheets = this.getSettingValue('pages').length;
        if (this.getSettingValue('duplex')) {
            sheets = Math.ceil(sheets / 2);
        }
        this.sheetCount_ = sheets * this.getSettingValue('copies');
    }
    isPdf_() {
        return !!this.destination &&
            this.destination.type === PrinterType.PDF_PRINTER;
    }
    updateSummary_() {
        switch (this.state) {
            case (State.PRINTING):
                this.summary_ =
                    loadTimeData.getString(this.isPdf_() ? 'saving' : 'printing');
                break;
            case (State.READY):
                this.updateSheetsSummary_();
                break;
            case (State.FATAL_ERROR):
                this.summary_ = this.getErrorMessage_();
                break;
            default:
                this.summary_ = null;
                break;
        }
    }
    /**
     * @return The error message to display.
     */
    getErrorMessage_() {
        switch (this.error) {
            case Error$1.PRINT_FAILED:
                return loadTimeData.getString('couldNotPrint');
            default:
                return '';
        }
    }
    updateSheetsSummary_() {
        if (this.sheetCount_ === 0) {
            this.summary_ = '';
            return;
        }
        const pageOrSheet = this.isPdf_() ? 'Page' : 'Sheet';
        PluralStringProxyImpl.getInstance()
            .getPluralString(`printPreview${pageOrSheet}SummaryLabel`, this.sheetCount_)
            .then(label => {
            this.summary_ = label;
        });
    }
}
customElements.define(PrintPreviewHeaderElement.is, PrintPreviewHeaderElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$g() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span id="layout-label" slot="title">$i18n{layoutLabel}</span>
  <div slot="controls">
    <select class="md-select" aria-labelledby="layout-label"
        ?disabled="${this.disabled}" .value="${this.selectedValue}"
        @change="${this.onSelectChange}">
      <option value="portrait" selected>$i18n{optionPortrait}</option>
      <option value="landscape">$i18n{optionLandscape}</option>
    </select>
  </div>
</print-preview-settings-section>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewLayoutSettingsElementBase = SettingsMixin(SelectMixin(CrLitElement));
class PrintPreviewLayoutSettingsElement extends PrintPreviewLayoutSettingsElementBase {
    static get is() {
        return 'print-preview-layout-settings';
    }
    static get styles() {
        return [
            getCss$x(),
            getCss$z(),
        ];
    }
    render() {
        return getHtml$g.bind(this)();
    }
    static get properties() {
        return {
            disabled: { type: Boolean },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.onLayoutSettingChange_(this.getSettingValue('layout'));
        this.addSettingObserver('layout.value', (newValue) => {
            this.onLayoutSettingChange_(newValue);
        });
    }
    onLayoutSettingChange_(newValue) {
        this.selectedValue = newValue ? 'landscape' : 'portrait';
    }
    onProcessSelectChange(value) {
        this.setSetting('layout', value === 'landscape');
    }
}
customElements.define(PrintPreviewLayoutSettingsElement.is, PrintPreviewLayoutSettingsElement);

let instance$f = null;
function getCss$f() {
    return instance$f || (instance$f = [...[getCss$x()], css `:host print-preview-settings-select{margin:0 calc(var(--print-preview-sidebar-margin) - 2px)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$f() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span id="media-size-label" slot="title">$i18n{mediaSizeLabel}</span>
  <div slot="controls">
    <print-preview-settings-select aria-label="$i18n{mediaSizeLabel}"
        .capability="${this.capability}" setting-name="mediaSize"
        ?disabled="${this.disabled}">
    </print-preview-settings-select>
  </div>
</print-preview-settings-section>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewMediaSizeSettingsElementBase = SettingsMixin(CrLitElement);
class PrintPreviewMediaSizeSettingsElement extends PrintPreviewMediaSizeSettingsElementBase {
    static get is() {
        return 'print-preview-media-size-settings';
    }
    static get styles() {
        return getCss$f();
    }
    render() {
        return getHtml$f.bind(this)();
    }
    static get properties() {
        return {
            capability: { type: Object },
            disabled: { type: Boolean },
        };
    }
    #capability_accessor_storage = null;
    get capability() { return this.#capability_accessor_storage; }
    set capability(value) { this.#capability_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    lastSelectedValue_ = '';
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('mediaSize.*', () => {
            this.onMediaSizeSettingChange_();
        });
        this.onMediaSizeSettingChange_();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('capability')) {
            this.onMediaSizeSettingChange_();
        }
    }
    onMediaSizeSettingChange_() {
        if (!this.capability) {
            return;
        }
        const valueToSet = JSON.stringify(this.getSettingValue('mediaSize'));
        for (const option of this.capability.option) {
            if (JSON.stringify(option) === valueToSet) {
                this.shadowRoot.querySelector('print-preview-settings-select')
                    .selectValue(valueToSet);
                this.lastSelectedValue_ = valueToSet;
                return;
            }
        }
        // If the sticky settings are not compatible with the initially selected
        // printer, reset this setting to the printer default. Only do this when
        // the setting changes, as occurs for sticky settings, and not for a printer
        // change which can also trigger this observer. The model is responsible for
        // setting a compatible media size value after printer changes.
        if (valueToSet !== this.lastSelectedValue_) {
            const defaultOption = this.capability.option.find(o => !!o.is_default) ||
                this.capability.option[0];
            this.setSetting('mediaSize', defaultOption, /*noSticky=*/ true);
        }
    }
}
customElements.define(PrintPreviewMediaSizeSettingsElement.is, PrintPreviewMediaSizeSettingsElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$e() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span id="margins-label" slot="title">$i18n{marginsLabel}</span>
  <div slot="controls">
    <select class="md-select" aria-labelledby="margins-label"
        ?disabled="${this.marginsDisabled_}"
        .value="${this.selectedValue}" @change="${this.onSelectChange}">
      <!-- The order of these options must match the natural order of their
      values, which come from MarginsType. -->
      <option value="${MarginsType.DEFAULT}" selected>
        $i18n{defaultMargins}
      </option>
      <option value="${MarginsType.NO_MARGINS}">
        $i18n{noMargins}
      </option>
      <option value="${MarginsType.MINIMUM}">
        $i18n{minimumMargins}
      </option>
      <option value="${MarginsType.CUSTOM}">
        $i18n{customMargins}
      </option>
    </select>
  </div>
</print-preview-settings-section><!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewMarginsSettingsElementBase = SettingsMixin(SelectMixin(CrLitElement));
class PrintPreviewMarginsSettingsElement extends PrintPreviewMarginsSettingsElementBase {
    static get is() {
        return 'print-preview-margins-settings';
    }
    static get styles() {
        return [
            getCss$x(),
            getCss$z(),
        ];
    }
    render() {
        return getHtml$e.bind(this)();
    }
    static get properties() {
        return {
            disabled: { type: Boolean },
            state: { type: Number },
            marginsDisabled_: { type: Boolean },
            pagesPerSheet_: { type: Number },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #state_accessor_storage = State.NOT_READY;
    get state() { return this.#state_accessor_storage; }
    set state(value) { this.#state_accessor_storage = value; }
    #marginsDisabled__accessor_storage = false;
    get marginsDisabled_() { return this.#marginsDisabled__accessor_storage; }
    set marginsDisabled_(value) { this.#marginsDisabled__accessor_storage = value; }
    #pagesPerSheet__accessor_storage = 1;
    get pagesPerSheet_() { return this.#pagesPerSheet__accessor_storage; }
    set pagesPerSheet_(value) { this.#pagesPerSheet__accessor_storage = value; }
    loaded_ = false;
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('pagesPerSheet.value', (newValue) => {
            this.pagesPerSheet_ = newValue;
        });
        this.pagesPerSheet_ = this.getSetting('pagesPerSheet').value;
        this.addSettingObserver('margins.value', (newValue) => {
            this.selectedValue = newValue.toString();
        });
        this.selectedValue = this.getSetting('margins').value.toString();
        this.addSettingObserver('mediaSize.value', this.onMediaSizeOrLayoutChange_.bind(this));
        this.addSettingObserver('layout.value', this.onMediaSizeOrLayoutChange_.bind(this));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('state')) {
            if (this.state === State.READY) {
                this.loaded_ = true;
            }
        }
        if (changedProperties.has('disabled') ||
            changedPrivateProperties.has('pagesPerSheet_')) {
            this.marginsDisabled_ = this.pagesPerSheet_ > 1 || this.disabled;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('pagesPerSheet_')) {
            if (this.pagesPerSheet_ > 1) {
                this.setSetting('margins', MarginsType.DEFAULT);
            }
        }
    }
    onMediaSizeOrLayoutChange_() {
        if (this.loaded_ &&
            this.getSetting('margins').value === MarginsType.CUSTOM) {
            this.setSetting('margins', MarginsType.DEFAULT);
        }
    }
    onProcessSelectChange(value) {
        this.setSetting('margins', parseInt(value, 10));
    }
}
customElements.define(PrintPreviewMarginsSettingsElement.is, PrintPreviewMarginsSettingsElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* List commonly used icons here to prevent duplication.
 * Do not add rarely used icons here; place those in your application.
 * Note that 20px and 24px icons are specified separately (size="", below).
 *
 * Icons are rendered at 20x20 px, but we don't have 20 px SVGs for everything.
 * The 24 px icons are used where 20 px icons are unavailable (which may appear
 * blurry at 20 px). Please use 20 px icons when available.
 */
const div = document.createElement('div');
div.innerHTML = getTrustedHTML `
<cr-iconset name="cr20" size="20">
  <svg>
    <defs>
      <!--
      Keep these in sorted order by id="".
      -->
      <g id="block">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M10 0C4.48 0 0 4.48 0 10C0 15.52 4.48 20 10 20C15.52 20 20 15.52 20 10C20 4.48 15.52 0 10 0ZM2 10C2 5.58 5.58 2 10 2C11.85 2 13.55 2.63 14.9 3.69L3.69 14.9C2.63 13.55 2 11.85 2 10ZM5.1 16.31C6.45 17.37 8.15 18 10 18C14.42 18 18 14.42 18 10C18 8.15 17.37 6.45 16.31 5.1L5.1 16.31Z">
        </path>
      </g>
      <g id="cloud-off">
        <path
          d="M16 18.125L13.875 16H5C3.88889 16 2.94444 15.6111 2.16667 14.8333C1.38889 14.0556 1 13.1111 1 12C1 10.9444 1.36111 10.0347 2.08333 9.27083C2.80556 8.50694 3.6875 8.09028 4.72917 8.02083C4.77083 7.86805 4.8125 7.72222 4.85417 7.58333C4.90972 7.44444 4.97222 7.30555 5.04167 7.16667L1.875 4L2.9375 2.9375L17.0625 17.0625L16 18.125ZM5 14.5H12.375L6.20833 8.33333C6.15278 8.51389 6.09722 8.70139 6.04167 8.89583C6 9.07639 5.95139 9.25694 5.89583 9.4375L4.83333 9.52083C4.16667 9.57639 3.61111 9.84028 3.16667 10.3125C2.72222 10.7708 2.5 11.3333 2.5 12C2.5 12.6944 2.74306 13.2847 3.22917 13.7708C3.71528 14.2569 4.30556 14.5 5 14.5ZM17.5 15.375L16.3958 14.2917C16.7153 14.125 16.9792 13.8819 17.1875 13.5625C17.3958 13.2431 17.5 12.8889 17.5 12.5C17.5 11.9444 17.3056 11.4722 16.9167 11.0833C16.5278 10.6944 16.0556 10.5 15.5 10.5H14.125L14 9.14583C13.9028 8.11806 13.4722 7.25694 12.7083 6.5625C11.9444 5.85417 11.0417 5.5 10 5.5C9.65278 5.5 9.31944 5.54167 9 5.625C8.69444 5.70833 8.39583 5.82639 8.10417 5.97917L7.02083 4.89583C7.46528 4.61806 7.93056 4.40278 8.41667 4.25C8.91667 4.08333 9.44444 4 10 4C11.4306 4 12.6736 4.48611 13.7292 5.45833C14.7847 6.41667 15.375 7.59722 15.5 9C16.4722 9 17.2986 9.34028 17.9792 10.0208C18.6597 10.7014 19 11.5278 19 12.5C19 13.0972 18.8611 13.6458 18.5833 14.1458C18.3194 14.6458 17.9583 15.0556 17.5 15.375Z">
        </path>
      </g>
      <g id="delete">
        <path
          d="M 5.832031 17.5 C 5.375 17.5 4.984375 17.335938 4.65625 17.011719 C 4.328125 16.683594 4.167969 16.292969 4.167969 15.832031 L 4.167969 5 L 3.332031 5 L 3.332031 3.332031 L 7.5 3.332031 L 7.5 2.5 L 12.5 2.5 L 12.5 3.332031 L 16.667969 3.332031 L 16.667969 5 L 15.832031 5 L 15.832031 15.832031 C 15.832031 16.292969 15.671875 16.683594 15.34375 17.011719 C 15.015625 17.335938 14.625 17.5 14.167969 17.5 Z M 14.167969 5 L 5.832031 5 L 5.832031 15.832031 L 14.167969 15.832031 Z M 7.5 14.167969 L 9.167969 14.167969 L 9.167969 6.667969 L 7.5 6.667969 Z M 10.832031 14.167969 L 12.5 14.167969 L 12.5 6.667969 L 10.832031 6.667969 Z M 5.832031 5 L 5.832031 15.832031 Z M 5.832031 5 ">
        </path>
      </g>
      <g id="domain" viewBox="0 -960 960 960">
        <path d="M96-144v-672h384v144h384v528H96Zm72-72h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm168 456h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm144 456h312v-384H480v80h72v72h-72v80h72v72h-72v80Zm168-232v-72h72v72h-72Zm0 152v-72h72v72h-72Z"></path>
      </g>
      <g id="kite">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M4.6327 8.00094L10.3199 2L16 8.00094L10.1848 16.8673C10.0995 16.9873 10.0071 17.1074 9.90047 17.2199C9.42417 17.7225 8.79147 18 8.11611 18C7.44076 18 6.80806 17.7225 6.33175 17.2199C5.85545 16.7173 5.59242 16.0497 5.59242 15.3371C5.59242 14.977 5.46445 14.647 5.22275 14.3919C4.98104 14.1369 4.66825 14.0019 4.32701 14.0019H4V12.6667H4.32701C5.00237 12.6667 5.63507 12.9442 6.11137 13.4468C6.58768 13.9494 6.85071 14.617 6.85071 15.3296C6.85071 15.6896 6.97867 16.0197 7.22038 16.2747C7.46209 16.5298 7.77488 16.6648 8.11611 16.6648C8.45735 16.6648 8.77014 16.5223 9.01185 16.2747C9.02396 16.2601 9.03607 16.246 9.04808 16.2319C9.08541 16.1883 9.12176 16.1458 9.15403 16.0947L9.55213 15.4946L4.6327 8.00094ZM10.3199 13.9371L6.53802 8.17116L10.3199 4.1814L14.0963 8.17103L10.3199 13.9371Z">
        </path>
      </g>
      <g id="menu">
        <path d="M2 4h16v2H2zM2 9h16v2H2zM2 14h16v2H2z"></path>
      </g>
      <g id="password">
        <path d="M5.833 11.667c.458 0 .847-.16 1.167-.479.333-.333.5-.729.5-1.188s-.167-.847-.5-1.167a1.555 1.555 0 0 0-1.167-.5c-.458 0-.854.167-1.188.5A1.588 1.588 0 0 0 4.166 10c0 .458.16.854.479 1.188.333.319.729.479 1.188.479Zm0 3.333c-1.389 0-2.569-.486-3.542-1.458C1.319 12.569.833 11.389.833 10c0-1.389.486-2.569 1.458-3.542C3.264 5.486 4.444 5 5.833 5c.944 0 1.813.243 2.604.729a4.752 4.752 0 0 1 1.833 1.979h7.23c.458 0 .847.167 1.167.5.333.319.5.708.5 1.167v3.958c0 .458-.167.854-.5 1.188A1.588 1.588 0 0 1 17.5 15h-3.75a1.658 1.658 0 0 1-1.188-.479 1.658 1.658 0 0 1-.479-1.188v-1.042H10.27a4.59 4.59 0 0 1-1.813 2A5.1 5.1 0 0 1 5.833 15Zm3.292-4.375h4.625v2.708H15v-1.042a.592.592 0 0 1 .167-.438.623.623 0 0 1 .458-.188c.181 0 .327.063.438.188a.558.558 0 0 1 .188.438v1.042H17.5V9.375H9.125a3.312 3.312 0 0 0-1.167-1.938 3.203 3.203 0 0 0-2.125-.77 3.21 3.21 0 0 0-2.354.979C2.827 8.298 2.5 9.083 2.5 10s.327 1.702.979 2.354a3.21 3.21 0 0 0 2.354.979c.806 0 1.514-.25 2.125-.75.611-.514 1-1.167 1.167-1.958Z"></path>
      </g>
      
  </svg>
</cr-iconset>

<!-- NOTE: In the common case that the final icon will be 20x20, export the SVG
     at 20px and place it in the section above. -->
<cr-iconset name="cr" size="24">
  <svg>
    <defs>
      <!--
      These icons are copied from Polymer's iron-icons and kept in sorted order.
      -->
      <g id="add">
        <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z" />
      </g>
      <g id="arrow-back">
        <path
          d="m7.824 13 5.602 5.602L12 20l-8-8 8-8 1.426 1.398L7.824 11H20v2Zm0 0">
        </path>
      </g>
      <g id="arrow-drop-up">
        <path d="M7 14l5-5 5 5z"></path>
      </g>
      <g id="arrow-drop-down">
        <path d="M7 10l5 5 5-5z"></path>
      </g>
      <g id="arrow-forward">
        <path
          d="M12 4l-1.41 1.41L16.17 11H4v2h12.17l-5.58 5.59L12 20l8-8z">
        </path>
      </g>
      <g id="arrow-right">
        <path d="M10 7l5 5-5 5z"></path>
      </g>
      <g id="cancel">
        <path
          d="M12 2C6.47 2 2 6.47 2 12s4.47 10 10 10 10-4.47 10-10S17.53 2 12 2zm5 13.59L15.59 17 12 13.41 8.41 17 7 15.59 10.59 12 7 8.41 8.41 7 12 10.59 15.59 7 17 8.41 13.41 12 17 15.59z">
        </path>
      </g>
      <g id="check">
        <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"></path>
      </g>
      <g id="check-circle" viewBox="0 -960 960 960">
        <path d="m424-296 282-282-56-56-226 226-114-114-56 56 170 170Zm56 216q-83 0-156-31.5T197-197q-54-54-85.5-127T80-480q0-83 31.5-156T197-763q54-54 127-85.5T480-880q83 0 156 31.5T763-763q54 54 85.5 127T880-480q0 83-31.5 156T763-197q-54 54-127 85.5T480-80Zm0-80q134 0 227-93t93-227q0-134-93-227t-227-93q-134 0-227 93t-93 227q0 134 93 227t227 93Zm0-320Z"></path>
      </g>
      <g id="chevron-left">
        <path d="M15.41 7.41L14 6l-6 6 6 6 1.41-1.41L10.83 12z"></path>
      </g>
      <g id="chevron-right">
        <path d="M10 6L8.59 7.41 13.17 12l-4.58 4.59L10 18l6-6z"></path>
      </g>
      <g id="clear">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="chrome-product" viewBox="0 -960 960 960">
        <path d="M336-479q0 60 42 102t102 42q60 0 102-42t42-102q0-60-42-102t-102-42q-60 0-102 42t-42 102Zm144 216q11 0 22.5-.5T525-267L427-99q-144-16-237.5-125T96-479q0-43 9.5-84.5T134-645l160 274q28 51 78 79.5T480-263Zm0-432q-71 0-126.5 42T276-545l-98-170q53-71 132.5-109.5T480-863q95 0 179 45t138 123H480Zm356 72q15 35 21.5 71t6.5 73q0 155-100 260.5T509-96l157-275q14-25 22-52t8-56q0-40-15-77t-41-67h196Z">
        </path>
      </g>
      <g id="close">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="computer">
        <path
          d="M20 18c1.1 0 1.99-.9 1.99-2L22 6c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2H0v2h24v-2h-4zM4 6h16v10H4V6z">
        </path>
      </g>
      <g id="create">
        <path
          d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z">
        </path>
      </g>
      <g id="delete" viewBox="0 -960 960 960">
        <path
          d="M309.37-135.87q-34.48 0-58.74-24.26-24.26-24.26-24.26-58.74v-474.5h-53.5v-83H378.5v-53.5h202.52v53.5h206.11v83h-53.5v474.07q0 35.21-24.26 59.32t-58.74 24.11H309.37Zm341.26-557.5H309.37v474.5h341.26v-474.5ZM379.7-288.24h77.5v-336h-77.5v336Zm123.1 0h77.5v-336h-77.5v336ZM309.37-693.37v474.5-474.5Z">
        </path>
      </g>
      <g id="domain">
        <path
          d="M12 7V3H2v18h20V7H12zM6 19H4v-2h2v2zm0-4H4v-2h2v2zm0-4H4V9h2v2zm0-4H4V5h2v2zm4 12H8v-2h2v2zm0-4H8v-2h2v2zm0-4H8V9h2v2zm0-4H8V5h2v2zm10 12h-8v-2h2v-2h-2v-2h2v-2h-2V9h8v10zm-2-8h-2v2h2v-2zm0 4h-2v2h2v-2z">
        </path>
      </g>
      <!-- source: https://fonts.google.com/icons?selected=Material+Symbols+Outlined:family_link:FILL@0;wght@0;GRAD@0;opsz@24&icon.size=24&icon.color=%23e8eaed -->
      <g id="kite" viewBox="0 -960 960 960">
        <path
          d="M390-40q-51 0-90.5-30.5T246-149q-6-23-25-37t-43-14q-16 0-30 6.5T124-175l-61-51q21-26 51.5-40t63.5-14q51 0 91 30t54 79q6 23 25 37t42 14q19 0 34-10t26-25l1-2-276-381q-8-11-11.5-23t-3.5-24q0-16 6-30.5t18-26.5l260-255q11-11 26-17t30-6q15 0 30 6t26 17l260 255q12 12 18 26.5t6 30.5q0 12-3.5 24T825-538L500-88q-18 25-48 36.5T390-40Zm110-185 260-360-260-255-259 256 259 359Zm1-308Z"/>
        </path>
      </g>
      <g id="error">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z">
        </path>
      </g>
      <g id="error-outline">
        <path
          d="M11 15h2v2h-2zm0-8h2v6h-2zm.99-5C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8z">
        </path>
      </g>
      <g id="expand-less">
        <path d="M12 8l-6 6 1.41 1.41L12 10.83l4.59 4.58L18 14z"></path>
      </g>
      <g id="expand-more">
        <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"></path>
      </g>
      <g id="extension">
        <path
          d="M20.5 11H19V7c0-1.1-.9-2-2-2h-4V3.5C13 2.12 11.88 1 10.5 1S8 2.12 8 3.5V5H4c-1.1 0-1.99.9-1.99 2v3.8H3.5c1.49 0 2.7 1.21 2.7 2.7s-1.21 2.7-2.7 2.7H2V20c0 1.1.9 2 2 2h3.8v-1.5c0-1.49 1.21-2.7 2.7-2.7 1.49 0 2.7 1.21 2.7 2.7V22H17c1.1 0 2-.9 2-2v-4h1.5c1.38 0 2.5-1.12 2.5-2.5S21.88 11 20.5 11z">
        </path>
      </g>
      <g id="file-download" viewBox="0 -960 960 960">
        <path d="M480-336 288-528l51-51 105 105v-342h72v342l105-105 51 51-192 192ZM263.72-192Q234-192 213-213.15T192-264v-72h72v72h432v-72h72v72q0 29.7-21.16 50.85Q725.68-192 695.96-192H263.72Z"></path>
      </g>
      <g id="fullscreen">
        <path
          d="M7 14H5v5h5v-2H7v-3zm-2-4h2V7h3V5H5v5zm12 7h-3v2h5v-5h-2v3zM14 5v2h3v3h2V5h-5z">
        </path>
      </g>
      <g id="group">
        <path
          d="M16 11c1.66 0 2.99-1.34 2.99-3S17.66 5 16 5c-1.66 0-3 1.34-3 3s1.34 3 3 3zm-8 0c1.66 0 2.99-1.34 2.99-3S9.66 5 8 5C6.34 5 5 6.34 5 8s1.34 3 3 3zm0 2c-2.33 0-7 1.17-7 3.5V19h14v-2.5c0-2.33-4.67-3.5-7-3.5zm8 0c-.29 0-.62.02-.97.05 1.16.84 1.97 1.97 1.97 3.45V19h6v-2.5c0-2.33-4.67-3.5-7-3.5z">
        </path>
      </g>
      <g id="help-outline">
        <path
          d="M11 18h2v-2h-2v2zm1-16C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm0-14c-2.21 0-4 1.79-4 4h2c0-1.1.9-2 2-2s2 .9 2 2c0 2-3 1.75-3 5h2c0-2.25 3-2.5 3-5 0-2.21-1.79-4-4-4z">
        </path>
      </g>
      <g id="history">
        <path
          d="M12.945312 22.75 C 10.320312 22.75 8.074219 21.839844 6.207031 20.019531 C 4.335938 18.199219 3.359375 15.972656 3.269531 13.34375 L 5.089844 13.34375 C 5.175781 15.472656 5.972656 17.273438 7.480469 18.742188 C 8.988281 20.210938 10.808594 20.945312 12.945312 20.945312 C 15.179688 20.945312 17.070312 20.164062 18.621094 18.601562 C 20.167969 17.039062 20.945312 15.144531 20.945312 12.910156 C 20.945312 10.714844 20.164062 8.855469 18.601562 7.335938 C 17.039062 5.816406 15.15625 5.054688 12.945312 5.054688 C 11.710938 5.054688 10.554688 5.339844 9.480469 5.902344 C 8.402344 6.46875 7.476562 7.226562 6.699219 8.179688 L 9.585938 8.179688 L 9.585938 9.984375 L 3.648438 9.984375 L 3.648438 4.0625 L 5.453125 4.0625 L 5.453125 6.824219 C 6.386719 5.707031 7.503906 4.828125 8.804688 4.199219 C 10.109375 3.566406 11.488281 3.25 12.945312 3.25 C 14.300781 3.25 15.570312 3.503906 16.761719 4.011719 C 17.949219 4.519531 18.988281 5.214844 19.875 6.089844 C 20.761719 6.964844 21.464844 7.992188 21.976562 9.167969 C 22.492188 10.34375 22.75 11.609375 22.75 12.964844 C 22.75 14.316406 22.492188 15.589844 21.976562 16.777344 C 21.464844 17.964844 20.761719 19.003906 19.875 19.882812 C 18.988281 20.765625 17.949219 21.464844 16.761719 21.976562 C 15.570312 22.492188 14.300781 22.75 12.945312 22.75 Z M 16.269531 17.460938 L 12.117188 13.34375 L 12.117188 7.527344 L 13.921875 7.527344 L 13.921875 12.601562 L 17.550781 16.179688 Z M 16.269531 17.460938">
        </path>
      </g>
      <g id="info">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z">
        </path>
      </g>
      <g id="info-outline">
        <path
          d="M11 17h2v-6h-2v6zm1-15C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zM11 9h2V7h-2v2z">
        </path>
      </g>
      <g id="insert-drive-file">
        <path
          d="M6 2c-1.1 0-1.99.9-1.99 2L4 20c0 1.1.89 2 1.99 2H18c1.1 0 2-.9 2-2V8l-6-6H6zm7 7V3.5L18.5 9H13z">
        </path>
      </g>
      <g id="location-on">
        <path
          d="M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7zm0 9.5c-1.38 0-2.5-1.12-2.5-2.5s1.12-2.5 2.5-2.5 2.5 1.12 2.5 2.5-1.12 2.5-2.5 2.5z">
        </path>
      </g>
      <g id="mic">
        <path
          d="M12 14c1.66 0 2.99-1.34 2.99-3L15 5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3zm5.3-3c0 3-2.54 5.1-5.3 5.1S6.7 14 6.7 11H5c0 3.41 2.72 6.23 6 6.72V21h2v-3.28c3.28-.48 6-3.3 6-6.72h-1.7z">
        </path>
      </g>
      <g id="more-vert">
        <path
          d="M12 8c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2zm0 2c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2zm0 6c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2z">
        </path>
      </g>
      <g id="open-in-new" viewBox="0 -960 960 960">
        <path
          d="M216-144q-29.7 0-50.85-21.15Q144-186.3 144-216v-528q0-29.7 21.15-50.85Q186.3-816 216-816h264v72H216v528h528v-264h72v264q0 29.7-21.15 50.85Q773.7-144 744-144H216Zm171-192-51-51 357-357H576v-72h240v240h-72v-117L387-336Z">
        </path>
      </g>
      <g id="person">
        <path
          d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z">
        </path>
      </g>
      <g id="phonelink">
        <path
          d="M4 6h18V4H4c-1.1 0-2 .9-2 2v11H0v3h14v-3H4V6zm19 2h-6c-.55 0-1 .45-1 1v10c0 .55.45 1 1 1h6c.55 0 1-.45 1-1V9c0-.55-.45-1-1-1zm-1 9h-4v-7h4v7z">
        </path>
      </g>
      <g id="print">
        <path
          d="M19 8H5c-1.66 0-3 1.34-3 3v6h4v4h12v-4h4v-6c0-1.66-1.34-3-3-3zm-3 11H8v-5h8v5zm3-7c-.55 0-1-.45-1-1s.45-1 1-1 1 .45 1 1-.45 1-1 1zm-1-9H6v4h12V3z">
        </path>
      </g>
      <g id="schedule">
        <path
          d="M11.99 2C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8zm.5-13H11v6l5.25 3.15.75-1.23-4.5-2.67z">
        </path>
      </g>
      <g id="search">
        <path
          d="M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z">
        </path>
      </g>
      <g id="security">
        <path
          d="M12 1L3 5v6c0 5.55 3.84 10.74 9 12 5.16-1.26 9-6.45 9-12V5l-9-4zm0 10.99h7c-.53 4.12-3.28 7.79-7 8.94V12H5V6.3l7-3.11v8.8z">
        </path>
      </g>
      <!-- The <g> IDs are exposed as global variables in Vulcanized mode, which
        conflicts with the "settings" namespace of MD Settings. Using an "_icon"
        suffix prevents the naming conflict. -->
      <g id="settings_icon">
        <path
          d="M19.43 12.98c.04-.32.07-.64.07-.98s-.03-.66-.07-.98l2.11-1.65c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.3-.61-.22l-2.49 1c-.52-.4-1.08-.73-1.69-.98l-.38-2.65C14.46 2.18 14.25 2 14 2h-4c-.25 0-.46.18-.49.42l-.38 2.65c-.61.25-1.17.59-1.69.98l-2.49-1c-.23-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64l2.11 1.65c-.04.32-.07.65-.07.98s.03.66.07.98l-2.11 1.65c-.19.15-.24.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1c.52.4 1.08.73 1.69.98l.38 2.65c.03.24.24.42.49.42h4c.25 0 .46-.18.49-.42l.38-2.65c.61-.25 1.17-.59 1.69-.98l2.49 1c.23.09.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.65zM12 15.5c-1.93 0-3.5-1.57-3.5-3.5s1.57-3.5 3.5-3.5 3.5 1.57 3.5 3.5-1.57 3.5-3.5 3.5z">
        </path>
      </g>
      <g id="star">
        <path
          d="M12 17.27L18.18 21l-1.64-7.03L22 9.24l-7.19-.61L12 2 9.19 8.63 2 9.24l5.46 4.73L5.82 21z">
        </path>
      </g>
      <g id="sync" viewBox="0 -960 960 960">
        <path
          d="M216-192v-72h74q-45-40-71.5-95.5T192-480q0-101 61-177.5T408-758v75q-63 23-103.5 77.5T264-480q0 48 19.5 89t52.5 70v-63h72v192H216Zm336-10v-75q63-23 103.5-77.5T696-480q0-48-19.5-89T624-639v63h-72v-192h192v72h-74q45 40 71.5 95.5T768-480q0 101-61 177.5T552-202Z">
        </path>
      </g>
      <g id="thumbs-down">
        <path
            d="M6 3h11v13l-7 7-1.25-1.25a1.454 1.454 0 0 1-.3-.475c-.067-.2-.1-.392-.1-.575v-.35L9.45 16H3c-.533 0-1-.2-1.4-.6-.4-.4-.6-.867-.6-1.4v-2c0-.117.017-.242.05-.375s.067-.258.1-.375l3-7.05c.15-.333.4-.617.75-.85C5.25 3.117 5.617 3 6 3Zm9 2H6l-3 7v2h9l-1.35 5.5L15 15.15V5Zm0 10.15V5v10.15Zm2 .85v-2h3V5h-3V3h5v13h-5Z">
        </path>
      </g>
      <g id="thumbs-down-filled">
        <path
            d="M6 3h10v13l-7 7-1.25-1.25a1.336 1.336 0 0 1-.29-.477 1.66 1.66 0 0 1-.108-.574v-.347L8.449 16H3c-.535 0-1-.2-1.398-.602C1.199 15 1 14.535 1 14v-2c0-.117.012-.242.04-.375.022-.133.062-.258.108-.375l3-7.05c.153-.333.403-.618.75-.848A1.957 1.957 0 0 1 6 3Zm12 13V3h4v13Zm0 0">
        </path>
      </g>
      <g id="thumbs-up">
        <path
            d="M18 21H7V8l7-7 1.25 1.25c.117.117.208.275.275.475.083.2.125.392.125.575v.35L14.55 8H21c.533 0 1 .2 1.4.6.4.4.6.867.6 1.4v2c0 .117-.017.242-.05.375s-.067.258-.1.375l-3 7.05c-.15.333-.4.617-.75.85-.35.233-.717.35-1.1.35Zm-9-2h9l3-7v-2h-9l1.35-5.5L9 8.85V19ZM9 8.85V19 8.85ZM7 8v2H4v9h3v2H2V8h5Z">
        </path>
      </g>
      <g id="thumbs-up-filled">
        <path
            d="M18 21H8V8l7-7 1.25 1.25c.117.117.21.273.29.477.073.199.108.39.108.574v.347L15.551 8H21c.535 0 1 .2 1.398.602C22.801 9 23 9.465 23 10v2c0 .117-.012.242-.04.375a1.897 1.897 0 0 1-.108.375l-3 7.05a2.037 2.037 0 0 1-.75.848A1.957 1.957 0 0 1 18 21ZM6 8v13H2V8Zm0 0">
      </g>
      <g id="videocam" viewBox="0 -960 960 960">
        <path
          d="M216-192q-29 0-50.5-21.5T144-264v-432q0-29.7 21.5-50.85Q187-768 216-768h432q29.7 0 50.85 21.15Q720-725.7 720-696v168l144-144v384L720-432v168q0 29-21.15 50.5T648-192H216Zm0-72h432v-432H216v432Zm0 0v-432 432Z">
        </path>
      </g>
      <g id="warning">
        <path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z"></path>
      </g>
    </defs>
  </svg>
</cr-iconset>`;
const iconsets = div.querySelectorAll('cr-iconset');
for (const iconset of iconsets) {
    document.head.appendChild(iconset);
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let hideInk = false;
document.addEventListener('pointerdown', function () {
    hideInk = true;
}, true);
document.addEventListener('keydown', function () {
    hideInk = false;
}, true);
/**
 * Attempts to track whether focus outlines should be shown, and if they
 * shouldn't, removes the "ink" (ripple) from a control while focusing it.
 * This is helpful when a user is clicking/touching, because it's not super
 * helpful to show focus ripples in that case. This is Polymer-specific.
 */
function focusWithoutInk(toFocus) {
    // |toFocus| does not have a 'noink' property, so it's unclear whether the
    // element has "ink" and/or whether it can be suppressed. Just focus().
    if (!('noink' in toFocus) || !hideInk) {
        toFocus.focus();
        return;
    }
    const toFocusWithNoInk = toFocus;
    // Make sure the element is in the document we're listening to events on.
    assert(document === toFocusWithNoInk.ownerDocument);
    const { noink } = toFocusWithNoInk;
    toFocusWithNoInk.noink = true;
    toFocusWithNoInk.focus();
    toFocusWithNoInk.noink = noink;
}

let instance$e = null;
function getCss$e() {
    return instance$e || (instance$e = [...[], css `:host{align-items:center;align-self:stretch;display:flex;margin:0;outline:none}:host(:not([effectively-disabled_])){cursor:pointer}:host(:not([no-hover],[effectively-disabled_]):hover){background-color:var(--cr-hover-background-color)}:host(:not([no-hover],[effectively-disabled_]):active){background-color:var(--cr-active-background-color)}:host(:not([no-hover],[effectively-disabled_])) cr-icon-button{--cr-icon-button-hover-background-color:transparent;--cr-icon-button-active-background-color:transparent}`]);
}

let instance$d = null;
function getCss$d() {
    return instance$d || (instance$d = [...[getCss$e()], css `:host([disabled]){opacity:0.65;pointer-events:none}:host([disabled]) cr-icon-button{display:var(--cr-expand-button-disabled-display,initial)}#label{flex:1;padding:var(--cr-section-vertical-padding) 0}cr-icon-button{--cr-icon-button-icon-size:var(--cr-expand-button-icon-size,20px);--cr-icon-button-size:var(--cr-expand-button-size,36px)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$d() {
    return html `
<div id="label" aria-hidden="true"><slot></slot></div>
<cr-icon-button id="icon" aria-labelledby="label" ?disabled="${this.disabled}"
    aria-expanded="${this.getAriaExpanded_()}"
    tabindex="${this.tabIndex}" part="icon" iron-icon="${this.getIcon_()}">
</cr-icon-button>`;
}

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'cr-expand-button' is a chrome-specific wrapper around a button that toggles
 * between an opened (expanded) and closed state.
 */
class CrExpandButtonElement extends CrLitElement {
    static get is() {
        return 'cr-expand-button';
    }
    static get styles() {
        return getCss$d();
    }
    render() {
        return getHtml$d.bind(this)();
    }
    static get properties() {
        return {
            /**
             * If true, the button is in the expanded state and will show the icon
             * specified in the `collapseIcon` property. If false, the button shows
             * the icon specified in the `expandIcon` property.
             */
            expanded: {
                type: Boolean,
                notify: true,
            },
            /**
             * If true, the button will be disabled and grayed out.
             */
            disabled: {
                type: Boolean,
                reflect: true,
            },
            /** A11y text descriptor for this control. */
            ariaLabel: { type: String },
            tabIndex: { type: Number },
            expandIcon: { type: String },
            collapseIcon: { type: String },
            expandTitle: { type: String },
            collapseTitle: { type: String },
        };
    }
    #expanded_accessor_storage = false;
    get expanded() { return this.#expanded_accessor_storage; }
    set expanded(value) { this.#expanded_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #expandIcon_accessor_storage = 'cr:expand-more';
    get expandIcon() { return this.#expandIcon_accessor_storage; }
    set expandIcon(value) { this.#expandIcon_accessor_storage = value; }
    #collapseIcon_accessor_storage = 'cr:expand-less';
    get collapseIcon() { return this.#collapseIcon_accessor_storage; }
    set collapseIcon(value) { this.#collapseIcon_accessor_storage = value; }
    #expandTitle_accessor_storage;
    get expandTitle() { return this.#expandTitle_accessor_storage; }
    set expandTitle(value) { this.#expandTitle_accessor_storage = value; }
    #collapseTitle_accessor_storage;
    get collapseTitle() { return this.#collapseTitle_accessor_storage; }
    set collapseTitle(value) { this.#collapseTitle_accessor_storage = value; }
    #tabIndex_accessor_storage = 0;
    get tabIndex() { return this.#tabIndex_accessor_storage; }
    set tabIndex(value) { this.#tabIndex_accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('click', this.toggleExpand_);
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('expanded') ||
            changedProperties.has('collapseTitle') ||
            changedProperties.has('expandTitle')) {
            this.title =
                (this.expanded ? this.collapseTitle : this.expandTitle) || '';
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('ariaLabel')) {
            this.onAriaLabelChange_();
        }
    }
    focus() {
        this.$.icon.focus();
    }
    getIcon_() {
        return this.expanded ? this.collapseIcon : this.expandIcon;
    }
    getAriaExpanded_() {
        return this.expanded ? 'true' : 'false';
    }
    onAriaLabelChange_() {
        if (this.ariaLabel) {
            this.$.icon.removeAttribute('aria-labelledby');
            this.$.icon.setAttribute('aria-label', this.ariaLabel);
        }
        else {
            this.$.icon.removeAttribute('aria-label');
            this.$.icon.setAttribute('aria-labelledby', 'label');
        }
    }
    toggleExpand_(event) {
        // Prevent |click| event from bubbling. It can cause parents of this
        // elements to erroneously re-toggle this control.
        event.stopPropagation();
        event.preventDefault();
        this.scrollIntoViewIfNeeded();
        this.expanded = !this.expanded;
        focusWithoutInk(this.$.icon);
    }
}
customElements.define(CrExpandButtonElement.is, CrExpandButtonElement);

let instance$c = null;
function getCss$c() {
    return instance$c || (instance$c = [...[getCss$K(), getCss$x()], css `:host{border-top:var(--print-preview-settings-border);display:block}:host([disabled]){pointer-events:none}div{align-items:center;display:flex;font:inherit;margin:0;min-height:48px}:host cr-expand-button{flex:1;padding-inline-end:calc(var(--print-preview-sidebar-margin) + 6px);padding-inline-start:var(--print-preview-sidebar-margin);--cr-expand-button-size:28px}:host([hidden]){display:none}:host([disabled]) #label{opacity:var(--cr-disabled-opacity)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$c() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div @click="${this.toggleExpandButton_}" actionable>
  <cr-expand-button aria-label="$i18n{moreOptionsLabel}"
      ?expanded="${this.settingsExpandedByUser}"
      @expanded-changed="${this.onSettingsExpandedByUserChanged_}"
      ?disabled="${this.disabled}">
    <div id="label">$i18n{moreOptionsLabel}</div>
  </cr-expand-button>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PrintPreviewMoreSettingsElement extends CrLitElement {
    static get is() {
        return 'print-preview-more-settings';
    }
    static get styles() {
        return getCss$c();
    }
    render() {
        return getHtml$c.bind(this)();
    }
    static get properties() {
        return {
            settingsExpandedByUser: {
                type: Boolean,
                notify: true,
            },
            disabled: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #settingsExpandedByUser_accessor_storage = false;
    get settingsExpandedByUser() { return this.#settingsExpandedByUser_accessor_storage; }
    set settingsExpandedByUser(value) { this.#settingsExpandedByUser_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    metrics_ = MetricsContext.printSettingsUi();
    /**
     * Toggles the expand button within the element being listened to.
     */
    toggleExpandButton_(e) {
        // The expand button handles toggling itself.
        const expandButtonTag = 'CR-EXPAND-BUTTON';
        if (e.target.tagName === expandButtonTag) {
            return;
        }
        if (!e.currentTarget.hasAttribute('actionable')) {
            return;
        }
        const expandButton = e.currentTarget.querySelector(expandButtonTag);
        assert(expandButton);
        expandButton.expanded = !expandButton.expanded;
        this.metrics_.record(this.settingsExpandedByUser ?
            PrintSettingsUiBucket.MORE_SETTINGS_CLICKED :
            PrintSettingsUiBucket.LESS_SETTINGS_CLICKED);
    }
    onSettingsExpandedByUserChanged_(e) {
        this.settingsExpandedByUser = e.detail.value;
    }
}
customElements.define(PrintPreviewMoreSettingsElement.is, PrintPreviewMoreSettingsElement);

let instance$b = null;
function getCss$b() {
    return instance$b || (instance$b = [...[getCss$x(), getCss$K()], css `print-preview-settings-section:not(.first-visible) .title{display:none}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$b() {
    // clang-format off
    return html `<!--_html_template_start_-->
${this.options_.map((item, index) => html `
  <print-preview-settings-section ?hidden="${!item.available}"
      class="${this.getClass_(index)}">
    <div slot="title">
      <span class="title">$i18n{optionsLabel}</span>
    </div>
    <div slot="controls" class="checkbox">
      <cr-checkbox id="${item.name}" data-index="${index}"
          ?disabled="${this.getDisabled_(item.managed)}"
          @change="${this.onChange_}" ?checked="${item.value}">
        <span>${this.i18n(item.label)}</span>
      </cr-checkbox>
    </div>
  </print-preview-settings-section>
`)}
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewOtherOptionsSettingsElementBase = SettingsMixin(I18nMixinLit(CrLitElement));
class PrintPreviewOtherOptionsSettingsElement extends PrintPreviewOtherOptionsSettingsElementBase {
    static get is() {
        return 'print-preview-other-options-settings';
    }
    static get styles() {
        return getCss$b();
    }
    render() {
        return getHtml$b.bind(this)();
    }
    static get properties() {
        return {
            disabled: { type: Boolean },
            options_: { type: Array },
            /**
             * The index of the checkbox that should display the "Options" title.
             */
            firstIndex_: { type: Number },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #options__accessor_storage = [
        { name: 'headerFooter', label: 'optionHeaderFooter' },
        { name: 'cssBackground', label: 'optionBackgroundColorsAndImages' },
        { name: 'rasterize', label: 'optionRasterize' },
        { name: 'selectionOnly', label: 'optionSelectionOnly' },
    ];
    get options_() { return this.#options__accessor_storage; }
    set options_(value) { this.#options__accessor_storage = value; }
    #firstIndex__accessor_storage = 0;
    get firstIndex_() { return this.#firstIndex__accessor_storage; }
    set firstIndex_(value) { this.#firstIndex__accessor_storage = value; }
    timeouts_ = new Map();
    previousValues_ = new Map();
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('headerFooter.*', this.onHeaderFooterSettingChange_.bind(this));
        this.onHeaderFooterSettingChange_();
        this.addSettingObserver('cssBackground.*', this.onCssBackgroundSettingChange_.bind(this));
        this.onCssBackgroundSettingChange_();
        this.addSettingObserver('rasterize.*', this.onRasterizeSettingChange_.bind(this));
        this.onRasterizeSettingChange_();
        this.addSettingObserver('selectionOnly.*', this.onSelectionOnlySettingChange_.bind(this));
        this.onSelectionOnlySettingChange_();
    }
    /**
     * @param settingName The name of the setting to updated.
     * @param newValue The new value for the setting.
     */
    updateSettingWithTimeout_(settingName, newValue) {
        const timeout = this.timeouts_.get(settingName);
        if (timeout !== null) {
            clearTimeout(timeout);
        }
        this.timeouts_.set(settingName, setTimeout(() => {
            this.timeouts_.delete(settingName);
            if (this.previousValues_.get(settingName) === newValue) {
                return;
            }
            this.previousValues_.set(settingName, newValue);
            this.setSetting(settingName, newValue);
            // For tests only
            this.fire('update-checkbox-setting', settingName);
        }, 200));
    }
    /**
     * @param index The index of the option to update.
     */
    updateOptionFromSetting_(index) {
        const setting = this.getSetting(this.options_[index].name);
        const option = this.options_[index];
        option.available = setting.available;
        option.value = setting.value;
        option.managed = setting.setByGlobalPolicy;
        this.requestUpdate();
        // Update first index
        const availableOptions = this.options_.filter(option => !!option.available);
        if (availableOptions.length > 0) {
            this.firstIndex_ = this.options_.indexOf(availableOptions[0]);
        }
    }
    /**
     * @param managed Whether the setting is managed by policy.
     * @return Whether the checkbox should be disabled.
     */
    getDisabled_(managed) {
        return managed || this.disabled;
    }
    onHeaderFooterSettingChange_() {
        this.updateOptionFromSetting_(0);
    }
    onCssBackgroundSettingChange_() {
        this.updateOptionFromSetting_(1);
    }
    onRasterizeSettingChange_() {
        this.updateOptionFromSetting_(2);
    }
    onSelectionOnlySettingChange_() {
        this.updateOptionFromSetting_(3);
    }
    /**
     * @param e Contains the checkbox item that was checked.
     */
    onChange_(e) {
        const index = Number.parseInt(e.target.dataset['index'], 10);
        const name = this.options_[index].name;
        this.updateSettingWithTimeout_(name, this.shadowRoot.querySelector(`#${name}`).checked);
    }
    /**
     * @param index The index of the settings section.
     * @return Class string containing 'first-visible' if the settings
     *     section is the first visible.
     */
    getClass_(index) {
        return index === this.firstIndex_ ? 'first-visible' : '';
    }
}
customElements.define(PrintPreviewOtherOptionsSettingsElement.is, PrintPreviewOtherOptionsSettingsElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$a() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span id="pages-per-sheet-label" slot="title">$i18n{pagesPerSheetLabel}
  </span>
  <div slot="controls">
    <select class="md-select" aria-labelledby="pages-per-sheet-label"
        ?disabled="${this.disabled}" .value="${this.selectedValue}"
        @change="${this.onSelectChange}">
      <option value="1" selected>1</option>
      <option value="2">2</option>
      <option value="4">4</option>
      <option value="6">6</option>
      <option value="9">9</option>
      <option value="16">16</option>
    </select>
  </div>
</print-preview-settings-section>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewPagesPerSheetSettingsElementBase = SettingsMixin(SelectMixin(CrLitElement));
class PrintPreviewPagesPerSheetSettingsElement extends PrintPreviewPagesPerSheetSettingsElementBase {
    static get is() {
        return 'print-preview-pages-per-sheet-settings';
    }
    static get styles() {
        return [
            getCss$x(),
            getCss$z(),
        ];
    }
    render() {
        return getHtml$a.bind(this)();
    }
    static get properties() {
        return {
            disabled: { type: Boolean },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.onPagesPerSheetSettingChange_(this.getSettingValue('pagesPerSheet'));
        this.addSettingObserver('pagesPerSheet.value', (newValue) => {
            this.onPagesPerSheetSettingChange_(newValue);
        });
    }
    /**
     * @param newValue The new value of the pages per sheet setting.
     */
    onPagesPerSheetSettingChange_(newValue) {
        this.selectedValue = newValue.toString();
    }
    onProcessSelectChange(value) {
        this.setSetting('pagesPerSheet', parseInt(value, 10));
    }
}
customElements.define(PrintPreviewPagesPerSheetSettingsElement.is, PrintPreviewPagesPerSheetSettingsElement);

let instance$a = null;
function getCss$a() {
    return instance$a || (instance$a = [...[getCss$x(), getCss$z()], css `:host([error-state_='0']) #pageSettingsCustomInput,:host([error-state_='3']) #pageSettingsCustomInput{--cr-input-error-display:none}:host([error-state_='1']) #customInputWrapper,:host([error-state_='2']) #customInputWrapper{margin-bottom:8px}#pageSettingsCustomInput{cursor:default;--cr-form-field-label-height:100%}:host #title{align-self:baseline}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$9() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span slot="title" id="pages-label">$i18n{pagesLabel}</span>
  <div slot="controls">
    <select class="md-select" aria-labelledby="pages-label"
        ?disabled="${this.controlsDisabled_}" .value="${this.selectedValue}"
        @change="${this.onSelectChange}" @blur="${this.onSelectBlur_}">
      <option value="${PagesValue.ALL}" selected>
        $i18n{optionAllPages}
      </option>
      <option value="${PagesValue.ODDS}"
          ?hidden="${this.isSinglePage_()}">
        $i18n{optionOddPages}
      </option>
      <option value="${PagesValue.EVENS}"
          ?hidden="${this.isSinglePage_()}">
        $i18n{optionEvenPages}
      </option>
      <option value="${PagesValue.CUSTOM}">
        $i18n{optionCustomPages}
      </option>
    </select>
  </div>
</print-preview-settings-section>
<cr-collapse ?opened="${this.shouldShowInput_()}"
    @transitionend="${this.onCollapseChanged_}">
  <print-preview-settings-section id="customInputWrapper">
    <div slot="title"></div>
    <div slot="controls">
      <cr-input id="pageSettingsCustomInput" class="stroked" type="text"
          data-timeout-delay="500" ?invalid="${this.hasError_}"
          ?disabled="${this.inputDisabled_()}"
          spellcheck="false" placeholder="$i18n{examplePageRangeText}"
          error-message="${this.getHintMessage_()}"
          @blur="${this.onCustomInputBlur_}">
      </cr-input>
    </div>
  </print-preview-settings-section>
</cr-collapse><!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var PagesInputErrorState;
(function (PagesInputErrorState) {
    PagesInputErrorState[PagesInputErrorState["NO_ERROR"] = 0] = "NO_ERROR";
    PagesInputErrorState[PagesInputErrorState["INVALID_SYNTAX"] = 1] = "INVALID_SYNTAX";
    PagesInputErrorState[PagesInputErrorState["OUT_OF_BOUNDS"] = 2] = "OUT_OF_BOUNDS";
    PagesInputErrorState[PagesInputErrorState["EMPTY"] = 3] = "EMPTY";
})(PagesInputErrorState || (PagesInputErrorState = {}));
var PagesValue;
(function (PagesValue) {
    PagesValue[PagesValue["ALL"] = 0] = "ALL";
    PagesValue[PagesValue["ODDS"] = 1] = "ODDS";
    PagesValue[PagesValue["EVENS"] = 2] = "EVENS";
    PagesValue[PagesValue["CUSTOM"] = 3] = "CUSTOM";
})(PagesValue || (PagesValue = {}));
/**
 * Used in place of Number.parseInt(), to ensure values like '1  2' or '1a2' are
 * not allowed.
 * @param value The value to convert to a number.
 * @return The value converted to a number, or NaN if it cannot be converted.
 */
function parseIntStrict(value) {
    if (/^\d+$/.test(value.trim())) {
        return Number(value);
    }
    return NaN;
}
const PrintPreviewPagesSettingsElementBase = WebUiListenerMixinLit(InputMixin(SettingsMixin(SelectMixin(CrLitElement))));
class PrintPreviewPagesSettingsElement extends PrintPreviewPagesSettingsElementBase {
    static get is() {
        return 'print-preview-pages-settings';
    }
    static get styles() {
        return getCss$a();
    }
    render() {
        return getHtml$9.bind(this)();
    }
    static get properties() {
        return {
            disabled: { type: Boolean },
            pageCount: { type: Number },
            controlsDisabled_: { type: Boolean },
            errorState_: {
                type: Number,
                reflect: true,
            },
            hasError_: { type: Boolean },
            inputString_: { type: String },
            pagesToPrint_: { type: Array },
            rangesToPrint_: { type: Array },
            selection_: { type: Number },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #pageCount_accessor_storage = 0;
    get pageCount() { return this.#pageCount_accessor_storage; }
    set pageCount(value) { this.#pageCount_accessor_storage = value; }
    #controlsDisabled__accessor_storage = false;
    get controlsDisabled_() { return this.#controlsDisabled__accessor_storage; }
    set controlsDisabled_(value) { this.#controlsDisabled__accessor_storage = value; }
    #errorState__accessor_storage = PagesInputErrorState.NO_ERROR;
    get errorState_() { return this.#errorState__accessor_storage; }
    set errorState_(value) { this.#errorState__accessor_storage = value; }
    #hasError__accessor_storage = false;
    get hasError_() { return this.#hasError__accessor_storage; }
    set hasError_(value) { this.#hasError__accessor_storage = value; }
    #inputString__accessor_storage = '';
    get inputString_() { return this.#inputString__accessor_storage; }
    set inputString_(value) { this.#inputString__accessor_storage = value; }
    #pagesToPrint__accessor_storage = [];
    get pagesToPrint_() { return this.#pagesToPrint__accessor_storage; }
    set pagesToPrint_(value) { this.#pagesToPrint__accessor_storage = value; }
    #rangesToPrint__accessor_storage = [];
    get rangesToPrint_() { return this.#rangesToPrint__accessor_storage; }
    set rangesToPrint_(value) { this.#rangesToPrint__accessor_storage = value; }
    #selection__accessor_storage = PagesValue.ALL;
    get selection_() { return this.#selection__accessor_storage; }
    set selection_(value) { this.#selection__accessor_storage = value; }
    /**
     * True if the user's last valid input should be restored to the custom
     * input field. Cleared when the input is set automatically, or the user
     * manually clears the field.
     */
    restoreLastInput_ = true;
    /**
     * Memorizes the user's last non-custom pages setting. Used when
     * `PagesValue.ODDS` and `PagesValue.EVEN` become invalid due to a changed
     * page count.
     */
    restorationValue_ = PagesValue.ALL;
    firstUpdated() {
        this.addEventListener('input-change', e => this.onInputChange_(e));
    }
    /**
     * Initialize |selectedValue| in connectedCallback() since this doesn't
     * observe settings.pages, because settings.pages is not sticky.
     */
    connectedCallback() {
        super.connectedCallback();
        this.selectedValue = PagesValue.ALL.toString();
        this.addSettingObserver('pages', () => this.onRangeChange_());
        this.addSettingObserver('pagesPerSheet.value', () => this.onRangeChange_());
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('disabled') ||
            changedPrivateProperties.has('hasError_')) {
            this.controlsDisabled_ = this.computeControlsDisabled_();
        }
        if (changedPrivateProperties.has('inputString_')) {
            this.updatePagesToPrint_();
        }
        if (changedPrivateProperties.has('pagesToPrint_')) {
            this.rangesToPrint_ = this.computeRangesToPrint_();
        }
        if (changedPrivateProperties.has('errorState_') ||
            changedPrivateProperties.has('rangesToPrint_')) {
            this.onRangeChange_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('pageCount')) {
            this.onPageCountChange_(this.pageCount, changedProperties.get('pageCount'));
        }
        if (changedPrivateProperties.has('selection_')) {
            this.onSelectionChange_();
        }
    }
    /** The cr-input field element for InputMixin. */
    getInput() {
        return this.$.pageSettingsCustomInput;
    }
    setSelectedValue_(value) {
        this.selectedValue = value.toString();
        this.shadowRoot.querySelector('select').dispatchEvent(new CustomEvent('change', { bubbles: true, composed: true }));
    }
    onInputChange_(e) {
        if (this.inputString_ !== e.detail) {
            this.restoreLastInput_ = true;
        }
        this.inputString_ = e.detail;
    }
    onProcessSelectChange(value) {
        this.selection_ = parseInt(value, 10);
    }
    onCollapseChanged_() {
        if (this.selection_ === PagesValue.CUSTOM) {
            this.$.pageSettingsCustomInput.inputElement.focus();
        }
    }
    /**
     * @return Whether the controls should be disabled.
     */
    computeControlsDisabled_() {
        // Disable the input if other settings are responsible for the error state.
        return !this.hasError_ && this.disabled;
    }
    /**
     * Updates pages to print and error state based on the validity and
     * current value of the input.
     */
    updatePagesToPrint_() {
        if (this.selection_ !== PagesValue.CUSTOM) {
            this.errorState_ = PagesInputErrorState.NO_ERROR;
            if (!this.pageCount) {
                this.pagesToPrint_ = [];
                return;
            }
            const first = this.selection_ === PagesValue.EVENS ? 2 : 1;
            const step = this.selection_ === PagesValue.ALL ? 1 : 2;
            assert(first === 1 || this.pageCount !== 1);
            const length = Math.floor(1 + (this.pageCount - first) / step);
            this.pagesToPrint_ = Array.from({ length }, (_, i) => step * i + first);
            return;
        }
        else if (this.inputString_ === '') {
            this.errorState_ = PagesInputErrorState.EMPTY;
            return;
        }
        const pages = [];
        const added = {};
        const ranges = this.inputString_.split(/,|\u3001/);
        const maxPage = this.pageCount;
        for (const range of ranges) {
            if (range === '') {
                this.errorState_ = PagesInputErrorState.INVALID_SYNTAX;
                this.onRangeChange_();
                return;
            }
            const limits = range.split('-');
            if (limits.length > 2) {
                this.errorState_ = PagesInputErrorState.INVALID_SYNTAX;
                this.onRangeChange_();
                return;
            }
            let min = parseIntStrict(limits[0]);
            if ((limits[0].length > 0 && Number.isNaN(min)) || min < 1) {
                this.errorState_ = PagesInputErrorState.INVALID_SYNTAX;
                this.onRangeChange_();
                return;
            }
            if (limits.length === 1) {
                if (min > maxPage) {
                    this.errorState_ = PagesInputErrorState.OUT_OF_BOUNDS;
                    this.onRangeChange_();
                    return;
                }
                if (!added.hasOwnProperty(min)) {
                    pages.push(min);
                    added[min] = true;
                }
                continue;
            }
            let max = parseIntStrict(limits[1]);
            if (Number.isNaN(max) && limits[1].length > 0) {
                this.errorState_ = PagesInputErrorState.INVALID_SYNTAX;
                this.onRangeChange_();
                return;
            }
            if (Number.isNaN(min)) {
                min = 1;
            }
            if (Number.isNaN(max)) {
                max = maxPage;
            }
            if (min > max) {
                this.errorState_ = PagesInputErrorState.INVALID_SYNTAX;
                this.onRangeChange_();
                return;
            }
            if (max > maxPage) {
                this.errorState_ = PagesInputErrorState.OUT_OF_BOUNDS;
                this.onRangeChange_();
                return;
            }
            for (let i = min; i <= max; i++) {
                if (!added.hasOwnProperty(i)) {
                    pages.push(i);
                    added[i] = true;
                }
            }
        }
        // Page numbers should be sorted to match the order of the pages in the
        // rendered PDF.
        pages.sort((left, right) => left - right);
        this.errorState_ = PagesInputErrorState.NO_ERROR;
        this.pagesToPrint_ = pages;
    }
    computeRangesToPrint_() {
        if (!this.pagesToPrint_ || this.pagesToPrint_.length === 0 ||
            this.pagesToPrint_[0] === -1 ||
            this.pagesToPrint_.length === this.pageCount) {
            return [];
        }
        let from = this.pagesToPrint_[0];
        let to = this.pagesToPrint_[0];
        const ranges = [];
        for (const page of this.pagesToPrint_.slice(1)) {
            if (page === to + 1) {
                to = page;
                continue;
            }
            ranges.push({ from: from, to: to });
            from = page;
            to = page;
        }
        ranges.push({ from: from, to: to });
        return ranges;
    }
    /**
     * @return The final page numbers, reflecting N-up setting.
     *     Page numbers are 1 indexed, since these numbers are displayed to the
     *     user.
     */
    getNupPages_() {
        const pagesPerSheet = this.getSettingValue('pagesPerSheet');
        if (pagesPerSheet <= 1 || this.pagesToPrint_.length === 0) {
            return this.pagesToPrint_;
        }
        const numPages = Math.ceil(this.pagesToPrint_.length / pagesPerSheet);
        const nupPages = new Array(numPages);
        for (let i = 0; i < nupPages.length; i++) {
            nupPages[i] = i + 1;
        }
        return nupPages;
    }
    /**
     * Updates the model with pages and validity, and adds error styling if
     * needed.
     */
    onRangeChange_() {
        if (this.errorState_ === PagesInputErrorState.EMPTY) {
            this.setSettingValid('pages', true);
            this.hasError_ = false;
            return;
        }
        if (this.errorState_ !== PagesInputErrorState.NO_ERROR) {
            this.hasError_ = true;
            this.setSettingValid('pages', false);
            return;
        }
        const nupPages = this.getNupPages_();
        const rangesChanged = !areRangesEqual(this.rangesToPrint_, this.getSettingValue('ranges'));
        if (rangesChanged ||
            nupPages.length !== this.getSettingValue('pages').length) {
            this.setSetting('pages', nupPages);
        }
        if (rangesChanged) {
            this.setSetting('ranges', this.rangesToPrint_);
        }
        this.setSettingValid('pages', true);
        this.hasError_ = false;
    }
    onSelectBlur_(event) {
        if (this.selection_ !== PagesValue.CUSTOM ||
            event.relatedTarget === this.$.pageSettingsCustomInput) {
            return;
        }
        this.onCustomInputBlur_();
    }
    async onCustomInputBlur_() {
        this.resetAndUpdate();
        await this.shadowRoot.querySelector('cr-input').updateComplete;
        if (this.errorState_ === PagesInputErrorState.EMPTY) {
            // Update with all pages.
            this.shadowRoot.querySelector('cr-input').value =
                this.getAllPagesString_();
            this.inputString_ = this.getAllPagesString_();
            this.resetString();
            this.restoreLastInput_ = false;
        }
        this.dispatchEvent(new CustomEvent('custom-input-blurred-for-test', { bubbles: true, composed: true }));
    }
    /**
     * @return Message to show as hint.
     */
    getHintMessage_() {
        if (this.errorState_ === PagesInputErrorState.NO_ERROR ||
            this.errorState_ === PagesInputErrorState.EMPTY) {
            return '';
        }
        let formattedMessage = '';
        if (this.errorState_ === PagesInputErrorState.INVALID_SYNTAX) {
            formattedMessage = loadTimeData.getStringF('pageRangeSyntaxInstruction', loadTimeData.getString('examplePageRangeText'));
        }
        else {
            formattedMessage = loadTimeData.getStringF('pageRangeLimitInstructionWithValue', this.pageCount);
        }
        return formattedMessage.replace(/<\/b>|<b>/g, '');
    }
    /**
     * @return Whether the document being printed has only one page.
     */
    isSinglePage_() {
        return this.pageCount === 1;
    }
    /**
     * @return Whether to disable the custom input.
     */
    inputDisabled_() {
        return this.selection_ !== PagesValue.CUSTOM || this.controlsDisabled_;
    }
    /**
     * @return Whether to display the custom input.
     */
    shouldShowInput_() {
        return this.selection_ === PagesValue.CUSTOM;
    }
    /**
     * @return A string representing the full page range.
     */
    getAllPagesString_() {
        if (this.pageCount === 0) {
            return '';
        }
        return this.pageCount === 1 ? '1' : `1-${this.pageCount}`;
    }
    onSelectionChange_() {
        const customSelected = this.selection_ === PagesValue.CUSTOM;
        if ((customSelected && !this.restoreLastInput_) ||
            this.errorState_ !== PagesInputErrorState.NO_ERROR) {
            this.restoreLastInput_ = true;
            this.inputString_ = '';
            this.shadowRoot.querySelector('cr-input').value = '';
            this.resetString();
        }
        this.updatePagesToPrint_();
    }
    onPageCountChange_(current, previous) {
        // Remember non-custom page settings when the page count changes to 1, so
        // they can be re-applied if the page count exceeds 1 again.
        if (this.selection_ !== PagesValue.CUSTOM) {
            if (current === 1) {
                this.restorationValue_ = this.selection_;
                this.setSelectedValue_(PagesValue.ALL);
            }
            else if (previous === 1) {
                assert(this.restorationValue_ !== PagesValue.CUSTOM);
                this.setSelectedValue_(this.restorationValue_);
            }
        }
        // Reset the custom input to the new "all pages" value if it is equal to the
        // full page range and was either set automatically, or would become invalid
        // due to the page count change.
        const resetCustom = this.selection_ === PagesValue.CUSTOM &&
            !!this.pagesToPrint_ && this.pagesToPrint_.length === previous &&
            (current < previous || !this.restoreLastInput_);
        if (resetCustom) {
            this.shadowRoot.querySelector('cr-input').value =
                this.getAllPagesString_();
            this.inputString_ = this.getAllPagesString_();
            this.resetString();
        }
        else {
            this.updatePagesToPrint_();
        }
    }
}
customElements.define(PrintPreviewPagesSettingsElement.is, PrintPreviewPagesSettingsElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$8() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span slot="title" id="scaling-label">$i18n{scalingLabel}</span>
  <div slot="controls">
    <select class="md-select" aria-labelledby="scaling-label"
        ?disabled="${this.dropdownDisabled_}" .value="${this.selectedValue}"
        @change="${this.onSelectChange}">
      <option value="${ScalingType.DEFAULT}"
          ?selected="${this.isSelected_(ScalingType.DEFAULT)}">
        $i18n{optionDefaultScaling}
      </option>
      <option value="${ScalingType.FIT_TO_PAGE}" ?hidden="${!this.isPdf}"
          ?disabled="${!this.isPdf}"
          ?selected="${this.isSelected_(ScalingType.FIT_TO_PAGE)}">
        $i18n{optionFitToPage}
      </option>
      <option value="${ScalingType.FIT_TO_PAPER}" ?hidden="${!this.isPdf}"
          ?disabled="${!this.isPdf}"
          ?selected="${this.isSelected_(ScalingType.FIT_TO_PAPER)}">
        $i18n{optionFitToPaper}
      </option>
      <option value="${ScalingType.CUSTOM}"
          ?selected="${this.isSelected_(ScalingType.CUSTOM)}">
        $i18n{optionCustomScaling}
      </option>
    </select>
  </div>
</print-preview-settings-section>
<cr-collapse ?opened="${this.customSelected_}"
    @transitionend="${this.onCollapseChanged_}">
  <print-preview-number-settings-section
      max-value="200" min-value="10" default-value="100"
      ?disabled="${this.inputDisabled_()}"
      current-value="${this.currentValue_}"
      @current-value-changed="${this.onCurrentValueChanged_}"
      ?input-valid="${this.inputValid_}"
      @input-valid-changed="${this.onInputValidChanged_}"
      hint-message="$i18n{scalingInstruction}">
  </print-preview-number-settings-section>
</cr-collapse><!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/*
 * Fit to page and fit to paper options will only be displayed for PDF
 * documents. If the custom option is selected, an additional input field will
 * appear to enter the custom scale factor.
 */
const PrintPreviewScalingSettingsElementBase = SettingsMixin(SelectMixin(CrLitElement));
class PrintPreviewScalingSettingsElement extends PrintPreviewScalingSettingsElementBase {
    static get is() {
        return 'print-preview-scaling-settings';
    }
    static get styles() {
        return [
            getCss$x(),
            getCss$z(),
        ];
    }
    render() {
        return getHtml$8.bind(this)();
    }
    static get properties() {
        return {
            disabled: { type: Boolean },
            isPdf: { type: Boolean },
            currentValue_: { type: String },
            customSelected_: { type: Boolean },
            scalingTypeValue_: { type: Number },
            scalingTypePdfValue_: { type: Number },
            inputValid_: { type: Boolean },
            dropdownDisabled_: { type: Boolean },
            settingKey_: { type: String },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #isPdf_accessor_storage = false;
    get isPdf() { return this.#isPdf_accessor_storage; }
    set isPdf(value) { this.#isPdf_accessor_storage = value; }
    #currentValue__accessor_storage = '';
    get currentValue_() { return this.#currentValue__accessor_storage; }
    set currentValue_(value) { this.#currentValue__accessor_storage = value; }
    #customSelected__accessor_storage = false;
    get customSelected_() { return this.#customSelected__accessor_storage; }
    set customSelected_(value) { this.#customSelected__accessor_storage = value; }
    #dropdownDisabled__accessor_storage = false;
    get dropdownDisabled_() { return this.#dropdownDisabled__accessor_storage; }
    set dropdownDisabled_(value) { this.#dropdownDisabled__accessor_storage = value; }
    #inputValid__accessor_storage = false;
    get inputValid_() { return this.#inputValid__accessor_storage; }
    set inputValid_(value) { this.#inputValid__accessor_storage = value; }
    #settingKey__accessor_storage = 'scalingType';
    get settingKey_() { return this.#settingKey__accessor_storage; }
    set settingKey_(value) { this.#settingKey__accessor_storage = value; }
    #scalingTypeValue__accessor_storage = ScalingType.DEFAULT;
    get scalingTypeValue_() { return this.#scalingTypeValue__accessor_storage; }
    set scalingTypeValue_(value) { this.#scalingTypeValue__accessor_storage = value; }
    #scalingTypePdfValue__accessor_storage = ScalingType.DEFAULT;
    get scalingTypePdfValue_() { return this.#scalingTypePdfValue__accessor_storage; }
    set scalingTypePdfValue_(value) { this.#scalingTypePdfValue__accessor_storage = value; }
    lastValidScaling_ = '';
    /**
     * Whether the custom scaling setting has been set to true, but the custom
     * input has not yet been expanded. Used to determine whether changes in the
     * dropdown are due to user input or sticky settings.
     */
    customScalingSettingSet_ = false;
    /**
     * Whether the user has selected custom scaling in the dropdown, but the
     * custom input has not yet been expanded. Used to determine whether to
     * auto-focus the custom input.
     */
    userSelectedCustomScaling_ = false;
    connectedCallback() {
        super.connectedCallback();
        this.addSettingObserver('scaling.value', this.onScalingSettingChanged_.bind(this));
        this.onScalingSettingChanged_();
        this.addSettingObserver('scalingType.*', () => {
            this.scalingTypeValue_ = this.getSettingValue('scalingType');
        });
        this.addSettingObserver('scalingTypePdf.*', () => {
            this.scalingTypePdfValue_ = this.getSettingValue('scalingTypePdf');
        });
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('isPdf')) {
            this.settingKey_ = this.isPdf ? 'scalingTypePdf' : 'scalingType';
        }
        if (changedPrivateProperties.has('settingKey_') ||
            changedPrivateProperties.has('scalingTypeValue_') ||
            changedPrivateProperties.has('scalingTypePdfValue_')) {
            this.customSelected_ = this.computeCustomSelected_();
            this.onScalingTypeSettingChanged_();
        }
        if (changedProperties.has('disabled') ||
            changedPrivateProperties.has('inputValid_')) {
            this.dropdownDisabled_ = this.disabled && this.inputValid_;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('inputValid_') ||
            changedPrivateProperties.has('currentValue_')) {
            this.onInputFieldChanged_();
        }
    }
    onProcessSelectChange(value) {
        const isCustom = value === ScalingType.CUSTOM.toString();
        if (isCustom && !this.customScalingSettingSet_) {
            this.userSelectedCustomScaling_ = true;
        }
        else {
            this.customScalingSettingSet_ = false;
        }
        const valueAsNumber = parseInt(value, 10);
        if (isCustom || value === ScalingType.DEFAULT.toString()) {
            this.setSetting('scalingType', valueAsNumber);
        }
        if (this.isPdf ||
            this.getSetting('scalingTypePdf').value === ScalingType.DEFAULT ||
            this.getSetting('scalingTypePdf').value === ScalingType.CUSTOM) {
            this.setSetting('scalingTypePdf', valueAsNumber);
        }
        if (isCustom) {
            this.setSetting('scaling', this.currentValue_);
        }
    }
    updateScalingToValid_() {
        if (!this.getSetting('scaling').valid) {
            this.currentValue_ = this.lastValidScaling_;
        }
        else {
            this.lastValidScaling_ = this.currentValue_;
        }
    }
    /**
     * Updates the input string when scaling setting is set.
     */
    onScalingSettingChanged_() {
        const value = this.getSetting('scaling').value;
        this.lastValidScaling_ = value;
        this.currentValue_ = value;
    }
    onScalingTypeSettingChanged_() {
        if (!this.settingKey_) {
            return;
        }
        const value = this.getSettingValue(this.settingKey_);
        if (value !== ScalingType.CUSTOM) {
            this.updateScalingToValid_();
        }
        else {
            this.customScalingSettingSet_ = true;
        }
        this.selectedValue = value.toString();
    }
    /**
     * Updates scaling settings based on the validity and current value of the
     * scaling input.
     */
    onInputFieldChanged_() {
        this.setSettingValid('scaling', this.inputValid_);
        if (this.currentValue_ !== undefined && this.currentValue_ !== '' &&
            this.inputValid_ &&
            this.currentValue_ !== this.getSettingValue('scaling')) {
            this.setSetting('scaling', this.currentValue_);
        }
    }
    /**
     * @return Whether the input should be disabled.
     */
    inputDisabled_() {
        return !this.customSelected_ || this.dropdownDisabled_;
    }
    /**
     * @return Whether the custom scaling option is selected.
     */
    computeCustomSelected_() {
        return !!this.settingKey_ &&
            this.getSettingValue(this.settingKey_) === ScalingType.CUSTOM;
    }
    onCollapseChanged_() {
        if (this.customSelected_ && this.userSelectedCustomScaling_) {
            this.shadowRoot.querySelector('print-preview-number-settings-section')
                .getInput()
                .focus();
        }
        this.customScalingSettingSet_ = false;
        this.userSelectedCustomScaling_ = false;
    }
    onCurrentValueChanged_(e) {
        this.currentValue_ = e.detail.value;
    }
    onInputValidChanged_(e) {
        this.inputValid_ = e.detail.value;
    }
    isSelected_(value) {
        return this.selectedValue === value.toString();
    }
}
customElements.define(PrintPreviewScalingSettingsElement.is, PrintPreviewScalingSettingsElement);

let instance$9 = null;
function getCss$9() {
    return instance$9 || (instance$9 = [...[], css `.throbber{background:url(chrome://resources/images/throbber_small.svg) no-repeat;display:inline-block;height:var(--throbber-size);width:var(--throbber-size)}`]);
}

let instance$8 = null;
function getCss$8() {
    return instance$8 || (instance$8 = [...[getCss$E(), getCss$9(), getCss$K()], css `:host{display:block}:host cr-icon-button{--cr-icon-button-icon-size:16px;--cr-icon-button-margin-end:-2px;--cr-icon-button-margin-start:0;--cr-icon-button-size:28px}.link:not([actionable]){pointer-events:none}.throbber{margin:8px;min-height:16px;min-width:16px}.link{align-items:center;display:flex;min-height:46px;padding:0 var(--print-preview-sidebar-margin)}#systemDialogLink{padding-top:0.5em}.label{flex:1}.link:not([actionable]) .label{opacity:var(--cr-disabled-opacity)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$7() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div class="link" id="systemDialogLink"
    ?actionable="${!this.systemDialogLinkDisabled_}"
    ?hidden="${!this.shouldShowSystemDialogLink_}"
    @click="${this.onSystemDialogClick_}">
  <div class="label">$i18n{systemDialogOption}</div>
  <cr-icon-button class="icon-external"
      ?hidden="${this.openingSystemDialog_}"
      ?disabled="${this.systemDialogLinkDisabled_}"
      aria-label="$i18n{systemDialogOption}"></cr-icon-button>
  <div id="systemDialogThrobber" ?hidden="${!this.openingSystemDialog_}"
      class="throbber"></div>
</div>

<div class="link" id="openPdfInPreviewLink" ?actionable="${!this.disabled}"
    @click="${this.onOpenInPreviewClick_}">
  <div class="label">$i18n{openPdfInPreviewOption}</div>
  <cr-icon-button class="icon-external"
      ?hidden="${this.openingInPreview_}" ?disabled="${this.disabled}"
      aria-label="$i18n{openPdfInPreviewOption}"></cr-icon-button>
  <div id="openPdfInPreviewThrobber" ?hidden="${!this.openingInPreview_}"
      class="throbber"></div>
</div>

<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PrintPreviewLinkContainerElement extends CrLitElement {
    static get is() {
        return 'print-preview-link-container';
    }
    static get styles() {
        return getCss$8();
    }
    render() {
        return getHtml$7.bind(this)();
    }
    static get properties() {
        return {
            appKioskMode: { type: Boolean },
            destination: { type: Object },
            disabled: { type: Boolean },
            shouldShowSystemDialogLink_: {
                type: Boolean,
                reflect: true,
            },
            systemDialogLinkDisabled_: { type: Boolean },
            openingSystemDialog_: { type: Boolean },
            openingInPreview_: { type: Boolean },
        };
    }
    #appKioskMode_accessor_storage = false;
    get appKioskMode() { return this.#appKioskMode_accessor_storage; }
    set appKioskMode(value) { this.#appKioskMode_accessor_storage = value; }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #shouldShowSystemDialogLink__accessor_storage = false;
    get shouldShowSystemDialogLink_() { return this.#shouldShowSystemDialogLink__accessor_storage; }
    set shouldShowSystemDialogLink_(value) { this.#shouldShowSystemDialogLink__accessor_storage = value; }
    #systemDialogLinkDisabled__accessor_storage = false;
    get systemDialogLinkDisabled_() { return this.#systemDialogLinkDisabled__accessor_storage; }
    set systemDialogLinkDisabled_(value) { this.#systemDialogLinkDisabled__accessor_storage = value; }
    #openingSystemDialog__accessor_storage = false;
    get openingSystemDialog_() { return this.#openingSystemDialog__accessor_storage; }
    set openingSystemDialog_(value) { this.#openingSystemDialog__accessor_storage = value; }
    #openingInPreview__accessor_storage = false;
    get openingInPreview_() { return this.#openingInPreview__accessor_storage; }
    set openingInPreview_(value) { this.#openingInPreview__accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('disabled')) {
            this.systemDialogLinkDisabled_ = this.computeSystemDialogLinkDisabled_();
        }
        if (changedProperties.has('appKioskMode') ||
            changedProperties.has('destination')) {
            this.shouldShowSystemDialogLink_ =
                this.computeShouldShowSystemDialogLink_();
        }
    }
    /**
     * @return Whether the system dialog link should be visible.
     */
    computeShouldShowSystemDialogLink_() {
        if (this.appKioskMode) {
            return false;
        }
        // 
        return true;
        // 
        // 
    }
    /**
     * @return Whether the system dialog link should be disabled
     */
    computeSystemDialogLinkDisabled_() {
        // 
        return false;
        // 
        // 
    }
    // 
    async onSystemDialogClick_() {
        if (!this.shouldShowSystemDialogLink_) {
            return;
        }
        this.openingSystemDialog_ = true;
        await this.updateComplete;
        this.fire('print-with-system-dialog');
    }
    // 
    // 
    // 
    async onOpenInPreviewClick_() {
        this.openingInPreview_ = true;
        await this.updateComplete;
        this.fire('open-pdf-in-preview');
    }
    // 
    /** @return Whether the system dialog link is available. */
    systemDialogLinkAvailable() {
        return this.shouldShowSystemDialogLink_ && !this.systemDialogLinkDisabled_;
    }
}
customElements.define(PrintPreviewLinkContainerElement.is, PrintPreviewLinkContainerElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview CrScrollObserverMixinLit has logic to add CSS classes based
 * on the scroll state of a scrolling container element specified by a
 * #container ID.
 *
 * Elements using this mixin are expected to define a #container element, which
 * is the element being scrolled. There should only be one such element in the
 * DOM.
 * <div id="container">...</div>
 * Alternatively, clients can set a container element by overriding the
 * getContainer() method. This method should always return the same element
 * throughout the life of the client. It is first called in connectedCallback().
 *
 * The mixin will toggles CSS classes on #container indicating the current
 * scroll state.
 * can-scroll: Has content to scroll to
 * scrolled-to-top: Scrolled all the way to the top
 * scrolled-to-bottom: Scrolled all the way to the bottom
 *
 * Clients can use these classes to define styles.
 */
const CrScrollObserverMixinLit = (superClass) => {
    class CrScrollObserverMixinLit extends superClass {
        intersectionObserver_ = null;
        topProbe_ = null;
        bottomProbe_ = null;
        connectedCallback() {
            super.connectedCallback();
            const container = this.getContainer();
            this.topProbe_ = document.createElement('div');
            this.bottomProbe_ = document.createElement('div');
            container.prepend(this.topProbe_);
            container.append(this.bottomProbe_);
            this.enableScrollObservation(true);
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            this.enableScrollObservation(false);
        }
        getContainer() {
            const container = this.shadowRoot.querySelector('#container');
            assert(container);
            return container;
        }
        getIntersectionObserver_() {
            const callback = (entries) => {
                // In some rare cases, there could be more than one entry per
                // observed element, in which case the last entry's result
                // stands.
                const container = this.getContainer();
                for (const entry of entries) {
                    const target = entry.target;
                    if (target === this.topProbe_) {
                        container.classList.toggle('scrolled-to-top', entry.intersectionRatio !== 0);
                        const canScroll = entry.intersectionRatio === 0 ||
                            !container.classList.contains('scrolled-to-bottom');
                        container.classList.toggle('can-scroll', canScroll);
                    }
                    if (target === this.bottomProbe_) {
                        container.classList.toggle('scrolled-to-bottom', entry.intersectionRatio !== 0);
                        const canScroll = entry.intersectionRatio === 0 ||
                            !container.classList.contains('scrolled-to-top');
                        container.classList.toggle('can-scroll', canScroll);
                    }
                }
            };
            return new IntersectionObserver(callback, { root: this.getContainer(), threshold: 0 });
        }
        /**
         * @param enable Whether to enable the mixin or disable it.
         *     This function does nothing if the mixin is already in the
         *     requested state.
         */
        enableScrollObservation(enable) {
            // Behavior is already enabled/disabled. Return early.
            if (enable === !!this.intersectionObserver_) {
                return;
            }
            if (!enable) {
                this.intersectionObserver_.disconnect();
                this.intersectionObserver_ = null;
                return;
            }
            this.intersectionObserver_ = this.getIntersectionObserver_();
            // Need to register the observer within a setTimeout() callback,
            // otherwise the drop shadow flashes once on startup, because of the
            // DOM modifications earlier in this function causing a relayout.
            window.setTimeout(() => {
                // In case this is already detached.
                if (!this.isConnected) {
                    return;
                }
                if (this.intersectionObserver_) {
                    assert(this.topProbe_);
                    assert(this.bottomProbe_);
                    this.intersectionObserver_.observe(this.topProbe_);
                    this.intersectionObserver_.observe(this.bottomProbe_);
                }
            });
        }
    }
    return CrScrollObserverMixinLit;
};

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview CrContainerShadowMixinLit holds logic for showing a drop shadow
 * near the top of a container element, when the content has scrolled. Inherits
 * from CrScrollObserverMixinLit.
 *
 * Lit version of CrContainerShadowMixin.
 *
 * Elements using this mixin are expected to define a #container element which
 * is the element being scrolled.
 *
 * If the #container element has a show-bottom-shadow attribute, a drop shadow
 * will also be shown near the bottom of the container element, when there
 * is additional content to scroll to. Examples:
 *
 * For both top and bottom shadows:
 * <div id="container" show-bottom-shadow>...</div>
 *
 * For top shadow only:
 * <div id="container">...</div>
 *
 * The mixin will take care of inserting an element with ID
 * 'cr-container-shadow-top' which holds the drop shadow effect, and,
 * optionally, an element with ID 'cr-container-shadow-bottom' which holds the
 * same effect. Note that the show-bottom-shadow attribute is inspected only
 * during connectedCallback() and any changes that occur after that point will
 * not be respected.
 *
 * Clients should either use the existing shared styling in
 * cr_shared_style.css, '#cr-container-shadow-[top/bottom]' and
 * '#cr-container-shadow-top:has(+ #container.can-scroll:not(.scrolled-to-top))'
 * and '#container.can-scroll:not(.scrolled-to-bottom) +
 *     #cr-container-shadow-bottom'
 * or define their own styles.
 */
var CrContainerShadowSide;
(function (CrContainerShadowSide) {
    CrContainerShadowSide["TOP"] = "top";
    CrContainerShadowSide["BOTTOM"] = "bottom";
})(CrContainerShadowSide || (CrContainerShadowSide = {}));
const CrContainerShadowMixinLit = (superClass) => {
    const superClassBase = CrScrollObserverMixinLit(superClass);
    class CrContainerShadowMixinLit extends superClassBase {
        dropShadows_ = new Map();
        sides_ = [];
        connectedCallback() {
            super.connectedCallback();
            const container = this.shadowRoot.querySelector('#container');
            assert(container);
            const hasBottomShadow = container.hasAttribute('show-bottom-shadow');
            this.sides_ = hasBottomShadow ?
                [CrContainerShadowSide.TOP, CrContainerShadowSide.BOTTOM] :
                [CrContainerShadowSide.TOP];
            this.sides_.forEach(side => {
                // The element holding the drop shadow effect to be shown.
                const shadow = document.createElement('div');
                shadow.id = `cr-container-shadow-${side}`;
                shadow.classList.add('cr-container-shadow');
                this.dropShadows_.set(side, shadow);
            });
            container.parentNode.insertBefore(this.dropShadows_.get(CrContainerShadowSide.TOP), container);
            if (hasBottomShadow) {
                container.parentNode.insertBefore(this.dropShadows_.get(CrContainerShadowSide.BOTTOM), container.nextSibling);
            }
        }
        /**
         * Toggles the force-shadow class. If |enabled| is true, shadows will be
         * forced to show regardless of scroll state when using the shared styles
         * in cr_shared_style.css. If false, shadows can be shown using classes set
         * by CrScrollObserverMixin.
         */
        setForceDropShadows(enabled) {
            assert(this.sides_.length > 0);
            for (const side of this.sides_) {
                this.dropShadows_.get(side).classList.toggle('force-shadow', enabled);
            }
        }
    }
    return CrContainerShadowMixinLit;
};

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * cr-lazy-render-lit helps with lazy rendering elements only when they are
 * actually needed (requested to be shown by the user). The lazy rendered
 * node is rendered right before the cr-lazy-render-lit node itself, such that
 * it can be fully styled by the parent, or use Lit bindings referring to the
 * parent's reactive properties.
 *
 * Example usage:
 *   <cr-lazy-render-lit id="menu"
 *       .template="${() => html`<heavy-menu></heavy-menu>`}">
 *   </cr-lazy-render-lit>
 *
 * Note that the provided template should create exactly one top-level DOM node,
 * otherwise the result of this.get() will not be correct.
 *
 *   this.$.menu.get().show();
 */
class CrLazyRenderLitElement extends CrLitElement {
    static get is() {
        return 'cr-lazy-render-lit';
    }
    static get properties() {
        return {
            template: { type: Object },
            rendered_: {
                type: Boolean,
                state: true,
            },
        };
    }
    #rendered__accessor_storage = false;
    get rendered_() { return this.#rendered__accessor_storage; }
    set rendered_(value) { this.#rendered__accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    child_ = null;
    render() {
        if (this.rendered_) {
            // Render items into the parent's DOM using the client provided template.
            render(this.template(), this.parentNode, {
                host: this.getRootNode().host,
                // Specify 'renderBefore', so that the lazy rendered node can be
                // easily located in get() later on.
                renderBefore: this,
            });
        }
        return html ``;
    }
    /**
     * Stamp the template into the DOM tree synchronously
     * @return Child element which has been stamped into the DOM tree.
     */
    get() {
        if (!this.rendered_) {
            this.rendered_ = true;
            this.performUpdate();
            this.child_ = this.previousElementSibling;
        }
        assert(this.child_);
        return this.child_;
    }
    /**
     * @return The element contained in the template, if it has
     *   already been stamped.
     */
    getIfExists() {
        return this.child_;
    }
}
customElements.define(CrLazyRenderLitElement.is, CrLazyRenderLitElement);

let instance$7 = null;
function getCss$7() {
    return instance$7 || (instance$7 = [...[], css `:host{display:block;position:relative}:host([chunk-size="0"]) #container>::slotted(*){box-sizing:border-box;contain-intrinsic-size:var(--list-item-size,100px) auto;content-visibility:auto;width:100%}:host(:not([chunk-size="0"])) #container>::slotted(.chunk){box-sizing:border-box;contain-intrinsic-size:calc(var(--chunk-size) * var(--list-item-size,100px)) auto;content-visibility:auto;width:100%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-lazy-list' is a component optimized for showing a list of
 * items that overflows the view and requires scrolling. For performance
 * reasons, the DOM items are incrementally added to the view as the user
 * scrolls through the list. The component expects a `scrollTarget` property
 * to be specified indicating the scrolling container. This container is
 * used for observing scroll events and resizes. The container should have
 * bounded height so that cr-lazy-list can determine how many HTML elements to
 * render initially.
 * If using a container that can shrink arbitrarily small to the height of the
 * contents, a 'minViewportHeight' property should also be provided specifying
 * the minimum viewport height to try to fill with items.
 * Each list item's HTML element is created using the `template` property,
 * which should be set to a function returning a TemplateResult corresponding
 * to a passed in list item and selection index.
 * Set `listItemHost` to the `this` context for any event handlers in this
 * template. If this property is not provided, cr-lazy-list is assumed to be
 * residing in a ShadowRoot, and the shadowRoot's |host| is used.
 * The `items` property specifies an array of list item data.
 * The `itemSize` property should be set to an estimate of the list item size.
 * This is used when setting contain-intrinsic-size styling for list items.
 * To restore focus to a specific item if it is focused when the items
 * array changes, set `restoreFocusItem` to that HTMLElement. If the element
 * is focused when the items array is updated, focus will be restored.
 * To set content-visibility on chunks of elements rather than on individual
 * elements, use the `chunkSize` property and specify the number of elements
 * to group. This is useful when rendering large numbers of short items, as
 * the intersection observers added by content-visibility: auto can slow down
 * the UI for very large numbers of elements.
 */
class CrLazyListElement extends CrLitElement {
    static get is() {
        return 'cr-lazy-list';
    }
    static get styles() {
        return getCss$7();
    }
    render() {
        const host = this.listItemHost === undefined ?
            this.getRootNode().host :
            this.listItemHost;
        // Render items into light DOM using the client provided template
        if (this.chunkSize === 0) {
            render(this.items.slice(0, this.numItemsDisplayed_).map((item, index) => {
                return this.template(item, index);
            }), this, { host });
        }
        else {
            const chunks = Math.ceil(this.numItemsDisplayed_ / this.chunkSize);
            const chunkArray = new Array(chunks).fill(0);
            // Render chunk divs.
            render(chunkArray.map((_item, index) => html `<div id="chunk-${index}" class="chunk">
                                     </div>`), this, { host });
            // Render items into chunk divs.
            for (let chunkIndex = 0; chunkIndex < chunks; chunkIndex++) {
                const start = chunkIndex * this.chunkSize;
                const end = Math.min(this.numItemsDisplayed_, (chunkIndex + 1) * this.chunkSize);
                const chunk = this.querySelector(`#chunk-${chunkIndex}`);
                assert(chunk);
                render(this.items.slice(start, end).map((item, index) => {
                    return this.template(item, start + index);
                }), chunk, { host });
            }
        }
        // Render container + slot into shadow DOM
        return html `<div id="container"><slot id="slot"></slot></div>`;
    }
    static get properties() {
        return {
            chunkSize: {
                type: Number,
                reflect: true,
            },
            items: { type: Array },
            itemSize: { type: Number },
            listItemHost: { type: Object },
            minViewportHeight: { type: Number },
            scrollOffset: { type: Number },
            scrollTarget: { type: Object },
            restoreFocusElement: { type: Object },
            template: { type: Object },
            numItemsDisplayed_: {
                state: true,
                type: Number,
            },
        };
    }
    #items_accessor_storage = [];
    get items() { return this.#items_accessor_storage; }
    set items(value) { this.#items_accessor_storage = value; }
    #itemSize_accessor_storage = undefined;
    get itemSize() { return this.#itemSize_accessor_storage; }
    set itemSize(value) { this.#itemSize_accessor_storage = value; }
    #listItemHost_accessor_storage;
    get listItemHost() { return this.#listItemHost_accessor_storage; }
    set listItemHost(value) { this.#listItemHost_accessor_storage = value; }
    #minViewportHeight_accessor_storage;
    get minViewportHeight() { return this.#minViewportHeight_accessor_storage; }
    set minViewportHeight(value) { this.#minViewportHeight_accessor_storage = value; }
    #scrollOffset_accessor_storage = 0;
    get scrollOffset() { return this.#scrollOffset_accessor_storage; }
    set scrollOffset(value) { this.#scrollOffset_accessor_storage = value; }
    #scrollTarget_accessor_storage = document.documentElement;
    get scrollTarget() { return this.#scrollTarget_accessor_storage; }
    set scrollTarget(value) { this.#scrollTarget_accessor_storage = value; }
    #restoreFocusElement_accessor_storage = null;
    get restoreFocusElement() { return this.#restoreFocusElement_accessor_storage; }
    set restoreFocusElement(value) { this.#restoreFocusElement_accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    #chunkSize_accessor_storage = 0;
    get chunkSize() { return this.#chunkSize_accessor_storage; }
    set chunkSize(value) { this.#chunkSize_accessor_storage = value; }
    #numItemsDisplayed__accessor_storage = 0;
    get numItemsDisplayed_() { return this.#numItemsDisplayed__accessor_storage; }
    set numItemsDisplayed_(value) { this.#numItemsDisplayed__accessor_storage = value; }
    // Internal state
    lastItemsLength_ = 0;
    lastRenderedHeight_ = 0;
    resizeObserver_ = null;
    scrollListener_ = () => this.onScroll_();
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('items')) {
            this.lastItemsLength_ = this.items.length;
            this.numItemsDisplayed_ = this.items.length === 0 ?
                0 :
                Math.min(this.numItemsDisplayed_, this.items.length);
        }
        else {
            assert(this.items.length === this.lastItemsLength_, 'Items array changed in place; rendered result may be incorrect.');
        }
        if (changedProperties.has('itemSize')) {
            this.style.setProperty('--list-item-size', `${this.itemSize}px`);
        }
        if (changedProperties.has('chunkSize')) {
            this.style.setProperty('--chunk-size', `${this.chunkSize}`);
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        let itemsChanged = false;
        if (changedProperties.has('items') ||
            changedProperties.has('minViewportHeight') ||
            changedProperties.has('scrollOffset')) {
            const previous = changedProperties.get('items');
            if (previous !== undefined || this.items.length !== 0) {
                this.onItemsChanged_();
                itemsChanged = true;
            }
        }
        if (changedProperties.has('scrollTarget')) {
            this.addRemoveScrollTargetListeners_(changedProperties.get('scrollTarget') || null);
            // Only re-render if there are items to display and we are not already
            // re-rendering for the items.
            if (this.scrollTarget && this.items.length > 0 && !itemsChanged) {
                this.fillCurrentViewport();
            }
        }
    }
    // Public API
    // Forces the list to fill the current viewport. Called when the viewport
    // size or position changes.
    fillCurrentViewport() {
        if (this.items.length === 0) {
            return Promise.resolve();
        }
        // Update the height if the previous height calculation was done when this
        // element was not visible or if new DOM items were added.
        return this.update_(this.$.container.style.height === '0px');
    }
    // Forces the list to render |numItems| items. If |numItems| are already
    // rendered, this is a no-op.
    async ensureItemRendered(index) {
        if (index < this.numItemsDisplayed_) {
            return this.domItems()[index];
        }
        assert(index < this.items.length);
        await this.updateNumItemsDisplayed_(index + 1);
        return this.domItems()[index];
    }
    // Private methods
    addRemoveScrollTargetListeners_(oldTarget) {
        if (oldTarget) {
            const target = oldTarget === document.documentElement ? window : oldTarget;
            target.removeEventListener('scroll', this.scrollListener_);
            assert(this.resizeObserver_);
            this.resizeObserver_.disconnect();
        }
        if (this.scrollTarget) {
            const target = this.scrollTarget === document.documentElement ?
                window :
                this.scrollTarget;
            target.addEventListener('scroll', this.scrollListener_);
            this.resizeObserver_ = new ResizeObserver(() => {
                requestAnimationFrame(() => {
                    const newHeight = this.getViewHeight_();
                    if (newHeight > 0 && newHeight !== this.lastRenderedHeight_) {
                        this.fillCurrentViewport();
                    }
                });
            });
            this.resizeObserver_.observe(this.scrollTarget);
        }
    }
    shouldRestoreFocus_() {
        if (!this.restoreFocusElement) {
            return false;
        }
        const active = getDeepActiveElement();
        return this.restoreFocusElement === active ||
            (!!this.restoreFocusElement.shadowRoot &&
                this.restoreFocusElement.shadowRoot.activeElement === active);
    }
    async onItemsChanged_() {
        if (this.items.length > 0) {
            const restoreFocus = this.shouldRestoreFocus_();
            await this.update_(true);
            if (restoreFocus) {
                // Async to allow clients to update in response to viewport-filled.
                setTimeout(() => {
                    // The element may have been removed from the DOM by the client.
                    if (!this.restoreFocusElement) {
                        return;
                    }
                    this.restoreFocusElement.focus();
                    this.fire('focus-restored-for-test');
                }, 0);
            }
        }
        else {
            // Update the container height to 0 since there are no items.
            this.$.container.style.height = '0px';
            this.fire('items-rendered');
            this.fire('viewport-filled');
        }
    }
    getScrollTop_() {
        return this.scrollTarget === document.documentElement ?
            window.pageYOffset :
            this.scrollTarget.scrollTop;
    }
    getViewHeight_() {
        const offsetHeight = this.scrollTarget === document.documentElement ?
            window.innerHeight :
            this.scrollTarget.offsetHeight;
        return this.getScrollTop_() - this.scrollOffset +
            Math.max(this.minViewportHeight || 0, offsetHeight);
    }
    async update_(forceUpdateHeight) {
        if (!this.scrollTarget) {
            return;
        }
        const height = this.getViewHeight_();
        if (height <= 0) {
            return;
        }
        const added = await this.fillViewHeight_(height);
        this.fire('items-rendered');
        if (added || forceUpdateHeight) {
            await this.updateHeight_();
            this.fire('viewport-filled');
        }
    }
    /**
     * @return Whether DOM items were created or not.
     */
    async fillViewHeight_(height) {
        this.fire('fill-height-start');
        this.lastRenderedHeight_ = height;
        // Ensure we have added enough DOM items so that we are able to estimate
        // item average height.
        assert(this.items.length);
        const initialDomItemCount = this.domItems().length;
        if (initialDomItemCount === 0) {
            await this.updateNumItemsDisplayed_(1);
        }
        const itemHeight = this.domItemAverageHeight_();
        // If this happens, the math below will be incorrect and we will render
        // all items. So return early, and correct |lastRenderedHeight_|.
        if (itemHeight === 0) {
            this.lastRenderedHeight_ = 0;
            return false;
        }
        const desiredDomItemCount = Math.min(Math.ceil(height / itemHeight), this.items.length);
        if (desiredDomItemCount > this.numItemsDisplayed_) {
            await this.updateNumItemsDisplayed_(desiredDomItemCount);
        }
        const added = initialDomItemCount !== desiredDomItemCount;
        if (added) {
            this.fire('fill-height-end');
        }
        return added;
    }
    async updateNumItemsDisplayed_(itemsToDisplay) {
        this.numItemsDisplayed_ = itemsToDisplay;
        if (this.numItemsDisplayed_ > 200 && this.chunkSize < 2) {
            console.warn(`cr-lazy-list: ${this.numItemsDisplayed_} list items rendered. ` +
                'If this is expected, consider chunking mode (chunkSize > 1) ' +
                'to improve scrolling performance.');
        }
        await this.updateComplete;
    }
    /**
     * @return The currently rendered list items, particularly useful for clients
     *     using chunking mode.
     */
    domItems() {
        return this.chunkSize === 0 ?
            this.$.slot.assignedElements() :
            Array.from(this.querySelectorAll('.chunk > *'));
    }
    /**
     * @return The average DOM item height.
     */
    domItemAverageHeight_() {
        // This logic should only be invoked if the list is non-empty and at least
        // one DOM item has been rendered so that an item average height can be
        // estimated. This is ensured by the callers.
        assert(this.items.length > 0);
        const domItems = this.domItems();
        assert(domItems.length > 0);
        const firstDomItem = domItems.at(0);
        const lastDomItem = domItems.at(-1);
        const lastDomItemHeight = lastDomItem.offsetHeight;
        if (firstDomItem === lastDomItem && lastDomItemHeight === 0) {
            // If there is only 1 item and it has a height of 0, return early. This
            // likely means the UI is still hidden or there is no content.
            return 0;
        }
        else if (this.itemSize) {
            // Once items are actually visible and have a height > 0, assume that it
            // is an accurate representation of the average item size.
            return this.itemSize;
        }
        let totalHeight = lastDomItem.offsetTop + lastDomItemHeight;
        if (this.chunkSize > 0) {
            // Add the parent's offsetTop. The offsetParent will be the chunk div.
            // Subtract the offsetTop of the first chunk div to avoid counting any
            // padding.
            totalHeight += lastDomItem.offsetParent.offsetTop -
                firstDomItem.offsetParent.offsetTop;
        }
        else {
            // Subtract the offsetTop of the first item to avoid counting any padding.
            totalHeight -= firstDomItem.offsetTop;
        }
        return totalHeight / domItems.length;
    }
    /**
     * Sets the height of the component based on an estimated average DOM item
     * height and the total number of items.
     */
    async updateHeight_() {
        // Await 1 cycle to ensure any child Lit elements have time to finish
        // rendering, or the height estimated below will be incorrect.
        await new Promise(resolve => setTimeout(resolve, 0));
        const estScrollHeight = this.items.length > 0 ?
            this.items.length * this.domItemAverageHeight_() :
            0;
        this.$.container.style.height = estScrollHeight + 'px';
    }
    /**
     * Adds additional DOM items as needed to fill the view based on user scroll
     * interactions.
     */
    async onScroll_() {
        const scrollTop = this.getScrollTop_();
        if (scrollTop <= 0 || this.numItemsDisplayed_ === this.items.length) {
            return;
        }
        await this.fillCurrentViewport();
    }
}
customElements.define(CrLazyListElement.is, CrLazyListElement);

let instance$6 = null;
function getCss$6() {
    return instance$6 || (instance$6 = [...[], css `:host{display:block;position:relative}:host([using-default-scroll-target]){overflow-y:auto}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-infinite-list' is a thin wrapper around 'cr-lazy-list' that
 * emulates some of the behavior of 'iron-list'.
 */
class CrInfiniteListElement extends CrLitElement {
    static get is() {
        return 'cr-infinite-list';
    }
    static get styles() {
        return getCss$6();
    }
    render() {
        // Render items into light DOM using the client provided template
        render(html `<cr-lazy-list id="list" .scrollTarget="${this.scrollTarget}"
          .chunkSize="${this.chunkSize}"
          .scrollOffset="${this.scrollOffset}"
          .listItemHost="${this.getRootNode().host}"
          .items="${this.items}" .itemSize="${this.itemSize}"
          .template="${(item, index) => this.template(item, index, index === this.focusedIndex ? 0 : -1)}"
          .restoreFocusElement="${this.focusedItem_}"
          @keydown="${this.onKeyDown_}"
          @focusin="${this.onItemFocus_}"
          @viewport-filled="${this.updateFocusedItem_}">
        </cr-lazy-list>`, this, {
            host: this,
        });
        return html `<slot></slot>`;
    }
    static get properties() {
        return {
            chunkSize: { type: Number },
            scrollOffset: { type: Number },
            scrollTarget: { type: Object },
            usingDefaultScrollTarget: {
                type: Boolean,
                reflect: true,
            },
            items: { type: Array },
            focusedIndex: { type: Number },
            itemSize: { type: Number },
            template: { type: Object },
            focusedItem_: { type: Object },
        };
    }
    #chunkSize_accessor_storage = 0;
    get chunkSize() { return this.#chunkSize_accessor_storage; }
    set chunkSize(value) { this.#chunkSize_accessor_storage = value; }
    #scrollOffset_accessor_storage = 0;
    get scrollOffset() { return this.#scrollOffset_accessor_storage; }
    set scrollOffset(value) { this.#scrollOffset_accessor_storage = value; }
    #scrollTarget_accessor_storage = this;
    get scrollTarget() { return this.#scrollTarget_accessor_storage; }
    set scrollTarget(value) { this.#scrollTarget_accessor_storage = value; }
    #usingDefaultScrollTarget_accessor_storage = true;
    get usingDefaultScrollTarget() { return this.#usingDefaultScrollTarget_accessor_storage; }
    set usingDefaultScrollTarget(value) { this.#usingDefaultScrollTarget_accessor_storage = value; }
    #items_accessor_storage = [];
    get items() { return this.#items_accessor_storage; }
    set items(value) { this.#items_accessor_storage = value; }
    #itemSize_accessor_storage = undefined;
    get itemSize() { return this.#itemSize_accessor_storage; }
    set itemSize(value) { this.#itemSize_accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    // Unlike cr-lazy-list, cr-infinite-list provides a tabindex parameter for
    // clients as is provided by iron-list. Like iron-list, cr-infinite-list will
    // pass 0 for this parameter if the list item should be keyboard focusable,
    // and -1 otherwise.
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    #focusedIndex_accessor_storage = -1;
    get focusedIndex() { return this.#focusedIndex_accessor_storage; }
    set focusedIndex(value) { this.#focusedIndex_accessor_storage = value; }
    #focusedItem__accessor_storage = null;
    get focusedItem_() { return this.#focusedItem__accessor_storage; }
    set focusedItem_(value) { this.#focusedItem__accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('scrollTarget')) {
            this.usingDefaultScrollTarget = this.scrollTarget === this;
        }
        if (changedProperties.has('items')) {
            if (this.focusedIndex >= this.items.length) {
                this.focusedIndex = this.items.length - 1;
            }
            else if (this.focusedIndex === -1 && this.items.length > 0) {
                this.focusedIndex = 0;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('focusedIndex')) {
            this.updateFocusedItem_();
        }
    }
    fillCurrentViewport() {
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        return list.fillCurrentViewport();
    }
    ensureItemRendered(index) {
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        return list.ensureItemRendered(index);
    }
    updateFocusedItem_() {
        if (this.focusedIndex === -1) {
            this.focusedItem_ = null;
            return;
        }
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        this.focusedItem_ =
            list.domItems()[this.focusedIndex + 1] ||
                null;
    }
    onItemFocus_(e) {
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        const renderedItems = list.domItems();
        const focusedIdx = Array.from(renderedItems).findIndex(item => {
            return item === e.target || item.shadowRoot?.activeElement === e.target;
        });
        if (focusedIdx !== -1) {
            this.focusedIndex = focusedIdx;
        }
    }
    /**
     * Handles key events when list item elements have focus.
     */
    async onKeyDown_(e) {
        // Do not interfere with any parent component that manages 'shift' related
        // key events.
        if (e.shiftKey || (e.key !== 'ArrowUp' && e.key !== 'ArrowDown')) {
            return;
        }
        e.stopPropagation();
        e.preventDefault();
        // Identify the new focused index.
        this.focusedIndex = e.key === 'ArrowUp' ?
            Math.max(0, this.focusedIndex - 1) :
            Math.min(this.items.length - 1, this.focusedIndex + 1);
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        const element = await list.ensureItemRendered(this.focusedIndex);
        element.focus();
        element.scrollIntoViewIfNeeded();
    }
}
customElements.define(CrInfiniteListElement.is, CrInfiniteListElement);

let instance$5 = null;
function getCss$5() {
    return instance$5 || (instance$5 = [...[getCss$K()], css `:host{align-items:center;cursor:default;display:flex;font-size:calc(12/13 * 1em);min-height:var(--destination-item-height);opacity:.87;padding-inline-end:2px;padding-inline-start:0;vertical-align:middle}:host>*{align-items:center;color:var(--cr-secondary-text-color);font-size:calc(10/12 * 1em);overflow:hidden;text-overflow:ellipsis;vertical-align:middle;white-space:nowrap}:host>span{margin-inline-start:1em}cr-icon{--icon-margin:calc((var(--search-icon-size) - var(--iron-icon-width))/2);fill:var(--google-grey-600);flex:0;height:var(--iron-icon-height);margin-inline-end:var(--icon-margin);margin-inline-start:var(--icon-margin);min-width:var(--iron-icon-width);transition:opacity 150ms}@media (prefers-color-scheme:dark){cr-icon{fill:var(--google-grey-500)}}:host .name{color:var(--cr-primary-text-color);font-size:1em;margin-inline-start:0;padding-inline-start:8px}.extension-controlled-indicator{display:flex;flex:1;justify-content:flex-end;min-width:150px;padding-inline-end:8px}.extension-icon{height:24px;margin-inline-start:1em;width:24px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$6() {
    // clang-format off
    return html `<!--_html_template_start_-->
${this.destination !== null ? html `
  <cr-icon icon="${this.destination.icon}"></cr-icon>
  <span class="name searchable">${this.destination.displayName}</span>
  <span class="search-hint searchable" ?hidden="${!this.searchHint_}">
    ${this.searchHint_}
  </span>
  <span class="extension-controlled-indicator"
      ?hidden="${!this.destination.isExtension}">
    <span class="extension-name searchable">
      ${this.destination.extensionName}
    </span>
    <span class="extension-icon" role="button" tabindex="0"
        title="${this.getExtensionPrinterTooltip_()}"></span>
  </span>
` : ''}
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PrintPreviewDestinationListItemElement extends CrLitElement {
    static get is() {
        return 'print-preview-destination-list-item';
    }
    static get styles() {
        return getCss$5();
    }
    render() {
        return getHtml$6.bind(this)();
    }
    static get properties() {
        return {
            destination: { type: Object },
            searchQuery: { type: Object },
            searchHint_: { type: String },
        };
    }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #searchQuery_accessor_storage = null;
    get searchQuery() { return this.#searchQuery_accessor_storage; }
    set searchQuery(value) { this.#searchQuery_accessor_storage = value; }
    #searchHint__accessor_storage = '';
    get searchHint_() { return this.#searchHint__accessor_storage; }
    set searchHint_(value) { this.#searchHint__accessor_storage = value; }
    highlights_ = [];
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('destination') ||
            changedProperties.has('searchQuery')) {
            this.updateSearchHint_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('destination')) {
            this.onDestinationPropertiesChange_();
        }
        if (changedProperties.has('destination') ||
            changedProperties.has('searchQuery')) {
            removeHighlights(this.highlights_);
            this.highlights_ = updateHighlights(this, this.searchQuery, new Map());
        }
    }
    onDestinationPropertiesChange_() {
        if (this.destination === null) {
            return;
        }
        this.title = this.destination.displayName;
        if (this.destination.isExtension) {
            const icon = this.shadowRoot.querySelector('.extension-icon');
            assert(icon);
            icon.style.backgroundImage = 'image-set(' +
                'url(chrome://extension-icon/' + this.destination.extensionId +
                '/24/1) 1x,' +
                'url(chrome://extension-icon/' + this.destination.extensionId +
                '/48/1) 2x)';
        }
    }
    updateSearchHint_() {
        if (this.destination === null) {
            return;
        }
        const matches = !this.searchQuery ?
            [] :
            this.destination.extraPropertiesToMatch.filter(p => p.match(this.searchQuery));
        this.searchHint_ = matches.length === 0 ?
            (this.destination.extraPropertiesToMatch.find(p => !!p) || '') :
            matches.join(' ');
    }
    getExtensionPrinterTooltip_() {
        assert(this.destination);
        if (!this.destination.isExtension) {
            return '';
        }
        return loadTimeData.getStringF('extensionDestinationIconTooltip', this.destination.extensionName);
    }
}
customElements.define(PrintPreviewDestinationListItemElement.is, PrintPreviewDestinationListItemElement);

let instance$4 = null;
function getCss$4() {
    return instance$4 || (instance$4 = [...[getCss$K(), getCss$9()], css `:host{display:flex;flex-direction:column;height:100%;overflow:hidden;user-select:none}#list{min-height:var(--destination-item-height)}.throbber-container{display:flex;margin-inline-start:calc((var(--search-icon-size) - var(--throbber-size))/2);min-height:var(--destination-item-height)}.throbber{align-self:center}.no-destinations-message{padding-bottom:8px;padding-inline-start:18px;padding-top:8px}.list-item{transition:background-color 150ms}.list-item:hover,.list-item:focus{background-color:rgb(228,236,247)}@media (prefers-color-scheme:dark){.list-item:-webkit-any(:hover,:focus){background-color:var(--cr-menu-background-focus-color)}}.list-item:focus{outline:none}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$5() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div class="no-destinations-message" ?hidden="${this.hasDestinations_}">
  $i18n{noDestinationsMessage}
</div>

<cr-infinite-list id="list" .items="${this.matchingDestinations_}" role="grid"
    aria-rowcount="${this.matchingDestinations_.length}"
    aria-label="$i18n{printDestinationsTitle}" ?hidden="${this.hideList_}"
    item-size="32" chunk-size="30"
    .template="${(item, index, tabIndex) => html `
      <div role="row" id="destination_${index}"
          aria-rowindex="${this.getAriaRowindex_(index)}" tabindex="${tabIndex}"
          @focus="${this.onDestinationRowFocus_}">
        <print-preview-destination-list-item class="list-item"
            .searchQuery="${this.searchQuery}" .destination="${item}"
            @click="${this.onDestinationSelected_}"
            @keydown="${this.onKeydown_}" tabindex="-1"
            role="gridcell">
        </print-preview-destination-list-item>
      </div>
    `}"></cr-infinite-list>
<div class="throbber-container" ?hidden="${this.throbberHidden_}">
  <div class="throbber"></div>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const DESTINATION_ITEM_HEIGHT = 32;
class PrintPreviewDestinationListElement extends CrLitElement {
    static get is() {
        return 'print-preview-destination-list';
    }
    static get styles() {
        return getCss$4();
    }
    render() {
        return getHtml$5.bind(this)();
    }
    static get properties() {
        return {
            destinations: { type: Array },
            searchQuery: { type: Object },
            loadingDestinations: { type: Boolean },
            matchingDestinations_: { type: Array },
            hasDestinations_: { type: Boolean },
            throbberHidden_: { type: Boolean },
            hideList_: { type: Boolean },
        };
    }
    #destinations_accessor_storage = [];
    get destinations() { return this.#destinations_accessor_storage; }
    set destinations(value) { this.#destinations_accessor_storage = value; }
    #searchQuery_accessor_storage = null;
    get searchQuery() { return this.#searchQuery_accessor_storage; }
    set searchQuery(value) { this.#searchQuery_accessor_storage = value; }
    #loadingDestinations_accessor_storage = false;
    get loadingDestinations() { return this.#loadingDestinations_accessor_storage; }
    set loadingDestinations(value) { this.#loadingDestinations_accessor_storage = value; }
    #matchingDestinations__accessor_storage = [];
    get matchingDestinations_() { return this.#matchingDestinations__accessor_storage; }
    set matchingDestinations_(value) { this.#matchingDestinations__accessor_storage = value; }
    #hasDestinations__accessor_storage = true;
    get hasDestinations_() { return this.#hasDestinations__accessor_storage; }
    set hasDestinations_(value) { this.#hasDestinations__accessor_storage = value; }
    #throbberHidden__accessor_storage = false;
    get throbberHidden_() { return this.#throbberHidden__accessor_storage; }
    set throbberHidden_(value) { this.#throbberHidden__accessor_storage = value; }
    #hideList__accessor_storage = false;
    get hideList_() { return this.#hideList__accessor_storage; }
    set hideList_(value) { this.#hideList__accessor_storage = value; }
    boundUpdateHeight_ = null;
    connectedCallback() {
        super.connectedCallback();
        this.boundUpdateHeight_ = () => this.updateHeight_();
        window.addEventListener('resize', this.boundUpdateHeight_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        window.removeEventListener('resize', this.boundUpdateHeight_);
        this.boundUpdateHeight_ = null;
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('destinations') ||
            changedProperties.has('searchQuery') ||
            changedProperties.has('loadingDestinations')) {
            this.updateMatchingDestinations_();
        }
    }
    updateHeight_(numDestinations) {
        const count = numDestinations === undefined ?
            this.matchingDestinations_.length :
            numDestinations;
        const maxDisplayedItems = this.offsetHeight / DESTINATION_ITEM_HEIGHT;
        const isListFullHeight = maxDisplayedItems <= count;
        // Update the throbber and "No destinations" message.
        this.hasDestinations_ = count > 0 || this.loadingDestinations;
        this.throbberHidden_ =
            !this.loadingDestinations || isListFullHeight || !this.hasDestinations_;
        this.hideList_ = count === 0;
        if (this.hideList_) {
            return;
        }
        const listHeight = isListFullHeight ? this.offsetHeight : count * DESTINATION_ITEM_HEIGHT;
        this.$.list.style.height = listHeight > DESTINATION_ITEM_HEIGHT ?
            `${listHeight}px` :
            `${DESTINATION_ITEM_HEIGHT}px`;
    }
    updateMatchingDestinations_() {
        if (this.destinations === undefined) {
            return;
        }
        const matchingDestinations = this.searchQuery ?
            this.destinations.filter(d => d.matches(this.searchQuery)) :
            this.destinations.slice();
        // Update the height before updating the list.
        this.updateHeight_(matchingDestinations.length);
        this.matchingDestinations_ = matchingDestinations;
    }
    onKeydown_(e) {
        if (e.key === 'Enter') {
            this.onDestinationSelected_(e);
            e.stopPropagation();
        }
    }
    /**
     * @param e Event containing the destination that was selected.
     */
    onDestinationSelected_(e) {
        if (e.composedPath()[0].tagName === 'A') {
            return;
        }
        const listItem = e.target;
        assert(listItem.destination);
        this.dispatchEvent(new CustomEvent('destination-selected', { bubbles: true, composed: true, detail: listItem.destination }));
    }
    /**
     * Returns a 1-based index for aria-rowindex.
     */
    getAriaRowindex_(index) {
        return index + 1;
    }
    onDestinationRowFocus_(e) {
        // Forward focus to the 'print-preview-destination-list-item'.
        const item = e.target.querySelector('.list-item');
        assert(!!item);
        item.focus();
    }
}
customElements.define(PrintPreviewDestinationListElement.is, PrintPreviewDestinationListElement);

let instance$3 = null;
function getCss$3() {
    return instance$3 || (instance$3 = [...[getCss$x(), getCss$K(), getCss$9()], css `:host-context([dir=rtl]) #manageIcon{transform:scaleX(-1)}#dialog::part(dialog){height:calc(100vh - 2 * var(--print-preview-dialog-margin));max-width:640px;width:calc(100vw - 2 * var(--print-preview-dialog-margin))}#dialog::part(wrapper){height:calc(100vh - 2 * var(--print-preview-dialog-margin))}#dialog::part(body-container){flex:1}print-preview-search-box{margin-bottom:16px;margin-top:6px}cr-dialog [slot=body]{display:flex;flex-direction:column;height:100%}div[slot='button-container']{justify-content:space-between}cr-button{font-size:calc(12 / 13 * 1em)}.cancel-button{margin-inline-end:0}cr-button cr-icon{--iron-icon-fill-color:currentColor;margin-inline-start:8px}#warning-message{color:var(--cr-primary-text-color)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$4() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-dialog id="dialog" @close="${this.onCloseOrCancel_}">
  <div slot="title" id="header">$i18n{destinationSearchTitle}</div>
  <div slot="body">
    <print-preview-search-box id="searchBox"
        label="$i18n{searchBoxPlaceholder}" .searchQuery="${this.searchQuery_}"
        @search-query-changed="${this.onSearchQueryChanged_}"
        autofocus>
    </print-preview-search-box>
    <print-preview-destination-list id="printList"
        ?loading-destinations="${this.loadingDestinations_}"
        .searchQuery="${this.searchQuery_}"
        @destination-selected="${this.onDestinationSelected_}">
    </print-preview-destination-list>
  </div>
  <div slot="button-container">
    <cr-button @click="${this.onManageButtonClick_}">
      $i18n{manage}
      <cr-icon icon="cr:open-in-new" id="manageIcon"></cr-icon>
    </cr-button>
    <cr-button class="cancel-button" @click="${this.onCancelButtonClick_}">
      $i18n{cancel}
    </cr-button>
  </div>
</cr-dialog>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PrintPreviewDestinationDialogElement extends CrLitElement {
    static get is() {
        return 'print-preview-destination-dialog';
    }
    static get styles() {
        return getCss$3();
    }
    render() {
        return getHtml$4.bind(this)();
    }
    static get properties() {
        return {
            destinationStore: { type: Object },
            loadingDestinations_: { type: Boolean },
            searchQuery_: { type: Object },
        };
    }
    #destinationStore_accessor_storage = null;
    get destinationStore() { return this.#destinationStore_accessor_storage; }
    set destinationStore(value) { this.#destinationStore_accessor_storage = value; }
    #loadingDestinations__accessor_storage = false;
    get loadingDestinations_() { return this.#loadingDestinations__accessor_storage; }
    set loadingDestinations_(value) { this.#loadingDestinations__accessor_storage = value; }
    #searchQuery__accessor_storage = null;
    get searchQuery_() { return this.#searchQuery__accessor_storage; }
    set searchQuery_(value) { this.#searchQuery__accessor_storage = value; }
    tracker_ = new EventTracker();
    initialized_ = false;
    firstUpdated() {
        this.addEventListener('keydown', (e) => this.onKeydown_(e));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('destinationStore')) {
            this.onDestinationStoreSet_();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.tracker_.removeAll();
    }
    onKeydown_(e) {
        e.stopPropagation();
        const searchInput = this.$.searchBox.getSearchInput();
        if (e.key === 'Escape' &&
            (e.composedPath()[0] !== searchInput || !searchInput.value.trim())) {
            this.$.dialog.cancel();
            e.preventDefault();
        }
    }
    onDestinationStoreSet_() {
        assert(!this.initialized_);
        assert(this.destinationStore);
        this.tracker_.add(this.destinationStore, DestinationStoreEventType.DESTINATIONS_INSERTED, this.updateDestinations_.bind(this));
        this.tracker_.add(this.destinationStore, DestinationStoreEventType.DESTINATION_SEARCH_DONE, this.updateDestinations_.bind(this));
        this.initialized_ = true;
    }
    updateDestinations_() {
        if (!this.destinationStore || !this.initialized_) {
            return;
        }
        this.$.printList.destinations = this.destinationStore.destinations();
        this.loadingDestinations_ =
            this.destinationStore.isPrintDestinationSearchInProgress;
    }
    onCloseOrCancel_() {
        if (this.searchQuery_) {
            this.$.searchBox.setValue('');
        }
    }
    onCancelButtonClick_() {
        this.$.dialog.cancel();
    }
    /**
     * @param e Event containing the selected destination.
     */
    onDestinationSelected_(e) {
        this.selectDestination_(e.detail);
    }
    selectDestination_(destination) {
        assert(this.destinationStore);
        this.destinationStore.selectDestination(destination);
        this.$.dialog.close();
    }
    show() {
        this.$.dialog.showModal();
        const loading = !this.destinationStore ||
            this.destinationStore.isPrintDestinationSearchInProgress;
        if (!loading) {
            // All destinations have already loaded.
            this.updateDestinations_();
        }
        this.loadingDestinations_ = loading;
    }
    /** @return Whether the dialog is open. */
    isOpen() {
        return this.$.dialog.hasAttribute('open');
    }
    onManageButtonClick_() {
        NativeLayerImpl.getInstance().managePrinters();
    }
    onSearchQueryChanged_(e) {
        this.searchQuery_ = e.detail.value;
    }
}
customElements.define(PrintPreviewDestinationDialogElement.is, PrintPreviewDestinationDialogElement);

let instance$2 = null;
function getCss$2() {
    return instance$2 || (instance$2 = [...[getCss$E(), getCss$x(), getCss$9(), getCss$z(), getCss$K()], css `:host{--printer-icon-side-padding:4px;--printer-icon-size:20px}select.md-select{background-position:var(--printer-icon-side-padding) center,calc(100% - var(--md-select-side-padding)) center;background-size:var(--printer-icon-size),var(--md-arrow-width);padding-inline-start:32px}:host-context([dir=rtl]) .md-select{background-position-x:calc(100% - var(--printer-icon-side-padding)),var(--md-select-side-padding)}.throbber-container{align-items:center;display:flex;overflow:hidden}.destination-additional-info,.destination-additional-info div{height:100%;min-height:0}.destination-status{color:var(--cr-secondary-text-color);font-size:calc(12/13 * 1em);padding:4px 0}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$3() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-settings-section>
  <span id="destination-label" slot="title">$i18n{destinationLabel}</span>
  <div slot="controls">
    <div class="throbber-container" ?hidden="${this.loaded}">
      <div class="throbber"></div>
    </div>
    <select class="md-select" aria-labelledby="destination-label"
        ?hidden="${!this.loaded}" ?disabled="${this.disabled}"
        .style="background-image:${this.getBackgroundImages_()};"
        @change="${this.onSelectChange}">
      ${this.recentDestinationList.map(item => html `
        <option value="${item.key}" ?selected="${this.isSelected_(item.key)}">
          ${item.displayName}
        </option>
      `)}
      <option value="${this.pdfDestinationKey_}"
          ?hidden="${this.pdfPrinterDisabled}"
          ?selected="${this.isSelected_(this.pdfDestinationKey_)}">
        $i18n{printToPDF}
      </option>
      <option value="noDestinations" ?hidden="${!this.noDestinations}"
          ?selected="${this.noDestinations}">
        $i18n{noDestinationsMessage}
      </option>
      <option value="seeMore" aria-label="$i18n{seeMoreDestinationsLabel}">
        $i18n{seeMore}
      </option>
    </select>
  </div>
</print-preview-settings-section>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewDestinationSelectElementBase = SelectMixin(CrLitElement);
class PrintPreviewDestinationSelectElement extends PrintPreviewDestinationSelectElementBase {
    static get is() {
        return 'print-preview-destination-select';
    }
    static get styles() {
        return getCss$2();
    }
    render() {
        return getHtml$3.bind(this)();
    }
    static get properties() {
        return {
            dark: { type: Boolean },
            destination: { type: Object },
            disabled: { type: Boolean },
            loaded: { type: Boolean },
            noDestinations: { type: Boolean },
            pdfPrinterDisabled: { type: Boolean },
            recentDestinationList: { type: Array },
            pdfDestinationKey_: { type: String },
        };
    }
    #dark_accessor_storage = false;
    get dark() { return this.#dark_accessor_storage; }
    set dark(value) { this.#dark_accessor_storage = value; }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #loaded_accessor_storage = false;
    get loaded() { return this.#loaded_accessor_storage; }
    set loaded(value) { this.#loaded_accessor_storage = value; }
    #noDestinations_accessor_storage = false;
    get noDestinations() { return this.#noDestinations_accessor_storage; }
    set noDestinations(value) { this.#noDestinations_accessor_storage = value; }
    #pdfPrinterDisabled_accessor_storage = false;
    get pdfPrinterDisabled() { return this.#pdfPrinterDisabled_accessor_storage; }
    set pdfPrinterDisabled(value) { this.#pdfPrinterDisabled_accessor_storage = value; }
    #recentDestinationList_accessor_storage = [];
    get recentDestinationList() { return this.#recentDestinationList_accessor_storage; }
    set recentDestinationList(value) { this.#recentDestinationList_accessor_storage = value; }
    #pdfDestinationKey__accessor_storage = PDF_DESTINATION_KEY;
    get pdfDestinationKey_() { return this.#pdfDestinationKey__accessor_storage; }
    set pdfDestinationKey_(value) { this.#pdfDestinationKey__accessor_storage = value; }
    focus() {
        this.shadowRoot.querySelector('.md-select').focus();
    }
    /** Sets the select to the current value of |destination|. */
    updateDestination() {
        this.selectedValue = this.destination?.key || '';
    }
    /**
     * Returns the iconset and icon for the selected printer. If printer details
     * have not yet been retrieved from the backend, attempts to return an
     * appropriate icon early based on the printer's sticky information.
     * @return The iconset and icon for the current selection.
     */
    getDestinationIcon_() {
        if (!this.selectedValue) {
            return '';
        }
        // If the destination matches the selected value, pull the icon from the
        // destination.
        if (this.destination && this.destination.key === this.selectedValue) {
            return this.destination.icon;
        }
        // Check for the Save as PDF id first.
        if (this.selectedValue === PDF_DESTINATION_KEY) {
            return 'cr:insert-drive-file';
        }
        // Otherwise, must be in the recent list.
        const recent = this.recentDestinationList.find(d => {
            return d.key === this.selectedValue;
        });
        if (recent && recent.icon) {
            return recent.icon;
        }
        // The key/recent destinations don't have information about what icon to
        // use, so just return the generic print icon for now. It will be updated
        // when the destination is set.
        return 'print-preview:print';
    }
    /**
     * @return An inline svg corresponding to the icon for the current
     *     destination and the image for the dropdown arrow.
     */
    getBackgroundImages_() {
        const icon = this.getDestinationIcon_();
        if (!icon) {
            return '';
        }
        let iconSetAndIcon = null;
        if (this.noDestinations) {
            iconSetAndIcon = ['cr', 'error'];
        }
        iconSetAndIcon = iconSetAndIcon || icon.split(':');
        const iconset = IconsetMap.getInstance().get(iconSetAndIcon[0]);
        assert(iconset);
        return getSelectDropdownBackground(iconset, iconSetAndIcon[1], this);
    }
    onProcessSelectChange(value) {
        this.fire('selected-option-change', value);
    }
    /**
     * Return the options currently visible to the user for testing purposes.
     */
    getVisibleItemsForTest() {
        return this.shadowRoot.querySelectorAll('option:not([hidden])');
    }
    isSelected_(destinationKey) {
        return this.selectedValue === destinationKey;
    }
}
customElements.define(PrintPreviewDestinationSelectElement.is, PrintPreviewDestinationSelectElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$2() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-destination-select id="destinationSelect"
    ?dark="${this.dark}"
    .destination="${this.destination}"
    ?disabled="${this.shouldDisableDropdown_()}"
    ?loaded="${this.loaded_}"
    ?no-destinations="${this.noDestinations_}"
    ?pdf-printer-disabled="${this.pdfPrinterDisabled_}"
    .recentDestinationList="${this.displayedDestinations_}"
    @selected-option-change="${this.onSelectedDestinationOptionChange_}">
</print-preview-destination-select>
<cr-lazy-render-lit id="destinationDialog" .template="${() => html `
  <print-preview-destination-dialog
      .destinationStore="${this.destinationStore_}"
      @close="${this.onDialogClose_}">
  </print-preview-destination-dialog>
`}">
</cr-lazy-render-lit>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var DestinationState;
(function (DestinationState) {
    DestinationState[DestinationState["INIT"] = 0] = "INIT";
    DestinationState[DestinationState["SET"] = 1] = "SET";
    DestinationState[DestinationState["UPDATED"] = 2] = "UPDATED";
    DestinationState[DestinationState["ERROR"] = 3] = "ERROR";
})(DestinationState || (DestinationState = {}));
/** Number of recent destinations to save. */
const NUM_PERSISTED_DESTINATIONS = 5;
/**
 * Number of unpinned recent destinations to display.
 * Pinned destinations include "Save as PDF" and "Save to Google Drive".
 */
const NUM_UNPINNED_DESTINATIONS = 3;
const PrintPreviewDestinationSettingsElementBase = I18nMixinLit(WebUiListenerMixinLit(SettingsMixin(CrLitElement)));
class PrintPreviewDestinationSettingsElement extends PrintPreviewDestinationSettingsElementBase {
    static get is() {
        return 'print-preview-destination-settings';
    }
    static get styles() {
        return [
            getCss$x(),
        ];
    }
    render() {
        return getHtml$2.bind(this)();
    }
    static get properties() {
        return {
            dark: { type: Boolean },
            destination: {
                type: Object,
                notify: true,
            },
            destinationState: {
                type: Number,
                notify: true,
            },
            disabled: { type: Boolean },
            error: {
                type: Number,
                notify: true,
            },
            firstLoad: { type: Boolean },
            state: { type: Number },
            destinationStore_: { type: Object },
            displayedDestinations_: { type: Array },
            isDialogOpen_: { type: Boolean },
            noDestinations_: { type: Boolean },
            pdfPrinterDisabled_: { type: Boolean },
            loaded_: { type: Boolean },
        };
    }
    #dark_accessor_storage = false;
    get dark() { return this.#dark_accessor_storage; }
    set dark(value) { this.#dark_accessor_storage = value; }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #destinationState_accessor_storage = DestinationState.INIT;
    get destinationState() { return this.#destinationState_accessor_storage; }
    set destinationState(value) { this.#destinationState_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #error_accessor_storage = null;
    get error() { return this.#error_accessor_storage; }
    set error(value) { this.#error_accessor_storage = value; }
    #firstLoad_accessor_storage = false;
    get firstLoad() { return this.#firstLoad_accessor_storage; }
    set firstLoad(value) { this.#firstLoad_accessor_storage = value; }
    #state_accessor_storage = State.NOT_READY;
    get state() { return this.#state_accessor_storage; }
    set state(value) { this.#state_accessor_storage = value; }
    #destinationStore__accessor_storage = null;
    get destinationStore_() { return this.#destinationStore__accessor_storage; }
    set destinationStore_(value) { this.#destinationStore__accessor_storage = value; }
    #displayedDestinations__accessor_storage = [];
    get displayedDestinations_() { return this.#displayedDestinations__accessor_storage; }
    set displayedDestinations_(value) { this.#displayedDestinations__accessor_storage = value; }
    #isDialogOpen__accessor_storage = false;
    get isDialogOpen_() { return this.#isDialogOpen__accessor_storage; }
    set isDialogOpen_(value) { this.#isDialogOpen__accessor_storage = value; }
    #noDestinations__accessor_storage = false;
    get noDestinations_() { return this.#noDestinations__accessor_storage; }
    set noDestinations_(value) { this.#noDestinations__accessor_storage = value; }
    #pdfPrinterDisabled__accessor_storage = false;
    get pdfPrinterDisabled_() { return this.#pdfPrinterDisabled__accessor_storage; }
    set pdfPrinterDisabled_(value) { this.#pdfPrinterDisabled__accessor_storage = value; }
    #loaded__accessor_storage = false;
    get loaded_() { return this.#loaded__accessor_storage; }
    set loaded_(value) { this.#loaded__accessor_storage = value; }
    tracker_ = new EventTracker();
    connectedCallback() {
        super.connectedCallback();
        this.destinationStore_ =
            new DestinationStore(this.addWebUiListener.bind(this));
        this.tracker_.add(this.destinationStore_, DestinationStoreEventType.DESTINATION_SELECT, this.onDestinationSelect_.bind(this));
        this.tracker_.add(this.destinationStore_, DestinationStoreEventType.SELECTED_DESTINATION_CAPABILITIES_READY, this.onDestinationCapabilitiesReady_.bind(this));
        this.tracker_.add(this.destinationStore_, DestinationStoreEventType.ERROR, this.onDestinationError_.bind(this));
        // Need to update the recent list when the destination store inserts
        // destinations, in case any recent destinations have been added to the
        // store. At startup, recent destinations can be in the sticky settings,
        // but they should not be displayed in the dropdown until they have been
        // fetched by the DestinationStore, to ensure that they still exist.
        this.tracker_.add(this.destinationStore_, DestinationStoreEventType.DESTINATIONS_INSERTED, this.updateDropdownDestinations_.bind(this));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.destinationStore_.resetTracker();
        this.tracker_.removeAll();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('error')) {
            this.onErrorChanged_();
        }
        if (changedProperties.has('destinationState') ||
            changedProperties.has('destination')) {
            this.loaded_ = this.computeLoaded_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('destinationState')) {
            this.updateDestinationSelect_();
        }
    }
    /**
     * @param defaultPrinter The system default printer ID.
     * @param pdfPrinterDisabled Whether the PDF printer is disabled.
     * @param serializedDefaultDestinationRulesStr String with rules
     *     for selecting a default destination.
     */
    init(defaultPrinter, pdfPrinterDisabled, serializedDefaultDestinationRulesStr) {
        this.pdfPrinterDisabled_ = pdfPrinterDisabled;
        let recentDestinations = this.getSettingValue('recentDestinations');
        recentDestinations = recentDestinations.slice(0, this.getRecentDestinationsDisplayCount_(recentDestinations));
        this.destinationStore_.init(this.pdfPrinterDisabled_, defaultPrinter, serializedDefaultDestinationRulesStr, recentDestinations);
    }
    /**
     * @param recentDestinations recent destinations.
     * @return Number of recent destinations to display.
     */
    getRecentDestinationsDisplayCount_(recentDestinations) {
        let numDestinationsToDisplay = NUM_UNPINNED_DESTINATIONS;
        for (let i = 0; i < recentDestinations.length; i++) {
            // Once all NUM_UNPINNED_DESTINATIONS unpinned destinations have been
            // found plus an extra unpinned destination, return the total number of
            // destinations found excluding the last extra unpinned destination.
            //
            // The extra unpinned destination ensures that pinned destinations
            // located directly after the last unpinned destination are included
            // in the display count.
            if (i > numDestinationsToDisplay) {
                return numDestinationsToDisplay;
            }
            // If a destination is pinned, increment numDestinationsToDisplay.
            if (isPdfPrinter(recentDestinations[i].id)) {
                numDestinationsToDisplay++;
            }
        }
        return Math.min(recentDestinations.length, numDestinationsToDisplay);
    }
    onDestinationSelect_() {
        if (this.state === State.FATAL_ERROR) {
            // Don't let anything reset if there is a fatal error.
            return;
        }
        const destination = this.destinationStore_.selectedDestination;
        this.destinationState = DestinationState.SET;
        // Notify observers that the destination is set only after updating the
        // destinationState.
        this.destination = destination;
        this.updateRecentDestinations_();
    }
    async onDestinationCapabilitiesReady_() {
        // Wait for any 'destination-changed' events to be fired first.
        await this.updateComplete;
        this.fire('destination-capabilities-changed', this.destinationStore_.selectedDestination);
        this.updateRecentDestinations_();
        if (this.destinationState === DestinationState.SET) {
            this.destinationState = DestinationState.UPDATED;
        }
    }
    onDestinationError_(e) {
        let errorType = Error$1.NONE;
        switch (e.detail) {
            case DestinationErrorType.INVALID:
                errorType = Error$1.INVALID_PRINTER;
                break;
            case DestinationErrorType.NO_DESTINATIONS:
                errorType = Error$1.NO_DESTINATIONS;
                this.noDestinations_ = true;
                break;
        }
        this.error = errorType;
    }
    onErrorChanged_() {
        if (this.error === Error$1.INVALID_PRINTER ||
            this.error === Error$1.NO_DESTINATIONS) {
            this.destinationState = DestinationState.ERROR;
        }
    }
    updateRecentDestinations_() {
        if (!this.destination) {
            return;
        }
        // Determine if this destination is already in the recent destinations,
        // where in the array it is located, and whether or not it is visible.
        const newDestination = makeRecentDestination(this.destination);
        const recentDestinations = this.getSettingValue('recentDestinations');
        let indexFound = -1;
        // Note: isVisible should be only be used if the destination is unpinned.
        // Although pinned destinations are always visible, isVisible may not
        // necessarily be set to true in this case.
        let isVisible = false;
        let numUnpinnedChecked = 0;
        for (let i = 0; i < recentDestinations.length; i++) {
            const recent = recentDestinations[i];
            if (recent.id === newDestination.id &&
                recent.origin === newDestination.origin) {
                indexFound = i;
                // If we haven't seen the maximum unpinned destinations already, this
                // destination is visible in the dropdown.
                isVisible = numUnpinnedChecked < NUM_UNPINNED_DESTINATIONS;
                break;
            }
            if (!isPdfPrinter(recent.id)) {
                numUnpinnedChecked++;
            }
        }
        // No change
        if (indexFound === 0 &&
            recentDestinations[0].capabilities === newDestination.capabilities) {
            return;
        }
        const isNew = indexFound === -1;
        // Shift the array so that the nth most recent destination is located at
        // index n.
        if (isNew && recentDestinations.length === NUM_PERSISTED_DESTINATIONS) {
            indexFound = NUM_PERSISTED_DESTINATIONS - 1;
        }
        // Create a clone first, otherwise array modifications will not be detected
        // by the underlying Observable instance.
        const recentDestinationsClone = recentDestinations.slice();
        if (indexFound !== -1) {
            // Remove from the list if it already exists, it will be re-added to the
            // front below.
            recentDestinationsClone.splice(indexFound, 1);
        }
        // Add the most recent destination
        recentDestinationsClone.splice(0, 0, newDestination);
        this.setSetting('recentDestinations', recentDestinationsClone);
        // The dropdown needs to be updated if a new printer or one not currently
        // visible in the dropdown has been added.
        if (!isPdfPrinter(newDestination.id) && (isNew || !isVisible)) {
            this.updateDropdownDestinations_();
        }
    }
    updateDropdownDestinations_() {
        const recentDestinations = this.getSettingValue('recentDestinations');
        const updatedDestinations = [];
        let numDestinationsChecked = 0;
        for (const recent of recentDestinations) {
            if (isPdfPrinter(recent.id)) {
                continue;
            }
            numDestinationsChecked++;
            const key = createRecentDestinationKey(recent);
            const destination = this.destinationStore_.getDestinationByKey(key);
            if (destination) {
                updatedDestinations.push(destination);
            }
            if (numDestinationsChecked === NUM_UNPINNED_DESTINATIONS) {
                break;
            }
        }
        this.displayedDestinations_ = updatedDestinations;
    }
    /**
     * @return Whether the destinations dropdown should be disabled.
     */
    shouldDisableDropdown_() {
        return this.state === State.FATAL_ERROR ||
            (this.destinationState === DestinationState.UPDATED && this.disabled &&
                this.state !== State.NOT_READY);
    }
    computeLoaded_() {
        return this.destinationState === DestinationState.ERROR ||
            this.destinationState === DestinationState.UPDATED ||
            (this.destinationState === DestinationState.SET && !!this.destination &&
                (!!this.destination.capabilities ||
                    this.destination.type === PrinterType.PDF_PRINTER));
    }
    /**
     * @param e Event containing the key of the recent destination that was
     *     selected, or "seeMore".
     */
    onSelectedDestinationOptionChange_(e) {
        const value = e.detail;
        if (value === 'seeMore') {
            this.destinationStore_.startLoadAllDestinations();
            this.$.destinationDialog.get().show();
            this.isDialogOpen_ = true;
        }
        else {
            this.destinationStore_.selectDestinationByKey(value);
        }
    }
    onDialogClose_() {
        // Reset the select value if the user dismissed the dialog without
        // selecting a new destination.
        this.updateDestinationSelect_();
        this.isDialogOpen_ = false;
    }
    updateDestinationSelect_() {
        if (this.destinationState === DestinationState.ERROR && !this.destination) {
            return;
        }
        if (this.destinationState === DestinationState.INIT) {
            return;
        }
        const shouldFocus = this.destinationState !== DestinationState.SET && !this.firstLoad;
        this.$.destinationSelect.updateDestination();
        if (shouldFocus) {
            this.$.destinationSelect.focus();
        }
    }
    getDestinationStoreForTest() {
        assert(this.destinationStore_);
        return this.destinationStore_;
    }
}
customElements.define(PrintPreviewDestinationSettingsElement.is, PrintPreviewDestinationSettingsElement);

let instance$1 = null;
function getCss$1() {
    return instance$1 || (instance$1 = [...[getCss$K(), getCss$E()], css `:host{background-color:white;border-inline-start:var(--print-preview-settings-border);display:flex;flex-direction:column}@media (prefers-color-scheme:dark){:host{background-color:rgba(255,255,255,.04)}}#container{color:var(--cr-primary-text-color);flex:1;overflow:overlay}:host #destinationSettings{margin-top:12px}:host #cr-container-shadow-top,:host #cr-container-shadow-bottom{box-shadow:inset 0 5px 3px -3px rgba(0,0,0,.2)}.settings-section{display:block;margin-bottom:16px;margin-top:16px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$1() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-header id="header" .destination="${this.destination}"
    error="${this.error}" .state="${this.state}"
    ?managed="${this.controlsManaged}">
</print-preview-header>
<div id="container" show-bottom-shadow>
  <print-preview-destination-settings id="destinationSettings"
      ?dark="${this.inDarkMode}"
      @destination-changed="${this.onDestinationChanged_}"
      @destination-state-changed="${this.onDestinationStateChanged_}"
      error="${this.error}" @error-changed="${this.onErrorChanged_}"
      ?first-load="${this.firstLoad_}"
      .state="${this.state}" ?app-kiosk-mode="${this.isInAppKioskMode_}"
      ?disabled="${this.controlsDisabled_}"
      available class="settings-section"
      @destination-capabilities-changed="${this.onDestinationCapabilitiesChanged_}">
  </print-preview-destination-settings>
  <print-preview-pages-settings
      page-count="${this.pageCount}" ?disabled="${this.controlsDisabled_}"
      ?hidden="${!this.settingsAvailable_.pages}" class="settings-section">
  </print-preview-pages-settings>
  <print-preview-copies-settings
      .capability="${this.destinationCapabilities_?.printer.copies || null}"
      ?disabled="${this.controlsDisabled_}"
      ?hidden="${!this.settingsAvailable_.copies}" class="settings-section">
  </print-preview-copies-settings>
  <print-preview-layout-settings
      ?disabled="${this.controlsDisabled_}"
      ?hidden="${!this.settingsAvailable_.layout}" class="settings-section">
  </print-preview-layout-settings>
  <print-preview-color-settings
      ?disabled="${this.controlsDisabled_}"
      ?hidden="${!this.settingsAvailable_.color}" class="settings-section">
  </print-preview-color-settings>
  <print-preview-more-settings
      ?settings-expanded-by-user="${this.settingsExpandedByUser_}"
      @settings-expanded-by-user-changed="${this.onSettingsExpandedByUserChanged_}"
      ?disabled="${this.controlsDisabled_}"
      ?hidden="${!this.shouldShowMoreSettings_}">
  </print-preview-more-settings>
  <cr-collapse id="moreSettings"
      ?opened="${this.shouldExpandSettings_()}">
    <print-preview-media-size-settings
        .capability="${this.destinationCapabilities_?.printer.media_size || null}"
        ?disabled="${this.controlsDisabled_}"
        ?hidden="${!this.settingsAvailable_.mediaSize}"
        class="settings-section">
    </print-preview-media-size-settings>
    <print-preview-pages-per-sheet-settings
        ?disabled="${this.controlsDisabled_}"
        ?hidden="${!this.settingsAvailable_.pagesPerSheet}"
        class="settings-section">
    </print-preview-pages-per-sheet-settings>
    <print-preview-margins-settings .state="${this.state}"
        ?disabled="${this.controlsDisabled_}"
        ?hidden="${!this.settingsAvailable_.margins}"
        class="settings-section">
    </print-preview-margins-settings>
    <print-preview-dpi-settings
        .capability="${this.destinationCapabilities_?.printer.dpi || null}"
        ?disabled="${this.controlsDisabled_}"
        ?hidden="${!this.settingsAvailable_.dpi}" class="settings-section">
    </print-preview-dpi-settings>
    <print-preview-scaling-settings
        ?disabled="${this.controlsDisabled_}" ?is-pdf="${this.isPdf}"
        ?hidden="${!this.settingsAvailable_.scaling}"
        class="settings-section">
    </print-preview-scaling-settings>
    <print-preview-duplex-settings
        ?disabled="${this.controlsDisabled_}" ?dark="${this.inDarkMode}"
        ?hidden="${!this.settingsAvailable_.duplex}"
        class="settings-section">
    </print-preview-duplex-settings>
    <print-preview-other-options-settings
        ?disabled="${this.controlsDisabled_}"
        ?hidden="${!this.settingsAvailable_.otherOptions}"
        class="settings-section">
    </print-preview-other-options-settings>
    <print-preview-advanced-options-settings
        .destination="${this.destination}"
        ?disabled="${this.controlsDisabled_}"
        ?hidden="${!this.settingsAvailable_.vendorItems}"
        class="settings-section">
    </print-preview-advanced-options-settings>
    <print-preview-link-container .destination="${this.destination}"
        ?app-kiosk-mode="${this.isInAppKioskMode_}"
        ?disabled="${this.controlsDisabled_}">
    </print-preview-link-container>
  </cr-collapse>
</div>
<print-preview-button-strip .destination="${this.destination}"
    .state="${this.state}" ?first-load="${this.firstLoad_}"
    @print-button-focused="${this.onPrintButtonFocused_}">
</print-preview-button-strip>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Number of settings sections to show when "More settings" is collapsed.
 */
const MAX_SECTIONS_TO_SHOW = 6;
// Settings to observe their `available` field to decide whether to show the
// "More settings" collapsible section.
const SETTINGS_TO_OBSERVE = [
    'color',
    'copies',
    'dpi',
    'duplex',
    'layout',
    'margins',
    'mediaSize',
    'otherOptions',
    'pages',
    'pagesPerSheet',
    'scaling',
    'vendorItems',
];
const PrintPreviewSidebarElementBase = CrContainerShadowMixinLit(WebUiListenerMixinLit(SettingsMixin(DarkModeMixin(CrLitElement))));
class PrintPreviewSidebarElement extends PrintPreviewSidebarElementBase {
    static get is() {
        return 'print-preview-sidebar';
    }
    static get styles() {
        return getCss$1();
    }
    render() {
        return getHtml$1.bind(this)();
    }
    static get properties() {
        return {
            controlsManaged: { type: Boolean },
            destination: {
                type: Object,
                notify: true,
            },
            destinationCapabilities_: { type: Object },
            destinationState: {
                type: Number,
                notify: true,
            },
            error: {
                type: Number,
                notify: true,
            },
            isPdf: { type: Boolean },
            pageCount: { type: Number },
            state: { type: Number },
            controlsDisabled_: { type: Boolean },
            firstLoad_: { type: Boolean },
            isInAppKioskMode_: { type: Boolean },
            settingsExpandedByUser_: { type: Boolean },
            shouldShowMoreSettings_: { type: Boolean },
            settingsAvailable_: { type: Object },
        };
    }
    #controlsManaged_accessor_storage = false;
    get controlsManaged() { return this.#controlsManaged_accessor_storage; }
    set controlsManaged(value) { this.#controlsManaged_accessor_storage = value; }
    #destination_accessor_storage = null;
    get destination() { return this.#destination_accessor_storage; }
    set destination(value) { this.#destination_accessor_storage = value; }
    #destinationCapabilities__accessor_storage = null;
    get destinationCapabilities_() { return this.#destinationCapabilities__accessor_storage; }
    set destinationCapabilities_(value) { this.#destinationCapabilities__accessor_storage = value; }
    #destinationState_accessor_storage = DestinationState.INIT;
    get destinationState() { return this.#destinationState_accessor_storage; }
    set destinationState(value) { this.#destinationState_accessor_storage = value; }
    #error_accessor_storage = null;
    get error() { return this.#error_accessor_storage; }
    set error(value) { this.#error_accessor_storage = value; }
    #isPdf_accessor_storage = false;
    get isPdf() { return this.#isPdf_accessor_storage; }
    set isPdf(value) { this.#isPdf_accessor_storage = value; }
    #pageCount_accessor_storage = 0;
    get pageCount() { return this.#pageCount_accessor_storage; }
    set pageCount(value) { this.#pageCount_accessor_storage = value; }
    #state_accessor_storage = State.NOT_READY;
    get state() { return this.#state_accessor_storage; }
    set state(value) { this.#state_accessor_storage = value; }
    #settingsAvailable__accessor_storage;
    get settingsAvailable_() { return this.#settingsAvailable__accessor_storage; }
    set settingsAvailable_(value) { this.#settingsAvailable__accessor_storage = value; }
    #controlsDisabled__accessor_storage = false;
    get controlsDisabled_() { return this.#controlsDisabled__accessor_storage; }
    set controlsDisabled_(value) { this.#controlsDisabled__accessor_storage = value; }
    #firstLoad__accessor_storage = true;
    get firstLoad_() { return this.#firstLoad__accessor_storage; }
    set firstLoad_(value) { this.#firstLoad__accessor_storage = value; }
    #isInAppKioskMode__accessor_storage = false;
    get isInAppKioskMode_() { return this.#isInAppKioskMode__accessor_storage; }
    set isInAppKioskMode_(value) { this.#isInAppKioskMode__accessor_storage = value; }
    #settingsExpandedByUser__accessor_storage = false;
    get settingsExpandedByUser_() { return this.#settingsExpandedByUser__accessor_storage; }
    set settingsExpandedByUser_(value) { this.#settingsExpandedByUser__accessor_storage = value; }
    #shouldShowMoreSettings__accessor_storage = false;
    get shouldShowMoreSettings_() { return this.#shouldShowMoreSettings__accessor_storage; }
    set shouldShowMoreSettings_(value) { this.#shouldShowMoreSettings__accessor_storage = value; }
    constructor() {
        super();
        this.settingsAvailable_ = {};
        for (const key of SETTINGS_TO_OBSERVE) {
            this.settingsAvailable_[key] = true;
        }
    }
    connectedCallback() {
        super.connectedCallback();
        for (const key of SETTINGS_TO_OBSERVE) {
            this.addSettingObserver(`${key}.available`, (value) => {
                this.settingsAvailable_[key] = value;
                this.updateShouldShowMoreSettings_();
                this.requestUpdate();
            });
        }
        this.updateShouldShowMoreSettings_();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('state')) {
            this.controlsDisabled_ = this.state !== State.READY;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('state')) {
            this.onStateChanged_();
        }
    }
    /**
     * @param defaultPrinter The system default printer ID.
     * @param serializedDestinationSelectionRulesStr String with rules
     *     for selecting the default destination.
     * @param pdfPrinterDisabled Whether the PDF printer is disabled.
     */
    init(appKioskMode, defaultPrinter, serializedDestinationSelectionRulesStr, pdfPrinterDisabled) {
        this.isInAppKioskMode_ = appKioskMode;
        pdfPrinterDisabled = this.isInAppKioskMode_ || pdfPrinterDisabled;
        this.$.destinationSettings.init(defaultPrinter, pdfPrinterDisabled, serializedDestinationSelectionRulesStr);
    }
    updateShouldShowMoreSettings_() {
        // Destination settings is always available. See if the total number of
        // available sections exceeds the maximum number to show.
        this.shouldShowMoreSettings_ =
            SETTINGS_TO_OBSERVE.reduce((count, setting) => {
                return this.settingsAvailable_[setting] ? count + 1 : count;
            }, 1) > MAX_SECTIONS_TO_SHOW;
    }
    /**
     * @return Whether the "more settings" collapse should be expanded.
     */
    shouldExpandSettings_() {
        if (this.settingsExpandedByUser_ === undefined ||
            this.shouldShowMoreSettings_ === undefined) {
            return false;
        }
        // Expand the settings if the user has requested them expanded or if more
        // settings is not displayed (i.e. less than 6 total settings available).
        return this.settingsExpandedByUser_ || !this.shouldShowMoreSettings_;
    }
    onPrintButtonFocused_() {
        this.firstLoad_ = false;
    }
    onStateChanged_() {
        if (this.state !== State.PRINTING) {
            return;
        }
        if (this.shouldShowMoreSettings_) {
            MetricsContext.printSettingsUi().record(this.settingsExpandedByUser_ ?
                PrintSettingsUiBucket.PRINT_WITH_SETTINGS_EXPANDED :
                PrintSettingsUiBucket.PRINT_WITH_SETTINGS_COLLAPSED);
        }
    }
    /** @return Whether the system dialog link is available. */
    systemDialogLinkAvailable() {
        const linkContainer = this.shadowRoot.querySelector('print-preview-link-container');
        return !!linkContainer && linkContainer.systemDialogLinkAvailable();
    }
    onDestinationChanged_(e) {
        this.destination = e.detail.value;
        this.destinationCapabilities_ = null;
    }
    onDestinationCapabilitiesChanged_(e) {
        assert(this.destination);
        assert(e.detail.id === this.destination.id);
        // When `this.destination.capabilities` changes it is always a new object.
        this.destinationCapabilities_ = this.destination.capabilities;
    }
    onDestinationStateChanged_(e) {
        this.destinationState = e.detail.value;
    }
    onErrorChanged_(e) {
        this.error = e.detail.value;
    }
    onSettingsExpandedByUserChanged_(e) {
        this.settingsExpandedByUser_ = e.detail.value;
    }
}
customElements.define(PrintPreviewSidebarElement.is, PrintPreviewSidebarElement);

let instance = null;
function getCss() {
    return instance || (instance = [...[], css `:host{display:flex;height:100%;user-select:none}@media (prefers-color-scheme:dark){:host{background:var(--google-grey-900)}}print-preview-sidebar{flex:none;width:var(--print-preview-sidebar-width)}#preview-area-container{align-items:center;background-color:var(--preview-area-background-color);flex:1}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml() {
    // clang-format off
    return html `<!--_html_template_start_-->
<print-preview-state id="state"
    @state-changed="${this.onStateChanged_}" error="${this.error_}"
    @error-changed="${this.onErrorChanged_}">
</print-preview-state>
<print-preview-model id="model"
    @settings-managed-changed="${this.onSettingsManagedChanged_}"
    .destination="${this.destination_}"
    .documentSettings="${this.documentSettings_}"
    .margins="${this.margins_}" .pageSize="${this.pageSize_}"
    @preview-setting-changed="${this.onPreviewSettingChanged_}"
    @sticky-setting-changed="${this.onStickySettingChanged_}"
    @setting-valid-changed="${this.onSettingValidChanged_}">
</print-preview-model>
<print-preview-document-info id="documentInfo"
    @document-settings-changed="${this.onDocumentSettingsChanged_}"
    @margins-changed="${this.onMarginsChanged_}"
    @page-size-changed="${this.onPageSizeChanged_}">
</print-preview-document-info>
<div id="preview-area-container">
  <print-preview-preview-area id="previewArea"
      .destination="${this.destination_}" error="${this.error_}"
      @error-changed="${this.onErrorChanged_}"
      ?document-modifiable="${this.documentSettings_.isModifiable}"
      .margins="${this.margins_}" .pageSize="${this.pageSize_}"
      state="${this.state}" .measurementSystem="${this.measurementSystem_}"
      @preview-state-changed="${this.onPreviewStateChanged_}"
      @preview-start="${this.onPreviewStart_}">
  </print-preview-preview-area>
</div>
<print-preview-sidebar id="sidebar"
    @destination-state-changed="${this.onDestinationStateChanged_}"
    ?controls-managed="${this.controlsManaged_}"
    error="${this.error_}" @error-changed="${this.onErrorChanged_}"
    ?is-pdf="${!this.documentSettings_.isModifiable}"
    page-count="${this.documentSettings_.pageCount}" state="${this.state}"
    @focus="${this.onSidebarFocus_}"
    @destination-changed="${this.onDestinationChanged_}"
    @destination-capabilities-changed="${this.onDestinationCapabilitiesChanged_}"
    @print-with-system-dialog="${this.onPrintWithSystemDialog_}"
    @print-requested="${this.onPrintRequested_}"
    @cancel-requested="${this.onCancelRequested_}">
</print-preview-sidebar>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PrintPreviewAppElementBase = WebUiListenerMixinLit(SettingsMixin(CrLitElement));
class PrintPreviewAppElement extends PrintPreviewAppElementBase {
    static get is() {
        return 'print-preview-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            state: { type: Number },
            controlsManaged_: { type: Boolean },
            destination_: { type: Object },
            destinationsManaged_: { type: Boolean },
            documentSettings_: { type: Object },
            error_: { type: Number },
            margins_: { type: Object },
            pageSize_: { type: Object },
            settingsManaged_: { type: Boolean },
            measurementSystem_: { type: Object },
        };
    }
    #state_accessor_storage = State.NOT_READY;
    get state() { return this.#state_accessor_storage; }
    set state(value) { this.#state_accessor_storage = value; }
    #controlsManaged__accessor_storage = false;
    get controlsManaged_() { return this.#controlsManaged__accessor_storage; }
    set controlsManaged_(value) { this.#controlsManaged__accessor_storage = value; }
    #destination__accessor_storage = null;
    get destination_() { return this.#destination__accessor_storage; }
    set destination_(value) { this.#destination__accessor_storage = value; }
    #destinationsManaged__accessor_storage = false;
    get destinationsManaged_() { return this.#destinationsManaged__accessor_storage; }
    set destinationsManaged_(value) { this.#destinationsManaged__accessor_storage = value; }
    #documentSettings__accessor_storage = createDocumentSettings();
    get documentSettings_() { return this.#documentSettings__accessor_storage; }
    set documentSettings_(value) { this.#documentSettings__accessor_storage = value; }
    #error__accessor_storage = null;
    get error_() { return this.#error__accessor_storage; }
    set error_(value) { this.#error__accessor_storage = value; }
    #margins__accessor_storage = null;
    get margins_() { return this.#margins__accessor_storage; }
    set margins_(value) { this.#margins__accessor_storage = value; }
    #pageSize__accessor_storage = new Size(612, 792);
    get pageSize_() { return this.#pageSize__accessor_storage; }
    set pageSize_(value) { this.#pageSize__accessor_storage = value; }
    #settingsManaged__accessor_storage = false;
    get settingsManaged_() { return this.#settingsManaged__accessor_storage; }
    set settingsManaged_(value) { this.#settingsManaged__accessor_storage = value; }
    #measurementSystem__accessor_storage = null;
    get measurementSystem_() { return this.#measurementSystem__accessor_storage; }
    set measurementSystem_(value) { this.#measurementSystem__accessor_storage = value; }
    nativeLayer_ = null;
    tracker_ = new EventTracker();
    cancelled_ = false;
    printRequested_ = false;
    startPreviewWhenReady_ = false;
    showSystemDialogBeforePrint_ = false;
    openPdfInPreview_ = false;
    isInKioskAutoPrintMode_ = false;
    whenReady_ = null;
    constructor() {
        super();
        // Regular expression that captures the leading slash, the content and the
        // trailing slash in three different groups.
        const CANONICAL_PATH_REGEX = /(^\/)([\/-\w]+)(\/$)/;
        const path = location.pathname.replace(CANONICAL_PATH_REGEX, '$1$2');
        if (path !== '/') { // There are no subpages in Print Preview.
            window.history.replaceState(undefined /* stateObject */, '', '/');
        }
    }
    firstUpdated() {
        FocusOutlineManager.forDocument(document);
        // 
        this.$.sidebar.addEventListener('open-pdf-in-preview', this.onOpenPdfInPreview_.bind(this));
        // 
    }
    connectedCallback() {
        super.connectedCallback();
        document.documentElement.classList.remove('loading');
        this.nativeLayer_ = NativeLayerImpl.getInstance();
        this.addWebUiListener('print-preset-options', this.onPrintPresetOptions_.bind(this));
        this.tracker_.add(window, 'keydown', this.onKeyDown_.bind(this));
        this.$.previewArea.setPluginKeyEventCallback(this.onKeyDown_.bind(this));
        this.whenReady_ = whenReady();
        this.nativeLayer_.getInitialSettings().then(this.onInitialSettingsSet_.bind(this));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.tracker_.removeAll();
        this.whenReady_ = null;
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('destinationsManaged_') ||
            changedPrivateProperties.has('settingsManaged_')) {
            this.controlsManaged_ =
                this.destinationsManaged_ || this.settingsManaged_;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('state')) {
            this.updateUiForStateChange_();
        }
        if (changedPrivateProperties.has('error_')) {
            if (this.error_ !== null && this.error_ !== Error$1.NONE) {
                this.nativeLayer_.recordInHistogram('PrintPreview.StateError', this.error_, Error$1.MAX_BUCKET);
            }
        }
    }
    onSidebarFocus_() {
        this.$.previewArea.hideToolbar();
    }
    /**
     * Consume escape and enter key presses and ctrl + shift + p. Delegate
     * everything else to the preview area.
     */
    onKeyDown_(e) {
        if (e.key === 'Escape' && !hasKeyModifiers(e)) {
            // On non-mac with toolkit-views, ESC key is handled by C++-side instead
            // of JS-side.
            if (isMac) {
                this.close_();
                e.preventDefault();
            }
            return;
        }
        // On Mac, Cmd+Period should close the print dialog.
        if (isMac && e.key === '.' && e.metaKey) {
            this.close_();
            e.preventDefault();
            return;
        }
        // Ctrl + Shift + p / Mac equivalent. Doesn't apply on Chrome OS.
        // On Linux/Windows, shift + p means that e.key will be 'P' with caps lock
        // off or 'p' with caps lock on.
        // On Mac, alt + p means that e.key will be unicode 03c0 (pi).
        if (e.key === 'P' || e.key === 'p' || e.key === '\u03c0') {
            if ((isMac && e.metaKey && e.altKey && !e.shiftKey && !e.ctrlKey) ||
                (!isMac && e.shiftKey && e.ctrlKey && !e.altKey && !e.metaKey)) {
                // Don't use system dialog if the link isn't available.
                if (!this.$.sidebar.systemDialogLinkAvailable()) {
                    return;
                }
                // Don't try to print with system dialog on Windows if the document is
                // not ready, because we send the preview document to the printer on
                // Windows.
                if (!isWindows || this.state === State.READY) {
                    this.onPrintWithSystemDialog_();
                }
                e.preventDefault();
                return;
            }
        }
        if ((e.key === 'Enter' || e.key === 'NumpadEnter') &&
            this.state === State.READY) {
            const activeElement = e.composedPath()[0];
            // activeElement may be undefined if this is a forwarded key event from
            // the plugin. Print Preview conventionally does not trigger a print for
            // Enter when the plugin is focused
            if (!activeElement) {
                return;
            }
            const activeElementTag = activeElement.tagName;
            if (['CR-BUTTON', 'BUTTON', 'SELECT', 'A', 'CR-CHECKBOX'].includes(activeElementTag)) {
                return;
            }
            this.onPrintRequested_();
            e.preventDefault();
            return;
        }
        // Pass certain directional keyboard events to the PDF viewer.
        this.$.previewArea.handleDirectionalKeyEvent(e);
    }
    onInitialSettingsSet_(settings) {
        if (!this.whenReady_) {
            // This element and its corresponding model were detached while waiting
            // for the callback. This can happen in tests; return early.
            return;
        }
        this.whenReady_.then(() => {
            this.$.documentInfo.init(settings.previewModifiable, settings.documentTitle, settings.documentHasSelection);
            this.$.model.setStickySettings(settings.serializedAppStateStr);
            this.$.model.setPolicySettings(settings.policies);
            this.measurementSystem_ = new MeasurementSystem(settings.thousandsDelimiter, settings.decimalDelimiter, settings.unitType);
            this.setSetting('selectionOnly', settings.shouldPrintSelectionOnly);
            this.$.sidebar.init(settings.isInAppKioskMode, settings.printerName, settings.serializedDefaultDestinationSelectionRulesStr, settings.pdfPrinterDisabled);
            this.destinationsManaged_ = settings.destinationsManaged;
            this.isInKioskAutoPrintMode_ = settings.isInKioskAutoPrintMode;
            // This is only visible in the task manager.
            let title = document.head.querySelector('title');
            if (!title) {
                title = document.createElement('title');
                document.head.appendChild(title);
            }
            title.textContent = settings.documentTitle;
        });
    }
    onDestinationStateChanged_(e) {
        const destinationState = e.detail.value;
        switch (destinationState) {
            case DestinationState.SET:
                if (this.state !== State.NOT_READY &&
                    this.state !== State.FATAL_ERROR) {
                    this.$.state.transitTo(State.NOT_READY);
                }
                break;
            case DestinationState.UPDATED:
                if (!this.$.model.initialized()) {
                    this.$.model.applyStickySettings();
                }
                this.$.model.applyPoliciesOnDestinationUpdate();
                this.startPreviewWhenReady_ = true;
                if (this.state === State.NOT_READY &&
                    this.destination_.type !== PrinterType.PDF_PRINTER) {
                    this.nativeLayer_.recordBooleanHistogram('PrintPreview.TransitionedToReadyState', true);
                }
                this.$.state.transitTo(State.READY);
                break;
            case DestinationState.ERROR:
                if (this.state === State.NOT_READY &&
                    this.destination_.type !== PrinterType.PDF_PRINTER) {
                    this.nativeLayer_.recordBooleanHistogram('PrintPreview.TransitionedToReadyState', false);
                }
                this.$.state.transitTo(State.ERROR);
                break;
        }
    }
    /**
     * @param e Event containing the new sticky settings.
     */
    onStickySettingChanged_(e) {
        this.nativeLayer_.saveAppState(e.detail);
    }
    async onPreviewSettingChanged_() {
        if (this.state === State.READY) {
            // Need to wait for rendering to finish, to ensure that the `destination`
            // is synced across print-preview-app, print-preview-model and
            // print-preview-area.
            await this.updateComplete;
            assert(this.destination_.id === this.$.previewArea.destination.id);
            assert(this.destination_.id === this.$.model.destination.id);
            this.$.previewArea.startPreview(false);
            this.startPreviewWhenReady_ = false;
        }
        else {
            this.startPreviewWhenReady_ = true;
        }
    }
    updateUiForStateChange_() {
        if (this.state === State.READY) {
            if (this.startPreviewWhenReady_) {
                this.$.previewArea.startPreview(false);
                this.startPreviewWhenReady_ = false;
            }
            if (this.isInKioskAutoPrintMode_ || this.printRequested_) {
                this.onPrintRequested_();
                // Reset in case printing fails.
                this.printRequested_ = false;
            }
        }
        else if (this.state === State.CLOSING) {
            this.remove();
            this.nativeLayer_.dialogClose(this.cancelled_);
        }
        else if (this.state === State.PRINT_PENDING) {
            assert(this.destination_);
            if (this.destination_.type !== PrinterType.PDF_PRINTER) {
                // Only hide the preview for local, non PDF destinations.
                this.nativeLayer_.hidePreview();
                this.$.state.transitTo(State.HIDDEN);
            }
        }
        else if (this.state === State.PRINTING) {
            assert(this.destination_);
            const whenPrintDone = this.nativeLayer_.doPrint(this.$.model.createPrintTicket(this.destination_, this.openPdfInPreview_, this.showSystemDialogBeforePrint_));
            const onError = this.destination_.type === PrinterType.PDF_PRINTER ?
                this.onFileSelectionCancel_.bind(this) :
                this.onPrintFailed_.bind(this);
            whenPrintDone.then(this.close_.bind(this), onError);
        }
    }
    onPrintRequested_() {
        if (this.state === State.NOT_READY) {
            this.printRequested_ = true;
            return;
        }
        this.$.state.transitTo(this.$.previewArea.previewLoaded() ? State.PRINTING :
            State.PRINT_PENDING);
    }
    onCancelRequested_() {
        this.cancelled_ = true;
        this.$.state.transitTo(State.CLOSING);
    }
    /**
     * @param e The event containing the new validity.
     */
    onSettingValidChanged_(e) {
        if (e.detail) {
            this.$.state.transitTo(State.READY);
        }
        else {
            this.error_ = Error$1.INVALID_TICKET;
            this.$.state.transitTo(State.ERROR);
        }
    }
    onFileSelectionCancel_() {
        this.$.state.transitTo(State.READY);
    }
    onPrintWithSystemDialog_() {
        // 
        // 
        this.nativeLayer_.showSystemDialog();
        this.$.state.transitTo(State.SYSTEM_DIALOG);
        // 
    }
    // 
    onOpenPdfInPreview_() {
        this.openPdfInPreview_ = true;
        this.$.previewArea.setOpeningPdfInPreview();
        this.onPrintRequested_();
    }
    // 
    /**
     * Called when printing to an extension printer fails.
     * @param httpError The HTTP error code, or -1 or a string describing
     *     the error, if not an HTTP error.
     */
    onPrintFailed_(httpError) {
        console.warn('Printing failed with error code ' + httpError);
        this.error_ = Error$1.PRINT_FAILED;
        this.$.state.transitTo(State.FATAL_ERROR);
    }
    onPreviewStateChanged_(e) {
        const previewState = e.detail.value;
        switch (previewState) {
            case PreviewAreaState.DISPLAY_PREVIEW:
            case PreviewAreaState.OPEN_IN_PREVIEW_LOADED:
                if (this.state === State.PRINT_PENDING || this.state === State.HIDDEN) {
                    this.$.state.transitTo(State.PRINTING);
                }
                break;
            case PreviewAreaState.ERROR:
                if (this.state !== State.ERROR && this.state !== State.FATAL_ERROR) {
                    this.$.state.transitTo(this.error_ === Error$1.INVALID_PRINTER ? State.ERROR :
                        State.FATAL_ERROR);
                }
                break;
        }
    }
    /**
     * Updates printing options according to source document presets.
     * @param disableScaling Whether the document disables scaling.
     * @param copies The default number of copies from the document.
     * @param duplex The default duplex setting from the document.
     */
    onPrintPresetOptions_(disableScaling, copies, duplex) {
        if (disableScaling) {
            this.$.documentInfo.updateIsScalingDisabled(true);
        }
        if (copies > 0 && this.getSetting('copies').available) {
            this.setSetting('copies', copies, true);
        }
        if (duplex === DuplexMode.UNKNOWN_DUPLEX_MODE) {
            return;
        }
        if (this.getSetting('duplex').available) {
            this.setSetting('duplex', duplex === DuplexMode.LONG_EDGE || duplex === DuplexMode.SHORT_EDGE, true);
        }
        if (duplex !== DuplexMode.SIMPLEX &&
            this.getSetting('duplexShortEdge').available) {
            this.setSetting('duplexShortEdge', duplex === DuplexMode.SHORT_EDGE, true);
        }
    }
    /**
     * @param e Contains the new preview request ID.
     */
    onPreviewStart_(e) {
        this.$.documentInfo.inFlightRequestId = e.detail;
    }
    close_() {
        this.$.state.transitTo(State.CLOSING);
    }
    onDestinationChanged_(e) {
        this.destination_ = e.detail.value;
    }
    onDestinationCapabilitiesChanged_() {
        this.$.model.updateSettingsFromDestination();
    }
    onStateChanged_(e) {
        this.state = e.detail.value;
    }
    onErrorChanged_(e) {
        this.error_ = e.detail.value;
    }
    onSettingsManagedChanged_(e) {
        this.settingsManaged_ = e.detail.value;
    }
    onDocumentSettingsChanged_(e) {
        this.documentSettings_ = e.detail.value;
    }
    onMarginsChanged_(e) {
        this.margins_ = e.detail.value;
    }
    onPageSizeChanged_(e) {
        this.pageSize_ = e.detail.value;
    }
}
customElements.define(PrintPreviewAppElement.is, PrintPreviewAppElement);

export { BackgroundGraphicsModeRestriction, ColorMode, ColorModeRestriction, CrButtonElement, CrCheckboxElement, CrIconButtonElement, CrInputElement, CustomMarginsOrientation, DEFAULT_MAX_COPIES, Destination, DestinationErrorType, DestinationOrigin, DestinationState, DestinationStore, DestinationStoreEventType, DuplexMode, DuplexModeRestriction, DuplexType, Error$1 as Error, GooglePromotedDestinationId, IconsetMap, Margins, MarginsType, MeasurementSystem, MeasurementSystemUnitType, NUM_PERSISTED_DESTINATIONS, NativeLayerImpl, Observable, PDF_DESTINATION_KEY, PagesValue, PluginProxyImpl, PreviewAreaState, PrintPreviewAdvancedSettingsDialogElement, PrintPreviewAdvancedSettingsItemElement, PrintPreviewAppElement, PrintPreviewButtonStripElement, PrintPreviewColorSettingsElement, PrintPreviewCopiesSettingsElement, PrintPreviewDestinationDialogElement, PrintPreviewDestinationListElement, PrintPreviewDestinationListItemElement, PrintPreviewDestinationSelectElement, PrintPreviewDestinationSettingsElement, PrintPreviewDpiSettingsElement, PrintPreviewDuplexSettingsElement, PrintPreviewHeaderElement, PrintPreviewLayoutSettingsElement, PrintPreviewLinkContainerElement, PrintPreviewMarginControlContainerElement, PrintPreviewMarginControlElement, PrintPreviewMarginsSettingsElement, PrintPreviewMediaSizeSettingsElement, PrintPreviewModelElement, PrintPreviewNumberSettingsSectionElement, PrintPreviewOtherOptionsSettingsElement, PrintPreviewPagesPerSheetSettingsElement, PrintPreviewPagesSettingsElement, PluralStringProxyImpl as PrintPreviewPluralStringProxyImpl, PrintPreviewPreviewAreaElement, PrintPreviewScalingSettingsElement, PrintPreviewSearchBoxElement, PrintPreviewSettingsSelectElement, PrintPreviewSidebarElement, PrinterType, ScalingType, SelectMixin, Size, State, VendorCapabilityValueType, createDestinationKey, createDocumentSettings, getInstance, getSelectDropdownBackground, getTrustedHTML, makeRecentDestination, setValueAtPath, whenReady };
//# sourceMappingURL=print_preview.rollup.js.map
