// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_auto_img/cr_auto_img.js';
import './description_section.js';
import './product_selector.js';
import './buying_options_section.js';
import './empty_section.js';
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_icons.css.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_hidden_style.css.js';
import 'chrome://resources/cr_elements/cr_tooltip/cr_tooltip.js';
import './shared_vars.css.js';
import { assert } from '//resources/js/assert.js';
import { getFaviconForPageURL } from '//resources/js/icon.js';
import { ShoppingServiceBrowserProxyImpl } from 'chrome://resources/cr_components/commerce/shopping_service_browser_proxy.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { DragAndDropManager } from './drag_and_drop_manager.js';
import { getCss } from './table.css.js';
import { getHtml } from './table.html.js';
import { WindowProxy } from './window_proxy.js';
export class TableElement extends CrLitElement {
    static get is() {
        return 'product-specifications-table';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            columns: { type: Array },
            draggingColumn: { type: Object },
            hoveredColumnIndex_: { type: Number },
        };
    }
    #columns_accessor_storage = [];
    get columns() { return this.#columns_accessor_storage; }
    set columns(value) { this.#columns_accessor_storage = value; }
    #draggingColumn_accessor_storage = null;
    get draggingColumn() { return this.#draggingColumn_accessor_storage; }
    set draggingColumn(value) { this.#draggingColumn_accessor_storage = value; }
    #hoveredColumnIndex__accessor_storage = null;
    get hoveredColumnIndex_() { return this.#hoveredColumnIndex__accessor_storage; }
    set hoveredColumnIndex_(value) { this.#hoveredColumnIndex__accessor_storage = value; }
    dragAndDropManager_;
    shoppingApi_ = ShoppingServiceBrowserProxyImpl.getInstance();
    constructor() {
        super();
        this.dragAndDropManager_ = new DragAndDropManager(this);
    }
    getDragAndDropManager() {
        return this.dragAndDropManager_;
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('columns')) {
            this.style.setProperty('--num-columns', String(this.columns.length));
        }
        this.dragAndDropManager_.tablePropertiesUpdated();
    }
    connectedCallback() {
        super.connectedCallback();
        // Prevent cursor from switching to not-allowed on Windows during drag and
        // drop.
        this.$.table.addEventListener('dragenter', (e) => e.preventDefault());
        this.$.table.addEventListener('dragleave', (e) => e.preventDefault());
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.dragAndDropManager_.destroy();
    }
    // Called by |dragAndDropManager|.
    async moveColumnOnDrop(fromIndex, dropIndex) {
        const columns = this.columns;
        const [draggingColumn] = columns.splice(fromIndex, 1);
        assert(draggingColumn);
        columns.splice(dropIndex, 0, draggingColumn);
        this.requestUpdate();
        await this.updateComplete;
        this.fire('url-order-update');
    }
    closeAllProductSelectionMenus() {
        const productSelectors = this.shadowRoot.querySelectorAll('product-selector');
        productSelectors.forEach(productSelector => productSelector.closeMenu());
    }
    // |this.draggingColumn| is set by |dragAndDropManager|.
    isDragging_(columnIndex) {
        return !!this.draggingColumn &&
            columnIndex === Number(this.draggingColumn.dataset['index']);
    }
    isFirstColumn_(columnIndex) {
        if (!this.draggingColumn) {
            return columnIndex === 0;
        }
        // While dragging, |dragAndDropManager| toggles this attribute, as the first
        // column shown in the table may have a non-zero `columnIndex`.
        return undefined;
    }
    getScrollSnapAlign_() {
        return !this.draggingColumn ? 'start' : 'none';
    }
    // Determines the number of rows needed in the grid layout.
    // This is the sum of:
    //   - 1 row for the product selector.
    //   - 1 row for the image container.
    //   - Number of product details.
    getRowCount_(numProductDetails) {
        return 2 + numProductDetails;
    }
    getUrls_() {
        return this.columns.map(column => column.selectedItem.url);
    }
    onHideOpenTabButton_() {
        this.hoveredColumnIndex_ = null;
    }
    onShowOpenTabButton_(e) {
        const currentTarget = e.currentTarget;
        this.hoveredColumnIndex_ = Number(currentTarget.dataset['index']);
    }
    showOpenTabButton_(columnIndex) {
        return !this.draggingColumn && this.hoveredColumnIndex_ !== null &&
            this.hoveredColumnIndex_ === columnIndex;
    }
    onOpenTabButtonClick_(e) {
        if (!WindowProxy.getInstance().onLine) {
            this.dispatchEvent(new Event('unavailable-action-attempted'));
            return;
        }
        const currentTarget = e.currentTarget;
        const columnIndex = Number(currentTarget.dataset['index']);
        this.shoppingApi_.switchToOrOpenTab({ url: this.columns[columnIndex]?.selectedItem.url || '' });
        chrome.metricsPrivate.recordUserAction('Commerce.Compare.ReopenedProductPage');
    }
    onSelectedUrlChange_(e) {
        const currentTarget = e.currentTarget;
        this.dispatchEvent(new CustomEvent('url-change', {
            bubbles: true,
            composed: true,
            detail: {
                url: e.detail.url,
                urlSection: e.detail.urlSection,
                index: Number(currentTarget.dataset['index']),
            },
        }));
    }
    onUrlRemove_(e) {
        const currentTarget = e.currentTarget;
        this.dispatchEvent(new CustomEvent('url-remove', {
            bubbles: true,
            composed: true,
            detail: {
                index: Number(currentTarget.dataset['index']),
            },
        }));
    }
    showRow_(title, rowIndex) {
        return this.rowHasNonEmptyAttributes_(title, rowIndex) ||
            this.rowHasNonEmptySummary_(title, rowIndex) ||
            this.rowHasText_(title, rowIndex) ||
            this.rowHasBuyingOptions_(rowIndex);
    }
    rowHasText_(title, rowIndex) {
        const rowDetails = this.columns.map(column => column.productDetails && column.productDetails[rowIndex]);
        return rowDetails.some(detail => detail && detail.title === title &&
            this.contentIsString_(detail.content));
    }
    rowHasNonEmptyAttributes_(title, rowIndex) {
        const rowDetails = this.columns.map(column => column.productDetails && column.productDetails[rowIndex]);
        return rowDetails.some(detail => {
            if (!detail || detail.title !== title ||
                !this.contentIsProductDescription_(detail.content)) {
                return false;
            }
            return detail.content && detail.content.attributes.some(attr => {
                return attr.value.length > 0 && attr.value !== 'N/A';
            });
        });
    }
    rowHasNonEmptySummary_(title, rowIndex) {
        const rowDetails = this.columns.map(column => column.productDetails && column.productDetails[rowIndex]);
        return rowDetails.some(detail => {
            return detail && detail.title === title && detail.content &&
                this.contentIsProductDescription_(detail.content) &&
                detail.content.summary.length > 0 &&
                detail.content.summary.some((summaryObj) => summaryObj.text !== 'N/A');
        });
    }
    rowHasBuyingOptions_(rowIndex) {
        const rowDetails = this.columns.map(column => column.productDetails && column.productDetails[rowIndex]);
        return rowDetails.some(detail => detail && this.contentIsBuyingOptions_(detail.content));
    }
    filterProductDescription_(productDesc, title, rowIndex) {
        // Hide product descriptions when all attributes/summaries in this row are
        // missing or marked "N/A".
        return {
            attributes: this.rowHasNonEmptyAttributes_(title, rowIndex) ?
                productDesc.attributes :
                [],
            summary: this.rowHasNonEmptySummary_(title, rowIndex) ?
                productDesc.summary :
                [],
        };
    }
    contentIsString_(content) {
        return (content && typeof content === 'string');
    }
    contentIsProductDescription_(content) {
        if (content) {
            const description = content;
            return Array.isArray(description.attributes) &&
                Array.isArray(description.summary);
        }
        return false;
    }
    contentIsBuyingOptions_(content) {
        if (content) {
            const buyingOptions = content;
            return typeof buyingOptions.price === 'string' &&
                typeof buyingOptions.jackpotUrl === 'string' &&
                buyingOptions.price.length > 0;
        }
        return false;
    }
    // This method provides a string that is intended to be used primarily in CSS.
    getFavicon_(url) {
        return getFaviconForPageURL(url, false, '', 32);
    }
}
customElements.define(TableElement.is, TableElement);
