export type ValueDataType = string;
export namespace ValueDataType {
    let HEXADECIMAL: string;
    let UTF8: string;
    let DECIMAL: string;
}
/**
 * A container for an array value that needs to be converted to multiple
 * display formats. Internally, the value is stored as an array and converted
 * to the needed display type at runtime.
 */
export class Value {
    /** @param {!Array<number>} initialValue */
    constructor(initialValue: Array<number>);
    /** @private {!Array<number>} */
    private value_;
    /**
     * Gets the backing array value.
     * @return {!Array<number>}
     */
    getArray(): Array<number>;
    /**
     * Sets the backing array value.
     * @param {!Array<number>} newValue
     */
    setArray(newValue: Array<number>): void;
    /**
     * Sets the value by converting the |newValue| string using the formatting
     * specified by |valueDataType|.
     * @param {!ValueDataType} valueDataType
     * @param {string} newValue
     */
    setAs(valueDataType: ValueDataType, newValue: string): void;
    /**
     * Gets the value as a string representing the given |valueDataType|.
     * @param {!ValueDataType} valueDataType
     * @return {string}
     */
    getAs(valueDataType: ValueDataType): string;
    /**
     * Converts the value to a hex string.
     * @return {string}
     * @private
     */
    private toHex_;
    /**
     * Sets the value from a hex string.
     * @param {string} newValue
     * @private
     */
    private setValueFromHex_;
    /**
     * Converts the value to a UTF-8 encoded text string.
     * @return {string}
     * @private
     */
    private toUTF8_;
    /**
     * Sets the value from a UTF-8 encoded text string.
     * @param {string} newValue
     * @private
     */
    private setValueFromUTF8_;
    /**
     * Converts the value to a decimal string with numbers delimited by '-'.
     * @return {string}
     * @private
     */
    private toDecimal_;
    /**
     * Sets the value from a decimal string delimited by '-'.
     * @param {string} newValue
     * @private
     */
    private setValueFromDecimal_;
}
/**
 * A set of inputs that allow a user to request reads and writes of values.
 * This control allows the value to be displayed in multiple forms
 * as defined by the |ValueDataType| array. Values must be written
 * in these formats. Read and write capability is controlled by a
 * 'properties' bitfield provided by the characteristic.
 * @constructor
 * @extends {HTMLDivElement}
 */
export class ValueControlElement extends HTMLDivElement {
    static get is(): string;
    static get template(): string | TrustedHTML;
    static get observedAttributes(): string[];
    /** @private {!Value} */
    private value_;
    /** @private {?string} */
    private deviceAddress_;
    /** @private {?string} */
    private serviceId_;
    /** @private {?string} */
    private characteristicId_;
    /** @private {?string|undefined} */
    private descriptorId_;
    /** @private {number} */
    private properties_;
    /** @private {!HTMLInputElement} */
    private valueInput_;
    /** @private {!HTMLSelectElement} */
    private typeSelect_;
    /** @private {!HTMLButtonElement} */
    private writeBtn_;
    /** @private {!HTMLButtonElement} */
    private readBtn_;
    /** @private {!HTMLElement} */
    private unavailableMessage_;
    connectedCallback(): void;
    /**
     * Sets the settings used by the value control and redraws the control to
     * match the read/write settings in |options.properties|. If properties
     * are not provided, no restrictions on reading/writing are applied.
     */
    attributeChangedCallback(name: any, oldValue: any, newValue: any): void;
    /**
     * Redraws the value control with updated layout depending on the
     * availability of reads and writes and the current cached value.
     */
    redraw(): void;
    /**
     * Sets the value of the control.
     * @param {!Array<number>} value
     */
    setValue(value: Array<number>): void;
    /**
     * Gets an error string describing the given |result| code.
     * @param {!GattResult} result
     * @private
     */
    private getErrorString_;
    /**
     * Called when the read button is pressed. Connects to the device and
     * retrieves the current value of the characteristic in the |service_id|
     * with id |characteristic_id|. If |descriptor_id| is defined,  the
     * descriptor value with |descriptor_id| is read instead.
     * @private
     */
    private readValue_;
    /**
     * Called when the write button is pressed. Connects to the device and
     * retrieves the current value of the characteristic in the
     * |service_id| with id |characteristic_id|. If |descriptor_id| is defined,
     * the descriptor value with |descriptor_id| is written instead.
     * @private
     */
    private writeValue_;
}
